#include "ft.h"


/* The predicate logic algorithm uses implication locking in the
following form: when an implication or equivalence is used, a barrier
is inserted in the implication list in the left premiss, before that
formula. When a barrier is encountered in the course of using the
formulas in the implication list, no further formula in the list is
used. When a new formula is transferred, the first barrier found is
removed from the implication list. (The implication list will in
general contain several barriers; IMPS and IFFS are treated as though
they formed one list with implications before equivalences.) There is
nothing corresponding to the transfer checking by equalfmla() in the
propositional algorithm: transfer checking is always by samepred(),
which is a very strict equality.


  The algorithm incorporates the following types of compaction: oror,
orsome, somesome, someor, (these on the right side), allall, allimp,
alliff (on the left side). The orflag and eflag are used for
implementing both locking and right compaction in accordance with the
rules stated below. The bracketed conditions for clearing the flags are
not in fact checked.

  ORFLAG:
  
   ->LOCK
     if orlock: at A=> and right premiss of ->=>, <->=>

   ->CLEAR
     [if orlock:] at |=> and left premiss of ->=>, <->=>

  
   ->SPLIT
     if oror: at =>|
     if someor: at =>E

   ->CLEAR
     [if oror or someor:] at =>->,=><->,=>&,=>A
     if [oror and] not someor: at =>E
     if [someor and] not oror: at =>|



  EFLAG: 

   ->LOCK
     if somelock: at A=> and right premiss of ->=>, <->=>

   ->CLEAR
     [if somelock:] at |=> and and E=> and left premiss of ->=>, <->=>

  
   ->SPLIT
     if somesome: at =>E
     if orsome: at =>|

   ->CLEAR
     [if somesome or orsome:] at =>->,=><->,=>&,=>A
     if [orsome and] not somesome: at =>E
     if [somesome and] not orsome: at =>|


  Two forms of sifting have been implemented: top level sifting and
two-premiss sifting. Top level sifting means that any answer bindings
given are saved, and further bindings obtained are checked against the
saved bindings before being presented as answers. Instances of saved
bindings are not presented. (If bit 0 is set in the variable tellme, a
dot is printed for each top level solution skipped. If bit 1 is set, the
number of solutions skipped at top level is reported.)

 Two-premiss sifting applies to two-premiss rules, i.e. =>&, |=>, ->=>, <->=> -
in the case of the two latter rules also when they are invoked in the course
of allimp or alliff compaction.

There are two principles of two-premiss sifting:

   i) In proving (doing or redoing) G1&G2, we save solutions of G1 as
"successes" or "failures", depending on whether or not G2 succeeds.
On backtracking to G1, we reject a solution if it is an instance of a
previous success, or an instance wrt the variables in G2 of a previous
failure.

   ii) If we get a solution of G1 which does not bind any variables in
G2, and G2 then fails, we don't backtrack to G1, but reject G1&G2.

   Note that if we have a solution of G1 which does not bind any
variables in G2, and G2 succeeds, we must still backtrack to G1 when
G2 fails on backtracking.

   In the case ii) we also utilize semi-invertibility (as in the
propositional algorithm): if we prove A->B,G=>A without binding any
variables in B,G=>C, and then fail to prove B,G=>C, we reject
A->B,G=>C. This too applies in allimp and alliff compaction.

   There is also a check of whether G1 was proved without binding any
variables at all. If so, the redoflag of G1 is set to prevent
backtracking into G1.


  SOME NOTES ON THE IMPLEMENTATION

  Backtracking in the algorithm is achieved by ad hoc means as
follows.  There are two main functions, doseq() and redo(). Each
operates on a frame, i.e. a consecutive sequence of pointers and flags
which represents an annotated sequent. doseq(p) is called when the proof of the
sequent p is first attempted. If doseq(p) succeeds, information
concerning how the proof succeeded is inserted in the frame p before
success is returned to the calling function. This information is given
in one or more of WORKF, REDOFLAG, PREM1, PREM2. For example, if the
sequent was proved by using an implication, REDOFLAG is set to IMPRD,
WORKF is set to point to the next implication to use, and PREM1 and
PREM2 are set to point to the premisses. Backtracking is then achieved
by calling redo(p).

  Frames are built on the main stack, which is where most of the action
takes place. There is also the formula stack, described below. The values
of the main stack pointer and formula stack pointer are saved in FSTACK
and FFSTACK when doseq() is entered. The main stack is reset to FSTACK(p)
when doseq(p) fails, when redo(p) fails, and on some other occasions.

  The formula stack is needed because of the destructive rewriting used
to implement most of the one-premiss invertible rules and the reshuffling
of formulas. The results of such rewriting must not be wiped out on
backtracking. Probably the trickiest aspect of the ad hoc routines below
is the question when it is admissible to reset the formula stack to FFSTACK.

  Two-premiss rules are the sensitive area.  A brief mnemonic: when we
prove G1&G2, the main stack looks like this:
                 
               G1&G2,G1,G2,..g1..,..*,g2..., 

where g1 is a proof of G1 and g2 a proof of G2. We must construct the
premisses G1,G2 before starting the proof of G1. When something in G1
fails we will cut away the stuff after *, but not G2. However, if we
want to rewrite G2 we can't store the new stuff (formulas, cells)
after *. So these new formulas are created on the formula stack. Thus
we may have stuff r(G2),r(G3) on the formula stack, where G3 is some
sequent created in the course of redoing G1.  We must not cut away
this when redo(G2) fails; hence we can't restore the formula stack on
redo failure. When do(G2) fails, however, it is ok to cut at
FFSTACK(G2), for in that case FFSTACK(G2) will point below r(G3).
Several functions under doseq() below can in fact also fail when
called by redo(): useuniv, useimp, useiff, doconor2, rotate: hence we
cannot reset the formula stack when these fail in the course of a
redo(). Since we need to cut the formula stack whenever possible, a
second argument - FORTH or BACK - has been added to doseq() and its
subfunctions, to allow them to decide whether or not it is ok to cut
the formula stack on failure.

  There is some use of the formula stack where it is probably not
necessary - in particular in the immediate treatment of strings of
existential quantifiers in useuniv() - and also it may be possible to
shrink the formula stack at some further points.

  Besides the main stack and the formula stack, prove() uses stack 4
to store top level solutions for top level sifting. Also, the
unification and instance checking make use of stack 3. (These two
latter do not use the main stack.) When two-premiss sifting is active,
blocks[] is also used, but this area is unproblematic in that it is
not used for any other purpose.


 the cells in UNIVSA, UNIVSD, IMPS and IFFS have the following form:

   cell[0]         pointer to formula
   cell[1]         contraction value
   cell[2]         pointer to next cell


  The flag SFLAG(p) is set to true to indicate that p has been proved
  without binding any variables - thus there is no saved solution of p.
  If bindings have been made, SFLAG(p) is set to false. SPOINT(p) - the
  pointer into the save stack of p - is NULL in two circumstances: when
  there is no save stack, and when saving on the save stack has ceased
  because of lack of space. SPOINT(p) is checked before calling marktrue()
  or markfalse(). savesolution(p) also checks that SPOINT(p) is non-NULL,
  and also sets it to NULL if the copy routines report error status. 

  Detailed comments on the sifting operations are given i connection with
  doantor() below. */



/*  ORFLAG and EFLAG values */

#define CLEAR 0L
#define LOCK 1L
#define SPLIT 2L


/* REDOFLAG values */

#define NOBACK 0L
#define ORLEFT 1L
#define ORRIGHT 2L
#define IMPRD 3L
#define IFFRD 4L
#define SECOND 5L
#define SOMERD 6L
#define ALLRD 7L
#define ALLIMPRD 8L
#define ALLIFFRD 9L
#define UNIFY1 10L
#define UNIFY2 11L
#define SPECIAL 12L




/********************** top level and initialization **********************/


long uconuse;       /* the working default contraction parameters         */
long iconuse;
long iffconuse;


frame initframe;    /* initial proof frame                                */
list savelist;      /* where (copies of) the top level answers are stored */
long *zerocon;      /* where the addresses of non-annotated ->, <->, A    */
                    /* in the baseformula are stored                      */
long skipped;       /* the number of skipped top level solutions          */
long *sp3base;      /* saved in prove()                                   */
long new;           /* flag used by getanswer()                           */


     
prove(q)
     formula q;
{
  long readtime();
  readtime();       /* ignore the time taken to parse the formula */
  uconuse=ucon;
  iconuse=icon;
  iffconuse=iffcon;
  zerocon=sp;
  fixconts((basefmla)q[0]); /* insert default contraction parameters */
  push(0L);                 /* indicating end of fixcont addresses */
  savelist=(list)NULL;
  initframe=sp;
  skipped=0;
  sp3base=sp3;
  setup(q);                 /* make initial frame */
  if (twosift)              /* clean up blocks used for internal sifting */
    clearblocks();
  return proveloop(q);
}



proveloop(q)
     formula q;
{
  int i,j;
  long time,readclock(),starttime;
#ifdef DEBUG
  if (trace) mess("\nproveloop");
#endif
  reportcont(); /* print present default contraction parameters */
  new=1;        /* we are seeking our first top level solution  */
  turnoffalarm();
  if (timeout)
    setalarm(timeout);
  starttime=readclock();
  while(1)
    {
      i=getanswer(q);
      time=starttime-readclock(); /* accumulated time */
      if (i==ABORT)
	{
	  push(time);
	  return i;
	}
      if (i<0)
	return i;
      if (i==0)      /* no (more,new) solutions */
        {
	  if (tellme&2)
	    {
	      if (skipped>1)
		printf("\n%d solutions skipped\n",skipped);
              if (skipped==1)
		printf("\n1 solution skipped\n");
	    }
	  skipped=0;
	  printf("\nno\n");
	  printf("%ld\n",time);  /* time includes top level sifting */
	  if (stuse)
	    reportstacks();
	  return 1;
        }
      if (topsift)
        {
	  if (isold())     /* this is an instance of an old solution */
	    {
	      skipped++;
	      if (tellme&1)
		mess(".");
	      continue;
	    }
	  if ((j=saveanswer())<0) /* stack 4 overflow - not likely */
	    return j;
        }
      showanswer();
      printf("\n%ld\n",time);
      if (stuse)
	reportstacks();
      if (infree[0]&&moreanswers())
	{
	  turnoffalarm();
	  if (timeout)
	    setalarm(timeout);
	  starttime=readclock(); /* timing starts over */
	  continue;
	}
      return 1;
    }
}



getanswer(q)
     formula q;
{
  int i;
#ifdef DEBUG
  if (trace) mess("\ngetanswer");
#endif
  while(1)
    {
      if (pcheck&&new)
	{
	  i=propcheck(initframe);
	  if(i<=0)
	    return i;
	}
      if (new)
        {
	  i=doseq(initframe,FORTH);
	  new=0;
        }
      else
	i=redo(initframe);
      if (i||forever==0||zerocon[0]==0L)
	return i;
      uconuse+=ustep;  /* deepen contraction and try again */
      iconuse+=istep;
      iffconuse+=iffstep;
      fsp=(long *)FFSTACK(initframe);
      sp=initframe;
      sp3=sp3base;
      refixconts();
      setup(q);
      new=1;
      if (twosift)
	clearblocks();
      reportcont();
    }
}


showanswer()
{
  if (tellme&2)
    {
      if (skipped>1)
	printf("\n%d solutions skipped\n",skipped);
      if (skipped==1)
	printf("\n1 solution skipped\n");
      skipped=0;
    }
  if (infree[0]==0L)
    printf("\nyes\n");
  else
    printvars();
}

moreanswers()
{
  int i,j,count;
  putchar('\n');
  while(1)
    {
      j=0;
      count=0;
      printf("More? (y/n) ");
      while((i=getchar())!='\n')
	{
	  j=i;
	  count++;
	}
      if (count>1)
	continue;
      if (j=='y')
	return 1;
#ifdef DEBUG
      if (j=='Y')
	{
	  trace=(dflag&2)?2:1;
	  return 1;
	}
#endif
      if (j=='n')
	return 0;
    }
}


reportcont()
{
  printf("\n%ld %ld %ld\n",iconuse,iffconuse,uconuse);
}


/* fixconts(q) goes through the base formula q, inserting default contraction
   values where none were supplied in the input (marked by 0 in the input).
   The addresses of the base formulas involved - implications, equivalences,
   universal formulas - are saved on the stack, for later use by refixconts().
   The value returned is the number of non-annotated implications,
   equivalences, and universal formulas in the input. */

fixconts(q)
     basefmla q;
{
  int i=0;
  switch((int)q[0])
    {
    case AND: return fixconts((basefmla)q[1])+fixconts((basefmla)q[2]);
    case OR:  return fixconts((basefmla)q[1])+fixconts((basefmla)q[2]);
    case IMP:
      {
	if (q[1]==0L)
	  {
	    q[1]=iconuse;
	    push((long)q);
	    i=1;
	  }
	return fixconts((basefmla)q[2])+fixconts((basefmla)q[3])+i;
      }
    case IFF:
      {
	if (q[1]==0L)
	  {
	    q[1]=iffconuse;
	    push((long)q);
	    i=1;
	  }
	return fixconts((basefmla)q[2])+fixconts((basefmla)q[3])+i;
      }
    case ALL:
      {
	if (q[1]==0)
	  {
	    q[1]=uconuse;
	    push((long)q);
	    i=1;
	  }
	return fixconts((basefmla)q[3])+i;
      }
    case SOME: return fixconts((basefmla)q[2]);
    default: return 0;
    }
}


/* refixconts() updates the contraction values of those subformulas
   of the input the addresses of which are stacked at zerocon. */


refixconts()
{
  long *adpoint;
  basefmla r;
#ifdef DEBUG
  if (trace) mess("\nrefixconts");
#endif
  for(adpoint=zerocon;adpoint[0];adpoint++)
    {
      r=(basefmla)adpoint[0];
      switch((int)r[0])
        {
        case IMP: r[1]=iconuse;break;
        case IFF: r[1]=iffconuse;break;
        case ALL: r[1]=uconuse;break;
        default: mess("\nNonsense value for fixconts");
        }
    }
}


setup(q)  /* make initial frame on stack */
     formula q;
{
  ANT(sp)=NULL;
  CONS(sp)=(long)q;
  IMPS(sp)=NULL;
  IFFS(sp)=NULL;
  ATOMS(sp)=NULL;
  TRANS(sp)=NULL;
  ORFLAG(sp)=CLEAR;
  EFLAG(sp)=CLEAR;
  UNIVSA(sp)=NULL;
  UNIVSD(sp)=NULL;
  PRM(sp)=0L;
  FFSTACK(sp)=(long)fsp;
  sp+=FRAME;
}

/* Note: we need not bother with the remaining items in the frame,
   which are set by a redo or doseq function before any possible
   use. FFSTACK is a special case here, since fsp may be reset from
   FFSTACK(initframe) in getanswer() whether or not a
   rewriting of the initial frame has been performed. */




/*************************** proving ****************************/



doseq(q,flag)
     frame q;
     int flag;
{
  formula r;
  basefmla rbase;
  list cell;
  int j;
#ifdef DEBUG
  if (trace)
    {
      printf("\ndoseq %ld:\n",(long)q);
      if (trace>1)
	printseq(q);
      else
	printfmla2((formula)CONS(q));
      fflush(stdout);
    }
#endif
  if (aflag)
    return TIMEOUT;
  if (cflag)
     {
       cflag=0;
       return ABORT;
     }
  if ((j=stackcheck())<0)    /* only checking of main stack     */
    return j;
  if ((j=fstackcheck())<0)   /* only checking of formula stack  */
    return j;
  if ((j=stack3check())<0)   /* only checking of variable stack */
    return j;
  FSTACK(q)=(long)sp;
  FFSTACK(q)=(long)fsp;
  if (ANT(q))
    {
      cell=(list)ANT(q);
      r=(formula)cell[0];
      rbase=(basefmla)r[0];
      switch((int)rbase[0])
        {
        case ALL: return doantall(q,flag);
        case SOME: return doantsome(q,flag);
        case AND: return doantand(q,flag);
        case OR:  return doantor(q,flag);
        case IMP: return doantimp(q,flag);
        case IFF: return doantiff(q,flag);
        case FALSUM:
	  {
	    REDOFLAG(q)=NOBACK;
	    return 1;          /* no minimal logic option */
	  }
        default: return doantatom(q,flag);
        }
    }
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];
  switch((int)rbase[0])
    {
    case ALL: return doconall(q,flag);
    case SOME: return doconsome(q,flag);
    case AND: return doconand(q,flag);
    case OR: return doconor(q,flag);
    case IMP: return implock?doconimpl(q,flag):doconimp(q,flag);
    case IFF: return doconiff(q,flag);
    case FALSUM: return useuniv(q,(list)UNIVSA(q),flag);
    default: return doconatom(q,flag);
    }
}


redo(q)
     frame q;
{
  int j;
#ifdef DEBUG
  if (trace)
    {
      printf("\nredo %ld:\n",(long)q);
      if (trace>1) printseq(q);
      else
	printfmla2((formula)CONS(q));
      fflush(stdout);
    }
#endif
  if (stuse)        /* to update counters */
    {
      if ((j=stackcheck())<0)
	return j;
      if ((j=stack3check())<0)
	return j;
    }
  if (aflag)
    return TIMEOUT;
  if (cflag)
    {
      cflag=0;
      return ABORT;
    }
  switch((int)REDOFLAG(q))
    {
    case NOBACK:
      {
	sp=(long *)FSTACK(q);
	return 0;
      }
    case ORLEFT: return orleftback(q);
    case ORRIGHT: return orrightback(q);
    case IMPRD: return imprdback(q);
    case IFFRD: return iffrdback(q);
    case SECOND: return secondback(q);
    case SOMERD: return somerdback(q);
    case ALLRD: return allrdback(q);
    case ALLIMPRD: return allimprdback(q);
    case ALLIFFRD: return alliffrdback(q);
    case UNIFY1: return unify1back(q);
    case UNIFY2: return unify2back(q);
    case SPECIAL: return specialback(q);
    default: return -101;
    }
}




      /********************** doant functions *******************/



/* A reminder: many lists are shared between different sequents. Thus
   we must not, as a general rule, make alterations to cells. The reason
   why IMPS, IFFS, UNIVSA, UNIVSD are copied when new frames are built
   is that changes are made in the contraction fields of those cells in
   the course of a proof. */


doantand(q,flag) /* rewrite */
     frame q;
     int flag;
{
  list cell;
  formula r;
  basefmla rbase;
#ifdef DEBUG
  if (trace) mess("\ndoantand");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];
  rbase=(basefmla)r[0];   /* AND A1 A2        */
  fsp[0]=rbase[1];        /* make A1          */
  fsp[1]=r[1];            /* same subtable    */
  fsp[2]=rbase[2];        /* make A2          */
  fsp[3]=r[1];
  fsp[4]=(long)fsp;       /* make two new cells on the formula stack */
  fsp[5]=(long)(fsp+6);
  fsp[6]=(long)(fsp+2);
  fsp[7]=cell[1];         /* link to old ANT, bypassing old first cell */
  ANT(q)=(long)(fsp+4);   /* install new ANT  */
  fsp+=8;
  return doseq(q,flag);
}


/* With two-premiss sifting active, we must compute VARS for the
  premisses in an application of |=> to A1|A2,G=>C.  This is done by
  first finding the variables that the two premisses have in common,
  i.e. the variables in G=>C and then adding the variables of A1 and
  A2 respectively. Similar manipulations in other cases. */


doantor(q,flag)
     frame q;
     int flag;
{
  list cell,copylist3();
  formula r;
  basefmla rbase;
  register frame p1,p2;
  int i,j;
  list addvars(),antorvars(),comvars;
#ifdef DEBUG
  if (trace) mess("\ndoantor");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];
  rbase=(basefmla)r[0];     /* OR A1 A2 */
  if (twosift)
    comvars=antorvars(q);   /* variables in formulas other than OR A1 A2 */
  /*********************/
  /* make left premiss */
  /*********************/
  sp[0]=rbase[1];         /* create formula A1           */
  sp[1]=r[1];
  sp[2]=(long)sp;         /* create cell                 */
  sp[3]=cell[1];          /* link to remainder of ANT(q) */
  sp+=4;
  p1=sp;                  /* now make new frame for first premiss  */
  sp+=FRAME;
  ANT(p1)=(long)(p1-2);
  CONS(p1)=CONS(q);
  IMPS(p1)=(long)copylist3((list)IMPS(q));
  IFFS(p1)=(long)copylist3((list)IFFS(q));
  ATOMS(p1)=ATOMS(q);
  TRANS(p1)=TRANS(q);
  ORFLAG(p1)=CLEAR;
  EFLAG(p1)=CLEAR;
  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
  PRM(p1)=PRM(q);
  if (twosift)
    VARS(p1)=(long)addvars((formula)(p1-4),comvars);
  /**********************/
  /* make right premiss */
  /**********************/
  sp[0]=rbase[2];    /* create formula A2 */
  sp[1]=r[1];
  sp[2]=(long)sp;    /* create cell */
  sp[3]=cell[1];     /* link to remainder of ANT(q) */
  sp+=4;
  p2=sp;             /* now make new frame */
  sp+=FRAME;
  ANT(p2)=(long)(p2-2);
  CONS(p2)=CONS(q);
  IMPS(p2)=(long)copylist3((list)IMPS(q));
  IFFS(p2)=(long)copylist3((list)IFFS(q));
  ATOMS(p2)=ATOMS(q);
  TRANS(p2)=TRANS(q);
  ORFLAG(p2)=CLEAR;
  EFLAG(p2)=CLEAR;
  UNIVSA(p2)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p2)=(long)copylist3((list)UNIVSD(q));
  PRM(p2)=PRM(q);
  if (twosift)
    VARS(p2)=(long)addvars((formula)(p2-4),comvars);
  /**********/
  /* proofs */
  /**********/
  if ((i=doseq(p1,flag))<=0)
    {
      fsp=(long *)FFSTACK(q);  /* ^ no need to check flag here */
      sp=(long *)FSTACK(q);
      return i;
    }
  /* sift case */
  if (twosift)
    return siftcase1(p1,p2,q,flag);
  /* no-sift case */
  while((j=doseq(p2,flag))<=0)
    {
      if (j<0)
	return j;
      if ((i=redo(p1))<=0)
	{
	  fsp=(long *)FFSTACK(q);
	  sp=(long *)FSTACK(q);
	  return i;
	}
    }
  PREM1(q)=(long)p1;
  PREM2(q)=(long)p2;
  REDOFLAG(q)=SECOND;
  return j;
}


siftcase1(p1,p2,q,flag)     /* antor and conand */
     frame p1,p2,q;
     int flag;
{
  int i,j;
  long *getblock();
  if (nobound((list)VARS(p1)))  /* p1 was proved without bindings */
    {
      SFLAG(p1)=1L;     /* Indicate that p1 was proved without bindings */
      SPOINT(p1)=NULL;  /* Indicate that p1 has no active save stack */
      REDOFLAG(p1)=NOBACK;     /* No need to redo p1 */
    }
  else
    {
      SFLAG(p1)=0L;     /* p1 was proved with bindings */
      if ((SAVE(p1)=(long)getblock())==NULL)  /* get a save stack */
	return ALLOCERR;
      SPOINT(p1)=SAVE(p1); /* initialize stack pointer */
      STORE(p1)=NULL;      /* initialize list of saved solutions */
      savesolution(p1);
    }
  while((j=doseq(p2,flag))<=0)  /* attempted proof of p2 fails */
    {
      if (j<0)
	return j;
      /* Check for the special case where p1 was proved without binding
	 any variables in p2. Note that this includes the case where
	 p1 was proved without binding any variables. */
      if (nobound((list)VARS(p2)))
	{
	  if (SFLAG(p1)==0L)    /* if p1 has a save stack, return it */
	    {
	      freeblock((long *)SAVE(p1));
	      SFLAG(p1)=1L;        /* ^ not necessary */
	      SPOINT(p1)=NULL;     /* ^ not necessary */
	    }
	  sp=(long *)FSTACK(q);
	  fsp=(long *)FFSTACK(q);
	  return 0;
	}
     /* if p1 has an active save stack mark the latest solution as a failure */
      if (SPOINT(p1))
	markfalse(p1);
     /* try p1 again until failure or error or a new solution is obtained */
      while((i=redo(p1))>0&&twoold(p1,p2))
	;
      if (i<=0)                            /* failure or error */
	{
	  if (SFLAG(p1)==0L) /* if p1 has a save stack, return it */
	    {               
	      freeblock((long *)SAVE(p1)); 
	      SFLAG(p1)=1L;                /* ^ not necessary */
	      SPOINT(p1)=NULL;             /* ^ not necessary */
	    }
	  sp=(long *)FSTACK(q);
	  fsp=(long *)FFSTACK(q);
	  return i;
	}
      /* we have a new solution of p1, so save it and try p2 again */
      savesolution(p1);
    }
  /* if p1 has an active save stack, mark the latest solution as a success */
  if (SPOINT(p1))
    marktrue(p1);
  PREM1(q)=(long)p1;
  PREM2(q)=(long)p2;
  REDOFLAG(q)=SECOND;
  return j;
}



doantimp(q,flag)    /* rewrite */
     frame q;
     int flag;
{
  list cell;
  formula r;
  basefmla rbase;
#ifdef DEBUG
  if (trace) mess("\ndoantimp");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];
  rbase=(basefmla)r[0]; /* IMP n A B */
  ANT(q)=cell[1];       /* chop away implication from ANT */
  fsp[0]=(long)r;        /* make 3-cell: formula */
  fsp[1]=rbase[1];       /* contraction value */
  fsp[2]=IMPS(q);        /* link to IMPS */
  IMPS(q)=(long)fsp;
  fsp+=3;
  return doseq(q,flag);
}


doantiff(q,flag)    /* rewrite */
     frame q;
     int flag;
{
  list cell;
  formula r;
  basefmla rbase;
#ifdef DEBUG
  if (trace) mess("\ndoantiff");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];
  rbase=(basefmla)r[0]; /* IFF n A B */
  ANT(q)=cell[1];       /* chop away iff from ANT */
  fsp[0]=(long)r;        /* make 3-cell: formula */
  fsp[1]=rbase[1];       /* contraction value */
  fsp[2]=IFFS(q);        /* link to IFFS */
  IFFS(q)=(long)fsp;
  fsp+=3;
  return doseq(q,flag);
}


doantall(q,flag)    /* rewrite */
     frame q;
     int flag;
{
  list cell;
  formula r;
  basefmla rbase;
#ifdef DEBUG
  if (trace) mess("\ndoantall");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];
  rbase=(basefmla)r[0];  /* ALL n x A */
  ANT(q)=cell[1];        /* chop away univ from ANT */
  fsp[0]=(long)r;         /* make 3-cell */
  fsp[1]=rbase[1];
  fsp[2]=UNIVSA(q);       /* link to active univs */
  UNIVSA(q)=(long)fsp;
  fsp+=3;
  return doseq(q,flag);
}



doantsome(q,flag)        /* rewrite */
     frame q;
     int flag;
{
  list cell;
  formula r;
  register long *rfsp=fsp;
  register long j;
  basefmla rbase;
  subtab rsub;
#ifdef DEBUG
  if (trace) mess("\ndoantsome");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];
  rbase=(basefmla)r[0];    /* existential: SOME x A(x) */
  rsub=(subtab)r[1];
  rfsp[0]=0L;        /* make parameter */
  rfsp[1]=PRM(q);
  rfsp[2]=rbase[2];    /* make substitution formula A(prm) */
  rfsp[3]=(long)(rfsp+4);  /* new subtable at fsp+4 */
  for(j=0;j<sublength;j++)
    rfsp[4+j]=rsub[j];     /* copy subtable */
  rfsp[4+rbase[1]]=(long)rfsp; /* do substitution */
  rfsp[4+sublength]=(long)(rfsp+2);  /* make new cell, just to be safe */
  rfsp[5+sublength]=cell[1];       /* link to rest of ANT */
  ANT(q)=(long)(rfsp+sublength+4);
  PRM(q)++;  /* adjust parameter value */
  fsp+=sublength+6;
  EFLAG(q)=CLEAR;
  return doseq(q,flag);
}


doantatom(q,flag)        /* possible rewrite */
     frame q;
     int flag;
{
  list cell,bind;
  formula r,consf;
  int i;
  basefmla consbase;
  subtab conssub;
  long *sp3sav=sp3;
#ifdef DEBUG
  if (trace) mess("\ndoantatom");
#endif
  cell=(list)ANT(q);
  r=(formula)cell[0];   /* atomic, not falsum */   
  consf=(formula)CONS(q);
  consbase=(basefmla)consf[0];
  if (consbase[0]>SOME) /* consequent is atomic */
    {
      conssub=(subtab)consf[1];
      bind=(list)NULL;
      i=unify((basefmla)r[0],(subtab)r[1],consbase,conssub,&bind);
      if (i)
        {
          REDOFLAG(q)=UNIFY1;
          PREM1(q)=(long)bind;     /* bindings saved here */
          PREM2(q)=(long)sp3sav;   /* to restore it on unbinding */
          return i;
        }
      unbind(bind);
      sp3=sp3sav;               /* note: unify() does not use the main stack */
    }
  /* no axiom, so just move the formula to ATOMS */
  ANT(q)=cell[1];       /* cut away first cell from ANT */
  fsp[0]=cell[0];       /* make new cell for first formula on formula stack */
  fsp[1]=ATOMS(q);      /* append it to ATOMS */
  ATOMS(q)=(long)fsp;   
  fsp+=2;
  return doseq(q,flag);
}



/*************************** docon functions ***********************/



doconand(q,flag)
     frame q;
     int flag;
{
  int i,j;
  formula r;
  register frame p1,p2;
  basefmla rbase;
  subtab rsub;
  list copylist3();
  list addvars(),conandvars(),comvars;
#ifdef DEBUG
  if (trace) mess("\ndoconand");
#endif
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* AND A1 A2 */
  rsub=(subtab)r[1];
  if (twosift)
    comvars=conandvars(q);
  /**********************/
  /* make first premiss */
  /**********************/
  sp[0]=rbase[1];   /* create formula A1 */
  sp[1]=(long)rsub;
  sp+=2;
  p1=sp;
  sp+=FRAME;
  ANT(p1)=ANT(q);
  CONS(p1)=(long)(p1-2);
  IMPS(p1)=(long)copylist3((list)IMPS(q));
  IFFS(p1)=(long)copylist3((list)IFFS(q));
  ATOMS(p1)=ATOMS(q);
  TRANS(p1)=TRANS(q);
  ORFLAG(p1)=CLEAR;
  EFLAG(p1)=CLEAR;
  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
  PRM(p1)=PRM(q);
  if (twosift)
    VARS(p1)=(long)addvars((formula)CONS(p1),comvars);
  /***********************/
  /* make second premiss */
  /***********************/
  sp[0]=rbase[2];   /* create formula A2 */
  sp[1]=(long)rsub;
  sp+=2;
  p2=sp;
  sp+=FRAME;
  ANT(p2)=ANT(q);
  CONS(p2)=(long)(p2-2);
  IMPS(p2)=(long)copylist3((list)IMPS(q));
  IFFS(p2)=(long)copylist3((list)IFFS(q));
  ATOMS(p2)=ATOMS(q);
  TRANS(p2)=TRANS(q);
  ORFLAG(p2)=CLEAR;
  EFLAG(p2)=CLEAR;
  UNIVSA(p2)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p2)=(long)copylist3((list)UNIVSD(q));
  PRM(p2)=PRM(q);
  if (twosift)
    VARS(p2)=(long)addvars((formula)CONS(p2),comvars);
  /**********/
  /* proofs */
  /**********/
  if ((i=doseq(p1,flag))<=0)
    {
      fsp=(long *)FFSTACK(q);             /* ^ ignore flag */
      sp=(long *)FSTACK(q);
      return i;
    }
  /* sift case */
  if (twosift)
    return siftcase1(p1,p2,q,flag);
  /* no-sift case */
   while((j=doseq(p2,flag))<=0)
     {
       if (j<0)
	 return j;
       if ((i=redo(p1))<=0)
	 {
	   fsp=(long *)FFSTACK(q);
	   sp=(long *)FSTACK(q);
	   return i;
	 }
     }
  PREM1(q)=(long)p1;
  PREM2(q)=(long)p2;
  REDOFLAG(q)=SECOND;
  return j;
 }


doconimp(q,flag)       /* rewrite - no implocking */
     frame q;
     int flag;
{
  formula r;
  basefmla rbase;
  subtab rsub;
  register long *rfsp=fsp;
#ifdef DEBUG
  if (trace) mess("\ndoconimp");
#endif
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];        /* IMP n A1 A2 */
  rsub=(subtab)r[1];
  rfsp[0]=rbase[2];      /* make A1 */
  rfsp[1]=(long)rsub;
  rfsp[2]=rbase[3];     /* make A2 */
  rfsp[3]=(long)rsub;
  rfsp[4]=(long)rfsp;    /* make new cell for A1 */
  rfsp[5]=NULL;
  ANT(q)=(long)(rfsp+4);
  CONS(q)=(long)(rfsp+2);
  fsp+=6;
  return doseq(q,flag);
}




/* doconimpl() cannot be treated as a rewrite because of the implication
   locking: the treatment of G=>A->B varies with the current bindings.
   Hence the present treatment. */

doconimpl(q,flag)
     frame q;
     int flag;
{
  formula r;
  basefmla rbase;
  subtab rsub;
  register frame p1;
  int i;
  list copylist3();
#ifdef DEBUG
  if (trace) mess("\ndoconimpl");
#endif
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];        /* IMP n A1 A2 */
  rsub=(subtab)r[1];
  sp[0]=rbase[2];      /* make A1 */
  sp[1]=(long)rsub;
  sp[2]=rbase[3];     /* make A2 */
  sp[3]=(long)rsub;
  sp+=4;
  p1=sp;
  sp+=FRAME;
  CONS(p1)=(long)(p1-2);
  IMPS(p1)=(long)copylist3((list)IMPS(q));
  IFFS(p1)=(long)copylist3((list)IFFS(q));
  ATOMS(p1)=ATOMS(q);
  ORFLAG(p1)=CLEAR;
  EFLAG(p1)=CLEAR;
  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
  PRM(p1)=PRM(q);
  /* the values of ANT,TRANS depend on the transcheck */
  if (transcheck(p1-4,(list)TRANS(q)))
    {
      ANT(p1)=ANT(q);
      TRANS(p1)=TRANS(q);
    }
  else
    {
      if (unbar((list)IMPS(p1))==0)    /* ^ check this out */
	unbar((list)IFFS(p1));
      sp[0]=(long)(p1-4);  /* make cell for A1 */
      sp[1]=ANT(q);  /* link it */
      ANT(p1)=(long)sp;
      sp[2]=(long)(p1-4);  /* also add A1 to TRANS */
      sp[3]=TRANS(q);
      TRANS(p1)=(long)(sp+2);
      sp+=4;
    }
  PREM1(q)=(long)p1;
  REDOFLAG(q)=SPECIAL;
  if (i= doseq(p1,flag))
    return i;
  sp=(long *)FSTACK(q);
  fsp=(long *)FFSTACK(q);
  return 0;
}



doconiff(q,flag)      /* rewrite */
     frame q;
     int flag;
{
  formula r;
  basefmla rbase;
  subtab rsub;
  register long *rfsp=fsp;
#ifdef DEBUG
  if (trace) mess("\ndoconiff");
#endif
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];        /* IFF n A1 A2 */
  rsub=(subtab)r[1];
  rfsp[0]=IMP;     /* make A1->A2 */
  rfsp[1]=1L;      /* contraction value doesn't matter */
  rfsp[2]=rbase[2];
  rfsp[3]=rbase[3];
  rfsp[4]=IMP;    /* make A2->A1 */
  rfsp[5]=1L;
  rfsp[6]=rbase[3];
  rfsp[7]=rbase[2];
  rfsp[8]=AND;    /* make conjunction */
  rfsp[9]=(long)rfsp;
  rfsp[10]=(long)(rfsp+4);
  rfsp[11]=(long)(rfsp+8);     /* make formula = base+subtab */
  rfsp[12]=(long)rsub;
  CONS(q)=(long)(rfsp+11);
  fsp+=13;
  return doconand(q,flag);
}



doconall(q,flag)     /* rewrite */
     frame q;
     int flag;
{
  int j;
  formula r;
  basefmla rbase;
  subtab rsub;
#ifdef DEBUG
  if (trace) mess("\ndoconall");
#endif
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* ALL n x A */
  rsub=(subtab)r[1];
  fsp[0]=0L;        /* make new parameter */
  fsp[1]=PRM(q);
  fsp[2]=rbase[3];      /* make new formula A(prm) */
  fsp[3]=(long)(fsp+4);  /* new subtable at sp+4 */
  for(j=0;j<sublength;j++)
    fsp[4+j]=rsub[j];     /* copy subtable */
  fsp[4+rbase[2]]=(long)fsp; /* do substitution */
  CONS(q)=(long)(fsp+2);
  fsp+=sublength+5;
  ORFLAG(q)=CLEAR;
  EFLAG(q)=CLEAR;
  PRM(q)++;
  return doseq(q,flag);
}



doconor(q,flag)
     frame q;
     int flag;
{
  int i;
  formula r;
  basefmla rbase;
  subtab rsub;
  frame p1;
  list copylist3();
#ifdef DEBUG
  if (trace) mess("\ndoconor");
#endif
  if (ORFLAG(q)==LOCK)
    return useuniv(q,(list)UNIVSA(q),flag);
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* OR A1 A2 */
  rsub=(subtab)r[1];
  sp[0]=rbase[1];        /* make A1 */
  sp[1]=(long)rsub;
  sp+=2;
  p1=sp;
  sp+=FRAME;
  ANT(p1)=ANT(q);
  CONS(p1)=(long)(p1-2);
  IMPS(p1)=(long)copylist3((list)IMPS(q));
  IFFS(p1)=(long)copylist3((list)IFFS(q));
  ATOMS(p1)=ATOMS(q);
  TRANS(p1)=TRANS(q);
  ORFLAG(p1)=oror?SPLIT:CLEAR;
  EFLAG(p1)=orsome?SPLIT:CLEAR;
  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
  PRM(p1)=PRM(q);
  if (pcheck)
    {
      i=propcheck(p1);
      if (i<0)
	return i;
      if (i==0)
	{
	  sp=(long *)FSTACK(q);
	  return doconor2(q,flag);
	}
    }
  if (i=doseq(p1,flag))
    {
      PREM1(q)=(long)p1;
      REDOFLAG(q)=ORLEFT;
      return i;
    }
  sp=(long *)FSTACK(q);
  return doconor2(q,flag);
}



doconor2(q,flag)
     frame q;
     int flag;
{
  int i;
  formula r;
  basefmla rbase;
  subtab rsub;
  frame p2;
  list copylist3();
#ifdef DEBUG
  if (trace) mess("\ndoconor2");
#endif
  if (ORFLAG(q)==LOCK)
    return useuniv(q,(list)UNIVSA(q),flag);
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* OR A1 A2 */
  rsub=(subtab)r[1];  
  sp[0]=rbase[2];        /* make A2 */
  sp[1]=(long)rsub;
  sp+=2;
  p2=sp;
  sp+=FRAME;
  ANT(p2)=ANT(q);
  CONS(p2)=(long)(p2-2);
  IMPS(p2)=(long)copylist3((list)IMPS(q));
  IFFS(p2)=(long)copylist3((list)IFFS(q));
  ATOMS(p2)=ATOMS(q);
  TRANS(p2)=TRANS(q);
  ORFLAG(p2)=oror?SPLIT:CLEAR;
  EFLAG(p2)=orsome?SPLIT:CLEAR;
  UNIVSA(p2)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p2)=(long)copylist3((list)UNIVSD(q));
  PRM(p2)=PRM(q);
  if (pcheck)
    {
      i=propcheck(p2);
      if (i<0)
	return i;
      if (i==0)
	{
	  sp=(long *)FSTACK(q);
	  return useuniv(q,(list)UNIVSA(q),flag);
	}
    }
  if (i=doseq(p2,flag))
    {
      PREM2(q)=(long)p2;
      REDOFLAG(q)=ORRIGHT;
      return i;
    }
  sp=(long *)FSTACK(q);
  if (ORFLAG(q)==SPLIT)
    {
      if (flag==FORTH)
	fsp=(long *)FFSTACK(q);
      return 0;
    }
  return useuniv(q,(list)UNIVSA(q),flag);
}



doconsome(q,flag)
     frame q;
{
  int i,j;
  formula r;
  basefmla rbase;
  subtab rsub;
  frame p1;
  list copylist3();
  long *spsave;
#ifdef DEBUG
  if (trace) mess("\ndoconsome");
#endif
  if (EFLAG(q)==LOCK)
    return useuniv(q,(list)UNIVSA(q),flag);
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* SOME x A(x) */
  rsub=(subtab)r[1];
  spsave=sp;
  sp[1]=(long)(sp+2);     /* pointer to substitution table */
  for (j=0;j<sublength;j++)
    sp[j+2]=rsub[j];     /* new subtable at spsave+2 */
  sp+=sublength+2;
  while(rbase[0]==SOME)
    {
      sp[0]=PRM(q)+1;  /* make new variable */
      sp[1]=NULL;
      (spsave+2)[rbase[1]]=(long)sp;    /* make substitution */
      sp+=2;
      if (somesome==0)
	{
	  spsave[0]=rbase[2];
	  break;
	}
      rbase=(basefmla)rbase[2];
      spsave[0]=(long)rbase;
    }
  p1=sp;
  sp+=FRAME;
  ANT(p1)=ANT(q);
  CONS(p1)=(long)spsave;
  IMPS(p1)=(long)copylist3((list)IMPS(q));
  IFFS(p1)=(long)copylist3((list)IFFS(q));
  ATOMS(p1)=ATOMS(q);
  TRANS(p1)=TRANS(q);
  ORFLAG(p1)=someor?SPLIT:CLEAR;
  EFLAG(p1)=somesome?SPLIT:CLEAR;
  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
  PRM(p1)=PRM(q);
  if (i=doseq(p1,flag))
    {
      PREM1(q)=(long)p1;
      REDOFLAG(q)=SOMERD;
      return i;
    }
  sp=(long *)FSTACK(q);
  if (EFLAG(q)==SPLIT)
    {
      fsp=(long *)FFSTACK(q);
      return 0;
    }
  return useuniv(q,(list)UNIVSA(q),flag);
}



doconatom(q,flag)
     frame q;
     int flag;
{
  formula r,s;
  basefmla rbase,sbase;
  subtab rsub,ssub;
  list cell,bind;
  long *sp3sav=sp3;
  int i;
#ifdef DEBUG
  if (trace) mess("\ndoconatom");
#endif
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* atomic, not falsum */
  rsub=(subtab)r[1];
  cell=(list)ATOMS(q);
  while(cell)
    {
      s=(formula)cell[0];
      sbase=(basefmla)s[0];
      ssub=(subtab)s[1];
      bind=(list)NULL;
      if (i=unify(rbase,rsub,sbase,ssub,&bind))
  	{
  	  PREM1(q)=(long)bind;     /* save bindings in PREM1 */
	  PREM2(q)=(long)sp3sav;   /* to restore on unbinding */
  	  REDOFLAG(q)=UNIFY2;
	  WORKF(q)=(long)cell;
	  return i;
  	}
      unbind(bind);
      sp3=sp3sav;
      cell=(list)cell[1];
    }
  return useuniv(q,(list)UNIVSA(q),flag);
}


useuniv(q,cell,flag)
     frame q;
     list cell;
     int flag;
{
  formula r;
  basefmla rbase;
  subtab rsub;
  long k,contsave;
  int i,j;
  register frame p1,p2;
  list addvars(),allimpvars(),alliffvars(),comvars,copylist3();
  int eclear=0;
  long *getblock();
#ifdef DEBUG
  if (trace) mess("\nuseuniv");
#endif
  while(cell)        /* formula contraction next */
    {
      if (cell[1]==0L)     /* contraction 0 - don't use it */
	{
	  cell=(list)cell[2];
	  continue;
	}
      r=(formula)cell[0];
      rbase=(basefmla)r[0];    /* ALL n x A */
      rsub=(subtab)r[1];
      /* first copy subtable for later substitutions */
      for(k=sublength-1;k>=0;k--)
	sp[k]=rsub[k];
      rsub=sp;
      sp+=sublength;
      while(rbase[0]==ALL)   
	{
	  /* make new variable */
	  sp[0]=PRM(q)+1;
	  sp[1]=NULL;
	  /* substitute */
	  rsub[rbase[2]]=(long)sp;
	  sp+=2;
	  rbase=(basefmla)rbase[3];
	  if (allall==0) break;
	}
      while(rbase[0]==SOME)       /* ^ formula stack not needed */
	{
	  /* make new parameter */
	  fsp[0]=0L;
	  fsp[1]=PRM(q);
	  /* substitute */
	  rsub[rbase[1]]=(long)fsp;
	  fsp+=2;
	  PRM(q)++;
	  eclear=1;
	  rbase=(basefmla)rbase[2];
	}
      FSTACK(q)=(long)sp;   /*  ^ these two probably not needed */
      FFSTACK(q)=(long)fsp;
      /*************** ALLIMP **************/
      if (rbase[0]==IMP && allimp)  /* IMP n C D */
	{
	  if (twosift)
	    comvars=allimpvars(q,cell,rbase,rsub);
	  /**********************/
	  /* build left premiss */
	  /**********************/
	  p1=sp;
	  sp+=FRAME;
	  ANT(p1)=NULL;
	  sp[0]=rbase[2];        /* make C */
	  sp[1]=(long)rsub;
	  CONS(p1)=(long)sp;
	  sp+=2;
	  IMPS(p1)=(long)copylist3((list)IMPS(q));
	  /* Note: we include C imp D even if it doesn't have any
	     contractions left, the reason being that we need the
	     barrier annotation. */
	  sp[0]=(long)rbase;       /* make C imp D */
	  sp[1]=(long)rsub;
	  sp[2]=(long)sp;           /* make cell for IMPS */
	  if (implock)
	    sp[3]=rbase[1]-1-BARVAL;   /* insert barrier */
	  else
	    sp[3]=rbase[1]-1;
	  sp[4]=(long)IMPS(p1);
	  IMPS(p1)=(long)(sp+2);
	  sp+=5;
	  IFFS(p1)=(long)copylist3((list)IFFS(q));
	  ATOMS(p1)=ATOMS(q);
	  TRANS(p1)=TRANS(q);
	  ORFLAG(p1)=CLEAR;
	  EFLAG(p1)=CLEAR;
	  contsave=cell[1];
	  cell[1]=0L;       /* remove working univ from active list */
	  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
	  cell[1]=contsave;
	  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
	  if (contsave>1)
	    {
	      sp[0]=cell[0];
	      sp[1]=contsave-1;
	      sp[2]=(long)UNIVSD(p1);
	      UNIVSD(p1)=(long)sp;    /* add it to dormant list */
	      sp+=3;
	    }
	  PRM(p1)=PRM(q);
	  if (twosift)
            VARS(p1)=(long)addvars((formula)CONS(p1),comvars);
	  /***********************/
	  /* make second premiss */
	  /***********************/
	  p2=sp;
	  sp+=FRAME;
	  sp[0]=rbase[3];          /* make D */
	  sp[1]=(long)rsub;
	  sp[2]=(long)sp;          /* make cell */
	  sp[3]=NULL;
	  ANT(p2)=(long)(sp+2);
	  sp+=4;
	  CONS(p2)=CONS(q);
	  IMPS(p2)=(long)copylist3((list)IMPS(q));  /* ignore C imp D here */
	  IFFS(p2)=(long)copylist3((list)IFFS(q));
	  ATOMS(p2)=ATOMS(q);
	  TRANS(p2)=TRANS(q);
	  ORFLAG(p2)=orlock?LOCK:CLEAR;
          EFLAG(p2)=somelock?LOCK:CLEAR;
	  UNIVSA(p2)=(long)copylist3((list)UNIVSA(p1));
	  UNIVSD(p2)=(long)copylist3((list)UNIVSD(p1));
	  PRM(p2)=PRM(q);
	  if (twosift)
	    {
	      VARS(p2)=(long)addvars((formula)CONS(p2),comvars);
	      if (rbase[1]==1)  /* i.e. if D wasn't included in comvars */
		VARS(p2)=(long)addvars((formula)(p2+FRAME),(list)VARS(p2));
	    }
	  /**********/
	  /* proofs */
	  /**********/
      if (pcheck)
	{
	  i=propcheck(p1);
	  if (i<0)
	    return i;
	  if (i==0)
	    {
	      sp=(long *)FSTACK(q);
	      cell=(list)cell[2];
	      continue;
	    }
	}
	  if((i=doseq(p1,flag))==0)
	    {
	      sp=(long *)FSTACK(q);          /* ^ formula stack? */
	      if (allinv)
		return 0;
	      cell=(list)cell[2];
	      continue;
	    }
	  if (i<0)
	    return i;
	  /* sift case */
	  if (twosift)
	    {
	      if (nobound((list)VARS(p1)))
		{
		  SFLAG(p1)=1L;
		  SPOINT(p1)=NULL;
		  REDOFLAG(p1)=NOBACK;
		}
	      else
		{
		  SFLAG(p1)=0L;
		  if ((SAVE(p1)=(long)getblock())==NULL)
		    return ALLOCERR;
		  SPOINT(p1)=SAVE(p1);
		  STORE(p1)=NULL;
		  savesolution(p1);
		}
	      while((j=doseq(p2,flag))<=0)
		{
		  if (j<0)
		    return j;
		  if (nobound((list)VARS(p2))) /* semi-invertibility */
		    {
		      if (SFLAG(p1)==0L)
			{
			  freeblock((long *)SAVE(p1));
			  SFLAG(p1)=1L;
			  SPOINT(p1)=NULL;
			}
		      sp=(long *)FSTACK(q);
		      return 0;
		    }
		  if (SPOINT(p1))
		    markfalse(p1);
		  while((i=redo(p1))>0&&twoold(p1,p2))
		    ;
		  if (i<0)
		    return i;
		  if (i==0)
		    {
		      if (SFLAG(p1)==0L)
			{
			  freeblock((long *)SAVE(p1));
			  SFLAG(p1)=1L;
			  SPOINT(p1)=NULL;
			}
		      sp=(long *)FSTACK(q);
		      cell=(list)cell[2];
		      break;
		    }
		  savesolution(p1);
		}
	      if (j==0)
		continue;
	      if (SPOINT(p1))
		marktrue(p1);
	      WORKF(q)=(long)cell;
	      REDOFLAG(q)=ALLIMPRD;
	      PREM1(q)=(long)p1;
	      PREM2(q)=(long)p2;
	      return j;
	    }
	  /* no-sift case */
	  while ((j=doseq(p2,flag))==0)
	    {
	      if ((i=redo(p1))==0)
		{
		    sp=(long *)FSTACK(q);
		    if (allinv)
		      return 0;
		    cell=(list)cell[2];
		    break;
		  }
	      if (i<0)
		return i;
	    }
	  if (j==0) continue;   /* check for break case */
	  WORKF(q)=(long)cell;
	  REDOFLAG(q)=ALLIMPRD;
	  PREM1(q)=(long)p1;
	  PREM2(q)=(long)p2;
	  return j;
	}
      /************** ALLIFF ***************/
      if (rbase[0]==IFF && alliff)  /* IFF n C D */
	{
	  if (twosift)
	    comvars=alliffvars(q,cell,rbase,rsub);
	  /******************************/
	  /* build left premiss G==>C|D */
	  /******************************/
	  p1=sp;
	  sp+=FRAME;
	  ANT(p1)=NULL;
	  sp[0]=OR;       /* new base formula */
	  sp[1]=rbase[2];
	  sp[2]=rbase[3];
	  sp[3]=(long)sp;   /* new formula */
	  sp[4]=(long)rsub;
	  CONS(p1)=(long)(sp+3);
	  sp+=5;
	  IMPS(p1)=(long)copylist3((list)IMPS(q));
	  IFFS(p1)=(long)copylist3((list)IFFS(q));
	  sp[0]=(long)rbase;       /* make C iff D */
	  sp[1]=(long)rsub;
	  sp[2]=(long)sp;          /* make cell for IFFS */
	  if (implock)
	    sp[3]=rbase[1]-1-BARVAL;  /* insert barrier */
	  else
	    sp[3]=rbase[1]-1;
	  sp[4]=(long)IFFS(p1);
	  IFFS(p1)=(long)(sp+2);
	  sp+=5;
	  ATOMS(p1)=ATOMS(q);
	  TRANS(p1)=TRANS(q);
	  ORFLAG(p1)=SPLIT;
	  EFLAG(p1)=CLEAR;
	  contsave=cell[1];
	  cell[1]=0L;         /* remove working univ from active list */
	  UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
	  cell[1]=contsave;
	  UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
	  if (contsave>1)
	    {
	      sp[0]=cell[0];
	      sp[1]=contsave-1;
	      sp[2]=(long)UNIVSD(p1);
	      UNIVSD(p1)=(long)sp;     /* add it to dormant list */
	      sp+=3;
	    }
	  PRM(p1)=PRM(q);
	  if (twosift)
            VARS(p1)=(long)comvars;
	  /***********************/
	  /* make second premiss */
	  /***********************/
	  p2=sp;
	  sp+=FRAME;
	  sp[0]=rbase[2];     /* make C */
	  sp[1]=(long)rsub;
	  sp[2]=rbase[3];     /* make D */
	  sp[3]=(long)rsub;
	  sp[4]=(long)sp;     /* make list */
	  sp[5]=(long)(sp+6);
	  sp[6]=(long)(sp+2);
	  sp[7]=NULL;
	  ANT(p2)=(long)(sp+4);
	  sp+=8;
	  CONS(p2)=CONS(q);
	  IMPS(p2)=(long)copylist3((list)IMPS(q));
	  IFFS(p2)=(long)copylist3((list)IFFS(q));  /* ignore C iff D */
	  ATOMS(p2)=ATOMS(q);
	  TRANS(p2)=TRANS(q);
          ORFLAG(p2)=orlock?LOCK:CLEAR;
	  EFLAG(p2)=somelock?LOCK:CLEAR;
	  UNIVSA(p2)=(long)copylist3((list)UNIVSA(p1));
	  UNIVSD(p2)=(long)copylist3((list)UNIVSD(p1));
	  PRM(p2)=PRM(q);
	  if (twosift)
	    VARS(p2)=(long)addvars((formula)CONS(p2),comvars);
	  /**********/
	  /* proofs */
	  /**********/
      if (pcheck)
	{
	  i=propcheck(p1);
	  if (i<0)
	    return i;
	  if (i==0)
	    {
	      sp=(long *)FSTACK(q);
	      cell=(list)cell[2];
	      continue;
	    }
	}
	  if ((i=doseq(p1,flag))==0)
	    {
	      sp=(long *)FSTACK(q);
	      if (allinv)
		return 0;
	      cell=(list)cell[2];
	      continue;
	    }
	  if (i<0)
	    return i;
	  /* sift case */
	  if (twosift)
	    {
	      if (nobound((list)VARS(p1)))
		{
		  SFLAG(p1)=1L;
		  SPOINT(p1)=NULL;
		  REDOFLAG(p1)=NOBACK;
		}
	      else
		{
		  SFLAG(p1)=0L;
		  if ((SAVE(p1)=(long)getblock())==NULL)
		    return ALLOCERR;
		  SPOINT(p1)=SAVE(p1);
		  STORE(p1)=NULL;
		  savesolution(p1);
		}
	      while((j=doseq(p2,flag))<=0)
		{
		  if (j<0)
		    return j;
		  if (nobound((list)VARS(p2))) /* semi-invertibility */
		    {
		      if (SFLAG(p1)==0L)
			{
			  freeblock((long *)SAVE(p1));
			  SFLAG(p1)=1L;
			  SPOINT(p1)=NULL;
			}
		      sp=(long *)FSTACK(q);
		      return 0;
		    }
		  if (SPOINT(p1))
		    markfalse(p1);
		  while((i=redo(p1))>0&&twoold(p1,p2))
		    ;
		  if (i<0)
		    return i;
		  if (i==0)
		    {
		      if (SFLAG(p1)==0L)
			{
			  freeblock((long *)SAVE(p1));
			  SFLAG(p1)=1L;
			  SPOINT(p1)=NULL;
			}
		      sp=(long *)FSTACK(q);
		      cell=(list)cell[2];
		      break;
		    }
		  savesolution(p1);
		}
	      if (j==0)
		continue;
	      if (SPOINT(p1))
		marktrue(p1);
	      WORKF(q)=(long)cell;
	      REDOFLAG(q)=ALLIFFRD;
	      PREM1(q)=(long)p1;
	      PREM2(q)=(long)p2;
	      return j;
	    }
	  /* no-sift case */
	  while ((j=doseq(p2,flag))==0)
	    {
	      if ((i=redo(p1))==0)
		{
		  sp=(long *)FSTACK(q);
		  if (allinv)
		    return 0;
		  cell=(list)cell[2];
		  break;
		}
	      if (i<0)
		return i;
	    }
	  if (j==0) continue;   /* check for break case */
	  WORKF(q)=(long)cell;
	  REDOFLAG(q)=ALLIFFRD;
	  PREM1(q)=(long)p1;
	  PREM2(q)=(long)p2;
	  return j;
	}
      /* make premiss for non-immediate treatment */
      p1=sp;
      sp+=FRAME;
      sp[0]=(long)rbase;      /* make formula */
      sp[1]=(long)rsub;
      sp[2]=(long)sp;         /* make cell */
      sp[3]=NULL;
      ANT(p1)=(long)(sp+2);
      sp+=4;
      CONS(p1)=CONS(q);
      IMPS(p1)=(long)copylist3((list)IMPS(q));
      IFFS(p1)=(long)copylist3((list)IFFS(q));
      ATOMS(p1)=ATOMS(q);
      TRANS(p1)=TRANS(q);
      ORFLAG(p1)=orlock?LOCK:CLEAR;
      if (eclear)
	EFLAG(p1)=CLEAR;
      else
        EFLAG(p1)=somelock?LOCK:CLEAR;
      contsave=cell[1];
      cell[1]=0L;
      UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
      cell[1]=contsave;
      UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
      if (contsave>1)
	{
          sp[0]=cell[0];    /* make cell for UNIVSD */
          sp[1]=contsave-1;
          sp[2]=UNIVSD(p1);
          UNIVSD(p1)=(long)sp;
          sp+=3;
	}
      PRM(p1)=PRM(q);
      /* proof */
      if ((i=doseq(p1,flag))==0)
	{
	  sp=(long *)FSTACK(q);
	  if (allinv)
	    return 0;
	  cell=(list)cell[2];
	  continue;
	}
      WORKF(q)=(long)cell;
      REDOFLAG(q)=ALLRD;
      PREM1(q)=(long)p1;
      return i;
    }
  return useimp(q,(list)IMPS(q),flag);
}




useimp(q,cell,flag)
     frame q;
     list cell;
     int flag;
{
  int i,j;
  long contsave;
  basefmla rbase;
  subtab rsub;
  formula r;
  register frame p1,p2;
  list addvars(),useimpvars(),comvars,copylist3();
  long *getblock();
#ifdef DEBUG
  if (trace) mess("\nuseimp");
#endif
  while(cell)                          /* formula cont next */
    {
      if (cell[1]<0)
	return rotate(q,flag);         /* barrier encountered */
      if (cell[1]==0L)                 /* contraction 0: don't use this one */
	{
	  cell=(list)cell[2];
	  continue;
	}
      r=(formula)cell[0];
      rbase=(basefmla)r[0];           /* IMP n A B */
      rsub=(subtab)r[1];
      if (twosift)
	comvars=useimpvars(q,cell);
      /**********************/
      /* build left premiss */
      /**********************/
      sp[0]=rbase[2];                                  /* make A */
      sp[1]=(long)rsub;
      sp+=2;
      p1=sp;
      sp+=FRAME;
      ANT(p1)=NULL;
      CONS(p1)=(long)(p1-2);
      cell[1]--;
      if (implock)
	cell[1]-=BARVAL;   /* mark it as barrier */
      IMPS(p1)=(long)copylist3((list)IMPS(q));
      if (implock)
	cell[1]+=BARVAL+1;
      else
	cell[1]++;
      IFFS(p1)=(long)copylist3((list)IFFS(q));
      ATOMS(p1)=ATOMS(q);
      TRANS(p1)=TRANS(q);
      ORFLAG(p1)=CLEAR;
      EFLAG(p1)=CLEAR;
      UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
      UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
      PRM(p1)=PRM(q);
      if (twosift)
        VARS(p1)=(long)addvars((formula)CONS(p1),comvars);
      /***********************/
      /* build right premiss */
      /***********************/
      sp[0]=rbase[3];      /* make B */
      sp[1]=(long)rsub;
      sp[2]=(long)sp;      /* make B-cell */
      sp[3]=NULL;
      sp+=4;
      p2=sp;
      sp+=FRAME;
      ANT(p2)=(long)(p2-2);
      CONS(p2)=CONS(q);
      contsave=cell[1];
      cell[1]=0L;
      IMPS(p2)=(long)copylist3((list)IMPS(q));
      cell[1]=contsave;
      IFFS(p2)=(long)copylist3((list)IFFS(q));
      ATOMS(p2)=ATOMS(q);
      TRANS(p2)=TRANS(q);
      ORFLAG(p2)=orlock?LOCK:CLEAR;
      EFLAG(p2)=somelock?LOCK:CLEAR;
      UNIVSA(p2)=(long)copylist3((list)UNIVSA(q));
      UNIVSD(p2)=(long)copylist3((list)UNIVSD(q));
      PRM(p2)=PRM(q);
      if (twosift)
	{
	  VARS(p2)=(long)addvars((formula)CONS(p2),comvars);
	  if (contsave==1) /* i.e. if B is not included in comvars */
	    VARS(p2)=(long)addvars((formula)(p2-4),(list)VARS(p2));
	}
      /**********/
      /* proofs */
      /**********/
      if (pcheck)
	{
	  i=propcheck(p1);
	  if (i<0)
	    return i;
	  if (i==0)
	    {
	      sp=(long *)FSTACK(q);
	      cell=(list)cell[2];
	      continue;
	    }
	}
      if ((i=doseq(p1,flag))==0)
	{
	  sp=(long *)FSTACK(q);       /* ^ formula stack? */
	  cell=(list)cell[2];
	  continue;
	}
      if (i<0)
	return i;
      /* sift case */
      if (twosift)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      SFLAG(p1)=0L;
	      if ((SAVE(p1)=(long)getblock())==NULL)
		return ALLOCERR;
	      SPOINT(p1)=SAVE(p1);
	      STORE(p1)=NULL;
	      savesolution(p1);
	    }
	  while((j=doseq(p2,flag))<=0)
	    {
	      if (j<0)
		return j;
	      if (nobound((list)VARS(p2)))  /* semi-invertibility */
		{
		  if (SFLAG(p1)==0L)
		    {
		      freeblock((long *)SAVE(p1));
		      SFLAG(p1)=1L;
		      SPOINT(p1)=NULL;
		    }
		  sp=(long *)FSTACK(q);
		  if (flag==FORTH)
		    fsp=(long *)FFSTACK(q);
		  return 0;
		}
	      if (SPOINT(p1))
		markfalse(p1);
	      while((i=redo(p1))>0&&twoold(p1,p2))
		;
	      if (i<0)
		return i;
	      if (i==0)
		{
		  if (SFLAG(p1)==0L)
		    {
		      freeblock((long *)SAVE(p1));
		      SFLAG(p1)=1L;
		      SPOINT(p1)=NULL;
		    }
		  sp=(long *)FSTACK(q);       /* ^ fsp? */
		  cell=(list)cell[2];
		  break;
		}
	      savesolution(p1);
	    }
	  if (j==0)    /* check because of break above */
	    continue;
	  if (SPOINT(p1))
	    marktrue(p1);
	  WORKF(q)=(long)cell;
	  REDOFLAG(q)=IMPRD;
	  PREM1(q)=(long)p1;
	  PREM2(q)=(long)p2;
	  return j;
	}
      /* no-sift case */
      while((j=doseq(p2,flag))==0)
	{
	  if ((i=redo(p1))==0)
	    {
	      sp=(long *)FSTACK(q);
	      cell=(list)cell[2];
	      break;
	    }
	  if (i<0)
	    return i;
	}
      if (j==0) continue;  /* check needed because of break */
      WORKF(q)=(long)cell;
      REDOFLAG(q)=IMPRD;
      PREM1(q)=(long)p1;
      PREM2(q)=(long)p2;
      return j;
    }
  sp=(long *)FSTACK(q);
  return useiff(q,(list)IFFS(q),flag);
}


useiff(q,cell,flag)
     frame q;
     list cell;
     int flag;
{
  int i,j;
  long contsave;
  basefmla rbase;
  subtab rsub;
  formula r;
  register frame p1,p2;
  list addvars(),useiffvars(),comvars,copylist3();
  long *getblock();
#ifdef DEBUG
  if (trace) mess("\nuseiff");
#endif
  while(cell)
    {
      if (cell[1]<0)
	return rotate(q,flag);         /* barrier encountered */
      if (cell[1]==0L)
	{
	  cell=(list)cell[2];
	  continue;
	}
      r=(formula)cell[0];
      rbase=(basefmla)r[0];   /* IFF n A1 A2 */
      rsub=(subtab)r[1];
      if (twosift)
	comvars=useiffvars(q);
      /*********************************************************/
      /* build left premiss for A1|A2 with orflag set to SPLIT */
      /*********************************************************/
      sp[0]=OR;               /* new base formula */
      sp[1]=rbase[2];
      sp[2]=rbase[3];
      sp[3]=(long)sp;     /* new formula */
      sp[4]=(long)rsub;
      sp+=5;
      p1=sp;
      sp+=FRAME;
      ANT(p1)=NULL;
      CONS(p1)=(long)(p1-2);
      IMPS(p1)=(long)copylist3((list)IMPS(q));
      cell[1]--;
      if (implock)
	cell[1]-=BARVAL;  /* mark it as barrier */
      IFFS(p1)=(long)copylist3((list)IFFS(q));
      if (implock)
	cell[1]+=BARVAL+1;
      else
	cell[1]++;
      ATOMS(p1)=ATOMS(q);
      TRANS(p1)=TRANS(q);
      ORFLAG(p1)=SPLIT;
      EFLAG(p1)=CLEAR;
      UNIVSA(p1)=(long)copylist3((list)UNIVSA(q));
      UNIVSD(p1)=(long)copylist3((list)UNIVSD(q));
      PRM(p1)=PRM(q);
      if (twosift)
        VARS(p1)=(long)comvars;
      /***********************/
      /* build right premiss */
      /***********************/
      sp[0]=rbase[2];      /* make A1 */
      sp[1]=(long)rsub;
      sp[2]=rbase[3];      /* make A2 */
      sp[3]=(long)rsub;
      sp[4]=(long)sp;      /* make A1-cell */
      sp[5]=(long)(sp+6);   /* link to A2-cell */
      sp[6]=(long)(sp+2);
      sp[7]=NULL;
      sp+=8;
      p2=sp;
      sp+=FRAME;
      ANT(p2)=(long)(p2-4);
      CONS(p2)=CONS(q);
      IMPS(p2)=(long)copylist3((list)IMPS(q));
      contsave=cell[1];
      cell[1]=0L;
      IFFS(p2)=(long)copylist3((list)IFFS(q));
      cell[1]=contsave;
      ATOMS(p2)=ATOMS(q);
      TRANS(p2)=TRANS(q);
      ORFLAG(p2)=orlock?LOCK:CLEAR;
      EFLAG(p2)=somelock?LOCK:CLEAR;
      UNIVSA(p2)=(long)copylist3((list)UNIVSA(q));
      UNIVSD(p2)=(long)copylist3((list)UNIVSD(q));
      PRM(p2)=PRM(q);
      SAVE(p2)=NULL;
      if (twosift)
	VARS(p2)=(long)addvars((formula)CONS(p2),comvars);
      /**********/
      /* proofs */
      /**********/
      if (pcheck)
	{
	  i=propcheck(p1);
	  if (i<0)
	    return i;
	  if (i==0)
	    {
	      sp=(long *)FSTACK(q);
	      cell=(list)cell[2];
	      continue;
	    }
	}
      if ((i=doseq(p1,flag))==0)
	{
	  sp=(long *)FSTACK(q);       /* ^ fsp? */
	  cell=(list)cell[2];
	  continue;
	}
      if (i<0)
	return i;
      /* sift case */
      if (twosift)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      SFLAG(p1)=0L;
	      if ((SAVE(p1)=(long)getblock())==NULL)
		return ALLOCERR;
	      SPOINT(p1)=SAVE(p1);
	      STORE(p1)=NULL;
	      savesolution(p1);
	    }
	  while((j=doseq(p2,flag))<=0)
	    {
	      if (j<0)
		return j;
	      if (nobound((list)VARS(p2)))  /* semi-invertibility */
		{
		  if (SFLAG(p1)==0L)
		    {
		      freeblock((long *)SAVE(p1));
		      SFLAG(p1)=1L;
		      SPOINT(p1)=NULL;
		    }
		  sp=(long *)FSTACK(q);
		  if (flag==FORTH)
		    fsp=(long *)FFSTACK(q);
		  return 0;
		}
	      if (SPOINT(p1))
		markfalse(p1);
	      while((i=redo(p1))>0&&twoold(p1,p2))
		;
	      if (i<0)
		return i;
	      if (i==0)
		{
		  if (SFLAG(p1)==0L)
		    {
		      freeblock((long *)SAVE(p1));
		      SFLAG(p1)=1L;
		      SPOINT(p1)=NULL;
		    }
		  sp=(long *)FSTACK(q);       /* ^ fsp? */
		  cell=(list)cell[2];
		  break;
		}
	      savesolution(p1);
	    }
	  if (j==0)    /* check because of break above */
	    continue;
	  if (SPOINT(p1))
	    marktrue(p1);
	  WORKF(q)=(long)cell;
	  REDOFLAG(q)=IFFRD;
	  PREM1(q)=(long)p1;
	  PREM2(q)=(long)p2;
	  return j;
	}
      /* no-sift case */
      while((j=doseq(p2,flag))==0)
	{
	  if ((i=redo(p1))==0)
	    {
	      sp=(long *)FSTACK(q);
	      cell=(list)cell[2];
	      break;
	    }
	  if (i<0)
	    return i;
	}
      if (j==0) continue;    /* check needed because of break above */
      WORKF(q)=(long)cell;
      PREM1(q)=(long)p1;
      PREM2(q)=(long)p2;
      REDOFLAG(q)=IFFRD;
      return j;
    }
  sp=(long *)FSTACK(q);
  return rotate(q,flag);
}



/* rotate can't be treated as a rewrite since the rotation is carried out
   only as an alternative to trying each of the active univs, imps and 
   iffs. */

rotate(q,flag)
     frame q;
     int flag;
{
  list revapp3(),copylist3();
  frame p1;
  int i;
#ifdef DEBUG
  if (trace) mess("\nrotate");
#endif
  if (UNIVSD(q))   /* only still active univs are transferred to UNIVSD */
    {
      p1=sp;
      sp+=FRAME;
      ANT(p1)=ANT(q);
      CONS(p1)=CONS(q);
      IMPS(p1)=(long)copylist3((list)IMPS(q));
      IFFS(p1)=(long)copylist3((list)IFFS(q));
      ATOMS(p1)=ATOMS(q);
      TRANS(p1)=TRANS(q);
      ORFLAG(p1)=ORFLAG(q);
      EFLAG(p1)=EFLAG(q);
      UNIVSA(p1)=
	(long)revapp3(copylist3((list)UNIVSD(q)),copylist3((list)UNIVSA(q)));
      UNIVSD(p1)=NULL;
      PRM(p1)=PRM(q);
      SPOINT(p1)=SPOINT(q);
      STORE(p1)=STORE(q);
      PREM1(q)=(long)p1;
      REDOFLAG(q)=SPECIAL;
      if (i=doseq(p1,flag))
	return i;
    }
  sp=(long *)FSTACK(q);
  if (flag==FORTH)
    fsp=(long *)FFSTACK(q);
  return 0;
}



/************************* redo functions *********************************/


/* no resetting of fsp on redo failure */

secondback(q)
     frame q;
{
  int i,j;
  frame p1,p2;
  long *getblock();
#ifdef DEBUG
  if (trace) mess("\nsecondback");
#endif
  p1=(frame)PREM1(q);
  p2=(frame)PREM2(q);
  /* sift case */
  if (twosift)
    {
      if (i=redo(p2))
	return i;
      while(SFLAG(p1)==0L&&(i=redo(p1))>0)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      freeblock((long *)SAVE(p1));
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      if (twoold(p1,p2))
		continue;
	      savesolution(p1);
	    }
	  if (j=doseq(p2,BACK))
	    {
	      if (j<0)
		return j;
	      if (SPOINT(p1))
		marktrue(p1);
	      return j;
	    }
	  if (SPOINT(p1))
	    markfalse(p1);
	}
      if (i<0)
	return i;
      if (SFLAG(p1))
	{
	  sp=(long *)FSTACK(q);
	  return 0;
	}
      freeblock((long *)SAVE(p1));
      SFLAG(p1)=1L;        /* ^ remove? */
      SPOINT(p1)=NULL;
      sp=(long *)FSTACK(q);
      return i;
    }
  /* no-sift case */
  if (i=redo(p2))
    return i;
  while(i=redo(p1))
    {
      if (i<0)
	return i;
      if (j=doseq(p2,BACK))
	return j;
    }
  sp=(long *)FSTACK(q);
  return 0;
}



orleftback(q)
     frame q;
{
  int i;
#ifdef DEBUG
  if (trace) mess("\norleftback");
#endif
  if (i=redo((frame)PREM1(q)))
    return i;
  sp=(long *)FSTACK(q);
  return doconor2(q,BACK);
}



orrightback(q)
     frame q;
{
  int i;
#ifdef DEBUG
  if (trace) mess("\norrightback");
#endif
  if (i=redo((frame)PREM2(q)))
    return i;
  sp=(long *)FSTACK(q);
  if (ORFLAG(q)==SPLIT)
    {
      return 0;
    }
  return useuniv(q,(list)UNIVSA(q),BACK);
}



somerdback(q)
     frame q;
{
  long i;
#ifdef DEBUG
  if (trace) mess("\nsomerdback");
#endif
  if (i=redo((frame)PREM1(q)))
    return i;
  sp=(long *)FSTACK(q);
  if (EFLAG(q)==SPLIT)
    {
          return 0;
    }
  return useuniv(q,(list)UNIVSA(q),BACK);
}



unify1back(q)
     frame q;
{
  list cell;
#ifdef DEBUG
  if (trace) mess("\nunify1back");
#endif
  unbind((list)PREM1(q));
  sp3=(long *)PREM2(q);
  cell=(list)ANT(q);
  ANT(q)=cell[1];        /* cut away first cell from ANT */
  fsp[0]=cell[0];        /* make new cell for first formula - note use of fsp*/
  fsp[1]=ATOMS(q);       /* append it to ATOMS */
  ATOMS(q)=(long)fsp;
  fsp+=2;
  return doseq(q,BACK);
}


unify2back(q)
     frame q;
{
  formula r,s;
  basefmla rbase,sbase;
  subtab rsub,ssub;
  list cell,m;
  long *sp3save;
  int i;
#ifdef DEBUG
  if (trace) mess("\nunify2back");
#endif
  unbind((list)PREM1(q));
  sp3=(long *)PREM2(q);
  sp3save=sp3;
  r=(formula)CONS(q);
  rbase=(basefmla)r[0];   /* atomic */
  rsub=(subtab)r[1];
  cell=(list)(((list)WORKF(q))[1]);  /* next atomic formula */
  while(cell)
    {
      s=(formula)cell[0];
      sbase=(basefmla)s[0];
      ssub=(subtab)s[1];
      m=(list)NULL;
      if (i=unify(rbase,rsub,sbase,ssub,&m))
  	{
	  if (i<0)
	    return i;
  	  PREM1(q)=(long)m;     /* save bindings in PREM1 */
	  PREM2(q)=(long)sp3save;
  	  REDOFLAG(q)=UNIFY2;
  	  WORKF(q)=(long)cell;
  	  return 1;
  	}
      unbind(m);
      sp3=sp3save;
      cell=(list)cell[1];
    }
  return useuniv(q,(list)UNIVSA(q),BACK);
}


specialback(q)      /* this is a redo of impcon or rotate */
     frame q;
{
  int i;
#ifdef DEBUG
  if (trace) mess("\nspecialback");
#endif
  if(i=redo((frame)PREM1(q)))
    return i;
  sp=(long *)FSTACK(q);
  return 0;
}


imprdback(q)
     frame q;
{
  frame p1,p2;
  list cell;
  int i,j;
#ifdef DEBUG
  if (trace) mess("\nimprdback");
#endif
  cell=(list)WORKF(q);
  p1=(frame)PREM1(q);
  p2=(frame)PREM2(q);
  /* sift case */
  if (twosift)
    {
      if (i=redo(p2))
	return i;
      while(SFLAG(p1)==0L&&(i=redo(p1))>0)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      freeblock((long *)SAVE(p1));
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      if (twoold(p1,p2))
		continue;
	      savesolution(p1);
	    }
	  if (j=doseq(p2,BACK))
	    {
	      if (j<0)
		return j;
	      if (SPOINT(p1))
		marktrue(p1);
	      return j;
	    }
	  if (SPOINT(p1))
	    markfalse(p1);
	}
      if (i<0)
	return i;
      if (SFLAG(p1))
	{
	  sp=(long *)FSTACK(q);
	  return 0;
	}
      freeblock((long *)SAVE(p1));
      SFLAG(p1)=1L;
      SPOINT(p1)=NULL;
      sp=(long *)FSTACK(q);
      return useimp(q,(list)cell[2],BACK);
    }
  /* no-sift case */
  if (i=redo(p2))
    return i;
  while(i=redo(p1))
    {
      if (i<0)
	return i;
      if (j=doseq(p2,BACK))
	return j;
    }
  sp=(long *)FSTACK(q);
  return useimp(q,(list)cell[2],BACK);
}


iffrdback(q)
     frame q;
{
  frame p1,p2;
  list cell;
  int i,j;
#ifdef DEBUG
  if (trace) mess("\niffrdback");
#endif
  cell=(list)WORKF(q);
  p1=(frame)PREM1(q);
  p2=(frame)PREM2(q);
  /* sift case */
  if (twosift)
    {
      if (i=redo(p2))
	return i;
      while(SFLAG(p1)==0L&&(i=redo(p1))>0)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      freeblock((long *)SAVE(p1));
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      if (twoold(p1,p2))
		continue;
	      savesolution(p1);
	    }
	  if (j=doseq(p2,BACK))
	    {
	      if (j<0)
		return j;
	      if (SPOINT(p1))
		marktrue(p1);
	      return j;
	    }
	  if (SPOINT(p1))
	    markfalse(p1);
	}
      if (i<0)
	return i;
      if (SFLAG(p1))
	{
	  sp=(long *)FSTACK(q);
	  return 0;
	}
      freeblock((long *)SAVE(p1));
      SFLAG(p1)=1L;
      SPOINT(p1)=NULL;
      sp=(long *)FSTACK(q);
      return useiff(q,(list)cell[2],BACK);
    }
  /* no-sift case */
  if (i=redo(p2))
    return i;
  while(i=redo(p1))
    {
      if (i<0)
	return i;
      if (j=doseq(p2,BACK))
	return j;
    }
  sp=(long *)FSTACK(q);
  return useiff(q,(list)cell[2],BACK);
}




allimprdback(q)
     frame q;
{
  list cell;
  int i,j;
  frame p1,p2;
#ifdef DEBUG
  if (trace) mess("\nallimprdback");
#endif
  p1=(frame)PREM1(q);
  p2=(frame)PREM2(q);
  cell=(list)WORKF(q);
  /* sift case */
  if (twosift)
    {
      if (i=redo(p2))
	return i;
      while(SFLAG(p1)==0L&&(i=redo(p1))>0)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      freeblock((long *)SAVE(p1));
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      if (twoold(p1,p2))
		continue;
	      savesolution(p1);
	    }
	  if (j=doseq(p2,BACK))
	    {
	      if (j<0)
		return j;
	      if (SPOINT(p1))
		marktrue(p1);
	      return j;
	    }
	  if (SPOINT(p1))
	    markfalse(p1);
	}
      if (i<0)
	return i;
      if (SFLAG(p1))
	{
	  sp=(long *)FSTACK(q);
	  return 0;
	}
      freeblock((long *)SAVE(p1));
      SFLAG(p1)=1L;
      SPOINT(p1)=NULL;
      sp=(long *)FSTACK(q);
      return useuniv(q,(list)cell[2],BACK);
    }
  /* no-sift case */
  if (i=redo(p2))
    return i;
  while(i=redo(p1))
    {
      if (i<0)
	return i;
      if (j=doseq(p2,BACK))
	return j;
    }
  sp=(long *)FSTACK(q);
  return useuniv(q,(list)cell[2],BACK);
}


alliffrdback(q)
     frame q;
{
  list cell;
  int i,j;
  frame p1,p2;
  long *getblock();
#ifdef DEBUG
  if (trace) mess("\nalliffrdback");
#endif
  p1=(frame)PREM1(q);
  p2=(frame)PREM2(q);
  cell=(list)WORKF(q);
  /* sift case */
  if (twosift)
    {
      if (i=redo(p2))
	return i;
      while(SFLAG(p1)==0L&&(i=redo(p1))>0)
	{
	  if (nobound((list)VARS(p1)))
	    {
	      freeblock((long *)SAVE(p1));
	      SFLAG(p1)=1L;
	      SPOINT(p1)=NULL;
	      REDOFLAG(p1)=NOBACK;
	    }
	  else
	    {
	      if (twoold(p1,p2))
		continue;
	      savesolution(p1);
	    }
	  if (j=doseq(p2,BACK))
	    {
	      if (j<0)
		return j;
	      if (SPOINT(p1))
		marktrue(p1);
	      return j;
	    }
	  if (SPOINT(p1))
	    markfalse(p1);
	}
      if (i<0)
	return i;
      if (SFLAG(p1))
	{
	  sp=(long *)FSTACK(q);
	  return 0;
	}
      freeblock((long *)SAVE(p1));
      SFLAG(p1)=1L;
      SPOINT(p1)=NULL;
      sp=(long *)FSTACK(q);
      return useuniv(q,(list)cell[2],BACK);
    }
  /* no-sift case */
  if (i=redo(p2))
    return i;
  while (i=redo(p1))
    {
      if (i<0)
	return i;
      if (j=doseq(p2,BACK))
	return j;
    }
  sp=(long *)FSTACK(q);
  return useuniv(q,(list)cell[2],BACK);
}


allrdback(q)
     frame q;
{
  frame p1;
  int i;
  list cell;
#ifdef DEBUG
  if (trace) mess("\nallrdback");
#endif
  p1=(frame)PREM1(q);
  cell=(list)WORKF(q);
  if (i=redo(p1)) return i;
  sp=(long *)FSTACK(q);
  return useuniv(q,(list)cell[2],BACK);
}



/************ various auxiliary functions *************/


/* revapp3(s,t) is the result of reverse-appending the 3-cell list s
   to the 3-cell list t. s must be non-empty. New cells are created on
   the formula stack. */


list revapp3(s,t)      /* Note use of fsp */
     list s,t;
{
  register list cell;
  register long i;
  cell=s;
  i=0;
  fsp[0]=cell[0];     /* make first cell, link to t */
  fsp[1]=cell[1];
  fsp[2]=(long)t;
  fsp+=3;
  cell=(list)cell[2];
  while(cell)
    {
      fsp[i]=cell[0];
      fsp[i+1]=cell[1];
      fsp[i+2]=(long)(fsp+i-3);
      i+=3;
      cell=(list)cell[2];
    }
  fsp+=i;
  return fsp-3;
}




/* transcheck(r,q) returns true if the list q contains a formula identical
   in the sense of samepred() with r. */


transcheck(r,q)
     formula r;
     list q;
{
  register list p=q;
#ifdef DEBUG
  if (trace) mess("\ntranscheck:");
#endif
  while(p)
    {
      if (samepred(r,(formula)p[0]))
	{
#ifdef DEBUG
	  if (trace)
	    mess("old");
#endif
	  return 1;
	}
      p=(list)p[1];
    }
#ifdef DEBUG
  if (trace) mess("new");
#endif
  return 0;
}



samepred(r,s)
     formula r,s;
{
  register long i;
  subtab rsub,ssub;
  if ((basefmla)r[0]!=(basefmla)s[0])
    return 0;
  rsub=(subtab)r[1];
  ssub=(subtab)s[1];
  for(i=sublength-1;i>=0;i--)
    {
      if (rsub[i]==NULL)
	continue;
      if (sameval((term)rsub[i],(term)ssub[i])==0)
	return 0;
    }
  return 1;
}



sameval(r,s)
     term r,s;
{
  register long i;
  subtab rsub,ssub;
  term getvalvar();
  if (var(r))
    {
      r=getvalvar(r);
      return (var(s)&&(r==getvalvar(s)));
    }
  if (var(s))
    return 0;
  if (param(r))
    return (param(s)&&r[1]==s[1]);
  if (param(s))
    return 0;
  if (r[1]!=s[1])
    return 0;  /* different base terms */
  rsub=(subtab)r[2];
  ssub=(subtab)s[2];
  for(i=sublength-1;i>=0;i--)
    {
      if (tbvarocc(i,(baseterm)r[1])==0)
	continue;  /* again to avoid loops */
      if (sameval((term)rsub[i],(term)ssub[i])==0)
	return 0;
    }
  return 1;
}

  
/* copylist3(s) copies the 3-cell list s to the stack, returning the
   address of the copy. */

list copylist3(s)
     list s;
{
  register list cell=s;
  register long *rsp=sp;
  if (cell==(list)NULL)
    return (list)NULL;
  while(cell)
    {
      rsp[0]=cell[0];
      rsp[1]=cell[1];
      if (cell=(list)cell[2])
	rsp[2]=(long)(rsp+3);
      else rsp[2]=NULL;
      rsp+=3;
    }
  cell=sp;
  sp=rsp;
  return cell;
}



/* unbar(q) removes the first barrier, if any, from the 3-list q, and
   returns 0 if there was no bar, otherwise 1. */

unbar(q)
     list q;
{
  register list r=q;
  while(r)
    {
      if (r[1]<0)
	{
	  r[1]+=BARVAL;
	  return 1;
	}
      r=(list)r[2];
    }
  return 0;
}



