#include "ft.h"


/* This is the propositional algorithm invoked when a formula without
  quantifiers or free variables is input.  Contraction values are
  ignored in the propositional algorithm. Formulas are just base
  formulas.

  The propositional algorithm uses generalized implication locking in
  the following form: when we use the implication A->B in
  At,I,A->B,I'=>C all implications I,A->B,I' are locked in the left
  premiss and not unlocked until something new has been transferred -
  and, in the propositional case, necessarily transferred from the
  relevant occurrence of A. (Hence the use of a global impflag rather
  than inserting barriers in the list as in the quantifier case.)  We
  prove the left premiss first. Semi-invertibility is still partly
  exploited, however: if the right premiss fails, the sequent fails.
  Similarly for equivalences A<->B, which are mixed up with the
  implications in the propositional algorithm.

  Orlocking and oror compaction is implemented through the orflag in
  accordance with the following rules:

   ORFLAG manipulations:

     -> CLEAR   at applications of |=> and left premiss in ->=>, <->=>
     -> LOCK    right premiss in ->=>, <->=> provided orlock is true

     -> CLEAR   at applications of =>& =>-> =><->
     -> SPLIT   in proving a premiss in an application of =>|,
                provided oror is true. Also when SPLITOR connective is
                encountered (introduced in or-gathering).

  Note that the checking of orlock and oror is done 
  only in the two cases where the flag is set to SPLIT or LOCK. SPLIT
  and LOCK are acted on without checking of oror and orlock.  Oror compaction
  is in practice far more important that orlocking. 

  The implementation incorporates optional or-gathering and
  and-gathering.  These are seldom beneficial. And-use and or-use
  checking are more often useful, in particular anduse.  The fact that
  the left premiss is proved first in ->=> and <->=> means that imp-use
  and iff-use checking is not implemented. However, anduse has the
  extended sense of general inconsistency checking: in ->=> and <->=> we
  check (if anduse is true) whether or not the consequent was used in
  the proof of the left premiss: if not, the right premiss is taken as
  proved. This checking is invoked only if an absurdity axiom was used
  at some point in the course of proving the left premiss. Information on
  this point is encoded in the value returned in case of success, which
  is 1 if no absurdity axiom has been used and 2 otherwise.

    Anduse and oruse have been made the default in version 1.1.

  The last control parameters in the propositional algorithm are ifftrans
  and the eq switch, which occasionally make a dramatic difference - see the
  documentation.

  Axioms A,G->A are looked for only for atomic formulas A. The reason for
  this is that anduse checking can be made a great deal more economic if only
  atomic subformulas have to be checked. */


#define ANTP (currseq[0])     /*  Pointer to list of antecedent formulas  */
#define CONSP (currseq[1])    /*  Pointer to consequent                   */
#define IMPSP (currseq[2])    /*  Pointer to list of imps and iffs        */
#define ATOMSP (currseq[3])   /*  Pointer to list of atoms                */
#define TRANSP (currseq[4])   /*  Pointer to list of transferred formulas */
#define ORFLAGP (currseq[5])  /*  Orflag                                  */
#define IMPFLAG (currseq[6])  /*  Impflag                                 */
  
#define FRAMEP 7

#define ORCLEAR {ORFLAGP=(long *)NULL;}
#define ORLOCK  {ORFLAGP=(long *)1;}
#define ORSPLIT {ORFLAGP=(long *)2;}
#define IMPCLEAR {IMPFLAG=(long *)NULL;}
#define IMPLOCK  {IMPFLAG=(long *)1;}

#define  ORISLOCK (ORFLAGP==(long *)1)
#define  ORISPLIT (ORFLAGP==(long *)2)
#define  IMPISLOCK ((long)IMPFLAG)
  
#define SPLITOR (-OR)      /* special connective used in orgathering */


/* SAVEP       Remember current sp and working sequent
   COPY        Copy the remembered sequent to the main stack
   RESTORE     Make remembered sequent and sp current   */


#define SAVEP {spsave=sp;seqsave=currseq;}
#define COPY {if ((cp=copyseq((long *)seqsave))<0) return cp;\
              currseq=(long **)spsave;}
#define RESTORE {sp=spsave;currseq=seqsave;}


/* The implementation is a lot simpler than in the predicate logic
case because there is no backtracking mechanism. Non-invertibility is
dealt with by simple logical disjunctions and a loop for trying each
implication or equivalence in turn. Hence there is a simple stack
discipline: each function uses the main stack for copying sequents and
building cells and formulas. A function need not clean up the main
stack on exit - this is dealt with by the calling function.

   If use checking is active, the atomic formulas used in proving a
sequent are stored on the formula stack. This is the only use of a
stack other than the main stack in pprove.c. The formula stack shrinks when
failure is detected in the functions consand, antor, puseimp. This is
not sufficient to prevent overflow, e.g. in the horrible example prop
8 ~~Ax(p(x)|~p(x)). When overflow occurs, use checking is just suspended
with a message. */



long **currseq;  /* pointer to the working sequent */
int use;         /* global flag to indicate whether use checking is active */


pprove(q)
     basefmla q;
{
  long j,readtime();
  int i;
  long ausave=anduse;
  long ousave=oruse;
  if (timeout)
    setalarm(timeout);
  psetup(q);
  i=redcon();
  j=readtime();    /* clock started by doformula or doprop */
  anduse=ausave;
  oruse=ousave;
  if (i<0)
    {
      if (i==ABORT)
	push(j);
      return i;
    }
  if (i)
    printf("\nyes\n");
  else
    printf("\nno\n");
  printf("%ld\n",j);
#ifdef DEBUG
  if (stuse)
    printf("\nstack use %ld\n",max-stack);
#endif
  return 1;
}


psetup(q)
     basefmla q;
{
  currseq=(long **)sp;
  ANTP=NULL;
  CONSP=q;
  IMPSP=NULL;
  ATOMSP=NULL;
  TRANSP=NULL;
  ORCLEAR
  IMPCLEAR
  sp+=FRAMEP;
  if (anduse || oruse)
    use=1;
  else
    use=0;
}



redcon()         /* Use the consequent */
{
  basefmla p;
  long readtime();
#ifdef DEBUG
  if (trace) mess("\nredcon");
#endif
  if (aflag)
    return TIMEOUT;
  if (cflag)
    return ABORT;
  p=CONSP;
  switch ((int)p[0])
    {
    case OR:   return consor(p);
    case SPLITOR: {ORSPLIT return consor(p);}
    case AND:  return consand(p);
    case IFF:  return consiff(p);
    case IMP:  return consimp(p);
    default:   return consatom(p);
    }
}


consor(p)
     basefmla p;     /* consequent A1|A2 */
{
  long *spsave,**seqsave;
  int stat,cp;
#ifdef DEBUG
  if (trace) mess("\nconsor");
#endif
  if (ORISLOCK)
    return puseimp();
  SAVEP
  COPY
  if (oror)
    ORSPLIT
  CONSP=(basefmla)p[1];      /* replace A1|A2 with A1 */
  if (stat=redcon())
    return stat;
  RESTORE
  COPY
  if (oror)
    ORSPLIT
  CONSP=(basefmla)p[2];     /* try A2 */
  if (stat=redcon())
    return stat;
  RESTORE
    if (ORISPLIT)
      return 0;
  return puseimp();
}


consand(p)
     basefmla p;      /* consequent A1&A2 */
{
  long *spsave,**seqsave;
  long *fspsave=fsp;
  int stat,stat2,cp;
#ifdef DEBUG
  if (trace) mess("\nconsand");
#endif
  ORCLEAR
  SAVEP
  COPY
  CONSP=(basefmla)p[1];    /* replace A1&A2 with A1 */
  if ((stat=redcon())<=0)
    {
      fsp=fspsave;
      return stat;
    }
  RESTORE
  if (anduse&&stat==2&&unused((basefmla)p[1],fspsave))
    return 2;
  CONSP=(basefmla)p[2];   /* replace A1&A2 with A2 */
  stat2=redcon();
  if (stat2<=0)
    {
      fsp=fspsave;
      return stat2;
    }
  if (stat2==2)
    return 2;
  return stat;      /* 1 or 2 */
}


/* consimp() searches TRANSP by address or by equalfmla depending on
   the value of the system variable eq. */

consimp(p)
     basefmla p;      /* consequent A1->nA2 */
{
#ifdef DEBUG
  if (trace) mess("\nconsimp");
#endif
  ORCLEAR
    if (eq?chainmemb(p[2],TRANSP):fmlamemb((basefmla)p[2],TRANSP))
    {
      CONSP=(basefmla)p[3];   /* replace A1->nA2 with A2, forget A1 */
      return redcon();
    }
  IMPCLEAR                  /* otherwise ok to use imp or iff again */
  sp[0]=p[2];               /* make new cell for TRANSP */
  sp[1]=(long)(TRANSP);
  sp[2]=p[2];               /* make cell for new ANTP */
  sp[3]=NULL;
  TRANSP=sp;                /* add A1 to list of transferred formulas */
  ANTP=sp+2;                /* ANTP is now [A1] */
  sp+=4;
  CONSP=(basefmla)p[3];     /* replace A1->nA2 with A2 */
  return redant();
}


consiff(p)
     basefmla p;   /* consequent A1<->nA2 */
{
  register long *r=sp;
#ifdef DEBUG
  if (trace) mess("\nconsiff");
#endif
  ORCLEAR
  r[0]=IMP;  /* build new formula on stack */
  r[1]=0L;  /* contraction value doesn't matter */
  r[2]=p[2];
  r[3]=p[3];
  r[4]=IMP;
  r[5]=0L;
  r[6]=p[3];
  r[7]=p[2];
  r[8]=AND;
  r[9]=(long)r;
  r[10]=(long)(r+4);
  sp+=11;
  return consand(r+8);
}


consatom(r)
     basefmla r;    /* atomic or falsum */
{
  basefmla s;
#ifdef DEBUG
  if (trace) mess("\nconsatom");
#endif
  if (fmlawhichmemb(r,ATOMSP,&s))  /* r is equal to the formula s in ATOMSP */
    {
      if (use)
        {
	  fsp[0]=(long)r;
	  fsp[1]=(long)s;
	  fsp+=2;
	  if (fstackcheck()<0)
	    {
	      mess("Suspending use checking\n");
	      anduse=0;
	      oruse=0;
	      use=0;
	    }
	}
      return 1;
    }
  return puseimp();
}


puseimp()
{
  long *spsave,**seqsave;
  long *fspsave=fsp;
  register list q;
  basefmla p;
  int stat,stat2,cp;
#ifdef DEBUG
  if (trace) mess("\npuseimp");
#endif
  if (IMPISLOCK)
    return 0;
  q=IMPSP;      /* all changes to cells must be undone before exiting */
  while(q)
    {
      p=(basefmla)q[0];
      if (p==NULL)      /* this formula has been deleted */
	{
	  q=(list)q[1];
	  continue;
	}      
      if (p[0]==IMP)   /* p is A1->nA2 */
        {
          SAVEP
          COPY
          ORCLEAR /* for the left premiss */
          IMPLOCK
          CONSP=(basefmla)p[2];
          if ((stat=redcon())<0)
	    return stat;
	  RESTORE
         if (stat)   /* left premiss proved */
	   {
	     if (anduse&&stat==2&&unused((basefmla)p[2],fspsave))
	       return 2;
	     q[0]=NULL;      /* don't use A1->nA2 in the right branch */
             if (orlock)
	       ORLOCK        /* for the right premiss */
             sp[0]=p[3];     /* make cell for [A2] as new ANTP */
             sp[1]=NULL;
             ANTP=sp;
             sp+=2;
             stat2=redant();
	     q[0]=(long)p;   /* restore A1->nA2 to IMPSP */
	     if (stat2<=0)
	       {
		 fsp=fspsave;
		 return stat2;     /* semi-invertibility */
	       }
	     if (stat2==2)
	       return 2;
	     return stat;      /* 1 or 2 */
	   }
	  fsp=fspsave;
	  q=(list)q[1];  /* left premiss failed, so we try the next formula */
	  continue;
	}
      if (p[0]==IFF)  /* p is A1<->nA2 */
	{
          SAVEP
          COPY
          ORCLEAR
          IMPLOCK
          CONSP=(basefmla)p[2];      /* first try deducing A1 */
          if ((stat=redcon())<0)
	    return stat;
          RESTORE
          if (stat)      /* Left premiss proved using A1 */
	    {
	      if (anduse&&stat==2&&unused((basefmla)p[2],fspsave))
		return 2;
	      if (orlock)
		ORLOCK  /* for right premiss */
              q[0]=NULL; /* throw away A1<->nA2 in right branch */
              sp[0]=p[2];
              sp[1]=(long)(sp+2);
              sp[2]=p[3];
              sp[3]=(long)NULL;
              ANTP=sp;                   /* antecedent [A1,A2] */
       	      sp+=4;
	      stat2=redant();
	      q[0]=(long)p;  /* restore A1<->nA2 to IMPSP */
	      if (stat2<=0)
		{
		  fsp=fspsave;
		  return stat2;       /* semi-invertibility */
		}
	      if (stat2==2)
		return 2;
	      return stat;        /* 1 or 2 */
	    }
	  fsp=fspsave;
	  COPY
          ORCLEAR
	  IMPLOCK
	  CONSP=(basefmla)p[3];   /* try A2 in left premiss */
          if ((stat=redcon())<0)
	    return stat;
	  RESTORE
	  if (stat)      /* Left premiss proved using A2 */
	    {
	      if (anduse&&stat==2&&unused((basefmla)p[3],fspsave))
		return 2;
	      if (orlock)
		ORLOCK  /* for right premiss */
              q[0]=NULL; /* throw away A1<->nA2 in right branch */
              sp[0]=p[2];
	      sp[1]=(long)(sp+2);
              sp[2]=p[3];
              sp[3]=(long)NULL;
              ANTP=sp;                   /* antecedent [A1,A2] */
       	      sp+=4;
	      stat2=redant();
	      q[0]=(long)p;      /* restore A1<->A2 to IMPSP */
	      if (stat2<=0)
		{
		  fsp=fspsave;
		  return stat2;      /* semi-invertibility */
		}
	      if (stat2==2)
		return 2;
	      return stat;      /* 1 or 2 */
	    }
	  fsp=fspsave;
	  q=(list)q[1];
	  continue;
	}
    }
  return 0;      /* every formula in IMPS has been tried */
}


redant()             /* use antecedent to prove sequent */
{
  long i,readtime();
  basefmla p;
#ifdef DEBUG
  if (trace) mess("\nredant");
#endif
  if (aflag)
    return TIMEOUT;
  if (cflag)
    return ABORT;
  if (ANTP==(list)NULL)
    return redcon();
  p=(basefmla)ANTP[0];
  i=p[0];
  if (i==FALSUM&&minimal==0)   /* absurdity axiom */
    {
      if (use)
	{
	  fsp[0]=(long)p;
	  fsp++;
	  if (fstackcheck()<0)
	    {
	      mess("Suspending use checking\n");
	      anduse=0;
	      oruse=0;
	      use=0;
	    }
	}
      return 2;
    }
  if ((p[0]>=EQU)&&equalatoms(p,CONSP))  /* only atomic axioms */
    {
      if (use)
        {
	  fsp[0]=(long)p;
	  fsp[1]=(long)CONSP;
	  fsp+=2;
	  if (fstackcheck()<0)
	    {
	      mess("Suspending use checking\n");
	      anduse=0;
	      oruse=0;
	      use=0;
	    }
	}
      return 1;
    }
  switch((int)i)
    {
    case OR:     return antor(p);
    case AND:    return antand(p);
    case IFF:    return antiff(p);
    case IMP:    return antimp(p);
    default:     return antatom(p);
    }
}


antor(p)
     basefmla p;     /* disjunction A1|A2 */
{
  long *spsave,**seqsave;
  long *fspsave=fsp;
  int stat,stat2,cp;
#ifdef DEBUG
  if (trace) mess("\nantor");
#endif
  ORCLEAR
  SAVEP
  COPY
  sp[0]=p[1];  /* put in A1 instead of A1|A2 */
  sp[1]=ANTP[1];
  ANTP=sp;
  sp+=2;
  if((stat=redant())<=0)
    {
      fsp=fspsave;
      return stat;
    }
  RESTORE
    if (oruse && unused((basefmla)p[1],fspsave))
      return stat;  /* 1 or 2 */
  sp[0]=p[2];  /* put in A2 instead of A1|A2 */
  sp[1]=ANTP[1];
  ANTP=sp;
  sp+=2;
  stat2=redant();
  if (stat2<=0)
    {
      fsp=fspsave;
      return stat2;
    }
  if (stat2==2)
    return 2;
  return stat;      /* 1 or 2 */
}


antand(p)
     long *p;        /* A1&A2 */
{
#ifdef DEBUG
  if (trace) mess("\nantand");
#endif
  sp[0]=p[2];
  sp[1]=ANTP[1];   /* ANTP with A1&A2 removed */
  sp[2]=p[1];
  sp[3]=(long)sp;
  ANTP=sp+2;
  sp+=4;
  return redant();
}


antimp(p)
     basefmla p;       /* A1->nA2 */
{
#ifdef DEBUG
  if (trace) mess("\nantimp");
#endif
  if (andgather)
    return andg(p);
  if (orgather)
    return org(p);
  return hoardimp(p);
}


hoardimp(p)            /* transfer implication to IMPSP */
     basefmla p;      /* A1->nA2 - n is ignored */
{
#ifdef DEBUG
  if (trace) mess("\nhoardimp");
#endif
  sp[0]=(long)p;
  sp[1]=(long)(IMPSP);
  IMPSP=sp;
  sp+=2;
  ANTP=(list)(ANTP)[1];
  return redant();
}


andg(p)
     basefmla p;     /* A1->mA2 */
{
  basefmla r;
  list q;
  int stat;
  register long *rsp=sp;
#ifdef DEBUG
  if (trace) mess("\nandg");
#endif
  q=IMPSP;
  while(q)
    {
      r=(basefmla)q[0];  /* pick an implication in IMPSP */
      if (r[0]==IMP && equalfmlas((basefmla)p[2],(basefmla)r[2]))
                    /* we have picked A1->nB */
        {
	  rsp[0]=AND;     /* make new formula A1->0A2&B */
	  rsp[1]=p[3];
	  rsp[2]=r[3];  
	  rsp[3]=IMP;
	  rsp[4]=0L;
	  rsp[5]=p[2];
	  rsp[6]=(long)rsp;
	  rsp[7]=(long)(rsp+3);        /* new IMPSP cell */
	  rsp[8]=(long)IMPSP;
	  IMPSP=rsp+7;
	  sp+=9;
	  ANTP=(list)ANTP[1];       /* remove A1->mA2 from ANTP */
	  q[0]=NULL;                  /* erase A1->nB from IMPSP */
	  stat=redant();
	  q[0]=(long)r;               /* restore A1->nB */
	  return stat;
	}
      q=(list)q[1];
    }
  if (orgather)
    return org(p);
  else
    return hoardimp(p);
}


org(p)
     basefmla p;       /* A1->nA2 */
{
  list q;
  basefmla r;
  int stat;
  register long *rsp=sp;
#ifdef DEBUG
  if (trace) mess("\norg");
#endif
  q=IMPSP;
  while(q)
    {
      r=(basefmla)q[0];  /* pick an implication in IMPSP */
      if (r[0]==IMP && equalfmlas((basefmla)p[3],(basefmla)r[3]))
                                    /* we have picked B->mA2 */
        {
	  rsp[0]=SPLITOR;     /* make new formula A1|B->0A2 */
	  rsp[1]=p[2];
	  rsp[2]=r[2];  
	  rsp[3]=IMP;
	  rsp[4]=0L;
	  rsp[5]=(long)rsp;
	  rsp[6]=p[3];
	  rsp[7]=(long)(rsp+3);    /* new IMPSP cell */
	  rsp[8]=(long)IMPSP;
	  IMPSP=rsp+7;
	  sp+=9;
	  ANTP=(list)ANTP[1];   /* remove A1->nA2 from ANTP */
	  q[0]=NULL;              /* erase B->mA2 from IMPS */
	  stat=redant();
	  q[0]=(long)r;           /* restore B->mA2 */
	  return stat;
	}
      q=(list)q[1];
    }
  return hoardimp(p);
}


antiff(p)
     basefmla p;       /* A1<->nA2 */
{
#ifdef DEBUG
  if (trace) mess("\nantiff");
#endif
  sp[0]=(long)p;            /* make cell */
  sp[1]=(long)(IMPSP);
  IMPSP=sp;                 /* attach to IMPSP */
  sp+=2;
  ANTP=(list)(ANTP)[1];     /* cut away from ANTP */
  return redant();
}


antatom(p)
     basefmla p;      /* atomic, not falsum */
{
#ifdef DEBUG
  if (trace) mess("\nantatom");
#endif
  sp[0]=(long)p;
  sp[1]=(long)(ATOMSP);  /* link address */
  ATOMSP=sp;        /* new first cell in ATOMSP */
  sp+=2;
  ANTP=(list)(ANTP)[1];
  return redant(); 
}



/*************************** auxiliary *****************************/


copyseq(p)
     long *p;
{
  register int i;
  for(i=FRAMEP-1;i>=0;i--)
    sp[i]=p[i];
  sp+=FRAMEP;
  return stackcheck(); /* checking of main stack only done here */
}

/* chainmemb(p,q) returns true iff p occurs in the chain q */
  
chainmemb(p,q)
     long p;
     list q;
{
  register long *r=q;
  while(r)
    {
      if(p==r[0])
	return 1;
      r=(list)r[1];
    }
  return 0;
}


/* fmlamemb(p,q) returns true iff the base formula p is identical with or
   equal to some formula in the list q of base formulas. It is used only
   in transfer checking (if eq is false). */

fmlamemb(p,q)
     basefmla p;
     list q;        /* list of base formulas */
{
  register list r=q;
  while(r)
    {
      if (p==(basefmla)r[0]||equalfmlas(p,(basefmla)r[0]))
	return 1;
      r=(list)r[1];
    }
  return 0;
}

/* fmlawhichmemb(p,q,&s) returns true iff an atomic base formula equal
  to p occurs in the list q of atomic base formulas, and sets s to point to the
  first such formula. */

fmlawhichmemb(p,q,s)
     basefmla p;
     list q;      
     formula *s;
{
  register list r=q;
  while(r)
    {
      if (equalatoms(p,(basefmla)r[0]))
	{
	  *s=(basefmla)r[0];
	  return 1;
	}
      r=(long *)r[1];
    }
  return 0;
}


/* equalfmlas() is used in transfer checking when eq is false, and also
   in orgathering and andgathering. Note that there is no checking for
   identity, so this should be done by the calling function, if it is
   a good idea (as in fmlamemb()). */

equalfmlas(p,q)
     basefmla p,q;
{
  register long i;
  register long args;
  i=p[0];
  if (i==OR||i==SPLITOR)
    {
      if ((q[0]!=OR)&&(q[0]!=SPLITOR))
	return 0;
    }
  else
    if(i!=q[0])
      return 0;
  if(i==FALSUM)
    return 1;
  if (i==IMP || i==IFF) /* difference in annotation doesn't count */
    return equalfmlas((basefmla)p[2],(basefmla)q[2]) &&
           equalfmlas((basefmla)p[3],(basefmla)q[3]);
  if (i==OR || i==AND)
    return equalfmlas((basefmla)p[1],(basefmla)q[1]) &&
           equalfmlas((basefmla)p[2],(basefmla)q[2]);
  for(args=p[1];args>0;args--)
    {
      if (equalterms((baseterm)p[1+args],(baseterm)q[1+args])==0)
	return 0;
    }
  return 1;
}



/* variables admitted in terms because of translation kludge in prop.c */

equalterms(p,q)
     baseterm p,q;
{
  long i=p[0];
  long args;
  if (i<=BMAX)      /* p is a syntactic variable */
    return q[0]==i;
  if (i!=q[0])      /* q and p have different function symbols */
    return 0;
  for(args=p[1];args>0;args--)
    {
      if (equalterms((baseterm)p[1+args],(baseterm)q[1+args])==0)
	return 0;
    }
  return 1;
}


/* unused(p,q) returns true iff no atomic subformula of p occurs on stack 4,
   where the search ends at the stack 4 address q. */

unused(p,q)
     basefmla p;
     long *q;
{
  register long *r;
  basefmla j;
  long k;
#ifdef DEBUG
  if (trace) mess("\nunused");
#endif
  k=(long)findatoms(p,sp);
  for(r=fsp-1;r>=q;--r)
    {
      j=(basefmla)r[0];
      if (atomfound((long)j,k))
	return 0;
    }
  return 1;
}

/* equalatoms(p,q) returns true iff p and q are the same atomic formula. It
   is only called with p an atomic formula. */


equalatoms(p,q)
     basefmla p,q;
{
  register long i;
  register long args;
  i=p[0];
  if (i!=q[0])
    return 0;
  if (i==FALSUM)
    return 1;
  for(args=p[1];args>0;args--)
    {
      if (equalterms((baseterm)p[1+args],(baseterm)q[1+args])==0)
	return 0;
    }
  return 1;
}


/* findatoms(p,q) writes consecutive pointers to the atomic subformulas of
   the base formula p to address q and returns the number of such 
   subformulas */


findatoms(p,q)
     basefmla p;
     long *q;
{
  int j1,j2;
  long i=p[0];
  if (i==IFF||i==IMP)
    {
      j1=findatoms((basefmla)p[2],q);
      j2=findatoms((basefmla)p[3],q+j1);
      return j1+j2;
    }
  if (i==OR||i==AND||i==SPLITOR)
    {
      j1=findatoms((basefmla)p[1],q);
      j2=findatoms((basefmla)p[2],q+j1);
      return j1+j2;
    }
  q[0]=(long)p;
  return 1;
}


/* atomfound(j,k) returns true iff one of the numbers sp[0],..sp[k-1] is j */

atomfound(j,k)
     long j;
     long k;
{
  register long u;
  for(u=0;u<k;u++)
    if (j==sp[u])
      return 1;
  return 0;
}


printcurrseq()
{
  printf("ANT:");
  printbaselist(ANTP);
  printf("IMPS:");
  printbaselist(IMPSP);
  printf("ATOMS:");
  printbaselist(ATOMSP);
  printf("CON:");
  printbasefmla(CONSP);
}
