#include "ft.h"


/*

Formulas are represented as two consecutive pointers, the first pointing
to the base formula, the second to the substitution table. The substitution
table indicates which substitutions have been made for syntactic variables
in the base formula. Thus all substitution tables have the same length,
and indexing into the substitution table gives a pointer to the term
substituted for a syntactic variable (NULL if there has been no substitution).

A term is either a variable, a parameter, or a substitution term. A variable
consists of two consecutive longs: the restriction value plus 1, followed
by a pointer to the current value of the variable, or NULL if the variable
is unbound. A parameter consists of NULL followed by the k for which the
parameter is prm(k). A substitution table consists of a pointer to a
base term followed by a pointer to a substitution table.

Base formulas and base terms are constructed from the input formula as
follows (where &W stands for a pointer to the representation of W):



   free input variables             These are represented as syntactic
                                    variables with unbound variables
                                    put into the initial substitution
                                    table.
                                    Because this is a late second thought,
                                    free input variables are first stored
                                    as m where m>FNMAX, and then fixed by
                                    the function makevartab().

   syntactic variables              n where 0<=n<=BMAX.

   The term f(t1,..tn)              m,n,&t1,..&tn where BMAX<m<=FNMAX

   Atomic formula p(t1,..tn)        m,n,&t1,..&tn where m>EQU=BMAX+1.
     where p is not =

   t1=t2                            EQU,2,&A1,&A2
   
   falsum                           FALSUM

   A1 & A2                          AND,&A1,&A2

   A1 | A2                          OR,&A1,&A2

   A1 ->n A2                        IMP,n,&A1,&A2
   
   A1 -> A2                         IMP,0,&A1,&A2

   A1 <-> n A2                      IFF,n,&A1,&A2

   A1 <-> A2                        IFF,0,&A1,&A2

   AnxB                             ALL,n,m,&B  where m>=0

   AxB                              ALL,0,m,&B  where m>=0

   ExB                              SOME,m,&B   where m>=0            



Negation is eliminated from the input: ~[n]A is read as A->[n]falsum.
All quantified variables are made unique in the base formula.
However, the use of ifftrans will lead to the same syntactic variable
being used in different quantifiers in a formula.This doesn't
matter, for the important thing is that there is no nesting of
quantifiers on the syntactic bound variable. As long as there is no such
nesting, we know that in order to substitute X for free x in F(x), obtained
from AxF(x), we substitute X for a l l occurrences of x in F(x).

The free variables in the input are stored on the stack after the
base formula as follows: the number of free variables, followed by the
variable names as strings, and then the variables themselves as 1,0
(restriction+1, pointer to value). infree is set to point to the
beginning of this table, infree2 to the first variable. initsub is
where the initial substitution table is. sublength is the length of a
substitution table = the number of quantifiers+the number of free
variables in the input. */


long *infree;      /* input free variable table */
long *infree2;     /* first free input variable */
long *initsub;     /* initial substitution table */
long sublength;    /* length of initial substitution table */ 


/* After the initial substitution table comes a table of input function
names at infunc, needed for presentation of answers. (See printvars().) */

long *infunc;


/* The symbol table is not constructed on the stack but in textbuf[] after
the input text. An entry in the symbol table consists of 5 longs: count of
characters, pointer to first occurrence in text, identifier, type
(BVAR, PRED, FUNC, FVAR), and finally the arity. */


static long *symtab;
static int symindex=0;

#define COUNT(p)    (symtab+p)[0]
#define OCC(p)      (symtab+p)[1]
#define ID(p)       (symtab+p)[2]
#define TYPE(p)     (symtab+p)[3]
#define ARITY(p)    (symtab+p)[4]

#define BVAR 0L        /* coding used in the symbol table */
#define PRED 1L
#define FUNC 2L
#define FVAR 3L

/* these used by bvarid() etc */

long bvarcount;   
long predcount;
long funcount;
long fvarcount;



doformula(p)
     char *p;
{
  long readtime();
  int i;
  formula q;
  basefmla skeleton();
  marktime();
  if ((i=makeformula(p,&q))<0)
    {
      readtime();
      return i;
    }
  if (syntax)
    {
      readtime();
      return 1;
    }
  bindall();  /* bind input variables to new variables on stack3... */
  if (*(basefmla)q[0]==EQU) /* identity s=t input */
    return dounify(q);
  if (sublength==0)         /* this is a propositional formula */
    return pprove((basefmla)q[0]);
  return prove(q);
}


dounify(q)
     formula q;
{
  baseterm base;
  subtab sub;
  list r=(list)NULL;
  long readtime(),time1,time2;
  int res;
  base=(basefmla)q[0];     /* base formula is =,2,s,t */
  sub=(subtab)q[1];
  time1=readtime();
  marktime();
  res=unify((baseterm)base[2],sub,(baseterm)base[3],sub,&r);
  time2=readtime();
  if (res==0)
    printf("\nNot unifiable");
  else
#ifdef DEBUG
    if (dflag&2==0)
#endif
      printvars();
  if (res==ABORT)
    {
      push(time2);
      return res;
    }
  if (res>0)
    printf("\n%ld %ld\n",time1,time2);
  return res;
}




/* makeformula(p,&q) tries to read the text at p as
   a formula. The internal representation of the formula is stored on
   the stack. The value returned is either an error number or 1. q is
   set to point to the analyzed formula in the style: pointer to
   base formula, pointer to substitution table. Also, the free variables in 
   the input are stored on the stack after the formula. */

makeformula(p,q)
     char *p;
     formula *q;
{
  int j;
  formula removeiff();
  initvars();
  if ((j=getmaxfmla(p,q))<0)
    return j;
  if (p[j])
    {
      push((long)p+j);
      return BADSYNTAX;
    }
  makevartab(*q);  /* includes setting initsub and fixing the formula */
  if (ifftrans)
    *q=removeiff(*q);
  push((long)*q);       /* pointer to base formula               */
  push((long)initsub);  /* pointer to initial substitution table */
  *q=sp-2;
  return 1;
}


initvars()
{
  symtab=(long *)(textp+4-(long)(textp)%4); /* ALIGNMENT */
  symtab[0]=0L;  /* marking the end of symtab */
  push3(-1L);    /* marking the end of the stacked scopes */
  symindex=0;
  bvarcount=0L;
  predcount=EQU+1; /* =BMAX+2 */
  funcount=BMAX+1;
  fvarcount=FNMAX+1;
}


/* getmaxfmla(p,&q) analyzes and stores the longest formula starting at
  p and returns the offset to the next character in p[], or an error
  number. q is set to point to the analyzed formula. */





getmaxfmla(p,q)
     char *p;
     formula *q;
{
  int i,m;
  if ((i=stackupfmla(p,&m))<0)
    return i;
  unstackfmla(m,q);
  return i;
}


/* stackupfmla(p,&m) stores the binary propositional skeleton of the maximal
   formula at p on stack 2 (reverse order) with the unit formulas 
   represented by their addresses on stack 1, and returns the offset to 
   the next character in p[] or an error number. m is set to the number
   of unit formulas stored. */


stackupfmla(p,m)
     char *p;
     int *m;
{
  int count,offs,i,j,k,a;
  long b;
  long *f;
  count=offs=0;
  while(1)
    {
      if ((i=get0fmla(p,&f))<0)
	return i;
      push2((long)f);
      count++;
      offs+=i;
      if ((j=getconn(p+i,&k))<0)
	break;
      if (k==IMP || k==IFF)   /* pack in contraction value */
        {
	  if ((a=getcont(p+i+j,&b))<0)
	    return a;
          k=10*b+k;
          offs+=a;
          p+=a;
	}
      push2((long)k);
      offs+=j;
      p+=i+j;
    }
  *m=count;
  return offs;
}


/* unstackfmla(m,&q) expects m addresses to formulas on stack 2, interspersed
   with connectives. A formula is made up from this material and stored as
   usual on stack 1 (no errors possible). q is set to point to the formula.
   Stack 2 is cleaned up. 
   Algorithm: the main connective is the first maximal connective. */

unstackfmla(m,q)
     int m;
     long **q;
{
  long *maxaddr,maxconn;  /* address in stack 2 and value of max connective */
  long *p,*ad1,*ad2,*q1,*spsav,pop2(),k;
  int left,right;
  if (m==1)
    {
      *q=(long *)pop2();
      return;  /* the one formula is already on stack 1 */
    }
  p=sp2+2*m-2;          /* location of first connective */
  maxaddr=p;
  maxconn=(*p);
  for(q1=p;q1>sp2;q1-=2)
    {
      if (*q1>maxconn)
	{
	  maxconn=(*q1);
	  maxaddr=q1;
	}
    }
  right=(maxaddr-sp2)/2;left=m-right;
  unstackfmla(right,&ad1);   /* second subfmla */
  *q=sp;
  k=pop2();   /* connective */
  if (k>=IMP)  /* unpack contraction */
    {
      push(k%10);
      push(k/10);
    }
  else
    push(k);
  spsav=sp;
  sp+=2;     /* make room for subformulas */
  unstackfmla(left,&ad2);   /* first subfmla */
  *spsav=((long)ad2);
  *(spsav+1)=((long)ad1);
}


/* get0fmla(p,&q) analyzes and stores the shortest formula starting at
  p and returns the offset to the next character in p[], or an error
  number. q is set to point to the analyzed formula. */

get0fmla(p,q)
     char *p;
     long **q;
{
  long *f,*storeimp(),*storeall(),*storesome(),pop3();
  int i;
  char c=(*p++);
  if (c=='(')  /* the case (F) */
    { 
      if ((i=getmaxfmla(p,q))<0)
	return i;
      if (p[i]!=')')
	{
	  push((long) p+i);
	  return BADSYNTAX;
	}
      return i+2;
    }
  if (c=='~')   /* the case ~[n]F */
    {
      long k;
      long j;
      if ((j=getcont(p,&k))<0)
	return j;  /* value in k, offset in j */
      if ((i=get0fmla(p+j,&f))<0)
	return i;
      push(FALSUM);
      /* different FALSUMs because of anduse and oruse */
      *q=storeimp(IMP,(long)k,(long)f,(long)(sp-1));
      return j+i+1;
    }
  if (c=='A')  /* the case A[n]xF */
    {
      int j,k,l;
      long m;
      if ((j=getcont(p,&m))<0)
	return j;  /* value in m, offset in j */
      if ((i=getbvar(p+j,&k))<0)
	return i; /* symtab index in k, offset in i */
      push3((long)k);  /* put variable on scope stack */
      if ((l=get0fmla(p+j+i,&f))<0)
	return l;
      pop3();   /* pop scope stack */
      *q=storeall(ALL,(long)m,ID(k),(long)f);
      return j+i+l+1;
    }
  if (c=='E')  /* the case ExF */
    {
      int k,l;
      if ((i=getbvar(p,&k))<0)
	return i;
      push3((long)k);
      if ((l=get0fmla(p+i,&f))<0)
	return l;
      pop3();
      *q=storesome(SOME,ID(k),(long)f);
      return i+l+1;
    }
  /* atomic formula */
  return getatom(--p,q);
}


/* getatom(p,&q) tries to make an atomic formula out of the text at
   p and returns an offset to the next character or an error number.
   q is set to point to the analyzed and stored atomic formula. Note:
   an error will be reported in certain cases when p[] does begin with
   an atomic formula (because of the calls to countargs and checkeq),
   but never when an initial atomic formula is followed by ) or
   a connective or \0. */


getatom(p,q)
     char *p;
     long **q;
{
  long *stacksave,*r,*r1,predid(),funcid();
  int i,j,eqflag,index,m,n,m1,offset,args;
  long id,type,funcid();
  if ((i=getfunctor(p))>=0)  /* new kludge */
    {
      if (isfalsum(p))
        {
	  push(FALSUM);
	  *q=sp-1;
          return i;
        }
    args=countargs(p+i); /* number of arguments of initial functor */
    eqflag=checkeq(p+i); /* if this is = eqflag is non-zero */
    if ((j=seek(p,i))>=0)  /* the main functor exists in symtab */
      {
        if (ARITY(j)!=args || (eqflag && TYPE(j)==PRED)
                           || (eqflag==0 && (TYPE(j)==FUNC||
                                             TYPE(j)==BVAR||
					     TYPE(j)==FVAR)))
          {
	    push((long)p);
            return INCSYNTAX;
          }
        index=j;
      }
      else
	{
	  type=eqflag ? FUNC : PRED;
          id=eqflag ? funcid() : predid();
          index=makesym((long)i,(long)p,(long)id,(long)type,(long)args);
	}
      stacksave=sp;
      *q=sp;
      offset=i;
    }
  else   /* if we have either bad syntax or a free variable  */
    {
      eqflag=1;
      stacksave=sp;
      *q=sp;
    }
  if (eqflag)       /* this is an equation or bad syntax */
    {
      push((long)EQU);
      push(2L);
      sp+=2;      /* make room for arguments */
      if ((m=getterm(p,&r))<0)
	return m;
      offset=m;
      if (p[m]!='=')
	{
	  push((long)p+m);
	  return BADSYNTAX;
	}
      offset+=1;
      if ((m1=getterm(p+m+1,&r1))<0)
	return m1;
      offset+=m1;
      stacksave[2]=(long) r;
      stacksave[3]=(long) r1;
      return offset;
    }
  push(ID(index));
  push((long)args);
  if (args==0)
    return offset;
  sp+=args;
  p+=i+1;      /* include ( */
  offset++;
  for(n=0;n<args;n++)
    {
      if ((i=getterm(p,&r))<0)
	return i;
      stacksave[2+n]=(long) r;
      offset+=i;
      p+=i;
      if (n<args-1 && *(p++)!=',')
        {
	  push((long)(p-1));
	  return BADSYNTAX;
        }
      if (n==args-1 && *(p++)!=')')
        {
	  push((long)(p-1));
	  return BADSYNTAX;
        }
      offset++;
    }
  return offset;
}


 
/* getterm(p,&r) constructs a term from text at p and returns offset to next
   character in p[] or error number. r is set to point to the analyzed
   term. */

getterm(p,r)
     char *p;
     long **r;
{
  int i,args,j,n,index,offset;
  long *spsave,*r1;
  long funcid();
  if (upper((int)*p))       /* free variable case treated separately */
    return getfvar(p,r);
  if ((i=getfunctor(p))<0)
    return i;
  offset=i;
  args=countargs(p+i);
  if ((j=seek(p,i))>=0)
    {
      if (TYPE(j)==PRED || ARITY(j)!=args)
        {
	  push((long)p);
	  return INCSYNTAX;
	}
      if (TYPE(j)==BVAR)
	return bvarcase(i,p,r); /* bvar case separately */
      index=j;
    }
  else /* if it isn't in symtab already, it must be a function symbol */
    index=makesym((long)i,(long)p,funcid(),FUNC,(long)args);
  spsave=sp;
  *r=sp;       /* this is where the term is */
  push(ID(index));
  push((long)args); 
  if (args==0)
    return offset;
  sp+=args;       /* make room for arguments */
  p+=i+1;         /* include ( */
  offset++;
  for(n=0;n<args;n++)
    {
      if ((i=getterm(p,&r1))<0)
	return i;
      spsave[2+n]=(long)r1;
      offset +=i;
      p+=i;
      if (n<args-1 && *(p++)!=',')
	{
	  push((long)(p-1));
	  return BADSYNTAX;
	}
      if (n==args-1 && *(p++)!=')')
	{
	  push((long)(p-1));
	  return BADSYNTAX;
	}
      offset++;
    }
  return offset;
}


/* bvarcase(i,p,&r): the first i chars of p[] have been used before as a
   bound variable. We must check that we are within the scope of a
   quantifier on that symbol, and give the present occurrence the
   identifier of the first such quantifier on the scope stack. */


bvarcase(i,p,r)
     int i;
     char *p;
     long **r;
{
  int j=1;
  long k;
  while((k=(*(sp3+j)))>=0)
    {
      if (COUNT(k)==i && same((char *)OCC(k),p,i))
        {
	  *r=sp;
          push(ID(k));
          return i;
        }
      j++;
    }
  push((long)p);
  return INCSYNTAX;
}


/* getfvar(p,&r) returns offset or error and sets r to free var */

getfvar(p,r)
     char *p;
     long **r;
{
  int i=0;
  int j;
  long id;
  long fvarid();
  if (upper((int)p[i++])==0)
    {
      push((long)p);
      return BADSYNTAX;
    }
  while(dig((int)p[i]))
    i++;
  if ((j=seek(p,i))>=0)
    ;
  else
    {
      id=fvarid();
      j=makesym((long)i,(long)p,(long)id,FVAR,0L);
    }
  *r=sp;
  push(ID(j));
  return i;
}



makevartab(q)
     formula q;  /* the analyzed input formula, for fixing input vars */
{
  char *varpoint;
  int index=0;
  long count=0;
  int j;
  long m,bvarid();
  infree=sp++;    /* make room for number of free vars in input */
  varpoint=(char *)sp;
  while(j=COUNT(index))
    {
      if (TYPE(index)==FVAR)
        {
	  count++;
          movevar(j,(char *)OCC(index),varpoint);
          varpoint+=j+1;
        }
      index+=5;
    }
  infree[0]=count;
  sp=(long *)(varpoint+4-(long)varpoint%4);  /* ALIGNMENT */
  index=0; /* inserted kludge begins here */
  initsub=sp; /* initial substitution table goes here */
  sublength=bvarcount+count;
  if (sublength==0)
    initsub=NULL;
  for(j=0;j<sublength;j++)
    sp[j]=NULL;
  sp+=sublength;
  infree2=sp; /* the actual variables begin at infree2 */
  while(COUNT(index))
    {
      if (TYPE(index)==FVAR)
        {
	  push(1L);   /* restriction is 0 */
	  push(0L);   /* variable is unbound */
	  m=bvarid();
          initsub[m]=(long)(sp-2); /* insert variable in substitution table */
	  fixvar(q,m,ID(index));  /* substitute for id of free var */
        }
      index+=5;
    }
  infunc=sp;    /* now make table of function names */
  varpoint=(char *)sp;      /* really funcpoint... */
  index=0;
  while(j=COUNT(index))
    {
      if (TYPE(index)==FUNC)
        {
	  movevar(j,(char *)OCC(index),varpoint);
	  varpoint+=j+1;
        }
      index+=5;
    }
  sp=(long *)(varpoint+4-(long)varpoint%4);   /* ALIGNMENT */
}


movevar(j,p,varpoint)
     int j;
     char *p,*varpoint;
{
  int i=0;
  while(j)
    {
      varpoint[i]=p[i];
      i++;
      j--;
    }
  varpoint[i]='\0';
}



/* fixvar(q,p,id) goes through the formula q and replaces every occurrence
   of id by p */


fixvar(q,p,id)
     formula q;
     long p,id;
{
  long i,args;
  if (q==NULL) return;
  i=q[0];
  if (i==IMP||i==IFF)
    {
      fixvar((long *)q[2],p,id);
      fixvar((long *)q[3],p,id);
      return;
    }
  if (i==OR||i==AND)
    {
      fixvar((long *)q[1],p,id);
      fixvar((long *)q[2],p,id);
      return;
    }
  if (i==ALL) {fixvar((long *)q[3],p,id);return;}
  if (i==SOME) {fixvar((long *)q[2],p,id);return;}
  if (i==FALSUM) return;
  args=q[1];
  for(i=0;i<args;i++)
    termfixvar((long *)q[2+i],p,id);
}


termfixvar(q,p,id)  /* substitute bvar p for fvar id in formula q */
     term q;
     long p,id;
{
  long i,args;
  if (q==NULL) return;
  i=q[0];
  if (i<=BMAX)
    return; /* includes bound vars and already fixed free vars */
  if (i==id)
    {
      q[0]=p;
      return;
    }
  if (i>FNMAX)
    return; /* free variable other than id */
  args=q[1];
  for (i=0;i<args;i++)
    termfixvar((long *)q[2+i],p,id);
}




bindall()
{
  long i;
  variable s;
  for(i=0;i<infree[0];i++)
    {
      s=invar(i);
      s[1]=(long)(sp3-1);
      sp3[-1]=s[0];
      sp3[0]=NULL;
      sp3-=2;
    }

}


/* getcont(p,&m) sets m to the positive integer read at p, or 0 if there
   isn't any integer at p, and returns offset to next char in p[], or
   error BADCONT if integer is non-positive or >= BARVAL. */

getcont(p,m)
     char *p;
     long *m;
{
  int k;
  long u,getnum();
  u=getnum(p,&k); /* in set.c - note: returns value or 0 */
  if (k!=0 && (u<=0||u>=BARVAL))
    {
      push((long) p);
      return BADCONT;
    }
  *m=u;
  return k;
}



/* getbvar(p,&m) sets m to the symtab index for the bound variable at
   p, and returns offset to the next char in p[] or error BADSYNTAX or
   INCSYNTAX.
   A new entry and id is always created, even if the variable already
   exists in symtab. */

getbvar(p,m)
     char *p;
     int *m;
{
  int i,j;
  long k,bvarid();
  if ((i=isbvar(p))<0)
    return i;
  if ((j=seek(p,i))>=0) /* the variable is in symtab */ 
    {
      if (TYPE(j)!=BVAR)
	{
	  push((long)p);
	  return INCSYNTAX;
	}
    }
  k=bvarid();
  *m=makesym((long)i,(long)p,(long)k,(long)BVAR,0L);
  return i;
}




isbvar(p)  /* returns offset to next char in p[] after bvar, or BADSYNTAX. */
     char *p;
{
  int i=0;
  char c=p[i++];
  if (lower((int)c))
    {
      while(dig((int)p[i]))i++;
      return i;
    }
  push((long)p);
  return BADSYNTAX;
}

/* getfunctor(p) checks that p[] starts with a predicate symbol or
   function symbol and returns offset or error. */

getfunctor(p)
     char *p;
{
  int i=0;
  while(lower((int)p[i]))
    i++;
  if (i==0)
    {
      push((long)p);
      return BADSYNTAX;
    }
  while(dig((int)p[i]))
    i++;
  return i;
}


/* seek(p,i) returns symtab index if p[] length i occurs in symtab, else
   -1. */


seek(p,i)
     char *p;
     int i;
{
  int j=0;
  while(COUNT(j))
    {
      if (COUNT(j)==i && same((char *)OCC(j),p,i))
	return j;
      j+=5;
    }
  return -1;
}



same(s,t,n)
     char *s,*t;
     int n;
{
  int i=0;
  while(i<n)
    {
      if (s[i]!=t[i])
	return 0;
      i++;
    }
  return 1;
}




/* countargs(p) computes the number of arguments to the functor
   preceding p, assuming the syntax to be ok. The number of arguments
   is computed as the number of commas at parenthesis level +1,
   plus 1. */

countargs(p)
     char *p;
{
  int paren=0;
  int commas=0;
  if (*p!='(')
    return 0;
  paren=1;
  while(paren>0 && *p!='\0') /* necessary since no syntax check */
    {
      int c;
      if ((c=(*++p))==',' && paren==1)
	commas++;
      else
	if (c=='(')
	  paren++;
	else
	  if (c==')')
	    paren--;
    }
  return commas+1;
}



/* checkeq(p) checks whether the atomic formula that continues at p
   (after an initial function or predicate symbol) is an equality.
   Again there is no syntax check; a non-zero return value only
   means that if we have an atomic formula, then it is an equality.
   What is checked is whether = is the first symbol after the
   closing parenthesis. */


checkeq(p)
     char *p;
{
  int paren=0;
  if (*p=='(')
    {
      paren=1;
      p++;
    }
  while(paren>0 && *p!='\0')
    {
      int c=(*p++);
      if (c=='(')
	paren++;
      if (c==')')
	paren--;
    }
  if (*p=='=')
    return 1;
  return 0;
}



/* getconn(p,&m) looks for a binary connective at p[] and returns
   an offset to the next char or an error number, and stores the
   connective in m */

getconn(p,m)
     char *p;
     int *m;
{
  char c=p[0];
  if (c=='&')
    {
      m[0]=AND;
      return 1;
    }
  if (c=='|')
    {
      m[0]=OR;
      return 1;
    }
  if (c=='-')
    {
      if (p[1]=='>')
	{
	  m[0]=IMP;
	  return 2;
	}
      push((long)p);
      return BADSYNTAX;
    }
  if (c=='<' && p[1]=='-' && p[2]=='>')
    {
      m[0]=IFF;
      return 3;
    }
  push((long)p);
  return BADSYNTAX;
}


makesym(count,occ,id,type,arity) /* returns index */
     long count,occ,id,type,arity;
{
  COUNT(symindex)=count;
  OCC(symindex)=occ;
  ID(symindex)=id;
  TYPE(symindex)=type;
  ARITY(symindex)=arity;
  symindex+=5;
  COUNT(symindex)=0;
  return symindex-5;
}



long *storeall(a,b,c,d)
     long a,b,c,d;
{
  push(a);
  push(b);
  push(c);
  push(d);
  return sp-4;
}

long *storesome(a,b,c)
     long a,b,c;
{
  push(a);
  push(b);
  push(c);
  return sp-3;
}

long *storeimp(a,b,c,d)
     long a,b,c,d;
{
  push(a);
  push(b);
  push(c);
  push(d);
  return sp-4;
}



long bvarid()
{
  return bvarcount++;
}


long predid()
{
  return predcount++;
}


long funcid()
{
  return funcount++;
}

long fvarid()
{
  return fvarcount++;
}


upper(i)
     int i;
{
  return('A'<=i && i<='Z');
}

lower(i)
     int i;
{
  return('a'<=i && i <='z');
}

dig(i)
     int i;
{
  return('0'<=i && i<='9');
}



formula removeiff(p)
     formula p;
{
  register formula r=p;
  formula f1,f2;
  switch((int)p[0])
    {
    case IFF:
      {
        f1=removeiff((long *)r[2]);
        f2=removeiff((long *)r[3]);
        sp[0]=IMP;
        sp[1]=r[1];
        sp[2]=(long)f1;
        sp[3]=(long)f2;
        sp[4]=IMP;
        sp[5]=r[1];
        sp[6]=(long)f2;
        sp[7]=(long)f1;
        sp[8]=AND;
        sp[9]=(long)sp;
        sp[10]=(long)(sp+4);
        sp+=11;
        return sp-3;
      }
    case IMP:
      {
        p[2]=(long)removeiff((long *)r[2]);
        p[3]=(long)removeiff((long *)r[3]);
        return p;
      }
    case AND:
      {
        p[1]=(long)removeiff((long *)r[1]);
        p[2]=(long)removeiff((long *)r[2]);
        return p;
      }
    case OR:
      {
        p[1]=(long)removeiff((long *)r[1]);
        p[2]=(long)removeiff((long *)r[2]);
        return p;
      }
    case ALL:
      {
        p[3]=(long)removeiff((long *)r[3]);
        return p;
      }
    case SOME:
      {
        p[2]=(long)removeiff((long *)r[2]);
        return p;
      }
    default:
      return p;
    }
}



isfalsum(p)
     char *p;
{
  return p[0]=='f'&&p[1]=='a'&&p[2]=='l'&&p[3]=='s'&&p[4]=='u'&&p[5]=='m';
}




