;; FRAPPS - unifylits.lsp

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;; Substitutions and unifiers, plus handling of literals.

;; =========== Functions for Handling Substitutions: ==========

;; Function "subst-s-exp" receives as input an arbitrary s-expression
;; (typically containing variables), and a binding list. It returns
;; the s-expression with the bindings of those variables appearing in
;; the binding list substituted in for the variables in the s-expression.
;;
;; NOTES: This function composes substitutions (i.e - if "x=y", and "y=a" 
;;        appear in the binding list, then "a" will be substituted in for 
;;        the variable "x" in the s-expr) by invoking "compose-bindings".
;;        Function "var-p" is invoked to determine if a particular item
;;        is a variable.

(defun subst-s-exp (item binding-list)
  (cond
    ((atom item) item)
    ((var-p item) (compose-bindings item binding-list))
    ((or (atom (car item)) (var-p (car item)))
     (cons 
       (if (var-p (car item))
	   (compose-bindings (car item) binding-list)
	 (car item))
       (subst-s-exp (cdr item) binding-list)))
    (T (cons (subst-s-exp (car item) binding-list)
	     (subst-s-exp (cdr item) binding-list)))))

       

;; Function "find-binding" receives as input a variable, and a binding
;; list. If the variable appears in the list, its binding is returned,
;; otherwise "NIL" is returned.  

(defun find-binding (var bindings)
  (cdr (assoc var bindings :test #'equal)))


;; Function "compose-bindings" receives as input a variable to bind, and a 
;; binding list. It returns the result of composing all of the bindings
;; (substitutions) that apply to the variable to be bound (e.g - if the 
;; var to bind is "x", and the binding list contains the bindings "x=y", 
;; and "y=a", then "a" will be returned as the value to substitute in
;; for the variable "x").
;;
;; NOTES: "Find-binding" is used to determine the binding of the variable
;;        passed to "compose-bindings".

(defun compose-bindings (item-to-bind binding-list)
  (let ((find-rslt nil))
    (cond 
      ((atom item-to-bind)
       item-to-bind)
      ((var-p item-to-bind) 
       (if (setq find-rslt (find-binding item-to-bind binding-list))
	   (compose-bindings find-rslt binding-list)
	 item-to-bind))
      (T (subst-s-exp item-to-bind binding-list)))))

;; "Collapses" a substitution to put it into its simplest form.

(defun collapse-subst (subst)
  (if (null subst) nil
      (if (equal (car (car subst)) (cdr (car subst)))
	  ;; NOTE that substitutions are of the form (cons <variable> term).
	  (collapse-subst (cdr subst))
	  (cons (car subst)
		(collapse-subst
		 (subst-s-exp (cdr subst) (list (car subst)))))
	  )))


;; Function "compose-substns" composes the two input substitutions theta1,
;; and theta2, and returns the resulting substitution.

(defun compose-substns (theta1 theta2)
  (let ((bind-list nil) (term-vars-list nil) (theta1-comp-theta2 nil)
	(theta1-var-lst nil) (subst-rslts nil))

    (dolist (var-term-pair theta1)

	    (setq theta1-var-lst (cons (car var-term-pair) theta1-var-lst))
	    (cond 
	      ((listp (cdr var-term-pair))
	       (setq term-vars-list (bld-var-list (cdr var-term-pair)))

	       (dolist (term-var term-vars-list)

		       (setq bind-list (cons (assoc term-var 
						    theta2 :test #'equal)
					     bind-list)))

	       (if (not (equal (car var-term-pair)
			       (setq subst-rslts 
				     (subst-s-exp (cdr var-term-pair)
						  bind-list))))
		   (setq theta1-comp-theta2 
			 (append theta1-comp-theta2 
				 (list (cons (car var-term-pair) 
					     subst-rslts)))))

	       (setq bind-list nil))
	      (T (setq theta1-comp-theta2 (append theta1-comp-theta2 
						  (list var-term-pair))))))

    (dolist (var-term-pair theta2 theta1-comp-theta2)

	    (if (not (member (car var-term-pair) theta1-var-lst :test #'equal))
		(setq theta1-comp-theta2 (append theta1-comp-theta2
						 (list var-term-pair)))))))




;;;; ====================== BEGIN UNIFICATION FUNCTIONS =======================

;;;;  The following functions are used in the implementation of the 
;;;;  ***UNIFICATION*** algorithm
;;;;

;;;; NORMAL unifier: (for hooked-on-FRAPPS)

(defun simple-unify (lit1 lit2 &optional const1 const2)
  ;; ignores CONSTRAINT parameters.
  (declare (ignore const1) (ignore const2))
  (let ((rslt (unify lit1 lit2)))
       (if (not (eq rslt 'FAIL))
	   (list (list rslt nil))
	   'FAIL)
       ))


;; this function provides a simple unification algorithm with occurs-check;
;; it does NOT compose bindings as it goes
;;
;;  NOTE: this function assumes that a variable has the form: 
;;
;;              (*var* <var. name> <var. subscript>)

(defun unify (pat1 pat2 &optional bindings)
  (cond 
    ((equal pat1 pat2)
     bindings)
    ((var-p pat1) 
     (var-unify pat1 pat2 bindings))
    ((var-p pat2) 
     (var-unify pat2 pat1 bindings))
    ((or (not (listp pat1)) (not (listp pat2))) 
     'FAIL)
    ((not (eq (setq bindings (unify (car pat1) (car pat2) bindings)) 'FAIL))
     (unify (cdr pat1) (cdr pat2) bindings))
    (t 'FAIL)))


;;  determines if the given variable will unify with the given pattern in the
;;  context of the given variable bindings

(defun var-unify (var pat bindings)
  (let ((val (find-binding var bindings)))
       (cond 
	(val (unify val pat bindings))
	((and (free-in var pat bindings) pat) ; pat can't be nil
	 (cons (cons var pat) bindings))
	((var-p pat)	;; Corrected bug in original version
	 (if (equal var (compose-bindings pat bindings))
	     bindings 'FAIL))
	(t 'FAIL))))



;;  determines if the given variable is "free in" (i.e. - does not occur in) 
;;  the given pattern in the context of the given bindings

(defun free-in (var pat bindings)
  (cond 
    ((null pat) t)
    ((equal var pat) nil)
    ((var-p pat) (free-in var (find-binding pat bindings) bindings))
    ((not (listp pat)) t)
    ((free-in var (car pat) bindings) (free-in var (cdr pat) bindings))))



;;  determines if the given item is a "variable"

(defun var-p (item)
  (and (listp item) (eq (car item) '*var*)))


;; unify-p is now in infprims.lsp / hinfprims.lsp


;;;; ====================== END UNIFICATION FUNCTIONS =========================



;; Function "bld-var-list" returns a list of the variables found in the
;; original s-expression passed to it. If no variables are found in the 
;; expression, "NIL" is returned. Note: Duplicates are not removed.
   
(defun bld-var-list (s-exp)
  (cond
    ((null s-exp) nil)
    ((var-p s-exp) (list s-exp))
    ((atom (car s-exp)) (bld-var-list (cdr s-exp)))
    ((var-p (car s-exp)) (cons (car s-exp) (bld-var-list (cdr s-exp))))
    (T (append (bld-var-list (car s-exp))
	       (bld-var-list (cdr s-exp))))))

;; (defun bld-var-set (s-exp)
  ;; (remove-duplicates (bld-var-list (s-exp)) :test #'equal)
  ;; )

;;  returns the predicate symbol of the given literal

(defun get-pred-sym (lit)
  (cond
    ((neg-lit-p lit) (second lit))
    (T (first lit))))


;;  returns a list of the predicate symbols found in the
;;  given literal list

(defun get-all-pred-syms (lit-list)
  (let ((pred-sym-list nil))
    (dolist (lit lit-list pred-sym-list)
	    (setq pred-sym-list
		  (union pred-sym-list (list (get-pred-sym lit)))))))



;;  returns the arguments of the given literal

(defun get-lit-args (lit)
  (if (neg-lit-p lit)
      (nthcdr 2 lit)
    (rest lit)))


;;  determines whether the given literal is negative

(defun neg-lit-p (lit)
  (eq (car lit) 'not))

;; NOTE: The following predicates used to take NODE-IDS as their arguments;
;; now they take clauses.

;;  determines whether the given clause is negative
;;  
;;  NOTE: if the clause contains *NO* literals (or no "non-answer" literals),
;;	T is returned

(defun neg-cls-p (lit-list)
  ;; (let ((lit-list (get-node-clause cls-id)))
    (dolist (lit lit-list T)
	    (if (and (not (neg-lit-p lit)) (not (ans-lit-p lit)))
		(return nil)
		))) ;; )

;;  determines whether the given clause is positive
;;
;;  NOTE: if the given clause contains *NO* literals (or no "non-answer"
;;        literals), then T is returned

(defun pos-cls-p (lit-list)
  ;; (let ((lit-list (get-node-clause cls-id)))
    (dolist (lit lit-list T)
	    (if (and (neg-lit-p lit) (not (ans-lit-p lit)))
		(return nil)
		)));;)


;;  determine whether the given clause is a definite clause, if it is 
;;  then the index of the POSITIVE LITERAL is returned, otherwise "nil" 
;;  is returned
;;
;;  NOTE: a base "1" index is assumed for the purposes of indexing the 
;;        clause literals

(defun def-cls-p (lit-list)
  (let (;; (lit-list (get-node-clause cls-id))
	(pos-lit-fnd nil)
	(i 1))
    (dolist (lit lit-list pos-lit-fnd)
	    (if (and (not (neg-lit-p lit)) (not (ans-lit-p lit)))
		(if pos-lit-fnd
		    (return nil)
		  (setq pos-lit-fnd i)))
	    (setq i (1+ i)))))


;;  pre-process the given clause; build its indexed predicate list,
;;  and determine its maximum subscript

(defun pre-process-cls (cls)
  (append (list (bld-pred-list cls))
	  (list (fnd-max-subscr cls))))



;;  build the indexed predicate list for the given clause
;;
;;  format: ( (p1 (pos p1 lits) (neg p1 lits))... (pn (pos pn lits) (neg pn lits)) )
;;
;;  where p1,p2,...,pn are the distinct predicate symbols
;;  in the clause; (pos pi lits) are the INDICES of the positive literals
;;  that the predicate "pi" occurs in, and (neg pi lits) are the INDICES of 
;;  the negative literals that the predicate "pi" occurs in.

(defun bld-pred-list (cls)
  (let ((pred-sym nil) (pred-list nil) (p-lst nil) (index 1))
    (dolist (lit cls pred-list)
	    (setq pred-sym (get-pred-sym lit))
	    (cond
	      ((setq p-lst (assoc pred-sym pred-list :test #'equal))
	       (setq pred-list 
		     (if (neg-lit-p lit)
			 (subst (list pred-sym 
				      (second p-lst) 
				      (append (third p-lst) (list index)))
				p-lst
				pred-list)
		       (subst (list pred-sym 
				    (append (second p-lst) (list index))
				    (third p-lst))
			      p-lst
			      pred-list))))
	      (T (setq pred-list
		       (if (neg-lit-p lit)
			   (append pred-list (list (list pred-sym nil (list index))))
			 (append pred-list (list (list pred-sym (list index) nil)))))))
	    (setq index (1+ index)))))




;;  build the indexed predicate list for the given clause
;;
;;  format: ( (p1 #1#...#n#) (p2 #1#...#n#)...(pn #1#...#n#) )
;;          where p1,p2,...,pn are the distinct predicate symbols
;;          in the clause; and #1#...#n# are the indices of the literals
;;          that predicate "pi" occurs in.

(defun old-bld-pred-list (cls)
  (let ((pred-sym nil) (pred-list nil) (p-lst nil) (index 1))
    (dolist (lit cls pred-list)
	    (setq pred-sym (get-pred-sym lit))
	    (cond
	      ((setq p-lst (assoc pred-sym pred-list :test #'equal))
	       (setq pred-list 
		     (subst (append p-lst (list index)) p-lst pred-list)))
	      (T (setq pred-list
		       (append pred-list (list (list pred-sym index))))))
	    (setq index (1+ index)))))




;;  determines the maximum subscript found in the given clause

(defun fnd-max-subscr (cls)
  (let ((max-subscr 1))
    (dolist (var (bld-var-list cls) max-subscr)
	    (cond
	      ((> (third var) max-subscr) (setq max-subscr (third var)))))))



;;  standardizes apart the variables in two
;;  LITERAL LISTS (rather than clause STRUCTURES)

(defun stndze-vars-apart (cls1 cls2)
  (let ((cls1-max-sub (fnd-max-subscr cls1))
	(cls2-max-sub (fnd-max-subscr cls2)))
       (inc-var-subscrs cls2
			(max cls1-max-sub cls2-max-sub))))



;;  determines if the two literals given as input are of opposite sign;
;;  returns following values:
;;  --> "0" if c1-lit is neg. and c2-lit is pos.
;;  --> "1" if c1-lit is pos. and c2-lit is neg.
;;  --> "nil" if the two literals are of the same sign

(defun opp-sign-p (c1-lit c2-lit)
  (cond 
    ((and (neg-lit-p c1-lit) (not (neg-lit-p c2-lit))) 0)
    ((and (not (neg-lit-p c1-lit)) (neg-lit-p c2-lit)) 1)
    (T nil)))


;; removes n-th literal from the clause given as input
;;
;; NOTE: clauses are indexed starting at "1"

(defun rmv-nth-lit (n cls)
  (cond
    ((or (<= n 0) (> n (length cls))) cls)
    ((= n 1) (cdr cls))
    (T (cons (car cls) (rmv-nth-lit (1- n) (cdr cls))))))


;;  merges left duplicate literals appearing in a clause

(defmacro merge-left (cls)
  `(remove-duplicates ,cls :from-end t :test #'equal))


;;  "strip" the variable names and subscripts from the given literal

(defun strip-var-names-and-subscrs (s-exp)
  (cond
   ((null s-exp) nil)
   ((atom s-exp) s-exp)
   ((var-p (first s-exp))
    (cons (list (first (first s-exp)))
	  (strip-var-names-and-subscrs (rest s-exp))))
   (T (cons (strip-var-names-and-subscrs (first s-exp))
	    (strip-var-names-and-subscrs (rest s-exp))))))

