;; FRAPPS - output.lsp

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Shared by H-Frapps.

;; Output functions.
;; Also includes macro abbreviations for long FRAPPS function names.

;; Some of these functions are not in the manual but
;; might be useful to the user...

;; 		=============================

;; Print ALL global variables visible to the user:

(defun print-globals ()
  (print-graph-globals)
  (format t "~%")
  (print-flags)
  )

;;  output all the clauses in the database by level up to, and including,
;;  the specified termination level:

(defun print-downto-level (term-level)
  (dotimes (level term-level) (print-level level)))

(defun print-level (level)
  (cond ((get-ids-at-level level)
	 ;; (format t "~% Level ~d:" level)
	 ;; (format t "~% ID  Parents  Clause ~%")
	 (print-cls-lst (gial level))
	 )
	)
  (values))

;; Was called "print-clause" before. Not visible to user, does not
;; do error check:

(defun print-node-clause (id)
  (let ((clause (get-node-clause id)))
       (cond (clause
	      (p-clause clause)
	      (user-print id)	;; In case user wants to print additional info.
	      )
	     (T (format t "{ }"))
	     )))

;; Visible to user:

(defun print-clause (id)
  (let ((cls-info (get-real-cls-info id)))
       (cond
	(cls-info
	 (if (clause-lit-list cls-info)
	     (p-clause (clause-lit-list cls-info))
	     (format t "{ }"))
	 (user-print id)
	 )
	(*print-access-errors* (err-messg-nonexistent id))
	))
  (values))

;; user-print:
;; This function can be used to print constraints (in the case of hooks),
;; user fields, etc.  Note that it takes an ID as an argument.
;; The following is the default.

(defun user-print (id)
  (format t " ~d" (get-node-active id)))

(defun print-var (var)
  (format t "?~d~d" (second var) (third var)))

(defun p-clause (clause)
  (cond
   ((atom clause) (if clause (format t "~d" clause)))
   ((var-p clause) (print-var clause))
   (T (format t "(")
	      (let ((i 0) (l (length clause)))
		   (dolist (elem clause)
			   (progn (p-clause elem)
				  (setf i (+ 1 i))
				  (if (< i l) (format t " ") nil)
				  )))
	      (format t ")")
      )))


;; prints out a list of clauses and their parents according to a list of ids: 
 
(defun print-cls-lst (id-lst)
  (dolist (id id-lst nil)
	  (format t " ~d:" id)
	  (format t " ~d " (get-node-parents id))
	  (print-node-clause id)
	  (format t "~%")
	  ))

;;  output the pertinent info of a clause structure
;; (defun output-reslvnt (cls)
  ;; (format t "~% clause id: ~d   clause: ~d ~%    left parent id:~d   right parent id:~d~%"
	  ;; (clause-id cls) (clause-lit-list cls) 
	  ;; (clause-left-par cls) (clause-right-par cls)))


;;  print the contents of a hash-table (for debugging)

(defun print-hash-table-contents (hash-table)
  (format t "~%  The hash-table contains ~d entries, they are..."
	  (hash-table-count hash-table))
  (maphash #'(lambda (key val)
	       (format t "~2% ==> key: ~d ~% ==>value: ~d" key val))
	   hash-table))


;;  open an output stream
;; (defun opn-str (str-name)
  ;; (setq *alt-str-ptr* *str-ptr*)
  ;; (setq *str-ptr* (open str-name :direction :output)))

;;  close an output stream
;; (defun clos-str ()
  ;; (close *str-ptr*))

;;  switch output streams
;; (defun swstr ()
  ;; (let ((tmp nil))
    ;; (setq tmp *str-ptr*)
    ;; (setq *str-ptr* *alt-str-ptr*)
    ;; (setq *alt-str-ptr* tmp)))


(defun get-derivation (node-id)
  (let ((cls (get-real-cls-info node-id)))
       (cond (cls
	      (make-proof-list (cons node-id nil) nil nil))
	     (*print-access-errors*
	      (err-messg-nonexistent node-id))
	     (T 'F-ERROR)
	     )))

(defun make-proof-list (pend list done)
  (if (null pend) list
    (let ((x (car pend)))
      (if (= (get-node-level x) 0) (make-proof-list (cdr pend) list done)
        (if (not (member x done))
	    (make-proof-list (append (get-node-parents x) (cdr pend))
                         (cons (cons (get-node-parents x) (list x)) list)
			 (cons x done)
			 )
	    (make-proof-list (cdr pend) list done)
	    )))))

(defun printnode (node-id)
  (printspace (get-node-level node-id))
  (format t " ~D:" node-id)
  (print-node-clause node-id)
  (if (> (get-node-level node-id) 0) (format t " ~d" (get-parents node-id)))
  (format t "~%")
  )


; (defun old-print-derivation (node-id)
  ; (let ((cls (get-cls-info node-id)))
       ; (cond ((not (null cls))
	      ; (dolist (pair (get-derivation node-id))
		      ; (printnode (first pair))
		      ; (printnode (second pair))
		      ; )
	      ; (printnode node-id))
	     ; (T (err-messg-nonexistent node-id))
	     ; ))
  ; (values))

;; The following one never prints a node twice, structure is clearer
;; (children always follow parents):

(defun print-derivation (node-id)
  (let ((cls (get-real-cls-info node-id))
	(printed nil))
       (cond ((not (null cls))
	      (dolist (x (get-derivation node-id))
		      (if (not (member (first (car x)) printed))
			  (printnode (first (car x))))
		      (if (not (member (second (car x)) printed))
			  (printnode (second (car x))))
		      (printnode (cadr x))
		      (setq printed (cons (cadr x) (union (car x) printed)))
		      )
	      ;; (printnode node-id)
	      )
	     (*print-access-errors* (err-messg-nonexistent node-id))
	     ))
  (values))


(defun printspace (n)
  (princ "-")
  (dotimes (x n)
         (princ "------")
	 ))


;;		 ====== ERROR Detection Functions =======

;;  determine the particular "error condition" (using the given clause(s) as
;;  evidence) that has been detected while attempting to derive new nodes

;; *print-inference-errrors* has already been checked.

(defun determine-error (cls1-id &optional (cls2-id nil cls2-id-flg))
  (cond
   ((and *print-access-errors* (not (is-a-node-p cls1-id)))
    (err-messg-nonexistent cls1-id))
   ((not (get-node-active cls1-id))
    (err-messg-deactivated cls1-id))
   )
  (cond
   ((and cls2-id-flg *print-access-errors* (not (is-a-node-p cls2-id)))
    (err-messg-nonexistent cls2-id))
   ((and cls2-id-flg (not (get-node-active cls2-id)))
    (err-messg-deactivated cls2-id))
   ))

;; Error and Miscellaneous Messages:

(defun err-messg-deactivated (id)
  ;; *print-inference-errors* is always checked before this one is called.
  (format t "~%  The node ~d has been DEACTIVATED.~%" id)
  'F-ERROR)

(defun err-messg-nonexistent (id)
  ;; *print-access-errors* is always checked before this one is called.
  (format t "~%  The node ~d does NOT exist in the derivation graph.~%" id)
  'F-ERROR)

;; The following is no longer in use since user-defined node-ids are not
;; allowed:

; (defun err-messg-wrong-id (id)
  ; (cond
   ; ((integerp id)
    ; (format t "~%  Warning: User-defined node-id's should not be integers."))
   ; ((not (atomp id))
    ; (format t "~%  Warning: User-defined node-id's must be atoms."))
   ; (T (format t "~%  Warning: The node ~d already exists." id))
   ; )
  ; (format t "~%  A new integer id will be assigned instead. Sorry.~%"))


(defun err-messg-subsume (function-name)
  (format t "~%  ~D:~%  The key <:degree> argument can only be 'full, 'instances, or 'variants.~%"
	  function-name)
  )

(defun err-messg-wrong-subsume-flags ()
  (format t "~%  Warning: *answer-subsume* and *weak-subsume* cannot be both ON")
  (format t "~%  at the same time. Reset one of them to nil.~%")
  )



;; Default message printed when backward subsumption deactivates a node:
;; First argument is a node-id, second is a clause.

(defun found-back-sub-msg (id clause)
  (format t "~% Node ~d back subsumed by resolvent " id)
  (p-clause clause)
  (format t "~%")
  )


;; Read an input "Y" or "N"; return NIL if input is "N" and T if it is Y:

(defun user-choice ()
  (do ((answer nil))
      ((or (string-equal answer "y") (string-equal answer "n"))
       (if (string-equal answer "y") T nil))
      (format t "~% Enter \"y\" or \"n\": ")
      (setq answer (read-line))
      ))


;;		 ------- Function Abbreviations and renamings: -------

(defun gial (level)
  (get-ids-at-level level))

(defun gni (id)
  (get-node-info id))

(defun gnal (level)
  (get-nodes-at-level level))

(defun gndl (&optional level)
  (get-nodes-downto-level level))

(defun gidl (&optional level)
  (get-ids-downto-level level))

(defun answer-lit-p (x)
  (ans-lit-p x))

(defun giol (n)
  (get-ids-of-length n))

