;; FRAPPS - infprims.lsp 

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;      ====> Inference Rule Primitives <====
;;;;
;;;;  This file contains those functions which together comprise the inference
;;;;  rule primitives. These functions are intended to provide a "foundation"
;;;;  for building more sophisticated inference rules.

;; NOTE: The depth of a clause is now computed in the "integrate-clause"
;; function in "dbss.lsp". It used to be carried around as an extra
;; parameter for quite a while, but can be more efficiently computed by
;; looking at the parents's depth when the resolvent or factor is integrated.


;;  ====> GENERAL RESOLUTION <====
;;
;;  The functions that support the implementation of general resolution are 
;;  broken down into two categories; those which support the implementation of
;;  BINARY RESOLUTION, and those which support the implementation of FACTORING.
;;
;;  The general resolution routine "resolve" implements the general
;;  resolution rule of inference by factoring, and then applying binary
;;  resolution to the original clauses and ALL their non-trivial factors




;;  computes **ALL** binary resolvents of the set of clauses consisting of the
;;  two input clauses and **ALL** of their non-trivial factors (*if* the 
;;  factoring option is enabled); the first input clause (and all its 
;;  non-trivial factors) is/are matched up against the second input clause 
;;  (and all its non-trivial factors) in a pairwise fashion.
;;
;;  NOTE: if the global flag *factor-flag* is NON-nil, then factoring
;;        will be performed, otherwise only BINARY resolution will be
;;        performed in any given resolution step
;;
;;        if *factor-flag* is NIL (i.e. - only binary resolution will be
;;        performed), then in addition to specifying the clauses to resolve
;;        together, the user has the OPTION of specifying the particular 
;;        LITERALS to resolve upon as well; if *factor-flag* is NON-nil, 
;;        then this option is NOT available (i.e. - any clash literal 
;;        specifications are ignored)
;;
;;        the clauses denoted by "c1-id" and "c2-id" MUST be integrated into
;;        the global clause database PRIOR to attempting to resolve them
;;        together; if either, or both, have not, then 'ERROR is returned

(defun resolve (cl1-id cl2-id &key c1-lit c2-lit)
  (let ((c1 (get-cls-info cl1-id))
	(c2 (get-cls-info cl2-id))
	;; NOTE: This copies the STRUCTURES.
	(cl1-f-lst nil) 
	(cl2-f-lst nil) 
	(resolvents nil)
	(mgus nil)
	(resolvents-list nil)
	(mgus-list nil))
    (cond
      ((and (not (or (null c1) (null c2)))
	    (clause-active c1) (clause-active c2))
       (cond
	 ((not *factor-flag*) (b-resolve c1 c2 :c1-lit c1-lit :c2-lit c2-lit))
	 (T (setq cl1-f-lst (append (list c1) (mapcar #'conv-res-info-factor-to-clause
						      (factor cl1-id))))
	    (setq cl2-f-lst (append (list c2) (mapcar #'conv-res-info-factor-to-clause
						      (factor cl2-id))))
	    (dolist (cl1 cl1-f-lst (values resolvents-list mgus-list))
		    (dolist (cl2 cl2-f-lst)
			    (multiple-value-setq (resolvents mgus)
						 (b-resolve cl1 cl2))
			    (setq resolvents-list 
				  (append resolvents-list resolvents))
			    (setq mgus-list (append mgus-list mgus)))))))
      (*print-inference-errors* (determine-error cl1-id cl2-id)))))

;; The following is a binary resolution function that ignores the factor-flag,
;; in case that the user wants to combine both binary and general
;; resolution.

(defun binary-resolve (cl1-id cl2-id &key c1-lit c2-lit)
  (let ((c1 (get-cls-info cl1-id))
	(c2 (get-cls-info cl2-id)))
       ;; NOTE: This copies the STRUCTURES.
       (cond
	((and (not (or (null c1) (null c2)))
	      (clause-active c1) (clause-active c2))
	 (if (null *factor-flag*) ;; Still have to do this to get label right!
	     (b-resolve c1 c2 :c1-lit c1-lit :c2-lit c2-lit)
	     (let ((rslvnts nil) (mgus nil))
		  (setq *factor-flag* nil)
		  (multiple-value-setq (rslvnts mgus)
				       (b-resolve c1 c2 :c1-lit c1-lit :c2-lit c2-lit))
		  (setq *factor-flag* 'T)
		  (values rslvnts mgus)
		  )
	     ))
	(*print-inference-errors* (determine-error cl1-id cl2-id)))))


;;  =====> BINARY RESOLUTION <=====
;;
;;  resolve the given clauses against one another; if the two optional 
;;  parameters are specified (i.e. - both of the clash literals are fully 
;;  specified), then attempt to resolve the clauses against one another 
;;  considering *only* the specified clash literals; if only one of the two 
;;  optional parameters is specified, then attempt to resolve the given 
;;  clauses together using the given clash literal of the one clause, and 
;;  *all* literals of the other clause; if neither of the optional parms. 
;;  is specified, then consider all possible pairs of literals (a pair 
;;  consisting of one literal from each clause) as potential clashing pairs;
;;  returns a list of clause STRUCTURES which represent the resolvents formed
;;  from the two parent clauses specified
;;
;;  NOTE: "c1" and "c2" refer to clause STRUCTURES
;;
;;        "c1" and "c2" are standardized apart from within this function

(defun b-resolve (c1 c2 &key c1-lit c2-lit)
  (setf (clause-lit-list c2) (struct-stndze-vars-apart c1 c2))  ;; rename vars. 
  ;; note that this is a "destructive" operation, so this is why the clause
  ;; STRUCTURES themselves should be copies of the original
  ;; (though not the internal fields).
  (cond
   ((and c1-lit c2-lit)
    (rslv-clsi-litm-clsj-litn c1 c1-lit c2 c2-lit))
   ((or c1-lit c2-lit)
    (if c1-lit
	(rslv-clsi-litm-clsj c1 c1-lit c2)
	(rslv-clsi-litm-clsj c2 c2-lit c1)))
   (T (rslv-clsi-clsj c1 c2))))




;;  resolve the two given clauses together, considering all possible pairs of
;;  literals between the two as potential clashing pairs; return the resulting
;;  list of resolvents
;;
;;  NOTE: If the two given clauses happen to be VARIANTS of each other,
;;        then NO DUPLICATE resolvents are produced, this is accomplished in
;;        the following way:
;;
;;         Since c1 and c2 are VARIANTS of each other, we must have:
;;
;;           c1 = L1 v L2 v ... v Ln and c2 = L1' v L2' v ... v Ln'
;;
;;         where Li and Li' are variants of each other;
;;
;;         thus, resolve(c1,i,c2,j) will be a PERMUTED VARIANT of 
;;         resolve(c1,j,c2,i), so we need not perform BOTH of the resolutions
;;

(defun rslv-clsi-clsj (c1 c2)
  (let* ((resolvents nil)
	 (mgus nil)
	 (resolvents-list nil)
	 (mgus-list nil))
    (dotimes (c1-lit-index (length (clause-lit-list c1)) (values resolvents-list mgus-list))
	     (multiple-value-setq (resolvents mgus)
				  (rslv-clsi-litm-clsj c1 (1+ c1-lit-index) c2))
	     (cond
	       (resolvents
		 (setq resolvents-list (append resolvents-list resolvents))
		 (setq mgus-list (append mgus-list mgus)))))))




;;  resolve the two given clauses together, using the specified literal as one
;;  of the clash literals, and considering all literals of the other clause
;;  as the other clash literal

(defun rslv-clsi-litm-clsj (c1 c1-index c2)
  (let* ((c1-id (clause-id c1))
	 (c1-lit-list (clause-lit-list c1))
	 (c1-clash-lit (nth (1- c1-index) c1-lit-list))
	 (c1-clash-lit-pred (get-pred-sym c1-clash-lit))
	 (c2-id (clause-id c2))
	 (c2-lit-list (clause-lit-list c2))
	 (c2-pred-list (clause-pred-list c2))
	 (c2-clash-pred-indx-lst 
	   (if (neg-lit-p c1-clash-lit)
	       (second (assoc c1-clash-lit-pred c2-pred-list :test #'equal))
	     (third (assoc c1-clash-lit-pred c2-pred-list :test #'equal))))
	 (c1-eq-c2 (eq c1-id c2-id))
	 (resolvent nil)
	 (mgu nil)
	 (resolvents-list nil)
	 (mgus-list nil))
    
    ;;;;(format t "~% ===&&&&&====> rslv-clsi-litm-clsj <====&&&&&====")
    ;;;;(format t "~% c1-lit-list: ~d " c1-lit-list)
    ;;;;(format t "~% c2-lit-list: ~d " c2-lit-list)
    ;;;;(format t "~% c1-index: ~d " c1-index)
    ;;;;(format t "~% c1-clash-lit: ~d " c1-clash-lit)
    ;;;;(format t "~% c2-clash-pred-indx-lst: ~d " c2-clash-pred-indx-lst) 
    ;;;;(format t "~% c1-eq-c2: ~d " c1-eq-c2)
    ;;;;(read)

    (cond
      ((null c2-clash-pred-indx-lst) (values nil nil))
      (T (dolist (c2-clash-lit-index c2-clash-pred-indx-lst (values resolvents-list 
								    mgus-list))

		 ;;;;(format t "~% c2-clash-lit-index: ~d " c2-clash-lit-index)
		 ;;;;(read)
		 
		 (cond
		   ((or (not c1-eq-c2) (<= c2-clash-lit-index c1-index))
		    (multiple-value-setq (resolvent mgu)
					 (compute-resolvent c1-id c1-lit-list c1-index
							    c2-id c2-lit-list c2-clash-lit-index
							    ;; (1+ (max (clause-depth c1) (clause-depth c2)))
							    ))
		    (cond
		      (resolvent
			(setq resolvents-list (append resolvents-list (list resolvent)))
			(setq mgus-list (append mgus-list (list mgu))))))
		   (T (return (values resolvents-list mgus-list)))))))))



;;  resolve the two given clauses together, using the specified clause indices
;;  as denoting the two clash literals

(defun rslv-clsi-litm-clsj-litn (c1 c1-index c2 c2-index)
  (let* ((c1-lit-list (clause-lit-list c1))
	 (c2-lit-list (clause-lit-list c2))
	 ;; (c1-clash-lit (nth (1- c1-index) c1-lit-list))
	 ;; (c2-clash-lit (nth (1- c2-index) c2-lit-list))
	 (resolvent nil)
	 (mgu nil))

    ;;;;(format t "~% =======> rslv-clsi-litm-clsj-litn <=============")
    ;;;;(format t "~% c1-lit-list: ~d " c1-lit-list)
    ;;;;(format t "~% c2-lit-list: ~d " c2-lit-list)
    ;;;;(format t "~% c1-index: ~d " c1-index)
    ;;;;(format t "~% c2-index: ~d " c2-index)
    ;;;;(format t "~% c1-clash-lit: ~d " c1-clash-lit)
    ;;;;(format t "~% c2-clash-lit: ~d " c2-clash-lit)
    ;;;;(read)

    (cond
      ((eq (get-pred-sym (nth (1- c1-index) c1-lit-list))
	   (get-pred-sym (nth (1- c2-index) c2-lit-list)))
       (multiple-value-setq (resolvent mgu)
			    (compute-resolvent (clause-id c1) c1-lit-list c1-index
					       (clause-id c2) c2-lit-list c2-index
					       ;; (1+ (max (clause-depth c1) (clause-depth c2)))
					       ))
       (if resolvent
	   (values (list resolvent) (list mgu))
	 (values nil nil)))
      (t (values nil nil)))))



;;  attempt to perform the completely specified resolution operation;
;;  if the resolution can be performed, then compute the resolvent, and 
;;  return the resulting clause structure, otherwise return "nil"

(defun compute-resolvent (c1-id c1-lit-list c1-index c2-id c2-lit-list c2-index)
  ;; depth
  (let* ((c1-clash-lit (nth (1- c1-index) c1-lit-list))
	 (c2-clash-lit (nth (1- c2-index) c2-lit-list))
	 (unify-rslts nil))

    ;;;;(format t "~% $$$$$$=======> COMPUTE RESOLVENT <=======$$$$$")
    ;;;;(format t "~% c1-lit-list: ~d " c1-lit-list)
    ;;;;(format t "~% c2-lit-list: ~d " c2-lit-list)
    ;;;;(format t "~% c1-index: ~d " c1-index)
    ;;;;(format t "~% c2-index: ~d " c2-index)
    ;;;;(format t "~% c1-clash-lit: ~d " c1-clash-lit)
    ;;;;(format t "~% c2-clash-lit: ~d " c2-clash-lit)
    ;;;;(read)
    
    (cond
      ((not (eq (setq unify-rslts (unify-p c1-clash-lit c2-clash-lit))
		'FAIL))
       (values (bld-resolvent c1-id c1-lit-list c1-index
		      c2-id c2-lit-list c2-index
		      unify-rslts ;; depth
		      )
	       unify-rslts))
      (T (values nil nil)))))




;;  builds the resolvent of two clauses which have been determined "a-priori"
;;  to resolve against one another using the specified clash literals

(defun bld-resolvent (c1-id c1-lit-list c1-index 
			    c2-id c2-lit-list c2-index 
			    unify-rslts ;; depth
			    )
  (construct-resolvent-info
   (merge-left (subst-s-exp (append (rmv-nth-lit c1-index c1-lit-list)
				    (rmv-nth-lit c2-index c2-lit-list))
			    unify-rslts))
   (list c1-id c2-id)
   (if *factor-flag* 'gr 'br))) ;; NEW




;; ====> Factoring <====
;;
;; The following functions support the implementation of FACTORING; some of the
;; functions used in implementing the functions below can be found in the file
;; "tpmisc.lsp" 



;;  produces *ALL* "non-trivial" factors of a given clause
;;
;;  Note: expects a clause ID corresponding to an integrated (i.e. - permanent)
;;  clause in the global clause database; if this is not the case, then a 
;;  "suitable" error message is returned; in addition, an error message is 
;;  returned if the specified clause has been "de-activated".
;;
;;        If the clause ID corresponds to a proposition (i.e. - a clause 
;;        containing no variables), then "nil" is returned if the clause 
;;        contains no duplicate literals (since the clause has no "nontrivial"
;;        factors), otherwise the clause is returned with all duplicate
;;        literals removed.
;;
;;        If factoring is being used in the "context" of general resolution 
;;        (i.e. - in conjunction with binary resolution), then ALL factors 
;;        produced will "inherit" their PARENT'S ID (i.e. - all factors will
;;        have the SAME ID as their PARENT). 
;;        If instead factoring is being used in a "stand-alone" fashion, then
;;        each factor will have a unique ID created for it.
;;
;;        Answer literals *ARE* factored upon!!
;;
;;;;     ===> THIS VERSION OF FACTOR **DOES NOT** COMPOSE SUBSTITUTIONS BEFORE
;;;;     ===> APPLYING THEM TO THE CLAUSE EXPRESSION, INSTEAD SUBSTITUTIONS ARE
;;;;     ===> SUCCESSIVELY APPLIED TO THE CLAUSE EXPRESSION ONE AFTER ANOTHER.


(defun factor (cls-id)
  (let ((cls (get-cls-info cls-id)))
    (cond 
      ((and (not (null cls)) (clause-active cls))
       (let ((cls-pred-list (clause-pred-list cls)) 
	     (cls-lit-list (clause-lit-list cls))
	     (merged-cls nil)
	     (factor-list nil))
	 (cond  
	   ((null (bld-var-list cls-lit-list))
	    (if (not (equal cls-lit-list (setq merged-cls (merge-left cls-lit-list))))
		(list (construct-resolvent-info 
			merged-cls 
			(list cls-id) 
			'f))
	      nil))
	   (T (dolist (pred-sublist cls-pred-list factor-list)
		      (setq factor-list
			    (append factor-list
				    (compute-factors cls-id (second pred-sublist))
				    (compute-factors cls-id (third pred-sublist)))))))))
      (*print-inference-errors* (determine-error cls-id))
      )))



(defun compute-factors (cls-id lit-indx-lst)
  (cond
    ((> (length lit-indx-lst) 1)
     (let* ((cls (get-cls-info cls-id))
	    (cls-lit-list (clause-lit-list cls))
	    ;; (cls-depth (clause-depth cls))
	    (unify-rslts nil))
       (cond
	 ((= (length lit-indx-lst) 2)
	  (setq unify-rslts
		(unify (nth (1- (first lit-indx-lst)) cls-lit-list)
		       (nth (1- (second lit-indx-lst)) cls-lit-list)))
	  (if (not (eq unify-rslts 'FAIL))
	      (list (bld-factor 
		      cls-id
		      (subst-s-exp cls-lit-list unify-rslts)
		      ;; cls-depth
		      ))
	    nil))
	 (T (do ((n (length lit-indx-lst))
		 (k 2 (1+ k))
		 (lit1-indx nil) (lit1 nil)
		 (lit2-indx nil) (lit2 nil)
		 (factor-list nil))
		
		((> k n) factor-list) 
		
		(do ((k-combn-tabl (gen-k-combs n k))
		     (i 0 (1+ i)) 
		     (factor-lit-list cls-lit-list cls-lit-list)
		     (num-tabl-rows (choose n k)))

		    ((= i num-tabl-rows))
		    
		    (setq unify-rslts nil)
		    (do ((j 0 (1+ j)))

			((or (= j (1- k)) (eq unify-rslts 'FAIL))) 
				
			(setq lit1-indx (nth (1- (aref k-combn-tabl i j)) lit-indx-lst))
			(setq lit2-indx (nth (1- (aref k-combn-tabl i (1+ j))) lit-indx-lst))
			(setq lit1 (nth (1- lit1-indx) factor-lit-list))
			(setq lit2 (nth (1- lit2-indx) factor-lit-list))
			(setq unify-rslts (unify lit1 lit2))
			(if (and (not (eq unify-rslts 'FAIL)) 
				 (not (eq unify-rslts '() )))
			    (setq factor-lit-list 
				  (subst-s-exp factor-lit-list unify-rslts))))
		    (cond 
		      ((not (eq unify-rslts 'FAIL))
		       (setq factor-list 
			     (append factor-list
				     (list 
				       (bld-factor
					 cls-id
					 factor-lit-list
					 ;; cls-depth
					 ))))))))))))))


(defun bld-factor (cls-id factor-lit-list) ;; cls-depth
  (construct-resolvent-info
    (merge-left factor-lit-list)
    (list cls-id)
    'f))

;; the following functions are slightly different in hooked-on-frapps:

;;  standardizes the variables in the two given clause STRUCTURES apart,
;;  returns the modified literal list of the structure cl2

(defun struct-stndze-vars-apart (cl1 cl2)
  (let ((cl1-max-sub (clause-max-subscr cl1))
	(cl2-max-sub (clause-max-subscr cl2)))
       (inc-var-subscrs (clause-lit-list cl2)
			(max cl1-max-sub cl2-max-sub))))

;;  uses information provided by "neg-lit-flag" (i.e. - which is the
;;  negative literal, and which is the positive literal of the given pair
;;  of clash literals) to apply the unification algorithm to the
;;  appropriate structures. Everything should have been renamed as
;;  necessary.

;; Should always return 'FAIL if the two literals are not unifiable,
;; including when the two are negative or positive.

(defun unify-p (c1-clash-lit c2-clash-lit)
  (let ((neg-lit-flag (opp-sign-p c1-clash-lit c2-clash-lit)))
       (if neg-lit-flag
	   (progn
	    ;; It's important that this be here!!!
	    (if (eq neg-lit-flag 0)
		(unify (cdr c1-clash-lit) c2-clash-lit)
		(unify c1-clash-lit (cdr c2-clash-lit))))
	   'FAIL)))

