;; hooked-on-FRAPPS - sorts.lsp

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Maintenance of a simple model for a Monomorphic tree for sorts.
;; Modeled after Charlene Bloch's sorted unification algorithm.

;; NOTE: Not the particularly most efficient way to do this.
;; A lot can be improved.

;;; In a clause, multiple occurrences of variables must appear identically,

;;; -- To specify one sort being a subsort of another, type
;;;         (def-subsort <sort1> <sort2>)
;;;         e.g. (def-subsort 'dogs 'animals)
;;;	<sort1> and <sort2> may or may not be already defined, as long as the
;;;	tree restriction is maintained.

;;; -- To specify the sort of a constant or function symbol, type
;;;         (def-sortobject <object> <sort>)
;;;         e.g. (def-sortobject 'daniel 'man)

;;; -- The sort tree is represented as a nested list with sorts at the same
;;;    level of the tree being at the same nesting level and their offspring
;;;    at a more nested level
;;;       e.g.
;;;                     X
;;;                 /   |   \
;;;               Y     Z     W
;;;              / \    |    / \
;;;             A   B   C   D   E
;;;                             |
;;;                             F

;;;    is represented as the list (X (Y (A B) Z (C) W (D E (F))))

;;; variables are assumed to have the format
;;; (*var* <name> subscript [sort])
;;; if [sort] is not included then the variable is assumed to be of
;;; the universal type.

;;; Global variables:

(defvar *sorts* nil)  ;; list of sorts defined.
(defvar *objects* nil) ;; list of objects defined.
(defvar *tree* nil)   ;; Monomorphic restriction for sorts.
(defvar *univ* 'univ) ;; Universal sort type.

;; hash table for getting subsorts of a sort:

(defvar *subsort-db* (make-hash-table :test #'equal))

;;; *tree* stores the tree of sort information as a nested list with 
;;; subsorts as sublists, sibling sorts being at the same level. 
;;; *sorts* is the list of known sorts.

(defun reset-sorts ()
  (setq *sorts* nil)
  (setq *objects* nil)
  (clrhash *subsort-db*)
  (setq *tree* nil)
  )

(setq *unifier* 'sorted-unify)

(defun update-sorts-clause (cls const)
  ;; adds sort info in "const" to the variables in the actual clause "cls"
  (cond
   ((atom cls) cls)
   ((var-p cls) (append cls
			(cdr (assoc cls const :test #'equal))))
   (T (cons (update-sorts-clause (car cls) const)
	     (update-sorts-clause (cdr cls) const)))
   ))

;; No constraints now, so do not have to rename them:

(defun increment-const-subscripts (const increment)
  (declare (ignore const) (ignore increment))
  nil)

;; The following funcion integrates the constraint information
;; into the clause, then calls H-FRAPPS's "def-clause" to actually
;; define the new clause.

;; therefore, when defining clauses, the constraints on variables only
;; have to be specified once, as the "const" field.
;; Note that the actual "constraints" field in H-FRAPPS will be nil
;; (not used).

(defun s-def-clause (cls &key sorts user-field support-set)
  (if sorts (setf cls (update-sorts-clause cls sorts)))
  (def-clause cls :user-field user-field :support-set support-set)
  )

(defun print-var (var)
  (format t "?~d~d" (second var) (third var))
  (if (fourth var)
      (format t ":~d" (fourth var)))
  )

(defun user-print (id)
  (print-constraints (get-node-const id))
  )

;; (defun print-constraints (x)
  ;; (print-sorts x))

(defun print-constraints (x)
  (declare (ignore x))
  nil)

(defun print-sorts (const)
  (dolist (x const)
	  (if (listp (first x))
	      (format t " ~d~d:~d"
		      (second (first x)) (third (first x)) (second x))
	      (format t " ~d:~d" (first x) (second x))
	      )))
  

;;; g-sort finds the sort of a constant, a variable or a term:

(defun get-sort (pat)
  (cond
   ((var-p pat) (get-var-sort pat))
   ((listp pat) (get-sort (car pat)))
   (T (let ((srt (get pat 'sort)))
	    (if srt srt *univ*)))
   ))

(defun def-sortobject (obj srt)
  (cond
   ((sort-p srt)
    (cond ((object-p obj)
	   (format t "~%WARNING: redefining sortobject ~d." obj))
	  (T (setq *objects* (cons obj *objects*))))
    (setf (get obj 'sort) srt))
   (T (format t "~%ERROR: The sort ~d has not been defined." srt))
   ))

(defun def-sortobject-list (l)
  (dolist (x l)
	  (if (listp (first x))
	      (dolist (y (first x))
		      (def-sortobject y (second x)))
	      (def-sortobject (first x) (second x)))))

(defun def-subsort-list (l)
  (dolist (x l)
	  (if (listp (first x))
	      (dolist (y (first x))
		      (def-subsort y (second x)))
	      (def-subsort (first x) (second x)))))

(defun object-p (o)
  (if (member o *objects*) T nil))

;; find sort of variable given a sort-list
;; if sort is NOT found, returns the universal sort

;; (defun get-var-sort (var sort-list)
  ;; (if (setq srt (cadr (assoc var sort-list :test #'equal)))
      ;; srt
      ;; *univ*))

(defmacro get-var-sort (var)
  `(if (fourth ,var) (fourth ,var)
      *univ*))

(defun sort-subset (s1 s2)
  (or (equal s1 s2)
      (equal s2 *univ*) 
      (member s1 (get-subsorts s2))))

(defmacro s-entails (p1 p2)
  `(sort-subset (get-sort ,p2) (get-sort ,p1)))

(defun get-subsorts (srt) ;; Returns a LIST with the subsorts of srt.
  (if (sort-p srt)
      (let ((sub (flatten (get-subtree srt))))
	   (if (atom sub) (cons sub nil)
	       sub))))

(defun compile-sorts ()
  ;; makes sorts a bit more efficient, using a hash-table for subsorts.
  (dolist (srt (cons *univ* *sorts*))
	  (setf (gethash srt *subsort-db*) (get-subsorts srt))
	  )
  ;; Redefine subsort and sort-p function:
  (defun get-subsorts (srt) (gethash srt *subsort-db*))
  (defun sort-p (srt) (gethash srt *subsort-db*))
  )

(defun get-subtree (srt) ;; Returns the LIST representing the subtree
 			;;  starting at srt.
  (if (equal *univ* srt) *tree*
      (let ((subtree (get-subtree1 srt *tree*)))
	   (if (null subtree)
	       (list srt)
	       (list srt subtree))
	   )))

(defun get-subtree1 (srt sort-tree)
  (if (null sort-tree)
      nil
      (cond ((listp (car sort-tree)) 
	     (or (get-subtree1 srt (car sort-tree))
		 (get-subtree1 srt (cdr sort-tree))))
	    ((equal srt (car sort-tree))
	     (if (listp (cadr sort-tree))
		 (cadr sort-tree)
		 nil))
	    (t (get-subtree1 srt (cdr sort-tree))))))


(defun move-sort1 (s1 s2 sortlist)
  (cond ((null sortlist) nil)
	((equal (car sortlist) s2) ; now add the subsort
	 (cons s2
	       (if (atom (cadr sortlist)) ;; s1 is the only subsort
		   (cons s1 (cdr sortlist)) 
		   (append (list (append s1 (cadr sortlist)))
			   (cddr sortlist))))) ;; BUG?
	((listp (car sortlist)) ;; still have branches to explore
	 (let ((newcar (move-sort1 s1 s2 (car sortlist))))
	      (if (equal newcar (car sortlist))
		  (cons (car sortlist) (move-sort1 s1 s2 (cdr sortlist)))
		  (cons newcar (cdr sortlist)))))
	(t (cons (car sortlist) (move-sort1 s1 s2 (cdr sortlist))))
	))

(defun del-sort (srt lst) ;; deletes sort srt from LIST lst.
  (cond
   ((null lst) nil)
   ((atom lst) lst)
   ((equal srt (car lst))
    (if (listp (cadr lst)) (cddr lst)
	(cdr lst)))
   (T
    (let ((newcar (del-sort srt (car lst))))
	 (if (equal newcar (car lst))
	     (cons (car lst) (del-sort srt (cdr lst)))
	     (if newcar (cons newcar (cdr lst)) (cdr lst))
	     ))
    )))

;; makes sort s1 a subsort of s2 when they're both already in the tree:

(defun move-sort (s1 s2)
  (cond 
   ((or (not (sort-p s1)) (not (sort-p s2)))
    (if (not (sort-p s1))
	(format t "~%ERROR:The sort type ~d is not defined." s1))
    (if (not (sort-p s2))
	(format t "~%ERROR:The sort type ~d is not defined." s2)))
   ((subsort-p s2 s1)
    (format t "~%ERROR:The sort ~d is a subsort of ~d." s2 s1))
   (T ;; can add s1 as a subsort of s2.
      (setq *tree* (move-sort1 (get-subtree s1) s2
			       (del-sort s1 *tree*)))
      )
   ))

(defun sort-p (srt)
  (cond
   ((equal srt *univ*) T)
   ((member srt *sorts*) T)
   (T nil)))

(defun def-subsort (s1 s2)		;; s1 is a subsort of s2
  (cond
   ((sort-p s2)
    (if (sort-p s1)
	(move-sort s1 s2)
	(progn
	 (setq *sorts* (cons s1 *sorts*))
	 (setq *tree* (addsub s1 s2 *tree*))
	 )))
   ((sort-p s1)
    (if (topsort-p s1) 
	(progn
	 (setq *sorts* (cons s2 *sorts*))
	 (setq *tree* (add-topsort s2 s1 *tree*))
	 )
	(format t "~%ERROR:The sort type ~d has already been defined as a subsort; use MOVE-SORT instead.~%" s1)))
   ((equal s2 *univ*)
    (setq *sorts* (cons s1 *sorts*))
    (setq *tree* (cons s1 *tree*))
    )
   (T (setq *sorts* (cons s1 (cons s2 *sorts*))) ;; Both s1 and s2 are new.
      (setq *tree* (cons (list s2 (list s1)) *tree*))
      )
   ))


(defmacro subsort-p (s1 s2) ;; returns T is s1 is a subsort of s2.
  `(if (member ,s1 (get-subsorts ,s2)) t nil))

(defun topsort-p (srt) ;; returns T if srt has no supersort in *tree*
  (if (assoc srt *tree*)
      t nil))

(defun add-topsort (topsort old-top tree)
  (if (null tree) nil
      (if (equal old-top (caar tree))
	  (cons (cons topsort (list (car tree))) (cdr tree))
	  (cons (car tree) (add-topsort topsort old-top (cdr tree)))
	  )))

(defun addsub (s1 s2 sortlist)
  (cond
   ((null sortlist) nil) ;; we've come down a wrong branch
   ((equal (car sortlist) s2) ; now add the subsort
    (cons s2
	  (if (atom (cadr sortlist)) ;; s1 is the only subsort BUG?
	      (cons (list s1) (cdr sortlist)) ;; add s1 to sibling subsorts
	      (append (list (cons s1 (cadr sortlist)))
		      (cddr sortlist)))))
   ((listp (car sortlist)) ; we have some branches to explore
    (let ((newcar (addsub s1 s2 (car sortlist)))) 
	 (if (equal newcar (car sortlist)) ; we've added nothing
	     (cons (car sortlist) (addsub s1 s2 (cdr sortlist)))
	     (cons newcar (cdr sortlist)))))
   (t (cons (car sortlist) (addsub s1 s2 (cdr sortlist))))))

