;; hooked-on-FRAPPS - hdbss.lsp

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;      ====> Database Support System functions <====
;;;;
;;;;  This file contains those functions which comprise the database support
;;;;  system.  Support for the integration into, and manipulation within, the
;;;;  clause, literal, and term indexing data structures are provided by the
;;;;  functions at this layer

;;  define a base-set clause
;;
;;  NOTE: "defined" clauses are automatically integrated into 
;;        the derivation graph as nodes

;;  ==> MUST ALSO **INSERT** NEWLY DEFINED CLAUSES INTO THE *PRIORITY-QUEUE*

(defun def-clause (cls &key const user-field support-set)
  (let ((lvl-lst (gethash 0 *level-db*))
	(cls-struct nil)
	(id (get-new-id)))

       (setf (gethash id *node-db*)
	     (setq cls-struct
		   (construct-clause id cls const nil nil 0 'ud user-field)))
       (add-cls-lits-to-lit-db id)
       (update-cls-length-db id)
       (if support-set 
	   (setq *support-set* (append *support-set* (list id))))
       (setq *base-set* (setf (gethash 0 *level-db*)
			      (append lvl-lst (list id))))
       (add-new-clauses-to-priority-queue (list id))
       ;; User defined clauses should usually not be empty or answer clauses,
       ;; but:
       (cond
	((null cls)
	 (setq *empty-cls-ids* (adjoin id *empty-cls-ids*)))
	((answer-clause-p cls)
	 (setq *answer-cls-ids* (adjoin id *answer-cls-ids*))))
       *base-set*))

;; "Tricky" function to reset everything except base-set:

(defun partial-reset ()
  (let ((base-set (gial 0))
	(base-clause-info nil))
       (dolist (id base-set)
	       (setq base-clause-info
		     (append base-clause-info
			     (list (list (get-node-clause id)
					 (get-node-const id) ;; New
					 (get-node-user-field id)
					 (if (member id *support-set*) T nil)
					 ))
			     )))
       (dbss-reset)
       (dolist (info base-clause-info)
	       (def-clause (first info)
			   :const (second info)
			   :user-field (third info)
			   :support-set (fourth info)))
       )
  *base-set*)


;;  integrates a resolvent-info STRUCTURE into the derivation graph, and the 
;;  node ID into the level database; also updates the global max depth
;;  counter if a node is being integrated at a level which before now
;;  had no nodes;
;;  returns the node-id assigned to the resolvent structure integrated

(defun integrate-clause (res-info)
  (let* ((node-id (get-new-id))
	 (parents (resolvent-info-parents res-info))
	 (level (if (eq (resolvent-info-deriv-mthd res-info) 'f)
		    ;; note that if clause has been obtained by factoring,
		    ;; then only level of first & only parent is examined:
		    (1+ (get-node-level (first parents)))
		    (1+ (max (get-node-level (first parents))
			     (get-node-level (second parents))))
		    ))
	 (lvl-lst (gethash level *level-db*)))
	(if (> level *depth*)
	    (setq *depth* level))
	(setf (gethash node-id *node-db*)
	      (construct-clause 
	       node-id
	       (resolvent-info-clause res-info)
	       (resolvent-info-constraints res-info) ;; Hooked
	       (first parents)
	       (second parents)
	       level
	       (resolvent-info-deriv-mthd res-info)
	       (resolvent-info-user-field res-info)))
	(setf (gethash level *level-db*)
	      (append lvl-lst (list node-id)))
	(add-cls-lits-to-lit-db node-id)
	(update-cls-length-db node-id)
	(add-new-clauses-to-priority-queue (list node-id))
	(update-children-info node-id (remove-duplicates parents))
	;; Check to see if this is the empty clause or an answer clause:
	(let ((new-clause (resolvent-info-clause res-info)))
	     (cond
	      ((null new-clause)
	       (setq *empty-cls-ids* (adjoin node-id *empty-cls-ids*)))
	      ((answer-clause-p new-clause)
	       (setq *answer-cls-ids* (adjoin node-id *answer-cls-ids*)))
	      ))
	node-id))


;; Function "bld-var-list" receives an arbitrary s-expression (typically
;; containing variables) and returns a list of the variables found in it.
;; If no variables are found in the expression, "NIL" is returned.

(defun old-bld-var-list (s-exp)
  (cond
    ((null s-exp) nil)
    ((var-p s-exp) (list s-exp))
    ((atom (car s-exp)) (bld-var-list (cdr s-exp)))
    ((var-p (car s-exp))
     (adjoin (car s-exp) (bld-var-list (cdr s-exp)) :test #'equal))
    (T (union (bld-var-list (car s-exp))
	      (bld-var-list (cdr s-exp)) :test #'equal))
   ))


;;;; ================== Structure Construction functions ======================


;;  constructs a clause structure with the specified attributes

(defun construct-clause (cls-id lit-lst const l-par-id r-par-id dpth driv-mthd &optional lmbda)
  (let ((proc-cls-rslts (pre-process-cls lit-lst)))
    (make-clause :id cls-id
		 :lit-list lit-lst
		 :constraints const ; hooked
		 :left-par l-par-id
		 :right-par r-par-id
		 :children nil
		 ;; NOTE: Clauses always have no children when constructed.
		 :max-subscr (second proc-cls-rslts)
		 :pred-list (first proc-cls-rslts)
		 :depth dpth
		 :user-field lmbda
		 :deriv-mthd driv-mthd
		 :active T
		 ;; NOTE: Clauses are always active when constructed.
		 )))



;;  constructs a node-info structure with the specified attributes

(defun construct-node-info (id cls const parnts childrn lvl driv-mthd usr-fld &optional (act T))
  (make-node-info :id id
		  :clause cls
		  :constraints const
		  :parents parnts
		  :children childrn
		  :level lvl
		  :deriv-mthd driv-mthd
		  :user-field usr-fld
		  :active act
		  ))

;; The following two global constants were used in the priority queue:
;; before the single-node cost function was added:

;; (defun set-pseudo-cls-globals ()
  ;; (setq *pseudo-cls-info*
	   ;; (construct-clause 'pseudo-cls nil nil ;; constraints
			     ;; nil nil 0 'UD nil))
  ;; (setq *pseudo-cls* (construct-node-info 'pseudo-cls nil nil ;; constraints
					     ;; nil nil ;; children
					     ;; 0 'sys-defd nil)))


;;  constructs a resolvent-info structure with the specified attributes:

(defun construct-resolvent-info (cls const parnts driv-mthd &optional usr-fld)
  (make-resolvent-info :clause cls
		       :constraints const
		       :parents parnts
		       :deriv-mthd driv-mthd
		       :user-field usr-fld))



;;;; ================== Structure Conversion functions ========================


(defun conv-res-info-factor-to-clause (resolvent-info)
  (let* ((cls-id (first (resolvent-info-parents resolvent-info)))
	 (cls-info (get-real-cls-info cls-id)))
	;; No copying; only used for each parent and depth.
    (construct-clause
      cls-id
      (resolvent-info-clause resolvent-info)
      (resolvent-info-constraints resolvent-info) ;; Hooked
      (clause-left-par cls-info)
      (clause-right-par cls-info)
      ;; note that resolvent-info has no children and that construct-clause
      ;; assumes this.
      (clause-depth cls-info)
      'f
      (resolvent-info-user-field resolvent-info)
      )))


;;;; ================================================================

;; All get-* access functions are in "dbshared.lsp" except for the ones
;; that access the constraint field:

(defun get-constraints (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info (clause-constraints cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-node-constraints (node-id)
  (clause-constraints (get-real-cls-info node-id)))

;; abbreviation macro for above:

(defmacro get-node-const (node-id)
  `(clause-constraints (get-real-cls-info ,node-id)))


(defun get-node-node-info (node-id)
  (let ((cls-info (get-cls-info node-id))) ;; note that this is a COPY
       (construct-node-info
	node-id
	(clause-lit-list cls-info)
	(clause-constraints cls-info)	;; hooked
	(cond
	 ((or (eq (clause-deriv-mthd cls-info) 'br)
	      (eq (clause-deriv-mthd cls-info) 'gr))
	  (list (clause-left-par cls-info)
		(clause-right-par cls-info)))
	 ((eq (clause-deriv-mthd cls-info) 'f)
	  (list (clause-left-par cls-info)))
	 ((eq (clause-deriv-mthd cls-info) 'ud)
	  nil))
	(clause-children cls-info)
	(clause-depth cls-info)
	(clause-deriv-mthd cls-info)
	(clause-user-field cls-info)
	(clause-active cls-info)
	)))


;;  ===========================================================================
;;  =                                                                         =
;;  =                     CLAUSE RETRIEVAL FUNCTIONS                          =
;;  =                                                                         =
;;  ===========================================================================


;;  returns the IDS of *ALL* clauses in the clause database which have literals
;;  that are *EQUAL* to the given literal

(defun ret-clauses-cont-equal-lits (lit)
  (let ((lit-db-key (bld-lit-db-key lit))
	(hash-val-fnd nil)
	(lit-args-ht nil)
	(lit-occurs-in-list nil))
    (multiple-value-setq (lit-args-ht hash-val-fnd)
			 (gethash lit-db-key *lit-db*))
    (cond
      (hash-val-fnd
	(multiple-value-setq (lit-occurs-in-list hash-val-fnd)
			     (gethash (bld-lit-args-ht-key lit) lit-args-ht))
	(cond
	  (hash-val-fnd lit-occurs-in-list)
	  (t nil)))
      (t nil))))



;;  returns the IDS of *ALL* clauses in the clause database which have literals
;;  that *UNIFY* with the given literal

;;  NOTE: NORMAL UNIFICATION is used in the following three functions!!!

(defun ret-clauses-cont-unifiable-lits (lit)
  (let ((ret-rslts (ret-clauses-matching-spec-lit-cond
		    #'(lambda (lit-args target-lit-args)
			      (if (not (eq (unify (stndze-vars-apart
						   lit-args
						   target-lit-args)
						  lit-args)
					   'FAIL))
				  t
				  nil))
		    lit)))
       (remove-duplicates ret-rslts :from-end t)))


;;  returns the IDS of *ALL* clauses in the clause database which have literals
;;  that are *INSTANCES* of the given literal

(defun ret-clauses-cont-instance-lits (lit)
  (let ((ret-rslts (ret-clauses-matching-spec-lit-cond
		     #'(lambda (lit-args target-args)
			 (let* ((stdzed-lit-args (stndze-vars-apart target-args
								    lit-args)))
			   (cond
			     ((not (eq (instance-p target-args stdzed-lit-args)
				       'FAIL))
			      t)
			     (t nil))))
		     lit)))
    (remove-duplicates ret-rslts :from-end t)))



;;  returns the IDS of *ALL* clauses in the clause database which have literals
;;  that are *MORE GENERAL THAN* the given literal

(defun ret-clauses-cont-more-general-lits (lit)
  (let ((ret-rslts (ret-clauses-matching-spec-lit-cond
		     #'(lambda (lit-args target-args)
			 (let* ((stdzed-lit-args (stndze-vars-apart target-args
								    lit-args)))
			   (cond
			     ((not (eq (instance-p stdzed-lit-args target-args)
				       'FAIL))
			      t)
			     (t nil))))
		     lit)))
    (remove-duplicates ret-rslts :from-end t)))



;;  return the IDS of *ALL* clauses in the clause database matching the
;;  specified literal condition "lit-cond"
;;
;;  NOTE: the three previous functions rely **heavily** on this function

(defun ret-clauses-matching-spec-lit-cond (lit-cond lit)
  (let ((lit-db-key (bld-lit-db-key lit))
	(hash-val-fnd nil)
	(lit-args-ht nil)
	(lit-args (get-lit-args lit))
	(cls-id-list nil))
       
       (multiple-value-setq (lit-args-ht hash-val-fnd)
			    (gethash lit-db-key *lit-db*))
       
       (cond
	(hash-val-fnd
	 (maphash #'(lambda (key val)
			    (if (apply lit-cond (list (bld-args key)
						      lit-args))
				(setq cls-id-list (append cls-id-list val))))
		  lit-args-ht)
	 cls-id-list)
	(t nil))))




;;  ====================== End Clause Retrieval Functions =====================


;;  increment the variable subscripts of the given clause by the increment
;;  specified

;;  Note: This generalizes the original version in FRAPPS, in that
;;  variables can have more than 3 elements. The subscript still has
;;  to be the third one, though.

(defun inc-var-subscrs (cls subscr-inc)
  (cond
   ((null cls) nil)
   ((var-p (car cls))
    ;; (cons (append (list (first (car cls)) (second (car cls)))
		  ;; (list (+ (third (car cls)) subscr-inc))
		  ;; (list (fourth (car cls))))
	  ;; (inc-var-subscrs (cdr cls) subscr-inc))
    (cons (cons (first (car cls))
		(cons (second (car cls))
		      (cons (+ (third (car cls)) subscr-inc)
			    (cdddr (car cls)))))
	  (inc-var-subscrs (cdr cls) subscr-inc))
    )
   ((atom (car cls)) (cons (car cls) (inc-var-subscrs (cdr cls) subscr-inc)))
   (T (cons (inc-var-subscrs (car cls) subscr-inc)
	    (inc-var-subscrs (cdr cls) subscr-inc)))))


;;  Setting the constraint field:

(defun set-constraints (id value)
  (let ((cls-info (get-real-cls-info id)))
       (cond
	(cls-info (setf (clause-constraints (gethash id *node-db*)) value))
	(*print-access-errors* (err-messg-nonexistent id))
	(T 'F-ERROR)
	)))

