;; FRAPPS - dbshared.lsp

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;      ====> Shared Database Support System functions <====
;;;;
;;;;  Database support functions shared between FRAPPS and H-FRAPPS

;;  reinitialize the entire system
;;  does NOT reinstate default values for general global variables.

(defun reset-frapps ()
  (dbss-reset)
  ;; (reset-globals)
  )

;;  Reset the database support system.
;;  Flags and parameters are NOT changed.

;; Note that *empty-cls-found* and *answer-cls-fnd* are subsumed by
;; *empty-cls-ids* and *answer-cls-ids*

(defun dbss-reset ()
  (clrhash *node-db*)
  (clrhash *level-db*)
  (clrhash *cls-length-db*)
  (clrhash *lit-db*)
  (clrhash *arg-retrv-db*)
  (clrhash *arg-dtct-db*)
  (setq *last-id* 0)		;; NOT 1
  (setq *depth* 0)
  (setq *max-db-length* 0)
  (setq *base-set* nil)
  (setq *support-set* nil)
  (setq *empty-cls-ids* nil)
  (setq *answer-cls-ids* nil)
  (setq *priority-queue* nil)
  )


;; WARNING: In general, copies of information in the different databases
;; are NOT made, so users should be careful
;; not to use destructive functions directly on the results
;; (or else copy them).

;; Note that the internal structure for clauses is called CLAUSE,
;; and is not visible to the user. The fields of NODE-INFO structures 
;; are only a subset of the CLAUSE structure fields.

;; ====================================================================

;; Obtain a previously unissued id # to uniquely identify a clause.
;;
;; NOTE: As a feature, node-id' are integers assigned in ascending order,
;; except for user-defined clauses, which can be given arbitrary ATOMS
;; as ID's (which should not be integers).
;; Users should NEVER change the value of *last-id*

;; No correctness check is done now since users are not allowed to
;; define integer id's themselves.
;; *user-id* is initialized at 0 when declared as a global variable.

(defmacro get-new-id ()
  `(setq *last-id* (1+ *last-id*))
  )

;;  Integrates a list of resolvent-info STRUCTURES into the derivation graph;
;;  returns a list of the corresponding node-id's:

(defun integrate-clause-list (resolvent-info-list)
  (let ((node-id-lst nil))
       (dolist (res-info resolvent-info-list node-id-lst)
	       (setq node-id-lst
		     (append node-id-lst (list (integrate-clause res-info)))
		     ))))


;; NOTE: Assumes that duplicates have already been deleted from
;; parent-id-list:

(defun update-children-info (id parent-id-list)
  (dolist (x parent-id-list)
	  (setf (clause-children (gethash x *node-db*))
		(append (clause-children (gethash x *node-db*)) (list id))
		)))


;;  given a particular clause id, this function retrieves a COPY of the 
;;  clause structure associated with the id from the global node database
;;
;;  NOTE1: if the given clause id DOES NOT refer to a clause in the
;;	   database, then NIL is returned.
;;
;;  NOTE2: Note that "copy-clause" does NOT copy the internal lists in
;;	   the structure.

(defmacro get-cls-info (cls-id)
  `(let ((cls-info (gethash ,cls-id *node-db*)))
       (if cls-info
	   (copy-clause cls-info)
	   )))


;; This function does NOT return a copy, but the real thing; have to be careful
;; with it. Not available to users.

(defmacro get-real-cls-info (cls-id)
  `(gethash ,cls-id *node-db*))


;;  builds a list containing all existing clause STRUCTURES (of the
;;  global database of clauses) down to the depth specified by the 
;;  input argument
;;  NOTE: This is never used...

;; (defun bld-cls-struct-lst (depth)
  ;; (conv-id-lst-to-struct-lst (bld-node-id-list depth)))


;;  builds a list containing the ID'S of all nodes existing in the
;;  derivation graph down to, and including, the depth specified by the 
;;  input argument 

(defun bld-node-id-list (depth)
  (let ((node-id-lst nil))
    (dotimes (level (1+ depth) node-id-lst)
	     (setq node-id-lst
		   (append node-id-lst (gethash level *level-db*))))))


;;  retrieves all nodes (i.e.- node-info structures) at the specified level of
;;  the derivation graph 

(defun get-nodes-at-level (level)
  (let ((node-id-list (gethash level *level-db*))
	(lvl-list nil)
	(fnd-node nil))
    (dolist (node-id node-id-list lvl-list)
	    (if (setq fnd-node (get-node-node-info node-id))
		(setq lvl-list (append lvl-list (list fnd-node)))))))


;;  Retrieves all node-info-structures down to the level specified
;;
;;  NOTE: if "level" isn't specified, then all nodes in the ENTIRE
;;        graph are returned  

(defun get-nodes-downto-level (&optional level)
  (let ((node-info-list nil))
       (if (null level) 
	   (setq level *depth*))
       (dotimes (depth (1+ level) node-info-list)
		(setq node-info-list
		      (append node-info-list (get-nodes-at-level depth)))
		)))


;;;; ==========================================================================


;;  convert an ID list to a CLAUSE STRUCTURE list.
;; NOTE: Never used...

;; (defun conv-id-lst-to-struct-lst (id-lst)
  ;; (let ((struct-lst nil))
    ;; (dolist (id id-lst struct-lst)
	    ;; (setq struct-lst 
		  ;; (append struct-lst
			  ;; (list (get-cls-info id)))))))



;; convert a CLAUSE STRUCTURE list to an ID list
;; NOTE: Never used...

;; (defun conv-struct-lst-to-id-lst (struct-lst)
  ;; (let ((id-lst nil))
    ;; (dolist (cls-struct struct-lst id-lst)
	    ;; (setq id-lst 
		  ;; (append id-lst
			  ;; (list (clause-id cls-struct)))))))



;;  convert an ID list to a NODE STRUCTURE list
;; NOTE: Never used...

;; (defun conv-id-lst-to-node-lst (id-lst)
  ;; (let ((node-lst nil))
    ;; (dolist (id id-lst node-lst)
	    ;; (setq node-lst 
		  ;; (append node-lst
			  ;; (list (get-node-node-info id)))))))


;; Decides if an id corresponds to a node in the derivation.
;; Not visible to user; not to be set to anything:

(defmacro is-a-node-p (id)
  `(gethash ,id *node-db*))



;;;; ===============> NODE Structure "access" functions <==================


;; The following functions are provided to the user and include error-checking.
;; The get-node-* functions do the same without the error checks.

;; This is what old versions looked like:

;; (defun get-clause (node-id)
  ;; (if (is-a-node-p node-id)
      ;; (copy-list (clause-lit-list (get-real-cls-info node-id)))
      ;; (err-messg-nonexistent node-id)))

;; New versions should be more efficient since they access the hash table
;; only once.
;; Also, now do NOT copy internal fields; users are responsible for copying
;; if necessary.

(defun get-clause (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info (clause-lit-list cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-parents (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info
	 ;; (list (clause-left-par cls-info) (clause-right-par cls-info))
	 ;; The following is needed to avoid NIL as element of returned list:
	 ;; Would have to be changed if more inference rules are added:
	 (cond
	  ((member (clause-deriv-mthd cls-info) '(br gr))
	   (list (clause-left-par cls-info)
		 (clause-right-par cls-info)))
	  ((eq (clause-deriv-mthd cls-info) 'f)
	   (list (clause-left-par cls-info)))
	  ((eq (clause-deriv-mthd cls-info) 'ud)
	   nil))
	 )
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-level (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info (clause-depth cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-deriv-mthd (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond 
	(cls-info (clause-deriv-mthd cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-user-field (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info (clause-user-field cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-children (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info (clause-children cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

(defun get-active (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       (cond
	(cls-info (clause-active cls-info))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	)))

;; This is the get-node-info version that should be visible to the user.
;; "get-node-node-info" is the internal one.

(defun get-node-info (node-id)
  (cond ((is-a-node-p node-id) (get-node-node-info node-id))
	(*print-access-errors* (err-messg-nonexistent node-id))
	(T 'F-ERROR)
	))


;;;; ==============> Re-named NODE structure "access" functions <==============
;;;;
;;;; These do not carry out any error checking.
;;;; Note that copies are NOT made.

;; DON'T use macros. These can have undesired side effects.

(defun get-node-clause (node-id)
  (clause-lit-list (get-real-cls-info node-id)))

(defun get-node-parents (node-id)
  (let ((cls-info (get-real-cls-info node-id)))
       ;; (list (clause-left-par cls-info) (clause-right-par cls-info))
       (cond
	((member (clause-deriv-mthd cls-info) '(br gr))
	 (list (clause-left-par cls-info)
	       (clause-right-par cls-info)))
	((eq (clause-deriv-mthd cls-info) 'f)
	 (list (clause-left-par cls-info)))
	((eq (clause-deriv-mthd cls-info) 'ud)
	 nil))
       ))

(defun get-node-level (node-id)
  (clause-depth (get-real-cls-info node-id)))

(defun get-node-deriv-mthd (node-id)
  (clause-deriv-mthd (get-real-cls-info node-id)))

(defun get-node-user-field (node-id)
  (clause-user-field (get-real-cls-info node-id)))

(defun get-node-active (node-id)
  (clause-active (get-real-cls-info node-id)))

(defun get-node-children (node-id)
  (clause-children (get-real-cls-info node-id)))

;;;; ==================== Node Modification functions =========================


;;  Allows the user to "modify" the user field of a 
;;  derivation graph node:

(defun set-user-field (id value)
  (let ((cls-info (get-real-cls-info id)))
       (cond
	(cls-info (setf (clause-user-field (gethash id *node-db*)) value))
	(*print-access-errors* (err-messg-nonexistent id))
	(T 'F-ERROR)
	)))

;; This function is no longer used; obsolete since the days of IDS.
;;
;; delete the specified node from the node derivation graph:
;;  NOTE:  MUST remove *all* traces of the node's information
;;         fields from the following databases maintained by FRAPPS:
;;
;;         ==> *node-db*
;;         ==> *level-db*
;;         ==> *cls-length-db* 
;;         ==> *lit-db* 

;;  and, ideally, the priority queue!

;; (defun delete-node-old (id)	
  ;; (cond
   ;; ((gethash id *node-db*)
    ;; (let* ((id-lvl (get-node-level id))
	   ;; (updated-lvl-lst (remove id (gethash id-lvl *level-db*)))
	   ;; (id-len (clause-length (get-node-clause id)))
	   ;; (updated-len-lst (remove id (gethash id-len *cls-length-db*))))
	  ;; (if (not (null updated-lvl-lst))
	      ;; (setf (gethash id-lvl *level-db*) updated-lvl-lst)
	      ;; (remhash id-lvl *level-db*))
	  ;; (if (not (null updated-len-lst))
	      ;; (setf (gethash id-len *cls-length-db*) updated-len-lst)
	      ;; (remhash id-len *cls-length-db*))
	  ;; (update-lit-db-del id)
	  ;; (remhash id *node-db*)
	  ;; (setq *support-set* (remove id *support-set*))
	  ;; (setq *base-set* (remove id *base-set*))))
   ;; (T T)))


;;  "de-activate" the specified derivation graph node.
;;
;;  NOTE: this prevents the node from being used to infer "new" information:

(defun deactivate-node (id)
  (let ((cls-info (get-real-cls-info id)))
       (cond
	(cls-info (setf (clause-active cls-info) nil))
	(*print-access-errors* (err-messg-nonexistent id))
	(T 'F-ERROR)
	)))


;; internal version, no check:

(defun deactivate-node-node (id)
  (setf (clause-active (get-real-cls-info id)) nil))


;;  "re-activate" the specified derivation graph node after it is has ALREADY
;;  been "DE-ACTIVATED"
;;
;;  NOTE: this allows the node to be used once again to infer "new" information

(defun reactivate-node (id)
  (let ((cls-info (get-real-cls-info id)))
       (cond
	(cls-info (setf (clause-active cls-info) T))
	(*print-access-errors* (err-messg-nonexistent id))
	(T 'F-ERROR)
	)))


;;;;  =========================================================================
;;;;  =                                                                       =
;;;;  =                                                                       =
;;;;  =                    LITERAL DATABASE FUNCTIONS                         =
;;;;  =                                                                       =
;;;;  =                                                                       =
;;;;  =========================================================================



;;  add the literals of the clause corresponding to the given clause ID to the 
;;  literal database

(defun add-cls-lits-to-lit-db (cls-id)
  (let ((lit-list (get-node-clause cls-id)))
    (dolist (lit lit-list)
	    (update-lit-db-add lit cls-id))))


;;  "update" the literal database by *ADDING* the specified literal, *OR*
;;  adding a "reference" to the specified literal (*if* the literal is 
;;  *already* in the database); the specified clause ID indicates
;;  the ID of the clause in which the specified literal occurs in

(defun update-lit-db-add (lit cls-id)
  (let ((lit-db-key (bld-lit-db-key lit))
	(lit-args-ht-key (bld-lit-args-ht-key lit))
	(get-rslts (get-clauses-cont-lit lit))
	(lit-args-ht nil))
    (cond
      ((null get-rslts)
       (setq lit-args-ht (make-hash-table :test #'equal))
       (setf (gethash lit-db-key *lit-db*) lit-args-ht)
       (setf (gethash lit-args-ht-key lit-args-ht) (list cls-id)))
      ((hash-table-p get-rslts)
       (setf (gethash lit-args-ht-key get-rslts) (list cls-id)))
      (t (setf (gethash lit-args-ht-key (gethash lit-db-key *lit-db*))
	       (cons cls-id get-rslts))))))




;;  "update" the literal database by *DELETING* all references to
;;  the given clause id that are associated with literals in the
;;  clause database.
;;
;;  NOTES: The list of literals associated with (i.e. - clause denoted by)
;;         cls-id are used to query the literal database for occurrences
;;         of the id "cls-id".  These occurences are deleted.

(defun update-lit-db-del (cls-id)
  (let ((lit-list (get-node-clause cls-id))
	(id-list nil)
	(lit-args-ht-key nil)
	(lit-args-ht nil)
	(lit-db-key nil))
    (dolist (lit lit-list)
	    (multiple-value-setq (id-list lit-args-ht-key lit-args-ht lit-db-key)
				 (get-clauses-cont-lit lit))
	    (setq id-list (remove cls-id id-list))
	    (cond
	      ((null id-list)
	       (remhash lit-args-ht-key lit-args-ht)
	       (if (eq (hash-table-count lit-args-ht) 0)
		   (remhash lit-db-key *lit-db*)))
	      (t (setf (gethash lit-args-ht-key lit-args-ht) id-list))))))




;;  attempts to return *ALL* clause IDS of those clauses in the clause
;;  database which contain the specified literal "lit", depending on the 
;;  outcome of the search, the following results are returned:
;;
;;      ==> "nil"  indicates that *NO* "literal arguments" hash-table exists
;;                 for the given literal
;;
;;      ==> <lit-args hash-table>  indicates that the "literal arguments"
;;                                 hash-table returned has *NO* value
;;                                 corresponding to the "key" constructed
;;                                 from the given literal
;;
;;      ==> <id-list>  indicates that a list of clauses were found which
;;                     contain the given literal
;;
;;  NOTES:  In the event that <id-list> is returned, this function also
;;          returns <lit-args hash-table-key>, <lit-args hash-table>, and
;;          <literal db-key> (in that order).

(defun get-clauses-cont-lit (lit)
  (let ((lit-db-key (bld-lit-db-key lit))
	(hash-val-fnd nil)
	(lit-args-ht nil)
	(lit-args-ht-key nil)
	(lit-occurs-in-list nil))
    (multiple-value-setq (lit-args-ht hash-val-fnd)
			 (gethash lit-db-key *lit-db*))
    (cond
      (hash-val-fnd
	(setq lit-args-ht-key (bld-lit-args-ht-key lit))
	(multiple-value-setq (lit-occurs-in-list hash-val-fnd)
			     (gethash lit-args-ht-key lit-args-ht))
	(cond
	  (hash-val-fnd (values lit-occurs-in-list lit-args-ht-key 
				lit-args-ht lit-db-key))
	  (t lit-args-ht)))
      (t nil))))


;;  builds a literal from the given literal database key, and arg. id list
;;  using the elements of the arg. id list to "index" into *arg-retrv-db* 
;;
;;  NOTE: this function ASSUMES that each element of arg-id-list appears
;;        as a KEY in *arg-retrv-db*

(defun bld-lit (lit-db-key arg-id-list)
  (append (butlast lit-db-key) (bld-args arg-id-list)))



;;  builds up a literal ARGUMENT LIST given a literal ARGUMENT ID LIST
;;
;;  NOTE: this function ASSUMES that each element of arg-id-list appears
;;        as a KEY in *arg-retrv-db*

(defun bld-args (arg-id-list)
  (let ((arg-list nil))
    (dolist (arg-id arg-id-list arg-list)
	    (setq arg-list
		  (append arg-list (list (gethash arg-id *arg-retrv-db*)))))))




;;  build a literal-database "key" given a literal

(defun bld-lit-db-key (lit)
  (cond
    ((neg-lit-p lit)
     (list (first lit) (second lit) (1- (length (rest lit)))))
    (t (list (first lit) (length (rest lit))))))




;;  build a literal arguments hash-table "key" given a literal

(defun bld-lit-args-ht-key (lit)
  (let ((hash-val-fnd nil)
	(arg-id nil)
	(arg-id-list nil))
    (dolist (arg (get-lit-args lit) arg-id-list)
	    (multiple-value-setq (arg-id hash-val-fnd)
				 (gethash arg *arg-dtct-db*))
	    (cond
	      (hash-val-fnd
		(setq arg-id-list (append arg-id-list (list arg-id))))
	      (t (setf (gethash arg *arg-dtct-db*)
		       (setq arg-id (gentemp "arg")))
		 (setf (gethash arg-id *arg-retrv-db*) arg)
		 (setq arg-id-list (append arg-id-list (list arg-id))))))))

;; New: Returns the list of predicates associated with the given node-id,
;; using the "pred-list" field of the clause structure.

(defun get-cls-preds (node-id)
  (mapcar 'car (clause-pred-list
		(get-real-cls-info node-id))))

;;;;  =========================================================================
;;;;  =                                                                       =
;;;;  =                                                                       =
;;;;  =                    CLAUSE LENGTH DATABASE FUNCTIONS                   =
;;;;  =                                                                       =
;;;;  =                                                                       =
;;;;  =========================================================================


;;  updates the *cls-length-db* hash table by adding
;;  the given node-id to the specified clause length list
;;  
;;  NOTE: ANSWER literals are IGNORED when determining the length of clauses:

(defun update-cls-length-db (node-id)
  (let* ((cls-length (clause-length (get-node-clause node-id)))
	 (cls-list (gethash cls-length *cls-length-db*)))
	(setf (gethash cls-length *cls-length-db*) 
	      (append cls-list (list node-id)))

	(if (> cls-length *max-db-length*)
	    (setq *max-db-length* cls-length))
	))


(defun get-ids-of-length (n)
  (gethash n *cls-length-db*))

;;;;  ================ end clause length database functions ===================

;;;; level access functions:

;;;; NOTE: Users should not use any destructive functions on these,
;;;; or else the database can get messed up.

(defun get-ids-at-level (level)
  (gethash level *level-db*))

(defun get-ids-downto-level (&optional level)
  (let ((node-info-list nil))
       (if (null level)
	   (setq level *depth*))
       (dotimes (current-depth (1+ level) node-info-list)
		(setq node-info-list (append node-info-list
					     (gethash current-depth *level-db*)
					     ))
		)))
