;; FRAPPS - costfns.lsp

 ;;
 ;; The Framework for Resolution-Based Automated Proof Procedure Systems
 ;;                         FRAPPS Version 2.0
 ;;    Authors: Alan M. Frisch, Michael K. Mitchell and Tomas E. Uribe
 ;;               (C) 1992 The Board of Trustees of the
 ;;                       University of Illinois
 ;;                        All Rights Reserved
 ;;
 ;;                              NOTICE
 ;;
 ;;   Permission to   use,  copy,  modify,  and   distribute  this
 ;;   software  and  its  documentation for educational, research,
 ;;   and non-profit purposes  is  hereby  granted  provided  that
 ;;   the   above  copyright  notice, the original authors  names,
 ;;   and this permission notice appear in all  such  copies   and
 ;;   supporting   documentation; that no charge be  made for such
 ;;   copies; and that  the name of  the University of Illinois or
 ;;   that  of  any  of the Authors not be used for advertising or
 ;;   publicity  pertaining  to   distribution   of  the  software
 ;;   without   specific  prior  written   permission. Any  entity 
 ;;   desiring  permission to incorporate   this   software   into
 ;;   commercial  products  should  contact   Prof.  A. M. Frisch,
 ;;   Department  of Computer  Science,  University  of  Illinois,
 ;;   1304  W.  Springfield Avenue, Urbana, IL 61801. The  Univer-
 ;;   sity of  Illinois and the Authors  make  no  representations
 ;;   about   the suitability  of this  software  for any purpose.
 ;;   It is provided "as is" without  express or implied warranty.
 ;;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Cost functions that implement various search strategies.
;; Shared by H-Frapps.

;; NOTE: Default single-node estimate now minus infinity, in case that
;; negative costs are being used.

;; NOTE: The arguments, cls1 and cls2, should be NODE-INFO structures.

;; 	 These functions can NOT be macros.

;; (setq *max-cost* (- most-positive-fixnum 1))
;; (setq *max-single-cost* (- most-positive-fixnum 1))

;; constants will be large enough for any reasonable search.

(defun zero-costfn (&rest l)
  (declare (ignore l))	;; (For the compiler)
  0)

(defun most-negative-costfn (&rest l)
  (declare (ignore l))
  most-negative-fixnum)

;; ====================================================================

;; clause length:
;;  determine the "heuristic" length of any potential resolvents formed
;;  from cls1 and cls2

(defun cls-length-costfn (cls1-info cls2-info)
  (+ (clause-length (node-info-clause cls1-info))
     (clause-length (node-info-clause cls2-info)))
  )

(defun min-length-costfn (cls1-info cls2-info)
  (min (clause-length (node-info-clause cls1-info))
       (clause-length (node-info-clause cls2-info)))
  )

(defun cls-length-single-costfn (cls1-info)
  (clause-length (node-info-clause cls1-info)))

;; ====================================================================

;; clause depth:

(defun cls-depth-costfn (cls1-info cls2-info)
  (max (node-info-level cls1-info) (node-info-level cls2-info)))

(defun cls-depth-single-costfn (cls1-info)
  (node-info-level cls1-info))

;; In all the following, *cost-function-single-components* is set
;; to be the clause depth, so as to get a general breadth-first search.

;; All these can, of course, be changed and/or combined. The possibilities
;; are endless...

;; ====================================================================

;; breadth-first input resolution:

(defun input-costfn (cls1-info cls2-info)
  (if (or (member (node-info-id cls1-info) *base-set*)
	  (member (node-info-id cls2-info) *base-set*)
	  )
      (max (node-info-level cls1-info) (node-info-level cls2-info))
      most-positive-fixnum)
  )


;; ====================================================================

;; Set of support: If we are going to implement it this way,
;; then can assume everything except *base-set* - *support-set*
;; is in the set of support:

(defun sos-costfn (cls1-info cls2-info)
  (if (and
       (member (node-info-id cls1-info) *base-set*)
       (member (node-info-id cls2-info) *base-set*)
       (not (member (node-info-id cls1-info) *support-set*))
       (not (member (node-info-id cls2-info) *support-set*))
       )
      most-positive-fixnum
      (max (node-info-level cls1-info) (node-info-level cls2-info))
      ))

;; better still, it can be "hardwired" in using this component
;; when loading the base set of clauses:

(defun sos-single-costfn (cls1-info)
  (if (member (node-info-id cls1-info) *support-set*)
      0
      most-positive-fixnum)
  )


;; ====================================================================

;; Unit resolution, breadth first:

(defun unit-costfn (cls1-info cls2-info)
  (if (or (eq (clause-length (node-info-clause cls1-info)) 1)
	  (eq (clause-length (node-info-clause cls2-info)) 1))
      (max (node-info-level cls1-info) (node-info-level cls2-info))
      most-positive-fixnum))

;; ====================================================================

;; Unit resolution, more or less breadth first, but giving priority to
;; short resolvents:

(defun unit-short-costfn (cls1-info cls2-info)
  (let ((l1 (clause-length (node-info-clause cls1-info)))
	(l2 (clause-length (node-info-clause cls2-info)))
	)
       (if (or (eq l1 1) (eq l2 1))
	   (max (node-info-level cls1-info) (node-info-level cls2-info))
	   (+ l1 l2)
	   ;; most-positive-fixnum
	   )
       ))

;; =========================================================================

;; P1 resolution:

(defun p1-costfn (cls1-info cls2-info)
  (if (or (pos-cls-p (node-info-clause cls1-info))
	  (pos-cls-p (node-info-clause cls2-info))
	  )
      (max (node-info-level cls1-info) (node-info-level cls2-info))
      most-positive-fixnum
      ))


;; =========================================================================

;; to do simple breadth-first:
(defun set-breadth-costfn ()
  (setq *cost-function-components* '(zero-costfn))
  (setq *cost-function-weights* '(0))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(1))
  )

;; to do simple depth-first search:
(defun set-depth-costfn ()
  (setq *cost-function-components* '(zero-costfn))
  (setq *cost-function-weights* '(0))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(-1))
  )

;; to do breadth-first input:
(defun set-input-costfn ()
  (setq *cost-function-components* '(input-costfn))
  (setq *cost-function-weights* '(1))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(1))
  )

;; for breadth-first set-of-support:

(defun set-sos-costfn ()
  (setq *cost-function-components* '(sos-costfn))
  (setq *cost-function-weights* '(1))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(1))
  )

;; for breadth-first unit:
(defun set-unit-costfn ()
  (setq *cost-function-components* '(unit-costfn))
  (setq *cost-function-weights* '(1))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(1))
  )

(defun set-unit-short-costfn ()
  (setq *cost-function-components* '(unit-short-costfn))
  (setq *cost-function-weights* '(1))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(1))
  )

;; for p1 resolution:
(defun set-p1-costfn ()
  (setq *cost-function-components* '(p1-costfn))
  (setq *cost-function-weights* '(1))
  (setq *cost-function-single-components*
	'(cls-depth-single-costfn))
  (setq *cost-function-single-weights* '(1))
  )

;; The following is useful for internal debugging:

;; (defun manual ()
  ;; (let ((pair (pop-priority-queue)))
       ;; (integrate-clause-list (resolve (first pair) (second pair)))
       ;; ))

