#include "header.h"

/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */


#define SYM_TAB_SIZE 50

static struct sym_ent *Sym_tab[SYM_TAB_SIZE];

/*************
 *
 *    print_term(file_ptr, term)  --  Print a term to a file.
 *
 *        Variables 0-5 are printed as x,y,z,u,v,w, and equalities
 *    and negated equalities are printed in infix.
 *
 *************/

void print_term(fp,t)
FILE *fp;
struct term *t;
{
    struct rel *r;
    
    if (t == NULL)
	fprintf(fp, "(nil)");
    else if (t->type == NAME) {           /* name */
	/* if (t->sym_num == Nil_sym_num)
	    fprintf(fp, "[]");
	else */
	    fprintf(fp, "%s", sn_to_str(t->sym_num));
	}
    else if (t->type == VARIABLE)               /* variable */
	print_variable(fp, t);
    else {  /* complex */
	    fprintf(fp, "%s", sn_to_str(t->sym_num));
	    fprintf(fp, "(");
	    r = t->farg;
	    while(r != NULL) {
		print_term(fp, r->argval);
		r = r->narg;
		if(r != NULL)
		    fprintf(fp, ",");
		}
	    fprintf(fp, ")");
	}
}  /* print_term */

void print_term_lisp(fp,t)
FILE *fp;
struct term *t;
{
    struct rel *r;
    
    /* if (t->sym_num != Ignore_sym_num) */

    if (t == NULL)
	fprintf(fp, "(nil)");
    else if (t->type == NAME) {           /* name */
	    fprintf(fp, "%s", sn_to_str(t->sym_num));
	}
    else if (t->type == VARIABLE) {	  /* variable */
	print_variable_lisp(fp, t);
	}
    else {  /* complex */
	    fprintf(fp, "(");
	    /* if (t->varnum == ANSWER) fprintf(fp, "Answer"); */
	    fprintf(fp, "%s", sn_to_str(t->sym_num));
	    fprintf(fp, " ");
	    r = t->farg;
	    while(r != NULL) {
		print_term_lisp(fp, r->argval);
		r = r->narg;
		if(r != NULL)
		    fprintf(fp, " ");
		}
	    fprintf(fp, ")");
	}
}  /* print_term_lisp */

/*************
 *
 *    print_term_nl(fp, term)  --  print_term followed by newline
 *
 *************/

void print_term_nl(fp, t)
FILE *fp;
struct term *t;
{
    print_term(fp, t);
    fprintf(fp,"\n");
}  /* print_term_nl */

/*************
 *
 *   print_variable(fp, variable)
 *
 *************/

void print_variable(fp, t)
FILE *fp;
struct term *t;
{
    int i;

    if (t->sym_num != 0)
	fprintf(fp, "%s", sn_to_str(t->sym_num));
    else if (Flags[PROLOG_STYLE_VARIABLES].val) {
	fprintf(fp, "%c", (t->varnum % 26) + 'A');
	i = t->varnum / 26;
	if (i > 0)
	    fprintf(fp, "%d", i);
	}
    else {
	if (t->varnum <= 2)
	    fprintf(fp, "%c", 'x'+t->varnum);
	else if (t->varnum <= 5)
	    fprintf(fp, "%c", 'r'+t->varnum);
	else
	    fprintf(fp, "%c%d", 'v', t->varnum);
	}
}  /* print_variable */

void print_variable_lisp(fp, t)
FILE *fp;
struct term *t;
{
    int i;

    fprintf(fp, "?");

    if (t->sym_num != 0)
	fprintf(fp, "%s", sn_to_str(t->sym_num));
    else if (Flags[PROLOG_STYLE_VARIABLES].val) {
	fprintf(fp, "%c", (t->varnum % 26) + 'A');
	i = t->varnum / 26;
	if (i > 0)
	    fprintf(fp, "%d", i);
	}
    else {
	if (t->varnum <= 2)
	    fprintf(fp, "%c", 'x'+t->varnum);
	else if (t->varnum <= 5)
	    fprintf(fp, "%c", 'r'+t->varnum);
	else
	    fprintf(fp, "%c%d", 'v', t->varnum);
	}
}  /* print_variable_lisp */

/*************
 *
 *    int str_to_sn(string, arity) -- Return a symbol number for string/arity.
 *
 *        If the given string/arity is already in the global symbol table,
 *    then return symbol number; else, create a new symbol table entry and
 *    return a new symbol number
 *
 *************/

int str_to_sn(s, arity)
char *s;
int arity;
{
    struct sym_ent *p, *r;
    int i;

    for (i = 0; i < SYM_TAB_SIZE; i++) {
	p = Sym_tab[i];
	while (p != NULL) {
	    if (str_ident(s, p->name) == 0)
	        p = p->next;
	    else if (p->arity == arity)
		return(p->sym_num);
	    else if (Flags[CHECK_ARITY].val)
		return(0);
	    else
		p = p->next;
	    }
	}
	
    r = get_sym_ent();
    str_copy(s, r->name);
    r->sym_num = new_sym_num();
    r->arity = arity;
    i = r->sym_num % SYM_TAB_SIZE;
    r->next = Sym_tab[i];
    Sym_tab[i] = r;
    return(r->sym_num);
}  /* str_to_sn */

/*************
 *
 *    print_syms(file_ptr) -- Display the symbol list.
 *
 *************/

void print_syms(fp)
FILE *fp;
{
    struct sym_ent *p;
    int i;

    for (i = 0; i < SYM_TAB_SIZE; i++) {
	p = Sym_tab[i];
	while (p != NULL) {
	    fprintf(fp, "%d  %s/%d \n", p->sym_num, p->name, p->arity);
	    p = p->next;
	    }
	}
}  /* print_syms */

/*************
 *
 *    p_syms()
 *
 *************/

void p_syms()
{
    print_syms(stdout);
}  /* p_syms */

/*************
 *
 *    free_sym_tab() -- free all symbols in the symbol table
 *  Called from "free_all_mem" in misc.c 
 *
 *************/

void free_sym_tab()
{
    struct sym_ent *p1, *p2;
    int i;

    for (i = 0; i < SYM_TAB_SIZE; i++) {
	p1 = Sym_tab[i];
	while (p1 != NULL) {
	    p2 = p1;
	    p1 = p1->next;
	    free_sym_ent(p2);
	    }
	Sym_tab[i] = NULL;
	}
}  /* free_sym_tab */

/*************
 *
 *    char *sn_to_str(sym_num)  --  given a symbol number, return the name
 * 
 *************/

char *sn_to_str(sym_num)
int sym_num;
{
    struct sym_ent *p;

    p = Sym_tab[sym_num % SYM_TAB_SIZE];
    while (p != NULL && p->sym_num != sym_num)
	p = p->next;
    if (p == NULL)
	return("");
    else
	return(p->name);
}  /* sn_to_str */

/*************
 *
 *    int in_sym_tab(s)  --  is s in the symbol table?
 *
 *************/

int in_sym_tab(s)
char *s;
{
    struct sym_ent *p;
    int i;

    for (i = 0; i < SYM_TAB_SIZE; i++) {
	p = Sym_tab[i];
	while (p != NULL) {
	    if (str_ident(p->name, s))
		return(1);
	    p = p->next;
	    }
	}
    return(0);
}  /* in_sym_tab */

/*************
 *
 *    str_copy(s, t)  --  Copy a string.
 *
 *************/
    
void str_copy(s, t)
char *s;
char *t;
{
    while (*t++ = *s++);
}  /* str_copy */

/*************
 *
 *     int str_ident(s, t) --  Identity of strings
 *
 *************/

int str_ident(s, t)
char *s;
char *t;
{
    for ( ; *s == *t; s++, t++)
	if (*s == '\0') return(1);
    return(0);
}  /* str_ident */

/*************
 *
 *     int initial_str(s, t)  --  Is s an initial substring of t?
 *
 *************/

int initial_str(s, t)
char *s;
char *t;
{
    for ( ; *s == *t; s++, t++)
	if (*s == '\0') return(1);
    return(*s == '\0');
}  /* initial_str */
    
/*************
 *
 *     int read_buf(file_ptr, buffer)
 *
 *    Read characters into buffer until one of the following:
 *        1.  '.' is reached ('.' goes into the buffer)
 *        2.  EOF is reached:  buf[0] = '\0' (an error occurs if
 *                 any nonwhite space precedes EOF)
 *        3.  MAX_BUF - 2 characters have been read (an error occurs)
 *
 *    If error occurs, return(0), else return(1).
 *
 *************/

int read_buf(fp, buf)
FILE *fp;
char buf[];
{
    int c, i, j;
    
    i = 0;
    c = getc(fp);
    while (c != '.' && c != EOF && i < MAX_BUF - 2) {
	if (c == '\n' || c == '\t')
	    c = ' ';
	if (c == '%') {  /* comment--flush rest of line */
	    c = getc(fp);
	    while (c != '\n' && c != EOF)
		c = getc(fp);
	    c = ' ';
	    }
	buf[i++] = c;
	c = getc(fp);
	}
    if (c == '.') {
	buf[i++] = '.';
	buf[i] = '\0';
	return(1);
	}
    else if (c == EOF) {
	j = 0;
	buf[i] = '\0';
	skip_white(buf, &j);
	if (i != j) {
	    fprintf(stdout, "ERROR, text after last period: %s\n", buf);
	    buf[0] = '\0';
	    return(0);
	    }
	else {
	    buf[0] = '\0';
	    return(1);
	    }
	}
    else {
	buf[i] = '\0';
	fprintf(stdout, "ERROR, input string more than %d characters, increase MAX_BUF : %s\n", MAX_BUF, buf);
	/* now flush and discard */
	c = getc(fp);
	while (c != EOF && c != '.')
	    c = getc(fp);
	return(0);
	}
}  /* read_buf */

/*************
 *
 *    skip_white(buffer, position)
 *
 *        Advance the buffer to the next nonwhite position.
 *
 *************/

void skip_white(buf, bufp)
char buf[];
int *bufp;
{
    char c;
    c = buf[*bufp];
    while (c == ' ' || c == '\t' || c == '\n')
	c = buf[++(*bufp)];
}  /* skip_white */

/*************
 *
 *    int is_delim(c)
 *
 *************/

int is_delim(c)
char c;
{
    return(c == '(' || c == ',' || c == ')' || c == '.'  || 
	   c == '|' ||
	   c == ' ' || c == '\t' ||
	   c == '[' || c == ']' ||
	   c == '\n');
}

/*************
 *
 *    get_word(buffer, position, word)
 *
 *************/

void get_word(buf, bufp, word)
char buf[];
int *bufp;
char word[];
{
    int i;
    char c;
    i = 0;
    skip_white(buf, bufp);
    c = buf[*bufp];
    while (i < MAX_NAME-1 && is_delim(c) == 0) {
	word[i++] = c;
	c = buf[++(*bufp)];
	}
    word[i] = '\0';
    if (is_delim(c))
	skip_white(buf, bufp);
    else {
	fprintf(stdout, "ERROR, word too big, max is %d; ", MAX_NAME-1);
	word[0] = '\0';
	}
}  /* get_word */

/*************
 *
 *    struct term *str_term(buffer, position) -- parse buffer and build term
 *
 *    If a syntax error is found, print message and return(NULL).
 *
 *************/

struct term *str_term(buf, bufp)
char buf[];
int *bufp;
{
    struct term *t1, *t2;
    struct rel *r1, *r2;
    char word[MAX_NAME];
    int i, save_pos;
    
    get_word(buf, bufp, word);
    if (word[0] == '\0') {
	    fprintf(stdout, "ERROR, bad word:\n");
	    print_error(stdout, buf, *bufp);
	    return(NULL);
	}
    else {
	t1 = get_term();
	if (buf[*bufp] != '(') {
	    t1->type = NAME;  /* decide later if variable */
	    t1->sym_num = str_to_sn(word, 0);
	    if (t1->sym_num == 0) {
	        fprintf(stdout, "ERROR, multiple arities :%s:\n", word);
	        print_error(stdout, buf, *bufp);
	        return(NULL);
	        }
	    else
	        return(t1);
	    }
	else {
	    t1->type = COMPLEX;
	    r1 = NULL;
	    i = 0;  /* count subterms to get arity */
	    save_pos = *bufp;  /* in case of error */
	    while (buf[*bufp] != ')') {
		i++;
		(*bufp)++;    /* skip past comma or open paren */
		t2 = str_term(buf, bufp);
		if (t2 == NULL)
		    return(NULL);
		else if (buf[*bufp] != ',' && buf[*bufp] != ')') {
		    fprintf(stdout, "ERROR, comma or ) expected:\n");
		    print_error(stdout, buf, *bufp);
		    return(NULL);
		    }
		else {
		    r2 = get_rel();
		    r2->argval = t2;
		    if (r1 == NULL)
			t1->farg = r2;
		    else
			r1->narg = r2;
		    r1 = r2;
		    }
		}
	    (*bufp)++;    /* skip past close paren */
	    skip_white(buf, bufp);
	    t1->sym_num = str_to_sn(word, i);  /* functor */
	    if (t1->sym_num == 0) {
		fprintf(stdout, "ERROR, multiple arities :%s:\n", word);
		print_error(stdout, buf, save_pos);
		return(NULL);
		}
	    else
		return(t1);
	    }
	}
}  /* str_term */

/*************
 *
 *    struct term *str_atom(buf, bufp, signp)
 *
 *************/

struct term *str_atom(buf, bufp, signp)
char buf[];
int *bufp;
int *signp;
{
    skip_white(buf, bufp);

    if (buf[*bufp] != '(') {
	*signp = 1;
	return(str_term(buf, bufp));
	}
    else {
	fprintf(stdout, "ERROR, bad term.\n");
	print_error(stdout, buf, *bufp);
	return(NULL);
	}
}  /* str_atom */

/*************
 *
 *    int set_vars(term)
 *
 *        Decide which of the names are really variables, and make
 *    into variables.  (This routine is used only on input terms.)
 *    Preserve the user's variable names by keeping the pointer into
 *    the symbol list.
 *
 *    If too many variabls, return(0); elase return(1).
 *
 *************/

int set_vars(t)
struct term *t;
{
    char *varnames[MAX_VARS];
    int i;
    
    for (i=0; i<MAX_VARS; i++)
	varnames[i] = NULL;
    
    return(set_vars_term(t, varnames));
}  /* set_vars */

/*************
 *
 *     int set_vars_term(term, varnames)
 *
 *************/

int set_vars_term(t, varnames)
struct term *t;
char *varnames[];
{
    struct rel *r;
    int i, rc;
    
    if (t->type == COMPLEX) {
	r = t->farg;
	rc = 1;
	while (rc && r != NULL) {
	    rc = set_vars_term(r->argval, varnames);
	    r = r->narg;
	    }
	return(rc);
	}
    else if (var_name(sn_to_str(t->sym_num)) == 0)
	return(1);
    else {
	i = 0;
	t->type = VARIABLE;
	while (i < MAX_VARS && varnames[i] != NULL &&
	       varnames[i] != sn_to_str(t->sym_num))
	    i++;
	if (i == MAX_VARS)
	    return(0);
	else {
	    if (varnames[i] == NULL)
		varnames[i] = sn_to_str(t->sym_num);
	    t->varnum = i;
	    return(1);
/* t->sym_num = 0;  include this to destroy input variable names */
	    }
	}
}  /* set_vars_term */

/*************
 *
 *    int var_name(string) -- Decide if a string represents a variable.
 *
 *        return("string is a variable")
 *
 *************/

int var_name(s)
char *s;
{
    if (Flags[PROLOG_STYLE_VARIABLES].val)
        return((*s >= 'A' && *s <= 'Z') || *s == '_');
    else
        return(*s >= 'u' && *s <= 'z');
}  /* var_name */

/*************
 *
 *    struct term *read_term(file_ptr, retcd_ptr) -- Read and return then next term.
 *
 *        It is assumed that the next term in the file is terminated
 *    with a period.   NULL is returned if EOF is reached first.
 *
 *    If an error is found, return(0); else return(1).
 *
 *************/

struct term *read_term(fp, rcp)
FILE *fp;
int *rcp;
{
    char buf[MAX_BUF];
    int p, rc;
    struct term *t;
    
    rc = read_buf(fp, buf);
    if (rc == 0) {  /* error */
	*rcp = 0;
	return(NULL);
	}
    else if (buf[0] == '\0') {  /* ok. EOF */
	*rcp = 1;
	return(NULL);
	}
    else {
	p = 0;
	t = str_term(buf, &p);
	if (t == NULL) {
	    *rcp = 0;
	    return(NULL);
	    }
	else {
	    skip_white(buf, &p);
	    if (buf[p] != '.') {
		fprintf(stdout, "ERROR, text after term:\n");
		print_error(stdout, buf, p);
		*rcp = 0;
		return(NULL);
		}
	    else {
		if (set_vars(t)) {
		    *rcp = 1;
		    return(t);
		    }
		else {
		    fprintf(stdout, "ERROR, too many variables, max is %d:\n%s\n",
				       MAX_VARS, buf);
		    *rcp = 0;
		    return(NULL);
		    }
		}
		    
	    }
	}
}  /* read_term */

/*************
 *
 *    print_error(fp, buf, pos)
 *
 *************/

void print_error(fp, buf, pos)
FILE *fp;
char buf[];
int pos;
{
    int i;

    fprintf(fp, "%s\n", buf);
    for (i = 0; i < pos; i++)
	fprintf(fp, "-");
    fprintf(fp, "^\n");
}  /* print_error */

/*************
 *
 *    int_str(int, str) -- translate an integer to a string
 *
 *************/

void int_str(i, s)
int i;
char s[];
{
    int j, sign;

    if ((sign = i) < 0)
	i = -i;
    
    j = 0;
    if (i == 0)
        s[j++] = '0';
    else {
	while (i > 0) {
            s[j++] = i % 10 + '0';
            i = i / 10;
	    }
        }
    if (sign < 0)
	s[j++] = '-';
    s[j] = '\0';
    reverse(s);
}  /* int_str */

/*************
 *
 *    reverse(s) -- reverse a string
 *
 *************/

void reverse(s)
char s[];
{
    int i, j;
    char temp;

    for (i = 0, j = strlen(s)-1; i<j; i++, j--) {
	temp = s[i];
	s[i] = s[j];
	s[j] = temp;
	}
}  /* reverse */

/*************
 *
 *    cat_str(s1, s2, s3)
 *
 *************/

void cat_str(s1,s2,s3)
char *s1;
char *s2;
char *s3;
{
    int i, j;

    for (i = 0; s1[i] != '\0'; i++)
        s3[i] = s1[i];
    for (j = 0; s2[j] != '\0'; j++, i++)
	s3[i] = s2[j];
    s3[i] = '\0';
}  /* cat_str */

