#include "header.h"

/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */


/*
 *
 *    This file has routines to input and output quantified formulas and
 *    to convert them to lists of clauses (Skolemization and CNF translation).
 *
 *    Well-formed-formulas:
 *        - an atom is a WFF
 *        - if F is a non-negated WFF, then -F is a WFF.
 *        - if F and G are WFFs, then (F <-> G) and (F -> G) are WFFs.
 *        - if F1, ..., Fn are WFFs, then (F1 & ... & Fn) and (F1 | ... | Fn)
 *          are WFFs.
 *        - if F is a WFF, Q1 ... Qn are quantifiers ("all" or "exists"),
 *          and X1 ... Xn are symbols, then (Q1 X1 ... Qn Xn F) is a WFF.
 *
 *    Note, double negations are not allowed, and all parentheses must be
 *    included:  --F is not a WFF, and (A & B -> C) is not a WFF.
 *
 */

/* Formulas are represented as terms.                           */
/* The scratch field is used to identify the type of formula.   */
/* All atoms and negated atoms have scratch=0.                  */
/* The scratch2 field gives the sign on an atom.                */

#define AND_FORM 1
#define OR_FORM 2
#define IMP_FORM 3
#define IFF_FORM 4
#define QUANT_FORM 5
#define EQ_FORM 6    /* a special case, infix =, not put into scratch */

/*************
 *
 *    print_formula(fp, t) -- print a formula to a file.
 *
 *************/

void print_formula(fp, t)
FILE *fp;
struct term *t;
{
    struct rel *r;
    char op[MAX_NAME];
    int f;

    if (t == NULL)
	fprintf(fp, "(nil)");
    else {
	    if (t->scratch2 == 0)
		fprintf(fp, "-");

	    if (t->scratch == 0 || t->type != COMPLEX)
		print_term(fp, t);
	    else if (t->scratch == AND_FORM && t->farg == NULL)
		fprintf(fp, "TRUE");
	    else if (t->scratch == OR_FORM && t->farg == NULL)
		fprintf(fp, "FALSE");
	    else {
		fprintf(fp, "(");
		f = t->scratch;

		if (f == AND_FORM)
		    str_copy("& ", op);
		else if (f == OR_FORM)
		    str_copy("| ", op);
		else if (f == IMP_FORM)
		    str_copy("-> ", op);
		else if (f == IFF_FORM)
		    str_copy("<-> ", op);
		else op[0] = '\0';

		r = t->farg;
		while (r != NULL) {
		    print_formula(fp, r->argval);
		    r = r->narg;
		    if (r != NULL)
			fprintf(fp, " %s", op);
		    }
		fprintf(fp, ")");
		}
	}
}  /* print_formula */

/*************
 *
 *    struct term *read_formula(fp, rcp) -- read a formula from a file
 *
 *    The return code *rcp:
 *        0 - an error was encountered and reported; NULL is returned.
 *        1 - OK; if EOF was found instead of a formula, NULL is returned.
 *
 *************/

struct term *read_formula(fp, rcp)
FILE *fp;
int *rcp;
{
    char buf[MAX_BUF];
    int p, rc;
    struct term *t;
    
    rc = read_buf(fp, buf);
    if (rc == 0) {  /* error */
	*rcp = 0;
	return(NULL);
	}
    else if (buf[0] == '\0') {  /* ok. EOF */
	*rcp = 1;
	return(NULL);
	}
    else {
	p = 0;
	t = str_formula(buf, &p);
	if (t == NULL) {
	    *rcp = 0;
	    return(NULL);
	    }
	else {
	    skip_white(buf, &p);
	    if (buf[p] != '.') {
		fprintf(stdout, "ERROR, text after formula:\n");
		print_error(stdout, buf, p);
		*rcp = 0;
		return(NULL);
		}
	    *rcp = 1;
	    return(t);
	    }
	}
}  /* read_formula */

/*************
 *
 *    struct term *str_formula(buf, bufp) -- convert part of a string in to a formula.
 *
 *    *bufp is an integer giving the current position in the string.
 *    *bufp is updated by this routine.
 *
 *************/

struct term *str_formula(buf, bufp)
char buf[];
int *bufp;
{
    struct term *t1, *t2;
    struct rel *r1, *r2;
    char word[MAX_NAME], *s;
    int i, save_pos, sign, formula_type, quant_done;
    
    skip_white(buf, bufp);
    if (buf[*bufp] == '-' && buf[(*bufp)+1] != '>') {
	sign = 0;
	(*bufp)++;
	skip_white(buf, bufp);
	if (buf[*bufp] == '-') {
	    fprintf(stdout, "ERROR, double negation:\n");
	    print_error(stdout, buf, *bufp);
	    return(NULL);
	    }
	}
    else
	sign = 1;

    if (buf[*bufp] == '(') {
	(*bufp)++;    /* skip past open paren */
	i = 0;  /* count arguments */
	formula_type = 0;
	t1 = get_term();
	t1->type = COMPLEX;
	r1 = NULL;
	save_pos = *bufp;  /* in case of error */
	while (buf[*bufp] != ')') {
	    i++;
	    save_pos = *bufp;
	    skip_white(buf, bufp);
	    /* '|' must be handled specially, because it is a delimiter */
	    if (buf[*bufp] == '|' && buf[(*bufp) + 1] == ' ') {
		(*bufp)++;  /* skip past '|' */
		skip_white(buf, bufp);
		t2 = get_term();
		t2->type = NAME;
		t2->sym_num = str_to_sn("|", 0);
		}
	    else
	        t2 = str_formula(buf, bufp);
	    if (t2 == NULL)
		return(NULL);  /* an error has already been handled */
	    else {
		s = sn_to_str(t2->sym_num);
		if (i == 1) {
		    if (t2->type == NAME && (str_ident(s,"all") || str_ident(s,"exists"))) {
			formula_type = QUANT_FORM;
			quant_done = 0;
			}
		    }
		else if (formula_type == QUANT_FORM) {
		    if (quant_done) {
			fprintf(stdout, "ERROR, bad quantified formula:\n");
			print_error(stdout, buf, save_pos);
			return(NULL);
			}
		    else if (i % 2 == 0 && t2->type != NAME) {
			fprintf(stdout, "ERROR, variable name expected:\n");
			print_error(stdout, buf, save_pos);
			return(NULL);
			}
		    else if (i % 2 == 1 && (str_ident(s,"all") == 0 && str_ident(s,"exists") == 0))
			quant_done = 1;
		    }
		else if (i == 2) {  /* must be AND, OR, IMP, or IFF, or =, or != */
		    if (str_ident(s, "&"))
			formula_type = AND_FORM;
		    else if (str_ident(s, "|"))
			formula_type = OR_FORM;
		    else if (str_ident(s, "->"))
			formula_type = IMP_FORM;
		    else if (str_ident(s, "<->"))
			formula_type = IFF_FORM;
		    else if (str_ident(s, "="))
			formula_type = EQ_FORM;
		    else if (str_ident(s, "!=")) {
			formula_type = EQ_FORM;
			sign = (sign ? 0 : 1);
			}
		    
		    if (formula_type == 0 || t2->type != NAME) {
			fprintf(stdout, "ERROR, logical operator, '=', or '!=' expected:\n");
			print_error(stdout, buf, save_pos);
			return(NULL);
			}
		    }
		else if (i > 3 && (formula_type == IMP_FORM ||
				   formula_type == IFF_FORM ||
				   formula_type == EQ_FORM)) {
			fprintf(stdout, "ERROR, too many arguments:\n");
			print_error(stdout, buf, save_pos);
			return(NULL);
			}
		else if (i % 2 == 0) {
		    if ((formula_type == AND_FORM && str_ident(s, "&") == 0) ||
		        (formula_type == OR_FORM  && str_ident(s, "|")  == 0)) {

			fprintf(stdout, "ERROR, operators switched:\n");
			print_error(stdout, buf, save_pos);
			return(NULL);
			}
		    }
		/* else ok: AND or OR, and odd-numbered argument */

		if (i % 2 == 1 || formula_type == QUANT_FORM) {
		    r2 = get_rel();
		    r2->argval = t2;
		    if (r1 == NULL)
			t1->farg = r2;
		    else
			r1->narg = r2;
		    r1 = r2;
		    }
		else
		    free_term(t2);  /* free operator */
		}
	    }
	if (i < 3) {
	    fprintf(stdout, "ERROR, too few arguments:\n");
	    print_error(stdout, buf, save_pos);
	    return(NULL);
	    }
	else {
	    (*bufp)++;    /* skip past close paren */
	    skip_white(buf, bufp);
	    /* if (formula_type == EQ_FORM)
		t1->sym_num = Eq_sym_num;
	    else */
	        t1->scratch = formula_type;
	    t1->scratch2 = sign;
	    return(t1);
	    }
	}
    else {
	i = *bufp;
	while (is_delim(buf[i]) == 0)
	    i++;
	if (buf[i] == ' ') {  /* next thing is not an atom with arguments */
	    get_word(buf, bufp, word);
	    if (word[0] == '\0') {
		fprintf(stdout, "ERROR, bad word:\n");
		print_error(stdout, buf, *bufp);
		return(NULL);
		}
	    else {
		t1 = get_term();
		t1->type = NAME;
		t1->sym_num = str_to_sn(word, 0);
	        t1->scratch2 = sign;
		return(t1);
		}
	    }
	else {  /* next thing is atom with arguments */
	    t1 = str_term(buf, bufp);
	    if (t1 == NULL)
		return(NULL);
	    else {
		t1->scratch2 = sign;
		return(t1);
		}
	    }
	}
}  /* str_formula */

/*************
 *
 *    negate(t) -- negate a formula
 *
 *************/

void negate(t)
struct term *t;
{
    t->scratch2 = (t->scratch2 ? 0 : 1);
}

/*************
 *
 *    struct term *nnf(t) -- convert to negation normal form
 *
 *    t is changed into its negation normal form; the correct way to call
 *    this routine is t = nnf(t).
 *
 *************/

struct term *nnf(t)
struct term *t;
{
    struct rel *r, *r1, *r2;
    struct term *t1, *t2;
    int sign, all, exists, i;

    switch (t->scratch) {
	case 0: 
	    return(t);  /* t is atomic */
	case IFF_FORM:
	    t2 = copy_term_scratch(t);
	    t1 = get_term();
	    r1 = get_rel(); r2 = get_rel();
	    t1->farg = r1; r1->narg = r2;
	    r1->argval = t; r2->argval = t2;
	    t1->scratch2 = t->scratch2;
	    t1->type = COMPLEX;
	    t->scratch2 = t2->scratch2 = 1;

	    if (t1->scratch2) {  /* (a iff b) => ((-a | b) and (a | -b)) */
		t1->scratch = AND_FORM;
		t->scratch = t2->scratch = OR_FORM;
		negate(t->farg->argval);
		negate(t2->farg->narg->argval);
		}
	    else {               /* -(a iff b) => -((a & b) | (-a & -b)) */
		t1->scratch = OR_FORM;
		t->scratch = t2->scratch = AND_FORM;
		negate(t2->farg->argval);
		negate(t2->farg->narg->argval);
		}

	    return(nnf(t1));
	case IMP_FORM:
	    t->scratch = OR_FORM;
	    negate(t->farg->argval);
	    return(nnf(t));
	case QUANT_FORM:
	    sign = t->scratch2;
	    if (sign == 0)
		t->scratch2 = 1;
	    all = str_to_sn("all", 0);
	    exists = str_to_sn("exists", 0);
	    r = t->farg;
	    i = 1;
	    while (r != NULL) {
		if (i % 2 == 1 && r->narg != NULL) {  /* if quantifier */
		    if (sign == 0)  /* if negate, then swith quantifier */
			r->argval->sym_num = (r->argval->sym_num == all ? exists : all);
		    }
		else if (r->narg == NULL) {  /* if formula */
		    if (sign == 0)
		        negate(r->argval);
		    r->argval = nnf(r->argval);
		    }
		r = r->narg;
		i++;
		}
	    return(t);
	case AND_FORM:
	case OR_FORM:
	    sign = t->scratch2;
	    if (sign == 0) {
		t->scratch2 = 1;
		t->scratch = (t->scratch == AND_FORM ? OR_FORM : AND_FORM);
		}
	    r = t->farg;
	    while (r != NULL) {
		if (sign == 0)
		    negate(r->argval);
		r->argval = nnf(r->argval);
		r = r->narg;
		}
	    return(t);
	}
    print_formula(stdout, t);
    printf(": bad op code.\n");
    return(t);
}  /* nnf */

/*************
 *
 *    skolemize(t) -- Skolemize a term
 *
 *    This routine assumes that t is in negation normal form.
 *    The existential quantifiers are deleted (which may result in a
 *    "quantified formula" with no quantifiers).
 *
 *************/

void skolemize(t)
struct term *t;
{
    struct term *vars;

    vars = get_term();
    vars->type = COMPLEX;
    skolem(t, vars);
    free_term(vars);
}  /* skolemize */

/*************
 *
 *    skolem(t, vars) -- skolemize t w.r.t universally quantified vars
 *
 *    Called by skolemize.
 *
 *************/

void skolem(t, vars)
struct term *t;
struct term *vars;
{
    struct rel *r, *r_save, *end, *r2, *r1, *prev;
    struct term *tr, *var;
    int all, sn;

    if (t->scratch != 0 && t->scratch2 == 0) {
	printf("ERROR, skolem gets negated non-atom: ");
	print_formula(stdout, t);
	printf("\n");
	}
    else if (t->scratch == IMP_FORM || t->scratch == IFF_FORM) {
	printf("ERROR, skolem gets: ");
	print_formula(stdout, t);
	printf("\n");
	}
    else if (t->scratch == AND_FORM || t->scratch == OR_FORM) {
	r = t->farg;
	while (r != NULL) {
	    skolem(r->argval, vars);
	    r = r->narg;
	    }
	}
    else if (t->scratch == QUANT_FORM) {
	/* first get formula in which to make replacements */
	r = t->farg;
	while (r != NULL) {
	    tr = r->argval;
	    r = r->narg;
	    }
	/* next get end of variable list */
	r = vars->farg;
	r_save = NULL;
	while (r != NULL) {
	    r_save = r;
	    r = r->narg;
	    }
	end = r_save;
	/* now go through arguments */
	all = str_to_sn("all", 0);
	prev = NULL;
	r = t->farg;
	/* each pass through loop processes one quantifier */
	while (r != NULL) {
	    if (r->narg != NULL) {  /* if quantifier */
		if (r->argval->sym_num == all) {
		    var = r->narg->argval;
		    if (occurs_in(var, vars)) {
			sn = new_var_name();
			subst_sn(var->sym_num, tr, sn, NAME);
			var->sym_num = sn;
			}
		    r2 = get_rel();
		    if (end == NULL)
			vars->farg = r2;
		    else
			end->narg = r2;
		    end = r2;
		    end->argval = var;
		    /* skip variable */
		    prev = r->narg;
		    r = prev->narg;
		    }
		else {  /* existential quantifier */
		    gen_sk_sym(vars);
		    subst_free(r->narg->argval, tr, vars);
		    /* delete quantifier and variable */
		    if (prev == NULL)
			t->farg = r->narg->narg;
		    else
			prev->narg = r->narg->narg;
		    r1 = r;
		    r = r->narg->narg;
		    free_term(r1->narg->argval);  /* variable */
		    free_term(r1->argval);  /* exists */
		    free_rel(r1->narg);
		    free_rel(r1);
		    }
		}
	    else  /* no action, quantified formula */
	        r = r->narg;  /* r == NULL after this statement */
	    }  /* while */

	skolem(tr, vars);  /* recursive call to skolemize quantified formula */

	/* free new universally quantified vars from vars */
	if (r_save == NULL) {
	    r2 = vars->farg; vars->farg = NULL;
	    }
	else {
	    r2 = r_save->narg; r_save->narg = NULL;
	    }
	while (r2 != NULL) {
	    r1 = r2;
	    r2 = r2->narg;
	    free_rel(r1);
	    }
	}
}  /* skolem */

/*************
 *
 *    subst_free(var, t, sk)
 *
 *    Substitute free occurrences of var in t with copies of sk.
 *
 *************/

void subst_free(var, t, sk)
struct term *var;
struct term *t;
struct term *sk;
{
    struct rel *r;
    struct term *t1;
    int i;

    if (t->type != COMPLEX)
	return;
    else {
	if (t->scratch == QUANT_FORM) {
	    r = t->farg;
	    i = 1;
	    while (r != NULL) {
		if (i % 2 == 0 && term_ident(var, r->argval))
		    return;
		t1 = r->argval;
		r = r->narg;
		i++;
		}
	    subst_free(var, t1, sk);
	    }
	r = t->farg;
	while (r != NULL) {
	    if (term_ident(var, r->argval)) {
		zap_term(r->argval);
		r->argval = copy_term(sk);
		}
	    else
		subst_free(var, r->argval, sk);
	    r = r->narg;
	    }
	}
}  /* subst_free */

/*************
 *
 *    gen_sk_sym(t) -- generate a fresh skolem symbol for term t.
 *
 *    Assign type field as well as sym_num field to term t.
 *    Check and make sure that the new symbol does not occur in the symbol table.
 *
 *************/

void gen_sk_sym(t)
struct term *t;
{
    int arity;
    struct rel *r;
    char s1[MAX_NAME], s2[MAX_NAME];

    static int func_num, const_num;

    arity = 0;
    r = t->farg;
    while (r != NULL) {
	arity++;
	r = r->narg;
	}

    if (arity == 0) {
	t->type = NAME;
	int_str(++const_num, s1);
        cat_str("c", s1, s2);
	while (in_sym_tab(s2)) {
	    int_str(++const_num, s1);
	    cat_str("c", s1, s2);
	    }
	}
    else {
	t->type = COMPLEX;
	int_str(++func_num, s1);
        cat_str("f", s1, s2);
	while (in_sym_tab(s2)) {
	    int_str(++func_num, s1);
	    cat_str("f", s1, s2);
	    }
	}

    t->sym_num = str_to_sn(s2, arity);
    /* mark_as_skolem(t->sym_num); */
  
}  /* gen_sk_sym */

/*************
 *
 *    int new_var_name() -- return a sym_num for a new VARIABLE symbol
 *
 *    Check and make sure that the new symbol does not occur in the symbol table.
 *
 *************/

int new_var_name()
{
    char s1[MAX_NAME], s2[MAX_NAME];

    static int var_num;
    char c[2];

    c[0] = (Flags[PROLOG_STYLE_VARIABLES].val ? 'X' : 'x');
    c[1] = '\0';

    int_str(++var_num, s1);
    cat_str(c, s1, s2);
    while (in_sym_tab(s2)) {
	int_str(++var_num, s1);
	cat_str(c, s1, s2);
	}

    return(str_to_sn(s2, 0));
  
}  /* new_var_name */

/*************
 *
 *    int new_functor_name(arity) -- return a sym_num for a new symbol.
 *
 *    Check and make sure that the new symbol does not occur in the symbol table.
 *
 *************/

int new_functor_name(arity)
int arity;
{
    char s1[MAX_NAME], s2[MAX_NAME];

    static int functor_num;

    int_str(++functor_num, s1);
    cat_str("k", s1, s2);
    while (in_sym_tab(s2)) {
	int_str(++functor_num, s1);
	cat_str("k", s1, s2);
	}

    return(str_to_sn(s2, arity));
  
}  /* new_functor_name */

/*************
 *
 *    unique_all(t, vars) -- make all universally quantified variables unique
 *
 *    It is assumed that t is in negation normal form and is Skolemized (no
 *    existential quantifiers).
 *
 *
 *************/

void unique_all(t)
struct term *t;
{
    struct term *vars;
    struct rel *r1, *r2;

    vars = get_term();
    vars->type = COMPLEX;
    uq_all(t, vars);
    r1 = vars->farg;
    while (r1 != NULL) {
	r2 = r1;
	r1 = r1->narg;
	free_rel(r2);
	}
    free_term(vars);
}  /* unique_all */

/*************
 *
 *    uq_all(t, vars) -- called by unique_all
 *
 *************/

void uq_all(t, vars)
struct term *t;
struct term *vars;
{
    struct rel *r, *r1;
    struct term *tr, *var;
    int sn;

    if (t->scratch == AND_FORM || t->scratch == OR_FORM) {
	r = t->farg;
	while (r != NULL) {
	    uq_all(r->argval, vars);
	    r = r->narg;
	    }
	}
    else if (t->scratch == QUANT_FORM) {
	/* get formula in which to make replacements */
	r = t->farg;
	while (r != NULL) {
	    tr = r->argval;
	    r = r->narg;
	    }
	r = t->farg;
	while (r->narg != NULL) {
	    var = r->narg->argval;
	    if (occurs_in(var, vars)) {
		sn = new_var_name();
		subst_sn(var->sym_num, tr, sn, NAME);
		var->sym_num = sn;
		}
	    else {
	        r1 = get_rel();
		r1->argval = var;
		r1->narg = vars->farg;
		vars->farg = r1;
		}
	    /* skip variable */
	    r = r->narg->narg;
	    }

	uq_all(tr, vars);  /* recursive call on quantified formula */

	}
}  /* uq_all */

/*************
 *
 *    struct term *zap_quant(t) -- delete quantifiers and mark quantified variables
 *
 *    It is assumed that t is skolemized nnf.  For each universal quantifier,
 *    mark all occurrences of the quantified variable by setting the type field
 *    to VARIABLE, then delete the quantifier.
 *    All QUANT_FORM nodes are deleted as well.
 *
 *************/

struct term *zap_quant(t)
struct term *t;
{
    struct rel *r, *r1;
    struct term *tr;

    if (t->scratch == 0)
	return(t);  /* atom or negated atom */
    else if (t->scratch == AND_FORM || t->scratch == OR_FORM) {
	r = t->farg;
	while (r != NULL) {
	    r->argval = zap_quant(r->argval);
	    r = r->narg;
	    }
	return(t);
	}
    else {  /* must be QUANT_FORM */
	r = t->farg;
	while(r != NULL) {
	    tr = r->argval;
	    r = r->narg;
	    }
	r = t->farg;
	while (r->narg != NULL) {
	    mark_free_var(r->narg->argval, tr);
	    r1 = r;
	    r = r->narg->narg;
	    free_term(r1->narg->argval);
	    free_term(r1->argval);
	    free_rel(r1->narg);
	    free_rel(r1);
	    }
	free_rel(r);
	free_term(t);
	return(zap_quant(tr));
	}
}  /* zap_quant */

/*************
 *
 *    mark_free_var(v, t) -- mark free occurrences of v in t 
 *
 *    Each free NAME in t with sym_num == v->sym_num is marked as
 *    a VARIABLE by setting the type field to VARIABLE.
 *
 *************/

void mark_free_var(v, t)
struct term *v;
struct term *t;
{
    struct rel *r;

    if (t->scratch == QUANT_FORM) {
	r = t->farg;
	while(r->narg != NULL) {
	    if (r->narg->argval->sym_num == v->sym_num)
		return;
	    r = r->narg->narg;
	    }
	mark_free_var(v,r->argval);
	}
    else if (t->type == NAME) {
	if (t->sym_num == v->sym_num)
	    t->type = VARIABLE;
	}
    else if (t->type == COMPLEX) {
	r = t->farg;
	while(r != NULL) {
	    mark_free_var(v, r->argval);
	    r = r->narg;
	    }
	}
}  /* mark_free_var */

/*************
 *
 *    flatten_top(t, f_type) -- flatten conjunctions or disjunctions
 *
 *    f_type should be either AND_FORM or OR_FORM.
 *
 *    t is flattened with respect to f_type.  Subtrees of type f_type below
 *    a node of the oppposite type are not flattened.  For example, in
 *    (a or (b and (c or (d or e)))), the formula (c or (d or e)) is never
 *    flattened.
 *
 *************/

void flatten_top(t, f_type)
struct term *t;
int f_type;
{
    struct rel *r1, *r, *prev;

    if (t->scratch == f_type) {
	r = t->farg;
	prev = NULL;
	while (r != NULL) {
	    flatten_top(r->argval, f_type);
	    if (r->argval->scratch == f_type) {
		r1 = r->argval->farg;
		if (r1 == NULL) {  /* empty f_type NODE (TRUE or FALSE) */
		    if (prev == NULL)
			t->farg = r->narg;
		    else
			prev->narg = r->narg;
		    /* prev stays the same */
		    }
		else {
		    while (r1->narg != NULL)
			r1 = r1->narg;
		    if (prev == NULL)
			t->farg = r->argval->farg;
		    else
			prev->narg = r->argval->farg;
		    r1->narg = r->narg;
		    prev = r1;
		    }
		r1 = r;
		r = r->narg;
		free_term(r1->argval);
		free_rel(r1);
		}
	    else {
		prev = r;
		r = r->narg;
		}
	    }
	}
}  /* flatten_top */

/*************
 *
 *    struct term *cnf(t) -- convert t to conjunctive normal form.
 *
 *    The standard calling sequance is t = cnf(t).
 *
 *************/

struct term *cnf(t)
struct term *t;
{
    struct rel *r;

    if (t->scratch == AND_FORM || t->scratch == OR_FORM) {

	/* first convert subterms to CNF */
	r = t->farg;
	while (r != NULL) {
	    r->argval = cnf(r->argval);
	    r = r->narg;
	    }

	if (Flags[SIMPLIFY_FOL].val)
	    ts_and_fs(t);  /* simplify immediate TRUEs and FALSEs */

	if (t->scratch == AND_FORM) {
	    flatten_top(t, AND_FORM);
	    if (Flags[SIMPLIFY_FOL].val) {
		t = subsume_conj(t);
		conflict_tautology(t);
		}
	    }
	else
	    t = distribute(t);
	}

    return(t);
}  /* cnf */

/*************
 *
 *    struct term *distribute(t) -- distribute OR over AND.
 *
 *    t is an OR node whose subterms are in CNF.  This routine returns
 *    a CNF of t.
 *
 *    A good way to call is t = distribute(t).
 *
 *************/

struct term *distribute(t)
struct term *t;
{
    struct term *t1, *temp;
    struct rel *r, *r1;
    int i, j;

    flatten_top(t, OR_FORM);

    if (Flags[SIMPLIFY_FOL].val) {
	t = subsume_disj(t);
	conflict_tautology(t);
	}

    if (t->scratch != OR_FORM)
	return(t);  /* reduced to TRUE by conflict_tautology */
    else {
	/* find first AND subterm */
	r = t->farg;
	i = 0;
	while (r != NULL && r->argval->scratch != AND_FORM) {
	    i++;
	    r = r->narg;
	    }
	if (r == NULL)
	    return(t);  /* nothing to distribute */
	else {
	    t1 = r->argval;
	    temp = t1->farg->argval;
	    t1->farg->argval = t;
	    r->argval = temp;
	    r1 = t1->farg->narg;
	    while (r1 != NULL) {
		temp = r1->argval;
		r1->argval = copy_term_scratch(t);
		r = r1->argval->farg;
		for (j = 0; j < i; j++)
		    r = r->narg;
		zap_term(r->argval);
		r->argval = temp;
		r1->argval = distribute(r1->argval);
		r1 = r1->narg;
		}
	    t1->farg->argval = distribute(t1->farg->argval);
	    flatten_top(t1, AND_FORM);
	    if (Flags[SIMPLIFY_FOL].val) {
		t1 = subsume_conj(t1);
		conflict_tautology(t1);
		}
	    return(t1);
	    }
	}
}  /* distribute */

/*************
 *
 *    rename_syms(t, tr)
 *    
 *    Rename NAMEs and VARIABLEs so that they conform to the rule for clauses:
 *    symbol is a variable iff it starst with u-z.
 *
 *************/

void rename_syms(t, tr)
struct term *t;
struct term *tr;
{
    struct rel *r;
    int sn;

    if (t->type == NAME) {
	if (var_name(sn_to_str(t->sym_num))) {
	    sn = new_functor_name(0);  /* with arity 0 */
	    subst_sn(t->sym_num, tr, sn, NAME);
	    }
	}
    else if (t->type == VARIABLE) {
	if (var_name(sn_to_str(t->sym_num)) == 0) {
	    sn = new_var_name();
	    subst_sn(t->sym_num, tr, sn, VARIABLE);
	    }
	}
    else {
	r = t->farg;
	while(r != NULL) {
	    rename_syms(r->argval, tr);
	    r = r->narg;
	    }
	}
}  /* rename_syms */

/*************
 *
 *    subst_sn(old_sn, t, new_sn, type)  --  called by rename_syms
 *
 *************/

void subst_sn(old_sn, t, new_sn, type)
int old_sn;
struct term *t;
int new_sn;
int type;
{
    struct rel *r;

    if (t->type == NAME) {
	if (type == NAME && t->sym_num == old_sn)
	    t->sym_num = new_sn;
	}
    else if (t->type == VARIABLE) {
	if (type == VARIABLE && t->sym_num == old_sn)
	    t->sym_num = new_sn;
	}
    else {
	r = t->farg;
	while (r != NULL) {
	    subst_sn(old_sn, r->argval, new_sn, type);
	    r = r->narg;
	    }
	}
}  /* subst_sn */

/*************
 *
 *    struct list *cnf_to_list(t)
 *
 *    Convert a CNF formula to a list of clauses.
 *    This includes assigning variable numbers to the varnum fileds of VARIABLES.
 *    An ABEND occurs if a clause has too many variables.
 *
 *************/

struct list *cnf_to_list(t)
struct term *t;
{
    struct list *l;
    struct rel *r, *prev;
    struct clause *c;

    l = get_list();
    if (t->scratch != AND_FORM) {
	c = disj_to_clause(t);
	append_cl(l, c);
	}
    else {  /* OR_FORM or atomic */
	r = t->farg;
	while (r != NULL) {
	    c = disj_to_clause(r->argval);
	    append_cl(l, c);
	    prev = r;
	    r = r->narg;
	    free_rel(prev);
	    }
	free_term(t);
	}
    return(l);
}  /* cnf_to_list */

/*************
 *
 *    struct clause *disj_to_clause(t)
 *
 *************/

struct clause *disj_to_clause(t)
struct term *t;
{
    struct clause *c;
    struct literal *lit, *prev;
    struct rel *r, *r2;
    struct term *t2;

    c = get_clause();
    if (t->scratch == 0) {
	lit = get_literal();
	lit->sign = t->scratch2;
	t->scratch2 = 0;
	lit->atom = t;
	t->occ.lit = lit;
	lit->container = c;
	mark_literal(lit);  /* atoms have varnum > 0 */
	clear_scratches(lit->atom);
	c->first_lit = lit;
	}
    else {  /* OR_FORM */
	prev = NULL;
	r = t->farg;
	while (r != NULL) {
	    t2 = r->argval;
	    lit = get_literal();
	    lit->sign = t2->scratch2;
	    t2->scratch2 = 0;
	    lit->atom = t2;
	    t2->occ.lit = lit;
	    lit->container = c;
	    mark_literal(lit);  /* atoms have varnum > 0 */
	    clear_scratches(lit->atom);
	    if (prev == NULL) 
		c->first_lit = lit;
	    else
		prev->next_lit = lit;
	    prev = lit;
	    r2 = r;
	    r = r->narg;
	    free_rel(r2);
	    }
	free_term(t);
	}
    cl_merge(c);  /* merge identical literals */
    if (set_vars_cl_2(c) == 0) {
	output_stats(stdout, 4);
	fprintf(stderr, "ABEND, too many variables in clause, max is %d.\007\n", MAX_VARS);
	fprintf(stdout, "ABEND, too many variables in clause, max is %d:\n", MAX_VARS);
	print_clause(stdout, c);
	exit(1);
	return(NULL);
	}
    return(c);
}  /* disj_to_clause */

/*************
 *
 *    clear_scratches(t) -- clear scratch areas
 *
 *************/

void clear_scratches(t)
struct term *t;
{
    struct rel *r;

    t->scratch = 0;
    t->scratch2 = 0;

    if (t->type == COMPLEX)
	for (r = t->farg; r != NULL; r = r->narg)
	    clear_scratches(r->argval);

}  /* clear_scratches */

/*************
 *
 *    int set_vars_cl_2(cl) -- give variables var_nums
 *
 *************/

int set_vars_cl_2(cl)
struct clause *cl;
{
    struct literal *lit;
    int sn[MAX_VARS];
    int i;

    for (i=0; i<MAX_VARS; i++)
	sn[i] = -1;
    lit = cl->first_lit;
    while (lit != NULL) {
	if (set_vars_term_2(lit->atom, sn))
	    lit = lit->next_lit;
	else
	    return(0);
	}
    return(1);
}  /* set_vars_cl_2 */

/*************
 *
 *     int set_vars_term_2(term, sn)
 *
 *************/

int set_vars_term_2(t, sn)
struct term *t;
int sn[];
{
    struct rel *r;
    int i, rc;
    
    if (t->type == COMPLEX) {
	r = t->farg;
	rc = 1;
	while (rc && r != NULL) {
	    rc = set_vars_term_2(r->argval, sn);
	    r = r->narg;
	    }
	return(rc);
	}
    else if (t->type == NAME)
	return(1);
    else {
	i = 0;
	while (i < MAX_VARS && sn[i] != -1 && sn[i] != t->sym_num)
	    i++;
	if (i == MAX_VARS)
	    return(0);
	else {
	    if (sn[i] == -1)
		sn[i] = t->sym_num;
	    t->varnum = i;
	    /*  include following to destroy input variable names 
            t->sym_num = 0;
	    */
	    return(1);
	    }
	}
}  /* set_vars_term_2 */

/*************
 *
 *    struct list *clausify(t) -- Skolem/CNF tranformation.
 *
 *    Convert a quantified formula to a list of clauses.
 *
 *************/

struct list *clausify(t)
struct term *t;
{
    struct list *l;

    t = nnf(t);
    skolemize(t);
    unique_all(t);
    t = zap_quant(t);
    rename_syms(t, t);
    t = cnf(t);
    l = cnf_to_list(t);
    return(l);

}  /* clausify */

/*************
 *
 *    struct list *clausify_formula_list(tp)
 *
 *    Clausify a set of formulas, and return a list of clauses.
 *    The set of formulas is deallocated.
 *
 *************/

struct list *clausify_formula_list(tp)
struct term_ptr *tp;
{
    struct list *l, *l1;
    struct term_ptr *tp1, *tp2;

    l = get_list();
    tp1 = tp;
    while (tp1 != NULL) {
	l1 = clausify(tp1->term);
	append_lists(l, l1);
	tp2 = tp1;
	tp1 = tp1->next;
	free_term_ptr(tp2);
	}
    return(l);
}  /* clausify_formula_list */

/*************
 *
 *    struct term_ptr *read_formula_list(file_ptr, errors_ptr) 
 *
 *    Read and return a list of quantified formulas.
 *
 *    The list must be terminated either with the term `end_of_list.'
 *    or with an actual EOF.
 *    Set errors_ptr to point to the number of errors found.
 *
 *************/

struct term_ptr *read_formula_list(fp, ep)
FILE *fp;
int *ep;
{
    struct term_ptr *p1, *p2, *p3;
    struct term *t;
    int rc;

    *ep = 0;
    p3 = NULL;
    p2 = NULL;
    t = read_formula(fp, &rc);
    while (rc == 0) {
	(*ep)++;
	t = read_formula(fp, &rc);
	}

    /* keep going until t == NULL || t is end marker */

    while (t != NULL && (t->type != NAME ||
			 str_ident(sn_to_str(t->sym_num), "end_of_list") == 0)) {
	p1 = get_term_ptr();
	p1->term = t;
	if (p2 == NULL)
	    p3 = p1;
	else
	    p2->next = p1;
	p2 = p1;
	t = read_formula(fp, &rc);
	while (rc == 0) {
	    (*ep)++;
	    t = read_formula(fp, &rc);
	    }
	}
    if (t != NULL)
	zap_term(t);
    return(p3);
}  /* read_formula_list */

/*************
 *
 *    print_formula_list(file_ptr, term_ptr) -- Print a list of quantified formulas.
 *
 *        The list is printed with periods after each quantified formula, and
 *    the list is terminated with `end_of_list.' so that it can
 *    be read with read_formula_list.
 *
 *************/

void print_formula_list(fp, p)
FILE *fp;
struct term_ptr *p;
{
    while (p != NULL) {
	fprintf(fp,";; ");
	print_formula(fp, p->term);
	fprintf(fp, ".\n");
	p = p->next;
	}
    fprintf(fp, ";; end_of_list.\n");
}  /* print_formula_list */

/*************
 *
 *    int gen_subsume(c, d) -- does c gen_subsume d?
 *
 *************/

int gen_subsume(c, d)
struct term *c;
struct term *d;
{
    struct rel *r;

    if (c->scratch == OR_FORM) {  /* return(each c_i gen_subsumes d */
	r = c->farg;
	while (r != NULL && gen_subsume(r->argval, d))
	    r = r->narg;
	return(r == NULL);
	}
    else if (d->scratch == AND_FORM) {  /* return(c gen_subsumes each d_i */
	r = d->farg;
	while (r != NULL && gen_subsume(c, r->argval))
	    r = r->narg;
	return(r == NULL);
	}
    else if (c->scratch == AND_FORM) {  /* return(one c_i gen_subsumes d */
	r = c->farg;
	while (r != NULL && gen_subsume(r->argval, d) == 0)
	    r = r->narg;
	return(r != NULL);
	}
    else if (d->scratch == OR_FORM) {  /* return(c gen_subsumes one d_i */
	r = d->farg;
	while (r != NULL && gen_subsume(c, r->argval) == 0)
	    r = r->narg;
	return(r != NULL);
	}
    else  /* c and d are both literals */
	return(c->scratch2 == d->scratch2 && term_ident(c, d));

}  /* gen_subsume */

/*************
 *
 *    struct term *subsume_conj(c)
 *
 *************/

struct term *subsume_conj(c)
struct term *c;
{
    struct rel *r, *prev, *r1;
    struct term *t;

    /* given a conjunction, discard subsumed conjuncts */
    /* the result is equivalent */

    if (c->scratch != AND_FORM)
	return(c);
    else {
	r = c->farg;
	prev = NULL;  /* only to quiet lint */
	while (r != NULL) {
	    /* first do forward subsumption of part already processed */
	    r1 = c->farg;
	    while (r1 != r && gen_subsume(r1->argval, r->argval) == 0)
	        r1 = r1->narg;
	    if (r1 != r) {  /* delete r */
		/* note that prev cannot be NULL */
		prev->narg = r->narg;
		zap_term(r->argval);
		free_rel(r);
		r = prev;
		}
	    else {
		/* back subsumption on part already processed */
		r1 = c->farg;
		prev = NULL;
		while (r1 != r && gen_subsume(r->argval, r1->argval) == 0) {
		    prev = r1;
		    r1 = r1->narg;
		    }
		if (r1 != r) {  /* delete r1 */
		    if (prev == NULL)
			c->farg = r1->narg;
		    else
			prev->narg = r1->narg;
		    zap_term(r1->argval);
		    free_rel(r1);
		    }
		}
	    prev = r;
	    r = r->narg;
	    }
	if (c->farg != NULL && c->farg->narg == NULL) {
	    t = c->farg->argval;
	    free_rel(c->farg);
	    free_term(c);
	    return(t);
	    }
	else
	    return(c);
	}
}  /* subsume_conj */

/*************
 *
 *    struct term *subsume_disj(c)
 *
 *************/

struct term *subsume_disj(c)
struct term *c;
{
    struct rel *r, *prev, *r1;
    struct term *t;

    /* given a disjunction, discard subsuming (stronger) disjuncts */
    /* the result is equivalent */

    if (c->scratch != OR_FORM)
	return(c);
    else {
	r = c->farg;
	prev = NULL;  /* only to quiet lint */
	while (r != NULL) {
	    /* first do forward subsumption of part already processed */
	    r1 = c->farg;
	    while (r1 != r && gen_subsume(r->argval, r1->argval) == 0)
	        r1 = r1->narg;
	    if (r1 != r) {  /* delete r */
		/* note that prev cannot be NULL */
		prev->narg = r->narg;
		zap_term(r->argval);
		free_rel(r);
		r = prev;
		}
	    else {
		/* back subsumption on part already processed */
		r1 = c->farg;
		prev = NULL;
		while (r1 != r && gen_subsume(r1->argval, r->argval) == 0) {
		    prev = r1;
		    r1 = r1->narg;
		    }
		if (r1 != r) {  /* delete r1 */
		    if (prev == NULL)
			c->farg = r1->narg;
		    else
			prev->narg = r1->narg;
		    zap_term(r1->argval);
		    free_rel(r1);
		    }
		}
	    prev = r;
	    r = r->narg;
	    }
	if (c->farg != NULL && c->farg->narg == NULL) {
	    t = c->farg->argval;
	    free_rel(c->farg);
	    free_term(c);
	    return(t);
	    }
	else
	    return(c);
	}
}  /* subsume_disj */

/*************
 *
 *    conflict_tautology(c)
 *
 *    If c is an AND_FORM, reduce to empty disjunction (FALSE) if conflicting conjuncts occur.
 *    If c is an OR_FORM,  reduce to empty conjunction (TRUE)  if conflicting disjuncts occur.
 *
 *************/

void conflict_tautology(c)
struct term *c;
{
    struct rel *r, *r1;

    if (c->scratch != AND_FORM && c->scratch != OR_FORM)
	return;
    else {
	r = c->farg;
	while (r != NULL) {
	    r1 = r->narg;
	    while (r1 != NULL && (r->argval->scratch2 == r1->argval->scratch2 ||
				  term_ident(r->argval, r1->argval) == 0))
		r1 = r1->narg;
	    if (r1 != NULL) {  /* reduce to either FALSE or TRUE */
		r = c->farg;
		while (r != NULL) {
		    r1 = r;
		    r = r->narg;
		    zap_term(r1->argval);
		    free_rel(r1);
		    }
		c->farg = NULL;
		c->scratch = (c->scratch == OR_FORM ? AND_FORM : OR_FORM);
		return;
		}
	    else
		r = r->narg;
	    }
	}
}  /* conflict_tautology */

/*************
 *
 *   ts_and_fs(f)
 *
 *************/

void ts_and_fs(f)
struct term *f;
{
    struct rel *r, *r1, *prev;
    struct term *t1;
    int f_type;

    f_type = f->scratch;
    if (f_type != AND_FORM && f_type != OR_FORM)
	return;
    else {
	r = f->farg;
	prev = NULL;
	while (r != NULL) {
	    t1 = r->argval;
	    if ((t1->scratch == AND_FORM || t1->scratch == OR_FORM) && t1->farg == NULL) {
		if (f_type != t1->scratch) {
		    r = f->farg;
		    while (r != NULL) {
			r1 = r;
			r = r->narg;
			zap_term(r1->argval);
			free_rel(r1);
			}
		    f->farg = NULL;
		    f->scratch = (f_type == AND_FORM ? OR_FORM : AND_FORM);
		    return;
		    }
		else {
		    if (prev == NULL)
			f->farg = r->narg;
		    else
			prev->narg = r->narg;
		    r1 = r;
		    r = r->narg;
		    zap_term(r1->argval);
		    free_rel(r1);
		    }
		}
	    else {
		prev = r;
		r = r->narg;
		}
	    }
	}
}  /* ts_and_fs */
