#include "header.h"

/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */

#define CLAUSE_TAB_SIZE 1000

/* hash table for accessing clauses by ID */
static struct clause_ptr *Clause_tab[CLAUSE_TAB_SIZE];

/* # of clauses integrated; used for clause id */
static int Int_clause_count;

/*************
 *
 *    cl_integrate(c) -- integrate a clause
 *
 *    This routine integrates most subterms of the atoms. (Incomming clause must
 *    already have back pointers from literal to clause and atom to literal.)
 *
 *    The atoms are not shared, and arguments of positive equality literals
 *    are not shared.
 *
 *    A clause is integrated iff its ID is > 0.
 *
 *************/

void cl_integrate(c)
struct clause *c;
{
    struct literal *lit;
    struct term *atom;
    struct rel *r;

    if (c->id != 0) {
	fprintf(stdout, "WARNING, cl_integrate, called with integrated clause: ");
	print_clause(stdout, c);
	}
    else {
	c->id = ++Int_clause_count;
	cl_insert_tab(c);
	lit = c->first_lit;
	while (lit != NULL) {
	    atom = lit->atom;
	    if (atom->type == COMPLEX) {
		r = atom->farg;
		while (r != NULL) {
		    r->argval = integrate_term(r->argval);
		    r->argof = atom;
		    r->nocc = r->argval->occ.rel;
		    r->argval->occ.rel = r;
		    r = r->narg;
		    }
		}
	    lit = lit->next_lit;
	    }
	}

}  /* cl_integrate */

/*************
 *
 *    cl_del_int(c) -- deallocate an integrated clause.
 *
 *************/

void cl_del_int(c)
struct clause *c;
{
    struct literal *lit, *plit;
    struct rel *r, *r2, *pr;
    struct term *atom;
    struct int_ptr *ip1, *ip2;

    lit = c->first_lit;
    while (lit != NULL) {
	atom = lit->atom;
	if (atom->type == COMPLEX) {
	    r = atom->farg;
	    while (r != NULL) {
		r2 = r->argval->occ.rel;
		pr = NULL;
		while (r2 != NULL && r2 != r) {
		    pr = r2;
		    r2 = r2->nocc;
		    }
		if (r2 == NULL) {
		    output_stats(stdout, 4);
		    fprintf(stderr, "ABEND, cl_del_int, bad clause.\007\n");
		    fprintf(stdout, "ABEND, cl_del_int, bad clause: ");
		    print_clause(stdout, c);
		    exit(1);
		    }
		if (pr == NULL)
		    r->argval->occ.rel = r->nocc;
		else
		    pr->nocc = r->nocc;
		if (r->argval->occ.rel == NULL)
		    disintegrate_term(r->argval);
		pr = r;
		r = r->narg;
		free_rel(pr);
		}
	    }
	free_term(atom);
	plit = lit;
	lit = lit->next_lit;
	free_literal(plit);
	}
    ip1 = c->parents;
    while (ip1 != NULL) {
	ip2 = ip1;
	ip1 = ip1->next;
	free_int_ptr(ip2);
	}
    cl_delete_tab(c);
    /* If there is other memory associated with clause, free it here */
    free_clause(c);
}  /* cl_del_int */

/*************
 *
 *    cl_del_non(c) -- deallocate a nonintegrated clause.
 *
 *************/

void cl_del_non(c)
struct clause *c;
{
    struct literal *lit, *plit;
    struct int_ptr *ip1, *ip2;

    lit = c->first_lit;
    while (lit != NULL) {
	lit->atom->occ.lit = NULL;
	zap_term(lit->atom);
	plit = lit;
	lit = lit->next_lit;
	free_literal(plit);
	}
    ip1 = c->parents;
    while (ip1 != NULL) {
	ip2 = ip1;
	ip1 = ip1->next;
	free_int_ptr(ip2);
	}
    /* If there is other memory associated with clause, free it here */
    free_clause(c);
}  /* cl_del_non */

/*************
 *
 *    struct list *read_cl_list(fp, errors_ptr)
 *
 *    Read clauses until EOF or the term `end_of_list' is reached.
 *
 *************/

struct list *read_cl_list(fp, ep)
FILE *fp;
int *ep;
{
    struct list *lst;
    struct clause *cl, *pcl;
    int rc;

    *ep = 0;
    lst = get_list();
    pcl = NULL;
    cl = read_clause(fp, &rc);
    while (rc == 0) {  /* while errors */
	(*ep)++;
	cl = read_clause(fp, &rc);
	}
    while (cl != NULL) {
	if (pcl == NULL)
	    lst->first_cl = cl;
	else
	    pcl->next_cl = cl;
	cl->prev_cl = pcl;
	cl->container = lst;
	pcl = cl;
	cl = read_clause(fp, &rc);
	while (rc == 0) {  /* while errors */
	    (*ep)++;
	    cl = read_clause(fp, &rc);
	    }
	}
    lst->last_cl = pcl;
    return(lst);
}  /* read_cl_list */
 
/*************
 *
 *    struct clause *read_clause(fp, retcd_ptr)
 *
 *************/

struct clause *read_clause(fp, rcp)
FILE *fp;
int *rcp;
{
    char buf[MAX_BUF];
    int p, go, sign, rc;
    struct term *t;
    struct clause *cl;
    struct literal *lit, *plit;

    rc = read_buf(fp, buf);
    if (rc == 0) {  /* error */
	*rcp = 0;
	return(NULL);
	}
    else if (buf[0] == '\0') {  /* ok. EOF */
	*rcp = 1;
	return(NULL);
	}
    else {
	cl = get_clause();
	plit = NULL;  /* prev lit */
	p = 0;
	skip_white(buf, &p);
	go = 1;
	while (go) {
	    sign = (buf[p] != '-');
	    if (sign == 0)
		p++;
	    t = str_atom(buf, &p, &rc);
	    if (rc == 0)
		sign = (sign ? 0 : 1);
	    if (t == NULL) {
		cl_del_non(cl);
		*rcp = 0;
		return(NULL);
		}
	    else if (t->type == NAME && str_ident(sn_to_str(t->sym_num),"end_of_list"))
		if (cl->first_lit == NULL) {
		    cl_del_non(cl);
		    free_term(t);
		    *rcp = 1;
		    return(NULL);
		    }
		else {
		    fprintf(stdout, "ERROR, bad literal:\n");
		    print_error(stdout, buf, p);
		    cl_del_non(cl);
		    free_term(t);
		    *rcp = 0;
		    return(NULL);
		    }
	    if (t->type == NAME && var_name(sn_to_str(t->sym_num))) {
		fprintf(stdout, "ERROR, literal is variable:\n");
		print_error(stdout, buf, p);
		cl_del_non(cl);
		free_term(t);
		*rcp = 0;
		return(NULL);
		}
	    lit = get_literal();
	    lit->atom = t;
	    lit->sign = sign;
	    lit->container = cl;  /* literal point up to clause */
	    t->occ.lit = lit;  /* atom point up to literal */
	    mark_literal(lit);  /* atoms have varnum > 0 */
	    if (plit == NULL)
		cl->first_lit = lit;
	    else
		plit->next_lit = lit;
	    plit = lit;
	    skip_white(buf, &p);
	    if (buf[p] == '|') {
		p++;
		skip_white(buf, &p);
		}
	    else if (buf[p] == '.')
		go = 0;
	    else {
		fprintf(stdout, "ERROR, | or . expected in clause:\n");
		print_error(stdout, buf, p);
		cl_del_non(cl);
		*rcp = 0;
		return(NULL);
		}
	    }
	if (set_vars_cl(cl)) {
	    Stats[CL_INPUT]++;
	    *rcp = 1;
	    return(cl);
	    }
	else {
	    fprintf(stdout, "ERROR, too many variables in clause, max is %d:\n%s\n",
			       MAX_VARS, buf);
	    cl_del_non(cl);
	    *rcp = 0;
	    return(NULL);
	    }
	}
}  /* read_clause */

/*************
 *
 *    int set_vars_cl(cl) -- decide which terms are variables
 *
 *************/

int set_vars_cl(cl)
struct clause *cl;
{
    struct literal *lit;
    char *varnames[MAX_VARS];
    int i;

    for (i=0; i<MAX_VARS; i++)
	varnames[i] = NULL;
    lit = cl->first_lit;
    while (lit != NULL) {
	if (set_vars_term(lit->atom, varnames))
	    lit = lit->next_lit;
	else
	    return(0);
	}
    return(1);
}

/*************
 *
 *    print_clause(fp, clause)
 *
 *************/

void print_clause(fp, cl)
FILE *fp;
struct clause *cl;
{
    struct literal *lit;
    struct int_ptr *ip;

    fprintf(fp, "%d ", cl->id);
    fprintf(fp,"[");
    if (cl->parents != NULL) {
	ip = cl->parents;
	while (ip != NULL) {
	    switch (ip->i) {
		/* case BINARY_RES_RULE  : fprintf(fp, "binary"); break;
		case FACTOR_RULE      : fprintf(fp, "factor"); break;
		case UNIT_DEL_RULE    : fprintf(fp, "unit_del"); break; */
		default               : fprintf(fp, "%d", ip->i); break;
		}
	    ip = ip->next;
	    if (ip != NULL)
		fprintf(fp, ",");
	    }
	}
    fprintf(fp, "] ");
    lit = cl->first_lit;
    while (lit != NULL) {
	if (lit->sign == 0 ) /* && lit->atom->sym_num != Eq_sym_num */
	    fprintf(fp, "-");
	print_term(fp, lit->atom);
	lit = lit->next_lit;
	if (lit != NULL)
	    fprintf(fp, " | ");
	}
    fprintf(fp, ".\n");
}  /* print_clause */

void print_clause_lisp(fp, cl, sup)
FILE *fp;
struct clause *cl;
int sup;
{
    struct literal *lit;
    struct rel *r;

    fprintf(fp, "(def-clause '(");
    lit = cl->first_lit;
    while (lit != NULL) {

	fprintf(fp, "(");

	if (lit->sign == 0 )
	    fprintf(fp, "not ");
	/* print_term_lisp(fp, lit->atom); */

	if (lit->atom->varnum == ANSWER) fprintf(fp, "answer");
	else fprintf(fp, "%s", sn_to_str(lit->atom->sym_num));

	r = lit->atom->farg;

	while (r != NULL) {
		fprintf(fp, " ");
		print_term_lisp(fp, r->argval);
		r = r->narg;
		} 

	fprintf(fp, ")");

	lit = lit->next_lit;

	if (lit != NULL)
	    fprintf(fp, " ");
	}

    fprintf(fp, ") ");
    if (sup == 1)
	fprintf(fp, " :support-set 't");
    fprintf(fp, ")\n");

}  /* print_clause_lisp */

/*************
 *
 *    print_cl_list(fp, lst)
 *
 *************/

void print_cl_list(fp, lst)
FILE *fp;
struct list *lst;
{
    struct clause *cl;

    if (lst == NULL)
	fprintf(fp, "(list nil)\n");
    else {
	cl = lst->first_cl;
	while (cl != NULL) {
	    print_clause(fp, cl);
	    cl = cl->next_cl;
	    }
	fprintf(fp, "end_of_list.\n");
	}
}  /* print_cl_list */

void print_cl_list_lisp(fp, lst, sup)
FILE *fp;
struct list *lst;
int sup;
{
    struct clause *cl;

    if (lst == NULL)
	fprintf(fp, ";; (list nil)\n");
    else {
	cl = lst->first_cl;
	while (cl != NULL) {
	    print_clause_lisp(fp, cl, sup);
	    cl = cl->next_cl;
	    }
	fprintf(fp, ";; end_of_list.\n");
	}
}  /* print_cl_list_lisp */

/*************
 *
 *    cl_merge(cl) -- merge identical literals (keep leftmost occurrence)
 *
 *************/

void cl_merge(c)
struct clause *c;
{
    struct literal *l1, *l2, *l_prev;

    l1 = c->first_lit;
    while (l1 != NULL) {
	l2 = l1->next_lit;
	l_prev = l1;
	while (l2 != NULL)
	    if (l1->sign == l2->sign && term_ident(l1->atom, l2->atom)) {
		l_prev->next_lit = l2->next_lit;
		l2->atom->occ.lit = NULL;
		zap_term(l2->atom);
		free_literal(l2);
		l2 = l_prev->next_lit;
		}
	    else {
		l_prev = l2;
		l2 = l2->next_lit;
		}
	l1 = l1->next_lit;
	}
}  /* cl_merge */

/*************
 *
 *    append_cl(lst, cl)
 *
 *************/

void append_cl(l, c)
struct list *l;
struct clause *c;
{
    c->next_cl = NULL;
    if (l->first_cl == NULL) {
	l->first_cl = c;
	c->prev_cl = NULL;
	}
    else {
	l->last_cl->next_cl = c;
	c->prev_cl = l->last_cl;
	}
    l->last_cl = c;
    c->container = l;
}  /* append_cl */

/*************
 *
 *    struct clause *extract_first_cl(l)
 *
 *************/

struct clause *extract_first_cl(l)
struct list *l;
{
    struct clause *c;

    if (l->first_cl == NULL)
	return(NULL);
    else {
	c = l->first_cl;
	rem_from_list(c);
	return(c);
	}
}  /* extract_first_cl */

/*************
 *
 *    struct clause *extract_last_cl(l)
 *
 *************/

struct clause *extract_last_cl(l)
struct list *l;
{
    struct clause *c;

    if (l->last_cl == NULL)
	return(NULL);
    else {
	c = l->last_cl;
	rem_from_list(c);
	return(c);
	}
}  /* extract_last_cl */

/*************
 *
 *    rem_from_list(c)
 *
 *************/

void rem_from_list(c)
struct clause *c;
{
    struct clause *p, *n;

    p = c->prev_cl;
    n = c->next_cl;
    if (n == NULL)
	c->container->last_cl = p; 
    else
	n->prev_cl = p;
    if (p == NULL)
	c->container->first_cl = n;
    else
	p->next_cl = n;
    c->container = NULL;
    c->prev_cl = NULL;
    c->next_cl = NULL;
}  /* rem_from_lists */

/*************
 *
 *    insert_clause(clause, *clause_ptr)
 *
 *    If not already there, insert clause into list of clause pointers.
 *
 *************/

void insert_clause(c, cpp)
struct clause *c;
struct clause_ptr **cpp;
{
    struct clause_ptr *curr, *prev, *new;

    curr = *cpp;
    prev = NULL;
    while (curr != NULL && curr->c->id > c->id) {
	prev = curr;
	curr = curr->next;
	}
    if (curr == NULL || curr->c->id != c->id) {
	new = get_clause_ptr();
	new->c = c;
	new->next = curr;
	if (prev != NULL) 
	    prev->next = new;
	else
	    *cpp = new;
	}
}  /* insert_clause */

/*************
 *
 *    remove_var_syms(t)
 *
 *    Variable terms normally do not have sym_nums.  This
 *    routine removes any that are present.
 *
 *************/

void remove_var_syms(t)
struct term *t;
{
    struct rel *r;

    if (t->type == VARIABLE)
        t->sym_num = 0;
    else if (t->type == COMPLEX)
	for (r = t->farg; r != NULL; r = r->narg)
	    remove_var_syms(r->argval);
}  /* remove_var_syms */


/*************
 *
 *    cl_insert_tab(c)
 *
 *************/

void cl_insert_tab(c)
struct clause *c;
{
    struct clause_ptr *cp1, *cp2, *cp3;
    int hashval, id;

    id = c->id;
    hashval = id % CLAUSE_TAB_SIZE;
    cp1 = Clause_tab[hashval];

    /* keep the chains ordered--increasing id */

    cp2 = NULL;
    while (cp1 != NULL && cp1->c->id < id) {
	cp2 = cp1;
	cp1 = cp1->next;
	}
    if (cp1 != NULL && cp1->c->id == id) {
	output_stats(stdout, 4);
	fprintf(stderr, "ABEND, cl_insert_tab, clause already there.\007\n");
	fprintf(stdout, "ABEND, cl_insert_tab, clause already there: ");
	print_clause(stdout, c);
	exit(1);
	}
    else {
	cp3 = get_clause_ptr();
	cp3->c = c;
	cp3->next = cp1;
	if (cp2 == NULL) 
	    Clause_tab[hashval] = cp3;
	else
	    cp2->next = cp3;
	}
}  /* cl_insert_tab */

/*************
 *
 *    cl_delete_tab(c)
 *
 *************/

void cl_delete_tab(c)
struct clause *c;
{
    struct clause_ptr *cp1, *cp2;
    int hashval, id;

    id = c->id;
    hashval = id % CLAUSE_TAB_SIZE;
    cp1 = Clause_tab[hashval];

    /* chains are ordered--increasing id */

    cp2 = NULL;
    while (cp1 != NULL && cp1->c->id < id) {
	cp2 = cp1;
	cp1 = cp1->next;
	}
    if (cp1 == NULL || cp1->c->id != id) {
	output_stats(stdout, 4);
	fprintf(stderr, "ABEND, cl_delete_tab, clause not found.\007\n");
	fprintf(stdout, "ABEND, cl_delete_tab, clause not found: ");
	print_clause(stdout, c);
	exit(1);
	}
    else {
	if (cp2 == NULL) 
	    Clause_tab[hashval] = cp1->next;
	else
	    cp2->next = cp1->next;
	free_clause_ptr(cp1);
	}
}  /* cl_delete_tab */

/*************
 *
 *    all_cont_cl(t, cpp) - insert containing clauses of t into *cpp
 *
 *************/

void all_cont_cl(t, cpp)
struct term *t;
struct clause_ptr **cpp;
{
    struct rel *r;
    struct clause *c;

    if (t->type != VARIABLE && t->varnum != 0) {  /* atom */
	c = t->occ.lit->container;
	insert_clause(c, cpp);
	}
    else {  /* term */
	r = t->occ.rel;
	while (r != NULL) {
	    all_cont_cl(r->argof, cpp);
	    r = r->nocc;
	    }
	}
}  /* all_cont_cl */

/*************
 *
 *    mark_literal(lit)
 *
 *    Atoms have varnum > 0.  This routine inserts the appropriate code.
 *
 *************/

void mark_literal(lit)
struct literal *lit;
{
    char *name;
    struct term *a;

    a = lit->atom;

    /* atoms have varnum > 0 */
    name = sn_to_str(a->sym_num);

    if (initial_str("$ANS", name) || initial_str("$Ans", name) ||
				     initial_str("$ans", name))
	a->varnum = ANSWER;  /* answer literal */
    else
	a->varnum = NORM_ATOM;  /* normal atom */

}  /* mark_literal */

/*************
 *
 *    clear_var_names(t) -- set sym_num field of all variables to NULL
 *
 *************/

void clear_var_names(t)
     struct term *t;
{
    struct rel *r;
    
    if (t->type == VARIABLE)
	t->sym_num = 0;
    else {
	for (r = t->farg; r != NULL; r = r->narg)
	    clear_var_names(r->argval);
	}
}  /* clear_var_names */

