;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Subgoals.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;
;;;	Provides	expand (subgoal), exhausted-p (subgoal),
;;;			propogate (subgoal)

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defmethod expand ((subgoal dtp-subgoal))
  (with-slots (conjuncts-to-propogate-to answers
	       ancestor-subgoals inferences used-inferences )
      subgoal

    ;; Model elimination
    (when (eq ancestor-subgoals :uninitialized)
      (setq ancestor-subgoals (subgoal-ancestors-of subgoal)) )
    (loop
	for ancestor = (pop ancestor-subgoals)
	while ancestor
	for answer = (model-elimination subgoal ancestor)
	when answer
	do (when (find :proofs *trace*)
	     (indent-line)
	     (format *debug-io* "Found ~A by model elimination~%" answer)
	     (indent-line)
	     (format *debug-io* " with ~A~%" ancestor) )
	   (propogate answer subgoal)
	   (return-from expand) )	      

    ;; Resolution
    (when (eq inferences :uninitialized)
      (setq inferences (norder-conjunctions (compute-inference subgoal))) )
    (if inferences
      (let ((conjunction (first inferences)))
	(with-slots (list binding-list label ae-binding-list) conjunction
	  (cond
	   ((null list)
	    (pop inferences)
	    (unless inferences
	      (setf (proof-subgoal-agenda *proof*)
		(remove subgoal (proof-subgoal-agenda *proof*)) ))
	    (add-to-end conjunction used-inferences)
	    (propogate
	     (make-answer
	      :binding-list binding-list :label label
	      :ae-binding-lists (when ae-binding-list (list ae-binding-list)) )
	     subgoal ))
	   ((exhausted-p conjunction)
	    (pop inferences)
	    (unless inferences
	      (setf (proof-subgoal-agenda *proof*)
		(remove subgoal (proof-subgoal-agenda *proof*)) ))
	    (add-to-end conjunction used-inferences) )
	   (t
	    (expand conjunction) ))	; cond
	  ))
      (dolist (conjunct conjuncts-to-propogate-to)
	(propogate :not-an-answer conjunct) ))
    ))

(defmethod expand :around ((subgoal dtp-subgoal))
  (when (find :proofs *trace*)
    (indent-line)
    (format *debug-io* "Expanding subgoal ~A~%"
	    (slot-value subgoal 'literal) ))
  (incf *depth*)
  (prog1
      (call-next-method)
    (decf *depth*) ))

;;;----------------------------------------------------------------------------

(defmethod exhausted-p ((subgoal dtp-subgoal))
  (null (slot-value subgoal 'inferences)) )

;;;----------------------------------------------------------------------------

(defmethod propogate (answer (subgoal dtp-subgoal))
  (unless (typep answer 'me-answer)	; Don't simplify model elimination
    (setq answer (nsimplify-binding-list (copy-answer answer) subgoal)) )
  (with-slots (answers conjuncts-to-propogate-to) subgoal
    (unless (find answer answers :test #'answer-equal-p)
      (add-to-end answer answers)
      (dolist (conjunct conjuncts-to-propogate-to)
	(propogate answer conjunct) ))))

(defmethod propogate :around (answer (subgoal dtp-subgoal))
  "For proof tracing"
  (when (find :proofs *trace*)
    (indent-line)
    (format *debug-io* "Propogating ~S to subgoal ~A~%"
	    answer (slot-value subgoal 'literal) ))
  (incf *depth*)
  (prog1
    (call-next-method)
    (decf *depth*) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Private

;;;----------------------------------------------------------------------------

(defun compute-inference (subgoal)
  "Returns list of conjunctions"
  (declare (type dtp-subgoal subgoal))
  (loop
      with sg-literal = (slot-value subgoal 'literal)
      with node-list-1 = (proof-goal-nodes *proof*)
      with node-list-2 =
	(active-theory-contents
	 :index-on (literal-relation (slot-value subgoal 'literal)) )
      for resolving-with-goal-p = node-list-1
      for kb-node =
	(if node-list-1 (pop node-list-1) (pop node-list-2))
      while kb-node
      appending
	(loop
	    with c-bl =
	      (multiple-value-list
		  (clause-rename-all-variables (kb-node-clause kb-node)) )
	    with clause = (first c-bl)
	    with tbl = (second c-bl)
	    for literal in (clause-literals clause)
	    for mgu = (literal-mgu literal sg-literal)
	    when mgu
	    collect
	      (resolve mgu sg-literal literal clause subgoal
		       (kb-node-id kb-node)
		       (when resolving-with-goal-p tbl) ))
	))

;;;----------------------------------------------------------------------------

(defun resolve (mgu sg-lit kb-lit clause parent kb-parent ans-ext-bl)
  "Returns conjunction"
  (declare (type binding-list mgu))
  (declare (type literal-node sg-lit kb-lit))
  (declare (type clause-node clause))
  (declare (type dtp-subgoal parent))
  (declare (type symbol kb-parent))
  (declare (type binding-list ans-ext-bl))
  (setq ans-ext-bl (remove '(t . t) ans-ext-bl :test #'equal))
  (let ((sg-vars (find-vars (literal-terms sg-lit)))
	conjunction )
    (declare (type dtp-conjunction conjunction))
    (setq clause (copy-clause-node clause))
    (setf (clause-literals clause) (remove kb-lit (clause-literals clause)))
    (setq clause (clause-plug clause mgu))
    (nclause-flip-negations clause)
    (setq conjunction
      (make-instance 'dtp-conjunction
	:parent parent
	:origin kb-parent
	:binding-list
	(remove-if-not
	 #'(lambda (binding)
	     (declare (type binding binding))
	     (find (binding-variable binding) sg-vars) )
	 mgu )
	:label (clause-label clause)
	:ae-binding-list (plug ans-ext-bl mgu) ))
    (setf (slot-value conjunction 'list)
      (mapcar
       #'(lambda (lit)
	   (declare (type literal-node lit))
	   (make-instance 'dtp-conjunct :literal lit :parent conjunction) )
       (clause-literals clause) ))
    conjunction ))

;;;----------------------------------------------------------------------------

(defmethod nsimplify-binding-list (answer (subgoal dtp-subgoal))
  "Remove any bindings for variables not in the subgoal literal"
  (let ((good-vars (literal-vars-in (slot-value subgoal 'literal))))
    (setf (answer-binding-list answer)
          (remove-if-not
           #'(lambda (binding)
	       (declare (type binding binding))
	       (find (binding-variable binding) good-vars) )
           (answer-binding-list answer) ))
    answer ))

;;;----------------------------------------------------------------------------

(defun subgoal-ancestors-of (subgoal)
  "List of all subgoals above SUBGOAL in the (possibly cyclic) proof graph"
  (declare (type (or dtp-subgoal null) subgoal))
  (when subgoal (subgoal-ancestors-of-internal subgoal nil)) )

(defun subgoal-ancestors-of-internal (subgoal children)
  (declare (type dtp-subgoal subgoal))
  (declare (type list children))
  (unless (find subgoal children)
    (loop
	with parents = (subgoal-parents-of subgoal)
	for parent in parents
	for parent-ancestors =
	  (subgoal-ancestors-of-internal parent (cons subgoal children))
	append parent-ancestors into ancestors
	finally
	  (return
	    (remove-duplicates (append parents ancestors) :from-end t) ))
    ))

;;;----------------------------------------------------------------------------

(defun subgoal-parents-of (subgoal)
  "List of all parents of SUBGOAL in the proof graph"
  (with-slots (conjuncts-to-propogate-to) subgoal
    (remove
     nil
     (mapcar
      #'(lambda (conjunction)
	  (declare (type dtp-conjunction conjunction))
	  (slot-value conjunction 'parent-subgoal) )
      (mapcar
       #'(lambda (conjunct)
	   (declare (type dtp-conjunct conjunct))
	   (slot-value conjunct 'parent-conjunction) )
       conjuncts-to-propogate-to )))
    ))

;;;----------------------------------------------------------------------------

(defun model-elimination (subgoal-1 subgoal-2)
  (let ((lit1 (slot-value subgoal-1 'literal))
	(lit2 (slot-value subgoal-2 'literal))
	mgu )
    (setq mgu (literal-negated-pair-p lit1 lit2))
    (when mgu (make-me-answer :binding-list mgu)) ))

;;;----------------------------------------------------------------------------
