;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Output.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;
;;;	Provides	show-proof-graph, show (conjunction), show (subgoal)

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(show-proof-graph show-contents) ))

;;;----------------------------------------------------------------------------

(defgeneric show (object &key fringe &allow-other-keys))

;;;----------------------------------------------------------------------------

(defun show-proof-graph (&optional (*proof* *proof*))
  (format t "~%")
  (if (or (proof-subgoal-agenda *proof*) (proof-query-conjunctions *proof*))
      (format t "Proof tree [in progress] ")
    (format t "Complete proof tree ") )
  (format t "using theory ~A for query~%" (proof-theory *proof*))
  (format t "     ~A~%" (proof-query *proof*))
  (format t "Found ~R answer~:P" (length (proof-answers *proof*)))
  (when (or (proof-subgoal-agenda *proof*) (proof-query-conjunctions *proof*))
    (format t " so far") )
  (format t ":~%")
  (dolist (answer (proof-answers *proof*))
    (format t "     ~A"
	    (plug (proof-query *proof*) (answer-binding-list answer)) )
    (when (answer-label answer)
      (format t " with label ~A" (label-value (answer-label answer))) )
    (format t "~%") )
  (format t "~%")

  (let ((*proof-line-count* 0)
	(*depth* 0)
	(*subgoal-map* nil) )
    (dolist (conjunction (proof-used-conjunctions *proof*))
      (show conjunction :fringe nil) )
    (dolist (conjunction (proof-query-conjunctions *proof*))
      (show conjunction :fringe t) )
    (dolist (conjunction (proof-forward-inference-conjunctions *proof*))
      (show conjunction :fringe nil) )
    )

  (format t "~%")
  (values) )

;;;----------------------------------------------------------------------------

(defmethod show ((conjunction dtp-conjunction) &key (fringe nil) (recurse t))
  (with-slots (list binding-list label) conjunction
    (indent-line :number t)
    (format t "Conj: ")
    (cond
     ((null list)
      (format t "True") )
     (t
      (print-literal-node (slot-value (first list) 'literal))
      (dolist (conjunct (rest list))
	(format t " and ")
	(print-literal-node (slot-value conjunct 'literal)) )))
    (when binding-list
      (format t " with")
      (print-binding-list binding-list) )
    (when label (format t " with label ~A" (label-value label)))
    (when (and fringe
	       (or (null list)
		   (and list
			(eq (slot-value (first list) 'subgoal)
			    :uninitialized ))))
      (format t "~50T [Fringe]") )
    (format t "~%")
    (when recurse
      (dolist (conjunct list)
	(with-slots (subgoal used-subgoals) conjunct
	  (incf *depth*)
	  (dolist (m-subgoal used-subgoals)
	    (show m-subgoal :fringe nil) )
	  (unless (eq subgoal :uninitialized)
	    (show subgoal :fringe t) )
	  (decf *depth*) )))
    ))

;;;----------------------------------------------------------------------------

(defmethod show ((subgoal dtp-subgoal) &key (fringe nil) (tbl nil) (recurse t))
  (indent-line :number t)
  (format t "Subg: ")
  (if (assoc subgoal *subgoal-map*)
      (progn
	(format t "[#~3,'0D" (cdr (assoc subgoal *subgoal-map*)))
	(when tbl
	  (format t " with")
	  (print-binding-list tbl) )
	(format t "]~%")
	(return-from show) )
    (push (cons subgoal *proof-line-count*) *subgoal-map*) )
  (with-slots (literal inferences used-inferences) subgoal
    (print-literal-node literal)
    (when (and fringe (eq inferences :uninitialized))
      (format t "~50T [Fringe]") )
    (format t "~%")
    (when recurse
      (incf *depth*)
      (dolist (conjunction used-inferences)
	(show conjunction :fringe nil) )
      (unless (eq inferences :uninitialized)
	(dolist (conjunction inferences)
	  (show conjunction :fringe t) ))
      (decf *depth*) )))

;;;----------------------------------------------------------------------------

(defun indent-line (&key (number nil))
  (when number
    (format *debug-io* "~3,'0D  " (incf *proof-line-count*)) )
  (dotimes (i *depth*)
    (format *debug-io* "  ") ))

;;;----------------------------------------------------------------------------

(defun show-contents (&optional (theory *default-theory*))
  (format t "~&")
  (dolist (node (theory-contents theory))
    (format t "~A~15T" (kb-node-id node))
    (print-clause-node (kb-node-clause node) :as-rule t)
    (format t "~%") )
  (values) )

;;;----------------------------------------------------------------------------

(defun print-binding-list (binding-list &key (s t))
  (dolist (binding-pair binding-list)
    (unless (eq (car binding-pair) 't)
      (format s " ~A->~A" (car binding-pair) (cdr binding-pair)) )))

;;;----------------------------------------------------------------------------
