;;; -*- Mode: LISP; Syntax: Common-lisp; Package: SME; Base: 10; Fonts: CPTFONT,TR12I -*-

;;;; Structure-Mapping Engine  (module windowing.lisp: machine dependent I/O routines, flavors, etc.)

;;; Copyright (C) 1986,  Brian C. Falkenhainer, Unversity of Illinois
;;; All rights reserved

;;; Fancy windowing interface routines

(in-package 'sme)
(shadow 'sme::class-name)
(shadow 'sme::defclass)
(shadow 'sme::class)
(shadow 'sme::sme-print)
(shadow 'sme::sme-terpri)
(use-package 'clos)

(defvar *RUNNING* nil)
(defvar *lisp-active* nil)
(defvar *std-o* *standard-output*)
;;; *********************************************************************************

(defvar *sme-frame* nil "holds the main frame window flavor")
(defvar *graphics-pane* nil "holds the graphics pane")
(defvar *scroll-pane* nil "holds the scroll pane")
(defvar *spare-scroll-pane* nil "holds the temporary, scratch scroll pane")
(defvar *lisp-pane* nil "holds the sme lisp listener")
(defvar *sme-title-pane* nil "holds the sme lisp listener")

(defvar *sme-graphics-output* nil "pane or stream where the graphics output should go")

;;(defvar *default-font* fonts:tvfont "default font for displaying and describing dgroups")
(defvar *default-font* nil)
(defvar *scroll-dump-format* :lpr)
(defvar *scroll-dump-comments* (format nil "~60,,,' A" "") "comments for scroll printout")

;;; *********************************************************************************
;;; make class for scrolling windows

(lucid-common-lisp:defclass scrolwin (clim::clx-window)
  ((itms :initarg :itm-lst
         :accessor item-list)))

;;; Accessor functions
(defmethod scroller-app ((win scrolwin) itm-list)
  (dolist (itm (reverse itm-list))
          (setf (item-list win) (cons (list itm) (item-list win))))
  (print-list win))

(defmethod scroller-set ((win scrolwin) itm-list)
  (setf (item-list win) itm-lst)
  (print-list win))

(defmethod scroller-top ((win scrolwin))
  (print-list win))

(defmethod scroller-clear ((win scrolwin))
  (setf (item-list win) nil)
;;  (clim:clear-output-record (clim:output-recording-stream-output-record win))
  (clim:window-clear win)
  (print-list win))

(defmethod print-list ((win scrolwin))
  (dolist (itm (item-list win))
          (if (listp itm) (clim:format win "~A~%" (car itm))
            (clim:format win "~A~%" itm))))

;;; *********************************************************************************

;;; Given a string with potential line-feeds in it, print out each line
;;;   to *sme-output-stream*. If *sme-output-stream* is a scroll-window,
;;;   it will append the items to the scroll items list. If it is not a
;;;   scroll window, it will merely use format.
;;;
;;;     A new line is always printed at the end of the string, whether
;;;     or not one was actually there....
;;;
(defun sme-print (string)
  (with-input-from-string (stream string)
                          (do ((line (read-line stream nil) (read-line stream nil)))
                              ((null line))
                              (if (typep *sme-output-stream* 'scrolwin)
                                  (scroller-app *sme-output-stream* (list line))
                                (clim:format *sme-output-stream* "~A~%" line)))))

(defun sme-terpri (&optional (n 1))
  (dotimes (i n) (sme-print (string #\newline))))

;;; *********************************************************************************

;;; Overall frame
;;;

(clim:define-application-frame sme () () ;; ((object-list :initform nil))
  (:panes 
    ((title :title
	    :display-string "SME") 
     (menu-pane :command-menu)
     (graphics-pane :application)
;;                    :display-function `(graphit))
     (scroll-pane :application
                  :window-class `scrolwin
                  :itm-lst '(("scroll window")))
;;                  :display-function `(scrol-disp))
     (spare-scroll-pane :application
                        :window-class `scrolwin
                        :itm-list '(("spare window")))
;;                        :display-function `(scrol-disp))
     (lisp-pane :application
               :display-function `(ldisp))
     (documentation :pointer-documentation)))
  (:layout ((scroll-config (:column :rest
                                    (menu-pane :compute)
                                    (scroll-pane 0.60)
                                    (lisp-pane :rest)
                                    (documentation :compute)))
            (double-scroll-config (:column :rest
                                           (menu-pane :compute)
                                           (:row 0.50
                                                 (scroll-pane :rest)
                                                 (spare-scroll-pane :rest))
                                           (lisp-pane :rest)
                                           (documentation :compute)))
            (graphics-config (:column :rest
                                      (menu-pane :compute)
                                      (graphics-pane 0.60)
                                      (lisp-pane :rest)
                                      (documentation :compute)))
            (large-graphics-config (:column :rest
                                            (menu-pane :compute)
                                            (graphics-pane 0.70)
                                            (lisp-pane :rest)
                                            (documentation :compute)))
            (split-config (:column :rest
                                   (menu-pane :compute)
                                   (:row 0.50
                                         (scroll-pane :rest)
                                         (graphics-pane :rest))
                                   (lisp-pane :rest)
                                   (documentation :compute))))))
  
;;; **********************************************************************
;;;; Display functions
(defmethod graphit ((application sme) stream) ()
  (clim:format stream "grapher"))

(defmethod scrol-disp ((application sme) stream) ()
  (clim:format stream "~A" (item-list stream)))

(defmethod ldisp ((application sme) stream) ()
   (clim:format stream "> "))

;;; *********************************************************************************

;;;; Top line menu

(define-sme-command (com-match :menu "Match" 
                               :name "Try to analogically match two chosen description-groups") ()
  (menu-match))

(define-sme-command (com-describe :menu "Describe DGroup"
                                  :name "Choose a description group for textual display") ()
  (menu-describe-dgroup))

(define-sme-command (com-utilities :menu "Utilities"
                                   :name  "Miscellaneous system utilities") ()
  (menu-utilities))

(define-sme-command (com-generalize :menu "Generalize" 
                                    :name "Form generalizations of the current analogy") ()
  (if (eq (clim:frame-current-layout *sme-frame*) 'graphics-config)
      (select-scroll))
  (let ((std-out *standard-output*))
    (setf *standard-output* *std-o*)
    (generalize (car (best-gmaps)) T)
    (setf *standard-ouput* std-out)))

(define-sme-command (com-display :menu "Display DGroup" 
                                 :name "Choose a description group for graphical display") ()
  (menu-display-dgroup))

(define-sme-command (com-load-dgroup :menu "Load Dgroup" 
                                     :name "Load a Description Group definition file") ()
  (get-dgroup))

(define-sme-command (com-select-windowing :menu "Alter Windowing"
                                          :name  "Choose a graphics and scroll window configuration") ()
  (select-windowing-configuration))

(define-sme-command (com-display-pairs :menu "Display DGroup Pairs" 
                                       :name "Graphically display two chosen description-groups") ()
  (menu-display-pairs))

(define-sme-command  (com-lisp :menu "Use lisp" :name "Activate Lisp listener window") ()
   (cond (*lisp-active* (setf *lisp-active* nil))
         (t (setf *lisp-active* t)
            (do-listener))))

(define-sme-command (com-bye :menu "Quit" :name "Quit") ()
  (throw 'exit-me nil))

;;; **************************************************************
;;;; SME System Utilities

;;; takes in list of vals (a b) and sets variable named a to b for each
(defmacro dochg (lst)
  `(progn ,@(mapcar #'(lambda (x) `(setf ,(first x) ,(second x))) (eval lst))))

(defSME-Utility "Inspect Evidence" (match-evidence-inspector))

(defSME-Utility "Dump Scroll Window"
		(if *windowing-active?*
		    (dump-scroll (dump-scroll-menu))))
(defSME-Utility "Clear Scroll Window"
  (if (menus:yes-no-dial "Are you sure you want to clear it?")
      (clear-scroll)))
(defSME-Utility "Alter Windowing" (select-windowing-configuration))
(defSME-Utility "Change Parameters"  (change-parms))
(defSME-Utility	"Load a Rules File"  (get-rules))
(defSME-Utility "Load a DGroup File" (get-dgroup))


;;; Bring up the System Utilities menu
;;;
(defun menu-utilities ()
  (eval (choose-facility *system-utilities-menu* "SME System Menu")))


;;; Load a file chosen from the list of provided files which matched the 
;;; template *sme-pathname*>file-pattern
;;;
(defun menu-get-file (directory file-pattern header &optional bin-file?  &aux file)
  (setq file (menus:submen (mapcar #'(lambda (x) (list x 'value x)) 
                                   (directory (merge-pathnames file-pattern directory)))
                           header))
  (let ((std-out *standard-output*))
    (setf *standard-output* *std-o*)
    (if (not (menus:aborted file))  
        (if bin-file?
;;;            #+IMACH (load (merge-pathnames ".ibin" file))
;;;            #-IMACH (load (merge-pathnames ".bin" file))
            (load file) (load file)))
    (setf *standard-output* std-out)))

(defun get-dgroup ()
  (menu-get-file *sme-dgroup-pathname* "*.dgroup" "Select a DGroup File to Load"))

(defun get-rules ()
  (menu-get-file *sme-rules-pathname* "*.rules" "Select a Rules File to Load" t))


;;; Change SME parameters
;;;

(defun change-parms ()
  (let ((vals (menus:domenun "SME System Parameters"
                             (mapcar #'(lambda (x) 
                                         (list (eval (first x)) (first x) 
                                               (second x) 
                                               (case (third x)
                                                     (:assoc 
                                                      (cons 'clim:member-alist (list
                                                        (mapcar #'(lambda (y) (list (car y) :value
                                                                                    (cdr y)))
                                                                (fourth x)))))
                                                     (:boolean
                                                      '(clim:member-alist (("yes" :value t) 
                                                                           ("no" :value nil))))
                                                     (:integer 'integer)
                                                     (:string 'string)
                                                     (:character 'character)
                                                     (t 'string))))
                                     *parameter-menu-options*)
                             )))
    (declare (special vals))
    t (dochg vals)))

;;; Compare two apparently identical Gmaps
;;;
(defun menu-compare-gmaps (&aux gmap1 gmap2)
  (when *gmaps*
        (let* ((maps (menus:domenun "Select Gmaps to compare"
                                    '((0 :gmap1 "Number of the first Gmap" integer)
                                      (1 :gmap2 "Number of the second Gmap" integer))))
               (gmap1 (cadr maps))
               (gmap2 (caddr maps)))
          (compare-gmaps (nth (1- gmap1) *gmaps*) (nth (1- gmap2) *gmaps*) T))))


;;; Dump the SME scroll window to a file.  This is just the menu to get it going.  
;;; See dump-scroll farther down in this file.
;;;
(defun dump-scroll-menu ()
  (let ((file "/u/siegle/aaa.out"))
    (declare (special file))
    (menus:domenu ((*scroll-dump-format* 
                    '(clim:member-alist ("Imprint and Line Printer" :value :lpr) ("Troff" :value :troff))
                    "Printer Format")
                   (file clim:pathname "output-file")
                   (*scroll-dump-comments* 'string "Comment String (as a FORMAT control string)"))
                  "Select Options for Scroll Dump")
    file))

;;; **************************************************************
;;;; SME matching operations


;;; These are intended to be commands relevant to creating and analyzing analogical mappings.
;;;  They will typically be the same commands that appear in the command pane.
;;;

(defSME-Operation "Display Two Dgroups" (menu-display-pairs))
(defSME-Operation "Display A Dgroup" (menu-display-dgroup))
(defSME-Operation "Describe A Dgroup" (menu-describe-dgroup))
(defSME-Operation "Compare Gmaps" (menu-compare-gmaps))
(defSME-Operation "Generalize" (progn (if (member (clim:frame-current-layout *sme-frame*)
						  '(graphics-config large-graphics-config))
					  (select-scroll))
                                      (let ((std-out *standard-output*))
                                        (setf *standard-output* *std-o*)
                                        (generalize (car (best-gmaps)) T)
                                        (setf *standard-output* std-out))))
(defSME-Operation "Match" (menu-match))


;;; Bring up the System Utilities menu
;;;
(defun menu-matching-operations ()
  (choose-facility *sme-operations-menu* "SME Operations Menu"))


;;; A general menu-choose facility which reuses a single flavor instance
;;;

(defun choose-facility (item-list &optional (label "Choose Item") (column-length 15))
  (let* ((items (mapcar #'(lambda (x) (if (listp x) (list (first x) :value (third x))
                                        (list x :value x))) item-list))
         (val (menus:submen items label)))
    (unless (menus:aborted val) val)))

;;; Give the user a pop-up menu of the known dgroups to choose from
;;;     Return a dgroup name or nil.
;;;
(defun DGroup-choose (&optional (label "Choose DGroup"))
  (choose-facility *description-groups* label 15))



;;; Provide the user with the list of currently loaded description-groups and graphically display the one choosen.
;;;
(defun menu-display-dgroup (&aux dgroup)
  (setq dgroup (fetch-dgroup (DGroup-choose)))
  (if dgroup  (display-dgroup dgroup)))

;;; Query the user for two description groups and graphically display them.
;;;
(defun menu-display-pairs (&aux dgroup1 dgroup2)
  (let ((l-choose (DGroup-choose "Choose Left DGroup")))
    (unless (or (null l-choose) (menus:aborted l-choose))
            (setq dgroup1 (fetch-dgroup l-choose)))
    (if dgroup1 (let ((r-choose (DGroup-choose "Choose Right DGroup")))
                  (unless (or (null r-choose) (menus:aborted r-choose))
                          (setq dgroup2 (fetch-dgroup r-choose)))))
    (if (and dgroup1 dgroup2) (display-dgroup dgroup1 dgroup2))))


;;; Provide the user with the list of currently loaded description-groups and textually describe the one choosen.
;;;
(defun menu-describe-dgroup (&optional dgroup)
  (labels ((terse? () (not (menus:submen-na '(("Terse Format (default)" :value nil) 
                                              ("Full Format" :value T))
                                            "Dgroup description format"))))
          (unless dgroup (if (setq dgroup (DGroup-choose))
                             (setq dgroup (fetch-dgroup dgroup))))
          (when dgroup
                (if  (and *windowing-active?*
                          (member (clim:frame-current-layout *sme-frame*)
                                  '(graphics-config large-graphics-config) :test #'eq))
                    (select-scroll))
                (sme-print (string #\Page))
                (describe-dgroup dgroup (terse?)))))


;;; In scroll window, show the result of trying to analogically match chosen base and target description-groups.
;;;
(defun menu-match (&aux base-name target-name)
  (setq base-name (DGroup-choose "Choose Base"))
  (if base-name (setq target-name (DGroup-choose "Choose Target")))
  (when (and base-name target-name)
        (if (eq (clim:frame-current-layout *sme-frame*) 'graphics-config)
            (select-scroll))
        (sme-print (string #\Page))
        (match base-name target-name T)
        ))


;;; **************************************************************
;;;; Inspecting the evidence links for a match

;;; Allow the user to query the source of evidence for Match Hypotheses and GMaps
;;;
(defun match-evidence-inspector ()
  (let ((std-out *standard-output*))
    (setf *standard-output* *std-o*)
    (unless (and *base* *target*)
            (format t "~%There is no analogy to inspect...")
            (return-from match-evidence-inspector t))
    (let* ((vals (menus:domenun "Evidence Inspector" 
                                '(((m g) :mode-flags "Types of Things to Examine" 
                                     (clim:subset-alist (("Match Hypotheses" :value m) 
                                                         ("GMaps" :value g))))
                                  (all :mh-base "Match Hypothesis Base Item (name or all)" 
                                       clim:expression)
                                  (all :mh-target "Match Hypothesis Target Item (name or all)"
                                       clim:expression)
                                  (999 :gmap-number "Gmap (id number or all (enter 999))" integer))))
           (mode-flags (cadr (assoc :mode-flags vals)))
           (mh-base (cadr (assoc :mh-base vals)))
           (mh-target (cadr (assoc :mh-target vals)))
           (gmap-number (if (eq 999 (cadr (assoc :gmap-number vals))) 
                            'all (cadr (assoc :gmap-number vals))))
           )
      (when mode-flags
            (if  (eq (clim:frame-current-layout *sme-frame*) 'graphics-config)  (select-scroll))
            (sme-print (with-output-to-string (stream)
                                              (format stream "~A~%~5TEvidence for Match from ~A to ~A."
                                                      #\Page (dgroup-name *base*) (dgroup-name *target*))))
            (when (member 'm mode-flags)
                  (sme-print 
                   (with-output-to-string 
                    (stream)
                    (cond ((eq mh-base 'all)
                           (if (eq mh-target 'all)
                               (dolist (mh *match-hypotheses*) (why-node (mh-bms-node mh) stream))
                             (dolist (pat (fetch
                                           `(mh ?b ,(fetch-expression mh-target *target* nil))))
                                     (why pat stream))))
                          ((eq mh-target 'all)
                           (dolist (pat (fetch `(mh ,(fetch-expression mh-base *base* nil) ?t)))
                                   (why pat stream)))
                          ((why (fetch `(mh ,(fetch-expression mh-base *base* nil)
                                            ,(fetch-expression mh-target *target* nil)))
                                stream))))))
            (when (member 'g mode-flags)
                  (sme-print 
                   (with-output-to-string 
                    (stream)
                    (cond ((numberp gmap-number)
                           (let ((gmap (find gmap-number *gmaps* :test #'= :key #'gm-id)))
                             (if gmap (why-node (gm-bms-node gmap) stream)
                               (format t "~%There is no GMap with ~A as an id number"
                                       gmap-number))))
                          ((dolist (gmap *gmaps*) (why-node (gm-bms-node gmap) stream)))))))
            (sme-terpri)))
    (setf *standard-output* std-out)))
  
  
;;; **************************************************************
;;;; Scroll pane printout routines

(defun clear-scroll (&optional spare-scroll-window?)
  (if spare-scroll-window?
      (scroller-clear *spare-scroll-pane*)
    (scroller-clear *scroll-pane*)))
	

;;; Dump the contents of the scroll window to a file
;;;
(defun dump-scroll (file-name &aux (first-line? t))
  (with-open-file (fn file-name :direction :output)
    (multiple-value-bind (sec min hour date month year) (get-decoded-time)
      (cond ((eq *scroll-dump-format* :troff)
	     (format fn ".he '\\fBSME Output\\fP'%'\\fB~2,'0D/~2,'0D/~2D  ~2,'0D:~2,'0D:~2,'0D'\\fP"
		     month date (mod year 100) hour min sec)
	     (format fn "~%.nf~%.ft TA~%.ss 22~%.po 0.5i~%.ll 6.9i~%.sz 9~%"))
	    ((format fn "SME Output~55@T~2,'0D/~2,'0D/~2D ~2,'0D:~2,'0D:~2,'0D~%~%"
		     month date (mod year 100) hour min sec)))
;;      (if (scl:string-search-not-set '(#\Space) *scroll-dump-comments*)
;;	  (format fn "~?~%" *scroll-dump-comments*))
      (format fn "~A~%" *scroll-dump-comments*)
      (dolist (a-line (map 'list #'identity (item-list *scroll-pane*)))
              (cond ((and (string/= "" (car a-line)) (string= (subseq (car a-line) 0 1) ""))
	       (cond (first-line?
		      (if (string/= "" (subseq (car a-line) 1))
			  (format fn "~A" (subseq (car a-line) 1)))
		      (when (cdr a-line)
			(dolist (item (cdr a-line)) (format fn "~A" item))))
		     ((eq *scroll-dump-format* :troff)
		      (format fn ".bp~%~A" (subseq (car a-line) 1))
		      (dolist (item (cdr a-line)) (format fn "~A" item)))
		     (t (format fn "~A~%" #\Page)
			(format fn "SME Output~55@T~2,'0D/~2,'0D/~2D ~2,'0D:~2,'0D:~2,'0D~%~%"
				month date (mod year 100) hour min sec)
			(format fn "~A" (subseq (car a-line) 1))
			(dolist (item (cdr a-line)) (format fn "~A" item))))
	       (if (or (string/= "" (subseq (car a-line) 1))  (cdr a-line))
		   (terpri fn)))
	      (t (dolist (item a-line) (format fn "~A" item))
		 (terpri fn)))
	(setq first-line? nil)))))


;;; **************************************************************
;;;; Windowing layouts and utilities


(defun select-windowing-configuration ()	;toggling between scroll window and graphics window
  (let ((config (clim:frame-current-layout *sme-frame*))                 
	(requested-config (menus:submen '(("Scroll Window" :value scroll-config)
                                          ("Graphics Window" :value graphics-config)
                                          ("LARGE Graphics Window" :value large-graphics-config)
                                          ("Scroll and Graphics" :value split-config)
                                          ("Double Scroll - dump to scroll window"
                                           :value ds-scroll)
                                          ("Double Scroll - dump to scratch pad"
                                           :value ds-scratch))
                                        "Choose a Window Configuration")))
    (when (not (menus:aborted requested-config))
          (case requested-config
                (ds-scratch (setq *sme-output-stream* *spare-scroll-pane*)
                            (clim:set-frame-layout *sme-frame* 'double-scroll-config))
                (ds-scroll (setq *sme-output-stream* *scroll-pane*)
                           (clim:set-frame-layout *sme-frame*  'double-scroll-config))
                (t (unless (eq config requested-config)
                           (clim:set-frame-layout *sme-frame* requested-config)))))))


(defun select-scroll () (clim:set-frame-layout *sme-frame* 'scroll-config))
(defun select-double-scroll () (clim:set-frame-layout *sme-frame* 'double-scroll-config))
(defun select-graphics () (clim:set-frame-layout *sme-frame* 'graphics-config))
(defun select-large-graphics ()
  (clim:set-frame-layout *sme-frame* 'large-graphics-config))
(defun select-split    () (clim:set-frame-layout *sme-frame* 'split-config))


;;;; **********************************************************************
(clim:define-command-table listener :inherit-from (clim:user-command-table))

;;;; This code pirated from ps-2 lisp listener demo and severely hacked
;;;; It also calls handler-bind and ignore-errors by specifying the
;;;; lucid-common-lisp package so it's not quite portable as is.

(defun do-listener ()
  #+ Lucid
  (let ((win (clim:get-frame-pane *sme-frame* 'lisp-pane))
        (command-table (clim:find-command-table 'listener)))
    (clim:with-input-focus (win)
     (progn (clim:terpri win)
            (clim:format win "Lisp Activated") (clim:terpri)
            (let* ((*standard-input* win)
                   (*standard-output* win)
                   (*query-io* win)
                   )
              (clim:with-command-table-keystrokes
               (keystrokes command-table)
               (loop while *lisp-active* do
                     (clim::catch-abort-gestures 
                      ("Return to ~A command level" (frame-pretty-name frame))
                      (clim:clear-input *standard-input*)
                      (clim:fresh-line *standard-input*)
                      (multiple-value-bind 
                       (clim:command-or-form type)
                       (block keystroke
                              (lucid-common-lisp:handler-bind ;;; yup... they redefine it.
                               ((clim::accelerator-gesture
                                 #'(lambda (c)
                                     (return-from keystroke
                                                  (values (clim::accelerator-gesture-event c)
                                                          ':keystroke)))))
                               (let ((clim::*accelerator-characters* keystrokes))
                                 (clim:accept `(clim:command-or-form :command-table 
                                                                     ,command-table)
                                              :stream *standard-input*
                                              :prompt "> " :prompt-mode :raw))))
                       (when (eql type ':keystroke)
                             (let ((command (clim:lookup-keystroke-command-item
                                             clim:command-or-form command-table)))
                               (unless (characterp command)
                                       (setq clim:command-or-form command
                                             type 'command))))
                       (cond ((eql type ':keystroke))
                             ((eql (clim:presentation-type-name type) 'command)
                              (clim:terpri)
                              (apply (command-name clim:command-or-form)
                                     (command-arguments clim:command-or-form))
                              (clim:terpri))
                             (t
                              (clim:terpri)
                              (let ((values (multiple-value-list 
                                             (lucid-common-lisp:ignore-errors 
                                              (eval clim:command-or-form)))))
                                (clim:fresh-line)
                                (dolist (value values)
                                        (clim:present value 'clim:expression :single-box 
                                                      :highlighting)
                                        (clim:terpri))
                                (setq - clim:command-or-form
                                      / values)
                                (shiftf *** ** * (first /))
                                (shiftf +++ ++ + -)))))))))))))

;;; **************************************************************
;;;; graphical displays

;;; Due to Release-7 incompatabilities, no graphical routines are currently supported.

;;; **************************************************************  
;;;; Routines to use the display on sme description groups


(defvar *sme-graph-type*			;this will be our Zgraph graph-type for histories
	(make-instance 'zg::graph-type
		:name 'sme-graph-type
		:traversal-function 'sme::dgroup-traversal-function
		   ;;Determines whether or not traversal function is to be applied
		   ;;recursively (depth first search from the root vertices), which is
		   ;;the default, or simply mapped onto the root vertices.
		:traverse-recursively? t
		   ;;controls how vertices and edges are printed
		:vertex-print-string-function 'sme::dgroup-vertex-print-function
		:edge-print-string-function 'zg::default-edge-print-string-function
		   ;;Specifies handling of mouse-selection of vertices and edges
		:vertex/edge-selection-handler 'zg::default-vertex/edge-selection-handler
		   ;;Specifies means of dumping to a scroll window a description of vertices and edges.
		   ;;This is only called in the Zgraph user interface when the user selects a vertex
		   ;;or edge to be described.
		:vertex/edge-description-function 'sme::sme-vertex/edge-description-function))


;;; Display a dgroup using Zgraph
;;;
(defun display-dgroup (dgroup1 &optional (dgroup2 nil))
  (let (
        #||
        (old-deb zg::*graph-debug-actions*)
        (old-sty zg::*graph-plotting-style* ) ;; will be plot-lattice
        (old-dis zg::*display*)
        (old-out zg::*graph-output*)
        
        (zg::*graph-debug-actions* nil)
	(zg::*graph-plotting-style* 'zg::plot-in-circles-for-bi-connected-components) 
        ;; will be plot-lattice
	(zg::*display* *graphics-pane*)
	(zg::*graph-output* *graphics-pane*) ;; should get the "real-window" of this? -- gjs
        ||#
	instance)
    (setf zg::*display* *graphics-pane*)
    (setf zg::*status* *scroll-pane*)
    (setf zg::*graph-debug-actions* nil)
    (setf zg::*graph-plotting-style* 'zg::plot-in-circles-for-bi-connected-components)
    (setf zg::*graph-output* *graphics-pane*)
    (setq zg::*description-output* *sme-output-stream*)  ;tell it where to put node description info
    (if (and *windowing-active?* (eq (clim:frame-current-layout *sme-frame*) 'scroll-config))
	(select-graphics))
    (cond ((null dgroup2)
	   (cond ((setq instance (dgroup-zgraph-instance dgroup1)))
		 (t (setq instance
			  (make-instance 'zg::graph
					 :name (dgroup-name dgroup1)
					 :root-vertices (list dgroup1)
					 :type *sme-graph-type*
;;                                         :default-root-finding-form (list dgroup1)
                                         ))
		    (zg::construct instance)
		    (zg::plot-vertices instance)
		    (setf (dgroup-zgraph-instance dgroup1) instance))))
	  (t (setq instance (make-instance 'zg::graph
					   :name (dgroup-name dgroup1)
					   :root-vertices (list dgroup1 dgroup2)
;;                                           :default-root-finding-form (list dgroup1)
					   :type *sme-graph-type*))
	     (zg::construct instance)
	     (zg::plot-vertices instance)))
    (zg::set-graph zg::zgraph-app instance)
    (zg::draw instance)))
;;;    (clim:window-clear *sme-graphics-output*)


;;; Dgroup traversal function for Zgraph
;;;
;;;** fix commutative to allow arg-type or (arg-name arg-type)
;;;
(defun dgroup-traversal-function (vertex &aux tmp)
  (labels ((commutative-arg-label (arg-list)
	     (if (consp arg-list)
		 (commutative-arg-label (car arg-list))
		 arg-list)))
    (cond ((Dgroup? vertex) (mapcar #'(lambda (root) (cons root "")) (dgroup-roots vertex)))
	  ((expression? vertex)
	   (if (> (expression-order vertex) 1)
	       (mapcar #'(lambda (arg arg-spec)
			   (cons arg (if (symbolp arg-spec)
					 (format nil arg-spec)
					 (format nil (car arg-spec)))))
		       (expression-arguments vertex)
		       (if (n-ary? (expression-functor vertex))
			   (setf (cdr (setq tmp
					    (list (commutative-arg-label
						    (arg-list (expression-functor vertex))))))
				 tmp)
			   (arg-list (expression-functor vertex)))))))))


(defun dgroup-vertex-print-function (vertex-struct)
  (labels ((get-name (item) (if (entity? item) (entity-name item) item)))
    (let ((sme-data (zg::vertex-data vertex-struct)))
      (cond ((Dgroup? sme-data)
	     (values (format nil "~A" (dgroup-name sme-data)) '(:swiss :bold :small)))
	    ((expression? sme-data)
	     (if (= (expression-order sme-data) 1)
		 (values
		   (concatenate 'string
				(format nil "~A(" (expression-functor sme-data))
				(if (= (length (expression-arguments sme-data)) 1)
				    (format nil "~A"
					    (entity-name
					       (car (expression-arguments sme-data))))
				    (string (reduce #'(lambda (a b)
							(format nil "~A,~A"
								(get-name a) (get-name b)))
						    (expression-arguments sme-data))))
				")")
		   '(:swiss :bold :small))
		 (values (format nil "~A<~A>" (expression-functor sme-data)
				              (expression-order sme-data))
			 '(:swiss :bold :small))))
	    ((entity? sme-data)
	     (values (format nil "~A" (entity-name sme-data)) '(:swiss :bold :small)))
	    ((values (format nil "~A" sme-data) '(:swiss :bold :small)))))))

(defun sme-vertex/edge-description-function (vertex-or-edge scroll-window)
  (cond ((typep vertex-or-edge 'zg::edge))
	((dgroup? (zg::vertex-data vertex-or-edge))
	 (menu-describe-dgroup (zg::vertex-data vertex-or-edge)))
	((let* (;;get a description string.
		(description
		  (with-output-to-string (stream)
		    (let ((*standard-output* stream))
		      (describe (if (typep vertex-or-edge 'zg::vertex)
				    (zg::vertex-data vertex-or-edge)
				    (zg::edge-data vertex-or-edge))
				T))))
		;;Break up the string into a list of lines of the description: ("line 1" "line 2"...)
		(description-lines
		  (with-input-from-string (stream description)
		    (loop for line = (read-line stream nil)
			  while line
			  collect line))))
	   (let ((separate-at (length (item-list scroll-window))))
	        ;;separate this description from previous ones.
             (scroller-app scroll-window '("------------------------"))
	        ;;Add in the lines of description.
	        ;;The scroll window items are entries of the form ({format string} . {format args}).
	        ;;We simply use "~a" to print out a description line.
	     (dolist (line description-lines)
                     (scroll-app scroll-window (list line)))
	        ;;Scroll to top of this description.
             (scroller-top scroll-window))))))

;;; **************************************************************  
;;; zgraph commands

(define-sme-command (com-main-select-graphics-object-edge)
    ((zg::edge 'zg::edge :gesture :describe))
  (zg::handle-selection-of-object (zg::graph-type (zg::graph zg::zgraph-app)) zg::edge zg::zgraph-app))

(define-sme-command (com-main-select-graphics-object-vertex)
     ((zg::vertex 'zg::vertex :gesture :describe))
  (zg::handle-selection-of-object (zg::graph-type (zg::graph zg::zgraph-app)) zg::vertex zg::zgraph-app))


(clim:define-command  (com-main-drag)
    ((x 'integer) (y 'integer))
  (zg::main-drag zg::zgraph-app x y)
  (zg::set-graph zg::zgraph-app (zg::graph zg::zgraph-app)))

(clim:define-presentation-to-command-translator pres-darg
  (clim:blank-area com-main-drag sme
              :gesture :select
              :tester
              ((x y window)
               T))
  (x y)
  `(,x ,y))


;;; **************************************************************  

(defun startup (&optional (reinit nil))
  (when reinit
        (setq *RUNNING* t)
        (princ "Initializing") (terpri)
        (setf *my-root* (clim:open-root-window :clx))
        (setf *sme-frame* (clim:make-application-frame 'sme
                                                     :parent *my-root*
                                                     :left 100
                                                     :right 960
                                                     :top 150
                                                     :bottom 600))
        (setq zg::zgraph-app (clim:make-application-frame 'zg::zgraph
                                                          :parent *my-root*
                                                          :width 700
                                                          :height 700)))
  (setq *windowing-active?* t)
  (setf clim:*application-frame* *sme-frame*)
  (setq	*sme-title-pane* (clim:get-frame-pane *sme-frame* 'title)
        *scroll-pane*       (clim:get-frame-pane *sme-frame* 'scroll-pane)
        *graphics-pane*     (clim:get-frame-pane *sme-frame* 'graphics-pane)
        *spare-scroll-pane* (clim:get-frame-pane *sme-frame* 'spare-scroll-pane)
        *lisp-pane*         (clim:get-frame-pane *sme-frame* 'lisp-pane))
  (setq *sme-graphics-output* *graphics-pane*)
  (setq *sme-output-stream* *scroll-pane*)  
  (setf (item-list *spare-scroll-pane*) '("spare pane"))
  (catch 'exit-me  
    (loop while t do
          (catch 'switch-zgraph (clim:run-frame-top-level *sme-frame*))
          (catch 'zg::exit-zgraph (clim:run-frame-top-level zg::zgraph-app))))
  (catch 'clim::resynchronize))


(if *RUNNING* (startup) (startup t))
