;;; -*- Mode: LISP; Syntax: Common-lisp; Package: SME; Base: 10; Fonts: CPTFONT,TR12I -*-

;;;; Structure-Mapping Engine  (module windowing.lisp: machine dependent I/O routines, flavors, etc.)

;;; Copyright (C) 1986,  Brian C. Falkenhainer, Unversity of Illinois
;;; All rights reserved

;;; Fancy windowing interface routines

(in-package 'sme)
(shadow 'sme::class-name)
(shadow 'sme::defclass)
(shadow 'sme::class)
(shadow 'sme::sme-print)
(shadow 'sme::sme-terpri)
(use-package 'clos)

(defvar *RUNNING* nil)
(defvar *lisp-active* nil)
(defvar *std-o* *standard-output*)
;;; *********************************************************************************
(defvar *sme-frame* nil "holds the main frame window flavor")
(defvar *graphics-pane* nil "holds the graphics pane")
(defvar *scroll-pane* nil "holds the scroll pane")
(defvar *spare-scroll-pane* nil "holds the temporary, scratch scroll pane")
(defvar *lisp-pane* nil "holds the sme lisp listener")
(defvar *sme-title-pane* nil "holds the sme lisp listener")

(defvar *sme-graphics-output* nil "pane or stream where the graphics output should go")

;;(defvar *default-font* fonts:tvfont "default font for displaying and describing dgroups")
(defvar *default-font* nil)
(defvar *scroll-dump-format* :lpr)
(defvar *scroll-dump-comments* (format nil "~60,,,' A" "") "comments for scroll printout")

(defvar *sme-extra-language-files* nil) ;; For adding language modules


(setq *dgroup-extension-name* "dgroup")
(setq *rule-extension-name* "rules")
(defvar *rule-set-index* nil)

(proclaim '(special *tre-rules-saver* *initial-assertions*
		    *mhc-intern-rules* *mhc-filter-rules*))
(proclaim '(special *current-rule-set*))

(defvar *current-rule-set* nil)
(defvar *stat-str* "hi there")

(defvar *drawing-location* :sme)
;;; *********************************************************************************
;;; make class for scrolling windows

(lucid-common-lisp:defclass scrolwin (clim::clx-window)
  ((itms :initarg :itm-lst
         :accessor item-list)))

;;; Accessor functions
(defmethod scroller-app ((win scrolwin) itm-list)
  (dolist (itm (reverse itm-list))
          (setf (item-list win) (cons (list itm) (item-list win))))
  (print-itms win itm-list))

(defmethod scroller-set ((win scrolwin) itm-list)
  (setf (item-list win) itm-lst)
  (print-list win))

(defmethod scroller-top ((win scrolwin))
  (print-list win))

(defmethod scroller-clear ((win scrolwin))
  (setf (item-list win) nil)
;;  (clim:clear-output-record (clim:output-recording-stream-output-record win))
  (clim:window-clear win)
  (print-list win))

(defmethod print-list ((win scrolwin))
  (dolist (itm (item-list win))
          (if (listp itm) (clim:format win "~A~%" (car itm))
            (clim:format win "~A~%" itm))))

(defmethod print-itms ((win scrolwin) item-list)
  (dolist (itm item-list)
          (if (listp itm) (clim:format win "~A~%" (car itm))
            (clim:format win "~A~%" itm))))

;;; *********************************************************************************
;;; Given a string with potential line-feeds in it, print out each line
;;;   to *sme-output-stream*. If *sme-output-stream* is a scroll-window,
;;;   it will append the items to the scroll items list. If it is not a
;;;   scroll window, it will merely use format.
;;;
;;;     A new line is always printed at the end of the string, whether
;;;     or not one was actually there....
;;;
(defun sme-print (string)
  (with-input-from-string (stream string)
                          (do ((line (read-line stream nil) (read-line stream nil)))
                              ((null line))
                              (if (typep *sme-output-stream* 'scrolwin)
                                  (scroller-app *sme-output-stream* (list line))
                                (clim:format *sme-output-stream* "~A~%" line)))))

(defun sme-terpri (&optional (n 1))
  (dotimes (i n) (sme-print (string #\newline))))

;;; *********************************************************************************
;;; Overall frame
;;;

(clim:define-application-frame sme () () ;; ((object-list :initform nil))
  (:panes 
    ((title :title
	    :display-string "SME") 
     (menu-pane :command-menu)
     (status     :application
                 :scroll-bars nil
                 :display-function `(stat-it)
                 :default-text-style `(:fix :bold :small))
     (graphics-pane :application 
                    :stream-background clim:+alice-blue+)
     (scroll-pane :application
                  :window-class `scrolwin
                  :itm-lst '((" "))
                  :stream-background clim:+light-cyan+)
     (spare-scroll-pane :application
                        :window-class `scrolwin
                        :itm-list '(("spare window"))
                        :stream-background clim:+light-cyan+)
     (lisp-pane :application
               :display-function `(ldisp)
               :stream-background clim:+lavender-blush+)
     (documentation :pointer-documentation)))
  (:layout ((scroll-config (:column :rest
                                    (menu-pane :compute)
                                    (status :compute)
                                    (scroll-pane 0.70)
                                    (lisp-pane :rest)
                                    (documentation :compute)))
            (double-scroll-config (:column :rest
                                           (menu-pane :compute)
                                           (status :compute)
                                           (:row 0.70
                                                 (scroll-pane :rest)
                                                 (spare-scroll-pane :rest))
                                           (lisp-pane :rest)
                                           (documentation :compute)))
            (graphics-config (:column :rest
                                      (menu-pane :compute)
                                      (status :compute)
                                      (graphics-pane 0.70)
                                      (lisp-pane :rest)
                                      (documentation :compute)))
            (large-graphics-config (:column :rest
                                            (menu-pane :compute)
                                            (status :compute)
                                            (graphics-pane 0.80)
                                            (lisp-pane :rest)
                                            (documentation :compute)))
            (split-config (:column :rest
                                   (menu-pane :compute)
                                   (status :compute)
                                   (:row 0.70
                                         (scroll-pane :rest)
                                         (graphics-pane :rest))
                                   (lisp-pane :rest)
                                   (documentation :compute))))))
  
;;; **********************************************************************
;;;; Display functions
(defmethod graphit ((application sme) stream) ()
  (clim:format stream "grapher"))

(defmethod scrol-disp ((application sme) stream) ()
  (clim:format stream "~A" (item-list stream)))

(defmethod ldisp ((application sme) stream) ()
   (clim:format stream "> "))

(defmethod stat-it ((application sme) stream) ()
  (clim:format stream "~A" *stat-str*))

(defun To-Status-Line (str) 
  (setq *stat-str* (format nil "~A" str))
  (stat-it *sme-frame* (clim:get-frame-pane *sme-frame* 'status)))

;;; *********************************************************************************
;;;; Top line menu

(define-sme-command (com-init :menu "File Operations" 
                              :name "Choose default files and Initialize SME")
  ()
  (let* ((std-out *standard-output*))
    (setf *standard-output* *std-o*)
    (eval (menus:submen '(("Initialize -- Clears SME & reloads language" 
                           :value (progn (sme-init)
                                         (setf *rule-set-index* (list (list *sme-default-rules* 
                                                                            nil nil nil nil)))
                                         (setf *current-rule-set* *sme-default-rules*)
                                         (cond ((listp *sme-extra-language-files*)
                                                (dolist (lfile *sme-extra-language-files*)
                                                        (load lfile))) (t nil))))
                          ("Change File Parameters" :value (change-fparms))
                          ("Load a Rules File" :value (get-rules))
                          ("Load a Dgroup File" :value (if *initialized?* (get-dgroup)
                                                         (to-status-line "Must Initialize first"))))
                        "File Operations"))
    (setf *standard-output* std-out)))

(define-sme-command (com-match :menu "Match Operations" 
                               :name "Try to analogically match two chosen description-groups") ()
  (if *initialized?* (menu-match) (to-status-line "Must Initialize first")))


(define-sme-command (com-utilities :menu "Utilities"
                                   :name  "Miscellaneous system utilities") ()
  (menu-utilities))

;;(define-sme-command (com-select-windowing :menu "Alter Windowing"
;;                                          :name  "Choose a graphics and scroll window configuration") ()
;;  (select-windowing-configuration))

#||
(define-sme-command (com-generalize :menu "Generalize" 
                                    :name "Form generalizations of the current analogy") ()
  (if (eq (clim:frame-current-layout *sme-frame*) 'graphics-config)
      (select-scroll))
  (let ((std-out *standard-output*))
    (setf *standard-output* *std-o*)
    (generalize (car (best-gmaps)) T)
    (setf *standard-ouput* std-out)))
||#

(define-sme-command (com-describe :menu "Describe DGroup"
                                  :name "Choose a description group for textual display") ()
  (menu-describe-dgroup))

(define-sme-command (com-display :menu "Display DGroup" 
                                 :name "Choose a description group for graphical display") ()
  (menu-display-dgroup))

(define-sme-command (com-display-pairs :menu "Display DGroup Pairs" 
                                       :name "Graphically display two chosen description-groups") ()
  (menu-display-pairs))

;;(define-sme-command (com-examine :menu "Examine Match" :name "Examine match results and statistics") ()
;;  (menu-examine-match-results))


(define-sme-command (com-insp :menu "Inspect Dgroups" :name "Displays information about current run") ()
  (menu-inspect))

(define-sme-command (com-reports :menu "Reports" :name "Dumps information about current run") ()
  (menu-reports))

(define-sme-command  (com-lisp :menu "Use lisp" :name "Activate Lisp listener window") ()
   (cond (*lisp-active* (setf *lisp-active* nil))
         (t (setf *lisp-active* t)
            (do-listener))))

(define-sme-command (com-bye :menu "Quit" :name "Quit") ()
  (throw 'exit-me nil))
;;; **************************************************************
;;; SME Change file parameters

(defun change-fparms ()
  (let ((vals (menus:domenun 
               "Change File Parameters"
               `((,*SME-LANGUAGE-FILE* *SME-LANGUAGE-FILE* "language definitions file" 
                                       clim:pathname)
;;                 (,*sme-extra-language-files* *sme-extra-language-files* 
;;                                              "Language extension files" string )
                 (,*sme-rules-pathname* *sme-rules-pathname* "Rules pathname" clim:pathname )
                 (,*sme-dgroup-pathname* *sme-dgroup-pathname* "Dgroup pathname" clim:pathname )
                 (,*rule-extension-name* *rule-extension-name* "Rule file extension" string)
                 (,*dgroup-extension-name* *dgroup-extension-name*  
                                           "Dgroup file extension" string)))))
    (eval `(progn ,@(mapcar #'(lambda (x) `(setf ,(first x) ,(second x))) vals)))))

#||
(defun change-fparms () 
  (flet ((bbb ()
          (format *std-o* "YUCH~%")
          (setq *query-io* *lisp-pane*)
          (menus:domenu ((sme::*SME-LANGUAGE-FILE* 'clim:pathname "language definitions file")
                         (sme::*sme-extra-language-files* 'clim:pathname "Language extension files")
                         (sme::*sme-rules-pathname* 'clim:pathname "Dgroup pathname")
                         (sme::*rule-extension-name* 'string "Rule file extension")
                         (sme::*dgroup-extension-name* 'string "Dgroup file extension"))
                        "Change File Parameters" t)))
        (bbb)))
||#
;;; **************************************************************
;;;; SME System Utilities

;;; takes in list of vals (a b) and sets variable named a to b for each
(defmacro dochg (&optional lst)
  `(progn ,@(mapcar #'(lambda (x) `(setf ,(first x) ,(second x))) (eval lst))))

(defSME-Utility "Switch Drawing Location" (menu-set-drawing-location))
(defSME-Utility "Zgraph Inspector" (throw 'switch-zgraph nil))
(defSME-Utility "Inspect Evidence" (match-evidence-inspector))

(defSME-Utility "Dump Scroll Window"
		(if *windowing-active?*
		    (dump-scroll (dump-scroll-menu))))
(defSME-Utility "Clear Scroll Window"
  (if (menus:yes-no-dial "Are you sure you want to clear it?")
      (clear-scroll)))
(defSME-Utility "Alter Windowing" (select-windowing-configuration))
;;(defSME-Utility "Change Match Parameters" (menu-set-match-parms))
(defSME-Utility "Change System Parameters" (change-parms))

;;; Bring up the System Utilities menu
;;;
(defun menu-utilities ()
  (eval (choose-facility *system-utilities-menu* "SME System Menu")))


;;; Load a file chosen from the list of provided files which matched the 
;;; template *sme-pathname*>file-pattern
;;;
(defun menu-get-file (directory file-pattern header &optional bin-file?  &aux file)
  (setq file (menus:submen (mapcar #'(lambda (x) (list x 'value x)) 
                                   (directory (merge-pathnames file-pattern directory)))
                           header))
  (unless (menus:aborted file)
          (let ((std-out *standard-output*))
            (setf *standard-output* *std-o*)
            (if (not (menus:aborted file))  
                (if bin-file?
;;;            #+IMACH (load (merge-pathnames ".ibin" file))
;;;            #-IMACH (load (merge-pathnames ".bin" file))
                    (load file) (load file)))
            (setf *standard-output* std-out) file)))
  
(defun get-dgroup ()
  (menu-get-file *sme-dgroup-pathname* "*.dgroup" "Select a DGroup File to Load"))

(defun get-rules ()
  (let* ((*tre-rules-saver* nil)
         (*initial-assertions* nil)
         (*mhc-intern-rules* nil)
         (*mhc-filter-rules* nil)
         (rset-name (format nil "~A" (menu-get-file *sme-rules-pathname* "*.rules" 
                                                    "Select a Rules File to Load")))
         (entry (assoc rset-name *rule-set-index*
                       :test #'string=)))
    (unless (string= rset-name "NIL")
            (when entry (setq *rule-set-index* (delete entry *rule-set-index*)))
            (push (list rset-name (list rset-name *mhc-intern-rules* *mhc-filter-rules*
                                        *tre-rules-saver* *initial-assertions*
                                        ))
                  *rule-set-index*))))

;;; Change SME parameters
;;;

(defun change-parms ()
  (let ((vals (menus:domenun "SME System Parameters"
                             (mapcar #'(lambda (x) 
                                         (list (eval (first x)) (first x) 
                                               (second x) 
                                               (case (third x)
                                                     (:assoc 
                                                      (cons 'clim:member-alist (list
                                                        (mapcar #'(lambda (y) (list (car y) :value
                                                                                    (cdr y)))
                                                                (fourth x)))))
                                                     (:boolean
                                                      '(clim:member-alist (("yes" :value t) 
                                                                           ("no" :value nil))))
                                                     (:integer 'integer)
                                                     (:string 'string)
                                                     (:character 'character)
                                                     (:expression 'clim:pathname)
                                                     (t 'string))))
                                     *parameter-menu-options*)
                             )))
    (eval `(progn ,@(mapcar #'(lambda (x) `(setf ,(first x) ,(second x))) vals)))))
;;    (declare (special vals))
;;    t (dochg vals)))

;;; Compare two apparently identical Gmaps
;;;
(defun menu-compare-gmaps (&aux gmap1 gmap2)
  (when *gmaps*
        (let* ((maps (menus:domenun "Select Gmaps to compare"
                                    '((1 :gmap1 "Number of the first Gmap" integer)
                                      (2 :gmap2 "Number of the second Gmap" integer))))
               (gmap1 (cadar maps))
               (gmap2 (cadadr maps)))
          (compare-gmaps (nth (1- gmap1) *gmaps*) (nth (1- gmap2) *gmaps*) T))))


;;; Dump the SME scroll window to a file.  This is just the menu to get it going.  
;;; See dump-scroll farther down in this file.
;;;
(defun dump-scroll-menu ()
  (let* ((file "~/aaa.out")
         (vals (menus:domenun "Select Options for Scroll Dump"
                   `((,*scroll-dump-format* *scroll-dump-format*                       
                      "Printer Format" (clim:member-alist (("Imprint and Line Printer" :value :lpr) 
                                                           ("Troff" :value :troff))))
                     (,file file "output-file" clim:pathname )
                     (,*scroll-dump-comments* *scroll-dump-comments* 
                      "Comment String (as a FORMAT control string)" string)))))
    (eval `(progn ,@(mapcar #'(lambda (x) `(setf ,(first x) ,(second x))) vals)))
    file))

(defun menu-set-drawing-location ()
  (let ((newloc (menus:submen '(("SME graphics pane" :value :sme) 
                                ("Zgraph Inspector" :value :zgraph))
                              "Choose location for drawing of graphs")))
    (unless (menus:aborted newloc) (setf *drawing-location* newloc))))

;;; **************************************************************
;;;; SME matching operations


;;; These are intended to be commands relevant to creating and analyzing analogical mappings.
;;;  They will typically be the same commands that appear in the command pane.
;;;

(defSME-Operation "Display Two Dgroups" (menu-display-pairs))
(defSME-Operation "Display A Dgroup" (menu-display-dgroup))
(defSME-Operation "Describe A Dgroup" (menu-describe-dgroup))
(defSME-Operation "Compare Gmaps" (menu-compare-gmaps))
(defSME-Operation "Generalize" (progn (if (member (clim:frame-current-layout *sme-frame*)
						  '(graphics-config large-graphics-config))
					  (select-scroll))
                                      (let ((std-out *standard-output*))
                                        (setf *standard-output* *std-o*)
                                        (generalize (car (best-gmaps)) T)
                                        (setf *standard-output* std-out))))
(defSME-Operation "Match" (menu-match))


;;; Bring up the System Utilities menu
;;;
(defun menu-matching-operations ()
  (choose-facility *sme-operations-menu* "SME Operations Menu"))


;;; A general menu-choose facility which reuses a single flavor instance
;;;

(defun choose-facility (item-list &optional (label "Choose Item") (column-length 15))
  (let* ((items (mapcar #'(lambda (x) (if (listp x) (list (first x) :value (third x))
                                        (list x :value x))) item-list))
         (val (menus:submen items label)))
    (unless (menus:aborted val) val)))

;;; Give the user a pop-up menu of the known dgroups to choose from
;;;     Return a dgroup name or nil.
;;;
(defun DGroup-choose (&optional (label "Choose DGroup"))
  (choose-facility *description-groups* label 15))

;;; Provide the user with the list of currently loaded description-groups and 
;;; graphically display the one choosen.
;;;
(defun menu-display-dgroup (&aux dgroup)
  (setq dgroup (fetch-dgroup (DGroup-choose)))
  (if dgroup  (display-dgroup dgroup)))

;;; Query the user for two description groups and graphically display them.
;;;
(defun menu-display-pairs (&aux dgroup1 dgroup2)
  (let ((l-choose (DGroup-choose "Choose Left DGroup")))
    (unless (or (null l-choose) (menus:aborted l-choose))
            (setq dgroup1 (fetch-dgroup l-choose)))
    (if dgroup1 (let ((r-choose (DGroup-choose "Choose Right DGroup")))
                  (unless (or (null r-choose) (menus:aborted r-choose))
                          (setq dgroup2 (fetch-dgroup r-choose)))))
    (if (and dgroup1 dgroup2) (display-dgroup dgroup1 dgroup2))))


;;; Provide the user with the list of currently loaded description-groups and 
;;; textually describe the one choosen.
;;;
(defun menu-describe-dgroup (&optional dgroup)
  (labels ((terse? () (not (menus:submen-na '(("Terse Format (default)" :value nil) 
                                              ("Full Format" :value T))
                                         "Dgroup description format"))))
          (unless dgroup (if (setq dgroup (DGroup-choose))
                             (setq dgroup (fetch-dgroup dgroup))))
          (when dgroup
                (if  (and *windowing-active?*
                          (member (clim:frame-current-layout *sme-frame*)
                                  '(graphics-config large-graphics-config) :test #'eq))
                    (select-scroll))
                (sme-print (string #\Page))
                (describe-dgroup dgroup (terse?)))))

;;; Allow the user to choose base and target dgroups for matching
;;;
(defun menu-set-match-parms (&aux base-name target-name)
  (if (null *base*) (setf *base* (first *description-groups*)))
  (if (null *target*) (setf *target* (first *description-groups*)))
  (cond ((and (> (length *description-groups*) 0) (> (length *rule-set-index*) 0))
   (let* ((vals
           (menus:domenun "set match parameters" `((,*base* :base "Base" 
                                            (clim:member-alist ,(mapcar #'(lambda (x) (list x :value x))
                                                                        *description-groups*)))
                                           (,*target* :target "Target"
                                            (clim:member-alist ,(mapcar #'(lambda (x) (list x :value x))
                                                                        *description-groups*)))
                                           (,*current-rule-set* :rules "Rule Set"
                                            (clim:member-alist ,(mapcar #'(lambda (x) (list x :value x))
                                                                        (mapcar #'first 
                                                                                *rule-set-index*)))))))
          (rset (cadr (find (cadr (assoc :rules vals)) *rule-set-index* 
                           :test #'(lambda (name entry)
                                     (string= name (car entry)))))))
     (when rset 
          (setq *tre-rules-saver* (fourth rset))
          (setq *initial-assertions* (fifth rset))
          (setq *current-rule-set* (first rset))
          (clear-sme)
          ;; Since SME doesn't provide a cache for match rules, I do.
          (setq *match-rules-file* (first rset))
          (setq *mhc-intern-rules* (second rset))
          (setq *mhc-filter-rules* (third rset)))
     (setf *base* (cadr (assoc :base vals)) *target* (cadr (assoc :target vals))) t))
   (t (to-status-line "Must have at least 1 dgroup and 1 rule set to match")
      nil)))

;; This splits out stuff hidden in MATCH and SME-RULES-FILE
;; to simplify switching between rule sets
;; Nuke SME internal pointers (to falken: why doesn't TRE-INIT do this??)
;;
(defun clear-sme ()
  (setq *mhc-intern-rules* nil)
  (setq *mhc-filter-rules* nil)
  (setq *match-hypotheses* nil)
  (setq *mh-identifier* (cons 1 nil))
  (setq *gmaps* nil)
  (setq *gmap-count* 0)
  (if *mh-hash-table*
      (clrhash *mh-hash-table*)
      (setq *mh-hash-table* (make-hash-table :test #'equal :size *mh-table-size*)))
  (tre-init))

;;; In scroll window, show the result of trying to analogically match 
;;; chosen base and target description-groups.
;;;
(defun menu-match (&aux base-name target-name)
  (eval (menus:submen '(("Set match parameters" :value (menu-set-match-parms))
                        ("Run Match" :value (when (menu-set-match-parms)
                                                  (if (eq (clim:frame-current-layout 
                                                           *sme-frame*) 'graphics-config)
                                                      (select-scroll))
                                                  (sme-print (string #\Page))
                                                  (match *base* *target* T)))
                        ("Examine Match" :value (menu-examine-match-results))
                        ("Compare Gmaps" :value (menu-compare-gmaps))
                        ("Generalize" :value (let ((std-out *standard-output*))
                                               (if (eq (clim:frame-current-layout *sme-frame*) 
                                                       'graphics-config)
                                                   (select-scroll))
                                               (setf *standard-output* *std-o*)
                                               (generalize (car (best-gmaps)) T)
                                               (setf *standard-ouput* std-out)))
                        ("Display Match" :value (print-match))
                        )
                      "Match operations" )))
;;; **************************************************************
;;;; Inspecting the evidence links for a match

;;; Allow the user to query the source of evidence for Match Hypotheses and GMaps
;;;
(defun match-evidence-inspector ()
  (let ((std-out *standard-output*))
    (setf *standard-output* *std-o*)
    (unless (and *base* *target*)
            (to-status-line "There is no analogy to inspect...")
            (return-from match-evidence-inspector t))
    (let* ((vals (menus:domenun "Evidence Inspector" 
                                '(((m g) :mode-flags "Types of Things to Examine" 
                                     (clim:subset-alist (("Match Hypotheses" :value m) 
                                                         ("GMaps" :value g))))
                                  (all :mh-base "Match Hypothesis Base Item (name or all)" 
                                       clim:expression)
                                  (all :mh-target "Match Hypothesis Target Item (name or all)"
                                       clim:expression)
                                  (999 :gmap-number "Gmap (id number or all (enter 999))" integer))))
           (mode-flags (cadr (assoc :mode-flags vals)))
           (mh-base (cadr (assoc :mh-base vals)))
           (mh-target (cadr (assoc :mh-target vals)))
           (gmap-number (if (eq 999 (cadr (assoc :gmap-number vals))) 
                            'all (cadr (assoc :gmap-number vals))))
           )
      (when mode-flags
            (if  (eq (clim:frame-current-layout *sme-frame*) 'graphics-config)  (select-scroll))
            (sme-print (with-output-to-string (stream)
                                              (format stream "~A~%~5TEvidence for Match from ~A to ~A."
                                                      #\Page (dgroup-name *base*) (dgroup-name *target*))))
            (when (member 'm mode-flags)
                  (sme-print 
                   (with-output-to-string 
                    (stream)
                    (cond ((eq mh-base 'all)
                           (if (eq mh-target 'all)
                               (dolist (mh *match-hypotheses*) (why-node (mh-bms-node mh) stream))
                             (dolist (pat (fetch
                                           `(mh ?b ,(fetch-expression mh-target *target* nil))))
                                     (why pat stream))))
                          ((eq mh-target 'all)
                           (dolist (pat (fetch `(mh ,(fetch-expression mh-base *base* nil) ?t)))
                                   (why pat stream)))
                          ((why (fetch `(mh ,(fetch-expression mh-base *base* nil)
                                            ,(fetch-expression mh-target *target* nil)))
                                stream))))))
            (when (member 'g mode-flags)
                  (sme-print 
                   (with-output-to-string 
                    (stream)
                    (cond ((numberp gmap-number)
                           (let ((gmap (find gmap-number *gmaps* :test #'= :key #'gm-id)))
                             (if gmap (why-node (gm-bms-node gmap) stream)
                               (to-status-line "~%There is no GMap with ~A as an id number"
                                       gmap-number))))
                          ((dolist (gmap *gmaps*) (why-node (gm-bms-node gmap) stream)))))))
            (sme-terpri)))
    (setf *standard-output* std-out)))
  
  
;;; **************************************************************
;;;; Scroll pane printout routines

(defun clear-scroll (&optional spare-scroll-window?)
  (if spare-scroll-window?
      (scroller-clear *spare-scroll-pane*)
    (scroller-clear *scroll-pane*)))
	

;;; Dump the contents of the scroll window to a file
;;;
(defun dump-scroll (file-name &aux (first-line? t))
  (with-open-file (fn file-name :direction :output)
    (multiple-value-bind (sec min hour date month year) (get-decoded-time)
      (cond ((eq *scroll-dump-format* :troff)
	     (format fn ".he '\\fBSME Output\\fP'%'\\fB~2,'0D/~2,'0D/~2D  ~2,'0D:~2,'0D:~2,'0D'\\fP"
		     month date (mod year 100) hour min sec)
	     (format fn "~%.nf~%.ft TA~%.ss 22~%.po 0.5i~%.ll 6.9i~%.sz 9~%"))
	    ((format fn "SME Output~55@T~2,'0D/~2,'0D/~2D ~2,'0D:~2,'0D:~2,'0D~%~%"
		     month date (mod year 100) hour min sec)))
;;      (if (scl:string-search-not-set '(#\Space) *scroll-dump-comments*)
;;	  (format fn "~?~%" *scroll-dump-comments*))
      (format fn "~A~%" *scroll-dump-comments*)
      (dolist (a-line (map 'list #'identity (item-list *scroll-pane*)))
              (cond ((and (string/= "" (car a-line)) (string= (subseq (car a-line) 0 1) ""))
	       (cond (first-line?
		      (if (string/= "" (subseq (car a-line) 1))
			  (format fn "~A" (subseq (car a-line) 1)))
		      (when (cdr a-line)
			(dolist (item (cdr a-line)) (format fn "~A" item))))
		     ((eq *scroll-dump-format* :troff)
		      (format fn ".bp~%~A" (subseq (car a-line) 1))
		      (dolist (item (cdr a-line)) (format fn "~A" item)))
		     (t (format fn "~A~%" #\Page)
			(format fn "SME Output~55@T~2,'0D/~2,'0D/~2D ~2,'0D:~2,'0D:~2,'0D~%~%"
				month date (mod year 100) hour min sec)
			(format fn "~A" (subseq (car a-line) 1))
			(dolist (item (cdr a-line)) (format fn "~A" item))))
	       (if (or (string/= "" (subseq (car a-line) 1))  (cdr a-line))
		   (terpri fn)))
	      (t (dolist (item a-line) (format fn "~A" item))
		 (terpri fn)))
	(setq first-line? nil)))))


;;; **************************************************************
;;;; Windowing layouts and utilities

(defun select-windowing-configuration ()	;toggling between scroll window and graphics window
  (let ((config (clim:frame-current-layout *sme-frame*))                 
	(requested-config (menus:submen '(("Scroll Window" :value scroll-config)
                                          ("Graphics Window" :value graphics-config)
                                          ("LARGE Graphics Window" :value large-graphics-config)
                                          ("Scroll and Graphics" :value split-config)
                                          ("Double Scroll - dump to scroll window"
                                           :value ds-scroll)
                                          ("Double Scroll - dump to scratch pad"
                                           :value ds-scratch))
                                        "Choose a Window Configuration")))
    (when (not (menus:aborted requested-config))
          (case requested-config
                (ds-scratch (setq *sme-output-stream* *spare-scroll-pane*)
                            (clim:set-frame-layout *sme-frame* 'double-scroll-config))
                (ds-scroll (setq *sme-output-stream* *scroll-pane*)
                           (clim:set-frame-layout *sme-frame*  'double-scroll-config))
                (t (unless (eq config requested-config)
                           (clim:set-frame-layout *sme-frame* requested-config)))))))


(defun select-scroll () (clim:set-frame-layout *sme-frame* 'scroll-config))
(defun select-double-scroll () (clim:set-frame-layout *sme-frame* 'double-scroll-config))
(defun select-graphics () (clim:set-frame-layout *sme-frame* 'graphics-config))
(defun select-large-graphics ()
  (clim:set-frame-layout *sme-frame* 'large-graphics-config))
(defun select-split    () (clim:set-frame-layout *sme-frame* 'split-config))


;;;; **********************************************************************
(clim:define-command-table listener :inherit-from (clim:user-command-table))

;;;; This code pirated from ps-2 lisp listener demo and severely hacked
;;;; It also calls handler-bind and ignore-errors by specifying the
;;;; lucid-common-lisp package so it's not quite portable as is.

(defun do-listener ()
  #+ Lucid
  (let ((win (clim:get-frame-pane *sme-frame* 'lisp-pane))
        (command-table (clim:find-command-table 'listener)))
    (clim:with-input-focus (win)
     (progn (clim:terpri win)
            (clim:format win "Lisp Activated") (clim:terpri)
            (to-status-line "Lisp Activated")
            (let* ((*standard-input* win)
                   (*standard-output* win)
                   (*query-io* win)
                   )
              (clim:with-command-table-keystrokes
               (keystrokes command-table)
               (loop while *lisp-active* do
                     (clim::catch-abort-gestures 
                      ("Return to ~A command level" (frame-pretty-name frame))
                      (clim:clear-input *standard-input*)
                      (clim:fresh-line *standard-input*)
                      (multiple-value-bind 
                       (clim:command-or-form type)
                       (block keystroke
                              (lucid-common-lisp:handler-bind ;;; yup... they redefine it.
                               ((clim::accelerator-gesture
                                 #'(lambda (c)
                                     (return-from keystroke
                                                  (values (clim::accelerator-gesture-event c)
                                                          ':keystroke)))))
                               (let ((clim::*accelerator-characters* keystrokes))
                                 (clim:accept `(clim:command-or-form :command-table 
                                                                     ,command-table)
                                              :stream *standard-input*
                                              :prompt "> " :prompt-mode :raw))))
                       (when (eql type ':keystroke)
                             (let ((command (clim:lookup-keystroke-command-item
                                             clim:command-or-form command-table)))
                               (unless (characterp command)
                                       (setq clim:command-or-form command
                                             type 'command))))
                       (cond ((eql type ':keystroke))
                             ((eql (clim:presentation-type-name type) 'command)
                              (clim:terpri)
                              (apply (command-name clim:command-or-form)
                                     (command-arguments clim:command-or-form))
                              (clim:terpri))
                             (t
                              (clim:terpri)
                              (let ((values (multiple-value-list 
                                             (lucid-common-lisp:ignore-errors 
                                              (eval clim:command-or-form)))))
                                (clim:fresh-line)
                                (dolist (value values)
                                        (clim:present value 'clim:expression :single-box 
                                                      :highlighting)
                                        (clim:terpri))
                                (setq - clim:command-or-form
                                      / values)
                                (shiftf *** ** * (first /))
                                (shiftf +++ ++ + -)))))))))))))

;;; **************************************************************
;;;; graphical displays

;;; Due to Release-7 incompatabilities, no graphical routines are currently supported.

;;; **************************************************************  
;;;; Routines to use the display on sme description groups


(defvar *sme-graph-type*			;this will be our Zgraph graph-type for histories
	(make-instance 'zg::graph-type
		:name 'sme-graph-type
		:traversal-function 'sme::dgroup-traversal-function
		   ;;Determines whether or not traversal function is to be applied
		   ;;recursively (depth first search from the root vertices), which is
		   ;;the default, or simply mapped onto the root vertices.
		:traverse-recursively? t
		   ;;controls how vertices and edges are printed
		:vertex-print-string-function 'sme::dgroup-vertex-print-function
		:edge-print-string-function 'zg::default-edge-print-string-function
		   ;;Specifies handling of mouse-selection of vertices and edges
		:vertex/edge-selection-handler 'zg::default-vertex/edge-selection-handler
		   ;;Specifies means of dumping to a scroll window a description of vertices and edges.
		   ;;This is only called in the Zgraph user interface when the user selects a vertex
		   ;;or edge to be described.
		:vertex/edge-description-function 'sme::sme-vertex/edge-description-function))


;;; Display a dgroup using Zgraph
;;;
(defun display-dgroup (dgroup1 &optional (dgroup2 nil))
  (let (instance)
    (case *drawing-location*
          (:sme
           (setf zg::*display* *graphics-pane*)
           (setf zg::*status* *scroll-pane*)
           (setf zg::*graph-debug-actions* nil)
           (setf zg::*docs* (clim:get-frame-pane *sme-frame* 'status))
;;           (setf zg::*graph-plotting-style* 'zg::plot-in-circles-for-bi-connected-components)
           (setf zg::*graph-plotting-style* 'zg::plot-lattice)
           (setf zg::*graph-output* *graphics-pane*)
           (setq zg::*description-output* *sme-output-stream*) ;tell it where to put node description info
           (if (and *windowing-active?* (eq (clim:frame-current-layout *sme-frame*) 'scroll-config))
               (select-graphics)))           
          (:zgraph
           (setf
            zg::*display* (clim:get-frame-pane zg::zgraph-app 'zg::display)
            zg::*status* (clim:get-frame-pane zg::zgraph-app 'zg::status)
            zg::*docs* (clim:get-frame-pane zg::zgraph-app 'zg::docs)
            zg::*interaction* (clim:get-frame-pane zg::zgraph-app 'zg::interaction)
            zg::*description* (clim:get-frame-pane zg::zgraph-app 'zg::description))))
    (cond ((null dgroup2)
	   (cond ((setq instance (dgroup-zgraph-instance dgroup1)))
		 (t (setq instance
			  (make-instance 'zg::graph
					 :name (dgroup-name dgroup1)
					 :root-vertices (list dgroup1)
					 :type *sme-graph-type*
                                         ))
		    (zg::construct instance)
		    (zg::plot-vertices instance)
		    (setf (dgroup-zgraph-instance dgroup1) instance))))
	  (t (setq instance (make-instance 'zg::graph
					   :name (dgroup-name dgroup1)
					   :root-vertices (list dgroup1 dgroup2)
					   :type *sme-graph-type*))
	     (zg::construct instance)
	     (zg::plot-vertices instance)))
    (zg::set-graph zg::zgraph-app instance)
    (zg::draw instance)))
;;;    (clim:window-clear *sme-graphics-output*)


;;; Dgroup traversal function for Zgraph
;;;
;;;** fix commutative to allow arg-type or (arg-name arg-type)
;;;
(defun dgroup-traversal-function (vertex &aux tmp)
  (labels ((commutative-arg-label (arg-list)
                                  (if (consp arg-list)
                                      (commutative-arg-label (car arg-list))
                                    arg-list)))
          (cond ((Dgroup? vertex) 
                 (mapcar #'(lambda (root) (cons root "")) (dgroup-roots vertex)))
                ((expression? vertex)
                 (if (> (expression-order vertex) 1)
                     (mapcar #'(lambda (arg arg-spec)
                                 (cons arg (if (symbolp arg-spec)
                                               (format nil "~A" arg-spec)
                                             (format nil "~A" (car arg-spec)))))
                             (expression-arguments vertex)
                             (if (n-ary? (expression-functor vertex))
                                 (setf (cdr (setq tmp
                                                  (list (commutative-arg-label
                                                         (arg-list (expression-functor vertex))))))
                                       tmp)
                               (arg-list (expression-functor vertex)))))))))


(defun dgroup-vertex-print-function (vertex-struct)
  (labels ((get-name (item) (if (entity? item) (entity-name item) item)))
    (let ((sme-data (zg::vertex-data vertex-struct)))
      (cond ((Dgroup? sme-data)
	     (values (format nil "~A" (dgroup-name sme-data)) '(:swiss :bold :small)))
	    ((expression? sme-data)
	     (if (= (expression-order sme-data) 1)
		 (values
		   (concatenate 'string
				(format nil "~A(" (expression-functor sme-data))
				(if (= (length (expression-arguments sme-data)) 1)
				    (format nil "~A"
					    (entity-name
					       (car (expression-arguments sme-data))))
				    (string (reduce #'(lambda (a b)
							(format nil "~A,~A"
								(get-name a) (get-name b)))
						    (expression-arguments sme-data))))
				")")
		   '(:swiss :bold :small))
		 (values (format nil "~A<~A>" (expression-functor sme-data)
				              (expression-order sme-data))
			 '(:swiss :bold :small))))
	    ((entity? sme-data)
	     (values (format nil "~A" (entity-name sme-data)) '(:swiss :bold :small)))
	    ((values (format nil "~A" sme-data) '(:swiss :bold :small)))))))

(defun sme-vertex/edge-description-function (vertex-or-edge scroll-window)
  (cond ((typep vertex-or-edge 'zg::edge))
	((dgroup? (zg::vertex-data vertex-or-edge))
	 (menu-describe-dgroup (zg::vertex-data vertex-or-edge)))
	((let* (;;get a description string.
		(description
		  (with-output-to-string (stream)
		    (let ((*standard-output* stream))
		      (describe (if (typep vertex-or-edge 'zg::vertex)
				    (zg::vertex-data vertex-or-edge)
				    (zg::edge-data vertex-or-edge))
				T))))
		;;Break up the string into a list of lines of the description: ("line 1" "line 2"...)
		(description-lines
		  (with-input-from-string (stream description)
		    (loop for line = (read-line stream nil)
			  while line
			  collect line))))
	   (let ((separate-at (length (item-list scroll-window))))
	        ;;separate this description from previous ones.
             (scroller-app scroll-window '("------------------------"))
	        ;;Add in the lines of description.
	        ;;The scroll window items are entries of the form ({format string} . {format args}).
	        ;;We simply use "~a" to print out a description line.
	     (dolist (line description-lines)
                     (scroll-app scroll-window (list line)))
	        ;;Scroll to top of this description.
             (scroller-top scroll-window))))))

;;; **************************************************************  
;;; Local versions of zgraph button commands

(define-sme-command (com-main-select-graphics-object-edge)
    ((zg::edge 'zg::edge :gesture :describe))
  (zg::handle-selection-of-object (zg::graph-type (zg::graph zg::zgraph-app)) zg::edge zg::zgraph-app))

(define-sme-command (com-main-select-graphics-object-vertex)
     ((zg::vertex 'zg::vertex :gesture :describe))
  (zg::handle-selection-of-object (zg::graph-type (zg::graph zg::zgraph-app)) zg::vertex zg::zgraph-app))


(define-sme-command  (com-main-drag)
    ((x 'integer) (y 'integer))
  (zg::main-drag zg::zgraph-app x y)
  (zg::set-graph zg::zgraph-app (zg::graph zg::zgraph-app)))

(clim:define-presentation-to-command-translator pres-darg
  (clim:blank-area com-main-drag sme
              :gesture :select
              :tester
              ((x y window)
               (equal window *graphics-pane*)))
  (x y)
  `(,x ,y))

(define-sme-command (com-main-vertex-move)
  ((vert 'zg::vertex :gesture :select))
  (zg::main-vertex-move zg::zgraph-app vert))


;;; **************************************************************  
;;; Routines to let the user examine a match
(defun menu-examine-match-results () 
  (let ((choice (menus:submen '(("Display match" :value :match)
                                ("Display base" :value :base)
                                ("Display target" :value :target)
                                ("Display Statistics" :value :stats)
                                )
                              "Examine What?")))
    (unless (menus:aborted choice)
            (if  (and *windowing-active?*
                          (member (clim:frame-current-layout *sme-frame*)
                                  '(graphics-config large-graphics-config) :test #'eq))
                (select-scroll))
            (case choice
                  (:match (display-match *base* *target*))
                  (:base (describe-dgroup *base*))
                  (:target (describe-dgroup *target*))
                  (:stats (display-match-statistics))))))

;;(defun menu-examine-match-results () 
;;    (to-status-line "This option not yet implemeted"))

;;; **************************************************************  
;;; Inspecting routines
(defun menu-inspect ()
  (let ((choice (menus:submen '(("Print dgroup" :value :print)
                                ("Dgroup concordance" :value :conc))
                              "Choose type of inspection"))
        dgroup)
    (unless (menus:aborted choice)
            (if (setq dgroup (DGroup-choose))
                (setq dgroup (fetch-dgroup dgroup)))
            (when dgroup
                  (if  (and *windowing-active?*
                            (member (clim:frame-current-layout *sme-frame*)
                                    '(graphics-config large-graphics-config) :test #'eq))
                      (select-scroll))
                  (case choice
                        (:print (pp-dgroup dgroup *std-o*))
                        (:conc (dgroup-concordance dgroup *std-o*)))))))
;;; **************************************************************  

;;;; Some useful SME-oriented print routines

(defun lispify-sme-thing (thing)
  (cond ((entity? thing) (sme::entity-name thing))
	((expression? thing)
	 (cons (sme::expression-functor thing)
	       (mapcar #'lispify-sme-thing 
		       (sme::expression-arguments thing))))))

(defun lispify-candidate-inferences (cis dgroup)
  (cond ((null cis) nil)
	((symbolp cis)
	 (let ((exp (get cis (dgroup-name dgroup))))
	   (if exp (lispify-sme-thing exp) cis)))
	((listp cis)
	 (cons (lispify-candidate-inferences (car cis) dgroup)
	       (lispify-candidate-inferences (cdr cis) dgroup)))
	(t cis)))

(defun dgroup-concordance (dgr &optional (stream *standard-output*))
  (labels ((print-concordance-divider (stream which)
	      (format stream "~%~A"
		      (if (eq which :boundary)
			  "========================================================="
			  "---------------------------------------------------------")))
	   (print-concordance-item (item stream)
				   (format stream "~%~A" (expression-name item))
				   (pprint (lispify-sme-thing item) stream)))
  (print-concordance-divider stream :boundary)
  (format stream "~% Concordance for Dgroup ~A." (dgroup-name dgr))
  (dolist (item (sort (copy-list (dgroup-expressions dgr))
		      #'(lambda (x y) (string< (symbol-name (expression-name x))
					       (symbol-name (expression-name y))))))
    (print-concordance-divider stream :Internal)
    (print-concordance-item item stream))
  (print-concordance-divider stream :Internal)
  (format stream "~% End of concordance for Dgroup ~A." (dgroup-name dgr))
  (print-concordance-divider stream :boundary)))

(defun dgroup-set-concordance (dgroup-list report-pathname)
  ;;; Produces a file containing the concordances for the listed dgroups.
  ;;; (Can be used with a single dgroup if needed, of course)
  (with-open-file (rout report-pathname :direction :output)
		  (do ((dgrs dgroup-list (cdr dgrs)))
		      ((null dgrs))
		    (dgroup-concordance (car dgrs) rout)
		    ;; Install a page break if needed between dgroups
		    (if (cdr dgrs) (format rout "~|")))))
;;; *************************************************************************
;;;; LaTeX'ing dgroup concordances
;; Use the "description" envirionment and alltt to produce a compact 
;; guide to match plots and decoding other arcane SME output.
;; All this needs to be better integrated.  

(defun LaTeX-dgroup-concordance (dgr &optional (stream *standard-input*) (borders? t))
  ;; Generates the LaTeX source for a single dgroup.
  (labels ((print-concordance-divider (stream which)
            (format stream "~%~A"
		    (case which
		      (:boundary "\\rule{\\textwidth}{0.04in}")
		      (:no-border "\\linebreak")
		      (t "\\hline" )))) ;; "\\rule{\\textwidth}{0.01in}"
	   (print-concordance-item (item stream)
	     (format stream "~%{\\bf ~A:} & ~%\\begin{minipage}{6in}~%\\begin{alltt}~A~%\\end{alltt}~%\\end{minipage} \\\\"
		     (expression-name item)
		     (with-output-to-string (foo) (pprint (lispify-sme-thing item) foo)))))
    (print-concordance-divider stream (if borders? :boundary :no-border))
    (format stream "~%{\\center {\\bf Concordance for Dgroup ~A}}~%" (dgroup-name dgr))
    (user::LaTeX-start stream "tabular" "{ll}")
    (dolist (item (sort (copy-list (dgroup-expressions dgr))
			#'(lambda (x y) (string< (symbol-name (expression-name x))
						 (symbol-name (expression-name y))))))
      (print-concordance-divider stream (if borders? :Internal :no-border))
      (print-concordance-item item stream))
    (print-concordance-divider stream (if borders? :Internal :no-border))
    (user::LaTeX-end stream "tabular")
    (format stream "~%{\\center {\\bf End of concordance for Dgroup ~A.}}~%"
	    (dgroup-name dgr))
    (if borders? (print-concordance-divider stream :boundary))))

(defun LaTeX-dgroup-set-concordance (dgroup-list report-pathname &optional (border? t))
  ;; Produces a file containing LaTeX source code for a fancy
  ;; dgroup concordance
  (with-open-file (rout report-pathname :direction :output)
	(user::start-LaTeX-file rout)
	(do ((dgrs dgroup-list (cdr dgrs)))
	    ((null dgrs))
	  (LaTeX-dgroup-concordance (car dgrs) rout border?)
	  (if (cdr dgrs) (user::New-LaTeX-Page rout)))
	(user::end-LaTeX-file rout)))
;;; *************************************************************************
;;;; Pretty-printing whole dgroups

(defun pp-dgroup (dgr &optional (stream *standard-output*))
  (dolist (form (lispify-dgroup dgr))
    (pprint form stream)))

(defun lispify-dgroup (dgr)
  (mapcar #'lispify-sme-thing (dgroup-roots dgr)))

(defun lispify-inferences (gm)
  (fully-expand-expression-form (gm-inferences gm) (gm-target gm)))

(defun dgroup-set-pp-listing (dgroup-list report-pathname)
  ;;; Produces a file containing the pretty-printed descriptions for the listed dgroups.
  ;;; (Can be used with a single dgroup if needed, of course)
  (let ((divider "========================================================="))
    (with-open-file (rout report-pathname :direction :output)
		    (do ((dgrs dgroup-list (cdr dgrs)))
			((null dgrs))
		      (format rout "~%~A" divider)
		      (format rout "~% Dgroup ~A:" (dgroup-name (car dgrs)))
		      (pp-dgroup (car dgrs) rout)
		      (format rout "~%~A" divider)
		      ;; Install a page break if needed between dgroups
		      (if (cdr dgrs) (format rout "~|"))))))
;;; *************************************************************************
;;;; Printing dgroup statistics

;; What is there to know?  Number of entities, attributes, relations, functions, and order vector.
;; All this could fit on one line, thus making a pretty concise report.

(defun dgroup-set-statistics (dgroup-list report-pathname)
  (with-open-file (rout report-pathname :direction :output)
		  (format rout "~% Dgroup statistics -- ~D in all.~% File = ~A"
			  (length dgroup-list) report-pathname)
  (format rout "~%~25@:<Name~>~6:<# Ent~>~6:<# Atr~>~6:<# Fun~>~6:<# Log~>~6:<# Rel~>  Order vector")
		  (do ((dgrs (sort (copy-list dgroup-list)
				   #'(lambda (dg1 dg2)
				       (string< (dgroup-name dg1) (dgroup-name dg2))))
			     (cdr dgrs)))
		      ((null dgrs))
		    (dgroup-stats (car dgrs) rout))
		  (format rout "~% End of dgroup statistics listing ~A" report-pathname)))

(defun dgroup-stats (dgr &optional (stream *standard-output*))
 ;; Number of entities, attributes, relations, functions, and order vector.
  (multiple-value-bind (nentities nattributes nfunctions nlogical nrelations order-vector)
		       (compute-dgroup-stats dgr)
  (format stream "~%~25:<~A~>~6:<~A~>~6:<~A~>~6:<~A~>~6:<~A~>~6:<~A~>  ~A"
	  (dgroup-name dgr) nentities nattributes nfunctions nlogical nrelations order-vector)))

(defun compute-dgroup-stats (dgroup)
  (let ((nattributes 0) (nfunctions 0) (nlogical 0)(nrelations 0)
	(order-vector (list (length (dgroup-entities dgroup)))))
    (dolist (exp (dgroup-expressions dgroup))
      (case (sme::predicate-type (sme::expression-functor exp))
	(ENTITY (error "Can't have an entity here -- ~A in ~A." exp dgroup))
	(FUNCTION (incf nfunctions))
	(ATTRIBUTE (incf nattributes))
	(RELATION (incf nrelations))
	(LOGICAL (incf nlogical))
	(t (error "Unknown expression type -- ~A is ~A in ~A."
		  exp (sme::predicate-type (sme::expression-functor exp))
		  dgroup)))
      (setq order-vector (extend-order-vector order-vector
					      (sme::expression-order exp))))
    (values (car order-vector) nattributes nfunctions nlogical
	    nrelations order-vector)))

(defun extend-order-vector (vector order &aux v)
  ;; First make sure it extends to that order
  (setq v vector)
  (dotimes (i order)
    (cond ((null (cdr v)) ;; Extend it
	   (setf (cdr v) (list 0))))
    (setq v (cdr v)))
  (incf (car v))
  vector)
;;; *************************************************************************
;;;; Inspecting matches in more detail

(defun fetch-form (form dgroup &aux result)
  (setq result (find form (dgroup-expressions dgroup)
		     :test #'(lambda (form exp)
			       (equal form (lispify-sme-thing exp)))))
  (if result (return-from FETCH-FORM result))
  (find form (dgroup-entities dgroup)
		   :test #'(lambda (form exp)
			     (equal form (lispify-sme-thing exp)))))

(defun find-mh (base-item target-item &optional (mhs sme::*match-hypotheses*))
  ;; Make not user-unfriendly, at least.
  (unless (or (entity? base-item) (expression? base-item))
    (setq base-item (fetch-form base-item sme::*base*)))
  (unless (or (entity? target-item) (expression? target-item))
    (setq target-item (fetch-form target-item sme::*target*)))
  (dolist (mh mhs)
    (when (and (eq base-item (mh-base-item mh))
	       (eq target-item (mh-target-item mh)))
      (return-from find-mh mh))))

(defun pp-mh (mh &optional (stream *standard-output*))
  (format stream "~%MH(~A,~A)~A, Order = ~D, SES = ~D" (sme::mh-base-item mh)
	  (sme::mh-target-item mh) (if (sme::mh-justifies-incomplete? mh) "*" "")
	  (sme::mh-order mh) (sme::node-belief+ (sme::mh-bms-node mh)))
  (format stream "~% B: ~A" (lispify-sme-thing (sme::mh-base-item mh)))
  (format stream "~% T: ~A" (lispify-sme-thing (sme::mh-target-item mh)))
  (format stream "~% #P: ~D #C: ~D." (length (sme::mh-justifications mh))
	  (length (sme::mh-justifies mh))))

(defun pp-gmap (gm &optional (details? nil) (stream *standard-output*))
  (format stream "~%GM~D: ~D correspondences, SES = ~D"
	  (gm-id gm) (length (gm-elements gm)) (node-belief+ (gm-bms-node gm)))
  (format stream "~%  Object mappings:")
  (dolist (emap (gm-emaps gm))
    (format stream "~%   ~A <-> ~A" (lispify-sme-thing (cadr emap))
	    (lispify-sme-thing (caddr emap))))
  (cond ((gm-inferences gm)
	 (format stream "~%  Candidate Inferences:")
	 (dolist (inf (lispify-candidate-inferences (gm-inferences gm) (gm-target gm)))
	   (pprint inf stream)))
	(t (format stream "~%  No candidate inferences.")))
  (when details?
    (dolist (mh (gm-elements gm)) (pp-mh mh stream)))
  gm)

                        
;;; **************************************************************  
;;; Reports
(defun menu-reports ()
  (let ((choice (menus:submen '(("Background Reports" :value :back)
                                ("Match Reports" :value :match))
                              "Choose report type")))
    (unless (menus:aborted choice)
            (case choice
                  (:back (menu-back-report))
                  (:match (menu-match-report))))))

(defvar *report-print-mode* :ASCII) ;; Alternative is LaTeX.
(defvar *latex-report-borders* t)

(defun menu-back-report ()
  (let* ((choice (menus:submen '(("Concordance for all dgroups" :value :conca)
                                 ;; ("Concordance for dgroups in memory" :value :concm)
                                 ("Concordance for some dgroups" :value :concs)
                                 ("PP all dgroups" :value :ppa)
                                 ;; ("PP memory"  :value :ppm)
                                 ("PP some dgroups" :value :pps))
                               "Report type"))
         (choicem (case choice
                        ((:conca :concm)
                         (menus:domenun 
                          "Choose options"
                          '(("~/report.sme" :fil  "File name?" clim:pathname)
                            (:Ascii :type "Ascii or Latex?" (clim:member-alist
                                                             (("Ascii" :value :Ascii)
                                                              ("Latex" :value :Latex))))
;                            (t :type "Latex Report-borders?"
;                               (clim:member-alist (("Yes" :value t) 
;                                                   ("No" :value nil))))
                            )))
                        (:concs
                         (menus:domenun 
                          "Choose options"
                          `(("~/report.sme" :fil  "File name?" clim:pathname)
                            (*report-print-mode* :type "Ascii or Latex?" (clim:member-alist
                                                                          (("Ascii" :value :Ascii)
                                                                           ("Latex" :value :Latex))))
;                           (*latex-report-borders* :type "Latex Report-borders?"
;                                                   (clim:member-alist ("Yes" :value t) 
;                                                                      ("No" :value nil)))
                            (nil :dgs "Dgroups?" (clim:subset-alist ,(mapcar #'(lambda (dg) 
                                                                                 (list
                                                                                  (format nil "~A" dg)
                                                                                  :value
                                                                                  (fetch-dgroup dg)))
                                                                             *description-groups*))))))
                        ((:ppa :ppm)
                         (menus:domenun 
                          "Choose file"
                          '(("~/report.sme" :fil  "File name?" clim:pathname))))
                        (:pps
                         (menus:domenun 
                          "Choose options"
                          `(("~/report.sme" :fil  "File name?" clim:pathname)
                            (nil :dgs "Dgroups?" (clim:subset-alist ,(mapcar #'(lambda (dg) 
                                                                                 (list
                                                                                  (format nil "~A" dg)
                                                                                  :value
                                                                                  (fetch-dgroup dg)))
                                                                             *description-groups*))))))
                        (t nil))))
    (case choice
          (:conca 
           (case (cadr (assoc :type choicem))
                 (:ASCII
                  (sme::dgroup-set-concordance
                   (mapcar #'sme::fetch-dgroup sme::*description-groups*) 
                   (cadr (assoc :fil choicem))))
                 (:Latex
                  (sme::LaTeX-dgroup-set-concordance
                   (mapcar #'sme::fetch-dgroup sme::*description-groups*) 
                   (cadr (assoc :fil choicem))))))
;;;                    (:concm (case (cadr (assoc :type choicem))
;;;                                  (:ASCII (sme::dgroup-set-concordance *memory* output-path))))
          (:concs
           (case (cadr (assoc :type choicem))
                 (:ASCII
                  (sme::dgroup-set-concordance
                   (cadr (assoc :dgs choicem))
                   (cadr (assoc :fil choicem))))
                 (:Latex
                  (sme::LaTeX-dgroup-set-concordance
                   (cadr (assoc :dgs choicem))
                   (cadr (assoc :fil choicem))))))
          (:ppa
           (sme::dgroup-set-pp-listing (mapcar #'sme::fetch-dgroup sme::*description-groups*)
                                       (cadr (assoc :fil choicem))))
          (:pps
           (sme::dgroup-set-pp-listing (cadr (assoc :dgs choicem))
                                       (cadr (assoc :fil choicem))))
          (t nil))))

                  
(defun menu-match-report () (to-status-line "Nothing here yet"))
;;; **************************************************************  
;;; Text-print of matches
;;; (not yet implemented)
(defun dgroup-trav (vertex &aux tmp)
  (labels ((commutative-arg-label (arg-list)
                                  (if (consp arg-list)
                                      (commutative-arg-label (car arg-list))
                                    arg-list)))
          (cond ((Dgroup? vertex) 
                 (mapcar #'(lambda (root) (cons root "")) (dgroup-roots vertex)))
                ((expression? vertex)
                 (if (> (expression-order vertex) 1)
                     (mapcar #'(lambda (arg arg-spec)
                                 (cons arg (if (symbolp arg-spec)
                                               (format nil "~A" arg-spec)
                                             (format nil "~A" (car arg-spec)))))
                             (expression-arguments vertex)
                             (if (n-ary? (expression-functor vertex))
                                 (setf (cdr (setq tmp
                                                  (list (commutative-arg-label
                                                         (arg-list (expression-functor vertex))))))
                                       tmp)
                               (arg-list (expression-functor vertex)))))))))


(defun node-print (sme-data)
  (labels ((get-name (item) (if (entity? item) (entity-name item) item)))
          (cond ((Dgroup? sme-data)
                 (format nil "~A" (dgroup-name sme-data)))
                ((expression? sme-data)
                 (if (= (expression-order sme-data) 1)
                     (concatenate 'string
                                  (format nil "~A(" (expression-functor sme-data))
                                  (if (= (length (expression-arguments sme-data)) 1)
                                      (format nil "~A"
                                              (entity-name
                                               (car (expression-arguments sme-data))))
                                    (string (reduce #'(lambda (a b)
                                                        (format nil "~A,~A"
                                                                (get-name a) (get-name b)))
                                                    (expression-arguments sme-data))))
                                  ")")
                   (values (format nil "~A<~A>" (expression-functor sme-data)
                                   (expression-order sme-data)))))
                ((entity? sme-data)
                 (format nil "~A" (entity-name sme-data)))
                (t (format nil "~A" sme-data)))))

(defun prt-node (num node)
  (clim:format *spare-scroll-pane* "~D : ~A~%" num (node-print node)))

(defvar *nodenum* 1)
(defvar *nodelist* nil)

(defun form-treeh (node level)
  (unless (null node)
          (setf *nodelist* (cons (list level *nodenum* node)  *nodelist* ))
          (incf *nodenum*)
          (let ((subs (dgroup-trav node)))
            (if (listp subs) (dolist (sub subs) (if (listp sub) (form-treeh (car sub) (1+ level)) 
                                                  (form-treeh sub (1+ level))))
              (form-treeh subs (1+ level))))))

(defun print-treeh (node level)
  (form-treeh node level)
  (let ((max-level 0))
    (dolist (item *nodelist*) (if (> (car item) max-level) (setf max-level (car item))))
    (dotimes (i (+ max-level 1))
             (clim:format *scroll-pane* "               ")
             (dolist (item (reverse *nodelist*)) 
                     (cond ((eq (car item) i) 
                            (clim:format *scroll-pane* "~A   " (cadr item))
                            (if (< (cadr item) 9) 
                                (clim:format *scroll-pane* " ")))
                           (t (clim:format *scroll-pane* "     "))))
             (clim:format *scroll-pane* "~%")))
  (dolist (item (reverse *nodelist*)) 
          (prt-node (cadr item) (third item))))

(defun print-treev (node level gm)
  (unless (null node)
          (dotimes (i level) (clim:format *scroll-pane* " "))
          ;;  (clim:format *scroll-pane* "~A~%" (node-print node))
          (clim:format *scroll-pane* "~A" *nodenum*)
          (dotimes (i (- 14 (+ level (length (format nil "~A~%" *nodenum*)))))
                   (clim:format *scroll-pane* " "))
          (clim:format *scroll-pane* "|")
          (let ((emaps (mapcar #'(lambda (gmp) (cdr (mh-form gmp))) 
                               (gm-elements (nth (- gm 1) *gmaps*)))))
            (dolist (item (reverse *nodelist*))
                    (if (member (third item) (assoc node (mapcar #'reverse emaps)))
                        (clim:format *scroll-pane* "x    ")
                      (clim:format *scroll-pane* "     "))))
          (clim:terpri *scroll-pane*)
          (prt-node *nodenum* node)
          (incf *nodenum*)
          (let ((subs (dgroup-trav node)))
            (if (listp subs) (dolist (sub subs) (if (listp sub) (print-treev (car sub) (1+ level) gm) 
                                                  (print-treev sub (1+ level) gm)))
              (print-treev subs (1+ level) gm)))))

(defun print-match () 
  (let ((gm (cadar (menus:domenun "What Gmap?" '((1 :gm "Gmap? --> " integer))))))
    (clim:format *scroll-pane* "~%Match for gmap ~D ~%--------------------------------------------~%" gm)
    (setf *nodenum* 1)
    (setf *nodelist* nil)
    (clim:format *spare-scroll-pane* "~%Base~%----------------------------------------~%")
    (print-treeh *base* 1)
    (clim:format *scroll-pane* "              ----------------------------------------~%")
    (setf *nodenum* 1)
    (clim:format *spare-scroll-pane* "~%Target~%----------------------------------------~%")
    (print-treev *target* 1 gm)
    (clim:terpri *scroll-pane*)
    ))

;;; **************************************************************  

(defun startup (&optional (reinit nil))
  (when reinit
        (setq *RUNNING* t)
        (princ "Initializing") (terpri)
        (setf *my-root* (clim:open-root-window :clx))
        (setf *sme-frame* (clim:make-application-frame 'sme
                                                     :parent *my-root*
                                                     :left 50
                                                     :right 1150
                                                     :top 150
                                                     :bottom 900))
        (setq zg::zgraph-app (clim:make-application-frame 'zg::zgraph
                                                          :parent *my-root*
                                                          :width 700
                                                          :height 700)))
  (setq *windowing-active?* t)
  (setf clim:*application-frame* *sme-frame*)
  (setq	*sme-title-pane* (clim:get-frame-pane *sme-frame* 'title)
        *scroll-pane*       (clim:get-frame-pane *sme-frame* 'scroll-pane)
        *graphics-pane*     (clim:get-frame-pane *sme-frame* 'graphics-pane)
        *spare-scroll-pane* (clim:get-frame-pane *sme-frame* 'spare-scroll-pane)
        *lisp-pane*         (clim:get-frame-pane *sme-frame* 'lisp-pane))
  (setq *sme-graphics-output* *graphics-pane*)
  (setq *sme-output-stream* *scroll-pane*)  
  (setf (item-list *spare-scroll-pane*) '("spare pane"))
  (catch 'exit-me  
    (loop while t do
          (catch 'switch-zgraph (clim:run-frame-top-level *sme-frame*))
          (catch 'zg::exit-zgraph (clim:run-frame-top-level zg::zgraph-app))))
  (catch 'clim::resynchronize))

(defun start-windows ()
  (if *RUNNING* (startup) (startup t)))
