;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: mf; -*-

(in-package 'USER)

;;;; Copyright (c) 1989, Kenneth D. Forbus, University of Illinois
;;;;  All rights reserved.

;; This code snarfed from /u/utils/new/panes.lisp, 6/23/89, for portability.
;; (I'm getting tired of updating things in N different directories when installing
;; MAC/FAC on different machines)
;; (Of course, now the screw is when I update one subsystem, need to update it everywhere..)

;;; This isn't supplied in LUCID environment, but is in their manual.
(defflavor property-list-mixin 
  ((property-list nil)) 
  ())

(defmethod (property-list-mixin :putprop) (value property) 
  (setf (getf property-list property) value))

(defmethod (property-list-mixin :get) (property &optional default)
  (getf property-list property default))

(defmethod (property-list-mixin :remprop) (property)
  (remf property-list property))


;;;; Frame for organizing windows
;;
;; Goal is to make a system for quickly putting together
;; simple but powerful user interfaces.
;;
;; A frame has several panes that are considered as a unit.
;; One of them can be a lisp listener.

;; The reason for wrapping panes around windows is to allow them to
;; have special properties.  Like scrollers, dumpers, etc.

;; A pane can also have several configurations.

;; Operations on a window frame
;; :EXPOSE -- exposes all in the current configuration.
;; :DEEXPOSE -- hides everything.
;; :SET-CONFIGURATION -- selects particular named configuration.
;; :SEND-PANE -- send rest of its arguments to the named pane in the configuration.
;; :CLEAR -- clears all windows in current configuration.  

(defflavor wframe ((name "")
		 (windows nil) ;; List of windows for all configurations
		 (configurations nil) ;; Alist of (name . <list of windows>)
		 (config nil)  ;; name of chosen configuration.
		 (exposed? nil) ;; Non-nil if exposed
		 )
  (property-list-mixin)
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables
  )

(defmethod (wframe :current-panes) ()
  (let ((entry (assoc (send self :config) (send self :configurations))))
    (unless entry (error "~A has illegal configuration -- ~A."
			 self (send self :config)))
    (cdr entry)))

(defmethod (wframe :get-pane) (pname)
  (dolist (pane windows)
    (when (eq (send pane :name) pname) (return pane))))

(defmethod (wframe :send-panes) (message &rest arguments) 
  (dolist (pane (send self :current-panes))
    (eval `(send ,pane ,message ,@ arguments))))

(defmethod (wframe :expose) () (send self :send-panes :expose))
(defmethod (wframe :deexpose) () (send self :send-panes :deexpose))

(defmethod (wframe :set-configuration) (new-config)
  (let ((entry (assoc new-config (send self :configurations))))
    (cond (entry 
	   (send self :deexpose)
	   (send self :set-config new-config)
	   (send self :expose))
	  (t (error "~A not a known configuration for ~A."
		    new-config self)))))

;;;; Defining configurations
;;; A configuration is a list of panes, or instances which include panes.
;;; The panes of a configuration are assumed to be non-overlapping and
;;; to completely cover the screen.

;;; Setting these up is a bit tricky.  Does one want shared windows between
;;; configurations?  Yes, because there can only be one lisp window.
;;; Is there any easy way to do this graphically?  Probably not.  But
;;; here are some utilities for making life easier. 

;; Remember the origin for screen coordinates is the upper left of the screen.
(defvar *screen-width* #-Megapel 1018 ;; Empirically determined for IBM 6155 display
  #+Megapel 1023) ;; Megapel

(defvar *screen-height* #-Megapel 750 ;; Empirically determined for IBM 6155 display
  #+Megapel 1023)

;; A list of entries in the following format is used to define panes:
;; (<name> <type to instantiate> . <arguments to make-instance>)
;; The :arguments field includes the viewport and bitmap sizes and locations, hence
;; specifying where that pane can be in all configurations.
;; The panes that comprise a configuration then are just by entries of the form:
;; ((<name> . <panes>) ...)

;; Yes, more manual labor than with Symbolics.  Nobody said life is fair. 

(defun create-wframe (name default configs panes)
  (let ((plist (mapcar #'(lambda (entry)
			   (cons (car entry)
				 (eval `(make-instance ',(cadr entry) ,@ (cddr entry)))))
		       panes))
	(clist nil))
    (dolist (config configs)
      (let ((panes nil) (pane nil))
	(dolist (pane-name (cdr config))
	  (setq pane (assoc pane-name plist))
	  (cond (pane (push (cdr pane) panes))
		(t (error "Pane ~A of configuration ~A not defined."
			  pane-name (car config)))))
	(push (cons (car config) panes) clist)))
    (make-instance 'wframe :name name :config default
		   :windows (mapcar #'cdr plist)
		   :configurations clist)))

(defflavor pane ((name "")
		 (bitmap-width 100)
		 (bitmap-height 100)
		 (screen-width 100)
		 (screen-height 100)
		 (x 0) ;; For top-left point of viewport
		 (y 0)
		 (activate nil)
		 (title "")
		 (title-font *default-font*)
		 (font *default-font*)
		 (scroll? t)
		 (window nil))
  ()
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables
  )

(defmethod (pane :AFTER :init) (ignore)
  (declare (ignore ignore))
  (send self :remake-window))

(defmethod (pane :clear-pane) () (clear-bitmap window))

(defmethod (pane :remake-window) ()
  (send self :set-window (make-window :x x :y y
				      :width bitmap-width :height bitmap-height
				      :viewport-width screen-width :viewport-height screen-height
				      :initial-font font :activate activate
				      :title-font title-font :title title :scroll scroll?)))

(defmethod (pane :expose) () (activate-viewport window) (expose-viewport window))
(defmethod (pane :deexpose) () (hide-viewport window))

;;;; Drawing commands
;;  Insulate user from Lucid-internals like "positions"

(defmethod (pane :draw-circle) (x y radius &key (width 1) (operation boole-1))
  (draw-circle window (make-position x y) radius :width width :operation operation))

(defmethod (pane :draw-line) (x1 y1 x2 y2 &key (width 1) (operation boole-1))
  (draw-line window (make-position x1 y1)
	     (make-position x2 y2) 
	     :width width :operation operation))

(defmethod (pane :draw-box) (xul yul xlr ylr &key (width 1) (operation boole-1))
  (let ((ul (make-position xul yul))
	(ur (make-position xlr yul))
	(ll (make-position xul ylr))
	(lr (make-position xlr ylr)))
    (draw-line window ul ur :width width :operation operation)
    (draw-line window ur lr :width width :operation operation)
    (draw-line window lr ll :width width :operation operation)
    (draw-line window ll ul :width width :operation operation)))

(defmethod (pane :draw-polyline) (positions &key (width 1) (operation boole-1))
  (draw-polyline window (mapcar #'(lambda (pair) (make-position (car pair) (cdr pair)))
				positions)
		 :width width :operation operation))

(defmethod (pane :draw-polypoint) (positions &key (width 1) (operation boole-1))
  (draw-polypoint window (mapcar #'(lambda (pair) (make-position (car pair) (cdr pair)))
				positions)
		  :width width :operation operation))

(defmethod (pane :draw-dashed-horizontal-line) (x1 x2 y &key (width 1) (operation boole-1)
						   (on-pixels 10) (off-pixels 5))
  (do ((x x1)(nx 0)
       (on? t (not on?)))
      ((= x x2))
    (setq nx (min x2 (+ x (if on? on-pixels off-pixels))))
    (if on? (send self :draw-line x y nx y :width width :operation operation))
    (setq x nx)))

;;;; Indicators and indicator panels.
;; An indicator consists of label, corresponding to its
;;    value, or perhaps a name and a value.
;; An indicator can have a STATE, which is LIT or UNLIT,
;;    corresponding to the label being highlighted or not.
;; The indicator's label can be updated as needed, via an
;;    :UPDATE message.
;; An indicator panel consists of a set of indicators.
;; Update messages can go to the entire panel at once or
;; to specific indicators in it.

(defflavor indicator ((name nil) ;; Assume a symbol here for lookup
		      (width 100)
		      (height 100)
		      (region nil)
		      (my-region nil)
		      (updater nil) ;; Should supply
		      (font *default-font*) 
		      (x 0) 
		      (y 0)
		      (state :UNLIT)
		      (justification :LEFT) ;; Default is start at initial position
		      (char-width 20) ;; maximum number of characters -- input
		      (label "") ;; 
		      (bitmap nil)
		      (window nil)) ;; What window it belongs to
  ()
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

;; From the number of characters and font, figure out how big to make the 
;; various parts of it.
(defvar *indicator-height-offset* 2.)
(defvar *indicator-width-offset* 2.)

(defmethod (indicator :AFTER :init) (ignore)
  (declare (ignore ignore))
  (send self :set-width (+ (* *indicator-width-offset* 2)
			   (* (font-fixed-width font) char-width)))
  (send self :set-height (+ (* *indicator-height-offset* 2)
			    (font-height font)))
  (send self :setup-region)
  (send self :set-my-region (make-region :x 0 :y 0 :width width :height height))
  (send self :set-bitmap (make-bitmap :width width :height height)))

(defmethod (indicator :setup-region) ()
  ;; Used also when changing the indicator's location
  (send self :set-region (make-region :x x :y y :width width :height height)))

(defmethod (indicator :reposition) (x y &optional (new-window window))
  (unless (eq new-window window) (send self :set-window new-window))
  (send self :set-x x)
  (send self :set-y y)
  (send self :setup-region))

;;; Things indicators can do

(defmethod (indicator :ON) ()
  (unless (eq state :LIT)
    (bitblt-region (viewport-bitmap window) region (viewport-bitmap window)
		   region boole-c1)
    (send self :set-state :LIT)))

(defmethod (indicator :OFF) ()
  (unless (eq state :UNLIT)
    (bitblt-region (viewport-bitmap window) region (viewport-bitmap window)
		   region boole-c1)
    (send self :set-state :UNLIT)))

(defmethod (indicator :CLEAR) () 
  (clear-bitmap window region)) ;; Clear out old contents

(defmethod (indicator :draw-label) ()
  (clear-bitmap bitmap)
  (stringblt bitmap (make-position (if (eq justification :LEFT) 0 
				       (- width (string-width label font)))
		     (- (font-height font) *indicator-height-offset*))
	     font label)
  (bitblt-region bitmap my-region (viewport-bitmap window) region boole-1))

(defmethod (indicator :UPDATE) ()
  (send self :clear)
  (when updater (send self :set-label (funcall updater)))
  (send self :draw-label)
  (if (eq state :lit) (send self :on)))

(defmethod (indicator :outline) ()
  (send self :draw-box (1+ x) (1+ y) (+ x width -1) (+ y height +1) :line-width 1 :operation boole-xor))

(defmethod (indicator :draw-box) (xul yul xlr ylr &key (line-width 1) (operation boole-1))
  (let ((ul (make-position xul yul))
	(ur (make-position xlr yul))
	(ll (make-position xul ylr))
	(lr (make-position xlr ylr)))
    (draw-line window ul ur :width line-width :operation operation)
    (draw-line window ur lr :width line-width :operation operation)
    (draw-line window lr ll :width line-width :operation operation)
    (draw-line window ll ul :width line-width :operation operation)))

(defmethod (indicator :left-tie) () ;; Center of left side of box
  (values x (+ y (round (/ height 2)))))
(defmethod (indicator :right-tie) () ;; Center of left side of box
  (values (+ x width) (+ y (round (/ height 2)))))

;;;; Indicator panels include some number of indicators.

(defflavor indicator-panel ((indicators nil))  ; list of indicators
  (pane) ;; includes a pane
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

(defmethod (indicator-panel :get-indicator) (name  &key (test #'eq))
  (dolist (ind indicators)
    (when (funcall test name (send ind :name)) (return ind))))

(defmethod (indicator-panel :update) ()
  (dolist (ind indicators) (send ind :update)))

(defmethod (indicator-panel :turn-off-indicators) ()
  (dolist (ind indicators) (send ind :set-state :UNLIT)))

;;; There are several possible ways to lay out indicators.
;;; For each technique, the idea is to re-build the underlying
;;; window (never letting it shrink below the viewport size, of course!).
;;; The simplest technique is to arrange the indicators in a single 
;;; column.

(defflavor 1d-indicator-panel ()
  (indicator-panel)
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

(defmethod (1d-indicator-panel :compute-layout) ()
  (let ((i-height 0) (i-pos 0))
    (dolist (ind indicators)
      (incf i-height (send ind :height)))
    (unless (> (send self :bitmap-height) i-height)
      (send self :set-bitmap-height i-height)
      (send self :remake-window))
    ;; Assign a position and new window to each indicator in turn.
    (dolist (ind indicators)
      (send ind :reposition 0 i-pos window)
      (incf i-pos (send ind :height)))))

(defmethod (1d-indicator-panel :center-on-name) (name)
  (let ((ind (send self :get-indicator name)))
    ;; Places the named indicator right in the middle of the
    ;; viewport.
    (setf (viewport-bitmap-y-offset window) 
	  (max 0 (- (+ (send ind :y)
		       (/ (send ind :height) 2)) ;; 1/2 of the indicator's height
		    (/ (region-height (viewport-bitmap-region window)) 2))))))

(defmethod (1d-indicator-panel :center-on) (ind)
  ;; Places the indicator right in the middle of the
  ;; viewport.
  (setf (viewport-bitmap-y-offset window) 
	(max 0 (- (+ (send ind :y)
		     (/ (send ind :height) 2)) ;; 1/2 of the indicator's height
		  (/ (region-height (viewport-bitmap-region window)) 2)))))

;;;; Lines
;; Making lines an explicit object is useful when you want to select or highlight them.

(defflavor explicit-line ((x1 0)
			  (y1 0)
			  (x2 0)
			  (y2 0)
			  (pane nil)
			  (operation boole-1) ;; Operation to use in drawing it by default
			  (width 1) ;; Width of line in pixels
			  (name nil) ;; Backpointer to external object
			  (ticks nil) ;; Drawing commands indicating tic-marks on the line
			  (state :UNLIT)) ;; Is it lit up?
  ()
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

(defmethod (explicit-line :draw) ()
  (send pane :draw-line x1 y1 x2 y2 :width width :operation operation)
  (dolist (tick ticks)
    (eval `(send ,pane ,@ tick)))
  (if (eq state :lit) (send self :ON)))

(defmethod (explicit-line :on) ()
  (unless (eq state :lit)
    (send pane :draw-line x1 y1 x2 y2 :width (+ width 2) :operation boole-xor)
    (send self :set-state :LIT)))

(defmethod (explicit-line :off) ()
  (unless (eq state :unlit)
    (send pane :draw-line x1 y1 x2 y2 :width (+ width 2) :operation boole-xor)
    (send self :set-state :unlit)))

;; For playing around
(defvar foo nil)
(defun make-test-window ()
       (setq foo (make-window :width 1000 :height 1000
			      :viewport-width 200 :viewport-height 200
			      :parent (root-viewport)
			      :title "Test window"
			      :scroll t))
  (hide-viewport foo))

(proclaim '(special *test-panes* *sconfig* *spane*))

(setq *test-panes* '((Top1 pane :title "Top Pane 1" :name 'TOP-PANE1
			    :screen-width 500 :screen-height 100
			    :bitmap-width 500 :bitmap-height 100 :x 0 :y 0 :activate nil
			    :title-font (find-font "SMALL-ROMAN") :font (find-font "SMALL-ROMAN"))
		       (Bottom1 pane :title "Bottom Pane 1" :name 'BOTTOM-PANE1
			    :screen-width 500 :screen-height 100
			    :bitmap-width 500 :bitmap-height 100 :x 0 :y 109 :activate nil
			    :title-font (find-font "SMALL-ROMAN") :font (find-font "SMALL-ROMAN"))
		       (Top2 pane :title "Top Pane 2" :name 'TOP-PANE2
			    :screen-width 500 :screen-height 200
			    :bitmap-width 500 :bitmap-height 200 :x 0 :y 0 :activate nil
			    :title-font (find-font "ITALIC") :font (find-font "MEDIUM-ROMAN"))
		       (Bottom2 pane :title "Bottom Pane 2" :name 'BOTTOM-PANE2
			    :screen-width 500 :screen-height 100
			    :bitmap-width 500 :bitmap-height 100 :x 0 :y 240 :activate nil
			    :title-font (find-font "ITALIC") :font (find-font "MEDIUM-ROMAN"))))

(defvar *test-configs* '((First Top1 Bottom1) (Second Top2 Bottom2)))

(setq *sconfig* '((Only One)))
(setq *spane* '((One pane :title "Test window" :name 'Test-window :screen-width 500 :screen-height 100
		     :bitmap-width 500 :bitmap-width 100 :x 0 :y 0 :activate nil
		     :title-font (find-font "ITALIC") :font (find-font "MEDIUM-ROMAN"))))
