;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: mf; -*-

;;;; MAC/FAC display system for IBM RT

;;; Uses PANES graphical system and EDMODE user interface

(in-package 'mf)

;;; From top to bottom, we have:
;;   title pane
;;        --  holds title, date, and time.
;;   parameter pane
;;       -- which matcher/selectors are chosen, processing status
;;   display 1 area (usually divided into three parts)
;;       -- memory, MAC results, FAC results
;;   display 2 area
;;       -- for showing more detail about aspects of the run
;;   help pane
;;       -- showing current bindings of the function keys

;;; Geometry -- We are going to be quick and dirty here, and just smash everything in
;;; as numerical parameters.  Sorry.

(setq *TITLE-HEIGHT* 50) ;; Estimate
(setq *PARAMETER-LINES* 4) ;; Number of lines in parameter window
(setq *display1-height* 200)
(setq *display2-height* 200)

;;; Cache lots of windows, so that the configuration changing mechanism will keep track of
;;; where things are for us.  (Maybe not the most efficient space-wise, but simplest)

(defvar *right-fudge* 40) ;; Move in from right margin for scroll bars

(setq *macfac-panes* `((Title user::pane :title "MAC/FAC" :title-font (user::find-font "BIG-ROMAN")
			      :name 'Title :x 0 :y 0 :activate nil
			      :screen-width ,user::*screen-width*
			      ;;; Works out to be about 25 high
			      :screen-height 0 :bitmap-width 0 :bitmap-height 0)
		       ;;; Parameters: Say four lines high, by two wide.
		       (Parameters user::1d-Indicator-Panel
				   :title "Parameters" :title-font (user::find-font "MEDIUM-ROMAN")
				   :name 'Parameters :x 0 :y ,*title-height* :activate nil
				   :screen-width ,user::*screen-width* :font (user::find-font "MEDIUM-ROMAN")
				   :screen-height ,(* *parameter-lines*
						      (+ (* 2 user::*indicator-height-offset*)
							 8))
				   :bitmap-width ,user::*screen-width*
				   :bitmap-height ,(* *parameter-lines*
						      (+ (* 2 user::*indicator-height-offset*)
							 8)))
		       (Load-Parameters user::1d-Indicator-Panel
			:title "Where MAC/FAC Components are found."
			:title-font (user::find-font "MEDIUM-ROMAN")
			:name 'Load-Parameters :x 0 :y ,*title-height* :activate nil
			:screen-width ,(- user::*screen-width* *right-fudge*)
					:font (user::find-font "MEDIUM-ROMAN")
			:screen-height ,(* *parameter-lines*
					   (+ (* 2 user::*indicator-height-offset*)
					      8))
			:indicators (list (user::make-instance 'user::indicator
							       :name 'MatchPath
							       :updater #'(lambda () 
									    (format nil "Matcher Pathname = ~A"
										    mf::*matchers-pathname*))
							       :char-width 80)
					  (user::make-instance 'user::indicator
							       :name 'MatchExt
							       :updater #'(lambda () 
									    (format nil "Matcher Extension = ~A"
										    mf::*matchers-extension*))
							       :char-width 80)
					  (user::make-instance 'user::indicator
							       :name 'SelectorPath
							       :updater #'(lambda () 
									    (format nil "Selector Pathname = ~A"
										    mf::*selectors-pathname*))
							       :char-width 80)
					  (user::make-instance 'user::indicator
							       :name 'MatchExt
							       :updater #'(lambda () 
									    (format nil "Selector Extension = ~A"
										    mf::*selectors-pathname*))
							       :char-width 80))
			:bitmap-width ,user::*screen-width*
			:bitmap-height ,(* *parameter-lines*
					   (+ (* 2 user::*indicator-height-offset*)
					      8)))
		       (Dgroup-Parameters user::1d-Indicator-Panel
			:title "Where Dgroups can be found"
			:title-font (user::find-font "MEDIUM-ROMAN")
			:name 'Dgroup-Parameters :x 0 :y ,*title-height* :activate nil
			:screen-width ,(- user::*screen-width* *right-fudge*)
					:font (user::find-font "MEDIUM-ROMAN")
			:screen-height ,(* *parameter-lines*
					   (+ (* 2 user::*indicator-height-offset*)
					      8))
			:indicators (list (user::make-instance 'user::indicator
							       :name 'DgroupPath
							       :updater #'(lambda () 
									    (format nil "Dgroup Pathname = ~A"
										    mf::*dgroups-pathname*))
							       :char-width 80)
					  (user::make-instance 'user::indicator
							       :name 'DgroupExt
							       :updater #'(lambda () 
									    (format nil "Dgroup Extension = ~A"
										    mf::*dgroups-extension*))
							       :char-width 80))
			:bitmap-width ,user::*screen-width*
			:bitmap-height ,(* *parameter-lines*
					   (+ (* 2 user::*indicator-height-offset*)
					      8)))
		       (Help user::Help-Pane
			     :title "Help" :title-font (user::find-font "MEDIUM-ROMAN")
			     :name 'Help :x 0 :y ,(+ *title-height* 12 ;; Three border widths
						     (* *parameter-lines*
							(+ (* 2 user::*indicator-height-offset*)
							   8))
						     *display1-height* *display2-height* 20 50)
			     :activate nil
			     :screen-width ,(- user::*screen-width* *right-fudge*)
			     :font (user::find-font "MEDIUM-ROMAN")
			     :screen-height ,(* 5 (+ (* 2 user::*indicator-height-offset*)
						     (user::font-height (user::find-font "MEDIUM-ROMAN"))))
			     :bitmap-width ,user::*screen-width*
			     :bitmap-height ,(* 5 (+ (* 2 user::*indicator-height-offset*) 4)))
		       (Display1 user::Indicator-Panel 
				 :name 'Display1 :x 0 :y ,(+ *title-height* 20 ;; border width
							     (* *parameter-lines*
								(+ (* 2 user::*indicator-height-offset*)
								   8))) :activate nil
				 :title "Display One" :title-font (user::find-font "MEDIUM-ROMAN") 
				 :screen-width ,(- user::*screen-width* *right-fudge*)
				 :font (user::find-font "MEDIUM-ROMAN")
				 :screen-height ,*display1-height*
				 :bitmap-width ,user::*screen-width* :bitmap-height ,*display1-height*)
		       (Display2 user::Indicator-Panel 
				 :name 'Display2 :x 0 :y ,(+ *title-height* 50 ;; two border widths
							     *display1-height*
							     (* *parameter-lines*
								(+ (* 2 user::*indicator-height-offset*)
								   8))) :activate nil
				 :title "Display Two" :title-font (user::find-font "MEDIUM-ROMAN") 
				 :screen-width ,(- user::*screen-width* *right-fudge*)
				 :font (user::find-font "MEDIUM-ROMAN")
				 :screen-height ,*display2-height*
				 :bitmap-width ,user::*screen-width* :bitmap-height ,*display2-height*)))

(setq *macfac-configurations* '((Basic Title Parameters Display1 Display2 Help)
				(Load Title Load-parameters Display1 Display2 Help)
				(Dgroups Title Dgroup-parameters Display1 Display2 Help)))


(defvar *macfac-frame* nil)

(defun make-macfac-display ()
  (setq *macfac-frame* (user::create-wframe "MAC/FAC Frame" 'Basic *macfac-configurations* *macfac-panes*))
  (user::send (user::send *macfac-frame* :get-pane 'load-parameters) :compute-layout)
  (user::send (user::send *macfac-frame* :get-pane 'dgroup-parameters) :compute-layout)
  (setq *d1* (user::send *macfac-frame* :get-pane 'display1))
  (setq *d2* (user::send *macfac-frame* :get-pane 'display2))
  (setq p (user::send *macfac-frame* :get-pane 'parameters))
  (setq title (user::send *macfac-frame* :get-pane 'title))
  (setq help (user::send *macfac-frame* :get-pane 'help))
  *macfac-frame* ) 

;;;; Setting up MAC/FAC screens and putting them away

(defun setup-macfac (&optional (new-window? nil) &rest arguments)
  (when (or (not (boundp '*macfac-frame*))
	    (null *macfac-frame*)
	    new-window?)
    (if (and (boundp '*macfac-frame*) *macfac-frame*) ;; if old one, retire it
	(user::send *macfac-frame* :deexpose)) 
    (setq *macfac-frame* (make-macfac-display)))
  (user::send help :compute-layout)
  (user::send help :update)
  (user::send *macfac-frame* :expose))

(defun putaway-macfac (&rest arguments)
  (if (and (boundp '*macfac-frame*)
	   *macfac-frame*)
      (user::send *macfac-frame* :deexpose)))

(defun change-macfac-mode (new-mode)
  (user::push-UI-mode new-mode)
  ;; These should be handled by a macfac frame instance, really
  (user::send help :update)
  (user::send help :deexpose)
  (user::send help :expose))

;;;; Editor menu setup 

(defun macfac-ui ()
  (change-macfac-mode "MAC/FAC Toplevel"))

;;; Esc is standard way to exit 

(user::DefInterface-Mode "MAC/FAC Toplevel"
			 mf::setup-macfac
			 mf::putaway-macfac
			 10
			 ("Initialize" g-clear-macfac :F1 "Clears MAC/FAC results.")
			 ("SME" sme::toplevel :F2 "Provides access to SME commands.")
			 ("Loading components" (lambda (ignore)
						 (change-macfac-mode "Load MAC/FAC components"))
			  :F3 "Load matchers and selectors")
			 ("Loading dgroups" (lambda (ignore) (change-macfac-mode "Load MAC/FAC dgroups"))
			  :F4 "Loading dgroups.")
			 ("Wire up MAC/FAC" (lambda (ignore) (change-macfac-mode "MAC/FAC wiring"))
			  :F5 "Choose matchers and selectors for MAC and FAC stages")
			 ("Arrange Memory" (lambda (ignore) (change-macfac-mode "Arrange MAC/FAC Memory"))
			  :F6 "Select what dgroups go into MAC/FAC's memory.")
			 ("Choose Probe" (lambda (ignore) (change-macfac-mode "Select MAC/FAC Probe"))
			  :F7 "Select dgroup to use as memory probe.")
			 ("Run MAC/FAC" (lambda (ignore) (change-macfac-mode "Run MAC/FAC"))
			  :F8 "Runs MAC/FAC on the given memory and probe.")
			 ("Report" (lambda (ignore) (change-macfac-mode "MAC/FAC Report Generation"))
			  :F9 "Records data from current MAC/FAC run.")
			 ("Exit" (lambda (ignore) (user::pop-UI-mode nil)) :F12
			  "Exit MAC/FAC Interface"))   

(defun g-clear-macfac (ignore)
  (clear-macfac)
  (editor::message "MAC/FAC cleared."))

;;; Loading components 

(user::DefInterface-Mode "Load MAC/FAC components"
			 mf::setup-macfac-loading
			 mf::putaway-macfac-loading
			 100
			 ("Pathname for matchers" g-change-matcher-pathname :F1
			  "Change pathname used to load matchers")
			 ("Pathname for selectors" g-change-selector-pathname :F2
			  "Change pathname used to load selectors")
			 ("Extension for matchers" g-change-matcher-extension :F3
			  "Change extension used for matchers")
			 ("Extension for selector" g-change-selector-extension :F4
			  "Change extension used for matchers")
			 ("Load some matchers" g-load-some-matchers :F5 "Load a matcher")
			 ("Load some selectors" g-load-some-selectors :F6 "Load a selector")
			 ("Load all matchers" g-load-all-matchers :F7 "Load all matchers")
			 ("Load all selectors" g-load-all-selectors :F8 "Load all selectors")
			 ("Exit" (lambda (ignore) (user::pop-UI-mode nil)) :F12
			  "Exit Loading Matchers/Selectors Menu."))

(defun setup-macfac-loading (ignore)
  (user::send *macfac-frame* :set-configuration 'Load)
  (user::send (user::send *macfac-frame* :get-pane 'Load-parameters) :update)
  (user::send *macfac-frame* :expose))

(defun putaway-macfac-loading (ignore)
  ;(user::send *macfac-frame* :set-configuration 'Basic)
  )

(defun update-macfac-indicator (pane ind)
  (user::send (user::send (user::send *macfac-frame* :get-pane pane)
			  :get-indicator ind) :update))

(defun g-change-matcher-pathname (ignore)
  (setq *matchers-pathname* (editor::prompt-for-string :prompt "Path for matchers "
						       :default *matchers-pathname*
						     :default-string *matchers-pathname*))
  (update-macfac-indicator 'load-parameters 'MatchPath))

(defun g-change-selector-pathname (ignore)
    (setq *selectors-pathname* (editor::prompt-for-string :prompt "Path for selectors "
							  :default *selectors-pathname*
						     :default-string *selectors-pathname*))
  (update-macfac-indicator 'load-parameters 'MatchExt))

(defun g-change-matcher-extension (ignore)
  (setq *matchers-extension* (editor::prompt-for-string :prompt "File extension for matchers "
							:default *matchers-extension*
							:default-string *matchers-extension*))
  (update-macfac-indicator 'load-parameters 'SelectPath))

(defun g-change-selector-extension (ignore)
    (setq *selectors-extension* (editor::prompt-for-string :prompt "File extension for selectors "
							   :default *selectors-extension*
							:default-string *selectors-extension*))
  (update-macfac-indicator 'load-parameters 'SelectExt))

(defvar *file-pmenu* nil)

(defun g-load-some-matchers (ignore)
  (unless *file-pmenu*
    (setq *file-pmenu* (user::make-instance 'user::1d-popup-scroll-menu
					    :title-font (user::find-font "ITALIC"))))
  (editor::message "Please choose which matchers to load: F1 to select, End when finished.")
  (user::send *file-pmenu* :set-title "Matchers to load:")
  (user::send *file-pmenu* :set-items (mapcar #'(lambda (file)
					    (cons (pathname-name file) file))
					(directory (concatenate 'string *matchers-pathname*
								"*." *matchers-extension*))))
  (user::send *file-pmenu* :set-when-done #'(lambda (files) (load-files-with-message files)))
  (user::send *file-pmenu* :make-choices))

(defun load-files-with-message (files)
  (dolist (file files) 
    (editor::message (format nil "Loading ~A..." file))
    (load file)))

(defun g-load-some-selectors (ignore)
  (unless *file-pmenu*
    (setq *file-pmenu* (user::make-instance 'user::1d-popup-scroll-menu
					    :title-font (user::find-font "ITALIC"))))
  (editor::message "Please choose which selectors to load: F1 to select, End when finished.")
  (user::send *file-pmenu* :set-items (mapcar #'(lambda (file)
					    (cons (pathname-name file) file))
					(directory (concatenate 'string *selectors-pathname*
								"*." *selectors-extension*))))
  (user::send *file-pmenu* :set-when-done #'(lambda (files) 
					        (load-files-with-message files)))
  (user::send *file-pmenu* :make-choices))

(defun g-load-all-matchers (ignore)
  (load-files-matching *matchers-pathname* *matchers-extension*))
(defun g-load-all-selectors (ignore)
  (load-files-matching *selectors-pathname* *selectors-extension*))


(user::DefInterface-Mode "Load MAC/FAC dgroups"
			 mf::setup-macfac-dgroups
			 mf::putaway-macfac-dgroups
			 100
			 ("Pathname for dgroups" g-change-dgroup-pathname :F1
			  "Change pathname to load dgroups from")
			 ("Extension for dgroups" g-change-dgroup-extension :F2
			  "Change extension used for dgroups")
			 ("Load some dgroups" g-load-some-dgroups :F3 "Selectively load dgroups")
			 ("Load all dgroups" g-load-all-dgroups :F4
			  "Load all dgroups which match the current file specifications")
			 ("Exit" (lambda (ignore) (user::pop-UI-mode nil)) :F12
			  "Exit Dgroup Loading Menu."))

(defun setup-macfac-dgroups (ignore)
  (user::send *macfac-frame* :set-configuration 'Dgroups)
  (user::send (user::send *macfac-frame* :get-pane 'Dgroup-parameters) :update))

(defun putaway-macfac-dgroups (ignore) nil)

(defvar *dgroups-pathname* #+KDF-RT "/u/dgroups/"
  #+QRG-RT "/usr/analogy/dgroups/"
  #+UIUC "rube:>macfac>dgroups>" )
(defvar *dgroups-extension* "dgr")

(defun g-change-dgroup-pathname (ignore)
  (setq *dgroups-pathname* (editor::prompt-for-string :prompt "Path for dgroups "
						       :default *dgroups-pathname*
						     :default-string *dgroups-pathname*))
  (update-macfac-indicator 'Dgroup-parameters 'DgroupPath))

(defun g-change-dgroup-extension (ignore)
    (setq *dgroups-extension* (editor::prompt-for-string :prompt "File extension for selectors "
							   :default *dgroups-extension*
							:default-string *dgroups-extension*))
  (update-macfac-indicator 'Dgroup-parameters 'DgroupExt))

(defun g-load-some-dgroups (ignore)
  (unless *file-pmenu*
    (setq *file-pmenu* (user::make-instance 'user::1d-popup-scroll-menu
					    :title-font (user::find-font "ITALIC"))))
  (editor::message "Please choose which dgroups to load: F1 to select, End when finished.")
  (user::send *file-pmenu* :set-items (mapcar #'(lambda (file)
					    (cons (pathname-name file) file))
					(directory (concatenate 'string *dgroups-pathname*
								"*." *dgroups-extension*))))
  (user::send *file-pmenu* :set-when-done #'(lambda (files) 
					        (load-files-with-message files)))
  (user::send *file-pmenu* :make-choices))

(defun g-load-all-dgroups (ignore)
  (load-files-matching *dgroups-pathname* *dgroups-extension*))

;;; YOU ARE HERE

;; What to do next:
;; 1. Make indicators for standard parameters
;; 4. Make special panes for: MEMORY, MAC OUTPUT, FAC OUTPUT, each being 1D indicator panels.
;; 5. Add commands using prompt stuff for parameters and file names.
;; 6. Add special pane for dgroup registry; use it as a menu for selecting contents of memory.
;; 7. Add report generator. 
;; 8. Make 2D-indicator-panel, for dgroup registry.
;; 9. Add ability to scroll on both 1D and 2D indicator panels -- some version of highlighting
;;    plus means of keeping track of what lies in particular directions.
;; 10. Make scrolling be activated by arrow keys in the appropriate UImode. 
