;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: user; -*-

(in-package 'user)

;;;; Plotting SME results

;;; Copyright (c) 1989, Kenneth D. Forbus, University of Illinois.
;;; All rights reserved.

;;   Assumes the rest of my simple graphics system for Lucid.

;;; Here's the layout:
;;; Left hand side has the items from the base, right hand side has target items.
;;; Primary sort order is order, secondary is alphabetical on name.
;;; Match hypotheses are represented by lines between the items.
;;; Gmaps are represented by vertical lines connecting their match hypotheses.

;; :SETUP causes the layout to be computed.
;; :REFRESH causes it to be displayed.
;; :SHOW-GMAP causes a particular gmap only to be displayed.
;; :STEP-GMAPS allows one to go through the gmaps one by one,
;;    using "f" to go forward, "b" to go backward, and "q" to quit.

(defflavor sme-match-display ((legend nil) ;; Lisp-readable identifier, for fetching
			      (base nil)
			      (target nil)
			      (mhs nil)
			      (gmaps nil)
			      ;; The next four are alists of (<item> . <graphics>)
			      (bindicators nil) 
			      (tindicators nil)
			      (mlines nil)
			      (glines nil)
			      (bind-width 0) ;; Max width of base item labels
			      (tind-width 0) ;; Max width of target item labels
			      )
  (indicator-panel)
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables)

(defvar *sme-match-display-width* 800.)
(defvar *sme-viewport-height* 500.)

(defun summarize-dgroup-items (dgr)
  (nconc (mapcar #'(lambda (ent) ;;(format nil "~A:~D" (sme::entity-name ent) 0) debugging
		     (list ent (format nil "~A" (sme::entity-name ent)) 0))
		 (sme::dgroup-entities dgr))
	 (mapcar #'(lambda (exp) ;;(format nil "~A:~D" (sme::expression-name exp) debugging
				 ;;	   (sme::expression-order exp))
		     (list exp (format nil "~A" (sme::expression-name exp)) 
			   (sme::expression-order exp)))
		 (sme::dgroup-expressions dgr))))

(defun sme-match-display-order (entry1 entry2)
  (cond ((< (third entry1) (third entry2)))
	((> (third entry1) (third entry2)) nil)
	(t (string< (second entry1) (second entry2)))))

;;; Setting it up

(defmethod (sme-match-display :setup) ()
   ;; Assumes base, target, mhs, and gmaps are known
  (send self :build-indicators)
  (send self :compute-layout)
  (send self :build-match-hypotheses)
  (send self :build-gmaps))

(defvar *SMD* nil) ;; Current match display

(defun show-sme-match ()
  (setq *SMD* (make-instance 'SME-match-display
			     :base sme::*base* :target sme::*target*
			     :mhs sme::*match-hypotheses*
			     :gmaps sme::*gmaps*))
  (send *SMD* :setup)
  (send *SMD* :refresh))

(defun dump-sme-plot (&optional (file-name nil))
  (when *SMD*
    (unless file-name
      (format t "~% File name for bitmap = ")
      (setq file-name (read-line)))
    (send *SMD* :dump-bitmap file-name)))

(defun hide-sme-match ()
  (when *SMD* (send *SMD* :deexpose))) 

(defun show-plot-dgroups-only ()
  (when *SMD* (send *SMD* :refresh nil nil)))

(defun show-plot-mhs-only ()
  (when *SMD* (send *SMD* :refresh t nil)))

(defun refresh-sme-plot ()
  (when *SMD* (send *SMD* :refresh)))

(defun dump-plot-stages (&optional (path nil) (prefix nil))
  (when *SMD* 
    (unless path
      (format t "~%Path to dump bitmaps to = ")
      (setq path (read-line)))
    (unless prefix
      (format t "~%Brief file prefix for the bitmap files = ")
      (setq prefix (read-line)))
    (show-plot-dgroups-only)
    (dump-sme-plot (concatenate 'string path prefix "dg.lbm"))
    (show-plot-mhs-only)
    (dump-sme-plot (concatenate 'string path prefix "mh.lbm"))
    (refresh-sme-plot)
    (dump-sme-plot (concatenate 'string path prefix "gm.lbm"))
    (hide-sme-match)))

;;;; Making displays for base and target items

(defmethod (sme-match-display :build-indicators) ()
  ;; Assumes base and target are initialized.
  ;; First snarf the items from the base and target dgroups
  (let ((bitems (summarize-dgroup-items (send self :base)))
	(titems (summarize-dgroup-items (send self :target)))
	(bwidth 0) (twidth 0))
    ;; Compute widths for base and target by finding the max of the labels
    (dolist (be bitems)
      (if (> (length (second be)) bwidth) (setq bwidth (length (second be)))))
    (dolist (te titems)
      (if (> (length (second te)) twidth) (setq twidth (length (second te)))))
    ;; Sort them by order and then by alphabetical order
    (setq bitems (sort bitems #'sme-match-display-order))
    (setq titems (sort titems #'sme-match-display-order))
    ;; Now walk down these lists building indicators, inserting
    ;; "null indicators" to take up space between items of different order.
    (do ((binds nil)
	 (tinds nil))
	((and (null bitems) (null titems))
	 (send self :set-bindicators (nreverse binds))
	 (send self :set-tindicators (nreverse tinds))
	 (send self :set-bind-width bwidth)
	 (send self :set-tind-width twidth))
;      (format t "~% Car(Bitems) = ~A, Car(Titems) = ~A"
;	      (car bitems) (car titems))
      (cond ((or (null bitems) 
		 (and titems
		      (> (third (car bitems)) (third (car titems)))))
	     ;; Make a dummy for the base
	     (push (cons (caar titems)
			 (make-match-item-indicator (car titems) twidth self))
		   tinds)
;	     (print "Dummy in the base.")
	     ;; Cache the order with the dummy to ensure order divisions are laid out correctly 
	     (push (cons (third (car titems))
			 (make-dummy-match-item-indicator bwidth self (third (car titems))))
		   binds)
	      (pop titems))
	    ((or (null titems) (< (third (car bitems)) (third (car titems))))
;	     (print "Dummy in the target.")
	     ;; Make a dummy for the target
	     (push (cons (caar bitems)
			 (make-match-item-indicator (car bitems) bwidth self))
		   binds)
	     ;; Cache the order with the dummy to ensure order divisions get laid out properly.
	     (push (cons (third (car bitems))
			 (make-dummy-match-item-indicator twidth self (third (car bitems))))
		   tinds)
	      (pop bitems))
	    (t ;; Same order, so just build indicators in both lists
	     (push (cons (caar bitems)
			 (make-match-item-indicator (car bitems) bwidth self))
		   binds)
	     (push (cons (caar titems)
			 (make-match-item-indicator (car titems) twidth self))
		   tinds)
	     (pop bitems) (pop titems))))))

(defun make-match-item-indicator (entry width panel)
  (make-instance 'indicator
		 :char-width width
		 :height (font-height (send panel :font))
		 :font (send panel :font)
		 :label (second entry)
		 :name (car entry) ;; Provide access
		 :updater #'(lambda () (second entry))))

(defun make-dummy-match-item-indicator (width panel &optional (order 0))
  (declare (ignore order))
  (make-instance 'indicator 
		 :char-width width
		 :height (font-height (send panel :font))
		 :font (send panel :font)
		 :label  "" ;; (format nil "DUMMY:~D" order)
		 :updater #'(lambda () "")))

;;;; Defining indicator layouts

(defvar *horizontal-margin* 5.)

(defmethod (sme-match-display :compute-layout) ()
  ;; Width is fixed, since we don't want to do horizontal scrolling.
  ;; Height of viewport is fixed, but height of window depends on how many
  ;; indicators.  This requires the base and target indicators to be set up
  (let ((bheight 0) (theight 0))
    ;; Set up indicators expected by normal panel operations 
    (send self :set-indicators (nconc (mapcar #'cdr (send self :bindicators))
				      (mapcar #'cdr (send self :tindicators))))
    (dolist (bind bindicators)
      (incf bheight (send (cdr bind) :height)))
    (dolist (tind tindicators)
      (incf theight (send (cdr tind) :height)))
    (send self :set-title (format nil "SME Match Plot: Base = ~A; Target = ~A" base target))
    (send self :set-bitmap-height (max bheight theight))
    (send self :set-bitmap-width *sme-match-display-width*)
    (send self :set-screen-height (min *sme-viewport-height* (max bheight theight)))
    (send self :set-screen-width *sme-match-display-width*)
    (send self :remake-window)
    ;; So with the window set up, position the indicators
    (setq bheight 0)
    (dolist (bind bindicators)
      (send (cdr bind) :set-justification :right)
      (send (cdr bind) :reposition *horizontal-margin* bheight window)
      (incf bheight (send (cdr bind) :height)))
    (setq theight 0)
    (dolist (tind tindicators)
      ;; Figure out how to compute the width for target items most easily
      (send (cdr tind) :set-justification :left)
      (send (cdr tind) :reposition (-  *sme-match-display-width* *horizontal-margin*
				       (* (font-fixed-width (send self :font))
					  (send self :tind-width)) )
	    theight window)
      (incf theight (send (cdr tind) :height)))))

(defmethod (sme-match-display :refresh) (&optional (draw-mhs? t) (draw-gmaps? t))
  (send self :expose)
  (send self :clear-pane)
  (dolist (bind bindicators)
    (send (cdr bind) :draw-label))
  (dolist (tind tindicators)
    (send (cdr tind) :draw-label))
  (send self :draw-order-divisions)
  (when draw-mhs? (send self :draw-mhs))
  (when draw-gmaps? (send self :draw-gms)))

;; Make horizontal divisions for order

(defvar *SMD-order-font* (find-font "SMALL-ROMAN"))

(defun grab-item-order (item)
  (cond ((fixnump item) item) ;; For dealing with dummies
	((sme::entity? item) 0)
	(t (sme::expression-order item))))

(defmethod (sme-match-display :draw-order-divisions) ()
  ;; Draws a dashed line across the display with a note "Order = ~A"
  ;; underneath to indicate order boundaries between the base and target
  ;; items.
  (do ((indicators bindicators (cdr indicators))
       (dheight 1)  ;; Where it is being drawn.
       (order -1)   ;; Current order
       (str "")   ;; The string itself
;;     (inds bindicators) ;; Since dummies have been introduced, either could be used.
       )
      ((null indicators))
      (unless (= order (grab-item-order (caar indicators)))
	(setq order (grab-item-order (caar indicators)))
	(setq str (format nil "Order = ~D" order))
	(setq dheight (- (send (cdar indicators) :y) 1))
	(send self :draw-dashed-horizontal-line 0 screen-width dheight)
	(stringblt window (make-position (- (round (/ screen-width 2))
					    (round (/ (string-width str font) 2)))
					 (+ dheight (font-height *SMD-order-font*) 2))
		   *SMD-order-font* str))))

;;; Adding match hypotheses

(defvar *SMD-mh-width* 1.)

(defmethod (sme-match-display :build-match-hypotheses) (&aux mh-line mh-lines)
  ;; Starts afresh, nuking old stuff, since we want to be able to
  ;; inspect subsets of the mh's at will.
  (dolist (mh mhs)
    (let ((bentry (assoc (sme::mh-base-item mh) bindicators))
	  (tentry (assoc (sme::mh-target-item mh) tindicators)))
      (unless bentry
	(error "Base item of ~A not found in ~A." mh base))
      (unless tentry
	(error "Target item of ~A not found in ~A." mh target))
      (multiple-value-bind (x1 y1)
			   (send (cdr bentry) :right-tie)
	(multiple-value-bind (x2 y2)
			     (send (cdr tentry) :left-tie)
	  (setq mh-line (make-instance 'explicit-line :name mh :x1 x1 :y1 y1
				       :x2 x2 :y2 y2 :width *SMD-mh-width*
				       :pane self))
	  (push (cons mh mh-line) mh-lines)))))
  (send self :set-mlines mh-lines))

(defmethod (sme-match-display :draw-mhs) ()
  (dolist (mentry mlines) (send (cdr mentry) :draw)))

(defmethod (sme-match-display :fetch-mh-line) (mh)
  (let ((entry (assoc (sme::mh-form mh) mlines
		      :test #'(lambda (key e) (equal key (sme::mh-form e))))))
    (cond (entry (values (cdr entry) (car entry)))
	  (t (values nil nil)))))

;;;; Displaying gmaps

(defvar *SMD-gmap-mh-diameter* 5.)
(defvar *SMD-gmap-margins* 30)

(defmethod (sme-match-display :build-gmaps) ()
  ;; Vertical lines, with circles on MH's they include.
  ;; Start and end them at *Gmap-margins* from the base and target items. 
  ;; Space them evenly.  
  (let ((field 0)(start 0)(width 0)(right nil) (left nil)
	(ngmaps (length gmaps))
	(gm-lines nil))
    (setq left (multiple-value-list (send (cdar bindicators) :right-tie)))
    (setq right (multiple-value-list (send (cdar tindicators) :left-tie)))
    (setq field (- (car right) (car left) (* 2 *SMD-gmap-margins*))
	  start (+ (car left) *SMD-gmap-margins*)
	  width (round (/ field (max 1 (1- ngmaps)))))
    ;; Next line generally not needed, but let's be paranoid!
    (send self :set-gmaps (sort (copy-list gmaps) #'(lambda (g1 g2) (< (sme::gm-id g1) (sme::gm-id g2)))))
    (dolist (gm gmaps)
      ;; Want to find lines associated with each mh in the gmap, and make a "tick" entry for it.
      (let ((ticks nil) (x start))
	(dolist (mh (sme::gm-elements gm))
	  (let ((line (send self :fetch-mh-line mh))
		(slope 0)(intercept 0))
	    (unless line
	      (error "No line found for match hypothesis ~A in gmap ~A." mh gm))
	    (setq slope (/ (- (send line :y1) (send line :y2))
			   (- (send line :x1) (send line :x2)))) ;safe, can't be vertical
	    (setq intercept (- (send line :y1) (* slope (send line :x1))))
	    (push `(:DRAW-CIRCLE ,(round x)
				 ,(round (+ (* slope x) intercept))
				 *SMD-gmap-mh-diameter*)
		  ticks)))
	(push (cons gm (make-instance 'explicit-line :name gm :pane self
				      :x1 x :x2 x :y1 0 :y2 (send self :bitmap-height)
				      :ticks ticks))
	      gm-lines))
      (setq start (+ start width)))
    (send self :set-glines (nreverse gm-lines))))

(defmethod (sme-match-display :draw-gms) ()
  (dolist (gentry glines) (send (cdr gentry) :draw)))

;;;; Drawing individual gmaps.

(defmethod (sme-match-display :show-gmap) (gm)
  ;; Assumes the graphics are all set up.
(when gm
  (send self :refresh nil nil) ;; Just draw the indicators
  (unless (member gm gmaps) (error "~A isn't a gmap of plot ~A" gm self))
  (let ((gline (cdr (assoc gm glines)))
	(mline nil))
    ;; Plot the corresponding match hypotheses
    (dolist (mh (sme::gm-elements gm))
      (setq mline (send self :fetch-mh-line mh))
      (unless mline (error "~A of ~A not recognized in ~A"
			   mh gm self))
      (send mline :draw))
    ;; Plot the gmap line itself, just to show what is happening
    (send gline :draw))))

(defmethod (sme-match-display :step-gmaps) ()
  (send self :refresh nil nil)
  (do ((rest gmaps)
       (current nil)
       (last-current nil)
       (done? nil)
       (response nil)
       (previous nil))
      (done? (send self :deexpose) self)
    (cond ((null current)
	   (cond (rest (setq current (car rest) rest (cdr rest)))
		 (previous (setq current (car previous) previous (cdr previous)))
		 (t (setq done? t))))
	  (t (unless (eq last-current current)
	       (send self :show-gmap current)
	       (setq last-current current))
	     (unwind-protect (progn (editor::tty_raw)
				    (setq response (read-char)))
	       (editor::tty_cooked))
	     (cond ((member response '(#\b #\B)) ;; Go backward
		    (when previous
		      (push current rest)
		      (setq last-current current
			    current (car previous)
			    previous (cdr previous))))
		   ((member response '(#\f #\F)) ;; Go forward
		    (when rest
		      (push current previous)
		      (setq last-current current
			    current (car rest)
			    rest (cdr rest))))
		   ((member response '(#\q #\Q)) ;; quit
		    (setq done? t))
		   (t nil))))))

;;;; Dumping the bitmap

(defmethod (sme-match-display :dump-bitmap) (filename)
  (store-bitmap (send self :window) filename))

