(in-package "UCPOP")

(defparameter *tests* nil)

(defstruct problem
  name
  domain
  inits
  goal
  (rank-fun #'rank3)
)

(defun bf-show (prob &optional (display "okanogan:0.0"))
  (when (symbolp prob)
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (record-vcr)
  (multiple-value-bind (plan stat)
      (plan (problem-inits prob) (problem-goal prob)
            :rank-fun (problem-rank-fun prob))            
    (when plan (display-plan plan))
    (display-stat stat))
  (load-vcr "bf-show")
  (play display))

(defun bf-control (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (multiple-value-bind (plan stat)
      (plan (problem-inits prob) (problem-goal prob) 
            :rank-fun (problem-rank-fun prob))
    (when plan (display-plan plan))
    (display-stat stat)
    (values plan stat)))

(defun ie-search (prob)
  (when (symbolp prob) 
    (setf prob (find prob *tests* :key #'problem-name)))
  (funcall (problem-domain prob))
  (multiple-value-bind (plan stat)
      (plan (problem-inits prob) (problem-goal prob)
	    :search-fun #'call-ie :rank-fun (problem-rank-fun prob))
    (when plan (display-plan plan))
    (display-stat stat)
    (values plan stat)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This file contains a set of test problems.  Some of them work,
;;;  and others are intractable for ucpop.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun blocks-world-domain ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  ;; Define step for placing one block on another.
  (define (operator puton)
      :parameters (?X ?Y ?Z)
      :precondition (:and (on ?X ?Z) (clear ?X) (clear ?Y)
			  (:neq ?Y ?Z) (:neq ?X ?Z)
			  (:neq ?X ?Y) (:neq ?X Table))
      :effects  
      ((:effect (:and (on ?X ?Y) (:not (on ?X ?Z))))
       (:effect (clear ?Z)        :when (:neq ?Z Table))
       (:effect (:not (clear ?Y)) :when (:neq ?Y Table)))))

;;;UCPOP(22): (bf-control 'sussman-anomaly)
;;;
;;;Initial  : ((BLOCK A) (BLOCK B) (BLOCK C) (BLOCK TABLE) (ON C A) (ON A TABLE)
;;;            (ON B TABLE) (CLEAR C) (CLEAR B) (CLEAR TABLE))
;;;
;;;Step 1  : (PUTON C TABLE A)      Created 2 
;;;           0  -> (ON C A)            
;;;           0  -> (CLEAR C)           
;;;           0  -> (CLEAR TABLE)       
;;;Step 2  : (PUTON B C TABLE)      Created 3 
;;;           0  -> (ON B TABLE)        
;;;           0  -> (CLEAR B)           
;;;           0  -> (CLEAR C)           
;;;Step 3  : (PUTON A B TABLE)      Created 1 
;;;           0  -> (ON A TABLE)        
;;;           2  -> (CLEAR A)           
;;;           0  -> (CLEAR B)           
;;;
;;;Goal    : (AND (ON B C) (ON A B))
;;;           3  -> (ON B C)            
;;;           1  -> (ON A B)            
;;;Complete!
;;;
;;;UCPOP (Init = 10 ; Goals = 3 ) => Win  (3 steps)     CPU 283      
;;;     Nodes (V = 51  ; Q = 25  ; C = 82  )             Branch 1.4901961 
;;;     Working Unifies: 481                             Bindings added: 202  
;;;NIL

(push (make-problem 
       :name 'sussman-anomaly
       :domain #'blocks-world-domain
       :inits '((block A) (block B) (block C) (block Table)
		(on C A) (on A Table) (on B Table) 
		(clear C) (clear B) (clear Table))
       :goal '(:and (on B C) (on A B)))
      *tests*)

(push (make-problem
       :name 'tower-invert
       :domain #'blocks-world-domain
       :inits '((block A) (block B) (block C) (block Table)
		(on a b) (on b c) (on c table)
		(clear a) (clear table))
       :goal '(:and (on b c) (on c a)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; traveling domain

(defun road-operators ()
  (reset-domain)
  (define (operator drive)
      :parameters (?vehicle ?location1 ?location2)
      :precondition (:and (at ?vehicle ?location1) (road ?location1 ?location2))
      :effects
      ((:effect (:and (at ?vehicle ?location2)
	      (:not (at ?vehicle ?location1))))))
  (define (operator cross)
      :parameters (?vehicle ?location1 ?location2)
      :precondition (:and (at ?vehicle ?location1) (bridge ?location1 ?location2))
      :effects
      ((:effect (:and (at ?vehicle ?location2)
	      (:not (at ?vehicle ?location1)))))))

(defun add-road (a b)
  `((road ,a ,b) (road ,b ,a)))

(defun add-bridge (a b)
  `((bridge ,a ,b) (bridge ,b ,a)))

;;;UCPOP(23): (bf-control (road-test))
;;;
;;;Initial  : ((VEHICLE JACK) (VEHICLE MARK) (PLACE A) (PLACE D) (PLACE G)
;;;            (AT JACK A) (AT MARK A) (BRIDGE A D) (BRIDGE D A) (ROAD D G)
;;;            (ROAD G D))
;;;
;;;Step 1  : (CROSS JACK A D)       Created 4 
;;;           0  -> (AT JACK A)         
;;;           0  -> (BRIDGE A D)        
;;;Step 2  : (CROSS MARK A D)       Created 2 
;;;           0  -> (AT MARK A)         
;;;           0  -> (BRIDGE A D)        
;;;Step 3  : (DRIVE JACK D G)       Created 3 
;;;           4  -> (AT JACK D)         
;;;           0  -> (ROAD D G)          
;;;Step 4  : (DRIVE MARK D G)       Created 1 
;;;           2  -> (AT MARK D)         
;;;           0  -> (ROAD D G)          
;;;
;;;Goal    : (AND (AT JACK G) (AT MARK G))
;;;           3  -> (AT JACK G)         
;;;           1  -> (AT MARK G)         
;;;Complete!
;;;
;;;UCPOP (Init = 11 ; Goals = 3 ) => Win  (4 steps)     CPU 133      
;;;     Nodes (V = 20  ; Q = 7   ; C = 28  )             Branch 1.35      
;;;     Working Unifies: 177                             Bindings added: 43   
;;;NIL

(defun road-test ()
  (make-problem
   :name 'road-test
   :domain #'road-operators
   :inits `((vehicle jack)(vehicle mark) 
			  (place a)(place d)(place g)
			  (at jack a) (at mark a)
			  ,@(add-bridge 'a 'd) ,@(add-road 'd 'g))
   :goal '(:and (at jack g) (at mark g))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Tower of hanoi

(defun hanoi-domain ()
  (reset-domain)
  (define (operator move-disk)
      :parameters ((disk ?disk) ?below-disk ?new-below-disk)
      :precondition (:and (smaller ?disk ?new-below-disk)  ;handles pegs!
			  (:neq ?new-below-disk ?below-disk)
			  (:neq ?new-below-disk ?disk)
			  (:neq ?below-disk ?disk)
			  (on ?disk ?below-disk)
			  (clear ?disk)
			  (clear ?new-below-disk))
      :effects 
      ((:effect (:and (clear ?below-disk)
	      (on ?disk ?new-below-disk)
	      (:not (on ?disk ?below-disk))
	      (:not (clear ?new-below-disk)))))))

;;;UCPOP(24): (bf-control (hanoi 2))
;;;
;;;Initial  : ((SMALLER D1 P1) (SMALLER D2 P1) (SMALLER D1 P2) (SMALLER D2 P2)
;;;            (SMALLER D1 P3) (SMALLER D2 P3) (SMALLER D1 D2) (CLEAR P1)
;;;            (CLEAR P2) (CLEAR D1) (DISK D1) (DISK D2) (ON D1 D2) (ON D2 P3))
;;;
;;;Step 1  : (MOVE-DISK D1 D2 P2)   Created 2 
;;;           0  -> (SMALLER D1 P2)     
;;;           0  -> (ON D1 D2)          
;;;           0  -> (CLEAR D1)          
;;;           0  -> (CLEAR P2)          
;;;           0  -> (DISK D1)           
;;;Step 2  : (MOVE-DISK D2 P3 P1)   Created 1 
;;;           0  -> (SMALLER D2 P1)     
;;;           0  -> (ON D2 P3)          
;;;           2  -> (CLEAR D2)          
;;;           0  -> (CLEAR P1)          
;;;           0  -> (DISK D2)           
;;;Step 3  : (MOVE-DISK D1 P2 D2)   Created 3 
;;;           0  -> (SMALLER D1 D2)     
;;;           2  -> (ON D1 P2)          
;;;           0  -> (CLEAR D1)          
;;;           2  -> (CLEAR D2)          
;;;           0  -> (DISK D1)           
;;;
;;;Goal    : (AND (ON D1 D2) (ON D2 P1))
;;;           3  -> (ON D1 D2)          
;;;           1  -> (ON D2 P1)          
;;;Complete!
;;;
;;;UCPOP (Init = 14 ; Goals = 3 ) => Win  (3 steps)     CPU 184      
;;;     Nodes (V = 32  ; Q = 14  ; C = 50  )             Branch 1.4375    
;;;     Working Unifies: 279                             Bindings added: 84   
;;;NIL

(defun hanoi (n)
  (make-problem 
   :name 'hanoi
   :domain #'hanoi-domain
   :inits
   (let* ((disks (subseq '(d1 d2 d3 d4 d5 d6 d7 d8 d9) 0 n))
	  (sizes (nconc (mapcar #'(lambda (d) `(smaller ,d p1)) disks)
			(mapcar #'(lambda (d) `(smaller ,d p2)) disks)
			(mapcar #'(lambda (d) `(smaller ,d p3)) disks)
			(mapcon
			 #'(lambda (d)
			     (mapcar #'(lambda (d2)
					 `(smaller ,(car d) ,d2))
				     (cdr d)))
			 disks)))
	  (initial (append '((clear p1)(clear p2)(clear d1))
			   (mapcar #'(lambda (d)
				       `(disk ,d)) disks)
			   (maplist
			    #'(lambda (d)
				(if (cdr d)
				    `(on ,(car d) ,(cadr d))
				  `(on ,(car d) p3)))
			    disks))))
     (nconc sizes initial))
   :goal 
   (let* ((disks (subseq '(d1 d2 d3 d4 d5 d6 d7 d8 d9) 0 n)))
     (cons :and (maplist #'(lambda (d)
			     (if (cdr d)
				 `(on ,(car d) ,(cadr d))
			       `(on ,(car d) p1)))
			 disks)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Ferry domain

(defun ferry-domain ()
  (reset-domain)
  (define (operator board)
      :parameters ((auto ?x)(place ?y))
      :precondition (:and (at ?x ?y)(at-ferry ?y)(empty-ferry))
      :effects  
      ((:effect (on ?x ferry))
       (:effect (:not (at ?x ?y)))
       (:effect (:not (empty-ferry)))))
  (define (operator sail)
      :parameters ((place ?x)(place ?y))
      :precondition (:and (at-ferry ?x) (:neq ?x ?y))
      :effects 
      ((:effect (at-ferry ?y))
       (:effect (:not (at-ferry ?x)))))
  (define (operator debark)
      :parameters ((auto ?x)(place ?y))
      :precondition (:and (on ?x ferry)(at-ferry ?y))
      :effects  
      ((:effect (:not (on ?x ferry)))
       (:effect (at ?x ?y))
       (:effect (empty-ferry)))))
  
;;;UCPOP(25): (bf-control 'test-ferry)
;;;
;;;Initial  : ((PLACE A) (PLACE B) (AUTO C1) (AUTO C2) (AT C1 A) (AT C2 A)
;;;            (AT-FERRY A) (EMPTY-FERRY))
;;;
;;;Step 1  : (BOARD C2 A)           Created 3 
;;;           0  -> (AT C2 A)           
;;;           0  -> (AT-FERRY A)        
;;;           0  -> (EMPTY-FERRY)       
;;;           0  -> (AUTO C2)           
;;;           0  -> (PLACE A)           
;;;Step 2  : (SAIL A B)             Created 2 
;;;           0  -> (AT-FERRY A)        
;;;           0  -> (PLACE A)           
;;;           0  -> (PLACE B)           
;;;Step 3  : (DEBARK C2 B)          Created 1 
;;;           3  -> (ON C2 FERRY)       
;;;           2  -> (AT-FERRY B)        
;;;           0  -> (AUTO C2)           
;;;           0  -> (PLACE B)           
;;;Step 4  : (SAIL B A)             Created 6 
;;;           2  -> (AT-FERRY B)        
;;;           0  -> (PLACE B)           
;;;           0  -> (PLACE A)           
;;;Step 5  : (BOARD C1 A)           Created 7 
;;;           0  -> (AT C1 A)           
;;;           6  -> (AT-FERRY A)        
;;;           1  -> (EMPTY-FERRY)       
;;;           0  -> (AUTO C1)           
;;;           0  -> (PLACE A)           
;;;Step 6  : (SAIL A B)             Created 5 
;;;           6  -> (AT-FERRY A)        
;;;           0  -> (PLACE A)           
;;;           0  -> (PLACE B)           
;;;Step 7  : (DEBARK C1 B)          Created 4 
;;;           7  -> (ON C1 FERRY)       
;;;           5  -> (AT-FERRY B)        
;;;           0  -> (AUTO C1)           
;;;           0  -> (PLACE B)           
;;;
;;;Goal    : (AND (AT C1 B) (AT C2 B))
;;;           4  -> (AT C1 B)           
;;;           1  -> (AT C2 B)           
;;;Complete!
;;;
;;;UCPOP (Init = 8  ; Goals = 3 ) => Win  (7 steps)     CPU 2633     
;;;     Nodes (V = 488 ; Q = 153 ; C = 786 )             Branch 1.3135246 
;;;     Working Unifies: 2194                            Bindings added: 362  
;;;NIL

(push (make-problem
       :name 'test-ferry
       :domain #'ferry-domain
       :inits '((place a) (place b) (auto c1) (auto c2)
		(at c1 a)(at c2 a)(at-ferry a)
		(empty-ferry))
       :goal '(:and (at c1 b)(at c2 b)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; molgen domain

(defun molgen-domain ()
  (reset-domain)
  
  ;; steps for building DNA molecules from mRNA
  
  (define (operator reverse-transcribe)
      :parameters (?x)
      :precondition (mRNA ?x)
      :effects 
      ((:effect (connected-cDNA-mRNA ?x))))
  (define (operator separate)
      :parameters (?x)
      :precondition (connected-cDNA-mRNA ?x)
      :effects 
      ((:effect (single-strand ?x))
       (:effect (:not (connected-cDNA-mRNA ?x)))))
  (define (operator polymerize)
      :parameters (?x)
      :precondition (single-strand ?x)
      :effects 
      ((:effect (hair-pin ?x))
       (:effect (:not (single-strand ?x)))))
  (define (operator digest)
      :parameters (?x)
      :precondition (hair-pin ?x)
      :effects 
      ((:effect (double-strand ?x))
       (:effect (:not (hair-pin ?x)))))

  ;; steps for splicing DNA molecules
  (define (operator ligate)
      :parameters (?x ?y)
      :precondition (:neq ?x ?y)
      :effects
      ((:effect (cleavable ?y)
	:when (:and (double-strand ?y) (:eq ?x LINKER)))
       (:effect (:and (contains ?x ?y) (cleavable ?y)
	      (:not (cleaved ?x)) (:not (cleaved ?y)))
	:when (:and (cleaved ?x) (cleaved ?y) (:neq ?x LINKER)))))
       
  (define (operator cleave)
      :parameters (?x)
      :precondition (cleavable ?x)
      :effects
      ((:effect (cleaved ?x))
       (:effect (:not (cleavable ?x)))))
  
  ;; Step for inserting a molecule into an organism
  (define (operator transform)
      :parameters (?x (bacterium ?y))
      :precondition (:and (:neq ?x ?y)
			  (cleavable ?x) ; molecule must be whole
			  (accepts ?x ?y)) ; Is molecule accepted?
      :effects
      ((:effect (contains ?x ?y))
       (:effect (:not (cleavable ?x)))))
  
  ;; purify a culture with an antibiotic
  (define (operator screen)
      :parameters ((bacterium ?x) ?y (antibiotic ?z))
      :precondition (:and (:neq ?x ?y) (:neq ?y ?z) (:neq ?x ?z)
			  (resists ?z ?y)(contains ?y ?x))
      :effects
      ((:effect (pure ?x)))))

;;;UCPOP(30): (bf-control 'rat-insulin)
;;;
;;;Initial  : ((MOLECULE INSULIN-GENE) (MOLECULE E-COLI-EXOSOME)
;;;            (MOLECULE JUNK-EXOSOME) (MOLECULE LINKER) (BACTERIUM E-COLI)
;;;            (BACTERIUM JUNK) (ANTIBIOTIC ANTIBIOTIC-1) (MRNA INSULIN-GENE)
;;;            (CLEAVABLE E-COLI-EXOSOME) (CLEAVABLE JUNK-EXOSOME)
;;;            (ACCEPTS JUNK-EXOSOME JUNK) (ACCEPTS E-COLI-EXOSOME E-COLI)
;;;            (RESISTS ANTIBIOTIC-1 E-COLI-EXOSOME))
;;;
;;;Step 1  : (REVERSE-TRANSCRIBE INSULIN-GENE)   Created 10
;;;           0  -> (MRNA INSULIN-GENE) 
;;;Step 2  : (SEPARATE INSULIN-GENE)   Created 9 
;;;           10 -> (CONNECTED-CDNA-MRNA INSULIN-GENE)
;;;Step 3  : (POLYMERIZE INSULIN-GENE)   Created 8 
;;;           9  -> (SINGLE-STRAND INSULIN-GENE)
;;;Step 4  : (DIGEST INSULIN-GENE)   Created 7 
;;;           8  -> (HAIR-PIN INSULIN-GENE)
;;;Step 5  : (LIGATE LINKER INSULIN-GENE)   Created 6 
;;;           7  -> (DOUBLE-STRAND INSULIN-GENE)
;;;Step 6  : (CLEAVE INSULIN-GENE)   Created 5 
;;;           6  -> (CLEAVABLE INSULIN-GENE)
;;;Step 7  : (CLEAVE E-COLI-EXOSOME)   Created 4 
;;;           0  -> (CLEAVABLE E-COLI-EXOSOME)
;;;Step 8  : (LIGATE INSULIN-GENE E-COLI-EXOSOME)   Created 3 
;;;           5  -> (CLEAVED INSULIN-GENE)
;;;           4  -> (CLEAVED E-COLI-EXOSOME)
;;;Step 9  : (TRANSFORM E-COLI-EXOSOME E-COLI)   Created 2 
;;;           3  -> (CLEAVABLE E-COLI-EXOSOME)
;;;           0  -> (ACCEPTS E-COLI-EXOSOME E-COLI)
;;;           0  -> (BACTERIUM E-COLI)  
;;;Step 10 : (SCREEN E-COLI E-COLI-EXOSOME ANTIBIOTIC-1)   Created 1 
;;;           0  -> (RESISTS ANTIBIOTIC-1 E-COLI-EXOSOME)
;;;           2  -> (CONTAINS E-COLI-EXOSOME E-COLI)
;;;           0  -> (BACTERIUM E-COLI)  
;;;           0  -> (ANTIBIOTIC ANTIBIOTIC-1)
;;;
;;;Goal    : (EXISTS ((BACTERIUM ?YGOAL) (MOLECULE ?XGOAL))
;;;           (AND (CONTAINS INSULIN-GENE ?XGOAL) (CONTAINS ?XGOAL ?YGOAL)
;;;            (PURE ?YGOAL)))
;;;           3  -> (CONTAINS INSULIN-GENE E-COLI-EXOSOME)
;;;           2  -> (CONTAINS E-COLI-EXOSOME E-COLI)
;;;           1  -> (PURE E-COLI)       
;;;           0  -> (BACTERIUM E-COLI)  
;;;           0  -> (MOLECULE E-COLI-EXOSOME)
;;;Complete!
;;;
;;;UCPOP (Init = 13 ; Goals = 3 ) => Win  (10 steps)     CPU 6850     
;;;     Nodes (V = 896 ; Q = 203 ; C = 1255)             Branch 1.2265625 
;;;     Working Unifies: 3176                            Bindings added: 1600 
;;;NIL

(push (make-problem
       :name 'rat-insulin
       :domain #'molgen-domain
       :inits '((molecule insulin-gene) (molecule e-coli-exosome) 
		(molecule junk-exosome) (molecule linker)
		(bacterium e-coli) (bacterium junk)
		(antibiotic antibiotic-1)
		(mRNA insulin-gene)
		(cleavable e-coli-exosome) 
		(cleavable junk-exosome) 
		(accepts junk-exosome junk) 
		(accepts e-coli-exosome e-coli) 
		(resists antibiotic-1 e-coli-exosome))
       :goal '(:exists ((bacterium ?y)(molecule ?x))
	               (:and (contains insulin-gene ?x)
		             (contains ?x ?y)
	                     (pure ?y))))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; robot domain (from Hank Wan [hsw@cs.cmu.edu])
(defun robot-domain ()
  (reset-domain)

  (define (operator pickup)
      :parameters ((object ?x) (location ?loc))
      :precondition (:and (:neq ?x robot) (empty-handed) (at ?x ?loc) (at robot ?loc))
      :effects
      ((:effect (grasping ?x))
       (:effect (:not (empty-handed)))))

  (define (operator drop)
      :parameters ((object ?x))
      :precondition (:and (:neq ?x robot) (grasping ?x))
      :effects
      ((:effect (empty-handed))
       (:effect (:not (grasping ?x)))))

  (define (operator move)
      :parameters ((location ?from) (location ?to))
      :precondition (:and (:neq ?from ?to) (at robot ?from) (connected ?from ?to))
      :effects
      ((:effect (at robot ?to)) 
       (:effect (:not (at robot ?from)))
       (:effect (:and (at ?x ?to) (:not (at ?x ?from)))
	:forall (?x)
	:when (:and (grasping ?x) (object ?x))))))

;;;UCPOP(31): (bf-control 'r-test1)
;;;
;;;Initial  : ((LOCATION RM1) (LOCATION RM2) (OBJECT BOX1) (OBJECT BOX2)
;;;            (OBJECT ROBOT) (CONNECTED RM1 RM2) (CONNECTED RM2 RM1)
;;;            (AT BOX1 RM2) (AT BOX2 RM2) (EMPTY-HANDED) (AT ROBOT RM1))
;;;
;;;Step 1  : (MOVE RM1 RM2)         Created 2 
;;;           0  -> (NOT (GRASPING BOX1))
;;;           0  -> (AT ROBOT RM1)      
;;;           0  -> (CONNECTED RM1 RM2) 
;;;           0  -> (LOCATION RM1)      
;;;           0  -> (LOCATION RM2)      
;;;Step 2  : (PICKUP BOX1 RM2)      Created 3 
;;;           0  -> (EMPTY-HANDED)      
;;;           0  -> (AT BOX1 RM2)       
;;;           2  -> (AT ROBOT RM2)      
;;;           0  -> (OBJECT BOX1)       
;;;           0  -> (LOCATION RM2)      
;;;Step 3  : (MOVE RM2 RM1)         Created 1 
;;;           3  -> (GRASPING BOX1)     
;;;           0  -> (OBJECT BOX1)       
;;;           2  -> (AT ROBOT RM2)      
;;;           0  -> (CONNECTED RM2 RM1) 
;;;           0  -> (LOCATION RM2)      
;;;           0  -> (LOCATION RM1)      
;;;
;;;Goal    : (AT BOX1 RM1)
;;;           1  -> (AT BOX1 RM1)       
;;;Complete!
;;;
;;;UCPOP (Init = 11 ; Goals = 3 ) => Win  (3 steps)     CPU 117      
;;;     Nodes (V = 20  ; Q = 11  ; C = 32  )             Branch 1.55      
;;;     Working Unifies: 157                             Bindings added: 25   
;;;NIL

(push (make-problem 
       :name 'r-test1
       :domain #'robot-domain
       :inits '((location rm1) (location rm2) 
		(object box1) (object box2) (object robot)
		(connected rm1 rm2)
		(connected rm2 rm1)
		(at box1 rm2) (at box2 rm2)
		(empty-handed)
		(at robot rm1))
       :goal '(at box1 rm1))
      *tests*)

(push (make-problem
       :name 'r-test2
       :domain #'robot-domain
       :inits '((location rm1) (location rm2) 
		(object box1) (object box2) (object robot)
		(connected rm1 rm2)
		(connected rm2 rm1)
		(at box1 rm2) (at box2 rm2)
		(empty-handed)
		(at robot rm1))
       :goal '(:and (at box1 rm1) (at box2 rm1)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Monkey domain (from prodigy)

(defun monkey-domain ()			; Comment: adding location caused fail
  (reset-domain)
  
  ;; movement and clinbing
  (define (operator GO-TO)
      :parameters (?x ?y)
      :precondition (:and (:neq ?y ?x) (on-floor) (at monkey ?y))
      :effects ((:effect (at monkey ?x)) 
		(:effect (:not (at monkey ?y)))))
  
  (define (operator CLIMB)
      :parameters (?x)
      :precondition (:and (at box ?x) (at monkey ?x))
      :effects ((:effect (onbox ?x)) 
		(:effect (:not (on-floor)))))
  
  (define (operator PUSH-BOX)
      :parameters (?x ?y)
      :precondition (:and (:neq ?y ?x) (at box ?y) (at monkey ?y) (on-floor))
      :effects ((:effect (:and (at monkey ?x) (:not (at monkey ?y))
			       (at box ?x)    (:not (at box ?y))))))

  ;; getting bananas
  (define (operator GET-KNIFE)
      :parameters (?y)
      :precondition (:and (at knife ?y) (at monkey ?y))
      :effects ((:effect (hasknife)) 
		(:effect (:not (at knife ?y)))))

  (define (operator GRAB-BANANAS)
      :parameters (?y)
      :precondition (:and (hasknife) (at bananas ?y) (onbox ?y))
      :effects ((:effect (hasbananas))))
  
  ;; getting water
  (define (operator PICKGLASS)
      :parameters (?y)
      :precondition (:and (at glass ?y) (at monkey ?y))
      :effects ((:effect (hasglass)) (:effect (:not (at glass ?y)))))
  
  (define (operator GETWATER)
      :parameters (?y)
      :precondition (:and (hasglass)
			  (at waterfountain ?y)
			  (at monkey ?y)
			  (onbox ?y))
      :effects ((:effect (haswater)))))
      
;;;UCPOP(32): (bf-control 'monkey-test1)
;;;
;;;Initial  : ((LOCATION P1) (LOCATION P2) (LOCATION P3) (LOCATION P4)
;;;            (AT MONKEY P1) (ON-FLOOR) (AT BOX P2) (AT BANANAS P3)
;;;            (AT KNIFE P4))
;;;
;;;Step 1  : (GO-TO P4 P1)          Created 5 
;;;           0  -> (ON-FLOOR)          
;;;           0  -> (AT MONKEY P1)      
;;;Step 2  : (GET-KNIFE P4)         Created 6 
;;;           0  -> (AT KNIFE P4)       
;;;           5  -> (AT MONKEY P4)      
;;;Step 3  : (GO-TO P2 P4)          Created 4 
;;;           0  -> (ON-FLOOR)          
;;;           5  -> (AT MONKEY P4)      
;;;Step 4  : (PUSH-BOX P3 P2)       Created 3 
;;;           0  -> (AT BOX P2)         
;;;           4  -> (AT MONKEY P2)      
;;;           0  -> (ON-FLOOR)          
;;;Step 5  : (CLIMB P3)             Created 2 
;;;           3  -> (AT BOX P3)         
;;;           3  -> (AT MONKEY P3)      
;;;Step 6  : (GRAB-BANANAS P3)      Created 1 
;;;           6  -> (HASKNIFE)          
;;;           0  -> (AT BANANAS P3)     
;;;           2  -> (ONBOX P3)          
;;;
;;;Goal    : (HASBANANAS)
;;;           1  -> (HASBANANAS)        
;;;Complete!
;;;
;;;UCPOP (Init = 9  ; Goals = 1 ) => Win  (6 steps)     CPU 850      
;;;     Nodes (V = 66  ; Q = 26  ; C = 103 )             Branch 1.3939394 
;;;     Working Unifies: 875                             Bindings added: 101  
;;;NIL

(push (make-problem
       :name 'monkey-test1
       :domain #'monkey-domain
       :inits '((location p1)(location p2)(location p3)(location p4)
		(at monkey p1)(on-floor)(at box p2)(at bananas p3)
		(at knife p4))
       :goal '(hasbananas))
      *tests*)

(push (make-problem
       :name 'monkey-test2 
       :domain #'monkey-domain
       :inits '((location p1)(location p2)(location p3)
		(location p4)(location p6)
		(at monkey p1)(on-floor)
		(at box p2)
		(at bananas p3)
		(at knife p4)
		(at waterfountain p3)(at glass p6))
       :goal '(:and (hasbananas) (haswater)))
      *tests*)

(push (make-problem
       :name 'monkey-test3
       :domain #'monkey-domain
       :inits '((location p1)(location p2)(location p3)
		(location p4)(location p6)
		(at monkey p1)(on-floor)
		(at box p2)
		(at bananas p3)
		(at knife p4)
		(at waterfountain p5)(at glass p6))
       :goal '(:and (hasbananas) (haswater)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Jscott's briefcase world

(defun briefcase-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  (define (operator mov-b)
      :parameters (?m ?l)
      :precondition (:and (:neq ?m ?l) (at B ?m))
      :effects 
      ((:effect (at b ?l))
       (:effect (:not (at B ?m)))
       (:effect (at P ?l) :when (in P))
       (:effect (:not (at P ?m)) :when (in P))
       (:effect (at D ?l) :when (in D))
       (:effect (:not (at D ?m)) :when (in D))))

  (define (operator take-out)
      :parameters (?x)
      :precondition (:neq ?x B)
      :effects
      ((:effect (:not (in ?x)))))
      
  (define (operator put-in)
      :parameters (?x ?l)
      :precondition (:neq ?x B)
      :effects
      ((:effect (in ?x) :when (:and (at ?x ?l) (at B ?l))))))

;;;UCPOP(33): (bf-control 'get-paid)
;;;
;;;Initial  : ((PLACE HOME) (PLACE OFFICE) (OBJECT P) (OBJECT D) (OBJECT B)
;;;            (AT B HOME) (AT P HOME) (AT D HOME) (IN P))
;;;
;;;Step 1  : (PUT-IN D HOME)        Created 3 
;;;           0  -> (AT D HOME)         
;;;           0  -> (AT B HOME)         
;;;Step 2  : (TAKE-OUT P)           Created 2 
;;;Step 3  : (MOV-B HOME OFFICE)    Created 1 
;;;           3  -> (IN D)              
;;;           0  -> (AT B HOME)         
;;;           2  -> (NOT (IN P))        
;;;
;;;Goal    : (AND (AT B OFFICE) (AT D OFFICE) (AT P HOME))
;;;           1  -> (AT B OFFICE)       
;;;           1  -> (AT D OFFICE)       
;;;           0  -> (AT P HOME)         
;;;Complete!
;;;
;;;UCPOP (Init = 9  ; Goals = 4 ) => Win  (3 steps)     CPU 134      
;;;     Nodes (V = 20  ; Q = 10  ; C = 31  )             Branch 1.5       
;;;     Working Unifies: 278                             Bindings added: 37   
;;;NIL

(push (make-problem
       :name 'get-paid
       :domain #'briefcase-world
       :inits '((place home) (place office) 
		(object p) (object d)(object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at B office) (at D office) (at P home)))
      *tests*)

(push (make-problem
       :name 'get-paid2
       :domain #'briefcase-world
       :inits '((place home) (place office) 
		(object p) (object d) (object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at P home) (at D office) (at B home)))
      *tests*)

(push (make-problem
       :name 'get-paid3
       :domain #'briefcase-world
       :inits '((place home) (place office) (place bank)
		(object p) (object d) (object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at P bank) (at D office) (at B home)))
      *tests*)

(push (make-problem
       :name 'get-paid4
       :domain #'briefcase-world
       :inits '((place home) (place office) (place bank)
		(object p) (object d) (object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at B home) (at D office) (at P bank)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; flat-tire domain (from Stuart Russell)

(defun init-flat-tire ()
  (reset-domain)
  
  (define (operator cuss)
      :effects ((:effect (:not (annoyed)))))
  
  (define (operator open)
      :parameters ((container ?x))
      :precondition (:and (:not (locked ?x)) (:not (open ?x)))
      :effects ((:effect (open ?x))))
  
  (define (operator close)
      :parameters ((container ?x))
      :precondition (open ?x)
      :effects ((:effect (:not (open ?x)))))
  
  (define (operator fetch)
      :parameters (?x (container ?y))
      :precondition (:and (:neq ?x ?y) (in ?x ?y) (open ?y))
      :effects ((:effect (have ?x)) 
		(:effect (:not (in ?x ?y)))))
  
  (define (operator put-away)
      :parameters (?x (container ?y))
      :precondition (:and (:neq ?x ?y) (have ?x) (open ?y))
      :effects ((:effect (in ?x ?y)) 
		(:effect (:not (have ?x)))))
  
  (define (operator loosen)
      :parameters ((nut ?x) (hub ?y))
      :precondition (:and (:neq ?x ?y) (have wrench) (tight ?x ?y) 
			  (on-ground ?y))
      :effects ((:effect (loose ?x ?y)) 
		(:effect (:not (tight ?x ?y)))))
  
  (define (operator tighten)
      :parameters ((nut ?x) (hub ?y))
      :precondition (:and (:neq ?x ?y) (have wrench) (loose ?x ?y) 
			  (on-ground ?y))
      :effects ((:effect (tight ?x ?y))
		(:effect (:not (loose ?x ?y)))))

  (define (operator jack-up)
      :parameters ((hub ?y))
      :precondition (:and (on-ground ?y) (have jack))
      :effects ((:effect (:not (on-ground ?y)))
		(:effect (:not (have jack)))))

  ;; jacking down wheel x on hub y (dependency would be better)
  (define (operator jack-down)
      :parameters ((hub ?x))
      :precondition (:not (on-ground ?x))
      :effects ((:effect (on-ground ?x))
		(:effect (have jack))))
  
  (define (operator undo)
      :parameters ((nut ?x) (hub ?y))
      :precondition (:and (:neq ?x ?y) 
			  (:not (on-ground ?y)) (:not (unfastened ?y))
			  (have wrench) (loose ?x ?y))
      :effects ((:effect (:and (have ?x) (unfastened ?y)
		       (:not (on ?x ?y)) (:not (loose ?x ?y))))))
  
  (define (operator do-up)
      :parameters ((nut ?x) (hub ?y))
      :precondition (:and (:neq ?x ?y)
			  (have wrench) (unfastened ?y)
			  (:not (on-ground ?y)) (have ?x))
      :effects 
      ((:effect (:and (loose ?x ?y) (:not (unfastened ?y)) (:not (have ?x))))))

  (define (operator remove-wheel)
      :parameters ((wheel ?x) (hub ?y))
      :precondition (:and (:neq ?x ?y) (:not (on-ground ?y))
			  (on ?x ?y) (unfastened ?y))
      :effects ((:effect (:and (have ?x) (free ?y) (:not (on ?x ?y))))))
  
  (define (operator put-on-wheel)
      :parameters ((wheel ?x) (hub ?y))
      :precondition (:and (:neq ?x ?y) (have ?x) (free ?y) (unfastened ?y)
			  (:not (on-ground ?y)))
      :effects 
      ((:effect (:and (on ?x ?y) (:not (have ?x)) (:not (free ?y))))))
  
  (define (operator inflate)
      :parameters ((wheel ?x))
      :precondition (:and (have pump) (:not (inflated ?x)) (intact ?x))
      :effects ((:effect (inflated ?x)))))

(push (make-problem
       :name 'fixit
       :domain #'init-flat-tire
       :inits '((wheel wheel1) (wheel wheel2) (hub hub) (nut nuts) 
		(container boot) (intact wheel2)
		(in jack boot) (in pump boot)
		(in wheel2 boot) (in wrench boot) 
		(on wheel1 hub) (on-ground hub) (tight nuts hub))
       :goal '(:and
	       (:not (open boot)) (in jack boot) (in pump boot)
	       (in wheel1 boot)
	       (in wrench boot) (inflated wheel2) (on wheel2 hub) 
	       (tight nuts hub)))
      *tests*)

(push (make-problem
       :name 'fix1
       :domain #'init-flat-tire
       :inits '((wheel wheel1) (wheel wheel2) (hub hub) (nut nuts) 
		(container boot) (intact wheel2)
		(in jack boot) (in pump boot)
		(in wheel2 boot) (in wrench boot) 
		(on wheel1 hub) (on-ground hub) (tight nuts hub))
       :goal '(:and (have jack) (have pump) (have wheel2)
	       (have wrench)))
      *tests*)

(push (make-problem
       :name 'fix2
       :domain #'init-flat-tire
       :inits '((wheel wheel1) (wheel wheel2) (hub hub) (nut nuts) 
		(container boot) (intact wheel2)
		(open boot)
		(have jack) (have pump) (have wheel2) (have wrench)
		(on wheel1 hub) (on-ground hub) (tight nuts hub))
       :goal '(:and
	       (inflated wheel2) (:not (on-ground hub))
	       (loose nuts hub)))
      *tests*)

(push (make-problem
       :name 'fix3
       :domain #'init-flat-tire
       :inits '((wheel wheel1) (wheel wheel2) (hub hub) (nut nuts)
		(container boot) (intact wheel2)
		(have pump) (have wheel2)
		(have wrench) (on wheel1 hub) (inflated wheel2)
		(loose nuts hub))
       :goal '(:and (tight nuts hub) (on-ground hub)
	       (on wheel2 hub) 
	       ))
      *tests*)

(push (make-problem
       :name 'fix4
       :domain #'init-flat-tire
       :inits '((wheel wheel1) (wheel wheel2) (hub hub) (nut nuts)
		(container boot) (intact wheel2)
		(have jack) (have pump) (have wheel1)
		(have wrench) (open boot)
		(inflated wheel2) 
		(on wheel2 hub) 
		(tight nuts hub) (on-ground hub)
		)
       :goal '(:and
	       (in jack boot) (in pump boot) (in wheel1 boot)
	       (in wrench boot) (inflated wheel2) (on wheel2 hub) 
	       (tight nuts hub)))
      *tests*)

(push (make-problem
       :name 'fix5
       :domain #'init-flat-tire
       :inits '((wheel wheel1) (wheel wheel2) (hub hub) (nut nuts) 
		(container boot)
		(open boot) (in jack boot) (in pump boot)
		(in wheel1 boot)
		(in wrench boot) (inflated wheel2) (on wheel2 hub) 
		(tight nuts hub))
       :goal '(:and
	       (:not (open boot)) (in jack boot) (in pump boot)
	       (in wheel1 boot)
	       (in wrench boot) (inflated wheel2) (on wheel2 hub) 
	       (tight nuts hub)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Here lies an encoding of Pednault's IJCAI-91 example
;;
;;  You bought a house and discover that after turning on water,
;;  it pours out of holes in the wall.  Using the three actions from
;;  below, the homeowner's problem is to find a way to have the water
;;  on without having all those holes in the wall.
;;
;;  1) A "Fixing the wall" action is only effective when the
;;     plumbing is good.
;;  2) A "Fixing the plumbing" action is only good when the water
;;     is off.
;;  2) A "Turning the faucet" action will bash the wall only
;;     when the plumbing is bad and when you turn it to "ON".
;;
;;  The first two actions are encoded as (FIX ?it).
;;  The second is (TURN-FAUCET ?how).
;;
;; J. Scott Penberthy 3/92
;;

(defun ho-world ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  ;; a FIX operator -- a handyman can do anything, within limits

  (define (operator fix)
      :parameters ((object ?it))
      :effects
      ((:effect (:not (holey-walls)) 
	:when (:and (:eq ?it wall) (good-plumbing)))
       (:effect (:not (holey-walls)) 
	:when (:and (:eq ?it wall) (:not (good-plumbing)) (water off)))
       (:effect (good-plumbing)
	:when (:and (:eq ?it plumbing) (water off)))))
  
  ;; another operator for turning the water on/off

  (define (operator turn-faucet)
      :parameters (?s ?how)
      :precondition (:neq ?s ?how)
      :effects
      ((:effect (:and (:not (water ?s)) (water ?how))
	:when (water ?s))
       (:effect (holey-walls)
	:when (:and (:eq ?how ON) (:not (good-plumbing)))))))
  
;;;UCPOP(40): (bf-control 'ho-demo)
;;;
;;;Initial  : ((OBJECT WALL) (OBJECT PLUMBING) (HOLEY-WALLS) (WATER ON))
;;;
;;;Step 1  : (TURN-FAUCET ON OFF)   Created 3 
;;;           0  -> (WATER ON)          
;;;Step 2  : (FIX PLUMBING)         Created 2 
;;;           3  -> (WATER OFF)         
;;;           0  -> (OBJECT PLUMBING)   
;;;Step 3  : (TURN-FAUCET OFF ON)   Created 4 
;;;           3  -> (WATER OFF)         
;;;Step 4  : (FIX WALL)             Created 1 
;;;           2  -> (GOOD-PLUMBING)     
;;;           0  -> (OBJECT WALL)       
;;;
;;;Goal    : (AND (WATER ON) (NOT (HOLEY-WALLS)))
;;;           4  -> (WATER ON)          
;;;           1  -> (NOT (HOLEY-WALLS)) 
;;;Complete!
;;;
;;;UCPOP (Init = 4  ; Goals = 3 ) => Win  (4 steps)     CPU 233      
;;;     Nodes (V = 72  ; Q = 16  ; C = 111 )             Branch 1.2222222 
;;;     Working Unifies: 285                             Bindings added: 75   
;;;NIL

(push (make-problem
       :name 'ho-demo
       :domain #'ho-world
       :inits '((object wall) (object plumbing)
		(holey-walls) (water on))
       :goal '(:and (water on) (:not (holey-walls))))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Dan's fridge domain
(defun fridge-domain ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)
  (define (operator unscrew)
      :parameters ((screw ?x) (backplane ?y))
      :precondition (:and (screwed ?X) (holds ?x ?y) )
      :effects 
      ((:effect (:not (screwed ?X)))))
  (define (operator screw)
      :parameters ((screw ?x) (backplane ?y))
      :precondition (:and (:not (screwed ?X)) (holds ?x ?y))
      :effects 
      ((:effect (screwed ?X))))
  (define (operator remove-backplane)
      :parameters ((backplane ?x) ?f ?a ?b ?c ?d)
      :precondition (:and (:neq ?a ?b) (:neq ?a ?c) (:neq ?a ?d)
			  (:neq ?b ?c) (:neq ?b ?d) (:neq ?c ?d)
			  (in-place ?x) (part-of ?x ?f) (:not (fridge-on ?f))
			  (holds ?a ?x)  (holds ?b ?x)  
			  (holds ?c ?x)  (holds ?d ?x)
			  (:not (screwed ?a)) (:not (screwed ?b)) 
			  (:not (screwed ?c)) (:not (screwed ?d)))
      :effects 
      ((:effect (:not (in-place ?X)))))
  (define (operator attach-backplane)
      :parameters ((backplane ?x) ?f ?a ?b ?c ?d)
      :precondition (:and (:neq ?a ?b) (:neq ?a ?c) (:neq ?a ?d)
			  (:neq ?b ?c) (:neq ?b ?d) (:neq ?c ?d)
			  (:not (in-place ?x))
			  (part-of ?x ?f) (:not (fridge-on ?f))
			  (holds ?a ?x)  (holds ?b ?x) 
			  (holds ?c ?x)  (holds ?d ?x)
			  (:not (screwed ?a)) (:not (screwed ?b))
			  (:not (screwed ?c)) (:not (screwed ?d)))
      :effects 
      ((:effect (in-place ?X))))
  (define (operator start-fridge)
      :parameters (?f ?a ?b ?c ?d ?x)
      :precondition (:and (:neq ?a ?b) (:neq ?a ?c) (:neq ?a ?d)
			  (:neq ?b ?c) (:neq ?b ?d) (:neq ?c ?d)
			  (backplane ?x) (in-place ?x) (part-of ?x ?f)
			  (holds ?a ?x)(holds ?b ?x)(holds ?c ?x)(holds ?d ?x)
			  (screwed ?a) (screwed ?b) (screwed ?c) (screwed ?d)
			  (:not (fridge-on ?f)))
      :effects 
      ((:effect (fridge-on ?f))))
  (define (operator stop-fridge)
      :parameters (?f)
      :precondition (fridge-on ?f)
      :effects 
      ((:effect (:not (fridge-on ?f)))))
  (define (operator change-compressor)
      :parameters (?x ?y ?a)
      :precondition (:and (:neq ?x ?y) (backplane ?a) (:not (in-place ?a)) (covers ?a ?x)
			  (compressor ?x) (compressor ?y) 
			  (attached ?x) (:not (attached ?y)))
      :effects 
      ((:effect (:and (:not (attached ?X)) (attached ?y)
		      (:not (covers ?a ?x)) (covers ?a ?y))))))

(push (make-problem
       :name 'fixa
       :domain #'fridge-domain
       :inits '((screw s1) (screw s2) (screw s3) (screw s4) 
		(backplane b1)
		(compressor c1) (compressor c2) (fridge f1)
		(covers b1 c1) (part-of b1 f1)
		(holds s1 b1)  (holds s2 b1)  (holds s3 b1)
		(holds s4 b1)
		(ok c1) (ok c2) (fridge-on f1)
		(screwed s1) (screwed s2) (screwed s3) (screwed s4)
		(in-place b1) (attached c1))
       :goal '(:and (attached c2) (ok c2)))
      *tests*)

(push (make-problem
       :name 'fixb
       :domain #'fridge-domain
       :inits '((screw s1) (screw s2) (screw s3) (screw s4) 
		(backplane b1)
		(compressor c1) (compressor c2) (fridge f1)
		(covers b1 c1) (part-of b1 f1)
		(holds s1 b1)  (holds s2 b1)  (holds s3 b1)
		(holds s4 b1)
		(ok c1) (ok c2) (fridge-on f1)
		(screwed s1) (screwed s2) (screwed s3) (screwed s4)
		(in-place b1) (attached c1))
       :goal '(:and (attached c2) (ok c2) (fridge-on f1)))
      *tests*)


(defun mcd-blocksworld ()
  (reset-domain)
  (define (operator puton)
      :parameters (?x ?y ?d)
      :precondition (:and (:neq ?x ?y) (:neq ?x table) (:neq ?d ?y) 
			  (on ?x ?d) 
			  (:or (:eq ?x Table)
			       (:forall ((block ?b)) (:not (on ?b ?x))))
			  (:or (:eq ?y Table)
			       (:forall ((block ?b)) (:not (on ?b ?y)))))
      :effects
      ((:effect (:and (on ?x ?y) (:not (on ?x ?d))))
       (:effect (above ?x ?c)
	:forall (?c)
	:when (:or (:eq ?y ?c) (above ?y ?c)))
       (:effect (:not (above ?x ?e))
	:forall (?e)
	:when (:and (above ?x ?e) (:neq ?y ?e) (:not (above ?y ?e)))))))
  
;;;UCPOP(41): (bf-control 'mcd-sussman-anomaly)
;;;
;;;Initial  : ((BLOCK A) (BLOCK B) (BLOCK C) (BLOCK TABLE) (ON C A) (ON B TABLE)
;;;            (ON A TABLE))
;;;
;;;Step 1  : (PUTON C TABLE A)      Created 2 
;;;           0  -> (ON C A)            
;;;           0  -> (NOT (ON TABLE C))  
;;;           0  -> (NOT (ON C C))      
;;;           0  -> (NOT (ON B C))      
;;;           0  -> (NOT (ON A C))      
;;;Step 2  : (PUTON B C TABLE)      Created 3 
;;;           0  -> (ON B TABLE)        
;;;           0  -> (NOT (ON TABLE B))  
;;;           0  -> (NOT (ON C B))      
;;;           0  -> (NOT (ON B B))      
;;;           0  -> (NOT (ON A B))      
;;;           0  -> (NOT (ON TABLE C))  
;;;           0  -> (NOT (ON C C))      
;;;           0  -> (NOT (ON B C))      
;;;           0  -> (NOT (ON A C))      
;;;Step 3  : (PUTON A B TABLE)      Created 1 
;;;           0  -> (ON A TABLE)        
;;;           0  -> (NOT (ON TABLE A))  
;;;           2  -> (NOT (ON C A))      
;;;           0  -> (NOT (ON B A))      
;;;           0  -> (NOT (ON A A))      
;;;           0  -> (NOT (ON TABLE B))  
;;;           0  -> (NOT (ON C B))      
;;;           0  -> (NOT (ON B B))      
;;;           0  -> (NOT (ON A B))      
;;;
;;;Goal    : (AND (ON B C) (ON A B))
;;;           3  -> (ON B C)            
;;;           1  -> (ON A B)            
;;;Complete!
;;;
;;;UCPOP (Init = 7  ; Goals = 3 ) => Win  (3 steps)     CPU 400      
;;;     Nodes (V = 54  ; Q = 25  ; C = 101 )             Branch 1.462963  
;;;     Working Unifies: 976                             Bindings added: 163  
;;;NIL

(push (make-problem 
       :name 'mcd-sussman-anomaly
       :domain #'mcd-blocksworld
       :inits '((block a) (block b) (block c) (block Table)
		(on c a) (on b table) (on a table))
       :goal '(:and (on b c) (on a b)))
      *tests*)

(push (make-problem 
       :name 'mcd-tower-invert
       :domain #'mcd-blocksworld
       :inits '((block A) (block B) (block C) (block D) (block E) (block Table)
		(clear a) (on a b) (on b c) (on c d) (on d e)(on e table)
		(clear table))
       :goal '(:and (on b c) (on c d) (on d e) (on e a)))
      *tests*)

(defun uni-bw ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)

  (define (operator mov-b)
    :parameters (?m ?l)
    :precondition (:and (at B ?m) (:neq ?m ?l))
    :effects 
    ((:effect (at b ?l))
     (:effect (:not (at B ?m)))
     (:effect (:and (at ?z ?l)  (:not (at ?z ?m)))
      :forall (?z)
      :when (:and (in ?z) (:neq ?z B))) ))
  
  (define (operator take-out)
      :parameters (?x)
      :precondition (:neq ?x B)
      :effects
      ((:effect (:not (in ?x)))))
      
  (define (operator put-in)
      :parameters (?x ?l)
      :precondition (:neq ?x B)
      :effects
      ((:effect (in ?x) :when (:and (at ?x ?l) (at B ?l))))))

;;;UCPOP(42): (bf-control 'uget-paid)
;;;
;;;Initial  : ((AT B HOME) (AT P HOME) (AT D HOME) (IN P))
;;;
;;;Step 1  : (PUT-IN D HOME)        Created 3 
;;;           0  -> (AT D HOME)         
;;;           0  -> (AT B HOME)         
;;;Step 2  : (TAKE-OUT P)           Created 2 
;;;Step 3  : (MOV-B HOME OFFICE)    Created 1 
;;;           3  -> (IN D)              
;;;           0  -> (AT B HOME)         
;;;           2  -> (NOT (IN P))        
;;;
;;;Goal    : (AND (AT B OFFICE) (AT D OFFICE) (AT P HOME))
;;;           1  -> (AT B OFFICE)       
;;;           1  -> (AT D OFFICE)       
;;;           0  -> (AT P HOME)         
;;;Complete!
;;;
;;;UCPOP (Init = 5  ; Goals = 4 ) => Win  (3 steps)     CPU 150      
;;;     Nodes (V = 23  ; Q = 11  ; C = 44  )             Branch 1.4782609 
;;;     Working Unifies: 254                             Bindings added: 65   
;;;NIL

(push (make-problem 
       :name 'uget-paid
       :domain #'uni-bw
       :inits '((at B home) (at P home) (at D home) (in P) )
       :goal '(:and (at B office) (at D office) (at P home)))
      *tests*)
      
(push (make-problem 
       :name 'uget-paid2
       :domain #'uni-bw
       :inits '((place home)(place office) (object p)(object d)(object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at P home) (at D office) (at B home)))
      *tests*)

(push (make-problem 
       :name 'uget-paid3
       :domain #'uni-bw
       :inits '((place home)(place office)(place bank)
		(object p)(object d)(object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at P bank) (at D office) (at B home)))
      *tests*)

(push (make-problem 
       :name 'uget-paid4
       :domain #'uni-bw
       :inits '((place home)(place office)(place bank)
		(object p)(object d)(object b)
		(at B home) (at P home) (at D home) (in P))
       :goal '(:and (at B home) (at D office) (at P bank)))
      *tests*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sched world

(defun sched-world-init (time i-state &aux (ret nil))
  (setf ret i-state)
  (dotimes (i time)
    (dolist (machine '(polisher roller lathe grinder punch drill-press
                       spray-painter immersion-painter))
      (push `(idle ,machine ,i) ret)))
  (dolist (size '(.1 .2 .3))
    (push `(has-bit ,size) ret))
  (dolist (color '(red black))
    (push `(have-paint-for-immersion ,color) ret)
    (push `(sprayable ,color) ret))
  (dotimes (i time)
    (dotimes (j i)
      (push `(later ,i ,j) ret)))
  ret)

(defun sched-world-domain2 ()
  ;; purge old domain prior to defining a new domain
  (reset-domain)
  
  (define (operator POLISH)
      :parameters (?x)
      :precondition (temperature ?x cold) 
      :effects 
      ((:effect (surface-condition ?x polished))
       (:effect (:not (surface-condition ?x ?oldsurf))
	:forall (?oldsurf)
	:when (:neq ?oldsurf polished))))

  (define (operator ROLL)
      :parameters (?x)
      :effects 
      ((:effect (temperature ?x hot))
       (:effect (shape ?x cylindrical))
       (:effect (painted ?x nil))
       (:effect (surface-condition ?x smooth))
       (:effect (:not (surface-condition ?x ?oldsurf))
	:forall (?oldsurf)
	:when (:neq ?oldsurf smooth))
       (:effect (:not (painted ?x ?oldpaint))
	:forall (?oldpaint)
	:when (:neq ?oldpaint nil))
       (:effect (:not (shape ?x ?oldshape))
	:forall (?oldshape)
	:when (:neq ?oldshape cylindrical))
       (:effect (:not (temperature ?x ?oldtemp))
	:forall (?oldtemp)
	:when (:neq ?oldtemp hot))
       (:effect (:not (has-hole ?x ?oldwidth ?old-orient))
	:forall (?oldwidth ?old-orient))))
		      
  (define (operator LATHE)
      :parameters (?x)
      :effects 
      ((:effect (surface-condition ?x rough))
       (:effect (shape ?x cylindrical))
       (:effect (painted ?x nil))
       (:effect (:not (surface-condition ?x ?oldsurf))
	:forall (?oldsurf)
	:when (:neq ?oldsurf rough))
       (:effect (:not (painted ?x ?oldpaint))
	:forall (?oldpaint)
	:when (:neq ?oldpaint nil))
       (:effect (:not (shape ?x ?oldshape))
	:forall (?oldshape)
	:when (:neq ?oldshape cylindrical))))

  (define (operator GRIND)
      :parameters (?x)
      :effects 
      ((:effect (surface-condition ?x smooth))
       (:effect (painted ?x nil))
       (:effect (:not (surface-condition ?x ?oldsurf))
	:forall (?oldsurf)
	:when (:neq ?oldsurf smooth))
       (:effect (:not (painted ?x ?oldpaint))
	:forall (?oldpaint)
	:when (:neq ?oldpaint nil))))

  (define (operator PUNCH)
      :parameters (?x ?width ?orient)
      :precondition (temperature ?x cold)
      :effects 
      ((:effect (has-hole ?x ?width ?orient))
       (:effect (surface-condition ?x rough))
       (:effect (:not (surface-condition ?x ?oldsurf))
	:forall (?oldsurf)
	:when (:neq ?oldsurf rough))))

  (define (operator DRILL-PRESS)
      :parameters (?x ?width ?orient)
      :precondition (:and (temperature ?x cold)
			  (have-bit ?width))
      :effects 
      ((:effect (has-hole ?x ?width ?orient))))

  (define (operator SPRAY-PAINT)
      :parameters (?x ?paint)
      :precondition (:and (temperature ?x cold)
			  (sprayable ?paint))
      :effects 
      ((:effect (painted ?x ?paint))))

  (define (operator IMMERSION-PAINT)
      :parameters (?x ?t ?paint)
      :precondition (have-paint-for-immersion ?paint)
      :effects 
      ((:effect (painted ?x ?paint)))))

(push (make-problem 
       :name 'sched-test1a
       :domain #'sched-world-domain2
       :inits (sched-world-init 5 '((shape obj-A oblong)
				    (temperature obj-A cold)
				    (surface-condition obj-A rough)
				    (painted obj-A none)
				    (has-hole obj-A 0 nil)
				    (last-scheduled obj-A 0)
				    (shape obj-B cylindrical)
				    (temperature obj-B cold)
				    (surface-condition obj-B smooth)
				    (painted obj-B red)
				    (has-hole obj-B 0 nil)
				    (last-scheduled obj-B 0)))
       :goal '(:and (shape Obj-A cylindrical)
	       (surface-condition Obj-B polished)))
      *tests*)

(push (make-problem 
       :name 'sched-test2a
       :domain #'sched-world-domain2
       :inits (sched-world-init 5 '((shape obj-A rectangular)
				    (temperature obj-A cold)
				    (surface-condition obj-A rough)
				    (painted obj-A blue)
				    (has-hole obj-A .2 front)
				    (last-scheduled obj-A 0)
				    
				    (shape obj-B flat)
				    (temperature obj-B cold)
				    (surface-condition obj-B polished)
				    (painted obj-B nil)
				    (has-hole obj-B 0 nil)
				    (last-scheduled obj-B 0)
				    
				    (shape obj-C oblong)
				    (temperature obj-C cold)
				    (surface-condition obj-C rough)
				    (painted obj-C nil)
				    (has-hole obj-C 0 nil)
				    (last-scheduled obj-C 0)))
       :goal '(:and 
	       (surface-condition Obj-B polished)
	       (surface-condition Obj-A smooth)
	       (shape Obj-A cylindrical)))
      *tests*)

(defun prodigy-bw ()
  (reset-domain)
  (define (operator pick-up)
      :parameters ((object ?ob1))
      :precondition (:and (clear ?ob1) (on-table ?ob1) (arm-empty))
      :effects
      ((:effect (:and (:not (on-table ?ob1))
		      (:not (clear ?ob1))
		      (:not (arm-empty))
		      (holding ?ob1)))))
  (define (operator put-down)
      :parameters ((object ?ob))
      :precondition (holding ?ob)
      :effects
      ((:effect (:and (:not (holding ?ob))
		      (clear ?ob)
		      (arm-empty)
		      (on-table ?ob)))))
  (define (operator stack)
      :parameters ((object ?sob)(object ?sunderob))
      :precondition (:and (holding ?sob) (clear ?sunderob))
      :effects
      ((:effect (:and (:not (holding ?sob))
		      (:not (clear ?sunderob))
		      (clear ?sob)
		      (arm-empty)
		      (on ?sob ?sunderob)))))
  (define (operator unstack)
      :parameters ((object ?sob)(object ?sunderob))
      :precondition (:and (on ?sob ?sunderob) (clear ?sob) (arm-empty))
      :effects
      ((:effect (:and (holding ?sob)
		      (clear ?sunderob)
		      (:not (clear ?sob))
		      (:not (arm-empty))
		      (:not (on ?sob ?sunderob)))))))

(push (make-problem 
       :name 'prodigy-sussman
       :domain #'prodigy-bw
       :inits '((object a) (object b) (object c) 
		(on-table a) (on-table b) (on c a)
		(clear b) (clear c) (arm-empty))
       :goal '(:and (on a b) (on b c)))
      *tests*)

(push (make-problem 
       :name 'prodigy-p22
       :domain #'prodigy-bw
       :inits '((object B) (clear B) (on-table B)
		(object G) (on-table G)
		(object F) (on F G)
		(object E) (on E F)
		(object D) (on D E)
		(object C) (on C D) (clear C)
		(object A) (on-table A) 
		(arm-empty))
       :goal '(:and (on B C) (on-table A) (on F A) (on C D)))
      *tests*)