;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /a/manic/n/u/hucka/Projects/Soar/Interface/Src/RCS/sde.el,v $
;;;; $Id: sde.el,v 0.108 1994/06/08 07:06:23 hucka Exp $
;;;; 
;;;; Description       : Main load file and common functions for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1993 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-el-version "$Revision: 0.108 $"
  "The revision number of sde.el.  The complete RCS id is:
      $Id: sde.el,v 0.108 1994/06/08 07:06:23 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Requirements and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Simple gensym.  Code based on cl.el of Emacs 18.58.
;;;; 5.  Often-used macros.
;;;; 6.  Hash table routines
;;;; 7.  Miscellaneous basic functions
;;;; 8.  Buffer and window handling
;;;; 9.  Search-related functions.
;;;; 10. Basic movement and production interpretation.
;;;; 11. Recordkeeping
;;;; 12. Error and diagnostic handling.
;;;; 13. Parenthesis handling.
;;;; 14. Indentation support.
;;;; 15. Sending, excising, etc., productions and regions of productions.
;;;; 16. Miscellaneous editing commands
;;;; 17. Comment support
;;;; 18. SDE mode
;;;; 19. Help support.
;;;; 20. Load commands for rest of SDE
;;;; 21. Functions for reporting bugs
;;;; 22. Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; ----------------
;;;; 0. Documentation
;;;; ----------------
;;;;
;;;; WARNING: This file redefines the function `describe-mode'.  See the
;;;; definition of `sde-describe-mode' below for details.
;;;;
;;;; This is the main file of the Soar Development Environment (SDE), an
;;;; editing and debugging environment featuring special tools to help manage
;;;; the complexities of writing Soar code and interacting with Soar.
;;;;
;;;; SDE is currently compatible with the following versions of Emacs: FSF
;;;; GNU Emacs 19 and Lucid Emacs 19.  Epoch 4.2 and Emacs 18 are not
;;;; supported, although they were supported at one time.
;;;;
;;;; This file contains the core of SDE, including basic functions, the
;;;; internal database for Soar task files, and the SDE editing mode
;;;; (`sde-mode').  Also contained here are the load instructions for the
;;;; rest of SDE.
;;;;
;;;; Certain function and data definitions that depend on the version of
;;;; Emacs being used are located in the following files:
;;;;
;;;;   FSF GNU Emacs 19:  sde-emacs19.el
;;;;   Lucid Emacs 19:    sde-lemacs.el
;;;;
;;;; The files listed above are loaded first by this file at both compile and
;;;; load time, hence they must not contain dependencies on the contents of
;;;; this file.


;;;----------------------------------------------------------------------------
;;; 1.  Requirements and miscellaneous setup.
;;;     Do not modify these.
;;;----------------------------------------------------------------------------

;; Fabulous hack to figure out where this file is located and add that
;; directory to the load-path.  This allows users simply to issue a
;;      (load "/foo/bar/soar")
;; and automatically get the /foo/bar directory added to their load-path if
;; it's not in load-path already.  Algorithm: look at where this file is
;; being loaded from, look at load-path, and if the current directory isn't
;; on load-path, add it.
;;
;; sde-directory is based on original code from Andy Norman:
;;
;;   From: Andy Norman <ange@hplb.hpl.hp.com>
;;   To: hucka@engin.umich.edu
;;   Subject: Re: How to get path info during load? 
;;   Date: Tue, 28 Jan 92 10:40:28 GMT

(defvar sde-directory nil
  "Directory contain SDE's files.")

(defun sde-directory ()
  "Guess the directory of the file currently being loaded, or return nil."
  (let* ((buf (get-buffer-create " *backtrace*"))
	 (standard-output buf)
	 file)
    (condition-case nil
	(save-excursion
	  (set-buffer buf)
	  (erase-buffer)
	  (backtrace)
	  (goto-char (point-min))
	  (if (re-search-forward "load(\"\\([^\"]+\\)\"" nil t)
	      (setq file (buffer-substring (match-beginning 1) (match-end 1))))
	  (kill-buffer buf))
      (error nil))
    (if (and file (file-name-directory file))
	(directory-file-name (file-name-directory file)))))


;; Now hack user's load-path if necessary.
(let ((path (sde-directory)))
  (if (null path)
      ;; Directory must be on load-path already & user did (load "sde").
      ;; Search for sde.el in one of the load-path directories. 
      (let ((lpath load-path))
	(while (and lpath
		    (not (file-exists-p (expand-file-name "sde.el" (car lpath)))))
	  (setq lpath (cdr lpath)))
	(if lpath
	    (setq sde-directory (car lpath))))
    ;; Found a dir in the load stmt.  Add to load-path if not there already.
    (setq sde-directory path)
    (if (not (member sde-directory load-path))
	(setq load-path (cons path load-path)))))


;; Requirements.  

(require 'cl)				; Common Lisp extensions for Emacs.
(require 'cl-19)			; Insures we get the correct version.
(require 'sde-version)

(eval-when (eval load compile)
  (when (string-lessp emacs-version "19")
    (error "SDE will only work in GNU Emacs 19 or Lucid Emacs 19.")))

(eval-when (eval load compile)
  (when window-system
    (require 'mouse))
  (cond ((string-match "Lucid" emacs-version)
	 (require 'sde-lemacs))
	((not (string-lessp emacs-version "19"))
	 (require 'sde-emacs19))))

(eval-when (eval)
  (proclaim '(optimize (safety 3))))

;; Provide is at the end.


;;;----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;----------------------------------------------------------------------------

;;; Configuration variables.

(defvar sde-file-types
  '("\\.soar$"  "\\.soar5$" "\\.soar6$" "\\.soa$"
    "\\.init\\.soar" "\\.reset.soar")
  "*List of file extensions that should use SDE.")

(defvar sde-inhibit-record-file-data nil
  "*If `nil', SDE will not parse Soar tasks as they are visited in Emacs.
SDE normally parses Soar files to determine the set of files and
productions in the tasks that you edit.  This is used to implement various
facilities such as finding production source code, string replacements across
tasks, context-sensitive operations, and more.  SDE is smart about parsing
files and uses caching and other techniques to try to delay performing a full
parse of a task until the information is really needed.  Unfortunately,
parsing files takes time and depending on your computer, the size of your
task, and your needs, you may want to turn it off by setting this variable
to `t'.  Doing so will turn off the facilities that depend on the task
information.")

(defvar sde-ignore-files-regexp-list
  '("\\.init\\.soar" "\\.reset\\.soar")
  "*List of regexps matching names of files that are not to be tracked.
Normally SDE maintains an internal database of the source files that comprise
the tasks being edited.  Some files, such as .init.soar files, should not be
tracked, for practical reasons.")

(defvar sde-use-multiple-frames
  (and window-system (string-match "19" emacs-version))
  "*If t (default in Emacs 19), use multiple frames if possible.
This is only possible in GNU Emacs 19 and Lucid Emacs 19, which are versions
of Emacs that support multiple screens (called \"frames\" in GNU Emacs 19)
under X windows.  A value of t for this variable means SDE will create a
separate frame/screen for the *output* buffer and for each Soar agent
process.  A value of 'output means SDE will only create a separate frame for
the *output* buffer, not for agent processes, while a value of nil means SDE
will never deliberately create a new frame for *output* or for agents.  Also
see the variable `sde-use-output-buffer'.")

(defvar sde-production-indent-offset 2
  "*Integer value indicating how far to indent production clauses.")

(defvar sde-arrow-indent-offset 0
  "*Integer value indicating how far to indent the arrow in a Soar production.
This value is relative to the indentation of the production as a whole.
Default: 0.  A negative value moves the arrow leftward.  A positive value
indents the arrow further to the right relative to the surrounding clauses.")

(defvar sde-sort-lists t
  "*Non-`nil' means to sort lists of files & productions before using them.
Some SDE operations, such as `sde-find-production-by-lhs' and
`sde-query-replace', display or operate on lists of files and productions.
The lists can become very long for large tasks.  When this flag is non-`nil'
\(the default), SDE will sort them alphabetically before presenting the lists
to the user.  However, this can take time.  To avoid the cost of sorting,
set this flag to `nil'.")

(defvar sde-soar-program "soar"
  "*String indicating the name of the Soar program to run.
This is used by the SDE command `soar' when starting up Soar.  The default
value is the string \"soar\".  It can be the name of a program, or an
absolute pathname (i.e., beginning with a '/' or '~').  If the value of this
variable is not an absolute pathname, the named program is found using normal
shell command searching conventions.")

(defvar sde-soar-starting-directory nil
  "*String indicating the default directory in which to start Soar.  
This is a useful variable to set because Soar looks in the starting directory
for init files.  Unless `\\[soar]' is invoked with a prefix argument, it will
cd to `sde-soar-default-directory' before starting Soar.")

(defvar sde-soar-switches nil
  "*List of strings passed as switches to Soar at start-up time.
As of version 6.1.1, standard Soar does *not* accept command line switches.
This variable is provided for use with modified Soar systems such as
ModSAF/SAFSoar.  Make sure that each individual element in this list
is a separate string.  Example of usage:

  \(setq sde-soar-switches '(\"-sde\" \"-nonet\" \"-terrain\" \"ocean-0001\"))

Also see the documentation for the variable `sde-prompt-for-soar-switches'.")

(defvar sde-prompt-for-soar-switches nil
  "*Whether to prompt for switches to Soar if `sde-soar-switches' is nil.
Normally, the `M-x soar' command will not prompt for switches unless it
is given a prefix argument, because standard Soar does not accept command
line switches.  However, if you wish `M-x soar' to prompt for switches
when `sde-soar-switches' is nil (just as it does for the other variables
such as `sde-soar-starting-directory'), set this variable to t.  This
may be useful if you are running a modified Soar, such as ModSAF/SAFSoar,
and you want to run it repeatedly in the same directory but with different
command line switches each time.")

(defvar sde-soar-use-ptys nil
  "Non-nil if SDE should make Emacs use a pty for communicating with Soar.
Default: nil.  A pseudo-tty is usually not necessary for SDE to communicate
with Soar, and it is more efficient to use pipes, hence you will normally want
this variable to be nil.  However, if you are running a Soar-derived system
such as ModSAF/SAFSoar, you may need to set this to t.")

(defvar sde-soar-defaults-file nil
  "*Pathname to the Soar default productions file (\"default.soar6\").
This is set in the SDE site initialization file, \"sde-site.el\".")

(defvar sde-soar-beep-after-setup nil
  "*Non-nil means beep after Soar has started up and is ready.  Default: nil.")

(defvar sde-soar-use-output-buffer t
  "*Whether output from most Soar commands should be put in a separate buffer.
If this variable is non-nil (the default), output from commands such as
querying commands will be placed in a buffer named *output* instead of being
dumped into the Soar agent buffer.  \(Exception: output from the commands go
and run is always placed in the Soar process buffer.)  If nil, all output
goes to the Soar process buffer.")

(defvar sde-soar-erase-output-buffer nil
  "*Whether to erase the Soar output buffer before each new output appears.
Default is nil.")

(defvar sde-soar-move-point-on-output nil
  "*Whether to move point to the bottom of new output upon each command.
Default is nil, which means that if your cursor is in a Soar agent or output
buffer, but not at the bottom, at the time you issue a command, point will
not be moved.  If this variable is t, then when output appears, the cursor
will be moved to the bottom to follow the output.  The default setting
results in less jerking around of the screen.")

(defvar sde-soar-agent-buffer-defaults
  '((height . 24)
    (width . 80))
  "Association list of default values for Soar agent buffer screens/frames.
This determines the attributes of new frames/screens created for Soar agents.
This is only effective if `sde-use-multiple-frames' is non-nil and you are
running in Emacs 19.22 or later or Lucid Emacs 19.  The value may be set in
your ~/.emacs init file, like this:
     
  (setq sde-soar-agent-buffer-defaults '((width . 80) (height . 55)))
     
In Emacs 19.22, the possible parameters are: 
  height                  -- Height of the window in characters.
  width                   -- Width of the window in characters.
  left                    -- Screen position of the left edge, in pixels.
  top                     -- Screen position of the top edge, in pixels.
  name                    -- Name of the frame.
  auto-raise              -- Whether selecting the frame raises it.
  auto-lower              -- Whether deselecting the frame raises it.
  vertical-scroll-bars    -- Whether frame has a vertical scroll bar.
  horizontal-scroll-bars  -- Whether frame has a horizontal scroll bar.
  border-width            -- The width in pixels of the window border.
  internal-border-width   -- The distance in pixels between text and border.
  menu-bar-lines          -- Number of lines to allocate for a menu bar.
  minibuffer              -- Whether to have a minibuffer in the frame
  cursor-color            -- The color of the text cursor that shows point.
  mouse-color             -- The color of the mouse cursor.
  foreground-color        -- The color of the inside of characters.
  background-color        -- The color of background of text.
  border-color            -- The color for the border of the frame.
  cursor-type             -- The way to display the cursor.
     There are two legitimate values: `bar' and `box'.  
     The value `bar' specifies a vertical bar between characters as the
     cursor.  The value `box' specifies an ordinary black box overlaying
     the character after point; that is the default.
  icon-type               -- Type of icon to use when frame it is iconified.
                             Non-`nil' = bitmap icon, `nil' = text icon.")

(defvar sde-soar-output-buffer-defaults 
  '((height . 20)
    (width . 80)
    (top . 20)
    (left . 20))
  "Association list of default values for Soar output buffer screens/frames.
This determines the attributes of the Soar output buffer screen/frame when
`sde-use-multiple-frames' is non-nil.  This is only effective if
`sde-use-multiple-frames' is non-nil and you are running in Emacs 19.22 or
later or Lucid Emacs 19.  The value may be set in your init file, like this:
     
  (setq sde-soar-output-buffer-defaults '((width . 80) (height . 55)))
     
In Emacs 19.22, the possible parameters are: 
  height                  -- Height of the window in characters.
  width                   -- Width of the window in characters.
  left                    -- Screen position of the left edge, in pixels.
  top                     -- Screen position of the top edge, in pixels.
  name                    -- Name of the frame.
  auto-raise              -- Whether selecting frame raises it.
  auto-lower              -- Whether deselecting frame raises it.
  vertical-scroll-bars    -- Whether frame has a vertical scroll bar.
  horizontal-scroll-bars  -- Whether frame has a horizontal scroll bar.
  border-width            -- The width in pixels of the window border.
  internal-border-width   -- The distance in pixels between text and border.
  menu-bar-lines          -- Number of lines to allocate for a menu bar.
  minibuffer              -- Whether to have a minibuffer in the frame
  cursor-color            -- The color of the text cursor that shows point.
  mouse-color             -- The color of the mouse cursor.
  foreground-color        -- The color of the inside of characters.
  background-color        -- The color of background of text.
  border-color            -- The color for the border of the frame.
  cursor-type             -- The way to display the cursor.
     There are two legitimate values: `bar' and `box'.  
     The value `bar' specifies a vertical bar between characters as the
     cursor.  The value `box' specifies an ordinary black box overlaying
     the character after point; that is the default.
  icon-type               -- Type of icon to use when frame it is iconified.
                             Non-`nil' = bitmap icon, `nil' = text icon.")

(defvar sde-soar-track-cd t
  "*Non-nil (the default) means track cd & pwd commands in the Soar process.")

(defvar sde-soar-input-ring-size 30
  "*Size of Soar process input history ring.")

(defvar sde-soar-input-ring-filter
  '(lambda (str) (not (string-match "\\`\\s *\\'" str)))
  "*Predicate for filtering additions to Soar process input history.
Only inputs answering true to this function are saved on the input
history list.  Default is to save anything that isn't all whitespace.")

(defvar sde-production-name-test-regexp
  "\*\\|chunk-[0-9]+\\|justification-[0-9]+"
  "*Regexp for determining whether a symbol likely is a production name.
It is used by the function that extracts production names near the cursor in
a buffer.  By default this regular expression tests that an extracted name
contains either the character '*' or a string of the form \"chunk-n\" or
\"justification-n\", where \"n\" is an integer.") 

(defvar sde-query-replace-highlight window-system
  "*Non-nil means to highlight words during query replacement.
This sets the value of `query-replace-highlight' in the string search functions
that use `sde-query-replace' and `sde-query-replace-regexp'.  It is turned on
by default when Emacs is running in a windowing environment.")

;; Keymaps

(defvar sde-mode-map nil
  "*Keymap for SDE Mode.")

(defvar sde-find-cmds-map nil
  "*Keymap for find commands in SDE.")

(defvar sde-view-cmds-map nil
  "*Keymap for view commands in SDE.")

(defvar sde-agent-cmds-map nil
  "*Keymap for agent commands in SDE.")

(defvar sde-region-cmds-map nil
  "*Keymap for region commands in SDE.")

(defvar sde-help-cmds-map nil
  "*Keymap for help commands in SDE.")  

(defvar sde-soar-mode-map nil
  "Keymap for Soar Mode.")

(defvar sde-soar-output-mode-map nil
  "Keymap for Soar output buffer.")

(defvar sde-mode-syntax-table nil
  "Syntax table used while in Soar Mode.")

(defvar sde-mode-abbrev-table nil
  "Table of abbreviations for Soar Mode.")

;;; Hook variables

(defvar sde-site-hook nil
  "*Hook run after loading the Soar Development Environment.
This is the place to put site-specific initializations, such as local
menus.")

(defvar sde-load-hook nil
  "*Hook run after loading the Soar Development Environment.
This is a good place to put customizations and key bindings.")

(defvar sde-mode-hook nil
  "*Hook run after starting sde-mode.  
This is a good place to put customizations and key bindings.")

(defvar sde-soar-mode-hook nil
  "*Hook for functions to be run after starting up a Soar process.
Good for customizing the interaction environment.")

(defvar sde-soar-output-mode-hook nil
  "*Hook of functions to be run after the Soar output buffer is created.")

(defvar sde-soar-hook nil
  "*Hook of functions to run after first prompt appears in Soar process buffer.")

(defvar sde-soar-error-hook nil
  "*Hook of functions to be run after an error is shown the Soar error buffer.")

;; Declared here for the compiler but really defined in sde-headers.el.
(defvar sde-header-hooks)
(put   'sde-header-hooks 'variable-documentation
 "*List of functions that insert each line of a file header in Soar files.")

;;; Defaults for optional arguments to commands

(defvar sde-go-args nil
  "*String containing default optional args for \"go\".")

(defvar sde-run-args nil
  "*String containing default optional args for \"run\".")

(defvar sde-matches-args nil
  "*String containing default optional args for \"matches\".")

(defvar sde-ms-args nil
  "*String containing default optional args for \"ms\".")

(defvar sde-firing-counts-args nil
  "*String containing default optional args for \"firing-counts\".")

(defvar sde-print-args nil
  "*String containing default optional args for \"soar-print\".")

(defvar sde-preferences-args nil
  "*String containing default optional args for \"preferences\".")

(defvar sde-list-productions-args nil
  "*String containing default optional args for \"list-productions\".")

(defvar sde-list-chunks-args nil
  "*String containing default optional args for \"list-chunks\".")

(defvar sde-list-justifications-args nil
  "*String containing default optional args for \"list-justifications\".")

(defvar sde-agent-go-args nil
  "*String containing default optional args for \"agent-go\".")

(defvar sde-schedule-args nil
  "*String containing default optional args for \"schedule\".")

(defvar sde-reset-args nil
  "*String containing default optional args for \"reset\".")

(defvar sde-explain-args nil
  "*String containing default optional args for \"explain\".")


;;;----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;----------------------------------------------------------------------------

(defconst sde-running-lemacs (and (string-match "Lucid" emacs-version) t)
  "Non-nil if running Lucid Emacs.")

(defconst sde-running-emacs19
    (and (string-match "\\`19" emacs-version) (not sde-running-lemacs))
  "Non-nil if running FSF GNU Emacs 19.")

(defconst sde-source-modes '(sde-mode)
  "List of mode names in SDE that deal with source files.")

(defconst sde-soar-modes '(sde-soar-mode sde-soar-output-mode)
  "List of mode names in SDE related to interacting with Soar.")

(defconst sde-modes (append sde-source-modes sde-soar-modes)
  "List of all modes that are related to SDE.")

(defvar sde-soar-agents nil
  "Alist of names of Soar agents and their associated buffers.
When running in single-agent mode, this variable is nil.  When running in
multi-agent mode, there are multiple agents and a separate buffer for each
agent, and the first name on the alist is the first agent created.  Format of
elements on the list is \( name .  buffer-name ).")

(defconst sde-soar-prompt-regexp "^Soar\\( agent \\([^>]+\\)\\)?> "
  "Regular expression to recognize prompts in the Soar process buffer.
If Soar is running in multi-agent mode, match-beginning 2 returns the start
of the current agent's name in the prompt string.")

;; In Emacs 18, such regexps wouldn't need the extra \( ... \) delimiters,
;; but they are needed in 19.

(defvar sde-sp-name-regexp "^[ \t]*(sp\\s +\\(\\(\\s_\\|\\sw\\)*\\)"
  "*Regexp to match the name part of a production.  
This must be a regular expression containing a grouping construct such that
SDE code can use \(match-beginning 1) and \(match-end 1) to extract the name
from within a matched string.")

(defvar sde-info-file "sde.info"
  "Name of the SDE info file.")

(defvar sde-soar-info-file "soar6-manual.info"
  "Name of the Soar Manual info file.")

(defvar sde-soar-release-notes-file nil
  "Filename of the release notes for Soar.")

(defvar sde-soar-user-notes-file nil
  "Filename of the user notes for Soar.")

(defvar sde-soar-state-string "no process"
  "String indicating the current status of Soar.  Used in mode lines.")

(defvar sde-commands-obarray (make-vector 103 0)
  "Obarray used for completion support for help involving Soar.")

(defvar sde-variables-obarray (make-vector 103 0)
  "Obarray used for completion support for help involving Soar.")

(defvar sde-topics-obarray (make-vector 211 0)
  "Obarray used for completion support involving SDE and Soar topics.")

(defvar sde-buffer-data nil
  "Buffer-local variable storing a structure of type `sde-buffer'.
Used to record information about a buffer, such as the task associated
with it, the agent associated with, etc.")

;; Things defined in sde-soar-mode.el or elsewhere, but declared here to
;; quiet the optimizing byte compiler.

(defvar sde-soar-buffer-mark)
(defvar make-header-hooks)
(defvar lisp-indent-hook)


;;;----------------------------------------------------------------------------
;;; 6.  Hash table routines
;;;----------------------------------------------------------------------------
;;;
;;; The CL extensions package implements hash tables, but unfortunately I have
;;; found the implementation slower in practice than the following simpler
;;; hash table code.
;;;
;;; Hash tables are implemented simply using the Emacs symbol table facility.
;;; They are equivalent to Emacs "obarrays".  


(defvar sde-hash-table-default-size 307
  "Default hash table size.  Should be a prime number.")


(defun sde-make-hash-table (&optional size)
  "Makes an obarray suitable for use as a hashtable.
SIZE, if supplied, should be a prime number."
  (make-vector (or size sde-hash-table-default-size) 0))


(defsubst sde-make-hash-key (key)
  "Returns a string hash key made from KEY."
  (if (stringp key)
      key
    (prin1-to-string key)))


(defsubst sde-puthash (name ht data)
  "Hashes the string NAME in table HT and sets its data value to DATA."
  (set (intern (sde-make-hash-key name) ht) data))


(defsubst sde-gethash (name ht)
  "Returns the value associated with the hashed string NAME in table HT.
Returns nil if NAME is not in the hash table."
  (let ((sym (intern-soft (sde-make-hash-key name) ht)))
    (and sym (symbol-value sym))))


;; There is actually no way to remove an entry from a hash table.
;; The only thing you can do is make the value nil.

(defsubst sde-remhash (name ht)
  "Nils the value associated with the hashed string NAME in table HT."
  (let ((sym (intern-soft (sde-make-hash-key name) ht)))
    (and sym (set sym nil))))


(defun sde-maphash (fun tbl)
  "Calls function FUN on each key and value in hash table TBL."
  (mapatoms (function 
	     (lambda (sym)
	       (funcall fun sym (symbol-value sym))))
	    tbl))


(defun sde-hash-table-count (ht)
  "Returns a count of the entries in hash table HT."
  (let ((count 0))
    (sde-maphash (function
		  (lambda (key val)
		    (when val
		      (setq count (1+ count)))))
		 ht)
    count))


;;;----------------------------------------------------------------------------
;;; .  Utilities for working with strings.
;;;----------------------------------------------------------------------------

;; Based on bbdb-string-trim in bbdb.el.

(defmacro sde-string-trim (string)
  "Remove leading and trailing whitespace from STRING."
  (let ((str (gensym "SDE")))
    (` (let* (((, str) (, string)))
	 (if (string-match "\\`[ \t\n]+" (, str))
	     (setq (, str) (substring (, str) (match-end 0))))
	 (if (string-match "[ \t\n]+\\'" (, str))
	     (substring (, str) 0 (match-beginning 0))
	     (, str))))))


(defmacro sde-substring (str match-num &optional match-end)
  "Do `substring' on string STR.  MATCH-NUM specifies the subexpression.
If MATCH-NUM is not followed with optional argument MATCH-END, this macro
expands into
    \(substring STR \(match-beginning MATCH-NUM) \(match-end MATCH-NUM))
Otherwise, if optional MATCH-END is supplied, this macro becomes
    \(substring STR \(match-beginning MATCH-NUM) \(match-end MATCH-END))"
  (if match-end
      (` (substring (, str) (match-beginning (, match-num)) (match-end (, match-end))))
      (` (substring (, str) (match-beginning (, match-num)) (match-end (, match-num))))))


(defmacro sde-buffer-substring (num)
  "Expands into \(buffer-substring \(match-beginning NUM) \(match-end NUM))."
  (` (buffer-substring (match-beginning (, num)) (match-end (, num)))))


(defmacro sde-unconcat (string)
  "Break up string at whitespaces into a list of strings and return the list."
  (` (let ((str (sde-string-trim (, string)))
	   (start 0)
	   (new))
       (while (string-match "[ \t]+" str start)
	 (setq new   (cons (substring str start (match-beginning 0)) new)
	       start (match-end 0)))
       (setq new (cons (substring str start) new))
       (nreverse new))))


;; This craziness is necessary when calling `message' with production names.
;; A production name can legally contain `%' characters, leading to
;; errors when such a string is passed to the Elisp `message' function.

(defmacro sde-quote-string (str)
  "Quote all instances of `%' characters in string STR and return the result."
  (` (let ((start 0) 
	   (new ""))
       (while (string-match "%" (, str) start)
	 (setq new (concat new (substring (, str) start (match-beginning 0)) "%%")
	       start (match-end 0)))
       (concat new (substring (, str) start)))))


;;;----------------------------------------------------------------------------
;;; Utilities for working with prefix arguments.
;;;----------------------------------------------------------------------------


(defun sde-prefix-arg-negative-p ()
  "Return non-nil if `current-prefix-arg' is a negative prefix."
  (and current-prefix-arg
       (or (symbolp current-prefix-arg)
	   (< (prefix-numeric-value current-prefix-arg) 0))))


(defun sde-prefix-arg-value ()
  "Return the absolute value of `current-prefix-arg' as a number, or nil."
  (and current-prefix-arg
       (not (symbolp current-prefix-arg))
       (let ((value (prefix-numeric-value current-prefix-arg)))
	 (if (< value 0)
	     (- value)
	     value))))


;;;----------------------------------------------------------------------------
;;; .  Utilities for working with files and directories
;;;----------------------------------------------------------------------------


(defsubst sde-find-file-noselect (file)
  (let ((default-major-mode 'fundamental-mode))
    (find-file-noselect file)))

;; This is kind of ugly.  It won't cons much, and it's not recursive, but
;; it's probably inefficient nevertheless.  Is it better to call substring
;; repeatedly, or walk down the strings letter-by-letter?

(defun sde-relative-pathname (path directory)
  "Make PATH relative to DIRECTORY."
  (setq path (expand-file-name path directory))
  (let* ((dir (expand-file-name directory))
	 (pindex 0)
	 (dindex 0)
	 (plength (length path))
	 (dlength (length dir))
	 (result ""))
    (while (and (< pindex plength)
		(< dindex dlength)
		(char-equal (aref path pindex) (aref dir dindex)))
      (setq pindex (1+ pindex)
	    dindex (1+ dindex)))
    ;; Now at the char where they're different.
    ;; Count how many dirs it takes to get from DIR,
    ;; then append what it takes to get to the PATH.
    (setq dindex (1+ dindex))
    (while (< dindex dlength)
      (if (char-equal (aref dir dindex) ?/)
	  (setq result (concat "../" result)))
      (setq dindex (1+ dindex)))
    ;; There may be 0 or at least 2 slashes.
    (if (string-equal "" result)
	(setq result "./"))
    (concat result (substring path pindex))))


(defsubst sde-directory-parent (dir)
  "Returns the parent directory of DIR.
The parent of \"/\" is taken to be nil for practical reasons."
  (if (string= dir "/")
      nil
    (file-name-directory (directory-file-name dir))))


(defun sde-cd (dir)
  "Make DIR become the current buffer's default directory.  
This function is like Emacs's `cd' but does its work silently."
  (interactive "DChange default directory: ")
  (setq dir (expand-file-name dir))
  (unless (eq system-type 'vax-vms)
    (setq dir (file-name-as-directory dir)))
  (unless (file-directory-p dir)
    (sde-log "Attempt to cd to %s\n" dir)
    (sde-log-error "%s is not a directory" dir))
  (setq default-directory dir))


;;;----------------------------------------------------------------------------
;;; Miscellaneous utility functions and macros.
;;;----------------------------------------------------------------------------


(defun sde-production-name-p (string)
  "Return non-nil if STRING is matched by `sde-production-name-test-regexp'."
  (and string
       (let ((match-data (match-data)))
	 (prog1
	     (string-match sde-production-name-test-regexp string)
	   (store-match-data match-data)))))


(defsubst sde-strip-multi-line-comments ()
  ;; Nuke multi line comments in this buffer, because they interfere
  ;; with scanning for things.
  (goto-char (point-min))
  (let (start end)
    ;; Make sure that both comment delimiters are found.
    (while (setq start (search-forward "#|" nil t)
		 end   (search-forward "|#" nil t))
      (delete-region (- start 2) end))))


(defun sde-search-buffer (buffer regexp &optional start)
  "Search forward in BUFFER for REGEXP, starting from the top.
Optional arg START specifies a starting position.  Returns the value of point
where pattern found, or nil."
  (save-excursion
    (set-buffer buffer)
    (goto-char (or start 1))
    (re-search-forward regexp nil t)))


(defun sde-search-file (filename regexp &optional inhibit-msg)
  "Read FILENAME into Emacs and search for REGEXP.
This reads the raw file, not setting its mode or reading its local vars.
FILENAME must not already be in a buffer somewhere.  Prints a message
unless optional argument INHIBIT-MSG is non-nil."
  (save-excursion
    (let ((kill-buffer-hook nil)	; For Emacs 19, for speed.
	  (buffer (create-file-buffer filename)))
      (set-buffer buffer)
      (erase-buffer)
      (prog1				; Return the result of next form.
	  (if (condition-case ()
		  (insert-file-contents filename t)
		(file-error nil))
	      (progn
		(or inhibit-msg (message "Scanning file %s..." buffer-file-name))
		;; Return the result of the search.
		(re-search-forward regexp nil t))
	    nil)
	(kill-buffer buffer)))))


(defun sde-skip-chars (end)
  "Skip past whitespace, comments, backslashed characters and strings in the
current buffer as long as you are before END.  This does move the point."
  (when (< (point) end)
    (let ((comment (and comment-start (string-to-char comment-start)))
	  (done nil)
	  char)
      (while (and (< (point) end)
		  (not done))
	(skip-chars-forward "\n\t " end)
	(setq char (char-after (point)))
	(cond ((eq char ?\")
	       (forward-sexp))
	      ((eq char comment)
	       (forward-char)
	       (skip-chars-forward "^\n" end))
	      ((eq char ?\\)
	       (forward-char 2))
	      (t
	       (setq done t)))))))


;; Modified from lisp-end-defun-text from ilisp-ext.el

(defun sde-end-sp-text (&optional at-start) 
  ;; Go the end of the text associated with the current sp and return
  ;; point.  The end is the last character before whitespace leading to
  ;; a left paren or ;;; at the left margin unless it is in a string.
  (unless at-start
    (sde-beginning-of-production))
  (let ((point (point))
	(boundary (sde-find-next-start))
	(final (save-excursion
		 (condition-case ()
		     (progn (forward-sexp) (point))
		   (error (point-max))))))
    ;; Find the next line starting at the left margin and then check
    ;; to see if it is in a string. 
    (while (progn
	     (skip-chars-forward "^\"" boundary) ;To the next string
	     (unless (= (point) boundary)	
	       (let ((string-boundary	;Start of next defun
		      (save-excursion
			(if (re-search-forward "^\(\\|^;;;" nil t)
			    (match-beginning 0)
			  (point-max)))))
		 (if (condition-case ()
			 (progn (forward-sexp) t)
		       (error (goto-char string-boundary) nil))
		     (if (>= (point) boundary)
			 ;; Boundary was in string
			 (if (> (point) string-boundary)
			     (progn	;String ended in next defun
			       (goto-char string-boundary)
			       nil)
			   (if (> (setq boundary
					(sde-find-next-start))
				  final)
			       ;; Normal defun
			       (progn (goto-char final) nil)
			     t))
		       t)
		   ;; Unclosed string
		   nil)))))
    (re-search-backward  "^[^; \t\n]\\|^[^;\n][ \t]*[^ \t\n]" point t)
    (end-of-line)
    (skip-chars-backward " \t")
    (if (< (point) point)
	(goto-char point)
      (when (save-excursion
	      (let ((point (point)))
		(beginning-of-line)
		(and comment-start (search-forward comment-start point t))))
	(next-line 1)
	(sde-indent-line))
      (point))))


(defun sde-find-next-start ()
  ;; Find the start of the next line at the left margin that starts with
  ;; something other than whitespace, a -, a \), three or more semicolons, or
  ;; a \( not followed by <, and return point.
  (if (eobp)
      (point-max)
    (save-excursion
      (forward-char)
      (if (re-search-forward "^\\(\\(;;;\\)\\|\\(\([^<]\\)\\|\\([^- ^( \t\n\);]\\)\\)" nil t)
	  (match-beginning 0)
	(point-max)))))


;;;----------------------------------------------------------------------------
;;; 8.  Buffer and window handling
;;;----------------------------------------------------------------------------

;;; NOTE: a large number of the sde-xxx functinos for window handling are
;;; defined either in sde-emacs19.el or sde-lemacs.el.  This permits handling
;;; the different names ("frame" vs "screen") with functions defined at load
;;; time instead of putting a lot of conditionals all over the place.

(defmacro sde-get-buffer-create (name)
  "Create buffer named NAME.
Like `get-buffer-create', but binds `default-major-mode' to `fundamental-mode'.
This is a hack necessary to counteract a problem for users who set their
default major mode to `text-mode' or other.  It prevents Emacs from running
`text-mode-hook' when a buffer is created."
  (` (let ((default-major-mode 'fundamental-mode))
       (get-buffer-create (, name)))))


(defmacro sde-buffer-file-name (buffer)
  "Return the value of variable `buffer-file-name' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) buffer-file-name)))


(defmacro sde-buffer-mode (buffer)
  "Return the value of variable `major-mode' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) major-mode)))


(defmacro sde-buffer-directory (buffer)
  "Return the value of `default-directory' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) default-directory)))


(defmacro sde-buffer-exists-p (buffer)
  "Return non-nil if BUFFER exists and is not killed.
BUFFER can be a buffer or a buffer name."
  (` (if (stringp (, buffer))
	 (and (bufferp (get-buffer (, buffer)))
	      (buffer-name (get-buffer (, buffer))))
	 (and (bufferp (, buffer))
	      (buffer-name (, buffer))))))


(defun sde-window-list (&optional all-frames)
  "Return a list of all windows excluding the minibuffer's windows.  
If optional arg ALL-FRAMES is non-nil, also lists windows on other
frames, including invisible and iconified frames."
  ;; Emacs 19.22's get-buffer-window completely ignores invisible windows,
  ;; so we have to do this the hard way.
  (let* ((first (selected-window))
	 (current first)
	 (windows nil))
    (while (progn
	     (setq windows (cons current windows)
		   current (sde-next-window-any current all-frames))
	     (not (eq current first))))
    windows))


(defun sde-create-buffer-frame (buffer param &rest args)
  "Create new frame/screen for BUFFER with parameters PARAM.
Optional third argument VISIBILITY determines whether the window is shown to
the user immediately\; the three possible values are `t' for visible, `nil'
for invisible, and `icon' for making the new frame iconified.  Default: t.
Optional fourth argument DEDICATED determines whether the frame is dedicated
to showing BUFFER's window\; `t' means yes, `nil' (the default) means no.

The argument PARAM is an alist specifying frame/screen parameters.  Any
parameters not mentioned in ALIST defaults according to the value of the
variable `default-frame-alist' in Emacs 19\; parameters not specified there
default from the standard X defaults file and X resources."
  (when (bufferp buffer)
    (message "Creating %s; one moment please..." (cond (sde-running-lemacs
							"screen")
						       (sde-running-emacs19
							"frame")))
    (let* ((orig-frame (sde-selected-frame))
	   (visibility (if (consp args) (car args) t))
	   (dedicated (if (consp args) (car (cdr args)) nil))
	   (frame (sde-make-frame
		   buffer (append (list (cons 'visibility visibility)) param))))
      (unwind-protect
	  (progn
	    (sde-select-frame frame)
	    (delete-other-windows)
	    (switch-to-buffer buffer)
	    (sde-set-window-dedicated-p (sde-frame-root-window frame) dedicated)
	    (when visibility
	      (sde-raise-frame frame)))
	(sde-select-frame orig-frame)))))


(defun sde-delete-buffer-frame (buffer)
  "Delete frames containing BUFFER.  All existing frames are examined.
Frames are only deleted if they contain one window and that window
is showing the given BUFFER."
  (let ((orig-frame (sde-selected-frame))
	(frames (sde-frame-list)))
    (when (cdr frames)			; If there's at least one frame.
      (unwind-protect
	  (mapcar (function
		   (lambda (frame)
		     (sde-select-frame frame)
		     (if (and (one-window-p)
			      (eq (window-buffer (selected-window)) buffer))
			 (sde-delete-frame frame))))
		  (sde-frame-list))
	(if (sde-frame-live-p orig-frame)
	    (sde-select-frame orig-frame))))))


(defun sde-show-frame (frame)
  "Make the given FRAME visible and raised if it is not already."
  (if (sde-frame-p frame)
      (sde-raise-frame (sde-make-frame-visible frame))))


;; sde-show-buffer is for buffers that are not special, i.e., not the
;; *output* and *error* buffers.  For those there are special functions in
;; sde-soar-mode.el.

(defun sde-show-buffer (buffer &optional switch here)
  "Display BUFFER, preferring a window in which it is already showing.  
If optional arg SWITCH is non-nil, switch to the buffer after displaying it.
If optional arg HERE is non-nil, show the buffer in the selected window
unless it is already being shown in another window.  If BUFFER is already
being shown in a window somewhere, make that window and frame visible if
necessary, and switch to it if SWITCH."
  (when (bufferp buffer)
    (let ((window (sde-buffer-window buffer t)))
      (cond (window
	     ;; There is a window currently displaying the buffer.  
	     ;; Make sure it's visible and switch to it if desired.
	     (when (or sde-running-lemacs sde-running-emacs19)
	       (sde-show-frame (sde-buffer-frame buffer)))
	     (when switch
	       (select-window window)
	       (set-buffer buffer)))
	    ;; There is no window displaying the buffer.
	    (here
	     (switch-to-buffer buffer))
	    (switch
	     (pop-to-buffer buffer)
	     (set-buffer buffer))
	    (t
	     (display-buffer buffer))))))


(defun sde-show-buffer-at-point (buffer point)
  "Display BUFFER with the window centered at line containing POINT.
This function will prefer a window in which BUFFER is already showing."
  (unless point
    (error "sde-show-buffer-at-point:  null location."))
  (sde-pop-to-buffer buffer)
  (goto-char point)
  (beginning-of-line)
  (recenter '(center)))


(defun sde-get-mouse-frame ()
  "Returns the frame in which the mouse cursor is located."
  (first (mouse-position)))


(defvar sde-last-buffer nil
  "The last used Soar buffer.")

(defun sde-pop-to-buffer (buffer)
  "Display BUFFER in a different window and switch to that window.
This function sets `sde-last-buffer' to the current buffer."
  (unless (equal (current-buffer) sde-last-buffer)
    (setq sde-last-buffer (current-buffer)))
  (when (sde-buffer-exists-p buffer)
    (sde-show-buffer buffer t)))


(defmacro sde-get-sde-buffers ()
  "Return a list of buffers that are SDE source mode buffers."
  (` (let ((buffers (buffer-list))
	   result)
       (while buffers
	 (if (memq (sde-buffer-mode (car buffers)) sde-source-modes)
	     (push (car buffers) result))
	 (setq buffers (cdr buffers)))
       (nreverse result))))


(defun sde-count-buffer-lines (buffer)
  "Count number of lines in BUFFER.
This version is simpler and somewhat faster than the standard Emacs
function, but is also potentially less accurate."
  (save-excursion			; This counts the number of lines in
    (set-buffer buffer)			;  buffer.  This is what Emacs'
    (goto-char 1)			;  `count-lines' essentially does.
    (- (buffer-size) (forward-line (buffer-size)))))


(defun sde-update-mode-lines ()
  ;; Updates mode lines.  Hackacious.
  (save-window-excursion
    (set-buffer-modified-p (buffer-modified-p))
    ;; Unless the following kind of hack is made, mode lines dont't seem to
    ;; get updated properly in 18.58.  Bug in Emacs?
    (select-window (selected-window))
    (sit-for 0)))			; Update mode lines.


;; Hacked from `save-some-buffers' from files.el in Emacs 19.22.
;; I don't fully understand what this is doing, but it seems to work as
;; desired.

(defvar view-exit-action)		; Declaration for the compiler.

(defun sde-save-sde-buffers (&optional buffers no-questions exiting)
  "Saves modified SDE source file-visiting buffers.
Optional argument BUFFERS specifies the list of buffers to use\; default
is all buffers that are in an SDE source mode.  Asks user about each one.
Optional second argument NO-QUESTIONS non-`nil' means save all with no
questions.  Optional second argument EXITING means ask about certain non-file
buffers as well as about file buffers."
  (interactive "P")
  (save-window-excursion
    (map-y-or-n-p
     (function
      (lambda (buffer)
	(and (buffer-modified-p buffer)
	     (or
	      (buffer-file-name buffer)
	      (and exiting
		   (progn
		     (set-buffer buffer)
		     (and buffer-offer-save (> (buffer-size) 0)))))
	     (if no-questions
		 t
	       (if (buffer-file-name buffer)
		   (format "Save file %s? "
			   (buffer-file-name buffer))
		 (format "Save buffer %s? "
			 (buffer-name buffer)))))))
     (function
      (lambda (buffer)
	(set-buffer buffer)
	(save-buffer)))
     (or buffers (sde-get-sde-buffers))
     '("buffer" "buffers" "save")
     (list (list ?\C-r (lambda (buf)
			 (view-buffer buf)
			 (setq view-exit-action
			       '(lambda (ignore)
				  (exit-recursive-edit)))
			 (recursive-edit)
			 ;; Return nil to ask about BUF again.
			 nil)
		 "display the current buffer"))
     )))


;;;----------------------------------------------------------------------------
;;; 12. Error and diagnostic handling.
;;;----------------------------------------------------------------------------

;;; Log facility based on dired.el from Emacs 19.24.

(defvar sde-log-buffer "*SDE log*")

(defun sde-why ()
  "Pop up a buffer with error log output from SDE.
A group of errors from a single command ends with a formfeed.
Thus, use \\[backward-page] to find the beginning of a group of errors."
  (interactive)
  (if (get-buffer sde-log-buffer)
      (let ((owindow (selected-window))
	    (window (display-buffer (get-buffer sde-log-buffer))))
	(unwind-protect
	    (progn
	      (select-window window)
	      (goto-char (point-max))
	      (recenter -1))
	  (select-window owindow)))))

(defun sde-log (log &rest args)
  ;; Log a message or the contents of a buffer.
  ;; If LOG is a string and there are more args, it is formatted with
  ;; those ARGS.  Usually the LOG string ends with a \n.
  ;; End each bunch of errors with (sde-log t): this inserts
  ;; current time and buffer, and a \f (formfeed).
  (let ((obuf (current-buffer)))
    (unwind-protect			; want to move point
	(progn
	  (set-buffer (get-buffer-create sde-log-buffer))
	  (goto-char (point-max))
	  (let (buffer-read-only)
	    (cond ((stringp log)
		   (insert (if args
			       (apply (function format) log args)
			     log)))
		  ((bufferp log)
		   (insert-buffer log))
		  ((eq t log)
		   (insert "\n" (current-time-string) "\n\f\n")))))		   
      (set-buffer obuf))))


(defun sde-log-message-aux (type string &optional failures)
  ;; Logs multiple messages to the log buffer, and prints a message.  TYPE
  ;; should be either 'error or 'message.  If 'error, then an error is
  ;; signaled and STRING is printed in the echo area.  If 'message, STRING is
  ;; printed to the echo area without an error signal.
  (let ((why-key (or (sde-where-is-internal 'sde-why nil t)
		     (substitute-command-keys "\\<sde-mode-map>\\[sde-why]"))))
    (sde-log (concat "\n" string))
    (sde-log t)
    (funcall (if (eq type 'error)
		 'error
	       'message)
	     (if failures
		 "%s--type `%s' for details (%s)"
	       "%s--type `%s' for details")
	     string why-key failures)))

(defun sde-log-message (string &optional failures)
  ;; Logs muliple messages to the log buffer.
  (sde-log-message-aux 'message string failures))

(defun sde-log-error (string &optional failures)
  ;; Logs muliple messages to the log buffer.
  (sde-log-message-aux 'error string failures))

;;; Functions for common error situations.

(defmacro sde-error-unless-site-var-set (var)
  (` (unless (symbol-value (, var))
       (sde-log "Variable `%s' should have been set to an appropriate\n\
value by your SDE site maintainer, but it was not.  Contact your maintainer.\n" (, var))
       (sde-log-error (format "Variable `%s' not set" (, var))))))


;;;----------------------------------------------------------------------------
;;; 10. Basic movement and production interpretation.
;;;----------------------------------------------------------------------------

;; It turns out that `scan-sexps' and related functions don't look at the
;; actual characters in matching delimiters -- they just count any open/close
;; delimiters.  That's unbelievable.  It means they will treat "( foo ]" as a
;; balanced expression.  That can be deadly when trying to extract
;; productions from buffers when the productions have unbalanced delimiters
;; (e.g., stray braces).  The hack below works around the problem.  It's
;; needed by `sde-end-of-production', but not `sde-beginning-of-production', because the
;; latter only searches backward for a particular string and ever actually
;; calls `forward-sexp' or other functions that rely on `scan-sexps'.  The
;; idea is to completely ignore [, ], {, and } characters, and only look at
;; parentheses.  That way, if a production contains a stray '{', etc., SDE
;; will still be able to extract properly, and Soar will get a chance to
;; signal a syntax error.  Without this, stray { and other characters in
;; productions really screw up SDE upon a `sde-send-production'.

(defmacro sde-with-hacked-delimiter-syntax (&rest forms)
  ;; Temporarily reset the syntax of '{' '}' and '[' ']' so that they
  ;; are considered whitespace, execute FORMS, and reset the syntax.
  (let ((otable (gensym "SDE"))
	(table (gensym "SDE")))
    (` (let (((, otable) (, (copy-syntax-table (syntax-table)))))
	 (unwind-protect
	     (let (((, table) (syntax-table)))
	       (modify-syntax-entry ?\{ " " (, table))
	       (modify-syntax-entry ?\} " " (, table))
	       (modify-syntax-entry ?\[ " " (, table))
	       (modify-syntax-entry ?\] " " (, table))
	       (,@ forms))
	   (set-syntax-table (, otable)))))))


;; Modified from Emacs 19's beginning-of-defun from lisp.el.  

(defun sde-beginning-of-production (&optional count stay)
  "Move backward to next beginning of an sp form and also return point.
Optional first argument COUNT means do it that many times.
Optional second argument STAY means don't move if already at the beginning
of an sp form.  Returns point unless search stops due to end of buffer."
  (interactive "p")
  (when (and count (< count 0))
    (forward-char 1))
  (unless (and stay (looking-at "^\\s(sp"))
    (when (re-search-backward (if (memq major-mode sde-soar-modes)
				  (concat "^\\s(\\|"
					  "\\(" sde-soar-prompt-regexp "\\)\\s *\\s(sp")
				"^\\s(sp")
			      nil 'move (or count 1))
      ;; This shoudn't be necessary, but if the point is on the opening paren
      ;; and you do the re-search-backward, point ends up moved by one char
      ;; (at least in 19.24).  Argh.
      (goto-char (match-beginning 0))))
  (point))


;; Modified from Emacs 19's end-of-defun.  I don't really understand the
;; point of all this code, but it currently works.  (Yeah, I know.)

(defun sde-end-of-production (&optional arg)
  "Move forward to next end of an sp form.  With arg, do it that many times.
Negative argument -N means move back to Nth preceding end of sp.

An end of an sp occurs right after the close-parenthesis that matches
the open-parenthesis that starts an sp; see `sde-beginning-of-production'.

Returns point unless an error occurs."
  (interactive "p")
  (when (or (null arg) (= arg 0))
    (setq arg 1))
  (sde-with-hacked-delimiter-syntax
      (let ((first t))
	(while (and (> arg 0) (< (point) (point-max)))
	  (let ((pos (point)))
	    (while (progn
		     (unless (and first
				  (progn
				    (forward-char 1)
				    (/= (sde-beginning-of-production 1) pos)))
		       (or (bobp) (forward-char -1))
		       (sde-beginning-of-production -1))
		     (setq first nil)
		     (forward-list 1)
		     (skip-chars-forward " \t")
		     (when (looking-at "\\s<\\|\n")
		       (forward-line 1))
		     (<= (point) pos))))
	  (setq arg (1- arg)))
	(while (< arg 0)
	  (let ((pos (point)))
	    (sde-beginning-of-production 1)
	    (forward-sexp 1)
	    (forward-line 1)
	    (when (>= (point) pos)
	      (if (/= (sde-beginning-of-production 2) pos)
		  (progn
		    (forward-list 1)
		    (skip-chars-forward " \t")
		    (if (looking-at "\\s<\\|\n")
			(forward-line 1)))
		(goto-char (point-min)))))
	  (setq arg (1+ arg)))
	(point))))    


(defun sde-production-body ()
  "Return as a string the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil."
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-production 1 t))
	  (sp-ending (sde-end-of-production)))
      (if (and (>= pt sp-beginning) (<= pt sp-ending))
	  (buffer-substring sp-beginning sp-ending)
	nil))))


(defun sde-production-name (&optional body)
  "Extract the name of the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil.  

Algorithm: store current point, scan backward for the beginning of an sp
form, then from there, scan forward for the end of that form.  If point is
within these bounds, extract the name of the production, else return nil.  "
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-production 1 t))
	  (sp-ending (sde-end-of-production)))
      (when (and (>= pt sp-beginning) (<= pt sp-ending))
	(goto-char sp-beginning)
	(when (looking-at sde-sp-name-regexp)
	  (forward-char 4)
	  (buffer-substring (point) (progn (forward-sexp 1) (point))))))))


(defun sde-region-name (start end)
  "Return a descriptive name for a region of productions or other sexps.
Format of the name is \"From START-SP to END-SP\", where START-SP and END-SP
are the names of the productions at the beginning and end." 
  (let (from to)
    (save-excursion
      (goto-char start)
      (sde-skip-chars end)
      (if (looking-at sde-sp-name-regexp)
	  (setq from (buffer-substring (match-beginning 1) (match-end 1)))
	  (setq from (buffer-substring (point) (progn (end-of-line) (point)))))
      (goto-char end)
      ;; See if we're in the blank space or comments following some expression
      (re-search-backward "^(" nil 'move)
      (forward-sexp 1)
      (sde-skip-chars end)
      (if (= (point) end)
	  (progn
	    (re-search-backward "^(" nil 'move)
	    (if (looking-at sde-sp-name-regexp)
		(setq to (buffer-substring (match-beginning 1) (match-end 1)))
		(setq to (buffer-substring (point) (progn (end-of-line) (point))))))
	  ;; Can't recognize where we are, so punt.
	  (progn
	    (beginning-of-line)
	    (setq to (buffer-substring (point) (progn (end-of-line) (point))))))
      (setq from (sde-string-trim from)
	    to   (sde-string-trim to))
      (if (string-equal from to)
	  from
	  (concat "From " from " to " to)))))


;; Modified from Emacs' mark-defun.

(defun sde-mark-production ()
  "Put mark at end of sp form, point at beginning."
  (interactive)
  (push-mark (point))
  (sde-end-of-production)
  (push-mark (point))
  (sde-beginning-of-production)
  (re-search-backward "^\n" (- (point) 1) t))


;;;----------------------------------------------------------------------------
;;; 10. Extracting symbols and other objects from buffers.
;;;----------------------------------------------------------------------------
;;;
;;; Soar 6.1 allows the following to be used as part of a production name:
;;;     alphanumeric characters
;;;     $ % & * - _ = + < > / : ? 
;;;
;;; The characters < and >, if appearing in a production name, cannot appear
;;; themselves; the name must have at least one alphanumeric character on either
;;; side of the < or >.   If the string "<>" appears in a production name, it
;;; must be flanked by an alphanumeric character on each side, otherwise Soar
;;; will not accept it as a production name.
;;;
;;; Soar stops reading a production name if it encounters one of the following:
;;;     @ ~  {   }  ! ^ ( ) . , # \ ' 
;;;
;;; The | character begins and ends a multi-line comment.
;;; The " character begins and ends a string.
;;;
;;; Extracting symbols is most efficiently done by using the trick of
;;; temporarily changing the syntax table in the buffer so that sexp scanning
;;; can be used effectively.  The ideas here are based on the completion.el
;;; package from J. Salem & B. Kahle at Thinking Machines Corp.  Characters
;;; are put into special classes; then `forward-word' and `scan-sexps' can be
;;; used to move across whole components quickly, without consing or
;;; much string-matching.
;;;
;;; Note that it is impossible to extract symbols from Soar process buffers
;;; without error, because the characters allowed in Soar production names
;;; include characters that appear in Soar output message, such as `pgs'
;;; listings.  In fact, you can define a production named "==>G:" in Soar!
;;; The approach here is to use some explicit tests for certain components
;;; in combination with two syntax tables.
;;;
;;; Since Soar itself rejects production names and other symbols if they
;;; contain certain characters, it is reasonable to assume SDE won't
;;; encounter them in normal operation.  So the algorithms here are not
;;; fool-proof; rather, they're oriented toward the most probably situations.
;;;----------------------------------------------------------------------------

(defvar sde-symbol-syntax-table
  "Syntax table used for extracting Soar symbols from buffers.")

;; symbol syntax      ("_") -- all chars allowed in production names
;; prefix syntax      ("'") -- the dash, which serves dual purposes
;; punctuation syntax (".") -- the up-arrow of attributes
;; whitespace syntax  (" ") -- everything else
;; misc. other syntax for things like the string delimiters.

(unless (syntax-table-p sde-symbol-syntax-table)
  (let ((table (make-vector 256 0))	; Default syntax is whitespace.
	(i 0))
    ;; Alphabetic chars.
    (while (< i 26)
      (modify-syntax-entry (+ ?a i) "_" table)
      (modify-syntax-entry (+ ?A i) "_" table)
      (setq i (1+ i)))
    ;; Digits.
    (setq i 0)
    (while (< i 10)
      (modify-syntax-entry (+ ?0 i) "_" table)
      (setq i (1+ i)))
    ;; Other characters.
    (mapcar (function
	     (lambda (char)
	       (modify-syntax-entry char "_" table)))
	    '(?$ ?% ?& ?* ?_ ?= ?+ ?< ?> ?/ ?: ??))
    (modify-syntax-entry ?\| "$|" table)
    (modify-syntax-entry ?-  "'"  table)
    (modify-syntax-entry ?^  "."  table)
    (modify-syntax-entry ?\" "\"" table)
    (setq sde-symbol-syntax-table table)))


(defun sde-extract-symbol (&optional start)
  "Returns list (SYMBOL BEGINNING END TIMETAG-P) for symbol under or to left of
point.  BEGINNING and END are the buffer positions of the SYMBOL, and
TIMETAG-P is a flag that is `t' if the symbol is actually believed to be a
timetag.  Optional argument START is where to move before prior to extracting
symbol\; point is left there after extracting the symbol.  Special cases: If
the symbol is an attribute name, the leading up-arrow of the attribute is
*not* included.  If the symbol under point is actually a timetag in a WME
printout, only the number is returned\; the trailing colon is omitted.
Otherwise, this function tries to returns whatever looks like a Soar symbol
or production name near point."
  (let ((old-syntax-table (syntax-table))
	(saved-point (or start (point))))
    (goto-char saved-point)
    (unwind-protect
	(let (beginning end timetag-p)
	  (set-syntax-table sde-symbol-syntax-table)
	  ;; First back up to something non-whitespace.
	  (when (= (char-syntax (following-char)) ? )
	    (skip-chars-backward " \t"))
	  ;; Find beginning and end of symbol.  If we're near the `^' or
	  ;; negation `-' of an attribute, or in the middle of something,
	  ;; find the boundaries by first moving forward.  Else, find the
	  ;; symbol's boundaries moving backward.
	  (if (or (memq (preceding-char) '(?- ?^))
		  (/= (char-syntax (following-char)) ? ))
	      (setq end       (scan-sexps (point) 1)
		    beginning (scan-sexps (or end 1) -1))
	    (setq beginning (scan-sexps (point) -1)
		  end       (scan-sexps (or beginning 1) 1)))
	  ;; Check if this may be a WME timetag.
	  (when (char-equal (char-after (1- end)) ?:)
	    (goto-char (1- end))
	    (skip-chars-backward "0-9") 
	    (if (= (point) beginning)	; Nothing but digits => timetag.
		(setq end (1- end)
		      timetag-p t)))
	  (if (and beginning end)
	      (list (buffer-substring beginning end) beginning end timetag-p)
	    nil))
      (goto-char saved-point)
      (set-syntax-table old-syntax-table))))


(defun sde-symbol-near-point (&optional predicate prompt-str)
  "Extract the symbol under or before point, and return it in string form.
If optional arg PREDICATE is non-nil, it must be a function.  PREDICATE is
funcall'ed with the extracted symbol string as its single argument, and it
should return non-nil if the string is acceptable, nil otherwise.  If
PREDICATE returns nil, the user is prompted with optional argument PROMPT-STR
and the extracted symbol as a default answer which the user can select just
by typing return.  If no symbol is found near the point, and PROMPT-STR is
nil, the user is prompted with the default prompt \"Target: \".  This
function returns the symbol string found near point or a new string typed by
the user.  If nothing is found and the user doesn't supply a symbol, an error
is signaled."
  (interactive)
  (if (eobp)
      (if (not (bobp))
	  (save-excursion
	    (backward-char 1)
	    (sde-symbol-near-point predicate prompt-str))
	(sde-log "This command requires a symbol on which to operate, but the\n\
current buffer is empty and you did not supply a symbol when prompted.\n")
	(sde-log-error "Empty buffer"))
    ;; Else, not eobp
    (let* ((prompt-str (or prompt-str "Target: "))
	   (sym (car (sde-extract-symbol))))
      ;; Check that we have something, and that if predicate is supplied it
      ;; returns non-nil on the candidate symbol.
      (when (or (null sym) (and predicate (not (funcall predicate sym))))
	(setq sym (sde-read-string
		   (if sym
		       (format "%s(default \"%s\") " prompt-str sym)
		     (format "%s" prompt-str))))
	(when (string= sym "")
	  (sde-log "This command requires a symbol on which to operate, but it was\n\
unable to find one in the buffer near the cursor and you did not\n\
supply one when prompted.\n")
	  (sde-log-error "No symbol found or provided")))
      sym)))


(defun sde-production-name-near-point (&optional prompt-str)
  "Extract the name of the production under or before point.
If point is inside a full sp expression, the production name is unambiguous.
If point is elsewhere, this function extracts the symbol under or before
point, and heuristically judges whether it is likely to be a production
name."
  (interactive)
  (or (sde-production-name)
      (sde-symbol-near-point 'sde-production-name-p "Production name: ")))
  

;; General function for extracting identifiers, attributes and values from
;; objects printed by Soar.
;;
;; The approach is attribute oriented.  The user should put point over or
;; near an attribute in the production LHS.  The following tries to grab the
;; attribute, the value following it (often it's a variable), and the id at
;; the beginning of the form.
;;
;; This uses the constants `sde-attribute-regexp', `sde-value-regex', etc.,
;; but negated attribute tests are handled explicitly (by looking for the
;; negation sign).
;; 
;; PROBLEMS: This doesn't handle the following constructs:
;;
;; 1)  (<id> ^ {  <> tried-tied-operator <sub-att>  }  <sub-val>)
;;
;; 2)  (<s> ^ {  << required-success success partial-success 
;;                draw lose win >> <svalue>  }  <eb> )
;;
;; But as it turns out we never have to deal with those cases, since those
;; only appear in productions and not in objects printed by Soar.

(defconst sde-attribute-regexp "\\^\\s *[^ \t(){}<]+"
  "Regexp to match an attribute.")

(defconst sde-id-regexp "[a-zA-Z][0-9]+"
  "Regexp to match an identifier.")

(defconst sde-variable-regexp "<[a-zA-Z0-9]+>"
  "Regexp to match a production variable.")

(defconst sde-id-or-variable-regexp
    (concat "\\(" sde-id-regexp "\\|" sde-variable-regexp "\\)")
  "Regexp to match either an identifier or a variable.")

(defconst sde-symbol-regexp "[^^ \t\n<>()]+"
  "Regexp to match a symbol used as a value in a production clause.")

(defconst sde-value-regexp
    (concat "\\(" sde-variable-regexp "\\|" sde-id-regexp
	    "\\|" sde-symbol-regexp "\\)")
  "Regexp to match an attribute value.")


(defun sde-extract-id-attrib-value ()
  "Return a list (id negation attribute value preference).
This function scans the text near the cursor and attempts to extract an
attribute, its value, and an identifier at the head of the clause.      

This function is only suitable for productions written with a simple syntax,
such as those printed out by Soar itself.  It can only understand simple
negated attribute tests and acceptable preferences (`+').

The list returned consists of strings.  The `negation' portion of the
returned list is equal to \"-\" if the attribute had a leading negation sign,
or nil otherwise.  The `attribute' portion of the list is only the attribute
name, and does not have a leading up-arrow.  The `preference' portion is
equal to the string \"+\" if the value in the clause was followed by an
acceptable preference, or nil otherwise."
  (let ((saved-point (point))
	(preference "")
	(negation "")
	current sym id attrib attrib-end value)
    (setq current (sde-extract-symbol))
    ;; Go to beginning of this symbol and see if it's an attribute.
    (goto-char (nth 1 current))
    (skip-chars-backward " \t")
    (if (char-equal (preceding-char) ?^)
	(setq attrib     (car current)
	      attrib-end (nth 2 current))
      ;; Not an attribute; maybe it's a value.  Back up further and try again.
      (setq current (sde-extract-symbol)) ; Get symbol we're now next to.
      (goto-char (nth 1 current))	  ; Go to its beginning
      (skip-chars-backward " \t")
      (when (char-equal (preceding-char) ?^)
	(setq attrib     (car current))
	(setq attrib-end (nth 2 current))))
    ;; At this point, we should be right after the up-arrow.  See if there's
    ;; a negation sign in front, then try to get the value after the attribute.
    (when attrib
      (backward-char 1)
      (skip-chars-backward " \t")
      (when (char-equal (preceding-char) ?-)
	(setq negation "-"))
      ;; Get the value.
      (setq current (sde-extract-symbol (scan-sexps attrib-end 1)))
      (setq sym     (car current))
      (when (string-match sde-value-regexp sym)
	(setq value sym)
	;; See if there's a preference following the value.
	(goto-char (nth 2 current))
	(skip-chars-forward " \t")
	(if (char-equal (following-char) ?+)
	    (setq preference "+"))))
    ;; Look for identifier.
    (search-backward "(" nil t)
    (forward-char 1)
    (setq current (sde-extract-symbol))
    (setq sym (car current))
    ;; Check if we actually have a timetag.
    (when (nth 3 current)
      ;; Yes, a timetag.  Move forward, try to get to the id.
      (goto-char (nth 2 current))	; End of timetag.
      (forward-char 1)			; Past the colon.
      (skip-chars-forward " \t")	; Past the whitespace.
      (setq current (sde-extract-symbol)) ; Get the id.
      (setq sym (car current)))	
    (when (string-match "goal\\|impasse\\|state" sym)
      (goto-char (scan-sexps (nth 2 current) 1))
      (setq current (sde-extract-symbol)
	    sym     (car current)))
    (when (string-match sde-id-or-variable-regexp sym)
      (setq id sym))
    (goto-char saved-point)
    (list id negation attrib value preference)))


;;;----------------------------------------------------------------------------
;;; 13. Parenthesis handling.
;;;----------------------------------------------------------------------------


(defun sde-close-all-sp (&optional arg) 
  "Insert enough parentheses to close the enclosing production.
If there are too many parens, excess parentheses are deleted.  The form
is also indented."
  (interactive "P")
  (let ((begin (sde-beginning-of-production 1 t))
	(end (sde-end-sp-text t))
	(count 0))
    (goto-char end)
    ;; Loop by going to the beginning, attempting to forward-sexp,
    ;; and inserting right parens until we get no more errors.
    (save-restriction
      (narrow-to-region (point) begin)
      (while (save-excursion
	       (progn (sde-beginning-of-production)
		      (condition-case nil
			  (progn (forward-sexp 1) nil)
			(error t))))
	(insert ?\))
	(setq count (1+ count)))
      ;; Delete the extra parens we inserted here.
      (unless (eq end (point))
	(delete-region end (point))))
    ;; Now insert the real parens.
    (goto-char end)
    (while (> count 0)
      (insert ?\))
      (setq count (1- count)))
    (goto-char begin)
    (sde-indent-sexp)
    (forward-sexp 1)))


;;;----------------------------------------------------------------------------
;;; 14. Indentation support.
;;;----------------------------------------------------------------------------
;;;
;;; Nearly all of the following functions are based on code from either 
;;; Emacs 19.19's lisp-mode.el and Ilisp 4.12.  
;;;
;;; The need to duplicate some lisp-mode.el functionality arose because the
;;; lisp indentation routines make assumptions about the structure of the
;;; code that are not true for Soar code.  The biggest problems is
;;; lisp-mode.el's indent-sexp works in such a way that only changes
;;; indentation levels when it encounters a nested sexp.  I.e.,
;;;
;;;	     (foo)
;;;	     (bar)
;;;
;;; end up indented the same way because none of the expressions introduce a
;;; new, more-nested sexp.  On the other hand,
;;;
;;;          (foo (bar)
;;;            (biff)
;;;
;;; introduces a new level of indentation because of the sexp (bar).  The
;;; problem for Soar code is that negated clauses need to be indented
;;; slightly differently, and as far as I can tell, there is no easy way to
;;; use the lisp-indent-function to make things work out properly.
;;; lisp-mode.el's indent-sexp will simply indent negated clauses at the same
;;; block level as the previous line,
;;; 
;;;          (<s> ^color purple)
;;;          -{(<s> ^biff y)}
;;;          (<s> ^free food)
;;;
;;; and that is ugly.  For now the only way out seems to be to copy
;;; indent-sexp and hack it to understand Soar productions more specifically.
;;; The functions lisp-indent-line and indent-sexp don't have any hooks on
;;; which the modifications could be installed, so I had to copy them and
;;; modify them appropriately.  
;;;
;;; Other code here is stolen from Ilisp, as usual, to avoid having to carry
;;; around ilisp-ext.el.
;;;
;;; All of this is messy and really should be improved, but I didn't have
;;; time to do better.


(defun sde-newline-and-indent ()
  "If at the end of the buffer and end of an sp, send the string back to the
process mark with no newline.  Otherwise, insert a newline, then indent.  In
a Soar process buffer the region is narrowed first.  See `newline-and-indent'
for more information."
  (interactive "*")
  (save-restriction
    (when (eq major-mode 'sde-soar-mode)
      (narrow-to-region sde-soar-buffer-mark (point-max)))
    (delete-region (point) (progn (skip-chars-backward " \t") (point)))
    (insert ?\n)
    (sde-indent-line-internal nil)))


(defun sde-indent-line (&optional whole-exp)
  "Indent current line as Soar code.  
With argument, indent any additional lines of the same expression rigidly
along with this one.  This is restricted to the current buffer input."
  (interactive "P")
  (save-restriction
    (when (eq major-mode 'sde-soar-mode)
      (narrow-to-region sde-soar-buffer-mark (point-max)))
    (sde-indent-line-internal whole-exp)))


;; Slightly modified lisp-indent-line from Emacs 19 lisp-mode.el.

(defun sde-indent-line-internal (&optional whole-exp)
  ;; Indent current line as Lisp code.  With argument, indent any additional
  ;; lines of the same expression rigidly along with this one.
  (let ((indent (calculate-lisp-indent)) shift-amt beg end
	(pos (- (point-max) (point))))
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (looking-at "\\s<\\s<\\s<")
	;; Don't alter indentation of a ;;; comment line.
	(goto-char (- (point-max) pos))
      (if (and (looking-at "\\s<") (not (looking-at "\\s<\\s<")))
	  ;; Single-semicolon comment lines should be indented
	  ;; as comment lines, not as code, except if they start
	  ;; in the left column.
	  (unless (bolp)
	    (indent-for-comment)
	    (forward-char -1))
	(when (listp indent)
	  (setq indent (car indent)))
	(setq indent    (sde-adjust-indent indent)
	      shift-amt (- indent (current-column)))
	(unless (zerop shift-amt)
	  (delete-region beg (point))
	  (indent-to indent)))
      ;; If initial point was within line's indentation,
      ;; position after the indentation.  Else stay at same point in text.
      (when (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
      ;; If desired, shift remaining lines of expression the same amount.
      (and whole-exp (not (zerop shift-amt))
	   (save-excursion
	     (goto-char beg)
	     (forward-sexp 1)
	     (setq end (point))
	     (goto-char beg)
	     (forward-line 1)
	     (setq beg (point))
	     (> end beg))
	   (indent-code-rigidly beg end shift-amt)))))


;; Slightly modified indent-sexp from Emacs 19 lisp-mode.el.

(defun sde-indent-sexp (&optional endpos)
  "Indent each line of the list starting just after point.
If optional arg ENDPOS is given, indent each line, stopping when
ENDPOS is encountered."
  (interactive)
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region sde-soar-buffer-mark (point-max)))
    (let ((indent-stack (list nil))
	  (next-depth 0)
	  (starting-point (point))
	  (last-point (point))
	  last-depth bol outer-loop-done inner-loop-done state this-indent)
      ;; Get error now if we don't have a complete sexp after point.
      (save-excursion (forward-sexp 1))
      (save-excursion
	(setq outer-loop-done nil)
	(while (if endpos (< (point) endpos)
		   (not outer-loop-done))
	  (setq last-depth next-depth
		inner-loop-done nil)
	  ;; Parse this line so we can learn the state to indent the next
	  ;; line.  This inner loop goes through only once unless a line ends
	  ;; inside a string.
	  (while (and (not inner-loop-done)
		      (not (setq outer-loop-done (eobp))))
	    (setq state (parse-partial-sexp (point) (progn (end-of-line) (point))
					    nil nil state))
	    (setq next-depth (car state))
	    ;; If the line contains a comment other than the sort that is
	    ;; indented like code, indent it now with indent-for-comment.
	    ;; Comments indented like code are right already.  In any case
	    ;; clear the in-comment flag in the state because
	    ;; parse-partial-sexp never sees the newlines.
	    (if (car (nthcdr 4 state))
		(progn (indent-for-comment)
		       (end-of-line)
		       (setcar (nthcdr 4 state) nil)))
	    ;; If this line ends inside a string, go straight to next line,
	    ;; remaining within the inner loop, and turn off the \-flag.
	    (if (car (nthcdr 3 state))
		(progn
		  (forward-line 1)
		  (setcar (nthcdr 5 state) nil))
		(setq inner-loop-done t)))
	  (and endpos
	       (<= next-depth 0)
	       (progn
		 (setq indent-stack (append indent-stack
					    (make-list (- next-depth) nil))
		       last-depth (- last-depth next-depth)
		       next-depth 0)))
	  (or outer-loop-done
	      (setq outer-loop-done (<= next-depth 0)))
	  (if outer-loop-done
	      (forward-line 1)
	      (while (> last-depth next-depth)
		(setq indent-stack (cdr indent-stack)
		      last-depth (1- last-depth)))
	      (while (< last-depth next-depth)
		(setq indent-stack (cons nil indent-stack)
		      last-depth (1+ last-depth)))
	      ;; Now go to the next line and indent it according
	      ;; to what we learned from parsing the previous one.
	      (forward-line 1)
	      (setq bol (point))
	      (skip-chars-forward " \t")
	      ;; Ignore blank lines.
	      (if (not (or (eobp) (looking-at "\\s<\\|\n")))
		  (progn
		    (if (and (car indent-stack)
			     (>= (car indent-stack) 0))
			(setq this-indent (car indent-stack))
			(let ((val (calculate-lisp-indent
				    (if (car indent-stack)
					(- (car indent-stack))
					starting-point))))
			  (if (integerp val)
			      (setcar indent-stack (setq this-indent val))
			      (progn
				(setcar indent-stack (- (car (cdr val))))
				(setq this-indent (car val))))))
		    ;; Modify the value of this-indent for special cases:
		    ;; single comments, negated clauses, or the arrow.
		    ;; Double-semicolon comments are indented as code.
		    (setq this-indent (sde-adjust-indent this-indent))
		    (if (/= (current-column) this-indent)
			(progn
			  (delete-region bol (point))
			  (indent-to this-indent))))))
	  (or outer-loop-done
	      (setq outer-loop-done (= (point) last-point))
	      (setq last-point (point))))))))


;; How this works:
;;
;; sde-adjust-indent gets called by sde-indent-sexp and by sde-indent-line.
;; At the time it is called, it is facing an about-to-be-indented line.  The
;; "given-indentation" argument is the indentation of the current block of
;; code, where a block is determined by s-expression nesting.
;; sde-adjust-indent is called on every line and is used to adjust the
;; indentation for special cases, such as negated clauses, the production
;; arrow, etc.
;;
;; sde-indent-hook is called by calculate-lisp-indent, which in turn is
;; called by sde-indent-sexp and sde-indent-line.  It is used to determine
;; the indentation for *new* nested blocks of code.
;;
;; sde-indent-line is called by indent-region by virtue of the fact that it
;; is the value of the variable indent-line-function (set in sde-mode).

(defun sde-adjust-indent (given-indentation)
  ;; Must be called with point facing the first non-blank character on a line
  ;; to be examined, and with GIVEN-INDENTATION the suggested indentation
  ;; value.  Returns an adjusted value based on what part of a Soar
  ;; production is on this line.  
  (cond ((looking-at "-->")		; The arrow.
	 (+ given-indentation sde-arrow-indent-offset))
	((looking-at "-\\s * { ")		; Conjunctive negation clause.
	 (- given-indentation 2))
	((looking-at "-\\(\\s *\\)[(^]") ; Single negated clause or attrib.
	 ;; Aggressive reformatting of code.  Remove extra spaces between
	 ;; the negation and the rest.  Indent so that up-arrows line up.
	 (if (> (match-end 1) (match-beginning 1))
	     (save-excursion
	       (goto-char (match-beginning 1))
	       (delete-region (match-beginning 1) (match-end 1))
	       (insert ? )
	       (- given-indentation 2))
	     (- given-indentation 1)))
	(t				; Make sure to return something.
	 given-indentation)))


;; Indent hook for Soar code.  This is used by the standard Emacs
;; indentation routines to determine the column to which the current
;; line should be indented.
;;
;; From the doc string of parse-partial-sexp:
;;
;; State is a list of seven elements describing final state of parsing:
;; 0. depth in parens.
;; 1. character address of start of innermost containing list; nil if none.
;; 2. character address of start of last complete sexp terminated.
;; 3. non-nil if inside a string.
;;    (it is the character that will terminate the string.)
;; 4. t if inside a comment.
;; 5. t if following a quote character.
;; 6. the minimum paren-depth encountered during this scan.

(defun sde-indent-hook (indent-point state)
  (let ((containing-form-start (elt state 1)))
    (goto-char containing-form-start)
    (cond ((looking-at "(sp")		; Start of sp.
	   (if (eq major-mode 'sde-soar-mode)
	       (+ 6 sde-production-indent-offset) ;!!! Bug
	       sde-production-indent-offset))
	  ((looking-at "(<[^>]+>")	; Variable name after "("
	   (if (save-excursion
		 (goto-char indent-point)
		 (looking-at "[ \t]*<[^>]+>"))
	       (let (tmp)
		 (goto-char indent-point)
		 (forward-line -1)
		 (end-of-line)
		 (forward-sexp -1)
		 ;; Stop when see either "^foo" or "^ {  <>" or "^ {  <<".
		 ;; Gross code.  There must be a cleaner way.
		 (while (not (looking-at "\\^\\(\\(\\sw\\|\\s_\\)+\\| { \\s *<<?>?\\)"))
		   (setq tmp (point))
		   (forward-sexp -1))
		 (if tmp (goto-char tmp))
		 (current-column))
	       (progn
		 (forward-char 1)
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (list (current-column) containing-form-start))))
	  ((looking-at "(\\(goal\\|state\\|impasse\\)")
	   (forward-char 1)
	   (forward-sexp 1)
	   (skip-chars-forward " \t\n")
	   (if (looking-at "<[^>]+>")
	       (progn
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (current-column))
	       (current-column)))
	  ((looking-at " { \\s *(")	; Beginning of grouped negation.
	   (skip-chars-forward "^(")
	   (current-column))
	  ((looking-at " { \\s *<<")	; Disjunction
	   (skip-chars-forward " {  \t<")
	   (current-column))
	  (t
	   (current-column)))))  


;; Slightly modified version of reindent-lisp from ilisp-ext.el, Ilisp v. 4.12.

(defmacro sde-point-in-comment (test-regexp)
  "Return t if in a comment determined by TEST-REGEXP."
  (` (progn
       (beginning-of-line)
       (and (looking-at (, test-regexp))
	    (/= (match-end 0) (progn (end-of-line) (point)))))))

;; The following marker keeps track of point so that it doesn't move during a
;; sde-reindent.  Made a global var to avoid calling make-marker repeatedly.

(defvar sde-fill-marker (make-marker))

(defun sde-reindent ()
  "Intelligently reindent the text under the cursor.
If in a comment, indent the comment paragraph bounded by non-comments, blank
lines or empty comment lines.  If in a string, indent the paragraph bounded
by string delimiters or blank lines.  Otherwise go to the containing sp form,
close it and reindent the code block."
  (interactive)
  (let ((region (sde-in-string))
	(comment (concat "[ \t]*" comment-start "+[ \t]*")))
    (set-marker sde-fill-marker (point))
    (back-to-indentation)
    (cond (region
	   (or (= (char-after (point)) ?\")
	       (and (< (point) (car region)) (goto-char (car region)))
	       (re-search-backward "^$" (car region) 'end))
	   (let ((begin (point))
		 (end (car (cdr region)))
		 (fill-prefix nil))
	     (forward-char)
	     (re-search-forward "^$" end 'end)
	     (if (= (point) end)
		 (progn (skip-chars-forward "^\n")
			(if (not (eobp)) (forward-char))))
	     (fill-region-as-paragraph begin (point))))
	  ((looking-at comment)
	   (let ((fill-prefix
		  (buffer-substring
		   (progn (beginning-of-line) (point))
		   (match-end 0))))
	     (while (and (not (bobp)) (sde-point-in-comment comment))
	       (forward-line -1))
	     (if (not (bobp)) (forward-line 1))
	     (let ((begin (point)))
	       (while (and (sde-point-in-comment comment) (not (eobp)))
		 (replace-match fill-prefix)
		 (forward-line 1))
	       (if (not (eobp))
		   (progn (forward-line -1)
			  (end-of-line)
			  (forward-char 1)))
	       (fill-region-as-paragraph begin (point)))))
	  (t
	   (goto-char sde-fill-marker)
	   (sde-close-all-sp)
	   (sde-beginning-of-production 1 t)
	   (sde-indent-sexp)))
  (goto-char sde-fill-marker)
  (set-marker sde-fill-marker nil)
  (message "Done.")))


(defun sde-in-string (&optional begin end)
  "Return the string region that immediately follows/precedes point or that
contains point in optional region BEGIN to END.  If point is in region, t
will be returned as well."
  (save-excursion
    (unless begin
      (save-excursion
	(setq begin (sde-beginning-of-production 1 t)
	      end (sde-end-sp-text t))))
    (let* ((point (progn (skip-chars-forward " \t") (point)))
	   (done nil))
      (goto-char begin)
      (while (and (< (point) end) (not done))
	(skip-chars-forward "^\"" end)
	(setq begin (point))
	(when (< begin end)
	  (if (and (not (bobp)) (= (char-after (1- begin)) ??))
	      (forward-char)
	    (if (condition-case () (progn (forward-sexp) (<= (point) end))
		  (error nil))
		(progn			;After string
		  (skip-chars-forward " \t")
		  (if (or (= begin point) (= point (point)))
		      (setq done (list begin (point) nil))
		    (when (and (< begin point) (< point (point)))
		      (setq done (list begin (point) t)))))
	      ;; In string at end of buffer
	      (setq done (list begin end t))))))
      done)))


;;;----------------------------------------------------------------------------
;;; 16. Miscellaneous editing commands
;;;----------------------------------------------------------------------------


;; Find unbalanced delimiters.
;; Originally from ilisp-ext.el by Chris McConnell.

(defun sde-find-unbalanced (arg)
  "Go to the point in buffer where there exists an extra delimiter.  
Point will be on the offending delimiter.  If called with a prefix, use the
current region.  Checks for '{' '}' and '(' ')' delimiters."
  (interactive "P")
  (if arg
      (call-interactively 'sde-find-unbalanced-region)
    (sde-find-unbalanced-region (point-min) (point-max))))


(defun sde-find-unbalanced-region (start end)
  "Go to the point in region where LEFT-DELIMITER and RIGHT-DELIMITER
become unbalanced.  Point will be on the offending delimiter."
  (interactive "r")
  (sde-count-pairs start end ?\{ ?\} )
  (sde-count-pairs start end ?\( ?\) )
  (beep)
  (message "Delimiters balance."))


(defun sde-count-pairs (begin end left-delimiter right-delimiter)
  "Return the number of top-level pairs of LEFT-DELIMITER and
RIGHT-DELIMITER between BEGIN and END.  If they don't match, the point
will be placed on the offending entry."
  (let ((old-point (point))
	(sexp 0))
    (goto-char begin)
    (sde-skip-chars end)
    (while (< (point) end)
      (let ((char (char-after (point))))
	(cond ((or (eq char left-delimiter)
		   ;; For things other than lists
		   (eq (char-after (1- (point))) ?\n))
	       (setq sexp (1+ sexp))
	       (condition-case ()
		   (progn (forward-sexp) nil)
		 (error 
		  (error "Extra %s" (char-to-string left-delimiter)))))
	      ((eq char right-delimiter)
	       (error "Extra %s" (char-to-string right-delimiter)))
	      ((< (point) end) (forward-char))))
      (sde-skip-chars end))
    (goto-char old-point)
    sexp))


;; Print information about the current file.
;; Originally based on functions from soar-mode v5.0.

(defun sde-region-count-productions (begin end)
  "Count the number of productions, lines & characters in the current region.
To count the productions in the whole buffer, first type `\\[mark-whole-buffer]'."
  (interactive "r")
  (when (interactive-p)
    (message "Counting soar productions..."))
  (let ((count 0)
	(lines (count-lines begin end)))
    (save-excursion
      (goto-char begin)
      (while (re-search-forward sde-sp-name-regexp end t)
	(setq count (1+ count))))
    (when (interactive-p)
      (message "%d production%s, %d line%s, %d character%s."
	       count (if (= count 1) "" "s")
	       lines (if (= lines 1) "" "s")
	       (- end begin) (if (= 1 (- end begin)) "" "s")))
    count))


;; Time/date stamps.  Based on some original code from a date.el that I
;; had written but never released.

(defconst sde-date-month-number
  '(("Jan" . 1) ("Feb" . 2) ("Mar" . 3) ("Apr" . 4) ("May" . 5) ("Jun" . 6)
    ("Jul" . 7) ("Aug" . 8) ("Sep" . 9) ("Oct" . 10) ("Nov" . 11) ("Dec" . 12))
  "Assoc list for looking up the month number from the month
abbreviation.")

(defun sde-insert-date-stamp (&optional arg)
  "Inserts the current date after point, in DD-MM-YY format.
With prefix argument, inserts the weekday first."
  (interactive)
  (let* ((s (current-time-string))
	 (day (substring s 0 3))
	 (month-name (substring s 4 7))
	 (date (if (equal ?\  (aref s 8)) ; Skip any leading space
		   (substring s 9 10)	;  in the day number.
		 (substring s 8 10)))
	 (year (substring s -4 nil)))
    (when current-prefix-arg
      (insert-before-markers (format "%s " day)))
    (insert-before-markers (format "%s-%s-%s" date month-name year))))


;;;----------------------------------------------------------------------------
;;; 17. Comment support
;;;----------------------------------------------------------------------------


;; Comment out regions.
;; Originally from ilisp-ext.el by Chris McConnell.

(defvar sde-comment-marker (make-marker) ; Declare global to avoid calling
  "Marker for end of a comment region.") ;  make-marker repeatedly.

(defun sde-region-comment (start end prefix)
  "If prefix is positive, insert prefix copies of `comment-start' at the
start and `comment-end' at the end of each line in region.  If prefix is
negative, remove all `comment-start' and `comment-end' strings from the
region."
  (interactive "r\np")
  (save-excursion
    (goto-char end)
    (if (and (/= start end) (bolp))
	(setq end (1- end)))
    (goto-char end)
    (beginning-of-line)
    (set-marker sde-comment-marker (point))
    (untabify start end)
    (goto-char start)
    (beginning-of-line)
    (let* ((count 1)
	   (comment comment-start)
	   (comment-end (if (not (equal comment-end "")) comment-end)))
      (if (> prefix 0)
	  (progn
	    (while (< count prefix)
	      (setq comment (concat comment-start comment)
		    count (1+ count)))
	    (while (<= (point) sde-comment-marker)
	      (beginning-of-line)
	      (insert comment)
	      (if comment-end (progn (end-of-line) (insert comment-end)))
	      (forward-line 1)))
	  (setq comment (concat comment "+"))
	  (while (<= (point) sde-comment-marker)
	    (back-to-indentation)
	    (if (looking-at comment) (replace-match ""))
	    (if comment-end
		(progn
		  (re-search-backward comment-end)
		  (replace-match "")))
	    (forward-line 1)))
      (set-marker sde-comment-marker nil))))


;;;----------------------------------------------------------------------------
;;;   Data structures for SDE Soar task database
;;;----------------------------------------------------------------------------
;;;
;;; Internal data is maintained in various forms.  The following are Common
;;; Lisp-style defstructs (made possible by the CL extensions package from
;;; Dave Gillespie) used to store major pieces of information.
;;;
;;; Locating components of a task is often difficult in a rule-based system
;;; because of the way information is split across individual rules and
;;; files.  Structuring a program appropriately helps, but as the rule base
;;; grows, so does the difficulty of locating information.  To help with this
;;; problem, SDE maintains information about rules and other task components
;;; as the user works with task files, which means that the user does not
;;; have to explicitly create indexes, and further, the information closely
;;; reflects the current state of the tasks.
;;;
;;; The primary database structure is as follows:
;;;
;;; 1) Each task is recorded using a defstruct structure that has fields
;;;    for the pathname to the top-level load file for the task, the pathname
;;;    to the .sde data file (if any) recording the task, a hash table of
;;;    defstructs for recording the files comprising the task, and a hash
;;;    table of defstructs for recording the productions in the task.  The
;;;    load file pathname serves as a unique key identifying the task.
;;;
;;; 2) Task data structures are stored in a list.
;;;
;;; 3) Each file defstruct records the pathname of the file that loads that
;;;    file.  (This may be extended in the future to record more info.)
;;;
;;; 4) Each production defstruct records: the pathname of the file that
;;;    contains the production, a parse of the production's LHS, a parse
;;;    of its RHS, and a defstruct storing information for implementing the
;;;    pbreak facility.
;;;
;;; 5) Each file buffer has a buffer-local variable, `sde-buffer-data', that
;;;    points to a defstruct that has fields for: the task to which the
;;;    buffer belongs, the agent (if any) associated with that buffer, and
;;;    a list of the names of the productions found in the buffer at the time
;;;    of the last file save.  The latter is updated at every file save and
;;;    is used in consistency checks.
;;;
;;; The picture is something like this:
;;;
;;;  `sde-known-tasks' (a list) = ( task1 task2 task3 ...)
;;;
;;;  "task1" (a defstruct of type `sde-task') =
;;;    load-file
;;;    files-table       -----> hash table: [ file1 file2 file3 ... ]
;;;    productions-table -----> hash table: [ production1 production2 ... ]
;;;    scanned
;;;    modified
;;;
;;;  "file1" (a defstruct of type `sde-file') =
;;;    load-file
;;;
;;;  "production1" (a defstruct of type `sde-production') =
;;;    file
;;;    lhs
;;;    rhs
;;;    pbreak-data
;;;
;;; Hash tables are used instead of (e.g.) lists for the files and
;;; productions , because the entries have to be unique, and Emacs doesn't
;;; have any built-in functions for efficiently doing member with string= on
;;; a long list of elements.  Using a hash table results in a set of unique
;;; entries, and we can use mapatoms to iterate over the elements.
;;;
;;; Each SDE source code buffer also has a local variable `sde-buffer-data'
;;; that stores info about the buffer, including a pointer to the task
;;; structure of the task that the buffer belongs to.


(defstruct (sde-buffer (:constructor sde-make-buffer-struct)
		       (:copier sde-copy-buffer-struct))
  task        	; Task structure, the task to which this buffer belongs.
  agent		; String, name of agent associated with current buffer.
  productions)	; List of productions last known to be defined in buffer.


(defstruct (sde-file (:constructor sde-make-file-struct)
		     (:copier sde-copy-file-struct))
  load-file)	; String, pathname of file that loads this one.


(defstruct (sde-production (:constructor sde-make-production-struct)
			   (:copier sde-copy-production-struct))
  file		; String, file in which production is defined.
  lhs		; List, the parse of the LHS.
  rhs		; List, the parse of the RHS.
  pbreak-data)


(defstruct (sde-task (:constructor sde-make-task-struct)
		     (:copier sde-copy-task-struct))
  load-file	; String, pathname to top load file
  root-dir      ; String, pathname to top directory (= dir of load-file)
  files		; Hash table of `sde-file' structures.
  productions	; Hash table of `sde-production' structures.
  productions-count ; Integer, count of productions in task.
  scanned       ; Boolean, whether the task has been fully scanned.
  modified)     ; Boolean, whether task modified & .sde file needs writing

(defconst sde-task-files-table-size 211)
(defconst sde-task-productions-table-size 2027)

(defvar sde-known-tasks nil)		; List of `sde-task' structures.

;;;
;;; Support functions for the core data structures.
;;;

(defun sde-reset-tasks ()
  (interactive)
  (setq sde-known-tasks nil))

(defun sde-add-task (lfile)
  ;; Creates a new task structure keyed by load file LFILE.
  ;; Adds it to `sde-known-tasks', and returns the task structure.
  ;; Files table is not set here and must be built up separately.
  (let ((tdata (sde-make-task-struct
		:load-file   lfile
		:root-dir    (file-name-directory lfile)
		:files       (sde-make-hash-table sde-task-files-table-size)
		:productions (sde-make-hash-table sde-task-productions-table-size)
		:scanned     nil
		:modified    t)))
    (setq sde-known-tasks (cons tdata sde-known-tasks))
    tdata))

(defun sde-get-task (lfile)
  ;; Return the task data structure keyed by load file LFILE.
  (let ((tasks sde-known-tasks))
    (while (and tasks (not (equal lfile (sde-task-load-file (first tasks)))))
      (setq tasks (rest tasks)))
    (first tasks)))

(defun sde-add-task-file (file tdata)
  ;; Adds a blank file data structure for filename FILE to a task.
  ;; Returns the new file data structure.
  (setf (sde-task-modified tdata) t)
  (sde-puthash file (sde-task-files tdata) (sde-make-file-struct)))

(defun sde-remove-task-file (file tdata)
  (sde-remhash file (sde-task-files tdata)))

(defun sde-add-task-production (pname tdata)
  ;; Adds a blank file production structure for filename FILE to a task.
  ;; Returns the new file production structure.
  (sde-puthash pname (sde-task-productions tdata) (sde-make-production-struct)))  

(defun sde-remove-task-production (pname tdata)
  (sde-remhash pname (sde-task-productions tdata)))

;; Notes about the following function to count the productions in a task:
;; A field could be added to the task structure to record the number of
;; productions in a task.  However, to keep the count accurate, the process
;; of adding a production would have to be changed to first test whether a
;; given production is already defined in the task, and the count left
;; unchanged in that case.  I'm wary of adding this extra step to
;; sde-add-task-production because of the time cost involved -- that function
;; gets called during file parsing and parsing is already annoyingly slow.
;; I should benchmark sde-add-task-production with and without code to
;; update a production count field and see whether the extra time is
;; significant.

(defun sde-count-task-productions (tdata)
  ;; Returns the number of productions in the task.
  (sde-hash-table-count (sde-task-productions tdata)))

(defun sde-get-production-task (pname)
  ;; Returns the task data structure for the task to which PNAME belongs.
  (let ((tasks sde-known-tasks))
    (while (and tasks (not (sde-gethash pname (sde-task-productions (first tasks)))))
      (setq tasks (rest tasks)))
    (first tasks)))

(defun sde-get-task-files-list (tdata)
  ;; Returns a list of files in the task TDATA.  If the variable
  ;; `sde-sort-lists' is non-`nil', the list will be sorted alphabetically.
  (let (files)
    ;; Construct a list of the files in the task.  Have to watch out for null
    ;; data fields, which indicate entries that have been removed.
    (sde-maphash (function (lambda (file fdata)
			     (when fdata
			       (push (symbol-name file) files))))
		 (sde-task-files tdata))
    (if sde-sort-lists
	(sort files 'string-lessp)
      files)))

(defun sde-map-task-files (func tdata)
  "Maps the function FUNC over each file of task TDATA.
FUNC is called with one argument, a file pathname.
Returns a list of the results of callilng FUNC on each file."
  (mapcar func (sde-get-task-files-list tdata)))

(defun sde-task ()
  "Returns the task for the current buffer."
  (and sde-buffer-data (sde-buffer-task sde-buffer-data)))

(defun sde-at-least-one-task ()
  ;; Make sure there's at least one known task.
  (unless sde-known-tasks
    (sde-add-task (sde-prompt-for-load-file))))

(defun sde-list-tasks ()
  (interactive)
  (with-output-to-temp-buffer "*tasks*"
    (princ "====== Tasks known to SDE ======")
    (mapcar
     (function
      (lambda (tdata)
	(princ "\n\nTask loaded by \"")
	(princ (sde-task-load-file tdata))
	(princ "\"\n")
	(princ "  ")
	(let ((numfiles (sde-hash-table-count (sde-task-files tdata))))
	  (princ numfiles)
	  (princ " file")
	  (if (> numfiles 1)
	      (princ "s")))		; Make it plural if > 1 files.
	(princ " in task\n")
	(if (sde-task-scanned tdata)
	    (progn
	      (princ "  ")
	      (princ (sde-hash-table-count (sde-task-productions tdata)))
	      (princ " productions\n")
	      (if (sde-task-modified tdata)
		  (princ "  Corresponding .sde file needs to be updated.\n")
		(princ "  Corresponding .sde file is up-to-date.\n")))
	  (princ "  Task has not been fully scanned yet.\n"))))
     sde-known-tasks)))

;; Debugging.

(defun sde-show-production-parse ()
  (interactive)
  (let* ((pname (sde-production-name-near-point))
	 (pdata (sde-get-production-task pname)))
    (with-output-to-temp-buffer "*parse*"
      (princ (save-excursion
	       (set-buffer (get-buffer-create "* sde tmp*"))
	       (erase-buffer)
	       (cl-prettyprint (sde-production-lhs pdata))
	       (buffer-string))))))      



;;;----------------------------------------------------------------------------
;;; 4.  Reading and maintaining .sde data files.
;;;----------------------------------------------------------------------------

;;; The .sde data file is used to track the load file and constituent files of
;;; a task.  This is really only used to avoid having to prompt the user for
;;; the information each time that a task is visited (e.g., in a new editing
;;; session), but that's important enough -- users got very annoyed at the
;;; constant prompting of an earlier version of SDE because it didn't have
;;; a good way of storing info across editing sessions.
;;;
;;; The .sde file has the following format:
;;;
;;;   ;; SDE data file -- DO NOT MODIFY.
;;;   ("load file1" 
;;;    "file1"
;;;    "file2"
;;;    ...)
;;;   ("load file2"
;;;    "file1"
;;;    "file2"
;;;    ...)
;;;
;;; For each task, the top-level load file is listed followed by the files
;;; that comprise the task.  This permits SDE to identify the load file for a
;;; task given a random file name, when a random file is visited by the user
;;; and the associated task has not yet been visited.  Once the task to which
;;; the random file belongs is identified, the info from the .sde file is not
;;; necessary, but the .sde file has to be kept up-to-date with the contents
;;; of the task.

(defconst sde-data-file-name ".sde")

(defsubst sde-read (stream)
  ;; Returns nil if end of stream is reached. 
  (condition-case nil
      (read stream)
    (end-of-file nil)))

(defsubst sde-data-format-check-p (data)
  ;; Return nil if the data is not formatted appropriately for a .sde entry.
  (and (first data) (rest data)))

(defun sde-read-directory-data-file (dir)
  ;; Reads the records stored in the .sde data file in DIR, and assimilates
  ;; the information into the SDE task database.  Returns a list of the
  ;; tasks found, nil otherwise.
  (let ((file (concat dir sde-data-file-name)))
    (when (file-readable-p file)
      (save-excursion
	(let (buffer found-tasks)
	  (cond ((setq buffer (find-buffer-visiting file))
		 ;; File is already in a buffer.  First check the disk file
		 ;; hasn't changed since last used.  Then read the task data
		 ;; as normal, being careful not to try to create a new task
		 ;; structure for a task that is already in the database.
		 (set-buffer buffer)
		 (when (not (verify-visited-file-modtime (current-buffer)))
		   (revert-buffer t t)
		   (setq buffer (set-buffer (find-buffer-visiting file)))))
		(t
		 ;; File is not in a buffer.  Read it.
		 (setq buffer (set-buffer (sde-find-file-noselect file)))))
	  ;; Read and incorporate info.  For extra safety, read in everything
	  ;; first and do some weak tests for proper format, *then* incorporate
	  ;; the data into the database.
	  (let ((ok t) data-list data)
	    (goto-char 1)
	    (while (setq data (sde-read buffer))
	      (if (sde-data-format-check-p data)
		  (push data data-list)
		(setq ok nil)))
	    (if ok
		(dolist (data data-list)
		  (let ((lfile (expand-file-name (first data) dir)))
		    ;; Don't add this if the task is already known. 
		    (unless (sde-get-task lfile)
		      (let* ((tdata  (sde-add-task lfile))
			     (ftable (sde-task-files tdata))
			     (files  (rest data)))
			(while files
			  (sde-puthash (expand-file-name (first files) dir)
				       ftable (sde-make-file-struct))
			  (setq files (rest files)))
			(push tdata found-tasks)))))
	      ;; Some part of the data in the .sde file was found wrong.
	      ;; Try to get rid of this .sde file.
	      (condition-case nil
		  (progn
		    (delete-file buffer-file-name)
		    (kill-buffer buffer))
		(error nil))))
	  found-tasks)))))


(defvar sde-data-file-header
  ";;; SDE data file -- generated automatically -- DO NOT EDIT THIS.\n"
  "Header for .sde files.")

(defun sde-insert-data-file-header ()
  ;; Insert the data file header at the top of an .sde file.
  (insert sde-data-file-header))

(defun sde-insert-data-file-task-data (tdata)
  (let ((dir (sde-task-root-dir tdata)))
    (insert "(\"" (sde-relative-pathname (sde-task-load-file tdata) dir) "\"")
    (insert "\n \"")
    (insert (mapconcat
	     (function (lambda (file) (sde-relative-pathname file dir)))
	     (sde-get-task-files-list tdata)
	     "\"\n \""))
    (insert "\")\n")))

(defun sde-insert-data-file-data (dir)
  (dolist (tdata sde-known-tasks)
    (when (equal (sde-task-root-dir tdata) dir)
      (sde-insert-data-file-task-data tdata)
      (setf (sde-task-modified tdata) nil))))

;;; Data files get written out when a new task is created, or when SDE
;;; discovers that an existing task (on disk) doesn't have an associated
;;; .sde data file.

(defun sde-write-directory-data-file (dir)
  ;; Writes out the .sde file in directory DIR, using the current contents of
  ;; the SDE database for the tasks whose load files are located in DIR. 
  (save-excursion
    (let ((file (concat dir sde-data-file-name)))
      (if (file-writable-p file)
	  (let ((buffer (or (find-buffer-visiting file)
			    (sde-find-file-noselect file))))
	    (if buffer
		(progn
		  (set-buffer buffer)
		  ;; Check that file hasn't been modified since we last wrote it.
		  (when (not (verify-visited-file-modtime buffer))
		    ;; Argh, it's been modified.  Read it anew to catch any possible
		    ;; new entries prior to writing it out again.
		    (sde-read-directory-data-file dir)
		    (setq buffer (set-buffer (find-buffer-visiting file))))
		  (erase-buffer)
		  (sde-insert-data-file-header)
		  (sde-insert-data-file-data dir)
		  (save-buffer))
	      ;; Couldn't read the file.  Why?
	      (message ".sde file exists but couldn't be read." dir)))
	;; File is not writable.
	(message "Warning: cannot write .sde file in %s" dir)))))

(defun sde-update-data-files ()
  ;; Run through known tasks and update the data files of those tasks whose
  ;; modified flag is set.
  (let (failure)
    (dolist (tdata sde-known-tasks)
      (when (sde-task-modified tdata)
	(condition-case err
	    (sde-write-directory-data-file (sde-task-root-dir tdata))
	  (error
	   (setq failure t)
	   (sde-log
	    (format "Encountered error while trying to save .sde file for task\n
loaded by %s\n%s\n" (sde-task-load-file tdata) err))))))
    (when failure
      (sde-log-message "Error saving data files"))))

(defsubst sde-file-in-task-p (file tdata)
  ;; Return non-nil if the FILE is believed to be part of task TDATA.
  (sde-gethash file (sde-task-files tdata)))

(defsubst sde-search-tasks-for-file (file tasks)
  ;; Search for FILE being a part of one of the tasks in list TLIST.
  ;; If found, return the task data structure, else return nil.
  (while (and tasks (not (sde-file-in-task-p file (car tasks))))
    (setq tasks (rest tasks)))
  (car tasks))

(defun sde-get-file-task (file &optional dont-scan)
  ;; Returns the task that contains FILE, or nil.  If the task is initially
  ;; not found, and optional argument DONT-SCAN is nil, it initiates a search
  ;; of .sde files beginning in the current directory and moving backward.
  (or (sde-search-tasks-for-file file sde-known-tasks)
      ;; Not in existing task database.  Initiate a search of .sde files.
      ;; Keep looking for .sde files and reading them (and by side-effect,
      ;; adding new task structures to `sde-known-tasks'), then checking if
      ;; the current file becomes known as a result.
      (and (null dont-scan)
	   (let ((dir (file-name-directory file))
		 found)
	     (while (and dir (not found))
	       (setq found (sde-search-tasks-for-file
			    file (sde-read-directory-data-file dir)))
	       (setq dir (sde-directory-parent dir)))
	     found))))

(defun sde-get-production-file (pname)
  ;; Returns the file pathname where production PNAME is found.  Since this
  ;; may require doing a full scan of each task, first all of the
  ;; already-scanned tasks are checked, then if the production is still not
  ;; found, the unscanned tasks are checked.
  (sde-at-least-one-task)
  (let ((tasks sde-known-tasks)
	found)
    ;; Go through the scanned tasks first.    
    (while tasks
      (let ((tdata (first tasks)))
	(if (and (sde-task-scanned tdata)
		 (sde-gethash pname (sde-task-productions tdata)))
	    (setq found tdata
		  tasks nil)
	  (setq tasks (rest tasks)))))
    (if found
	(sde-production-file (sde-gethash pname (sde-task-productions found)))
      ;; Not found in the scanned tasks.  Go through the unscanned ones.
      (setq tasks sde-known-tasks)
      (while tasks
	(let ((tdata (first tasks)))
	  (unless (sde-task-scanned tdata)
	    (sde-parse-task tdata)
	    (if (sde-gethash pname (sde-task-productions tdata))
		(setq found tdata
		      tasks nil)))
	  (setq tasks (rest tasks))))
      (when found
	(sde-production-file (sde-gethash pname (sde-task-productions found)))))))


;;;----------------------------------------------------------------------------
;;; 6.  Functions for prompting the user.
;;;----------------------------------------------------------------------------

(defvar sde-y-or-n-p-default-help-msg
  "Type SPC or `y' to accept;
DEL or `n' to say no;
ESC or `q' to exit.")

(defun sde-y-or-n-p (prompt &optional help-msg arg)
  "Ask user a \"y or n\" question.  Return t if answer is \"y\".
Takes one argument, which is the string to display to ask the question.
It should not end in a space; `sde-y-or-n-p' adds ` (y, n, ?, C-h) ' to it.
No confirmation of the answer is requested; a single character is enough.
Also accepts Space to mean yes, or Delete to mean no.
Optional argument HELP-MSG is a message to display if the user types
`?' or `C-h'.  Optional second argument ARG should be an argument that
is used in (funcall help-msg arg) to format the help message presented
to the user\; in that case, help-msg should contain a %s where the
value of ARG should be substituted."
  (save-window-excursion
    (let* ((hchar (key-description (vector help-char)))
	   (prompt (format "%s (y, n, ?, %s)" prompt hchar)))
      (let (answer)
	(while
	    (let (char)
	      ;; Prompt.
	      (message prompt)
	      (setq char (read-event))
	      ;; Show the user's input.
	      (message (concat prompt (format " %s" (single-key-description char))))
	      (cond ((memq char '(?y ?Y ?\ ))
		     (setq answer t)
		     nil)
		    ((memq char '(?n ?N ?\C-? delete))
		     (setq answer nil)
		     nil)
		    ((memq char '(?? ?\C-h help))
		     (with-output-to-temp-buffer "*Help*"
		       (if help-msg
			   (princ (funcall 'format help-msg arg))
			 (princ sde-y-or-n-p-default-help-msg)))
		     ;; Go back around the loop.
		     t)
		    (t
		     (beep)
		     (message "Type %s for help" hchar)
		     (sit-for 1)
		     ;; Go back around the loop.
		     t))))
	(message "")			; Erase last prompt.
	answer))))


;;; These help hide the differences in user prompting between Emacs 18 with
;;; gmhist and Emacs 19.  There are a few enhancements, such as making
;;; `sde-completing-read' return nil when the user types just a return
;;; (instead of returning "" as the regular function does).


(defun sde-read-string (prompt &optional hist initial)
  (if (and hist (featurep 'gmhist))
      (read-with-history-in hist prompt initial)
    (read-from-minibuffer prompt initial nil nil hist)))


(defun sde-completing-read (prompt &optional hist table pred mustmatch initial)
  (let ((answer (if (and hist (featurep 'gmhist))
		    (completing-read-with-history-in
		     hist prompt table pred mustmatch initial)
		  (completing-read prompt table pred mustmatch initial hist))))
    (if (string= answer "")
	nil
      answer)))


(defun sde-read-file-name (prompt &optional hist dir default mustmatch initial)
  (if (and hist (featurep 'gmhist))
      (read-file-name-with-history-in hist prompt dir default mustmatch initial)
    (read-file-name prompt dir default mustmatch initial)))

 
(defun sde-prompt (prompt &optional hist initial)
  "If `current-prefix-arg' is non-nil, prompt user with PROMPT and history HIST.  
Otherwise, if HIST is provided, return the first item on the list, or if 
HIST is not provided, return INITIAL."
  (if hist
      (if current-prefix-arg
	  (if (featurep 'gmhist)
	      (read-with-history-in hist prompt initial)
	    (read-from-minibuffer prompt initial nil nil hist))
	(car (eval hist)))
    (if current-prefix-arg
	(read-from-minibuffer prompt initial)
      initial)))


;;; Functions for prompting for common values.

(defvar sde-prompt-for-load-file-hist nil)
(defvar sde-prompt-for-ps-name-hist nil)
(defvar sde-prompt-for-op-name-hist nil)


(defun sde-prompt-for-load-file (&optional file)
  (save-window-excursion
    (when file
      (select-frame (first (frame-list)))
      (switch-to-buffer (get-file-buffer file)))
    (let ((answer
	   (substitute-in-file-name
	    (sde-read-file-name "Top-level load file: "
				'sde-prompt-for-load-file-hist nil nil t))))
      (if (string= answer "")
	  nil
	answer))))


(defun sde-prompt-for-op-name (&optional must-match)
  (if must-match
      (sde-completing-read "Operator: (? = list) "
			   'sde-prompt-for-op-name-hist
			   (sde-get-task-ops (sde-get-task-data (sde-task))))
    (read-no-blanks-input "Operator name (must be one symbol): ")))


(defun sde-prompt-for-ps-name (&optional must-match initial)
  (if must-match
      (sde-completing-read "Problem space: (? = list) "
			   'sde-prompt-for-ps-name-hist
			   (sde-get-task-pss (sde-get-task-data (sde-task)))
			   nil t initial)
    (read-no-blanks-input "Problem space name (must be one symbol): ")))


;;;----------------------------------------------------------------------------
;;; 4.  File scanning and parsing.
;;;----------------------------------------------------------------------------

;;; SDE production parsing code for Soar 6 syntax.
;;;
;;; The idea here is to create an internal representation of the object
;;; structure and attribute names mentioned in the condition side of a
;;; production.  A graph is constructed for each production in each file of a
;;; task when the files of the task are first read in, and thereafter
;;; whenever a buffer is saved.  This is done to permit fast searching of
;;; production bodies.
;;;
;;; The condition side of a production can be regarded as a graph, and the
;;; internal representation here is basically a graph structure stored in
;;; list form.  The nodes are the (object) variables or constants, and the
;;; arcs of the graph are the attribute names.  So something like
;;;
;;; (goal <g> ^object <sg> ^operator <o>)
;;; (goal <sg> ^object nil ^impasse no-change)
;;; (<o> ^name foo ^object <x>)
;;; (<x> ^color red ^size big)
;;;
;;; is conceptually represented as
;;;
;;;                        /--- ^impasse -- no-change
;;;                       /
;;;     /- ^object --- <sg> --- ^object --- nil
;;;    /
;;;   /
;;; <g> ------ ^operator - <o> -- ^name ----- foo
;;;                         \
;;;                          \--- ^object ---- <x> --- ^color -- red
;;;                                              \
;;;                                               \--- ^size -- big
;;;
;;; and in internal list form as
;;;
;;; (<g> ^object <sg> ^operator <o>)
;;; (<sg> ^object nil ^impasse no-change)
;;; (<o> ^name foo ^object <x>)
;;; (<x> ^color red ^size big)
;;;
;;; where the first item in the list is always the lowest goal tested in the
;;; production condition.  Note that attribute names are recorded as symbols,
;;; including their leading up-arrows, because that's the form they end up in
;;; when they are read in using the Emacs Lisp `read' function.
;;;
;;; This internal graph then permits pattern-matching search on the condition
;;; sides of productions.
;;;
;;; Only a limited number of aspects of a production LHS are actually
;;; recorded.  Typical LHS clause constructs are translated as follows:
;;;
;;; (<x> ^att1 const)              --> (<x> ^att1 const)
;;; (<x> -^att1 const)             --> (<x> ^att1 const)
;;; -{(<x> ^att1 const)}           --> (<x> ^att1 const)
;;;
;;;    (Negated attribute tests are listed, because in a sense, the condition
;;;    is testing that attribute.  In fact the presense of negations is
;;;    ignored completely.)
;;;
;;; (<x> ^{ <z> <> foo } <y>)      --> (<x> ^foo <y>)
;;;
;;;    (Variable attributes are basically ignored.  In compound tests like
;;;    this, it tries to extract whatever it can, such as the attribute name
;;;    foo, because the condition is testing that attribute in some sense.)
;;;
;;; (<x> ^<z> <y>)                 --> ignored
;;;
;;; (<x> ^ << att1 att2 >> <y>)    --> (<x> ^att1 <y> ^att2 <y>)
;;; (<x> ^ << att1 att2 >> <> <y>) --> (<x> ^att1 nil ^att2 nil)
;;;
;;;    (As an attribute value, nil signifies the end of that graph path.
;;;    This is currently an arbitrary limitation on the parsing and matching
;;;    code.)
;;;
;;; (<x> ^att1.att2 <y>)           --> (<x> ^att1 <a>)
;;;                                    (<a> ^att2 <y>)
;;;
;;; The full Soar 6 grammar for condition sides of productions is as follows,
;;; in BNF notation (taken from the Soar 6 User's Manual):
;;;
;;; <condition-side>   ::= <cond>+
;;; <cond>             ::= <positive_cond> | - <positive_cond>
;;; <positive_cond>    ::= <conds_for_one_id> | { <cond>+ } 
;;; <conds_for_one_id> ::= ( [goal|impasse] [<id_test>] <attr_value_tests>* )
;;; <id_test>          ::= <test>
;;; <attr_value_tests> ::= [-] ^ <attr_test> [.<attr_test>]* <value_test>*
;;; <attr_test>        ::= <test>
;;; <value_test>       ::= <test> [+] | <conds_for_one_id> [+]
;;;
;;; <test>             ::= <conjunctive_test> | <simple_test>
;;; <conjunctive_test> ::= { <simple_test>+ } 
;;; <simple_test>      ::= <disjunction_test> | <relational_test>
;;; <disjunction_test> ::= << <constant>* >>
;;; <relational_test>  ::= [<relation>] <single_test>
;;; <relation>         ::= <> | < | > | <= | >= | = | <=>
;;; <single_test>      ::= variable | <constant>
;;; <constant>         ::= sym_constant | int_constant | float_constant
;;;

(defvar sde-cond nil)
(defvar sde-parsed nil)

(defsubst sde-name-string (sym)
  (if (numberp sym)
      (number-to-string sym)
    (symbol-name sym)))

(defsubst sde-variable-p (sym)
  "Is SYM a Soar variable (a symbol beginning with '<' and ending with '>')?"
  (and sym
       (symbolp sym)
       (char-equal (aref (symbol-name sym) 0) ?<)
       (not (memq sym '(<< < <= <=> <>)))))

(defsubst sde-attribute-p (sym)
  "Is SYM a Soar attribute (a symbol beginning with '^')?"
  (and sym
       (symbolp sym)
       (char-equal (aref (symbol-name sym) 0) ?^)))

(defsubst sde-dotted-p (sym)
  (and (symbolp sym)
       (string-match "\\." (symbol-name sym))))

(defsubst sde-gensym-new-var (letter-string)
  (intern (concat "<" (symbol-name (gensym letter-string)) ">")))


;;; The following set of functions handle parsing of 
;;;   <test> ::= <conjunctive_test> | <simple_test>
;;; The entry point is `sde-parse-test', later on.  Nearly all functions
;;; are open-coded for speed.

(defsubst sde-parse-relational-test (front)
  ;; Parse "<relational_test>  ::= [<relation>] <single_test>".
  ;;
  (when (memq front '(<> < > <=> <= => =))
    ;; Just eat the relation.
    (setq sde-cond (rest sde-cond))
    (setq front    (first sde-cond)))
  ;; Move pointer past "<single_test>"...
  (setq sde-cond (rest sde-cond))
  ;; ... and return the "<single_test>".
  (list front))


(defsubst sde-parse-disjunctive-test (front)
  ;; Parse "<disjunction_test> ::= << <constant>* >>".
  ;;
  (let (syms)
    (setq sde-cond (rest sde-cond))	; Eat "<<".
    (while (and sde-cond (not (eq (first sde-cond) '>>)))
      (push (first sde-cond) syms)
      (setq sde-cond (rest sde-cond)))
    (if (eq (first sde-cond) '>>)
	(setq sde-cond (rest sde-cond))) ; Eat ">>".
    syms))


(defsubst sde-parse-simple-test (front)
  ;; Parse "<simple_test> ::= <disjunction_test> | <relational_test>".
  ;;
  (cond ((eq front '^<<)
	 (setq sde-cond (cons '<< (rest sde-cond)))
	 (sde-parse-disjunctive-test front))

	((eq front '<<)
	 (sde-parse-disjunctive-test front))

	(t
	 (sde-parse-relational-test front))))


(defsubst sde-parse-conjunctive-test (front)
  ;; Parse "<conjunctive_test> ::= { <simple_test>+ }"
  ;;
  (let (vars-and-constants)
    (setq sde-cond (rest sde-cond)) ; Eat the "{".
    (while (and (setq front (first sde-cond))
		(not (eq front '} )))
      (setq vars-and-constants
	    (nconc (sde-parse-simple-test front) vars-and-constants)))
    (setq sde-cond (rest sde-cond)) ; Eat the " } ".
    vars-and-constants))


(defun sde-parse-test ()
  ;; Parse "<test> ::= <conjunctive_test> | <simple_test>".
  ;; Returns a list of the variables and constants mentioned.
  (let ((front (first sde-cond)))
    (if (memq front '( { ^{ ))
	(sde-parse-conjunctive-test front)
      (sde-parse-simple-test front))))


(defsubst sde-parse-test-1 (front)
  (if (or (sde-variable-p front) (sde-attribute-p front))
      (progn
	(setq sde-cond (rest sde-cond))
	(list front))
    (sde-parse-test)))


(defsubst sde-parse-value-test (front)
  ;; Parses "<value_test> ::= <test> [+] | <conds_for_one_id> [+]"
  ;;
  (if (consp front)
      ;; Structured value notation.
      (let ((tmp sde-cond)
	    (result (sde-parse-conds-for-one-id front)))
	(setq sde-cond (rest tmp))
	(list (first result)))
    ;; Parse a test, optionally followed by a "+".
    (prog1
	(sde-parse-test-1 front)
      ;; sde-cond will have been changed, hence `front' will no longer
      ;; refer to the correct element of the list.  Use (first sde-cond).
      (when (eq (first sde-cond) '+)
	(setq sde-cond (rest sde-cond))))))


(defsubst sde-parse-value-test-star ()
  ;; Parses "<value_test>*" 
  ;;
  (let (front results)
    (while (and sde-cond
		(not (eq (setq front (first sde-cond)) '-))
		(not (sde-attribute-p front)))
      (setq results (nconc (sde-parse-value-test front) results)))
    (or results (list nil))))


(defsubst sde-parse-dotted-attr-value-tests ()
  ;; Parse "<attr_value_tests> ::= [-] ^ <attr_test> [.<attr_test>]* <value_test>*"
  ;; for the case where the attribute is dotted.  This is messy because the
  ;; attribute is read as a symbol, but the only way to decompose it is to
  ;; transform it into a string.
  (let* ((attr (first sde-cond))
	 (var1 (sde-gensym-new-var "v"))
	 new)
    ;; First check some common cases.
    (cond ((eq attr '^problem-space\.name)
	   (setq sde-cond (rest sde-cond))
	   (let ((values (sde-parse-value-test-star)))
	     (while
		 (setq new    (cons '^name (cons (car values) new))
		       values (cdr values))))
	   (setq sde-parsed (nconc sde-parsed (list (cons var1 new))))
	   (list '^problem-space var1))

	  ((eq attr '^operator\.name)
	   (setq sde-cond (rest sde-cond))
	   (let ((values (sde-parse-value-test-star)))
	     (while
		 (setq new    (cons '^name (cons (car values) new))
		       values (cdr values))))
	   (setq sde-parsed (nconc sde-parsed (list (cons var1 new))))
	   (list '^operator var1))

	  (t
	   ;; Ok to use symbol-name here because the caller will
	   ;; have called `sde-dotted-p' before branching here,
	   ;; and that will have checked that the symbol is not actually
	   ;; a number.
	   (let* ((attr-str (symbol-name attr))
		  (match (string-match "\\." attr-str))
		  var2)
	     (prog1
		 ;; The first component of the dotted attribute belongs to the current
		 ;; id whose cond is being parsed, so that's what we return.
		 (list (intern (substring attr-str 0 match)) var1)
	       (setq attr-str (concat "^" (substring attr-str (1+ match))))
	       (while (setq match (string-match "\\." attr-str))
		 (setq attr (intern (substring attr-str 0 match)))
		 (setq attr-str (concat "^" (substring attr-str (1+ match))))
		 (setq var2 (sde-gensym-new-var "v"))
		 (setq sde-parsed (nconc sde-parsed (list (list var1 (cons attr var2)))))
		 (setq var1 var2))
	       ;; The last component is the one that gets the value tests.
	       (setq attr (intern attr-str))
	       (setq sde-cond (rest sde-cond))
	       (let ((values (sde-parse-value-test-star)))
		 (while
		     (setq new    (cons attr (cons (car values) new))
			   values (cdr values))))
	       (setq sde-parsed (nconc sde-parsed (list (cons var1 new))))))))))


(defun sde-parse-attr-value-tests ()
  ;; Parse "<attr_value_tests> ::= [-] ^ <attr_test> [.<attr_test>]* <value_test>*"
  ;; Algorithm:
  ;;  Deconstruct the attribute test(s), watching out for possible dotted
  ;;  notation used in the attribute name.  Store the constants and throw
  ;;  away any variables in the attribute tests.  Then parse the value
  ;;  tests, looking for the main variable or constant in the value-tests.
  ;;  Finally, add new structures to `sde-parsed', keyed by the variables.
  (if (sde-dotted-p (first sde-cond))
      (sde-parse-dotted-attr-value-tests)
    ;; Not dotted.
    (let ((vars-and-constants (sde-parse-test-1 (first sde-cond)))
	  (value-tests (sde-parse-value-test-star))
	  (pairs))
      (when vars-and-constants
	(if value-tests
	    (dolist (attr vars-and-constants)
	      ;; Ignore variable attributes.
	      (unless (sde-variable-p attr)
		(unless (eq (aref (sde-name-string attr) 0) ?^)
		  (setq attr (intern (concat "^" (sde-name-string attr)))))
		(dolist (val value-tests)
		  (setq pairs (cons val pairs))
		  (setq pairs (cons attr pairs)))))
	  ;; No value tests.
	  (dolist (attr vars-and-constants)
	    (unless (sde-variable-p attr)
	      (unless (eq (aref (sde-name-string attr) 0) ?^)
		(setq attr (intern (concat "^" (sde-name-string attr)))))
	      (setq pairs (cons nil pairs))
	      (setq pairs (cons attr pairs))))))
      pairs)))


(defsubst sde-parse-attr-value-tests-star ()
  ;; Parses tests, returns list.
  (let (front tests)
    (while (setq front (first sde-cond))
      (cond ((eq front '-)
	     (setq sde-cond (rest sde-cond)))

	    ((eq front '^)
	     ;; Assume an attribute name with a separated "^".  Massage it.
	     (setq sde-cond
		   (cons (intern (concat "^" (sde-name-string (second sde-cond))))
			 (rest (rest sde-cond))))
	     (setq tests (nconc tests (sde-parse-attr-value-tests))))

	    (t
	     (setq tests (nconc tests (sde-parse-attr-value-tests))))))
    tests))


(defsubst sde-parse-head-of-cond (first-letter-if-no-id)
  ;; Returns a list of one or more variables and constants mentioned
  ;; in the leading portion of a condition clause, i.e., the portion
  ;; "( [goal | impasse] [ <id_test> ] ".  If an id-test is found,
  ;; and it mentions more than one variable, the first variable on the
  ;; list returned is the main variable in the test.
  (let ((front (first sde-cond)))
    (cond ((memq front '(goal impasse state))
	   ;; !!! Assumes no test involving goal/impasse/state id.
	   (prog1
	       (list (first (rest sde-cond)))
	     (setq sde-cond (rest (rest sde-cond)))))
	  ((or (null sde-cond) (sde-attribute-p front))
	   ;; No id found.
	   (list (sde-gensym-new-var first-letter-if-no-id)))
	  (t
	   (sde-parse-test-1 front)))))


(defun sde-parse-conds-for-one-id (cond &optional first-letter-if-no-id)
  ;; Parses a condition.
  ;; Updates sde-parsed.
  ;; Ignore constants and add vars to sde-parsed.
  ;; Returns the list for the one id.
  (setq sde-cond cond)
  (let* ((vars-and-constants (sde-parse-head-of-cond first-letter-if-no-id))
	 (main (first vars-and-constants)))
    (dolist (sym vars-and-constants)
      (when (and (sde-variable-p sym) (not (assq sym sde-parsed)))
	;; Leave the first entry (assumed to be the goal) where it is.
	(setq sde-parsed (nconc sde-parsed (list (list sym))))))
    (if (and main (sde-variable-p main))
	(let ((var-data (assq main sde-parsed)))
	  (if var-data
	      (setcdr var-data (nconc (rest var-data)
				      (sde-parse-attr-value-tests-star))))
	  var-data))))


(defvar sde-conds nil)

(defsubst sde-parse-cond (clause)
  ;; Parses the combination
  ;;  <cond>             ::= <positive_cond> | - <positive_cond>
  ;;  <positive_cond>    ::= <conds_for_one_id> |  {  <cond>+  } 
  ;; Argument CONDS is the remaining list of conditions in the LHS.
  (cond ((eq clause '-)
	 ;; Ignore negations.
	 (setq sde-conds (rest sde-conds)))

	((eq clause '{)
	 ;; Parse "{  <cond>+  }".
	 (setq sde-conds (rest sde-conds)) ; Eat the "{".
	 (sde-parse-cond-plus)
	 (setq sde-conds (rest sde-conds))) ; Eat the "}".

	((consp clause)
	 (sde-parse-conds-for-one-id clause "g")
	 (setq sde-conds (rest sde-conds)))))


(defun sde-parse-cond-plus ()
  ;; Parses "<condition-side>   ::= <cond>+"
  ;; Argument CONDS is the remaining list of conditions in the LHS.
  (let (c)
    (while (or (consp (setq c (first sde-conds))) (memq c '(- {)))
      (sde-parse-cond c))))


;;; Grammar for RHS:
;;;
;;;   <rhs>             ::= <rhs_action>*
;;;   <rhs_action>      ::= ( variable <attr_value_make>+ ) | <function_call>
;;;   <function_call>   ::= ( <function_name> <rhs_value>* )
;;;   <function_name>   ::= sym_constant | + | -
;;;   <rhs_value>       ::= <constant> | <function_call> | variable
;;;   <constant>        ::= sym_constant | int_constant | float_constant
;;;   <attr_value_make> ::= ^ <variable_or_sym_constant> <value_make>+
;;;   <variable_or_sym_constant> ::= variable | sym_constant
;;;   <value_make>      ::= <rhs_value> <preference_specifier>*
;;;
;;;   <preference-specifier>       ::= <naturally-unary-preference>
;;;                                  | <forced-unary-preference>
;;;                                  | <binary-preference> <rhs_value>
;;;   <naturally-unary-preference> ::= + | - | ! | ~ | @
;;;   <binary-preference>          ::= > | = | < | &
;;;   <forced-unary-preference>    ::= <binary-preference> {, | ) | ^}
;;;     ;but the parser doesn't consume the ")" or "^" lexeme here


(defun sde-parse-production (form file tdata)
  (setq form (rest form))		; Skip 'sp' part.
  (unless (not (symbolp (first form)))	; Production name should be a symbol.
    ;; Store production data under production name, retain pointer to data.
    (let ((pdata (sde-add-task-production (first form) tdata)))
      (setf (sde-production-file pdata) file)
      (setq form (rest form))		; Move forward past name.
      (if (stringp (first form))	; Skip doc string if present.
	  (setq form (rest form)))
      (if (symbolp (first form))	; Skip flag if present.
	  (setq form (rest form)))
      ;; Parse condition side and action side.
      (setq sde-conds form)
      (setq sde-parsed nil)
      (sde-parse-cond-plus)
      (setf (sde-production-lhs pdata) sde-parsed))))


(defsubst sde-parse-preclean ()
  (sde-strip-multi-line-comments)
  ;; Separate certain tokens with whitespace
  (goto-char 1)
  (while (re-search-forward "\\({\\|}\\)" nil t)
    (replace-match " \\1 "))
  (goto-char 1)
  (while (search-forward "-^" nil t)
    (replace-match "- ^")))


;; There are two variants of functions for parsing files, for greater
;; efficiency.  One is used to scan recursively, and thus it pays attention
;; to load and cd statements, and the other ignores load statements but
;; builds a list of the productions encountered.  Duplicating code like this
;; is rather poor programming practice, but I wanted as much speed out of
;; this as possible.

(defun sde-parse-file (file parent tdata dir)
  ;; FILE and DIR must be expanded.
  (save-excursion
    (let ((kill-buffer-hook nil)	; For Emacs 19, for speed.
	  (buffer (create-file-buffer file)))
      (set-buffer buffer)
      (erase-buffer)
      (if (condition-case nil
	      (insert-file-contents file t)
	    (file-error nil))
	  (let ((fdata (sde-add-task-file file tdata)))
	    (setf (sde-file-load-file fdata) parent)
	    (buffer-disable-undo buffer)
	    (message "Parsing %s ..." file)
	    (sde-parse-preclean)
	    (goto-char 1)
	    ;; `read' generates an `end-of-file' error upon trying to read
	    ;; past the last form in a buffer.  Hence the condition-case.
	    (condition-case nil
		(while
		    (let ((form (read buffer)))
		      (cond ((listp form)
			     (let ((first (first form)))
			       (cond ((eq first 'sp)
				      (sde-parse-production form file tdata))
				     ((eq first 'load)
				      (sde-parse-file
				       (expand-file-name (second form) dir)
				       file tdata dir))
				     ((memq first '(cd chdir))
				      (setq dir (expand-file-name (second form) dir))))))
			    ((eq form 'load)
			     (sde-parse-file
			      (expand-file-name (read buffer) dir)
			      file tdata dir))
			    ((memq form '(cd chdir))
			     (setq dir (expand-file-name (read buffer) dir))))
		      form))
	      (invalid-read-syntax nil)
	      (end-of-file nil))))
      (set-buffer-modified-p nil)
      (kill-buffer buffer))))


(defun sde-parse-buffer-productions ()
  ;; Scans only the productions in the current buffer.
  ;; Updates the task database.
  ;; Updates the sde-buffer-productions field of sde-buffer-data.
  (let ((kill-buffer-hook nil)
	(buffer (current-buffer))
	(file   buffer-file-name)
	(tdata  (sde-task))
	(tmpbuf (get-buffer-create " *sde scan temp*"))
	productions)
    (message "Parsing %s ..." file)
    (save-excursion
      (set-buffer tmpbuf)
      (erase-buffer)
      (insert-buffer buffer)
      (buffer-disable-undo tmpbuf)
      (sde-parse-preclean)
      (goto-char 1)
      ;; `read' generates an `end-of-file' error upon trying to read past the
      ;; last form in a buffer.  Hence the condition-case.
      (condition-case nil
	  (while
	      (let ((form (read tmpbuf)))
		(when (and (listp form) (eq (car form) 'sp))
		  (push (second form) productions) ; Record the name.
		  (sde-parse-production form file tdata))
		form))
	(invalid-read-syntax nil)
	(end-of-file nil))
      (set-buffer-modified-p nil))
    ;; Back in the original buffer.
    (setf (sde-buffer-productions sde-buffer-data) productions)
    (kill-buffer tmpbuf)))


(defun sde-parse-task (tdata &optional dir)
  ;; Scan the files of the task TDATA starting with its load file.
  ;; Optional DIR specifies the current directory.
  (setq dir (or dir default-directory))
  (let ((file (expand-file-name (sde-task-load-file tdata) dir)))
    (when (file-exists-p file)
      (sde-parse-file file file tdata (file-name-directory file))
      (setf (sde-task-scanned tdata) t)
      (setf (sde-task-modified tdata) t)
      (sde-update-data-files))))


(defun sde-check-task-parsed (tdata)
  ;; Verify that task TDATA has had a full scan.
  (when (null tdata)
    (let ((lfile (sde-prompt-for-load-file)))
      (cond ((setq tdata (sde-get-file-task lfile))
	     ;; User supplied load file & it's known.  Implies the current
	     ;; file should also have been known.  Assume it's an extra.
	     (sde-add-task-file lfile tdata))

	    (t
	     ;; Load file is unknown.  Create new task, scan task files.
	     (setq tdata (sde-add-task lfile))
	     (sde-parse-task tdata)))))
  (unless (sde-task-scanned tdata)
    (sde-parse-task tdata))
  tdata)

;;; For profiling with th ELP package.
;;; (setq elp-function-list '(sde-parse-value-test sde-parse-value-test-star sde-parse-dotted-attr-value-tests sde-parse-attr-value-tests sde-parse-attr-value-tests-star sde-parse-disjunctive-test sde-parse-relational-test sde-parse-simple-test sde-parse-conjunctive-test sde-parse-test sde-parse-head-of-cond sde-parse-conds-for-one-id sde-parse-cond-plus sde-parse-cond sde-parse-production sde-parse-preclean))

(defun sde-list-parses ()
  (interactive)
  (unless (fboundp 'cl-prettyprint)
    (load-library "cl-extra"))
  (with-output-to-temp-buffer "*productions*"
    (princ "====== Productions parsed ======")
    (mapcar (function
	     (lambda (tdata)
	       (princ "\n\nTask \"")
	       (princ (sde-task-load-file tdata))
	       (princ "\":\n")
	       (sde-maphash (function
			     (lambda (name pdata)
			       (princ "  ")
			       (princ name)
			       (princ (save-excursion
					(set-buffer (get-buffer-create "* sde tmp*"))
					(erase-buffer)
					(cl-prettyprint (sde-production-lhs pdata))
					(goto-char 1)
					(while (re-search-forward "^" nil t)
					  (replace-match "    "))
					(buffer-string)))
			       (princ "\n")))
			    (sde-task-productions tdata))))
	    sde-known-tasks)))


;;;----------------------------------------------------------------------------
;;; 4.  Recording files and buffers into the database.
;;;----------------------------------------------------------------------------

;;; As a backup measure, and so that users who set sde-inhibit-record-file-data
;;; can still get some use out of sde-find-production-by-name, SDE records each
;;; file in one global table of file names.

(defvar sde-known-files (sde-make-hash-table))

(defun sde-add-to-known-files (file)
  ;; Add FILE to the table of known files.
  (sde-puthash file sde-known-files t))

(defun sde-ignore-file-p (file)
  ;; Returns non-nil if FILE is one that shouldn't be tracked.
  (string-match
   (mapconcat 'identity sde-ignore-files-regexp-list "\\|")
   file))

(defun sde-record-buffer (buffer)
  ;; Record info about the given buffer.
  (save-excursion
    (set-buffer buffer)
    (unless (sde-ignore-file-p buffer-file-name)
;      (let* ((file (file-truename buffer-file-name))
      (let* ((file buffer-file-name)
	     (task (sde-get-file-task file)))
	(when (null task)
	  ;; If we didn't find it mentioned in a .sde file, ask user.
	  (let ((lfile (sde-prompt-for-load-file buffer-file-name)))
	    (cond ((null lfile)
		   ;; No load file given ==> this file should be treated as
		   ;; being alone.  Create a new task for it.
		   (setq task (sde-add-task file))
		   (setq lfile file)
		   (sde-parse-task task))

		  ((setq task (sde-get-file-task lfile))
		   ;; User supplied load file & it's known.  Implies the
		   ;; current file should also have been known.  Assume
		   ;; it's an extra.
		   (sde-add-task-file file task))

		  (t
		   ;; Load file is unknown.  Create new task, scan task files.
		   (setq task (sde-add-task lfile))
		   (sde-parse-task task)))))
	;; Update fields in the sde-buffer structure
	(when (null sde-buffer-data)
	  (setq sde-buffer-data (sde-make-buffer-struct)))
	(setf (sde-buffer-task sde-buffer-data) task)
	(sde-parse-buffer-productions)))))

(defun sde-record-file (file)
  ;; If the file is not already known to SDE, record info about it.
  (unless (sde-get-file-task file 'dont-scan)
    (save-excursion
      (let ((kill-buffer-hook nil)	; For Emacs 19, for speed.
	    (buffer (create-file-buffer file)))
	(set-buffer buffer)
	(erase-buffer)
	(prog1
	    (if (condition-case nil
		    (insert-file-contents file t)
		  (file-error nil))
		(sde-record-buffer buffer))
	  (kill-buffer buffer)))))
  ;; Also record this in a separate table, as a backup.  Used by
  ;; sde-find-production-using-heuristics.
  (sde-add-to-known-files file))


;;;----------------------------------------------------------------------------
;;; find-file-hook, local-write-file-hook, kill-emacs-hook
;;;----------------------------------------------------------------------------

;;; Function `sde-find-file-hook' is added to the Emacs `find-file-hooks' so
;;; that when a Soar source file is first read in, it can be recorded into
;;; the database.  When invoked, `sde-find-file-hook' also adds a local
;;; write file hook to the file's buffer, so that the database can be updated
;;; whenever the file is written out to disk.

(add-hook 'find-file-hooks 'sde-find-file-hook-fn)
(add-hook 'kill-emacs-hook 'sde-kill-emacs-hook-fn)

(defvar sde-inhibit-write-file-data nil)

(defun sde-find-file-hook-fn ()
  ;; Assumes buffer is current.
  (unless sde-inhibit-record-file-data
    (condition-case ()
	(when (memq major-mode sde-source-modes)
	  (sde-record-buffer (current-buffer))
	  (add-hook 'local-write-file-hooks 'sde-local-write-file-hook-fn))
      (error nil))))

(defun sde-local-write-file-hook-fn ()
  ;; Assumes buffer is current.
  ;; This function is added to a buffer's `local-write-file-hooks'
  ;; by a function on `find-file-hooks'.
  (condition-case err
      (when (memq major-mode sde-source-modes) ; Sanity check.
	(let ((tdata (sde-buffer-task sde-buffer-data))
	      (previous-list (sde-buffer-productions sde-buffer-data))
	      (current-list))
	  ;; Update internal database
	  (sde-parse-buffer-productions)
	  (setq current-list (sde-buffer-productions sde-buffer-data))
	  ;; Remove from the task database the productions that are no longer in
	  ;; this buffer.
	  (mapcar (function
		   (lambda (production)
		     (unless (member production current-list)
		       (sde-remove-task-production production tdata))))
		  previous-list))
	;; Must return nil for Emacs. 
	nil)
    (error
     (sde-log "Encountered error while trying to write %s.\n\
This may indicate a bug in SDE.  Please use `M-x sde-feedback' to report\n\
what happened to the maintainers of SDE.  Please be specific and detailed\n\
in your email message." buffer-file-name)
     (sde-log-message (format "Error writing file %s" 
			      (file-name-nondirectory buffer-file-name))))))

(defun sde-kill-emacs-hook-fn ()
  ;; Runs through and saves .sde files if necessary before quitting Emacs.
  (message "Saving SDE data files...")
  (sde-update-data-files)
  (message "Saving SDE data files... Done."))


;;;----------------------------------------------------------------------------
;;; Search/find utilities.
;;;----------------------------------------------------------------------------

;;; This is based largely on the tags-loop-continue facility in etags.el
;;; of Emacs 19.24.

(defvar sde-next-match-initialized nil
  "Flag that is non-`nil' after the first time a search or find is executed.")

(defvar sde-next-match-operate nil
  "Form for `sde-next-match' to eval to change one file.")

(defvar sde-next-match-scan
  '(error (substitute-command-keys
	   "No find or search operation in progress."))
  "Form for `sde-next-match' to eval to scan one file.
If it returns non-nil, this file needs processing by evalling
\`sde-next-match-operate'.  Otherwise, move on to the next file.")

(defvar sde-next-match-task nil
  "Task being searched currently by `sde-next-match'.")

(defvar sde-next-match-file-list nil
  "List of files for \\[sde-next-match-file] to process.")

(defvar sde-next-match-production-list nil
  "List of productions for \\[sde-next-match-production] to process.")

;; Users found it annoying, after performing a replacement across 100 files,
;; that the files would be unsaved and then upon exit Emacs would prompt for
;; saving each of those 100 files.  The `sde-next-match-visited-buffers'
;; variable allows SDE to track the files that were visited during the course
;; of (e.g.) `sde-query-replace', and lets SDE prompt the user to save all the
;; files at once after the operation is completed.  The variable is actually
;; used in `sde-next-match-next-file'.

(defvar sde-next-match-visited-buffers nil
  "List of buffers visited in the course of a next-match operation.")

(defvar sde-next-match-save-help-msg 
  "As a result of the query-replace operation you have just performed, there
may be modified buffers that have not been saved to their respective files.
Type 'y' if you wish to have SDE save all the buffers now.
Type `n', `q', `ESC' or `DEL' to not save them at this time.
If you choose not to save the buffers now, Emacs will eventually ask you
again before it lets you exit this editing session.")

(defun sde-next-match-buffers-were-modified ()
  ;; Returns non-`nil' if any of the buffers on the list
  ;; sde-next-match-visited-buffers is modified.
  (dolist (buffer sde-next-match-visited-buffers)
    (when (buffer-modified-p buffer)
      (return t))))

(defun sde-next-match-maybe-save-buffers ()
  (when (and (sde-next-match-buffers-were-modified)
	     (sde-y-or-n-p "All files processed.  Save modified buffers?"
			   sde-next-match-save-help-msg))
    (sde-save-sde-buffers sde-next-match-visited-buffers 'no-query)))

(defun sde-next-match-next-file (&optional initialize novisit)
  ;; Select next file among files in the task currently being searched.  A
  ;; first argument of t (prefix arg, if interactive) initializes to the
  ;; first file in the task's file table.  If the argument is neither nil
  ;; nor t, it is eval'ed to initialize the list of files.  Non-nil second
  ;; argument NOVISIT means use a temporary buffer to save time and avoid
  ;; uninteresting warnings.  Value is nil if the file was already visited;
  ;; if the file was newly read in, the value is the filename.
  (interactive "P")
  (cond ((not initialize)
	 ;; Not the first run.
	 nil)
	((eq initialize t)
	 ;; Initialize the list from the database.
	 (setq sde-next-match-file-list (sde-get-task-files-list sde-next-match-task))
	 (setq sde-next-match-visited-buffers nil))
	(t
	 ;; Initialize the list by evalling the argument.
	 (setq sde-next-match-file-list (eval initialize))
	 (setq sde-next-match-visited-buffers nil)))
  (unless sde-next-match-file-list
    (if sde-next-match-visited-buffers
	(progn
	  (sde-next-match-maybe-save-buffers)
	  (error "Done."))
      (error "All files processed.")))
  (let ((new (not (find-buffer-visiting (car sde-next-match-file-list)))))
    (if (not (and new novisit))
	(set-buffer (find-file-noselect (car sde-next-match-file-list) novisit))
      ;; Like find-file, but avoids random warning messages.
      (set-buffer (get-buffer-create " *sde-next-file*"))
      (kill-all-local-variables)
      (erase-buffer)
      (setq new (car sde-next-match-file-list))
      (insert-file-contents new nil))
    (setq sde-next-match-file-list (cdr sde-next-match-file-list))
    new))

;; Basically this is a hacked version of tags-loop-continue from etags.el.

(defun sde-next-match-file (&optional first-time)
  "Move to the next file being searched."
  (let ((messaged nil)
	new)
    (while
	(progn
	  ;; Scan files quickly for the first or next interesting one.
	  (while (or first-time
		     (save-restriction
		       (widen)
		       (not (eval sde-next-match-scan))))
	    (setq new (sde-next-match-next-file first-time t))
	    ;; If NEW is non-nil, we got a temp buffer,
	    ;; and NEW is the file name.
	    (if (or messaged
		    (and (not first-time)
			 (> baud-rate search-slow-speed)
			 (setq messaged t)))
		(message "Searching file %s..." (or new buffer-file-name)))
	    (setq first-time nil)
	    (goto-char (point-min)))

	  ;; If we visited it in a temp buffer, visit it now for real.
	  (if new
	      (let ((pos (point)))
		(erase-buffer)
		(set-buffer (find-file-noselect new))
		(widen)
		(goto-char pos)))

	  (switch-to-buffer (current-buffer))
	  (pushnew (current-buffer) sde-next-match-visited-buffers)

	  ;; Now operate on the file.
	  ;; If value is non-nil, continue to scan the next file.
	  (eval sde-next-match-operate)))
    (and messaged
	 (null sde-next-match-operate)
	 (message "Searching file %s...found" buffer-file-name))))

(defun sde-next-match-production (&optional first-time)
  (if sde-next-match-production-list
      (sde-find-production-by-name (pop sde-next-match-production-list)
				   sde-next-match-task 'no-msg)
    (error "No more matching productions.")))

;; If the variable `sde-next-match-scan' is `nil', the first element on
;; `sde-next-match-productions-list' is taken to be the next production to be
;; found and show in another buffer.  If `sde-next-match-productions-list' is
;; `nil', the list of productions is assumed to have been exhausted and
;; nothing is shown.
;; 
;; If `sde-next-match-scan' is non-`nil', it should be a form that is evaluated
;; to search for something in each file mentioned on the list
;; `sde-next-match-file-list'.  Call `sde-next-match' noninteractively with a
;; non-`nil' argument to create the file list.  The argument is passed to
;; `sde-next-match-file' (which see).  The variable `sde-next-match-operate' is
;; eval'ed when `sde-next-match-scan' evaluates to a non-`nil' value.  If
;; `sde-next-match-operate' returns nil, SDE moves on to the next file.

(defun sde-next-match (&optional first-time)
  "Continue the last find or search command by moving to the next match."
  (interactive)
  (if sde-next-match-scan
      (sde-next-match-file first-time)
    (sde-next-match-production first-time)))


;;;----------------------------------------------------------------------------
;;; 19. Help support.
;;;----------------------------------------------------------------------------

;; None of these use a separate command history, because we want the same
;; symbols to be available from one command to another.  E.g., the user might
;; start off typing sde-apropos on something, then try sde-topic-help, and it's
;; desirable to be able to retrieve the item off the history list.  

(defvar sde-help-hist nil)


(defun sde-apropos (&optional regexp)
  (interactive (list (sde-read-string "Apropos: " 'sde-help-hist)))
  (with-output-to-temp-buffer "*Help*"
    (sde-apropos-print-matches
     (if sde-running-emacs19
	 (apropos-internal regexp 'sde-apropos-match)))))


(defun sde-apropos-match (symbol)
  ;; Determine if given symbol is in sde-commands-obarray or
  ;; sde-variables-obarray and matches a given regexp.
  (or (intern-soft (symbol-name symbol) sde-commands-obarray)
      (intern-soft (symbol-name symbol) sde-variables-obarray)))


;; Following two functions very heavily borrowed from fast-apropos.el,
;; by Joe Wells <jbw@bigbird.bu.edu> (now a standard part of Emacs 19).

(defun sde-apropos-print-matches (matches &optional spacing)
  ;;  Helper function for fast-apropos and super-apropos.  Depends on
  ;;  standard-output having been rebound appropriately.  Prints the symbols
  ;;  and documentation in alist MATCHES of form ((symbol fn-doc var-doc) ...).
  ;;  Displays in the buffer pointed to by standard-output.  Optional argument
  ;;  SPACING means put blank lines in between each symbol's documentation.
  ;;  This should only be called within a with-output-to-temp-buffer.
  (sde-apropos-get-doc matches)
  (setq matches (sort matches (function
			       (lambda (a b)
				 (string-lessp (car a) (car b))))))
  (let ((p matches)
	(old-buffer (current-buffer))
	item symbol str)
    (save-excursion
      (set-buffer standard-output)
      (unless matches (princ "No matches found."))
      (while (consp p)
	(setq item (car p)
	      symbol (car item)
	      p (cdr p))
	(unless (or (not spacing) (bobp))
	  (terpri))
	(princ symbol)		        ; Print symbol name.
	(when (commandp symbol)
	  (let (keys)
	    (save-excursion
	      (set-buffer old-buffer)
	      (setq keys (sde-where-is symbol)))
	    (indent-to 30 1)
	    (princ (or keys "(not bound to any keys)"))))
	(terpri)
	(when (setq str (nth 1 item))
	  (princ "  Function: ")
	  (princ (substitute-command-keys str)))
	(unless (bolp)
	  (terpri))
	(when (setq str (nth 2 item))
	  (princ "  Variable: ")
	  (princ (substitute-command-keys str)))
	(unless (bolp)
	  (terpri))	
	(terpri))))
  t)


(defun sde-apropos-get-doc (sym-list)
  ;; Helper for sde-apropos.
  ;; Takes SYM-LIST of symbols and adds documentation.  Modifies SYM-LIST in
  ;; place.  Resulting alist is of form ((symbol fn-doc var-doc) ...).
  ;; Returns SYM-LIST.
  (let ((p sym-list)
	fn-doc var-doc symbol)
    (while (consp p)
      (setq symbol (car p)
	    fn-doc (if (fboundp symbol)
		       (documentation symbol))
	    var-doc (documentation-property symbol 'variable-documentation)
	    fn-doc (and fn-doc
			(substring fn-doc 0 (string-match "\n" fn-doc)))
	    var-doc (and var-doc
			 (substring var-doc 0 (string-match "\n" var-doc))))
      (setcar p (list symbol fn-doc var-doc))
      (setq p (cdr p)))
    sym-list))


;; Modified from bbdb-info from bdbb-com.el. 

(defvar Info-directory)			; D. Gillespie's info.el defines this.

(defun sde-display-info-file (file)
  (require 'info)
  (let ((Info-directory (and (boundp 'Info-directory) Info-directory))
	(try (expand-file-name file sde-directory)))
    (or (file-exists-p try)
	(setq try (concat file ".info")))
    (or (file-exists-p try)  
	(setq try (expand-file-name file Info-directory)))
    (or (file-exists-p try)
	(setq try (concat try ".info")))
    (or (file-exists-p try)
	(error "Info file %s doesn't exist" file))
    (let ((Info-directory (file-name-directory try)))
      (Info-goto-node (format "(%s)Top" try)))))


(defun sde-info ()
  "Run the Emacs Info system on the SDE documentation."
  (interactive)
  (sde-display-info-file sde-info-file))


(defun sde-news ()
  "Display news about SDE, including recent changes."
  (interactive)
  (sde-info)
  (Info-goto-node "News"))


(defun sde-soar-info ()
  "Run the Emacs Info system on the on-line Soar manual."
  (interactive)
  (sde-display-info-file sde-soar-info-file))


(defmacro sde-display-notes-file (file)
  ;; Helper macro.  This needs to be a macro because we need the actual
  ;; symbol passed as argument for use in error messages. 
  (` (progn
       (sde-error-unless-site-var-set (quote (, file)))
       (let ((buffer (create-file-buffer (, file))))
	 (save-excursion
	   (set-buffer buffer)
	   (erase-buffer)
	   (condition-case err
	       (insert-file-contents (, file) t)
	     (error
	      (sde-log "Couldn't read file \"%s\" because of error:\n" (, file))
	      (sde-log (format "%s" err))
	      (sde-log-error (format "Couldn't read file \"%s.\"" (, file)))))
	   (setq buffer-read-only t))
	 (sde-pop-to-buffer buffer)))))


(defun sde-soar-release-notes ()
  "Read the most recent release notes for Soar.
This depends on the variable `sde-soar-release-notes-file' to be set properly." 
  (interactive)
  (sde-display-notes-file sde-soar-release-notes-file))


(defun sde-soar-user-notes ()
  "Read the most recent release notes for Soar.
This depends on the variable `sde-soar-user-notes-file' to be set properly." 
  (interactive)
  (sde-display-notes-file sde-soar-user-notes-file))


;; The topics obarray works like this.  Initially, when this file is loaded,
;; sde-topics-obarray is set to the union of sde-commands-obarray and
;; sde-variables-obarray.  (See bottom of this file.)  Whenever Soar is
;; restarted, the function sde-record-soar-help-topics is called; this
;; resets sde-topics-obarray to also include the help topics that Soar knows
;; about.  This means that the user won't have the full set of help topics
;; available until Soar is started at least once.

(defun sde-topic-help (topic)
  (interactive (list (sde-completing-read
		      "Topic (? = list): "
		      'sde-help-hist sde-topics-obarray)))
  (if topic
      (cond ((intern-soft topic sde-commands-obarray)
	     (sde-command-help topic))
	    ((intern-soft topic sde-variables-obarray)
	     (sde-variable-help topic))
	    (t
	     (sde-soar-help topic)))
    (error "No topic.")))


(defun sde-command-help (cmd-name)
  "Prompts for an SDE command name (with completion) and displays help about it.
This is almost identical to the Emacs `describe-function' command, except that
completion is limited to user commands in SDE.  If called from a program,
takes argument CMD-NAME, the string name of the variable."
  (interactive (list (sde-completing-read
		      "Command name (? = list): "
		      'sde-help-hist sde-commands-obarray)))
  (if cmd-name
      (describe-function (car (read-from-string cmd-name)))
    (error "No command name.")))


(defun sde-variable-help (var-name)
  "Prompts for an SDE variable name (with completion) and displays help about it.
This is almost identical to the Emacs `describe-variable' command, except that
completion is limited to user variables in SDE.  If called from a program,
takes argument VAR-NAME, the string name of the variable."
  (interactive (list (sde-completing-read
		      "Variable (? = list): "
		      'sde-help-hist sde-variables-obarray)))
  (if var-name
      (describe-variable (car (read-from-string var-name)))
    (error "No variable name.")))


;; Following code partly taken from rp-describe-function.el, written by
;; Robert Potter, rpotter@grip.cis.upenn.edu.

(defun sde-where-is (cmd)
  "Prompts for an SDE command (with completion) and shows its key bindings.
This command differs from the regular Emacs `where-is' by the fact that 
it completes only on SDE commands and its output is more specific.  If
called from a Lisp program, returns a string of the form 
   \"locally on `keystroke1' and globally on `keystroke2'\"
or nil if the command has no key binding."
  (interactive (list (car (read-from-string
			   (sde-completing-read
			    "Where is command (? = list): "
			    'sde-help-hist sde-commands-obarray)))))
  ;; This finds local key bindings and global bindings separately
  ;; so that we can tell the user exactly where things are.
  (let* ((global-map (current-global-map))
	 (global (sde-where-is-internal cmd))
	 (local (unwind-protect
		    (progn
		      (use-global-map (make-keymap))
		      (sde-where-is-internal cmd (current-local-map)))
		  (use-global-map global-map)))
	 (text (if (or global local)
		   (format "%s%s%s"
			   (if local
			       (format "locally on `%s'"
				       (mapconcat 'key-description local "' and `"))
			     "")
			   (if (and global local)
			       " and "
			     "")
			   (if global
			       (format "globally on `%s'"
				       (mapconcat 'key-description global "' and `"))
			     "")))))
    (when (interactive-p)
      (if text
	  (message "%s is available %s" cmd text)
	(message "%s has no key bindings." cmd)))
    text))
  

;; What follows is an ugly thing to do.  I wanted three features for the mode
;; help in SDE:
;;
;;   1) Print a message at beginning and end, to let user know that
;;	*something* is happening -- some mode docs are long and Emacs takes
;;	forever to format them.
;;	
;;   2) Cache the mode help string, to speed up help as much as possible.
;;
;;   3) Fix an inconsistency in Emacs 18 about how it prints the key
;;	binding information for (1) commands that have meta bindings and (2)
;;	commands that have no bindings.  In case (1), the mode help shows a
;;	binding of "ESC c" but in case (2), it shows "M-x foo" instead.
;;	Lucid 19 consistently uses "M-c", but FSF 19 (wouldn't you know it)
;;	uses both "ESC c" and "M-c" in describe-mode and "M-c" in the manual.
;;      I'm going to make an effort to point out this inconsistency to the
;;	GNU folks, and in the meantime, the functions below change everything
;;	to "M-c" to be consistent with the Emacs (and SDE) manuals.
;;
;; Now the problem is how to hook this into the describe-mode function.
;; Experience has shown that simply giving "\C-hm" a new binding in sde-mode-map
;; results in problems for people who must rebind C-h to act as delete
;; because their terminals are stupid.  (Creating a new binding for "\C-hm"
;; automatically introduces a new intermediate local map for "\C-h", which is
;; what causes the problems.  There is no way to create a binding only for
;; "\C-hm" without also creating a local map for C-h.)  So this code resorts
;; to resetting the function definition of describe-mode, a bad thing to do,
;; but if it's done carefully, maybe the repercussions will be small.

(defvar sde-describe-mode-text-alist nil
  "Association list of mode-names and their associated documentation strings,
storing the result of doing a sde-describe-mode for each mode.  The list
consists of dotted pairs of the form \(major-mode . documentation-string).")

(defvar sde-true-describe-mode (symbol-function 'describe-mode)
  "The original function definition of `describe-mode'.")

(fset 'sde-true-describe-mode sde-true-describe-mode)

(defun sde-describe-mode ()
  "Display documentation of current major mode."
  (interactive)
  (if (not (memq major-mode sde-modes))
      (sde-true-describe-mode)
    (with-output-to-temp-buffer "*Help*"
      (let ((mode-help (assoc major-mode sde-describe-mode-text-alist)))
	(if mode-help
	    (princ (cdr mode-help))
	  (let (tmp case-fold-search case-replace)
	    (message "Formatting documentation ...")
	    (princ mode-name)
	    (princ " Mode:\n")
	    (princ (documentation major-mode))
	    (save-excursion
	      (set-buffer "*Help*")
	      (sde-massage-key-descriptions)
	      (setq tmp (buffer-string)))
	    (message "Formatting documentation ...done.")
	    (push (cons major-mode tmp) sde-describe-mode-text-alist))))
      (print-help-return-message))))

(fset 'describe-mode (symbol-function 'sde-describe-mode))


(defvar sde-massage-key-descriptions-list
  '(("\\bLFD\\b" . "C-j")
    ("\\bESC \\([][~a-zA-Z0-9@\;\\`!\"#$%^(){}*+='.,|/<>?-]\\|LFD\\)" . "M-\\1") 
    ("\\bM-C-\\([][~a-zA-Z0-9@\;\\`!\"#$%^(){}*+='.,|/<>?-]\\|LFD\\)" . "C-M-\\1")
    ;; SDE-specific things.
    ("\\bC-c \\? \\(C-.\\|\\?\\|TAB\\|RET\\|LFD\\)" . "C-c C-h \\1")
    ("\\bC-c\\( C-h\\| C-v\\) ?RET" . "C-c\\1 C-m")
    ("\\bC-c C-v LFD" . "C-c C-v C-j")
    ("\\bC-c C-h TAB" . "C-c C-h C-i")))


(defun sde-massage-key-descriptions ()
  ;; Fix inconsistencies in the output of describe-bindings.
  (save-excursion
    (let ((translations sde-massage-key-descriptions-list))
      (while translations
	(goto-char 1)
	(while (re-search-forward (car (car translations)) nil t)
	  (replace-match (cdr (car translations)) t))
	(setq translations (cdr translations))))))


(defun sde-describe-bindings (&optional prefix)
  "Show a list of all defined keys, and their definitions.
The list is put in a buffer, which is displayed.

In Emacs 19, when called from a Lisp program, can take an optional argument
PREFIX, which if non-nil, should be a key sequence; then we display only
bindings that start with that prefix."
  (interactive)
  (if sde-running-emacs19
      (describe-bindings prefix)
    (describe-bindings))
  (save-excursion
    (set-buffer "*Help*")
    (sde-massage-key-descriptions)))


;; Modified version of make-help-screen, originally by Lynn Slater, now
;; part of Emacs 19.  Modified so that keys such as C-v are not intercepted.
;; It's stupid to have to do this.

(defmacro sde-make-help-screen (fname help-line help-text helped-map)
  "Constructs function FNAME that when invoked shows HELP-LINE and if a help
character is requested, shows HELP-TEXT. The user is prompted for a character
from the HELPED-MAP and the corresponding interactive function is executed."
  (` (defun (, fname) ()
	   (, help-text)
	   (interactive)
	   (let ((line-prompt (substitute-command-keys (, help-line))))
	     (message line-prompt)
	     (let ((char (read-char))
		   config)
	       (unwind-protect
		   (progn
		     (if (or (char-equal char ??) (char-equal char help-char))
			 (progn
			   (setq config (current-window-configuration))
			   (switch-to-buffer-other-window "*Help*")
			   (erase-buffer)
			   (set-window-hscroll (selected-window) 0)
			   (insert (documentation (quote (, fname))))
			   (goto-char (point-min))
			   (while (memq char (cons help-char '(?? ?\ ?\177)))
			     (if (char-equal char ?\ ) 
				 (scroll-up))
			     (if (char-equal char ?\177)
				 (scroll-down))
			     (message "%s%s: "
				      line-prompt
				      (if (pos-visible-in-window-p (point-max))
					  "" " or Space to scroll"))
			     (let ((cursor-in-echo-area t))
			       (setq char (read-char))))))
		     (let ((defn (cdr (assq (if (integerp char)
						(downcase char)
						char)
					    (, helped-map)))))
		       (if defn
			   (if (keymapp defn)
			       (error "sorry, this command cannot be run from the help screen.  Start over.")
			     (if config
				 (progn
				   (set-window-configuration config)
				   (setq config nil)))
			     (call-interactively defn))
			 (if (listp char) ; Emacs 19
			     (setq unread-command-events
				   (cons char unread-command-events)
				   config nil)
			   (ding)))))
		 (if config
		     (set-window-configuration config))))))))


;;;----------------------------------------------------------------------------
;;; 18. SDE mode
;;;----------------------------------------------------------------------------

(if (not (keymapp sde-find-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-find-cmds)
      (define-key map "\C-h" 'sde-help-for-find-cmds)
      (define-key map "\C-t" 'sde-find-task)
      (define-key map "\C-r" 'sde-find-production-by-rhs)
      (define-key map "\C-p" 'sde-find-problem-space)
      (define-key map "\C-o" 'sde-find-operator)
      (define-key map "\C-n" 'sde-find-production-by-name)
      (define-key map "\C-l" 'sde-find-production-by-lhs)
      (define-key map "\C-b" 'sde-find-production-by-body)
      (setq sde-find-cmds-map map)))


(sde-make-help-screen sde-help-for-find-cmds
  "C-b, C-l, C-n, C-o, C-p, C-r, C-t. [C-h = more help]"
  "Commands for finding components of Soar tasks:

C-n  find production by name
C-b  find production by searching whole body
C-l  find production by searching only left-hand side (i.e., condition-side)
C-r  find production by searching only right-hand side (i.e., action-side)

C-o  find operator
C-p  find problem-space
C-t  find task

?    help on find commands
C-h  help on find commands
C-g  cancel"
  sde-find-cmds-map)


(if (not (keymapp sde-view-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-view-cmds)
      (define-key map "\C-h" 'sde-help-for-view-cmds)
      (define-key map "\C-w" 'sde-view-working-memory)
      (define-key map "\C-t" 'sde-view-ptraces)
      (define-key map "\C-s" 'sde-view-pgs)
      (define-key map "\C-p" 'sde-view-productions)
      (define-key map "\C-o" 'sde-view-pgso)
      (define-key map "\C-m" 'sde-view-ms)
      (define-key map "\C-j" 'sde-view-justifications)
      (define-key map "\C-c" 'sde-view-chunks)
      (define-key map "\C-b" 'sde-view-pbreaks)
      (setq sde-view-cmds-map map)))


(sde-make-help-screen sde-help-for-view-cmds
  "C-b, C-c, C-j, C-m, C-p, C-s, C-t, C-w. [C-h = more help]"
  "Commands for viewing various Soar data:

C-b  view pbreaks
C-c  view chunks
C-j  view justifications
C-m  view match set (Soar \"ms\")
C-o  view operator stack (Soar \"pgso\")
C-p  view productions
C-s  view goal stack (Soar \"pgs\")
C-t  view ptraces
C-w  view contents of Soar working memory

?    help on view commands
C-h  help on view commands
C-g  cancel"
  sde-view-cmds-map)


(if (not (keymapp sde-agent-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-agent-cmds)
      (define-key map "\C-h" 'sde-help-for-agent-cmds)
      (define-key map "\C-s" 'select-agent)
      (define-key map "\C-l" 'list-agents)
      (define-key map "\C-d" 'destroy-agent)
      (define-key map "\C-c" 'create-agent)
      (define-key map "\C-a" 'agent-go)
      (setq sde-agent-cmds-map map)))


(sde-make-help-screen sde-help-for-agent-cmds
  "C-a, C-c, C-d, C-l, C-s. [C-h = more help]"
  "Commands for managing multiple Soar agents:

C-a  agent \"go\" parameters
C-c  create agent
C-d  destroy agent
C-l  list agents known by SDE
C-s  select agent

?    help on agent commands
C-h  help on agent commands
C-g  cancel"
  sde-agent-cmds-map)


(if (not (keymapp sde-region-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-region-cmds)
      (define-key map "\C-h" 'sde-help-for-region-cmds)
      (define-key map "="    'sde-region-count-productions)
      (define-key map ";"    'sde-region-comment)
      (define-key map "\C-x" 'sde-region-excise)
      (define-key map "\C-t" 'sde-region-ptrace)
      (define-key map "\C-s" 'sde-region-send)
      (define-key map "\C-b" 'sde-region-pbreak)
      (setq sde-region-cmds-map map)))


(sde-make-help-screen sde-help-for-region-cmds
  "C-b, C-e, C-t, C-x, \;, =. [C-h = more help]"
  "Commands for operating on regions of Soar code:

C-b  pbreak all productions in region
C-s  send (i.e., load) region to Soar
C-t  ptrace all productions in region
C-x  excise all productions in region
;    comment or uncomment region
=    count productions, lines and characters in region

?    help on region commands
C-h  help on region commands
C-g  cancel"
  sde-region-cmds-map)


(if (not (keymapp sde-help-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-help-cmds)
      (define-key map "\C-h" 'sde-help-for-help-cmds)
      (define-key map "\C-w" 'sde-where-is)
      (define-key map "\C-u" 'sde-soar-user-notes)
      (define-key map "\C-t" 'sde-topic-help)
      (define-key map "\C-s" 'sde-soar-info)
      (define-key map "\C-r" 'sde-soar-release-notes)
      (define-key map "\C-m" 'sde-describe-mode)
      (define-key map "\C-k" 'describe-key)
      (define-key map "\C-i" 'sde-info)
      (define-key map "\C-b" 'sde-describe-bindings)
      (define-key map "\C-a" 'sde-apropos)
      (setq sde-help-cmds-map map)))


(sde-make-help-screen sde-help-for-help-cmds
  "C-a, C-b, C-i, C-k, C-m, C-s, C-t, C-u, C-w. [C-h = more help]"
  "Help interface for SDE and Soar.  Press one of the following keys:

C-m  mode         -- Describe the current mode.
C-b  bindings     -- List all SDE key bindings for the current mode.
C-k  describe key -- Describe the SDE command invoked by a given key sequence.
C-w  where-is     -- Show the key that invokes a given SDE command.

C-a  apropos      -- Briefly list SDE commands and variables matching a string.
C-t  topic        -- Display help on any topic, SDE or Soar.

C-i  info         -- Read the SDE User's Manual using the Info system.

C-s  Soar manual  -- Read the on-line Soar User's Manual using the Info system.
C-r  rel. notes   -- Read the Release Notes for the most recent version of Soar
C-u  user notes   -- Read the User Notes for the most recent version of Soar.

?    help on help commands
C-h  help on help commands
C-g  cancel

The `topic' and `where-is' commands offer completion.  
The `topic' command knows about both Soar and SDE commands and variables, and
will query Soar for help if asked about a Soar topic."
  sde-help-cmds-map)


;; Certain keys are reserved for use in Soar Mode:
;; 
;; c-c c-w  backward-kill-word    	   Like ^w in shells
;; C-c C-o  sde-kill-output                Delete last batch of process output
;; C-c C-u  sde-kill-input                 Like ^u in shells
;; M-p      sde-previous-input             Cycle backwards in input history
;; M-n      sde-next-input                 Cycle forwards
;; M-s      sde-previous-similar-input     Previous similar input
;; M-C-r    sde-previous-input-matching    Search backwards in input history
;;
;; Two key bindings get rebound in Soar Mode to something entirely different:
;;
;; C-c C-r  sde-show-output
;; M-C-r    sde-previous-input-matching

;; The order of the definitions in the following code matter.  When
;; `describe-bindings' is invoked, it will print the items in
;; *reverse* order from the order of their definition.  So keys must
;; be defined in reverse alphabetical order to get them to be printed
;; in alphabetical order when `describe-bindings' and other functions
;; are invoked. In addition, some duplicate keys are purposefully
;; defined earlier than their alternative (for example `C-M-x' before
;; `C-c C-e') so that when `substitute-key-bindings' is run by Emacs
;; for various help functions, it gets the desired key binding (in
;; this case "C-c C-e") before the alternative, because the
;; alterative may not be as mnemonic.  Is that clear? :-)

(unless (keymapp sde-mode-map)
  (let ((map (make-sparse-keymap)))
    (define-key map "\eq"      'sde-reindent)
    (define-key map "\ej"      'indent-new-comment-line)
    (define-key map "\e\;"     'indent-for-comment)
    (define-key map "\e\C-x"   'sde-send-production) ; Like eval-defun
    (define-key map "\e\C-r"   'sde-reposition-window)
    (define-key map "\e\C-q"   'sde-indent-sexp)
    (define-key map "\e\C-j"   'indent-new-comment-line)
    (define-key map "\e\C-h"   'sde-mark-production) ; Like mark-defun
    (define-key map "\e\C-e"   'sde-end-of-production)
    (define-key map "\e\C-a"   'sde-beginning-of-production)

    (define-key map "\C-c\C-z" 'sde-switch-to-soar)
    (define-key map "\C-c\C-y" 'wm)
    (define-key map "\C-c\C-x" 'excise)
    (define-key map "\C-c\C-v" sde-view-cmds-map)
    (define-key map "\C-c\C-t" 'ptrace)
    (define-key map "\C-c\C-s" 'sde-send-production)
    (define-key map "\C-c\C-r" sde-region-cmds-map)
    (define-key map "\C-c\C-n" 'sde-next-match)
    (define-key map "\C-c\C-q" 'firing-counts)
    (define-key map "\C-c\C-p" 'print-soar)
    (define-key map "\C-c\C-l" 'load-soar)
    (define-key map "\C-c\C-h" sde-help-cmds-map)
    (define-key map "\C-c\C-f" sde-find-cmds-map)
    (define-key map "\C-c\C-e" 'explain)
    (define-key map "\C-c\C-c" 'sde-interrupt-soar)
    (define-key map "\C-c\C-b" 'pbreak)
    (define-key map "\C-c\C-a" sde-agent-cmds-map)
    (define-key map "\C-c1"    'sde-hide-soar-error-window)
    (define-key map "\C-c0"    'init-soar)
    (define-key map "\C-c+"    'sde-insert-date-stamp)
    (define-key map "\C-c%"    'sde-query-replace)
    (define-key map "\C-c$"    'sde-search)
    (define-key map "\C-c)"    'sde-find-unbalanced)
    (define-key map "\C-c]"    'sde-close-all-sp)
    (define-key map "\C-c\;"   'sde-region-comment)
    (define-key map "\C-c/"    'preferences)
    (define-key map "\C-c'"    'matches-1)
    (define-key map "\C-c\""   'matches-2)
    (define-key map "\C-c,"    'schedule)
    (define-key map "\C-c."    'go)
    (define-key map "\C-c?"    'sde-why)

    (define-key map "\C-m"     'sde-newline-and-indent)
    (define-key map "\C-j"     'sde-newline-and-indent)
    (define-key map "\t"       'sde-indent-line)
    (define-key map "\177"     'backward-delete-char-untabify)
    (setq sde-mode-map map)))


;; Syntax table.

(unless (syntax-table-p sde-mode-syntax-table)
  (let ((table (copy-syntax-table lisp-mode-syntax-table)))
    (modify-syntax-entry ?\t " "  table)
    (modify-syntax-entry ?\f " "  table)
    ;; Need newline to be comment-end for some of the functions that look
    ;; for productions components to work properly.  Unfortunately, this
    ;; causes "\\s " regexp's to fail to consider newline to be whitespace!
    (modify-syntax-entry ?\n ">"  table)
    (modify-syntax-entry ?   " "  table)
    (modify-syntax-entry ?~  " "  table)
    (modify-syntax-entry ?!  " "  table)
    (modify-syntax-entry ?@  " "  table)
    (modify-syntax-entry ?#  " "  table)
    (modify-syntax-entry ?$  "_"  table) 
    (modify-syntax-entry ?%  "_"  table) 
    (modify-syntax-entry ?^  "_"  table)
    (modify-syntax-entry ?&  "_"  table)
    (modify-syntax-entry ?*  "_"  table)
    (modify-syntax-entry ?\( "()" table)
    (modify-syntax-entry ?\) ")(" table)
    (modify-syntax-entry ?-  "_"  table) 
    (modify-syntax-entry ?_  "_"  table) 
    (modify-syntax-entry ?=  "_"  table)
    (modify-syntax-entry ?+  "_"  table) 
    (modify-syntax-entry ?\{ "(}" table)
    (modify-syntax-entry ?\} "){" table)
    (modify-syntax-entry ?\[ "(]" table)
    (modify-syntax-entry ?\] ")[" table)
    (modify-syntax-entry ?\| "$|" table)
    (modify-syntax-entry ?\  " "  table) 
    (modify-syntax-entry ?\; "<"  table)
    (modify-syntax-entry ?:  "_"  table)
    (modify-syntax-entry ?'  "'"  table)
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?.  " "  table)
    (modify-syntax-entry ?,  " "  table)
    ;; We can't make '<' and '>' be opening and closing delimiters because
    ;; '>' is used in the production arrow, and Emacs would get confused by
    ;; the right angle bracket in "-->".  Too bad.
    (modify-syntax-entry ?\< "'"  table)
    (modify-syntax-entry ?\> "'"  table)
    (modify-syntax-entry ?/  "_"  table)
    (modify-syntax-entry ??  "_"  table) 
    (setq sde-mode-syntax-table table)))

;; Abbrev table.

(when (null sde-mode-abbrev-table)
  (define-abbrev-table 'sde-mode-abbrev-table ()))


;; The actual SDE mode.
;;
;; Note that the key bindings for next and previous history cannot be written
;; as \\[gmhist-previous], etc., because we don't use gmhist in GNU Emacs 19.
;; I had to hardwire the descriptions as "ESC p" and "ESC n".  Some other key
;; descriptions are hardwired too, mainly because FSF 19 seems to catch the
;; menu definition instead of the key binding first, and consequently prints
;; `menu foo-bar biff-wack', which is ugly and almost useless.

(defun sde-mode ()
  "The Soar Development Environment (SDE), a major mode for writing Soar programs.
\\<sde-mode-map>
`\\[sde-beginning-of-production]' moves the cursor to the beginning of a production.
`\\[sde-end-of-production]' moves the cursor to the end of a production.

Commands that move the cursor by words, such as `\\[forward-word]' (forward-word), 
and `\\[backward-word]' (backward-word), consider whitespace and the characters
'^', '*', '-', '_', '<', '>' and '.' to be word delimiters.  In addition, such
commands ignore parentheses, curly braces and whitespace.  Commands that move
the cursor by balanced expressions, such as `\\[forward-sexp]' (forward-sexp) and
`\\[backward-sexp]' (backward-sexp), consider parentheses and curly braces to
be parts of expressions.

`M-x soar' will start Soar in a new buffer.  A prefix argument (such as
\\[universal-argument]) will make it prompt for the name of the Soar program, the
starting directory, and optional command line arguments to pass to Soar.

`\\[sde-send-production]' will send (that is, load) the current production
under the cursor to the current Soar process.  With a prefix argument, it
also switches to Soar after sending the production.

`\\[load-soar]' will prompt for a file name and load it into Soar\; you may use
this to load the current buffer's file into Soar as well as other files.
`M-x load-defaults' will load the most current version of the Soar defaults
file.  `M-x excise-file' will prompt for a file name and will excise all of
the productions found in that file.

The region commands perform common operations on a region in the buffer.
`\\[sde-region-send]' sends everything within the current region to Soar.
`\\[sde-region-pbreak]' issues a pbreak for every production in the region.
`\\[sde-region-ptrace]' issues a ptrace for every production in the region.
`\\[sde-region-excise]' excises every production in the region.
`C-c C-r ;' comments out the region between mark and point.  If given a 
positive prefix arg, it inserts that many semicolons at the beginning of 
each line \(defaulting to 1).  If given a negative prefix argument, it 
uncomments the current region.

If Soar is running in multi-agent mode, and the current buffer is not
associated with any agent, the first command that needs a target agent will
prompt for an agent name and will set the buffer's default to the answer.
To change default agents, or to set the initial agent explicitly, use the
select-agent command (`\\[select-agent]').

`\\[sde-reindent]' reindents whatever the cursor is in.  If the cursor is in a comment,
it reindents the comment.  If in a production, it reindents the whole
production, closing it first if necessary.  If the cursor is in the
documentation string of a production, it reindents the documentation string.
`\\[sde-indent-line]' indents just the current line.  `RET' moves to the next
line and automatically indentations appropriately.  Variable
`sde-production-indent-offset' controls how far clauses are indented underneath
the first line of the production\; the default is 2 spaces.  Variable
`sde-arrow-indent-offset' determines whether the arrow is indented differently\;
negative offsets move it leftward, positive offsets indent the arrow further.

SDE tracks the files that comprise your tasks.  When you visit a Soar file
that is part of a task that is not yet known to SDE, it will prompt you for
the main (i.e., top-most) load file for the task, and then it will
recursively parse each file of the task to build an internal database of the
files and productions that comprise the task.  It will also cache the list of
files into a special file named \".sde\" in the directory where the main load
file is located.  In future editing sessions, SDE will read the \".sde\" file
first and delay the full parsing of files until the information is needed for
certain types of searches \(specifically, for the `sde-find-xxx' commands, 
described below).

All interactive Soar commands are available via `M-x COMMANDNAME'.  Some
commands in SDE have no Soar equivalents, e.g., `excise-file'.  Many key
bindings give direct access to the most common commands.  Some notable
bindings include the following, and a complete list appears later below.  
Use describe-key (\\<global-map>`\\[describe-key]'\\<sde-mode-map>) \(the standard Emacs help facility) to
get detailed help on each command.

  Running and Initializing Soar:

    go                      \\[go]
    schedule                \\[schedule]
    interrupt Soar          \\[sde-interrupt-soar]
    init-soar               \\[init-soar]

  Manipulating Productions:

    send production         \\[sde-send-production]
    send region             \\[sde-region-send]
    load file               \\[load-soar]
    excise production       \\[excise]
    excise region           \\[sde-region-excise]
    
  Tracing and breaking:

    ptrace production       \\[ptrace]
    ptrace region           \\[sde-region-ptrace]
    pbreak production       \\[pbreak]
    pbreak region           \\[sde-region-pbreak]

  Querying Soar for information about specific objects:

    print                   \\[print-soar]
    explain                 \\[explain]
    matches 1               \\[matches-1]
    matches 2               \\[matches-2]
    preferences             \\[preferences]
    firing-counts           \\[firing-counts]
    wm                      \\[wm]

  Querying Soar for general information:

    view ptraces            \\[sde-view-ptraces]
    view pbreaks            \\[sde-view-pbreaks]
    view goal stack (pgs)   \\[sde-view-pgs]
    view op. stack (pgso)   \\[sde-view-pgso]
    view match set (ms)     \\[sde-view-ms]
    view chunks             \\[sde-view-chunks]
    view productions        \\[sde-view-productions]
    view justifications     \\[sde-view-justifications]
    view Soar WM            \\[sde-view-working-memory]

  Finding source code:

    find production by name \\[sde-find-production-by-name]
    find by body (LHS/RHS)  \\[sde-find-production-by-body]
    find by LHS pattern     \\[sde-find-production-by-lhs]
    find by RHS pattern     \\[sde-find-production-by-rhs]
    find operator           \\[sde-find-operator]
    find problem-space      \\[sde-find-problem-space]
    find next match         \\[sde-next-match]

  Searching and replace across whole tasks:

    search for a string     \\[sde-search]
    search for a regexp     \\[sde-search-regexp]
    query-replace           \\[sde-query-replace]
    query-replace-regexp    \\[sde-query-replace-regexp]
    continue search         \\[sde-next-match]

  Controlling multiple agents:
  
    agent-go                \\[agent-go]
    create-agent            \\[create-agent]
    destroy-agent           \\[destroy-agent] (Unavailable prior to Soar 6.1.1)
    list agents             \\[list-agents]
    select-agent            \\[select-agent]

When running in the X Window System, the following mouse buttons are defined:

  Button 	    	      Action
  ------------------    -------------------------------------------------
  SHIFT-left            Execute `print' on item under the cursor
  SHIFT-middle          Execute `matches' on item under the cursor
  SHIFT-right           Execute `preferences' on attribute under the cursor

  META-CONTROL-left     Send to Soar the production under the cursor
  META-CONTROL-middle   Find the file containing the production under cursor
  META-CONTROL-right    Execute `explain' on chunk condition under the cursor

Many Soar commands, such as print, require \"target\" arguments, i.e.,
something to act on.  Those commands that need a production name as target
will use the name of the production that the cursor is in currently, or else
the name of the preceeding sp form.  Those commands that can take any symbol
as target will try to use the symbol under the cursor regardless of whether
it looks like a production name.

Many of the standard Soar commands also take optional arguments -- for
example, the go and print commands.  Giving a prefix argument (such as `\\[universal-argument]')
to most commands makes them prompt for optional arguments.  Optional
arguments are remembered on history lists.  Typing `M-p' and `M-n' at a 
prompt moves backward and forward through the history.

Optional arguments are automatically reused upon subsequent invocations of a
command until they are changed.  This allows you to set an option and then
apply a command repeatedly to different target arguments without having to
respecify the optional arguments each time.  For example, you can type
`\\[universal-argument] \\[print-soar]', type \":depth 3\" to the resulting prompt for optional
arguments, and then repeat this command with the same arguments using
just `\\[print-soar]' on different targets.

The commands ptrace, pbreak, preferences, and explain have different
conventions.  Without arguments, ptrace (`\\[ptrace]') and pbreak (`\\[pbreak]')
act on the production under the cursor.  With a positive argument (such as
`\\[universal-argument]') they perform unptrace or unpbreak, respectively, on the production
under the cursor.  With a negative argument (e.g., `\\[negative-argument]'), they undo all 
ptraces or pbreaks, respectively.  You can print out the current list of
active ptraces and pbreaks using the view commands `\\[sde-view-ptraces]' and 
`\\[sde-view-pbreaks]', respectively.

The preferences command (`\\[preferences]') attempts to extract its arguments
using heuristics.  Place the cursor over the attribube in a printed oject,
for example on \"^io-state\" in the WME
                    (8: S1 ^io-state S1)
and preferences will automatically use the attribute and the WME's identifier 
as arguments.  If the cursor is not over an attribute, it will attempt to
extract the symbol nearest the cursor, and if it is an identifier, then 
preferences will prompt for the attribute.  In addition, if preferences is
given a prefix argument (`\\[universal-argument]'), it will prompt for the optional
\"level\" argument.

The explain command (`\\[explain]') also attempts to extract its arguments
using heuristics.  Remember that before explain may be used, Soar must be
instructed to track the necessary information using \"explain :on\" \(or the
SDE equivalent `M-x explain-on').  The easiest way to use explain in SDE
is to print out a chunk and then place the cursor over an attribute in
the condition side of the chunk.  Explain will then query Soar with the
name of the chunk and the condition that tests that attribute.  \(It does
this by constructing a condition clause consisting of the appropriate
identifier-attribute-value slot, and silently querying Soar to determine
the correct condition number, then finally sending the command 
\"explain <production-name> <condition-number>\" to Soar.\)  You may also
put the cursor over a chunk name and invoke explain, and SDE will extract
just the production name and issue \"explain <name>\" to Soar.  Finally,
if you are looking at the text of a previous \"explain <name>\" and you
wish to do \"explain <name> <condition>\", simply put the cursor on the
line containing the condition number and invoke explain, and SDE will issue
\"explain <production-name> <condition-number>\" to Soar.  If given a prefix
argument, explain will prompt for its arguments, offering as initial defaults
any production name and condition clause it can find.

Commands that only query Soar for information place the output in in a
separate buffer for viewing, unless the variable `sde-soar-use-output-buffer'
is nil, in which case all output goes to the agent process buffer.
New output is appended to the output buffer, each batch separated by a
title line, unless the variable `sde-soar-erase-output-buffer' is non-nil,
in which case the output buffer's existing contents will be cleared before
new output is written out.

When running in Emacs 19 or Lucid Emacs 19, SDE will attempt to use a
separate X Window screen (called a \"frame\" in Emacs 19) for the output 
buffer, unless the variable `sde-use-multiple-frames' is nil.  The
initial attributes of this buffer are determined by the list of values in
the variable `sde-soar-output-buffer-defaults'.  Use Emacs help on this
variable (`\\[describe-variable]') for more information about how to use
this feature.  

The command `\\[sde-find-production-by-name]' locates the source code for a
production whose name is under the cursor.  Put the cursor on a production in
your task, either in a Soar process buffer or an editing buffer, type
`\\[sde-find-production-by-name]', and it will jump to the place in your
files where the production is defined.

The command `\\[sde-find-production-by-lhs]' (`sde-find-production-by-lhs') allows you 
to search a task for all productions that contain a certain pattern in their
condition sides.  When invoked, the command prompts you for a search pattern.
The pattern must be a list similar in form to production conditions.
For example,
             ( (goal <g> ^problem-space <p>) (<p> ^name top-ps) )

is a pattern that will match all productions whose conditions test for a
problem space named \"top-ps\".  \(The actual variable names that you use are
irrelevant because the system is doing true pattern-matching search.)  The
names of the productions will be listed in a separate window, and the source
code of the first production will be shown in a second window.  You can view
the source code of each successive production by repeatedly typing
`\\[sde-next-match]' (`sde-next-match').  Alternatively, you can move the
cursor over a production name and use `\\[sde-find-production-by-name]'.

The command `\\[sde-search]' (`sde-search') lets you search the files of a task for a
string.  When invoked, this command prompts you for a search string and then
searches the files of the current task, stopping at the first match.  Use the
command `\\[sde-next-match]' to find the next occurrence of the string in the
files.  The command `\\[sde-search-regexp]' is identical, but lets you supply
a regular expression instead of a plain string.  Files are searched in
alphabetical order by default, unless the variable `sde-sort-lists' is set
to `nil'.

The command `\\[sde-query-replace]' performs an interactive query-replace
across the files of a task.  When invoked, this command prompts you for a
string to search for and a string with which to replace it.  If given a
prefix argument, it also prompts for the task in which to search\; otherwise,
it uses the task with which the current buffer is associated.  It then
performs a `query-replace' in each file of the task.  Type \\[help-command]
at a prompt for directions about what to do.  If you exit the query-replace
\(using `\\[keyboard-quit]' or ESC), you can resume the query-replace with the
command `\\[sde-next-match]'.  The command `\\[sde-query-replace-regexp]' is
similar but allows you to supply a regular expression instead of a plain
string.  The commands `\\[sde-replace-string]' and 
`\\[sde-replace-string-regexp]' are noninteractive versions of these commands.

The SDE on-line help facility is available on the `C-c C-h' prefix.
`\\[sde-apropos]' (apropos) prompts for a string and lists a brief description of
all SDE user commands and user variables that contain the given string.
`\\[sde-topic-help]' (topic) prompts for a command, variable or Soar help topic, and 
displays information about it.  This command interfaces to the Soar help 
facility, and will query Soar for help on topics that are not already defined
in SDE.  Note that `\\[sde-topic-help]' provides completion, which means you
can type `?' any time at the prompt to list possible completions of what you 
have typed so far, and you can type `TAB' to have Emacs complete the rest of
what you have typed based on known help topics.

`\\[sde-soar-info]' will bring up the Soar 6 User's manual using the Emacs
Info browser.  `C-c C-h C-i' will bring up the SDE manual.  `\\[sde-soar-release-notes]' 
will bring up the Release Notes for the latest revision of Soar, while 
`\\[sde-soar-user-notes]' will bring up the User Notes.  `\\[sde-describe-bindings]' will list the 
local bindings in the current SDE mode, and `\\[sde-where-is]' will prompt for
an SDE command name (with completion) and report which key binding
invokes it.

`\\[sde-close-all-sp]' will close the current sp form by adding or removing right parentheses
as necessary.

`\\[sde-region-count-productions]' will count the number of productions, lines
and characters the mark and the point, and report the answer.

`\\[backward-delete-char-untabify]' converts tabs to spaces as it moves back.

`\\[sde-reposition-window]' will make the current production and/or comment visible.
If the production is fully onscreen, it is moved to the top of the window.
If it is partly offscreen, the window is scrolled to get the definition (or
as much as will fit) onscreen, unless point is in a comment which is also
partly offscreen, in which case the scrolling attempts to get as much of the
comment onscreen as possible.  Repeated invocations of `\\[sde-reposition-window]' move the
production to the top of the window or toggle the visibility of comments that
precede it.

`\\[sde-find-unbalanced]' searches for unbalanced parentheses in the current buffer.

`\\[sde-switch-to-soar]' will switch you to the Soar process buffer.

To send a bug report, questions or other feedback to the authors and 
maintainers of the Soar Development Environment, please use `\\[sde-feedback]'.

Entry to this mode will run the functions on the list variable `sde-mode-hook'.

Here is a list of all the special bindings in Soar mode.  The name of the key
map is `sde-mode-map'.  For further help on individual commands, type
\\<global-map>`\\[describe-key] KEY' where KEY is the keystroke.

\\<sde-mode-map>\\{sde-mode-map}"

  (interactive)
  (sde-mode-internal)
  ;; !!! This is a hack that must be removed when the header code
  ;; is cleaned up.  This is to enable users to set sde-header-hooks
  ;; now, even though header.el uses "make-header-hooks".
  (make-local-variable 'make-header-hooks)
  (when (boundp 'sde-header-hooks)
    (setq make-header-hooks sde-header-hooks))
  (run-hooks 'sde-mode-hook))


(defun sde-mode-internal ()
  "Function for performing initializations common to derivatives of SDE mode.
It does almost everything that sde-mode does, except for running hooks and
certain other initializations that only make sense for file buffers."
  ;; Set standard Emacs variables.
  (setq mode-name "SDE"
	major-mode 'sde-mode)
  (use-local-map sde-mode-map)
  ;; Build menus if appropriate.  Although in FSF 19 we could build menus at
  ;; load time, in Lucid the menus have to be defined for each buffer, so
  ;; it's easiest just to let a function handle it for both cases.
  (when (and window-system (or sde-running-emacs19 sde-running-lemacs))
    (sde-define-menus 'sde-mode sde-mode-map))
  (set-syntax-table sde-mode-syntax-table)
  (setq local-abbrev-table sde-mode-abbrev-table)

  (make-local-variable 'comment-column)
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'comment-indent-function)
  (make-local-variable 'indent-line-function)
  (make-local-variable 'paragraph-start)
  (make-local-variable 'paragraph-separate)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq comment-column       40
	comment-start        ";"
	comment-start-skip   ";+ *"
	indent-line-function 'sde-indent-line
	paragraph-start      "^(sp\\|^;+\\|^[ \t\f\n]*$"
	paragraph-separate   "^;+\\|^[ \t\f\n]*$"
	parse-sexp-ignore-comments t)
  (setq comment-indent-function  'lisp-comment-indent)
  (make-local-variable 'lisp-indent-function)
  (setq lisp-indent-function 'sde-indent-hook)

  ;; SDE Soar Mode variables
  (make-local-variable 'sde-buffer-data)
  (setq sde-buffer-data (sde-make-buffer-struct))

  ;; Set up the mode line.
  (setq	mode-line-modified '("-%1*%1* ")
	mode-line-buffer-identification '("%24b")
	mode-line-format
	(list "" 'mode-line-modified 'mode-line-buffer-identification
	      "  {"
	      '(sde-soar-agents
		;; Multi-agent case
		((sde-soar-buffer-agent sde-soar-buffer-agent "no agent")
		 (sde-soar-buffer-agent ": ")
		 (sde-soar-buffer-agent sde-soar-state-string))
		;; Single-agent case
		sde-soar-state-string)
	      "}  "
	      'global-mode-string
	      " %[("
	      'mode-name 'minor-mode-alist
	      ")%n%] --"
	      (if (or sde-running-emacs19 sde-running-lemacs)
		  '(line-number-mode "L%l--")
		"")
	      '(-3 . "%p") "-%-")))


;;;----------------------------------------------------------------------------
;;; 20. Mouse support under the X Window System.
;;;----------------------------------------------------------------------------


(defconst sde-mouseable-modes
    (append sde-source-modes sde-soar-modes)
  "List of SDE modes in which mouse actions are permissible.")


(defun sde-mouse-in-sde-buffer (event)
  ;; Return non-nil if the mouse is over an SDE buffer of some sort.
  ;; Lucid Emacs 19 version.
  (interactive "@e")
  (save-excursion
    (set-buffer (window-buffer (selected-window)))
    (memq major-mode sde-mouseable-modes)))


(defmacro sde-with-mouse-click (arg &rest body)
  (` (if (sde-mouse-in-sde-buffer (, arg))
	 (progn
	   (require 'sde-soar-mode)
	   (mouse-set-point (, arg))
	   (,@ body)))))


(defun sde-mouse-print-soar (args)
  "Execute Soar \"print\" on the item under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'print-soar)))


(defun sde-mouse-matches (args)
  "Execute Soar \"matches\" on the item under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'matches)))


(defun sde-mouse-preferences (args)
  "Execute Soar \"preferences\" on the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'preferences)))


(defun sde-mouse-find-production-by-name (args)
  "Execute `sde-find-production' on the production name under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (call-interactively 'sde-find-production-by-name)))


(defun sde-mouse-send-production (args)
  "Send (load) the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'sde-send-production)))


(defun sde-mouse-ptrace-production (args)
  "Execute Soar \"ptrace\" on the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'ptrace)))


(defun sde-mouse-pbreak-production (args)
  "Execute SDE's \"pbreak\" on the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'pbreak)))


(defun sde-mouse-explain (args)
  "Execute SDE's \"explain\" on the chunk condition under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'explain)))


(when window-system
  (cond (sde-running-lemacs
	 ;; Use the down events.  Lucid Emacs 19.6 only defines actions for the
	 ;; down events of:  button1, (shift button1), (control button1), and
	 ;; (control shift button1).
	 (define-key sde-mode-map '(shift button1)        'sde-mouse-print-soar)
	 (define-key sde-mode-map '(shift button2)        'sde-mouse-matches)
	 (define-key sde-mode-map '(shift button3)        'sde-mouse-preferences)
	 (define-key sde-mode-map '(control meta button1) 'sde-mouse-send-production)
	 (define-key sde-mode-map '(control meta button2) 'sde-mouse-find-production-by-name)
	 (define-key sde-mode-map '(control meta button3) 'sde-mouse-explain))

	(sde-running-emacs19
	 (define-key sde-mode-map [S-mouse-1] 'sde-mouse-print-soar)
	 (define-key sde-mode-map [S-mouse-2] 'sde-mouse-matches)
	 (define-key sde-mode-map [S-mouse-3] 'sde-mouse-preferences)
	 (define-key sde-mode-map [C-M-mouse-1] 'sde-mouse-send-production)
	 (define-key sde-mode-map [C-M-mouse-2] 'sde-mouse-find-production-by-name)
	 (define-key sde-mode-map [C-M-mouse-3] 'sde-mouse-explain))))


;;;----------------------------------------------------------------------------
;;; 21. Font and color support.
;;;----------------------------------------------------------------------------

(when window-system
  (require 'sde-highlight))


;;;----------------------------------------------------------------------------
;;; 21. Load and set-up commands for SDE
;;;----------------------------------------------------------------------------

;;; THIS MUST ALWAYS BE UPDATED TO REFLECT CURRENT SET OF COMMANDS AND VARIABLES.

;;; The following list serves two main purposes:
;;;   1) set up autoloading of other functions
;;;   2) create completion tables, for user input completion during help, etc.
;;;
;;; Using an explicit list like this is not ideal, because it means it must
;;; be updated whenever the associated files are modified.  Unfortunately I
;;; cannot find a better way.  We need some of the symbols to be available
;;; all the time for input completion, but their definitions may be located
;;; in separate files, and thus may not be loaded until later.  This rules
;;; out, for example, putting lists right with the individual files and
;;; concatenating them when the files are loaded, or defining new
;;; defun/defvar functions that automatically update the completion tables.
;;; We're left with a bit of a maintenance headache.  Is it worth it?  Don't
;;; know yet.

(defvar sde-symbols
  ;; Lists of lists.  File name, then triples of (symbol autoload help) 
  ;; The autoload field can only be t for functions, and
  ;; the completion field is one of: function, variable, or nil.

  '(("sde-compat"
     ;; Symbol			    Autoload?  Completion?
     (sde-compat-auto-update		nil	variable)
     (production-editor-mode		t	function)
     (soar-mode				t	function))

    ("sde-feedback"
     (sde-feedback-cc			nil	variable)
     (sde-feedback-self-blind		nil	variable)
     (sde-feedback-archive-file-name 	nil	variable)
     (sde-feedback-setup-hook		nil	variable)
     (sde-feedback			t	function))

    ("sde-find"
     (sde-find-production-by-name       t       function)
     (sde-find-production-by-body       t       function)
     (sde-find-production-by-lhs	t       function)
     (sde-find-production-by-rhs	t       function)
     (sde-find-production-operator	t       function)
     (sde-find-production-problem-space t       function))

    ("sde-header"
     (sde-header-hooks			nil	variable)
     (sde-header-soar-version		t	function))

    ("sde-repos"
     (sde-reposition-window		t	function))

    ("sde-search"
     (sde-query-replace-highlight	nil	variable)
     (sde-search			t	function)
     (sde-search-regexp			t	function)
     (sde-query-replace			t	function)
     (sde-query-replace-regexp		t	function)
     (sde-replace-string		t	function)
     (sde-replace-regexp		t	function))

    ("sde-soar-mode"
     (sde-soar-mode			t 	function)
     (sde-soar-output-mode	        t	function)
     (soar				t	function)
     (go				t	function)
     (run				t	function)
     (run-soar				t	function)
     (reset				t	function)     
     (load-errors			t	function)
     (load-errors-on			t	function)
     (load-errors-off			t	function)
     (init-soar				t	function)
     (pbreak				t	function)
     (unpbreak				t	function)
     (ptrace				t	function)
     (unptrace				t	function)
     (explain				t	function)
     (explain-on			t	function)
     (explain-off			t	function)
     (firing-counts			t	function)
     (list-chunks			t	function)
     (list-justifications		t	function)
     (list-productions			t	function)
     (matches				t	function)
     (matches-1				t	function)
     (matches-2				t	function)
     (memory-stats			t	function)
     (ms				t	function)
     (pgs				t	function)
     (pgso				t	function)
     (preferences			t	function)
     (print-soar			t	function)
     (print-stats			t	function)
     (rete-stats			t	function)
     (wm				t	function)
     (sde-view-chunks			t	function)
     (sde-view-pbreaks			t	function)
     (sde-view-ptraces			t	function)
     (sde-view-working-memory		t	function)
     (excise				t	function)
     (excise-chunks			t	function)
     (excise-task			t	function)
     (excise-all			t	function)
     (excise-file			t	function)
     (init-soar				t	function)
     (load-soar				t	function)
     (load-defaults 			t	function)
     (select-agent			t	function)
     (schedule				t	function)
     (agent-go				t	function)
     (create-agent			t	function)
     (destroy-agent			t	function)
     (list-agents			t	function)     
     (soarnews				t	function)
     (max-elaborations			t	function)
     (max-chunks			t	function)
     (max-chunks-on			t	function)
     (max-chunks-off			t	function)
     (max-elaborations-on		t	function)
     (max-elaborations-off		t	function)
     (sde-switch-to-soar		t	function)
     (sde-close-and-send		t	function)
     (sde-send-production		t	function)
     (sde-region-send			t	function)
     (sde-region-pbreak			t	function)
     (sde-region-ptrace			t	function)
     (sde-region-excise			t	function)
     (sde-previous-input		t	function)
     (sde-next-input			t	function)
     (sde-next-input-matching		t	function)
     (sde-previous-input-matching	t	function)
     (sde-previous-similar-input	t	function)
     (sde-interrupt-soar		t	function)
     (sde-kill-output			t	function)
     (sde-show-output			t	function)
     (sde-kill-input			t	function)
     (sde-backward-prompt		t	function)
     (sde-forward-prompt		t	function)
     (sde-return			t	function)
     (sde-bol				t 	function)
     (sde-soar-track-cd-toggle		t	function)
     (track-cd-toggle			t	function)
     (sde-kill-buffer-agent-hook        t       nil)
     (sde-pop-up-error-buffer		t	nil)
     (sde-hide-soar-error-window	t	nil)
     (sde-soar-error			t	nil)
     (sde-agent				t	nil)
     (sde-process-input-start		t 	nil)
     (sde-get-old-input			t 	nil)
     (sde-soar-cmd			t	nil)
     (sde-check-soar			t	nil)
     (sde-check-soar-multi-agent	t	nil)
     (sde-soar-is-alive			t	nil)
     (sde-agent				t	nil)
     (sde-pbreak-in-effect		t	nil)
     (sde-unpbreak-production		t	nil)
     (sde-update-pbreak-list		t	nil)
     (sde-soar-help			t	nil))

    ("sde"
     (sde-site-hook			nil	variable)
     (sde-load-hook			nil	variable)
     (sde-mode-hook			nil	variable)
     (sde-soar-mode-hook		nil	variable)
     (sde-soar-hook			nil	variable)
     (sde-soar-output-hook		nil	variable)
     (sde-soar-error-hook		nil	variable)
     (sde-soar-program			nil	variable)
     (sde-soar-starting-directory	nil	variable)
     (sde-soar-switches			nil	variable)
     (sde-prompt-for-soar-switches	nil	variable)
     (sde-soar-use-ptys			nil	variable)
     (sde-soar-defaults-file		nil	variable)
     (sde-file-types			nil	variable)
     (sde-inhibit-record-file-data	nil	variable)
     (sde-use-multiple-frames		nil	variable)
     (sde-production-indent-offset	nil	variable)
     (sde-arrow-indent-offset		nil	variable)
     (sde-soar-beep-after-setup		nil	variable)
     (sde-soar-use-output-buffer	nil	variable)
     (sde-soar-erase-output-buffer	nil	variable)
     (sde-soar-move-point-on-output	nil	variable)
     (sde-soar-output-buffer-defaults	nil	variable)
     (sde-soar-error-buffer-defaults 	nil	variable)
     (sde-soar-track-cd			nil	variable)
     (sde-soar-input-ring-size		nil	variable)
     (sde-soar-input-ring-filter	nil	variable)
     (sde-production-name-test-regexp 	nil	variable)
     (sde-mode-map			nil	variable)
     (sde-view-cmds-map			nil	variable)
     (sde-agent-cmds-map		nil	variable)
     (sde-region-cmds-map		nil	variable)
     (sde-help-cmds-map			nil	variable)
     (sde-soar-mode-map			nil	variable)
     (sde-soar-output-mode-map		nil	variable)
     (sde-mode-syntax-table		nil	variable)
     (sde-mode-abbrev-table		nil	variable)
     (sde-go-args			nil	variable)
     (sde-run-args			nil	variable)
     (sde-matches-args			nil	variable)
     (sde-ms-args			nil	variable)
     (sde-firing-counts-args		nil	variable)
     (sde-print-args			nil	variable)
     (sde-preferences-args		nil	variable)
     (sde-list-productions-args		nil	variable)
     (sde-list-chunks-args		nil	variable)
     (sde-list-justifications-args	nil	variable)
     (sde-agent-go-args			nil	variable)
     (sde-schedule-args			nil	variable)
     (sde-info-file			nil	variable)
     (sde-soar-info-file		nil	variable)
     (sde-soar-release-notes-file	nil	variable)
     (sde-soar-user-notes-file		nil	variable)
     (sde-beginning-of-production	nil	function)
     (sde-end-of-production		nil	function)
     (sde-mark-production		nil	function)
     (sde-cd				nil	function)
     (sde-close-all-sp			nil	function)
     (sde-newline-and-indent		nil	function)
     (sde-indent-line			nil	function)
     (sde-indent-sexp			nil	function)
     (sde-reindent			nil	function)
     (sde-region-comment		nil	function)
     (sde-region-count-productions	nil	function)
     (sde-close-and-send		nil	function)
     (sde-find-unbalanced		nil	function)
     (sde-find-unbalanced-region	nil	function)
     (sde-reposition-window		nil	function)
     (sde-insert-date-stamp		nil	function)
     (sde-task				nil 	function)
     (sde-reset-tasks			nil 	function)
     (sde-list-tasks			nil 	function)
     (sde-next-match			nil 	function)
     (sde-mode				nil	function)
     (sde-apropos			nil	function)
     (sde-info				nil	function)
     (sde-news				nil	function)
     (sde-soar-info			nil	function)
     (sde-soar-release-notes		nil	function)
     (sde-topic-help			nil	function)
     (sde-command-help                  nil     function)
     (sde-variable-help			nil     function)
     (sde-where-is			nil	function)
     (sde-describe-mode			nil	function)
     (sde-describe-bindings		nil 	function)
     (sde-mouse-print-soar		nil	function)
     (sde-mouse-matches			nil	function)
     (sde-mouse-preferences		nil	function)
     (sde-mouse-find-production-byd-name nil	function)
     (sde-mouse-send-production		nil	function)
     (sde-mouse-ptrace-production	nil	function)
     (sde-mouse-explain			nil	function))))


(defmacro sde-set-up-file-symbols (data)
  (` (let ((blurb "A yet-to-be-loaded Soar Development Environment command.")
	   (file (car (, data)))
	   (type nil)
	   (sym nil))
       (mapcar (function
		(lambda (triple)
		  (setq sym  (car triple)
			type (car (cdr (cdr triple))))
		  (when (car (cdr triple))
		    (autoload sym file blurb t))
		  ;; Type field non-nil ==> include in completion tables.
		  (cond ((eq type 'function)
			 (intern (prin1-to-string sym) sde-commands-obarray))
			((eq type 'variable)
			 (intern (prin1-to-string sym) sde-variables-obarray)))))
	       (cdr (, data))))))


;; Now set it all up.

(mapcar (function (lambda (x) (sde-set-up-file-symbols x)))
         sde-symbols)


;;;----------------------------------------------------------------------------
;;; 22. Closing statements.
;;;----------------------------------------------------------------------------

;; Attempt to load site-specific file.  If it doesn't exist, don't complain.

(load "sde-site.el" 'failure-ok)

;; Provide now, in case user's code checks this.

(provide 'sde)

;; Run user hooks.

(run-hooks 'sde-site-hook)
(run-hooks 'sde-load-hook)

;; Set up the file extensions which invoke soar-mode.  The variable
;; `sde-file-types' contains the list of file suffixes.  If you wish to add
;; to this list, do a setq of `sde-file-types' prior to loading this file.
;; (For example, you can do this in the site-hook or load-hook.)

(mapcar (function
	 (lambda (suffix)
	   (unless (assoc suffix auto-mode-alist)
	     (setq auto-mode-alist (cons (cons suffix 'sde-mode)
					 auto-mode-alist)))))
	sde-file-types)

;; Set up initial help topics list.  Whenever Soar is started up, this
;; is reset to include the help topics that Soar knows about.  

(mapatoms
 (function (lambda (name) (intern (prin1-to-string name) sde-topics-obarray)))
 sde-commands-obarray)

(mapatoms
 (function (lambda (name) (intern (prin1-to-string name) sde-topics-obarray)))
 sde-variables-obarray)

;;; Emacs indentation support for macros and form hooks for edebug.
;;;
;;; Local Variables:
;;; eval:(put 'sde-with-hacked-delimiter-syntax 'lisp-indent-function 1)
;;; eval:(put 'sde-with-hacked-delimiter-syntax 'lisp-indent-hook 1)
;;; eval:(put 'sde-with-hacked-delimiter-syntax 'edebug-form-spec '(&rest form))
;;; eval:(put 'sde-make-help-screen    'lisp-indent-function 1)
;;; eval:(put 'sde-make-help-screen    'lisp-indent-hook 1)
;;; eval:(put 'sde-with-mouse-click    'lisp-indent-function 1)
;;; eval:(put 'sde-with-mouse-click    'lisp-indent-hook 1)
;;; eval:(put 'sde-make-hash-key       'edebug-form-spec '(form))
;;; eval:(put 'sde-puthash             'edebug-form-spec '(form form &optional form))
;;; eval:(put 'sde-gethash             'edebug-form-spec '(form &optional form))
;;; eval:(put 'sde-map-hashtable       'edebug-form-spec '(form form))
;;; eval:(put 'sde-substring           'edebug-form-spec '(form integerp &optional form))
;;; eval:(put 'sde-buffer-substring    'edebug-form-spec '(integerp))
;;; eval:(put 'sde-prompt              'edebug-form-spec '(form form))
;;; eval:(put 'sde-get-buffer-create   'edebug-form-spec '(form))
;;; eval:(put 'sde-buffer-file-name    'edebug-form-spec '(form))
;;; eval:(put 'sde-make-name-regexp    'edebug-form-spec '(form))
;;; eval:(put 'sde-with-mouse-click    'edebug-form-spec '(form &rest (form)))
;;; eval:(put 'sde-set-up-file-symbols 'edebug-form-spec '(form))
;;; End:
