;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /u/hucka/Projects/Soar/Interface/Src/RCS/sde-soar-mode.el,v $
;;;; $Id: sde-soar-mode.el,v 0.165 1994/06/08 04:54:18 hucka Exp $
;;;; 
;;;; Description       : Soar Mode portion of SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;; 
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1993 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-soar-mode-el-version "$Revision: 0.165 $"
  "The revision number of sde-soar-mode.el.  The complete RCS id is:
      $Id: sde-soar-mode.el,v 0.165 1994/06/08 04:54:18 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Macros
;;;; 5.  Soar Mode
;;;; 6.  Interactive Soar commands
;;;; 6.A.  Commands for execution control
;;;; 6.B.  Commands for tracing and breaking
;;;; 6.C.  Commands for querying Soar
;;;; 6.D.  Commands for manipulating Soar memories
;;;; 6.E.  Commands for dealing with multiple agents
;;;; 6.F.  Help commands
;;;; 6.G.  Misc. commands
;;;; 7.  Command line interface in Soar process buffers.
;;;; 8.  Interface to Soar
;;;; 9.  Support for pbreak
;;;; 10. Support for tracking productions
;;;; 11. Miscellaneous support code
;;;; 12. Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; -----------------
;;;; 0.  Documentation
;;;; -----------------
;;;;
;;;; Soar process interaction and basic Soar editing commands.
;;;;
;;;; This was at first implement using comint-mode, but when multi-agent Soar
;;;; handling was added, comint-mode functions had to be replaced.  The
;;;; problem is that the comint functions assume there is only one buffer per
;;;; process.  To handle multi-agent Soar, we need to provide multiple
;;;; "pseudo" process buffers, one for each Soar agent.  The comint functions
;;;; make assumptions about being able to do (get-process (current-buffer))
;;;; in the current buffer, which then wouldn't work.  So, nearly all the
;;;; comint functions had to be rewritten.
;;;;
;;;; This module therefore reimplements comint-like functionality in the
;;;; process buffers, at the unfortunate cost of recreating a lot of code.
;;;;
;;;; .... (more documentation under construction here)...
;;;; 


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Requirements

(require 'sde)
(require 'comint)
(require 'assoc)

;; Provide is at the end.


;;;-----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;-----------------------------------------------------------------------------

;;; Maintained in sde.el


;;;-----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;-----------------------------------------------------------------------------


(defvar sde-soar-default-name "soar"
  "String that is the default name for single-agent Soar communication buffers.
If Soar is running in single agent mode, the buffer is named
\"*SDE-SOAR-DEFAULT-NAME*\".  If in multi-agent mode, agent buffers are named
\"*AGENT-NAME*\" where AGENT-NAME is the name of the agent.  For example, a
multi-agent Soar running two agents named \"egg-head\" and \"tough-guy\"
might have two associated buffers named \"*egg-head*\" and \"*tough-guy*\".")

(defconst sde-soar-started-multi-regexp
    (concat "Multiple agent mode selected\\|Soar agent \\([^>]+\\)?>")
  "Regexp to recognize when Soar has started up in multi-agent mode.")

(defconst sde-soar-sched-regexp "\nSelecting agent \\(\\(\\s_\\|\\sw\\)*\\)"
  "Regexp to recognize when multi-agent Soar switches agents.
This must include a leading newline, because the printf statement in the
Soar source code includes a leading newline.")

(defconst sde-soar-agent-creation-regexp "Creating agent \\([^.\n]+\\)[.\n]"
  "Regexp to recognize the names of agents when Soar says they're created.")

(defconst sde-soar-agent-destruction-regexp "Destroying agent \\([^.\n]+\\)[.\n]"
  "Regexp to recognize the names of agents when Soar says they're destroyed.")

(defconst sde-soar-loading-regexp "^Loading \\([^ \t\n]+\\)"
  "Regexp to match Soar messages of loading files.")

(defconst sde-soar-chdir-regexp "^Changing to directory:?\\s *\\([^ \t\n]+\\)"
  "Regexp to match Soar saying it changed to a directory.")

(defconst sde-soar-returning-regexp "^Returning from directory\\s *\\([^ \t\n]+\\)"
  "Regexp to match Soar saying it returned from a directory.")

(defconst sde-soar-loading-or-chdir-regexp
  (concat sde-soar-loading-regexp "\\|" sde-soar-chdir-regexp "\\|"
	  sde-soar-returning-regexp)
  "Regexp to match either a loading file message or a chdir message.")

(defconst sde-soar-pwd-output-regexp "Current directory:\\s *\\([^ \t\n]+\\)"
  "Regexp to match Soar reply to pwd commands.")

(defconst sde-soar-first-goal-regexp "^[ \t]*0: ==>G:\\|^=>WM:"
  "Regexp to match sign of Soar starting a run.")

(defconst sde-soar-running-regexp
  (concat sde-soar-sched-regexp "\\|" sde-soar-first-goal-regexp
	  "\\|" sde-soar-agent-creation-regexp)
  "Regexp to recognize when Soar has starting running without immediately
issuing a prompt.  This happens if the user has a .init file containing load,
run, go or schedule statements.")

(defconst sde-soar-startup-error-regexp
  "Couldn't exec\\|syntax error"
  "Regexp to match errors in starting up Soar.")

(defconst sde-soar-runtime-error-regexp
  "exited abnormally\\|core dumped"
  "Regexp to match errors in the process sentinel `sde-soar-process-sentinel'.")

(defconst sde-soar-load-error-regexp
  "^Continue\\?\nEnter 'n' or 'N' to Stop, <Return> to Continue\n"
  "Regexp to match error message issued during loads when load-errors is on.")

(defconst sde-soar-explain-error-regexp
  "Could not find the chunk.  Maybe explain was not on when it was created."
  "Regex to match error message issued when `explain' is invoked wrongly.") 

(defvar sde-soar-error-regexp
  "^Error: \\|^Internal error: \\|^Expected \\|^No production named \
\\|Expected --> in production\
\\|^Ignoring \\|^There is no \\|^The current [^ \t]+ is not\
\\|^Matches 'level' must be \\|^Illegal argument\\|^Bad argument\
\\|^Invalid argument\\|^Illegal value\\|^'Level' argument must be \
\\|^MS level of detail must be\
\\|^That goal stack level doesn't exist right now\
\\|^No RHS function named\\|^Function [^ \t]+ cannot be used\
\\|^Function [ \t]+ can only be used\
\\|^Could not find the chunk\
\\|^Wrong number of arguments"
  "Regular expression to match error messages from Soar.")

(defconst sde-soar-interrupt-regexp
  "\*\*\* Ctrl-C Interrupt \*\*\*"
  "Regexp to match message printed by Soar when it's interrupted.")

(defconst sde-soar-interrupt-fmt
  "\n*** Ctrl-C Interrupt ***\n"
  "Format string for fake ^C interrupt messages.")

(defconst sde-soar-prompt-str "Soar> "
  "Format string for fake prompts in a single agent process buffer.")

(defconst sde-soar-agent-prompt-fmt "Soar agent %s> "
  "Format string for fake prompts issued in Soar agent interaction buffers.
This should imitate whatever Soar really uses for a prompt string.")

(defconst sde-soar-agent-select-fmt "select-agent %s"
  "Format string for commands to select the current agent in Soar.")

(defconst sde-soar-version-regexp
  "Soar version:\\s *\\(\\([67]\\)\.\\(-?[0-9]+\\)\.?\\(-?[0-9]*\\)?\\)"
  "Regexp to match the number in the string returned by the \"version\" cmd.
It is a multi-element match:  
  * match element 1 is the whole version number
  * match element 2 is the X in a version number such as x.y.z
  * match element 3 is the Y in x.y.z
  * match element 4 is the Z in x.y.z.")

(defconst sde-soar-output-truncation-threshold 1000000
  "Maximum size of a process buffer before SDE starts truncating it.")

(defconst sde-soar-output-truncation-amount 100000
  "Number of bytes by which old Soar output is removed from Soar process
buffers when the buffer size exceeds `sde-soar-output-truncation-threshold.")

(defvar sde-soar-version "6"		; This gets reset once Soar starts.
  "Soar's version number in string form.")

(defvar sde-soar-version-at-least-6-1-1 nil
  "True if the currently running copy of Soar is version 6.1.1 or later.")

(defvar sde-soar-input-ring nil
  "Input ring for process interaction buffers.")

(defvar sde-soar-input-ring-index
  "Index variable for process input history ring.")

(defvar sde-soar-last-input-match ""
  "Last string searched for by input history search, for defaulting.
Buffer local variable.") 

(defvar sde-soar-last-input-end 0
  "Location of last input in buffer.")

(defvar sde-soar-last-dir/file nil
  "List of previously accessed files in Soar process or edit buffers.")

(defvar sde-soar-buffer-mark nil
  "Buffer-local mark for current start of input in a Soar Mode buffer.")

(make-variable-buffer-local 'sde-soar-buffer-mark)

(defvar sde-soar-process nil
  "The Soar process.")

(defvar sde-soar-buffer nil
  "The Soar process buffer.")


;;;-----------------------------------------------------------------------------
;;; 4.  Internal utilities for interactive commands and command history
;;;-----------------------------------------------------------------------------

;; Some of this is for use with the gmhist package.

(defmacro sde-set-default (item hist-symbol)
  (` (progn
       (put (, hist-symbol) 'default (, item))
       (set (, hist-symbol) (append (list (, item)) (eval (, hist-symbol)))))))


(defmacro sde-reset-hist (hist-symbol hist-initial)
  (` (progn
       (set (quote (, hist-symbol)) nil)
       (sde-set-default (, hist-initial) (quote (, hist-symbol))))))


(defmacro sde-args (&rest rest)
  ;; Used in calls to (interactive) to check Soar is okay before attempting
  ;; to prompt or extract arguments.
  (` (progn (sde-check-soar) (list (,@ rest)))))


;;;-----------------------------------------------------------------------------
;;; 5.  Internal utilities for managing multiple agents.
;;;-----------------------------------------------------------------------------


(defmacro sde-buffer-of-agent (name)
  ;; Returns buffer associated with named agent.
  (` (cdr (assoc (, name) sde-soar-agents))))


(defmacro sde-agent-of-buffer (buffer)
  ;; Returns name of agent associated with given buffer, if any.
  (` (car (rassoc (get-buffer (, buffer)) sde-soar-agents))))


(defmacro sde-make-agent-buffer-name (name)
  (` (concat "*" (, name) "*")))


(defmacro sde-add-new-agent-buffer-pair (name buffer)
  ;; Helper macro for adding (name . buffer) pairs to sde-soar-agents.
  (` (setq sde-soar-agents
	   (append sde-soar-agents (list (cons (, name) (, buffer)))))))


(defun sde-agent-exists-p (name)
  ;; Return non-nil if agent named NAME exists.
  (assoc name sde-soar-agents))


(defun sde-agent ()
  ;; Determines name of current agent, or prompts the user.
  ;; As a side-effect, changes the buffer agent if it has to prompt the user.
  (and sde-soar-agents
       (or (sde-buffer-agent sde-buffer-data)
	   (setf (sde-buffer-agent sde-buffer-data) (sde-ask-for-agent)))))


(defvar sde-select-agent-hist nil)

(defun sde-ask-for-agent ()
  ;; Prompt the user to choose an agent, and return their choice.
  (if (sde-buffer-agent sde-buffer-data)
      (sde-set-default (sde-buffer-agent sde-buffer-data) 'sde-select-agent-hist))
  (sde-completing-read "Agent name: "
		       'sde-select-agent-hist ; History.
		       sde-soar-agents	      ; Possible choices.
		       nil t))		      ; Require a match.


;; The following function creates a new agent buffer and updates internal SDE
;; information to match.  It does not change the agent buffer's directory to
;; ROOT-DIR/AGENT-NAME, as a previous version of SDE did.  Soar doesn't do
;; this to the agent; Soar leaves the agent's current working directory where
;; it was created.  If it loads a .init.soar file in an agent subdirectory,
;; it will print a message like
;;
;;   Creating agent one.
;; 
;;   Changing to directory one/.
;;   Loading .init.soar
;;
;;   Returning from directory one/.
;;
;; But the agent's directory ends up being the same as the parent agent's.

(defun sde-add-agent (name root-dir &optional initial-output)
  ;; Create a new agent buffer for agent NAME and initialize appropriately.
  ;; Directory ROOT-DIR is the directory in which Soar was started and where
  ;; there may be a subdirectory for the agent.
  ;; Optional arg INITIAL-OUTPUT specifies what to print in the
  ;; buffer when it first comes up.  This is for Soar's initial output
  ;; message.
  ;; If `sde-use-multiple-frames' is non-nil, this also creates a new screen
  ;; for the agent buffer.
  (save-excursion
    (let* ((agent-buf-name (sde-make-agent-buffer-name name)))
      (if (sde-buffer-exists-p (get-buffer agent-buf-name))
	  (kill-buffer (get-buffer agent-buf-name)))
      (set-buffer (sde-get-buffer-create agent-buf-name))
      (setq default-directory (file-name-as-directory root-dir))
      (sde-soar-mode)
      (goto-char (point-max))
      (if initial-output
	  (insert initial-output))
      (insert "\n")
      (setq sde-soar-buffer-mark    (point-marker)
	    sde-soar-last-input-end (point))
      (setf (sde-buffer-agent sde-buffer-data) name)
      (sde-add-new-agent-buffer-pair name (current-buffer))
      ;; If a new agent is created, it is likely that a .init file will be
      ;; read, so make sure that we catch load statements in this agent buffer.
      (sde-add-soar-cmd-hook name 'sde-record-soar-loads)
      (if (and sde-use-multiple-frames (not (eq sde-use-multiple-frames 'output)))
	  (sde-create-buffer-frame (current-buffer) sde-soar-agent-buffer-defaults)
	  (sde-show-buffer (current-buffer)))
      (run-hooks 'sde-soar-hook))))


(defun sde-remove-agent (agent-name)
  ;; Removes the named agent, its buffer, window, and frame if there is one.
  (let ((buffer (sde-buffer-of-agent agent-name)))
    (if buffer
	(progn
	  (setq sde-soar-agents (adelete agent-name sde-soar-agents))
	  (if (and sde-use-multiple-frames (not (eq sde-use-multiple-frames 'output)))
	      (sde-delete-buffer-frame buffer))
	  (delete-windows-on buffer)
	  (kill-buffer buffer)))))


(defun sde-delete-agent-buffers ()
  ;; Delete all Soar agent buffers. 
  (if (sde-buffer-exists-p sde-soar-buffer)
      (kill-buffer sde-soar-buffer))
  (mapcar '(lambda (data) (sde-remove-agent (car data)))
	  (copy-alist sde-soar-agents)))


;;;-----------------------------------------------------------------------------
;;; 5.  Hook for kill-buffer-hook.
;;;-----------------------------------------------------------------------------
;;;
;;; In order to be able to catch killing of agent buffers, and properly remove
;;; the associated agent info from SDE, we have to hook into kill-buffer.  
;;; This is only used for `sde-soar-mode' buffers.


(defun sde-kill-buffer-agent-hook-fn ()
  "Removes the agent associated with the buffer to be killed.
This is used on `kill-buffer-hook' (Emacs 19) or `sde-kill-buffer-hook'
\(Emacs 18).  The hook, if not void, is a list of functions to be called,
with no arguments, before the buffer is actually killed.  The buffer to be
killed is current when the hook functions are called.  If the buffer to be
removed is an SDE Soar agent buffer, this function calls on Soar to destroy
the agent and updates internal SDE variables to reflect the agent's removal."
  (if sde-soar-agents
      (let ((agent-name (sde-agent-of-buffer (current-buffer))))
	(if (and agent-name (sde-soar-is-alive)
		 sde-soar-version-at-least-6-1-1)
	    (if (destroy-agent agent-name)
		(sde-remove-agent agent-name)
		(error "Buffer not killed."))))))


;;;----------------------------------------------------------------------------
;;; Error and diagnostic handling.
;;;----------------------------------------------------------------------------


(defun sde-error-unless-soar-multi-agent ()
  "Check that Soar is running in multi-agent mode.  If not, signal error."
  (unless sde-soar-agents
    (error "Soar is not running in multiple-agent mode.")))


;;;-----------------------------------------------------------------------------
;;; 5.  Soar Mode
;;;-----------------------------------------------------------------------------

;; Soar Mode is patterned after comint and uses the same basic functional key
;; bindings.  Unfortunately nearly all the comint commands needed to be 
;; rewritten because they will not work in a multi-agent Soar environment.
;;
;; The following functions and bindings are taken straight from comint-mode:
;;
;; c-a     comint-bol                      Beginning of line; skip prompt.
;; c-c c-w backward-kill-word    	   Like ^w in shells
;;
;; The following key bindings are the same as comint-mode's but use
;; new functions specific for Soar Mode:
;;
;; M-p      sde-previous-input             Cycle backwards in input history
;; M-n      sde-next-input                 Cycle forwards
;; M-r      sde-previous-input-matching    Search backwards in input history
;; M-s      sde-next-input-matching        Search forwards in input history
;; C-c C-c  sde-interrupt-soar             Send ^c to Soar
;; C-c C-o  sde-kill-output                Delete last batch of process output
;; C-c C-r  sde-show-output                Show last batch of process output
;; C-c C-u  sde-kill-input                 Like ^u in shells

(if (not (keymapp sde-soar-mode-map))
    (let ((map (copy-keymap sde-mode-map)))
      (define-key map "\es"      'sde-next-input-matching)
      (define-key map "\er"      'sde-previous-input-matching)
      (define-key map "\ep"      'sde-previous-input)
      (define-key map "\en"      'sde-next-input)

      (define-key map "\C-c\C-w" 'backward-kill-word)
      (define-key map "\C-c\C-u" 'sde-kill-input)
      (define-key map "\C-c\C-r" nil)    
      (define-key map "\C-c\C-r" 'sde-show-output)
      (define-key map "\C-c\C-o" 'sde-kill-output)

      (define-key map "\C-c>"    'sde-forward-prompt)
      (define-key map "\C-c<"    'sde-backward-prompt)

      (define-key map "\C-m"     'sde-return)
      (define-key map "\C-a"     'sde-bol)
      (setq sde-soar-mode-map map)))


;; Soar Mode.
;;
;; Note that the key bindings for next and previous history cannot be written
;; as \\[gmhist-previous], etc., because we don't use gmhist in GNU Emacs 19.
;; I had to hardwire the descriptions as "ESC p" and "ESC n".  Some other key
;; descriptions are hardwired too, mainly because FSF 19 seems to catch the
;; menu definition instead of the key binding first, and consequently prints
;; `menu foo-bar biff-wack', which is ugly and almost useless.

(defun sde-soar-mode ()
  "The Soar interaction component of the Soar Development Environment (SDE).
\\<sde-soar-mode-map>
Typing `\\[sde-return]' sends the text from the end of the last Soar prompt to the
end of the current line.  `\\[sde-kill-input]' erases the input typed so far (like
`C-u' in Unix shells), while `\\[backward-kill-word]' erases the last word typed 
\(like `C-w' in Unix shells).

`\\[sde-interrupt-soar]' interrupts whatever Soar is doing currently.

The Soar process buffer has a command history associated with it.
`\\[sde-previous-input]' and `\\[sde-next-input]' cycle forward and backward through the input history.  
The size of the input history is determined by the variable
`sde-soar-input-ring-size'.

`\\[load-soar]' will prompt for a file name and load the file into Soar.
`M-x load-defaults' will load the most current version of the Soar defaults
file.  `M-x excise-file' will prompt for a file name and will excise all of
the productions found in that file.

SDE tracks the files that comprise your tasks.  When you visit a Soar file
that is part of a task that is not yet known to SDE, it will prompt you for
the main (i.e., top-most) load file for the task, and then it will
recursively parse each file of the task to build an internal database of the
files and productions that comprise the task.  It will also cache the list of
files into a special file named \".sde\" in the directory where the main load
file is located.  In future editing sessions, SDE will read the \".sde\" file
first and delay the full parsing of files until the information is needed for
certain types of searches \(specifically, for the `sde-find-xxx' commands, 
described below).

All interactive Soar commands are available via `M-x COMMANDNAME'.  Some
commands in SDE have no Soar equivalents, e.g., `excise-file'.  Many key
bindings give direct access to the most common commands.  Some notable
bindings include the following, and a complete list appears later below.  
Use describe-key (\\<global-map>`\\[describe-key]') \(the standard Emacs help facility) to
get detailed help on each command.\\<sde-soar-mode-map>

  Running and Initializing Soar:

    go                      \\[go]
    schedule                \\[schedule]
    interrupt Soar          \\[sde-interrupt-soar]
    init-soar               \\[init-soar]

  Manipulating Productions:

    send production         \\[sde-send-production]
    load file               \\[load-soar]
    excise production       \\[excise]
    
  Tracing and breaking:

    ptrace production       \\[ptrace]
    pbreak production       \\[pbreak]

  Querying Soar for information about specific objects:

    print                   \\[print-soar]
    explain                 \\[explain]
    matches 1               \\[matches-1]	
    matches 2               \\[matches-2]	
    preferences             \\[preferences]	
    firing-counts           \\[firing-counts]	
    wm                      \\[wm]

  Querying Soar for general information:

    view ptraces            \\[sde-view-ptraces]
    view pbreaks            \\[sde-view-pbreaks]
    view goal stack (pgs)   \\[sde-view-pgs]
    view op. stack (pgso)   \\[sde-view-pgso]
    view match set (ms)     \\[sde-view-ms]
    view chunks             \\[sde-view-chunks]
    view productions        \\[sde-view-productions]
    view justifications     \\[sde-view-justifications]
    view Soar WM            \\[sde-view-working-memory]

  Finding source code:

    find production by name \\[sde-find-production-by-name]
    find by body (LHS/RHS)  \\[sde-find-production-by-body]
    find by LHS pattern     \\[sde-find-production-by-lhs]
    find by RHS pattern     \\[sde-find-production-by-rhs]
    find operator           \\[sde-find-operator]
    find problem-space      \\[sde-find-problem-space]
    find next match         \\[sde-next-match]

  Searching and replace across whole tasks:

    search for a string     \\[sde-search]
    search for a regexp     \\[sde-search-regexp]
    query-replace           \\[sde-query-replace]
    query-replace-regexp    \\[sde-query-replace-regexp]
    continue search         \\[sde-next-match]

  Controlling multiple agents:
  
    agent-go                \\[agent-go]
    create-agent            \\[create-agent]
    destroy-agent           \\[destroy-agent] (Unavailable prior to 6.1.1)
    list agents             \\[list-agents]
    select-agent            \\[select-agent]

  Moving around in the process buffer:
\\<sde-soar-mode-map>
    show last output        \\[sde-show-output]
    move to previous prompt \\[sde-backward-prompt]
    move to next prompt     \\[sde-forward-prompt]

When running in the X Window System, the following mouse buttons are defined:

  Button 	    	      Action
  ------------------    -------------------------------------------------
  SHIFT-left            Execute `print' on item under the cursor
  SHIFT-middle          Execute `matches' on item under the cursor
  SHIFT-right           Execute `preferences' on attribute under the cursor

  META-CONTROL-left     Send to Soar the production under the cursor
  META-CONTROL-middle   Find the file containing the production under cursor
  META-CONTROL-right    Execute `explain' on chunk condition under the cursor

Many Soar commands, such as print, require \"target\" arguments, i.e.,
something to act on.  Those commands that need a production name as target
will use the name of the production that the cursor is in currently, or else
the name of the preceeding sp form.  Those commands that can take any symbol
as target will try to use the symbol under the cursor regardless of whether
it looks like a production name.

Many of the standard Soar commands also take optional arguments -- for
example, the go and print commands.  Giving a prefix argument (such as `\\[universal-argument]')
to most commands makes them prompt for optional arguments.  Optional
arguments are remembered on history lists.  Typing `M-p' and `M-n' at a 
prompt moves backward and forward through the history.

Optional arguments are automatically reused upon subsequent invocations of a
command until they are changed.  This allows you to set an option and then
apply a command repeatedly to different target arguments without having to
respecify the optional arguments each time.  For example, you can type
`\\[universal-argument] \\[print-soar]', type \":depth 3\" to the resulting prompt for optional
arguments, and then repeat this command with the same arguments using
just `\\[print-soar]' on different targets.

The commands ptrace, pbreak, preferences, and explain have different
conventions.  Without arguments, ptrace (`\\[ptrace]') and pbreak (`\\[pbreak]')
act on the production under the cursor.  With a positive argument (such as
`\\[universal-argument]') they perform unptrace or unpbreak, respectively, on the production
under the cursor.  With a negative argument (e.g., `\\[negative-argument]'), they undo all 
ptraces or pbreaks, respectively.  You can print out the current list of
active ptraces and pbreaks using the view commands `\\[sde-view-ptraces]' and 
`\\[sde-view-pbreaks]', respectively.

The preferences command (`\\[preferences]') attempts to extract its arguments
using heuristics.  Place the cursor over the attribube in a printed oject,
for example on \"^io-state\" in the WME
                    (8: S1 ^io-state S1)
and preferences will automatically use the attribute and the WME's identifier 
as arguments.  If the cursor is not over an attribute, it will attempt to
extract the symbol nearest the cursor, and if it is an identifier, then 
preferences will prompt for the attribute.  In addition, if preferences is
given a prefix argument (`\\[universal-argument]'), it will prompt for the optional
\"level\" argument.

The explain command (`\\[explain]') also attempts to extract its arguments
using heuristics.  Remember that before explain may be used, Soar must be
instructed to track the necessary information using \"explain :on\" \(or the
SDE equivalent `M-x explain-on').  The easiest way to use explain in SDE
is to print out a chunk and then place the cursor over an attribute in
the condition side of the chunk.  Explain will then query Soar with the
name of the chunk and the condition that tests that attribute.  \(It does
this by constructing a condition clause consisting of the appropriate
identifier-attribute-value slot, and silently querying Soar to determine
the correct condition number, then finally sending the command 
\"explain <production-name> <condition-number>\" to Soar.\)  You may also
put the cursor over a chunk name and invoke explain, and SDE will extract
just the production name and issue \"explain <name>\" to Soar.  Finally,
if you are looking at the text of a previous \"explain <name>\" and you
wish to do \"explain <name> <condition>\", simply put the cursor on the
line containing the condition number and invoke explain, and SDE will issue
\"explain <production-name> <condition-number>\" to Soar.  If given a prefix
argument, explain will prompt for its arguments, offering as initial defaults
any production name and condition clause it can find.

Commands that only query Soar for information place the output in in a
separate buffer for viewing, unless the variable `sde-soar-use-output-buffer'
is nil, in which case all output goes to the agent process buffer.
New output is appended to the output buffer, each batch separated by a
title line, unless the variable `sde-soar-erase-output-buffer' is non-nil,
in which case the output buffer's existing contents will be cleared before
new output is written out.

When running in Emacs 19 or Lucid Emacs 19, SDE will attempt to use a
separate X Window screen (called a \"frame\" in Emacs 19) for the output 
buffer, unless the variable `sde-use-multiple-frames' is nil.  The
initial attributes of this buffer are determined by the list of values in
the variable `sde-soar-output-buffer-defaults'.  Use Emacs help on this
variable (`\\[describe-variable]') for more information about how to use
this feature.  

The command `\\[sde-find-production-by-name]' locates the source code for a
production whose name is under the cursor.  Put the cursor on a production in
your task, either in a Soar process buffer or an editing buffer, type
`\\[sde-find-production-by-name]', and it will jump to the place in your
files where the production is defined.

The command `\\[sde-find-production-by-lhs]' (`sde-find-production-by-lhs') allows you 
to search a task for all productions that contain a certain pattern in their
condition sides.  When invoked, the command prompts you for a search pattern.
The pattern must be a list similar in form to production conditions.
For example,
             ( (goal <g> ^problem-space <p>) (<p> ^name top-ps) )

is a pattern that will match all productions whose conditions test for a
problem space named \"top-ps\".  \(The actual variable names that you use are
irrelevant because the system is doing true pattern-matching search.)  The
names of the productions will be listed in a separate window, and the source
code of the first production will be shown in a second window.  You can view
the source code of each successive production by repeatedly typing
`\\[sde-next-match]' (`sde-next-match').  Alternatively, you can move the
cursor over a production name and use `\\[sde-find-production-by-name]'.

The command `\\[sde-search]' (`sde-search') lets you search the files of a task for a
string.  When invoked, this command prompts you for a search string and then
searches the files of the current task, stopping at the first match.  Use the
command `\\[sde-next-match]' to find the next occurrence of the string in the
files.  The command `\\[sde-search-regexp]' is identical, but lets you supply
a regular expression instead of a plain string.  Files are searched in
alphabetical order by default, unless the variable `sde-sort-lists' is set
to `nil'.

The command `\\[sde-query-replace]' performs an interactive query-replace
across the files of a task.  When invoked, this command prompts you for a
string to search for and a string with which to replace it.  If given a
prefix argument, it also prompts for the task in which to search\; otherwise,
it uses the task with which the current buffer is associated.  It then
performs a `query-replace' in each file of the task.  Type \\[help-command]
at a prompt for directions about what to do.  If you exit the query-replace
\(using `\\[keyboard-quit]' or ESC), you can resume the query-replace with the
command `\\[sde-next-match]'.  The command `\\[sde-query-replace-regexp]' is
similar but allows you to supply a regular expression instead of a plain
string.  The commands `\\[sde-replace-string]' and 
`\\[sde-replace-string-regexp]' are noninteractive versions of these commands.

The SDE on-line help facility is available on the `C-c C-h' prefix.
`\\[sde-apropos]' (apropos) prompts for a string and lists a brief description of
all SDE user commands and user variables that contain the given string.
`\\[sde-topic-help]' (topic) prompts for a command, variable or Soar help topic, and 
displays information about it.  This command interfaces to the Soar help 
facility, and will query Soar for help on topics that are not already defined
in SDE.  Note that `\\[sde-topic-help]' provides completion, which means you
can type `?' any time at the prompt to list possible completions of what you 
have typed so far, and you can type `TAB' to have Emacs complete the rest of
what you have typed based on known help topics.

`\\[sde-soar-info]' will bring up the Soar 6 User's manual using the Emacs
Info browser.  `C-c C-h C-i' will bring up the SDE manual.  `\\[sde-soar-release-notes]' 
will bring up the Release Notes for the latest revision of Soar, while 
`\\[sde-soar-user-notes]' will bring up the User Notes.  `\\[sde-describe-bindings]' will list the 
local bindings in the current SDE mode, and `\\[sde-where-is]' will prompt for
an SDE command name (with completion) and report which key binding
invokes it.

The cd, pushd and popd commands, when given to Soar, is watched by SDE to
keep this buffer's default directory the same as Soar's working directory.
`\\[track-cd-toggle]' turns directory tracking on and off.  If SDE ever gets
confused, you can resync with Soar by just typing `pwd' directly to Soar
in the buffer.

`\\[sde-bol]' will move the cursor to the beginning of the current line, and
will skip the Soar prompt if the current line is at a prompt.

`\\[sde-return]' knows about prompts and Soar productions.  If a production is not
complete, `\\[sde-return]' will indent it properly.  When an entire production form is
complete, `\\[sde-return]' sends it to the Soar process together with a new line.

`\\[sde-backward-prompt]' moves the cursor backward to the next previous prompt line in the
current buffer, and displays that line at the top of the window.
`\\[sde-forward-prompt]' moves the cursor forward to the next prompt line and displays that
line at the top of the window.  These commands are handy for navigating 
through Soar output in the process buffer.

`\\[sde-close-all-sp]' will close the current sp form by adding or removing right parentheses
as necessary.

`\\[backward-delete-char-untabify]' converts tabs to spaces as it moves back.

`\\[sde-reposition-window]' will make the current production and/or comment visible.
If the production is fully onscreen, it is moved to the top of the window.
If it is partly offscreen, the window is scrolled to get the definition (or
as much as will fit) onscreen, unless point is in a comment which is also
partly offscreen, in which case the scrolling attempts to get as much of the
comment onscreen as possible.  Repeated invocations of `\\[sde-reposition-window]' move the
production to the top of the window or toggle the visibility of comments that
precede it.

The command history in the process interaction buffer is filtered by the
function on the variable `sde-soar-input-ring-filter'.  (This variable is
analogous to Comint Mode's `comint-input-filter'.)  Commands for which the
function returns true, and that do not match the immediately prior command,
will be added to the input history.  The default function on
`sde-soar-input-ring-filter' checks that the command is not all whitespace.

Soar Mode is modeled after Comint Mode but does not use it directly.
Comint Mode's basic key bindings and functions are available in new forms
adapted for multi-agent Soar interaction.

To send a bug report, questions or other feedback to the authors and maintainers
of the Soar Development Environment, please use `\\[sde-feedback]'.

Entry to this mode will run the functions on the list variable
`sde-soar-mode-hook'.

Here is a list of all the special bindings in Soar mode.  The name of the key
map is `sde-soar-mode-map'.  For further help on individual commands, type
\\<global-map>`\\[describe-key] KEY' where KEY is the keystroke.

\\<sde-soar-mode-map>\\{sde-soar-mode-map}"

  (interactive)
  (let ((old-ring (and (assq 'sde-soar-input-ring (buffer-local-variables))
		       (boundp 'sde-soar-input-ring)
		       sde-soar-input-ring)))
    (sde-soar-mode-internal)
    ;; Run hooks first and then create input ring, to allow user to
    ;; set the SDE-SOAR-INPUT-RING-SIZE in the hook.  The test is to prevent
    ;; loosing history if sde-soar-mode is run twice in a buffer.
    (run-hooks 'sde-soar-mode-hook)
    (setq sde-soar-input-ring (if (ring-p old-ring) old-ring
				  (make-ring sde-soar-input-ring-size)))))


(defun sde-soar-mode-internal ()
  ;; The function for doing common soar-mode initializations.
  (sde-mode-internal)
  (setq mode-name "SDE-Soar"
	major-mode 'sde-soar-mode)
  (use-local-map sde-soar-mode-map)
  ;; Build menus if appropriate.  Although in FSF 19 we could build menus at
  ;; load time, in Lucid the menus have to be defined for each buffer, so
  ;; it's easiest just to let a function handle it for both cases.
  (if (and window-system (or sde-running-emacs19 sde-running-lemacs))
      (sde-define-menus 'sde-soar-mode sde-soar-mode-map))
  ;; Set up the mode line for Soar Mode buffers.
  (setq mode-line-modified "--- "
	mode-line-buffer-identification '("%20b")
	mode-line-format 
	(list "" 'mode-line-modified 'mode-line-buffer-identification
	      " {"
	      'sde-soar-state-string
	      "}  "
	      'global-mode-string
	      " %[("
	      'mode-name 'minor-mode-alist
	      ")%n%] --"
	      (if (or sde-running-emacs19 sde-running-lemacs)
		  '(line-number-mode "L%l--")
		"")
	      '(-3 . "%p") "-%-"))
  ;; Soar
  (make-local-variable 'sde-soar-last-input-match)
  (make-local-variable 'sde-soar-last-input-end)
  (make-local-variable 'sde-soar-input-ring-size)
  (make-local-variable 'sde-soar-input-ring)
  (make-local-variable 'sde-soar-input-ring-index)
  (setq sde-soar-last-input-match  ""
	sde-soar-input-ring-index 0
	sde-soar-last-input-end 0
	sde-soar-buffer-mark (make-marker))
  ;; Handle cleaning up after user kills agent buffer.
  (add-hook 'kill-buffer-hook 'sde-kill-buffer-agent-hook-fn))


;;;-----------------------------------------------------------------------------
;;; 6.  Soar output buffer.
;;;-----------------------------------------------------------------------------
;;;
;;; The output buffer has its own mode, slightly modified from sde-soar-mode.
;;; Having a separate mode for the output buffer seems excessive, but it's the
;;; easiest way I've found to allow screen customizations for the user.  Who
;;; knows; eventually we may find more uses for it.
;;;
;;; Basic desired behavior:
;;; 
;;; If the user deletes the output buffer, it must be recreated when new
;;; output is sent to it.
;;;
;;; If we're running with multiple windows (e.g. in Emacs 19):
;;;   * Only allow one instance of the output buffer on the screen.
;;;   * The output buffer should be raised/deiconified whenever new output
;;;     is sent to it.
;;;   * Operations such as find-production should never end up shown in the
;;;     output buffer window, even if the user initiated the operation from
;;;     that window.


(defconst sde-soar-output-buffer-name "*output*"
  "Name of pop-up output buffer.")

(defvar sde-soar-output-buffer nil
  "The Soar output buffer.")

(defvar sde-soar-output-buffer-title nil
  "Title for the next batch of output shown in the output buffer.")

(defvar sde-soar-output-buffer-title-fmt "====== %s ======\n\n"
  "Format string for title lines in the output buffer.")

(defvar sde-soar-output-buffer-data nil
  "Pointer to the buffer data of the buffer that originated the creation of
this output buffer.")


;; Keymap is same as SDE Soar Mode, except that RET just types a plain
;; newline in the buffer, instead of trying to send something to Soar.

(if (not (keymapp sde-soar-output-mode-map))
    (let (map)
      (if sde-running-lemacs
	  (progn
	    (setq map (make-sparse-keymap))
	    (set-keymap-name   map 'sde-soar-mode-map)
	    (set-keymap-parent map sde-mode-map))
	  (setq map (copy-keymap sde-soar-mode-map)))
      (define-key map "\C-m"     'newline)
      (setq sde-soar-output-mode-map map)))


(defun sde-soar-output-mode ()
  "Major mode for the SDE Soar output buffer.
The mode's keymap (`sde-soar-output-mode-map') and other settings are
based on sde-soar-mode's.  Entry to this mode runs the hooks on variable
`sde-soar-output-mode-hook'.  The local key bindings are:\\<sde-soar-output-mode-map>
\\{sde-soar-output-mode-map}"
  (interactive)
  (sde-soar-mode-internal)
  (setq major-mode 'sde-soar-output-mode
	mode-name  "SDE Soar Output")
  (use-local-map sde-soar-output-mode-map)
  ;; Build menus if appropriate.  Although in FSF 19 we could build menus at
  ;; load time, in Lucid the menus have to be defined for each buffer, so
  ;; it's easiest just to let a function handle it for both cases.
  (if (and window-system (or sde-running-emacs19 sde-running-lemacs))
      (sde-define-menus 'sde-soar-output-mode sde-soar-mode-map))
  (if sde-running-lemacs
      (progn
	;; The default of "Output: *output*" is not attractive
	(if (equal screen-title-format "%S: %b")
	    (set (make-local-variable 'screen-title-format) "%b"))
	(put 'sde-soar-output-mode 'screen-name 'output)
	(put 'sde-soar-output-mode 'instance-limit 1)
	(put 'sde-soar-output-mode 'screen-defaults
	     sde-soar-output-buffer-defaults)))
  (run-hooks 'sde-soar-output-mode-hook))


(defun sde-title-soar-output-buffer (agent title)
  ;; Record the agent and title to be used for the next output to the output
  ;; buffer.
  (setq sde-soar-output-buffer-title title
	sde-soar-output-buffer-data sde-buffer-data))


;; A note about font-lock: in FSF Emacs 19 it's implemented using the variable
;; `after-change-function'.  Every insertion, or part of an insertion, causes
;; the function to be called.  So it turns out to be more efficient to
;; do something like (insert (format foo-fmt foo-contents)) than something like
;; (insert "===== " foo-contents "====\n") because the latter results in 3
;; calls to `after-change-function', while the former only results in 1. 

(defun sde-init-soar-output-buffer ()
  ;; Create (if necessary) and initialize the sde-soar-output-buffer.  If
  ;; `sde-soar-output-buffer-title' is non-nil, point is moved to the top of
  ;; the output buffer and there its value (a string) is printed, unless
  ;; `sde-soar-erarse-output-buffer' is t, in which case the string is never
  ;; printed.  Thus a non-nil value of `sde-soar-output-buffer-title' is
  ;; taken to be a signal that we're dealing with a new batch of output.
  (let ((exists (sde-buffer-exists-p sde-soar-output-buffer)))
    (setq sde-soar-output-buffer
	  (sde-get-buffer-create sde-soar-output-buffer-name))
    (set-buffer sde-soar-output-buffer)
    (or exists
	(sde-soar-output-mode))
    (setf sde-buffer-data
	  (sde-copy-buffer-struct sde-soar-output-buffer-data))
    (if sde-soar-output-buffer-title
	(progn
	  (goto-char (point-min))
	  (if sde-soar-erase-output-buffer ; Erase if desired.
	      (erase-buffer)
	    (insert (format sde-soar-output-buffer-title-fmt
			    sde-soar-output-buffer-title)))
	  (set-marker sde-soar-buffer-mark (point))
	  (setq sde-soar-output-buffer-title nil)
	  (if sde-use-multiple-frames
	      (let ((frame (sde-buffer-frame sde-soar-output-buffer t)))
		(if frame 
		    (sde-show-frame frame)
		  (sde-create-buffer-frame sde-soar-output-buffer
					   sde-soar-output-buffer-defaults
					   t nil)))
	    ;; In Emacs 18 or deliberately not using frames.
	    (sde-show-buffer sde-soar-output-buffer))))))


(defun sde-update-soar-output-buffer-window ()
  ;; Move point in the output buffer to the top.  Otherwise, the window
  ;; contents are left at the bottom of the last output in the buffer.
  (let ((window (sde-buffer-window sde-soar-output-buffer t)))
    (if window
	(progn
	  (set-window-start window 1 t)
	  (set-window-point window 1)))))


(defmacro sde-in-soar-output-buffer (&rest forms)
  ;; Usage: (sde-in-soar-output-buffer BODYFORMS ...)
  ;; Use list syntax for this one to avoid a bug in Emacs 18's backquote.el.
  (list 'let '((--sde-obuf-- (current-buffer)))
	(list 'unwind-protect
	      (append '(progn
			(sde-init-soar-output-buffer)
			(set-buffer sde-soar-output-buffer))
		      forms
		      (list '(sde-update-soar-output-buffer-window)))
	      '(and (not (eq --sde-obuf-- (current-buffer)))
		    (sde-buffer-exists-p --sde-obuf--)
		    (set-buffer --sde-obuf--)))))


;;;-----------------------------------------------------------------------------
;;; 6.  Soar error buffer.
;;;-----------------------------------------------------------------------------


(defconst sde-soar-error-buffer-name "*error*"
  "Name of pop-up error buffer.")

(defun sde-key-for-command (cmd)
  "Return a string description of the binding for function CMD.
If there is no key binding for the command, returns the string \"M-x cmd\"."
  (let ((key (key-description
	      (sde-where-is-internal cmd (current-local-map) 'non-ascii))))
    (if (and key (not (string= key "")))
	key
      (concat "M-x " (symbol-name cmd)))))


(defun sde-pop-up-error-buffer (buffer)
  ;; If there's already a window showing this BUFFER, do nothing.
  ;; Else, split the currently selected window, display BUFFER in the
  ;; top-most of the two resulting windows, resize that window so that
  ;; it only shows as many lines as are in BUFFER, and finally try to
  ;; minimize the amount of screen motion in the second window.
  (or (sde-buffer-window buffer)
      (let ((orig-buffer (current-buffer))
	    (start (window-start))	; Pos in current window.
	    ;; Need get key binding now, in the current buffer.
	    (key (sde-key-for-command 'sde-hide-soar-error-window))
	    ;; Rebind this temporarily to allow smaller window.  Mainly for 19.
	    (window-min-height 3)
	    (window))
	;; Split window creates two windows on the same buffer and leaves the
	;; top-most one selected.  That's where we put the output buffer.
	(split-window nil 
		      (max (sde-count-buffer-lines buffer)
			   window-min-height))
	(set-window-buffer (setq window (selected-window)) buffer)
	(set-window-start window 1)
	(set-window-point window 1)
	(set-buffer buffer)		; Go to error buffer
	(sde-soar-mode-internal)
	(setq mode-line-buffer-identification '("%9b")
	      mode-line-format 
	      (list "" 'mode-line-modified 'mode-line-buffer-identification
		    (format "[%s to hide window]" key)
		    "   %[("
		    'mode-name 'minor-mode-alist
		    ")%n%] --"
		    '(-3 . "%p") "-%-"))
	(set-buffer orig-buffer)
	;; Move back to the window we started from (I hope).
	(select-window (next-window window 'no))
	;; Move window so that just the lines at top are covered, instead of
	;; letting Emacs jerk the contents around.  Code from popper.el.
	(save-excursion
	  (set-window-start (selected-window) start)
	  (move-to-window-line (window-height window))
	  (set-window-start (selected-window) (point))))))


(defun sde-soar-show-error-buffer (buffer)
  (if (and sde-use-multiple-frames
	   (sde-buffer-exists-p sde-soar-output-buffer)
	   (eq (selected-window) (get-buffer-window sde-soar-output-buffer)))
      (let* ((window (selected-window))
	     (dedicated (window-dedicated-p window)))
	(set-window-dedicated-p window nil)
	(sde-pop-up-error-buffer buffer)
	(set-window-dedicated-p window dedicated))
      (sde-pop-up-error-buffer buffer)))


;; Windows in an Emacs frame have a certain ordering.  If there are no
;; horizontally split windows, then `next-window' returns the window downward
;; from the given window, and `previous-window' returns the window upward
;; from the given window.  In FSF 19, `delete-window' gives lines back to the
;; window upward from the deleted window.

(defun sde-hide-soar-error-window ()
  "Hide the Soar error buffer window."
  (interactive)
  (let* ((buffer (get-buffer sde-soar-error-buffer-name))
	 (window (and (bufferp buffer) (sde-buffer-window buffer)))
	 (old-w (selected-window)))
    (if (and (bufferp buffer) (windowp window))
	(unwind-protect
	    (let* ((height (window-height window))
		   (next (next-window window 'no))
		   (prev (previous-window window)))
	      (bury-buffer buffer)
	      ;; Select previous window so lines are given to the next one.
	      ;; The "next one" should the one that was split for the error.
	      (select-window prev)
	      (delete-window window)
	      (unless (or (coordinates-in-window-p '(0 . 1) next)
			  (one-window-p))
		(shrink-window height))
	      (select-window next)
	      (scroll-down height))
	  (select-window old-w)))))


(defun sde-soar-error (output msg)
  ;; Beep and show Soar's OUTPUT in a pop-up buffer and print message MSG in
  ;; the minibuffer.
  (let ((temp-buffer-show-function 'sde-soar-show-error-buffer))
    (with-output-to-temp-buffer sde-soar-error-buffer-name
      (princ output))
    (message (sde-quote-string msg))
    (beep)))


;; Use internally to clean up.

(defun sde-delete-soar-error-buffer ()
  ;; Delete the error buffer & window
  (let ((buffer (get-buffer sde-soar-error-buffer-name)))
    (if (sde-buffer-exists-p buffer)
	(progn
	  (delete-windows-on buffer)
	  (kill-buffer buffer)))))


;;;-----------------------------------------------------------------------------
;;; 7.  Interactive Soar commands
;;;
;;; The various interactive Soar commands are very idiosyncratic in their use
;;; of arguments.  For the purposes of implementing a sensible interface to
;;; them in Emacs, I divided arguments into two categories: target arguments
;;; and optional arguments.  "Target arguments" are arguments that a Soar
;;; command directly acts on.  For example, print needs a target argument.
;;; All other arguments are "optional arguments"; they modify the behavior of
;;; a command.  For example, print takes optional args ":depth" and
;;; ":internal".
;;;
;;; The interface implemented here attempts to remember optional args from
;;; one invocation of a command to the next, so that if in subsequent
;;; invocations an optional arg is not given to a command, the optional args
;;; from the last invocation are reused.  Target arguments are never
;;; remembered.  The idea is that a user is likely to want to set some
;;; behavior a command and then execute that same command on several
;;; different objects before changing the optional args.
;;;
;;; Associated with each command that can take an optional argument is a
;;; variable that the user can set to a string that is to be the initial
;;; value of the argument (i.e., before a new value is supplied in a
;;; session).  These variables are "sde-go-args", etc., in Section ii at the
;;; top of this file.
;;;
;;; The general scheme for supplying arguments is summarized in the following
;;; rules; the rules are not mutually exclusive:
;;;
;;; 1) If a command takes optional args, then:
;;;
;;;      a) If this is the first time the command is invoked and no prefix
;;;         argument is given, the value of the command's associated
;;;         "sde-xxx-args" variable is used as the argument.  If this
;;;         variable is nil, the command is issued to Soar without any
;;;         arguments and Soar will do whatever its default behavior is.
;;;
;;;      b) If given a prefix argument, it prompts for the arguments to be
;;;         passed to the corresponding Soar command.  (Exception: ptrace,
;;;         pbreak, and firing-counts have different conventions.)
;;;
;;;      c) If no prefix argument is given, then the last optional argument
;;;         given to that particular command is used.
;;;
;;; 2) If a command takes target args, then the symbol under the cursor or
;;;    the nearest symbol left of the cursor is examined as a potential
;;;    target. 
;;;
;;;      a) If the command needs a production name, the symbol is
;;;         checked to see if it looks like a production name.  If so,
;;;         it is automatically used as the target argument.  If it does
;;;         not seem like a production name, the user is prompted for a
;;;         production name.
;;;
;;;      b) If the command just needs any symbol (e.g., an id or "<o>"
;;;         or whatever), then the symbol under the cursor is
;;;         automatically used.
;;;
;;; Note that all commands can be called from other elisp functions, and
;;; nearly all take as first argument the destination agent name.
;;;
;;; The function `sde-user-input-check' is called by `sde-return' to
;;; verify the input typed by the user and catch mistakes, primarily to
;;; avoid some situations that would screw up SDE, such as the user
;;; selecting an agent in an agent buffer.
;;
;;; The function `sde-soar-update-args' (see code section below) watches
;;; commands typed directly to the Soar process and remembers the optional
;;; arguments.  Thus, if a user types a command directly in the process
;;; window and subsequently uses the key binding for that command, the same
;;; optional args are reused.  The goal is to maximize consistency.
;;;
;;; The general output behavior of commands is as follows:
;;;
;;; 1. All commands are echoed in the minibuffer, to provide feedback to the
;;;    user.
;;;
;;; 2. When sde-soar-use-output-buffer is t, the output produced by query
;;;    commands is caught and displayed in the Soar output buffer; nothing
;;;    appears in the process buffer.  Query commands are things like print,
;;;    etc.  All other commands' output is shown directly in the process
;;;    buffer.
;;;
;;;    Exceptions:  pbreak, select-agent, create-agent are special.
;;;
;;; 3. The commands for execution control always switch the user to the
;;;    process buffer (because they are important to understanding the current
;;;    state of Soar).
;;;
;;;-----------------------------------------------------------------------------

;; The "-hist" variables are used internally to remember histories for
;; individual commands; the "-args" variables are advertised to the user
;; as something they can use to set default values.

(defvar sde-go-hist		     nil)
(defvar sde-run-hist		     nil)
(defvar sde-matches-hist	     nil)
(defvar sde-ms-hist		     nil)
(defvar sde-firing-counts-hist	     nil)
(defvar sde-print-hist		     nil)
(defvar sde-preferences-hist	     nil)
(defvar sde-list-productions-hist    nil)
(defvar sde-list-chunks-hist	     nil)     
(defvar sde-list-justifications-hist nil)
(defvar sde-agent-go-hist	     nil)
(defvar sde-schedule-hist            nil)
(defvar sde-explain-hist             nil)
(defvar sde-max-elaborations-hist    nil)
(defvar sde-max-chunks-hist          nil)

(defun sde-reset-soar-histories ()
  ;; Reset to initial defaults all of the Soar command arg history lists.
  (sde-reset-hist sde-go-hist		       sde-go-args)
  (sde-reset-hist sde-run-hist		       sde-run-args)
  (sde-reset-hist sde-matches-hist	       sde-matches-args)
  (sde-reset-hist sde-ms-hist		       sde-ms-args)
  (sde-reset-hist sde-firing-counts-hist       sde-firing-counts-args)
  (sde-reset-hist sde-print-hist	       sde-print-args)
  (sde-reset-hist sde-preferences-hist	       sde-preferences-args)
  (sde-reset-hist sde-list-productions-hist    sde-list-productions-args)
  (sde-reset-hist sde-list-chunks-hist	       sde-list-chunks-args)
  (sde-reset-hist sde-list-justifications-hist sde-list-justifications-args)
  (sde-reset-hist sde-agent-go-hist	       sde-agent-go-args)
  (sde-reset-hist sde-schedule-hist	       sde-schedule-args)
  (sde-reset-hist sde-explain-hist	       sde-explain-args)
  (sde-reset-hist sde-max-elaborations-hist    sde-explain-args)
  (sde-reset-hist sde-max-chunks-hist	       sde-explain-args))


;; Aliases for commands

(fset 'run-soar			'soar)
(fset 'sde-soar			'soar)
(fset 'sde-go   		'go)
(fset 'sde-run			'run)
(fset 'sde-reset		'reset)
(fset 'sde-load-errors		'load-errors)
(fset 'sde-load-errors-on	'load-errors-on)
(fset 'sde-load-errors-off	'load-errors-off)
(fset 'sde-pbreak		'pbreak)
(fset 'sde-unpbreak             'unpbreak)
(fset 'sde-ptrace		'ptrace)
(fset 'sde-unptrace             'unptrace)
(fset 'sde-explain		'explain)
(fset 'sde-explain-on		'explain-on)
(fset 'sde-explain-off		'explain-off)
(fset 'sde-firing-counts	'firing-counts)
(fset 'sde-list-chunks		'list-chunks)
(fset 'sde-list-justifications	'list-justifications)
(fset 'sde-list-productions	'list-productions)
(fset 'sde-matches		'matches)
(fset 'sde-matches-1		'matches-1)
(fset 'sde-matches-2		'matches-2)
(fset 'sde-matches-3		'matches-3)
(fset 'sde-memory-stats		'memory-stats)
(fset 'sde-ms			'ms)
(fset 'sde-pgs			'pgs)
(fset 'sde-pgso			'pgso)
(fset 'sde-preferences		'preferences)
(fset 'sde-print		'print)
(fset 'sde-print-stats		'print-stats)
(fset 'sde-rete-stats		'rete-stats)
(fset 'sde-wm			'wm)
(fset 'sde-excise		'excise)
(fset 'sde-excise-chunks	'excise-chunks)
(fset 'sde-excise-task		'excise-task)
(fset 'sde-excise-all		'excise-all)
(fset 'sde-excise-file		'excise-file)
(fset 'sde-init-soar		'init-soar)
(fset 'sde-load-soar		'load-soar)
(fset 'sde-load-defaults	'load-defaults)
(fset 'sde-select-agent		'select-agent)
(fset 'sde-schedule		'schedule)
(fset 'sde-agent-go		'agent-go)
(fset 'sde-create-agent		'create-agent)
(fset 'sde-destroy-agent	'destroy-agent)
(fset 'sde-list-agents		'list-agents)
(fset 'sde-soarnews		'soarnews)

;; Currently unimplemented:
;;
;; list-help-topics		user-select
;; print-all-help		object-trace-format
;; watch			time
;; log				multi-attribute
;; learn			add-wme
;; default-print-depth		remove-wme
;; d                            stack-trace-format
;; chunk-free-problem-spaces    trace-format-escapes


;;;
;;; 7.A.  Commands for execution control
;;;
;;; All commands that start & stop executing are here.
;;;


(defvar sde-start-soar-args nil)

(defun soar (&optional dir program switches)
  "Runs Soar in an Emacs buffer.
Unless given a prefix argument, starts the Soar program indicated by the
variable `sde-soar-program' in the directory indicated by the variable
`sde-soar-starting-directory' and passing command line arguments indicated by
the variable `sde-soar-switches'.  In this context \"program\" means a Soar
executable binary to start up, and not a file of Soar productions.  If the
pathname `sde-soar-program' is an absolute pathname (i.e., it begins with a
slash '/' or '~' character) then it specifies exactly the program to start
up.  Otherwise, the program indicated by `sde-soar-program' is searched for
using the normal search rules of a Unix shell.  (Note that the program must
be in a directory listed in your shell's $path variable, otherwise Emacs will
not be able to find it unless you specify an absolute pathname.)  The
variable `sde-soar-starting-directory' specifies the directory to cd to prior
to starting up `sde-soar-program' and thus determines where Soar will look
for initialization files such as .init-soar.  

If given a prefix argument, prompts for a directory to cd to, the pathname of
the Soar program, and command line switches to pass to the program, before
starting Soar.  If no prefix argument is given, and one or both of
`sde-soar-program' and `sde-soar-starting-directory' are not set, prompts for
the missing values.  It will not prompt for command line switches unless you
supply a prefix argument, since standard Soar does not accept command line
arguments.  (The switches feature is provided for use with modified Soar
systems such as ModSAF/SAFSoar.)

Also, if given new values for the program name, starting directory and
command line switches, the corresponding variables (`sde-soar-program',
`sde-soar-starting-directory', `sde-soar-switches') are reset to those new
values.  This means that after setting new values, you can subsequently start
the same Soar combination without supplying a prefix argument and answering
the prompts again.

The variable `sde-soar-switches' must be set to a list of strings
representing individual arguments that are to be passed to Soar at start-up
time.  Invoke help on this variable (`\\[describe-variable]') for more info.

Do a describe-mode command (`\\[describe-mode]') in the Soar buffer, once Soar
is running, to get help on SDE Soar Mode.  Further help is available using the
SDE help key, `C-c C-h'."
  (interactive)
  (if (sde-soar-is-alive)
      (sde-pop-to-buffer sde-soar-buffer)
      (let ((default-dir (file-name-as-directory
			  (substitute-in-file-name
			   (or dir sde-soar-starting-directory default-directory))))
	    (program (or program sde-soar-program ""))
	    (switches (or switches sde-soar-switches)))
	(if (or current-prefix-arg
		(and (null dir) (null sde-soar-starting-directory)))
	    (setq dir (sde-read-file-name
		       "Starting directory: " 'sde-start-soar-args
		       default-dir default-dir)))
	(setq dir (file-name-as-directory
		   (substitute-in-file-name (or dir default-dir))))
	(if (or current-prefix-arg (string= program ""))
	    (setq program (sde-read-file-name
			   "Name of soar program: " 'sde-start-soar-args
			   dir program)))
	(setq program (substitute-in-file-name program))
	;; Prompt for switches only if prefix arg given.
	;; Switches must be maintained as a list of strings.
	(if (or current-prefix-arg
		(and sde-prompt-for-soar-switches (null switches)))
	    (setq switches
		  (sde-unconcat
		   (sde-read-string
		    "Switches: " nil
		    (mapconcat 'identity sde-soar-switches " ")))))
	(sde-start-soar program dir switches)
	;; If we get this far, assume there were no errors, so
	;; reset these variables to new values they may have been given.
	(setq sde-soar-program program
	      sde-soar-starting-directory dir
	      sde-soar-switches switches))))


(defun go (&optional agent args)
  "Execute Soar's \"go\" command.
If given a prefix argument, prompts for optional arguments to give to \"go\".
If not given a prefix argument, uses the same argument given to this command
the last time it was invoked.

This is the most general command for running Soar.  It accepts up to two
optional arguments, in the following two formats:

  go forever
  go N TYPE

The first form, `go forever', runs Soar until it halts or until it is
interrupted by \\<sde-soar-mode-map>\\[sde-interrupt-soar].  The second form, `go N TYPE', permits the
following specifications:
     
   * [N] An integer interpreted according to the value of TYPE.

   * [TYPE] One of the following:
     
        * [`p'] Run Soar for N phases, where a \"phase\" is any of the
          following: an input phase, a preference phase, a working memory
          phase, an output phase, or a quiescence phase.

        * [`e'] Run Soar for N elaboration cycles.  For purposes of this
          command, a quiescence phase is counted as an elaboration cycle.

        * [`d'] Run Soar for N decision cycles.
          
        * [`g'] Run Soar until the Nth time a goal is selected.
          
        * [`ps'] Run Soar until the Nth time a problem space is selected.
          
        * [`s'] Run Soar until the Nth time a state is selected.
          
        * [`o'] Run Soar until the Nth time an operator is selected.
          
        * [`v'] Run Soar until the Nth time a selection is made for the context
          slot indicated by `v', or until the context stack pops to above the
          context indicated by `v'.  The possible values of `v' are:
          
                 `<g>'      `<p>'      `<s>'      `<o>'
                 `<sg>'     `<sp>'     `<ss>'     `<so>'
                 `<ssg>'    `<ssp>'    `<sss>'    `<sso>'

Examples:
  (go 5 d)   --> run for 5 decision cycles
  (go e)     --> run for another 5 elaboration cycles
  (go 1 g)   --> run until the next goal is selected (i.e., until the next
                 time an impasse arises)
  (go <so>)  --> run until the next superoperator is selected (or until the
                 supergoal goes away)
  (go 3 <o>) --> run for 3 operator selections at this level (continuing
                 through any subgoals that arise)
"
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'go': " 'sde-go-hist)))
  (sde-soar-cmd agent (concat "go " args) nil nil t))


(defun run (&optional agent args)
  "Execute Soar's \"run\" command.  
If given a prefix argument, prompts for optional arguments to give to
\"run\".  If not given a prefix argument, uses the same argument given to
this command the last time it was invoked.

This command runs Soar for that number of elaboration cycles.  (For this
command, quiescence phase is counted as an elaboration cycle.)  With no
arguments, this runs Soar forever (or until Soar halts, receives an
interrupt, etc.)."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'run': " 'sde-run-hist)))
  (sde-soar-cmd agent (concat "run " args) nil nil t))


(defun reset (&optional agent file)
  "Executes the Soar \"reset\" command.
Causes Soar to load the file \".reset.soar\".  If given a prefix 
argument, prompts for an alternative file name to use instead of
\".reset.soar\".

The intention is for this file to contain any function calls or other
commands needed to reset the system at the end of a run.  Examples would
include calling init-soar, excising-chunks, restarting a simulator etc."
  (interactive (if sde-soar-version-at-least-6-1-1
		   (sde-args
		     (sde-agent)
		     (if current-prefix-arg
			 (car (comint-get-source "Load file: " sde-soar-last-dir/file
						 sde-source-modes t))))
		   (error "The reset facility is unavailable prior to Soar 6.1.1.")))
  (sde-add-soar-cmd-hook agent 'sde-update-pbreak-list)
  (sde-add-soar-cmd-hook agent 'sde-record-soar-loads)
  (if file
      (sde-soar-cmd agent (concat "reset" file))
      (sde-soar-cmd agent "reset")))


(defun load-errors (&optional agent on-off)
  "Executes the Soar \"load-errors\" command.
With no arguments, this command prints out the current load-errors status.
If given a prefix argument, prompts for an argument, which should be either
\"on\" or \"off\".  If \"on\" is given as the argument, then if any errors occur
while loading an input file, Soar will ask the user whether to continue 
loading.  If \"off\" is given as the argument, then he user will not be prompted
if any errors occur when loading an input file.  The default status is on."
  (interactive (if sde-soar-version-at-least-6-1-1
		   (sde-args
		     (sde-agent)
		     (if current-prefix-arg
			 (sde-completing-read
			  "New load-errors setting (on/off): " nil
			  '(("on") ("off")) nil t)))
		   (error "The load-errors facility is unavailable prior to Soar 6.1.1.")))
  (if on-off
      (sde-soar-cmd (sde-agent) (concat "load-errors " on-off))
      (sde-soar-query (sde-agent) "load-errors")))


;; These are for use in menus:

(defun load-errors-on ()
  "Turn on the Soar load-errors setting."
  (interactive)
  (or sde-soar-version-at-least-6-1-1
      (error "The load-errors facility is unavailable prior to Soar 6.1.1."))
  (load-errors (sde-agent) "on"))


(defun load-errors-off ()
  "Turn off the Soar load-errors setting."
  (interactive)
  (or sde-soar-version-at-least-6-1-1
      (error "The load-errors facility is unavailable prior to Soar 6.1.1."))
  (load-errors (sde-agent) "off"))


(defun warnings (&optional agent on-off)
  "Executes the Soar \"warnings\" command.
With no arguments, this command prints out the current `warnings' setting, 
which determines whether warnings are printed.  If given a prefix argument,
it prompts for an argument, which should be either \"on\" or \"off\", and
sets the Soar `warnings' settings appropriately."
  (interactive (sde-args
		 (sde-agent)
		 (if current-prefix-arg
		     (sde-completing-read
		      "New warnings setting (on/off): " nil
		      '(("on") ("off")) nil t))))
  (if on-off
      (sde-soar-cmd (sde-agent) (concat "warnings " on-off))
      (sde-soar-query (sde-agent) "warnings")))


;; These are for use in menus:

(defun warnings-on ()
  "Sets Soar `warnings' to \"on\"."
  (interactive)
  (warnings (sde-agent) "on"))


(defun warnings-off ()
  "Sets Soar `warnings' to \"off\"."
  (interactive)
  (warnings (sde-agent) "off"))



;;;
;;; 7.B.  Commands for tracing and breaking
;;; 


(defun pbreak (&optional agent name unpbreak)
  "Cause a production to issue a break.  I.e., redefine the production to
interrupt Soar when it fires.  Uses as target the name of the production that
the cursor is in, or the next-previous production, in the current buffer.  If
given a positive prefix arg, undoes the pbreak on the production.  If given a
negative prefix arg, undoes all currently active pbreaks.  Pbreaks are
remembered on a per-agent basis, so pbreak'ing a production in one agent will
not automatically pbreak that production in any other agents that may also
share the same production.  To list the currently active pbreaks, use
\\<sde-soar-mode-map>\\[sde-view-pbreaks]

Pbreaks persist until undone by giving this command an argument, or by using 
`unpbreak' directly, or by reloading the production.

If called from a program, takes three arguments: AGENT NAME UNPBREAK.
Argument UNPBREAK should be nil to pbreak the current production, non-nil to
unpbreak the current production, or 'all to unpbreak all productions."
  (interactive (sde-args
		 (sde-agent)
		 ;; Don't look for production if undoing all.
		 (or (sde-prefix-arg-negative-p)
		     (sde-production-name-near-point))
		 (if (sde-prefix-arg-negative-p)
		     'all
		     (sde-prefix-arg-value))))
  (if unpbreak
      ;; Note that if unpbreaking all, name will be nil, but that's okay.
      (unpbreak agent name (eq unpbreak 'all))
      (if (sde-pbreak-in-effect agent name)
	  (error "Production \"%s\" is already being pbreak'ed." name)
	  (progn
	    (sde-pbreak-production agent name)
	    (message "pbreak %s" (sde-quote-string name))))))


(defun unpbreak (&optional agent name unpbreak)
  "Undo pbreak on a production.
Uses as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer.  If given a prefix
argument, undoes all pbreaks active in the current agent.  Pbreaks are
remembered on a per-agent basis, so pbreak'ing a production in one agent will
not automatically pbreak that production in any other agents that may also
share the same production.  To list the currently active pbreaks, use
\\<sde-soar-mode-map>\\[sde-view-pbreaks]"
  (interactive (sde-args
		 (sde-agent)
		 ;; Don't look for production if undoing all.
		 (or current-prefix-arg (sde-production-name-near-point))
		 current-prefix-arg))
  (cond (unpbreak
	 (sde-unpbreak-all)
	 (message "All pbreaks disabled."))
	((sde-pbreak-in-effect agent name)
	 (sde-unpbreak-production agent name)
	 (message "unpbreak %s" (sde-quote-string name)))
	(t
	 (error "Production \"%s\" is not under the effect of a pbreak." name))))


(defun ptrace (&optional agent name unptrace)
  "Execute the Soar \"ptrace\" command.
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer.  If given a positive prefix
arg, undoes the effects of a ptrace on the production.  If given a negative
prefix arg, disables all currently active ptraces.  To list all the currently
active ptraces, use \\<sde-soar-mode-map>\\[sde-view-ptraces]

Ptrace enables and disables tracing the firings and retractions of individual
productions.  This mechanism is orthogonal to the watch :firings mechanism.
Tracing persists until disabled by an unptrace command, or until the
production is excised.  (Note that redefining a production results in Soar
excising the previous definition.)

If called from a program, takes three arguments: AGENT NAME UNPTRACE.
Argument UNPTRACE should be nil to ptrace the current production, non-nil to
unptrace the current production, or 'all to unptrace all productions."
  (interactive (sde-args
		 (sde-agent)
		 ;; Don't look for production if undoing all.
		 (or (sde-prefix-arg-negative-p)
		     (sde-production-name-near-point))
		 (if (sde-prefix-arg-negative-p)
		     'all
		     (sde-prefix-arg-value))))
  (if unptrace
      ;; Note that if unptracing all, name will be nil, but that's okay.
      (unptrace agent name (eq unptrace 'all))
      ;; Ptrace current production.
      (sde-soar-cmd agent (concat "ptrace " name))))


(defun unptrace (&optional agent name unptrace)
  "Undo ptrace on a production.
Uses as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer.  If given a prefix
argument, undoes all ptraces active in the current agent.  Ptraces are
remembered on a per-agent basis, so tracing a production in one agent will
not automatically ptrace that production in any other agents that may also
share the same production.  To list the currently active traces, use
\\<sde-soar-mode-map>\\[sde-view-ptraces]"
  (interactive (sde-args
		 (sde-agent)
		 ;; Don't look for production if undoing all.
		 (or current-prefix-arg (sde-production-name-near-point))
		 current-prefix-arg))
  (if unptrace
      (sde-soar-query agent "unptrace")
      (sde-soar-cmd agent (concat "unptrace " name))))


;;;
;;; 7.C.  Commands for querying Soar
;;; 
;;; This includes the following Soar commands:
;;;   pgs, pgso               print, p, spr, wm
;;;   ms                      preferences
;;;   firing-counts           stats, memory-stats, rete-stats
;;;   matches                 watch, stack-trace-format, object-trace-format
;;;   list-chunks, -productions, -justifications
;;;   explain
;;;


(defun explain-on (&optional agent)
  "Turn on the Soar \"explain\" facility (Soar 6.1.1 or later).
This is equivalent to issuing a \"explain :on\" command to Soar."
  (interactive (sde-args (sde-agent)))
  (or sde-soar-version-at-least-6-1-1
      (error "The explain facility is unavailable prior to Soar 6.1.1."))
  (sde-soar-cmd (sde-agent) "explain :on"))


(defun explain-off (&optional agent)
  "Turn off the Soar \"explain\" facility (Soar 6.1.1 or later).
This is equivalent to issuing a \"explain :off\" command to Soar."
  (interactive (sde-args (sde-agent)))
  (or sde-soar-version-at-least-6-1-1
      (error "The explain facility is unavailable prior to Soar 6.1.1."))
  (sde-soar-cmd (sde-agent) "explain :off"))


(defun explain (&optional agent name cond args)
  "Execute the Soar \"explain\" command (Soar 6.1.1 or later).  
Explain provides some interpretation of backtraces generated during chunking.
Before this facility may be used, Soar must be instructed to track the
necessary information.  This must be done prior to when a given
chunk/justification is *created* or no explanation will be available for it.
The default for explain mode is off\; when explain mode is on, more memory is
used, and building chunks/justifications will be slower.  To turn explain on
and off, you may either

 1) Use the SDE commands \\[explain-on] and \\[explain-off], or
 2) Invoke this command with a prefix argument, which will make it prompt
    for optional arguments to give to \"explain\", and give it \":on\" or
    \":off\".

The SDE interface to explain attempts to determine its arguments in an
intelligent way using various heuristics.  When the cursor appears to be
over the name of a production or chunk, this command will perform

   explain <name>

This lists all of the conditions for the named junk or justification and the
\"ground\" which was included.  A \"ground\" is a WME which was tested in the
supergoal.  Just knowing which WME was tested may be enough to explain why
the chunk/justification exists.

You may ask Soar to explain why a particular condition was included in a
chunk/justification.  Place the cursor over an attribute in a condition
element in the LHS of a chunk, then invoke explain\; it will attempt to
extract the condition element consisting of just that attribute, and then
ask Soar to perform

   explain <name> <condition-number>

by first deducing the name of the containing production, then determining the
number for the particular condition under the cursor by silently querying
Soar, and finally issuing the full command \"explain <name>
<condition-number>\".  Soar will provide a list of the productions that
fired, and why they matched, in producing the given condition.

If SDE is unable to extract a production name and/or a condition element,
it will prompt the user for its arguments.  

If this command is given a prefix argument, it will prompt the user for
arguments, offering as initial defaults any <name> and <condition> that it
can find.  This allows you to edit and/or add optional arguments to explain.
For example, to issue

   explain :trace

use a prefix argument and delete any initial defaults provided (e.g., using
\\[beginning-of-line] followed by \\[kill-line]), then type \":trace\".
To issue the command

   explain <name> :trace

again use a prefix argument, and append \":trace\".

Backward compability note: this command is not available in Soar versions
prior to 6.1.1."
  (interactive)
  (or sde-soar-version-at-least-6-1-1
      (error "The explain facility is unavailable prior to Soar 6.1.1."))
  (let ((agent (sde-agent))
	(p-name (car (sde-extract-symbol)))
	(p-cond nil))
    ;; If point is over a production name, that's all we do.
    ;; Else, try to extract condition clause info.
    (or (sde-production-name-p p-name)
	(progn
	  (setq p-name nil)		; In case got some random string.
	  (cond ((setq p-cond (sde-explain-extract-condition-number))
		 ;; Point inside a production condition, & we extracted a number,
		 ;; which means it was in the output of a previous explain.
		 (setq p-name (sde-explain-hunt-sp-name)))

		((setq p-cond (sde-explain-extract-clause-near-point))
		 ;; Assume inside a production.  Try to extract production name.
		 (or (not (string= "" (setq p-name (sde-production-name-near-point))))
		     (not (string= "" (setq p-name (sde-read-string "Production :"))))
		     (error "Failed to find a production name."))
		 ;; Try to extract condition clause text, do an explain on it
		 ;; behind the user's back, match up the condition to the numbered
		 ;; conditions in the info, and finally use the name and the number
		 ;; as arguments.
		 (or (setq p-cond (sde-explain-ask-soar-for-number agent p-name p-cond))
		     (error "Unable to match condition element to output from `explain'."))))))
    (if (or (null p-name) current-prefix-arg)
	(setq p-name (sde-read-string "Production: " nil (or p-name ""))))
    (if (or (null p-name) (string= p-name ""))
	(error "No production name found or given."))
    (if current-prefix-arg
	(progn
	  (setq p-cond (sde-read-string "Condition number: " nil p-cond)
		args   (sde-read-string "Additional arguments: " 'sde-explain-hist))))
    ;; If we get this far, we have something usable; send it to Soar.
    (sde-soar-query agent (concat "explain " p-name " " (or p-cond "")))))


(defconst sde-condition-number-regexp "[ \t]*\\([0-9]+\\)\\s ?:"
  "Regexp to match condition numbers as printed by Soar `explain'.")

(defun sde-explain-hunt-sp-name ()
  ;; This is a brittle algorithm.
  (save-excursion
    (beginning-of-line)
    (and (looking-at sde-condition-number-regexp)
	 (sde-beginning-of-production)
	 (sde-production-name))))


(defun sde-explain-extract-clause-near-point ()
  (let ((data (sde-extract-id-attrib-value)))
    (concat (nth 0 data) " " (nth 1 data) "^" (nth 2 data) " " (nth 3 data))))


(defun sde-explain-extract-condition-number ()
  ;; Assume point is on a condition line from the output of explain.
  ;; Extract the condition number as a string and return it, or return nil.
  (save-excursion
    (beginning-of-line)
    (when (looking-at sde-condition-number-regexp)
      (sde-buffer-substring 1))))


(defun sde-explain-ask-soar-for-number (agent p-name p-cond)
  ;; Given production name P-NAME and condition clause P-COND, talk to
  ;; Soar and figure out the condition's number.
  (save-excursion
    (set-buffer (sde-get-buffer-create " *sde temp*"))
    (erase-buffer)
    (insert (sde-soar-silent-cmd agent (concat "explain " p-name)))
    (goto-char (point-min))
    (if (looking-at sde-soar-explain-error-regexp)
	(sde-soar-error (buffer-string) (concat "explain " p-name p-cond))
      (goto-char (point-max))
      (when (or (search-backward p-cond nil t)
		;; Remove any "goal", "state" or "impasse" and try again.
		(and (string-match "\\(goal\\|impasse\\|state\\) " p-cond)
		     (setq p-cond (concat "(" (substring p-cond (match-end 0))))
		     (search-backward p-cond nil t)))
	(beginning-of-line)
	(when (looking-at sde-condition-number-regexp)
	  (sde-buffer-substring 1))))))


(defun firing-counts (&optional agent args)
  "Execute the Soar \"firing-counts\" command.
This command prints how many times certain productions have fired.  With no
prefix argument, executes 'firing-counts' on the production under or immediately
before the cursor.  With prefix argument, prompts for a number (call it k)
and does `firing-counts k'.  If an integer argument (k) is given, only the
top k productions are listed.  If k=0, only the productions which haven't
fired at all are listed.  Note that firing counts are not reset by an
\(init-soar) -- the counts indicate the number of firings since the productions
were loaded or built."
  (interactive (sde-args
		 (sde-agent)
		 (if current-prefix-arg
		     (sde-prompt "Integer for 'firing-counts N': "
				 'sde-firing-counts-hist)
		     (sde-production-name-near-point))))
  (sde-soar-query agent (concat "firing-counts " args)))


(defun list-chunks (&optional agent args)
  "Execute the Soar \"list-chunks\" command.
If given a prefix arg, prompts for optional arguments to list-chunks.

List-chunks lists all the chunks in the system.

The optional [:internal] argument tells Soar to print productions in their
internal reordered (RETE net) form.

If a filename is given, the productions are printed to that file, otherwise
they are printed to the screen.  If :append is given, the file is appended
to, rather than overwritten."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'list-chunks': "
			     'sde-list-chunks-hist)))
  (sde-soar-query agent (concat "list-chunks " args)))


(defun list-justifications (&optional agent args)
  "Execute the Soar \"list-justifications\" command.
If given a prefix arg, prompts for optional arguments to list-justifications.

List-justifications lists all the justifications in the system.

The optional [:internal] argument tells Soar to print productions in their
internal reordered (RETE net) form.

If a filename is given, the productions are printed to that file, otherwise
they are printed to the screen.  If :append is given, the file is appended
to, rather than overwritten."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'list-justifications': "
			     'sde-list-justifications-hist)))
  (sde-soar-query agent (concat "list-justifications " args)))
  

(defun list-productions (&optional agent args)
  "Execute the Soar \"list-productions\" command.  
If given a prefix arg, prompts for optional arguments to list-productions.

List-productions lists all the productions in the system.

The optional [:internal] argument tells Soar to print productions in their
internal reordered (RETE net) form.

If a filename is given, the productions are printed to that file, otherwise
they are printed to the screen.  If :append is given, the file is appended
to, rather than overwritten."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'list-productions': "
			     'sde-list-productions-hist)))
  (sde-soar-query agent (concat "list-productions " args)))  


(defun matches (&optional agent name args)
  "Execute the Soar \"matches\" command.
Print partial match information for a production.  By default, uses the name
of the production that the cursor is in.  If given a prefix argument, prompts
for an additional argument to specify the level of detail wanted:

  0 (default) prints out just the partial match counts
  1 also prints the timetags of wmes at the first failing condition
  2 prints the wmes rather than just their timetags.

If not given a prefix argument, uses the same level argument given to this
command the last time it was invoked."
  (interactive (let ((sp-name (sde-production-name-near-point)))
		 (sde-args
		   (sde-agent)
		   sp-name
		   (sde-prompt (if sp-name
				   (concat "Arguments to 'matches " sp-name "': ")
				   "Arguments to 'matches': ")
			       'sde-matches-hist))))
  (sde-soar-query agent (concat "matches " name " " args)))


(defun matches-1 (&optional agent name)
  "Execute the Soar \"matches\" command with a \"level\" argument of 1.  
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer."
  (interactive (sde-args (sde-agent) (sde-production-name-near-point)))
  (sde-soar-query agent (concat "matches " name " 1")))


(defun matches-2 (&optional agent name)
  "Execute the Soar \"matches\" command with \"level\" argument of 2.  
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer."
  (interactive (sde-args (sde-agent) (sde-production-name-near-point)))
  (sde-soar-query agent (concat "matches " name " 2")))


(defun matches-3 (&optional agent name)
  "Execute the Soar \"matches\" command with \"level\" argument of 3.  
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer."
  (interactive (sde-args (sde-agent) (sde-production-name-near-point)))
  (sde-soar-query agent (concat "matches " name " 3")))


(defun memory-stats (&optional agent)
  "Execute the Soar \"memory-stats\" command.
This command prints out statistics on memory usage.  See also
\"rete-stats\" and \"stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "memory-stats"))


(defun ms (&optional agent args)
  "Execute the Soar \"ms\" command.
This command prints the current Match Set, i.e., a list of productions
that are about to fire or retract in the next preference phase.

With no arguments, this command prints out the production names for both 
Assertions and Retractions.

If given a prefix argument, prompts for additional optional argument, which can
be a combination of one character and/or one integer.

An optional character specifies listing of either Assertions or Retractions.

The optional integer specifies the level of detail wanted:  0 (the default)
prints out just the production names; 1 also prints the timetags of wmes
matched; and 2 prints the wmes rather than just their timetags."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'ms': " 'sde-ms-hist)))
  (if (or (null args)
	  (string-match "\\`\\s *\\(a\\|r\\)?\\s +\\(0\\|1\\|2\\)?\\s *\\'" args))
      (sde-soar-query agent (concat "ms " args) )
      (error "'ms' only accepts optional arguments 'a' or 'r' and/or '0', '1', or '2'")))


(defun pgs (&optional agent)
  "Execute the Soar \"pgs\" command.
Pgs stands for \"print goal stack\"\; it prints Soar's current context stack."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "pgs"))


(defun pgso (&optional agent)
  "Execute the Soar \"pgso\" command.
Pgs stands for \"print goal stack operators\"\; it prints only the operators
in Soar's current context stack." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "pgso"))


(defun preferences (&optional agent id attribute level)
  "Execute the Soar \"preferences\" command.
The SDE interface to the preferences command tries to extract its arguments
in an intelligent way.  If the cursor is sitting near a Soar identifier and
an attribute, for example when the cursor is between \"S1\" and \"^io-state\"
in the WM
                    (8: S1 ^io-state S1)
then it uses both the Soar identifier and the attribute as arguments.  If the
cursor is not near such a combination, this command will use as target
argument just the symbol under the cursor or the symbol nearest the cursor
from the left and will prompt for the attribute.  If it is given a
prefix argument, it prompts for the optional \"level\" argument.

This command prints all the preferences for the slot given by the \"id\" and
\"attribute\" arguments.  The optional \"level\" argument must be 0, 1, 2, or 3
\(0 is the default) and it indicates the level of detail requested:
  level 0 -- prints just the preferences themselves
  level 1 -- also prints the names of the productions that generated them
  level 2 -- also prints the timetags of the wmes matched by the productions
  level 3 -- prints the whole wmes, not just their timetags."
  (interactive (let ((data (sde-extract-id-attrib-value)))
		 (sde-args
		   (sde-agent)
		   (nth 0 data)		; Id.
		   (nth 2 data)		; Attribute.
		   (sde-prompt "Level of detail (0,1,2,3): " 'sde-preferences-hist))))
  (sde-soar-query agent (concat "preferences " id " " attribute " " level)))


(defun print-soar (&optional agent symbol args)
  "Execute the Soar \"print\" command.
Use as target the symbol under the cursor.  If given a prefix argument,
prompts for optional arguments to \"print\".

The print command is used to print items from production memory or working
memory.  It can take several kinds of arguments:

  arg ::= production-name  (print that production)
  arg ::= identifier       (id of the object to print)
  arg ::= integer          (timetag of wme--the identifier from the wme
                            indicates the object to be printed)
  arg ::= pattern          (pattern--same as if you listed as arguments
                            the timetags of all wmes matching the pattern)

  pattern ::= ( {identifier | '*'} ^ { attribute | '*'} { value | '*' } [+])

The optional argument :depth n overrides default-print-depth.

The optional argument :internal tells Soar to print things in their
internal form.  For productions, this means leaving conditions in their
reordered (RETE net) form.  For wmes, this means printing the individual
wmes with their timetags, rather than the objects.

:depth 0 is meaningful only for integer and pattern arguments, and only
when used along with :internal.  It causes just the matching wmes to be
printed, instead of all wmes whose id is an id in one of the matching wmes."
  (interactive (let ((symb (sde-symbol-near-point)))
		 (sde-args
		   (sde-agent)
		   symb
		   (sde-prompt (if symb
				   (concat "Arguments to 'print " symb "': ")
				 "Arguments to 'print': ")
			       'sde-print-hist))))
  (sde-soar-query agent (concat "print " args " " symbol)))

;; Soar `spr' is same thing as Soar `print'.

(fset 'spr 'print-soar)


(defun print-stats (&optional agent)
  "Execute the Soar \"print-stats\" command.
This command prints out some statistics on the current Soar run.
See also:  \"memory-stats\", \"rete-stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "print-stats"))


(defun rete-stats (&optional agent)
  "Execute the Soar \"rete-stats\".  
This command prints out statistics on the rete net.  See
also:  \"memory-stats\", \"stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "rete-stats"))


(defun wm (&optional agent symbol)
  "Execute the Soar \"wm\" command, which is shorthand for
     \"print :depth 0 :internal ...\"."
  (interactive (sde-args (sde-agent) (sde-symbol-near-point)))
  (sde-soar-query agent (concat "wm " symbol)))


;;;
;;; The view commands.  These are query commands grouped together to
;;; provide a more common interface.  Some are identical to other commands.
;;; 


(fset 'sde-view-chunks         'list-chunks)
(fset 'sde-view-pgs            'pgs)
(fset 'sde-view-pgso	       'pgso)
(fset 'sde-view-justifications 'list-justifications)
(fset 'sde-view-ms             'ms)
(fset 'sde-view-productions    'list-productions)


(defun sde-view-stats (&optional agent)
  "View the output of \"print-stats\", \"memory-stats\" and \"rete-stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-title-soar-output-buffer agent "print-stats, memory-stats, rete-stats")
  (sde-in-soar-output-buffer
    (insert (sde-soar-silent-cmd agent "print-stats"))
    (insert (sde-soar-silent-cmd agent "memory-stats"))
    (insert (sde-soar-silent-cmd agent "rete-stats"))))


(defun sde-view-pbreaks (&optional agent)
  "List all the productions currently under the effect of a pbreak."
  (interactive (sde-args (sde-agent)))
  (sde-print-pbreak-list))


(defun sde-view-ptraces (&optional agent)
  "List all the productions currently under the effect of a ptrace."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "ptrace"))


(defun sde-view-working-memory (&optional agent)
  "Print the contents of working memory, as would \"print (* ^ * *)\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "print (* ^ * *)"))



;;;
;;; 7.D.  Commands for manipulating Soar memories
;;; 

(defun excise (&optional agent name)
  "Execute the Soar \"excise\" command on the production under the cursor.
This removes the given production from Soar's memory.  
See also: \"excise-chunks\", \"excise-task\", \"excise-all\"." 
  (interactive (sde-args (sde-agent) (sde-production-name-near-point)))
  (sde-soar-cmd agent (concat "excise " name)))


(defun excise-chunks (&optional agent)
  "Execute the Soar \"excise-chunks\" command.
This command removes all chunks and justifications from the system.
See also: \"excise\", \"excise-task\", \"excise-all\"." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "excise-chunks"))


(defun excise-task (&optional agent)
  "Execute the Soar \"excise-task\" command. This command removes all
non-default productions from the system.  It also does an (init-soar).
See also: \"excise\", \"excise-chunks\", \"excise-all\"." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "excise-task"))


(defun excise-all (&optional agent)
  "Execute the Soar \"excise-all\" command.  This command removes all
productions from the system.  It also does an (init-soar).
See also: \"excise\", \"excise-chunks\", \"excise-task\"." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "excise-all"))


(defun excise-file (&optional agent file)
  "Excise the productions found in a file.  Prompts for file name.
This function searches the given file for all production definitions and
executes the Soar \"excise\" command on them in the current Soar agent."
  (interactive (sde-args
		 (sde-agent)
		 (car (comint-get-source "Excise file: " sde-soar-last-dir/file
				      sde-source-modes t))))
  (comint-check-source file)
  ;; Read file into Emacs, but don't bother setting mode, etc.  Loop,
  ;; searching for production names, create a list of them.  Then send excise
  ;; command to Soar.
  (let (buffer excise-list i names msg)
    (unwind-protect 
	 (save-excursion
	   (setq buffer (create-file-buffer file))
	   (set-buffer buffer)
	   (erase-buffer)
	   (if (condition-case ()
		   (insert-file-contents file t)
		 (file-error nil))
	       (progn
		 (message "Scanning file %s..." file)
		 (sde-strip-multi-line-comments)
		 (set-buffer-modified-p nil)
		 (goto-char (point-min))
		 (while (re-search-forward sde-sp-name-regexp nil t)
		   (push (sde-buffer-substring 1) excise-list))
		 (if (null excise-list)
		     (error "No productions found in %s" file))
		 (setq msg (format "Excising file %s " file))
		 ;; Soar cannot accept lines longer than 1000 chars.  Have to
		 ;; break up excises into small groups.  Some people have
		 ;; very long production names.  Groups of 10 seems reasonable.
		 (while excise-list
		   (setq i 0
			 names "")
		   (while (and (< i 10) excise-list)
		     (setq i           (+ i 1)
			   names       (concat names " " (car excise-list))
			   excise-list (cdr excise-list)))
		   (sde-soar-cmd agent (concat "excise" names)
				 nil nil nil nil msg)
		   (setq msg 'no-message)
		   (sde-soar-wait))
		 (message "Excising file %s...Done" file))))
      (kill-buffer buffer))))


(defun init-soar (&optional agent)
  "Execute Soar's \"init-soar\" command.  Takes no arguments.
This command re-initializes Soar.  It removes all wmes from working memory,
wiping out the goal stack, and resets all statistics (except the counts of
how many times each individual production has fired, used by the
\"firing-counts\" command)."
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "init-soar"))


(defun load-soar (&optional agent file switch)
  "Invoke the Soar \"load\" command on a file.  Prompts for the file name.
Automatically switches to the process buffer unless given a prefix argument.

The load command tells Soar to read commands from the given file instead of
the keyboard.  Soar will read and execute each command in the file, and then
go back to the keyboard.  Loads may be nested, i.e., the given file may
contain a command to load another file, and so on."
  (interactive (sde-args
		 (sde-agent)
		 (car (comint-get-source "Load file: " sde-soar-last-dir/file
					 sde-source-modes t))
		 (and current-prefix-arg t)))
  (comint-check-source file)
  (sde-add-soar-cmd-hook agent 'sde-update-pbreak-list)
  (sde-add-soar-cmd-hook agent 'sde-record-soar-loads)
  (sde-soar-cmd agent (concat "load \"" file "\"") nil nil (not switch)))


(defun load-defaults (&optional agent file switch)
  "Load the Soar defaults file into Soar.
Automatically switches to the process buffer unless given a prefix argument.

This determines the pathname to the defaults file from the value of the
variable `sde-soar-defaults-file'.  This variable should be set to the latest
copy of the defaults.soar6 file by the installer of SDE."
  (interactive (sde-args
		 (sde-agent)
		 sde-soar-defaults-file
		 (and current-prefix-arg t)))
  (or file (sde-error-unless-site-var-set 'sde-soar-defaults-file))
  (load-soar agent file switch))



;;;
;;; 7.E.  Commands for dealing with multiple agents
;;; 

(defun select-agent (&optional agent-name)
  "Select an agent to be the current recipient of commands in this buffer."
  (interactive)
  (sde-check-soar)
  (sde-error-unless-soar-multi-agent)
  (cond ((eq major-mode 'sde-soar-mode)
	 ;; Don't allow the user to reset a process buffer's agent.
	 ;; Instead, switch the agent buffer.
	 (switch-to-buffer (sde-buffer-of-agent (sde-ask-for-agent))))

	((memq major-mode sde-modes)
	 ;; A mode derived from Soar Mode.  It's okay to set an agent.
	 (setf (sde-buffer-agent sde-buffer-data) (or agent-name (sde-ask-for-agent)))
	 (message "Agent %s selected." (sde-buffer-agent sde-buffer-data)))

	(t
	 ;; We're not in a Soar Mode-based mode.  Makes no sense.
	 (error "Cursor is not in a buffer that communicates with Soar."))))


(defun schedule (&optional cycles)
  "Execute the Soar \"schedule\" command in multi-agent Soar.
Schedule agents in a multi-agent soar setting.  If given a prefix argument,
prompts for how many cycles to run the scheduler.  If given a blank in
response to the prompt, the scheduler runs until interrupted with a control-c
or all agents have completed processing.  If not given a prefix argument,
this command reuses the same argument given the last time it was invoked."
  (interactive (progn
		 (sde-check-soar)
		 (sde-error-unless-soar-multi-agent)
		 (sde-args (sde-prompt "Cycles to run: " 'sde-schedule-hist))))
  (sde-soar-cmd (sde-buffer-agent sde-buffer-data) (concat "schedule " cycles) nil nil t))


(defun agent-go (&optional agent args)
  "Execute the Soar \"agent-go\" command.  Uses the current buffer's agent
by default; if given a command prefix, prompts the user for an agent name
as well as the go arguments.

This command operates exactly the same as the \"go\" command, EXCEPT that
this only defines the go settings for a agent when it is selected to
run by the multi-agent scheduler.  Hence, the go settings are defined
for subsequent runs, but no runs are made when the command is read.
See also:  \"go\", \"run\"."
  (interactive (progn
		 (sde-check-soar)
		 (sde-error-unless-soar-multi-agent)
		 (sde-args
		   (if current-prefix-arg
		       (sde-ask-for-agent)
		       (sde-agent))
		   (sde-read-string "Arguments to 'agent-go': "
				    'sde-agent-go-hist))))
  (sde-soar-cmd agent (concat "agent-go " agent " " args)))


;; The next two are different depending on whether the user is running
;; Soar version 6.1.1 or earlier.  The approach taken here is to
;; implement the latest variant in terms of what the user has to use,
;; but make them backward compatible as much as possible.

(defun create-agent (&optional agent-name)
  "Execute the Soar \"create-agent\" command.
Prompts for a name, and creates a single soar agent identified by the given
name.  It also creates an interaction buffer in Emacs for the new agent.

Note that since the X Window interface is not used when running Soar
under SDE, the optional \"agent-display-type\" variable that can be given
to \"create-agent\" in the X Windows version is not available.

Backward compatibility note: in versions of Soar prior to 6.1.1, this command
was named \"create-agents\".  SDE only implements the one-agent-at-a-time
version of this command, which corresponds to Soar versions 6.1.1 or later.
If you are using an earlier release of Soar, this command will work properly, 
but you will still only be able to create one agent at a time."
  (interactive (progn
		 (sde-check-soar)
		 (sde-error-unless-soar-multi-agent)
		 (sde-args (sde-read-string "Agent name: "))))
  ;; Note: this is a special case.  The process filter notices the creation
  ;; of agents, so nothing more is needed here beyond sending the command.
  (sde-soar-cmd nil (concat (if sde-soar-version-at-least-6-1-1
				"create-agent "
				"create-agents ")
			    agent-name))
  (sde-pop-to-buffer (sde-buffer-of-agent agent-name)))


(defun destroy-agent (&optional agent-name)
  "Execute the Soar \"destroy-agent\" command.
Prompts for an agent name and removes that agent from Soar.  If this command
is issued in the window of the agent being destroyed, then control switches to
the global control agent.  

Backward compability note: this command is not available in Soar versions
prior to 6.1.1."
  (interactive
   ;; Messy.
   (if sde-soar-version-at-least-6-1-1
       (progn
	 (sde-check-soar)
	 (sde-error-unless-soar-multi-agent)
	 (sde-args (sde-ask-for-agent)))
       (error "Soar versions prior to 6.1.1 do not implement destroy-agent.")))
  (cond ((not (string= agent-name "control"))
	 ;; Either the process filter or `sde-udpate-on-output' will catch the
	 ;; deletion, so nothing more is needed here.
	 (sde-add-soar-cmd-hook agent-name 'sde-update-on-output)
	 (sde-soar-cmd nil (concat "destroy-agent " agent-name))
	 t)

	((yes-or-no-p "Destroying the control agent will exit Soar.  Confirm: ")
	 t)

	(t nil)))


(defun list-agents ()
  "List the Soar agents that SDE believes exist."
  (interactive)
  (sde-check-soar)
  (sde-error-unless-soar-multi-agent)
  (sde-title-soar-output-buffer nil "list-agents")
  (sde-in-soar-output-buffer
    (insert "  Agent name")
    (indent-to 24)
    (insert "Buffer\n")
    (insert "  ----------")
    (indent-to 24)
    (insert "------\n")
    (let ((agents sde-soar-agents))
      (while agents
	(insert "  " (car (car agents)))
	(indent-to 24)
	(insert (buffer-name (cdr (car agents))) "\n")
	(setq agents (cdr agents))))
    (insert "\n")))


;;;
;;; 7.F.  Help commands
;;; 


(defun soarnews ()
  "Invoke the Soar \"soarnews\" command and place the result in a buffer.
This command prints news about the current release of Soar."
  (interactive)
  (sde-check-soar)
  (sde-soar-query (sde-agent) "soarnews"))



;;;
;;; 7.G.  Misc. commands
;;; 

(defun max-elaborations (&optional agent num)
  "Execute the Soar `max-elaborations' command.

With no argument, this command prints the current value of the Soar variable
`max-elaborations'.  With a prefix argument, it prompts for an integer and
sets the current value.  

The `max-elaborations' variable controls the maximum number of elaboration
cycles allowed in a single decision cycle.  After this many elabloration
cycles have been executed, Soar proceeds to quiescence phase even if
quiescence hasn't really been reached yet.  (Max-elaborations is initially
100.)"
  (interactive (sde-args
		 (sde-agent)
		 (if current-prefix-arg
		     (sde-prompt "Integer for 'max-elaborations N': "
				 'sde-max-elaborations-hist)
		   nil)))
  (if num
      (sde-soar-cmd (sde-agent) (concat "max-elaborations " (int-to-string num)))
    (sde-soar-query agent "max-elaborations ")))


(defun max-chunks (&optional agent num)
  "Execute the Soar `max-chunks' command.

With no argument, this command prints the current value of the Soar variable
`max-chunks'.  With a prefix argument, it prompts for an integer and
sets the current value.  

The `max-chunks' variable controls the maximum number of chunks
allowed in a single decision cycle.  After this many elabloration
cycles have been formed, Soar proceeds to quiescence phase even if
quiescence hasn't really been reached yet.  (Max-chunks is initially
50.)"
  (interactive (sde-args
		 (sde-agent)
		 (if current-prefix-arg
		     (sde-prompt "Integer for 'max-chunks N': "
				 'sde-max-chunks-hist)
		   nil)))
  (if num
      (sde-soar-cmd (sde-agent) (concat "max-chunks " (int-to-string num)))
    (sde-soar-query agent "max-chunks ")))



(defun sde-switch-to-soar (eob-p)
  "Switch to the current Soar process.  
If already in a Soar process buffer, switch back to the buffer that last
switched to the process buffer.  With argument, position cursor at end of
buffer.  In cases where it is hard to determine from which buffer the user
last switched to the process buffer (e.g., because sde-switch-to-soar wasn't
used to do it) it will heuristically attempt to find the last-used SDE source
buffer."
  (interactive "P")
  (sde-check-soar)
  (if (eq major-mode 'sde-soar-mode)
      (if (and sde-last-buffer (not (equal sde-last-buffer (current-buffer))))
	  ;; In a soar buffer and we know where we were last.
	  (sde-pop-to-buffer sde-last-buffer)
	  ;; In a soar buffer but we don't know where we were last.  Punt.
	  (sde-pop-to-buffer (or (sde-previous-sde-buffer)
				 (other-buffer (current-buffer)))))
      ;; Not in a Soar mode buffer.
      (progn
	(setq sde-last-buffer (current-buffer))
 	(sde-show-buffer (or (sde-buffer-of-agent (sde-agent)) sde-soar-buffer) t)
 	(if eob-p
	    (goto-char (point-max))))))


(defun sde-previous-sde-buffer ()
  ;; Return the first buffer after this buffer in the buffer list that is
  ;; an SDE mode buffer.
  (let ((buffers (sde-get-sde-buffers)))
    (if (equal (current-buffer) (car buffers))
	(cdr buffers)
	(car buffers))))


(defun sde-close-and-send (arg)
  "Close and indent the current sp form, then send it to Soar."
  (interactive "P")
  (sde-reindent)
  (sde-close-all-sp arg)
  (if (eq major-mode 'sde-soar-mode)
      (sde-return)
    (save-excursion			; We're past last paren of sp,
      (backward-char 1)			;  so back up one char before
      (sde-send-production)		;  trying to eval.
      (forward-char 1))))


;;;----------------------------------------------------------------------------
;;; . Sending, excising, etc., productions and regions of productions.
;;;----------------------------------------------------------------------------


(defun sde-send-production ()
  "Send the production under the cursor to Soar.  
With prefix argument, send the production and switch to Soar.  If the new
production is a changed version of an already-known production of the same
name, Soar will automatically excise the previous definition and update it
with the new one.  Note that loading a production into Soar will unpbreak the
production, if a pbreak was in effect."
  (interactive)
  (sde-check-soar)
  (let ((body (sde-production-body)))
    (if body
	(let* ((name (sde-production-name))
	       (msg (concat "sp " name))
	       (agent (sde-agent)))
	  (sde-soar-cmd agent body nil nil current-prefix-arg msg msg)
	  ;; Clear possible pbreaks for this production.
	  (when (sde-pbreak-in-effect agent name)
	    (sde-unpbreak-production agent name)))
      (error "Point is not inside a production."))))


;; The next function actually sends the whole region, instead of extracting
;; the productions in the region and sending only those.  This seems more
;; flexible, allowing users to send more than productions in a buffer.  E.g.,
;; they may have load statements.  Without this facility, there would be no
;; way to send a non-production statement, except by sending the whole file.

(defun sde-region-send (start end)
  "Send the region between point and mark to the Soar process.  
With prefix argument, send the region and switch to Soar.  Note that doing so
will unpbreak all productions in the region, if pbreaks were in effect for
any of them."
  (interactive "r")
  (sde-check-soar)
  (let ((name (sde-region-name start end)))
    (sde-soar-cmd (sde-agent) (buffer-substring start end) 
		  nil nil current-prefix-arg name name)
    (sde-update-pbreak-list)))


(defun sde-region-sp-names (start end)
  "Return a list of the names of all the productions in the given region."
  (if (and start end)			; Check for non-nil param
      (save-excursion
	(save-restriction
	  (narrow-to-region start end)
	  (goto-char (point-max))
	  (let (lst done pt)
	    (while (not done)		; Work backwards from end of region.
	      (setq pt (sde-beginning-of-production)) ; This also moves point.
	      (when (looking-at "^(sp\\s +")
		(goto-char (match-end 0))
		(push (buffer-substring (point) (progn (forward-sexp 1)
						       (point)))
		      lst)
		(goto-char pt))
	      (setq done (= (point) (point-min)))) ; Reached top of region yet?
	    ;; Return list.
	    lst)))
    ;; Bad params.
    nil))


(defun sde-region-pbreak (start end)
  "Pbreak each production in the region.  
I.e., redefine each production to interrupt Soar when it fires.  If given a
positive prefix arg, undoes the pbreaks on each production in the region.  If
given a negative prefix arg, undoes all currently active pbreaks (whether for
productions in the region or not).  Pbreaks are remembered on a per-agent
basis, so pbreak'ing a production in one agent will not automatically pbreak
that production in any other agents that may also share the same production.
To list the currently active pbreaks, use `\\[sde-view-pbreaks]'"
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (pbreak (sde-agent) name current-prefix-arg)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-ptrace (start end)
  "Ptrace each production in the region.  
If given a positive prefix arg, undoes ptraces on all the productions in the
region.  If given a negative prefix arg, undoes all currently active ptraces
(whether for productions in the region or not).  To list all the currently
active ptraces, use `\\[sde-view-ptraces]'"
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (ptrace (sde-agent) name current-prefix-arg)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-excise (start end)
  "Excise each production in the region."
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (excise (sde-agent) name)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


;;;-----------------------------------------------------------------------------
;;; 8.  Command line interface in Soar process buffers.
;;;-----------------------------------------------------------------------------

;;; Input history.
;;;
;;; These are modifications of commands from comint.el.  The comint mode
;;; commands can't be used directly since they assume that there is only
;;; one buffer associated with a process.  In Soar Mode we may have multiple
;;; Soar agents, hence multiple pseudo-process-buffers in which we want these
;;; commands available.
;;;
;;; M-p     sde-previous-input             Cycle backwards in input history
;;; M-n     sde-next-input                 Cycle forwards
;;; M-s     sde-previous-similar-input     Previous similar input
;;; M-C-r   sde-previous-input-matching    Search backwards in input history


;; This code was stolen from comint.el and put here for portability.

(defun sde-skip-prompt ()
  "Skip past the text matching regexp sde-soar-prompt-regexp.
If this takes us past the end of the current line, don't skip at all."
  (if (and (looking-at sde-soar-prompt-regexp)
	   (<= (match-end 0) (save-excursion (end-of-line) (point))))
      (goto-char (match-end 0))))


(defun sde-bol (arg)
  "Goes to the beginning of line, then skips past the prompt, if any.
If a prefix argument is given (\\[universal-argument]), then no prompt skip 
-- go straight to column 0.

The prompt skip is done by skipping text matching the regular expression
`sde-soar-prompt-regexp'.

If you don't like this command, reset \\<sde-soar-mode-map>\\[sde-bol] to
`beginning-of-line' in your mode hook, `sde-soar-mode-hook'."
  (interactive "P")
  (beginning-of-line)
  (if (null arg) (sde-skip-prompt)))


(defun sde-mod (n m)
  "Returns N mod M. M is positive. Answer is guaranteed to be non-negative, 
and less than m."
  (let ((n (% n m)))
    (if (>= n 0) n
	(+ n
	   (if (>= m 0) m (- m)))))) ; (abs m)


(defun sde-previous-input (arg)
  "Cycle backwards through input history."
  (interactive "*p")
  (let ((len (ring-length sde-soar-input-ring)))
    (cond ((<= len 0)
	   (message "Empty input ring.")
	   (ding))
	  ((< (point) (marker-position sde-soar-buffer-mark))
	   (message "Not after process mark.")
	   (ding))
	  (t
	   (cond ((eq last-command 'sde-previous-input)
		  (delete-region (mark) (point)))
		 ((eq last-command 'sde-previous-similar-input)
		  (delete-region (marker-position sde-soar-buffer-mark) (point)))
		 (t                          
		  (setq sde-soar-input-ring-index
			(if (> arg 0) -1
			    (if (< arg 0) 1 0)))
		  (push-mark (point))))
	   (setq sde-soar-input-ring-index (sde-mod (+ sde-soar-input-ring-index arg) len))
	   (message "%d" (1+ sde-soar-input-ring-index))
	   (insert (ring-ref sde-soar-input-ring sde-soar-input-ring-index))
	   (setq this-command 'sde-previous-input)))))

	 
(defun sde-next-input (arg)
  "Cycle forwards through input history."
  (interactive "*p")
  (sde-previous-input (- arg)))


(defun sde-previous-input-matching (str)
  "Searches backwards through input history for substring match."
  (interactive (let* ((last-command last-command) ; Preserve around r-f-m.
		      (s (sde-read-string 
			  (format "Command substring (default %s): "
				  sde-soar-last-input-match))))
		 (list (if (string= s "") sde-soar-last-input-match s))))
  (setq sde-soar-last-input-match str)	; Update default.
  (if (not (eq last-command 'sde-previous-input))
      (setq sde-soar-input-ring-index -1))
  (let ((str (regexp-quote str))
        (len (ring-length sde-soar-input-ring))
	(n (+ sde-soar-input-ring-index 1)))
    (while (and (< n len) (not (string-match str (ring-ref sde-soar-input-ring n))))
      (setq n (+ n 1)))
    (cond ((< n len)
	   (sde-previous-input (- n sde-soar-input-ring-index)))
	  (t (if (eq last-command 'sde-previous-input) 
		 (setq this-command 'sde-previous-input))
	     (message "Not found.")
	     (ding)))))


(defvar sde-last-similar-string "" 
  "The string last used in an sde-previous-similar-input search.")

(defun sde-previous-similar-input (arg)
  "Reenters the last input that matches the string typed so far.  If repeated
successively older inputs are reentered.  If arg is 1, it will go back in the
history, if -1 it will go forward."
  (interactive "p")
  (if (< (point) (marker-position sde-soar-buffer-mark))
      (error "Not after process mark."))
  (if (not (eq last-command 'sde-previous-similar-input))
      (setq sde-soar-input-ring-index -1
	    sde-last-similar-string (buffer-substring
				     (marker-position sde-soar-buffer-mark)
				     (point))))
  (let* ((size (length sde-last-similar-string))
	 (len (ring-length sde-soar-input-ring))
	 (n (+ sde-soar-input-ring-index arg))
	 entry)
    (while (and (< n len) 
		(or (< (length (setq entry (ring-ref sde-soar-input-ring n))) size)
		    (not (equal sde-last-similar-string 
				(substring entry 0 size)))))
      (setq n (+ n arg)))
    (cond ((< n len)
	   (setq sde-soar-input-ring-index n)
	   (if (eq last-command 'sde-previous-similar-input)
	       (delete-region (mark) (point)) ; repeat
	       (push-mark (point)))	      ; 1st time
	   (insert (substring entry size)))
	  (t (message "Not found.") (ding) (sit-for 1)))
    (message "%d" (1+ sde-soar-input-ring-index))))


;;; Direct I/O in Soar buffers.
;;;
;;; C-c C-c sde-interrupt-soar
;;; C-c C-o sde-kill-output
;;; C-c C-r sde-show-output
;;; C-c C-u sde-kill-input
;;; RET     sde-return


;; Make sure not to do `sde-soar-wait' in the following, because if Soar
;; is doing something like loading files when the user attempts to
;; interrupt it, the `sde-soar-wait' will cause a hang.  Printing the prompts
;; is taken care of automatically by the process filter functions.

(defun sde-interrupt-soar ()
  "Interrupt the Soar process.
A limitation in Soar is that it can be interrupted only while it is running
productions, and not while it is loading a file."
  (interactive)
  (sde-check-soar)
  (if sde-soar-agents
      (sde-add-soar-cmd-hook nil 'sde-update-agents-after-interrupt))
  (interrupt-process sde-soar-process sde-soar-use-ptys))


(defun sde-kill-output ()
  "Kill all output from interpreter since last input."
  (interactive)
  (kill-region sde-soar-last-input-end (marker-position sde-soar-buffer-mark))
  (goto-char (marker-position sde-soar-buffer-mark))
  (insert "*** output flushed ***\n")
  (sde-soar-show-prompts)
  (set-marker sde-soar-buffer-mark (point)))


(defun sde-show-output ()
  "Display start of this batch of interpreter output at top of window.
Also put cursor there."
  (interactive)
  (goto-char sde-soar-last-input-end)
  (backward-char)
  (beginning-of-line)
  (set-window-start (selected-window) (point))
  (end-of-line))


(defun sde-kill-input ()
  "Kill all text from last stuff output by interpreter to point."
  (interactive)
  (if (> (point) (marker-position sde-soar-buffer-mark))
      (kill-region sde-soar-buffer-mark (point))))


;; The following are based on code from "Joseph S. Mertz" <jm7l+@andrew.cmu.edu>
;; submitted for CMU-released soar-mode v5.0.  Originally called
;; last-cl-top-window and next-cl-top-window.

(defun sde-backward-prompt ()
  "Move cursor to the previous prompt in the buffer and display that line at top."
  (interactive)
  (beginning-of-line)
  (if (not (re-search-backward sde-soar-prompt-regexp (point-min) t))
      (error "Cannot find any more prompts in this buffer."))
  (recenter 0)
  (end-of-line)
  (sde-bol nil))


(defun sde-forward-prompt ()
  "Move cursor to the next prompt in the buffer and display that line at the top."
  (interactive)
  (end-of-line)
  (if (not (re-search-forward sde-soar-prompt-regexp (point-max) t))
      (error "Cannot find any more prompts in this buffer."))
  (recenter 0)
  (end-of-line)
  (sde-bol nil))


(defun sde-return ()
  "Grab the current expression and send it to Soar.
If we have a complete sexp, send it.  Otherwise, indent appropriately."
  (interactive)
  (sde-check-soar)
  (let ((input (sde-get-old-input)))
    (if input
	;; Input is complete
	(if (sde-user-input-check input)
	    ;; Only proceed if filter returns true.
	    (progn
	      (if (>= (point) (marker-position sde-soar-buffer-mark))
		  (goto-char (point-max))
		  (progn
		    (goto-char (marker-position sde-soar-buffer-mark))
		    (insert input)))
	      (insert ?\n)
	      (set-marker sde-soar-buffer-mark (point))
	      (setq sde-soar-last-input-end (point))
	      (sde-update-on-input input)
	      (sde-add-to-input-history input)
	      (sde-add-soar-cmd-hook (sde-agent) 'sde-update-on-output)
	      (sde-soar-cmd (sde-agent) input nil (current-buffer) nil
			    'no-message 'no-message)))
	;; Input is not complete
	(if (= (marker-position sde-soar-buffer-mark) (point-max))
	    ;; Send a blank line.
	    (progn
	      (insert ?\n)
	      (set-marker sde-soar-buffer-mark (point))
	      (setq sde-soar-last-input-end (point))
	      ;; Send a blank string because sde-soar-cmd will append \n.
	      (sde-soar-cmd (sde-agent) "" nil (current-buffer) nil
			    'no-message 'no-message)
	      ;; Stupid hack because Soar versions prior to 6.1.1 didn't
	      ;; echo a prompt after a blank line.
	      (if (and (not sde-soar-version-at-least-6-1-1)
		       ;; This looser condition is for modsaf, in case we
		       ;; couldn't get the version number initially.  If
		       ;; use-ptys is set, assume we *are* in 6.1.1 or later.
		       (not sde-soar-use-ptys))
		  (progn
		    (sde-soar-show-prompts)
		    (sde-soar-update-state 'ready))))
	    ;; Dont' send anything, just reindent.
	    (progn
	      (insert ?\n)
	      (save-restriction
		(narrow-to-region sde-soar-buffer-mark (point-max))
		(sde-indent-line)))))))


(defun sde-add-to-input-history (cmd)
  ;; Add the CMD to this buffer's sde-soar-input-ring.
  (if (and (funcall sde-soar-input-ring-filter cmd)
	   (or (ring-empty-p sde-soar-input-ring)
	       (not (string= (ring-ref sde-soar-input-ring 0) cmd))))
      (ring-insert sde-soar-input-ring cmd)))


(defun sde-get-old-input ()
  ;; Return the sexp starting at the nearest previous prompt, or nil if none.
  (save-excursion
    (goto-char sde-soar-buffer-mark)
    (let* ((once (save-excursion (end-of-line) (point)))
	   (begin (point))
	   (end nil)
	   (done nil))
      (condition-case nil
	  (while (and (not done) (< (point) (point-max)))
	    (forward-sexp)
	    (setq end (point))
	    (skip-chars-forward " \t\n")
	    (if (and once (>= (point) once))
		(setq done t)))
 	(error (setq end nil)))
      (if end
	  (buffer-substring begin end)))))


;;; Command filtering, based in part on directory tracking code of cmushell.el.
;;;
;;; sde-user-input-check checks and validates inputs, and possibly signals
;;; errors.  For example, in multi-agent Soar we don't want the user to
;;; select a different agent in an agent buffer (accidentally or
;;; deliberately).  The input filter checks for that.
;;;
;;; sde-update-on-input looks at the arguments passed to the Soar command
;;; before it's sent to Soar, and updates the argument history list for the
;;; particular command and other internal information.  This is necessary to
;;; insure that when the user next invokes a particular command from a key
;;; binding, the default arguments are consistent with what they last typed
;;; to the process.  Also, the user may have typed commands into the Soar
;;; process buffer before ever invoking the command via key binding or M-X.
;;; We want the defaults to be consistent no matter which way the command is
;;; invoked first.
;;;
;;; Note that it is necessary to call sde-update-on-input before Soar gets the
;;; command, to insure that the argument history is set prior to commands
;;; that may generate long output.  This is important if the user wants to
;;; type a *new* command while a command is still executing.  Otherwise, if
;;; sde-update-on-input was made part of the sde-update-on-output, the command
;;; argument histories wouldn't get updated until Soar finished.
;;;
;;; sde-update-on-output is called after each command issued by the user.
;;; It examines the output and performs any necessary post-command
;;; processing.  Other modes, such as cmushell, simply use the
;;; comint-input-sentinel to match commands like cd and then extract the
;;; arguments to the command that the user typed.  However, we have to get
;;; Soar's actual response to the commands.  For example, we don't want to
;;; just change directories to the directory specified in a cd command,
;;; because while the user may have typed a valid directory, Soar may not be
;;; able to cd to it.  So we have to get Soar's own output.  A simple way to
;;; do this is to run this postprocessing function.  sde-return invokes
;;; sde-soar-cmd with sde-update-on-output as a hook function to be called
;;; by the process filters.


(defmacro sde-match-soar-cmd (regexp input)
  (` (let (start end)
       (string-match "^(?\\s *" (, input))	  ; Skip whitespace.
       (setq start (match-end 0))
       (and (string-match (, regexp) (, input) start)
	    (progn
	      (setq end (match-end 0))
	      (cond ((eq end (string-match "\\s *\\(\)\\|$\\)" (, input) end))
		     "")			  ; No arg.
		    ((eq end (string-match "\\s +\\([^)]*\\)\)?$" (, input) end))
		     (sde-substring (, input) 1)) ; Have arg, return it.
		    (t nil)))))))	          ; Something else.


(defun sde-user-input-check (input)
  ;; The input filter checks and validates inputs, and possibly signals
  ;; errors.  It is called on every input to Soar.
  (cond ((and (sde-match-soar-cmd "select-agent" input) sde-soar-agents)
	 ;; Don't allow changing agents in agent buffers.
	 (error "A process buffer's agent is always fixed.  Switch agents using `%s'."
		(sde-key-for-command 'select-agent)))
	(t t)))


(defun sde-update-on-input (input)
  ;; Updates the argument history list for the command in the INPUT string,
  ;; and/or performs any other processing necessary prior to sending a
  ;; command.  This function is called on each input passed to the Soar
  ;; process.
  (condition-case err
      (let (args)
	(cond ((setq args (sde-match-soar-cmd "go" input))
	       (sde-set-default args 'sde-go-hist))

	      ((setq args (sde-match-soar-cmd "run" input))
	       (sde-set-default args 'sde-run-hist))

	      ((setq args (sde-match-soar-cmd "p\\|print" input))
	       ;; Ugly match string to find :depth and :internal arguments
	       ;; regardless of their order in the argument.
	       (if (string-match "\\(:depth\\s +[0-9]\\|:internal\\)\\s +\\(:depth\\s +[0-9]\\|:internal\\)?" args)
		   (if (match-beginning 2)
		       (sde-set-default (sde-substring args 1 2) 'sde-print-hist)
		       (sde-set-default (sde-substring args 1) 'sde-print-hist))))

	      ((setq args (sde-match-soar-cmd "matches" input))
	       (if (string-match "\\(\\S +\\)\\s +\\([0-9]+\\)" args)
		   (sde-set-default (sde-substring args 2)
				    'sde-matches-hist)))

	      ((setq args (sde-match-soar-cmd "ms" input))
	       (sde-set-default args 'sde-ms-hist))

	      ((setq args (sde-match-soar-cmd "firing-counts" input))
	       (or (string= args "")
		   (let ((first-elem (car (read-from-string args))))
		     (if (integerp first-elem)
			 (sde-set-default first-elem 'sde-firing-counts-hist)))))

	      ((setq args (sde-match-soar-cmd "preferences" input))
	       ;; 3rd arg, if present, is the optional level specification
	       (if (and (string-match "\\(\\S +\\)\\s +\\(\\S +\\)\\s +\\([0-3]+\\)" args)
			(match-beginning 3))
		   (sde-set-default (sde-substring args 3)
				    'sde-preferences-hist)))

	      ((setq args (sde-match-soar-cmd "list-chunks" input))
	       (sde-set-default args 'sde-list-chunks-hist))

	      ((setq args (sde-match-soar-cmd "list-justifications" input))
	       (sde-set-default args 'sde-list-justifications-hist))

	      ((setq args (sde-match-soar-cmd "list-productions" input))
	       (sde-set-default args 'sde-list-productions-hist))
	      ))
    (error
     (message (car (cdr err))))))


;; Note that sde-update-on-output is called via `sde-run-soar-cmd-hook'.
;; which means that it will only be called on *successful* Soar commands, not
;; on commands that incur errors.  This is important so that we don't have to
;; double check here whether the commands succeeded.

(defun sde-update-on-output ()
  ;; This function is called *after* the input has been processed by Soar.
  ;; It watches for commands such as pwd and performs special-case
  ;; processing, such as setting this buffer's directory to match the output
  ;; of pwd.  
  (condition-case err
      (let (args)
	(cond ((setq args (sde-match-soar-cmd "load" sde-soar-cmd-input))
	       (sde-update-pbreak-list)
	       (sde-record-soar-loads))

	      ((and sde-soar-track-cd
		    (sde-match-soar-cmd "cd\\|chdir\\|pushd\\|popd\\|pwd"
					sde-soar-cmd-input))
	       (sde-soar-check-pwd))

	      ((setq args (sde-match-soar-cmd "destroy-agent" sde-soar-cmd-input))
	       ;; A successful destroy-agent command will mean that Soar will
	       ;; already have removed the agent, so just need to do bookkeeping.
	       ;; Note that Soar after 6.1.1alpha will print a message that will
	       ;; be caught by the process filter, so this code is really for
	       ;; older versions of Soar.
	       (if sde-soar-agents
		   (sde-remove-agent args)
		   (message "Cannot create or destroy agents in a single-agent Soar.")))

              ((sde-match-soar-cmd "create-agents?" sde-soar-cmd-input)
	       ;; The process filter will notice agent creation.  Here just do
	       ;; something intelligent for errors.
	       (if (not sde-soar-agents)
		   (message "Cannot create or destroy agents in a single-agent Soar.")))
	      ))
    (error
     (message (car (cdr err))))))


(defun sde-soar-track-cd-toggle ()
  "Turn directory tracking on and off in a Soar process buffer."
  (interactive)
  (setq sde-soar-track-cd (not sde-soar-track-cd))
  (message "directory tracking %s."
	   (if sde-soar-track-cd "on" "off")))

(fset 'track-cd-toggle 'sde-soar-track-cd-toggle)

(defun sde-soar-check-pwd ()
  (condition-case nil
      (let ((output (sde-soar-silent-cmd (sde-agent) "pwd")))
	(string-match sde-soar-pwd-output-regexp output)
	(setq default-directory (file-name-as-directory (sde-substring output 1)))
	(sde-soar-pwd-message))
    (error nil)))


(defun sde-soar-pwd-message ()
  ;; Show the current directory on the message line.
  ;; Pretty up dirs a bit by changing "/usr/jqr/foo" to "~/foo".
  (let ((dir default-directory))
    (if (string-match (format "^%s\\(/\\|$\\)" (getenv "HOME")) dir)
	(setq dir (concat "~/" (substring default-directory (match-end 0)))))
    (if (string-equal dir "~/")
	(setq dir "~"))
    (message dir)))


;;;-----------------------------------------------------------------------------
;;; 9.  Interface to Soar
;;;-----------------------------------------------------------------------------
;;;
;;; General notes:
;;; 
;;; This interface uses asynchronous communication with Soar.  This means
;;; that Emacs functions that send input to Soar come back immediately, and
;;; Soar's output must be gathered by a process filter.  The control of
;;; communication with Soar is therefore split between several routines,
;;; which makes it harder to understand and debug.
;;;
;;; There are different filters for different states of Soar, such as when
;;; Soar is just starting up or when it is running productions.  Output
;;; received by SDE from Soar is not contiguous.  Each filter may get called
;;; many, many times before Soar is finished with a given command.  The only
;;; way for a filter to know that Soar has finished production output for a
;;; particular command is to watch the output stream and look for some sign,
;;; in particular the appearance of a Soar prompt.
;;;
;;; A problem involving the prompt is that since there is only one physical
;;; Soar process, Soar only prints one prompt after each command is finished,
;;; even if that command is "schedule".  This is problematic if we're trying
;;; to give each agent its own buffer, since we need a separate prompt for
;;; each buffer.  The solution here is to catch and remove the prompts printed
;;; by Soar, and instead stuff an explicit prompt in each agent buffer that
;;; needs it after each command.
;;;
;;; In the main (real) process buffer, `sde-soar-buffer-mark' is the
;;; process-mark.  In agent buffers, `sde-soar-buffer-mark' is a marker for
;;; where to dump new output.  In both cases, code uniformly refers to
;;; `sde-soar-buffer-mark' rather than making a distinction between the
;;; buffer types.
;;;
;;; The main internal function for sending commands to Soar is `sde-soar-cmd'.
;;; It calls a low-level function `sde-soar-send' that does the real work of
;;; sending input to Soar.  Several global variables are used to communicate
;;; between `sde-soar-send' and the process filters.  The main global
;;; variables are below and have names all prefixed with `sde-soar-cmd-'.
;;; All of these variables are part of the current command being sent to Soar.
;;; Using such separate variables is stylistically poor.  It would have been
;;; better to gather up these separate variables into a defstruct, but the
;;; defstruct facility in Emacs Lisp is inefficient and would require loading
;;; a lot more than we need.  Using a vector is another approach but that
;;; complicates the code more.
;;;-----------------------------------------------------------------------------

(defvar sde-soar-cmd-input          ""  "String sent to Soar.")
(defvar sde-soar-cmd-dest           nil "Destination buffer for this cmd.")
(defvar sde-soar-cmd-last-dest      nil "Last buffer that received output.")
(defvar sde-soar-cmd-output-handler nil "Filter for handling output.")
(defvar sde-soar-cmd-output         nil "Output from a command when gathered.")
(defvar sde-soar-cmd-error          nil "Whether Soar signaled an error.")

(defvar sde-soar-output-start 0
  "Starting pos of output in sde-soar-buffer from new Soar processes.")

(defvar sde-soar-banner ""
  "Initial message put up by Soar when it starts up.")

(defconst sde-soar-output-handlers
    '((display-buffers . sde-handle-soar-output-to-agents)
      (display-special . sde-handle-soar-output-diverted)
      (gather          . sde-handle-soar-output-gathered))
  "Alist mapping output actions to functions that handle them.")

(defvar sde-newline-hack nil)


;; Variable `sde-soar-state' indicates the status of Soar.  It is a superset
;; of the process-status of the process and can be set in two ways: either by
;; the process-sentinel, or by calling `sde-soar-update-state'.  The process
;; sentinel is called by Emacs whenever the Soar process changes state.
;; Function `sde-soar-update-state' is also called explicitly throughout this
;; code.  Possible values of `sde-soar-state' are:
;;
;;   'starting   -- Soar started but before any prompts or load statements.
;;   'ready      -- No commands pending; Soar waiting at a prompt.
;;   'running    -- While Soar is running production cycles or loading files.
;;   'load-error -- Soar encountered an error while loading a file.
;;   'stop       -- Process is stopped but continuable.
;;   'exit       -- Process has exited naturally or due to an error.
;;   'signal     -- Process received a fatal signal and has exited.
;;
;; SDE thinks of Soar as being in one of these possible states at any given
;; moment.  Soar may make the following transitions:
;;
;;    starting   -> ready, running, exit
;;    running    -> ready, load-error, exit
;;    ready      -> running, exit
;;    load-error -> running, exit
;;    exit       -> starting

(defvar sde-soar-state 'ready
  "The current state of Soar.")

(defun sde-soar-update-state (&optional status)
  ;; Updates the status indicator in all Soar mode lines.  If given optional
  ;; arg STATUS, sets status to that value.  `sde-soar-state-string' is set
  ;; to a string equivalent of the status indicator, for use in mode lines.
  (setq sde-soar-state        (or status (process-status sde-soar-process))
	sde-soar-state-string (format "%s" sde-soar-state))
  (sde-update-mode-lines)
  sde-soar-state)


(defun sde-soar-process-sentinel (process status)
  "Main process sentinel for Soar process."
  (let ((match-data (match-data)))
    (if (string-match sde-soar-runtime-error-regexp status)
	(progn
	  (sde-pop-to-buffer sde-soar-buffer)
	  (store-match-data match-data)
	  (error "Soar has terminated abnormally."))
	(let ((inhibit-quit t))
	  (sde-soar-update-state)
	  (store-match-data match-data)))))


;; Hacked from comint-exec from comint.el 2.03.

(defun sde-exec (buffer name command switches)
  ;; Fire up a process in BUFFER named NAME using COMMAND with SWITCHES.
  ;; Blasts any old process running in the buffer.
  ;; Doesn't set the buffer mode.
  (save-excursion
    (set-buffer buffer)
    (if (get-buffer-process buffer)	; Blast any old process.
	(delete-process (get-buffer-process buffer)))
    ;; Crank up a new process.
    (let ((proc (sde-exec-1 name buffer command switches)))
      ;; Jump to the end, and set the process mark.
      (goto-char (point-max))
      (set-marker (process-mark proc) (point)))
    buffer))


;; Stolen from comint.el 2.03, for portability.  The comments:

;; This auxiliary function cranks up the process for comint-exec in
;; the appropriate environment. It is twice as long as it should be
;; because emacs has two distinct mechanisms for manipulating the
;; process environment, selected at compile time with the
;; MAINTAIN-ENVIRONMENT #define. In one case, process-environment
;; is bound; in the other it isn't.

(defun sde-exec-1 (name buffer command switches)
  (if (boundp 'process-environment)	; Not a completely reliable test.
      (let ((process-environment
	     (sde-update-env process-environment
			     (list (format "TERMCAP=emacs:co#%d:tc=unknown"
					   (screen-width))
				   "TERM=emacs"
				   "EMACS=t"))))
	(apply 'start-process name buffer command switches))

      (let ((tcapv (getenv "TERMCAP"))
	    (termv (getenv "TERM"))
	    (emv   (getenv "EMACS")))
	(unwind-protect
	     (progn (setenv "TERMCAP" (format "emacs:co#%d:tc=unknown"
					      (screen-width)))
		    (setenv "TERM" "emacs")
		    (setenv "EMACS" "t")
		    (apply 'start-process name buffer command switches))
	  (setenv "TERMCAP" tcapv)
	  (setenv "TERM"    termv)
	  (setenv "EMACS"   emv)))))
	     

;; Stolen from comint.el 2.03.  The original comments:

;; This is just (append new old-env) that compresses out shadowed entries.
;; It's also pretty ugly, mostly due to elisp's horrible iteration structures.

(defun sde-update-env (old-env new)
  (let ((ans (reverse new))
	(vars (mapcar (function (lambda (vv)
			(and (string-match "^[^=]*=" vv)
			     (substring vv 0 (match-end 0)))))
		      new)))
    (while old-env
      (let* ((vv (car old-env)) ; vv is var=value
	     (var (and (string-match "^[^=]*=" vv)
		       (substring vv 0 (match-end 0)))))
	(setq old-env (cdr old-env))
	(cond ((not (and var (member var vars)))
	       (if var (setq var (cons var vars)))
	       (setq ans (cons vv ans))))))
    (nreverse ans)))


(defun sde-start-soar (program start-dir switches)
  ;; Starts soar program PROGRAM in directory START-DIR. 
  ;; First kills any old Soar buffers.
  (if (and sde-use-multiple-frames (not (eq sde-use-multiple-frames 'output))
	   (sde-buffer-exists-p sde-soar-buffer))
      (sde-delete-buffer-frame sde-soar-buffer))
  (sde-delete-agent-buffers)
  (sde-delete-soar-error-buffer)
  (message "Starting Soar (\"%s\") . . ." program)
  ;; Reset certain internal variables
  (sde-reset-soar-histories)
  (sde-reset-pbreak-list)
  ;; Initialize buffer and start process.
  (save-excursion
    (setq sde-soar-buffer (sde-get-buffer-create
			   (sde-make-agent-buffer-name sde-soar-default-name)))
    (set-buffer sde-soar-buffer)
    ;; Make sure to start in the right directory.
    (sde-cd (file-name-as-directory start-dir))
    (sde-soar-mode)
    (goto-char (point-max))		 ; Move point where new output will go.
    (setq sde-soar-output-start (point)	 ; Must set this before starting Soar.
	  sde-soar-banner            ""
	  sde-soar-version           "6" ; Reset to some value
	  sde-soar-last-input-end    sde-soar-output-start
	  sde-soar-cmd-dest          sde-soar-buffer
	  sde-soar-cmd-last-dest     sde-soar-buffer
	  sde-soar-cmd-output-handler 'sde-handle-soar-output-to-agents
	  sde-newline-hack    nil)
    (sde-reset-soar-cmd-hook)
    (sde-add-soar-cmd-hook nil 'sde-record-soar-version)
    (sde-add-soar-cmd-hook nil 'sde-record-soar-loads)
    (sde-add-soar-cmd-hook nil 'sde-record-soar-help-topics)
    (sde-add-soar-cmd-hook nil 'sde-finish-starting-soar)
    (let ((process-connection-type sde-soar-use-ptys))
      (sde-exec sde-soar-buffer sde-soar-default-name program switches))
    (sde-soar-update-state 'starting)
    (setq sde-soar-process     (get-buffer-process sde-soar-buffer)
	  sde-soar-buffer-mark (process-mark sde-soar-process)
	  sde-soar-agents      nil)
    (set-process-filter sde-soar-process 'sde-soar-starting-filter)
    (set-process-sentinel sde-soar-process 'sde-soar-process-sentinel)
    (if (and sde-use-multiple-frames (not (eq sde-use-multiple-frames 'output)))
	(sde-create-buffer-frame sde-soar-buffer sde-soar-agent-buffer-defaults))))


(defun sde-finish-starting-soar ()
  (message "Soar %s running in %s mode."
	   sde-soar-version (if sde-soar-agents "multi-agent" "single-agent"))
  (if sde-soar-beep-after-setup
      (beep t))
  (sde-update-buffer-window sde-soar-buffer)
  (run-hooks 'sde-soar-hook))


(defun sde-soar-starting-filter (process output)
  ;; Filter installed immediately after Soar is started up.  Runs until
  ;; Soar changes state to 'running, 'ready, or 'exit.
  ;; Gathers first output from Soar and sets things up appropriately.
  (let ((inhibit-quit t)
	(orig-buffer (current-buffer))
	(match-data (match-data))
	(continuing nil))	 
    (unwind-protect
	 (progn
	   (set-buffer (process-buffer process))
	   (if (string-match sde-soar-started-multi-regexp output)
	       ;; Perform initializations for multi-agent Soar.
	       (progn
		 (rename-buffer (sde-make-agent-buffer-name "control"))
		 (setq sde-soar-buffer        (current-buffer)
		       sde-soar-cmd-last-dest sde-soar-buffer)
		 (setf (sde-buffer-agent sde-buffer-data) "control")
		 (set-buffer (process-buffer process))
		 (sde-add-new-agent-buffer-pair "control" sde-soar-buffer)))
	   (goto-char (point-max))
	   (cond ((string-match sde-soar-startup-error-regexp output)
		  ;; Error during startup.  This means Soar didn't start and
		  ;; the process sentinel will have been called and it will
		  ;; have set the status appropriately.
		  (insert output)
		  (set-marker sde-soar-buffer-mark (point-max))
		  (set-process-filter process 'sde-soar-running-filter)
		  (sde-show-buffer sde-soar-buffer)
		  (error "Error occurred while starting up Soar."))
		 
		 ((string-match sde-soar-loading-regexp output)
		  ;; Loading a .init file.  Note that Soar can only go from
		  ;; the starting state to the running state as a result of
		  ;; loading a .init file.  Show the output up to the end of
		  ;; the "Loading ..." message, and take the rest of the
		  ;; output and hand it to the main process filter.
		  (setq continuing (match-end 0)
			sde-soar-banner
			(concat (buffer-string)
				(substring output 0 (1- (match-beginning 0)))))
		  (insert (substring output 0 (match-end 0)))
		  (insert "\n")
		  (set-marker sde-soar-buffer-mark (point-max))
		  (set-process-filter process 'sde-soar-running-filter)
		  (sde-show-buffer sde-soar-buffer)
		  (sde-soar-update-state 'running))

		 ((string-match sde-soar-prompt-regexp output)
		  ;; Finished starting & waiting at a prompt.  This case won't
		  ;; match if the previous case ever matches, because the
		  ;; previous case will reset the process filter.
		  (setq sde-soar-banner
			(concat (buffer-string)
				(substring output 0 (1- (match-beginning 0)))))
		  (insert output)
		  (set-marker sde-soar-buffer-mark (point-max))
		  (sde-show-buffer sde-soar-buffer)
		  (sde-add-soar-cmd-hook nil 'sde-reset-after-send)
		  (sde-run-soar-cmd-hook))

		 (t
		  ;; Use insert-before-markers to get window updated.
		  (insert-before-markers output)
		  (set-marker sde-soar-buffer-mark (point-max)))))
      (store-match-data match-data)
      (if (sde-buffer-exists-p orig-buffer)
	  (set-buffer orig-buffer))
      (if continuing
	  ;; Make sure to process remainder of what `output' contained,
	  ;; but do it outside the unwind-protected set-buffer calls.
	  (sde-soar-running-filter process (substring output continuing))))))


;; Sending commands to the Soar process.

(defun sde-soar-query (agent input)
  ;; Wrapper around sde-soar-cmd for a common operation:  querying Soar
  ;; for something and dumping the output into the Soar output buffer.
  (sde-soar-cmd agent input nil 'special))


(defun sde-soar-silent-cmd (agent input)
  ;; Send command to Soar, silently.  Useful for querying Soar behind the
  ;; scenes.  Ignores errors.
  (sde-soar-cmd agent input 'wait 'silent))


;; sde-soar-cmd has a horrible parameter scheme, but it was hard to figure
;; out how to get the necessary functionality without resorting to many
;; similar-but-different versions of sde-soar-cmd.  Still, this is a good
;; candidate for revision sometime in the future.
;;
;; A tough issue here is how to handle errors.  If sde-soar-cmd doesn't wait,
;; it can't catch errors.  However, you don't want the process filters
;; necessarily to catch errors, because in some cases (e.g., sde-return) you
;; never want an error popup but instead want errors just to be passed to the
;; buffer.  In other cases, we want the errors to be ignored.  The solution
;; adopted here is (1) for "special" sends, errors are caught & displayed by
;; process-special-filter; (2) for all other sends, errors are handled by
;; sde-soar-cmd and not by process filters, and then only if WAIT-P is
;; non-nil.

(defun sde-soar-cmd (agent input &optional wait-p dest switch m-msg b-msg)
  ;; To agent AGENT, send INPUT. Optional arguments:
  ;; Wait until Soar returns if WAIT-P.
  ;; If WAIT-P is nil, errors are only caught if DEST is 'special.
  ;; If DEST is nil, dumps output to appropriate agent buffers.
  ;; If DEST is 'special, dumps output to Soar output buffer, doesn't print
  ;;   any messages in the process buffer.  If DEST is 'special but
  ;;   sde-soar-use-output-buffer is nil, the 'special setting is ignored.
  ;; If DEST is 'silent, doesn't dump output anywhere, doesn't print any
  ;;   messages in the process buffer.
  ;; Switch to destination buffer if SWITCH is non-nil.
  ;; M-MSG specifies a string to echo in the minibuffer.
  ;; If M-MSG is nil, INPUT is echoed in the minibuffer instead.
  ;; If M-MSG is 'no-message, no message is echoed under any circumstance.
  ;; B-MSG specifies a string to print in the agent process buffer.
  ;; If B-MSG is nil, INPUT is printed in the process buffer, unless
  ;;   DEST is 'special or 'silent.
  ;; If B-MSG is 'no-message, nothing is echoed under any circumstance.
  (let* ((inhibit-quit t)
	 (special (and (eq dest 'special) sde-soar-use-output-buffer))
	 (silent (eq dest 'silent))
	 (output-action (cond (wait-p 'gather)
			      (special 'display-special)
			      (t 'display-buffers)))
	 (dest (cond ((or silent special) nil)
		     ((and sde-soar-agents (sde-buffer-of-agent agent))
		      (sde-buffer-of-agent agent))
		     (t sde-soar-buffer)))
	 (m-msg (and (not silent) (not (eq m-msg 'no-message)) (or m-msg input)))
	 (b-msg (and (not silent) (not (eq b-msg 'no-message)) (or b-msg input))))
    ;; When `sde-soar-cmd' is called, it's always for a new command and hence
    ;; we need to initialize the output buffer.
    (if special
	(sde-title-soar-output-buffer agent (or m-msg "")))
    (if m-msg
	(message (sde-quote-string m-msg)))
    ;; Go to the destination buffer, move point to the start of where new
    ;; output should go, and if there's a b-msg to display, print it.
    (if (and (not silent) (not special))
	(save-excursion
	  (set-buffer dest)
	  (goto-char (marker-position sde-soar-buffer-mark))
	  (setq sde-soar-last-input-end (point))
	  (if b-msg
	      (progn
		(insert-before-markers (concat ";;; " b-msg "\n"))
		(set-marker sde-soar-buffer-mark (point))))))
    ;; Send command.
    (sde-soar-send agent input dest wait-p output-action)
    ;; Post-processing.
    (cond ((and (eq output-action 'gather) sde-soar-cmd-error (not silent))
	   (sde-soar-error sde-soar-cmd-output (or m-msg b-msg "")))
	  ((and switch (not silent) (not special))
	   (sde-pop-to-buffer dest))
	  ((and (eq output-action 'gather) (not silent))
	   (sde-soar-show-prompts)
	   sde-soar-cmd-output)
	  ((eq output-action 'gather)
	   ;; If we get an error in this mode, just return nil.
	   (and (not sde-soar-cmd-error) sde-soar-cmd-output)))))


;; `sde-soar-send' does the job of actually sending strings to Soar,
;; first setting up the handlers and filters to catch the output.

(defun sde-soar-send (agent input dest wait-p action)
  ;; Low-level send function.
  ;; We set the status here to 'running, create the sde-soar-input record,
  ;; send the input to Soar, and let the process-filter gather the results
  ;; and reset the status to nil when Soar returns with a prompt.
  (setq sde-soar-cmd-input     input
	sde-soar-cmd-output-handler (cdr (assoc action sde-soar-output-handlers))
	sde-soar-cmd-dest      dest
	sde-soar-cmd-output    ""
	sde-soar-cmd-last-dest nil
	sde-soar-cmd-error     nil)
  ;; Make sure to reset the process filter and other things after the command
  ;; is finished.  Must do this as the last thing after a command *and* the
  ;; commands that *it* sends have finished.  Do it using a hook, because
  ;; other hooks may recursively send commands to Soar.  This may result in
  ;; the reset function getting called more than once, but since it always
  ;; gets called at the end of a chain of commands, that should be okay.
  (sde-add-soar-cmd-hook nil 'sde-reset-after-send)
  ;; If we're running multiple agents, we normally first have to specify
  ;; which agent is to receive this command.  This is done by prepending a
  ;; select-agent command to the real command.  Unfortunately we then have to
  ;; nuke the extra prompt introduced by the initial select-agent command;
  ;; here this is done by using a special process filter whose purpose is to
  ;; eat the prompt and then install the real filter.  Exceptions: if we're
  ;; still starting up or we got a load-error, don't prepend the
  ;; select-agent, because Soar won't be able to take it.
  (if (and agent (not (memq sde-soar-state '(starting load-error))))
      (progn
	(setq input (concat (format sde-soar-agent-select-fmt agent) "\n" input))
	(set-process-filter sde-soar-process 'sde-soar-nuke-prompt-filter))
      (set-process-filter sde-soar-process 'sde-soar-running-filter))
  (sde-soar-update-state 'running)
  (sde-send-string sde-soar-process (concat input "\n"))
  (let ((inhibit-quit nil))
    (if wait-p
	(sde-soar-wait))))


(defun sde-reset-after-send ()
  ;; Reset filters to something sane, because some Soar systems may
  ;; start producing output that is not in response to something SDE sends.
  ;; Otherwise, the last thing SDE sent may leave funny filters installed.
  (set-process-filter sde-soar-process 'sde-soar-running-filter)
  (setq sde-soar-cmd-output-handler 'sde-handle-soar-output-to-agents
	sde-newline-hack nil)
  (sde-soar-update-state 'ready))


;; More code from comint.el 2.03, for portability to different Emacses.

(defvar sde-input-chunk-size 512
  "Long inputs sent to Soar processes are broken up into chunks of this size.")

(defun sde-send-string (proc str)
  ;; Send PROCESS the contents of STRING as input.  This is equivalent to
  ;; `process-send-string', except that long input strings are broken up into
  ;; chunks of size `sde-input-chunk-size'.  Processes are given a chance to
  ;; output between chunks.  This can help prevent processes from hanging when
  ;; you send them long inputs on some OS's.
  (let* ((len (length str))
	 (i (min len sde-input-chunk-size)))
    (process-send-string proc (substring str 0 i))
    (while (< i len)
      (let ((next-i (+ i sde-input-chunk-size)))
	(accept-process-output)
	(process-send-string proc (substring str i (min len next-i)))
	(setq i next-i)))))


(defun sde-soar-wait ()
  ;; Wait for Soar to be ready for input.
  (while (not (eq sde-soar-state 'ready))
    (accept-process-output)
    (sit-for 0)))


;; Emacs receives the output from Soar in spurts.  The process filters have
;; to gather up all the output (or write it out to buffers, as appropriate)
;; and keep checking for the appearance of a prompt in the output from Soar.
;; Once the prompt is seen the process filter assumes that the current
;; command has been finished.  Of course, the filter also needs to check the
;; output stream for signs of errors.
;;
;; Emacs apparently will queue the calls to process filters in a smart way,
;; so that even if a new filter has been installed, the current invocation of
;; the current filter will finish before the next filter function gets called
;; on the continuing output.
;;
;; Prompts have to be handled specially in multi-agent case.  First,
;; `sde-soar-send' has to issue a "select-agent" command prior to each real
;; command in the multi-agent case, to make sure the proper context is set
;; for the actual command.  The fast way to issue the select-agent is to
;; simply prepend it to the real command string.  (See `sde-soar-send'
;; above.)  But this means that in Soar's subsequent output there will be one
;; initial prompt string that must be removed before getting the real output.
;; This is handled by using an initial filter, 
;; `sde-soar-nuke-prompt-filter'.

(defun sde-soar-nuke-prompt-filter (process output)
  ;; Filter used in multi-agent situations to remove the extra prompt string
  ;; that will be the first thing to appear in the output stream from Soar.
  ;; This simply finds the end of the prompt string, resets the process
  ;; filter to the real process filter, and then hands the rest of the
  ;; initial output to that filter prior to exiting.
  (let* ((inhibit-quit t)
         (match-data (match-data))
	 (prompt (string-match sde-soar-prompt-regexp output))
	 (end (match-end 0)))
    (store-match-data match-data)
    (set-process-filter process 'sde-soar-running-filter)
    (if prompt
	(sde-soar-running-filter process (substring output end))
	(sde-soar-running-filter process output))))


;; This filter really needs to be as efficient as possible.  Unfortunately
;; it also needs to handle a lot.
;;
;; This filter receives an indeterminate amount of output from Soar at
;; any given invocation.  Sometimes it may get two or more "Creating agent..."
;; lines in the same chunk of output, sometimes it may get just one line,
;; etc.  It's very important that none of the output is dropped.  The filter
;; calls itself recursively to process pieces of Soar output when the
;; output includes messages such as creation and destruction of agents.
;;
;; Handling load errors: if a load error is encountered, the call to
;; `sde-soar-running-filter' that contains the error message will presumably
;; be the last one in the queue, since then Soar will be waiting for input.
;; So we flag the error by setting the status to 'load-error, which will let
;; `sde-soar-send' and other functions take special cautions the next time
;; they're called.  The next command to Soar will automatically result in the
;; status getting changed to 'running, so nothing more is needed for that.
;;
;; The business with `sde-newline-hack' is a bogosity to get around a
;; problem created by differences in Soar's output from platform to platform.
;; It is most important for catching the scheduling string.  What happens is
;; that sometimes SDE will receive
;;
;;      \nScheduling agent ...
;;
;; as *one* chunk of output, and sometimes it will receive
;;
;;      \n
;;      Scheduling agent ...
;;
;; as *two* chunks of output.  We really want this to be received
;; consistently as one chunk of output, but despite that the C code in Soar
;; is written that way, it doesn't always happen.  This creates a problem for
;; SDE, which gets output from Soar asynchronously, because a single newline
;; received from Soar can be a legitimate output that should be dumped into
;; the agent buffer.  But given a single newline, SDE can't tell if it's part
;; of a "Scheduling ..." statement, or part of the user's program.  So we
;; have to keep this extra bit of state around and hack around the problem in
;; `sde-soar-running-filter': we don't print single newlines immediately, but
;; rather remember they were seen and prepend them to the following bit of
;; output.  This permits pattern matching regexps to work as expected.


(defun sde-soar-running-filter (process output)
  (if (string= "\n" output)
      (setq sde-newline-hack t)
      (let ((inhibit-quit t)
	    (match-data (match-data)))
	(unwind-protect
	     (let ((dest (or sde-soar-cmd-last-dest sde-soar-cmd-dest
			     sde-soar-buffer)))
	       (if sde-newline-hack
		   (setq output (concat "\n" output)
			 sde-newline-hack nil))
	       ;; Trim buffer from the top if it exceeds a certain size.
	       (if (> (buffer-size) sde-soar-output-truncation-threshold)
		   (delete-region 1 sde-soar-output-truncation-amount))
	       (cond
		 ;; If we see that something has created a new agent, have to
		 ;; process the input up to the creation message, create the
		 ;; agent, redirect output, and handle the rest of the output.
		 ((string-match sde-soar-agent-creation-regexp output)
		  (let ((agent  (sde-substring output 1))
			(end    (match-end 0))
			(leader (substring output 0 (match-beginning 0))))
		    ;; Seems safe to skip this whitespace.
		    (or (string-match "\\`[ \t\n]*\\'" leader)
			(sde-soar-running-filter process leader))
		    (if (sde-handle-creation agent (sde-buffer-directory dest))
			(setq sde-soar-cmd-last-dest (sde-buffer-of-agent agent)))
		    (sde-soar-running-filter process (substring output end))))
		 
		 ;; If we see that something has destroyed an agent, remove it
		 ;; from SDE, along with its buffer.  This is drastic.
		 ((string-match sde-soar-agent-destruction-regexp output)
		  (let* ((agent (sde-substring output 1))
			 ;; Have to determine is-dest prior to removing agent.
			 (is-dest (eq dest (sde-buffer-of-agent agent)))
			 (end (match-end 0))
			 (leader (substring output 0 (match-beginning 0))))
		    ;; Seems safe to skip this whitespace.
		    (or (string-match "\\`[ \t\n]*\\'" leader)
			(sde-soar-running-filter process leader))
		    (if (and (sde-handle-destruction agent) is-dest)
			(setq sde-soar-cmd-last-dest sde-soar-buffer))
		    (sde-soar-running-filter process (substring output end))))

		 ;; Otherwise process output normally.
		 (t
		  (let ((prompt (string-match sde-soar-prompt-regexp output)))
		    (if prompt
			(setq output (substring output 0 prompt)))
		    (funcall sde-soar-cmd-output-handler output dest prompt)
		    (cond ((string-match sde-soar-load-error-regexp output)
			   (sde-soar-update-state 'load-error)
			   (sde-update-buffer-window dest)
			   (sde-pop-to-buffer dest))
			  (sde-soar-cmd-error
			   (sde-soar-update-state 'ready))
			  (prompt
			   (setq sde-newline-hack nil)
			   (sde-run-soar-cmd-hook)))))))
	  (store-match-data match-data)))))


(defun sde-handle-creation (agent-name root-dir)
  ;; Creates agent named AGENT-NAME if it doesn't already exist.
  ;; Returns nil if agent existed, non-nil otherwise.
  (and (not (sde-agent-exists-p agent-name))
       (progn
	 (sde-add-agent agent-name root-dir sde-soar-banner)
	 (message "Agent %s created." agent-name))))


(defun sde-handle-destruction (agent-name)
  ;; Destroys agent named AGENT-NAME if it exists.
  ;; Returns nil if agent does not exist, non-nil otherwise.
  (and (sde-agent-exists-p agent-name)
       (progn
	 (sde-remove-agent agent-name)
	 (message "Agent %s destroyed." agent-name))))


(defun sde-handle-soar-output-to-agents (output dest prompt)
  ;; Output handler for dumping output straight to agent buffers.
  ;; Loop on output string, looking for indication of which agent is supposed
  ;; to receive each piece of the output.
  (while (and (not (string= output ""))
	      (string-match sde-soar-sched-regexp output))
    (sde-insert-in-buffer dest (substring output 0 (match-beginning 0)))
    (setq dest   (sde-buffer-of-agent (sde-substring output 1))
	  output (substring output (match-end 0))))
  (setq sde-soar-cmd-last-dest dest)
  ;; Following handles both left-over output in multi-agents case and all
  ;; output in single agent case.
  (if (not (string= output ""))
      (sde-insert-in-buffer dest output))
  (if prompt
      (sde-soar-show-prompts)))


(defun sde-handle-soar-output-diverted (output dest prompt)
  ;; Output handler for displaying output in the Soar output buffer.  Notices
  ;; and flags errors, and in that case doesn't display output buffer.
  (if (string-match sde-soar-error-regexp output)
      ;; I hope that when errors occur, the output contains the complete
      ;; error message!
      (progn
	(setq sde-soar-cmd-error  t
	      sde-soar-cmd-output output)
	(sde-soar-error sde-soar-cmd-output sde-soar-cmd-input))
      (sde-in-soar-output-buffer
	(goto-char sde-soar-buffer-mark)
	(insert-before-markers output))))

  
(defun sde-handle-soar-output-gathered (output dest prompt)
  ;; Output handler for storing Soar's output without displaying it.
  ;; Ignores errors. 
  (setq sde-soar-cmd-output (concat sde-soar-cmd-output output)))


(defun sde-update-buffer-window (&optional buffer)
  (let ((window (sde-buffer-window buffer t)))
    (and window
	 (let ((inhibit-quit t)
	       (orig-buffer (current-buffer))
	       (orig-window (selected-window)))
	   (if buffer
	       (set-buffer buffer))
	   (select-window window)
	   (goto-char sde-soar-buffer-mark)
	   (recenter)
	   (if buffer
	       (set-buffer orig-buffer))
	   (select-window orig-window)))))


(defun sde-update-agents-after-interrupt ()
  ;; Walk through all agent windows and print a message that an interrupt
  ;; was sent to Soar.  This is needed because otherwise, in a multi-agent
  ;; setting, only one agent will have gotten the text from Soar that
  ;; says '*** Ctrl-C Interrupt ***'.
  (let ((buffers (if sde-soar-cmd-last-dest
		     (remove* sde-soar-cmd-last-dest sde-soar-agents
			      :test (function (lambda (x y)
						(equal x (cdr y)))))
		   sde-soar-agents)))
    (while (save-excursion
	     (set-buffer (cdr (car buffers)))
	     (goto-char (point-max))
	     (beginning-of-line)
	     (if (looking-at sde-soar-prompt-regexp)
		 (delete-region (match-beginning 0) (match-end 0)))
	     (insert-before-markers sde-soar-interrupt-fmt)
	     (set-marker sde-soar-buffer-mark (point))
	     (setq buffers (cdr buffers))))
    (sde-soar-show-prompts)))


;; The tricky thing about the next function is that it has to handle the case
;; where the user is running multiple agents but is only interacting with one
;; agent.  Then, the remaining agents' prompts will (probably) still be
;; showing, and we don't want to insert another prompt string.  So this
;; function has to find where a prompt should go, and then check that there
;; isn't already a prompt there.

(defun sde-soar-show-prompts ()
  ;; Cycle through agent buffers and check that they all show a prompt for
  ;; the user, if appropriate.    
  (save-excursion
    (if sde-soar-agents
	(let ((agents sde-soar-agents)
	      newline)
	  (while agents
	    (set-buffer (cdr (car agents)))
	    (goto-char (point-max))
	    (beginning-of-line)
	    (if (not (looking-at sde-soar-prompt-regexp))
		(progn
		  (setq newline (progn
				  (forward-line -1)
				  (char-equal (following-char) ?\n)))
		  (goto-char (point-max))
		  (if (not newline)
		      (insert-before-markers "\n"))
		  (insert-before-markers
		   (format sde-soar-agent-prompt-fmt (car (car agents))))
		  (set-marker sde-soar-buffer-mark (point)))
		(goto-char (point-max)))
	    (setq agents (cdr agents))))
	;; Single agent case
	(progn
	  (set-buffer sde-soar-buffer)
	  (goto-char (point-max))
	  (or sde-soar-version-at-least-6-1-1
	      (insert-before-markers "\n"))
	  (insert-before-markers sde-soar-prompt-str)
	  (set-marker sde-soar-buffer-mark (point))))))


;; Based on bridge-insert from bridge.el.

(defun sde-insert-in-buffer (dest-buffer output)
  ;; Insert into BUFFER the given OUTPUT.  The buffer point is moved if
  ;; `sde-soar-move-point-on-output' is non-nil.  This function depends on
  ;; each buffer having its own local value of `sde-soar-buffer-mark' so that
  ;; this function can know where to put output.
  (let* ((dest-window (selected-window))
	 (orig-buffer (current-buffer))
	 (orig-window (selected-window))
	 (at-end nil))
    (unwind-protect
	 (progn
	   ;; Switch to buffer to access its local vars.
	   (set-buffer dest-buffer)
	   (if (eq (window-buffer dest-window) dest-buffer)
	       (setq at-end (= (point) sde-soar-buffer-mark))
	       (setq dest-window (sde-buffer-window dest-buffer t)))
	   (save-excursion
	     (goto-char sde-soar-buffer-mark)
	     (insert-before-markers output)
	     (set-marker sde-soar-buffer-mark (point)))
	   (if (and dest-window sde-soar-move-point-on-output)
	       (progn
		 (or at-end
		     (goto-char sde-soar-buffer-mark))
		 (if (not (pos-visible-in-window-p (point) dest-window))
		     (progn
		       (set-buffer dest-buffer)
		       (select-window dest-window)
		       (goto-char sde-soar-buffer-mark)
		       (recenter '(center))
		       (select-window orig-window)
		       (set-buffer orig-buffer))))))
      (set-buffer orig-buffer))))


;;;-----------------------------------------------------------------------------
;;; 5.  Internal utilities for running functions after Soar finishes a command.
;;;-----------------------------------------------------------------------------


(defvar sde-soar-cmd-hook-alist nil
  "Alist of agent buffers and functions to be called after a Soar cmd.") 


(defun sde-add-soar-cmd-hook (agent func)
  ;; For AGENT append FUNC to the list of functions run on
  ;; `sde-soar-cmd-hook-alist' but only if it is not already on the list.
  (let ((buffer (or (sde-buffer-of-agent agent) sde-soar-buffer))
	(pair))
    (or (setq pair (assoc buffer sde-soar-cmd-hook-alist))
	(setq pair (list buffer)
	      sde-soar-cmd-hook-alist (cons pair sde-soar-cmd-hook-alist)))
    (or (member func (cdr pair))
	(setcdr pair (append (cdr pair) (list func))))))


(defun sde-run-soar-cmd-hook ()
  ;; Run `sde-soar-cmd-hook-alist' if Soar didn't encounter an error.
  ;; Resets `sde-soar-cmd-hook-alist' to nil.
  (if (and sde-soar-cmd-hook-alist (not sde-soar-cmd-error))
      (let ((alist sde-soar-cmd-hook-alist)
	    (orig-buffer (current-buffer))
	    pair)
	;; Have to set `sde-soar-cmd-hook-alist' to nil now, before executing
	;; the hooks, otherwise if the hook functions send commands to Soar,
	;; we would get called again and an infinite loop would result. 
	(setq sde-soar-cmd-hook-alist nil)
	(unwind-protect
	     (while (progn
		      (setq pair (car alist))
		      (and (sde-buffer-exists-p (car pair))
			   (set-buffer (car pair))
			   (mapcar 'funcall (cdr pair)))
		      (setq alist (cdr alist))))
	  (set-buffer orig-buffer)))))


(defun sde-reset-soar-cmd-hook ()
  (setq sde-soar-cmd-hook-alist nil))


;;;-----------------------------------------------------------------------------
;;; 10. Support for pbreak
;;;-----------------------------------------------------------------------------


(defvar sde-soar-pbreak-list nil
  "List of productions currently under the effect of a pbreak.  Each element of
the list is a quadruples of the form (NAME ORIGINAL AGENT NEW), where NAME is
the name of the production, ORIGINAL is the full original production
definition, AGENT is the name of the agent in which the pbreak is in effect,
and NEW is the new definition of the production according to Soar.")


(defun sde-pbreak-in-effect (agent name)
  ;; Return entry from sde-soar-pbreak-list if there is a pbreak in effect
  ;; in AGENT for production named NAME.
  (let ((pbreaks sde-soar-pbreak-list)
	(data nil)
	(found nil))
    ;; There may be more than one instance of the same production having
    ;; a pbreak, but for different agents; this has to work in such cases.
    (while
	(not (if (and (setq data (car pbreaks))
		      (string= name (nth 0 data)) 
		      (string= agent (nth 2 data)))
		 (setq found data)
		 (null (setq pbreaks (cdr pbreaks))))))
    found))


(defun sde-pbreak-production (agent name)
  ;; In agent AGENT, establish a pbreak for production NAME, and add it to
  ;; the sde-soar-pbreak-list.
  (let ((sp (sde-get-production agent name)))
    (if (not sp)
	(error "Production \"%s\" not yet defined in Soar." name))
    ;; Search for last closing paren and add interrupt stmt. 
    (sde-soar-silent-cmd agent (concat
				(substring sp 0 (string-match ")[ \t\n]*\\'" sp))
				"(interrupt) )"))
    ;; Store away info, including Soar's idea of the new production.
    (push (list name sp agent (sde-get-production agent name))
	      sde-soar-pbreak-list)))


(defun sde-unpbreak-production (agent name)
  ;; Undo the pbreak on the named production.
  (let ((pbreaks sde-soar-pbreak-list)
	data new)
    (while (setq data (car pbreaks))
      (if (and (string= name (nth 0 data)) (string= agent (nth 2 data)))
	  ;; Redefine back to original.
	  (sde-soar-silent-cmd (nth 2 data) (nth 1 data))
	  (push data new))
      (setq pbreaks (cdr pbreaks)))
    (setq sde-soar-pbreak-list new)))
	  

(defun sde-unpbreak-all ()
  ;; Undo all pbreaks.
  (while sde-soar-pbreak-list
    (sde-soar-silent-cmd (nth 2 (car sde-soar-pbreak-list))  ; agent
			 (nth 1 (car sde-soar-pbreak-list))) ; sp body
    (setq sde-soar-pbreak-list (cdr sde-soar-pbreak-list))))


(defun sde-print-pbreak-list ()
  ;; Show which productions are under the effect of a pbreak.
  (sde-title-soar-output-buffer nil "pbreak")
  (sde-in-soar-output-buffer
    (let ((pbreaks sde-soar-pbreak-list))
      (while pbreaks
	(if sde-soar-agents
	    (insert "  [agent: " (nth 2 (car pbreaks)) "]"))
	(insert "  " (nth 0 (car pbreaks)) "\n")
	(setq pbreaks (cdr pbreaks)))
      (insert "\n"))))


(defun sde-reset-pbreak-list ()
  ;; Reset the list.
  (setq sde-soar-pbreak-list nil))


(defun sde-update-pbreak-list ()
  ;; Check with Soar the status of each of the productions thought to be
  ;; in pbreak mode.  If Soar's definition doesn't match the stored definition,
  ;; assume something has redefined the production, and clear its pbreak list
  ;; entry.
  (let ((pbreaks sde-soar-pbreak-list)
	data new)
    (while (setq data (car pbreaks))
      (if (string= (nth 3 data) (sde-get-production (nth 2 data) (nth 0 data)))
	  (push data new))
      (setq pbreaks (cdr pbreaks)))
    (setq sde-soar-pbreak-list new)))



;;;-----------------------------------------------------------------------------
;;; 11. Support for Help
;;;-----------------------------------------------------------------------------


(defun sde-soar-help (topic)
  ;; Query Soar for help on the given topic, and display in *Help* buffer.
  (sde-check-soar)
  (let ((text (sde-soar-silent-cmd nil (concat "help " topic))))
    (with-output-to-temp-buffer "*Help*"
      (princ text))))


(defun sde-record-soar-help-topics ()
  ;; Query Soar for the names of all commands available under help,
  ;; and add these to sde-topics-obarray.
  (let ((topics (sde-soar-silent-cmd nil "list-help-topics")))
    ;; Skip the lead-in text.
    (string-match "topics:" topics)
    (setq topics (substring topics (match-end 0)))
    ;; Loop through the text, extracting topic names.
    (while (string-match "\\s *\\([^ ,\n]+\\)," topics)
      (intern (sde-substring topics 1) sde-topics-obarray)
      (setq topics (substring topics (match-end 0))))))


;;;-----------------------------------------------------------------------------
;;; 12. Support for tracking productions
;;;-----------------------------------------------------------------------------

;;; All files loaded into Soar are tracked, so that operations such as finding
;;; production source code can work more effectively.  A hash table is used
;;; instead of a list, because the entries have to be unique, and Emacs doesn't
;;; have any built-in functions for efficiently doing member with string= on
;;; a long list of elements.  Using a hash table results in a set of unique
;;; entries, and we can use mapatoms to iterate over the elements.

(defun sde-record-soar-loads ()
  ;; Check through current Soar buffer for signs of Soar having loaded files.
  ;; Record their pathnames.
  (save-excursion
    (goto-char sde-soar-last-input-end)
    (let ((dir default-directory)
	  files-loaded)
      (while (re-search-forward sde-soar-loading-or-chdir-regexp nil t)
	(beginning-of-line)
	(cond ((looking-at sde-soar-loading-regexp)
	       ;; Loading a file.
	       (push (expand-file-name (sde-buffer-substring 1) dir) files-loaded))

	      ((looking-at sde-soar-returning-regexp)
	       ;; Soar moved up from a directory.
	       ;; The math in the substring below removes the trailing "/."
	       (let ((trailing (buffer-substring (match-beginning 1) (- (match-end 1) 2))))
		 (if (string-match (concat "\\`\\(.*/\\)" trailing "\\'") dir)
		     (setq dir (sde-substring dir 1)))))

	      ((looking-at sde-soar-chdir-regexp)
	       ;; Soar moved down into a directory.
	       (setq dir (expand-file-name (sde-buffer-substring 1) dir))))
	;; Move on, to avoid getting stuck on same line!
	(end-of-line))
      ;; Now go through the files, beginning with the first, and record it.
      (setq files-loaded (nreverse files-loaded))
      (dolist (file files-loaded)
	(sde-record-file file))
      ;; Assign task name based on what was loaded.  Start from the beginning
      ;; and look for something we're not ignoring.  Assume it's a load file
      ;; for the task.  !!! Bug: very fragile heuristic.
      (dolist (file files-loaded)
	(when (not (sde-ignore-file-p file))
	  (setf (sde-buffer-task sde-buffer-data) (sde-get-file-task file))
	  (return))))))


;; To do:
;; If user does excise-task, we're in trouble.  Need a resync loads list.


;;;-----------------------------------------------------------------------------
;;; 13. Miscellaneous support code
;;;-----------------------------------------------------------------------------


(defun sde-get-production (agent name)
  ;; Get named production from Soar.
  (sde-soar-silent-cmd agent (concat "print " name)))


(defun sde-soar-is-alive ()
  ;; Return non-nil if a Soar process is running, nil otherwise.
  (and sde-soar-process
       (memq (process-status sde-soar-process) '(run stop))))


(defun sde-check-soar ()
  "Check if there's a Soar running, and complain if there isn't."
  (or (sde-soar-is-alive) (sde-complain-no-soar-process)))


(defun sde-complain-no-soar-process ()
  "Complain that there isn't a running Soar, and ask to start one."
  (interactive)
  (beep)
  (if (cond ((null sde-soar-process)
	     (y-or-n-p "No Soar process running.  Start one? "))
	    ((eq sde-soar-state 'exit)
	     (y-or-n-p "Soar has quit.  Restart Soar? "))
	    ((eq sde-soar-state 'signal)
	     (y-or-n-p "Soar has received a fatal signal.  Restart Soar? ")))
      (let ((buffer (buffer-name (current-buffer))))
	;; Protect against deletion of buffers, resulting from restarting of
	;; Soar, by trying to reset our current buffer to what it was before
	;; we restarted Soar.
	(soar)
	(sde-soar-wait)
	(if (get-buffer buffer)
	    (switch-to-buffer buffer)
	    (switch-to-buffer sde-soar-buffer))) ; Punt.
      (error "No Soar process running.")))


(defun sde-record-soar-version ()
  ;; Get version number of Soar process.
  (let ((version-str (sde-soar-silent-cmd nil "version")))
    (if (and version-str (not (string= version-str "")))
	(let (minor subminor)
	  (string-match sde-soar-version-regexp version-str)
	  (setq sde-soar-version (sde-substring version-str 1)
		;; Deconstruct the minor numbers.
		minor    (string-to-int (or (sde-substring version-str 3) "0"))
		subminor (string-to-int (or (sde-substring version-str 4) "0"))
		sde-soar-version-at-least-6-1-1
		         (or (> minor 1) (and (= minor 1) (>= subminor 1)))))
	;; If couldn't extract a version number, must still set
	;; sde-soar-version to something, to get around the rare case where
	;; someone runs a post-6.0 version of Soar, quits it, and then starts
	;; a pre-6.0 version.  We don't want to leave sde-soar-version with
	;; an incorrect value, so this sets it to a default.
	(setq sde-soar-version               "6"
	      sde-soar-version-at-least-6-1-1 nil))))


;;;-----------------------------------------------------------------------------
;;; 14. Closing statements.
;;;-----------------------------------------------------------------------------

;; Provide

(provide 'sde-soar-mode)

;;; Emacs indentation support for macros and form hooks for edebug.
;;;
;;; Local Variables:
;;; eval:(put 'sde-in-soar-output-buffer 'lisp-indent-function 0)
;;; eval:(put 'sde-in-soar-output-buffer 'lisp-indent-hook 0)
;;; eval:(put 'sde-args                  'lisp-indent-function 0)
;;; eval:(put 'sde-args                  'lisp-indent-hook 0)
;;; eval:(put 'sde-in-soar-output-buffer 'edebug-form-spec '(&rest form))
;;; eval:(put 'sde-buffer-of-agent       'edebug-form-spec '(form))
;;; eval:(put 'sde-set-default           'edebug-form-spec '(form symbol))
;;; eval:(put 'sde-args                  'edebug-form-spec '(&rest form))
;;; eval:(put 'sde-prompt                'edebug-form-spec '(form form))
;;; End:
