;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-compat.el,v $
;;;; $Id: sde-compat.el,v 0.5 1994/05/28 05:34:31 hucka Exp $
;;;; 
;;;; Description       : SDE compatibility functions to handle older versions
;;;;                     of soar-mode and SDE itself.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs 18.58: Copyright (C) 1985-1991 Free Software Foundation, Inc.
;;;; Soar-mode 5.0    Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:      Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.46:       Copyright (C) 1991-1992 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25    Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:     Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu

(defconst sde-compat-el-version "$Revision: 0.5 $"
  "The revision number of sde-compat.el.  The complete RCS id is:
      $Id: sde-compat.el,v 0.5 1994/05/28 05:34:31 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  General internal constants and variables
;;;; 4.  Main code.
;;;; 5.  Closing statements
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; ----------------
;;;; 0. Documentation
;;;; ----------------
;;;;
;;;; Cheap support for existing files that try to use "soar-mode" or
;;;; "production-editor-mode".  In old versions of the Emacs interface,
;;;; "soar-mode" was the name of the mode used for editing files.  In SDE, we
;;;; refer to the editing component as sde-mode.  Worse, an early release
;;;; of SDE defined "Production Editor" as the editing component.  Many
;;;; existing .soar files contain a -*- line at the top that makes Emacs set
;;;; the buffer's mode to either soar-mode or production-editor-mode, which
;;;; no longer exist.
;;;;
;;;; The awful hacks below arrange to automatically fix files that end with .soar
;;;; extensions but try to use soar-mode or production-editor-mode.  


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Requirements

(require 'sde)


;;;-----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;-----------------------------------------------------------------------------


(defvar sde-compat-auto-update t
  "*If non-nil, SDE will automatically convert files that try to use
\"soar-mode\" or \"production-editor\" to use SDE instead.  The files' 
local variables lists or the -*- header lines will be edited to use
\"sde-mode\".")


;;;-----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;-----------------------------------------------------------------------------

(defconst sde-compat-sde-mode-name "sde"
  "Mode name to insert into files' -*- line or their local variables list.")

(defvar sde-compat-file-needs-updating nil
  "Buffer-local variable that is set true if SDE determines that the buffer
needs to have its file local variables or the -*- header line updated.")

(make-variable-buffer-local 'sde-compat-file-needs-updating)


;;;-----------------------------------------------------------------------------
;;; 4.  Main code.
;;;-----------------------------------------------------------------------------

;; Hack to support early SDE design.  The Production Editor doesn't exist
;; anymore; SDE *is* the production editor.

(defun production-editor-mode ()
  "SDE compatibility function to support older versions of SDE.  
This function will get called by files that try to use
\"production-editor-mode\" in their local variables list or the -*- header
line."
  (interactive)
  (cond ((interactive-p)
	 (error "\"production-editor\" no longer exists; please use SDE mode instead."))
	;; Not interactive; assume called in a file's -*- line or similar.
	((and buffer-file-name
	      (string-match "\\.soar[56]?$" (file-name-sans-versions buffer-file-name)))
	 (sde-compat-mark-for-update)
	 (sde-mode))
	(t
	 ;; What did the user do to get us here?
	 (error "\"production-editor\" no longer exists -- do you mean sde-mode?"))))


;; Hack to support files from soar-mode, the Emacs Soar interface prior to SDE.

(defun soar-mode ()
  "SDE compatibility function to support older versions of SDE.  
This function will get called by files that try to use \"soar-mode\" in their
local variables list or the -*- header line."
  (interactive)
  (cond ((interactive-p)
	 (error "\"soar-mode\" no longer exists; please use SDE mode instead."))
	;; Not interactive; assume called in a file's -*- line or similar.
	((and buffer-file-name
	      (string-match "\\.soar[56]?$" (file-name-sans-versions buffer-file-name)))
	 (sde-compat-mark-for-update)
	 (sde-mode))
	(t
	 ;; What did the user do to get us here?
	 (error "\"soar-mode\" no longer exists -- do you mean sde-mode?"))))


(defun sde-compat-mark-for-update ()
  (setq sde-compat-file-needs-updating t)
  (make-local-variable 'write-file-hooks)
  (add-hook 'write-file-hooks 'sde-compat-update-on-write))


;; Based on set-auto-mode from Emacs 18.58 files.el.  Ugly code.

(defun sde-compat-update-on-write ()
  ;; Update file -*- header or local variables list before writing.
  ;; Gets added to a buffer's write-file-hooks.
  ;; Must return nil, so that Emacs runs the other functions on
  ;; write-file-hooks.
  (let ((case-fold-search t)
	beg end mode)
    (save-excursion
      (goto-char (point-min))
      (skip-chars-forward " \t\n")
      (if (and (search-forward "-*-" (save-excursion (end-of-line) (point)) t)
	       (progn
		 (skip-chars-forward " \t")
		 (setq beg (point))
		 (search-forward "-*-" (save-excursion (end-of-line) (point)) t))
	       (progn
		 (forward-char -3)
		 (skip-chars-backward " \t")
		 (setq end (point))
		 (goto-char beg)
		 (if (search-forward ":" end t)
		     (progn
		       (goto-char beg)
		       (if (search-forward "mode:" end t)
			   (progn
			     (skip-chars-forward " \t")
			     (setq beg (point))
			     (if (search-forward ";" end t)
				 (forward-char -1)
				 (goto-char end))
			     (skip-chars-backward " \t")
			     (setq end (point))
			     (setq mode (downcase (buffer-substring beg end)))))))))
	  ;; Found a "-*-" style mode definition.
	  (if (and (assoc mode '(("soar-mode") ("production-editor")
				 ("soar") ("production-editor-mode")))
		   sde-compat-file-needs-updating
		   buffer-file-name
		   (file-writable-p buffer-file-name))
	      (progn
		(delete-region beg end)
		(insert sde-compat-sde-mode-name))))))
  nil)


;;;-----------------------------------------------------------------------------
;;; 4.  Closing statements.
;;;-----------------------------------------------------------------------------

