/* =====================================================================
	       SimTime Routines for simulated real time
	  and automatic maintenance of resource limitations

   These routines provide a simulated real time mechanism to replace
   Mark Weismeyer's EDT, including a few new user-interface functions
   that will allow you to initialize and utilize the system.  They
   have also been extended to include the ability to initiate events
   that may tie up resources for specified durations.
===================================================================== */

#include <strings.h>
#include <sys/time.h>
#include "soar.h"
#include "simtime.h"
#define TIME_ATTRIBUTE     "real-time"

char before_op_name[80]; /* Name of operator present before current DC.	*/
int simtime = 0;	/* Simulated real time in ms			*/
int inittime = 0;	/* Time to initialize to at soar-init		*/
static int lasttime = -1; /* Used to keep track of when to print	*/


/* This determines whether system does anything other than load.	*/
/* The system cannot be switched on without at least one of the oplist	*/
/* or the defaults being specified.					*/
int ST_on = 0;		/* Simulated time system in use.		*/

int ST_print_all = 0;	/* Print time value every DC.			*/
int ST_print_delta = 0;	/* Print time value whenever updated.		*/
int ST_wm_time = 0;	/* Insert the time value into WM.		*/
int ST_elab_def = 0;	/* Default time passed for each EC.		*/
int ST_decis_def = 0;	/* Default time passed for each DC.		*/
int ST_oper_def = 0;	/* Default time passed for each operator.	*/
wme *ST_real_time_WME = NULL;	/* WME for holding real time.		*/
char CPM_logfile[80];	/* Filename for CPM log trace.			*/
FILE *CPM_logfp;	/* FILE pointer for CPM log file.		*/

POpDur opdurlist = NULL;
PSTEvent eventlist = NULL;
PSTEvent nextevent = NULL;
PSTEvent ceventlist = NULL;
hash_table *ST_variable_hash_table;
hash_table *ST_wme_hash_table;

bool user_simtime(void);
void ReadTiming(char *fname);
void ReadEventTimes(char *fname);
void PrintStatus(void);
void PrintTime(void);
void PrintTiming(void);
void PrintEventTimes(void);
void CheckForEvents(void);
char *tilde_expand(char *fname);
void FreeOpList(POpDur list);
void FreeEventList(PSTEvent nextevent);
void UpdateTimeWME(void);
void AddInputWME(char *args);
void RemoveInputWME(char *args);
void ReclaimVariables(void);
void ReclaimWMEs(void);
void TallyNewOps(void);
ST_variable *ST_find_variable (char *name);
Symbol *ST_make_variable (char *name);
ST_wme *ST_find_wme (char *name);
void ST_make_wme (char *name, Symbol *id, Symbol *attr, Symbol *val);
unsigned long hash_ST_variable (void *item, short num_bits);
unsigned long hash_ST_wme_name (void *item, short num_bits);
/* Stolen from symtab.h:						*/
extern unsigned long current_symbol_hash_id;
#define get_next_symbol_hash_id() (current_symbol_hash_id += 137)
unsigned long hash_variable_raw_info (char *name, short num_bits);

/* Adds user-interface commands for simulated time system.		*/
void add_simtime_commands(void) {
  add_command("simtime", user_simtime);
  /* Initialize hash tables for storing SimTime-created inputs.		*/
  ST_variable_hash_table = make_hash_table (0, hash_ST_variable);
  ST_wme_hash_table = make_hash_table (0, hash_ST_wme_name);
  /* I don't think we should use a memory pool, because things won't be */
  /* coming and going that often.					*/ 
} 

/* Initialize simulated time system.					*/
void init_simtime(void) {
  PSTEvent temp;
  struct timeval tv;
  char *date;

  /* Remove real-time WME before top state vanishes. */
  if (top_state && ST_real_time_WME) {
    remove_input_wme(ST_real_time_WME);
    ST_real_time_WME = NIL;
  }

  /* Discard any program-generated events. */
  while (ceventlist) {
    temp = ceventlist;
    ceventlist = ceventlist->next;
    free(temp);
  }

  /* Reset the time and event-stream if info is available. */
  if (eventlist) {
    simtime = inittime;
    nextevent = eventlist;
  }

  /* Empty out the hash tables before trying to init-soar. */
  ReclaimVariables();
  ReclaimWMEs();

  /* Close old logging file. */
  if (CPM_logfp) fclose(CPM_logfp);
  CPM_logfp = NULL;

  /* Don't do the following if switched off. */
  if (!ST_on) return;

  lasttime = simtime-1;

  /* Begin logging again if desired. */
  if (*CPM_logfile) {
    CPM_logfp = fopen(CPM_logfile, "a");
    if (CPM_logfp) {
      /* Get time-of-day string. */
      gettimeofday(&tv, NULL);
      date = ctime(&(tv.tv_sec));
      date[24] = '\0';
      fprintf(CPM_logfp, "%d\tinit-soar %s\n", simtime, date);
    }
  }
}

void ST_before_DC() {
  Symbol *g, *o, *opsym;
  slot *curslot;
  POpDur oplist;

  /* Do nothing if system not switched on. */
  if (!ST_on) return;

  /* Find the lowest goal. */
  for (g = top_goal; g->id.lower_goal != NIL; g = g->id.lower_goal);

  /* If it doesn't have an operator, return. */
  if (g->id.operator_slot == NIL ||
      g->id.operator_slot->wmes == NIL ||
      g->id.operator_slot->wmes->value == NIL) return;
  
  /* Get its operator. */
  o = g->id.operator_slot->wmes->value;

/*  print_with_symbols("\nBottom operator: %y\n", o); */

  /* Get its operator's name. */
  for (curslot = o->id.slots; curslot != NULL; curslot = curslot->next) {
    if (!strcmp(curslot->attr->sc.name,"name")) break;
  }

  if (!curslot) return;

  opsym = curslot->wmes->value;
  /* Pick the first one, if multiple names. 	*/
  switch (opsym->common.symbol_type) {
  case SYM_CONSTANT_SYMBOL_TYPE:
    strcpy(before_op_name, opsym->sc.name);
    break;
  case IDENTIFIER_SYMBOL_TYPE:
    sprintf(before_op_name, "%c%d", opsym->id.name_letter, opsym->id.name_number);
    break;
  default:
    fprintf(stderr, "\nSimTime encountered unexpected operator name type %d.\n", opsym->common.symbol_type);
    strcpy(before_op_name, "UNKNOWN");
    break;
  }
  return;
}

/* Add appropriate time for last operator selected, and time the whole	*/
/* decision cycle regardless of operator change.  This might need to	*/
/* take name of operator into account.					*/ 
void ST_after_DC() {
  /* Ignore if system is not on. */
  if (!ST_on) return;

  simtime += ST_decis_def;
  if (ST_oper_def || opdurlist) TallyNewOps();
  PrintTime();
  UpdateTimeWME();
  if (top_state) CheckForEvents();
}

void ST_after_EC() {
  /* Ignore if system is not on. */
  if (!ST_on) return;

  simtime += ST_elab_def;
  if (ST_print_delta) PrintTime();
  UpdateTimeWME();
  if (top_state) CheckForEvents();
}


void TallyNewOps() {
  POpDur oplist;

#ifdef DEBUG_SIMTIME
  fprintf(stderr,"\nTallyNewOps.");
#endif /* DEBUG_SIMTIME */

  /* If we have not yet encountered a new operator, return. */
  if (*before_op_name == '\0') return;

  /* It is an operator, so write this out in CPM log. */
  if (CPM_logfp) 
    fprintf(CPM_logfp, "%d	operator %s\n", simtime, before_op_name);
  
  /* Add default operator duration */
  simtime += ST_oper_def;

  for (oplist = opdurlist; oplist != NULL; oplist = oplist->next) {
    if (!strcmp(before_op_name, oplist->name)) {

#ifdef DEBUG_SIMTIME
    fprintf(stderr, "got operator %s, duration %dms\n",
	    before_op_name, oplist->duration);
#endif DEBUG_SIMTIME

      simtime += oplist->duration;
    }
  }
  *before_op_name = '\0';
}

/* Check to see if any new events have happened. */
void CheckForEvents(void) {
  int i;
  Symbol *timeval;
  PSTEvent temp;

  /* Iterate through user loaded events, which must be preserved.	*/
  while (nextevent && nextevent->time <= simtime) {

#ifdef DEBUG_SIMTIME
    fprintf(stderr, "\nTime has come for %s %s...",
	    nextevent->type, nextevent->args);
#endif DEBUG_SIMTIME

    if (CPM_logfp) 
      fprintf(CPM_logfp, "%d	event %s\n", simtime, nextevent->type);
  
    for (i=0; st_funcs[i].type; i++) {
      if (!strcmp(st_funcs[i].type, nextevent->type)) {
	(*(st_funcs[i].fun))(nextevent->args);
	break;
      }
    }
    if (st_funcs[i].type == NULL) {
      if (!strcmp(nextevent->type, "ADD-WME"))
	AddInputWME(nextevent->args);
      else if (!strcmp(nextevent->type, "REMOVE-WME"))
	RemoveInputWME(nextevent->args);
      else
	print("\nError: Event type %s not defined in st_funcs.\n",
	      nextevent->type);
    }
    nextevent = nextevent->next;
  }

  /* Iterate through program created events, which must be discarded	*/
  /* as they are used up.						*/
  while (ceventlist && ceventlist->time <= simtime) {

#ifdef DEBUG_SIMTIME
    fprintf(stderr, "\nTime has come for %s %s...",
	    ceventlist->type, ceventlist->args);
#endif /* DEBUG_SIMTIME */

    for (i=0; st_funcs[i].type; i++) {
      if (!strcmp(st_funcs[i].type, ceventlist->type)) {
	(*(st_funcs[i].fun))(ceventlist->args);
	break;
      }
    }
    if (st_funcs[i].type == NULL) {
      if (!strcmp(ceventlist->type, "ADD-WME"))
	AddInputWME(ceventlist->args);
      else if (!strcmp(ceventlist->type, "REMOVE-WME"))
	RemoveInputWME(ceventlist->args);
      else
	print("\nError: Event type %s not defined in st_funcs.\n",
	      ceventlist->type);
    }
    temp = ceventlist;
    ceventlist = ceventlist->next;
    free(temp);
  }
}

void ReadTiming(char *fname) {
  char *expanded_string;
  FILE *fp;
  char opname[80];
  int time;
  POpDur newlist, temp;

  expanded_string = tilde_expand(fname);

  newlist = NULL;
  if ((fp = fopen(expanded_string, "r")) != NULL) {
    ST_elab_def = ST_decis_def = 0;
    while (fscanf(fp, "%s %d\n", opname, &time) != EOF) {
      if (!strcmp(opname, "ELABORATION")) ST_elab_def = time;
      else if (!strcmp(opname, "DECISION")) ST_decis_def = time;
      else if (!strcmp(opname, "OPERATOR")) ST_oper_def = time;
      else {
	temp = newlist;
	newlist = (POpDur) malloc(sizeof(OpDur));
	newlist->name = (char *)malloc(sizeof(char)*(1+strlen(opname)));
	strcpy(newlist->name, opname);
	newlist->duration = time;
	newlist->next = temp;
      }
    }
    if (opdurlist) FreeOpList(opdurlist);
    opdurlist = newlist;
  } else print ("Error: unable to open file %s\n", expanded_string);
  free(expanded_string);
}

void ReadEventTimes(char *fname) {
  char *expanded_string;
  FILE *fp;
  int time, pc;
  char funtype[80], *args, c, *sp;
  PSTEvent newlist, *temp;

  expanded_string = tilde_expand(fname);

  newlist = NULL;
  temp = &newlist;

  if ((fp = fopen(expanded_string, "r")) != NULL) {
    fscanf(fp, "%d\n", &inittime);
    while ((pc = getc(fp)) != EOF) {
      if (pc == '\n') continue;
      if (pc == ';' || pc == '#') {
	do { pc = getc(fp); } while (pc != '\n' && pc != EOF);
	continue;
      } else ungetc(pc, fp);

      fscanf(fp, "%d %s", &time, funtype);
      /* Get argument list. 	*/
      args = fgets_long(fp);
      /* Allocate space for block 	*/
      *temp = (PSTEvent) malloc(sizeof(STEvent));

      /* Copy time 	*/
      (*temp)->time = time;

      /* Allocate space for function type and copy it. 	*/
      (*temp)->type = (char *)malloc(sizeof(char)*(1+strlen(funtype)));
      strcpy((*temp)->type, funtype);

      /* Copy argument string, eliminating blank space at beginning. 	*/
      (*temp)->args = args+1;
      
      /* Update next pointer and set temp to point there for next iteration 	*/
      (*temp)->next = NULL;
      temp = &((*temp)->next);
    }
    if (eventlist) FreeEventList(eventlist);
    nextevent = eventlist = newlist;
  } else print ("Error: unable to open file %s\n", expanded_string);
  free(expanded_string);
  init_simtime();
}

void PrintStatus(void) {
  print("Simulated time system is %s", ST_on?"on":"off");
  print(ST_print_all?", printing time every DC.\n":
	(ST_print_delta?", printing time when changed.\n":".\n"));
  if (ST_wm_time)
    print("Time is being directly inserted into working memory as %s.\n",
	  TIME_ATTRIBUTE);
  print("Initial time is %d, current time is %d.\n", inittime, simtime);
  if (opdurlist && eventlist)
    print("Timing and event lists are currently loaded.\n");
  else if (opdurlist) 
    print("Timing list is currently loaded.\n");
  else if (eventlist)
    print("An event list is currently loaded.\n");
  else print("No timing or event list is currently loaded.\n");
  if (ST_elab_def || ST_decis_def)
    print("Using cycle times of DC = %d, EC = %d.\n",
	  ST_decis_def, ST_elab_def);
}


void PrintTime(void) {
  if (ST_print_all || (ST_print_delta && lasttime != simtime))
    print("\nReal time: %d", simtime);
  lasttime = simtime;
}

void PrintTiming(void) {
  POpDur p;

  if (ST_oper_def)
    print("Base operator duration: %dms\n", ST_oper_def);
  if (ST_decis_def) 
    print("Base decision cycle duration: %dms\n", ST_decis_def);
  if (ST_elab_def)
    print("Base elaboration cycle duration: %dms\n", ST_elab_def);

  if (!opdurlist) {
    print("No operators loaded.\n");
    return;
  }

  for (p = opdurlist; p != NULL; p = p->next) {
    print("%30s %dms\n", p->name, p->duration);
  }
}

void PrintEventTimes(void) {
  PSTEvent p;

  if (!eventlist && !ceventlist) {
    print("No event list loaded.\n");
    return;
  }
  
  for (p = eventlist; p != NULL; p = p->next) {
    print("%6d", p->time);
    if (p == nextevent) print("* "); else print("  ");
    print("%s %s\n", p->type, p->args);
  }

  for (p = ceventlist; p != NULL; p = p->next) {
    print("%6d", p->time);
    if (p == ceventlist) print("* "); else print("  ");
    print("%s %s\n", p->type, p->args);
  }
}

void FreeOpList(POpDur list) {
  if (list == NULL) return;
  FreeOpList(list->next);
  free(list);
}

void FreeEventList(PSTEvent list) {
  if (list == NULL) return;
  FreeEventList(list->next);
  free(list);
}

bool user_simtime(void) {
  char *arg;
  char filetype;

  set_lexer_allow_ids(FALSE);
  get_lexeme();			/* Move past command name. */
  if (lexeme.type == R_PAREN_LEXEME) {
    PrintStatus();
    return TRUE;
  } else if (lexeme.type == SYM_CONSTANT_LEXEME) {
    if (!strcmp(lexeme.string, "off")) {
      if (ST_on) print("Simulated time system switched off.\n");
      else print("Simulated time system already off.\n");
      ST_on = 0;
      get_lexeme();
      return TRUE;
    } else if (!strcmp(lexeme.string, "on"))
      if (opdurlist || ST_elab_def || ST_decis_def || ST_oper_def) {
	if (!ST_on) print("Simulated time system switched on.\n");
	else print("Simulated time system already on.\n");
	ST_on = 1;
	get_lexeme();
	return TRUE;
      } else {
	print ("Cannot turn simulated time system on without operators or defaults.\n");
	print_location_of_most_recent_lexeme();
	return FALSE;
      }
    else if (!strcmp(lexeme.string, "print")) {
      get_lexeme();		/* Move past 'print' lexeme. */
      if (lexeme.type == SYM_CONSTANT_LEXEME) {
	if (!strcmp(lexeme.string, "on") ||
	    !strcmp(lexeme.string, "changes")) {
	  ST_print_delta = TRUE;
	  ST_print_all = FALSE;
	  get_lexeme();
	  return TRUE;
	} else if (!strcmp(lexeme.string, "always")) {
	  ST_print_all = TRUE;
	  get_lexeme();
	  return TRUE;
	} else if (!strcmp(lexeme.string, "off")) {
	  ST_print_delta = ST_print_all = FALSE;
	  get_lexeme();
	  return TRUE;
	}
      }
    }
    else if (!strcmp(lexeme.string, "wme")) {
      get_lexeme();		/* Move past 'wme' lexeme. */
      if (lexeme.type == SYM_CONSTANT_LEXEME) {
	if (!strcmp(lexeme.string, "on")) {
	  ST_wm_time = TRUE;
	  get_lexeme();
	  return TRUE;
	} else if (!strcmp(lexeme.string, "off")) {
	  ST_wm_time = FALSE;
	  get_lexeme();
	  return TRUE;
	}
      }
    } else if (!strcmp(lexeme.string, "set")) {
      get_lexeme();		/* Move past 'set' lexeme. */
      if (lexeme.type == INT_CONSTANT_LEXEME) {
	simtime = lexeme.int_val;
	get_lexeme();
	return TRUE;
      }
    } else if (!strcmp(lexeme.string, "operators") ||
	       !strcmp(lexeme.string, "timing") ||
	       !strcmp(lexeme.string, "events") ||
	       !strcmp(lexeme.string, "cpm")) {
      filetype = lexeme.string[0];

      get_lexeme();	/* Move past 'operators', 'events', or 'cpm' lexeme. */
      if (lexeme.type == QUOTED_STRING_LEXEME) {
	switch(filetype) {
	case 'o':	/* Backwards compatbility */
	case 't':
	  ReadTiming(lexeme.string);
	  break;
	case 'e':
	  ReadEventTimes(lexeme.string);
	  break;
	case 'c':
	  strcpy(CPM_logfile, lexeme.string);
	  break;
	default:
	  return FALSE;
	}
	get_lexeme();
	return TRUE;
      } else if (lexeme.type == R_PAREN_LEXEME) {
	switch(filetype) {
	case 'o':	/* Backwards compatibility */
	case 't':
	  PrintTiming();
	  break;
	case 'e':
	  PrintEventTimes();
	  break;
	case 'c':
	  if (CPM_logfp) fclose(CPM_logfp);
	  *CPM_logfile = '\0';
	  CPM_logfp = NULL;
	  break;
	default:
	  return FALSE;
	}
	return TRUE;
      } else {
	print ("Expected string in quotes for filename\n");
	print_location_of_most_recent_lexeme();
	return FALSE;
      }
    }
  }
  print_location_of_most_recent_lexeme();
  print ("\nExpected 'on', 'off', 'print [always|on|changes|off]', 'wme [on|off]',");
  print ("\n'set <num>', or [events|timing|cpm] and a filename for argument to");
  print ("\n'simtime' command.");
  return FALSE;
}

void UpdateTimeWME(void) {
  Symbol *IC_real_time, *timeval;

  if (ST_wm_time && top_state) {
    if (ST_real_time_WME) remove_input_wme(ST_real_time_WME);
    IC_real_time = get_io_sym_constant(TIME_ATTRIBUTE);
    timeval = get_io_int_constant(simtime);
    ST_real_time_WME = add_input_wme(top_state, IC_real_time, timeval);
  }
}
      

/* Mechanism for recording tied-up resources.				*/
void CPM_schedule_resource(char *resname, int duration) {
  struct timeval tv;
  char *wmename, AIWargs[100];
  PSTEvent newevent, temp;
  int time;

  /* Tally operator that initiated this before recording. */
  TallyNewOps();

  printf("\n%7d	resource %s %d", simtime, resname, duration);

  if (CPM_logfp) 
    fprintf(CPM_logfp, "%d	resource %s %d\n", simtime, resname, duration);
  
  /* Generate unique identifer for WME we are about to create. 	*/
  /* If these do not come out unique, you *will* get core dumps.  Ugh... */
  gettimeofday(&tv, NULL);
  wmename = (char *)malloc(sizeof(char)*18);
  sprintf(wmename, "%10d%06d", tv.tv_sec, tv.tv_usec);
  
  sprintf(AIWargs, "%s top-state resource-in-use %s", wmename, resname);
  AddInputWME(AIWargs);

  /* Create new event. 	*/
  newevent = (PSTEvent) malloc(sizeof(STEvent));

  /* Calculate free-up time 	*/
  newevent->time = simtime+duration;

  /* Allocate space for function type and copy it. 	*/
  newevent->type = (char *)malloc(sizeof(char)*12);
  strcpy(newevent->type, "REMOVE-WME");

  /* Copy pointer to wmename. 	*/
  newevent->args = wmename;
      
  /* Move current pointer in list back if this event is earlier, 	*/
  /* otherwise insert in list in order. 	*/
  if (!ceventlist) {
    ceventlist = newevent;
    newevent->next = NULL;
  } else if (ceventlist->time > newevent->time) {
    newevent->next = ceventlist;
    ceventlist = newevent;
  } else {
    temp = ceventlist;
    while (temp->next && temp->next->time < newevent->time) temp = temp->next;
    newevent->next = temp->next;
    temp->next = newevent;
  }
}
  
/* Default user interface function:  ADD-WME				*/
void AddInputWME(char *args) {
  char *sloc;
  int slen;
  char *wmeid, *id, *attr, *val;
  Symbol *Sid, *Sattr, *Sval;
  char NRid, NRattr, NRval;

#ifdef DEBUG_SIMTIME
  fprintf(stderr, "\nEntering AddInputWME...");
#endif
  sloc = args+strspn(args, " \t");
  slen = strcspn(sloc, " \t");
  wmeid = (char *)malloc(sizeof(char)*(slen+1));
  strncpy(wmeid, sloc, slen); wmeid[slen] = '\0';
  sloc = sloc+slen+strspn(sloc+slen, " \t");
  slen = strcspn(sloc, " \t");
  id = (char *)malloc(sizeof(char)*(slen+1));
  strncpy(id, sloc, slen); id[slen] = '\0';
  sloc = sloc+slen+strspn(sloc+slen, " \t");
  slen = strcspn(sloc, " \t");
  attr = (char *)malloc(sizeof(char)*(slen+1));
  strncpy(attr, sloc, slen); attr[slen] = '\0';
  sloc = sloc+slen+strspn(sloc+slen, " \t");
  slen = strcspn(sloc, " \t\n");
  val = (char *)malloc(sizeof(char)*(slen+1));
  strncpy(val, sloc, slen); val[slen] = '\0';
  if (val[0] == '\0') {
    fprintf(stderr, "\nADD-WME %s: not enough arguments.", args);
    return;
  }
#ifdef DEBUG_SIMTIME
  fprintf(stderr, "\nAdding WME %s: (%s ^%s %s)\n", wmeid, id, attr, val);
#endif
  NRid = NRattr = NRval = 0;
  if (!strcmp(id, "top-state")) Sid = top_state;
  else {
    if (id[0] == '<' && id[strlen(id)-1] == '>') {
      Sid = ST_make_variable(id);
    } else {
      fprintf(stderr, "\n\"%s\" is not a legal id.", id);
      free(id); free(attr); free(val);
      return;
    }
  }

  if (attr[0] == '<' || attr[strlen(attr)-1] == '>') {
    fprintf(stderr, "\n\"%s\" is not a legal attribute.", attr);
    free(id); free(attr); free(val);
    if (NRid) release_io_symbol(Sid);
    return;
  } else {
    slen = strspn(attr, "0123456789");
    /* Currently doesn't handle floats. 	*/
    if (slen == strlen(attr)) {
      Sattr = get_io_int_constant(atoi(attr));
      NRattr = 1;
    } else {
      Sattr = get_io_sym_constant(attr);
      NRattr = 1;
    }
  }

  if (val[0] == '<' || val[strlen(val)-1] == '>') {
    Sval = ST_make_variable(val);
  } else {
    slen = strspn(val, "0123456789");
    /* Currently doesn't handle floats. 	*/
    if (slen == strlen(val)) {
      Sval = get_io_int_constant(atoi(val));
      NRval = 1;
    } else {
      Sval = get_io_sym_constant(val);
      NRval = 1;
    }
  }
  
  ST_make_wme(wmeid, Sid, Sattr, Sval);

  free(id); free(attr); free(val);
  if (NRid) release_io_symbol(Sid);
  if (NRattr) release_io_symbol(Sattr);
  if (NRval) release_io_symbol(Sval);
}

/* Default user interface function:  REMOVE-WME				*/
void RemoveInputWME(char *args) {
  char *sloc;
  int slen;
  char *wmeid;
  ST_wme *elt;

#ifdef DEBUG_SIMTIME
  fprintf(stderr, "\nEntering RemoveInputWME...");
#endif
  sloc = args+strspn(args, " \t");
  slen = strcspn(sloc, " \t\n");
  wmeid = (char *)malloc(sizeof(char)*(slen+1));
  strncpy(wmeid, sloc, slen); wmeid[slen] = '\0';

  if (wmeid[0] == '\0') {
    fprintf(stderr, "\nREMOVE-WME %s: not enough arguments.", args);
    return;
  }
#ifdef DEBUG_SIMTIME
  fprintf(stderr, "\nRemoving WME %s.", wmeid);
#endif
  
  if ((elt = ST_find_wme(wmeid)) != NIL) {
    remove_input_wme(elt->value);
    remove_from_hash_table(ST_wme_hash_table, elt);
    free_memory_block_for_string(elt->name);
  } else 
    fprintf(stderr, "\nNo WME %s was present to be removed.", wmeid);
}

ST_variable *ST_find_variable (char *name) {
  unsigned long hash_value;
  ST_variable *var;

  hash_value = hash_variable_raw_info(name, ST_variable_hash_table->log2size);
  var = (ST_variable *) (*(ST_variable_hash_table->buckets + hash_value));
  for ( ; var != NIL; var = var->next) {
    if (!strcmp(var->name, name)) return var;
  }
  return NIL;
}

Symbol *ST_make_variable (char *name) {
  ST_variable *var;

  if ((var = ST_find_variable(name)) == NIL) {
    var = (ST_variable *) malloc(sizeof(ST_variable));
    var->name = make_memory_block_for_string(name);
    var->value = get_new_io_identifier((name[1] > 96)?(name[1]-32):name[1]);
    add_to_hash_table(ST_variable_hash_table, var);
  }
  return var->value;
}

ST_wme *ST_find_wme (char *name) {
  unsigned long hash_value;
  ST_wme *elt;

  hash_value = hash_variable_raw_info(name, ST_wme_hash_table->log2size);
  elt = (ST_wme *) (*(ST_wme_hash_table->buckets + hash_value));
  for ( ; elt != NIL; elt = elt->next) {
    if (!strcmp(elt->name, name)) return elt;
  }
  return NIL;
}

void ST_make_wme (char *name, Symbol *id, Symbol *attr, Symbol *val) {
  ST_wme *elt;

  if ((elt = ST_find_wme(name)) == NIL) {
    elt = (ST_wme *) malloc(sizeof(ST_wme));
    elt->name = make_memory_block_for_string(name);
    elt->value = add_input_wme(id, attr, val);
    add_to_hash_table(ST_wme_hash_table, elt);
  } else {
    fprintf(stderr, "\nError -- the WME %s already exists.", name);
  }
}

#define BLOCKSIZE 80
char *fgets_long(FILE *fp) {
  char *str, *p, *temp;
  int len, c;

  /* Get initial amount of space of specified block size. 	*/
  len = BLOCKSIZE;
  p = str = (char *)malloc(len);

  /* Until we explictly return because we got a newline or an EOF... 	*/
  while (1) {

    /* Read in characters until we fill up our current block. 	*/
    for (; p-str < len; p++) {

      /* Using an int here so that we can test for EOF 	*/
      c = getc(fp);
      if (c == EOF && p == str) {
	free(str);
	return NULL;
      }

      /* In either of these cases, just write the correct string 	*/
      /* terminator and return the string. 	*/
      if (c == '\n' || c == EOF) {
        *p = '\0';
	return realloc(str, sizeof(char)*(p-str+1));
      } else *p = c;
    }

    /* We've run out of space; increase the amount and try to realloc 	*/
    len += BLOCKSIZE;

    /* If realloc() fails, generate an error message and exit.  If it 	*/
    /* succeeds, we may need to relocate the pointers p and str. 	*/
    if ((temp = (char *)realloc(str, sizeof(char)*len)) == NULL) {
      fprintf(stderr, 
	      "fgets_long: failed to get block of memory!  Exiting.\n");
      exit(-1);
    } else {
      p = temp + (p - str);
      str = temp;
    }
  }
}

unsigned long hash_ST_variable (void *item, short num_bits) {
  ST_variable *var;
  var = item;
  return compress (hash_string(var->name),num_bits);
}

unsigned long hash_ST_wme_name (void *item, short num_bits) {
  ST_wme *wme;
  wme = item;
  return compress (hash_string(wme->name), num_bits);
}

void ReclaimVariables (void) {
  unsigned long hash_value;
  ST_variable *item;
  hash_table *ht;

#ifdef DEBUG_SIMTIME
  fprintf(stderr,"\nReclaiming variables:");
#endif DEBUG_SIMTIME
  ht = ST_variable_hash_table;
  if (ht->count == 0) return;
  
  for (hash_value=0; hash_value < ht->size; hash_value++) {
    item = (ST_variable *) (*(ht->buckets + hash_value));
    while (item != NIL) {
#ifdef DEBUG_SIMTIME
  fprintf(stderr,"\n\t%s", item->name);
#endif DEBUG_SIMTIME
      free_memory_block_for_string(item->name);
      release_io_symbol(item->value);
      item = item->next;
      free(item);
    }
    *(ht->buckets + hash_value) = NIL;
  }
  free_memory(ht->buckets, HASH_TABLE_MEM_USAGE);
  ht->count = 0;
  ht->log2size = ht->minimum_log2size;
  ht->size = (((unsigned long)1) << (ht->log2size));
  ht->buckets = allocate_memory_and_zerofill (ht->size * sizeof(char *),
                                              HASH_TABLE_MEM_USAGE);

}
  
void ReclaimWMEs (void) {
  unsigned long hash_value;
  ST_wme *item;
  hash_table *ht;

#ifdef DEBUG_SIMTIME
  fprintf(stderr,"\nReclaiming WMEs:");
#endif DEBUG_SIMTIME
  ht = ST_wme_hash_table;
  if (ht->count == 0) return;
  
  for (hash_value=0; hash_value < ht->size; hash_value++) {
    item = (ST_wme *) (*(ht->buckets + hash_value));
    while (item != NIL) {
#ifdef DEBUG_SIMTIME
  fprintf(stderr,"\n\t%s", item->name);
#endif DEBUG_SIMTIME
      free_memory_block_for_string(item->name);
      remove_input_wme(item->value);
      item = item->next;
      free(item);
    }
    *(ht->buckets + hash_value) = NIL;
  }
  free_memory(ht->buckets, HASH_TABLE_MEM_USAGE);
  ht->count = 0;
  ht->log2size = ht->minimum_log2size;
  ht->size = (((unsigned long)1) << (ht->log2size));
  ht->buckets = allocate_memory_and_zerofill (ht->size * sizeof(char *),
                                              HASH_TABLE_MEM_USAGE);
}

bool print_var_ent(void *item) {
  ST_variable *var;
  var = (ST_variable *) item;

  print("Name: %s\t", var->name);
  print_with_symbols("Value: %y\n", var->value);
}

bool print_wme_ent(void *item) {
  ST_wme *elt;
  elt = (ST_wme *) item;

  print("Name: %s\tValue: ", elt->name);
  print_wme(elt->value);
  print("\n");
}

void DumpHTs(void) {
  do_for_all_items_in_hash_table (ST_variable_hash_table, print_var_ent);
  do_for_all_items_in_hash_table (ST_wme_hash_table, print_wme_ent);
}

