/* -----------------------------------------------------------------------
    Switch flipping Soar I/O code, converted to C

    USAGE NOTES:

    Insert the following line into function system_startup_hook()
    in file hooks.c:

       init_flip_io_stuff()

------------------------------------------------------------------------ */

#include "soar.h"

/* -----------------------------------------------------------------------
   Symbols we'll need a lot a run time.  Rather than call
   get_io_sym_constant() a lot a run time, we'll create them once at system
   startup time and use them forever. 
------------------------------------------------------------------------ */

symbol *input_link_sym,
  *object_sym,
  *command_sym,
  *flip_switch_sym,
  *arg1_sym, *arg2_sym,
  *position_sym, *down_sym, *up_sym, *status_sym, *on_sym, *off_sym;

/* -----------------------------------------------------------------------
                             Global Variables

   1.  Variables recording the current (simulated) world status.
   2.  Pointers to certain identifiers & wmes in the current input structure
       that we need to update the input structure later.
------------------------------------------------------------------------ */

bool switch_status = FALSE;  /* TRUE iff switch is up */
bool bulb_status = FALSE;    /* TRUE iff bulb is lit */

bool prev_switch_status = FALSE;  /* need these so the input function can
                                     tell when something has changed */
bool prev_bulb_status = FALSE;


symbol *top_input_link_id;  /* identifier of top input link */
symbol *switch_ob_id;       /* identifier of the switch OBJECT */
symbol *bulb_ob_id;         /* identifier of the bulb OBJECT */
wme *top_input_link_wme;    /* wme: (S1 ^input-link I37) */
wme *switch_position_wme;   /* wme: (S37 ^position up/down) */
wme *bulb_status_wme;       /* wme: (B29 ^status on/off) */

/* -----------------------------------------------------------------------
                     Create initial input structures   

   This function creates the initial input structures off the top state.
   It sets top_input_link_id, switch_ob_id, bulb_ob_id, top_input_link_wme,
   switch_position_wme, and bulb_status_wme for later use.
----------------------------------------------------------------------- */

void create_initial_input_structures (void) {
  symbol *name_sym, *isa_sym, *switch1_sym, *switch_sym;
  symbol *bulb_sym, *bulb1_sym, *associated_bulb_sym, *associated_switch_sym;

  /* --- create symbols that we'll use below --- */
  name_sym = get_io_sym_constant ("name");
  isa_sym = get_io_sym_constant ("isa");
  switch1_sym = get_io_sym_constant ("switch1");
  switch_sym = get_io_sym_constant ("switch");
  bulb_sym = get_io_sym_constant ("bulb");
  bulb1_sym = get_io_sym_constant ("bulb1");
  associated_switch_sym = get_io_sym_constant ("assocated-switch");
  associated_bulb_sym = get_io_sym_constant ("assocated-bulb");

  /* --- create the top input link --- */
  top_input_link_id = get_new_io_identifier ('S');
  top_input_link_wme = add_input_wme (top_state,
                                      input_link_sym,
                                      top_input_link_id);
  
  /* --- create first object off that link, for the switch --- */
  switch_ob_id = get_new_io_identifier ('S');
  bulb_ob_id = get_new_io_identifier ('B');

  add_input_wme (top_input_link_id, object_sym, switch_ob_id);
  add_input_wme (switch_ob_id, name_sym, switch1_sym);
  add_input_wme (switch_ob_id, isa_sym, switch_sym);
  switch_position_wme = add_input_wme (switch_ob_id, position_sym, down_sym);
  add_input_wme (switch_ob_id, associated_bulb_sym, bulb_ob_id);

  /* --- create second object off that link, for the bulb --- */
  add_input_wme (top_input_link_id, object_sym, bulb_ob_id);
  add_input_wme (bulb_ob_id, name_sym, bulb1_sym);
  add_input_wme (bulb_ob_id, isa_sym, bulb_sym);
  bulb_status_wme = add_input_wme (bulb_ob_id, status_sym, off_sym);
  add_input_wme (bulb_ob_id, associated_switch_sym, switch_ob_id);  

  /* --- release the temporary symbols now that we're done --- */
  release_io_symbol (name_sym);
  release_io_symbol (isa_sym);
  release_io_symbol (switch1_sym);
  release_io_symbol (switch_sym);
  release_io_symbol (bulb_sym);
  release_io_symbol (bulb1_sym);
  release_io_symbol (associated_switch_sym);
  release_io_symbol (associated_bulb_sym);
}

/* -----------------------------------------------------------------------
                       Update input structures   

   Updates the input structures if anything has changed.   (The only things
   that can change are the bulb status and the switch position.)
----------------------------------------------------------------------- */

void update_input_structures (void) {
  if (bulb_status!=prev_bulb_status) {
    remove_input_wme (bulb_status_wme);
    bulb_status_wme = add_input_wme (bulb_ob_id,
                                     status_sym,
                                     bulb_status ? on_sym : off_sym);
  }
  if (switch_status!=prev_switch_status) {
    remove_input_wme (switch_position_wme);
    switch_position_wme = add_input_wme (switch_ob_id,
                                         position_sym,
                                         switch_status ? up_sym : down_sym);
  }
  bulb_status = prev_bulb_status;
  switch_status = prev_switch_status;
}

/* -----------------------------------------------------------------------

                         The Input Function

----------------------------------------------------------------------- */

void my_input_function (int mode) {
  switch (mode) {
  case TOP_STATE_JUST_CREATED:
    create_initial_input_structures();
    break;
  case TOP_STATE_JUST_REMOVED:
    release_io_symbol (top_input_link_id);
    release_io_symbol (bulb_ob_id);
    release_io_symbol (switch_ob_id);
    break;
  case NORMAL_INPUT_CYCLE:
    update_input_structures ();
    break;
  }
}

/* -----------------------------------------------------------------------

                   The Output Function (and helpers)

----------------------------------------------------------------------- */

/* ---------
   snarfed from database.c...

    Scans through the io_wme chain looking for a wme whose id/attr match
    the given id/attr.  Either id or attr or both may be given as NIL,
    meaning "don't care".  Returns the value from the wme if a matching
    wme is found, or NIL if no match is found.
---------- */
extern symbol *get_output_value (io_wme *outputs, symbol *id, symbol *attr);


void do_flip_simulator_switch (io_wme *outputs, symbol *link_id) {
  symbol *object_id, *position;

  object_id = get_output_value (outputs, link_id, arg1_sym);
  /* ignore object_id, as there's only one possible switch */
  position = get_output_value (outputs, link_id, arg2_sym);
  if (position == up_sym) {
    switch_status = TRUE;
    bulb_status = TRUE;
  } else {
    switch_status = FALSE;
    bulb_status = FALSE;
  }
}

void my_output_function (int mode, io_wme *outputs) {
  symbol *output_link_id, *command_name;
  
  if (mode!=ADDED_OUTPUT_COMMAND) return;
  output_link_id = get_output_value (outputs, top_state, NIL);
  if (!output_link_id) return;
  command_name = get_output_value (outputs, output_link_id, command_sym);
  if (!command_name) return;
  if (command_name == flip_switch_sym)
    do_flip_simulator_switch (outputs, output_link_id);
  else
    print ("\nI don't know that command!!\n");
}

/* -----------------------------------------------------------------------

                       Initialization Code

----------------------------------------------------------------------- */

void init_flip_io_stuff (void) {
  /* --- tell the system about my I/O functions --- */
  add_input_function (my_input_function);
  add_output_function ("output-link", my_output_function);
  /* --- create symbols that we'll need to use later (might as well do
     this now at system startup time, so as to avoid doing it later) --- */
  input_link_sym = get_io_sym_constant ("input-link");
  object_sym = get_io_sym_constant ("object");
  command_sym = get_io_sym_constant ("command");
  flip_switch_sym = get_io_sym_constant ("flip-switch");
  arg1_sym = get_io_sym_constant ("arg1");
  arg2_sym = get_io_sym_constant ("arg2");

  position_sym = get_io_sym_constant ("position");
  down_sym = get_io_sym_constant ("down");
  up_sym = get_io_sym_constant ("up");
  status_sym = get_io_sym_constant ("status");
  on_sym = get_io_sym_constant ("on");
  off_sym = get_io_sym_constant ("off");
}
