;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-search.el,v $
;;;; $Id: sde-search.el,v 0.5 1994/06/15 20:30:36 hucka Exp $
;;;; 
;;;; Description       : String search and replace across tasks.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-search-el-version "$Revision: 0.5 $"
  "The revision number of sde-search.el.  The complete RCS id is:
      $Id: sde-search.el,v 0.5 1994/06/15 20:30:36 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Requirements and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables.
;;;; 4.  Main code.
;;;; 5.  Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; ----------------
;;; 0. Documentation
;;; ----------------
;;;
;;; This file contains functions for performing string-oriented search and
;;; replace operations.  It is mostly based on etags.el.  It does not require
;;; parsing productions -- this is entirely string-based and is orthogonal to
;;; the `sde-find-xxx' commands.  This means it will not force a scan of a
;;; task, unless there is no .sde file for the task.


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

(require 'sde-next-match)

;; Provide is at the end.


;;;----------------------------------------------------------------------------
;;; 4.  Main code.
;;;----------------------------------------------------------------------------

(defun sde-search-read-args (prompt)
  (let (string tdata)
    (setq string (read-from-minibuffer (format "%s: " prompt)
				       nil nil nil 'query-replace-history))
    (setq tdata (if current-prefix-arg
		    (sde-get-file-task (sde-prompt-for-load-file))
		  (sde-task)))
    (list string tdata)))


(defun sde-search (string tdata)
  "Search for STRING through all the files of a task.

When called interactively, this command prompts you for a search string.  If
given a prefix argument, it also prompts for the task in which to search\;
otherwise, it uses the task with which the current buffer is associated.  It
then searches in the files of the task and stops when a match is found.  To
continue searching for the next match, use `\\[sde-next-match]'.  Files are searched
in alphabetical order unless the variable `sde-sort-lists' is `nil'."
  (interactive (sde-search-read-args "Search"))
  (if (and (equal string "")
	   (eq (car sde-next-match-scan) 'search-forward)
	   (null sde-next-match-operate))
      ;; Continue last search.
      (sde-next-match nil)
    ;; Start new search.
    (setq sde-next-match-task tdata)
    (setq sde-next-match-scan (list 'search-forward string nil t))
    (setq sde-next-match-operate nil)
    (sde-next-match t)))

(defun sde-search-regexp (regexp tdata)
  "Search for REGEXP through all the files of a task.

When called interactively, this command prompts you for a string which can be
a regular expression.  If given a prefix argument, it also prompts for the
task in which to search\; otherwise, it uses the task with which the current
buffer is associated.  It then searches in the files of the task and stops
when a match is found.  To continue searching for the next match, use
`\\[sde-next-match]'.  Files are searched in alphabetical order unless the
variable `sde-sort-lists' is `nil'.

See also the command `sde-search'."
  (interactive (sde-search-read-args "Search (regexp)"))
  (if (and (equal regexp "")
	   (eq (car sde-next-match-scan) 're-search-forward)
	   (null sde-next-match-operate))
      ;; Continue last search.
      (sde-next-match nil)
    ;; Start new search.
    (setq sde-next-match-task tdata)
    (setq sde-next-match-scan (list 're-search-forward regexp nil t))
    (setq sde-next-match-operate nil)
    (sde-next-match t)))

(defun sde-query-replace-read-args (prompt)
  (let (from to tdata)
    (setq from (read-from-minibuffer (format "%s: " prompt)
				     nil nil nil 'query-replace-history))
    (setq to (read-from-minibuffer (format "%s %s with: " prompt from)
				   nil nil nil 'query-replace-history))
    (setq tdata (if current-prefix-arg
		    (sde-get-file-task (sde-prompt-for-load-file))
		  (sde-task)))
    (list from to tdata)))

(defun sde-replace-string (from to tdata)
  "Noninteractively perform `replace-string' throughout the files of a task.

When invoked, this command prompts you for a string to search for and a
string with which to replace it.  If given a prefix argument, it also prompts
for the task in which to search\; otherwise, it uses the task with which the
current buffer is associated.  It then performs a noninteractive
`replace-string' in each file of the task.  Files are searched in alphabetical
order unless the variable `sde-sort-lists' is `nil'.

This command preserves case in each replacement if the variables
`case-replace' and `case-fold-search' are non-nil, and the FROM string has no
uppercase letters.

See also the command `sde-replace-regexp'."
  (interactive (sde-query-replace-read-args "Replace string"))
  (setq sde-next-match-task tdata)
  (setq sde-next-match-scan
	(list 'prog1
	      (list 'if (list 'search-forward from nil t)
		    ;; When we find a match, move back to the
		    ;; beginning of it so perform-replace will
		    ;; see it.
		    '(goto-char (match-beginning 0)))))
  (setq sde-next-match-operate
	(list 'perform-replace from to nil nil nil))
  (sde-next-match t)
  (or unread-command-events (message "Done")))

(defun sde-replace-regexp (regexp to-string tdata)
  "Noninteractively perform `replace-regexp' throughout the files of a task.

When invoked, this command prompts you for a regular expression to search for
and a string with which to replace it.  If given a prefix argument, it also
prompts for the task in which to search\; otherwise, it uses the task with
which the current buffer is associated.  It then performs a noninteractive
`replace-regexp' in each file of the task.  Files are searched in alphabetical
order unless the variable `sde-sort-lists' is `nil'.

In TO-STRING, `\\&' stands for whatever matched the whole of REGEXP, and
`\\=\\N' (where N is a digit) stands for whatever what matched the Nth
`\\(...\\)' in REGEXP.

This command preserves case in each replacement if the variables
`case-replace' and `case-fold-search' are non-nil, and the REGEXP has no
uppercase letters.

See also the command `sde-replace-string'."
  (interactive (sde-query-replace-read-args "Replace regexp"))
  (setq sde-next-match-task tdata)
  (setq sde-next-match-scan
	(list 'prog1
	      (list 'if (list 're-search-forward regexp nil t)
		    ;; When we find a match, move back to the
		    ;; beginning of it so perform-replace will
		    ;; see it.
		    '(goto-char (match-beginning 0)))))
  (setq sde-next-match-operate
	(list 'perform-replace regexp to-string nil t nil))
  (sde-next-match t)
  (or unread-command-events (message "Done")))

;; The following is a hacked version of `perform-replace' from replace.el in
;; Emacs 19.24.  This is gross, but the way `perform-replace' is written
;; prevents any other solutions as far as I can see.  The problem is that if
;; the user is performing an `sde-replace-regexp' across multiple files and
;; they type `!', they really want to have all remaining instances across all
;; remaining files to be replaced.  What `tags-replace' does is only replace
;; all remaining instances in the current file, but then will start prompting
;; again in the next file.  The standard version of `perform-replace' does
;; not allow a way of fixing this, since it completely encapsulates the
;; user's response.
;;
;; The only difference between this and the original version is that the
;; global variable `sde-perform-replace-query-flag' is set `t' or `nil'
;; depending on whether the user hit the `!' key during the course of a
;; replacement session.

(defvar sde-perform-replace-query-flag t)

(defun sde-perform-replace (from-string replacements
                            query-flag regexp-flag delimited-flag
			    &optional repeat-count map)
  "Subroutine of `query-replace'.  Its complexity handles interactive queries.
Don't use this in your own program unless you want to query and set the mark
just as `query-replace' does.  Instead, write a simple loop like this:
  (while (re-search-forward \"foo[ \t]+bar\" nil t)
    (replace-match \"foobar\" nil nil))
which will run faster and probably do exactly what you want."
  (or map (setq map query-replace-map))
  (let ((nocasify (not (and case-fold-search case-replace
			    (string-equal from-string
					  (downcase from-string)))))
	(literal (not regexp-flag))
	(search-function (if regexp-flag 're-search-forward 'search-forward))
	(search-string from-string)
	(real-match-data nil)		; the match data for the current match
	(next-replacement nil)
	(replacement-index 0)
	(keep-going t)
	(stack nil)
	(next-rotate-count 0)
	(replace-count 0)
	(lastrepl nil)			;Position after last match considered.
	(match-again t)
	(message
	 (if query-flag
	     (substitute-command-keys
	      "Query replacing %s with %s: (\\<query-replace-map>\\[help] for help) "))))
    (if (stringp replacements)
	(setq next-replacement replacements)
      (or repeat-count (setq repeat-count 1)))
    (if delimited-flag
	(setq search-function 're-search-forward
	      search-string (concat "\\b"
				    (if regexp-flag from-string
				      (regexp-quote from-string))
				    "\\b")))
    (push-mark)
    (undo-boundary)
    (unwind-protect
	;; Loop finding occurrences that perhaps should be replaced.
	(while (and keep-going
		    (not (eobp))
		    (funcall search-function search-string nil t)
		    ;; If the search string matches immediately after
		    ;; the previous match, but it did not match there
		    ;; before the replacement was done, ignore the match.
		    (if (or (eq lastrepl (point))
			    (and regexp-flag
				 (eq lastrepl (match-beginning 0))
				 (not match-again)))
			(if (eobp)
			    nil
			  ;; Don't replace the null string 
			  ;; right after end of previous replacement.
			  (forward-char 1)
			  (funcall search-function search-string nil t))
		      t))

	  ;; Save the data associated with the real match.
	  (setq real-match-data (match-data))

	  ;; Before we make the replacement, decide whether the search string
	  ;; can match again just after this match.
	  (if regexp-flag
	      (setq match-again (looking-at search-string)))
	  ;; If time for a change, advance to next replacement string.
	  (if (and (listp replacements)
		   (= next-rotate-count replace-count))
	      (progn
		(setq next-rotate-count
		      (+ next-rotate-count repeat-count))
		(setq next-replacement (nth replacement-index replacements))
		(setq replacement-index (% (1+ replacement-index) (length replacements)))))
	  (if (not query-flag)
	      (progn
		(store-match-data real-match-data)
		(replace-match next-replacement nocasify literal)
		(setq replace-count (1+ replace-count)))
	    (undo-boundary)
	    (let (done replaced key def)
	      ;; Loop reading commands until one of them sets done,
	      ;; which means it has finished handling this occurrence.
	      (while (not done)
		(replace-highlight (match-beginning 0) (match-end 0))
		(message message from-string next-replacement)
		(setq key (read-event))
		(setq key (vector key))
		(setq def (lookup-key map key))
		;; Restore the match data while we process the command.
		(store-match-data real-match-data)
		(cond ((eq def 'help)
		       (with-output-to-temp-buffer "*Help*"
			 (princ
			  (concat "Query replacing "
				  (if regexp-flag "regexp " "")
				  from-string " with "
				  next-replacement ".\n\n"
				  (substitute-command-keys
				   query-replace-help)))))
		      ((eq def 'exit)
		       (setq keep-going nil)
		       (setq done t))
		      ((eq def 'backup)
		       (if stack
			   (let ((elt (car stack)))
			     (goto-char (car elt))
			     (setq replaced (eq t (cdr elt)))
			     (or replaced
				 (store-match-data (cdr elt)))
			     (setq stack (cdr stack)))
			 (message "No previous match")
			 (ding 'no-terminate)
			 (sit-for 1)))
		      ((eq def 'act)
		       (or replaced
			   (replace-match next-replacement nocasify literal))
		       (setq done t replaced t))
		      ((eq def 'act-and-exit)
		       (or replaced
			   (replace-match next-replacement nocasify literal))
		       (setq keep-going nil)
		       (setq done t replaced t))
		      ((eq def 'act-and-show)
		       (if (not replaced)
			   (progn
			     (replace-match next-replacement nocasify literal)
			     (setq replaced t))))
		      ((eq def 'automatic)
		       (or replaced
			   (replace-match next-replacement nocasify literal))
		       (setq done t query-flag nil replaced t)
		       (setq sde-perform-replace-query-flag nil))
		      ((eq def 'skip)
		       (setq done t))
		      ((eq def 'recenter)
		       (recenter nil))
		      ((eq def 'edit)
		       (store-match-data
			(prog1 (match-data)
			  (save-excursion (recursive-edit))))
		       ;; Before we make the replacement,
		       ;; decide whether the search string
		       ;; can match again just after this match.
		       (if regexp-flag
			   (setq match-again (looking-at search-string))))
		      ((eq def 'delete-and-edit)
		       (delete-region (match-beginning 0) (match-end 0))
		       (store-match-data
			(prog1 (match-data)
			  (save-excursion (recursive-edit))))
		       (setq replaced t))
		      (t
		       (setq keep-going nil)
		       (setq unread-command-events
			     (append (listify-key-sequence key)
				     unread-command-events))
		       (setq done t))))
	      ;; Record previous position for ^ when we move on.
	      ;; Change markers to numbers in the match data
	      ;; since lots of markers slow down editing.
	      (setq stack
		    (cons (cons (point)
				(or replaced
				    (mapcar (lambda (elt)
					      (and elt
						   (prog1 (marker-position elt)
						     (set-marker elt nil))))
				     (match-data))))
			  stack))
	      (if replaced (setq replace-count (1+ replace-count)))))
	  (setq lastrepl (point)))
      (replace-dehighlight))
  (and keep-going (or stack (not query-flag)))))

(defun sde-query-replace (from to tdata)
  "Query-replace through all the files of a task the string FROM with TO.

When invoked, this command prompts you for a string to search for and a
string with which to replace it.  If given a prefix argument, it also prompts
for the task in which to search\; otherwise, it uses the task with which the
current buffer is associated.  It then performs a `query-replace' in each
file of the task.  Files are searched in alphabetical order unless the variable
`sde-sort-lists' is `nil'.

As each match is found, you must type a character saying what to do with it.
For directions, type `\\[help-command]' at that time.  If you exit the query-replace
\(using `\\[keyboard-quit]' or ESC), you can resume the query-replace with the
command `\\[sde-next-match]'.

This command preserves case in each replacement if the variables
`case-replace' and `case-fold-search' are non-nil, and the REGEXP has no
uppercase letters.

This command is similar to `tags-query-replace'.  See also the command
`sde-query-replace-regexp'."
  (interactive (sde-query-replace-read-args "Query replace"))
  (setq sde-next-match-task tdata)
  (setq sde-next-match-scan
	(list 'prog1
	      (list 'if (list 'search-forward from nil t)
		    ;; When we find a match, move back to the
		    ;; beginning of it so perform-replace will
		    ;; see it.
		    '(goto-char (match-beginning 0)))))
  (setq sde-perform-replace-query-flag t)
  (setq sde-next-match-operate
	(` (let ((query-replace-highlight sde-query-replace-highlight))
	     (sde-perform-replace (, from) (, to)
				  sde-perform-replace-query-flag nil nil))))
  (sde-next-match t)
  (or unread-command-events (message "Done")))

(defun sde-query-replace-regexp (regexp to-string tdata)
  "Query-replace-regexp through the files of a task.

When invoked, this command prompts you for a regular expression to search for
and a string with which to replace it.  If given a prefix argument, it also
prompts for the task in which to search\; otherwise, it uses the task with
which the current buffer is associated.  It then performs a
`query-replace-regexp' in each file of the task.  Files are searched in
alphabetical order unless the variable `sde-sort-lists' is `nil'.

As each match is found, you must type a character saying what to do with it.
For directions, type `\\[help-command]' at that time.  If you exit the query-replace
\(using `\\[keyboard-quit]' or ESC), you can resume the query-replace with the
command `\\[sde-next-match]'.

In the TO-STRING string, `\\&' stands for whatever matched the whole of REGEXP,
and `\\=\\N' (where N is a digit) stands for whatever what matched
the Nth `\\(...\\)' in REGEXP.

This command preserves case in each replacement if the variables
`case-replace' and `case-fold-search' are non-nil, and the REGEXP has no
uppercase letters.

This command is analogous to `tags-query-replace'.  See also the command
`sde-query-replace'."
  (interactive (sde-query-replace-read-args "Query replace regexp"))
  (setq sde-next-match-task tdata)
  (setq sde-next-match-scan
	(list 'prog1
	      (list 'if (list 're-search-forward regexp nil t)
		    ;; When we find a match, move back to the
		    ;; beginning of it so perform-replace will
		    ;; see it.
		    '(goto-char (match-beginning 0)))))
  (setq sde-perform-replace-query-flag t)
  (setq sde-next-match-operate
	(` (let ((query-replace-highlight sde-query-replace-highlight))
	     (sde-perform-replace (, regexp) (, to-string)
				  sde-perform-replace-query-flag t nil))))
  (sde-next-match t)
  (or unread-command-events (message "Done")))


;;;-----------------------------------------------------------------------------
;;; 5.  Closing statements.
;;;-----------------------------------------------------------------------------

(provide 'sde-search)
