;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-next-match.el,v $
;;;; $Id: sde-next-match.el,v 0.2 1994/06/23 20:05:48 hucka Exp $
;;;; 
;;;; Description       : Support for continuing searches & find operations.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-next-match-el-version "$Revision: 0.2 $"
  "The revision number of sde-next-match.el.  The complete RCS id is:
      $Id: sde-next-match.el,v 0.2 1994/06/23 20:05:48 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation.
;;;; 1.  Internal constants and variables.
;;;; 2.  Implementation of sde-next-match and related code.
;;;; 3.  Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; -----------------
;;; 0. Documentation.
;;; -----------------
;;;
;;; This file contains code for implementing commands for continuing searches
;;; and find operations.  The code is based on the `tags-loop-continue'
;;; facility in etags.el of Emacs 19.24.


;;;----------------------------------------------------------------------------
;;; 1.  Internal constants and variables.
;;;----------------------------------------------------------------------------

(defvar sde-next-match-initialized nil
  "Flag that is non-`nil' after the first time a search or find is executed.")

(defvar sde-next-match-operate nil
  "Form for `sde-next-match' to eval to change one file.")

(defvar sde-next-match-scan
  '(error (substitute-command-keys
	   "No find or search operation in progress."))
  "Form for `sde-next-match' to eval to scan one file.
If it returns non-nil, this file needs processing by evalling
\`sde-next-match-operate'.  Otherwise, move on to the next file.")

(defvar sde-next-match-task nil
  "Task being searched currently by `sde-next-match'.")

(defvar sde-next-match-file-list nil
  "List of files for \\[sde-next-match-file] to process.")

(defvar sde-next-match-production-list nil
  "List of productions for \\[sde-next-match-production] to process.")

;; Users found it annoying, after performing a replacement across 100 files,
;; that the files would be unsaved and then upon exit Emacs would prompt for
;; saving each of those 100 files.  The `sde-next-match-visited-buffers'
;; variable allows SDE to track the files that were visited during the course
;; of (e.g.) `sde-query-replace', and lets SDE prompt the user to save all the
;; files at once after the operation is completed.  The variable is actually
;; used in `sde-next-match-next-file'.

(defvar sde-next-match-visited-buffers nil
  "List of buffers visited in the course of a next-match operation.")


;;;----------------------------------------------------------------------------
;;; 2.  Implementation of sde-next-match and related code.
;;;----------------------------------------------------------------------------

(defun sde-next-match-buffers-were-modified ()
  ;; Returns non-`nil' if any of the buffers on the list
  ;; sde-next-match-visited-buffers is modified.
  (dolist (buffer sde-next-match-visited-buffers)
    (when (buffer-modified-p buffer)
      (return t))))

(defvar sde-next-match-save-help-msg 
  "As a result of the query-replace operation you have just performed, there
may be modified buffers that have not been saved to their respective files.
Type 'y' if you wish to have SDE save all the buffers now.
Type `n', `q', `ESC' or `DEL' to not save them at this time.
If you choose not to save the buffers now, Emacs will eventually ask you
again before it lets you exit this editing session.")

(defun sde-next-match-maybe-save-buffers ()
  (when (and (sde-next-match-buffers-were-modified)
	     (sde-y-or-n-p "All files processed.  Save modified buffers?"
			   sde-next-match-save-help-msg))
    (sde-save-sde-buffers sde-next-match-visited-buffers 'no-query)))

;;; This is based largely on the tags-loop-continue facility in etags.el
;;; of Emacs 19.24.

(defun sde-next-match-next-file (&optional initialize novisit)
  ;; Select next file among files in the task currently being searched.  A
  ;; first argument of t (prefix arg, if interactive) initializes to the
  ;; first file in the task's file table.  If the argument is neither nil
  ;; nor t, it is eval'ed to initialize the list of files.  Non-nil second
  ;; argument NOVISIT means use a temporary buffer to save time and avoid
  ;; uninteresting warnings.  Value is nil if the file was already visited;
  ;; if the file was newly read in, the value is the filename.
  (interactive "P")
  (cond ((not initialize)
	 ;; Not the first run.
	 nil)
	((eq initialize t)
	 ;; Initialize the list from the database.
	 (setq sde-next-match-file-list (sde-get-task-files-list sde-next-match-task))
	 (setq sde-next-match-visited-buffers nil))
	(t
	 ;; Initialize the list by evalling the argument.
	 (setq sde-next-match-file-list (eval initialize))
	 (setq sde-next-match-visited-buffers nil)))
  (unless sde-next-match-file-list
    (if sde-next-match-visited-buffers
	(progn
	  (sde-next-match-maybe-save-buffers)
	  (error "Done."))
      (error "All files processed.")))
  (let ((new (not (get-file-buffer (car sde-next-match-file-list)))))
    (if (not (and new novisit))
	(set-buffer (find-file-noselect (car sde-next-match-file-list) novisit))
      ;; Like find-file, but avoids random warning messages.
      (set-buffer (get-buffer-create " *sde-next-file*"))
      (kill-all-local-variables)
      (erase-buffer)
      (setq new (car sde-next-match-file-list))
      (insert-file-contents new nil))
    (setq sde-next-match-file-list (cdr sde-next-match-file-list))
    new))

;; Basically this is a hacked version of tags-loop-continue from etags.el.

(defun sde-next-match-file (&optional first-time)
  "Move to the next file being searched."
  (let ((messaged nil)
	new)
    (while
	(progn
	  ;; Scan files quickly for the first or next interesting one.
	  (while (or first-time
		     (save-restriction
		       (widen)
		       (not (eval sde-next-match-scan))))
	    (setq new (sde-next-match-next-file first-time t))
	    ;; If NEW is non-nil, we got a temp buffer,
	    ;; and NEW is the file name.
	    (if (or messaged
		    (and (not first-time)
			 (> baud-rate search-slow-speed)
			 (setq messaged t)))
		(message "Searching file %s..." (or new buffer-file-name)))
	    (setq first-time nil)
	    (goto-char (point-min)))

	  ;; If we visited it in a temp buffer, visit it now for real.
	  (if new
	      (let ((pos (point)))
		(erase-buffer)
		(set-buffer (find-file-noselect new))
		(widen)
		(goto-char pos)))

	  (switch-to-buffer (current-buffer))
	  (pushnew (current-buffer) sde-next-match-visited-buffers)

	  ;; Now operate on the file.
	  ;; If value is non-nil, continue to scan the next file.
	  (eval sde-next-match-operate)))
    (and messaged
	 (null sde-next-match-operate)
	 (message "Searching file %s...found" buffer-file-name))))

(defun sde-next-match-production (&optional first-time)
  (if sde-next-match-production-list
      (sde-find-production-by-name (pop sde-next-match-production-list)
				   sde-next-match-task 'no-msg)
    (error "No more matching productions.")))

;; If the variable `sde-next-match-scan' is `nil', the first element on
;; `sde-next-match-productions-list' is taken to be the next production to be
;; found and show in another buffer.  If `sde-next-match-productions-list' is
;; `nil', the list of productions is assumed to have been exhausted and
;; nothing is shown.
;; 
;; If `sde-next-match-scan' is non-`nil', it should be a form that is evaluated
;; to search for something in each file mentioned on the list
;; `sde-next-match-file-list'.  Call `sde-next-match' noninteractively with a
;; non-`nil' argument to create the file list.  The argument is passed to
;; `sde-next-match-file' (which see).  The variable `sde-next-match-operate' is
;; eval'ed when `sde-next-match-scan' evaluates to a non-`nil' value.  If
;; `sde-next-match-operate' returns nil, SDE moves on to the next file.

(defun sde-next-match (&optional first-time)
  "Continue the last find or search command by moving to the next match."
  (interactive)
  (if sde-next-match-scan
      (sde-next-match-file first-time)
    (sde-next-match-production first-time)))


;;;----------------------------------------------------------------------------
;;; 3.  Closing statements.
;;;----------------------------------------------------------------------------

(provide 'sde-next-match)
