;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/flamingo/y/soar/projects/hucka/sde/RCS/sde-indent.el,v $
;;;; $Id: sde-indent.el,v 0.1 1994/06/15 20:27:54 hucka Exp $
;;;; 
;;;; Description       : Indentation support for Soar programming.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-indent-el-version "$Revision: 0.1 $"
  "The revision number of sde-indent.el.  The complete RCS id is:
      $Id: sde-indent.el,v 0.1 1994/06/15 20:27:54 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation.
;;;; 1.  Internal constants and variables.
;;;; 2.  Functions for indentation support.
;;;; 3.  Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; -----------------
;;; 0. Documentation.
;;; -----------------
;;;
;;; This file contains code for indenting Soar productions.


;;;----------------------------------------------------------------------------
;;; 1.  Internal constants and variables.
;;;----------------------------------------------------------------------------


;;;----------------------------------------------------------------------------
;;; 2.  Functions for indentation support.
;;;----------------------------------------------------------------------------
;;;
;;; Nearly all of the following functions are based on code from either 
;;; Emacs 19.19's lisp-mode.el and Ilisp 4.12.  
;;;
;;; The need to duplicate some lisp-mode.el functionality arose because the
;;; lisp indentation routines make assumptions about the structure of the
;;; code that are not true for Soar code.  The biggest problems is
;;; lisp-mode.el's indent-sexp works in such a way that only changes
;;; indentation levels when it encounters a nested sexp.  I.e.,
;;;
;;;	     (foo)
;;;	     (bar)
;;;
;;; end up indented the same way because none of the expressions introduce a
;;; new, more-nested sexp.  On the other hand,
;;;
;;;          (foo (bar)
;;;            (biff)
;;;
;;; introduces a new level of indentation because of the sexp (bar).  The
;;; problem for Soar code is that negated clauses need to be indented
;;; slightly differently, and as far as I can tell, there is no easy way to
;;; use the lisp-indent-function to make things work out properly.
;;; lisp-mode.el's indent-sexp will simply indent negated clauses at the same
;;; block level as the previous line,
;;; 
;;;          (<s> ^color purple)
;;;          -{(<s> ^biff y)}
;;;          (<s> ^free food)
;;;
;;; and that is ugly.  For now the only way out seems to be to copy
;;; indent-sexp and hack it to understand Soar productions more specifically.
;;; The functions lisp-indent-line and indent-sexp don't have any hooks on
;;; which the modifications could be installed, so I had to copy them and
;;; modify them appropriately.  
;;;
;;; Other code here is stolen from Ilisp, as usual, to avoid having to carry
;;; around ilisp-ext.el.
;;;
;;; All of this is messy and really should be improved, but I didn't have
;;; time to do better.

(defvar sde-soar-buffer-mark)		; For the compiler.

(defun sde-newline-and-indent ()
  "If at the end of the buffer and end of an sp, send the string back to the
process mark with no newline.  Otherwise, insert a newline, then indent.  In
a Soar process buffer the region is narrowed first.  See `newline-and-indent'
for more information."
  (interactive "*")
  (save-restriction
    (when (eq major-mode 'sde-soar-mode)
      (narrow-to-region sde-soar-buffer-mark (point-max)))
    (delete-region (point) (progn (skip-chars-backward " \t") (point)))
    (insert ?\n)
    (sde-indent-line-internal nil)))

(defun sde-indent-line (&optional whole-exp)
  "Indent current line as Soar code.  
With argument, indent any additional lines of the same expression rigidly
along with this one.  This is restricted to the current buffer input."
  (interactive "P")
  (save-restriction
    (when (eq major-mode 'sde-soar-mode)
      (narrow-to-region sde-soar-buffer-mark (point-max)))
    (sde-indent-line-internal whole-exp)))

;; Slightly modified lisp-indent-line from Emacs 19 lisp-mode.el.

(defun sde-indent-line-internal (&optional whole-exp)
  ;; Indent current line as Lisp code.  With argument, indent any additional
  ;; lines of the same expression rigidly along with this one.
  (let ((indent (calculate-lisp-indent)) shift-amt beg end
	(pos (- (point-max) (point))))
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (looking-at "\\s<\\s<\\s<")
	;; Don't alter indentation of a ;;; comment line.
	(goto-char (- (point-max) pos))
      (if (and (looking-at "\\s<") (not (looking-at "\\s<\\s<")))
	  ;; Single-semicolon comment lines should be indented
	  ;; as comment lines, not as code, except if they start
	  ;; in the left column.
	  (unless (bolp)
	    (indent-for-comment)
	    (forward-char -1))
	(when (listp indent)
	  (setq indent (car indent)))
	(setq indent    (sde-adjust-indent indent)
	      shift-amt (- indent (current-column)))
	(unless (zerop shift-amt)
	  (delete-region beg (point))
	  (indent-to indent)))
      ;; If initial point was within line's indentation,
      ;; position after the indentation.  Else stay at same point in text.
      (when (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
      ;; If desired, shift remaining lines of expression the same amount.
      (and whole-exp (not (zerop shift-amt))
	   (save-excursion
	     (goto-char beg)
	     (forward-sexp 1)
	     (setq end (point))
	     (goto-char beg)
	     (forward-line 1)
	     (setq beg (point))
	     (> end beg))
	   (indent-code-rigidly beg end shift-amt)))))

;; Slightly modified indent-sexp from Emacs 19 lisp-mode.el.

(defun sde-indent-sexp (&optional endpos)
  "Indent each line of the list starting just after point.
If optional arg ENDPOS is given, indent each line, stopping when
ENDPOS is encountered."
  (interactive)
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region sde-soar-buffer-mark (point-max)))
    (let ((indent-stack (list nil))
	  (next-depth 0)
	  (starting-point (point))
	  (last-point (point))
	  last-depth bol outer-loop-done inner-loop-done state this-indent)
      ;; Get error now if we don't have a complete sexp after point.
      (save-excursion (forward-sexp 1))
      (save-excursion
	(setq outer-loop-done nil)
	(while (if endpos (< (point) endpos)
		   (not outer-loop-done))
	  (setq last-depth next-depth
		inner-loop-done nil)
	  ;; Parse this line so we can learn the state to indent the next
	  ;; line.  This inner loop goes through only once unless a line ends
	  ;; inside a string.
	  (while (and (not inner-loop-done)
		      (not (setq outer-loop-done (eobp))))
	    (setq state (parse-partial-sexp (point) (progn (end-of-line) (point))
					    nil nil state))
	    (setq next-depth (car state))
	    ;; If the line contains a comment other than the sort that is
	    ;; indented like code, indent it now with indent-for-comment.
	    ;; Comments indented like code are right already.  In any case
	    ;; clear the in-comment flag in the state because
	    ;; parse-partial-sexp never sees the newlines.
	    (if (car (nthcdr 4 state))
		(progn (indent-for-comment)
		       (end-of-line)
		       (setcar (nthcdr 4 state) nil)))
	    ;; If this line ends inside a string, go straight to next line,
	    ;; remaining within the inner loop, and turn off the \-flag.
	    (if (car (nthcdr 3 state))
		(progn
		  (forward-line 1)
		  (setcar (nthcdr 5 state) nil))
		(setq inner-loop-done t)))
	  (and endpos
	       (<= next-depth 0)
	       (progn
		 (setq indent-stack (append indent-stack
					    (make-list (- next-depth) nil))
		       last-depth (- last-depth next-depth)
		       next-depth 0)))
	  (or outer-loop-done
	      (setq outer-loop-done (<= next-depth 0)))
	  (if outer-loop-done
	      (forward-line 1)
	      (while (> last-depth next-depth)
		(setq indent-stack (cdr indent-stack)
		      last-depth (1- last-depth)))
	      (while (< last-depth next-depth)
		(setq indent-stack (cons nil indent-stack)
		      last-depth (1+ last-depth)))
	      ;; Now go to the next line and indent it according
	      ;; to what we learned from parsing the previous one.
	      (forward-line 1)
	      (setq bol (point))
	      (skip-chars-forward " \t")
	      ;; Ignore blank lines.
	      (if (not (or (eobp) (looking-at "\\s<\\|\n")))
		  (progn
		    (if (and (car indent-stack)
			     (>= (car indent-stack) 0))
			(setq this-indent (car indent-stack))
			(let ((val (calculate-lisp-indent
				    (if (car indent-stack)
					(- (car indent-stack))
					starting-point))))
			  (if (integerp val)
			      (setcar indent-stack (setq this-indent val))
			      (progn
				(setcar indent-stack (- (car (cdr val))))
				(setq this-indent (car val))))))
		    ;; Modify the value of this-indent for special cases:
		    ;; single comments, negated clauses, or the arrow.
		    ;; Double-semicolon comments are indented as code.
		    (setq this-indent (sde-adjust-indent this-indent))
		    (if (/= (current-column) this-indent)
			(progn
			  (delete-region bol (point))
			  (indent-to this-indent))))))
	  (or outer-loop-done
	      (setq outer-loop-done (= (point) last-point))
	      (setq last-point (point))))))))

;; How this works:
;;
;; sde-adjust-indent gets called by sde-indent-sexp and by sde-indent-line.
;; At the time it is called, it is facing an about-to-be-indented line.  The
;; "given-indentation" argument is the indentation of the current block of
;; code, where a block is determined by s-expression nesting.
;; sde-adjust-indent is called on every line and is used to adjust the
;; indentation for special cases, such as negated clauses, the production
;; arrow, etc.
;;
;; sde-indent-hook is called by calculate-lisp-indent, which in turn is
;; called by sde-indent-sexp and sde-indent-line.  It is used to determine
;; the indentation for *new* nested blocks of code.
;;
;; sde-indent-line is called by indent-region by virtue of the fact that it
;; is the value of the variable indent-line-function (set in sde-mode).

(defun sde-adjust-indent (given-indentation)
  ;; Must be called with point facing the first non-blank character on a line
  ;; to be examined, and with GIVEN-INDENTATION the suggested indentation
  ;; value.  Returns an adjusted value based on what part of a Soar
  ;; production is on this line.  
  (cond ((looking-at "-->")		; The arrow.
	 (+ given-indentation sde-arrow-indent-offset))
	((looking-at "-\\s * { ")		; Conjunctive negation clause.
	 (- given-indentation 2))
	((looking-at "-\\(\\s *\\)[(^]") ; Single negated clause or attrib.
	 ;; Aggressive reformatting of code.  Remove extra spaces between
	 ;; the negation and the rest.  Indent so that up-arrows line up.
	 (if (> (match-end 1) (match-beginning 1))
	     (save-excursion
	       (goto-char (match-beginning 1))
	       (delete-region (match-beginning 1) (match-end 1))
	       (insert ? )
	       (- given-indentation 2))
	     (- given-indentation 1)))
	(t				; Make sure to return something.
	 given-indentation)))

;; Indent hook for Soar code.  This is used by the standard Emacs
;; indentation routines to determine the column to which the current
;; line should be indented.
;;
;; From the doc string of parse-partial-sexp:
;;
;; State is a list of seven elements describing final state of parsing:
;; 0. depth in parens.
;; 1. character address of start of innermost containing list; nil if none.
;; 2. character address of start of last complete sexp terminated.
;; 3. non-nil if inside a string.
;;    (it is the character that will terminate the string.)
;; 4. t if inside a comment.
;; 5. t if following a quote character.
;; 6. the minimum paren-depth encountered during this scan.

(defun sde-indent-hook (indent-point state)
  (let ((containing-form-start (elt state 1)))
    (goto-char containing-form-start)
    (cond ((looking-at "(sp")		; Start of sp.
	   (if (eq major-mode 'sde-soar-mode)
	       (+ 6 sde-production-indent-offset) ;!!! Bug
	       sde-production-indent-offset))
	  ((looking-at "(<[^>]+>")	; Variable name after "("
	   (if (save-excursion
		 (goto-char indent-point)
		 (looking-at "[ \t]*<[^>]+>"))
	       (let (tmp)
		 (goto-char indent-point)
		 (forward-line -1)
		 (end-of-line)
		 (forward-sexp -1)
		 ;; Stop when see either "^foo" or "^ {  <>" or "^ {  <<".
		 ;; Gross code.  There must be a cleaner way.
		 (while (not (looking-at "\\^\\(\\(\\sw\\|\\s_\\)+\\| { \\s *<<?>?\\)"))
		   (setq tmp (point))
		   (forward-sexp -1))
		 (if tmp (goto-char tmp))
		 (current-column))
	       (progn
		 (forward-char 1)
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (list (current-column) containing-form-start))))
	  ((looking-at "(\\(goal\\|state\\|impasse\\)")
	   (forward-char 1)
	   (forward-sexp 1)
	   (skip-chars-forward " \t\n")
	   (if (looking-at "<[^>]+>")
	       (progn
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (current-column))
	       (current-column)))
	  ((looking-at " { \\s *(")	; Beginning of grouped negation.
	   (skip-chars-forward "^(")
	   (current-column))
	  ((looking-at " { \\s *<<")	; Disjunction
	   (skip-chars-forward " {  \t<")
	   (current-column))
	  (t
	   (current-column)))))  

;; Slightly modified version of reindent-lisp from ilisp-ext.el, Ilisp v. 4.12.

(defmacro sde-point-in-comment (test-regexp)
  "Return t if in a comment determined by TEST-REGEXP."
  (` (progn
       (beginning-of-line)
       (and (looking-at (, test-regexp))
	    (/= (match-end 0) (progn (end-of-line) (point)))))))

;; The following marker keeps track of point so that it doesn't move during a
;; sde-reindent.  Made a global var to avoid calling make-marker repeatedly.

(defvar sde-fill-marker (make-marker))

(defun sde-reindent ()
  "Intelligently reindent the text under the cursor.
If in a comment, indent the comment paragraph bounded by non-comments, blank
lines or empty comment lines.  If in a string, indent the paragraph bounded
by string delimiters or blank lines.  Otherwise go to the containing sp form,
close it and reindent the code block."
  (interactive)
  (let ((region (sde-in-string))
	(comment (concat "[ \t]*" comment-start "+[ \t]*")))
    (set-marker sde-fill-marker (point))
    (back-to-indentation)
    (cond (region
	   (or (= (char-after (point)) ?\")
	       (and (< (point) (car region)) (goto-char (car region)))
	       (re-search-backward "^$" (car region) 'end))
	   (let ((begin (point))
		 (end (car (cdr region)))
		 (fill-prefix nil))
	     (forward-char)
	     (re-search-forward "^$" end 'end)
	     (if (= (point) end)
		 (progn (skip-chars-forward "^\n")
			(if (not (eobp)) (forward-char))))
	     (fill-region-as-paragraph begin (point))))
	  ((looking-at comment)
	   (let ((fill-prefix
		  (buffer-substring
		   (progn (beginning-of-line) (point))
		   (match-end 0))))
	     (while (and (not (bobp)) (sde-point-in-comment comment))
	       (forward-line -1))
	     (if (not (bobp)) (forward-line 1))
	     (let ((begin (point)))
	       (while (and (sde-point-in-comment comment) (not (eobp)))
		 (replace-match fill-prefix)
		 (forward-line 1))
	       (if (not (eobp))
		   (progn (forward-line -1)
			  (end-of-line)
			  (forward-char 1)))
	       (fill-region-as-paragraph begin (point)))))
	  (t
	   (goto-char sde-fill-marker)
	   (sde-close-all-sp)
	   (sde-beginning-of-production 1 t)
	   (sde-indent-sexp)))
  (goto-char sde-fill-marker)
  (set-marker sde-fill-marker nil)
  (message "Done.")))

(defun sde-in-string (&optional begin end)
  "Return the string region that immediately follows/precedes point or that
contains point in optional region BEGIN to END.  If point is in region, t
will be returned as well."
  (save-excursion
    (unless begin
      (save-excursion
	(setq begin (sde-beginning-of-production 1 t)
	      end (sde-end-sp-text t))))
    (let* ((point (progn (skip-chars-forward " \t") (point)))
	   (done nil))
      (goto-char begin)
      (while (and (< (point) end) (not done))
	(skip-chars-forward "^\"" end)
	(setq begin (point))
	(when (< begin end)
	  (if (and (not (bobp)) (= (char-after (1- begin)) ??))
	      (forward-char)
	    (if (condition-case () (progn (forward-sexp) (<= (point) end))
		  (error nil))
		(progn			;After string
		  (skip-chars-forward " \t")
		  (if (or (= begin point) (= point (point)))
		      (setq done (list begin (point) nil))
		    (when (and (< begin point) (< point (point)))
		      (setq done (list begin (point) t)))))
	      ;; In string at end of buffer
	      (setq done (list begin end t))))))
      done)))


;;;----------------------------------------------------------------------------
;;; 3.  Closing statements.
;;;----------------------------------------------------------------------------
