;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-help.el,v $
;;;; $Id: sde-help.el,v 0.4 1994/06/23 20:36:14 hucka Exp $
;;;; 
;;;; Description       : Help support for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-help-el-version "$Revision: 0.4 $"
  "The revision number of sde-help.el.  The complete RCS id is:
      $Id: sde-help.el,v 0.4 1994/06/23 20:36:14 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation.
;;;; 1.  Internal constants and variables.
;;;; 2.  Help commands.
;;;; 3.  Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; -----------------
;;; 0. Documentation.
;;; -----------------
;;;
;;; This file contains code for implementing various help commands in SDE.


;;;----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;----------------------------------------------------------------------------

(eval-when-compile
  (require 'info))


;;;----------------------------------------------------------------------------
;;; 2.  Internal constants and variables.
;;;----------------------------------------------------------------------------

(defvar sde-commands-obarray (make-vector 103 0)
  "Obarray used for completion support for help involving Soar.")

(defvar sde-variables-obarray (make-vector 103 0)
  "Obarray used for completion support for help involving Soar.")

(defvar sde-topics-obarray (make-vector 211 0)
  "Obarray used for completion support involving SDE and Soar topics.")

;; None of these use a separate command history, because we want the same
;; symbols to be available from one command to another.  E.g., the user might
;; start off typing sde-apropos on something, then try sde-topic-help, and it's
;; desirable to be able to retrieve the item off the history list.  

(defvar sde-help-hist nil)


;;;----------------------------------------------------------------------------
;;; 2. Help commands.
;;;----------------------------------------------------------------------------

(defun sde-apropos (&optional regexp)
  (interactive (list (sde-read-string "Apropos: " 'sde-help-hist)))
  (with-output-to-temp-buffer "*Help*"
    (sde-apropos-print-matches
     (if sde-running-emacs19
	 (apropos-internal regexp 'sde-apropos-match)))))

(defun sde-apropos-match (symbol)
  ;; Determine if given symbol is in sde-commands-obarray or
  ;; sde-variables-obarray and matches a given regexp.
  (or (intern-soft (symbol-name symbol) sde-commands-obarray)
      (intern-soft (symbol-name symbol) sde-variables-obarray)))

;; Following two functions very heavily borrowed from fast-apropos.el,
;; by Joe Wells <jbw@bigbird.bu.edu> (now a standard part of Emacs 19).

(defun sde-apropos-print-matches (matches &optional spacing)
  ;;  Helper function for fast-apropos and super-apropos.  Depends on
  ;;  standard-output having been rebound appropriately.  Prints the symbols
  ;;  and documentation in alist MATCHES of form ((symbol fn-doc var-doc) ...).
  ;;  Displays in the buffer pointed to by standard-output.  Optional argument
  ;;  SPACING means put blank lines in between each symbol's documentation.
  ;;  This should only be called within a with-output-to-temp-buffer.
  (sde-apropos-get-doc matches)
  (setq matches (sort matches (function
			       (lambda (a b)
				 (string-lessp (car a) (car b))))))
  (let ((p matches)
	(old-buffer (current-buffer))
	item symbol str)
    (save-excursion
      (set-buffer standard-output)
      (unless matches (princ "No matches found."))
      (while (consp p)
	(setq item (car p)
	      symbol (car item)
	      p (cdr p))
	(unless (or (not spacing) (bobp))
	  (terpri))
	(princ symbol)		        ; Print symbol name.
	(when (commandp symbol)
	  (let (keys)
	    (save-excursion
	      (set-buffer old-buffer)
	      (setq keys (sde-where-is symbol)))
	    (indent-to 30 1)
	    (princ (or keys "(not bound to any keys)"))))
	(terpri)
	(when (setq str (nth 1 item))
	  (princ "  Function: ")
	  (princ (substitute-command-keys str)))
	(unless (bolp)
	  (terpri))
	(when (setq str (nth 2 item))
	  (princ "  Variable: ")
	  (princ (substitute-command-keys str)))
	(unless (bolp)
	  (terpri))	
	(terpri))))
  t)

(defun sde-apropos-get-doc (sym-list)
  ;; Helper for sde-apropos.
  ;; Takes SYM-LIST of symbols and adds documentation.  Modifies SYM-LIST in
  ;; place.  Resulting alist is of form ((symbol fn-doc var-doc) ...).
  ;; Returns SYM-LIST.
  (let ((p sym-list)
	fn-doc var-doc symbol)
    (while (consp p)
      (setq symbol (car p)
	    fn-doc (if (fboundp symbol)
		       (documentation symbol))
	    var-doc (documentation-property symbol 'variable-documentation)
	    fn-doc (and fn-doc
			(substring fn-doc 0 (string-match "\n" fn-doc)))
	    var-doc (and var-doc
			 (substring var-doc 0 (string-match "\n" var-doc))))
      (setcar p (list symbol fn-doc var-doc))
      (setq p (cdr p)))
    sym-list))

;; Modified from bbdb-info from bdbb-com.el. 

(defvar Info-directory)			; D. Gillespie's info.el defines this.

(defun sde-display-info-file (file)
  (require 'info)
  (let ((Info-directory (and (boundp 'Info-directory) Info-directory))
	(try (expand-file-name file sde-directory)))
    (unless (file-exists-p try)
      (setq try (concat file ".info")))
    (unless (file-exists-p try)  
      (setq try (expand-file-name file Info-directory)))
    (unless (file-exists-p try)
      (setq try (concat try ".info")))
    (unless (file-exists-p try)
      (sde-error
       (format "Info file %s doesn't exist" file)
       (format "SDE could not find the info file \"%s\"\n\
in the places it normally looks.  This almost certainly indicates a problem\n\
with your local installation of SDE.  You should report this to your local SDE
site maintainer." file)))
    (let ((Info-directory (file-name-directory try)))
      (Info-goto-node (format "(%s)Top" try)))))

(defun sde-info ()
  "Run the Emacs Info system on the SDE documentation."
  (interactive)
  (sde-display-info-file sde-info-file))

(defun sde-news ()
  "Display news about SDE, including recent changes."
  (interactive)
  (sde-info)
  (Info-goto-node "News"))

(defun sde-soar-info ()
  "Run the Emacs Info system on the on-line Soar manual."
  (interactive)
  (sde-display-info-file sde-soar-info-file))

(defmacro sde-display-notes-file (file)
  ;; Helper macro.  This needs to be a macro because we need the actual
  ;; symbol passed as argument for use in error messages. 
  (` (progn
       (sde-error-unless-site-var-set (quote (, file)))
       (let ((buffer (create-file-buffer (, file))))
	 (save-excursion
	   (set-buffer buffer)
	   (erase-buffer)
	   (condition-case err
	       (insert-file-contents (, file) t)
	     (error
	      (sde-error 
	       (format "Couldn't read file \"%s.\"" (, file))
	       (format "Couldn't read file \"%s\" because of error:\n%s"
		       (, file) err))))
	   (setq buffer-read-only t))
	 (sde-pop-to-buffer buffer)))))

(defun sde-soar-release-notes ()
  "Read the most recent release notes for Soar.
This depends on the variable `sde-soar-release-notes-file' to be set properly." 
  (interactive)
  (sde-display-notes-file sde-soar-release-notes-file))

(defun sde-soar-user-notes ()
  "Read the most recent release notes for Soar.
This depends on the variable `sde-soar-user-notes-file' to be set properly." 
  (interactive)
  (sde-display-notes-file sde-soar-user-notes-file))

;; The topics obarray works like this.  Initially, when this file is loaded,
;; sde-topics-obarray is set to the union of sde-commands-obarray and
;; sde-variables-obarray.  (See bottom of this file.)  Whenever Soar is
;; restarted, the function sde-record-soar-help-topics is called; this
;; resets sde-topics-obarray to also include the help topics that Soar knows
;; about.  This means that the user won't have the full set of help topics
;; available until Soar is started at least once.

(defun sde-topic-help (topic)
  (interactive (list (sde-completing-read
		      "Topic (? = list): "
		      'sde-help-hist sde-topics-obarray)))
  (if topic
      (cond ((intern-soft topic sde-commands-obarray)
	     (sde-command-help topic))
	    ((intern-soft topic sde-variables-obarray)
	     (sde-variable-help topic))
	    (t
	     (sde-soar-help topic)))
    (error "No topic.")))

(defun sde-command-help (cmd-name)
  "Prompts for an SDE command name (with completion) and displays help about it.
This is almost identical to the Emacs `describe-function' command, except that
completion is limited to user commands in SDE.  If called from a program,
takes argument CMD-NAME, the string name of the variable."
  (interactive (list (sde-completing-read
		      "Command name (? = list): "
		      'sde-help-hist sde-commands-obarray)))
  (if cmd-name
      (describe-function (car (read-from-string cmd-name)))
    (error "No command name.")))

(defun sde-variable-help (var-name)
  "Prompts for an SDE variable name (with completion) and displays help about it.
This is almost identical to the Emacs `describe-variable' command, except that
completion is limited to user variables in SDE.  If called from a program,
takes argument VAR-NAME, the string name of the variable."
  (interactive (list (sde-completing-read
		      "Variable (? = list): "
		      'sde-help-hist sde-variables-obarray)))
  (if var-name
      (describe-variable (car (read-from-string var-name)))
    (error "No variable name.")))

;; This is used by SDE Soar Mode to record Soar help items.

(defun sde-add-help-topic (topic)
  (intern topic sde-topics-obarray))

;; Following code partly taken from rp-describe-function.el, written by
;; Robert Potter, rpotter@grip.cis.upenn.edu.

(defun sde-where-is (cmd)
  "Prompts for an SDE command (with completion) and shows its key bindings.
This command differs from the regular Emacs `where-is' by the fact that 
it completes only on SDE commands and its output is more specific.  If
called from a Lisp program, returns a string of the form 
   \"locally on `keystroke1' and globally on `keystroke2'\"
or nil if the command has no key binding."
  (interactive (list (car (read-from-string
			   (sde-completing-read
			    "Where is command (? = list): "
			    'sde-help-hist sde-commands-obarray)))))
  ;; This finds local key bindings and global bindings separately
  ;; so that we can tell the user exactly where things are.
  (let* ((global-map (current-global-map))
	 (global (sde-where-is-internal cmd global-map))
	 (local (unwind-protect
		    (progn
		      (use-global-map (make-keymap))
		      (sde-where-is-internal cmd (current-local-map)))
		  (use-global-map global-map)))
	 (text (if (or global local)
		   (format "%s%s%s"
			   (if local
			       (format "locally on `%s'"
				       (mapconcat 'key-description local "' and `"))
			     "")
			   (if (and global local)
			       " and "
			     "")
			   (if global
			       (format "globally on `%s'"
				       (mapconcat 'key-description global "' and `"))
			     "")))))
    (when (interactive-p)
      (if text
	  (message "%s is available %s" cmd text)
	(message "%s has no key bindings." cmd)))
    text))
  
;; What follows is an ugly thing to do.  I wanted three features for the mode
;; help in SDE:
;;
;;   1) Print a message at beginning and end, to let user know that
;;	*something* is happening -- some mode docs are long and Emacs takes
;;	forever to format them.
;;	
;;   2) Cache the mode help string, to speed up help as much as possible.
;;
;;   3) Fix an inconsistency in Emacs 18 about how it prints the key
;;	binding information for (1) commands that have meta bindings and (2)
;;	commands that have no bindings.  In case (1), the mode help shows a
;;	binding of "ESC c" but in case (2), it shows "M-x foo" instead.
;;	Lucid 19 consistently uses "M-c", but FSF 19 (wouldn't you know it)
;;	uses both "ESC c" and "M-c" in describe-mode and "M-c" in the manual.
;;      I'm going to make an effort to point out this inconsistency to the
;;	GNU folks, and in the meantime, the functions below change everything
;;	to "M-c" to be consistent with the Emacs (and SDE) manuals.
;;
;; Now the problem is how to hook this into the describe-mode function.
;; Experience has shown that simply giving "\C-hm" a new binding in sde-mode-map
;; results in problems for people who must rebind C-h to act as delete
;; because their terminals are stupid.  (Creating a new binding for "\C-hm"
;; automatically introduces a new intermediate local map for "\C-h", which is
;; what causes the problems.  There is no way to create a binding only for
;; "\C-hm" without also creating a local map for C-h.)  So this code resorts
;; to resetting the function definition of describe-mode, a bad thing to do,
;; but if it's done carefully, maybe the repercussions will be small.

(defvar sde-describe-mode-text-alist nil
  "Association list of mode-names and their associated documentation strings,
storing the result of doing a sde-describe-mode for each mode.  The list
consists of dotted pairs of the form \(major-mode . documentation-string).")

(defvar sde-true-describe-mode (symbol-function 'describe-mode)
  "The original function definition of `describe-mode'.")

(fset 'sde-true-describe-mode sde-true-describe-mode)

(defun sde-describe-mode ()
  "Display documentation of current major mode."
  (interactive)
  (if (not (memq major-mode sde-modes))
      (sde-true-describe-mode)
    (with-output-to-temp-buffer "*Help*"
      (let ((mode-help (assoc major-mode sde-describe-mode-text-alist)))
	(if mode-help
	    (princ (cdr mode-help))
	  (let (tmp case-fold-search case-replace)
	    (message "Formatting documentation ...")
	    (princ mode-name)
	    (princ " Mode:\n")
	    (princ (documentation major-mode))
	    (save-excursion
	      (set-buffer "*Help*")
	      (sde-massage-key-descriptions)
	      (setq tmp (buffer-string)))
	    (message "Formatting documentation ...done.")
	    (push (cons major-mode tmp) sde-describe-mode-text-alist))))
      (print-help-return-message))))

(fset 'describe-mode (symbol-function 'sde-describe-mode))

(defvar sde-massage-key-descriptions-list
  '(("\\bLFD\\b" . "C-j")
    ("\\bESC \\([][~a-zA-Z0-9@\;\\`!\"#$%^(){}*+='.,|/<>?-]\\|LFD\\)" . "M-\\1") 
    ("\\bM-C-\\([][~a-zA-Z0-9@\;\\`!\"#$%^(){}*+='.,|/<>?-]\\|LFD\\)" . "C-M-\\1")
    ;; SDE-specific things.
    ("\\bC-c \\? \\(C-.\\|\\?\\|TAB\\|RET\\|LFD\\)" . "C-c C-h \\1")
    ("\\bC-c\\( C-h\\| C-v\\) ?RET" . "C-c\\1 C-m")
    ("\\bC-c C-v LFD" . "C-c C-v C-j")
    ("\\bC-c C-h TAB" . "C-c C-h C-i")))

(defun sde-massage-key-descriptions ()
  ;; Fix inconsistencies in the output of describe-bindings.
  (save-excursion
    (let ((translations sde-massage-key-descriptions-list))
      (while translations
	(goto-char 1)
	(while (re-search-forward (car (car translations)) nil t)
	  (replace-match (cdr (car translations)) t))
	(setq translations (cdr translations))))))

(defun sde-describe-bindings (&optional prefix)
  "Show a list of all defined keys, and their definitions.
The list is put in a buffer, which is displayed.

In Emacs 19, when called from a Lisp program, can take an optional argument
PREFIX, which if non-nil, should be a key sequence; then we display only
bindings that start with that prefix."
  (interactive)
  (if sde-running-emacs19
      (describe-bindings prefix)
    (describe-bindings))
  (save-excursion
    (set-buffer "*Help*")
    (sde-massage-key-descriptions)))

;; Special items just for menus.

(defvar sde-about-main-text 
  "The Soar Development Environment (SDE) is an integrated environment for
developing rule-based programs in the Soar language.  It is designed to
provide editing and debugging facilities, commands for creating skeleton
source code for common constructs, facilities for locating the definitions of
task components, and other features.  It emphasizes context-sensitive
operation by maintaining information about the program being developed.  Our
goal is to provide users of Soar with a development environment that not only
helps them to work more efficiently, but that also helps them to cope with
problems of complexity and program management of Soar programming.

More information about SDE is available in the on-line manual, which you
can access with the command `C-c C-h i' or from the \"Help\" menu using
the item \"Describe Mode\".")

(defvar sde-about-sde-mode-text
  "You are currently in SDE Mode, an Emacs mode for writing Soar programs.")

(defvar sde-about-soar-mode-text 
  "You are currently in SDE Soar Mode, an Emacs mode for interacting with Soar.")

(defun sde-about-sde ()
  "Help function for displaying info about SDE."
  (interactive)
  (save-excursion
    (let ((mode major-mode))
      (with-output-to-temp-buffer "*Help*"
	(set-buffer (get-buffer-create "*Help*"))
	(sde-insert-faced 'bold-italic "The Soar Development Environment (SDE).\n")
	(sde-insert-faced 'italic (format "Version %s of %s.\n\n" sde-version sde-version-date))
	(sde-insert-faced 'default (substitute-command-keys sde-about-main-text))
	(princ "\n\n")
	(if (eq mode 'sde-soar-mode)
	    (princ sde-about-soar-mode-text)
	  (princ sde-about-sde-mode-text))))))


;;;----------------------------------------------------------------------------
;;; 3.  Closing statements.
;;;----------------------------------------------------------------------------

;;; THIS MUST ALWAYS BE UPDATED TO REFLECT CURRENT SET OF COMMANDS AND VARIABLES.

;;; This list is used to construct the tables of functions and commands defined
;;; in SDE, for use with completion functions and the help commands.
;;;
;;; Using an explicit list like this is not ideal, because it means it must
;;; be updated whenever the associated files are modified.  But we need some
;;; of the symbols to be available all the time for input completion; their
;;; definitions may be located in separate files, and thus may not be loaded
;;; until later.  This rules out, for example, putting lists right with the
;;; individual files and concatenating them when the files are loaded, or
;;; defining new defun/defvar functions that automatically update the
;;; completion tables.  We're left with a bit of a maintenance headache.  Is
;;; it worth it?  Don't know yet.
;;;
;;; A better way might be to implement something like what is used in Emacs'
;;; autoload.el scheme: namely, put magic cookie comments in the source files
;;; next to the definitions of symbols that are to be put in these lists,
;;; then run a function over the source files that looks for the magic cookies
;;; and constructs the lists below.

(defvar sde-user-variables
  '(sde-file-types-regexp-list sde-ignore-files-regexp-list
    sde-inhibit-record-file-data
    sde-use-multiple-frames sde-production-indent-offset
    sde-arrow-indent-offset sde-sort-lists sde-soar-program
    sde-soar-starting-directory sde-soar-switches
    sde-prompt-for-soar-switches sde-soar-use-ptys
    sde-soar-defaults-file sde-soar-beep-after-setup
    sde-soar-use-output-buffer sde-soar-erase-output-buffer
    sde-soar-move-point-on-output sde-soar-agent-buffer-defaults
    sde-soar-output-buffer-defaults sde-soar-track-cd
    sde-soar-input-ring-size sde-soar-input-ring-filter
    sde-production-name-test-regexp sde-compat-auto-update
    sde-query-replace-highlight sde-font-lock-list
    sde-names-face sde-variables-face sde-attributes-face sde-values-face
    sde-flags-face sde-soar-prompt-face sde-soar-output-buffer-title-face
    sde-mode-map sde-find-cmds-map sde-view-cmds-map
    sde-agent-cmds-map sde-region-cmds-map sde-help-cmds-map
    sde-soar-mode-map  sde-soar-output-mode-map
    sde-mode-syntax-table sde-mode-abbrev-table sde-site-hook
    sdde-load-hook sde-mode-hook sde-soar-mode-hook
    sde-soar-output-mode-hook sde-soar-hook sde-soar-error-hook
    sde-header-hooks sde-go-args sde-run-args sde-matches-args
    sde-ms-args sde-firing-counts-args sde-print-args
    sde-preferences-args sde-list-productions-args sde-list-chunks-args
    sde-list-justifications-args sde-agent-go-args sde-schedule-args
    sde-reset-args sde-explain-args sde-feedback-dcc
    sde-feedback-self-blind sde-feedback-archive-file-name
    sde-feedback-setup-hook))

(defvar sde-user-commands
  '(soar-mode production-editor-mode sde-feedback
    sde-find-production-by-name sde-find-production-by-lhs
    sde-find-production-by-rhs sde-find-production-by-body
    sde-find-operator sde-find-problem-space
    sde-header-soar-version sde-reposition-window
    sde-search sde-search-regexp sde-query-replace
    sde-query-replace-regexp sde-replace-string sde-replace-regexp
    sde-soar-mode sde-soar-output-mode soar go run run-soar reset
    load-errors load-errors-on load-errors-off init-soar pbreak
    unpbreak ptrace unptrace explain explain-on explain-off
    firing-counts list-chunks list-justifications list-productions
    matches matches-1 matches-2 matches-3 memory-stats ms pgs pgso
    preferences print-soar print-stats rete-stats wm sde-view-chunks
    sde-view-pbreaks sde-view-ptraces sde-view-working-memory
    excise excise-chunks excise-task excise-all excise-file
    init-soar load-soar load-defaults select-agent schedule agent-go
    create-agent destroy-agent list-agents soarnews max-elaborations
    max-chunks sde-switch-to-soar sde-close-and-send sde-send-production
    sde-region-send sde-region-pbreak sde-region-ptrace sde-region-excise
    sde-previous-input sde-next-input sde-next-input-matching
    sde-previous-input-matching sde-previous-similar-input
    sde-interrupt-soar sde-kill-output sde-kill-input sde-show-output
    sde-backward-prompt sde-forward-prompt sde-return sde-bol
    sde-soar-track-cd-toggle track-cd-toggle
    sde-beginning-of-production sde-end-of-production
    sde-mark-production sde-cd
    sde-close-all-sp sde-newline-and-indent sde-indent-line
    sde-indent-sexp sde-reindent sde-region-comment
    sde-region-count-productions sde-close-and-send sde-find-unbalanced
    sde-find-unbalanced-region sde-reposition-window sde-insert-date-stamp
    sde-task sde-reset-tasks sde-list-tasks sde-next-match sde-mode
    sde-apropos sde-info sde-news sde-soar-info sde-soar-release-notes
    sde-topic-help sde-command-help sde-variable-help sde-where-is
    sde-describe-mode sde-describe-bindings sde-mouse-print-soar
    sde-mouse-matches sde-mouse-preferences sde-mouse-find-production-by-name
    sde-mouse-send-production sde-mouse-ptrace-production sde-mouse-explain
    sde-why sde-about-sde sde-popup-menu))
    
;;; Note: when Soar is started up, the `sde-topics-obarray' is updated to
;;; include things defined by Soar.

(eval-when (load eval)
  ;; Variables.
  (dolist (sym sde-user-variables)
    (intern (prin1-to-string sym) sde-variables-obarray)
    (intern (prin1-to-string sym) sde-topics-obarray))
  ;; Commands.
  (dolist (sym sde-user-commands)
    (intern (prin1-to-string sym) sde-commands-obarray)
    (intern (prin1-to-string sym) sde-topics-obarray)))
