;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-find.el,v $
;;;; $Id: sde-find.el,v 0.20 1994/06/22 08:54:33 hucka Exp $
;;;; 
;;;; Description       : Production findinding facilities for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;; 
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-find-el-version "$Revision: 0.20 $"
  "The revision number of sde-find.el.  The complete RCS id is:
      $Id: sde-find.el,v 0.20 1994/06/22 08:54:33 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; -----------------
;;; 0.  Documentation
;;; -----------------
;;;
;;; sde-find-production-by-name   C-c C-f C-n
;;; sde-find-production-by-lhs    C-c C-f C-l
;;; sde-find-production-by-rhs    C-c C-f C-r
;;; sde-find-production-by-body   C-c C-f C-b
;;; sde-find-task                 C-c C-f C-t
;;; sde-find-operator             C-c C-f C-o
;;; sde-find-problem-space        C-c C-f C-p


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Requirements

(require 'sde-next-match)

;; This is wrapped so that tagify.el is loaded at compile time for the
;; compiler, but does not need to be included at load time of this file
;; except when certain functions are actually called.  So the needed
;; functions are autoloaded later in this file, and that tagify.el is only
;; loaded at run-time if really necessary.  (Every little speed tweak counts.)

(eval-when (compile)
  (require 'tagify))

;; Provide is at the end.


;;;----------------------------------------------------------------------------
;;; 2.  Pattern matcher
;;;----------------------------------------------------------------------------
;;;
;;; This implementation of a pattern matcher borrows some ideas from the
;;; pattern matcher presented by Peter Norvig in Chapter 6 of his book
;;;
;;;    Paradigms of Artificial Intelligence Programming: Case Studies in
;;;    Common Lisp.  (Morgan Kaufmann, 1992.)
;;;
;;; The basic idea here is to allow pattern-matching searches of the LHS and
;;; RHS of productions.  

;;; The input lists (the parsed production LHSs & RHSs) look like this:
;;;
;;;    (<g> ^problem-space <p> ^operator <o> ^object nil)
;;;    (<p> ^name foo)
;;;    (<o> ^arg bar)
;;;
;;; The pattern matcher permits search patterns such as
;;;
;;;    "(<g> ^operator <o>) (<o> ^arg <x>)"
;;;
;;; where names surrounded by '<' '>' are variables.  The idea is to let the
;;; user specify a single path down the condition graph.  Variables must be
;;; allowed, and the syntax of the search pattern should be that of Soar
;;; productions.

(defconst sde-fail nil)			; Indicates pattern matcher failure.
(defconst sde-no-bindings '((t . t)))	; Indicates succes, with no variables.

;;; Utilities used in the following code.

(defsubst sde-make-binding (var val)
  ;; Creates a new binding, a pair consisting of (VAR . VAL).
  (cons var val))

(defsubst sde-extend-bindings (var val bindings)
  ;; Add the pair (VAR . VAL) to the binding list BINDINGS.
  (cons (sde-make-binding var val)
	;; When a real binding is added, remove the dummy sde-no-bindings.
        (if (eq bindings sde-no-bindings)
            nil
	  bindings)))

(defsubst sde-get-binding (var bindings)
  ;; Looks for & returns a binding for variable VAR in the given BINDINGS list.
  (rest (assq var bindings)))

;; I made the pattern matcher accept only simply formatted search patterns
;; consisting of lists of id-attribute-value triples.  However, for the
;; user's convenience, the user can type slightly more normal patterns that
;; look like Soar conditions.  E.g.,
;;
;;   '( (goal <g> ^operator <o> ^problem-space <p>)
;;      (<o> ^name foo) )
;;
;; The input patterns are reformatted by the simple code below.  Another
;; operation performed by the reformatting code is to rename all variables
;; such as "<x>" to gensym'ed symbols of the form "_SDENNNN", to minimize the
;; chances of an accidental name clash between a search variable and a
;; condition clause element.
;;
;; The reformatting code doesn't have to be too efficient, I think, because
;; it will be called only once per search.

(defun sde-new-match-variable ()
  ;; Return a new, gensym'ed variable name.
  (gensym "_SDE"))

(defsubst sde-match-variable-p (sym)
  ;; Returns t if SYM is a variable in a pattern.
  (and sym (symbolp sym) (char-equal (aref (symbol-name sym) 0) ?_)))

(defun sde-rename-variables (pattern &optional bindings)
  ;; Walk down pattern, renaming variables to use new gensym'ed symbols.
  ;; Returns the same pattern but with variables renamed.
  (cons (mapcar
	 (function (lambda (sym)
		     (if (sde-variable-p sym)
			 (or (sde-get-binding sym bindings)
			     (let ((new (sde-new-match-variable)))
			       (setq bindings
				     (sde-extend-bindings sym new bindings))
			       new))
		       ;; Not a variable.
		       sym)))
	 ;; Go through each element of the first clause.
	 (first pattern))
	(unless (null (rest pattern))
	  (sde-rename-variables (rest pattern) bindings))))

(defun sde-reformat-one (clause &optional id)
  ;; Reformat one clause by turning it into a list of triples.
  (cond ((null clause) nil)
	(id
	 (cons (list id (first clause) (second clause))
	       (sde-reformat-one (nthcdr 2 clause) id)))
	(t
	 (cons (list (first clause) (second clause) (third clause))
	       (sde-reformat-one (nthcdr 3 clause) (first clause))))))

(defun sde-reformat-pattern-aux (pattern)
  (let ((front (first pattern)))  
    (cond ((null pattern) nil)
	  ((fourth front)		; More than 3 elements
	   (nconc (sde-reformat-one front)
		  (sde-reformat-pattern-aux (rest pattern))))
	  (t
	   (cons front (sde-reformat-pattern-aux (rest pattern)))))))

(defun sde-reformat-pattern (pattern)
  ;; Simplifies the syntax of a pattern by changing multiple condition tests
  ;; into a list of several single condition tests.  E.g., turns
  ;;   (goal <g> ^operator <o> ^object foo)
  ;; into
  ;;   (<g> ^operator <o>) (<g> ^object foo)
  ;; where the first condition is assumed to be the one testing the goal.
  ;; Returns the reformatted pattern, a list of triples.
  (when (memq (first (first pattern)) '(goal state impasse))
    (setq pattern (cons (rest (first pattern)) (rest pattern))))
  (sde-reformat-pattern-aux (sde-rename-variables pattern)))

(defun sde-error-if-bad-pattern (pattern &optional not-first)
  ;; Check a single Soar-like condition clause for legality (within the
  ;; limited sense of "legal" for this pattern matcher).
  (let ((clause (first pattern)))  
    (if not-first
	(cond ((memq (first clause) '(goal state impasse))
	       (error "Invalid inclusion of `%s' in clause: %s"
		      (first clause) clause))
	      ((evenp (length clause))	; Clauses should have odd length.
	       (error "Malformed clause: %s" clause)))
      ;; This is the first condition clause.
      (cond ((not (memq (first clause) '(goal state impasse)))
	     (error "First clause should begin with `goal': %s" pattern))
	    ((oddp (length clause))
	     (error "Malformed first clause: %s" clause))))
    (unless (null (rest pattern))
      (sde-error-if-bad-pattern (rest pattern) t))))

;;; The pattern matcher proper.

(defun sde-match-attr (attr clause)
  ;; Returns a list of values following each matched attribute ATTR in CLAUSE.
  (let ((m (memq attr clause)))
    (when m
      (cons (second m) (sde-match-attr attr (nthcdr 2 m))))))

(defun sde-match-cond (pattern clauses bindings)
  (let* ((triple  (first pattern))
	 (pvar    (first triple))
	 (pattr   (second triple))
	 (pval    (third triple))
	 (binding (sde-get-binding pvar bindings))
	 ;; If have binding, can only succeed by matching the clauses bound.
	 (subset  (if binding
		      (list (assq binding clauses))
		    clauses)))
    (or (dolist (clause subset)
	  (if (memq pattr clause)
	      ;; Have to loop over the matched attributes, because one
	      ;; may lead to success while another may not. 
	      (return (dolist (val (sde-match-attr pattr clause))
			(cond ((equal val pval)
			       (return bindings))
			      ((sde-match-variable-p pval)
			       (if (sde-get-binding pval bindings)
				   (if (equal (sde-get-binding pval bindings) val)
				       (return bindings)
				     (return sde-fail))
				 ;; Try if this binding would lead to success.
				 (let ((b2 (sde-match-pattern
					    (rest pattern) clauses
					    (sde-extend-bindings
					     pval val
					     (if binding
						 bindings
					       (sde-extend-bindings
						pval (first clause) bindings))))))
				   ;; If this didn't lead to success, go on to
				   ;; next item in innermost dolist.  If it
				   ;; lead to success, return the new bindings.
				   (if (eq b2 sde-fail)
				       nil
				     (return b2))))))))))
	;; If we get here, we didn't find a match.
	sde-fail)))

(defsubst sde-cond-p (pattern)
  (and (consp pattern) (= (length pattern) 3)))

(defun sde-match-pattern (pattern clauses &optional &rest bindings)
  ;; Pattern must be in internal format.
  (setq bindings (if (null bindings)	; No bindings arg given.
		     sde-no-bindings
		   (first bindings)))
  (cond ((eq bindings sde-fail)
	 sde-fail)
        ((sde-cond-p (first pattern))
	 (sde-match-cond pattern clauses bindings))
	((null pattern)
	 bindings)
        (t sde-fail)))

;;; The interfaces to the pattern matcher.

(defun sde-match-p (pattern clauses)
  (sde-error-if-bad-pattern pattern)
  (sde-match-pattern (sde-reformat-pattern pattern) clauses))

(defvar sde-prompt-for-match-pattern-hist nil)

(defun sde-prompt-for-match-pattern ()
  "Prompts for a pattern to use for searching productions.
Checks the pattern for general correctness and reformats it into internal form.
Returns the reformatted version of the pattern."
  (let ((pattern (read-from-minibuffer "Search pattern: "
				       nil nil 'read
				       'sde-prompt-for-match-pattern-hist)))
    (sde-error-if-bad-pattern pattern)
    (sde-reformat-pattern pattern)))


;;;----------------------------------------------------------------------------
;;; 3.  Finding productions
;;;----------------------------------------------------------------------------

(defvar sde-find-production-args nil)

(defvar sde-find-production-by-name-msg
  "SDE does not directly know in which file the production named
  \"%s\"
is defined.  Possible reasons:
 * The production appears in a file, but it is commented out.
 * The production has been deleted from the file that contained it originally.
 * The file no longer exists.
 * You are encountering a problem in SDE itself.

SDE can use alternative, heuristic methods for locating productions.
Type `y' to have SDE employ those heuristics now.
Type `n', `q', `ESC' or `DEL' to exit.")


(defsubst sde-make-production-name-regexp (name)
  "Take a production name and make a suitable regexp out of it."
  (format "^[ \t]*(sp\\s +%s" (regexp-quote name)))

(defun sde-find-production-by-name (name &optional tdata no-msg)
  "Find the definition of the production whose name appears under the cursor."
  (interactive (list (sde-production-name-near-point)))
  (let ((file (sde-get-production-file name)))
    (cond (file
	   (let ((buffer (find-file-noselect file))
		 (regexp (sde-make-production-name-regexp name)))
	     (sde-show-buffer-at-point buffer (sde-search-buffer buffer regexp))
	     (unless no-msg
	       (message "Found in file %s" file))))

	  ;; Hmm.  Don't know which file it's in.  Try last-ditch effort.
	  ((and (interactive-p)
		(sde-y-or-n-p "Don't know location; try heuristics? "
			      sde-find-production-by-name-msg name))
	   (sde-find-production-using-heuristics name))

	  (t
	   (error "Could not find production %s" name)))))

(defun sde-find-print-results (msg productions tdata)
  (let ((buffer (sde-get-buffer-create "*productions*")))
    (save-excursion
      (set-buffer buffer)
      (erase-buffer)
      (sde-mode-internal)
      (setf (sde-buffer-task sde-buffer-data) tdata)
      (insert (format "===== %s =====\n\n" msg))
      (let ((pt (point)))
	(dolist (p productions)
	  (insert p "\n"))
	(goto-char pt)))
    (sde-show-buffer buffer)))

(defun sde-find-production-by-lhs (pattern tdata)
  "Prompts for a search pattern and lists productions with matching conditions.
To view the definitions of the productions, you can repeatedly type
`\\<sde-mode-map>>\\[sde-next-match]' and SDE will find each production's definition
in turn, or you can move the cursor directly over a particular production name
and use `\\<sde-mode-map>\\[sde-find-production-by-name]' to find that production's
definition.  If given a prefix argument, prompts for the task in which to look
for productions\; otherwise, it uses the task with which the current buffer is
associated.

The pattern you use must be a list of conditions similar to Soar condition
clauses.  For example,
             ( (goal <g> ^problem-space <p>) (<p> ^name top-ps) )

is a pattern that will match all productions whose conditions test for a
problem space named \"top-ps\".  The actual variable names that you use are
irrelevant because the system is doing true pattern-matching search."
  (interactive (list (sde-prompt-for-match-pattern)
		     (if current-prefix-arg
			 (sde-get-file-task (sde-prompt-for-load-file))
		       (sde-task))))
  (sde-check-task-parsed tdata)
  (let ((total (sde-count-task-productions tdata))
	(count 0)
	(productions))
    ;; The list being constructed is really only needed for sde-next-match.
    (sde-maphash (function
		  (lambda (name pdata)
		    (when pdata
		      (setq count (1+ count))
		      (let ((percent (/ (* 100 count) total)))
			(message "Searching...%d%%" percent))
		      (when (sde-match-pattern pattern (sde-production-lhs pdata))
			(push (symbol-name name) productions)))))
		 (sde-task-productions tdata))
    ;; Compute length of the productions list here, instead of keeping an
    ;; explicit counting variable incremented during the loop, for loop speed.
    (message "Found %d matching productions (out of %d)."
	     (length productions) total)
    (when sde-sort-lists
      (setq productions (sort productions 'string-lessp)))
    (setq sde-next-match-production-list productions)
    (setq sde-next-match-task tdata)
    (setq sde-next-match-operate nil)
    (setq sde-next-match-scan nil)
    (when productions
      ;; Show the first match in the next window.  This may pop up a frame.
      (sde-next-match t)
      ;; Now show the list of productions.
      (sde-find-print-results "LHS pattern search" productions tdata))))

(defun sde-find-production-by-rhs (pattern &optional task)
  "Prompts for a search pattern and lists productions with matching actions.
To view the definitions of the productions, you can repeatedly type
`\\<sde-mode-map>>\\[sde-next-match]' and SDE will find each production's definition
in turn, or you can move the cursor directly over a particular production name
and use `\\<sde-mode-map>\\[sde-find-production-by-name]' to find that production's
definition.  If given a prefix argument, prompts for the task in which to look
for productions\; otherwise, it uses the task with which the current buffer is
associated."
  (interactive (list (sde-prompt-for-search-pattern)
		     (if current-prefix-arg
			 (sde-prompt-for-task-name)
		       (sde-task))))
  (error "Sorry; not yet implemented."))

(defun sde-find-production-by-body (pattern &optional task)
  "Prompts for a search pattern and lists productions with matching LHS or RHS.
Move the cursor to one of the production names and use `\\<sde-mode-map>\\[sde-find-production-by-name]'
to find that production's definition.  If given a prefix argument, prompts
for the task in which to look for productions\; otherwise, it uses the task
with which the current buffer is associated."
  (interactive (list (sde-prompt-for-search-pattern)
		     (if current-prefix-arg
			 (sde-prompt-for-task-name)
		       (sde-task))))
  (error "Sorry; not yet implemented."))


;;;----------------------------------------------------------------------------
;;; 4.  Finding operators
;;;----------------------------------------------------------------------------

(defun sde-find-operator (name)
  "Find the file that defines this operator, or do a Dired on the root directory."
  (interactive)
  (error "Sorry; not yet implemented."))


;;;----------------------------------------------------------------------------
;;; 5.  Finding problem spaces
;;;----------------------------------------------------------------------------

(defun sde-find-problem-space (name)
  "Find the file that defines this problem space, or do a Dired on the root directory."
  (interactive)
  (error "Sorry; not yet implemented."))


;;;-----------------------------------------------------------------------------
;;; 6.  Heuristic production finding.
;;;-----------------------------------------------------------------------------

;;; This code used to be the way that find-production was implemented before
;;; SDE was changed to track files and productions as soon as a file is visited.

;;; The control of execution here is messy, because we're using multiple
;;; heuristics one after another as each one fails.  It is further
;;; complicated by the need to allow continuation of searches from the last
;;; place where a string was found.
;;;
;;; The overall search procedure is split among several functions, and global
;;; variables are used to maintain state information and communicate between
;;; the functions.
;;;
;;; The general algorithm is:
;;;
;;; 1) Every production that is found is recorded in a hash table along with
;;;    the buffer and file pathname where it was found.  find-production
;;;    first checks to see if hashed info is available for a production.  If
;;;    info is available, it tries to find the production in the buffer that
;;;    it was last found in.  If the buffer has been killed or the production
;;;    isn't in the buffer any more (e.g., the user deleted it), it looks in
;;;    the file.  If the production isn't in the file anymore, it removes
;;;    the production's hashed info entirely and resorts to searching as if
;;;    the production was never encountered before.
;;;
;;; 2) If a production's info hasn't been cached, or the buffer it was last
;;;    in no longer exists, or the production isn't in the file it was last
;;;    found in, find-production next searches all Emacs buffers that are in
;;;    SDE mode.  If found, the production location is cached.
;;;
;;; 3) If the production isn't found in any buffer, find-production next
;;;    tries using tags.  If found, the production location is cached.
;;;
;;; 4) If the tags lookup fails, find-production next searches all files in
;;;    the table sde-known-files.  If found, the production location is
;;;    cached. 
;;;
;;; 5) If this fails, find-production derives a set of directories from the
;;;    file pathnames in sde-known-files, then derives a table of all files
;;;    in those directories that have SDE file extensions, and then performs
;;;    file search as in step (4).  If found, the production location is
;;;    cached.
;;;
;;; Searched-for strings are always cached even if they are not complete
;;; production names.  There is no certain way to determine whether a given
;;; string is part of a production name or a full name.
;;;
;;; When doing continuation of a search, the last-found location is
;;; cached, instead of the first.  Not only is it slightly easier to
;;; implement, but this also seems like the natural thing to do, since the
;;; user is likely to continue searching mainly if the first thing found is
;;; not the desired one.
;;;
;;; The use of tags is limited to the standard Emacs tags facility.  I was
;;; tempted to use some of the enhanced tags packages available (particularly
;;; Joe Wells' version) but that would force the user to adopt some changes
;;; to the tags behavior that they may not want.  The user should be able
;;; to use their favorite tags enhancement package if they wish.
;;;
;;; Notes and lessons from failed attempts:
;;;
;;; When constructing lists of file and directory names, we generally want
;;; only unique entries.  Emacs doesn't have good primitives for this -- it
;;; doesn't even have a built-in member function (one that uses equal for
;;; comparison) prior to Emacs 19.  In the end I decided to use hash tables,
;;; which can be implemented cheaply using Emacs' symbol table functions.
;;; The hash table functions are in the main file, sde.el.

;; Global variables to keep state information across calls.  The
;; sde-find-last-xxx variables are for implementing sde-find-next functionality.

(defvar sde-find-production-args nil)
(defvar sde-find-last-buffer nil)
(defvar sde-find-last-point  nil)
(defvar sde-find-last-name   nil)
(defvar sde-find-buffer-list nil)

(defvar sde-find-hash-table (sde-make-hash-table 523)
  "Caches production location info.")

;; The next two functions are the main entry points to this module.

(defun sde-find-production-using-heuristics (name &optional continue)
  "Find the source code of the production whose name is under the cursor.
This will also find productions whose names are longer than and contain the
name under the cursor.  If given a prefix argument, prompts for a string and
searches for productions whose names contain the given string.

This command uses a number of heuristics to attempt to locate productions in
their source files.  The following describes the search procedure:

  \(a) Search through existing Emacs buffers for the first match.
  \(b) If fail, attempt to look up the name using the standard Emacs tag
       facility.  You must have built a tags table for your Soar files in
       order for this to work.
  \(c) If fail, search through all files known to have either been edited in
       SDE or loaded into a Soar subprocess in Soar Mode.
  \(d) If fail, search through all files found in the directories of the
       files from (c).

Since some production names may be substrings of other, longer production
names, the first match found by this command may not be the production you
are looking for.  Use the command `sde-find-next-production' \(\\[sde-find-next-production])
to continue searching for the same name from the current file and file
position."
  (interactive (list (if current-prefix-arg
			 (sde-prompt "Find production: " 'sde-find-production-args)
			 (sde-production-name-near-point))
		     nil))
  (let ((hashdata (sde-gethash name sde-find-hash-table))
	(regexp (sde-make-production-name-regexp name))
	buffer file pos)
    (setq sde-find-last-name        name
	  sde-find-last-buffer      nil
	  sde-find-last-point       nil
	  sde-find-buffer-list (sde-get-sde-buffers))
    (if hashdata
	(if (and (setq buffer (car hashdata))
		 (sde-buffer-exists-p buffer)
		 (setq pos (sde-search-buffer buffer regexp)))
	    (progn
	      (sde-show-buffer-at-point buffer pos)
	      (setq sde-find-last-buffer buffer
		    sde-find-last-point  pos))
	    ;; Buffer doesn't exist or production not in that buffer anymore.
	    ;; Try the cached file name.
	    (if (and (setq file (car (cdr hashdata)))
		     (setq buffer (find-file-noselect file))
		     (setq pos (sde-search-buffer buffer regexp)))
		(progn
		  (sde-puthash name sde-find-hash-table (list buffer file))
		  (sde-show-buffer-at-point buffer pos)
		  (setq sde-find-last-buffer buffer
			sde-find-last-point  pos))
		;; Not in buffer, not in file.  Remove the hash entry and
		;; resort to searching.
		(progn
		  (sde-puthash name sde-find-hash-table nil)
		  (sde-find-by-search name))))
	;; No hash entry, so just search.
	(sde-find-by-search name))))

(defun sde-find-next-production-using-heuristics ()
  "Find the next production whose name was last searched-for.  
Strings searched for using `sde-find-production' (\\[sde-find-production]) are remembered.
This function allows you to search for the next occurrence of a string
in your files.  This is useful because production names may not be unique
and/or may be substrings of other, longer production names, and the first
match found may not be the desired one."
  (interactive)
  (sde-find-by-search sde-find-last-name 'continue))

(defun sde-find-by-search (name &optional continue)
  ;; The main control loop of the find-production search operations.
  ;; Searches for production named NAME through buffers, files and
  ;; directories.  Optional arg CONTINUE indicates that this is a
  ;; continuation of the last find-production search.
  (let ((regexp (sde-make-production-name-regexp name)))
    (cond ((sde-find-in-buffers     name regexp continue))
	  ((sde-find-tag            name regexp continue))
	  ((sde-find-in-files       name regexp))
	  ((sde-find-in-directories name regexp))
	  ((if continue
	       (error "No more occurrences of %s" name)
	       (error "Cannot find %s" name))))))

(defun sde-find-in-buffers (name pattern &optional continue)
  (let (found)
    (if continue
	(cond ((null sde-find-last-buffer)
	       (setq sde-find-buffer-list (sde-get-sde-buffers)))
	      
	      ((not (sde-buffer-exists-p sde-find-last-buffer))
	       ;; We're continuing, but the buffer has been killed.
	       (setq sde-find-last-buffer nil)
	       (setq sde-find-buffer-list (cdr sde-find-buffer-list)))
	      
	      ((setq sde-find-last-point
		     (sde-search-buffer sde-find-last-buffer pattern
					sde-find-last-point))
	       (sde-show-buffer-at-point sde-find-last-buffer sde-find-last-point)
	       (setq found t))

	      (t
	       (setq sde-find-buffer-list
		     (cdr (member sde-find-last-buffer sde-find-buffer-list))))))
    (if (not found)
	(let ((buffers sde-find-buffer-list)
	      buf pos)
	  (while (setq buf (car buffers))
	    (if (setq pos (sde-search-buffer buf pattern))
		(progn
		  (sde-puthash name sde-find-hash-table (list buf (sde-buffer-file-name buf)))
		  (sde-show-buffer-at-point buf pos)
		  (setq sde-find-last-buffer buf
			sde-find-last-point  pos
			buffers              nil
			found                t))
		(setq buffers (cdr buffers))))))
    found))

(defvar tag-table-alist)		; For the compiler.

(defun sde-find-tag (name pattern &optional continue)
  ;; Look up the string NAME in the current tag table.  If optional argument
  ;; CONTINUE is non-nil, do a tags-loop-continue.  Returns t if found
  ;; definition.
  (if (or (and (boundp 'tags-file-name) tags-file-name)
	  (and (boundp 'tag-table-alist) tag-table-alist)) ; Lucid version.
      ;; Try tags.
      (if continue
	  (if (tags-loop-continue)
	      (progn
		(sde-puthash name sde-find-hash-table (list (current-buffer) buffer-file-name))
		(setq sde-find-last-buffer (current-buffer)
		      sde-find-last-point  (point))
		t))
	  ;; Else, first time using tags.  Don't let errors stop the whole
	  ;; search -- just return nil.
	  (condition-case nil
	      (if (if (fboundp 'find-tag-regexp)
		      (find-tag-regexp pattern continue)
		      (find-tag pattern continue))
		  (message "Found in file %s" buffer-file-name))
	    (error nil)))))

(defun sde-find-in-files (name pattern &optional file-table)
  ;; Search for string NAME through the files known to SDE to be related to
  ;; the current task.  Does this by searching all the files in
  ;; sde-known-files which don't already have buffers.  Returns non-nil if
  ;; found.
  (let* ((where nil)
	 (found
	  (catch 'found
	    (sde-maphash (function 
			  (lambda (entry val)
			    (when val
			      (let ((file (symbol-name entry)))
				(when (and (not (get-file-buffer file))
					   (setq where (sde-search-file file pattern)))
				  (throw 'found file))))))
			 (or file-table sde-known-files))
	    nil)))
    (if found
	(if (condition-case nil
		(set-buffer (find-file-noselect found))
	      (file-error nil))
	    ;; Found file and was able to read it okay.
	    (progn
	      (setq sde-find-last-buffer (current-buffer)
		    sde-find-last-point  where)
	      (sde-puthash name sde-find-hash-table (list sde-find-last-buffer buffer-file-name))
	      (sde-show-buffer-at-point sde-find-last-buffer where)
	      (message "Found in file %s" found))
	  ;; !!! Should do something if found file but couldn't read it.
	  ))
    found))

(defun sde-find-in-directories (name pattern)
  (let ((dirs-table (sde-make-hash-table 103))
	(files-table (sde-make-hash-table 103)))
    ;; Create a table of directories.
    (sde-maphash (function
		  (lambda (entry val)
		    (when val
		      (let* ((file (symbol-name entry))
			     (dir-name (file-name-directory file)))
			(sde-puthash dir-name dirs-table t)))))
		 sde-known-files)
    ;; Create a list of files in those directories.
    (sde-maphash (function
		  (lambda (entry val)
		    (let* ((file (symbol-name entry))
			   (files (directory-files file t (sde-make-file-types-regexp))))
		      (while files
			(sde-puthash (car files) files-table t)
			(setq files (cdr files))))))
		 dirs-table)
    ;; Search through files for pattern.
    (sde-find-in-files name pattern files-table)))


;;;-----------------------------------------------------------------------------
;;; 7.  Tags support.
;;;
;;; SDE provides a way to build tags files directly within Emacs, in case the
;;; C program "stags" is not available.  This uses the tagify.el package from
;;; Dan LaLiberte.
;;; -----------------------------------------------------------------------------

;; Defined in tagify.el.  Mentioned here for the byte-compiler.

(defvar tagify-mode-alist)

(autoload 'tagify-files "tagify"
  "Make a tags file from files listed in FILENAMES." t)

(autoload 'retagify-files "tagify"
  "Update the TAGS file replacing entries only for files that have changed." t)

(defun sde-make-tags (filenames)
  "Create TAGS file for files of Soar productions.
This is an alternative to using the stags C program\; it is slower, but serves
as a back-up in case stags is unavailable."
  (interactive "sMake TAGS for files: ")
  (or (assoc 'sde-mode tagify-mode-alist)
      (setq tagify-mode-alist (cons (cons 'sde-mode 'sde-make-soar-tag-internal)
				    tagify-mode-alist)))
  (tagify-files filenames))

(defun sde-remake-tags ()
  "Update the TAGS file, replacing entries only for files that
have changed since the TAGS file was saved.  Note that this only considers
the files that are already mentioned in the TAGS file."
  (interactive)
  (require 'tags)
  (retagify-files))

(defun sde-make-soar-tag-internal ()
  (if (re-search-forward sde-sp-name-regexp nil t)
      (sde-buffer-substring 1)))


;;;-----------------------------------------------------------------------------
;;; 8.  Closing statements.
;;;-----------------------------------------------------------------------------

(provide 'sde-find)
