;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-data.el,v $
;;;; $Id: sde-data.el,v 0.1 1994/06/23 20:03:35 hucka Exp $
;;;; 
;;;; Description       : Core data structures and related functions.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-comment-el-version "$Revision: 0.1 $"
  "The revision number of sde-comment.el.  The complete RCS id is:
      $Id: sde-data.el,v 0.1 1994/06/23 20:03:35 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation.
;;;; 1.  Data structures for SDE Soar task database
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; -----------------
;;; 0. Documentation.
;;; -----------------
;;;
;;; This file contains code to implement a database system for tracking info
;;; about tasks, such as the productions and files that comprise them.
;;;
;;; Locating components of a task is often difficult in a rule-based system
;;; because of the way information is split across individual rules and
;;; files.  Structuring a program appropriately helps, but as the rule base
;;; grows, so does the difficulty of locating information.  To help with this
;;; problem, SDE maintains information about rules and other task components
;;; as the user works with task files, which means that the user does not
;;; have to explicitly create indexes, and further, the information closely
;;; reflects the current state of the tasks.
;;;
;;; The primary database structure is as follows:
;;;
;;; 1) Each task is recorded using a defstruct structure that has fields
;;;    for the pathname to the top-level load file for the task, the pathname
;;;    to the .sde data file (if any) recording the task, a hash table of
;;;    defstructs for recording the files comprising the task, and a hash
;;;    table of defstructs for recording the productions in the task.  The
;;;    load file pathname serves as a unique key identifying the task.
;;;
;;; 2) Task data structures are stored in a list.
;;;
;;; 3) Each file defstruct records the pathname of the file that loads that
;;;    file.  (This may be extended in the future to record more info.)
;;;
;;; 4) Each production defstruct records: the pathname of the file that
;;;    contains the production, a parse of the production's LHS, a parse
;;;    of its RHS, and a defstruct storing information for implementing the
;;;    pbreak facility.
;;;
;;; 5) Each file buffer has a buffer-local variable, `sde-buffer-data', that
;;;    points to a defstruct that has fields for: the task to which the
;;;    buffer belongs, the agent (if any) associated with that buffer, and
;;;    a list of the names of the productions found in the buffer at the time
;;;    of the last file save.  The latter is updated at every file save and
;;;    is used in consistency checks.
;;;
;;; The picture is something like this:
;;;
;;;  `sde-known-tasks' (a list) = ( task1 task2 task3 ...)
;;;
;;;  "task1" (a defstruct of type `sde-task') =
;;;    load-file
;;;    files-table       -----> hash table: [ file1 file2 file3 ... ]
;;;    productions-table -----> hash table: [ production1 production2 ... ]
;;;    scanned
;;;    modified
;;;
;;;  "file1" (a defstruct of type `sde-file') =
;;;    load-file
;;;
;;;  "production1" (a defstruct of type `sde-production') =
;;;    file
;;;    lhs
;;;    rhs
;;;    pbreak-data
;;;
;;; Hash tables are used instead of (e.g.) lists for the files and
;;; productions , because the entries have to be unique, and Emacs doesn't
;;; have any built-in functions for efficiently doing member with string= on
;;; a long list of elements.  Using a hash table results in a set of unique
;;; entries, and we can use mapatoms to iterate over the elements.


;;;----------------------------------------------------------------------------
;;; 1.  Data structures for SDE Soar task database
;;;----------------------------------------------------------------------------
;;;
;;; Internal data is maintained in various forms.  The following are Common
;;; Lisp-style defstructs (made possible by the CL extensions package from
;;; Dave Gillespie) used to store major pieces of information.

(defstruct (sde-buffer (:constructor sde-make-buffer-struct)
		       (:copier sde-copy-buffer-struct))
  task        	; Task structure, the task to which this buffer belongs.
  agent		; String, name of agent associated with current buffer.
  productions)	; List of productions last known to be defined in buffer.


(defstruct (sde-file (:constructor sde-make-file-struct)
		     (:copier sde-copy-file-struct))
  load-file)	; String, pathname of file that loads this one.


(defstruct (sde-production (:constructor sde-make-production-struct)
			   (:copier sde-copy-production-struct))
  file		; String, file in which production is defined.
  lhs		; List, the parse of the LHS.
  rhs		; List, the parse of the RHS.
  pbreak-data)


(defstruct (sde-task (:constructor sde-make-task-struct)
		     (:copier sde-copy-task-struct))
  load-file	; String, pathname to top load file
  root-dir      ; String, pathname to top directory (= dir of load-file)
  files		; Hash table of `sde-file' structures.
  productions	; Hash table of `sde-production' structures.
  scanned       ; Boolean, whether the task has been fully scanned.
  modified)     ; Boolean, whether task modified & .sde file needs writing

(defconst sde-task-files-table-size 211)
(defconst sde-task-productions-table-size 2027)

(defvar sde-known-tasks nil)		; List of `sde-task' structures.

;;;
;;; Support functions for the core data structures.
;;;

(defun sde-reset-tasks ()
  (interactive)
  (setq sde-known-tasks nil))

(defun sde-add-task (lfile)
  ;; Creates a new task structure keyed by load file LFILE.
  ;; Adds it to `sde-known-tasks', and returns the task structure.
  ;; Files table is not set here and must be built up separately.
  (let ((tdata (sde-make-task-struct
		:load-file   lfile
		:root-dir    (file-name-directory lfile)
		:files       (sde-make-hash-table sde-task-files-table-size)
		:productions (sde-make-hash-table sde-task-productions-table-size)
		:scanned     nil
		:modified    t)))
    (setq sde-known-tasks (cons tdata sde-known-tasks))
    tdata))

(defun sde-get-task (lfile)
  ;; Return the task data structure keyed by load file LFILE.
  (let ((tasks sde-known-tasks))
    (while (and tasks (not (equal lfile (sde-task-load-file (first tasks)))))
      (setq tasks (rest tasks)))
    (first tasks)))

(defun sde-add-task-file (file tdata)
  ;; Adds a blank file data structure for filename FILE to a task.
  ;; Returns the new file data structure.
  (setf (sde-task-modified tdata) t)
  (sde-puthash file (sde-task-files tdata) (sde-make-file-struct)))

(defun sde-remove-task-file (file tdata)
  (sde-remhash file (sde-task-files tdata)))

(defun sde-add-task-production (pname tdata)
  ;; Adds a blank file production structure for filename FILE to a task.
  ;; Returns the new file production structure.
  (sde-puthash pname (sde-task-productions tdata) (sde-make-production-struct)))  

(defun sde-remove-task-production (pname tdata)
  (sde-remhash pname (sde-task-productions tdata)))

;; Notes about the following function to count the productions in a task:
;; A field could be added to the task structure to record the number of
;; productions in a task.  However, to keep the count accurate, the process
;; of adding a production would have to be changed to first test whether a
;; given production is already defined in the task, and the count left
;; unchanged in that case.  I'm wary of adding this extra step to
;; sde-add-task-production because of the time cost involved -- that function
;; gets called during file parsing and parsing is already annoyingly slow.
;; I should benchmark sde-add-task-production with and without code to
;; update a production count field and see whether the extra time is
;; significant.

(defun sde-count-task-productions (tdata)
  ;; Returns the number of productions in the task.
  (sde-hash-table-count (sde-task-productions tdata)))

(defun sde-get-production-task (pname)
  ;; Returns the task data structure for the task to which PNAME belongs.
  (let ((tasks sde-known-tasks))
    (while (and tasks (not (sde-gethash pname (sde-task-productions (first tasks)))))
      (setq tasks (rest tasks)))
    (first tasks)))

(defun sde-get-task-files-list (tdata)
  ;; Returns a list of files in the task TDATA.  If the variable
  ;; `sde-sort-lists' is non-`nil', the list will be sorted alphabetically.
  (let (files)
    ;; Construct a list of the files in the task.  Have to watch out for null
    ;; data fields, which indicate entries that have been removed.
    (sde-maphash (function (lambda (file fdata)
			     (when fdata
			       (push (symbol-name file) files))))
		 (sde-task-files tdata))
    (if sde-sort-lists
	(sort files 'string-lessp)
      files)))

(defun sde-map-task-files (func tdata)
  "Maps the function FUNC over each file of task TDATA.
FUNC is called with one argument, a file pathname.
Returns a list of the results of callilng FUNC on each file."
  (mapcar func (sde-get-task-files-list tdata)))

(defun sde-task ()
  "Returns the task for the current buffer."
  (and sde-buffer-data (sde-buffer-task sde-buffer-data)))

(defun sde-at-least-one-task ()
  ;; Make sure there's at least one known task.
  (unless sde-known-tasks
    (sde-add-task (sde-prompt-for-load-file))))

(defun sde-list-tasks ()
  (interactive)
  (with-output-to-temp-buffer "*tasks*"
    (princ "====== Tasks known to SDE ======")
    (mapcar
     (function
      (lambda (tdata)
	(princ "\n\nTask loaded by \"")
	(princ (sde-task-load-file tdata))
	(princ "\"\n")
	(princ "  ")
	(let ((numfiles (sde-hash-table-count (sde-task-files tdata))))
	  (princ numfiles)
	  (princ " file")
	  (if (> numfiles 1)
	      (princ "s")))		; Make it plural if > 1 files.
	(princ " in task\n")
	(if (sde-task-scanned tdata)
	    (progn
	      (princ "  ")
	      (princ (sde-hash-table-count (sde-task-productions tdata)))
	      (princ " productions\n")
	      (if (sde-task-modified tdata)
		  (princ "  Corresponding .sde file needs to be updated.\n")
		(princ "  Corresponding .sde file is up-to-date.\n")))
	  (princ "  Task has not yet been fully scanned.\n"))))
     sde-known-tasks)))

;; Debugging.

(defun sde-show-production-parse ()
  (interactive)
  (let* ((pname (sde-production-name-near-point))
	 (pdata (sde-get-production-task pname)))
    (with-output-to-temp-buffer "*parse*"
      (princ (save-excursion
	       (set-buffer (get-buffer-create "* sde tmp*"))
	       (erase-buffer)
	       (cl-prettyprint (sde-production-lhs pdata))
	       (buffer-string))))))      



;;;----------------------------------------------------------------------------
;;; 4.  Reading and maintaining .sde data files.
;;;----------------------------------------------------------------------------

;;; The .sde data file is used to track the load file and constituent files of
;;; a task.  This is really only used to avoid having to prompt the user for
;;; the information each time that a task is visited (e.g., in a new editing
;;; session), but that's important enough -- users got very annoyed at the
;;; constant prompting of an earlier version of SDE because it didn't have
;;; a good way of storing info across editing sessions.
;;;
;;; The .sde file has the following format:
;;;
;;;   ;; SDE data file -- generated automatically -- DO NOT EDIT THIS.
;;;   ("load file1" 
;;;    "file1"
;;;    "file2"
;;;    ...)
;;;   ("load file2"
;;;    "file1"
;;;    "file2"
;;;    ...)
;;;
;;; For each task, the top-level load file is listed followed by the files
;;; that comprise the task.  This permits SDE to identify the load file for a
;;; task given a random file name, when a random file is visited by the user
;;; and the associated task has not yet been visited.  Once the task to which
;;; the random file belongs is identified, the info from the .sde file is not
;;; necessary, but the .sde file has to be kept up-to-date with the contents
;;; of the task.
;;;
;;; About visiting files.  Unless the variable `find-file-visit-truename' is
;;; set non-`nil', Emacs doesn't expand symbolic links out to the full file
;;; pathname when it sets the `buffer-file-name' variable.  This is a problem
;;; for SDE when the user has a lot of symbolic links all over the place,
;;; because they may end up visiting the same files through different paths.
;;; SDE must know the true file pathname of each file, both the .sde file and
;;; Soar source code files.  One way would be to require the user to set the
;;; Emacs variable `find-file-visit-truename', but this has some annoying
;;; consequences for the user, as well as being unreliable (e.g., what if a
;;; user didn't set the variable as recommended?)  So we can't rely on
;;; `buffer-file-name' being the true pathname of a buffer.  Luckily, Emacs
;;; 19.22+ and Lucid Emacs 19.10+ *do* also automatically find the expanded
;;; true filename of a file and they set `buffer-file-truename' for each
;;; buffer.  So SDE carefully works off this truename rather than
;;; `buffer-file-name', thus avoiding having to make the user set the
;;; variable `find-file-visit-truename'.  The user can find files as normal,
;;; the file buffer names may be unexpanded and use symbolic links, and SDE
;;; can still get at the true pathname without a lot of work.
;;; This issue shows up again in `sde-record-buffer'.

(defconst sde-data-file-name ".sde")

(defsubst sde-read (stream)
  ;; Returns nil if end of stream is reached. 
  (condition-case nil
      (read stream)
    (end-of-file nil)))

(defsubst sde-data-format-check-p (data)
  ;; Return nil if the data is not formatted appropriately for a .sde entry.
  (and (first data) (rest data)))

(defun sde-read-directory-data-file (dir)
  ;; Reads the records stored in the .sde data file in DIR, and assimilates
  ;; the information into the SDE task database.  Returns a list of the
  ;; tasks found, nil otherwise.
  (let ((file (concat dir sde-data-file-name)))
    (when (file-readable-p file)
      (save-excursion
	(let (buffer found-tasks)
	  (cond ((setq buffer (get-file-buffer file))
		 ;; File is already in a buffer.  First check the disk file
		 ;; hasn't changed since last used.  Then read the task data
		 ;; as normal, being careful not to try to create a new task
		 ;; structure for a task that is already in the database.
		 (set-buffer buffer)
		 (when (not (verify-visited-file-modtime (current-buffer)))
		   (revert-buffer t t)
		   (setq buffer (set-buffer (get-file-buffer file)))))
		(t
		 ;; File is not in a buffer.  Read it.
		 (setq buffer (set-buffer (sde-find-file-noselect file)))))
	  ;; Read and incorporate info.  For extra safety, read in everything
	  ;; first and do some weak tests for proper format, *then* incorporate
	  ;; the data into the database.
	  (let ((ok t) data-list data)
	    (goto-char 1)
	    (while (setq data (sde-read buffer))
	      (if (sde-data-format-check-p data)
		  (push data data-list)
		(setq ok nil)))
	    (if ok
		(dolist (data data-list)
		  (let ((lfile (sde-expand-file-name (first data) dir)))
		    ;; Don't add this if the task is already known. 
		    (unless (sde-get-task lfile)
		      (let* ((tdata  (sde-add-task lfile))
			     (ftable (sde-task-files tdata))
			     (files  (rest data)))
			(while files
			  (sde-puthash (sde-expand-file-name (first files) dir)
				       ftable (sde-make-file-struct))
			  (setq files (rest files)))
			(push tdata found-tasks)))))
	      ;; Some part of the data in the .sde file was found wrong.
	      ;; Try to get rid of this .sde file.
	      (condition-case nil
		  (progn
		    (delete-file buffer-file-name)
		    (kill-buffer buffer))
		(error nil))))
	  found-tasks)))))


(defvar sde-data-file-header
  ";;; SDE data file -- generated automatically -- DO NOT EDIT THIS.\n"
  "Header for .sde files.")

(defun sde-insert-data-file-header ()
  ;; Insert the data file header at the top of an .sde file.
  (insert sde-data-file-header))

(defun sde-insert-data-file-task-data (tdata)
  (let ((dir (sde-task-root-dir tdata)))
    (insert "(\"" (sde-relative-pathname (sde-task-load-file tdata) dir) "\"")
    (insert "\n \"")
    (insert (mapconcat
	     (function (lambda (file) (sde-relative-pathname file dir)))
	     (sde-get-task-files-list tdata)
	     "\"\n \""))
    (insert "\")\n")))

(defun sde-insert-data-file-data (dir)
  (dolist (tdata sde-known-tasks)
    (when (equal (sde-task-root-dir tdata) dir)
      (sde-insert-data-file-task-data tdata)
      (setf (sde-task-modified tdata) nil))))

;;; Data files get written out when a new task is created, or when SDE
;;; discovers that an existing task (on disk) doesn't have an associated
;;; .sde data file.

(defun sde-write-directory-data-file (dir)
  ;; Writes out the .sde file in directory DIR, using the current contents of
  ;; the SDE database for the tasks whose load files are located in DIR. 
  (save-excursion
    (let ((file (concat dir sde-data-file-name)))
      (if (file-writable-p file)
	  (let ((buffer (or (get-file-buffer file)
			    (sde-find-file-noselect file))))
	    (if buffer
		(progn
		  (set-buffer buffer)
		  ;; Check that file hasn't been modified since we last wrote it.
		  (when (not (verify-visited-file-modtime buffer))
		    ;; Argh, it's been modified.  Read it anew to catch any possible
		    ;; new entries prior to writing it out again.
		    (sde-read-directory-data-file dir)
		    (setq buffer (set-buffer (get-file-buffer file))))
		  (erase-buffer)
		  (sde-insert-data-file-header)
		  (sde-insert-data-file-data dir)
		  (save-buffer))
	      ;; Couldn't read the file.  Why?
	      (message ".sde file exists but couldn't be read." dir)))
	;; File is not writable.
	(message "Warning: cannot write .sde file in %s" dir)))))

(defun sde-update-data-files ()
  ;; Run through known tasks and update the data files of those tasks whose
  ;; modified flag is set.
  (let (failures)
    (dolist (tdata sde-known-tasks)
      (when (sde-task-modified tdata)
	(condition-case err
	    (sde-write-directory-data-file (sde-task-root-dir tdata))
	  (error
	   (push tdata failures)))))
    (when failures
      (sde-error
       "Error saving data files"
       (concat
	"Encountered error while trying to save the .sde file(s) for the task(s)\n\
loaded by:\n" (mapconcat 'sde-task-load-file failures "\n"))))))

(defsubst sde-file-in-task-p (file tdata)
  ;; Return non-nil if the FILE is believed to be part of task TDATA.
  (sde-gethash file (sde-task-files tdata)))

(defsubst sde-search-tasks-for-file (file tasks)
  ;; Search for FILE being a part of one of the tasks in list TLIST.
  ;; If found, return the task data structure, else return nil.
  (while (and tasks (not (sde-file-in-task-p file (car tasks))))
    (setq tasks (rest tasks)))
  (car tasks))

(defun sde-get-file-task (file &optional dont-scan)
  ;; Returns the task that contains FILE, or nil.  If the task is initially
  ;; not found, and optional argument DONT-SCAN is nil, it initiates a search
  ;; of .sde files beginning in the current directory and moving backward.
  ;; FILE is assumed to already be a truename.
  (or (sde-search-tasks-for-file file sde-known-tasks)
      ;; Not in existing task database.  Initiate a search of .sde files.
      ;; Keep looking for .sde files and reading them (and by side-effect,
      ;; adding new task structures to `sde-known-tasks'), then checking if
      ;; the current file becomes known as a result.
      (and (null dont-scan)
	   (let ((dir (file-name-directory file))
		 found)
	     (while (and dir (not found))
	       (setq found (sde-search-tasks-for-file
			    file (sde-read-directory-data-file dir)))
	       (setq dir (sde-directory-parent dir)))
	     found))))

(defun sde-get-production-file (pname)
  ;; Returns the file pathname where production PNAME is found.  Since this
  ;; may require doing a full scan of each task, first all of the
  ;; already-scanned tasks are checked, then if the production is still not
  ;; found, the unscanned tasks are checked.
  (sde-at-least-one-task)
  (let ((tasks sde-known-tasks)
	found)
    ;; Go through the scanned tasks first.    
    (while tasks
      (let ((tdata (first tasks)))
	(if (and (sde-task-scanned tdata)
		 (sde-gethash pname (sde-task-productions tdata)))
	    (setq found tdata
		  tasks nil)
	  (setq tasks (rest tasks)))))
    (if found
	(sde-production-file (sde-gethash pname (sde-task-productions found)))
      ;; Not found in the scanned tasks.  Go through the unscanned ones.
      (setq tasks sde-known-tasks)
      (while tasks
	(let ((tdata (first tasks)))
	  (unless (sde-task-scanned tdata)
	    (sde-parse-task tdata)
	    (if (sde-gethash pname (sde-task-productions tdata))
		(setq found tdata
		      tasks nil)))
	  (setq tasks (rest tasks))))
      (when found
	(sde-production-file (sde-gethash pname (sde-task-productions found)))))))

;;;----------------------------------------------------------------------------
;;; 6.  Functions for prompting the user.
;;;----------------------------------------------------------------------------

(defvar sde-y-or-n-p-default-help-msg
  "Type SPC or `y' to accept;
DEL or `n' to say no;
ESC or `q' to exit.")

(defun sde-y-or-n-p (prompt &optional help-msg arg)
  "Ask user a \"y or n\" question.  Return t if answer is \"y\".
Takes one argument, which is the string to display to ask the question.
It should not end in a space; `sde-y-or-n-p' adds ` (y, n, ?, C-h) ' to it.
No confirmation of the answer is requested; a single character is enough.
Also accepts Space to mean yes, or Delete to mean no.
Optional argument HELP-MSG is a message to display if the user types
`?' or `C-h'.  Optional second argument ARG should be an argument that
is used in (funcall help-msg arg) to format the help message presented
to the user\; in that case, help-msg should contain a %s where the
value of ARG should be substituted."
  (save-window-excursion
    (let* ((hchar (key-description (vector help-char)))
	   (prompt (format "%s (y, n, ?, %s)" prompt hchar)))
      (let (answer)
	(while
	    (let (char)
	      ;; Prompt.
	      (message prompt)
	      (setq char (read-event))
	      ;; Show the user's input.
	      (message (concat prompt (format " %s" (single-key-description char))))
	      (cond ((memq char '(?y ?Y ?\ ))
		     (setq answer t)
		     nil)
		    ((memq char '(?n ?N ?\C-? delete))
		     (setq answer nil)
		     nil)
		    ((memq char '(?? ?\C-h help))
		     (with-output-to-temp-buffer "*Help*"
		       (if help-msg
			   (princ (funcall 'format help-msg arg))
			 (princ sde-y-or-n-p-default-help-msg)))
		     ;; Go back around the loop.
		     t)
		    (t
		     (beep)
		     (message "Type %s for help" hchar)
		     (sit-for 1)
		     ;; Go back around the loop.
		     t))))
	(message "")			; Erase last prompt.
	answer))))


;;; These help hide the differences in user prompting between Emacs 18 with
;;; gmhist and Emacs 19.  There are a few enhancements, such as making
;;; `sde-completing-read' return nil when the user types just a return
;;; (instead of returning "" as the regular function does).


(defun sde-read-string (prompt &optional hist initial)
  (if (and hist (featurep 'gmhist))
      (read-with-history-in hist prompt initial)
    (read-from-minibuffer prompt initial nil nil hist)))


(defun sde-completing-read (prompt &optional hist table pred mustmatch initial)
  (let ((answer (if (and hist (featurep 'gmhist))
		    (completing-read-with-history-in
		     hist prompt table pred mustmatch initial)
		  (completing-read prompt table pred mustmatch initial hist))))
    (if (string= answer "")
	nil
      answer)))


(defun sde-read-file-name (prompt &optional hist dir default mustmatch initial)
  (if (and hist (featurep 'gmhist))
      (read-file-name-with-history-in hist prompt dir default mustmatch initial)
    (read-file-name prompt dir default mustmatch initial)))

 
(defun sde-prompt (prompt &optional hist initial)
  "If `current-prefix-arg' is non-nil, prompt user with PROMPT and history HIST.  
Otherwise, if HIST is provided, return the first item on the list, or if 
HIST is not provided, return INITIAL."
  (if hist
      (if current-prefix-arg
	  (if (featurep 'gmhist)
	      (read-with-history-in hist prompt initial)
	    (read-from-minibuffer prompt initial nil nil hist))
	(car (eval hist)))
    (if current-prefix-arg
	(read-from-minibuffer prompt initial)
      initial)))


;;; Functions for prompting for common values.

(defvar sde-prompt-for-load-file-hist nil)
(defvar sde-prompt-for-ps-name-hist nil)
(defvar sde-prompt-for-op-name-hist nil)


(defun sde-prompt-for-load-file (&optional buffer)
  (save-window-excursion
    (when buffer
      (select-frame (first (frame-list)))
      (switch-to-buffer buffer))
    (let ((answer
	   (substitute-in-file-name
	    (sde-read-file-name "Top-level load file: "
				'sde-prompt-for-load-file-hist nil nil t))))
      (if (string= answer "")
	  nil
	answer))))


;;;----------------------------------------------------------------------------
;;; 4.  File scanning and parsing.
;;;----------------------------------------------------------------------------

;;; SDE production parsing code for Soar 6 syntax.
;;;
;;; The idea here is to create an internal representation of the object
;;; structure and attribute names mentioned in the condition side of a
;;; production.  A graph is constructed for each production in each file of a
;;; task when the files of the task are first read in, and thereafter
;;; whenever a buffer is saved.  This is done to permit fast searching of
;;; production bodies.
;;;
;;; The condition side of a production can be regarded as a graph, and the
;;; internal representation here is basically a graph structure stored in
;;; list form.  The nodes are the (object) variables or constants, and the
;;; arcs of the graph are the attribute names.  So something like
;;;
;;; (goal <g> ^object <sg> ^operator <o>)
;;; (goal <sg> ^object nil ^impasse no-change)
;;; (<o> ^name foo ^object <x>)
;;; (<x> ^color red ^size big)
;;;
;;; is conceptually represented as
;;;
;;;                        /--- ^impasse -- no-change
;;;                       /
;;;     /- ^object --- <sg> --- ^object --- nil
;;;    /
;;;   /
;;; <g> ------ ^operator - <o> -- ^name ----- foo
;;;                         \
;;;                          \--- ^object ---- <x> --- ^color -- red
;;;                                              \
;;;                                               \--- ^size -- big
;;;
;;; and in internal list form as
;;;
;;; (<g> ^object <sg> ^operator <o>)
;;; (<sg> ^object nil ^impasse no-change)
;;; (<o> ^name foo ^object <x>)
;;; (<x> ^color red ^size big)
;;;
;;; where the first item in the list is always the lowest goal tested in the
;;; production condition.  Note that attribute names are recorded as symbols,
;;; including their leading up-arrows, because that's the form they end up in
;;; when they are read in using the Emacs Lisp `read' function.
;;;
;;; This internal graph then permits pattern-matching search on the condition
;;; sides of productions.
;;;
;;; Only a limited number of aspects of a production LHS are actually
;;; recorded.  Typical LHS clause constructs are translated as follows:
;;;
;;; (<x> ^att1 const)              --> (<x> ^att1 const)
;;; (<x> -^att1 const)             --> (<x> ^att1 const)
;;; -{(<x> ^att1 const)}           --> (<x> ^att1 const)
;;;
;;;    (Negated attribute tests are listed, because in a sense, the condition
;;;    is testing that attribute.  In fact the presense of negations is
;;;    ignored completely.)
;;;
;;; (<x> ^{ <z> <> foo } <y>)      --> (<x> ^foo <y>)
;;;
;;;    (Variable attributes are basically ignored.  In compound tests like
;;;    this, it tries to extract whatever it can, such as the attribute name
;;;    foo, because the condition is testing that attribute in some sense.)
;;;
;;; (<x> ^<z> <y>)                 --> ignored
;;;
;;; (<x> ^ << att1 att2 >> <y>)    --> (<x> ^att1 <y> ^att2 <y>)
;;; (<x> ^ << att1 att2 >> <> <y>) --> (<x> ^att1 nil ^att2 nil)
;;;
;;;    (As an attribute value, nil signifies the end of that graph path.
;;;    This is currently an arbitrary limitation on the parsing and matching
;;;    code.)
;;;
;;; (<x> ^att1.att2 <y>)           --> (<x> ^att1 <a>)
;;;                                    (<a> ^att2 <y>)
;;;
;;; The full Soar 6 grammar for condition sides of productions is as follows,
;;; in BNF notation (taken from the Soar 6 User's Manual):
;;;
;;; <condition-side>   ::= <cond>+
;;; <cond>             ::= <positive_cond> | - <positive_cond>
;;; <positive_cond>    ::= <conds_for_one_id> | { <cond>+ } 
;;; <conds_for_one_id> ::= ( [goal|impasse] [<id_test>] <attr_value_tests>* )
;;; <id_test>          ::= <test>
;;; <attr_value_tests> ::= [-] ^ <attr_test> [.<attr_test>]* <value_test>*
;;; <attr_test>        ::= <test>
;;; <value_test>       ::= <test> [+] | <conds_for_one_id> [+]
;;;
;;; <test>             ::= <conjunctive_test> | <simple_test>
;;; <conjunctive_test> ::= { <simple_test>+ } 
;;; <simple_test>      ::= <disjunction_test> | <relational_test>
;;; <disjunction_test> ::= << <constant>* >>
;;; <relational_test>  ::= [<relation>] <single_test>
;;; <relation>         ::= <> | < | > | <= | >= | = | <=>
;;; <single_test>      ::= variable | <constant>
;;; <constant>         ::= sym_constant | int_constant | float_constant
;;;

(defvar sde-cond nil)
(defvar sde-parsed nil)

(defsubst sde-name-string (sym)
  (if (numberp sym)
      (number-to-string sym)
    (symbol-name sym)))

(defsubst sde-variable-p (sym)
  "Is SYM a Soar variable (a symbol beginning with '<' and ending with '>')?"
  (and sym
       (symbolp sym)
       (char-equal (aref (symbol-name sym) 0) ?<)
       (not (memq sym '(<< < <= <=> <>)))))

(defsubst sde-attribute-p (sym)
  "Is SYM a Soar attribute (a symbol beginning with '^')?"
  (and sym
       (symbolp sym)
       (char-equal (aref (symbol-name sym) 0) ?^)))

(defsubst sde-dotted-p (sym)
  (and (symbolp sym)
       (string-match "\\." (symbol-name sym))))

(defsubst sde-gensym-new-var (letter-string)
  (intern (concat "<" (symbol-name (gensym letter-string)) ">")))


;;; The following set of functions handle parsing of 
;;;   <test> ::= <conjunctive_test> | <simple_test>
;;; The entry point is `sde-parse-test', later on.  Nearly all functions
;;; are open-coded for speed.

(defsubst sde-parse-relational-test (front)
  ;; Parse "<relational_test>  ::= [<relation>] <single_test>".
  ;;
  (when (memq front '(<> < > <=> <= => =))
    ;; Just eat the relation.
    (setq sde-cond (rest sde-cond))
    (setq front    (first sde-cond)))
  ;; Move pointer past "<single_test>"...
  (setq sde-cond (rest sde-cond))
  ;; ... and return the "<single_test>".
  (list front))


(defsubst sde-parse-disjunctive-test (front)
  ;; Parse "<disjunction_test> ::= << <constant>* >>".
  ;;
  (let (syms)
    (setq sde-cond (rest sde-cond))	; Eat "<<".
    (while (and sde-cond (not (eq (first sde-cond) '>>)))
      (push (first sde-cond) syms)
      (setq sde-cond (rest sde-cond)))
    (if (eq (first sde-cond) '>>)
	(setq sde-cond (rest sde-cond))) ; Eat ">>".
    syms))


(defsubst sde-parse-simple-test (front)
  ;; Parse "<simple_test> ::= <disjunction_test> | <relational_test>".
  ;;
  (cond ((eq front '^<<)
	 (setq sde-cond (cons '<< (rest sde-cond)))
	 (sde-parse-disjunctive-test front))

	((eq front '<<)
	 (sde-parse-disjunctive-test front))

	(t
	 (sde-parse-relational-test front))))


(defsubst sde-parse-conjunctive-test (front)
  ;; Parse "<conjunctive_test> ::= { <simple_test>+ }"
  ;;
  (let (vars-and-constants)
    (setq sde-cond (rest sde-cond)) ; Eat the "{".
    (while (and (setq front (first sde-cond))
		(not (eq front '} )))
      (setq vars-and-constants
	    (nconc (sde-parse-simple-test front) vars-and-constants)))
    (setq sde-cond (rest sde-cond)) ; Eat the " } ".
    vars-and-constants))


(defun sde-parse-test ()
  ;; Parse "<test> ::= <conjunctive_test> | <simple_test>".
  ;; Returns a list of the variables and constants mentioned.
  (let ((front (first sde-cond)))
    (if (memq front '( { ^{ ))
	(sde-parse-conjunctive-test front)
      (sde-parse-simple-test front))))


(defsubst sde-parse-test-1 (front)
  (if (or (sde-variable-p front) (sde-attribute-p front))
      (progn
	(setq sde-cond (rest sde-cond))
	(list front))
    (sde-parse-test)))


(defsubst sde-parse-value-test (front)
  ;; Parses "<value_test> ::= <test> [+] | <conds_for_one_id> [+]"
  ;;
  (if (consp front)
      ;; Structured value notation.
      (let ((tmp sde-cond)
	    (result (sde-parse-conds-for-one-id front)))
	(setq sde-cond (rest tmp))
	(list (first result)))
    ;; Parse a test, optionally followed by a "+".
    (prog1
	(sde-parse-test-1 front)
      ;; sde-cond will have been changed, hence `front' will no longer
      ;; refer to the correct element of the list.  Use (first sde-cond).
      (when (eq (first sde-cond) '+)
	(setq sde-cond (rest sde-cond))))))


(defsubst sde-parse-value-test-star ()
  ;; Parses "<value_test>*" 
  ;;
  (let (front results)
    (while (and sde-cond
		(not (eq (setq front (first sde-cond)) '-))
		(not (sde-attribute-p front)))
      (setq results (nconc (sde-parse-value-test front) results)))
    (or results (list nil))))


(defsubst sde-parse-dotted-attr-value-tests ()
  ;; Parse "<attr_value_tests> ::= [-] ^ <attr_test> [.<attr_test>]* <value_test>*"
  ;; for the case where the attribute is dotted.  This is messy because the
  ;; attribute is read as a symbol, but the only way to decompose it is to
  ;; transform it into a string.
  (let* ((attr (first sde-cond))
	 (var1 (sde-gensym-new-var "v"))
	 new)
    ;; First check some common cases.
    (cond ((eq attr '^problem-space\.name)
	   (setq sde-cond (rest sde-cond))
	   (let ((values (sde-parse-value-test-star)))
	     (while
		 (setq new    (cons '^name (cons (car values) new))
		       values (cdr values))))
	   (setq sde-parsed (nconc sde-parsed (list (cons var1 new))))
	   (list '^problem-space var1))

	  ((eq attr '^operator\.name)
	   (setq sde-cond (rest sde-cond))
	   (let ((values (sde-parse-value-test-star)))
	     (while
		 (setq new    (cons '^name (cons (car values) new))
		       values (cdr values))))
	   (setq sde-parsed (nconc sde-parsed (list (cons var1 new))))
	   (list '^operator var1))

	  (t
	   ;; Ok to use symbol-name here because the caller will
	   ;; have called `sde-dotted-p' before branching here,
	   ;; and that will have checked that the symbol is not actually
	   ;; a number.
	   (let* ((attr-str (symbol-name attr))
		  (match (string-match "\\." attr-str))
		  var2)
	     (prog1
		 ;; The first component of the dotted attribute belongs to the current
		 ;; id whose cond is being parsed, so that's what we return.
		 (list (intern (substring attr-str 0 match)) var1)
	       (setq attr-str (concat "^" (substring attr-str (1+ match))))
	       (while (setq match (string-match "\\." attr-str))
		 (setq attr (intern (substring attr-str 0 match)))
		 (setq attr-str (concat "^" (substring attr-str (1+ match))))
		 (setq var2 (sde-gensym-new-var "v"))
		 (setq sde-parsed (nconc sde-parsed (list (list var1 (cons attr var2)))))
		 (setq var1 var2))
	       ;; The last component is the one that gets the value tests.
	       (setq attr (intern attr-str))
	       (setq sde-cond (rest sde-cond))
	       (let ((values (sde-parse-value-test-star)))
		 (while
		     (setq new    (cons attr (cons (car values) new))
			   values (cdr values))))
	       (setq sde-parsed (nconc sde-parsed (list (cons var1 new))))))))))


(defun sde-parse-attr-value-tests ()
  ;; Parse "<attr_value_tests> ::= [-] ^ <attr_test> [.<attr_test>]* <value_test>*"
  ;; Algorithm:
  ;;  Deconstruct the attribute test(s), watching out for possible dotted
  ;;  notation used in the attribute name.  Store the constants and throw
  ;;  away any variables in the attribute tests.  Then parse the value
  ;;  tests, looking for the main variable or constant in the value-tests.
  ;;  Finally, add new structures to `sde-parsed', keyed by the variables.
  (if (sde-dotted-p (first sde-cond))
      (sde-parse-dotted-attr-value-tests)
    ;; Not dotted.
    (let ((vars-and-constants (sde-parse-test-1 (first sde-cond)))
	  (value-tests (sde-parse-value-test-star))
	  (pairs))
      (when vars-and-constants
	(if value-tests
	    (dolist (attr vars-and-constants)
	      ;; Ignore variable attributes.
	      (unless (sde-variable-p attr)
		(unless (eq (aref (sde-name-string attr) 0) ?^)
		  (setq attr (intern (concat "^" (sde-name-string attr)))))
		(dolist (val value-tests)
		  (setq pairs (cons val pairs))
		  (setq pairs (cons attr pairs)))))
	  ;; No value tests.
	  (dolist (attr vars-and-constants)
	    (unless (sde-variable-p attr)
	      (unless (eq (aref (sde-name-string attr) 0) ?^)
		(setq attr (intern (concat "^" (sde-name-string attr)))))
	      (setq pairs (cons nil pairs))
	      (setq pairs (cons attr pairs))))))
      pairs)))


(defsubst sde-parse-attr-value-tests-star ()
  ;; Parses tests, returns list.
  (let (front tests)
    (while (setq front (first sde-cond))
      (cond ((eq front '-)
	     (setq sde-cond (rest sde-cond)))

	    ((eq front '^)
	     ;; Assume an attribute name with a separated "^".  Massage it.
	     (setq sde-cond
		   (cons (intern (concat "^" (sde-name-string (second sde-cond))))
			 (rest (rest sde-cond))))
	     (setq tests (nconc tests (sde-parse-attr-value-tests))))

	    (t
	     (setq tests (nconc tests (sde-parse-attr-value-tests))))))
    tests))


(defsubst sde-parse-head-of-cond (first-letter-if-no-id)
  ;; Returns a list of one or more variables and constants mentioned
  ;; in the leading portion of a condition clause, i.e., the portion
  ;; "( [goal | impasse] [ <id_test> ] ".  If an id-test is found,
  ;; and it mentions more than one variable, the first variable on the
  ;; list returned is the main variable in the test.
  (let ((front (first sde-cond)))
    (cond ((memq front '(goal impasse state))
	   ;; !!! Assumes no test involving goal/impasse/state id.
	   (prog1
	       (list (first (rest sde-cond)))
	     (setq sde-cond (rest (rest sde-cond)))))
	  ((or (null sde-cond) (sde-attribute-p front))
	   ;; No id found.
	   (list (sde-gensym-new-var first-letter-if-no-id)))
	  (t
	   (sde-parse-test-1 front)))))


(defun sde-parse-conds-for-one-id (cond &optional first-letter-if-no-id)
  ;; Parses a condition.
  ;; Updates sde-parsed.
  ;; Ignore constants and add vars to sde-parsed.
  ;; Returns the list for the one id.
  (setq sde-cond cond)
  (let* ((vars-and-constants (sde-parse-head-of-cond first-letter-if-no-id))
	 (main (first vars-and-constants)))
    (dolist (sym vars-and-constants)
      (when (and (sde-variable-p sym) (not (assq sym sde-parsed)))
	;; Leave the first entry (assumed to be the goal) where it is.
	(setq sde-parsed (nconc sde-parsed (list (list sym))))))
    (if (and main (sde-variable-p main))
	(let ((var-data (assq main sde-parsed)))
	  (if var-data
	      (setcdr var-data (nconc (rest var-data)
				      (sde-parse-attr-value-tests-star))))
	  var-data))))


(defvar sde-conds nil)

(defsubst sde-parse-cond (clause)
  ;; Parses the combination
  ;;  <cond>             ::= <positive_cond> | - <positive_cond>
  ;;  <positive_cond>    ::= <conds_for_one_id> |  {  <cond>+  } 
  ;; Argument CONDS is the remaining list of conditions in the LHS.
  (cond ((eq clause '-)
	 ;; Ignore negations.
	 (setq sde-conds (rest sde-conds)))

	((eq clause '{)
	 ;; Parse "{  <cond>+  }".
	 (setq sde-conds (rest sde-conds)) ; Eat the "{".
	 (sde-parse-cond-plus)
	 (setq sde-conds (rest sde-conds))) ; Eat the "}".

	((consp clause)
	 (sde-parse-conds-for-one-id clause "g")
	 (setq sde-conds (rest sde-conds)))))


(defun sde-parse-cond-plus ()
  ;; Parses "<condition-side>   ::= <cond>+"
  ;; Argument CONDS is the remaining list of conditions in the LHS.
  (let (c)
    (while (or (consp (setq c (first sde-conds))) (memq c '(- {)))
      (sde-parse-cond c))))


;;; Grammar for RHS:
;;;
;;;   <rhs>             ::= <rhs_action>*
;;;   <rhs_action>      ::= ( variable <attr_value_make>+ ) | <function_call>
;;;   <function_call>   ::= ( <function_name> <rhs_value>* )
;;;   <function_name>   ::= sym_constant | + | -
;;;   <rhs_value>       ::= <constant> | <function_call> | variable
;;;   <constant>        ::= sym_constant | int_constant | float_constant
;;;   <attr_value_make> ::= ^ <variable_or_sym_constant> <value_make>+
;;;   <variable_or_sym_constant> ::= variable | sym_constant
;;;   <value_make>      ::= <rhs_value> <preference_specifier>*
;;;
;;;   <preference-specifier>       ::= <naturally-unary-preference>
;;;                                  | <forced-unary-preference>
;;;                                  | <binary-preference> <rhs_value>
;;;   <naturally-unary-preference> ::= + | - | ! | ~ | @
;;;   <binary-preference>          ::= > | = | < | &
;;;   <forced-unary-preference>    ::= <binary-preference> {, | ) | ^}
;;;     ;but the parser doesn't consume the ")" or "^" lexeme here


(defun sde-parse-production (form file tdata)
  (setq form (rest form))		; Skip 'sp' part.
  (unless (not (symbolp (first form)))	; Production name should be a symbol.
    ;; Store production data under production name, retain pointer to data.
    (let ((pdata (sde-add-task-production (first form) tdata)))
      (setf (sde-production-file pdata) file)
      (setq form (rest form))		; Move forward past name.
      (if (stringp (first form))	; Skip doc string if present.
	  (setq form (rest form)))
      (if (symbolp (first form))	; Skip flag if present.
	  (setq form (rest form)))
      ;; Parse condition side and action side.
      (setq sde-conds form)
      (setq sde-parsed nil)
      (sde-parse-cond-plus)
      (setf (sde-production-lhs pdata) sde-parsed))))


(defsubst sde-parse-preclean ()
  (sde-strip-multi-line-comments)
  ;; Separate certain tokens with whitespace
  (goto-char 1)
  (while (re-search-forward "\\({\\|}\\)" nil t)
    (replace-match " \\1 "))
  (goto-char 1)
  (while (search-forward "-^" nil t)
    (replace-match "- ^")))


;; There are two variants of functions for parsing files, for greater
;; efficiency.  One is used to scan recursively, and thus it pays attention
;; to load and cd statements, and the other ignores load statements but
;; builds a list of the productions encountered.  Duplicating code like this
;; is rather poor programming practice, but I wanted as much speed out of
;; this as possible.

(defun sde-parse-file (file parent tdata dir)
  ;; FILE and DIR must be expanded.
  ;; Returns FILE.
  (save-excursion
    (let ((kill-buffer-hook nil)	; For Emacs 19, for speed.
	  (buffer (create-file-buffer file)))
      (set-buffer buffer)
      (erase-buffer)
      (if (condition-case nil
	      (insert-file-contents file t)
	    (file-error nil))
	  (let ((fdata (sde-add-task-file file tdata)))
	    (setf (sde-file-load-file fdata) parent)
	    (buffer-disable-undo buffer)
	    (message "Parsing %s ..." file)
	    (sde-parse-preclean)
	    (goto-char 1)
	    ;; `read' generates an `end-of-file' error upon trying to read
	    ;; past the last form in a buffer.  Hence the condition-case.
	    (condition-case nil
		(while
		    (let ((form (read buffer)))
		      (cond ((listp form)
			     (let ((first (first form)))
			       (cond ((eq first 'sp)
				      (sde-parse-production form file tdata))
				     ((eq first 'load)
				      (sde-parse-file
				       (sde-expand-file-name (second form) dir)
				       file tdata dir))
				     ((memq first '(cd chdir))
				      (setq dir (sde-expand-file-name (second form) dir))))))
			    ((eq form 'load)
			     (sde-parse-file
			      (sde-expand-file-name (read buffer) dir)
			      file tdata dir))
			    ((memq form '(cd chdir))
			     (setq dir (sde-expand-file-name (read buffer) dir))))
		      form))
	      (invalid-read-syntax nil)
	      (end-of-file nil))))
      (set-buffer-modified-p nil)
      (kill-buffer buffer)))
  file)


(defun sde-parse-buffer-productions ()
  ;; Scans only the productions in the current buffer.
  ;; Updates the task database.
  ;; Updates the sde-buffer-productions field of sde-buffer-data.
  (let ((kill-buffer-hook nil)
	(buffer (current-buffer))
	(file   buffer-file-truename)
	(tdata  (sde-task))
	(tmpbuf (get-buffer-create " *sde scan temp*"))
	productions)
    (message "Parsing %s ..." file)
    (save-excursion
      (set-buffer tmpbuf)
      (erase-buffer)
      (insert-buffer buffer)
      (buffer-disable-undo tmpbuf)
      (sde-parse-preclean)
      (goto-char 1)
      ;; `read' generates an `end-of-file' error upon trying to read past the
      ;; last form in a buffer.  Hence the condition-case.
      (condition-case nil
	  (while
	      (let ((form (read tmpbuf)))
		(when (and (listp form) (eq (car form) 'sp))
		  (push (second form) productions) ; Record the name.
		  (sde-parse-production form file tdata))
		form))
	(invalid-read-syntax nil)
	(end-of-file nil))
      (set-buffer-modified-p nil))
    ;; Back in the original buffer.
    (setf (sde-buffer-productions sde-buffer-data) productions)
    (kill-buffer tmpbuf)))


(defun sde-parse-task (tdata &optional dir)
  ;; Scan the files of the task TDATA starting with its load file.
  ;; Optional DIR specifies the current directory.
  ;; Returns TDATA.
  (setq dir (or dir default-directory))
  (let ((file (sde-expand-file-name (sde-task-load-file tdata) dir)))
    (when (file-exists-p file)
      (sde-parse-file file file tdata (file-name-directory file))
      (setf (sde-task-scanned tdata) t)
      (setf (sde-task-modified tdata) t)
      (sde-update-data-files)))
  tdata)


(defun sde-check-task-parsed (tdata)
  ;; Verify that task TDATA has had a full scan.
  (when (null tdata)
    (let ((lfile (file-truename (sde-prompt-for-load-file))))
      (cond ((setq tdata (sde-get-file-task lfile))
	     ;; User supplied load file & it's known.  Implies the current
	     ;; file should also have been known.  Assume it's an extra.
	     (sde-add-task-file lfile tdata))

	    (t
	     ;; Load file is unknown.  Create new task, scan task files.
	     (setq tdata (sde-add-task lfile))
	     (sde-parse-task tdata)))))
  (unless (sde-task-scanned tdata)
    (sde-parse-task tdata))
  tdata)


;; For debugging purposes.

(defun sde-list-parses ()
  (interactive)
  (unless (fboundp 'cl-prettyprint)
    (load-library "cl-extra"))
  (with-output-to-temp-buffer "*productions*"
    (princ "====== Productions parsed ======")
    (mapcar (function
	     (lambda (tdata)
	       (princ "\n\nTask \"")
	       (princ (sde-task-load-file tdata))
	       (princ "\":\n")
	       (sde-maphash (function
			     (lambda (name pdata)
			       (princ "  ")
			       (princ name)
			       (princ (save-excursion
					(set-buffer (get-buffer-create "* sde tmp*"))
					(erase-buffer)
					(cl-prettyprint (sde-production-lhs pdata))
					(goto-char 1)
					(while (re-search-forward "^" nil t)
					  (replace-match "    "))
					(buffer-string)))
			       (princ "\n")))
			    (sde-task-productions tdata))))
	    sde-known-tasks)))


;;;----------------------------------------------------------------------------
;;; 4.  Recording files and buffers into the database.
;;;----------------------------------------------------------------------------

;;; As a backup measure, and so that users who set sde-inhibit-record-file-data
;;; can still get some use out of sde-find-production-by-name, SDE records each
;;; file in one global table of file names.

(defvar sde-known-files (sde-make-hash-table))

(defun sde-add-to-known-files (file)
  ;; Add FILE to the table of known files.
  (sde-puthash file sde-known-files t))

;;; Files that test true to `sde-load-files-regexp-list' are automatically
;;; considered top-level load files.  Typically these are files like
;;; .init.soar.

(defsubst sde-load-file-p (file)
  (string-match (sde-make-load-files-regexp) (file-name-nondirectory file)))

(defsubst sde-record-new-task (lfile)
  ;; Creates a new task based on load file LFILE, and parses the task files.
  (sde-parse-task (sde-add-task lfile) (file-name-directory lfile)))

(defun sde-record-file (file &optional buffer)
  ;; Record info about the given file name.
  ;; Assumes FILE is a file truename.
  ;; Optional arg BUFFER is the buffer of FILE.
  ;; Returns FILE.
  (let ((task (sde-get-file-task file)))
    (when (null task)
      ;; We don't now about this file.
      (if (sde-load-file-p file)	; Should we assume it's a load file?
	  (setq task (sde-record-new-task file))
	;; No.  Have to ask user.
	(let ((lfile (sde-expand-file-name (sde-prompt-for-load-file buffer))))
	  (cond ((null lfile)
		 ;; No load file given ==> this file should be treated as
		 ;; being alone.  Create a new task for it.
		 (setq task (sde-parse-task (sde-add-task file)))
		 (setq lfile file))

		((setq task (sde-get-file-task lfile))
		 ;; User supplied load file & it's known.  Implies the
		 ;; current file should also have been known.  Assume
		 ;; it's an extra.
		 (sde-add-task-file file task))

		(t
		 ;; Load file is unknown.  Create new task, scan task files.
		 (setq task (sde-record-new-task lfile)))))))
    ;; Update fields in the sde-buffer structure
    (when buffer
      (save-excursion
	(set-buffer buffer)
	(when (null sde-buffer-data)
	  (setq sde-buffer-data (sde-make-buffer-struct)))
	(setf (sde-buffer-task sde-buffer-data) task)
	(sde-parse-buffer-productions)))
    ;; Add to table of files used as backup for sde-find-production-by-name.
    (sde-add-to-known-files file)
    file))

(defun sde-record-buffer (buffer)
  ;; Record info about the given BUFFER.
  ;; Returns BUFFER.
  (sde-record-file (sde-buffer-file-truename buffer) buffer)
  buffer)


;;;----------------------------------------------------------------------------
;;; find-file-hook, local-write-file-hook, kill-emacs-hook
;;;----------------------------------------------------------------------------

;;; Function `sde-find-file-hook' is added to the Emacs `find-file-hooks' so
;;; that when a Soar source file is first read in, it can be recorded into
;;; the database.  When invoked, `sde-find-file-hook' also adds a local
;;; write file hook to the file's buffer, so that the database can be updated
;;; whenever the file is written out to disk.

(add-hook 'find-file-hooks 'sde-find-file-hook-fn)
(add-hook 'kill-emacs-hook 'sde-kill-emacs-hook-fn)

(defvar sde-inhibit-write-file-data nil)

(defun sde-find-file-hook-fn ()
  ;; Assumes buffer is current.
  (unless sde-inhibit-record-file-data
    (condition-case ()
	(when (memq major-mode sde-source-modes)
	  (sde-record-buffer (current-buffer))
	  (add-hook 'local-write-file-hooks 'sde-local-write-file-hook-fn))
      (error nil))))

(defun sde-local-write-file-hook-fn ()
  ;; Assumes buffer is current.
  ;; This function is added to a buffer's `local-write-file-hooks'
  ;; by a function on `find-file-hooks'.
  (condition-case err
      (when (memq major-mode sde-source-modes) ; Sanity check.
	(let ((tdata (sde-buffer-task sde-buffer-data))
	      (previous-list (sde-buffer-productions sde-buffer-data))
	      (current-list))
	  ;; Update internal database
	  (sde-parse-buffer-productions)
	  (setq current-list (sde-buffer-productions sde-buffer-data))
	  ;; Remove from the task database the productions that are no longer in
	  ;; this buffer.
	  (mapcar (function
		   (lambda (production)
		     (unless (member production current-list)
		       (sde-remove-task-production production tdata))))
		  previous-list))
	;; Must return nil for Emacs. 
	nil)
    (error
     (sde-error
      (format "Error writing file \"%s\"" (file-name-nondirectory buffer-file-name))
      (format "Encountered error while trying to write %s.\n\
This may indicate a bug in SDE.  Please use `M-x sde-feedback' to report\n\
what happened to the maintainers of SDE.  Please be specific and detailed\n\
in your email message." buffer-file-name)))))

(defun sde-kill-emacs-hook-fn ()
  ;; Runs through and saves .sde files if necessary before quitting Emacs.
  (message "Saving SDE data files...")
  (sde-update-data-files)
  (message "Saving SDE data files... Done."))

