;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-basics.el,v $
;;;; $Id: sde-basics.el,v 0.120 1994/06/22 08:50:43 hucka Exp $
;;;; 
;;;; Description       : Basic functions for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993, 1994 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;;
;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1994 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.50:      Copyright (C) 1991-1994 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; VM 5.72:        Copyright (C) 1989-1994 Kyle E. Jones
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-basics-el-version "$Revision: 0.120 $"
  "The revision number of sde-basics.el.  The complete RCS id is:
      $Id: sde-basics.el,v 0.120 1994/06/22 08:50:43 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Requirements and miscellaneous setup.
;;;; 3.  Internal constants and variables
;;;; 4.  Simple gensym.  Code based on cl.el of Emacs 18.58.
;;;; 5.  Often-used macros.
;;;; 6.  Hash table routines
;;;; 7.  Miscellaneous basic functions
;;;; 8.  Buffer and window handling
;;;; 9.  Search-related functions.
;;;; 10. Basic movement and production interpretation.
;;;; 11. Recordkeeping
;;;; 12. Error and diagnostic handling.
;;;; 13. Parenthesis handling.
;;;; 14. Indentation support.
;;;; 15. Sending, excising, etc., productions and regions of productions.
;;;; 16. Miscellaneous editing commands
;;;; 17. Comment support
;;;; 18. SDE mode
;;;; 19. Help support.
;;;; 20. Load commands for rest of SDE
;;;; 21. Functions for reporting bugs
;;;; 22. Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;; ----------------
;;; 0. Documentation
;;; ----------------
;;;
;;; WARNING: This file redefines the function `describe-mode'.  See the
;;; definition of `sde-describe-mode' below for details.
;;;
;;; This is the main file of the Soar Development Environment (SDE), an
;;; editing and debugging environment featuring special tools to help manage
;;; the complexities of writing Soar code and interacting with Soar.
;;;
;;; SDE is currently compatible with the following versions of Emacs: FSF
;;; GNU Emacs 19 and Lucid Emacs 19.  Epoch 4.2 and Emacs 18 are not
;;; supported, although they were supported at one time.
;;;
;;; This file contains the core of SDE, including basic functions, the
;;; internal database for Soar task files, and the SDE editing mode
;;; (`sde-mode').  Also contained here are the load instructions for the
;;; rest of SDE.
;;;
;;; Certain function and data definitions that depend on the version of
;;; Emacs being used are located in the following files:
;;;
;;;   FSF GNU Emacs 19:  sde-emacs19.el
;;;   Lucid Emacs 19:    sde-lemacs.el
;;;
;;; The files listed above are loaded first by this file at both compile and
;;; load time, hence they must not contain dependencies on the contents of
;;; this file.


;;;----------------------------------------------------------------------------
;;; 1.  Requirements and miscellaneous setup.
;;;     Do not modify these.
;;;----------------------------------------------------------------------------


;;;----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;----------------------------------------------------------------------------

(defconst sde-running-lemacs (and (string-match "Lucid" emacs-version) t)
  "Non-nil if running Lucid Emacs.")

(defconst sde-running-emacs19
    (and (string-match "\\`19" emacs-version) (not sde-running-lemacs))
  "Non-nil if running FSF GNU Emacs 19.")

(defconst sde-source-modes '(sde-mode)
  "List of mode names in SDE that deal with source files.")

(defconst sde-soar-modes '(sde-soar-mode sde-soar-output-mode)
  "List of mode names in SDE related to interacting with Soar.")

(defconst sde-modes (append sde-source-modes sde-soar-modes)
  "List of all modes that are related to SDE.")

(defvar sde-soar-agents nil
  "Alist of names of Soar agents and their associated buffers.
When running in single-agent mode, this variable is nil.  When running in
multi-agent mode, there are multiple agents and a separate buffer for each
agent, and the first name on the alist is the first agent created.  Format of
elements on the list is \( name .  buffer-name ).")

(defconst sde-soar-prompt-regexp "^Soar\\( agent \\([^>]+\\)\\)?> "
  "Regular expression to recognize prompts in the Soar process buffer.
If Soar is running in multi-agent mode, match-beginning 2 returns the start
of the current agent's name in the prompt string.")

;; In Emacs 18, such regexps wouldn't need the extra \( ... \) delimiters,
;; but they are needed in 19.

(defvar sde-sp-name-regexp "^[ \t]*(sp\\s +\\(\\(\\s_\\|\\sw\\)*\\)"
  "*Regexp to match the name part of a production.  
This must be a regular expression containing a grouping construct such that
SDE code can use \(match-beginning 1) and \(match-end 1) to extract the name
from within a matched string.")

(defvar sde-info-file "sde.info"
  "Name of the SDE info file.")

(defvar sde-soar-info-file "soar6-manual.info"
  "Name of the Soar Manual info file.")

(defvar sde-soar-release-notes-file nil
  "Filename of the release notes for Soar.")

(defvar sde-soar-user-notes-file nil
  "Filename of the user notes for Soar.")

(defvar sde-soar-state-string "no process"
  "String indicating the current status of Soar.  Used in mode lines.")

(defvar sde-buffer-data nil
  "Buffer-local variable storing a structure of type `sde-buffer'.
Used to record information about a buffer, such as the task associated
with it, the agent associated with, etc.")

;; Things defined in sde-soar-mode.el or elsewhere, but declared here to
;; quiet the optimizing byte compiler.

(defvar sde-header-hooks)
(defvar make-header-hooks)


;;;----------------------------------------------------------------------------
;;; 6.  Hash table routines
;;;----------------------------------------------------------------------------
;;;
;;; The CL extensions package implements hash tables, but unfortunately I have
;;; found the implementation slower in practice than the following simpler
;;; hash table code.
;;;
;;; Hash tables are implemented simply using the Emacs symbol table facility.
;;; They are equivalent to Emacs "obarrays".  

(defvar sde-hash-table-default-size 307
  "Default hash table size.  Should be a prime number.")

(defun sde-make-hash-table (&optional size)
  "Makes an obarray suitable for use as a hashtable.
SIZE, if supplied, should be a prime number."
  (make-vector (or size sde-hash-table-default-size) 0))

(defsubst sde-make-hash-key (key)
  "Returns a string hash key made from KEY."
  (if (stringp key)
      key
    (prin1-to-string key)))

(defsubst sde-puthash (name ht data)
  "Hashes the string NAME in table HT and sets its data value to DATA."
  (set (intern (sde-make-hash-key name) ht) data))

(defsubst sde-gethash (name ht)
  "Returns the value associated with the hashed string NAME in table HT.
Returns nil if NAME is not in the hash table."
  (let ((sym (intern-soft (sde-make-hash-key name) ht)))
    (and sym (symbol-value sym))))

;; There is actually no way to remove an entry from a hash table.
;; The only thing you can do is make the value nil.

(defsubst sde-remhash (name ht)
  "Nils the value associated with the hashed string NAME in table HT."
  (let ((sym (intern-soft (sde-make-hash-key name) ht)))
    (and sym (set sym nil))))

(defun sde-maphash (fun tbl)
  "Calls function FUN on each key and value in hash table TBL."
  (mapatoms (function 
	     (lambda (sym)
	       (funcall fun sym (symbol-value sym))))
	    tbl))

(defun sde-hash-table-count (ht)
  "Returns a count of the entries in hash table HT."
  (let ((count 0))
    (sde-maphash (function
		  (lambda (key val)
		    (when val
		      (setq count (1+ count)))))
		 ht)
    count))


;;;----------------------------------------------------------------------------
;;; .  Utilities for working with strings.
;;;----------------------------------------------------------------------------

;; This is bbdb-string-trim from bbdb.el.

(defsubst sde-string-trim (string)
  "Remove leading and trailing whitespace from STRING."
  (if (string-match "\\`[ \t\n]+" string)
      (setq string (substring string (match-end 0))))
  (if (string-match "[ \t\n]+\\'" string)
      (substring string 0 (match-beginning 0))
    string))

(defsubst sde-substring (string match-num &optional match-end)
  "Do `substring' on string STRING.  MATCH-NUM specifies the subexpression.
If MATCH-NUM is not followed with optional argument MATCH-END, this macro
expands into
    \(substring STRING \(match-beginning MATCH-NUM) \(match-end MATCH-NUM))
Otherwise, if optional MATCH-END is supplied, this macro becomes
    \(substring STRING \(match-beginning MATCH-NUM) \(match-end MATCH-END))"
  (if match-end
      (substring string (match-beginning match-num) (match-end match-end))
    (substring string (match-beginning match-num) (match-end match-num))))

(defsubst sde-buffer-substring (num)
  "Expands into \(buffer-substring \(match-beginning NUM) \(match-end NUM))."
  (buffer-substring (match-beginning num) (match-end num)))

(defsubst sde-unconcat (string)
  "Break up string at whitespaces into a list of strings and return the list."
  (let ((str (sde-string-trim string))
	(start 0)
	(new))
    (while (string-match "[ \t]+" str start)
      (setq new   (cons (substring str start (match-beginning 0)) new)
	    start (match-end 0)))
    (setq new (cons (substring str start) new))
    (nreverse new)))

;; This craziness is necessary when calling `message' with production names.
;; A production name can legally contain `%' characters, leading to
;; errors when such a string is passed to the Elisp `message' function.

(defsubst sde-quote-string (str)
  "Quote all instances of `%' characters in string STR and return the result."
  (let ((start 0) 
	(new ""))
    (while (string-match "%" str start)
      (setq new (concat new (substring str start (match-beginning 0)) "%%")
	    start (match-end 0)))
    (concat new (substring str start))))


;;;----------------------------------------------------------------------------
;;; Utilities for working with prefix arguments.
;;;----------------------------------------------------------------------------

(defun sde-prefix-arg-negative-p ()
  "Return non-nil if `current-prefix-arg' is a negative prefix."
  (and current-prefix-arg
       (or (symbolp current-prefix-arg)
	   (< (prefix-numeric-value current-prefix-arg) 0))))

(defun sde-prefix-arg-value ()
  "Return the absolute value of `current-prefix-arg' as a number, or nil."
  (and current-prefix-arg
       (not (symbolp current-prefix-arg))
       (let ((value (prefix-numeric-value current-prefix-arg)))
	 (if (< value 0)
	     (- value)
	     value))))


;;;----------------------------------------------------------------------------
;;; Miscellaneous utility functions and macros.
;;;----------------------------------------------------------------------------

(defun sde-key-for-command (cmd)
  "Return a string description of the binding for function CMD.
If there is no key binding for the command, returns the string \"M-x cmd\"."
  (let ((key (key-description (sde-where-is-internal cmd nil 'non-ascii t))))
    (if (and key (not (string= key "")))
	key
      (concat "M-x " (symbol-name cmd)))))

(defun sde-production-name-p (string)
  "Return non-nil if STRING is matched by `sde-production-name-test-regexp'."
  (and string
       (let ((match-data (match-data)))
	 (prog1
	     (string-match sde-production-name-test-regexp string)
	   (store-match-data match-data)))))

(defsubst sde-strip-multi-line-comments ()
  ;; Nuke multi line comments in this buffer, because they interfere
  ;; with scanning for things.
  (goto-char (point-min))
  (let (start end)
    ;; Make sure that both comment delimiters are found.
    (while (setq start (search-forward "#|" nil t)
		 end   (search-forward "|#" nil t))
      (delete-region (- start 2) end))))

(defun sde-search-buffer (buffer regexp &optional start)
  "Search forward in BUFFER for REGEXP, starting from the top.
Optional arg START specifies a starting position.  Returns the value of point
where pattern found, or nil."
  (save-excursion
    (set-buffer buffer)
    (goto-char (or start 1))
    (re-search-forward regexp nil t)))

(defun sde-search-file (filename regexp &optional inhibit-msg)
  "Read FILENAME into Emacs and search for REGEXP.
This reads the raw file, not setting its mode or reading its local vars.
FILENAME must not already be in a buffer somewhere.  Prints a message
unless optional argument INHIBIT-MSG is non-nil."
  (save-excursion
    (let ((kill-buffer-hook nil)	; For Emacs 19, for speed.
	  (buffer (create-file-buffer filename)))
      (set-buffer buffer)
      (erase-buffer)
      (prog1				; Return the result of next form.
	  (if (condition-case ()
		  (insert-file-contents filename t)
		(file-error nil))
	      (progn
		(or inhibit-msg (message "Scanning file %s..." buffer-file-name))
		;; Return the result of the search.
		(re-search-forward regexp nil t))
	    nil)
	(kill-buffer buffer)))))

(defun sde-skip-chars (end)
  "Skip past whitespace, comments, backslashed characters and strings in the
current buffer as long as you are before END.  This does move the point."
  (when (< (point) end)
    (let ((comment (and comment-start (string-to-char comment-start)))
	  (done nil)
	  char)
      (while (and (< (point) end)
		  (not done))
	(skip-chars-forward "\n\t " end)
	(setq char (char-after (point)))
	(cond ((eq char ?\")
	       (forward-sexp))
	      ((eq char comment)
	       (forward-char)
	       (skip-chars-forward "^\n" end))
	      ((eq char ?\\)
	       (forward-char 2))
	      (t
	       (setq done t)))))))

;; Modified from lisp-end-defun-text from ilisp-ext.el

(defun sde-end-sp-text (&optional at-start) 
  ;; Go the end of the text associated with the current sp and return
  ;; point.  The end is the last character before whitespace leading to
  ;; a left paren or ;;; at the left margin unless it is in a string.
  (unless at-start
    (sde-beginning-of-production))
  (let ((point (point))
	(boundary (sde-find-next-start))
	(final (save-excursion
		 (condition-case ()
		     (progn (forward-sexp) (point))
		   (error (point-max))))))
    ;; Find the next line starting at the left margin and then check
    ;; to see if it is in a string. 
    (while (progn
	     (skip-chars-forward "^\"" boundary) ;To the next string
	     (unless (= (point) boundary)	
	       (let ((string-boundary	;Start of next defun
		      (save-excursion
			(if (re-search-forward "^\(\\|^;;;" nil t)
			    (match-beginning 0)
			  (point-max)))))
		 (if (condition-case ()
			 (progn (forward-sexp) t)
		       (error (goto-char string-boundary) nil))
		     (if (>= (point) boundary)
			 ;; Boundary was in string
			 (if (> (point) string-boundary)
			     (progn	;String ended in next defun
			       (goto-char string-boundary)
			       nil)
			   (if (> (setq boundary
					(sde-find-next-start))
				  final)
			       ;; Normal defun
			       (progn (goto-char final) nil)
			     t))
		       t)
		   ;; Unclosed string
		   nil)))))
    (re-search-backward  "^[^; \t\n]\\|^[^;\n][ \t]*[^ \t\n]" point t)
    (end-of-line)
    (skip-chars-backward " \t")
    (if (< (point) point)
	(goto-char point)
      (when (save-excursion
	      (let ((point (point)))
		(beginning-of-line)
		(and comment-start (search-forward comment-start point t))))
	(next-line 1)
	(sde-indent-line))
      (point))))

(defun sde-find-next-start ()
  ;; Find the start of the next line at the left margin that starts with
  ;; something other than whitespace, a -, a \), three or more semicolons, or
  ;; a \( not followed by <, and return point.
  (if (eobp)
      (point-max)
    (save-excursion
      (forward-char)
      (if (re-search-forward "^\\(\\(;;;\\)\\|\\(\([^<]\\)\\|\\([^- ^( \t\n\);]\\)\\)" nil t)
	  (match-beginning 0)
	(point-max)))))

;; Modified version of make-help-screen, originally by Lynn Slater, now
;; part of Emacs 19.  Modified so that keys such as C-v are not intercepted.
;; It's stupid to have to do this.

(defmacro sde-make-help-screen (fname help-line help-text helped-map)
  "Constructs function FNAME that when invoked shows HELP-LINE and if a help
character is requested, shows HELP-TEXT. The user is prompted for a character
from the HELPED-MAP and the corresponding interactive function is executed."
  (` (defun (, fname) ()
	   (, help-text)
	   (interactive)
	   (let ((line-prompt (substitute-command-keys (, help-line))))
	     (message line-prompt)
	     (let ((char (read-char))
		   config)
	       (unwind-protect
		   (progn
		     (if (or (char-equal char ??) (char-equal char help-char))
			 (progn
			   (setq config (current-window-configuration))
			   (switch-to-buffer-other-window "*Help*")
			   (erase-buffer)
			   (set-window-hscroll (selected-window) 0)
			   (insert (documentation (quote (, fname))))
			   (goto-char (point-min))
			   (while (memq char (cons help-char '(?? ?\ ?\177)))
			     (if (char-equal char ?\ ) 
				 (scroll-up))
			     (if (char-equal char ?\177)
				 (scroll-down))
			     (message "%s%s: "
				      line-prompt
				      (if (pos-visible-in-window-p (point-max))
					  "" " or Space to scroll"))
			     (let ((cursor-in-echo-area t))
			       (setq char (read-char))))))
		     (let ((defn (cdr (assq (if (integerp char)
						(downcase char)
						char)
					    (, helped-map)))))
		       (if defn
			   (if (keymapp defn)
			       (error "sorry, this command cannot be run from the help screen.  Start over.")
			     (if config
				 (progn
				   (set-window-configuration config)
				   (setq config nil)))
			     (call-interactively defn))
			 (if (listp char) ; Emacs 19
			     (setq unread-command-events
				   (cons char unread-command-events)
				   config nil)
			   (ding)))))
		 (if config
		     (set-window-configuration config))))))))


;;;----------------------------------------------------------------------------
;;; .  Utilities for working with files and directories
;;;----------------------------------------------------------------------------
    
;; Things for recognizing Soar files.  Note that the file-types-regexp-list
;; and load-files-regexp-list are concatenated into full regexps at run time,
;; rather than being cached, so that if the values are changed at run time or
;; in the user's .emacs file, the changes will be seen by SDE code using
;; these variables.  (Whether this is a useful things is not clear yet.)

(defun sde-make-file-types-regexp ()
  "Return a regexp for matching files with `sde-file-types-regexp-list'."
  (mapconcat 'identity sde-file-types-regexp-list "\\|"))

(defun sde-make-load-files-regexp ()
  "Return a regexp for matching files with `sde-load-files-regexp-list'."  
  (mapconcat 'identity sde-load-files-regexp-list "\\|"))

;; General things.

(defsubst sde-find-file-noselect (file)
  (let ((default-major-mode 'fundamental-mode))
    (find-file-noselect file)))

;; Stupid Emacs 19.25 expand-file-name sometimes leaves a leading "//".

(defsubst sde-expand-file-name (file &optional dir)
  (substitute-in-file-name (expand-file-name file dir)))

;; This is kind of ugly.  It won't cons much, and it's not recursive, but
;; it's probably inefficient nevertheless.  Is it better to call substring
;; repeatedly, or walk down the strings letter-by-letter?

(defun sde-relative-pathname (path directory)
  "Make PATH relative to DIRECTORY."
  (setq path (expand-file-name path directory))
  (let* ((dir (expand-file-name directory))
	 (pindex 0)
	 (dindex 0)
	 (plength (length path))
	 (dlength (length dir))
	 (result ""))
    (while (and (< pindex plength)
		(< dindex dlength)
		(char-equal (aref path pindex) (aref dir dindex)))
      (setq pindex (1+ pindex)
	    dindex (1+ dindex)))
    ;; Now at the char where they're different.
    ;; Count how many dirs it takes to get from DIR,
    ;; then append what it takes to get to the PATH.
    (setq dindex (1+ dindex))
    (while (< dindex dlength)
      (if (char-equal (aref dir dindex) ?/)
	  (setq result (concat "../" result)))
      (setq dindex (1+ dindex)))
    ;; There may be 0 or at least 2 slashes.
    (if (string-equal "" result)
	(setq result "./"))
    (concat result (substring path pindex))))

(defsubst sde-directory-parent (dir)
  "Returns the parent directory of DIR.
The parent of \"/\" is taken to be nil for practical reasons."
  (if (string= dir "/")
      nil
    (file-name-directory (directory-file-name dir))))

(defun sde-cd (dir)
  "Make DIR become the current buffer's default directory.  
This function is like Emacs's `cd' but does its work silently."
  (interactive "DChange default directory: ")
  (setq dir (expand-file-name dir))
  (unless (eq system-type 'vax-vms)
    (setq dir (file-name-as-directory dir)))
  (if (file-directory-p dir)
      (setq default-directory dir)
    (sde-error (format "\"%s\" is not a directory" dir)
	       (format "Attempt to cd to \"%s\"" dir))))


;;;----------------------------------------------------------------------------
;;; 8.  Buffer and window handling
;;;----------------------------------------------------------------------------

;;; NOTE: a large number of the sde-xxx functions for window handling are
;;; defined either in sde-emacs19.el and sde-lemacs.el.  This permits handling
;;; the different names ("frame" vs "screen") with functions defined at load
;;; time instead of putting a lot of conditionals all over the place.

(defsubst sde-get-buffer-create (name)
  "Create buffer named NAME.
Like `get-buffer-create', but binds `default-major-mode' to `fundamental-mode'.
This is a hack necessary to counteract a problem for users who set their
default major mode to `text-mode' or other.  It prevents Emacs from running
`text-mode-hook' when a buffer is created."
  (let ((default-major-mode 'fundamental-mode))
    (get-buffer-create name)))

(defsubst sde-buffer-file-name (buffer)
  "Return the value of variable `buffer-file-name' in BUFFER."
  (save-excursion (set-buffer buffer) buffer-file-name))

(defsubst sde-buffer-file-truename (buffer)
  "Return the value of variable `buffer-file-truename' in BUFFER."
  (save-excursion (set-buffer buffer) buffer-file-truename))

(defsubst sde-buffer-mode (buffer)
  "Return the value of variable `major-mode' in BUFFER."
  (save-excursion (set-buffer buffer) major-mode))

(defsubst sde-buffer-directory (buffer)
  "Return the value of `default-directory' in BUFFER."
  (save-excursion (set-buffer buffer) default-directory))

(defsubst sde-buffer-exists-p (buffer)
  "Return non-nil if BUFFER exists and is not killed.
BUFFER can be a buffer or a buffer name."
  (if (stringp buffer)
      (and (bufferp (get-buffer buffer))
	   (buffer-name (get-buffer buffer)))
    (and (bufferp buffer)
	 (buffer-name buffer))))

(defun sde-create-buffer-frame (buffer param &rest args)
  "Create new frame/screen for BUFFER with parameters PARAM.
Optional third argument VISIBILITY determines whether the window is shown to
the user immediately\; the three possible values are `t' for visible, `nil'
for invisible, and `icon' for making the new frame iconified.  Default: t.
Optional fourth argument DEDICATED determines whether the frame is dedicated
to showing BUFFER's window\; `t' means yes, `nil' (the default) means no.
Returns the newly created frame unless an error occurred.

The argument PARAM is an alist specifying frame/screen parameters.  Any
parameters not mentioned in ALIST defaults according to the value of the
variable `default-frame-alist' in Emacs 19\; parameters not specified there
default from the standard X defaults file and X resources."
  (when (bufferp buffer)
    (message "Creating %s; one moment please..." (cond (sde-running-lemacs
							"screen")
						       (sde-running-emacs19
							"frame")))
    (let* ((orig-frame (sde-selected-frame))
	   (visibility (if (consp args) (car args) t))
	   (dedicated (if (consp args) (car (cdr args)) nil))
	   (frame (sde-make-frame
		   buffer (append (list (cons 'visibility visibility)) param))))
      (unwind-protect
	  (progn
	    (sde-select-frame frame)
	    (delete-other-windows)
	    (switch-to-buffer buffer)
	    (sde-set-window-dedicated-p (sde-frame-root-window frame) dedicated)
	    (when visibility
	      (sde-raise-frame frame))
	    frame)
	(sde-select-frame orig-frame)))))

(defun sde-delete-buffer-frame (buffer)
  "Delete frames containing BUFFER.  All existing frames are examined.
Frames are only deleted if they contain one window and that window
is showing the given BUFFER."
  (let ((orig-frame (sde-selected-frame))
	(frames (sde-frame-list)))
    (when (cdr frames)			; If there's at least one frame.
      (unwind-protect
	  (mapcar (function
		   (lambda (frame)
		     (sde-select-frame frame)
		     (if (and (one-window-p)
			      (eq (window-buffer (selected-window)) buffer))
			 (sde-delete-frame frame))))
		  (sde-frame-list))
	(if (sde-frame-live-p orig-frame)
	    (sde-select-frame orig-frame))))))

(defun sde-show-frame (frame)
  "Make the given FRAME visible and raised if it is not already.
Returns FRAME."
  (when (sde-frame-p frame)
    (sde-raise-frame (sde-make-frame-visible frame)))
  frame)

;; sde-show-buffer is for buffers that are not special, i.e., not the
;; *output* and *error* buffers.  For those there are special functions in
;; sde-soar-mode.el.

(defun sde-show-buffer (buffer &optional switch here)
  "Display BUFFER, preferring a window in which it is already showing.  
If optional arg SWITCH is non-nil, switch to the buffer after displaying it.
If optional arg HERE is non-nil, show the buffer in the selected window
unless it is already being shown in another window.  If BUFFER is already
being shown in a window somewhere, make that window and frame visible if
necessary, and switch to it if SWITCH.  This function returns the window
showing BUFFER."
  (when (bufferp buffer)
    (let ((window (sde-buffer-window buffer t)))
      (cond (window
	     ;; There is a window currently displaying the buffer.  
	     ;; Make sure it's visible and switch to it if desired.
	     (when (or sde-running-lemacs sde-running-emacs19)
	       (sde-show-frame (sde-buffer-frame buffer)))
	     (when switch
	       (select-window window)
	       (set-buffer buffer))
	     window)
	    ;; There is no window displaying the buffer.
	    (here
	     (switch-to-buffer buffer)
	     (selected-window))
	    (switch
	     (pop-to-buffer buffer)
	     (set-buffer buffer)
	     (get-buffer-window buffer))
	    (t
	     (display-buffer buffer))))))

(defun sde-show-buffer-at-point (buffer point)
  "Display BUFFER with the window centered at line containing POINT.
This function will prefer a window in which BUFFER is already showing.
Returns the window showing BUFFER."
  (unless point
    (error "sde-show-buffer-at-point:  null location."))
  (sde-pop-to-buffer buffer)
  (goto-char point)
  (beginning-of-line)
  (recenter '(center))
  (get-buffer-window buffer))

(defvar sde-last-buffer nil
  "The last used Soar buffer.")

(defun sde-pop-to-buffer (buffer)
  "Display BUFFER in a different window and switch to that window.
This function sets `sde-last-buffer' to the current buffer.
Returns the window showing BUFFER."
  (unless (equal (current-buffer) sde-last-buffer)
    (setq sde-last-buffer (current-buffer)))
  (when (sde-buffer-exists-p buffer)
    (sde-show-buffer buffer t)))

(defun sde-get-sde-buffers ()
  "Return a list of buffers that are SDE source mode buffers."
  (let ((buffers (buffer-list))
	result)
    (while buffers
      (if (memq (sde-buffer-mode (car buffers)) sde-source-modes)
	  (push (car buffers) result))
      (setq buffers (cdr buffers)))
    (nreverse result)))

(defun sde-count-buffer-lines (buffer)
  "Count number of lines in BUFFER.
This version is simpler and somewhat faster than the standard Emacs
function, but is also potentially less accurate."
  (save-excursion			; This counts the number of lines in
    (set-buffer buffer)			;  buffer.  This is what Emacs'
    (goto-char (point-min))		;  `count-lines' essentially does.
    (- (buffer-size) (forward-line (buffer-size)))))

(defun sde-update-mode-lines ()
  ;; Updates mode lines.  Hackacious.
  (save-window-excursion
    (set-buffer-modified-p (buffer-modified-p))
    ;; Unless the following kind of hack is made, mode lines dont't seem to
    ;; get updated properly in 18.58.  Bug in Emacs?
    (select-window (selected-window))
    (sit-for 0)))			; Update mode lines.

(defun sde-get-mouse-frame ()
  "Returns the frame in which the mouse cursor is located."
  (first (mouse-position)))

;; Hacked from `save-some-buffers' from files.el in Emacs 19.22.
;; I don't fully understand what this is doing, but it seems to work as
;; desired.

(defvar view-exit-action)		; Declaration for the compiler.

(defun sde-save-sde-buffers (&optional buffers no-questions exiting)
  "Saves modified SDE source file-visiting buffers.
Optional argument BUFFERS specifies the list of buffers to use\; default
is all buffers that are in an SDE source mode.  Asks user about each one.
Optional second argument NO-QUESTIONS non-`nil' means save all with no
questions.  Optional second argument EXITING means ask about certain non-file
buffers as well as about file buffers."
  (interactive "P")
  (save-window-excursion
    (map-y-or-n-p
     (function
      (lambda (buffer)
	(and (buffer-modified-p buffer)
	     (or
	      (buffer-file-name buffer)
	      (and exiting
		   (progn
		     (set-buffer buffer)
		     (and buffer-offer-save (> (buffer-size) 0)))))
	     (if no-questions
		 t
	       (if (buffer-file-name buffer)
		   (format "Save file %s? "
			   (buffer-file-name buffer))
		 (format "Save buffer %s? "
			 (buffer-name buffer)))))))
     (function
      (lambda (buffer)
	(set-buffer buffer)
	(save-buffer)))
     (or buffers (sde-get-sde-buffers))
     '("buffer" "buffers" "save")
     (list (list ?\C-r (lambda (buf)
			 (view-buffer buf)
			 (setq view-exit-action
			       '(lambda (ignore)
				  (exit-recursive-edit)))
			 (recursive-edit)
			 ;; Return nil to ask about BUF again.
			 nil)
		 "display the current buffer"))
     )))


;;;----------------------------------------------------------------------------
;;; 12. Utilities for highlighting.
;;;----------------------------------------------------------------------------

(defvar sde-highlight-overlay nil)

(defun sde-highlight-region (start end)
  "Highlight the region between START and END."
  (unless sde-highlight-overlay
    (setq sde-highlight-overlay (make-overlay start end))
    (overlay-put sde-highlight-overlay 'face 'highlight))
  (move-overlay sde-highlight-overlay start end (current-buffer)))

(defun sde-dehighlight-region ()
  (when sde-highlight-overlay
    (delete-overlay sde-highlight-overlay)
    (setq sde-highlight-overlay nil)))


;;;----------------------------------------------------------------------------
;;; 12. Message window handling.
;;;----------------------------------------------------------------------------

;; This implements a facility for creating small windows for displaying brief
;; messages.  It borrows some ideas from Chris McConnell's Popper facility.

(defvar sde-message-buffer nil
  "The buffer most recently shown in a message window."  )

(defvar sde-message-default-dismiss-function 'sde-dismiss-message
  "Default function for dismissing the message window.")

(defvar sde-message-default-position-function 'sde-message-position-function
  "Default function for positioning the text in the message window.")

(defun sde-message-position-function (window)
  ;; Default function for position the message within a message buffer.
  ;; This just moves the window to show the first character.
  (set-window-start window (point-min))
  (set-window-point window (point-min)))

(defun sde-show-message (buffer)
  ;; Splits the currently selected window, displays BUFFER in the top-most of
  ;; the two resulting windows, resizes that window so that it only shows as
  ;; many lines as are in BUFFER, and finally tries to minimize the amount of
  ;; screen motion in the second window.  If there is already a message
  ;; window anywhere on any frame, this removes it from there and displays a
  ;; message window on the current frame.  Returns the window showing BUFFER.
  (when sde-message-buffer
    (sde-dismiss-message))
  (let* ((old-window (selected-window))
	 (dedicated (window-dedicated-p old-window))
	 (start (window-start))		; Pos in current window.
	 ;; Need get key binding now, in the current buffer.
	 (key (sde-key-for-command sde-message-default-dismiss-function))
	 ;; Rebind this temporarily to allow smaller window.  For FSF 19.
	 (window-min-height 3)
	 height window)
    (setq sde-message-buffer buffer)
    ;; Regardless of whether it's dedicated, undedicate it.
    ;; This is so that we can show error messages in the soar output buffer.
    (set-window-dedicated-p old-window nil)
    ;; Figure out how many lines we need to show of the buffer.
    (save-excursion
      (set-buffer buffer)
      (save-restriction
	(widen)
	(goto-char (point-max))
	(narrow-to-page)
	;; Add one for the mode line.
	(setq height (1+ (sde-count-buffer-lines buffer)))))
    ;; Split window creates two windows on the same buffer and leaves the
    ;; top-most one selected.  That's where we want to display our buffer.
    (split-window nil (max height window-min-height))
    (setq window (selected-window))
    (set-window-buffer window buffer)
    (set-buffer buffer)			; Go to the buffer.
    (sde-soar-mode-internal)		; Give it a reasonable mode.
    (funcall sde-message-default-position-function window)
    (setq mode-line-buffer-identification '("%9b")
	  mode-line-format 
	  (list "" 'mode-line-modified 'mode-line-buffer-identification
		(format " [`%s' to dismiss]" key)
		"   %[("
		'mode-name 'minor-mode-alist
		")%n%] --"
		'(-3 . "%p") "-%-"))
    (setq old-window (next-window window 'no))
    (select-window old-window)
    ;; Move window so that just the lines at top are covered, instead of
    ;; letting Emacs jerk the contents around.  Code from popper.el.
    (save-excursion
      (set-window-start old-window start)
      (move-to-window-line (window-height window))
      (set-window-start old-window (point)))
    ;; Reset things.
    (set-window-dedicated-p old-window dedicated)
    window))

;; Windows in an Emacs frame have a certain ordering.  If there are no
;; horizontally split windows, then `next-window' returns the window downward
;; from the given window, and `previous-window' returns the window upward
;; from the given window.  In FSF 19, `delete-window' gives lines back to the
;; window upward from the deleted window.

(defun sde-dismiss-message ()
  "Hide the message buffer window."
  (interactive)
  (when (and sde-message-buffer (sde-buffer-exists-p sde-message-buffer))
    (let ((window (sde-buffer-window sde-message-buffer t))
	  (old-w (selected-window)))
      (when (window-live-p window)
	(unwind-protect
	    (let* ((height (window-height window))
		   (next (next-window window 'no))
		   (prev (previous-window window)))
	      (bury-buffer sde-message-buffer)
	      ;; Select previous window so lines are given to the next one.
	      ;; The "next one" should the one that was split for the error.
	      (select-window prev)
	      (delete-window window)
	      (unless (or (coordinates-in-window-p '(0 . 1) next)
			  (one-window-p))
		(shrink-window height))
	      (select-window next)
	      (scroll-down height))
	  (select-window old-w)))))
  (setq sde-message-buffer nil))
    

;;;----------------------------------------------------------------------------
;;; 12. Error logging and display handling.
;;;----------------------------------------------------------------------------

;;; This error facility was inspired by dired-log of dired.el from Emacs
;;; 19.24, but is actually completely different.  This implementation has
;;; more features and implements a different program interface, and unlike
;;; dired's facility, is designed to allow `sde-error' to be called only once
;;; for a given error situation.  I believe this implementation produces much
;;; more readable log messages.
;;;
;;; Format of messages: each message looks like this:
;;; 
;;;    ^L
;;;    <current-time-string>
;;;    Command: <command-name>
;;;      Error: <message>
;;;
;;;    Text of log message.
;;;

(defconst sde-error-buffer-name "*errors*")

(defun sde-error (summary explanation &optional command view-only)
  (let ((old-buffer (current-buffer))
	(buffer (get-buffer-create sde-error-buffer-name))
	(why-key (sde-key-for-command 'sde-why)))
    (unwind-protect			; Make sure to reset buffer.
	(progn
	  (set-buffer buffer)
	  (goto-char (point-max))
	  (let ((buffer-read-only nil))
	    (insert "\n\f\n" (current-time-string) "\n")
	    (sde-insert-faced 'italic "Command: ")
	    (sde-insert-faced 'default (format "%s\n" (or command this-command)))
	    (sde-insert-faced 'italic "  Error: ")
	    (sde-insert-faced 'bold (sde-string-trim summary))
	    (insert "\n\n")
	    (sde-insert-faced 'default (sde-string-trim explanation) "\n")))
      (set-buffer old-buffer)
      (if view-only
	  ;; View it right away, don't signal an error, and don't show
	  ;; message in the echo area.
	  (progn
	    (sde-why)
	    (beep))
	;; Tell user about it, signal an error, but don't view the log now.
	(error "%s--type `%s' for details" summary why-key)))))
  
;; Things for viewing.

(defun sde-error-message-position-function (window)
  (let ((old-window (selected-window)))
    (unwind-protect
	(progn
	  (select-window window)
	  (goto-char (point-max))	
	  (backward-page)		; Move to beginning of last message.
	  (forward-line 2)		; Skip the ^L and the time string.
	  (set-window-start window (point)))
      (select-window old-window))))

(defun sde-why ()
  "Toggle viewing of last error message from SDE or Soar and its explanation.
The last message is displayed at the bottom of the message buffer.
Use \\[backward-page] to move backward to previous messages."
  (interactive)
  (let ((buffer (get-buffer sde-error-buffer-name)))
    (if (and (bufferp buffer) (window-live-p (sde-buffer-window buffer)))
	;; It's already showing; hide it.
	(sde-dismiss-message)
      ;; Not showing, so show it.
      (let ((sde-message-default-position-function 'sde-error-message-position-function)
	    (sde-message-default-dismiss-function 'sde-why))	     
	(sde-show-message buffer)))))

;;; Functions for common error situations.

(defmacro sde-error-unless-site-var-set (var)
  (` (unless (symbol-value (, var))
       (sde-error
	(format "Variable `%s' not set" (, var))
	(format "Variable `%s' should have been set to an appropriate\n\
value by your SDE site maintainer, but it was not.  Contact your maintainer." 
		(, var))))))


;;;----------------------------------------------------------------------------
;;; 10. Basic movement and production interpretation.
;;;----------------------------------------------------------------------------

;; It turns out that `scan-sexps' and related functions don't look at the
;; actual characters in matching delimiters -- they just count any open/close
;; delimiters.  That's unbelievable.  It means they will treat "( foo ]" as a
;; balanced expression.  That can be deadly when trying to extract
;; productions from buffers when the productions have unbalanced delimiters
;; (e.g., stray braces).  The hack below works around the problem.  It's
;; needed by `sde-end-of-production', but not `sde-beginning-of-production', because the
;; latter only searches backward for a particular string and ever actually
;; calls `forward-sexp' or other functions that rely on `scan-sexps'.  The
;; idea is to completely ignore [, ], {, and } characters, and only look at
;; parentheses.  That way, if a production contains a stray '{', etc., SDE
;; will still be able to extract properly, and Soar will get a chance to
;; signal a syntax error.  Without this, stray { and other characters in
;; productions really screw up SDE upon a `sde-send-production'.

(defmacro sde-with-hacked-delimiter-syntax (&rest forms)
  ;; Temporarily reset the syntax of '{' '}' and '[' ']' so that they
  ;; are considered whitespace, execute FORMS, and reset the syntax.
  (let ((otable (gensym "SDE"))
	(table (gensym "SDE")))
    (` (let (((, otable) (, (copy-syntax-table (syntax-table)))))
	 (unwind-protect
	     (let (((, table) (syntax-table)))
	       (modify-syntax-entry ?\{ " " (, table))
	       (modify-syntax-entry ?\} " " (, table))
	       (modify-syntax-entry ?\[ " " (, table))
	       (modify-syntax-entry ?\] " " (, table))
	       (,@ forms))
	   (set-syntax-table (, otable)))))))


;; Modified from Emacs 19's beginning-of-defun from lisp.el.  

(defun sde-beginning-of-production (&optional count stay)
  "Move backward to next beginning of an sp form and also return point.
Optional first argument COUNT means do it that many times.
Optional second argument STAY means don't move if already at the beginning
of an sp form.  Returns point unless search stops due to end of buffer."
  (interactive "p")
  (when (and count (< count 0))
    (forward-char 1))
  (unless (and stay (looking-at "^\\s(sp"))
    (when (re-search-backward (if (memq major-mode sde-soar-modes)
				  (concat "^\\s(\\|"
					  "\\(" sde-soar-prompt-regexp "\\)\\s *\\s(sp")
				"^\\s(sp")
			      nil 'move (or count 1))
      ;; This shoudn't be necessary, but if the point is on the opening paren
      ;; and you do the re-search-backward, point ends up moved by one char
      ;; (at least in 19.24).  Argh.
      (goto-char (match-beginning 0))))
  (point))


;; Modified from Emacs 19's end-of-defun.  I don't really understand the
;; point of all this code, but it currently works.  (Yeah, I know.)

(defun sde-end-of-production (&optional arg)
  "Move forward to next end of an sp form.  With arg, do it that many times.
Negative argument -N means move back to Nth preceding end of sp.

An end of an sp occurs right after the close-parenthesis that matches
the open-parenthesis that starts an sp; see `sde-beginning-of-production'.

Returns point unless an error occurs."
  (interactive "p")
  (when (or (null arg) (= arg 0))
    (setq arg 1))
  (sde-with-hacked-delimiter-syntax
      (let ((first t))
	(while (and (> arg 0) (< (point) (point-max)))
	  (let ((pos (point)))
	    (while (progn
		     (unless (and first
				  (progn
				    (forward-char 1)
				    (/= (sde-beginning-of-production 1) pos)))
		       (or (bobp) (forward-char -1))
		       (sde-beginning-of-production -1))
		     (setq first nil)
		     (forward-list 1)
		     (skip-chars-forward " \t")
		     (when (looking-at "\\s<\\|\n")
		       (forward-line 1))
		     (<= (point) pos))))
	  (setq arg (1- arg)))
	(while (< arg 0)
	  (let ((pos (point)))
	    (sde-beginning-of-production 1)
	    (forward-sexp 1)
	    (forward-line 1)
	    (when (>= (point) pos)
	      (if (/= (sde-beginning-of-production 2) pos)
		  (progn
		    (forward-list 1)
		    (skip-chars-forward " \t")
		    (if (looking-at "\\s<\\|\n")
			(forward-line 1)))
		(goto-char (point-min)))))
	  (setq arg (1+ arg)))
	(point))))    


(defun sde-production-body ()
  "Return as a string the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil."
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-production 1 t))
	  (sp-ending (sde-end-of-production)))
      (if (and (>= pt sp-beginning) (<= pt sp-ending))
	  (buffer-substring sp-beginning sp-ending)
	nil))))


(defun sde-production-name (&optional body)
  "Extract the name of the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil.  

Algorithm: store current point, scan backward for the beginning of an sp
form, then from there, scan forward for the end of that form.  If point is
within these bounds, extract the name of the production, else return nil.  "
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-production 1 t))
	  (sp-ending (sde-end-of-production)))
      (when (and (>= pt sp-beginning) (<= pt sp-ending))
	(goto-char sp-beginning)
	(when (looking-at sde-sp-name-regexp)
	  (forward-char 4)
	  (buffer-substring (point) (progn (forward-sexp 1) (point))))))))


(defun sde-region-name (start end)
  "Return a descriptive name for a region of productions or other sexps.
Format of the name is \"From START-SP to END-SP\", where START-SP and END-SP
are the names of the productions at the beginning and end." 
  (let (from to)
    (save-excursion
      (goto-char start)
      (sde-skip-chars end)
      (if (looking-at sde-sp-name-regexp)
	  (setq from (buffer-substring (match-beginning 1) (match-end 1)))
	  (setq from (buffer-substring (point) (progn (end-of-line) (point)))))
      (goto-char end)
      ;; See if we're in the blank space or comments following some expression
      (re-search-backward "^(" nil 'move)
      (forward-sexp 1)
      (sde-skip-chars end)
      (if (= (point) end)
	  (progn
	    (re-search-backward "^(" nil 'move)
	    (if (looking-at sde-sp-name-regexp)
		(setq to (buffer-substring (match-beginning 1) (match-end 1)))
		(setq to (buffer-substring (point) (progn (end-of-line) (point))))))
	  ;; Can't recognize where we are, so punt.
	  (progn
	    (beginning-of-line)
	    (setq to (buffer-substring (point) (progn (end-of-line) (point))))))
      (setq from (sde-string-trim from)
	    to   (sde-string-trim to))
      (if (string-equal from to)
	  from
	  (concat "From " from " to " to)))))


;; Modified from Emacs' mark-defun.

(defun sde-mark-production ()
  "Put mark at end of sp form, point at beginning."
  (interactive)
  (push-mark (point))
  (sde-end-of-production)
  (push-mark (point))
  (sde-beginning-of-production)
  (re-search-backward "^\n" (- (point) 1) t))


;;;----------------------------------------------------------------------------
;;; 10. Extracting symbols and other objects from buffers.
;;;----------------------------------------------------------------------------
;;;
;;; Soar 6.1 allows the following to be used as part of a production name:
;;;     alphanumeric characters
;;;     $ % & * - _ = + < > / : ? 
;;;
;;; The characters < and >, if appearing in a production name, cannot appear
;;; themselves; the name must have at least one alphanumeric character on either
;;; side of the < or >.   If the string "<>" appears in a production name, it
;;; must be flanked by an alphanumeric character on each side, otherwise Soar
;;; will not accept it as a production name.
;;;
;;; Soar stops reading a production name if it encounters one of the following:
;;;     @ ~  {   }  ! ^ ( ) . , # \ ' 
;;;
;;; The | character begins and ends a multi-line comment.
;;; The " character begins and ends a string.
;;;
;;; Extracting symbols is most efficiently done by using the trick of
;;; temporarily changing the syntax table in the buffer so that sexp scanning
;;; can be used effectively.  The ideas here are based on the completion.el
;;; package from J. Salem & B. Kahle at Thinking Machines Corp.  Characters
;;; are put into special classes; then `forward-word' and `scan-sexps' can be
;;; used to move across whole components quickly, without consing or
;;; much string-matching.
;;;
;;; Note that it is impossible to extract symbols from Soar process buffers
;;; without error, because the characters allowed in Soar production names
;;; include characters that appear in Soar output message, such as `pgs'
;;; listings.  In fact, you can define a production named "==>G:" in Soar!
;;; The approach here is to use some explicit tests for certain components
;;; in combination with two syntax tables.
;;;
;;; Since Soar itself rejects production names and other symbols if they
;;; contain certain characters, it is reasonable to assume SDE won't
;;; encounter them in normal operation.  So the algorithms here are not
;;; fool-proof; rather, they're oriented toward the most probably situations.
;;;----------------------------------------------------------------------------

(defvar sde-symbol-syntax-table
  "Syntax table used for extracting Soar symbols from buffers.")

;; symbol syntax      ("_") -- all chars allowed in production names
;; prefix syntax      ("'") -- the dash, which serves dual purposes
;; punctuation syntax (".") -- the up-arrow of attributes
;; whitespace syntax  (" ") -- everything else
;; misc. other syntax for things like the string delimiters.

(unless (syntax-table-p sde-symbol-syntax-table)
  (let ((table (make-vector 256 0))	; Default syntax is whitespace.
	(i 0))
    ;; Alphabetic chars.
    (while (< i 26)
      (modify-syntax-entry (+ ?a i) "_" table)
      (modify-syntax-entry (+ ?A i) "_" table)
      (setq i (1+ i)))
    ;; Digits.
    (setq i 0)
    (while (< i 10)
      (modify-syntax-entry (+ ?0 i) "_" table)
      (setq i (1+ i)))
    ;; Other characters.
    (mapcar (function
	     (lambda (char)
	       (modify-syntax-entry char "_" table)))
	    '(?$ ?% ?& ?* ?_ ?= ?+ ?< ?> ?/ ?: ??))
    (modify-syntax-entry ?\| "$|" table)
    (modify-syntax-entry ?-  "'"  table)
    (modify-syntax-entry ?^  "."  table)
    (modify-syntax-entry ?\" "\"" table)
    (setq sde-symbol-syntax-table table)))


(defun sde-extract-symbol (&optional start)
  "Returns list (SYMBOL BEGINNING END TIMETAG-P) for symbol under or to left of
point.  BEGINNING and END are the buffer positions of the SYMBOL, and
TIMETAG-P is a flag that is `t' if the symbol is actually believed to be a
timetag.  Optional argument START is where to move before prior to extracting
symbol\; point is left there after extracting the symbol.  Special cases: If
the symbol is an attribute name, the leading up-arrow of the attribute is
*not* included.  If the symbol under point is actually a timetag in a WME
printout, only the number is returned\; the trailing colon is omitted.
Otherwise, this function tries to returns whatever looks like a Soar symbol
or production name near point."
  (let ((old-syntax-table (syntax-table))
	(saved-point (or start (point))))
    (goto-char saved-point)
    (unwind-protect
	(let (beginning end timetag-p)
	  (set-syntax-table sde-symbol-syntax-table)
	  ;; First back up to something non-whitespace.
	  (when (= (char-syntax (following-char)) ? )
	    (skip-chars-backward " \t"))
	  ;; Find beginning and end of symbol.  If we're near the `^' or
	  ;; negation `-' of an attribute, or in the middle of something,
	  ;; find the boundaries by first moving forward.  Else, find the
	  ;; symbol's boundaries moving backward.
	  (if (or (memq (preceding-char) '(?- ?^))
		  (/= (char-syntax (following-char)) ? ))
	      (setq end       (scan-sexps (point) 1)
		    beginning (scan-sexps (or end 1) -1))
	    (setq beginning (scan-sexps (point) -1)
		  end       (scan-sexps (or beginning 1) 1)))
	  ;; Check if this may be a WME timetag.
	  (when (char-equal (char-after (1- end)) ?:)
	    (goto-char (1- end))
	    (skip-chars-backward "0-9") 
	    (if (= (point) beginning)	; Nothing but digits => timetag.
		(setq end (1- end)
		      timetag-p t)))
	  (if (and beginning end)
	      (list (buffer-substring beginning end) beginning end timetag-p)
	    nil))
      (goto-char saved-point)
      (set-syntax-table old-syntax-table))))


(defun sde-symbol-near-point (&optional predicate prompt-str)
  "Extract the symbol under or before point, and return it in string form.
If optional arg PREDICATE is non-nil, it must be a function.  PREDICATE is
funcall'ed with the extracted symbol string as its single argument, and it
should return non-nil if the string is acceptable, nil otherwise.  If
PREDICATE returns nil, the user is prompted with optional argument PROMPT-STR
and the extracted symbol as a default answer which the user can select just
by typing return.  If no symbol is found near the point, and PROMPT-STR is
nil, the user is prompted with the default prompt \"Target: \".  This
function returns the symbol string found near point or a new string typed by
the user.  If nothing is found and the user doesn't supply a symbol, an error
is signaled."
  (interactive)
  (if (eobp)
      (if (not (bobp))
	  (save-excursion
	    (backward-char 1)
	    (sde-symbol-near-point predicate prompt-str))
	(sde-error
	 "Empty buffer"
	 "This command requires a symbol on which to operate, but the\n\
current buffer is empty and you did not supply a symbol when prompted."))
    ;; Else, not eobp
    (let* ((prompt-str (or prompt-str "Target: "))
	   (sym (car (sde-extract-symbol))))
      ;; Check that we have something, and that if predicate is supplied it
      ;; returns non-nil on the candidate symbol.
      (when (or (null sym) (and predicate (not (funcall predicate sym))))
	(setq sym (sde-read-string
		   (if sym
		       (format "%s(default \"%s\") " prompt-str sym)
		     (format "%s" prompt-str))))
	(when (string= sym "")
	  (sde-error
	   "No symbol found or provided"
	   "This command requires a symbol on which to operate, but it was\n\
unable to find one in the buffer near the cursor and you did not\n\
supply one when prompted.")))
      ;; No error; return symbol.
      sym)))


(defun sde-production-name-near-point (&optional prompt-str)
  "Extract the name of the production under or before point.
If point is inside a full sp expression, the production name is unambiguous.
If point is elsewhere, this function extracts the symbol under or before
point, and heuristically judges whether it is likely to be a production
name."
  (interactive)
  (or (sde-production-name)
      (sde-symbol-near-point 'sde-production-name-p "Production name: ")))
  

;; General function for extracting identifiers, attributes and values from
;; objects printed by Soar.
;;
;; The approach is attribute oriented.  The user should put point over or
;; near an attribute in the production LHS.  The following tries to grab the
;; attribute, the value following it (often it's a variable), and the id at
;; the beginning of the form.
;;
;; This uses the constants `sde-attribute-regexp', `sde-value-regex', etc.,
;; but negated attribute tests are handled explicitly (by looking for the
;; negation sign).
;; 
;; PROBLEMS: This doesn't handle the following constructs:
;;
;; 1)  (<id> ^ {  <> tried-tied-operator <sub-att>  }  <sub-val>)
;;
;; 2)  (<s> ^ {  << required-success success partial-success 
;;                draw lose win >> <svalue>  }  <eb> )
;;
;; But as it turns out we never have to deal with those cases, since those
;; only appear in productions and not in objects printed by Soar.

(defconst sde-attribute-regexp "\\^\\s *[^ \t(){}<]+"
  "Regexp to match an attribute.")

(defconst sde-id-regexp "[a-zA-Z][0-9]+"
  "Regexp to match an identifier.")

(defconst sde-variable-regexp "<[a-zA-Z0-9]+>"
  "Regexp to match a production variable.")

(defconst sde-id-or-variable-regexp
    (concat "\\(" sde-id-regexp "\\|" sde-variable-regexp "\\)")
  "Regexp to match either an identifier or a variable.")

(defconst sde-symbol-regexp "[^^ \t\n<>()]+"
  "Regexp to match a symbol used as a value in a production clause.")

(defconst sde-value-regexp
    (concat "\\(" sde-variable-regexp "\\|" sde-id-regexp
	    "\\|" sde-symbol-regexp "\\)")
  "Regexp to match an attribute value.")


(defun sde-extract-id-attrib-value ()
  "Return a list (id negation attribute value preference).
This function scans the text near the cursor and attempts to extract an
attribute, its value, and an identifier at the head of the clause.      

This function is only suitable for productions written with a simple syntax,
such as those printed out by Soar itself.  It can only understand simple
negated attribute tests and acceptable preferences (`+').

The list returned consists of strings.  The `negation' portion of the
returned list is equal to \"-\" if the attribute had a leading negation sign,
or nil otherwise.  The `attribute' portion of the list is only the attribute
name, and does not have a leading up-arrow.  The `preference' portion is
equal to the string \"+\" if the value in the clause was followed by an
acceptable preference, or nil otherwise."
  (let ((saved-point (point))
	(preference "")
	(negation "")
	current sym id attrib attrib-end value)
    (setq current (sde-extract-symbol))
    ;; Go to beginning of this symbol and see if it's an attribute.
    (goto-char (nth 1 current))
    (skip-chars-backward " \t")
    (if (char-equal (preceding-char) ?^)
	(setq attrib     (car current)
	      attrib-end (nth 2 current))
      ;; Not an attribute; maybe it's a value.  Back up further and try again.
      (setq current (sde-extract-symbol)) ; Get symbol we're now next to.
      (goto-char (nth 1 current))	  ; Go to its beginning
      (skip-chars-backward " \t")
      (when (char-equal (preceding-char) ?^)
	(setq attrib     (car current))
	(setq attrib-end (nth 2 current))))
    ;; At this point, we should be right after the up-arrow.  See if there's
    ;; a negation sign in front, then try to get the value after the attribute.
    (when attrib
      (backward-char 1)
      (skip-chars-backward " \t")
      (when (char-equal (preceding-char) ?-)
	(setq negation "-"))
      ;; Get the value.
      (setq current (sde-extract-symbol (scan-sexps attrib-end 1)))
      (setq sym     (car current))
      (when (string-match sde-value-regexp sym)
	(setq value sym)
	;; See if there's a preference following the value.
	(goto-char (nth 2 current))
	(skip-chars-forward " \t")
	(if (char-equal (following-char) ?+)
	    (setq preference "+"))))
    ;; Look for identifier.
    (search-backward "(" nil t)
    (forward-char 1)
    (setq current (sde-extract-symbol))
    (setq sym (car current))
    ;; Check if we actually have a timetag.
    (when (nth 3 current)
      ;; Yes, a timetag.  Move forward, try to get to the id.
      (goto-char (nth 2 current))	; End of timetag.
      (forward-char 1)			; Past the colon.
      (skip-chars-forward " \t")	; Past the whitespace.
      (setq current (sde-extract-symbol)) ; Get the id.
      (setq sym (car current)))	
    (when (string-match "goal\\|impasse\\|state" sym)
      (goto-char (scan-sexps (nth 2 current) 1))
      (setq current (sde-extract-symbol)
	    sym     (car current)))
    (when (string-match sde-id-or-variable-regexp sym)
      (setq id sym))
    (goto-char saved-point)
    (list id negation attrib value preference)))


;;;----------------------------------------------------------------------------
;;; 18. SDE mode
;;;----------------------------------------------------------------------------

(if (not (keymapp sde-find-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-find-cmds)
      (define-key map "\C-h" 'sde-help-for-find-cmds)
      (define-key map "\C-t" 'sde-find-task)
      (define-key map "\C-r" 'sde-find-production-by-rhs)
      (define-key map "\C-p" 'sde-find-problem-space)
      (define-key map "\C-o" 'sde-find-operator)
      (define-key map "\C-n" 'sde-find-production-by-name)
      (define-key map "\C-l" 'sde-find-production-by-lhs)
      (define-key map "\C-b" 'sde-find-production-by-body)
      (setq sde-find-cmds-map map)))


(sde-make-help-screen sde-help-for-find-cmds
  "C-b, C-l, C-n, C-o, C-p, C-r, C-t. [C-h = more help]"
  "Commands for finding components of Soar tasks:

C-n  find production by name
C-b  find production by searching whole body
C-l  find production by searching only left-hand side (i.e., condition-side)
C-r  find production by searching only right-hand side (i.e., action-side)

C-o  find operator
C-p  find problem-space
C-t  find task

?    help on find commands
C-h  help on find commands
C-g  cancel"
  sde-find-cmds-map)


(if (not (keymapp sde-view-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-view-cmds)
      (define-key map "\C-h" 'sde-help-for-view-cmds)
      (define-key map "\C-w" 'sde-view-working-memory)
      (define-key map "\C-t" 'sde-view-ptraces)
      (define-key map "\C-s" 'sde-view-pgs)
      (define-key map "\C-p" 'sde-view-productions)
      (define-key map "\C-o" 'sde-view-pgso)
      (define-key map "\C-m" 'sde-view-ms)
      (define-key map "\C-j" 'sde-view-justifications)
      (define-key map "\C-c" 'sde-view-chunks)
      (define-key map "\C-b" 'sde-view-pbreaks)
      (setq sde-view-cmds-map map)))


(sde-make-help-screen sde-help-for-view-cmds
  "C-b, C-c, C-j, C-m, C-p, C-s, C-t, C-w. [C-h = more help]"
  "Commands for viewing various Soar data:

C-b  view pbreaks
C-c  view chunks
C-j  view justifications
C-m  view match set (Soar \"ms\")
C-o  view operator stack (Soar \"pgso\")
C-p  view productions
C-s  view goal stack (Soar \"pgs\")
C-t  view ptraces
C-w  view contents of Soar working memory

?    help on view commands
C-h  help on view commands
C-g  cancel"
  sde-view-cmds-map)


(if (not (keymapp sde-agent-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-agent-cmds)
      (define-key map "\C-h" 'sde-help-for-agent-cmds)
      (define-key map "\C-s" 'select-agent)
      (define-key map "\C-l" 'list-agents)
      (define-key map "\C-d" 'destroy-agent)
      (define-key map "\C-c" 'create-agent)
      (define-key map "\C-a" 'agent-go)
      (setq sde-agent-cmds-map map)))


(sde-make-help-screen sde-help-for-agent-cmds
  "C-a, C-c, C-d, C-l, C-s. [C-h = more help]"
  "Commands for managing multiple Soar agents:

C-a  agent \"go\" parameters
C-c  create agent
C-d  destroy agent
C-l  list agents known by SDE
C-s  select agent

?    help on agent commands
C-h  help on agent commands
C-g  cancel"
  sde-agent-cmds-map)


(if (not (keymapp sde-region-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-region-cmds)
      (define-key map "\C-h" 'sde-help-for-region-cmds)
      (define-key map "="    'sde-region-count-productions)
      (define-key map ";"    'sde-region-comment)
      (define-key map "\C-x" 'sde-region-excise)
      (define-key map "\C-t" 'sde-region-ptrace)
      (define-key map "\C-s" 'sde-region-send)
      (define-key map "\C-b" 'sde-region-pbreak)
      (setq sde-region-cmds-map map)))


(sde-make-help-screen sde-help-for-region-cmds
  "C-b, C-e, C-t, C-x, \;, =. [C-h = more help]"
  "Commands for operating on regions of Soar code:

C-b  pbreak all productions in region
C-s  send (i.e., load) region to Soar
C-t  ptrace all productions in region
C-x  excise all productions in region
;    comment or uncomment region
=    count productions, lines and characters in region

?    help on region commands
C-h  help on region commands
C-g  cancel"
  sde-region-cmds-map)


(if (not (keymapp sde-help-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-help-cmds)
      (define-key map "\C-h" 'sde-help-for-help-cmds)
      (define-key map "\C-w" 'sde-where-is)
      (define-key map "\C-u" 'sde-soar-user-notes)
      (define-key map "\C-t" 'sde-topic-help)
      (define-key map "\C-s" 'sde-soar-info)
      (define-key map "\C-r" 'sde-soar-release-notes)
      (define-key map "\C-m" 'sde-describe-mode)
      (define-key map "\C-k" 'describe-key)
      (define-key map "\C-i" 'sde-info)
      (define-key map "\C-b" 'sde-describe-bindings)
      (define-key map "\C-a" 'sde-apropos)
      (setq sde-help-cmds-map map)))


(sde-make-help-screen sde-help-for-help-cmds
  "C-a, C-b, C-i, C-k, C-m, C-s, C-t, C-u, C-w. [C-h = more help]"
  "Help interface for SDE and Soar.  Press one of the following keys:

C-m  mode         -- Describe the current mode.
C-b  bindings     -- List all SDE key bindings for the current mode.
C-k  describe key -- Describe the SDE command invoked by a given key sequence.
C-w  where-is     -- Show the key that invokes a given SDE command.

C-a  apropos      -- Briefly list SDE commands and variables matching a string.
C-t  topic        -- Display help on any topic, SDE or Soar.

C-i  info         -- Read the SDE User's Manual using the Info system.

C-s  Soar manual  -- Read the on-line Soar User's Manual using the Info system.
C-r  rel. notes   -- Read the Release Notes for the most recent version of Soar
C-u  user notes   -- Read the User Notes for the most recent version of Soar.

?    help on help commands
C-h  help on help commands
C-g  cancel

The `topic' and `where-is' commands offer completion.  
The `topic' command knows about both Soar and SDE commands and variables, and
will query Soar for help if asked about a Soar topic."
  sde-help-cmds-map)


;; Certain keys are reserved for use in Soar Mode:
;; 
;; c-c c-w  backward-kill-word    	   Like ^w in shells
;; C-c C-o  sde-kill-output                Delete last batch of process output
;; C-c C-u  sde-kill-input                 Like ^u in shells
;; M-p      sde-previous-input             Cycle backwards in input history
;; M-n      sde-next-input                 Cycle forwards
;; M-s      sde-previous-similar-input     Previous similar input
;; M-C-r    sde-previous-input-matching    Search backwards in input history
;;
;; Two key bindings get rebound in Soar Mode to something entirely different:
;;
;; C-c C-r  sde-show-output
;; M-C-r    sde-previous-input-matching

;; The order of the definitions in the following code matter.  When
;; `describe-bindings' is invoked, it will print the items in
;; *reverse* order from the order of their definition.  So keys must
;; be defined in reverse alphabetical order to get them to be printed
;; in alphabetical order when `describe-bindings' and other functions
;; are invoked. In addition, some duplicate keys are purposefully
;; defined earlier than their alternative (for example `C-M-x' before
;; `C-c C-e') so that when `substitute-key-bindings' is run by Emacs
;; for various help functions, it gets the desired key binding (in
;; this case "C-c C-e") before the alternative, because the
;; alterative may not be as mnemonic.  Is that clear? :-)

(unless (keymapp sde-mode-map)
  (let ((map (make-sparse-keymap)))
    (define-key map "\eq"      'sde-reindent)
    (define-key map "\ej"      'indent-new-comment-line)
    (define-key map "\e\;"     'indent-for-comment)
    (define-key map "\e\C-x"   'sde-send-production) ; Like eval-defun
    (define-key map "\e\C-r"   'sde-reposition-window)
    (define-key map "\e\C-q"   'sde-indent-sexp)
    (define-key map "\e\C-j"   'indent-new-comment-line)
    (define-key map "\e\C-h"   'sde-mark-production) ; Like mark-defun
    (define-key map "\e\C-e"   'sde-end-of-production)
    (define-key map "\e\C-a"   'sde-beginning-of-production)

    (define-key map "\C-c\C-z" 'sde-switch-to-soar)
    (define-key map "\C-c\C-y" 'wm)
    (define-key map "\C-c\C-x" 'excise)
    (define-key map "\C-c\C-v" sde-view-cmds-map)
    (define-key map "\C-c\C-t" 'ptrace)
    (define-key map "\C-c\C-s" 'sde-send-production)
    (define-key map "\C-c\C-r" sde-region-cmds-map)
    (define-key map "\C-c\C-n" 'sde-next-match)
    (define-key map "\C-c\C-q" 'firing-counts)
    (define-key map "\C-c\C-p" 'print-soar)
    (define-key map "\C-c\C-l" 'load-soar)
    (define-key map "\C-c\C-h" sde-help-cmds-map)
    (define-key map "\C-c\C-f" sde-find-cmds-map)
    (define-key map "\C-c\C-e" 'explain)
    (define-key map "\C-c\C-c" 'sde-interrupt-soar)
    (define-key map "\C-c\C-b" 'pbreak)
    (define-key map "\C-c\C-a" sde-agent-cmds-map)
    (define-key map "\C-c0"    'init-soar)
    (define-key map "\C-c+"    'sde-insert-date-stamp)
    (define-key map "\C-c%"    'sde-query-replace)
    (define-key map "\C-c$"    'sde-search)
    (define-key map "\C-c)"    'sde-find-unbalanced)
    (define-key map "\C-c]"    'sde-close-all-sp)
    (define-key map "\C-c\;"   'sde-region-comment)
    (define-key map "\C-c/"    'preferences)
    (define-key map "\C-c'"    'matches-1)
    (define-key map "\C-c\""   'matches-2)
    (define-key map "\C-c,"    'schedule)
    (define-key map "\C-c."    'go)
    (define-key map "\C-c?"    'sde-why)

    (define-key map "\C-m"     'sde-newline-and-indent)
    (define-key map "\C-j"     'sde-newline-and-indent)
    (define-key map "\t"       'sde-indent-line)
    (define-key map "\177"     'backward-delete-char-untabify)
    
    ;; Mouse buttons
    (when window-system
      (cond (sde-running-lemacs
	     (define-key map '(button3)       'sde-popup-menu)
	     (define-key map '(shift button1) 'sde-mouse-print-soar)
	     (define-key map '(shift button2) 'sde-mouse-matches)
	     (define-key map '(shift button3) 'sde-mouse-preferences))

	    (sde-running-emacs19
	     (define-key map [down-mouse-3]   'sde-popup-menu)
	     (define-key map [S-mouse-1]      'sde-mouse-print-soar)
	     (define-key map [S-mouse-2]      'sde-mouse-matches)
	     (define-key map [S-mouse-3]      'sde-mouse-preferences))))

    (setq sde-mode-map map)))


;; Syntax table.

(unless (syntax-table-p sde-mode-syntax-table)
  (let ((table (copy-syntax-table lisp-mode-syntax-table)))
    (modify-syntax-entry ?\t " "  table)
    (modify-syntax-entry ?\f " "  table)
    ;; Need newline to be comment-end for some of the functions that look
    ;; for productions components to work properly.  Unfortunately, this
    ;; causes "\\s " regexp's to fail to consider newline to be whitespace!
    (modify-syntax-entry ?\n ">"  table)
    (modify-syntax-entry ?   " "  table)
    (modify-syntax-entry ?~  " "  table)
    (modify-syntax-entry ?!  " "  table)
    (modify-syntax-entry ?@  " "  table)
    (modify-syntax-entry ?#  " "  table)
    (modify-syntax-entry ?$  "_"  table) 
    (modify-syntax-entry ?%  "_"  table) 
    (modify-syntax-entry ?^  "_"  table)
    (modify-syntax-entry ?&  "_"  table)
    (modify-syntax-entry ?*  "_"  table)
    (modify-syntax-entry ?\( "()" table)
    (modify-syntax-entry ?\) ")(" table)
    (modify-syntax-entry ?-  "_"  table) 
    (modify-syntax-entry ?_  "_"  table) 
    (modify-syntax-entry ?=  "_"  table)
    (modify-syntax-entry ?+  "_"  table) 
    (modify-syntax-entry ?\{ "(}" table)
    (modify-syntax-entry ?\} "){" table)
    (modify-syntax-entry ?\[ "(]" table)
    (modify-syntax-entry ?\] ")[" table)
    (modify-syntax-entry ?\| "$|" table)
    (modify-syntax-entry ?\  " "  table) 
    (modify-syntax-entry ?\; "<"  table)
    (modify-syntax-entry ?:  "_"  table)
    (modify-syntax-entry ?'  "'"  table)
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?.  " "  table)
    (modify-syntax-entry ?,  " "  table)
    ;; We can't make '<' and '>' be opening and closing delimiters because
    ;; '>' is used in the production arrow, and Emacs would get confused by
    ;; the right angle bracket in "-->".  Too bad.
    (modify-syntax-entry ?\< "'"  table)
    (modify-syntax-entry ?\> "'"  table)
    (modify-syntax-entry ?/  "_"  table)
    (modify-syntax-entry ??  "_"  table) 
    (setq sde-mode-syntax-table table)))

;; Abbrev table.

(when (null sde-mode-abbrev-table)
  (define-abbrev-table 'sde-mode-abbrev-table ()))


;; The actual SDE mode.
;;
;; Note that the key bindings for next and previous history cannot be written
;; as \\[gmhist-previous], etc., because we don't use gmhist in GNU Emacs 19.
;; I had to hardwire the descriptions as "ESC p" and "ESC n".  Some other key
;; descriptions are hardwired too, mainly because FSF 19 seems to catch the
;; menu definition instead of the key binding first, and consequently prints
;; `menu foo-bar biff-wack', which is ugly and almost useless.

(defun sde-mode ()
  "The Soar Development Environment (SDE), a major mode for writing Soar programs.
\\<sde-mode-map>
`\\[sde-beginning-of-production]' moves the cursor to the beginning of a production.
`\\[sde-end-of-production]' moves the cursor to the end of a production.

Commands that move the cursor by words, such as `\\[forward-word]' (forward-word), 
and `\\[backward-word]' (backward-word), consider whitespace and the characters
'^', '*', '-', '_', '<', '>' and '.' to be word delimiters.  In addition, such
commands ignore parentheses, curly braces and whitespace.  Commands that move
the cursor by balanced expressions, such as `\\[forward-sexp]' (forward-sexp) and
`\\[backward-sexp]' (backward-sexp), consider parentheses and curly braces to
be parts of expressions.

`M-x soar' will start Soar in a new buffer.  A prefix argument (such as
\\[universal-argument]) will make it prompt for the name of the Soar program, the
starting directory, and optional command line arguments to pass to Soar.

`\\[sde-send-production]' will send (that is, load) the current production
under the cursor to the current Soar process.  With a prefix argument, it
also switches to Soar after sending the production.

`\\[load-soar]' will prompt for a file name and load it into Soar\; you may use
this to load the current buffer's file into Soar as well as other files.
`M-x load-defaults' will load the most current version of the Soar defaults
file.  `M-x excise-file' will prompt for a file name and will excise all of
the productions found in that file.

The region commands perform common operations on a region in the buffer.
`\\[sde-region-send]' sends everything within the current region to Soar.
`\\[sde-region-pbreak]' issues a pbreak for every production in the region.
`\\[sde-region-ptrace]' issues a ptrace for every production in the region.
`\\[sde-region-excise]' excises every production in the region.
`C-c C-r ;' comments out the region between mark and point.  If given a 
positive prefix arg, it inserts that many semicolons at the beginning of 
each line \(defaulting to 1).  If given a negative prefix argument, it 
uncomments the current region.

If Soar is running in multi-agent mode, and the current buffer is not
associated with any agent, the first command that needs a target agent will
prompt for an agent name and will set the buffer's default to the answer.
To change default agents, or to set the initial agent explicitly, use the
select-agent command (`\\[select-agent]').

`\\[sde-reindent]' reindents whatever the cursor is in.  If the cursor is in a comment,
it reindents the comment.  If in a production, it reindents the whole
production, closing it first if necessary.  If the cursor is in the
documentation string of a production, it reindents the documentation string.
`\\[sde-indent-line]' indents just the current line.  `RET' moves to the next
line and automatically indentations appropriately.  Variable
`sde-production-indent-offset' controls how far clauses are indented underneath
the first line of the production\; the default is 2 spaces.  Variable
`sde-arrow-indent-offset' determines whether the arrow is indented differently\;
negative offsets move it leftward, positive offsets indent the arrow further.

SDE tracks the files that comprise your tasks.  When you visit a Soar file
that is part of a task that is not yet known to SDE, it will prompt you for
the main (i.e., top-most) load file for the task, and then it will
recursively parse each file of the task to build an internal database of the
files and productions that comprise the task.  It will also cache the list of
files into a special file named \".sde\" in the directory where the main load
file is located.  In future editing sessions, SDE will read the \".sde\" file
first and delay the full parsing of files until the information is needed for
certain types of searches \(specifically, for the `sde-find-xxx' commands, 
described below).

All interactive Soar commands are available via `M-x COMMANDNAME'.  Some
commands in SDE have no Soar equivalents, e.g., `excise-file'.  Many key
bindings give direct access to the most common commands.  Some notable
bindings include the following, and a complete list appears later below.  
Use describe-key (\\<global-map>`\\[describe-key]'\\<sde-mode-map>) \(the standard Emacs help facility) to
get detailed help on each command.

  Running and Initializing Soar:

    go                      \\[go]
    schedule                \\[schedule]
    interrupt Soar          \\[sde-interrupt-soar]
    init-soar               \\[init-soar]

  Manipulating Productions:

    send production         \\[sde-send-production]
    send region             \\[sde-region-send]
    load file               \\[load-soar]
    excise production       \\[excise]
    excise region           \\[sde-region-excise]
    
  Tracing and breaking:

    ptrace production       \\[ptrace]
    ptrace region           \\[sde-region-ptrace]
    pbreak production       \\[pbreak]
    pbreak region           \\[sde-region-pbreak]

  Querying Soar for information about specific objects:

    print                   \\[print-soar]
    explain                 \\[explain]
    matches 1               \\[matches-1]
    matches 2               \\[matches-2]
    preferences             \\[preferences]
    firing-counts           \\[firing-counts]
    wm                      \\[wm]

  Querying Soar for general information:

    view ptraces            \\[sde-view-ptraces]
    view pbreaks            \\[sde-view-pbreaks]
    view goal stack (pgs)   \\[sde-view-pgs]
    view op. stack (pgso)   \\[sde-view-pgso]
    view match set (ms)     \\[sde-view-ms]
    view chunks             \\[sde-view-chunks]
    view productions        \\[sde-view-productions]
    view justifications     \\[sde-view-justifications]
    view Soar WM            \\[sde-view-working-memory]

  Finding source code:

    find production by name \\[sde-find-production-by-name]
    find by body (LHS/RHS)  \\[sde-find-production-by-body]
    find by LHS pattern     \\[sde-find-production-by-lhs]
    find by RHS pattern     \\[sde-find-production-by-rhs]
    find operator           \\[sde-find-operator]
    find problem-space      \\[sde-find-problem-space]
    find next match         \\[sde-next-match]

  Searching and replace across whole tasks:

    search for a string     \\[sde-search]
    search for a regexp     \\[sde-search-regexp]
    query-replace           \\[sde-query-replace]
    query-replace-regexp    \\[sde-query-replace-regexp]
    continue search         \\[sde-next-match]

  Controlling multiple agents:
  
    agent-go                \\[agent-go]
    create-agent            \\[create-agent]
    destroy-agent           \\[destroy-agent] (Unavailable prior to Soar 6.1.1)
    list agents             \\[list-agents]
    select-agent            \\[select-agent]

When running in the X Window System, the following mouse buttons are defined:

  Button 	    	      Action
  ------------------    -------------------------------------------------
  right                 SDE popup menu

  SHIFT-left            Execute `print' on item under the cursor
  SHIFT-middle          Execute `matches' on item under the cursor
  SHIFT-right           Execute `preferences' on attribute under the cursor

Many Soar commands, such as print, require \"target\" arguments, i.e.,
something to act on.  Those commands that need a production name as target
will use the name of the production that the cursor is in currently, or else
the name of the preceeding sp form.  Those commands that can take any symbol
as target will try to use the symbol under the cursor regardless of whether
it looks like a production name.

Many of the standard Soar commands also take optional arguments -- for
example, the go and print commands.  Giving a prefix argument (such as `\\[universal-argument]')
to most commands makes them prompt for optional arguments.  Optional
arguments are remembered on history lists.  Typing `M-p' and `M-n' at a 
prompt moves backward and forward through the history.

Optional arguments are automatically reused upon subsequent invocations of a
command until they are changed.  This allows you to set an option and then
apply a command repeatedly to different target arguments without having to
respecify the optional arguments each time.  For example, you can type
`\\[universal-argument] \\[print-soar]', type \":depth 3\" to the resulting prompt for optional
arguments, and then repeat this command with the same arguments using
just `\\[print-soar]' on different targets.

The commands ptrace, pbreak, preferences, and explain have different
conventions.  Without arguments, ptrace (`\\[ptrace]') and pbreak (`\\[pbreak]')
act on the production under the cursor.  With a positive argument (such as
`\\[universal-argument]') they perform unptrace or unpbreak, respectively, on the production
under the cursor.  With a negative argument (e.g., `\\[negative-argument]'), they undo all 
ptraces or pbreaks, respectively.  You can print out the current list of
active ptraces and pbreaks using the view commands `\\[sde-view-ptraces]' and 
`\\[sde-view-pbreaks]', respectively.

The preferences command (`\\[preferences]') attempts to extract its arguments
using heuristics.  Place the cursor over the attribube in a printed oject,
for example on \"^io-state\" in the WME
                    (8: S1 ^io-state S1)
and preferences will automatically use the attribute and the WME's identifier 
as arguments.  If the cursor is not over an attribute, it will attempt to
extract the symbol nearest the cursor, and if it is an identifier, then 
preferences will prompt for the attribute.  In addition, if preferences is
given a prefix argument (`\\[universal-argument]'), it will prompt for the optional
\"level\" argument.

The explain command (`\\[explain]') also attempts to extract its arguments
using heuristics.  Remember that before explain may be used, Soar must be
instructed to track the necessary information using \"explain :on\" \(or the
SDE equivalent `M-x explain-on').  The easiest way to use explain in SDE
is to print out a chunk and then place the cursor over an attribute in
the condition side of the chunk.  Explain will then query Soar with the
name of the chunk and the condition that tests that attribute.  \(It does
this by constructing a condition clause consisting of the appropriate
identifier-attribute-value slot, and silently querying Soar to determine
the correct condition number, then finally sending the command 
\"explain <production-name> <condition-number>\" to Soar.\)  You may also
put the cursor over a chunk name and invoke explain, and SDE will extract
just the production name and issue \"explain <name>\" to Soar.  Finally,
if you are looking at the text of a previous \"explain <name>\" and you
wish to do \"explain <name> <condition>\", simply put the cursor on the
line containing the condition number and invoke explain, and SDE will issue
\"explain <production-name> <condition-number>\" to Soar.  If given a prefix
argument, explain will prompt for its arguments, offering as initial defaults
any production name and condition clause it can find.

Commands that only query Soar for information place the output in in a
separate buffer for viewing, unless the variable `sde-soar-use-output-buffer'
is nil, in which case all output goes to the agent process buffer.
New output is appended to the output buffer, each batch separated by a
title line, unless the variable `sde-soar-erase-output-buffer' is non-nil,
in which case the output buffer's existing contents will be cleared before
new output is written out.

When running in Emacs 19 or Lucid Emacs 19, SDE will attempt to use a
separate X Window screen (called a \"frame\" in Emacs 19) for the output 
buffer, unless the variable `sde-use-multiple-frames' is nil.  The
initial attributes of this buffer are determined by the list of values in
the variable `sde-soar-output-buffer-defaults'.  Use Emacs help on this
variable (`\\[describe-variable]') for more information about how to use
this feature.  

The command `\\[sde-find-production-by-name]' locates the source code for a
production whose name is under the cursor.  Put the cursor on a production in
your task, either in a Soar process buffer or an editing buffer, type
`\\[sde-find-production-by-name]', and it will jump to the place in your
files where the production is defined.

The command `\\[sde-find-production-by-lhs]' (`sde-find-production-by-lhs') allows you 
to search a task for all productions that contain a certain pattern in their
condition sides.  When invoked, the command prompts you for a search pattern.
The pattern must be a list similar in form to production conditions.
For example,
             ( (goal <g> ^problem-space <p>) (<p> ^name top-ps) )

is a pattern that will match all productions whose conditions test for a
problem space named \"top-ps\".  \(The actual variable names that you use are
irrelevant because the system is doing true pattern-matching search.)  The
names of the productions will be listed in a separate window, and the source
code of the first production will be shown in a second window.  You can view
the source code of each successive production by repeatedly typing
`\\[sde-next-match]' (`sde-next-match').  Alternatively, you can move the
cursor over a production name and use `\\[sde-find-production-by-name]'.

The command `\\[sde-search]' (`sde-search') lets you search the files of a task for a
string.  When invoked, this command prompts you for a search string and then
searches the files of the current task, stopping at the first match.  Use the
command `\\[sde-next-match]' to find the next occurrence of the string in the
files.  The command `\\[sde-search-regexp]' is identical, but lets you supply
a regular expression instead of a plain string.  Files are searched in
alphabetical order by default, unless the variable `sde-sort-lists' is set
to `nil'.

The command `\\[sde-query-replace]' performs an interactive query-replace
across the files of a task.  When invoked, this command prompts you for a
string to search for and a string with which to replace it.  If given a
prefix argument, it also prompts for the task in which to search\; otherwise,
it uses the task with which the current buffer is associated.  It then
performs a `query-replace' in each file of the task.  Type \\[help-command]
at a prompt for directions about what to do.  If you exit the query-replace
\(using `\\[keyboard-quit]' or ESC), you can resume the query-replace with the
command `\\[sde-next-match]'.  The command `\\[sde-query-replace-regexp]' is
similar but allows you to supply a regular expression instead of a plain
string.  The commands `\\[sde-replace-string]' and 
`\\[sde-replace-string-regexp]' are noninteractive versions of these commands.

The SDE on-line help facility is available on the `C-c C-h' prefix.
`\\[sde-apropos]' (apropos) prompts for a string and lists a brief description of
all SDE user commands and user variables that contain the given string.
`\\[sde-topic-help]' (topic) prompts for a command, variable or Soar help topic, and 
displays information about it.  This command interfaces to the Soar help 
facility, and will query Soar for help on topics that are not already defined
in SDE.  Note that `\\[sde-topic-help]' provides completion, which means you
can type `?' any time at the prompt to list possible completions of what you 
have typed so far, and you can type `TAB' to have Emacs complete the rest of
what you have typed based on known help topics.

`\\[sde-soar-info]' will bring up the Soar 6 User's manual using the Emacs
Info browser.  `C-c C-h C-i' will bring up the SDE manual.  `\\[sde-soar-release-notes]' 
will bring up the Release Notes for the latest revision of Soar, while 
`\\[sde-soar-user-notes]' will bring up the User Notes.  `\\[sde-describe-bindings]' will list the 
local bindings in the current SDE mode, and `\\[sde-where-is]' will prompt for
an SDE command name (with completion) and report which key binding
invokes it.

`\\[sde-close-all-sp]' will close the current sp form by adding or removing right parentheses
as necessary.

`\\[sde-region-count-productions]' will count the number of productions, lines
and characters the mark and the point, and report the answer.

`\\[backward-delete-char-untabify]' converts tabs to spaces as it moves back.

`\\[sde-reposition-window]' will make the current production and/or comment visible.
If the production is fully onscreen, it is moved to the top of the window.
If it is partly offscreen, the window is scrolled to get the definition (or
as much as will fit) onscreen, unless point is in a comment which is also
partly offscreen, in which case the scrolling attempts to get as much of the
comment onscreen as possible.  Repeated invocations of `\\[sde-reposition-window]' move the
production to the top of the window or toggle the visibility of comments that
precede it.

`\\[sde-find-unbalanced]' searches for unbalanced parentheses in the current buffer.

`\\[sde-switch-to-soar]' will switch you to the Soar process buffer.

To send a bug report, questions or other feedback to the authors and 
maintainers of the Soar Development Environment, please use `\\[sde-feedback]'.

Entry to this mode will run the functions on the list variable `sde-mode-hook'.

Here is a list of all the special bindings in Soar mode.  The name of the key
map is `sde-mode-map'.  For further help on individual commands, type
\\<global-map>`\\[describe-key] KEY' where KEY is the keystroke.

\\<sde-mode-map>\\{sde-mode-map}"

  (interactive)
  (sde-mode-internal)
  ;; !!! This is a hack that must be removed when the header code
  ;; is cleaned up.  This is to enable users to set sde-header-hooks
  ;; now, even though header.el uses "make-header-hooks".
  (make-local-variable 'make-header-hooks)
  (when (boundp 'sde-header-hooks)
    (setq make-header-hooks sde-header-hooks))
  (run-hooks 'sde-mode-hook))


(defun sde-mode-internal ()
  "Function for performing initializations common to derivatives of SDE mode.
It does almost everything that sde-mode does, except for running hooks and
certain other initializations that only make sense for file buffers."
  ;; Set standard Emacs variables.
  (setq mode-name "SDE"
	major-mode 'sde-mode)
  (use-local-map sde-mode-map)
  ;; Build menus if appropriate.  Although in FSF 19 we could build menus at
  ;; load time, in Lucid the menus have to be defined for each buffer, so
  ;; it's easiest just to let a function handle it for both cases.
  (when (and window-system (or sde-running-emacs19 sde-running-lemacs))
    (sde-define-menus 'sde-mode sde-mode-map))
  (set-syntax-table sde-mode-syntax-table)
  (setq local-abbrev-table sde-mode-abbrev-table)

  (make-local-variable 'comment-column)
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'comment-indent-function)
  (make-local-variable 'indent-line-function)
  (make-local-variable 'paragraph-start)
  (make-local-variable 'paragraph-separate)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq comment-column       40
	comment-start        ";"
	comment-start-skip   ";+ *"
	indent-line-function 'sde-indent-line
	paragraph-start      "^(sp\\|^;+\\|^[ \t\f\n]*$"
	paragraph-separate   "^;+\\|^[ \t\f\n]*$"
	parse-sexp-ignore-comments t)
  (setq comment-indent-function  'lisp-comment-indent)
  (make-local-variable 'lisp-indent-function)
  (setq lisp-indent-function 'sde-indent-hook)

  ;; SDE Soar Mode variables
  (make-local-variable 'sde-buffer-data)
  (setq sde-buffer-data (sde-make-buffer-struct))

  ;; Set up the mode line.
  (setq	mode-line-modified '("-%1*%1* ")
	mode-line-buffer-identification '("%24b")
	mode-line-format
	(list "" 'mode-line-modified 'mode-line-buffer-identification
	      "  {"
	      '(sde-soar-agents
		;; Multi-agent case
		((sde-soar-buffer-agent sde-soar-buffer-agent "no agent")
		 (sde-soar-buffer-agent ": ")
		 (sde-soar-buffer-agent sde-soar-state-string))
		;; Single-agent case
		sde-soar-state-string)
	      "}  "
	      'global-mode-string
	      " %[("
	      'mode-name 'minor-mode-alist
	      ")%n%] --"
	      (if (or sde-running-emacs19 sde-running-lemacs)
		  '(line-number-mode "L%l--")
		"")
	      '(-3 . "%p") "-%-")))


;;;----------------------------------------------------------------------------
;;; 22. Closing statements.
;;;----------------------------------------------------------------------------


;;; Emacs indentation support for macros and form hooks for edebug.
;;;
;;; Local Variables:
;;; eval:(put 'sde-with-hacked-delimiter-syntax 'lisp-indent-function 1)
;;; eval:(put 'sde-with-hacked-delimiter-syntax 'lisp-indent-hook 1)
;;; eval:(put 'sde-with-hacked-delimiter-syntax 'edebug-form-spec '(&rest form))
;;; eval:(put 'sde-make-help-screen    'lisp-indent-function 1)
;;; eval:(put 'sde-make-help-screen    'lisp-indent-hook 1)
;;; eval:(put 'sde-with-mouse-click    'lisp-indent-function 1)
;;; eval:(put 'sde-with-mouse-click    'lisp-indent-hook 1)
;;; eval:(put 'sde-make-hash-key       'edebug-form-spec '(form))
;;; eval:(put 'sde-puthash             'edebug-form-spec '(form form &optional form))
;;; eval:(put 'sde-gethash             'edebug-form-spec '(form &optional form))
;;; eval:(put 'sde-map-hashtable       'edebug-form-spec '(form form))
;;; eval:(put 'sde-substring           'edebug-form-spec '(form integerp &optional form))
;;; eval:(put 'sde-buffer-substring    'edebug-form-spec '(integerp))
;;; eval:(put 'sde-prompt              'edebug-form-spec '(form form))
;;; eval:(put 'sde-get-buffer-create   'edebug-form-spec '(form))
;;; eval:(put 'sde-buffer-file-name    'edebug-form-spec '(form))
;;; eval:(put 'sde-make-name-regexp    'edebug-form-spec '(form))
;;; eval:(put 'sde-with-mouse-click    'edebug-form-spec '(form &rest (form)))
;;; eval:(put 'sde-set-up-file-symbols 'edebug-form-spec '(form))
;;; End:
