/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  FILE: x-interface.c                                                    */
/*                                                                         */
/*  This file defines the X window system interface to the multi-agent     */
/*  version of Soar.                                                       */
/*                                                                         */
/*  This file exports the following functions:                             */
/*    void create_display (void);                                          */
/*    void refresh_display (void);                                         */
/*    void destroy_display (void);                                         */
/*                                                                         */
/*  Initially coded by: Karl Schwamb USC/ISI                               */
/*  Initial version:    14 October 1992                                    */
/*                                                                         */
/*-------------------------------------------------------------------------*/

/*- I N C L U D E S -------------------------------------------------------*/

#include "soar.h"                              /* Imports stop_soar and    */
                                               /* reason_for_stopping.     */

#ifdef USE_X_DISPLAY

#include <X11/Xlib.h>                          /* Standard X headers.      */
#include <X11/Xutil.h>
#include <X11/keysym.h>                        /* Imports cursor key codes */

#include <stdio.h>

#ifdef USE_STDARGS
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include <math.h>                              /* Imports atan2, sin, cos  */



/*- G L O B A L S ---------------------------------------------------------*/
/*--------------- X   I N F O ---------------------------------------------*/ 

static Display * display;         /* Vital X information.                  */
static char * fontname = "6x13";

/*--------------- W I N D O W   P A R A M E T E R S -----------------------*/

#define WINDOW_UPPER_LEFT_X 0
#define WINDOW_UPPER_LEFT_Y 0
#define WINDOW_WIDTH         6*80
#define WINDOW_HEIGHT       13*15

#define CHAR_WIDTH   6
#define CHAR_HEIGHT 13

/*--------------- O T H E R -----------------------------------------------*/

#define BUFFER_SIZE 200           /* Handy char buffer decl.               */
typedef char buffer[BUFFER_SIZE];

typedef char * string;            /* Constant strings used in display.     */
static string hi = "Click!";

char * x_input_buffer       = NIL;
int    x_input_buffer_index = 0;

/*- F U N C T I O N S -----------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  redraw_x_display: Redraws the display showing the current state of the */
/*                    simulation.                                          */

void 
redraw_soar_x_display (Window soar_window) 
{
/*  XClearWindow(display, soar_window);       Clear window first of all.   */
  XFlush(display);            /* Flush events so updates seen at X server  */
}


void
note_window_resized (Window window, int width, int height)
{
  cons * c;
  agent * the_agent;

  for (c = all_soar_agents; c != NIL; c = c->rest) {
    the_agent = (agent *) c->first;
    if (the_agent->X->window == window) {
      the_agent->X->width = width;
      the_agent->X->height = height;
    }
  }
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  handle_cursor_event: This function is given a cursor key event to      */
/*                       process.  Currently, the cursors handle management*/
/*                       of the panning paramters.                         */

static
void
handle_cursor_event(KeySym key) 
{
  print("\nThat cursor key doesn't mean anything to Soar");
  print("-- ignoring keystroke.\n");
}


bool 
command_is_complete (x_info * xi) 
{
  int i;
  int paren_count = 0;

  for (i=0; i < xi->input_buffer_index; i++) {
    switch (xi->input_buffer[i]) {
    case '(':       
      paren_count++;
      break;
    case ')':       
      paren_count--;
      break;
    }
  }
  return paren_count == 0;
}


bool
scroll_if_necessary (x_info * window)
{
  if (window->window_y >= window->height - CHAR_HEIGHT) {
    window->window_x = 0;
    window->window_y -= CHAR_HEIGHT;
    XCopyArea(display, window->window, window->window,
	      window->gc, 0, CHAR_HEIGHT,
	      window->width, window->height, 0, 0);
    XClearArea(display, window->window,
	       0, window->height - CHAR_HEIGHT,
	       0, 0, FALSE);
    return TRUE;
  } else {
    return FALSE;
  }
}


void
print_x_string(x_info * current_window, char * s)
{ char * c;
  static char buf[2000];
  int buf_start = 0;
  int buf_index = 0;
  int x_so_far;

  if (current_window) {
    x_so_far = current_window->window_x;
    for (c=s; *c!='\0'; c++) {
      if (x_so_far >= current_window->width - CHAR_WIDTH) {
	buf[buf_index] = '\0';
	XDrawImageString(display, current_window->window, current_window->gc,
			   current_window->window_x, current_window->window_y,
			   &buf[buf_start], buf_index - buf_start);
	buf_index++;
	buf_start = buf_index;
	current_window->window_x = 0;
	current_window->window_y += CHAR_HEIGHT;
	x_so_far = 0;
      }
      if (scroll_if_necessary(current_window)) {
	x_so_far = 0;
      }
      switch (*c) {
      case '\12':                         /* line feed       */
      case '\15':                         /* carriage return */
	if (buf_index) {
	  buf[buf_index] = '\0';
	  XDrawImageString(display, current_window->window, current_window->gc,
			   current_window->window_x, current_window->window_y,
			   &buf[buf_start], buf_index - buf_start);
	buf_index++;
	buf_start = buf_index;
	}
	current_window->window_x  =  0;
	current_window->window_y += CHAR_HEIGHT;
	x_so_far = 0;
	break;
      default:
	buf[buf_index++] = *c;
	x_so_far += CHAR_WIDTH;
/*	current_window->window_x += CHAR_WIDTH; */
      }
    }
    if (buf_index - buf_start) {
      buf[buf_index] = '\0';
      XDrawImageString(display, current_window->window, current_window->gc,
		       current_window->window_x, current_window->window_y,
		       &buf[buf_start], buf_index - buf_start);
      current_window->window_x += CHAR_WIDTH * (buf_index - buf_start);
    }
    XFlush(display);
  } else {
    fputs (s, stdout);
    fflush(stdout);
  }
}


#ifdef USE_STDARGS

void 
print_x_format_string (x_info * window, char * format, ...) {
  va_list args;
  buffer buf;

  va_start (args, format);
#else
void 
print_x_format_string (va_alist) va_dcl { 
  va_list args;
  x_info * window;
  char *format;
  buffer buf;

  va_start (args);
  window = va_arg(args, x_info *);
  format = va_arg(args, char *);
#endif
  vsprintf (buf, format, args);
  va_end (args);
  print_x_string (window, buf);
}


void
print_agent_prompt (agent * agent_for_prompt)
{
  print_x_format_string (agent_for_prompt->X, "\nSoar agent %s> ",
                         agent_for_prompt->name);
}


void
add_buffer_to_text_io_queue (agent * agent_receiving_text_io, x_info * x_data)
{
  
  queue_add(agent_receiving_text_io->text_input_queue,
	    make_memory_block_for_string (x_data->text_input_buffer));
  x_data->text_input_buffer_index = 0;
}

bool text_io_mode;

void 
send_buffer_to_soar_agent (agent * agent_to_get_command) 
{
  x_info * xi;
  agent * prev_agent;

  prev_agent = soar_agent;
  soar_agent = agent_to_get_command; 

  xi = agent_to_get_command->X;

  if (xi->input_buffer_index != 0) {
    xi->input_buffer[xi->input_buffer_index] = '\n';

    x_input_buffer = xi->input_buffer;
    x_input_buffer_index = 0;
  
    /* --- consume rparen from previous command, get start of next cmd. --- */
    get_lexeme();
    if (current_agent(lexeme).type==EOF_LEXEME) return;

    /* --- if not lparen, fake one at end of the current line --- */
    if (current_agent(lexeme).type==L_PAREN_LEXEME) {
      get_lexeme(); /* consume lparen */
    } else {
      fake_rparen_at_next_end_of_line ();
    }
    
    if (current_agent(lexeme).type==SYM_CONSTANT_LEXEME) {
      text_io_mode = TRUE;
      dispatch_command();
      text_io_mode = FALSE;
    } else {
      print ("Error:  unknown command %s\n", current_agent(lexeme).string);
      print_location_of_most_recent_lexeme();
      skip_ahead_to_balanced_parentheses(0);
    }

  }

  x_input_buffer = NIL;    

  xi->input_buffer_index = 0;

  print_agent_prompt(soar_agent);

  soar_agent = prev_agent;
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  handle_keystroke_event: This procedure handles a single character key- */
/*                          stroke event.                                  */

static
void
handle_keystroke_event(Window soar_window, char keystroke)
{
  x_info * x_data;
  char buf[2];
  cons * c;
  agent * agent_for_window;
  bool window_found = FALSE;

  if (global_agent->X->window == soar_window) {
    window_found = TRUE;
    x_data = global_agent->X;
    agent_for_window = global_agent;
  } else {
    for(c = all_soar_agents; c != NIL; c = c->rest) {
      agent_for_window = (agent *) c->first;
      x_data = agent_for_window->X;
      if (x_data->window == soar_window) {
	window_found = TRUE;
	break;
      }
    }
  }

/*
  if (agent_for_window->stop_soar) {
    text_io_mode = FALSE;
  } else {
    text_io_mode = TRUE;
  }
*/
  if (window_found) {
    switch (keystroke) {
    case '\3' :                        /* \3 is a cntl-C which stops Soar   */
      control_c_handler (0);
      break;
    case '\15':                        /* \15 is a carriage return          */
      x_data->window_x = 0;
      x_data->window_y += CHAR_HEIGHT;
      scroll_if_necessary (x_data);

      if (text_io_mode) {
	x_data->text_input_buffer[x_data->text_input_buffer_index++] 
	  = 0;
	add_buffer_to_text_io_queue(agent_for_window, x_data);
      } else {
	if (command_is_complete(x_data)) {
	  send_buffer_to_soar_agent(agent_for_window);
	} else {
	  x_data->input_buffer[x_data->input_buffer_index++] = ' ';
	}
      }
      break;
    case 0:                           /* Ignore null char events.           */
      break;
    case '\10':                       /* Backspace */
    case '\177':                      /* DEL       */
      if (x_data->window_x > 0 
	  && (   (text_io_mode && x_data->text_input_buffer_index > 0)
	      || (!text_io_mode && x_data->input_buffer_index > 0)))
	{
	buf[0] = ' ';
	buf[1] = 0;
	if (text_io_mode) {
	  x_data->text_input_buffer[--x_data->text_input_buffer_index] = buf[0];
	} else {
	  x_data->input_buffer[--x_data->input_buffer_index] = buf[0];
	}
	x_data->window_x -= CHAR_WIDTH;  /* font char width */
	XDrawImageString(display, soar_window, x_data->gc, 
			 x_data->window_x, x_data->window_y, 
			 buf, 1);
	XFlush(display);
      }
      break;
    default:
      buf[0] = keystroke;
      buf[1] = 0;
      XDrawImageString(display, soar_window, x_data->gc, 
		       x_data->window_x, x_data->window_y, 
		       buf, 1);
      XFlush(display);
      if (text_io_mode) {
	if (x_data->text_input_buffer_index >= MAX_TEXT_INPUT_LINE_LENGTH) {
	  print ("\nText Input Error: input line too long, truncated it.\n");	
	} else {
	  x_data->text_input_buffer[x_data->text_input_buffer_index++] 
	    = keystroke;
	}
      } else {
	x_data->input_buffer[x_data->input_buffer_index++] = keystroke;
      }
      x_data->window_x += CHAR_WIDTH;  /* font char width */
    }
  }
}
  

/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  handle_x_events: Reads and processes any pending X events for the      */
/*                   graphics window.  This routine uses non-wait I/O.     */

void 
handle_soar_x_events(void)
{
  XEvent event;               /* X event sent to graphics window           */
  KeySym  key;                /* Code assoc. with user's keystroke         */
  int event_count;            /* Number of events pending in event queue   */
  int char_count;             /* Number of characters in keystroke         */
  buffer text;                /* Actual characters in keystroke            */

                              /* Process all events pending in the X-event */
                              /* queue.                                    */

  for (event_count = XEventsQueued(display, QueuedAfterFlush) 
       ; event_count != 0 
       ; event_count = XEventsQueued(display, QueuedAlready) ) {
 
    XNextEvent(display, &event);      /* Get the next X event in the queue */

    switch (event.type) {
    case Expose:                      /* Window was exposed... redraw:     */
      if (!event.xexpose.count) {     
	redraw_soar_x_display(event.xexpose.window);           
      } 
      break;
    case MappingNotify:               /* Don't ask.                        */
      XRefreshKeyboardMapping(&(event.xmapping));
      break;
    case ButtonPress:                 /* Stub for mouse clicks.            */
      XDrawImageString(event.xbutton.display, event.xbutton.window, 
		       global_agent->X->gc, 
		       event.xbutton.x, event.xbutton.y, hi, strlen(hi));
      break;
    case KeyPress:                    /* Regular keystroke...              */
      char_count = XLookupString(&(event.xkey), text, BUFFER_SIZE, &key, 0);

      if (IsCursorKey(key))           /* Cursor movement keys alter the    */
	handle_cursor_event(key);     /* pan settings.                     */
      else if (IsModifierKey(key))
        ;                             /* Ignore modifier keys */ 
      else if (char_count == 1)       /* Here we've got other actions      */
	handle_keystroke_event(event.xkey.window, text[0]);
      break;
    case GraphicsExpose:              /* XCopyArea side effect - ignore.   */
    case NoExpose:
      break;
    case CirculateNotify:
    case DestroyNotify:
    case GravityNotify:
    case MapNotify:
    case ReparentNotify:
    case UnmapNotify:
      break;
    case ConfigureNotify:
      note_window_resized (event.xconfigure.window,
			   event.xconfigure.width,
			   event.xconfigure.height);
      break;
    default:                          /* Unknown event seen, so stop.      */
      fprintf(stderr, "\nUnknown event seen!  Type = %d\n", event.type); 
      return;
    } /* switch */
  } /* for */
}


void 
wait_for_exposure (Window window)
{
  XEvent event;               /* X event sent to graphics window           */

  while(TRUE)
    {
      XNextEvent(display, &event);      /* Get the next X event in the queue */

      if (event.type == Expose) {       /* Window was exposed... redraw:     */
	if (event.xexpose.window == window && !event.xexpose.count) {     
	  return;
	} 
      }
    }
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  create_agent_window: This procedure creates the window used to display */
/*                      the command line interaction with each agent.      */

void
create_agent_window(agent * soar_agent)
{
  buffer title;
                              /* Various X window vars decl'd for calls to */
                              /* X init routines:                          */
  int screen;                 /* Code for current screen.                  */
  unsigned long foreground, background;                /* Colors */
  XSizeHints  h;              /* Window sizeing information.               */
  int argc;                   /* Used to simulate command line input below */
  char *argv[1];
  XGCValues values;           /* Graphics context values.                  */
  unsigned long valuemask;    /* var used to set window values.            */
  XSetWindowAttributes xswa;  /* Window attribute helper var.              */
  x_info *        axi;
  Font font;

  screen = DefaultScreen(display);           /* Get the current screen.    */

  background = WhitePixel(display, screen);    
  foreground = BlackPixel(display, screen);
                             
  h.x = WINDOW_UPPER_LEFT_X;
  h.y = WINDOW_UPPER_LEFT_Y 
        + ((soar_agent->index_num) * (WINDOW_HEIGHT + 30));
  h.width = WINDOW_WIDTH;
  h.height = WINDOW_HEIGHT;
  h.flags = PPosition | PSize;

  soar_agent->X = (x_info *) malloc (sizeof(x_info));
  soar_agent->X->window  = XCreateSimpleWindow(display, 
					    DefaultRootWindow(display),
					    h.x, h.y, h.width, h.height, 5, 
					    foreground, background);

  argc = 1;                  /* Setup fake command line to set up window   */
  argv[0] = "soar-agent";    /* title bar.                                 */
  sprintf(title, "Agent %s", soar_agent->name);
  XSetStandardProperties(display, soar_agent->X->window, title, title,
			 None, argv, argc, &h);

                             /* Set up mode for drawing.                   */
  valuemask = GCFunction | GCForeground;
  values.function = GXcopy;
  values.foreground = background ^ foreground;
  soar_agent->X->gc = XCreateGC(display, soar_agent->X->window, 
				valuemask, &values);

  font = XLoadFont (display, fontname);
  XSetFont(display, soar_agent->X->gc, font);

                             /* Setup fore/background colors.              */
  XSetBackground(display, soar_agent->X->gc, background);
  XSetForeground(display, soar_agent->X->gc, foreground);

                             /* Set events we'll be accepting.             */
  XSelectInput(display, soar_agent->X->window, 
        KeyPressMask | ExposureMask | StructureNotifyMask);

                             /* Set backing store to cut down on expose    */
                             /* events and produce faster restoration of   */
                             /* window.                                    */
  xswa.backing_store = WhenMapped;
  valuemask = CWBackingStore;
  XChangeWindowAttributes(display, soar_agent->X->window, valuemask, &xswa);

  soar_agent->X->input_buffer_index = 0;
  soar_agent->X->text_input_buffer_index = 0;
  soar_agent->X->window_x = 0;
  soar_agent->X->window_y = 0;
  soar_agent->X->width = h.width;
  soar_agent->X->height = h.height;

  XMapRaised(display, soar_agent->X->window);
  wait_for_exposure(soar_agent->X->window);
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  create_display: Creates and shows to the user the X window used for    */
/*                  the graphic display.                                   */

void 
create_global_display (void) 
{
  int i, j;                   /* Loop indices.                             */

  string title = "Global Soar Control";
                              /* Various X window vars decl'd for calls to */
                              /* X init routines:                          */
  int screen;                 /* Code for current screen.                  */
  unsigned long foreground, background;                /* Colors */
  XSizeHints  h;              /* Window sizeing information.               */
  int argc;                   /* Used to simulate command line input below */
  char *argv[1];
  XGCValues values;           /* Graphics context values.                  */
  unsigned long valuemask;    /* var used to set GC values.                */
  XSetWindowAttributes xswa;  /* Window attribute helper var.              */
  Colormap cmap;              /* Color map.                                */
  XColor exact;               /* Color struct.                             */
  static char dot[] = {2,2};  /* Encoding for (pseudo)dotted lines.        */
  Font font;  
  char * displayname = "";

  display = XOpenDisplay(displayname);       /* Open the X display         */

  if (display == (Display *) 0) {
    fprintf (stderr, "Cannot establish display connection.\n");
    fprintf (stderr, "Is your DISPLAY environment variable set properly?\n");
    fprintf (stderr, "Terminating program...\n");
    exit(1);
  }

  screen = DefaultScreen(display);           /* Get the current screen.    */
                                             /* Get fore/background colors */
  background = WhitePixel(display, screen);    
  foreground = BlackPixel(display, screen);

  h.x = WINDOW_UPPER_LEFT_X;  /* Set the window position and dimensions.   */
  h.y = WINDOW_UPPER_LEFT_Y;
  h.width  = WINDOW_WIDTH;
  h.height = WINDOW_HEIGHT;
  h.flags = PPosition | PSize;

  global_agent = soar_agent;

                             /* Create a window using the above data.      */
  global_agent->X = (x_info *) malloc (sizeof(x_info));

  global_agent->X->window = 
    XCreateSimpleWindow(display, DefaultRootWindow(display),
			h.x, h.y, h.width, h.height, 5, 
			foreground, background);

  global_agent->X->input_buffer_index = 0;
  global_agent->X->window_x = 0;
  global_agent->X->window_y = 0;
  global_agent->X->width = h.width;
  global_agent->X->height = h.height;

  argc = 1;                  /* Setup fake command line to set up window   */
  argv[0] = "soar";          /* title bar.                                 */
  XSetStandardProperties(display, global_agent->X->window, 
			 title, title, None, argv, argc, &h);

                             /* Set up Xor mode for drawing.               */
  valuemask = GCFunction | GCForeground;
  values.function = GXcopy;
  values.foreground = background ^ foreground;
  global_agent->X->gc = XCreateGC(display, global_agent->X->window, valuemask, &values);

  font = XLoadFont (display, fontname);
  XSetFont(display, global_agent->X->gc, font);

                             /* Set up video.                              */
  XSetBackground(display, global_agent->X->gc, background);
  XSetForeground(display, global_agent->X->gc, foreground);
                             /* Set events we'll be accepting.             */
  XSelectInput(display, global_agent->X->window, 
       KeyPressMask | ExposureMask | StructureNotifyMask);
                             /* Set backing store to cut down on expose    */
                             /* events and produce faster restoration of   */
                             /* window.                                    */
  xswa.backing_store = WhenMapped;
  valuemask = CWBackingStore;
  XChangeWindowAttributes(display, global_agent->X->window, valuemask, &xswa);
                             /* Find colors to use in displaying graphic   */
                             /* objects.                                   */
                             /* Set dot/dash style to use in some of the   */
                             /* line drawing options.                      */

                             /* Construct array of grid lines.             */

  XMapRaised(display, global_agent->X->window);
  wait_for_exposure(global_agent->X->window);
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*  destroy_display: Destroys the X window used for the graphics display   */
/*                   and performs other standard X cleanup actions.        */

void 
destroy_soar_display (void) 
{
  cons * c;
  agent * ag;

  XFreeGC(display, global_agent->X->gc);
  XDestroyWindow(display, global_agent->X->window);

  for (c = all_soar_agents; c != NIL; c = c->rest) {
    ag = (agent *) c->first;
    XFreeGC(display, ag->X->gc);
    XDestroyWindow(display, ag->X->window);
  }

  XCloseDisplay(display);
}

#endif

/*- E N D   O F   F I L E -------------------------------------------------*/

