/* include file for all of Soar 6 */

#include <stdio.h>
#include <ctype.h>
#ifdef USE_STDARGS
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <memory.h>
#include <errno.h>

/* ------------------------------ */
/* Global type declarations, etc. */
/* ------------------------------ */

typedef unsigned char byte;
typedef unsigned long ulong;
typedef unsigned short ushort;
typedef char bool;
#define TRUE (-1)
#define FALSE (0)

#define NIL (0)

/* ------------------------------------- */
/* memory allocation and other utilities */
/* ------------------------------------- */

extern void *allocate_memory (ulong size);
extern void *allocate_memory_and_zerofill (ulong size);
extern void free_memory (void *mem);

extern char *make_memory_block_for_string (char *s);

/* ------------------------------ */
/* Lexer definitions and routines */
/* ------------------------------ */

typedef enum lexer_token_type_enum {
  EOF_LEXEME,
  IDENTIFIER_LEXEME,
  VARIABLE_LEXEME,
  SYM_CONSTANT_LEXEME,
  INT_CONSTANT_LEXEME,
  FLOAT_CONSTANT_LEXEME,
  L_PAREN_LEXEME,
  R_PAREN_LEXEME,
  L_BRACE_LEXEME,
  R_BRACE_LEXEME,
  PLUS_LEXEME,
  MINUS_LEXEME,
  RIGHT_ARROW_LEXEME,
  GREATER_LEXEME,
  LESS_LEXEME,
  EQUAL_LEXEME,
  AMPERSAND_LEXEME,
  LESS_EQUAL_LEXEME,
  GREATER_EQUAL_LEXEME,
  NOT_EQUAL_LEXEME,
  LESS_EQUAL_GREATER_LEXEME,
  LESS_LESS_LEXEME,
  GREATER_GREATER_LEXEME,
  AT_LEXEME,
  TILDE_LEXEME,
  UP_ARROW_LEXEME,
  EXCLAMATION_POINT_LEXEME,
  COMMA_LEXEME,
  APOSTROPHE_LEXEME,
  DOUBLE_BACKSLASH_LEXEME,
  QUOTED_STRING_LEXEME } lexer_token_type;

#define LENGTH_OF_LONGEST_SPECIAL_LEXEME 3

extern void init_lexer (void);
extern void get_lexeme(void);

extern int max_lexeme_length;

extern lexer_token_type lexeme_type;
extern char lexeme_string[];
extern long lexeme_int;
extern float lexeme_float;
extern char lexeme_id_letter;
extern ulong lexeme_id_number;

extern bool lexer_allow_ids;

extern int current_lexer_parentheses_level (void);
extern void skip_ahead_to_balanced_parentheses (int parentheses_level);

/* ------------------------- */
/* working memory management */
/* ------------------------- */

/* WARNING: preference types must be numbered 0..(NUM_PREFERENCE_TYPES-1) */
#define NUM_PREFERENCE_TYPES 13  /* number of different preference types */

#define ACCEPTABLE_PREFERENCE_TYPE 0
#define REQUIRE_PREFERENCE_TYPE 1
#define REJECT_PREFERENCE_TYPE 2
#define PROHIBIT_PREFERENCE_TYPE 3
#define RECONSIDER_PREFERENCE_TYPE 4
#define UNARY_INDIFFERENT_PREFERENCE_TYPE 5
#define UNARY_PARALLEL_PREFERENCE_TYPE 6
#define BEST_PREFERENCE_TYPE 7
#define WORST_PREFERENCE_TYPE 8
#define BINARY_INDIFFERENT_PREFERENCE_TYPE 9
#define BINARY_PARALLEL_PREFERENCE_TYPE 10
#define BETTER_PREFERENCE_TYPE 11
#define WORSE_PREFERENCE_TYPE 12
#define preference_is_unary(p) ((p)<9)
#define preference_is_binary(p) ((p)>8)

/* --------------- */
/* top-level stuff */
/* --------------- */

extern void abort_with_fatal_error (void);

char supported_prod_name[5000];
bool o_support;
bool support_declaration_found;
bool inside_default = FALSE;

bool currently_commenting = FALSE;
int last_commented_line = -1;

/* ====================================================================

                   Basic Memory Allocation Utilities

==================================================================== */

/* Uncomment the following line to debug stuff */
#define DEBUG_MEMORY
#define GARBAGE_DEBUGGING_CHAR 0xBB

extern void *malloc (); /* BUGBUG include header file <stdlib.h> instead */

void *allocate_memory (ulong size) {
  char *p;

#ifdef DEBUG_MEMORY
  size += 4;
#endif
  
  p = (char *) malloc (size);
  if (p==NULL) {
    fprintf (stderr,"\n\nError:  Tried but failed to allocate %lu bytes of memory.\n", size);
    abort_with_fatal_error ();
    }

#ifdef DEBUG_MEMORY
  memset (p, GARBAGE_DEBUGGING_CHAR, size);
  *((ulong *)p) = size;
  p += 4;
#endif
  
  return (void *)p;
}

void *allocate_memory_and_zerofill (ulong size) {
  void *p;

  p = allocate_memory (size);
  memset (p, 0, size);
  return p;
}

void free_memory (void *mem) {

#ifdef DEBUG_MEMORY
  mem = ((char *)mem)-4;
  memset (mem, GARBAGE_DEBUGGING_CHAR, *((ulong *)mem));
#endif

  free (mem);
}

char *make_memory_block_for_string (char *s) {
  char *p;

  p = allocate_memory (strlen(s)+1);
  strcpy(p,s);
  return p;
}

/* =================================================================

                  Printing Utility Routines for Soar 6B

================================================================= */

#ifdef USE_STDARGS
void error (const char *format, ...) {
  va_list args;
  
  va_start (args, format);
#else
void error (va_alist) va_dcl {
  va_list args;
  char *format;

  va_start(args);
  format = va_arg(args, char *);
#endif
  fprintf (stdout, "\n; CONVERSION WARNING: ");
  vfprintf (stdout, format, args);
  va_end (args);

#ifdef USE_STDARGS
  va_start (args, format);
#else
  va_start(args);
  format = va_arg(args, char *);
#endif
  vfprintf (stderr, format, args);
  va_end (args);
}






/* lexer for Soar 6 */

bool constituent_char[256];
bool whitespace[256];
bool number_starters[256];

#define MAX_LINE_LENGTH 1024
#define MAX_LEXEME_LENGTH (MAX_LINE_LENGTH+5)  /* a little bigger, to avoid
                                                  any off-by-one-errors */
int max_lexeme_length = MAX_LEXEME_LENGTH;

lexer_token_type lexeme_type;
int lexeme_length;
char lexeme_string[MAX_LEXEME_LENGTH+1];
long lexeme_int;
float lexeme_float;
char lexeme_id_letter;
ulong lexeme_id_number;

bool lexer_allow_ids = FALSE;

char current_char;

#define BUFSIZE (MAX_LINE_LENGTH+2)  /* +2 for newline and null at end */

typedef struct lexer_source_file_struct {
  char *filename;
  FILE *file;
  int parentheses_level;
  int current_column;   /* column number of next char to read (0-based) */
  int column_of_start_of_last_lexeme;
  ulong current_line;   /* line number of line in buffer (1-based) */
  ulong line_of_start_of_last_lexeme;
  char buffer[BUFSIZE];
} lexer_source_file;

lexer_source_file current_file;

int current_lexer_parentheses_level (void) {
  return current_file.parentheses_level;
}

void get_next_char () {
  char *s;
  
  current_char = current_file.buffer [current_file.current_column++];
  if (current_char) return;

  if ((current_file.current_column == BUFSIZE) &&
      (current_file.buffer[BUFSIZE-2] != '\n') &&
      (current_file.buffer[BUFSIZE-2] != EOF)) {
    fprintf (stderr, "Error:  line too long (max allowed is %d chars)\n",
             MAX_LINE_LENGTH);
    fprintf (stderr, "Line %lu\n", current_file.current_line);
    abort_with_fatal_error();
  }

  s = fgets (current_file.buffer, BUFSIZE, current_file.file);
  
  if (s) {
    current_file.current_line++;
  } else {
    /* s==NIL means immediate eof encountered or read error occurred */
    if (! feof(current_file.file)) {
      fprintf (stderr, "I/O error while reading standard input!\n");
      abort_with_fatal_error();
    }
    current_file.buffer[0] = EOF;
    current_file.buffer[1] = 0;
  }
  current_char = current_file.buffer[0];
  current_file.current_column = 1;
}

void print_location_of_most_recent_lexeme (void) {
  int i;
  
  if (current_file.line_of_start_of_last_lexeme ==
      current_file.current_line) {
    /* --- error occurred on current line, so print out the line --- */
    fprintf (stderr, "Line %lu:\n", current_file.current_line);
    if (current_file.buffer[strlen(current_file.buffer)-1]=='\n')
      fprintf (stderr, "%s",current_file.buffer);
    else
      fprintf (stderr, "%s\n",current_file.buffer);
    for (i=0; i<current_file.column_of_start_of_last_lexeme; i++)
      fprintf (stderr, "-");
    fprintf (stderr, "^\n\n");
  } else {
    /* --- error occurred on a previous line, so just give the position --- */
    fprintf (stderr, "Line %lu, column %lu.\n\n",
             current_file.line_of_start_of_last_lexeme,
             current_file.column_of_start_of_last_lexeme + 1);
  }
}

#define record_position_of_start_of_lexeme() { \
  current_file.column_of_start_of_last_lexeme = \
    current_file.current_column - 1; \
  current_file.line_of_start_of_last_lexeme = \
    current_file.current_line; }




void (*(lexer_routines[256]))(void);




#define store_and_advance() { \
  lexeme_string[lexeme_length++] = (isupper(current_char) ? \
                                    tolower(current_char) : \
                                    current_char); \
  get_next_char(); }

#define finish() { lexeme_string[lexeme_length]=0; }

void read_constituent_string (void) {
  while ((current_char!=EOF) && constituent_char[current_char])
    store_and_advance();
  finish();
}  

double my_strtod (char *ch, char **p, int base) {
  /* BUGBUG doesn't set errno on overflow */
  return atof(ch);
}

/* ======================================================================
               Determine possible symbol types for string

This is a utility routine which figures out what kind(s) of symbol a 
given string could represent.  At entry:  s, length_of_s represent the
string.  At exit:  possible_xxx is set to TRUE/FALSE to indicate
whether the given string could represent that kind of symbol; rereadable
is set to TRUE indicating whether the lexer would read the given string
as a symbol with exactly the same name (as opposed to treating it as a
special lexeme like "+", changing upper to lower case, etc.
====================================================================== */

void determine_possible_symbol_types_for_string (char *s,
                                                 int length_of_s,
                                                 bool *possible_id,
                                                 bool *possible_var,
                                                 bool *possible_sc,
                                                 bool *possible_ic,
                                                 bool *possible_fc,
                                                 bool *rereadable) {
  char *ch;
  bool rereadability_dead, rereadability_questionable;

  *possible_id = FALSE;
  *possible_var = FALSE;
  *possible_sc = FALSE;
  *possible_ic = FALSE;
  *possible_fc = FALSE;
  *rereadable = FALSE;

  /* --- make sure it's entirely constituent characters --- */
  for (ch=s; *ch!=0; ch++) if (! constituent_char[*ch]) return;

  /* --- check for rereadability --- */
  rereadability_questionable = FALSE;
  rereadability_dead = FALSE;
  for (ch=s; *ch!=0; ch++) {
    if (islower(*ch) || isdigit(*ch)) continue; /* these guys are fine */
    if (isupper(*ch)) { rereadability_dead = TRUE; break; }
    rereadability_questionable = TRUE;
  }
  if (! rereadability_dead) {
    if ((! rereadability_questionable) ||
        (length_of_s >= LENGTH_OF_LONGEST_SPECIAL_LEXEME) ||
        ((length_of_s==1)&&(*s=='*')))
      *rereadable = TRUE;
  }

  /* --- any string of constituents could be a sym constant --- */
  *possible_sc = TRUE;
  
  /* --- check whether it's a variable --- */
  if ((*s=='<')&&(*(s+length_of_s-1)=='>')) *possible_var = TRUE;

  /* --- check if it's an integer or floating point number --- */
  if (number_starters[*s]) {
    ch = s;
    if ((*ch=='+')||(*ch=='-')) ch++;  /* optional leading + or - */
    while (isdigit(*ch)) ch++;         /* string of digits */
    if ((*ch==0)&&(isdigit(*(ch-1))))
      *possible_ic = TRUE;
    if (*ch=='.') {
      ch++;                              /* decimal point */
      while (isdigit(*ch)) ch++;         /* string of digits */
      if ((*ch=='e')||(*ch=='E')) {
        ch++;                              /* E */
        if ((*ch=='+')||(*ch=='-')) ch++;  /* optional leading + or - */
        while (isdigit(*ch)) ch++;         /* string of digits */
      }
      if (*ch==0) *possible_fc = TRUE;
    }
  }

  /* --- check if it's an identifier --- */
  if (isalpha(*s)) {
    /* --- is the rest of the string an integer? --- */
    ch = s+1;
    while (isdigit(*ch)) ch++;         /* string of digits */
    if ((*ch==0)&&(isdigit(*(ch-1)))) *possible_id = TRUE;
  }
}




void determine_type_of_constituent_string (void) {
  bool possible_id, possible_var, possible_sc, possible_ic, possible_fc;
  bool rereadable;
  
  determine_possible_symbol_types_for_string (lexeme_string,
                                              lexeme_length,
                                              &possible_id,
                                              &possible_var,
                                              &possible_sc,
                                              &possible_ic,
                                              &possible_fc,
                                              &rereadable);

  /* --- check whether it's a variable --- */
  if (possible_var) {
    lexeme_type = VARIABLE_LEXEME;
    return;
  }

  /* --- check whether it's an integer --- */
  if (possible_ic) {
    errno = 0;
    lexeme_type = INT_CONSTANT_LEXEME;
    lexeme_int = strtol (lexeme_string,NULL,10);
    if (errno) {
      error ("Error: bad integer (probably too large)\n");
      print_location_of_most_recent_lexeme();
      lexeme_int = 0;
    }
    return;
  }
    
  /* --- check whether it's a floating point number --- */
  if (possible_fc) {
    errno = 0;
    lexeme_type = FLOAT_CONSTANT_LEXEME;
    /* BUGBUG should use strtod, but I can't find it at CMU */
    lexeme_float = my_strtod (lexeme_string,NULL,10); 
    if (errno) {
      error ("Error: bad floating point number\n");
      print_location_of_most_recent_lexeme();
      lexeme_float = 0.0;
    }
    return;
  }
  
  /* --- otherwise it must be a symbolic constant --- */
  if (possible_sc) {
    lexeme_type = SYM_CONSTANT_LEXEME;
    return;
  }

  fprintf (stderr, "Internal error: can't determine_type_of_constituent_string\n");
  abort_with_fatal_error();
}






void lex_eof (void) {
  store_and_advance();
  finish();
  lexeme_type = EOF_LEXEME;
}

void lex_at (void) {
  store_and_advance();
  finish();
  lexeme_type = AT_LEXEME;
}

void lex_tilde (void) {
  store_and_advance();
  finish();
  lexeme_type = TILDE_LEXEME;
}

void lex_up_arrow (void) {
  store_and_advance();
  finish();
  lexeme_type = UP_ARROW_LEXEME;
}

void lex_lbrace (void) {
  store_and_advance();
  finish();
  lexeme_type = L_BRACE_LEXEME;
}

void lex_rbrace (void) {
  store_and_advance();
  finish();
  lexeme_type = R_BRACE_LEXEME;
}

void lex_exclamation_point (void) {
  store_and_advance();
  finish();
  lexeme_type = EXCLAMATION_POINT_LEXEME;
}

void lex_equal (void) {
  store_and_advance();
  finish();
  lexeme_type = EQUAL_LEXEME;
}

void lex_ampersand (void) {
  /* Lexeme might be "&", or symbol */
  /* Note: this routine relies on & being a constituent character */

  read_constituent_string();
  if (lexeme_length==1) { lexeme_type = AMPERSAND_LEXEME; return; }
  determine_type_of_constituent_string();
}

void lex_comma (void) {
  store_and_advance();
  finish();
  lexeme_type = COMMA_LEXEME;
}

void lex_lparen (void) {
  store_and_advance();
  finish();
  lexeme_type = L_PAREN_LEXEME;
  current_file.parentheses_level++;
}

void lex_rparen (void) {
  store_and_advance();
  finish();
  lexeme_type = R_PAREN_LEXEME;
  current_file.parentheses_level--;
}

void lex_apostrophe (void) {
  store_and_advance();
  finish();
  lexeme_type = APOSTROPHE_LEXEME;
}

extern void output_current_char(void);
extern bool output_previous_lexeme;

void lex_unknown (void) {
  fprintf (stderr, "Warning:  Unknown character encountered by lexer, code=%d, char='%c'\n", current_char, current_char);
  fprintf (stderr, "Line %lu, column %lu.\n", current_file.current_line,
           current_file.current_column);
  output_current_char();
  get_next_char();
  output_previous_lexeme = FALSE;
  get_lexeme();
}

void lex_backslash (void) {
  /* only valid lexeme is double backslash "\\" */
  store_and_advance();
  if(current_char != '\\') {
    error ("A '\\' was followed by something other than another '\\'\n");
    print_location_of_most_recent_lexeme();
  }
  store_and_advance();
  finish();
  lexeme_type = DOUBLE_BACKSLASH_LEXEME;
}

void lex_greater (void) {
  /* Lexeme might be ">", ">=", ">>", or symbol */
  /* Note: this routine relies on =,> being constituent characters */

  read_constituent_string();
  if (lexeme_length==1) { lexeme_type = GREATER_LEXEME; return; }
  if (lexeme_length==2) {
    if (lexeme_string[1]=='>') { lexeme_type = GREATER_GREATER_LEXEME; return;}
    if (lexeme_string[1]=='=') { lexeme_type = GREATER_EQUAL_LEXEME; return; }
  }
  determine_type_of_constituent_string();
}
    
void lex_less (void) {
  /* Lexeme might be "<", "<=", "<=>", "<>", "<<", or variable */
  /* Note: this routine relies on =,<,> being constituent characters */

  read_constituent_string();
  if (lexeme_length==1) { lexeme_type = LESS_LEXEME; return; }
  if (lexeme_length==2) {
    if (lexeme_string[1]=='>') { lexeme_type = NOT_EQUAL_LEXEME; return; }
    if (lexeme_string[1]=='=') { lexeme_type = LESS_EQUAL_LEXEME; return; }
    if (lexeme_string[1]=='<') { lexeme_type = LESS_LESS_LEXEME; return; }
  }
  if (lexeme_length==3) {
    if ((lexeme_string[1]=='=')&&(lexeme_string[2]=='>'))
      { lexeme_type = LESS_EQUAL_GREATER_LEXEME; return; }
  }
  determine_type_of_constituent_string();
}

void lex_plus (void) {
  /* Lexeme might be +, number, or symbol */
  /* Note: this routine relies on various things being constituent chars */
  
  read_constituent_string();
  if (lexeme_length==1) { lexeme_type = PLUS_LEXEME; return; }
  determine_type_of_constituent_string();
}
      
void lex_minus (void) {
  /* Lexeme might be -, -->, number, or symbol */
  /* Note: this routine relies on various things being constituent chars */

  read_constituent_string();
  if (lexeme_length==1) { lexeme_type = MINUS_LEXEME; return; }
  if (lexeme_length==3) {
    if ((lexeme_string[1]=='-')&&(lexeme_string[2]=='>'))
      { lexeme_type = RIGHT_ARROW_LEXEME; return; }
  }
  determine_type_of_constituent_string();
}

void lex_constituent_string (void) {
  read_constituent_string();
  determine_type_of_constituent_string();
}

void lex_vbar (void) {
  lexeme_type = SYM_CONSTANT_LEXEME;
  get_next_char();
  do {
    if ((current_char==EOF)||(lexeme_length==MAX_LEXEME_LENGTH)) {
      error ("Opening '|' without closing '|'\n");
      print_location_of_most_recent_lexeme();
      /* BUGBUG if reading from top level, don't want to signal EOF */
      lexeme_type = EOF_LEXEME;
      lexeme_string[0]=EOF;
      lexeme_string[1]=0;
      lexeme_length = 1;
      return;
    }
    if (current_char=='\\') {
      get_next_char();
      lexeme_string[lexeme_length++]=current_char;
      get_next_char();
    } else if (current_char=='|') {
      get_next_char();
      break;
    } else {
      lexeme_string[lexeme_length++]=current_char;
      get_next_char();
    }
  } while(TRUE);
  lexeme_string[lexeme_length]=0;
}

void lex_quote (void) {
  lexeme_type = QUOTED_STRING_LEXEME;
  get_next_char();
  do {
    if ((current_char==EOF)||(lexeme_length==MAX_LEXEME_LENGTH)) {
      error ("Opening '\"' without closing '\"'\n");
      print_location_of_most_recent_lexeme();
      /* BUGBUG if reading from top level, don't want to signal EOF */
      lexeme_type = EOF_LEXEME;
      lexeme_string[0]=EOF;
      lexeme_string[1]=0;
      lexeme_length = 1;
      return;
    }
    if (current_char=='\\') {
      get_next_char();
      lexeme_string[lexeme_length++]=current_char;
      get_next_char();
    } else if (current_char=='"') {
      get_next_char();
      break;
    } else {
      lexeme_string[lexeme_length++]=current_char;
      get_next_char();
    }
  } while(TRUE);
  lexeme_string[lexeme_length]=0;
}



void output_current_char (void) {
  /* In order to get the comment characters at the beginning of the line, I
   * make the assumption that you won't ever be commenting out something in
   * the middle of a command.. if this is a bogus assumption, it can most
   * likely be worked around later with an extra variable.
   */
  if(currently_commenting &&
     last_commented_line != current_file.current_line &&
     current_file.parentheses_level != 0)  {
    last_commented_line = current_file.current_line;
    printf(";; ");
  }
  printf ("%c",current_char);
}

void output_lexeme_string_without_periods(void) {
  int i;

  for(i=0; i < lexeme_length; i++) {
    if(lexeme_string[i] != '.')
      printf("%c", lexeme_string[i]);
  }
}

void output_lexeme_string_with_escapes (char first_and_last_char) {
  int i;
  
  printf ("%c",first_and_last_char);
  for (i=0; i<lexeme_length; i++) {
    if ((lexeme_string[i]==first_and_last_char) ||
        (lexeme_string[i]=='\\')) printf ("\\");
    printf ("%c",lexeme_string[i]);
  }
  printf ("%c",first_and_last_char);
}

void output_current_lexeme (void) {
  char *c;
  bool possible_id, possible_var, possible_sc, possible_ic, possible_fc;
  bool rereadable;

  /* this doesn't need the check for parentheses level because you cannot get
   * a straight lexeme at the top level that hasn't already been handled and
   * decided whether it should be commented or not at the top level read and
   * dispatch.
   */
  if(currently_commenting &&
     last_commented_line != current_file.current_line)  {
    last_commented_line = current_file.current_line;
    printf(";; ");
  }
  switch (lexeme_type) {
  case SYM_CONSTANT_LEXEME:
    determine_possible_symbol_types_for_string (lexeme_string,
                                                strlen (lexeme_string),
                                                &possible_id,
                                                &possible_var,
                                                &possible_sc,
                                                &possible_ic,
                                                &possible_fc,
                                                &rereadable);
    if ((!possible_sc) || possible_var || possible_ic || possible_fc ||
        (!rereadable)) {
      output_lexeme_string_with_escapes ('|');
    } else if(possible_sc && index(lexeme_string, '.') != NULL) {
      output_lexeme_string_with_escapes ('|');
    } else {
      printf ("%s",lexeme_string);
    }
    return;
    
  case QUOTED_STRING_LEXEME:
    output_lexeme_string_with_escapes ('"');
    return;
   
  case VARIABLE_LEXEME:
    output_lexeme_string_without_periods();
    return;
 
  default:  /* for everything else */
    printf ("%s",lexeme_string);
    return;
  }
}



/* Comments in Soar source code:  from semicolon to end-of-line, or  */
/* from '#|' to next '|#' */


bool output_previous_lexeme = FALSE;

void get_lexeme (void) {
  if (output_previous_lexeme) output_current_lexeme();
  lexeme_length = 0;
  lexeme_string[0] = 0;
  while (TRUE) {
    if (current_char==EOF) break;
    if (whitespace[current_char]) {
      if (output_previous_lexeme) output_current_char();
      get_next_char();
      continue;
    }
    if (current_char==';') {
      output_previous_lexeme = TRUE;
      while ((current_char!='\n')&&(current_char!=EOF)) {
        output_current_char();
        get_next_char();
      }
      continue;
    }

    if (current_char=='#') {
      output_previous_lexeme = TRUE;
      record_position_of_start_of_lexeme(); /* in case of later error mesg. */
      output_current_char();
      get_next_char();
      if (current_char!='|') {
        error ("Error: '#' not followed by '|'\n");
        print_location_of_most_recent_lexeme();
        continue;
      }
      output_current_char();
      get_next_char();  /* consume the vbar */
      while (TRUE) {
        if (current_char==EOF) {
          error ("Error: '#|' without terminating '|#'\n");
          print_location_of_most_recent_lexeme();
          break;
        }
        if (current_char!='|') {
          output_current_char();
          get_next_char();
          continue;
        }
        output_current_char();
        get_next_char();
        if (current_char=='#') break;
      }
      output_current_char();
      get_next_char();  /* consume the closing '#' */
      continue; /* continue outer while(TRUE), reading more whitespace */
    }
    break; /* if no whitespace, ';', or '#|' found, break out of the loop */
  }
  record_position_of_start_of_lexeme();
  if (current_char!=EOF) (*(lexer_routines[current_char]))();
    else lex_eof();
  if (lexeme_type==SYM_CONSTANT_LEXEME)
    if ( (!strcmp(lexeme_string,"soar::*unbound*")) ||
         (!strcmp(lexeme_string,"SOAR::*UNBOUND*")) ) {
      output_previous_lexeme = FALSE;
      get_lexeme();
    }
  output_previous_lexeme = TRUE;
}
  
  

char constituents[] = "$%&*+-./:<=>?_";

void init_lexer (void) {
  int i;

  /* --- setup constituent_char array --- */
  for (i=0; i<256; i++) constituent_char[i]=FALSE;
  for (i='0'; i<='9'; i++) constituent_char[i]=TRUE;
  for (i='A'; i<='Z'; i++) constituent_char[i]=TRUE;
  for (i='a'; i<='z'; i++) constituent_char[i]=TRUE;
  for (i=0; i<strlen(constituents); i++) {
    constituent_char[constituents[i]]=TRUE;
  }
  
  /* --- setup whitespace array --- */
  for (i=0; i<256; i++) whitespace[i]=FALSE;
  for (i=1; i<13; i++) whitespace[i]=TRUE;
  whitespace[' ']=TRUE;

  /* --- setup number_starters array --- */
  for (i=0; i<256; i++) number_starters[i]=FALSE;
  for (i='0'; i<='9'; i++) number_starters[i]=TRUE;
  number_starters['+']=TRUE;
  number_starters['-']=TRUE;
  number_starters['.']=TRUE;

  /* --- setup lexer_routines array --- */
  for (i=0; i<256; i++) lexer_routines[i] = lex_unknown;
  for (i=0; i<256; i++)
    if (constituent_char[i]) lexer_routines[i] = lex_constituent_string;
  lexer_routines['@'] = lex_at;
  lexer_routines[','] = lex_comma;
  lexer_routines['('] = lex_lparen;
  lexer_routines[')'] = lex_rparen;
  lexer_routines['+'] = lex_plus;
  lexer_routines['-'] = lex_minus;
  lexer_routines['~'] = lex_tilde;
  lexer_routines['^'] = lex_up_arrow;
  lexer_routines['{'] = lex_lbrace;
  lexer_routines['}'] = lex_rbrace;
  lexer_routines['!'] = lex_exclamation_point;
  lexer_routines['>'] = lex_greater;
  lexer_routines['<'] = lex_less;
  lexer_routines['='] = lex_equal;
  lexer_routines['&'] = lex_ampersand;
  lexer_routines['|'] = lex_vbar;
  lexer_routines['"'] = lex_quote;
  lexer_routines['\''] = lex_apostrophe;
  lexer_routines['\\'] = lex_backslash;

  /* --- initially we're reading from the standard input --- */
  current_file.file = stdin;
  current_file.parentheses_level = 0;
  current_file.column_of_start_of_last_lexeme = 0;
  current_file.line_of_start_of_last_lexeme = 0;
  current_file.current_line = 0;
  current_file.current_column = 0;
  current_file.buffer[0] = 0;
  current_char = ' ';    /* whitespace, return--to force read of first line */
}


void skip_ahead_to_balanced_parentheses (int parentheses_level) {
  while (TRUE) {
    output_previous_lexeme = TRUE;
    if (lexeme_type==EOF_LEXEME) return;
    if ((lexeme_type==R_PAREN_LEXEME) &&
        (parentheses_level==current_file.parentheses_level)) return;
    get_lexeme();
  }
}

void eat_until_balanced_parentheses (int parentheses_level) {
  while(TRUE) {
    output_previous_lexeme = FALSE;
    if (lexeme_type==EOF_LEXEME) return;
    if ((lexeme_type==R_PAREN_LEXEME) &&
        (parentheses_level==current_file.parentheses_level)) return;
    get_lexeme();
  }
}

/* this function should not be called except at the top level.  Trying to
 * comment out parts inside a command currently won't work.
 * However, the modifications to make it work inside a command if needed
 * should be trivial, so we'll leave the parameter here.  Currently though
 * it's always 0.
 */
void comment_until_balanced_parentheses (int parentheses_level) {
  currently_commenting = TRUE;
  while(TRUE) {
    output_previous_lexeme = TRUE;
    if (lexeme_type==EOF_LEXEME) return;
    if ((lexeme_type==R_PAREN_LEXEME) &&
        (parentheses_level==current_file.parentheses_level)) return;
    get_lexeme();
  }
}

/* Production parser */

/* =================================================================

 Grammar for left hand sides of productions

 <lhs> ::= <cond>+
 <cond> ::= <pos_cond> | - <pos_cond>
 <pos_cond> ::= <conds_for_one_id> | { <cond>+ }
 <conds_for_one_id> ::= ( <id_test> <attr_value_test>* )
 <id_test> ::= <test>
 <attr_value_test> ::= [-] ^ <attr_test> <value_test>*
 <attr_test> ::= <test>
 <value_test> ::= <test> [+]

 <test> ::= <conjunctive_test> | <simple_test>
 <conjunctive_test> ::= { <simple_test>+ }
 <simple_test> ::= <disjunction_test> | <relational_test>
 <disjunction_test> ::= << <constant>* >>
 <relational_test> ::= [<relation>] <single_test>
 <relation> ::= <> | < | > | <= | >= | = | <=>
 <single_test> ::= variable | <constant>
 <constant> ::= sym_constant | int_constant | float_constant

================================================================= */



/* =================================================================

                          Routines for Tests

================================================================= */

bool parse_relational_test (void) {
  
  /* --- read optional relation symbol --- */
  switch(lexeme_type) {
  case EQUAL_LEXEME:
  case NOT_EQUAL_LEXEME:
  case LESS_LEXEME:
  case GREATER_LEXEME:
  case LESS_EQUAL_LEXEME:
  case GREATER_EQUAL_LEXEME:
  case LESS_EQUAL_GREATER_LEXEME:
    get_lexeme();
    break;

  default:
    break;
  }

  /* --- read variable or constant --- */
  switch(lexeme_type) {
  case SYM_CONSTANT_LEXEME:
  case INT_CONSTANT_LEXEME:
  case FLOAT_CONSTANT_LEXEME:
  case VARIABLE_LEXEME:
    get_lexeme();
    return TRUE;

  default:
    error ("Expected variable or constant for test\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
}

bool parse_disjunction_test (void) {

  if (lexeme_type!=LESS_LESS_LEXEME) {
    error ("Expected << to begin disjunction test\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();

  while (lexeme_type!=GREATER_GREATER_LEXEME) {
    switch (lexeme_type) {
    case SYM_CONSTANT_LEXEME:
    case INT_CONSTANT_LEXEME:
    case FLOAT_CONSTANT_LEXEME:
      get_lexeme();
      break;
    default:
      error ("Expected constant or >> while reading disjunction test\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
  }
  get_lexeme();  /* consume the >> */
  return TRUE;
}

bool parse_simple_test (void) {
  if (lexeme_type==LESS_LESS_LEXEME)
    return parse_disjunction_test();
  return parse_relational_test();
}

bool parse_test (void) {
  bool temp;

  if (lexeme_type!=L_BRACE_LEXEME)
    return parse_simple_test();
  /* --- parse and return conjunctive test --- */
  get_lexeme();
  while (lexeme_type!=R_BRACE_LEXEME) {
    temp = parse_simple_test();
    if (!temp) {
      return FALSE;
    }
  }
  get_lexeme();
  return TRUE;
}

/* =================================================================

                        Routines for Conditions

================================================================= */

bool parse_value_test_star (void) {
  bool value_test;
  bool acceptable;

  if ((lexeme_type==MINUS_LEXEME) ||
      (lexeme_type==UP_ARROW_LEXEME) ||
      (lexeme_type==R_PAREN_LEXEME)) {
    /* --- value omitted, so create dummy (blank) value test --- */
    return TRUE;
  }

  do {  /* --- read <value_test>, build condition for it --- */
    value_test = parse_test();
    if (!value_test) {
      return FALSE;
    }
    if (lexeme_type==PLUS_LEXEME) {
      acceptable = TRUE;
      get_lexeme();
    } else {
      acceptable = FALSE;
    }
  } while ((lexeme_type!=MINUS_LEXEME) &&
           (lexeme_type!=UP_ARROW_LEXEME) &&
           (lexeme_type!=R_PAREN_LEXEME));
  return TRUE;
}

bool parse_conds_for_one_id (void) {
  bool c;
  bool id_test, attr_test;
  bool negate_it;
  bool eat_class_name;

  if (lexeme_type!=L_PAREN_LEXEME) {
    error ("Expected ( to begin condition element\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();

  eat_class_name = TRUE;
  if (!strcmp(lexeme_string,"goal")) eat_class_name = FALSE;
  else if (!strcmp(lexeme_string,"impasse")) eat_class_name = FALSE;
  if (lexeme_type!=SYM_CONSTANT_LEXEME) {
    error ("Found a non-constant in the class field of a condition.\n");
    error ("Classes are not used in Soar6.  You must hand-convert this.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  if (eat_class_name) output_previous_lexeme = FALSE;
  get_lexeme();
  
  id_test = parse_test();
  if (!id_test) return FALSE;

  while (lexeme_type!=R_PAREN_LEXEME) {
    if (lexeme_type==MINUS_LEXEME) {
      negate_it = TRUE;
      get_lexeme();
    } else {
      negate_it = FALSE;
    }
    if (lexeme_type!=UP_ARROW_LEXEME) {
      error ("Expected ^ after identifier\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
    get_lexeme();
    attr_test = parse_test();
    if (!attr_test) {
      return FALSE;
    }
    c = parse_value_test_star ();
    if (!c) {
      return FALSE;
    }
  }

  /* --- reached the end of the condition --- */
  get_lexeme();       /* consume the right parenthesis */
  return TRUE;
}

extern bool parse_cond_plus (void);

bool parse_cond (void) {
  bool c;
  bool negate_it;

  negate_it = FALSE;
  if (lexeme_type==MINUS_LEXEME) { negate_it = TRUE; get_lexeme(); }
  if (lexeme_type==L_BRACE_LEXEME) {
    /* --- read conjunctive condition --- */
    get_lexeme();
    c = parse_cond_plus();
    if (!c) return NIL;
    if (lexeme_type!=R_BRACE_LEXEME) {
      error ("Expected } to end conjunctive condition\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
    get_lexeme();  /* consume the R_BRACE */
  } else {
    /* --- read conds for one id --- */
    c = parse_conds_for_one_id();
    if (!c) return FALSE;
  }
  return TRUE;
}

bool parse_cond_plus (void) {
  bool c;

  do {
    /* --- get individual <cond> --- */
    c = parse_cond();
    if (!c) {
      return FALSE;
    }
    /* --- find last condition in c, make it point to prev_c --- */
  } while ((lexeme_type==MINUS_LEXEME) ||
           (lexeme_type==L_PAREN_LEXEME) ||
           (lexeme_type==L_BRACE_LEXEME));
  return TRUE;
}

bool parse_lhs (void) {
  return parse_cond_plus ();
}



/* =====================================================================

 Grammar for right hand sides of productions

 <rhs> ::= <rhs_action>+
 <rhs_action> ::= ( variable <attr_value_make>+ ) | <function_call>
 <function_call> ::= ( <function_name> <rhs_value>* )
 <function_name> ::= sym_constant | + | -
    BUGBUG might need others here if lexer changes
 <rhs_value> ::= <constant> | <function_call> | variable
 <constant> ::= sym_constant | int_constant | float_constant
 <attr_value_make> ::= ^ <variable_or_sym_constant> <value_make>+
 <variable_or_sym_constant> ::= variable | sym_constant
 <value_make> ::= <rhs_value> <preference_specifier>*

 <preference-specifier> ::= <naturally-unary-preference>
                          | <forced-unary-preference>
                          | <binary-preference> <rhs_value>
 <naturally-unary-preference> ::= + | - | ! | ~ | @
 <binary-preference> ::= > | = | < | &
 <forced-unary-preference> ::= <binary-preference> {, | ) | ^}  
    ;but the parser shouldn't consume the ")" or "^" lexeme here

===================================================================== */

bool parse_crlf_function (void) {
  printf ("(");
  get_lexeme();  /* consume the "crlf" */
  get_lexeme();  /* consume the rparen */
  return TRUE;
}
  
bool parse_tabto_function (void) {
  printf ("(");
  error ("Tabto function not implemented in Soar6\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}
  
bool parse_call2_function (void) {
  printf ("(");
  error ("Call2's must be hand-converted to Soar6\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

typedef struct expr_tree_struct {
  bool isa_leaf;
  lexer_token_type lexeme_type;  /* leaf argument, or operator token */
  char *lexeme_string;                /* leaf argument, or operator token */
  struct expr_tree_struct *op1, *op2;
} expr_tree;

extern expr_tree *parse_expression (void);

/*
   Grammar for compute's:

      <exp> ::= <factor> | <factor> <op> <exp>
      <factor> ::= numeric_constant | variable | ( <exp> )
      <op> ::= + | - | * | / | \
      <compute> ::= ( compute <exp> )

*/

expr_tree *parse_factor (void) {
  expr_tree *result;

  switch (lexeme_type) {
  case INT_CONSTANT_LEXEME:
  case FLOAT_CONSTANT_LEXEME:
  case VARIABLE_LEXEME:
    result = allocate_memory (sizeof(expr_tree));
    result->isa_leaf = TRUE;
    result->lexeme_type = lexeme_type;
    result->lexeme_string = make_memory_block_for_string(lexeme_string);
    output_previous_lexeme = FALSE;
    get_lexeme();
    return result;
    
  case L_PAREN_LEXEME:
    output_previous_lexeme = FALSE;
    get_lexeme();
    result = parse_expression();
    if (!result) return NIL;
    if (lexeme_type!=R_PAREN_LEXEME) {
      error ("Expected ) to end computed expression\n");
      print_location_of_most_recent_lexeme();
      return NIL;
    }
    output_previous_lexeme = FALSE;
    get_lexeme();
    return result;
    
  default:
    error ("Bad expression syntax\n");
    print_location_of_most_recent_lexeme();
    return NIL;
  }
}

expr_tree *parse_expression (void) {
  expr_tree *factor_tree, *result;

  factor_tree = parse_factor();
  if (!factor_tree) return NIL;
  switch (lexeme_type) {
  case PLUS_LEXEME: break;
  case MINUS_LEXEME: break;
  case DOUBLE_BACKSLASH_LEXEME: break;
  case SYM_CONSTANT_LEXEME:
    if (!strcmp(lexeme_string,"*")) break;
    if (!strcmp(lexeme_string,"/")) break;
    return factor_tree;
  default:
    return factor_tree;
  }
  result = allocate_memory (sizeof(expr_tree));
  result->isa_leaf = FALSE;
  result->lexeme_type = lexeme_type;
  if(lexeme_type == DOUBLE_BACKSLASH_LEXEME)
    result->lexeme_string = make_memory_block_for_string("mod");
  else
    result->lexeme_string = make_memory_block_for_string(lexeme_string);
  result->op1 = factor_tree;
  output_previous_lexeme = FALSE;
  get_lexeme();
  factor_tree = parse_expression();
  if (!factor_tree) return NIL;
  result->op2 = factor_tree;
  return result;
}

void output_expression (expr_tree *exp) {
  if (exp->isa_leaf) {
    lexeme_type = exp->lexeme_type;
    strcpy(lexeme_string, exp->lexeme_string);
    lexeme_length = strlen(lexeme_string);
    output_current_lexeme();
  } else {
    printf ("(");
    lexeme_type = exp->lexeme_type;
    strcpy(lexeme_string, exp->lexeme_string);
    lexeme_length = strlen(lexeme_string);
    output_current_lexeme();
    printf (" ");
    output_expression (exp->op1);
    printf (" ");
    output_expression (exp->op2);
    printf (")");
  }
}
  
bool parse_compute_function (void) {
  expr_tree *exp;
  
  output_previous_lexeme = FALSE;  /* eat the "compute" */
  get_lexeme();
  exp = parse_expression();
  if (!exp) return FALSE;
  if (lexeme_type!=R_PAREN_LEXEME) {
    error ("Expected ) to close computed expression\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  output_expression (exp);
  printf (" ");
  output_previous_lexeme = FALSE;
  get_lexeme();  /* consume the rparen */
}
  
bool parse_accept_function (void) {
  printf ("(");
  get_lexeme();  /* consume the "accept" */
  get_lexeme();  /* consume the rparen */
  return TRUE;
}
  

bool parse_rhs_value (void) {
  if (lexeme_type==L_PAREN_LEXEME) {
    output_previous_lexeme = FALSE;
    get_lexeme();
    if (!strcmp(lexeme_string,"crlf"))
      return parse_crlf_function();
    else if (!strcmp(lexeme_string,"tabto"))
      return parse_tabto_function();
    else if (!strcmp(lexeme_string,"compute"))
      return parse_compute_function();
    else if (!strcmp(lexeme_string,"accept"))
      return parse_accept_function();
    else if (!strcmp(lexeme_string,"call2"))
      return parse_call2_function();
    error ("Unknown RHS function name\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  if ((lexeme_type==SYM_CONSTANT_LEXEME) ||
      (lexeme_type==INT_CONSTANT_LEXEME) ||
      (lexeme_type==FLOAT_CONSTANT_LEXEME) ||
      (lexeme_type==VARIABLE_LEXEME)) {
    get_lexeme();
    return TRUE;
  }
  if (lexeme_type==QUOTED_STRING_LEXEME) {
    lexeme_type = SYM_CONSTANT_LEXEME;  /* change RHS strings into symbols */
    get_lexeme();
    return TRUE;
  }
  error ("Illegal value for RHS value\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

byte parse_preference_specifier (void) {
  switch (lexeme_type) {
    
  case PLUS_LEXEME:
    get_lexeme();
    return ACCEPTABLE_PREFERENCE_TYPE;
    
  case MINUS_LEXEME:
    get_lexeme();
    return REJECT_PREFERENCE_TYPE;
    
  case EXCLAMATION_POINT_LEXEME:
    get_lexeme();
    return REQUIRE_PREFERENCE_TYPE;
    
  case TILDE_LEXEME:
    get_lexeme();
    return PROHIBIT_PREFERENCE_TYPE;
    
  case AT_LEXEME:
    get_lexeme();
    return RECONSIDER_PREFERENCE_TYPE;
    
  case GREATER_LEXEME:
    get_lexeme();
    if ((lexeme_type!=COMMA_LEXEME) &&
        (lexeme_type!=R_PAREN_LEXEME) &&
        (lexeme_type!=UP_ARROW_LEXEME))
      return BETTER_PREFERENCE_TYPE;
    /* --- forced unary preference --- */
    if (lexeme_type==COMMA_LEXEME) get_lexeme();
    return BEST_PREFERENCE_TYPE;
    
  case EQUAL_LEXEME:
    get_lexeme();
    if ((lexeme_type!=COMMA_LEXEME) &&
        (lexeme_type!=R_PAREN_LEXEME) &&
        (lexeme_type!=UP_ARROW_LEXEME))
      return BINARY_INDIFFERENT_PREFERENCE_TYPE;
    /* --- forced unary preference --- */
    if (lexeme_type==COMMA_LEXEME) get_lexeme();
    return UNARY_INDIFFERENT_PREFERENCE_TYPE;
    
  case LESS_LEXEME:
    get_lexeme();
    if ((lexeme_type!=COMMA_LEXEME) &&
        (lexeme_type!=R_PAREN_LEXEME) &&
        (lexeme_type!=UP_ARROW_LEXEME))
      return WORSE_PREFERENCE_TYPE;
    /* --- forced unary preference --- */
    if (lexeme_type==COMMA_LEXEME) get_lexeme();
    return WORST_PREFERENCE_TYPE;
    
  case AMPERSAND_LEXEME:
    get_lexeme();
    if ((lexeme_type!=COMMA_LEXEME) &&
        (lexeme_type!=R_PAREN_LEXEME) &&
        (lexeme_type!=UP_ARROW_LEXEME))
      return BINARY_PARALLEL_PREFERENCE_TYPE;
    /* --- forced unary preference --- */
    if (lexeme_type==COMMA_LEXEME) get_lexeme();
    return UNARY_PARALLEL_PREFERENCE_TYPE;
    
  default:
    /* --- if no preference given, make it an acceptable preference --- */
    return ACCEPTABLE_PREFERENCE_TYPE;
  } /* end of switch statement */
}

bool parse_preference_specifier_star (void) {
  byte preference_type;
  bool saw_plus_sign;
  
  /* --- Note: this routine is set up so if there's not preference type
     indicator at all, we return a single acceptable preference make --- */

  preference_type = parse_preference_specifier ();
  
  while (TRUE) {
    /* --- read referent --- */
    if (preference_is_binary(preference_type)) {
      if (! parse_rhs_value()) {
        return FALSE;
      }
    }

    /* --- look for another preference type specifier --- */
    saw_plus_sign = (lexeme_type==PLUS_LEXEME);
    preference_type = parse_preference_specifier ();
    
    /* --- exit loop when done reading preferences --- */
    if ((preference_type==ACCEPTABLE_PREFERENCE_TYPE) &&
        (! saw_plus_sign))
      return TRUE;
  }
}

bool parse_rhs_attr_value_make (void) {
  bool new_actions;
  
  if (lexeme_type!=UP_ARROW_LEXEME) {
    error ("Expected ^ in RHS make action\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();
  if ((lexeme_type!=VARIABLE_LEXEME)&&(lexeme_type!=SYM_CONSTANT_LEXEME)) {
    error ("Expected variable or symbolic constant for attribute\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();
  
  do {
    if (!parse_rhs_value()) {
      return FALSE;
    }
    new_actions = parse_preference_specifier_star ();
    if (!new_actions) {
      return FALSE;
    }
  } while ((lexeme_type!=R_PAREN_LEXEME) &&
           (lexeme_type!=UP_ARROW_LEXEME));

  return TRUE;
}

bool parse_write1_action (void) {
  printf ("write ");
  output_previous_lexeme = FALSE;
  get_lexeme();  /* consume "write1" */
  while (lexeme_type!=R_PAREN_LEXEME) {
    if (!parse_rhs_value()) return FALSE;
    printf (" | | ");  /* add an extra space after each argument */
  }
  get_lexeme();  /* consume the rparen */
  return TRUE;  
}

bool parse_write2_action (void) {
  printf ("write ");
  output_previous_lexeme = FALSE;
  get_lexeme();  /* consume "write2" */
  while (lexeme_type!=R_PAREN_LEXEME) {
    if (!parse_rhs_value()) return FALSE;
  }
  get_lexeme();  /* consume the rparen */
  return TRUE;
}
  
bool parse_write1plus_action (void) {
  error ("Write1+ is not implemented in Soar6.  You must hand-convert it.\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}
  
bool parse_write2plus_action (void) {
  error ("Write2+ is not implemented in Soar6.  You must hand-convert it.\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

bool parse_tabstop_action (void) {
  error ("Tabstop is not implemented in Soar6.\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

bool parse_halt_action (void) {
  get_lexeme();  /* consume the "halt" */
  get_lexeme();  /* consume the rparen */
  return TRUE;
}
  
bool parse_bind_action (void) {
  error ("Bind is not implemented in Soar6.\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}
  
bool parse_call2_action (void) {
  error ("Call2's must be hand-converted to Soar6.\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

bool parse_rhs_action (void) {
  bool new_actions;
  
  if (lexeme_type!=L_PAREN_LEXEME) {
    if(lexeme_type!=R_PAREN_LEXEME) {
      error ("Expected ( to begin RHS action\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    } else
      return TRUE;
  }
  get_lexeme();
  
  if (!strcmp(lexeme_string,"write1"))
    return parse_write1_action();
  else if (!strcmp(lexeme_string,"write2"))
    return parse_write2_action();
  else if (!strcmp(lexeme_string,"write1+"))
    return parse_write1plus_action();
  else if (!strcmp(lexeme_string,"write2+"))
    return parse_write2plus_action();
  else if (!strcmp(lexeme_string,"tabstop"))
    return parse_tabstop_action();
  else if (!strcmp(lexeme_string,"halt"))
    return parse_halt_action();
  else if (!strcmp(lexeme_string,"bind"))
    return parse_bind_action();
  else if (!strcmp(lexeme_string,"call2"))
    return parse_call2_action();

  /* --- regular make actions --- */
  if (lexeme_type==SYM_CONSTANT_LEXEME) {
    output_previous_lexeme = FALSE;  /* eat the class name */
    get_lexeme();
  } else if (lexeme_type==VARIABLE_LEXEME) {
    error ("Unable to convert to Soar6, because there is a\n");
    error ("variable in the class field of a preference make\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  } else {
    error ("Expected a constant (symbol) for the class field of this preference make\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
    
  if (lexeme_type!=VARIABLE_LEXEME) {
    error ("There must be a variable in the id field of this preference make\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();

  while (lexeme_type!=R_PAREN_LEXEME) {
    new_actions = parse_rhs_attr_value_make ();
    if (! new_actions) return FALSE;
  }
  get_lexeme();  /* consume the right parenthesis */
  return TRUE;
}

bool parse_rhs (void) {
  bool new_actions;

  do {
    new_actions = parse_rhs_action ();
    if (! new_actions) return FALSE;
  } while (lexeme_type!=R_PAREN_LEXEME);
  return TRUE;
}





void init_parser (void) {

}



bool parse_production (void) {
  int parentheses_level;
  bool lhs;
  bool rhs;
  bool flag;
  bool declared_o_support, declared_no_o_support;
  
  parentheses_level = current_lexer_parentheses_level();
  
  if (lexeme_type!=SYM_CONSTANT_LEXEME) {
    error ("Expected symbol for production name\n");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return FALSE;
  }
  declared_o_support = FALSE;
  declared_no_o_support = FALSE;

  if (support_declaration_found) {
    support_declaration_found = FALSE;
    if (strcmp(supported_prod_name,lexeme_string)) {
      error ("Previous support declaration isn't for this production\n");
      print_location_of_most_recent_lexeme();
    } else {
      if (o_support) declared_o_support = TRUE;
      else declared_no_o_support = TRUE;
    }
  }

  get_lexeme();
  
  if (lexeme_type==QUOTED_STRING_LEXEME) {
    get_lexeme();
  }

  if (declared_o_support) printf ("  :o-support\n");
  if (declared_no_o_support) printf ("  :no-o-support\n");
  if (inside_default) printf ("  :default\n");
  lhs = parse_lhs();
  if (! lhs) {
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return FALSE;
  }

  if (lexeme_type!=RIGHT_ARROW_LEXEME) {
    error ("Expected --> in production\n");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return FALSE;
  }
  get_lexeme();
  
  rhs = parse_rhs();
  if (!rhs) {
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return FALSE;
  }
  
  if (lexeme_type!=R_PAREN_LEXEME) {
    error ("Expected ) to end production\n");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return FALSE;
  }

  /* --- everything parsed okay, so make the production structure --- */
  return TRUE;
}





/* ===================================================================
   
                            Exiting Soar

=================================================================== */

void abort_with_fatal_error (void) {
  fprintf (stderr, "Soar cannot recover from this error.  Aborting...\n");
  exit (1);
}


/* ===================================================================
   
                            Main Loop

=================================================================== */

void setf_interface_routine(int called_as_setf) {
  bool is_chunk_free = FALSE;
  bool inside_setf = FALSE;

  get_lexeme(); /* consume setf */
  output_previous_lexeme = FALSE;
  if(lexeme_type != SYM_CONSTANT_LEXEME) {
    error ("Illegal value to %s.\n", (called_as_setf) ? "setf" : "setq");
    printf ("(%s ", (called_as_setf) ? "setf" : "setq");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (0);
  }
  if (!strcmp(lexeme_string, "*chunk-free-problem-spaces*")){
    is_chunk_free = TRUE;
  } else if (strcmp(lexeme_string, "*watch-free-problem-spaces*")){
    printf ("(%s ", (called_as_setf) ? "setf" : "setq");
    error ("Illegal value to %s.\n", (called_as_setf) ? "setf" : "setq");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (0);
    return;
  }
  get_lexeme(); /* chew the name */
  output_previous_lexeme = FALSE;
  if (lexeme_type != APOSTROPHE_LEXEME) {
    printf("(%s *%s* ", (called_as_setf) ? "setf" : "setq",
                        (is_chunk_free ?
			 "chunk-free-problem-spaces" :
			 "watch-free-problem-spaces"));
    error("Bogus %s command, expecting apostrophe.\n",
          (called_as_setf) ? "setf" : "setq");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (0);
    return;
  }
  get_lexeme(); /* chew the ' */
  output_previous_lexeme = FALSE;
  if(lexeme_type != L_PAREN_LEXEME) {
    printf("(%s *%s* '", (called_as_setf) ? "setf" : "setq",
                         (is_chunk_free ?
			  "chunk-free-problem-spaces" :
			  "watch-free-problem-spaces"));
    error("Bogus %s command, expecting left paren.\n",
          (called_as_setf) ? "setf" : "setq");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (0);
    return;
  }
  get_lexeme();
  output_previous_lexeme = FALSE;
  while(lexeme_type != R_PAREN_LEXEME && lexeme_type != EOF_LEXEME) {
    if(inside_setf)
      printf(")\n");
    inside_setf = TRUE;
    printf("(%s :add %s", (is_chunk_free ?
                           "chunk-free-problem-spaces" :
			   "watch-free-problem-spaces"), lexeme_string);
    get_lexeme();
    output_previous_lexeme = FALSE;
  }
  get_lexeme(); /* eat up innermost right_paren */
}

void sp_interface_routine (void) {
  bool p;

  get_lexeme();  /* consume "sp", advance to production name */
  p = parse_production();
}

void repeatedly_read_and_dispatch_commands (void) {

  if (current_lexer_parentheses_level()!=0) {
    fprintf (stderr, "Internal error:  misbalanced paren's in main loop.\n");
    abort_with_fatal_error();
  }
    
  while (TRUE) {
    get_lexeme();  /* consume rparen from previous command */
    /* since we're at the top level, turn off all the commenting */
    currently_commenting = FALSE;
    if (lexeme_type==EOF_LEXEME) break;
    if (lexeme_type!=L_PAREN_LEXEME) {
      error ("Expected ( to begin command\n");
      print_location_of_most_recent_lexeme();
      continue;
    }
    output_previous_lexeme = FALSE;
    get_lexeme(); /* consume lparen */
    
    if (lexeme_type==SYM_CONSTANT_LEXEME) {
      if (!strcmp(lexeme_string,"sp")) {
        printf ("(");
        sp_interface_routine();
      } else if (!strcmp(lexeme_string,"op-apps")) {
        if (support_declaration_found) {
          error ("Support declarations must immediately precede the\n");
          error ("corresponding productions.  You have two in a row here.\n");
          print_location_of_most_recent_lexeme();
          skip_ahead_to_balanced_parentheses (0);
        }
        support_declaration_found = TRUE;
        output_previous_lexeme = FALSE;
        get_lexeme();
        strcpy (supported_prod_name,lexeme_string);
        o_support = TRUE;
        output_previous_lexeme = FALSE;        
        get_lexeme();
        if (lexeme_type!=R_PAREN_LEXEME) {
          error ("Expected ) to end op-apps command\n");
          print_location_of_most_recent_lexeme();
          skip_ahead_to_balanced_parentheses (0);
        }
        output_previous_lexeme = FALSE;
      } else if (!strcmp(lexeme_string,"op-no-apps")) {
        if (support_declaration_found) {
          error ("Support declarations must immediately precede the\n");
          error ("corresponding productions.  You have two in a row here.\n");
          print_location_of_most_recent_lexeme();
          skip_ahead_to_balanced_parentheses (0);
        }
        support_declaration_found = TRUE;
        output_previous_lexeme = FALSE;
        get_lexeme();
        strcpy (supported_prod_name,lexeme_string);
        o_support = FALSE;
        output_previous_lexeme = FALSE;
        get_lexeme();
        if (lexeme_type!=R_PAREN_LEXEME) {
          error ("Expected ) to end op-apps command\n");
          print_location_of_most_recent_lexeme();
          skip_ahead_to_balanced_parentheses (0);
        }
        output_previous_lexeme = FALSE;
      } else if (!strcmp(lexeme_string, "user-select")) {
	printf("(");
        skip_ahead_to_balanced_parentheses (0);
      } else if (!strcmp(lexeme_string, "init-soar")) {
	printf("(");
	skip_ahead_to_balanced_parentheses (0);
      } else if (!strcmp(lexeme_string, "excise")) {
	printf("(");
	skip_ahead_to_balanced_parentheses (0);
      } else if (!strcmp(lexeme_string, "multi-attributes")) {
        error("Unable to convert multi-attributes, commenting out.\n");
	printf(";; (");
	last_commented_line = current_file.current_line;
	comment_until_balanced_parentheses (0);
      } else if (!strcmp(lexeme_string, "trace-attributes")) {
        error("Unable to convert trace-attributes, commenting out.\n");
        printf(";; (");
        last_commented_line = current_file.current_line;
        comment_until_balanced_parentheses (0);
      } else if (!strcmp(lexeme_string, "setf")) {
	output_previous_lexeme = FALSE;
	setf_interface_routine(TRUE);
      } else if (!strcmp(lexeme_string, "setq")) {
	output_previous_lexeme = FALSE;
	setf_interface_routine(FALSE);
      } else if (!strcmp(lexeme_string, "defun")) {
        error("Unable to convert defun declarations, commenting out.\n");
        printf(";; (");
        last_commented_line = current_file.current_line;
        comment_until_balanced_parentheses (0);
      } else if (!strcmp(lexeme_string, "defvar")) {
        error("Unable to convert defvar declarations, commenting out.\n");
        printf(";; (");
        last_commented_line = current_file.current_line;
        comment_until_balanced_parentheses (0);
      } else if(!strcmp(lexeme_string, "start-default")) {
        inside_default = TRUE;
        output_previous_lexeme = FALSE;
	get_lexeme();
        output_previous_lexeme = FALSE;
      } else if(!strcmp(lexeme_string, "stop-default")) {
        inside_default = FALSE;
        output_previous_lexeme = FALSE;
	get_lexeme();
        output_previous_lexeme = FALSE;
      } else {
        printf ("(");
        error ("Unable to convert this command\n");
        print_location_of_most_recent_lexeme();
        skip_ahead_to_balanced_parentheses (0);
      }
      if (lexeme_type==EOF_LEXEME) break;
    } else {
      printf ("(");
      error ("Unable to convert this command\n");
      print_location_of_most_recent_lexeme();
      skip_ahead_to_balanced_parentheses (0);
    }
  } /* end of while TRUE */
}

int main (int argc, char *(argv[]))
{
  init_lexer ();
  init_parser ();
  support_declaration_found = FALSE;
  
  repeatedly_read_and_dispatch_commands ();
  exit(0);
}
