;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde.el,v $
;;;; $Id: sde.el,v 0.89 1994/03/21 07:48:55 hucka Exp $
;;;; 
;;;; Description       : Main load file and common functions for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs:      Copyright (C) 1985-1993 Free Software Foundation, Inc.
;;;; Soar-mode 5.0:  Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:     Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.46:      Copyright (C) 1991-1992 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25:  Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:    Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu
;;;; Calc 2.02b:     Copyright (C) 1990-1993 Free Software Foundation, Inc.
;;;; Edebug 3.2:     Copyright (C) 1988-1993 Free Software Foundation, Inc.
;;;; rp-describe-function:  Copyright (C) 1991 Robert D. Potter.

(defconst sde-el-version "$Revision: 0.89 $"
  "The revision number of sde.el.  The complete RCS id is:
      $Id: sde.el,v 0.89 1994/03/21 07:48:55 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Simple gensym.  Code based on cl.el of Emacs 18.58.
;;;; 5.  Often-used macros.
;;;; 6.  Hash table routines
;;;; 7.  Miscellaneous basic functions
;;;; 8.  Buffer and window handling
;;;; 9.  Search-related functions.
;;;; 10. Basic movement and production interpretation.
;;;; 11. Recordkeeping
;;;; 12. Error and diagnostic handling.
;;;; 13. Parenthesis handling.
;;;; 14. Indentation support.
;;;; 15. Sending, excising, etc., productions and regions of productions.
;;;; 16. Miscellaneous editing commands
;;;; 17. Comment support
;;;; 18. SDE mode
;;;; 19. Help support.
;;;; 20. Load commands for rest of SDE
;;;; 21. Functions for reporting bugs
;;;; 22. Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; ----------------
;;;; 0. Documentation
;;;; ----------------
;;;;
;;;; WARNING:  This file redefines the functions `describe-mode' and, in
;;;; Emacs 18, `kill-buffer'.  See the definitions of `sde-describe-mode'
;;;; and `sde-kill-buffer-emacs18' below for details.
;;;;
;;;; This is the main file of the Soar Development Environment (SDE), an
;;;; editing and debugging environment featuring special tools to help manage
;;;; the complexities of writing Soar code and interacting with Soar.  It
;;;; is currently compatible with the following versions of Emacs:  FSF GNU
;;;; Emacs 18, FSF GNU Emacs 19, and Lucid Emacs 19.  Epoch 4.2 is only
;;;; weakly supported and largely untested.
;;;;
;;;; This file contains the core of SDE, implemented in Emacs as a mode
;;;; called "sde-mode".  Also contained here are basic macros and functions
;;;; used in the other SDE modules, and load instructions for the rest of
;;;; SDE.
;;;;
;;;; Certain function and data definitions that depend on the version of
;;;; Emacs being used are located in the following files:
;;;;
;;;;   FSF GNU Emacs 18:  sde-emacs18.el
;;;;   FSF GNU Emacs 19:  sde-emacs19.el
;;;;   Lucid Emacs 19:    sde-lemacs.el
;;;;
;;;; The files listed above are loaded first by this file at both compile and
;;;; load time, hence they must not contain dependencies on the contents of
;;;; this file.


;;;----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;----------------------------------------------------------------------------

;; Fabulous hack to figure out where this file is located and add that
;; directory to the load-path.  This allows users simply to issue a
;;      (load "/foo/bar/soar")
;; and automatically get the /foo/bar directory added to their load-path if
;; it's not in load-path already.  Algorithm: look at where this file is
;; being loaded from, look at load-path, and if the current directory isn't
;; on load-path, add it.
;;
;; sde-directory is based on original code from Andy Norman:
;;
;;   From: Andy Norman <ange@hplb.hpl.hp.com>
;;   To: hucka@engin.umich.edu
;;   Subject: Re: How to get path info during load? 
;;   Date: Tue, 28 Jan 92 10:40:28 GMT

(defvar sde-directory nil
  "Directory contain SDE's files.")

(defun sde-directory ()
  "Guess the directory of the file currently being loaded, or return nil."
  (let* ((buf (get-buffer-create " *backtrace*"))
	 (standard-output buf)
	 file)
    (condition-case nil
	(save-excursion
	  (set-buffer buf)
	  (erase-buffer)
	  (backtrace)
	  (goto-char (point-min))
	  (if (re-search-forward "load(\"\\([^\"]+\\)\"" nil t)
	      (setq file (buffer-substring (match-beginning 1) (match-end 1))))
	  (kill-buffer buf))
      (error nil))
    (if (and file (file-name-directory file))
	(directory-file-name (file-name-directory file)))))
   

(let ((path (sde-directory))
      (lpath load-path))
  (if (null path)
      ;; Directory must be on load-path already.
      (progn
	(while (and lpath
		    (not (file-exists-p (expand-file-name "sde.el" (car lpath)))))
	  (setq lpath (cdr lpath)))
	(if lpath
	    (setq sde-directory (car lpath))))
      ;; Found a dir in the load stmt.  Add to load-path if not there already.
      (progn
	(setq sde-directory path)
	(while (and lpath (not (equal path (car lpath))))
	  (setq lpath (cdr lpath)))
	(if (null lpath)
	    (setq load-path (cons path load-path))))))


;; Requirements.  
;; Gross code to get conditional evaluation both at compile time and eval
;; time.  This is the only way to do it in unmodified Emacs 18.

(require 'sde-version)
(require (progn
	   (if window-system
	       (if (and (string-lessp emacs-version "19")
			(not (boundp 'epoch::version)))
		   (require 'x-mouse)
		   (require 'mouse)))
	   (cond ((string-match "Lucid" emacs-version)
		  (require 'sde-lemacs))
		 ((not (string-lessp emacs-version "19"))
		  (require 'sde-emacs19))
		 (t			; Emacs 18 or epoch.
		  (require 'gmhist)
		  (require 'sde-emacs18)))
	   ;; Dummy
	   'sde-version))

;; Provide

(provide 'sde)


;;;----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;----------------------------------------------------------------------------

;;; Configuration variables.

(defvar sde-soar-program "soar"
  "*String indicating the name of the Soar program to run.
This is used by the SDE command `soar' when starting up Soar.  The default
value is the string \"soar\".  It can be the name of a program, or an
absolute pathname (i.e., beginning with a '/' or '~').  If the value of this
variable is not an absolute pathname, the named program is found using normal
shell command searching conventions.")

(defvar sde-soar-starting-directory nil
  "*String indicating the default directory in which to start Soar.  
This is a useful variable to set because Soar looks in the starting directory
for init files.  Unless `\\[soar]' is invoked with a prefix argument, it will
cd to `sde-soar-default-directory' before starting Soar.")

(defvar sde-soar-switches nil
  "*List of strings passed as switches to Soar at start-up time.
As of version 6.1.1, standard Soar does *not* accept command line switches.
This variable is provided for use with modified Soar systems such as
ModSAF/SAFSoar.  Make sure that each individual element in this list
is a separate string.  Example of usage:

  \(setq sde-soar-switches '(\"-sde\" \"-nonet\" \"-terrain\" \"ocean-0001\"))

Also see the documentation for the variable `sde-prompt-for-soar-switches'.")

(defvar sde-prompt-for-soar-switches nil
  "*Whether to prompt for switches to Soar if `sde-soar-switches' is nil.
Normally, the `M-x soar' command will not prompt for switches unless it
is given a prefix argument, because standard Soar does not accept command
line switches.  However, if you wish `M-x soar' to prompt for switches
when `sde-soar-switches' is nil (just as it does for the other variables
such as `sde-soar-starting-directory'), set this variable to t.  This
may be useful if you are running a modified Soar, such as ModSAF/SAFSoar,
and you want to run it repeatedly in the same directory but with different
command line switches each time.")

(defvar sde-soar-use-ptys nil
  "Non-nil if SDE should make Emacs use a pty for communicating with Soar.
Default: nil.  A pseudo-tty is usually not necessary for SDE to communicate
with Soar, and it is more efficient to use pipes, hence you will normally want
this variable to be nil.  However, if you are running a Soar-derived system
such as ModSAF/SAFSoar, you may need to set this to t.")

(defvar sde-soar-defaults-file nil
  "*Pathname to the Soar default productions file (\"default.soar6\").
This is set in the SDE site initialization file, \"sde-site.el\".")

(defvar sde-file-types
  '("\\.soar$"  "\\.soar5$" "\\.soar6$" "\\.soa$"
    "\\.init\\.soar" "\\.reset.soar")
  "*List of file extensions that should use SDE.")

(defvar sde-production-indent-offset 2
  "*Integer value indicating how far to indent production clauses.")

(defvar sde-arrow-indent-offset 0
  "*Integer value indicating how far to indent the arrow in a Soar production.
This value is relative to the indentation of the production as a whole.
Default: 0.  A negative value moves the arrow leftward.  A positive value
indents the arrow further to the right relative to the surrounding clauses.")

(defvar sde-soar-beep-after-setup nil
  "*Non-nil means beep after Soar has started up and is ready.  Default: nil.")

(defvar sde-soar-use-output-buffer t
  "*Whether output from most Soar commands should be put in a separate buffer.
If this variable is non-nil (the default), output from commands such as
querying commands will be placed in a buffer named *output* instead of being
dumped into the Soar agent buffer.  \(Exception: output from the commands go
and run is always placed in the Soar process buffer.)  If nil, all output
goes to the Soar process buffer.")

(defvar sde-soar-erase-output-buffer nil
  "*Whether to erase the Soar output buffer before each new output appears.
Default is nil.")

(defvar sde-soar-move-point-on-output nil
  "*Whether to move point to the bottom of new output upon each command.
Default is nil, which means that if your cursor is in a Soar agent or output
buffer, but not at the bottom, at the time you issue a command, point will
not be moved.  If this variable is t, then when output appears, the cursor
will be moved to the bottom to follow the output.  The default setting
results in less jerking around of the screen.")

(defvar sde-use-multiple-frames
  (and window-system
       (or (and (boundp 'epoch::version) (stringp (symbol-value 'epoch::version)))
	   (string-match "19" emacs-version))
       t)
  "*If t (default in Epoch & Emacs 19), use multiple frames if possible.
This is only possible in Epoch, GNU Emacs 19 and Lucid Emacs 19, which are
versions of Emacs that support multiple screens (called \"frames\" in GNU
Emacs 19) under X windows.  A value of t for this variable means SDE will
create a separate frame/screen for the *output* buffer and for each Soar 
agent process.  A value of 'output means SDE will only create a separate
frame for the *output* buffer, not for agent processes, while a value of
nil means SDE will never deliberately create a new frame for *output*
or for agents.  Also see the variable `sde-use-output-buffer'.")

(defvar sde-soar-agent-buffer-defaults
  '((height . 24)
    (width . 80))
  "Association list of default values for Soar agent buffer screens/frames.
This determines the attributes of new frames/screens created for Soar agents.
This is only effective if `sde-use-multiple-frames' is non-nil and you are
running in Emacs 19.22 or later or Lucid Emacs 19.  The value may be set in
your ~/.emacs init file, like this:
     
  (setq sde-soar-agent-buffer-defaults '((width . 80) (height . 55)))
     
In Emacs 19.22, the possible parameters are: 
  height                  -- Height of the window in characters.
  width                   -- Width of the window in characters.
  left                    -- Screen position of the left edge, in pixels.
  top                     -- Screen position of the top edge, in pixels.
  name                    -- Name of the frame.
  auto-raise              -- Whether selecting the frame raises it.
  auto-lower              -- Whether deselecting the frame raises it.
  vertical-scroll-bars    -- Whether frame has a vertical scroll bar.
  horizontal-scroll-bars  -- Whether frame has a horizontal scroll bar.
  border-width            -- The width in pixels of the window border.
  internal-border-width   -- The distance in pixels between text and border.
  menu-bar-lines          -- Number of lines to allocate for a menu bar.
  minibuffer              -- Whether to have a minibuffer in the frame
  cursor-color            -- The color of the text cursor that shows point.
  mouse-color             -- The color of the mouse cursor.
  foreground-color        -- The color of the inside of characters.
  background-color        -- The color of background of text.
  border-color            -- The color for the border of the frame.
  cursor-type             -- The way to display the cursor.
     There are two legitimate values: `bar' and `box'.  
     The value `bar' specifies a vertical bar between characters as the
     cursor.  The value `box' specifies an ordinary black box overlaying
     the character after point; that is the default.
  icon-type               -- Type of icon to use when frame it is iconified.
                             Non-`nil' = bitmap icon, `nil' = text icon.")

(defvar sde-soar-output-buffer-defaults 
  '((height . 20)
    (width . 80)
    (top . 20)
    (left . 20))
  "Association list of default values for Soar output buffer screens/frames.
This determines the attributes of the Soar output buffer screen/frame when
`sde-use-multiple-frames' is non-nil.  This is only effective if
`sde-use-multiple-frames' is non-nil and you are running in Emacs 19.22 or
later or Lucid Emacs 19.  The value may be set in your init file, like this:
     
  (setq sde-soar-output-buffer-defaults '((width . 80) (height . 55)))
     
In Emacs 19.22, the possible parameters are: 
  height                  -- Height of the window in characters.
  width                   -- Width of the window in characters.
  left                    -- Screen position of the left edge, in pixels.
  top                     -- Screen position of the top edge, in pixels.
  name                    -- Name of the frame.
  auto-raise              -- Whether selecting frame raises it.
  auto-lower              -- Whether deselecting frame raises it.
  vertical-scroll-bars    -- Whether frame has a vertical scroll bar.
  horizontal-scroll-bars  -- Whether frame has a horizontal scroll bar.
  border-width            -- The width in pixels of the window border.
  internal-border-width   -- The distance in pixels between text and border.
  menu-bar-lines          -- Number of lines to allocate for a menu bar.
  minibuffer              -- Whether to have a minibuffer in the frame
  cursor-color            -- The color of the text cursor that shows point.
  mouse-color             -- The color of the mouse cursor.
  foreground-color        -- The color of the inside of characters.
  background-color        -- The color of background of text.
  border-color            -- The color for the border of the frame.
  cursor-type             -- The way to display the cursor.
     There are two legitimate values: `bar' and `box'.  
     The value `bar' specifies a vertical bar between characters as the
     cursor.  The value `box' specifies an ordinary black box overlaying
     the character after point; that is the default.
  icon-type               -- Type of icon to use when frame it is iconified.
                             Non-`nil' = bitmap icon, `nil' = text icon.")

(defvar sde-soar-track-cd t
  "*Non-nil (the default) means track cd & pwd commands in the Soar process.")

(defvar sde-soar-input-ring-size 30
  "*Size of Soar process input history ring.")

(defvar sde-soar-input-ring-filter
  '(lambda (str) (not (string-match "\\`\\s *\\'" str)))
  "*Predicate for filtering additions to Soar process input history.
Only inputs answering true to this function are saved on the input
history list.  Default is to save anything that isn't all whitespace.")

(defvar sde-production-name-test-regexp
  "\*\\|chunk-[0-9]+\\|justification-[0-9]+"
  "*Regexp for determining whether a symbol likely is a production name.
It is used by the function that extracts production names near the cursor in
a buffer.  By default this regular expression tests that an extracted name
contains either the character '*' or a string of the form \"chunk-n\" or
\"justification-n\", where \"n\" is an integer.") 

;; Keymaps

(defvar sde-mode-map nil
  "*Keymap for SDE Mode.")

(defvar sde-view-cmds-map nil
  "*Keymap for view commands in SDE.")

(defvar sde-agent-cmds-map nil
  "*Keymap for agent commands in SDE.")

(defvar sde-region-cmds-map nil
  "*Keymap for region commands in SDE.")

(defvar sde-help-cmds-map nil
  "*Keymap for help commands in SDE.")  

(defvar sde-mouse-map nil
  "*Mouse map used when running under Epoch 4.2.")

(defvar sde-soar-mode-map nil
  "Keymap for Soar Mode.")

(defvar sde-soar-output-mode-map nil
  "Keymap for Soar output buffer.")

(defvar sde-mode-syntax-table nil
  "Syntax table used while in Soar Mode.")

(defvar sde-mode-abbrev-table nil
  "Table of abbreviations for Soar Mode.")

;;; Hook variables

(defvar sde-site-hook nil
  "*Hook run after loading the Soar Development Environment.
This is the place to put site-specific initializations, such as local
menus.")

(defvar sde-load-hook nil
  "*Hook run after loading the Soar Development Environment.
This is a good place to put customizations and key bindings.")

(defvar sde-mode-hook nil
  "*Hook run after starting sde-mode.  
This is a good place to put customizations and key bindings.")

(defvar sde-soar-mode-hook nil
  "*Hook for functions to be run after starting up a Soar process.
Good for customizing the interaction environment.")

(defvar sde-soar-output-mode-hook nil
  "*Hook of functions to be run after the Soar output buffer is created.")

(defvar sde-soar-hook nil
  "*Hook of functions to run after first prompt appears in Soar process buffer.")

(defvar sde-soar-error-hook nil
  "*Hook of functions to be run after an error is shown the Soar error buffer.")

;; Declared here for the compiler but really defined in sde-headers.el.
(defvar sde-header-hooks)
(put   'sde-header-hooks 'variable-documentation
 "*List of functions that insert each line of a file header in Soar files.")

;;; Defaults for optional arguments to commands

(defvar sde-go-args nil
  "*String containing default optional args for \"go\".")

(defvar sde-run-args nil
  "*String containing default optional args for \"run\".")

(defvar sde-matches-args nil
  "*String containing default optional args for \"matches\".")

(defvar sde-ms-args nil
  "*String containing default optional args for \"ms\".")

(defvar sde-firing-counts-args nil
  "*String containing default optional args for \"firing-counts\".")

(defvar sde-print-args nil
  "*String containing default optional args for \"soar-print\".")

(defvar sde-preferences-args nil
  "*String containing default optional args for \"preferences\".")

(defvar sde-list-productions-args nil
  "*String containing default optional args for \"list-productions\".")

(defvar sde-list-chunks-args nil
  "*String containing default optional args for \"list-chunks\".")

(defvar sde-list-justifications-args nil
  "*String containing default optional args for \"list-justifications\".")

(defvar sde-agent-go-args nil
  "*String containing default optional args for \"agent-go\".")

(defvar sde-schedule-args nil
  "*String containing default optional args for \"schedule\".")

(defvar sde-reset-args nil
  "*String containing default optional args for \"reset\".")

(defvar sde-explain-args nil
  "*String containing default optional args for \"explain\".")


;;;----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;----------------------------------------------------------------------------


(defconst sde-running-epoch
    (and (boundp 'epoch::version) (stringp (symbol-value 'epoch::version)))
  "Non-nil if running Epoch.")

(defconst sde-running-lemacs (and (string-match "Lucid" emacs-version) t)
  "Non-nil if running Lucid Emacs.")

(defconst sde-running-emacs19
    (and (string-match "\\`19" emacs-version) (not sde-running-lemacs))
  "Non-nil if running FSF GNU Emacs 19.")

(defconst sde-running-emacs18
    (and (string-lessp emacs-version "19") (not sde-running-epoch))
  "Non-nil if running GNU Emacs 18.")

(defconst sde-source-modes '(sde-mode)
  "List of mode names in SDE that deal with source files.")

(defconst sde-soar-modes '(sde-soar-mode sde-soar-output-mode)
  "List of mode names in SDE related to interacting with Soar.")

(defconst sde-modes (append sde-source-modes sde-soar-modes)
  "List of all modes that are related to SDE.")

(defvar sde-kill-buffer-hook nil
  "Hook to be run (by `run-hooks', which see) when a buffer is killed.
The buffer being killed will be current while the hook is running.
This is unused in Emacs 19, but is implemented by SDE for Emacs 18.")

(defvar sde-soar-agents nil
  "Alist of names of Soar agents and their associated buffers.
When running in single-agent mode, this variable is nil.  When running in
multi-agent mode, there are multiple agents and a separate buffer for each
agent, and the first name on the alist is the first agent created.  Format of
elements on the list is \( name .  buffer-name ).")

(defconst sde-soar-prompt-regexp "^Soar\\( agent \\([^>]+\\)\\)?> "
  "Regular expression to recognize prompts in the Soar process buffer.
If Soar is running in multi-agent mode, match-beginning 2 returns the start
of the current agent's name in the prompt string.")

;; In Emacs 18, such regexps wouldn't need the extra \( ... \) delimiters,
;; but they are needed in 19.

(defvar sde-sp-name-regexp "^[ \t]*(sp\\s +\\(\\(\\s_\\|\\sw\\)*\\)"
  "*Regexp to match the name part of a production.  
This must be a regular expression containing a grouping construct such that
SDE code can use \(match-beginning 1) and \(match-end 1) to extract the name
from within a matched string.")

(defvar sde-info-file "sde.info"
  "Name of the SDE info file.")

(defvar sde-soar-info-file "soar6-manual.info"
  "Name of the Soar Manual info file.")

(defvar sde-soar-release-notes-file nil
  "Filename of the release notes for Soar.")

(defvar sde-soar-user-notes-file nil
  "Filename of the user notes for Soar.")

(defvar sde-known-files nil
  "Hash table of files known to have been seen during a session with SDE.  
File names are hashed directly and an entry's value is just its string name.
Used in facilities like `sde-find-production'.")

(defvar sde-soar-buffer-agent nil
  "Buffer-local var indicating agent responsible for I/O in current buffer.")

(defvar sde-soar-state-string "no process"
  "String indicating the current status of Soar.  Used in mode lines.")

(defvar sde-commands-obarray (make-vector 103 0)
  "Obarray used for completion support for help involving Soar.")

(defvar sde-variables-obarray (make-vector 103 0)
  "Obarray used for completion support for help involving Soar.")

(defvar sde-topics-obarray (make-vector 211 0)
  "Obarray used for completion support involving SDE and Soar topics.")

;; Things defined in sde-soar-mode.el or elsewhere, but declared here to
;; quiet the optimizing byte compiler.

(defvar sde-soar-buffer-mark)
(defvar make-header-hooks)
(defvar lisp-indent-hook)


;;;----------------------------------------------------------------------------
;;; 4.  Simple gensym.  Code based on cl.el of Emacs 18.58.
;;;----------------------------------------------------------------------------

;; sde-gensym is used later in a macro, so we have to resort to this evil
;; hack using require and provide.  sde-gensym will end up uncompiled, which
;; is okay given how it's used currently.

(require				; Evaluated at load and compile time.
 (progn
   (provide 'sde-gensym-require-hack)

   (defvar sde-gensym-index 0
     "Integer used by gensym to produce new names.")

   (defvar sde-gensym-prefix "sde$$"
     "Names generated by gensym begin with this string by default.")

   (defun sde-gensym (&optional prefix)
     "Generate a fresh uninterned symbol.  
Optional argument PREFIX becomes the string that begins the new name.
Warning: this could conflict with other gensyms.  Care should be taken to
insure that sde-gensym-prefix is unique to a given set of files."
     (setq prefix (concat (or prefix sde-gensym-prefix) "_"))
     (let ((newsymbol nil)
	   (newname   ""))
       (while (not newsymbol)
	 (setq newname          (concat prefix sde-gensym-index)
	       sde-gensym-index (+ sde-gensym-index 1))
	 (if (not (intern-soft newname))
	     (setq newsymbol (make-symbol newname))))
       newsymbol))

   'sde-gensym-require-hack))


;;;----------------------------------------------------------------------------
;;; 5.  General purpose macros.
;;;----------------------------------------------------------------------------


(defmacro sde-member (elt lst)
  "Look for ELT in LST; return cdr whose car is `equal' to ITEM."
  (let ((e (sde-gensym "SDE"))
	(l (sde-gensym "SDE")))
    (` (let* (((, e) (, elt))		; Avoid evaluating elt & lst more than once.
	      ((, l) (, lst)))
	 (while (and (, l) (not (equal (, e) (car (, l)))))
	   (setq (, l) (cdr (, l))))
	 (, l)))))


(defmacro sde-rassoc (elt lst)
  "Find the first item whose cdr is `equal' to item ELT in list LST.
List LST must be a list of dotted pairs, not of regular conses."
  (let ((e (sde-gensym "SDE"))
	(l (sde-gensym "SDE")))
    (` (let* (((, e) (, elt))		; Avoid evaluating elt & lst more than once.
	      ((, l) (, lst)))
	 (while (and (, l) (not (equal (, e) (cdr (car (, l))))))
	   (setq (, l) (cdr (, l))))
	 (car (, l))))))


(defmacro sde-adelete (key alist)
  "Delete pairs that have car `equal' to KEY in ALIST and return a new alist.
This is a non-destructive operation.  ALIST elements must be dotted pairs."
  (let ((k (sde-gensym "SDE"))
	(lst (sde-gensym "SDE"))
	(tmp (sde-gensym "SDE")))
    (` (let* (((, k) (, key))
	      ((, lst) (, alist))
	      (, tmp))
	 (while (, lst)
	   (if (equal (, k) (car (car (, lst))))
	       (setq (, lst) (cdr (, lst)))
	       (setq (, tmp) (cons (car (, lst)) (, tmp))
		     (, lst) (cdr (, lst)))))
	 (and (, tmp) (nreverse (, tmp)))))))


(defmacro sde-radelete (key alist)
  "Delete pairs that have cdr `equal' to KEY in ALIST and return a new alist.
This is a non-destructive operation.  ALIST elements must be dotted pairs."
  (let ((k (sde-gensym "SDE"))
	(lst (sde-gensym "SDE"))
	(tmp (sde-gensym "SDE")))
    (` (let* (((, k) (, key))
	      ((, lst) (, alist))
	      (, tmp))
	 (while (, lst)
	   (if (equal (, k) (cdr (car (, lst))))
	       (setq (, lst) (cdr (, lst)))
	       (setq (, tmp) (cons (car (, lst)) (, tmp))
		     (, lst) (cdr (, lst)))))
	 (and (, tmp) (nreverse (, tmp)))))))


(defmacro sde-push (elt lst)
  "Do like \(setq ELT \(cons ELT LST))."
  (if (symbolp lst)
      (` (setq (, lst) (cons (, elt) (, lst))))
      (error "Second argument of push must be a symbol:  %s"
	     lst)))
  

(defmacro sde-pushnew (elt lst)
  "If ELT is not already a member of LST, do \(setq ELT \(cons ELT LST))."
  (if (symbolp lst)
      (let ((e (sde-gensym "SDE")))
	(` (let (((, e) (, elt)))	; Avoid evaluating elt more than once.
	     (if (not (sde-member (, e) (, lst)))
		 (setq (, lst) (cons (, e) (, lst)))
		 (, lst)))))
      (error "Second argument of pushnew must be a symbol:  %s"
	     lst)))

;; Based on bbdb-string-trim in bbdb.el.

(defmacro sde-string-trim (string)
  "Remove leading and trailing whitespace from STRING."
  (let ((str (sde-gensym "SDE")))
    (` (let* (((, str) (, string)))
	 (if (string-match "\\`[ \t\n]+" (, str))
	     (setq (, str) (substring (, str) (match-end 0))))
	 (if (string-match "[ \t\n]+\\'" (, str))
	     (substring (, str) 0 (match-beginning 0))
	     (, str))))))


(defmacro sde-substring (str match-num &optional match-end)
  "Do `substring' on string STR.  MATCH-NUM specifies the subexpression.
If MATCH-NUM is not followed with optional argument MATCH-END, this macro
expands into
    \(substring STR \(match-beginning MATCH-NUM) \(match-end MATCH-NUM))
Otherwise, if optional MATCH-END is supplied, this macro becomes
    \(substring STR \(match-beginning MATCH-NUM) \(match-end MATCH-END))"
  (if match-end
      (` (substring (, str) (match-beginning (, match-num)) (match-end (, match-end))))
      (` (substring (, str) (match-beginning (, match-num)) (match-end (, match-num))))))


(defmacro sde-buffer-substring (num)
  "Expands into \(buffer-substring \(match-beginning NUM) \(match-end NUM))."
  (` (buffer-substring (match-beginning (, num)) (match-end (, num)))))


(defmacro sde-unconcat (string)
  "Break up string at whitespaces into a list of strings and return the list."
  (` (let ((str (sde-string-trim (, string)))
	   (start 0)
	   (new))
       (while (string-match "[ \t]+" str start)
	 (setq new   (cons (substring str start (match-beginning 0)) new)
	       start (match-end 0)))
       (setq new (cons (substring str start) new))
       (nreverse new))))


;; This craziness is necessary when calling `message' with production names.
;; A production name can legally contain `%' characters, leading to
;; errors when such a string is passed to the Elisp `message' function.

(defmacro sde-quote-string (str)
  "Quote all instances of `%' characters in string STR and return the result."
  (` (let ((start 0) 
	   (new ""))
       (while (string-match "%" (, str) start)
	 (setq new (concat new (substring (, str) start (match-beginning 0)) "%%")
	       start (match-end 0)))
       (concat new (substring (, str) start)))))


;; Modified from lisp-in-comment from ilisp-ext.el

(defmacro sde-point-in-comment (test-regexp)
  "Return t if in a comment determined by TEST-REGEXP."
  (` (progn
       (beginning-of-line)
       (and (looking-at (, test-regexp))
	    (/= (match-end 0) (progn (end-of-line) (point)))))))


(defmacro sde-key-for-command (cmd)
  "Return a string description of the binding for function CMD.
If there is no key binding for the command, returns the string \"M-x cmd\"."
  (` (let ((key (key-description
		 (cond (sde-running-emacs19
			(where-is-internal (, cmd) (current-local-map) nil t))
		       (sde-running-emacs18
			(where-is-internal (, cmd) (current-local-map) t))
		       (sde-running-lemacs
			(where-is-internal (, cmd) (current-local-map)
					   (current-global-map) t))))))
       (if (and key (not (string= key "")))
	   key
	   (concat "M-x " (symbol-name (, cmd)))))))


(defmacro sde-make-name-regexp (name)
  "Take a production name and make a suitable regexp out of it."
  (` (format "^[ \t]*(sp\\s +\\(\\(\\s_\\|\\sw\\)*\\)%s" (regexp-quote (, name)))))


(defmacro sde-make-file-extensions-regexp ()
  "Return a regexp for matching files with extensions in `sde-file-types'."
  (` (mapconcat 'identity sde-file-types "\\|")))


(defmacro sde-get-sde-buffers ()
  "Return a list of buffers that are SDE source mode buffers."
  (` (let ((buffers (buffer-list))
	   result)
       (while buffers
	 (if (memq (sde-buffer-mode (car buffers)) sde-source-modes)
	     (sde-push (car buffers) result))
	 (setq buffers (cdr buffers)))
       (nreverse result))))


;;;----------------------------------------------------------------------------
;;; 6.  Input history handling, for different versions of Emacs
;;;----------------------------------------------------------------------------
;;;
;;; These macros help hide the differences in user prompting between Emacs 18
;;; with gmhist and Emacs 19.


(defmacro sde-read-string (prompt &optional hist initial)
  (if hist
      (` (if (featurep 'gmhist)
	     (read-with-history-in (, hist) (, prompt) (, initial))
	     (read-from-minibuffer (, prompt) (, initial) nil nil (, hist))))
      (` (read-from-minibuffer (, prompt) (, initial)))))


(defmacro sde-read-file-name (prompt &optional hist dir default mustmatch initial)
  (if hist
      (` (if (featurep 'gmhist)
	     (read-file-name-with-history-in
	      (, hist) (, prompt) (, dir) (, default) (, mustmatch) (, initial))
	     (read-file-name
	      (, prompt) (, dir) (, default) (, mustmatch) (, initial))))
      (` (read-file-name
	  (, prompt) (, dir) (, default) (, mustmatch) (, initial)))))


(defmacro sde-completing-read (prompt &optional hist table pred req-match init)
  (if hist
      (` (if (featurep 'gmhist)
	     (completing-read-with-history-in
	      (, hist) (, prompt) (, table) (, pred) (, req-match) (, init))
	     (completing-read
	      (, prompt) (, table) (, pred) (, req-match) (, init) (, hist))))
      (` (completing-read
	  (, prompt) (, table) (, pred) (, req-match) (, init)))))


(defmacro sde-prompt (prompt &optional hist initial)
  "If `current-prefix-arg' is non-nil, prompt user with PROMPT and history HIST.  
Otherwise, if HIST is provided, return the first item on the list, or if 
HIST is not provided, return INITIAL."
  (if hist
      (` (if current-prefix-arg
	     (if (featurep 'gmhist)
		 (read-with-history-in (, hist) (, prompt) (, initial))
		 (read-from-minibuffer (, prompt) (, initial) nil nil (, hist)))
	     (car (eval (, hist)))))
      (` (if current-prefix-arg
	     (read-from-minibuffer (, prompt) (, initial))
	     (, initial)))))


;;;----------------------------------------------------------------------------
;;; 6.  Hash table routines
;;;----------------------------------------------------------------------------
;;;
;;; Hash tables are implemented simply using the Emacs symbol table facility.
;;; They are equivalent to Emacs "obarrays", which are basically vectors.  I
;;; call them hash tables instead of obarrays because that's how they're being
;;; used here.


(defvar sde-name-table-size 2027
  "Size of the sde-name-table hash table.  Should be a prime number.")

(defvar sde-name-table nil
  "Hash table mapping production names to data about the productions.")


(defun sde-make-hashtable (&optional size)
  "Make an obarray suitable for use as a hashtable.
SIZE, if supplied, should be a prime number."
  (make-vector (or size sde-name-table-size) 0))


(defmacro sde-make-hash-key (key)
  "Return a string hash key made from KEY."
  (` (if (stringp (, key))
	 (, key)
	 (prin1-to-string (, key)))))


(defmacro sde-puthash (name data &optional ht)
  "Hash the string NAME and sets its data value to DATA.
Optional argument HT specifies the hash table to use.  The default hash table
is `sde-name-table'."
  (` (set
      (intern (sde-make-hash-key (, name)) (or (, ht) sde-name-table))
      (, data))))


(defmacro sde-gethash (name &optional ht)
  "Return the value associated with the hashed string NAME.
Optional argument HT specifies the hash table to use.  The default hash table
is `sde-name-table'.  Returns nil if NAME is not in the hash table."
  (let ((sym (sde-gensym "SDE")))
    (` (let (((, sym) (intern-soft (sde-make-hash-key (, name)) (or (, ht) sde-name-table))))
	 (and (, sym) (symbol-value (, sym)))))))


(defun sde-map-hashtable (fun tbl)
  "Call function FUN on each key and value in hash table TBL."
  (mapatoms
   (function 
    (lambda (sym)
     (funcall fun sym (symbol-value sym))))
   tbl))


;;;----------------------------------------------------------------------------
;;; .  Modifications to Emacs kill-buffer.
;;;----------------------------------------------------------------------------
;;;
;;; In order to be able to catch killing of agent buffers, and properly remove
;;; the associated agent info from SDE, we have to hook into kill-buffer.  In
;;; Emacs 19 this is easily done using the kill-buffer-hook, but in 18 it's
;;; necessary to redefine kill-buffer.  


(defvar sde-true-kill-buffer (symbol-function 'kill-buffer)
  "The original function definition of `kill-buffer'.")


(defun sde-kill-buffer-emacs18 (bufname)
  "One arg, a string or a buffer.  Get rid of the specified buffer.
Any processes that have this buffer as the `process-buffer' are killed
with `delete-process'.

This functions has been modified for the Soar Development Environment (SDE).
When called, if a multiple-agent version of Soar is running, this function
first runs the hooks on `sde-kill-buffer-hook' before deleting the buffer.
The buffer to be deleted is current when the hook is called."
  (interactive "bKill buffer: ")
  (if sde-soar-agents                   ; This condition means this won't get
      (if (sde-buffer-exists-p bufname) ; executed unless a Soar is running.
	  (save-excursion
	    (set-buffer (get-buffer bufname))
	    (run-hooks 'sde-kill-buffer-hook))))
  (funcall sde-true-kill-buffer (get-buffer bufname)))


;; For Emacs 19, `sde-soar-mode' sets up a buffer-local `kill-buffer-hook'.
;; For Emacs 18, have to do this:

(if sde-running-emacs18
    (fset 'kill-buffer 'sde-kill-buffer-emacs18))


;;;----------------------------------------------------------------------------
;;; 10. Basic movement and production interpretation.
;;;----------------------------------------------------------------------------

;; The parser in Emacs 18 is broken and cannot properly ignore code inside of
;; comments in languages (such as Lisp) that use newlines as comment enders.
;; In Emacs 18 you're supposed to set `parse-sexp-ignore-comments' to nil for
;; such languages.  However, doing so leads to the annoying bug that
;; forward-sexp/forward-list operations fail to work properly if a comment
;; contains an unbalanced delimiter.  If you set `parse-sexp-ignore-comments'
;; to t, then forward-sexp/forward-list *do* seem to work properly but
;; backward-moving operations don't.  (Exasperation.)
;;
;; The following set of hacks work around the problem to some degree.
;; Function sde-mode sets `parse-sexp-ignore-comments' to t, and redefines
;; backward-moving functions to bind `parse-sexp-ignore-comments' temporarily
;; to nil.  This allows forward moving operations to work, and is more
;; efficient than leaving `parse-sexp-ignore-comments' nil and having
;; forward-moving operations bind it to t temporarily, because forward-moving
;; functions are used more frequently in this SDE code.
;;
;; The corresponding functions are in lisp.el.  These are really only used
;; in interactive commands, so efficiency is not a critical issue.

(defun sde-backward-sexp (&optional arg)
  "Move forward across one balanced expression (sexp).
With argument, do it that many times.  Negative arg -N means
move backward across N balanced expressions."
  (interactive "p")
  (let* ((parse-sexp-ignore-comments sde-running-emacs19))
    (forward-sexp (- arg))))


(defun sde-backward-list (&optional arg)
  "Move forward across one balanced group of parentheses.
With argument, do it that many times.
Negative arg -N means move backward across N groups of parentheses."
  (interactive "p")
  (let* ((parse-sexp-ignore-comments sde-running-emacs19))
    (forward-list (- arg))))


(defun sde-backward-up-list (arg)
  "Move forward out of one level of parentheses.
With argument, do this that many times.
A negative argument means move backward but still to a less deep spot.
In Lisp programs, an argument is required."
  (interactive "p")
  (let* ((parse-sexp-ignore-comments sde-running-emacs19))
    (up-list (- arg))))


(defun sde-backward-kill-sexp (arg)
  "Kill the sexp (balanced expression) preceding the cursor.
With argument, kill that many sexps before the cursor.
Negative arg -N means kill N sexps after the cursor."
  (interactive "p")
  (let* ((parse-sexp-ignore-comments sde-running-emacs19))
    (kill-sexp (- arg))))

;; These equivalences are only included so that when users list key bindings,
;; they don't wonder why the forward-moving functions are "missing"...

(fset 'sde-forward-sexp 'forward-sexp)
(fset 'sde-forward-list 'forward-list)
(fset 'sde-kill-sexp    'kill-sexp)

;; Modified from Emacs 19's beginning-of-defun from lisp.el.  

(defun sde-beginning-of-sp (&optional count stay)
  "Move backward to next beginning of an sp form and also return point.
Optional first argument COUNT means do it that many times.
Optional second argument STAY means don't move if already at the beginning
of an sp form.  Returns point unless search stops due to end of buffer."
  (interactive "p")
  (and count (< count 0) (forward-char 1))
  (or (and stay (looking-at "^\\s("))
      (and (re-search-backward (if (memq major-mode sde-soar-modes)
				   (concat "^\\s(\\|"
					   "\\(" sde-soar-prompt-regexp "\\)\\s *\\s(")
				   "^\\s(")
			       nil 'move (or count 1))
	   (progn (goto-char (- (match-end 0) 1)) t)))
  (point))


;; Modified from Emacs 19's end-of-defun.

(defun sde-end-of-sp (&optional arg)
  "Move forward to next end of an sp form.  With arg, do it that many times.
Negative argument -N means move back to Nth preceding end of sp.

An end of an sp occurs right after the close-parenthesis that matches
the open-parenthesis that starts an sp; see `sde-beginning-of-sp'."
  (interactive "p")
  (if (or (null arg) (= arg 0)) (setq arg 1))
  (let ((first t))
    (while (and (> arg 0) (< (point) (point-max)))
      (let ((pos (point)))
	(while (progn
		 (if (and first
			  (progn
			    (forward-char 1)
			    (/= (sde-beginning-of-sp 1) pos)))
		     nil
		     (or (bobp) (forward-char -1))
		     (sde-beginning-of-sp -1))
		 (setq first nil)
		 (forward-list 1)
		 (skip-chars-forward " \t")
		 (if (looking-at "\\s<\\|\n")
		     (forward-line 1))
		 (<= (point) pos))))
      (setq arg (1- arg)))
    (while (< arg 0)
      (let ((pos (point)))
	(sde-beginning-of-sp 1)
	(forward-sexp 1)
	(forward-line 1)
	(if (>= (point) pos)
	    (if (/= (sde-beginning-of-sp 2) pos)
		(progn
		  (forward-list 1)
		  (skip-chars-forward " \t")
		  (if (looking-at "\\s<\\|\n")
		      (forward-line 1)))
		(goto-char (point-min)))))
      (setq arg (1+ arg)))))


(defun sde-sp-body ()
  "Return as a string the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil."
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-sp 1 t))
	  (sp-ending (progn (forward-sexp 1) (point))))
      (if (and (>= pt sp-beginning) (<= pt sp-ending))
	  (buffer-substring sp-beginning sp-ending)))))


(defun sde-sp-name (&optional body)
  "Extract the name of the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil.  

Algorithm: store current point, scan backward for the beginning of an sp
form, then from there, scan forward for the end of that form.  If point is
within these bounds, extract the name of the production, else return nil.  "
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-sp 1 t))
	  (sp-ending (progn (forward-sexp 1) (point))))
      (if (and (>= pt sp-beginning) (<= pt sp-ending))
	  (progn
	    (goto-char sp-beginning)
	    (forward-char 4)
	    (buffer-substring (point) (progn (forward-sexp 1) (point))))))))


(defun sde-region-name (start end)
  "Return a descriptive name for a region of productions or other sexps.
Format of the name is \"From START-SP to END-SP\", where START-SP and END-SP
are the names of the productions at the beginning and end." 
  (let (from to)
    (save-excursion
      (goto-char start)
      (sde-skip-chars end)
      (if (looking-at sde-sp-name-regexp)
	  (setq from (buffer-substring (match-beginning 1) (match-end 1)))
	  (setq from (buffer-substring (point) (progn (end-of-line) (point)))))
      (goto-char end)
      ;; See if we're in the blank space or comments following some expression
      (re-search-backward "^(" nil 'move)
      (forward-sexp 1)
      (sde-skip-chars end)
      (if (= (point) end)
	  (progn
	    (re-search-backward "^(" nil 'move)
	    (if (looking-at sde-sp-name-regexp)
		(setq to (buffer-substring (match-beginning 1) (match-end 1)))
		(setq to (buffer-substring (point) (progn (end-of-line) (point))))))
	  ;; Can't recognize where we are, so punt.
	  (progn
	    (beginning-of-line)
	    (setq to (buffer-substring (point) (progn (end-of-line) (point))))))
      (setq from (sde-string-trim from)
	    to   (sde-string-trim to))
      (if (string-equal from to)
	  from
	  (concat "From " from " to " to)))))


;; Modified from Emacs' mark-defun.

(defun sde-mark-sp ()
  "Put mark at end of sp form, point at beginning."
  (interactive)
  (push-mark (point))
  (sde-end-of-sp)
  (push-mark (point))
  (sde-beginning-of-sp)
  (re-search-backward "^\n" (- (point) 1) t))


;;;----------------------------------------------------------------------------
;;; 9.  Search-related functions.
;;;----------------------------------------------------------------------------


(defun sde-skip-chars (end)
  "Skip past whitespace, comments, backslashed characters and strings in the
current buffer as long as you are before END.  This does move the point."
  (if (< (point) end)
      (let ((comment (and comment-start (string-to-char comment-start)))
	    (done nil)
	    char)
	(while (and (< (point) end)
		    (not done))
	  (skip-chars-forward "\n\t " end)
	  (setq char (char-after (point)))
	  (cond ((eq char ?\")
		 (forward-sexp))
		((eq char comment)
		 (forward-char)
		 (skip-chars-forward "^\n" end))
		((eq char ?\\)
		 (forward-char 2))
		(t
		 (setq done t)))))))


(defun sde-skip-regexp-forward (regexp)
  "Keep moving point forward over REGEXP, stopping when the next characters in
the buffer do not match REGEXP."
  (while (looking-at regexp)
    (goto-char (match-end 0))))


(defun sde-skip-regexp-backward (regexp)
  "Keep moving point backward over REGEXP, with REGEXP *ending* at the current
point, until a character that is not matched by REGEXP."
  (let ((point (point)))
    (while (progn
	     (re-search-backward regexp (point-min) t)
	     (looking-at regexp)	; Get the match data
	     (= (match-end 0) point))
      (goto-char (match-beginning 0))
      (setq point (point)))
    (goto-char point)))


(defun sde-search-buffer (buffer regexp &optional start)
  "Search forward in BUFFER for REGEXP, starting from the top.
Optional arg START specifies a starting position.  Returns the value of point
where pattern found, or nil."
  (save-excursion
    (set-buffer buffer)
    (if start
	(goto-char start)
	(goto-char 1))
    (if (re-search-forward regexp nil t)
	(point))))


(defun sde-search-file (filename regexp &optional inhibit-msg)
  "Read FILENAME into Emacs and search for REGEXP.
This reads the raw file, not setting its mode or reading its local vars.
FILENAME must not already be in a buffer somewhere.  Prints a message
unless optional argument INHIBIT-MSG is non-nil."
  (let ((kill-buffer-hook nil)		; For Emacs 19, for speed.
	results buffer)
    (save-excursion
      (setq buffer (create-file-buffer filename))
      (set-buffer buffer)
      (erase-buffer)
      (if (condition-case ()
	      (insert-file-contents filename t)
	    (file-error nil))
	  (progn
	    (goto-char (point-min))
	    (or inhibit-msg (message "Scanning file %s..." buffer-file-name))
	    (if (re-search-forward regexp nil t)
		(setq results (point)))))
      (funcall sde-true-kill-buffer buffer))
    results))


;;;----------------------------------------------------------------------------
;;; 10. Extracting symbols and other objects from buffers.
;;;----------------------------------------------------------------------------
;;;
;;; Soar 6.1 allows the following to be used as part of a production name:
;;;     alphanumeric characters
;;;     $ % & * - _ = + < > / : ? 
;;;
;;; The characters < and >, if appearing in a production name, cannot appear
;;; themselves; the name must have at least one alphanumeric character on either
;;; side of the < or >.   If the string "<>" appears in a production name, it
;;; must be flanked by an alphanumeric character on each side, otherwise Soar
;;; will not accept it as a production name.
;;;
;;; Soar stops reading a production name if it encounters one of the following:
;;;     @ ~ { } ! ^ ( ) . , # \ ' 
;;;
;;; The | character begins and ends a multi-line comment.
;;; The " character begins and ends a string.
;;;
;;; Extracting symbols is most efficiently done by using the trick of
;;; temporarily changing the syntax table in the buffer so that sexp scanning
;;; can be used effectively.  The ideas here are based on the completion.el
;;; package from J. Salem & B. Kahle at Thinking Machines Corp.  Characters
;;; are put into special classes; then `forward-word' and `scan-sexps' can be
;;; used to move across whole components quickly, without consing or
;;; much string-matching.
;;;
;;; Note that it is impossible to extract symbols from Soar process buffers
;;; without error, because the characters allowed in Soar production names
;;; include characters that appear in Soar output message, such as `pgs'
;;; listings.  In fact, you can define a production named "==>G:" in Soar!
;;; The approach here is to use some explicit tests for certain components
;;; in combination with two syntax tables.
;;;
;;; Since Soar itself rejects production names and other symbols if they
;;; contain certain characters, it is reasonable to assume SDE won't
;;; encounter them in normal operation.  So the algorithms here are not
;;; fool-proof; rather, they're oriented toward the most probably situations.
;;;----------------------------------------------------------------------------

(defvar sde-symbol-syntax-table
  "Syntax table used for extracting Soar symbols from buffers.")

;; symbol syntax      ("_") -- all chars allowed in production names
;; prefix syntax      ("'") -- the dash, which serves dual purposes
;; punctuation syntax (".") -- the up-arrow of attributes
;; whitespace syntax  (" ") -- everything else
;; misc. other syntax for things like the string delimiters.

(if (not (syntax-table-p sde-symbol-syntax-table))
    (let ((table (make-vector 256 0))	; Default syntax is whitespace.
	  (i 0))
      ;; Alphabetic chars.
      (while (< i 26)
	(modify-syntax-entry (+ ?a i) "_" table)
	(modify-syntax-entry (+ ?A i) "_" table)
	(setq i (1+ i)))
      ;; Digits.
      (setq i 0)
      (while (< i 10)
	(modify-syntax-entry (+ ?0 i) "_" table)
	(setq i (1+ i)))
      ;; Other characters.
      (mapcar (function
	       (lambda (char)
		(modify-syntax-entry char "_" table)))
	      '(?$ ?% ?& ?* ?_ ?= ?+ ?< ?> ?/ ?: ??))
      (modify-syntax-entry ?\| "$|" table)
      (modify-syntax-entry ?-  "'"  table)
      (modify-syntax-entry ?^  "."  table)
      (modify-syntax-entry ?\" "\"" table)
      (setq sde-symbol-syntax-table table)))


(defun sde-extract-symbol (&optional start)
  "Returns list (SYMBOL BEGINNING END) for symbol under or to left of point.
BEGINNING and END are the buffer positions of the SYMBOL.  Optional argument
START is where to move before prior to extracting symbol\; point is left
there after extracting the symbol.  Special cases: If the symbol is an
attribute name, the leading up-arrow of the attribute is *not* included.  If
the symbol under point is actually a timetag in a WME printout, only the
number is returned\; the trailing colon is omitted.  Otherwise, this function
tries to returns whatever looks like a Soar symbol or production name near
point."
  (let ((old-syntax-table (syntax-table))
	(saved-point (or start (point))))
    (goto-char saved-point)
    (unwind-protect
	 (let (beginning end)
	   (set-syntax-table sde-symbol-syntax-table)
	   ;; First back up to something non-whitespace.
	   (if (= (char-syntax (following-char)) ? )
	       (skip-chars-backward " \t"))
	   ;; Find beginning and end of symbol.  If we're near the `^' or
	   ;; negation `-' of an attribute, or in the middle of something,
	   ;; find the boundaries by first moving forward.  Else, find the
	   ;; symbol's boundaries moving backward.
	   (if (or (memq (preceding-char) '(?- ?^))
		   (/= (char-syntax (following-char)) ? ))
	       (setq end       (scan-sexps (point) 1)
		     beginning (scan-sexps end -1))
	       (setq beginning (scan-sexps (point) -1)
		     end       (scan-sexps beginning 1)))
	   ;; Check if this may be a WME timetag.
	   (if (char-equal (char-after (1- end)) ?:)
	       (progn
		 (goto-char (1- end))
		 (skip-chars-backward "0-9") 
		 (if (= (point) beginning) ; Nothing but digits => timetag.
		     (setq end (1- end)))))
	   (list (buffer-substring beginning end) beginning end))
      (goto-char saved-point)
      (set-syntax-table old-syntax-table))))


(defun sde-symbol-near-point (&optional predicate prompt-str)
  "Extract the symbol under or before point, and return it in string form.
If optional arg PREDICATE is non-nil, it must be a function.  PREDICATE is
funcall'ed with the extracted symbol string as its single argument, and it
should return non-nil if the string is acceptable, nil otherwise.  If
PREDICATE returns nil, the user is prompted with optional argument PROMPT-STR
and the extracted symbol as a default answer which the user can select just
by typing return.  If no symbol is found near the point, and PROMPT-STR is
nil, the user is prompted with the default prompt \"Target: \".  This
function returns the symbol string found near point or a new string typed by
the user.  If nothing is found and the user doesn't supply a symbol, an error
is signaled."
  (interactive)
  (if (eobp)
      (if (not (bobp))
	  (save-excursion
	    (backward-char 1)
	    (sde-symbol-near-point predicate prompt-str))
	  (error "Empty buffer."))
      (let* ((prompt-str (or prompt-str "Target: "))
	     (sym (car (sde-extract-symbol))))
	;; Check that we have something, and that if predicate is supplied it
	;; returns non-nil on the candidate symbol.
	(if (or (null sym) (and predicate (not (funcall predicate sym))))
	    (if (string= "" (setq sym (sde-read-string
				       (if sym
					   (format "%s(default \"%s\") " prompt-str sym)
					   (format "%s" prompt-str)))))
		(error "No symbol found or provided.")))
	sym)))


(defun sde-sp-name-near-point (&optional prompt-str)
  "Extract the name of the production under or before point.
If point is inside a full sp expression, the production name is unambiguous.
If point is elsewhere, this function extracts the symbol under or before
point, and heuristically judges whether it is likely to be a production
name."
  (interactive)
  (or (sde-sp-name)
      (sde-symbol-near-point 'sde-production-name-p "Production name: ")))
  

;; General function for extracting identifiers, attributes and values from
;; objects printed by Soar.
;;
;; The approach is attribute oriented.  The user should put point over or
;; near an attribute in the production LHS.  The following tries to grab the
;; attribute, the value following it (often it's a variable), and the id at
;; the beginning of the form.
;;
;; This uses the constants `sde-attribute-regexp', `sde-value-regex', etc.,
;; but negated attribute tests are handled explicitly (by looking for the
;; negation sign).
;; 
;; PROBLEMS: This doesn't handle the following constructs:
;;
;; 1)  (<id> ^{ <> tried-tied-operator <sub-att> } <sub-val>)
;;
;; 2)  (<s> ^{ << required-success success partial-success 
;;                draw lose win >> <svalue> } <eb> )
;;
;; But as it turns out we never have to deal with those cases, since those
;; only appear in productions and not in objects printed by Soar.

(defconst sde-attribute-regexp "\\^\\s *[^ \t(){}<]+"
  "Regexp to match an attribute.")

(defconst sde-id-regexp "[a-zA-Z][0-9]+"
  "Regexp to match an identifier.")

(defconst sde-variable-regexp "<[a-zA-Z0-9]+>"
  "Regexp to match a production variable.")

(defconst sde-id-or-variable-regexp
    (concat "\\(" sde-id-regexp "\\|" sde-variable-regexp "\\)")
  "Regexp to match either an identifier or a variable.")

(defconst sde-symbol-regexp "[^^ \t\n<>()]+"
  "Regexp to match a symbol used as a value in a production clause.")

(defconst sde-value-regexp
    (concat "\\(" sde-variable-regexp "\\|" sde-id-regexp
	    "\\|" sde-symbol-regexp "\\)")
  "Regexp to match an attribute value.")


(defun sde-extract-id-attrib-value ()
  "Return a list (id negation attribute value preference).
This function scans the text near the cursor and attempts to extract an
attribute, its value, and an identifier at the head of the clause.      

This function is only suitable for productions written with a simple syntax,
such as those printed out by Soar itself.  It can only understand simple
negated attribute tests and acceptable preferences (`+').

The list returned consists of strings.  The `negation' portion of the
returned list is equal to \"-\" if the attribute had a leading negation sign,
or nil otherwise.  The `attribute' portion of the list is only the attribute
name, and does not have a leading up-arrow.  The `preference' portion is
equal to the string \"+\" if the value in the clause was followed by an
acceptable preference, or nil otherwise."
  (let ((saved-point (point))
	(preference "")
	(negation "")
	current sym id attrib attrib-end value)
    (setq current (sde-extract-symbol))
    ;; Go to beginning of this symbol and see if it's an attribute.
    (goto-char (nth 1 current))
    (skip-chars-backward " \t")
    (if (char-equal (preceding-char) ?^)
	(setq attrib     (car current)
	      attrib-end (nth 2 current))
	;; Not an attribute; maybe it's a value.  Back up further and try again.
	(progn
	  (setq current (sde-extract-symbol)) ; Get symbol we're now next to.
	  (goto-char (nth 1 current))	      ; Go to its beginning
	  (skip-chars-backward " \t")
	  (if (char-equal (preceding-char) ?^)
	      (setq attrib     (car current)
		    attrib-end (nth 2 current)))))
    ;; At this point, we should be right after the up-arrow.  See if there's
    ;; a negation sign in front, then try to get the value after the attribute.
    (if attrib
	(progn
	  (backward-char 1)
	  (skip-chars-backward " \t")
	  (if (char-equal (preceding-char) ?-)
	      (setq negation "-"))
	  ;; Get the value.
	  (setq current (sde-extract-symbol (scan-sexps attrib-end 1))
		sym     (car current))
	  (if (string-match sde-value-regexp sym)
	      (progn
		(setq value sym)
		;; See if there's a preference following the value.
		(goto-char (nth 2 current))
		(skip-chars-forward " \t")
		(if (char-equal (following-char) ?+)
		    (setq preference "+"))))))
    ;; Look for identifier.
    (search-backward "(" nil t)
    (forward-char 1)
    (setq current (sde-extract-symbol)
	  sym     (car current))
    (if (string-match "goal\\|impasse" sym)
	(progn
	  (goto-char (scan-sexps (nth 2 current) 1))
	  (setq current (sde-extract-symbol)
		sym     (car current))))
    (if (string-match sde-id-or-variable-regexp sym)
	(setq id sym))
    (goto-char saved-point)
    (list id negation attrib value preference)))


;;;----------------------------------------------------------------------------
;;; 8.  Buffer and window handling
;;;----------------------------------------------------------------------------

;;; NOTE: a large number of the sde-xxx functinos for window handling are
;;; defined either in sde-emacs19.el or sde-lemacs.el.  This permits handling
;;; the different names ("frame" vs "screen") with functions defined at load
;;; time instead of putting a lot of conditionals all over the place.

(defmacro sde-get-buffer-create (name)
  "Create buffer named NAME.
Like `get-buffer-create', but binds `default-major-mode' to `fundamental-mode'.
This is a hack necessary to counteract a problem for users who set their
default major mode to `text-mode' or other.  It prevents Emacs from running
`text-mode-hook' when a buffer is created."
  (` (let ((default-major-mode 'fundamental-mode))
       (get-buffer-create (, name)))))


(defmacro sde-buffer-file-name (buffer)
  "Return the value of variable `buffer-file-name' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) buffer-file-name)))


(defmacro sde-buffer-mode (buffer)
  "Return the value of variable `major-mode' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) major-mode)))


(defmacro sde-buffer-directory (buffer)
  "Return the value of `default-directory' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) default-directory)))


(defmacro sde-buffer-exists-p (buffer)
  "Return non-nil if BUFFER exists and is not killed.
BUFFER can be a buffer or a buffer name."
  (` (if (stringp (, buffer))
	 (and (bufferp (get-buffer (, buffer)))
	      (buffer-name (get-buffer (, buffer))))
	 (and (bufferp (, buffer))
	      (buffer-name (, buffer))))))


(defun sde-window-list (&optional all-frames)
  "Return a list of all windows excluding the minibuffer's windows.  
If optional arg ALL-FRAMES is non-nil, also lists windows on other
frames, including invisible and iconified frames."
  ;; Emacs 19.22's get-buffer-window completely ignores invisible windows,
  ;; so we have to do this the hard way.
  (let* ((first (selected-window))
	 (current first)
	 (windows nil))
    (while (progn
	     (setq windows (cons current windows)
		   current (sde-next-window-any current all-frames))
	     (not (eq current first))))
    windows))


(defun sde-create-buffer-frame (buffer param &rest args)
  "Create new frame/screen for BUFFER with parameters PARAM.
Optional third argument VISIBILITY determines whether the window is shown to
the user immediately\; the three possible values are `t' for visible, `nil'
for invisible, and `icon' for making the new frame iconified.  Default: t.
Optional fourth argument DEDICATED determines whether the frame is dedicated
to showing BUFFER's window\; `t' means yes, `nil' (the default) means no.

The argument PARAM is an alist specifying frame/screen parameters.  Any
parameters not mentioned in ALIST defaults according to the value of the
variable `default-frame-alist' in Emacs 19\; parameters not specified there
default from the standard X defaults file and X resources."
  (if (bufferp buffer)
      (let* ((orig-frame (sde-selected-frame))
	     (visibility (if (consp args) (car args) t))
	     (dedicated (if (consp args) (car (cdr args)) nil))
	     (frame (sde-make-frame
		     buffer (append (list (cons 'visibility visibility)) param))))
	(unwind-protect
	     (progn
	       (sde-select-frame frame)
	       (delete-other-windows)
	       (switch-to-buffer buffer)
	       (sde-set-window-dedicated-p (sde-frame-root-window frame) dedicated)
	       (if visibility
		   (sde-raise-frame frame)))
	  (sde-select-frame orig-frame)))))


(defun sde-delete-buffer-frame (buffer)
  "Delete frames containing BUFFER.  All existing frames are examined.
Frames are only deleted if they contain one window and that window
is showing the given BUFFER."
  (let ((orig-frame (sde-selected-frame))
	(frames (sde-frame-list)))
    (if (cdr frames)			; If there's at least one frame.
	(unwind-protect
	     (mapcar (function
		      (lambda (frame)
		       (sde-select-frame frame)
		       (if (and (one-window-p)
				(eq (window-buffer (selected-window)) buffer))
			   (sde-delete-frame frame))))
		     (sde-frame-list))
	  (if (sde-frame-live-p orig-frame)
	      (sde-select-frame orig-frame))))))


(defun sde-show-frame (frame)
  "Make the given FRAME visible and raised if it is not already.
This function can only work in FSF Emacs 19 and Lucid Emacs 19.
In Emacs 18 it does nothing."
  (if (sde-frame-p frame)
      (sde-raise-frame (sde-make-frame-visible frame))))


;; sde-show-buffer is for buffers that are not special, i.e., not the
;; *output* and *error* buffers.  For those there are special functions in
;; sde-soar-mode.el.

(defun sde-show-buffer (buffer &optional switch here)
  "Display BUFFER, preferring a window in which it is already showing.  
If optional arg SWITCH is non-nil, switch to the buffer after displaying it.
If optional arg HERE is non-nil, show the buffer in the selected window
unless it is already being shown in another window.  If BUFFER is already
being shown in a window somewhere, make that window and frame visible if
necessary, and switch to it if SWITCH."
  (if (bufferp buffer)
      (let ((window (sde-buffer-window buffer t)))
	(cond (window
	       ;; There is a window currently displaying the buffer.  
	       ;; Make sure it's visible and switch to it if desired.
	       (if (or sde-running-lemacs sde-running-emacs19)
		   (sde-show-frame (sde-buffer-frame buffer)))
	       (if switch
		   (progn
		     (select-window window)
		     (set-buffer buffer))))
	      ;; There is no window displaying the buffer.
	      (here
	       (switch-to-buffer buffer))
	      (switch
	       (pop-to-buffer buffer)
	       (set-buffer buffer))
	      (t
	       (display-buffer buffer))))))


(defvar sde-last-buffer nil
  "The last used Soar buffer.")

(defun sde-pop-to-buffer (buffer)
  "Display BUFFER in a different window and switch to that window.
This function sets `sde-last-buffer' to the current buffer."
  (if (not (equal (current-buffer) sde-last-buffer))
      (setq sde-last-buffer (current-buffer)))
  (if (sde-buffer-exists-p buffer)
      (sde-show-buffer buffer t)))


;;;----------------------------------------------------------------------------
;;; 7.  Miscellaneous basic functions
;;;----------------------------------------------------------------------------

;; Emacs 18 doesn't provide add-hook.  Things are safer if we define our own,
;; even though it's duplicating code available in many other packages.  Based
;; on code posted by Tom May <uw-nsr!uw-warp!tom@beaver.cs.washington.edu>.
;; Differs in that it puts the new function on the front of the list.

(defun sde-add-hook (hook-var hook-fun)
  "Two arguments, HOOK-VAR and HOOK-FUN.  Adds HOOK-FUN to the *front* of the
list of hooks in HOOK-VAR if it is not already present.  `sde-add-hook' is very
tolerant: HOOK-VAR need not be previously defined, its value doesn't have to
be a list, lamda expressions are cool, etc."
  (or (boundp hook-var)
      (set hook-var nil))
  (let ((hook-var-val (symbol-value hook-var)))
    (or (listp hook-var-val)
	(setq hook-var-val (cons hook-var-val nil)))
    (if (eq (car hook-var-val) 'lambda)
	(setq hook-var-val (cons hook-var-val nil)))
    (or (sde-member hook-fun hook-var-val)
	(set hook-var (cons hook-fun hook-var-val)))))


;; `remove-hook' is unavailable prior to Emacs 19.  The following is from
;; subr.el from Emacs 19.22.

(defun sde-remove-hook (hook function)
  "Remove from the value of HOOK the function FUNCTION.
HOOK should be a symbol, and FUNCTION may be any valid function.  If
FUNCTION isn't the value of HOOK, or, if FUNCTION doesn't appear in the
list of hooks to run in HOOK, then nothing is done.  See `add-hook'."
  (if (or (not (boundp hook))		;unbound symbol, or
	  (null (symbol-value hook))	;value is nil, or
	  (null function))		;function is nil, then
      nil				;Do nothing.
    (let ((hook-value (symbol-value hook)))
      (if (consp hook-value)
	  (setq hook-value (delete function hook-value))
	(if (eq hook-value function)
	    (setq hook-value nil)))
      (set hook hook-value))))


(defun sde-prefix-arg-negative-p ()
  "Return non-nil if `current-prefix-arg' is a negative prefix."
  (and current-prefix-arg
       (or (symbolp current-prefix-arg)
	   (< (prefix-numeric-value current-prefix-arg) 0))))


(defun sde-prefix-arg-value ()
  "Return the absolute value of `current-prefix-arg' as a number, or nil."
  (and current-prefix-arg
       (not (symbolp current-prefix-arg))
       (let ((value (prefix-numeric-value current-prefix-arg)))
	 (if (< value 0)
	     (- value)
	     value))))


(defun sde-cd (dir)
  "Make DIR become the current buffer's default directory.  
This function is like Emacs's `cd' but does its work silently."
  (interactive "DChange default directory: ")
  (setq dir (expand-file-name dir))
  (if (not (eq system-type 'vax-vms))
      (setq dir (file-name-as-directory dir)))
  (if (not (file-directory-p dir))
      (error "%s is not a directory" dir)
      (setq default-directory dir)))


(defun sde-count-buffer-lines (buffer)
  "Count number of lines in BUFFER.
This version is simpler and somewhat faster than the standard Emacs
function, but is also potentially less accurate."
  (save-excursion			; This counts the number of lines in
    (set-buffer buffer)			;  buffer.  This is what Emacs'
    (goto-char 1)			;  `count-lines' essentially does.
    (- (buffer-size) (forward-line (buffer-size)))))


(defun sde-integer-string-p (string)
  "Return non-nil if STRING is the string representation of an integer."
  (and string
       (not (string= string ""))
       (setq string (car (read-from-string string)))
       (numberp string)))


(defun sde-production-name-p (string)
  "Return non-nil if STRING is matched by `sde-production-name-test-regexp'."
  (and string
       (let ((match-data (match-data)))
	 (prog1
	     (string-match sde-production-name-test-regexp string)
	   (store-match-data match-data)))))


;; Modified from lisp-in-string from ilisp-ext.el

(defun sde-in-string (&optional begin end)
  "Return the string region that immediately follows/precedes point or that
contains point in optional region BEGIN to END.  If point is in region, t
will be returned as well."
  (save-excursion
    (if (not begin)
	(save-excursion
	  (setq begin (sde-beginning-of-sp 1 t)
		end (sde-end-sp-text t))))
    (let* ((point (progn (skip-chars-forward " \t") (point)))
	   (done nil))
      (goto-char begin)
      (while (and (< (point) end) (not done))
	(skip-chars-forward "^\"" end)
	(setq begin (point))
	(if (< begin end)
	    (if (and (not (bobp)) (= (char-after (1- begin)) ??))
		(forward-char)
		(if (condition-case () (progn (forward-sexp) (<= (point) end))
		      (error nil))
		    (progn		;After string
		      (skip-chars-forward " \t")
		      (if (or (= begin point) (= point (point)))
			  (setq done (list begin (point) nil))
			  (if (and (< begin point) (< point (point)))
			      (setq done (list begin (point) t)))))
		    ;; In string at end of buffer
		    (setq done (list begin end t))))))
      done)))


;; Modified from lisp-end-defun-text from ilisp-ext.el

(defun sde-end-sp-text (&optional at-start) 
  ;; Go the end of the text associated with the current sp and return
  ;; point.  The end is the last character before whitespace leading to
  ;; a left paren or ;;; at the left margin unless it is in a string.
  (if (not at-start)
      (sde-beginning-of-sp))
  (let ((point (point))
	(boundary (sde-find-next-start))
	(final (save-excursion
		 (condition-case ()
		     (progn (forward-sexp) (point))
		   (error (point-max))))))
    ;; Find the next line starting at the left margin and then check
    ;; to see if it is in a string. 
    (while (progn
	     (skip-chars-forward "^\"" boundary) ;To the next string
	     (if (= (point) boundary)	
		 nil			;No quote found and at limit
		 (let ((string-boundary ;Start of next defun
			(save-excursion
			  (if (re-search-forward "^\(\\|^;;;" nil t)
			      (match-beginning 0)
			      (point-max)))))
		   (if (condition-case ()
			   (progn (forward-sexp) t)
			 (error (goto-char string-boundary) nil))
		       (if (>= (point) boundary)
			   ;; Boundary was in string
			   (if (> (point) string-boundary)
			       (progn	;String ended in next defun
				 (goto-char string-boundary)
				 nil)
			       (if (> (setq boundary
					    (sde-find-next-start))
				      final)
				   ;; Normal defun
				   (progn (goto-char final) nil)
				   t))
			   t)
		       ;; Unclosed string
		       nil)))))
    (re-search-backward  "^[^; \t\n]\\|^[^;\n][ \t]*[^ \t\n]" point t)
    (end-of-line)
    (skip-chars-backward " \t")
    (if (< (point) point)
	(goto-char point)
	(if (save-excursion
	      (let ((point (point)))
		(beginning-of-line)
		(if comment-start (search-forward comment-start point t))))
	    (progn (next-line 1) (sde-indent-line)))
	(point))))


(defun sde-find-next-start ()
  ;; Find the start of the next line at the left margin that starts with
  ;; something other than whitespace, a -, a \), three or more semicolons, or
  ;; a \( not followed by <, and return point.
  (if (eobp)
      (point-max)
      (save-excursion
	(forward-char)
	(if (re-search-forward "^\\(\\(;;;\\)\\|\\(\([^<]\\)\\|\\([^-^( \t\n\);]\\)\\)" nil t)
	    (match-beginning 0)
	    (point-max)))))


;;;----------------------------------------------------------------------------
;;; 11. Recordkeeping
;;;----------------------------------------------------------------------------


(defun sde-record-file (filename)
  "Record file FILENAME on sde-known-files."
  (sde-puthash filename filename sde-known-files))


;;;----------------------------------------------------------------------------
;;; 12. Error and diagnostic handling.
;;;----------------------------------------------------------------------------


(defun sde-error-soar-not-multi-agent ()
  ;; Generate an error that Soar isn't running in multi-agent mode.
  (error "Soar is not running in multiple-agent mode."))


(defun sde-error-unset-site-var (var)
  (error "Variable `%s' not set; contact your SDE maintainer." var))    


;;;----------------------------------------------------------------------------
;;; 13. Parenthesis handling.
;;;----------------------------------------------------------------------------


(defun sde-close-all-sp (&optional arg) 
  "Insert enough parentheses to close the enclosing production.
If there are too many parens, excess parentheses are deleted.  The form
is also indented."
  (interactive "P")
  (let ((begin (sde-beginning-of-sp 1 t))
	(end (sde-end-sp-text t))
	(count 0))
    (goto-char end)
    ;; Loop by going to the beginning, attempting to forward-sexp,
    ;; and inserting right parens until we get no more errors.
    (save-restriction
      (narrow-to-region (point) begin)
      (while (save-excursion
	       (progn (sde-beginning-of-sp)
		      (condition-case nil
			  (progn (forward-sexp 1) nil)
			(error t))))
	(insert ?\))
	(setq count (1+ count)))
      ;; Delete the extra parens we inserted here.
      (if (not (eq end (point)))
	  (delete-region end (point))))
    ;; Now insert the real parens.
    (goto-char end)
    (while (> count 0)
      (insert ?\))
      (setq count (1- count)))
    (goto-char begin)
    (sde-indent-sexp)
    (forward-sexp 1)))


;;;----------------------------------------------------------------------------
;;; 14. Indentation support.
;;;----------------------------------------------------------------------------
;;;
;;; Nearly all of the following functions are based on code from either 
;;; Emacs lisp-mode.el (both Emacs 18 and 19.19) and Ilisp 4.12.  
;;;
;;; The need to duplicate some lisp-mode.el functionality arose because the
;;; lisp indentation routines make assumptions about the structure of the
;;; code that are not true for Soar code.  The biggest problems is
;;; lisp-mode.el's indent-sexp works in such a way that only changes
;;; indentation levels when it encounters a nested sexp.  I.e.,
;;;
;;;	     (foo)
;;;	     (bar)
;;;
;;; end up indented the same way because none of the expressions introduce a
;;; new, more-nested sexp.  On the other hand,
;;;
;;;          (foo (bar)
;;;            (biff)
;;;
;;; introduces a new level of indentation because of the sexp (bar).  The
;;; problem for Soar code is that negated clauses need to be indented
;;; slightly differently, and as far as I can tell, there is no easy way to
;;; use the lisp-indent-function to make things work out properly.
;;; lisp-mode.el's indent-sexp will simply indent negated clauses at the same
;;; block level as the previous line,
;;; 
;;;          (<s> ^color purple)
;;;          -{(<s> ^biff y)}
;;;          (<s> ^free food)
;;;
;;; and that is ugly.  For now the only way out seems to be to copy
;;; indent-sexp and hack it to understand Soar productions more specifically.
;;; The functions lisp-indent-line and indent-sexp don't have any hooks on
;;; which the modifications could be installed, so I had to copy them and
;;; modify them appropriately.  
;;;
;;; Other code here is stolen from Ilisp, as usual, to avoid having to carry
;;; around ilisp-ext.el.
;;;
;;; All of this is messy and really should be improved, but I didn't have
;;; time to do better.


(defun sde-newline-and-indent ()
  "If at the end of the buffer and end of an sp, send the string back to the
process mark with no newline.  Otherwise, insert a newline, then indent.  In
a Soar process buffer the region is narrowed first.  See `newline-and-indent'
for more information."
  (interactive "*")
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region sde-soar-buffer-mark (point-max)))
    (delete-region (point) (progn (skip-chars-backward " \t") (point)))
    (insert ?\n)
    (sde-indent-line-internal nil)))


(defun sde-indent-line (&optional whole-exp)
  "Indent current line as Soar code.  
With argument, indent any additional lines of the same expression rigidly
along with this one.  This is restricted to the current buffer input."
  (interactive "P")
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region sde-soar-buffer-mark (point-max)))
    (sde-indent-line-internal whole-exp)))


;; Slightly modified lisp-indent-line from Emacs 19 lisp-mode.el.

(defun sde-indent-line-internal (&optional whole-exp)
  ;; Indent current line as Lisp code.  With argument, indent any additional
  ;; lines of the same expression rigidly along with this one.
  (let ((indent (calculate-lisp-indent)) shift-amt beg end
	(pos (- (point-max) (point))))
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (looking-at "\\s<\\s<\\s<")
	;; Don't alter indentation of a ;;; comment line.
	(goto-char (- (point-max) pos))
      (if (and (looking-at "\\s<") (not (looking-at "\\s<\\s<")))
	  ;; Single-semicolon comment lines should be indented
	  ;; as comment lines, not as code, except if they start
	  ;; in the left column.
	  (or (bolp)
	      (progn
		(indent-for-comment)
		(forward-char -1)))
	(if (listp indent) (setq indent (car indent)))
	(setq indent    (sde-adjust-indent indent)
	      shift-amt (- indent (current-column)))
	(if (zerop shift-amt)
	    nil
	  (delete-region beg (point))
	  (indent-to indent)))
      ;; If initial point was within line's indentation,
      ;; position after the indentation.  Else stay at same point in text.
      (if (> (- (point-max) pos) (point))
	  (goto-char (- (point-max) pos)))
      ;; If desired, shift remaining lines of expression the same amount.
      (and whole-exp (not (zerop shift-amt))
	   (save-excursion
	     (goto-char beg)
	     (forward-sexp 1)
	     (setq end (point))
	     (goto-char beg)
	     (forward-line 1)
	     (setq beg (point))
	     (> end beg))
	   (indent-code-rigidly beg end shift-amt)))))


;; Slightly modified indent-sexp from Emacs 19 lisp-mode.el.

(defun sde-indent-sexp (&optional endpos)
  "Indent each line of the list starting just after point.
If optional arg ENDPOS is given, indent each line, stopping when
ENDPOS is encountered."
  (interactive)
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region sde-soar-buffer-mark (point-max)))
    (let ((indent-stack (list nil))
	  (next-depth 0)
	  (starting-point (point))
	  (last-point (point))
	  last-depth bol outer-loop-done inner-loop-done state this-indent)
      ;; Get error now if we don't have a complete sexp after point.
      (save-excursion (forward-sexp 1))
      (save-excursion
	(setq outer-loop-done nil)
	(while (if endpos (< (point) endpos)
		   (not outer-loop-done))
	  (setq last-depth next-depth
		inner-loop-done nil)
	  ;; Parse this line so we can learn the state to indent the next
	  ;; line.  This inner loop goes through only once unless a line ends
	  ;; inside a string.
	  (while (and (not inner-loop-done)
		      (not (setq outer-loop-done (eobp))))
	    (setq state (parse-partial-sexp (point) (progn (end-of-line) (point))
					    nil nil state))
	    (setq next-depth (car state))
	    ;; If the line contains a comment other than the sort that is
	    ;; indented like code, indent it now with indent-for-comment.
	    ;; Comments indented like code are right already.  In any case
	    ;; clear the in-comment flag in the state because
	    ;; parse-partial-sexp never sees the newlines.
	    (if (car (nthcdr 4 state))
		(progn (indent-for-comment)
		       (end-of-line)
		       (setcar (nthcdr 4 state) nil)))
	    ;; If this line ends inside a string, go straight to next line,
	    ;; remaining within the inner loop, and turn off the \-flag.
	    (if (car (nthcdr 3 state))
		(progn
		  (forward-line 1)
		  (setcar (nthcdr 5 state) nil))
		(setq inner-loop-done t)))
	  (and endpos
	       (<= next-depth 0)
	       (progn
		 (setq indent-stack (append indent-stack
					    (make-list (- next-depth) nil))
		       last-depth (- last-depth next-depth)
		       next-depth 0)))
	  (or outer-loop-done
	      (setq outer-loop-done (<= next-depth 0)))
	  (if outer-loop-done
	      (forward-line 1)
	      (while (> last-depth next-depth)
		(setq indent-stack (cdr indent-stack)
		      last-depth (1- last-depth)))
	      (while (< last-depth next-depth)
		(setq indent-stack (cons nil indent-stack)
		      last-depth (1+ last-depth)))
	      ;; Now go to the next line and indent it according
	      ;; to what we learned from parsing the previous one.
	      (forward-line 1)
	      (setq bol (point))
	      (skip-chars-forward " \t")
	      ;; Ignore blank lines.
	      (if (not (or (eobp) (looking-at "\\s<\\|\n")))
		  (progn
		    (if (and (car indent-stack)
			     (>= (car indent-stack) 0))
			(setq this-indent (car indent-stack))
			(let ((val (calculate-lisp-indent
				    (if (car indent-stack)
					(- (car indent-stack))
					starting-point))))
			  (if (integerp val)
			      (setcar indent-stack (setq this-indent val))
			      (progn
				(setcar indent-stack (- (car (cdr val))))
				(setq this-indent (car val))))))
		    ;; Modify the value of this-indent for special cases:
		    ;; single comments, negated clauses, or the arrow.
		    ;; Double-semicolon comments are indented as code.
		    (setq this-indent (sde-adjust-indent this-indent))
		    (if (/= (current-column) this-indent)
			(progn
			  (delete-region bol (point))
			  (indent-to this-indent))))))
	  (or outer-loop-done
	      (setq outer-loop-done (= (point) last-point))
	      (setq last-point (point))))))))


;; How this works:
;;
;; sde-adjust-indent gets called by sde-indent-sexp and by sde-indent-line.
;; At the time it is called, it is facing an about-to-be-indented line.  The
;; "given-indentation" argument is the indentation of the current block of
;; code, where a block is determined by s-expression nesting.
;; sde-adjust-indent is called on every line and is used to adjust the
;; indentation for special cases, such as negated clauses, the production
;; arrow, etc.
;;
;; sde-indent-hook is called by calculate-lisp-indent, which in turn is
;; called by sde-indent-sexp and sde-indent-line.  It is used to determine
;; the indentation for *new* nested blocks of code.
;;
;; sde-indent-line is called by indent-region by virtue of the fact that it
;; is the value of the variable indent-line-function (set in sde-mode).

(defun sde-adjust-indent (given-indentation)
  ;; Must be called with point facing the first non-blank character on a line
  ;; to be examined, and with GIVEN-INDENTATION the suggested indentation
  ;; value.  Returns an adjusted value based on what part of a Soar
  ;; production is on this line.  
  (cond ((looking-at "-->")		; The arrow.
	 (+ given-indentation sde-arrow-indent-offset))
	((looking-at "-\\s *{")		; Conjunctive negation clause.
	 (- given-indentation 2))
	((looking-at "-\\(\\s *\\)[(^]") ; Single negated clause or attrib.
	 ;; Aggressive reformatting of code.  Remove extra spaces between
	 ;; the negation and the rest.  Indent so that up-arrows line up.
	 (if (> (match-end 1) (match-beginning 1))
	     (save-excursion
	       (goto-char (match-beginning 1))
	       (delete-region (match-beginning 1) (match-end 1))
	       (insert ? )
	       (- given-indentation 2))
	     (- given-indentation 1)))
	(t				; Make sure to return something.
	 given-indentation)))


;; Indent hook for Soar code.  This is used by the standard Emacs
;; indentation routines to determine the column to which the current
;; line should be indented.
;;
;; From the doc string of parse-partial-sexp:
;;
;; State is a list of seven elements describing final state of parsing:
;; 0. depth in parens.
;; 1. character address of start of innermost containing list; nil if none.
;; 2. character address of start of last complete sexp terminated.
;; 3. non-nil if inside a string.
;;    (it is the character that will terminate the string.)
;; 4. t if inside a comment.
;; 5. t if following a quote character.
;; 6. the minimum paren-depth encountered during this scan.

(defun sde-indent-hook (indent-point state)
  (let ((containing-form-start (elt state 1)))
    (goto-char containing-form-start)
    (cond ((looking-at "(sp")		; Start of sp.
	   (if (eq major-mode 'sde-soar-mode)
	       (+ 6 sde-production-indent-offset)
	       sde-production-indent-offset))
	  ((looking-at "(<[^>]+>")	; Variable name after "("
	   (if (save-excursion
		 (goto-char indent-point)
		 (looking-at "[ \t]*<[^>]+>"))
	       (let (tmp)
		 (goto-char indent-point)
		 (forward-line -1)
		 (end-of-line)
		 (forward-sexp -1)
		 ;; Stop when see either "^foo" or "^{ <>" or "^{ <<".
		 ;; Gross code.  There must be a cleaner way.
		 (while (not (looking-at "\\^\\(\\(\\sw\\|\\s_\\)+\\|{\\s *<<?>?\\)"))
		   (setq tmp (point))
		   (forward-sexp -1))
		 (if tmp (goto-char tmp))
		 (current-column))
	       (progn
		 (forward-char 1)
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (list (current-column) containing-form-start))))
	  ((looking-at "(goal")		; The goal test clause.
	   (forward-char 1)
	   (forward-sexp 1)
	   (skip-chars-forward " \t\n")
	   (if (looking-at "<[^>]+>")
	       (progn
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (current-column))
	       (current-column)))
	  ((looking-at "{\\s *(")	; Beginning of grouped negation.
	   (skip-chars-forward "^(")
	   (current-column))
	  ((looking-at "{\\s *<<")	; Disjunction
	   (skip-chars-forward "{ \t<")
	   (current-column))
	  (t
	   (current-column)))))  


;; Slightly modified version of reindent-lisp from ilisp-ext.el, Ilisp v. 4.12.

;; The following marker keeps track of point so that it doesn't move during a
;; sde-reindent.  Made a global var to avoid calling make-marker repeatedly.

(defvar sde-fill-marker (make-marker))

(defun sde-reindent ()
  "Intelligently reindent the text under the cursor.
If in a comment, indent the comment paragraph bounded by non-comments, blank
lines or empty comment lines.  If in a string, indent the paragraph bounded
by string delimiters or blank lines.  Otherwise go to the containing sp form,
close it and reindent the code block."
  (interactive)
  (let ((region (sde-in-string))
	(comment (concat "[ \t]*" comment-start "+[ \t]*")))
    (set-marker sde-fill-marker (point))
    (back-to-indentation)
    (cond (region
	   (or (= (char-after (point)) ?\")
	       (and (< (point) (car region)) (goto-char (car region)))
	       (re-search-backward "^$" (car region) 'end))
	   (let ((begin (point))
		 (end (car (cdr region)))
		 (fill-prefix nil))
	     (forward-char)
	     (re-search-forward "^$" end 'end)
	     (if (= (point) end)
		 (progn (skip-chars-forward "^\n")
			(if (not (eobp)) (forward-char))))
	     (fill-region-as-paragraph begin (point))))
	  ((looking-at comment)
	   (let ((fill-prefix
		  (buffer-substring
		   (progn (beginning-of-line) (point))
		   (match-end 0))))
	     (while (and (not (bobp)) (sde-point-in-comment comment))
	       (forward-line -1))
	     (if (not (bobp)) (forward-line 1))
	     (let ((begin (point)))
	       (while (and (sde-point-in-comment comment) (not (eobp)))
		 (replace-match fill-prefix)
		 (forward-line 1))
	       (if (not (eobp))
		   (progn (forward-line -1)
			  (end-of-line)
			  (forward-char 1)))
	       (fill-region-as-paragraph begin (point)))))
	  (t
	   (goto-char sde-fill-marker)
	   (sde-close-all-sp)
	   (sde-beginning-of-sp 1 t)
	   (sde-indent-sexp)))
  (goto-char sde-fill-marker)
  (set-marker sde-fill-marker nil)
  (message "Done.")))


;;;----------------------------------------------------------------------------
;;; 15. Sending, excising, etc., productions and regions of productions.
;;;----------------------------------------------------------------------------


(defun sde-send-production ()
  "Send the production under the cursor to Soar.  
With prefix argument, send the production and switch to Soar.  If the new
production is a changed version of an already-known production of the same
name, Soar will automatically excise the previous definition and update it
with the new one.  Note that loading a production into Soar will unpbreak the
production, if a pbreak was in effect."
  (interactive)
  (sde-check-soar)
  (let ((body (sde-sp-body)))
    (if body
	(let* ((name (sde-sp-name))
	       (msg (concat "sp " name))
	       (agent (sde-agent)))
	  (sde-soar-cmd agent body nil nil current-prefix-arg msg msg)
	  ;; Clear possible pbreaks for this production.
	  (if (sde-pbreak-in-effect agent name)
	      (sde-unpbreak-production agent name)))
	(error "Point is not inside a production."))))


;; The next function actually sends the whole region, instead of extracting
;; the productions in the region and sending only those.  This seems more
;; flexible, allowing users to send more than productions in a buffer.  E.g.,
;; they may have load statements.  Without this facility, there would be no
;; way to send a non-production statement, except by sending the whole file.

(defun sde-region-send (start end)
  "Send the region between point and mark to the Soar process.  
With prefix argument, send the region and switch to Soar.  Note that doing so
will unpbreak all productions in the region, if pbreaks were in effect for
any of them."
  (interactive "r")
  (sde-check-soar)
  (let ((name (sde-region-name start end)))
    (sde-soar-cmd (sde-agent) (buffer-substring start end) 
		  nil nil current-prefix-arg name name)
    (sde-update-pbreak-list)))


(defun sde-region-pbreak (start end)
  "Pbreak each production in the region.  
I.e., redefine each production to interrupt Soar when it fires.  If given a
positive prefix arg, undoes the pbreaks on each production in the region.  If
given a negative prefix arg, undoes all currently active pbreaks (whether for
productions in the region or not).  Pbreaks are remembered on a per-agent
basis, so pbreak'ing a production in one agent will not automatically pbreak
that production in any other agents that may also share the same production.
To list the currently active pbreaks, use `\\[sde-view-pbreaks]'"
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (pbreak (sde-agent) name current-prefix-arg)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-ptrace (start end)
  "Ptrace each production in the region.  
If given a positive prefix arg, undoes ptraces on all the productions in the
region.  If given a negative prefix arg, undoes all currently active ptraces
(whether for productions in the region or not).  To list all the currently
active ptraces, use `\\[sde-view-ptraces]'"
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (ptrace (sde-agent) name current-prefix-arg)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-excise (start end)
  "Excise each production in the region."
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (excise (sde-agent) name)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-sp-names (start end)
  "Return a list of the names of all the productions in the given region."
  (if (and start end)			; Check for non-nil param
      (save-excursion
	(save-restriction
	  (narrow-to-region start end)
	  (goto-char (point-max))
	  (let (lst done pt)
	    (while (not done)		; Work backwards from end of region.
	      (setq pt (sde-beginning-of-sp)) ; This also moves point.
	      (if (looking-at "^(sp\\s +")
		  (progn
		    (goto-char (match-end 0))
		    (sde-push (buffer-substring (point) (progn (forward-sexp 1)
							       (point)))
			      lst)
		    (goto-char pt)))
	      (setq done (= (point) (point-min)))) ; Reached top of region yet?
	    ;; Return list.
	    lst)))))


;;;----------------------------------------------------------------------------
;;; 16. Miscellaneous editing commands
;;;----------------------------------------------------------------------------


(defun sde-close-and-send (arg)
  "Close and indent the current sp form, then send it to Soar."
  (interactive "P")
  (sde-reindent)
  (sde-close-all-sp arg)
  (if (eq major-mode 'sde-soar-mode)
      (sde-return)
      (save-excursion			; We're past last paren of sp,
	(backward-char 1)		;  so back up one char before
	(sde-send-production)		;  trying to eval.
	(forward-char 1))))


;; Find unbalanced delimiters.
;; Originally from ilisp-ext.el by Chris McConnell.

(defun sde-find-unbalanced (arg)
  "Go to the point in buffer where there exists an extra delimiter.  
Point will be on the offending delimiter.  If called with a prefix, use the
current region.  Checks for '{' '}' and '(' ')' delimiters."
  (interactive "P")
  (if arg
      (call-interactively 'sde-find-unbalanced-region)
      (sde-find-unbalanced-region (point-min) (point-max))))


(defun sde-find-unbalanced-region (start end)
  "Go to the point in region where LEFT-DELIMITER and RIGHT-DELIMITER
become unbalanced.  Point will be on the offending delimiter."
  (interactive "r")
  (sde-count-pairs start end ?\{ ?\} )
  (sde-count-pairs start end ?\( ?\) )
  (beep)
  (message "Delimiters balance."))


(defun sde-count-pairs (begin end left-delimiter right-delimiter)
  "Return the number of top-level pairs of LEFT-DELIMITER and
RIGHT-DELIMITER between BEGIN and END.  If they don't match, the point
will be placed on the offending entry."
  (let ((old-point (point))
	(sexp 0))
    (goto-char begin)
    (sde-skip-chars end)
    (while (< (point) end)
      (let ((char (char-after (point))))
	(cond ((or (eq char left-delimiter)
		   ;; For things other than lists
		   (eq (char-after (1- (point))) ?\n))
	       (setq sexp (1+ sexp))
	       (if (condition-case ()
		       (progn (forward-sexp) nil)
		     (error t))
		   (error "Extra %s" (char-to-string left-delimiter))))
	      ((eq char right-delimiter)
	       (error "Extra %s" (char-to-string right-delimiter)))
	      ((< (point) end) (forward-char))))
      (sde-skip-chars end))
    (goto-char old-point)
    sexp))


;; Originally from reposition-window.el by Michael Ernst & adapted for SDE.
;; Newsgroups: gnu.emacs.sources
;; From: mernst@@theory.lcs.mit.edu (Michael Ernst)
;; Subject: reposition-window.el (again) (corrected posting)
;; Date: 22 Feb 91 11:30:49

(defun sde-reposition-window (&optional arg)
  "Move current production & its preceding comments to beginning of window.
Further invocations move first line of production to the top of the window
or toggle the visibility of comments that precede it.
  Point is left unchanged unless prefix ARG is supplied.
  If the production is fully onscreen, it is moved to the top of the
window.  If it is partly offscreen, the window is scrolled to get the
definition (or as much as will fit) onscreen, unless point is in a comment
which is also partly offscreen, in which case the scrolling attempts to get
as much of the comment onscreen as possible.
  Initially `sde-reposition-window' attempts to make both the definition and
preceding comments visible.  Further invocations toggle the visibility of
the comment lines.
  If ARG is non-nil, point may move in order to make the whole production
visible (if only part could otherwise be made so), to make the production
line visible (if point is in code and it could not be made so, or if only
comments, including the first comment line, are visible), or to make the
first comment line visible (if point is in a comment)."
  (interactive "P")
  (let* ((here (point))
	 ;; change this name once I've gotten rid of references to ht.
	 ;; this is actually the number of the last screen line
	 (ht (- (window-height (selected-window)) 2))
	 (line (sde-repos-count-lines (window-start) (point)))
	 (comment-height
	  ;; The call to max deals with the case of cursor between defuns.
	  (max 0
	       (sde-repos-count-lines-signed
		;; the beginning of the preceding comment
		(save-excursion
		  (forward-char 1)
		  (sde-beginning-of-sp 2)
		  (sde-end-of-sp)
		  ;; Skip whitespace, newlines, and form feeds.
		  (re-search-forward "[^\\s \n\014]")
		  (backward-char 1)
		  (point))
		here)))
	 (defun-height 
	     (sde-repos-count-lines-signed
	      (save-excursion
		(sde-end-of-sp 1)	; so comments associate with following defuns
		(sde-beginning-of-sp 1)
		(point))
	      here))
	 ;; This must be positive, so don't use the signed version.
	 (defun-depth (sde-repos-count-lines here
					     (save-excursion
					       (sde-end-of-sp 1)
					       (point))))
	 (defun-line-onscreen-p
	     (and (<= defun-height line)
		  (<= (- line defun-height) ht))))
    (cond ((or (= comment-height line)
	       (and (= line ht)
		    (> comment-height line)
		    ;; if defun line offscreen, we should be in case 4
		    defun-line-onscreen-p))
	   ;; Either first comment line is at top of screen or (point at
	   ;; bottom of screen, defun line onscreen, and first comment line
	   ;; off top of screen).  That is, it looks like we just did
	   ;; recenter-definition, trying to fit as much of the comment
	   ;; onscreen as possible.  Put defun line at top of screen; that
	   ;; is, show as much code, and as few comments, as possible.

	   (if (and arg (> defun-depth (1+ ht)))
	       ;; Can't fit whole defun onscreen without moving point.
	       (progn (sde-end-of-sp) (sde-beginning-of-sp) (recenter 0))
	       (recenter (max defun-height 0))))

	  ((or (= defun-height line)
	       (= line 0)
	       (and (< line comment-height)
		    (< defun-height 0)))
	   ;; Defun line or cursor at top of screen, OR cursor in comment
	   ;; whose first line is offscreen.
	   ;; Avoid moving definition up even if defun runs offscreen;
	   ;; we care more about getting the comment onscreen.
	   
	   (cond ((= line ht)
		  ;; cursor on last screen line (and so in a comment)
		  (if arg (progn (sde-end-of-sp) (sde-beginning-of-sp)))
		  (recenter 0))
		 

		 ;; This condition, copied from case 4, may not be quite right
		 
		 ((and arg (< ht comment-height))
		  ;; Can't get first comment line onscreen.
		  ;; Go there and try again.
		  (forward-line (- comment-height))
		  (beginning-of-line)
		  (recenter 0))
		 (t
		  (recenter (min ht comment-height)))))

	  ((and (> (+ line defun-depth -1) ht)
		defun-line-onscreen-p)
	   ;; Defun runs off the bottom of the screen and the defun line
	   ;; is onscreen.
	   ;; Move the defun up.
	   (recenter (max 0 (1+ (- ht defun-depth)) defun-height)))

	  (t
	   ;; If on the bottom line and comment start is offscreen
	   ;; then just move all comments offscreen, or at least as
	   ;; far as they'll go.
	   ;; Try to get as much of the comments onscreen as possible.
	   (if (and arg (< ht comment-height))
	       ;; Can't get defun line onscreen; go there and try again.
	       (progn (forward-line (- defun-height))
		      (beginning-of-line)
		      (reposition-window))
	       (recenter (min ht comment-height)))))))


;; Return number of screen lines between START and END.

(defun sde-repos-count-lines (start end)
  (save-excursion
    (save-restriction
      (narrow-to-region start end)
      (goto-char (point-min))
      (vertical-motion (- (point-max) (point-min))))))


;; Return number of screen lines between START and END; returns a negative
;; number if END precedes START.

(defun sde-repos-count-lines-signed (start end)
  (let ((lines (sde-repos-count-lines start end)))
    (if (< start end)
	lines
	(- lines))))


;; Print information about the current file.
;; Originally based on functions from soar-mode v5.0.

(defun sde-region-count-productions (begin end)
  "Count the number of productions, lines & characters in the current region.
To count the productions in the whole buffer, first type `\\[mark-whole-buffer]'."
  (interactive "r")
  (if (interactive-p)
      (message "Counting soar productions..."))
  (let ((count 0)
	(lines (count-lines begin end)))
    (save-excursion
      (goto-char begin)
      (while (re-search-forward sde-sp-name-regexp end t)
	(setq count (1+ count))))
    (if (interactive-p)
	(message "%d production%s, %d line%s, %d character%s."
		 count (if (= count 1) "" "s")
		 lines (if (= lines 1) "" "s")
		 (- end begin) (if (= 1 (- end begin)) "" "s")))
    count))


;; Time/date stamps.  Based on some original code from a date.el that I
;; had written but never released.

(defconst sde-date-month-number
  '(("Jan" . 1) ("Feb" . 2) ("Mar" . 3) ("Apr" . 4) ("May" . 5) ("Jun" . 6)
    ("Jul" . 7) ("Aug" . 8) ("Sep" . 9) ("Oct" . 10) ("Nov" . 11) ("Dec" . 12))
  "Assoc list for looking up the month number from the month
abbreviation.")

(defun sde-insert-date-stamp (&optional arg)
  "Inserts the current date after point, in DD-MM-YY format.
With prefix argument, inserts the weekday first."
  (interactive)
  (let* ((s (current-time-string))
	 (day (substring s 0 3))
	 (month-name (substring s 4 7))
	 (date (if (equal ?\  (aref s 8)) ; Skip any leading space
		   (substring s 9 10)	;  in the day number.
		   (substring s 8 10)))
	 (year (substring s -4 nil)))
    (if current-prefix-arg
	(insert-before-markers (format "%s " day)))
    (insert-before-markers (format "%s-%s-%s" date month-name year))))


;;;----------------------------------------------------------------------------
;;; 17. Comment support
;;;----------------------------------------------------------------------------


;; Comment out regions.
;; Originally from ilisp-ext.el by Chris McConnell.

(defvar sde-comment-marker (make-marker) ; Declare global to avoid calling
  "Marker for end of a comment region.") ;  make-marker repeatedly.

(defun sde-region-comment (start end prefix)
  "If prefix is positive, insert prefix copies of `comment-start' at the
start and `comment-end' at the end of each line in region.  If prefix is
negative, remove all `comment-start' and `comment-end' strings from the
region."
  (interactive "r\np")
  (save-excursion
    (goto-char end)
    (if (and (/= start end) (bolp))
	(setq end (1- end)))
    (goto-char end)
    (beginning-of-line)
    (set-marker sde-comment-marker (point))
    (untabify start end)
    (goto-char start)
    (beginning-of-line)
    (let* ((count 1)
	   (comment comment-start)
	   (comment-end (if (not (equal comment-end "")) comment-end)))
      (if (> prefix 0)
	  (progn
	    (while (< count prefix)
	      (setq comment (concat comment-start comment)
		    count (1+ count)))
	    (while (<= (point) sde-comment-marker)
	      (beginning-of-line)
	      (insert comment)
	      (if comment-end (progn (end-of-line) (insert comment-end)))
	      (forward-line 1)))
	  (setq comment (concat comment "+"))
	  (while (<= (point) sde-comment-marker)
	    (back-to-indentation)
	    (if (looking-at comment) (replace-match ""))
	    (if comment-end
		(progn
		  (re-search-backward comment-end)
		  (replace-match "")))
	    (forward-line 1)))
      (set-marker sde-comment-marker nil))))


;;;----------------------------------------------------------------------------
;;; 19. Help support.
;;;----------------------------------------------------------------------------

;; None of these use a separate command history, because we want the same
;; symbols to be available from one command to another.  E.g., the user might
;; start off typing sde-apropos on something, then try sde-topic-help, and it's
;; desirable to be able to retrieve the item off the history list.  

(defvar sde-help-hist nil)


(defun sde-apropos (&optional regexp)
  (interactive (list (sde-read-string "Apropos: " 'sde-help-hist)))
  (with-output-to-temp-buffer "*Help*"
    (sde-apropos-print-matches
     (cond (sde-running-emacs19
	    (apropos-internal regexp 'sde-apropos-match))
	   (sde-running-emacs18
	    (apropos regexp 'sde-apropos-match t))))))


(defun sde-apropos-match (symbol)
  ;; Determine if given symbol is in sde-commands-obarray or
  ;; sde-variables-obarray and matches a given regexp.
  (or (intern-soft (symbol-name symbol) sde-commands-obarray)
      (intern-soft (symbol-name symbol) sde-variables-obarray)))


;; Following two functions very heavily borrowed from fast-apropos.el,
;; by Joe Wells <jbw@bigbird.bu.edu> (now a standard part of Emacs 19).

(defun sde-apropos-print-matches (matches &optional spacing)
  ;;  Helper function for fast-apropos and super-apropos.  Depends on
  ;;  standard-output having been rebound appropriately.  Prints the symbols
  ;;  and documentation in alist MATCHES of form ((symbol fn-doc var-doc) ...).
  ;;  Displays in the buffer pointed to by standard-output.  Optional argument
  ;;  SPACING means put blank lines in between each symbol's documentation.
  ;;  This should only be called within a with-output-to-temp-buffer.
  (sde-apropos-get-doc matches)
  (setq matches (sort matches (function
			       (lambda (a b)
				(string-lessp (car a) (car b))))))
  (let ((p matches)
	(old-buffer (current-buffer))
	item symbol str)
    (save-excursion
      (set-buffer standard-output)
      (or matches (princ "No matches found."))
      (while (consp p)
	(setq item (car p)
	      symbol (car item)
	      p (cdr p))
	(or (not spacing) (bobp) (terpri))
	(princ symbol)		        ; Print symbol name.
	(if (commandp symbol)
	    (let (keys)
	      (save-excursion
		(set-buffer old-buffer)
		(setq keys (sde-where-is symbol)))
	      (indent-to 30 1)
	      (princ (or keys "(not bound to any keys)"))))
	(terpri)
	(if (setq str (nth 1 item))
	    (progn
	      (princ "  Function: ")
	      (princ (substitute-command-keys str))))
	(or (bolp) (terpri))
	(if (setq str (nth 2 item))
	    (progn
	      (princ "  Variable: ")
	      (princ (substitute-command-keys str))))
	(or (bolp) (terpri))	
	(terpri))))
  t)


(defun sde-apropos-get-doc (sym-list)
  ;; Helper for sde-apropos.
  ;; Takes SYM-LIST of symbols and adds documentation.  Modifies SYM-LIST in
  ;; place.  Resulting alist is of form ((symbol fn-doc var-doc) ...).
  ;; Returns SYM-LIST.
  (let ((p sym-list)
	fn-doc var-doc symbol)
    (while (consp p)
      (setq symbol (car p)
	    fn-doc (if (fboundp symbol)
		       (documentation symbol))
	    var-doc (documentation-property symbol 'variable-documentation)
	    fn-doc (and fn-doc
			(substring fn-doc 0 (string-match "\n" fn-doc)))
	    var-doc (and var-doc
			 (substring var-doc 0 (string-match "\n" var-doc))))
      (setcar p (list symbol fn-doc var-doc))
      (setq p (cdr p)))
    sym-list))


;; Modified from bbdb-info from bdbb-com.el. 

(defvar Info-directory)			; D. Gillespie's info.el defines this.

(defun sde-display-info-file (file)
  (require 'info)
  (let ((Info-directory (and (boundp 'Info-directory) Info-directory))
	(try (expand-file-name file sde-directory)))
    (or (file-exists-p try)
	(setq try (concat file ".info")))
    (or (file-exists-p try)  
	(setq try (expand-file-name file Info-directory)))
    (or (file-exists-p try)
	(setq try (concat try ".info")))
    (or (file-exists-p try)
	(error "Info file %s doesn't exist" file))
    (let ((Info-directory (file-name-directory try)))
      (Info-goto-node (format "(%s)Top" try)))))


(defun sde-info ()
  "Run the Emacs Info system on the SDE documentation."
  (interactive)
  (sde-display-info-file sde-info-file))


(defun sde-news ()
  "Display news about SDE, including recent changes."
  (interactive)
  (sde-info)
  (Info-goto-node "News"))


(defun sde-soar-info ()
  "Run the Emacs Info system on the on-line Soar manual."
  (interactive)
  (sde-display-info-file sde-soar-info-file))


(defmacro sde-display-notes-file (file)
  ;; Helper macro. 
  (` (if (null (, file))
	 (sde-error-unset-site-var (quote (, file)))
	 (let ((buffer (create-file-buffer (, file))))
	   (save-excursion
	     (set-buffer buffer)
	     (erase-buffer)
	     (condition-case ()
		 (insert-file-contents (, file) t)
	       (file-error
		(error "Couldn't read file \"%s.\"" (, file))))
	     (setq buffer-read-only t))
	   (sde-pop-to-buffer buffer)))))


(defun sde-soar-release-notes ()
  "Read the most recent release notes for Soar.
This depends on the variable `sde-soar-release-notes-file' to be set properly." 
  (interactive)
  (sde-display-notes-file sde-soar-release-notes-file))


(defun sde-soar-user-notes ()
  "Read the most recent release notes for Soar.
This depends on the variable `sde-soar-user-notes-file' to be set properly." 
  (interactive)
  (sde-display-notes-file sde-soar-user-notes-file))


;; The topics obarray works like this.  Initially, when this file is loaded,
;; sde-topics-obarray is set to the union of sde-commands-obarray and
;; sde-variables-obarray.  (See bottom of this file.)  Whenever Soar is
;; restarted, the function sde-record-soar-help-topics is called; this
;; resets sde-topics-obarray to also include the help topics that Soar knows
;; about.  This means that the user won't have the full set of help topics
;; available until Soar is started at least once.

(defun sde-topic-help (topic)
  (interactive (list (sde-completing-read
		      "Topic (? = list): "
		      'sde-help-hist sde-topics-obarray)))
  (cond ((intern-soft topic sde-commands-obarray)
	 (sde-command-help topic))
	((intern-soft topic sde-variables-obarray)
	 (sde-variable-help topic))
	(t
	 (sde-soar-help topic))))


(defun sde-command-help (cmd-name)
  "Prompts for an SDE command name (with completion) and displays help about it.
This is almost identical to the Emacs `describe-function' command, except that
completion is limited to user commands in SDE.  If called from a program,
takes argument CMD-NAME, the string name of the variable."
  (interactive (list (sde-completing-read
		      "Command name (? = list): "
		      'sde-help-hist sde-commands-obarray)))
  (describe-function (car (read-from-string cmd-name))))


(defun sde-variable-help (var-name)
  "Prompts for an SDE variable name (with completion) and displays help about it.
This is almost identical to the Emacs `describe-variable' command, except that
completion is limited to user variables in SDE.  If called from a program,
takes argument VAR-NAME, the string name of the variable."
  (interactive (list (sde-completing-read
		      "Variable (? = list): "
		      'sde-help-hist sde-variables-obarray)))
  (describe-variable (car (read-from-string var-name))))


;; Following code partly taken from rp-describe-function.el, written by
;; Robert Potter, rpotter@grip.cis.upenn.edu.

(defun sde-where-is (cmd)
  "Prompts for an SDE command (with completion) and shows its key bindings.
This command differs from the regular Emacs `where-is' by the fact that 
it completes only on SDE commands and its output is more specific.  If
called from a Lisp program, returns a string of the form 
   \"locally on `keystroke1' and globally on `keystroke2'\"
or nil if the command has no key binding."
  (interactive (list (car (read-from-string
			   (sde-completing-read
			    "Where is command (? = list): "
			    'sde-help-hist sde-commands-obarray)))))
  ;; This finds local key bindings and global bindings separately
  ;; so that we can tell the user exactly where things are.
  (let* ((global-map (current-global-map))
	 (global (where-is-internal cmd))
	 (local (unwind-protect
		     (progn
		       (use-global-map (make-keymap))
		       (where-is-internal cmd (current-local-map)))
		  (use-global-map global-map)))
	 (text (if (or global local)
		   (format "%s%s%s"
			   (if local
			       (format "locally on `%s'"
				       (mapconcat 'key-description local "' and `"))
			       "")
			   (if (and global local)
			       " and "
			       "")
			   (if global
			       (format "globally on `%s'"
				       (mapconcat 'key-description global "' and `"))
			       "")))))
    (if (interactive-p)
	(if text
	    (message "%s is available %s" cmd text)
	    (message "%s has no key bindings." cmd)))
    text))
  

;; What follows is an ugly thing to do.  I wanted three features for the mode
;; help in SDE:
;;
;;   1) Print a message at beginning and end, to let user know that
;;	*something* is happening -- some mode docs are long and Emacs takes
;;	forever to format them.
;;	
;;   2) Cache the mode help string, to speed up help as much as possible.
;;
;;   3) Fix an inconsistency in Emacs 18 about how it prints the key
;;	binding information for (1) commands that have meta bindings and (2)
;;	commands that have no bindings.  In case (1), the mode help shows a
;;	binding of "ESC c" but in case (2), it shows "M-x foo" instead.
;;	Lucid 19 consistently uses "M-c", but FSF 19 (wouldn't you know it)
;;	uses both "ESC c" and "M-c" in describe-mode and "M-c" in the manual.
;;      I'm going to make an effort to point out this inconsistency to the
;;	GNU folks, and in the meantime, the functions below change everything
;;	to "M-c" to be consistent with the Emacs (and SDE) manuals.
;;
;; Now the problem is how to hook this into the describe-mode function.
;; Experience has shown that simply giving "\C-hm" a new binding in sde-mode-map
;; results in problems for people who must rebind C-h to act as delete
;; because their terminals are stupid.  (Creating a new binding for "\C-hm"
;; automatically introduces a new intermediate local map for "\C-h", which is
;; what causes the problems.  There is no way to create a binding only for
;; "\C-hm" without also creating a local map for C-h.)  So this code resorts
;; to resetting the function definition of describe-mode, a bad thing to do,
;; but if it's done carefully, maybe the repercussions will be small.

(defvar sde-describe-mode-text-alist nil
  "Association list of mode-names and their associated documentation strings,
storing the result of doing a sde-describe-mode for each mode.  The list
consists of dotted pairs of the form \(major-mode . documentation-string).")

(defvar sde-true-describe-mode (symbol-function 'describe-mode)
  "The original function definition of `describe-mode'.")

(fset 'sde-true-describe-mode sde-true-describe-mode)

(defun sde-describe-mode ()
  "Display documentation of current major mode."
  (interactive)
  (if (not (memq major-mode sde-modes))
      (sde-true-describe-mode)
      (with-output-to-temp-buffer "*Help*"
	(let ((mode-help (assoc major-mode sde-describe-mode-text-alist)))
	  (if mode-help
	      (princ (cdr mode-help))
	      (let (tmp case-fold-search case-replace)
		(message "Formatting documentation ...")
		(princ mode-name)
		(princ " Mode:\n")
		(princ (documentation major-mode))
		(save-excursion
		  (set-buffer "*Help*")
		  (sde-massage-key-descriptions)
		  (setq tmp (buffer-string)))
		(message "Formatting documentation ...done.")
		(sde-push (cons major-mode tmp) sde-describe-mode-text-alist))))
	(print-help-return-message))))

(fset 'describe-mode (symbol-function 'sde-describe-mode))


(defvar sde-massage-key-descriptions-list
  '(("\\bLFD\\b" . "C-j")
    ("\\bESC \\([][~a-zA-Z0-9@\;\\`!\"#$%^(){}*+='.,|/<>?-]\\|LFD\\)" . "M-\\1") 
    ("\\bM-C-\\([][~a-zA-Z0-9@\;\\`!\"#$%^(){}*+='.,|/<>?-]\\|LFD\\)" . "C-M-\\1")
    ;; SDE-specific things.
    ("\\bC-c \\? \\(C-.\\|\\?\\|TAB\\|RET\\|LFD\\)" . "C-c C-h \\1")
    ("\\bC-c\\( C-h\\| C-v\\) ?RET" . "C-c\\1 C-m")
    ("\\bC-c C-v LFD" . "C-c C-v C-j")
    ("\\bC-c C-h TAB" . "C-c C-h C-i")))


(defun sde-massage-key-descriptions ()
  ;; Fix inconsistencies in the output of describe-bindings.
  (save-excursion
    (let ((translations sde-massage-key-descriptions-list))
      (while translations
	(goto-char 1)
	(while (re-search-forward (car (car translations)) nil t)
	  (replace-match (cdr (car translations)) t))
	(setq translations (cdr translations))))))


(defun sde-describe-bindings (&optional prefix)
  "Show a list of all defined keys, and their definitions.
The list is put in a buffer, which is displayed.

In Emacs 19, when called from a Lisp program, can take an optional argument
PREFIX, which if non-nil, should be a key sequence; then we display only
bindings that start with that prefix."
  (interactive)
  (if sde-running-emacs19
      (describe-bindings prefix)
      (describe-bindings))
  (save-excursion
    (set-buffer "*Help*")
    (sde-massage-key-descriptions)))


;; Modified version of make-help-screen, originally by Lynn Slater, now
;; part of Emacs 19.  Modified so that keys such as C-v are not intercepted.
;; It's stupid to have to do this.

(defmacro sde-make-help-screen (fname help-line help-text helped-map)
  "Constructs function FNAME that when invoked shows HELP-LINE and if a help
character is requested, shows HELP-TEXT. The user is prompted for a character
from the HELPED-MAP and the corresponding interactive function is executed."
  (` (defun (, fname) ()
	   (, help-text)
	   (interactive)
	   (let ((line-prompt (substitute-command-keys (, help-line))))
	     (message line-prompt)
	     (let ((char (read-char))
		   config)
	       (unwind-protect
		   (progn
		     (if (or (char-equal char ??) (char-equal char help-char))
			 (progn
			   (setq config (current-window-configuration))
			   (switch-to-buffer-other-window "*Help*")
			   (erase-buffer)
			   (set-window-hscroll (selected-window) 0)
			   (insert (documentation (quote (, fname))))
			   (goto-char (point-min))
			   (while (memq char (cons help-char '(?? ?\ ?\177)))
			     (if (char-equal char ?\ ) 
				 (scroll-up))
			     (if (char-equal char ?\177)
				 (scroll-down))
			     (message "%s%s: "
				      line-prompt
				      (if (pos-visible-in-window-p (point-max))
					  "" " or Space to scroll"))
			     (let ((cursor-in-echo-area t))
			       (setq char (read-char))))))
		     (let ((defn (cdr (assq (if (integerp char)
						(downcase char)
						char)
					    (, helped-map)))))
		       (if defn
			   (if (keymapp defn)
			       (error "sorry, this command cannot be run from the help screen.  Start over.")
			     (if config
				 (progn
				   (set-window-configuration config)
				   (setq config nil)))
			     (call-interactively defn))
			 (if (listp char) ; Emacs 19
			     (setq unread-command-events
				   (cons char unread-command-events)
				   config nil)
			   (ding)))))
		 (if config
		     (set-window-configuration config))))))))


;;;----------------------------------------------------------------------------
;;; 18. SDE mode
;;;----------------------------------------------------------------------------

(if (not (keymapp sde-view-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-view-cmds)
      (define-key map "\C-h" 'sde-help-for-view-cmds)
      (define-key map "\C-w" 'sde-view-working-memory)
      (define-key map "\C-t" 'sde-view-ptraces)
      (define-key map "\C-s" 'sde-view-pgs)
      (define-key map "\C-p" 'sde-view-productions)
      (define-key map "\C-o" 'sde-view-pgso)
      (define-key map "\C-m" 'sde-view-ms)
      (define-key map "\C-j" 'sde-view-justifications)
      (define-key map "\C-c" 'sde-view-chunks)
      (define-key map "\C-b" 'sde-view-pbreaks)
      (setq sde-view-cmds-map map)))


(sde-make-help-screen sde-help-for-view-cmds
  "C-b, C-c, C-j, C-m, C-p, C-s, C-t, C-w. [C-h = more help]"
  "Commands for viewing various Soar data:

C-b  view pbreaks
C-c  view chunks
C-j  view justifications
C-m  view match set (Soar \"ms\")
C-o  view operator stack (Soar \"pgso\")
C-p  view productions
C-s  view goal stack (Soar \"pgs\")
C-t  view ptraces
C-w  view contents of Soar working memory

?    help on view commands
C-h  help on view commands
C-g  cancel"
  sde-view-cmds-map)


(if (not (keymapp sde-agent-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-agent-cmds)
      (define-key map "\C-h" 'sde-help-for-agent-cmds)
      (define-key map "\C-s" 'select-agent)
      (define-key map "\C-l" 'list-agents)
      (define-key map "\C-d" 'destroy-agent)
      (define-key map "\C-c" 'create-agent)
      (define-key map "\C-a" 'agent-go)
      (setq sde-agent-cmds-map map)))


(sde-make-help-screen sde-help-for-agent-cmds
  "C-a, C-c, C-d, C-l, C-s. [C-h = more help]"
  "Commands for managing multiple Soar agents:

C-a  agent \"go\" parameters
C-c  create agent
C-d  destroy agent
C-l  list agents known by SDE
C-s  select agent

?    help on agent commands
C-h  help on agent commands
C-g  cancel"
  sde-agent-cmds-map)


(if (not (keymapp sde-region-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-region-cmds)
      (define-key map "\C-h" 'sde-help-for-region-cmds)
      (define-key map "="    'sde-region-count-productions)
      (define-key map ";"    'sde-region-comment)
      (define-key map "\C-x" 'sde-region-excise)
      (define-key map "\C-t" 'sde-region-ptrace)
      (define-key map "\C-s" 'sde-region-send)
      (define-key map "\C-b" 'sde-region-pbreak)
      (setq sde-region-cmds-map map)))


(sde-make-help-screen sde-help-for-region-cmds
  "C-b, C-e, C-t, C-x, \;, =. [C-h = more help]"
  "Commands for operating on regions of Soar code:

C-b  pbreak all productions in region
C-s  send (i.e., load) region to Soar
C-t  ptrace all productions in region
C-x  excise all productions in region
;    comment or uncomment region
=    count productions, lines and characters in region

?    help on region commands
C-h  help on region commands
C-g  cancel"
  sde-region-cmds-map)


(if (not (keymapp sde-help-cmds-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "?"    'sde-help-for-help-cmds)
      (define-key map "\C-h" 'sde-help-for-help-cmds)
      (define-key map "\C-w" 'sde-where-is)
      (define-key map "\C-u" 'sde-soar-user-notes)
      (define-key map "\C-t" 'sde-topic-help)
      (define-key map "\C-s" 'sde-soar-info)
      (define-key map "\C-r" 'sde-soar-release-notes)
      (define-key map "\C-m" 'sde-describe-mode)
      (define-key map "\C-k" 'describe-key)
      (define-key map "\C-i" 'sde-info)
      (define-key map "\C-b" 'sde-describe-bindings)
      (define-key map "\C-a" 'sde-apropos)
      (setq sde-help-cmds-map map)))


(sde-make-help-screen sde-help-for-help-cmds
  "C-a, C-b, C-i, C-k, C-m, C-s, C-t, C-u, C-w. [C-h = more help]"
  "Help interface for SDE and Soar.  Press one of the following keys:

C-m  mode         -- Describe the current mode.
C-b  bindings     -- List all SDE key bindings for the current mode.
C-k  describe key -- Describe the SDE command invoked by a given key sequence.
C-w  where-is     -- Show the key that invokes a given SDE command.

C-a  apropos      -- Briefly list SDE commands and variables matching a string.
C-t  topic        -- Display help on any topic, SDE or Soar.

C-i  info         -- Read the SDE User's Manual using the Info system.

C-s  Soar manual  -- Read the on-line Soar User's Manual using the Info system.
C-r  rel. notes   -- Read the Release Notes for the most recent version of Soar
C-u  user notes   -- Read the User Notes for the most recent version of Soar.

?    help on help commands
C-h  help on help commands
C-g  cancel

The `topic' and `where-is' commands offer completion.  
The `topic' command knows about both Soar and SDE commands and variables, and
will query Soar for help if asked about a Soar topic."
  sde-help-cmds-map)


;; Certain keys are reserved for use in Soar Mode:
;; 
;; c-c c-w  backward-kill-word    	   Like ^w in shells
;; C-c C-o  sde-kill-output                Delete last batch of process output
;; C-c C-u  sde-kill-input                 Like ^u in shells
;; M-p      sde-previous-input             Cycle backwards in input history
;; M-n      sde-next-input                 Cycle forwards
;; M-s      sde-previous-similar-input     Previous similar input
;; M-C-r    sde-previous-input-matching    Search backwards in input history
;;
;; Two key bindings get rebound in Soar Mode to something entirely different:
;;
;; C-c C-r  sde-show-output
;; M-C-r    sde-previous-input-matching

;; The order of the definitions in the following code matter.  When
;; `describe-bindings' is invoked, it will print the items in
;; *reverse* order from the order of their definition.  So keys must
;; be defined in reverse alphabetical order to get them to be printed
;; in alphabetical order when `describe-bindings' and other functions
;; are invoked. In addition, some duplicate keys are purposefully
;; defined earlier than their alternative (for example `C-M-x' before
;; `C-c C-e') so that when `substitute-key-bindings' is run by Emacs
;; for various help functions, it gets the desired key binding (in
;; this case "C-c C-e") before the alternative, because the
;; alterative may not be as mnemonic.  Is that clear? :-)

(if (not (keymapp sde-mode-map))
    (let ((map (make-sparse-keymap)))
      (define-key map "\eq"      'sde-reindent)
      (define-key map "\ej"      'indent-new-comment-line)
      (define-key map "\e\;"     'indent-for-comment)
      (define-key map "\e\C-x"   'sde-send-production) ; Like eval-defun
      (define-key map "\e\C-u"   'sde-backward-up-list)
      (define-key map "\e\C-r"   'sde-reposition-window)
      (define-key map "\e\C-q"   'sde-indent-sexp)
      (define-key map "\e\C-p"   'sde-backward-list)
      (define-key map "\e\C-n"   'sde-forward-list)
      (define-key map "\e\C-k"   'sde-kill-sexp)
      (define-key map "\e\C-j"   'indent-new-comment-line)
      (define-key map "\e\C-h"   'sde-mark-sp)         ; Like mark-defun
      (define-key map "\e\C-f"   'sde-forward-sexp)
      (define-key map "\e\C-e"   'sde-end-of-sp)
      (define-key map "\e\C-b"   'sde-backward-sexp)
      (define-key map "\e\C-a"   'sde-beginning-of-sp)

      (define-key map "\C-c?"    sde-help-cmds-map)
      (define-key map "\C-c\C-z" 'sde-switch-to-soar)
      (define-key map "\C-c\C-y" 'wm)
      (define-key map "\C-c\C-x" 'excise)
      (define-key map "\C-c\C-v" sde-view-cmds-map)
      (define-key map "\C-c\C-t" 'ptrace)
      (define-key map "\C-c\C-s" 'sde-send-production)
      (define-key map "\C-c\C-r" sde-region-cmds-map)
      (define-key map "\C-c\C-q" 'firing-counts)
      (define-key map "\C-c\C-p" 'print-soar)
      (define-key map "\C-c\C-n" 'sde-find-next-production)
      (define-key map "\C-c\C-l" 'load-soar)
      (define-key map "\C-c\C-h" sde-help-cmds-map)
      (define-key map "\C-c\C-f" 'sde-find-production)
      (define-key map "\C-c\C-e" 'explain)
      (define-key map "\C-c\C-c" 'sde-interrupt-soar)
      (define-key map "\C-c\C-b" 'pbreak)
      (define-key map "\C-c\C-a" sde-agent-cmds-map)
      (define-key map "\C-c1"    'sde-hide-soar-error-window)
      (define-key map "\C-c0"    'init-soar)
      (define-key map "\C-c+"    'sde-insert-date-stamp)
      (define-key map "\C-c)"    'sde-find-unbalanced)
      (define-key map "\C-c]"    'sde-close-all-sp)
      (define-key map "\C-c\;"   'sde-region-comment)
      (define-key map "\C-c/"    'preferences)
      (define-key map "\C-c'"    'matches-1)
      (define-key map "\C-c\""   'matches-2)
      (define-key map "\C-c,"    'schedule)
      (define-key map "\C-c."    'go)

      (define-key map "\C-m"     'sde-newline-and-indent)
      (define-key map "\C-j"     'sde-newline-and-indent)
      (define-key map "\t"       'sde-indent-line)
      (define-key map "\177"	 'backward-delete-char-untabify)
      (setq sde-mode-map map)))


;; Syntax table.

(if (not (syntax-table-p sde-mode-syntax-table))
    (let ((table (copy-syntax-table lisp-mode-syntax-table)))
      (modify-syntax-entry ?\t " "  table)
      (modify-syntax-entry ?\f " "  table)
      ;; Need newline to be comment-end for some of the functions that look
      ;; for productions components to work properly.  Unfortunately, this
      ;; causes "\\s " regexp's to fail to consider newline to be whitespace!
      (modify-syntax-entry ?\n ">"  table)
      (modify-syntax-entry ?   " "  table)
      (modify-syntax-entry ?~  " "  table)
      (modify-syntax-entry ?!  " "  table)
      (modify-syntax-entry ?@  " "  table)
      (modify-syntax-entry ?#  " "  table)
      (modify-syntax-entry ?$  "_"  table) 
      (modify-syntax-entry ?%  "_"  table) 
      (modify-syntax-entry ?^  "_"  table)
      (modify-syntax-entry ?&  "_"  table)
      (modify-syntax-entry ?*  "_"  table)
      (modify-syntax-entry ?\( "()" table)
      (modify-syntax-entry ?\) ")(" table)
      (modify-syntax-entry ?-  "_"  table) 
      (modify-syntax-entry ?_  "_"  table) 
      (modify-syntax-entry ?=  "_"  table)
      (modify-syntax-entry ?+  "_"  table) 
      (modify-syntax-entry ?\{ "(}" table)
      (modify-syntax-entry ?\} "){" table)
      (modify-syntax-entry ?\[ "(]" table)
      (modify-syntax-entry ?\] ")[" table)
      (modify-syntax-entry ?\| "$|" table)
      (modify-syntax-entry ?\  " "  table) 
      (modify-syntax-entry ?\; "<"  table)
      (modify-syntax-entry ?:  "_"  table)
      (modify-syntax-entry ?'  "'"  table)
      (modify-syntax-entry ?\" "\"" table)
      (modify-syntax-entry ?.  " "  table)
      (modify-syntax-entry ?,  " "  table)
      ;; We can't make '<' and '>' be opening and closing delimiters because
      ;; '>' is used in the production arrow, and Emacs would get confused by
      ;; the right angle bracket in "-->".  Too bad.
      (modify-syntax-entry ?\< "'"  table)
      (modify-syntax-entry ?\> "'"  table)
      (modify-syntax-entry ?/  "_"  table)
      (modify-syntax-entry ??  "_"  table) 
      (setq sde-mode-syntax-table table)))

;; Abbrev table.

(if (null sde-mode-abbrev-table)
    (define-abbrev-table 'sde-mode-abbrev-table ()))


;; The actual SDE mode.
;;
;; Note that the key bindings for next and previous history cannot be written
;; as \\[gmhist-previous], etc., because we don't use gmhist in GNU Emacs 19.
;; I had to hardwire the descriptions as "ESC p" and "ESC n".  Some other key
;; descriptions are hardwired too, mainly because FSF 19 seems to catch the
;; menu definition instead of the key binding first, and consequently prints
;; `menu foo-bar biff-wack', which is ugly and almost useless.

(defun sde-mode ()
  "The Soar Development Environment (SDE), a major mode for writing Soar programs.
\\<sde-mode-map>
`\\[sde-beginning-of-sp]' moves the cursor to the beginning of a production.
`\\[sde-end-of-sp]' moves the cursor to the end of a production.

Commands that move the cursor by words, such as `\\[forward-word]' (forward-word), 
and `\\[backward-word]' (backward-word), consider whitespace and the characters
'^', '*', '-', '_', '<', '>' and '.' to be word delimiters.  In addition, such
commands ignore parentheses, curly braces and whitespace.  Commands that move
the cursor by balanced expressions, such as `\\[sde-forward-sexp]' (forward-sexp) and
`\\[sde-backward-sexp]' (backward-sexp), consider parentheses and curly braces to
be parts of expressions.

`M-x soar' will start Soar in a new buffer.  A prefix argument (such as
\\[universal-argument]) will make it prompt for the name of the Soar program, the
starting directory, and optional command line arguments to pass to Soar.

`\\[sde-send-production]' will send (that is, load) the current production
under the cursor to the current Soar process.  With a prefix argument, it
also switches to Soar after sending the production.

`\\[load-soar]' will prompt for a file name and load it into Soar\; you may use
this to load the current buffer's file into Soar as well as other files.
`M-x load-defaults' will load the most current version of the Soar defaults
file.  `M-x excise-file' will prompt for a file name and will excise all of
the productions found in that file.

The region commands perform common operations on a region in the buffer.
`\\[sde-region-send]' sends everything within the current region to Soar.
`\\[sde-region-pbreak]' issues a pbreak for every production in the region.
`\\[sde-region-ptrace]' issues a ptrace for every production in the region.
`\\[sde-region-excise]' excises every production in the region.

If Soar is running in multi-agent mode, and the current buffer is not
associated with any agent, the first command that needs a target agent will
prompt for an agent name and will set the buffer's default to the answer.
To change default agents, or to set the initial agent explicitly, use the
select-agent command (`\\[select-agent]').

`\\[sde-reindent]' reindents whatever the cursor is in.  If the cursor is in a comment,
it reindents the comment.  If in a production, it reindents the whole
production, closing it first if necessary.  If the cursor is in the
documentation string of a production, it reindents the documentation string.
`\\[sde-indent-line]' indents just the current line.  `RET' moves to the next
line and automatically indentations appropriately.  Variable
`sde-production-indent-offset' controls how far clauses are indented underneath
the first line of the production\; the default is 2 spaces.  Variable
`sde-arrow-indent-offset' determines whether the arrow is indented differently\;
negative offsets move it leftward, positive offsets indent the arrow further.

All interactive Soar commands are available via `M-x COMMANDNAME'.  Some
commands in SDE have no Soar equivalents, e.g., `excise-file'.  Many key
bindings give direct access to the most common commands.  Some notable
bindings include the following, and a complete list appears later below.  
Use describe-key (\\<global-map>`\\[describe-key]'\\<sde-mode-map>) \(the standard Emacs help facility) to
get detailed help on each command.

  Running and Initializing Soar:

    go                      \\[go]
    schedule                \\[schedule]
    interrupt Soar          \\[sde-interrupt-soar]
    init-soar               \\[init-soar]

  Manipulating Productions:

    send production         \\[sde-send-production]
    send region             \\[sde-region-send]
    load file               \\[load-soar]
    excise production       \\[excise]
    excise region           \\[sde-region-excise]
    
  Tracing and breaking:

    ptrace production       \\[ptrace]
    ptrace region           \\[sde-region-ptrace]
    pbreak production       \\[pbreak]
    pbreak region           \\[sde-region-pbreak]

  Querying Soar for information about specific objects:

    print                   \\[print-soar]
    explain                 \\[explain]
    matches 1               \\[matches-1]
    matches 2               \\[matches-2]
    preferences             \\[preferences]
    firing-counts           \\[firing-counts]
    wm                      \\[wm]

  Querying Soar for general information:

    view ptraces            \\[sde-view-ptraces]
    view pbreaks            \\[sde-view-pbreaks]
    view goal stack (pgs)   \\[sde-view-pgs]
    view op. stack (pgso)   \\[sde-view-pgso]
    view match set (ms)     \\[sde-view-ms]
    view chunks             \\[sde-view-chunks]
    view productions        \\[sde-view-productions]
    view justifications     \\[sde-view-justifications]
    view Soar WM            \\[sde-view-working-memory]

  Finding source code:

    find production source  \\[sde-find-production]
    find next source        \\[sde-find-next-production]

  Controlling multiple agents:
  
    agent-go                \\[agent-go]
    create-agent            \\[create-agent]
    destroy-agent           \\[destroy-agent] (Unavailable prior to Soar 6.1.1)
    list agents             \\[list-agents]
    select-agent            \\[select-agent]

When running in the X Window System, the following mouse buttons are defined:

  Button 	    	      Action
  ------------------    -------------------------------------------------
  SHIFT-left            Execute `print' on item under the cursor
  SHIFT-middle          Execute `matches' on item under the cursor
  SHIFT-right           Execute `preferences' on attribute under the cursor

  META-CONTROL-left     Send to Soar the production under the cursor
  META-CONTROL-middle   Find the file containing the production under cursor
  META-CONTROL-right    Execute `explain' on chunk condition under the cursor

Many Soar commands, such as print, require \"target\" arguments, i.e.,
something to act on.  Those commands that need a production name as target
will use the name of the production that the cursor is in currently, or else
the name of the preceeding sp form.  Those commands that can take any symbol
as target will try to use the symbol under the cursor regardless of whether
it looks like a production name.

Many of the standard Soar commands also take optional arguments -- for
example, the go and print commands.  Giving a prefix argument (such as `\\[universal-argument]')
to most commands makes them prompt for optional arguments.  Optional
arguments are remembered on history lists.  Typing `M-p' and `M-n' at a 
prompt moves backward and forward through the history.

Optional arguments are automatically reused upon subsequent invocations of a
command until they are changed.  This allows you to set an option and then
apply a command repeatedly to different target arguments without having to
respecify the optional arguments each time.  For example, you can type
`\\[universal-argument] \\[print-soar]', type \":depth 3\" to the resulting prompt for optional
arguments, and then repeat this command with the same arguments using
just `\\[print-soar]' on different targets.

The commands ptrace, pbreak, preferences, and explain have different
conventions.  Without arguments, ptrace (`\\[ptrace]') and pbreak (`\\[pbreak]')
act on the production under the cursor.  With a positive argument (such as
`\\[universal-argument]') they perform unptrace or unpbreak, respectively, on the production
under the cursor.  With a negative argument (e.g., `\\[negative-argument]'), they undo all 
ptraces or pbreaks, respectively.  You can print out the current list of
active ptraces and pbreaks using the view commands `\\[sde-view-ptraces]' and 
`\\[sde-view-pbreaks]', respectively.

The preferences command (`\\[preferences]') attempts to extract its arguments
using heuristics.  Place the cursor over the attribube in a printed oject,
for example on \"^io-state\" in the WME
                    (8: S1 ^io-state S1)
and preferences will automatically use the attribute and the WME's identifier 
as arguments.  If the cursor is not over an attribute, it will attempt to
extract the symbol nearest the cursor, and if it is an identifier, then 
preferences will prompt for the attribute.  In addition, if preferences is
given a prefix argument (`\\[universal-argument]'), it will prompt for the optional
\"level\" argument.

The explain command (`\\[explain]') also attempts to extract its arguments
using heuristics.  Remember that before explain may be used, Soar must be
instructed to track the necessary information using \"explain :on\" \(or the
SDE equivalent `M-x explain-on').  The easiest way to use explain in SDE
is to print out a chunk and then place the cursor over an attribute in
the condition side of the chunk.  Explain will then query Soar with the
name of the chunk and the condition that tests that attribute.  \(It does
this by constructing a condition clause consisting of the appropriate
identifier-attribute-value slot, and silently querying Soar to determine
the correct condition number, then finally sending the command 
\"explain <production-name> <condition-number>\" to Soar.\)  You may also
put the cursor over a chunk name and invoke explain, and SDE will extract
just the production name and issue \"explain <name>\" to Soar.  Finally,
if you are looking at the text of a previous \"explain <name>\" and you
wish to do \"explain <name> <condition>\", simply put the cursor on the
line containing the condition number and invoke explain, and SDE will issue
\"explain <production-name> <condition-number>\" to Soar.  If given a prefix
argument, explain will prompt for its arguments, offering as initial defaults
any production name and condition clause it can find.

Commands that only query Soar for information place the output in in a
separate buffer for viewing, unless the variable `sde-soar-use-output-buffer'
is nil, in which case all output goes to the agent process buffer.
New output is appended to the output buffer, each batch separated by a
title line, unless the variable `sde-soar-erase-output-buffer' is non-nil,
in which case the output buffer's existing contents will be cleared before
new output is written out.

When running in Emacs 19 or Lucid Emacs 19, SDE will attempt to use a
separate X Window screen (called a \"frame\" in Emacs 19) for the output 
buffer, unless the variable `sde-use-multiple-frames' is nil.  The
initial attributes of this buffer are determined by the list of values in
the variable `sde-soar-output-buffer-defaults'.  Use Emacs help on this
variable (`\\[describe-variable]') for more information about how to use
this feature.  

The find-production facility locates the source code for a production
whose name is under the cursor or given as argument.  Put the cursor on a
production in your task, either in a Soar process buffer or an editing buffer,
type `\\[sde-find-production]', and it will jump to the place in your files where the
production is defined.  With a prefix argument, it will prompt for a string
instead of using the production name under the cursor. You don't need to have
created a tags file, you don't even need to have visited the file in question
yet.  It uses several heuristics to locate productions: (1) search through
existing Emacs buffers that are Soar source code buffers; (2) if fail, and a
tag file exists, do tags search; (3) if fail, search through files known to
have been loaded into Soar (it watches Soar's output for signs of file
loading); (4) if fail, search through files found in directories of Soar
files that have been loaded or visited so far.  In addition,
`sde-find-production' caches information about the location of productions that
it finds, so that subsequent finds on the same productions are much faster.
`\\[sde-find-next-production]' goes to the next occurrence of the last searched-for production
name string in your files.

The SDE on-line help facility is available on the `C-c C-h' prefix.
`\\[sde-apropos]' (apropos) prompts for a string and lists a brief description of
all SDE user commands and user variables that contain the given string.
`\\[sde-topic-help]' (topic) prompts for a command, variable or Soar help topic, and 
displays information about it.  This command interfaces to the Soar help 
facility, and will query Soar for help on topics that are not already defined
in SDE.  Note that `\\[sde-topic-help]' provides completion, which means you
can type `?' any time at the prompt to list possible completions of what you 
have typed so far, and you can type `TAB' to have Emacs complete the rest of
what you have typed based on known help topics.

`\\[sde-soar-info]' will bring up the Soar 6 User's manual using the Emacs
Info browser.  `C-c C-h C-i' will bring up the SDE manual.  `\\[sde-soar-release-notes]' 
will bring up the Release Notes for the latest revision of Soar, while 
`\\[sde-soar-user-notes]' will bring up the User Notes.  `\\[sde-describe-bindings]' will list the 
local bindings in the current SDE mode, and `\\[sde-where-is]' will prompt for
an SDE command name (with completion) and report which key binding
invokes it.

`\\[sde-close-all-sp]' will close the current sp form by adding or removing right parentheses
as necessary.

`\\[sde-region-count-productions]' will count the number of productions, lines
and characters the mark and the point, and report the answer.

`\\[sde-region-comment]' comments out the region between mark and point.  If given a positive
prefix arg, it inserts that many semicolons at the beginning of each line
\(defaulting to 1).  If given a negative prefix argument, it uncomments
the current region.

`\\[backward-delete-char-untabify]' converts tabs to spaces as it moves back.

`\\[sde-reposition-window]' will make the current production and/or comment visible.
If the production is fully onscreen, it is moved to the top of the window.
If it is partly offscreen, the window is scrolled to get the definition (or
as much as will fit) onscreen, unless point is in a comment which is also
partly offscreen, in which case the scrolling attempts to get as much of the
comment onscreen as possible.  Repeated invocations of `\\[sde-reposition-window]' move the
production to the top of the window or toggle the visibility of comments that
precede it.

`\\[sde-find-unbalanced]' searches for unbalanced parentheses in the current buffer.

`\\[sde-switch-to-soar]' will switch you to the Soar process buffer.

To send a bug report, questions or other feedback to the authors and 
maintainers of the Soar Development Environment, please use `\\[sde-feedback]'.

Entry to this mode will run the functions on the list variable sde-mode-hook.

Here is a list of all the special bindings in Soar mode.  The name of the key
map is `sde-mode-map'.  For further help on individual commands, type
\\<global-map>`\\[describe-key] KEY' where KEY is the keystroke.

\\<sde-mode-map>\\{sde-mode-map}"

  (interactive)
  (sde-mode-internal)
  ;; Internal bookeeping for SDE facilities.
  (sde-record-file buffer-file-name)
  ;; !!! This is a hack that must be removed when the header code
  ;; is cleaned up.  This is to enable users to set sde-header-hooks
  ;; now, even though header.el uses "make-header-hooks".
  (make-local-variable 'make-header-hooks)
  (if (boundp 'sde-header-hooks)
      (setq make-header-hooks sde-header-hooks))
  (run-hooks 'sde-mode-hook))


(defun sde-mode-internal ()
  "Function for performing initializations common to derivatives of SDE mode.
It does almost everything that sde-mode does, except for running hooks and
certain other initializations that only make sense for file buffers."
  ;; Set standard Emacs variables.
  (setq mode-name "SDE"
	major-mode 'sde-mode)
  (use-local-map sde-mode-map)
  ;; Build menus if appropriate.  Although in FSF 19 we could build menus at
  ;; load time, in Lucid the menus have to be defined for each buffer, so
  ;; it's easiest just to let a function handle it for both cases.
  (if (and window-system (or sde-running-emacs19 sde-running-lemacs))
      (sde-define-menus 'sde-mode sde-mode-map))
  (if sde-running-epoch
      (use-local-mouse-map sde-mouse-map))
  (set-syntax-table sde-mode-syntax-table)
  (setq local-abbrev-table sde-mode-abbrev-table)

  (make-local-variable 'comment-column)
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-start-skip)
  (if sde-running-emacs19
      (make-local-variable 'comment-indent-function)
      (make-local-variable 'comment-indent-hook))
  (make-local-variable 'indent-line-function)
  (make-local-variable 'paragraph-start)
  (make-local-variable 'paragraph-separate)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq comment-column       40
	comment-start        ";"
	comment-start-skip   ";+ *"
	indent-line-function 'sde-indent-line
	paragraph-start      "^(sp\\|^;+\\|^[ \t\f\n]*$"
	paragraph-separate   "^;+\\|^[ \t\f\n]*$"
	parse-sexp-ignore-comments t)	; See comments near sde-backward-sexp.
  (if sde-running-emacs19
      (setq comment-indent-function  'lisp-comment-indent)
      (setq comment-indent-hook  'lisp-comment-indent))
  (if (or sde-running-lemacs sde-running-emacs19)
      (progn
	(make-local-variable 'lisp-indent-function)
	(setq lisp-indent-function 'sde-indent-hook))
      (progn
	(make-local-variable 'lisp-indent-hook)
	(setq lisp-indent-hook 'sde-indent-hook)))

  ;; SDE Soar Mode variables
  (make-local-variable 'sde-soar-buffer-agent)
  (setq sde-soar-buffer-agent nil)

  ;; Set up the mode line.
  (setq	mode-line-modified '("-%1*%1* ")
	mode-line-buffer-identification '("%24b")
	mode-line-format
	(list "" 'mode-line-modified 'mode-line-buffer-identification
	      "  {"
	      '(sde-soar-agents
		;; Multi-agent case
		((sde-soar-buffer-agent sde-soar-buffer-agent "no agent")
		 (sde-soar-buffer-agent ": ")
		 (sde-soar-buffer-agent sde-soar-state-string))
		;; Single-agent case
		sde-soar-state-string)
	      "}  "
	      'global-mode-string
	      " %[("
	      'mode-name 'minor-mode-alist
	      ")%n%] --"
	      (if (or sde-running-emacs19 sde-running-lemacs)
		  '(line-number-mode "L%l--")
		  "")
	      '(-3 . "%p") "-%-")))


;;;----------------------------------------------------------------------------
;;; 20. Mouse support under the X Window System.
;;;----------------------------------------------------------------------------
;;;
;;; Because Emacs 18, Epoch 4.2 and Emacs 19 handle the mouse in different
;;; ways, we must implement different variants of the same operations.  An
;;; important thing to watch out for in the code below is the fset's that
;;; define basic functions used throughout the rest of the code.  
;;;
;;; Note the use of (interactive "@e") in the following functions, a
;;; specification which is illegal in regular Emacs and Epoch.  The sneaky
;;; trick here is that Regular Emacs and Epoch don't call the mouse functions
;;; interactively, hence the interactive declaration is ignored.  But in Lucid
;;; Emacs, which needs (interactive "@e"), the declaration is used.


(defconst sde-mouseable-modes
    (append sde-source-modes sde-soar-modes)
  "List of SDE modes in which mouse actions are permissible.")

;; In Epoch, mouse event dispatching uses two tables, one global and the
;; other local to each buffer.  When a mouse event is dispatched, the local
;; table is checked first.  If the table is missing or the entry is `nil',
;; then the global table is used.  If a function is found, then it is
;; dispatched.  The argument passed to a mouse function is a list of the form
;;    (point buffer window screen)
;; These all refer to the character location at which the button on the mouse
;; was pressed or released.  At the time the handler is called, no change to
;; point, current buffer, current window or current screen has been made.  It
;; is entirely up to the handler how much to change such information.

(defun sde-mouse-in-sde-buffer-epoch (args)
  ;; Return non-nil if the mouse is over an SDE buffer of some sort.
  ;; Epoch 4.2 version.  2nd element of args list is the buffer over 
  ;; which the mouse was positioned.
  (memq (sde-buffer-mode (car (cdr args))) sde-mouseable-modes))


(defun sde-mouse-in-sde-buffer-lemacs (event)
  ;; Return non-nil if the mouse is over an SDE buffer of some sort.
  ;; Lucid Emacs 19 version.
  (interactive "@e")
  (save-excursion
    (set-buffer (window-buffer (selected-window)))
    (memq major-mode sde-mouseable-modes)))


(defun sde-mouse-in-sde-buffer-emacs18 (args)
  ;; Return non-nil if the mouse is over an SDE buffer of some sort.
  (let* ((first-window (selected-window))
	 (window (next-window first-window))
	 found)
    (while (not (or (setq found (coordinates-in-window-p args window))
		    (eq window first-window)))
      (setq window (next-window window)))
    (if found
	(memq (sde-buffer-mode (window-buffer window)) sde-mouseable-modes))))


;; Define the function names actually used.

(if window-system
    (cond (sde-running-epoch		; Epoch 4.2
	   (fset 'sde-mouse-in-sde-buffer 'sde-mouse-in-sde-buffer-epoch)
	   (fset 'sde-mouse-set-point     'mouse::set-point))

	  ((or sde-running-lemacs sde-running-emacs19)
	   (fset 'sde-mouse-in-sde-buffer 'sde-mouse-in-sde-buffer-lemacs)
	   (fset 'sde-mouse-set-point     'mouse-set-point))

	  (t				; Regular Emacs 18
	   (fset 'sde-mouse-in-sde-buffer 'sde-mouse-in-sde-buffer-emacs18)
	   (fset 'sde-mouse-set-point     'x-mouse-set-point))))


;;;
;;; Now the real mouse handling code.
;;;

(defmacro sde-with-mouse-click (arg &rest body)
  (` (if (sde-mouse-in-sde-buffer (, arg))
	 (progn
	   (require 'sde-soar-mode)
	   (sde-mouse-set-point (, arg))
	   (,@ body)))))


(defun sde-mouse-print-soar (args)
  "Execute Soar \"print\" on the item under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'print-soar)))


(defun sde-mouse-matches (args)
  "Execute Soar \"matches\" on the item under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'matches)))


(defun sde-mouse-preferences (args)
  "Execute Soar \"preferences\" on the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'preferences)))


(defun sde-mouse-find-production (args)
  "Execute `sde-find-production' on the production name under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (call-interactively 'sde-find-production)))


(defun sde-mouse-send-production (args)
  "Send (load) the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'sde-send-production)))


(defun sde-mouse-ptrace-production (args)
  "Execute Soar \"ptrace\" on the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'ptrace)))


(defun sde-mouse-pbreak-production (args)
  "Execute SDE's \"pbreak\" on the production under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'pbreak)))


(defun sde-mouse-explain (args)
  "Execute SDE's \"explain\" on the chunk condition under the cursor."
  (interactive "@e")
  (sde-with-mouse-click args
    (sde-check-soar)
    (call-interactively 'explain)))


;;;
;;; Now define the bindings.
;;;

;; More defvar's to shut up the byte compiler.  These are used in different
;; versions of Emacs.

(defvar mouse::global-map)
(defvar mouse-left)
(defvar mouse-shift)
(defvar mouse-middle)
(defvar mouse-right)
(defvar mouse-meta-control)
(defvar mouse-map)
(defvar x-button-s-left)
(defvar x-button-s-middle)
(defvar x-button-s-right)
(defvar x-button-c-m-left)
(defvar x-button-c-m-middle)
(defvar x-button-c-m-right)
(defvar x-button-s-left-up)
(defvar x-button-s-middle-up)
(defvar x-button-s-right-up)
(defvar x-button-c-m-left-up)
(defvar x-button-c-m-middle-up)
(defvar x-button-c-m-right-up)

(if window-system
    (cond (sde-running-epoch
	   (setq sde-mouse-map (create-mouse-map mouse::global-map))
	   ;; Use the down events.
	   (define-mouse sde-mouse-map mouse-left   mouse-shift        'sde-mouse-print-soar)
	   (define-mouse sde-mouse-map mouse-middle mouse-shift        'sde-mouse-matches)
	   (define-mouse sde-mouse-map mouse-right  mouse-shift        'sde-mouse-preferences)
	   (define-mouse sde-mouse-map mouse-left   mouse-meta-control 'sde-mouse-send-production)
	   (define-mouse sde-mouse-map mouse-middle mouse-meta-control 'sde-mouse-find-production)
	   (define-mouse sde-mouse-map mouse-right  mouse-meta-control 'sde-mouse-explain))

	  (sde-running-lemacs
	   ;; Use the down events.  Lucid Emacs 19.6 only defines actions for the
	   ;; down events of:  button1, (shift button1), (control button1), and
	   ;; (control shift button1).
	   (define-key sde-mode-map '(shift button1)        'sde-mouse-print-soar)
	   (define-key sde-mode-map '(shift button2)        'sde-mouse-matches)
	   (define-key sde-mode-map '(shift button3)        'sde-mouse-preferences)
	   (define-key sde-mode-map '(control meta button1) 'sde-mouse-send-production)
	   (define-key sde-mode-map '(control meta button2) 'sde-mouse-find-production)
	   (define-key sde-mode-map '(control meta button3) 'sde-mouse-explain))

	  (sde-running-emacs19
	   (define-key sde-mode-map [S-mouse-1] 'sde-mouse-print-soar)
	   (define-key sde-mode-map [S-mouse-2] 'sde-mouse-matches)
	   (define-key sde-mode-map [S-mouse-3] 'sde-mouse-preferences)
	   (define-key sde-mode-map [C-M-mouse-1] 'sde-mouse-send-production)
	   (define-key sde-mode-map [C-M-mouse-2] 'sde-mouse-find-production)
	   (define-key sde-mode-map [C-M-mouse-3] 'sde-mouse-explain))

	  (sde-running-emacs18
	   ;; I'm no longer sure why, but here we needed to use the up events.
	   ;; Erase down map events to avoid interference with existing bindings.
	   (define-key mouse-map x-button-s-left 'x-mouse-ignore)
	   (define-key mouse-map x-button-s-middle 'x-mouse-ignore)
	   (define-key mouse-map x-button-s-right 'x-mouse-ignore)
	   (define-key mouse-map x-button-c-m-left 'x-mouse-ignore)
	   (define-key mouse-map x-button-c-m-middle 'x-mouse-ignore)
	   (define-key mouse-map x-button-c-m-right 'x-mouse-ignore)
       
	   (define-key mouse-map x-button-s-left-up     'sde-mouse-print-soar)
	   (define-key mouse-map x-button-s-middle-up   'sde-mouse-matches)
	   (define-key mouse-map x-button-s-right-up    'sde-mouse-preferences)
	   (define-key mouse-map x-button-c-m-left-up   'sde-mouse-send-production)
	   (define-key mouse-map x-button-c-m-middle-up 'sde-mouse-find-production)
	   (define-key mouse-map x-button-c-m-right-up  'sde-mouse-explain))))


;;;----------------------------------------------------------------------------
;;; 21. Font and color support.
;;;----------------------------------------------------------------------------

(if (and window-system (or sde-running-lemacs sde-running-emacs19))
    (require 'sde-highlight))


;;;----------------------------------------------------------------------------
;;; 21. Load and set-up commands for SDE
;;;----------------------------------------------------------------------------

;;; THIS MUST ALWAYS BE UPDATED TO REFLECT CURRENT SET OF COMMANDS AND VARIABLES.

;;; The following list serves two main purposes:
;;;   1) set up autoloading of other functions
;;;   2) create completion tables, for user input completion during help, etc.
;;;
;;; Using an explicit list like this is not ideal, because it means it must
;;; be updated whenever the associated files are modified.  Unfortunately I
;;; cannot find a better way.  We need some of the symbols to be available
;;; all the time for input completion, but their definitions may be located
;;; in separate files, and thus may not be loaded until later.  This rules
;;; out, for example, putting lists right with the individual files and
;;; concatenating them when the files are loaded, or defining new
;;; defun/defvar functions that automatically update the completion tables.
;;; We're left with a bit of a maintenance headache.  Is it worth it?  Don't
;;; know yet.

(defvar sde-symbols
  ;; Lists of lists.  File name, then triples of (symbol autoload help) 
  ;; The autoload field can only be t for functions, and
  ;; the completion field is one of: function, variable, or nil.

  '(("sde-compat"
     ;; Symbol			    Autoload?  Completion?
     (sde-compat-auto-update		nil	variable)
     (production-editor-mode		t	function)
     (soar-mode				t	function))

    ("sde-feedback"
     (sde-feedback-cc			nil	variable)
     (sde-feedback-self-blind		nil	variable)
     (sde-feedback-archive-file-name 	nil	variable)
     (sde-feedback-setup-hook		nil	variable)
     (sde-feedback			t	function))

    ("sde-find"
     (sde-find-production		t	function)
     (sde-find-next-production		t	function)
     (sde-make-tags			t	function)
     (sde-remake-tags			t	function))

    ("sde-header"
     (sde-header-hooks			nil	variable)
     (sde-header-soar-version		t	function))

    ("sde-soar-mode"
     (sde-soar-mode			t 	function)
     (sde-soar-output-mode	        t	function)
     (soar				t	function)
     (go				t	function)
     (run				t	function)
     (run-soar				t	function)
     (reset				t	function)     
     (load-errors			t	function)
     (load-errors-on			t	function)
     (load-errors-off			t	function)
     (init-soar				t	function)
     (pbreak				t	function)
     (unpbreak				t	function)
     (ptrace				t	function)
     (unptrace				t	function)
     (explain				t	function)
     (explain-on			t	function)
     (explain-off			t	function)
     (firing-counts			t	function)
     (list-chunks			t	function)
     (list-justifications		t	function)
     (list-productions			t	function)
     (matches				t	function)
     (matches-1				t	function)
     (matches-2				t	function)
     (memory-stats			t	function)
     (ms				t	function)
     (pgs				t	function)
     (pgso				t	function)
     (preferences			t	function)
     (print-soar			t	function)
     (print-stats			t	function)
     (rete-stats			t	function)
     (wm				t	function)
     (sde-view-chunks			t	function)
     (sde-view-pbreaks			t	function)
     (sde-view-ptraces			t	function)
     (sde-view-working-memory		t	function)
     (excise				t	function)
     (excise-chunks			t	function)
     (excise-task			t	function)
     (excise-all			t	function)
     (excise-file			t	function)
     (init-soar				t	function)
     (load-soar				t	function)
     (load-defaults 			t	function)
     (select-agent			t	function)
     (schedule				t	function)
     (agent-go				t	function)
     (create-agent			t	function)
     (destroy-agent			t	function)
     (list-agents			t	function)     
     (soarnews				t	function)
     (sde-switch-to-soar		t	function)
     (sde-previous-input		t	function)
     (sde-next-input			t	function)
     (sde-next-input-matching		t	function)
     (sde-previous-input-matching	t	function)
     (sde-previous-similar-input	t	function)
     (sde-interrupt-soar		t	function)
     (sde-kill-output			t	function)
     (sde-show-output			t	function)
     (sde-kill-input			t	function)
     (sde-backward-prompt		t	function)
     (sde-forward-prompt		t	function)
     (sde-return			t	function)
     (sde-bol				t 	function)
     (sde-soar-track-cd-toggle		t	function)
     (track-cd-toggle			t	function)
     (sde-kill-buffer-agent-hook        t       nil)
     (sde-pop-up-error-buffer		t	nil)
     (sde-hide-soar-error-window	t	nil)
     (sde-soar-error			t	nil)
     (sde-agent				t	nil)
     (sde-process-input-start		t 	nil)
     (sde-get-old-input			t 	nil)
     (sde-soar-cmd			t	nil)
     (sde-check-soar			t	nil)
     (sde-check-soar-multi-agent	t	nil)
     (sde-soar-is-alive			t	nil)
     (sde-agent				t	nil)
     (sde-pbreak-in-effect		t	nil)
     (sde-unpbreak-production		t	nil)
     (sde-update-pbreak-list		t	nil)
     (sde-soar-help			t	nil))

    ("sde"
     (sde-site-hook			nil	variable)
     (sde-load-hook			nil	variable)
     (sde-mode-hook			nil	variable)
     (sde-soar-mode-hook		nil	variable)
     (sde-soar-hook			nil	variable)
     (sde-soar-output-hook		nil	variable)
     (sde-soar-error-hook		nil	variable)
     (sde-soar-program			nil	variable)
     (sde-soar-starting-directory	nil	variable)
     (sde-soar-switches			nil	variable)
     (sde-prompt-for-soar-switches	nil	variable)
     (sde-soar-use-ptys			nil	variable)
     (sde-soar-defaults-file		nil	variable)
     (sde-file-types			nil	variable)
     (sde-use-multiple-frames		nil	variable)
     (sde-production-indent-offset	nil	variable)
     (sde-arrow-indent-offset		nil	variable)
     (sde-soar-beep-after-setup		nil	variable)
     (sde-soar-use-output-buffer	nil	variable)
     (sde-soar-erase-output-buffer	nil	variable)
     (sde-soar-move-point-on-output	nil	variable)
     (sde-soar-output-buffer-defaults	nil	variable)
     (sde-soar-error-buffer-defaults 	nil	variable)
     (sde-soar-track-cd			nil	variable)
     (sde-soar-input-ring-size		nil	variable)
     (sde-soar-input-ring-filter	nil	variable)
     (sde-production-name-test-regexp 	nil	variable)
     (sde-mode-map			nil	variable)
     (sde-view-cmds-map			nil	variable)
     (sde-agent-cmds-map		nil	variable)
     (sde-region-cmds-map		nil	variable)
     (sde-help-cmds-map			nil	variable)
     (sde-mouse-map			nil	variable)
     (sde-soar-mode-map			nil	variable)
     (sde-soar-output-mode-map		nil	variable)
     (sde-mode-syntax-table		nil	variable)
     (sde-mode-abbrev-table		nil	variable)
     (sde-go-args			nil	variable)
     (sde-run-args			nil	variable)
     (sde-matches-args			nil	variable)
     (sde-ms-args			nil	variable)
     (sde-firing-counts-args		nil	variable)
     (sde-print-args			nil	variable)
     (sde-preferences-args		nil	variable)
     (sde-list-productions-args		nil	variable)
     (sde-list-chunks-args		nil	variable)
     (sde-list-justifications-args	nil	variable)
     (sde-agent-go-args			nil	variable)
     (sde-schedule-args			nil	variable)
     (sde-info-file			nil	variable)
     (sde-soar-info-file		nil	variable)
     (sde-soar-release-notes-file	nil	variable)
     (sde-soar-user-notes-file		nil	variable)
     (sde-backward-sexp			nil	function)
     (sde-backward-list			nil	function)
     (sde-backward-up-list		nil	function)
     (sde-backward-kill-sexp		nil	function)
     (sde-forward-sexp			nil 	function)
     (sde-forward-list			nil 	function)
     (sde-kill-sexp			nil 	function)
     (sde-beginning-of-sp		nil	function)
     (sde-end-of-sp			nil	function)
     (sde-mark-sp			nil	function)
     (sde-cd				nil	function)
     (sde-close-all-sp			nil	function)
     (sde-newline-and-indent		nil	function)
     (sde-indent-line			nil	function)
     (sde-indent-sexp			nil	function)
     (sde-reindent			nil	function)
     (sde-send-production		nil	function)
     (sde-region-send			nil	function)
     (sde-region-pbreak			nil	function)
     (sde-region-ptrace			nil	function)
     (sde-region-excise			nil	function)
     (sde-region-comment		nil	function)
     (sde-region-count-productions	nil	function)
     (sde-close-and-send		nil	function)
     (sde-find-unbalanced		nil	function)
     (sde-find-unbalanced-region	nil	function)
     (sde-reposition-window		nil	function)
     (sde-insert-date-stamp		nil	function)
     (sde-mode				nil	function)
     (sde-apropos			nil	function)
     (sde-info				nil	function)
     (sde-news				nil	function)
     (sde-soar-info			nil	function)
     (sde-soar-release-notes		nil	function)
     (sde-topic-help			nil	function)
     (sde-command-help                  nil     function)
     (sde-variable-help			nil     function)
     (sde-where-is			nil	function)
     (sde-describe-mode			nil	function)
     (sde-describe-bindings		nil 	function)
     (sde-mouse-print-soar		nil	function)
     (sde-mouse-matches			nil	function)
     (sde-mouse-preferences		nil	function)
     (sde-mouse-find-production		nil	function)
     (sde-mouse-send-production		nil	function)
     (sde-mouse-ptrace-production	nil	function)
     (sde-mouse-explain			nil	function))))


(defmacro sde-set-up-file-symbols (data)
  (` (let ((blurb "A yet-to-be-loaded Soar Development Environment command.")
	   (file (car (, data)))
	   (type nil)
	   (sym nil))
       (mapcar (function
		(lambda (triple)
		 (setq sym  (car triple)
		       type (car (cdr (cdr triple))))
		 (if (car (cdr triple))
		     (autoload sym file blurb t))
		 ;; Type field non-nil ==> include in completion tables.
		 (cond ((eq type 'function)
			(intern (prin1-to-string sym) sde-commands-obarray))
		       ((eq type 'variable)
			(intern (prin1-to-string sym) sde-variables-obarray)))))
	       (cdr (, data))))))


;; Now set it all up.

(mapcar (function (lambda (x) (sde-set-up-file-symbols x)))
         sde-symbols)	  


;;;----------------------------------------------------------------------------
;;; 22. Closing statements.
;;;----------------------------------------------------------------------------

;; Attempt to load site-specific file.  If it doesn't exist, don't complain.

(load "sde-site.el" 'failure-ok)

;; Run user hooks.

(run-hooks 'sde-site-hook)
(run-hooks 'sde-load-hook)

;; Set up the file extensions which invoke soar-mode.  The variable
;; `sde-file-types' contains the list of file suffixes.  If you wish to add
;; to this list, do a setq of `sde-file-types' prior to loading this file.
;; (For example, you can do this in the site-hook or load-hook.)

(mapcar '(lambda (suffix)
	  (if (not (assoc suffix auto-mode-alist))
	      (setq auto-mode-alist (cons (cons suffix 'sde-mode)
					  auto-mode-alist))))
	sde-file-types)

;; Create various hash tables.

(setq sde-name-table (sde-make-hashtable)
      sde-known-files (sde-make-hashtable 503))

;; Set up initial help topics list.  Whenever Soar is started up, this
;; is reset to include the help topics that Soar knows about.  

(mapatoms
 (function (lambda (name) (sde-puthash name nil sde-topics-obarray)))
 sde-commands-obarray)

(mapatoms
 (function (lambda (name) (sde-puthash name nil sde-topics-obarray)))
 sde-variables-obarray)

;;; Emacs indentation support for macros and form hooks for edebug.
;;;
;;; Local Variables:
;;; eval:(put 'sde-make-help-screen    'lisp-indent-function 1)
;;; eval:(put 'sde-make-help-screen    'lisp-indent-hook 1)
;;; eval:(put 'sde-with-mouse-click    'lisp-indent-function 1)
;;; eval:(put 'sde-with-mouse-click    'lisp-indent-hook 1)
;;; eval:(put 'sde-make-hash-key       'edebug-form-spec '(form))
;;; eval:(put 'sde-puthash             'edebug-form-spec '(form form &optional form))
;;; eval:(put 'sde-gethash             'edebug-form-spec '(form &optional form))
;;; eval:(put 'sde-map-hashtable       'edebug-form-spec '(form form))
;;; eval:(put 'sde-member              'edebug-form-spec '(form form))
;;; eval:(put 'sde-rassoc              'edebug-form-spec '(form form))
;;; eval:(put 'sde-push                'edebug-form-spec '(form form))
;;; eval:(put 'sde-pushnew             'edebug-form-spec '(form form))
;;; eval:(put 'sde-substring           'edebug-form-spec '(form integerp &optional form))
;;; eval:(put 'sde-buffer-substring    'edebug-form-spec '(integerp))
;;; eval:(put 'sde-prompt              'edebug-form-spec '(form form))
;;; eval:(put 'sde-get-buffer-create   'edebug-form-spec '(form))
;;; eval:(put 'sde-buffer-file-name    'edebug-form-spec '(form))
;;; eval:(put 'sde-make-name-regexp    'edebug-form-spec '(form))
;;; eval:(put 'sde-with-mouse-click    'edebug-form-spec '(form &rest (form)))
;;; eval:(put 'sde-set-up-file-symbols 'edebug-form-spec '(form))
;;; End:
