;;;; -*- Mode: Emacs-Lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-feedback.el,v $
;;;; $Id: sde-feedback.el,v 0.4 1994/03/03 16:13:27 hucka Exp $
;;;; 
;;;; Description       : SDE user functions for reporting feedback.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs 18.58: Copyright (C) 1985-1993 Free Software Foundation, Inc.
;;;; Soar-mode 5.0    Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:      Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.46:       Copyright (C) 1991-1992 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25    Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:     Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu

(defconst sde-feedback-el-version "$Revision: 0.4 $"
  "The revision number of sde-feedback.el.  The complete RCS id is:
      $Id: sde-feedback.el,v 0.4 1994/03/03 16:13:27 hucka Exp $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Main code.
;;;; 5.  Closing statements
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; ----------------
;;;; 0. Documentation
;;;; ----------------
;;;; This file provides an interface for emailing feedback about SDE.  It is
;;;; largely base on the package "reporter.el" v1.20 by Barry A. Warsaw
;;;; (warsaw@anthem.nlm.nih.gov).  After an initial attempt at using
;;;; reporter, I resorted to copying it and hacking it up, turning it into a
;;;; special feature for SDE only.  This was necessary in order to implement
;;;; some of the desired functionality, such as `sde-feedback-cc'.
;;;;
;;;; It might be possible to rework reporter.el to incorporate some of the
;;;; ideas included here, but I don't have time to do this now.
;;;;
;;;; The basic idea is to provide users with an interface to composing mail
;;;; messages for bug reporters, questions, etc., about SDE.  Invoking
;;;; `sde-feedback' creates a mail buffer just like `mail' does, but
;;;; includes information about the state of SDE at the time the message was
;;;; composed.  Several variables allow the user to customize the behavior,
;;;; such as the inclusion of cc'ed recipients via `sde-feedback-cc', a
;;;; special archive file name via `sde-feedback-archive-file-name', etc.
;;;;
;;;; Beta test users complained that the state information appended by
;;;; sde-feedback in a message was annoying.  One idea was to have two
;;;; functions, `sde-feedback' and `sde-bug', with only the latter including
;;;; state information in a message.  After some agonizing over what should
;;;; be the right approach, I decided to leave it as one function and force
;;;; the state info.  The problem is basically that we can't trust the user's
;;;; judgement.  They may decide to send a question or other message using
;;;; the command that doesn't include state info, but the state info may be
;;;; important to answering the question.  Further, it's easier to remember
;;;; one command than two.  Hence, although it's admittedly ugly, it seems
;;;; best to always force the inclusion of state information and to provide
;;;; the user with only one feedback/mail/bug reporting interface.


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Requirements

(require 'sde)
(require 'sendmail)


;;;-----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;-----------------------------------------------------------------------------

;; Leave the next few vars unbound by default.  This makes it possible to
;; determine whether the user has set them at all.

(defvar sde-feedback-cc nil
  "*Address(es) that will be carbon-copied automatically by sde-feedback.
This variable should be set to a string containing addresses separated by
commas.")

(defvar sde-feedback-self-blind)
(put   'sde-feedback-self-blind 'variable-documentation
  "*If non-nil, a blind carbon-copy (bcc) field to yourself is inserted into
the sde-message when it is initialized.  Setting this variable overrides the
standard Emacs `mail-self-blind' variable in sde-feedback messages.")

(defvar sde-feedback-archive-file-name)
(put   'sde-feedback-archive-file-name 'variable-documentation
  "*If non-nil, it should be a string naming a file in which the sde-feedback
message will be written at the same time it is mailed.  Setting this variable
overrides the standard `mail-archive-file-name' in sde-feedback messages.
Here is an example of how to set this in your .emacs file:

      \(setq 'sde-feedback-archive-file-name \"~/mail/sde-mail\"")

(defvar sde-feedback-setup-hook nil
  "*If non-nil, it should be a function or list of functions to call after a
message is initialized.  Hook functions get called with no arguments.")



;;;-----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;-----------------------------------------------------------------------------

(defvar sde-feedback-address nil
  "Who to send bug reports to.  This should be set in the site file.")

(defvar sde-feedback-subject-line "Feedback about SDE"
  "String to put in the subject line of a feedback message.")

(defvar sde-feedback-salutation "Dear SDE maintainers,"
  "String inserted at the top of the mail buffer as a salutation to the
recipients of the message.  Point is left after the saluation.")

(defvar sde-feedback-interesting-variables
  '(sde-version
    sde-el-version
    sde-compat-el-version
    sde-find-el-version
    sde-header-el-version
    sde-soar-mode-el-version
    sde-feedback-el-version
    sde-highlight-el-version
    sde-running-emacs18
    sde-running-emacs19
    sde-running-lemacs
    sde-running-epoch
    sde-site-hook
    sde-load-hook
    sde-mode-hook
    sde-soar-mode-hook
    sde-soar-hook
    sde-soar-output-mode-hook
    sde-header-hooks
    sde-soar-program
    sde-soar-starting-directory
    sde-directory
    sde-file-types
    sde-soar-version
    sde-soar-version-at-least-6-1-1
    sde-soar-beep-after-setup
    sde-soar-use-output-buffer
    sde-soar-pop-up-output-buffer
    sde-soar-erase-output-buffer
    sde-soar-output-buffer-name
    sde-soar-output-buffer
    sde-soar-track-cd
    sde-soar-input-ring-filter
    sde-soar-input-ring-size
    sde-soar-default-name
    sde-soar-use-ptys
    sde-show-soar-status
    sde-production-indent-offset
    sde-arrow-indent-offset
    sde-feedback-address
    sde-feedback-cc
    sde-feedback-self-blind
    sde-feedback-archive-file-name
    sde-feedback-setup-hook
    sde-feedback-eval-buffer
    sde-feedback-buffer-name
    sde-go-args
    sde-run-args
    sde-matches-args
    sde-ms-args
    sde-firing-counts-args
    sde-print-args
    sde-preferences-args
    sde-list-production-args
    sde-list-chunks-args
    sde-list-justifications-args
    sde-agent-go-args
    sde-schedule-args
    sde-explain-args
    sde-use-multiple-windows
    sde-soar-move-point-on-output
    sde-soar-output-buffer-screen-defaults
    sde-soar-status
    sde-soar-agents
    sde-source-modes
    sde-soar-modes
    sde-modes
    sde-sp-name-regexp
    sde-info-file
    sde-soar-info-file
    sde-soar-release-notes-file
    sde-soar-user-notes-file
    sde-soar-state
    sde-soar-cmd-hook-alist
    sde-soar-pbreak-list
    sde-font-lock-list
    sde-names-face
    sde-variables-face
    sde-attributes-face
    sde-values-face
    sde-flags-face
    sde-soar-prompt-face
    sde-soar-output-buffer-title-face
    font-lock-mode-hook
    comint-version
    comint-output-filter
    comint-prompt-regexp
    input-ring-size
    tab-width
    window-system
    window-system-version
    temp-buffer-show-hook
    load-path
    )
  "List of variables whose state is included in messages sent by sde-feedback.")

(defvar sde-feedback-eval-buffer nil
  "Buffer to retrieve variable's value from.  This is necessary to properly
support the printing of buffer-local variables.  Current buffer will always
be the mail buffer being composed.  This variable is used through dynamic scoping.")

(defvar sde-feedback-buffer-name "*mail*"
  "Name of the SDE feedback message buffer.")


;;;-----------------------------------------------------------------------------
;;; 4.  Main code.
;;;-----------------------------------------------------------------------------

(autoload 'mail-setup "sendmail")

(defun sde-feedback (&optional noerase)
  "Mail a feedback message to the authors and maintainers of SDE.  
Invoking this command will create a mail buffer and insert the values of a
number of state variables useful in bug reports.  Simply write your comments,
questions, bug reports or other information, then type \\[mail-send-and-exit]
to send the message.  To abort the message, delete the buffer with \\[kill-buffer].
Supplying an argument when invoking this command means to resume editing any
existing messages.

The buffer is placed in standard Emacs mail mode, which is like Text Mode but
with these additional special commands:

\\[mail-send]  mail-send (send the message)	
\\[mail-send-and-exit]  mail-send-and-exit

\\[mail-to]  Move to \"To:\" line
\\[mail-subject]  Move to \"Subject:\" line
\\[mail-cc]  Move to \"Cc:\" line
\\[mail-bcc]  Move to \"BCC:\" line

The following SDE variables control the behavior of sde-feedback and how it
interacts with Emacs mail mode:

  If `sde-feedback-cc' is non-nil, it should be a string containing the 
    mail addresses of additional users to be carbon-copied in the
    feedback message.  The addresses in the string should be separated by
    commas.  Here is an example of how to set this in your .emacs file:

      \(setq sde-feedback-cc \"jim, barbara\"\)

  If `sde-feedback-self-blind' is t, a blind carbon-copy (bcc) to
    yourself is inserted when the message is initialized.  Setting this
    variable overrides the standard 'mail-self-blind' in sde-feedback
    messages. 

  If `sde-feedback-archive-file-name' is non-nil, it should be a string
    naming a file in which the sde-feedback message will be written at the
    same time it is mailed.  Setting this variable overrides the standard
    `mail-archive-file-name' in sde-feedback messages.  Here is an example
    of how to set this in your .emacs file:

      \(setq sde-feedback-archive-file-name \"~/mail/sde-mail\"\)

The following variables are standard Emacs mail variables and behave normally
in sde-feedback mail buffers:

  If mail-default-reply-to is non-nil, it should be a string containing 
    your return address.  A \"Reply-to:\" field with that address will be
    inserted.  You should not normally need to set this.

  If `mail-insert-signature' is non-nil, the signature file, denoted by
    the variable `mail-signature-file', is automatically inserted at the
    end of the message before sending.  \(Otherwise you must explicitly
    use \\[mail-signature] \)

Invoking sde-feedback calls the hooks on variables `mail-setup-hook
and `sde-feedback-setup-hook', in that order, if those variables are bound.
The hook functions can be used to add more default fields."

  (interactive "P")
  ;; Relying on dynamic scoping is poor style, but useful in
  ;; the case of the next three variables.
  (let ((sde-feedback-eval-buffer (current-buffer))
	(mail-self-blind (if (boundp 'sde-feedback-self-blind)
			     sde-feedback-self-blind
			     mail-self-blind))
	(mail-archive-file-name (if (boundp 'sde-feedback-archive-file-name)
				    sde-feedback-archive-file-name
				    mail-archive-file-name))
	(return-point nil))
    (require 'sendmail)
    (pop-to-buffer sde-feedback-buffer-name)
    (sde-feedback-setup noerase)
    ;; Different mailers use different separators, some may not even
    ;; use m-h-s, but sendmail.el stuff must have m-h-s bound.
    (goto-char (point-min))
    (let* ((mail-header-separator
	    (save-excursion
	      (re-search-forward
	       (concat
		"^\\("			;beginning of line
		(mapconcat
		 'identity
		 (list "[        ]*"	;simple SMTP form
		       "-+"		;mh-e form
		       mail-header-separator) ;sendmail.el form
		 "\\|")			;or them together
		"\\)$")			;end of line
	       nil
	       'move)			;search for and move
	      (buffer-substring (match-beginning 0) (match-end 0))))
	   (buf-id (and (stringp (car mode-line-buffer-identification))
			(car mode-line-buffer-identification)))
	   (sendkey "C-c C-c")		;can this be generalized like below?
	   (killkey-whereis (where-is-internal 'kill-buffer nil t))
	   (killkey (if killkey-whereis
			(key-description killkey-whereis)
			"ESC x kill-buffer")))
	    
      (re-search-forward mail-header-separator (point-max) 'move)
      (forward-line 1)
      (insert sde-feedback-salutation "\n\n")
      (setq return-point (point))	; Remember where to put point.
      (insert "\n\n")
      (sde-feedback-dump-state sde-feedback-interesting-variables)
      (goto-char return-point)
      ;; Put some informative text in the modeline.
      ;; For some reason, binding these in the let doesn't work.  Need to
      ;; setq them explicitly.
      (setq mode-line-buffer-identification
	    ;; Try to shorten the space allocated to the buffer name.
	    (if (and buf-id
		     (string-match "\\`\\([^0-9]*\\)[0-9]+\\(.*\\)\\'" buf-id))
		(list (concat
		       (substring buf-id (match-beginning 1) (match-end 1))
		       "8"
		       (substring buf-id (match-beginning 2) (match-end 2))))
		mode-line-buffer-identification))
      (setq mode-line-format
	    (list "" 'mode-line-modified 'mode-line-buffer-identification
		  (concat "(Type C-c C-c to send message, "
			  (substitute-command-keys "\\[kill-buffer]")
			  " to abort.)")
		  "  %[("
		  'mode-name 'minor-mode-alist "%n"
		  ")%]----"
		  '(-3 . "%p") "-%-"))
      (message "Please type in your report.  Type %s to send, %s to abort. "
	       sendkey killkey))))


(defun sde-feedback-setup (noerase)
  ;; Following several lines based mainly on sendmail.el's `mail' function.
  (auto-save-mode auto-save-default)
  (mail-mode)
  (and (not noerase)
       (or (not (buffer-modified-p))
	   (y-or-n-p "Unsent message being composed; erase it? "))
       (progn (erase-buffer)
	      (if (or sde-running-lemacs sde-running-emacs19)
		  (mail-setup sde-feedback-address      ; to
			      sde-feedback-subject-line	; subject
			      nil			; in-reply-to
			      sde-feedback-cc		; cc
			      nil			; reply-buffer
			      nil)			; actions
		  ;; Emacs 18 takes different arguments.
		  (mail-setup sde-feedback-address	; to
			      sde-feedback-subject-line	; subject
			      nil			; in-reply-to
			      sde-feedback-cc		; cc
			      nil))			; reply-buffer
	      t))
  ;; Set these so that this functions' documentation string is used as the
  ;; mode help, instead of the mail-mode's doc string.
  (setq major-mode 'sde-feedback)
  (setq mode-name "SDE Feedback"))


(defun sde-feedback-dump-state (varlist &optional pre-hooks post-hooks)
  "Dump the state of the mode specific variables.
VARLIST is the list of variables to dump.  Each element in VARLIST can
be a variable symbol, or a cons cell.  If a symbol, this will be
passed to `sde-feedback-dump-variable' for insertion into the mail buffer.
If a cons cell, the car must be a variable symbol and the cdr must be
a function which will be `funcall'd with the symbol. Use this to write
your own custom variable value printers for specific variables.

Note that the global variable `sde-feedback-eval-buffer' will be bound to
the buffer in which `sde-feedback-submit-bug-report' was invoked.  If you
want to print the value of a buffer local variable, you should wrap
the `eval' call in your custom printer inside a `set-buffer' (and
probably a `save-excursion'). `sde-feedback-dump-variable' handles this
properly.

PRE-HOOKS is run after the emacs-version is inserted, but before the VARLIST
is dumped.  POST-HOOKS is run after the VARLIST is dumped."

  (let ((buffer (current-buffer)))
    (set-buffer buffer)
    (insert
     "\n== Current state ==============================================================\n")
    (insert (emacs-version) "\n\n")
    (run-hooks 'pre-hooks)
    (mapcar
     (function
      (lambda (varsym-or-cons-cell)
	(let ((varsym (or (car-safe varsym-or-cons-cell)
			  varsym-or-cons-cell))
	      (printer (or (cdr-safe varsym-or-cons-cell)
			   'sde-feedback-dump-variable)))
	  (funcall printer varsym)
	  )))
     varlist)
    (insert
     "===============================================================================\n")
    (run-hooks 'post-hooks)))


(defun sde-feedback-dump-variable (varsym)
  "Pretty-print the value of the variable in symbol VARSYM."
  (let ((val (if (and (symbolp varsym) (boundp varsym))
		 (save-excursion
		   (set-buffer sde-feedback-eval-buffer)
		   (eval varsym))
		 'unbound))
	(sym (symbol-name varsym))
	(print-escape-newlines t))
    (insert sym " "
	    (cond
	      ((memq val '(t nil)) "")
	      ((listp val)         "'")
	      ((symbolp val)       "'")
	      (t                   ""))
	    (prin1-to-string val)
	    "\n")))



;;;-----------------------------------------------------------------------------
;;; 5.  Closing statements.
;;;-----------------------------------------------------------------------------
