/*
 * $Id: osupport.c,v 1.1 1993/06/17 20:48:17 jtraub Exp $
 * $Log: osupport.c,v $
 * Revision 1.1  1993/06/17  20:48:17  jtraub
 * Released
 *
 * Revision 0.1  1993/06/17  20:22:10  jtraub
 * 6.1_checkin
 *
 * Revision 9.1  1993/05/10  20:31:37  jtraub
 * Initial checking
 *
 */

/* =========================================================================
             O Support calculation routines.
========================================================================= */

#include <ctype.h>
#include "soar.h"

extern list *collect_root_variables(condition *, tc_number, bool);
extern void add_bound_variables_in_test (test t, tc_number tc, list **var_list);

/* -----------------------------------------------------------------------
                  O-Support Transitive Closure Routines

   These routines are used by the o-support calculations to mark transitive
   closures through TM (= WM+PM) plus (optionally) the RHS-generated pref's.

   The caller should first call begin_os_tc (rhs_prefs_or_nil).  Then
   add_to_os_tc (id) should be called any number of times to add stuff
   to the TC.  (Note that the rhs_prefs shouldn't be modified between the
   begin_os_tc() call and the last add_to_os_tc() call.)

   Each identifier in the TC is marked with id.tc_num=o_support_tc; the
   caller can check for TC membership by looking at id.tc_num on any id.
----------------------------------------------------------------------- */

#define add_to_os_tc_if_needed(sym) \
  { if ((sym)->common.symbol_type==IDENTIFIER_SYMBOL_TYPE) \
      add_to_os_tc (sym); }

void add_to_os_tc (Symbol *id) {
  slot *s;
  preference *pref;
  wme *w;

  /* --- if id is already in the TC, exit; else mark it as in the TC --- */
  if (id->id.tc_num==current_agent(o_support_tc)) return;
  id->id.tc_num = current_agent(o_support_tc);
  
  /* --- scan through all preferences and wmes for all slots for this id --- */
  for (w=id->id.input_wmes; w!=NIL; w=w->next)
    add_to_os_tc_if_needed (w->value);
  for (s=id->id.slots; s!=NIL; s=s->next) {
    for (pref=s->all_preferences; pref!=NIL; pref=pref->all_of_slot_next) {
      add_to_os_tc_if_needed (pref->value);
      if (preference_is_binary(pref->type))
        add_to_os_tc_if_needed (pref->referent);
    }
    for (w=s->wmes; w!=NIL; w=w->next)
      add_to_os_tc_if_needed (w->value);
  } /* end of for slots loop */
  /* --- now scan through RHS prefs and look for any with this id --- */
  for (pref=current_agent(rhs_prefs_from_instantiation); pref!=NIL; pref=pref->inst_next) {
    if (pref->id==id) {
      add_to_os_tc_if_needed (pref->value);
      if (preference_is_binary(pref->type))
        add_to_os_tc_if_needed (pref->referent);
    }
  }
  /* We don't need to worry about goal/impasse wmes here, since o-support tc's
     never start there and there's never a pointer to a goal or impasse from
     something else. */
}

void begin_os_tc (preference *rhs_prefs_or_nil) {
  current_agent(o_support_tc) = get_new_tc_number();
  current_agent(rhs_prefs_from_instantiation) = rhs_prefs_or_nil;
}

/* -----------------------------------------------------------------------
           Utilities for Testing Inclusion in the O-Support TC

   After a TC has been marked with the above routine, these utility
   routines are used for checking whether certain things are in the TC.
   Test_has_id_in_os_tc() checks whether a given test contains an equality
   test for any identifier in the TC, other than the identifier
   "excluded_sym".  Id_or_value_of_condition_list_is_in_os_tc() checks whether
   any id or value test in the given condition list (including id/value tests
   inside NCC's) has a test for an id in the TC.  In the case of value tests,
   the id is not allowed to be "sym_excluded_from_value".
----------------------------------------------------------------------- */

bool test_has_id_in_os_tc (test t, Symbol *excluded_sym) {
  cons *c;
  Symbol *referent;
  complex_test *ct;

  if (test_is_blank_test(t)) return FALSE;
  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent->common.symbol_type==IDENTIFIER_SYMBOL_TYPE)
      if (referent->id.tc_num==current_agent(o_support_tc))
        if (referent!=excluded_sym)
          return TRUE;
    return FALSE;
  }
  ct = complex_test_from_test(t);
  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if (test_has_id_in_os_tc (c->first, excluded_sym)) return TRUE;
    return FALSE;
  }
  return FALSE;
}

bool id_or_value_of_condition_list_is_in_os_tc (condition *conds,
                                        Symbol *sym_excluded_from_value) {
  for ( ; conds!=NIL; conds=conds->next) {
    switch (conds->type) {
    case POSITIVE_CONDITION:
    case NEGATIVE_CONDITION:
      if (test_has_id_in_os_tc (conds->data.tests.id_test, NIL))
        return TRUE;
      if (test_has_id_in_os_tc (conds->data.tests.value_test,
                                sym_excluded_from_value))
        return TRUE;
      break;
    case CONJUNCTIVE_NEGATION_CONDITION:
      if (id_or_value_of_condition_list_is_in_os_tc (conds->data.ncc.top,
                                             sym_excluded_from_value))
        return TRUE;
      break;
    }
  }
  return FALSE;
}

/* ------------------------------------------------------
                    Run-Time O-Support Calculation

   This routine calculates o-support for each preference for the given
   instantiation, filling in pref->o_supported (TRUE or FALSE) on each one.

   The following predicates are used for support calculations.  In the
   following, "lhs has some elt. ..." means the lhs has some id or value
   at any nesting level.

     lhs_oa_support:
       (1) does lhs test (match_goal ^operator match_operator NO) ?
       (2) mark TC (match_operator) using TM;
           does lhs has some elt. in TC but != match_operator ?
       (3) mark TC (match_state) using TM;
           does lhs has some elt. in TC ?
     lhs_oc_support:
       (1) mark TC (match_state) using TM;
           does lhs has some elt. in TC but != match_state ?
     lhs_om_support:
       (1) does lhs tests (match_goal ^operator) ?
       (2) mark TC (match_state) using TM;
           does lhs has some elt. in TC but != match_state ?

     rhs_oa_support:
       mark TC (match_state) using TM+RHS;
       if pref.id is in TC, give support
     rhs_oc_support:
       mark TC (inst.rhsoperators) using TM+RHS;
       if pref.id is in TC, give support
     rhs_om_support:
       mark TC (inst.lhsoperators) using TM+RHS;
       if pref.id is in TC, give support

   BUGBUG the code does a check of whether the lhs tests the match state via
          looking just at id and value fields of top-level positive cond's.
          It doesn't look at the attr field, or at any negative or NCC's.
          I'm not sure whether this is right or not.  (It's a pretty
          obscure case, though.)
----------------------------------------------------------------------- */

void calculate_support_for_instantiation_preferences (instantiation *inst) {
  Symbol *match_goal, *match_state, *match_operator;
  wme *match_state_wme, *match_operator_wme;
  bool lhs_tests_operator_installed;
  bool lhs_tests_operator_acceptable_or_installed;
  bool lhs_tests_match_state;
  bool lhs_is_known_to_test_something_off_match_state;
  bool lhs_is_known_to_test_something_off_match_operator;
  bool rhs_has_some_non_goal_preference;
  bool rhs_does_an_operator_creation;
  bool oc_support_possible;
  bool om_support_possible;
  bool oa_support_possible;
  preference *rhs, *pref;
  wme *w;
  condition *lhs, *c;
#ifdef DETAILED_TIMING_STATS
  struct timeval saved_start_tv;
#endif

#ifdef DETAILED_TIMING_STATS
  start_timer (&saved_start_tv);
#endif

  /* --- initialize by giving everything NO o_support --- */  
  for (pref=inst->preferences_generated; pref!=NIL; pref=pref->inst_next)
    pref->o_supported = FALSE;

  /* --- find the match goal, match state, and match operator --- */
  match_goal = inst->match_goal;
  if (!match_goal) goto o_support_done;  /* nothing gets o-support */

  match_state_wme = match_goal->id.state_slot->wmes;
  if (! match_state_wme) goto o_support_done; /* no state --> no o-support */
  match_state = match_state_wme->value;

  match_operator_wme = match_goal->id.operator_slot->wmes;
  if (match_operator_wme)
    match_operator = match_operator_wme->value;
  else
    match_operator = NIL;

  lhs = inst->top_of_instantiated_conditions;
  rhs = inst->preferences_generated;
  
  /* --- scan through rhs to look for various things --- */
  rhs_has_some_non_goal_preference = FALSE;
  rhs_does_an_operator_creation = FALSE;  

  for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
    if (! pref->id->id.isa_goal) rhs_has_some_non_goal_preference = TRUE;
    if ((pref->id==match_goal) &&
        (pref->attr==current_agent(operator_symbol)) &&
        ((pref->type==ACCEPTABLE_PREFERENCE_TYPE) ||
         (pref->type==REQUIRE_PREFERENCE_TYPE)) )
      rhs_does_an_operator_creation = TRUE;
  }

  /* --- if all rhs preferences are goal aug's, there's no o-support --- */
  if (! rhs_has_some_non_goal_preference) goto o_support_done;
  
  /* --- scan through lhs to look for various tests --- */
  lhs_tests_operator_acceptable_or_installed = FALSE;
  lhs_tests_operator_installed = FALSE;
  lhs_tests_match_state = FALSE;
  lhs_is_known_to_test_something_off_match_state = FALSE;
  lhs_is_known_to_test_something_off_match_operator = FALSE;

  for (c=lhs; c!=NIL; c=c->next) {
    if (c->type!=POSITIVE_CONDITION) continue;
    w = c->bt.wme;
    if (w->value==match_state) lhs_tests_match_state = TRUE;
    if (w->id==match_state)
      lhs_is_known_to_test_something_off_match_state = TRUE;
    if (w->id==match_operator)
      lhs_is_known_to_test_something_off_match_operator = TRUE;
    if (w==match_operator_wme) lhs_tests_operator_installed = TRUE;
    if ((w->id==match_goal)&&(w->attr==current_agent(operator_symbol)))
      lhs_tests_operator_acceptable_or_installed = TRUE;
  }

  /* --- calcluate lhs support flags --- */
  oa_support_possible = lhs_tests_operator_installed;
  oc_support_possible = rhs_does_an_operator_creation; 
  om_support_possible = lhs_tests_operator_acceptable_or_installed;

  if ((!oa_support_possible)&&(!oc_support_possible)&&(!om_support_possible))
    goto o_support_done;

  if (! lhs_is_known_to_test_something_off_match_state) {
    begin_os_tc (NIL);
    add_to_os_tc (match_state);
    if (! id_or_value_of_condition_list_is_in_os_tc (lhs, match_state)) {
      oc_support_possible = FALSE;
      om_support_possible = FALSE;
      if (! lhs_tests_match_state) oa_support_possible = FALSE;
    }
  }

  if (oa_support_possible) {
    if (! lhs_is_known_to_test_something_off_match_operator) {
      begin_os_tc (NIL);
      add_to_os_tc (match_operator);
      if (! id_or_value_of_condition_list_is_in_os_tc (lhs, match_operator))
        oa_support_possible = FALSE;
    }
  }

  /* --- look for rhs oa support --- */
  if (oa_support_possible) {
    begin_os_tc (rhs);
    add_to_os_tc (match_state);
    for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
      if (pref->id->id.tc_num==current_agent(o_support_tc))
        pref->o_supported = TRUE;
    }
  }

  /* --- look for rhs oc support --- */
  if (oc_support_possible) {
    begin_os_tc (rhs);
    for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
      if ((pref->id==match_goal) &&
          (pref->attr==current_agent(operator_symbol)) &&
          ((pref->type==ACCEPTABLE_PREFERENCE_TYPE) ||
           (pref->type==REQUIRE_PREFERENCE_TYPE)) ) {
          add_to_os_tc (pref->value);
      }
    }
    for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
      if (pref->id->id.tc_num==current_agent(o_support_tc))
        pref->o_supported = TRUE;
    }
  }
  
  /* --- look for rhs om support --- */
  if (om_support_possible) {
    begin_os_tc (rhs);
    for (c=inst->top_of_instantiated_conditions; c!=NIL; c=c->next)
      if (c->type==POSITIVE_CONDITION) {
        w = c->bt.wme;
        if ((w->id==match_goal) && (w->attr==current_agent(operator_symbol)))
          add_to_os_tc (w->value);
      }
    for (pref=rhs; pref!=NIL; pref=pref->inst_next)
      if (pref->id->id.tc_num==current_agent(o_support_tc))
        pref->o_supported = TRUE;
  }

  o_support_done:  {}
#ifdef DETAILED_TIMING_STATS
  stop_timer (&saved_start_tv, &current_agent(o_support_cpu_time));
#endif
}

/* *********************************************************************

                   Compile-Time O-Support Calculations

********************************************************************* */

/* ------------------------------------------------------------------
                         Test Is For Symbol

   This function determines whether a given symbol could be the match
   for a given test.  It returns YES if the symbol is the only symbol
   that could pass the test (i.e., the test *forces* that symbol to be
   present in WM), NO if the symbol couldn't possibly pass the test,
   and MAYBE if it can't tell for sure.  The symbol may be a variable;
   the test may contain variables.
------------------------------------------------------------------ */

typedef enum yes_no_maybe_enum { YES, NO, MAYBE } yes_no_maybe;

yes_no_maybe test_is_for_symbol (test t, Symbol *sym) {
  cons *c;
  yes_no_maybe temp;
  bool maybe_found;
  complex_test *ct;
  Symbol *referent;

  if (test_is_blank_test(t)) return MAYBE;

  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent==sym) return YES;
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE) return MAYBE;
    if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE) return MAYBE;
    return NO;
  }

  ct = complex_test_from_test(t);
  
  switch (ct->type) {
  case DISJUNCTION_TEST:
    if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE) return MAYBE;
    if (member_of_list (sym, ct->data.disjunction_list)) return MAYBE;
    return NO;
  case CONJUNCTIVE_TEST:
    maybe_found = FALSE;
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest) {
      temp = test_is_for_symbol (c->first, sym);
      if (temp==YES) return YES;
      if (temp==MAYBE) maybe_found = TRUE;
    }
    if (maybe_found) return MAYBE;
    return NO;
  default:  /* goal/impasse tests, relational tests other than equality */
    return MAYBE;
  }
}

/* ------------------------------------------------------------------
                         Find Known Goals

   This routine looks at the LHS and returns a list of variables that
   are certain to be bound to goals.

   Note:  this uses the TC routines and clobbers any existing TC.
                         
   BUGBUG should follow ^object links up the goal stack if possible
------------------------------------------------------------------ */

list *find_known_goals (condition *lhs) {
  tc_number tc;
  list *vars;
  condition *c;

  tc = get_new_tc_number();
  vars = NIL;
  for (c=lhs; c!=NIL; c=c->next) {
    if (c->type != POSITIVE_CONDITION) continue;
    if (test_includes_goal_or_impasse_id_test (c->data.tests.id_test,
                                               TRUE,
                                               FALSE))
      add_bound_variables_in_test (c->data.tests.id_test, tc, &vars);
  }
  return vars;
}

/* ------------------------------------------------------------------
                  Find Compile Time Match Goal

   Given the LHS and a list of known goals (i.e., variables that must
   be bound to goals at run-time), this routine tries to determine
   which variable will be the match goal.  If successful, it returns
   that variable; if it can't tell which variable will be the match
   goal, it returns NIL.

   Note:  this uses the TC routines and clobbers any existing TC.
------------------------------------------------------------------ */

Symbol *find_compile_time_match_goal (condition *lhs, list *known_goals) {
  tc_number tc;
  list *roots;
  list *root_goals;
  int num_root_goals;
  cons *c, *prev_c, *next_c;
  Symbol *result;
  condition *cond;
  
  /* --- find root variables --- */
  tc = get_new_tc_number();
  roots = collect_root_variables (lhs, tc, FALSE);
  
  /* --- intersect roots with known_goals, producing root_goals --- */
  root_goals = NIL;
  num_root_goals = 0;
  for (c=roots; c!=NIL; c=c->rest)
    if (member_of_list (c->first, known_goals)) {
      push (c->first, root_goals);
      num_root_goals++;
    }
  free_list (roots);

  /* --- if more than one goal, remove any with "^object nil" --- */
  if (num_root_goals > 1) {
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if ((cond->type==POSITIVE_CONDITION) &&
          (test_is_for_symbol(cond->data.tests.attr_test,current_agent(object_symbol))==YES)&&
          (test_is_for_symbol(cond->data.tests.value_test,current_agent(nil_symbol))==YES)) {
        prev_c = NIL;
        for (c=root_goals; c!=NIL; c=next_c) {
          next_c = c->rest;
          if (test_is_for_symbol (cond->data.tests.id_test, c->first)==YES) {
            /* --- remove c from the root_goals list --- */
            if (prev_c) prev_c->rest = next_c; else root_goals = next_c;
            free_cons (c);
            num_root_goals--;
            if (num_root_goals==1) break; /* be sure not to remove them all */
          } else {
            prev_c = c;
          }
        } /* end of for (c) loop */
        if (num_root_goals==1) break; /* be sure not to remove them all */
      }
    } /* end of for (cond) loop */
  }
  
  /* --- if there's only one root goal, that's it! --- */
  if (num_root_goals==1)
    result = root_goals->first;
  else
    result = NIL;

  /* --- clean up and return result --- */
  free_list (root_goals);
  return result;      
}

/* ------------------------------------------------------------------
                       Find Thing Off Goal

   Given the LHS and a the match goal variable, this routine looks
   for a positive condition testing (goal ^attr) for the given attribute
   "attr".  If such a condition exists, and the value field contains
   an equality test for a variable, then that variable is returned.
   (If more than one such variable exists, one is chosen arbitrarily
   and returned.)  Otherwise the function returns NIL.

   Note:  this uses the TC routines and clobbers any existing TC.
------------------------------------------------------------------ */

Symbol *find_thing_off_goal (condition *lhs, Symbol *goal, Symbol *attr) {
  condition *c;
  list *vars;
  tc_number tc;
  Symbol *result;

  for (c=lhs; c!=NIL; c=c->next) {
    if (c->type != POSITIVE_CONDITION) continue;
    if (test_is_for_symbol (c->data.tests.id_test, goal) != YES) continue;
    if (test_is_for_symbol (c->data.tests.attr_test, attr) != YES) continue;
    if (c->test_for_acceptable_preference) continue;
    tc = get_new_tc_number();
    vars = NIL;
    add_bound_variables_in_test (c->data.tests.value_test, tc, &vars);
    if (vars) {
      result = vars->first;
      free_list (vars);
      return result;
    }
  }
  return NIL;
}

/* ------------------------------------------------------------------
                 Condition List Has Id Test For Sym

   This checks whether a given condition list has an equality test for
   a given symbol in the id field of any condition (at any nesting level
   within NCC's).
------------------------------------------------------------------ */

bool condition_list_has_id_test_for_sym (condition *conds, Symbol *sym) {
  for ( ; conds!=NIL; conds=conds->next) {
    switch (conds->type) {
    case POSITIVE_CONDITION:
    case NEGATIVE_CONDITION:
      if (test_includes_equality_test_for_symbol (conds->data.tests.id_test,
                                                  sym))
        return TRUE;
      break;
    case CONJUNCTIVE_NEGATION_CONDITION:
      if (condition_list_has_id_test_for_sym (conds->data.ncc.top, sym))
        return TRUE;
      break;
    }
  }
  return FALSE;
}

/* ------------------------------------------------------------------
                      Add TC Through LHS and RHS

   This enlarges a given TC by adding to it any connected conditions
   in the LHS or actions in the RHS.
------------------------------------------------------------------ */

void add_tc_through_lhs_and_rhs (condition *lhs, action *rhs, tc_number tc,
                                 list **id_list, list **var_list) {
  condition *c;
  action *a;
  bool anything_changed;
  
  for (c=lhs; c!=NIL; c=c->next) c->already_in_tc = FALSE;
  for (a=rhs; a!=NIL; a=a->next) a->already_in_tc = FALSE;

  /* --- keep trying to add new stuff to the tc --- */  
  while (TRUE) {
    anything_changed = FALSE;
    for (c=lhs; c!=NIL; c=c->next)
      if (! c->already_in_tc)
        if (cond_is_in_tc (c, tc)) {
          add_cond_to_tc (c, tc, id_list, var_list);
          c->already_in_tc = TRUE;
          anything_changed = TRUE;
        }
    for (a=rhs; a!=NIL; a=a->next)
      if (! a->already_in_tc)
        if (action_is_in_tc (a, tc)) {
          add_action_to_tc (a, tc, id_list, var_list);
          a->already_in_tc = TRUE;
          anything_changed = TRUE;
        }
    if (! anything_changed) break;
  }
}

/* -----------------------------------------------------------------------
                   Calculate Compile Time O-Support

   This takes the LHS and RHS, and fills in the a->support field in each
   RHS action with either UNKNOWN_SUPPORT, O_SUPPORT, or NO_O_SUPPORT.
   (Actually, it only does this for MAKE_ACTION's--for FUNCALL_ACTION's,
   the support doesn't matter.)
----------------------------------------------------------------------- */

void calculate_compile_time_o_support (condition *lhs, action *rhs) {
  list *known_goals;
  cons *c;
  Symbol *match_goal, *match_state, *match_operator;
  yes_no_maybe lhs_oa_support, lhs_oc_support, lhs_om_support;
  action *a;
  condition *cond;
  yes_no_maybe ynm;
  bool operator_found, possible_operator_found;
  tc_number tc;

  /* --- initialize:  mark all rhs actions as "unknown" --- */
  for (a=rhs; a!=NIL; a=a->next)
    if (a->type==MAKE_ACTION) a->support=UNKNOWN_SUPPORT;

  /* --- if "operator" doesn't appear in any LHS attribute slot, and there
         are no RHS +/! makes for "operator", then nothing gets support --- */
  operator_found = FALSE;
  possible_operator_found = FALSE;
  for (cond=lhs; cond!=NIL; cond=cond->next) {
    if (cond->type != POSITIVE_CONDITION) continue;
    ynm = test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol));
    if (ynm==YES) { operator_found = possible_operator_found = TRUE; break; }
    if (ynm==MAYBE) possible_operator_found = TRUE;
  }
  if (! operator_found)
    for (a=rhs; a!=NIL; a=a->next) {
      if (a->type != MAKE_ACTION) continue;
      if (a->attr==current_agent(operator_symbol))
        { operator_found = possible_operator_found = TRUE; break; }
      if (a->attr->common.symbol_type==VARIABLE_SYMBOL_TYPE)
        possible_operator_found = TRUE;
    }
  if (! possible_operator_found) {
    for (a=rhs; a!=NIL; a=a->next)
      if (a->type == MAKE_ACTION) a->support=NO_O_SUPPORT;
    return;
  }

  /* --- find known goals; RHS augmentations of goals get no support --- */
  known_goals = find_known_goals (lhs);
  for (c=known_goals; c!=NIL; c=c->rest)
    for (a=rhs; a!=NIL; a=a->next)
      if (a->type == MAKE_ACTION)
        if (a->id == c->first) a->support = NO_O_SUPPORT;

  /* --- find match goal, state, and operator --- */
  match_goal = find_compile_time_match_goal (lhs, known_goals);
  free_list (known_goals);
  if (!match_goal) return;
  match_state = find_thing_off_goal (lhs, match_goal, current_agent(state_symbol));
  if (!match_state) return;
  match_operator = find_thing_off_goal (lhs, match_goal, current_agent(operator_symbol));

  /* --- If when checking (above) for "operator" appearing anywhere, we
     found a possible operator but not a definite operator, now go back and
     see if the possible operator was actually the match goal or match state;
     if so, it's not a possible operator.  (Note:  by "possible operator" I
     mean something appearing in the *attribute* field that might get bound
     to the symbol "operator".)  --- */
  if (possible_operator_found && !operator_found) {
    possible_operator_found = FALSE;
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if (cond->type != POSITIVE_CONDITION) continue;
      ynm = test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol));
      if ((ynm!=NO) &&
          (test_is_for_symbol (cond->data.tests.attr_test, match_goal)!=YES) &&
          (test_is_for_symbol (cond->data.tests.attr_test, match_state)!=YES))
        { possible_operator_found = TRUE; break; }
    }
    if (! possible_operator_found) {
      for (a=rhs; a!=NIL; a=a->next) {
        if (a->type != MAKE_ACTION) continue;
        /* we're looking for "operator" augs of goals only, and match_state
           couldn't get bound to a goal */
        if (a->id == match_state) continue;
        if ((a->attr->common.symbol_type==VARIABLE_SYMBOL_TYPE) &&
            (a->attr != match_goal) &&
            (a->attr != match_state))
          { possible_operator_found = TRUE; break; }
      }
    }
    if (! possible_operator_found) {
      for (a=rhs; a!=NIL; a=a->next)
        if (a->type == MAKE_ACTION) a->support=NO_O_SUPPORT;
      return;
    }
  }
  
  /* --- calculate LHS support predicates --- */
  lhs_oa_support = MAYBE;
  if (match_operator)
    if (condition_list_has_id_test_for_sym (lhs, match_operator))
      lhs_oa_support = YES;

  lhs_oc_support = MAYBE;
  lhs_om_support = MAYBE;
  if (condition_list_has_id_test_for_sym (lhs, match_state)) {
    lhs_oc_support = YES;
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if (cond->type != POSITIVE_CONDITION) continue;
      if (test_is_for_symbol (cond->data.tests.id_test, match_goal) != YES)
        continue;
      if (test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol))
          != YES)
        continue;
      lhs_om_support = YES;
      break;
    }
  }     

  if (lhs_oa_support == YES) {    /* --- look for RHS o-a support --- */
    /* --- do TC(match_state) --- */
    tc = get_new_tc_number();
    add_symbol_to_tc (match_state, tc, NIL, NIL);
    add_tc_through_lhs_and_rhs (lhs, rhs, tc, NIL, NIL);

    /* --- any action with id in the TC gets support --- */
    for (a=rhs; a!=NIL; a=a->next)
      if (action_is_in_tc (a, tc)) a->support = O_SUPPORT;
  }

  if (lhs_oc_support == YES) {    /* --- look for RHS o-c support --- */
    /* --- do TC(rhs operators) --- */
    tc = get_new_tc_number();
    for (a=rhs; a!=NIL; a=a->next) {
      if (a->type != MAKE_ACTION) continue;
      if ((a->id==match_goal) &&
          (a->attr==current_agent(operator_symbol)) &&
          ((a->preference_type==ACCEPTABLE_PREFERENCE_TYPE) ||
           (a->preference_type==REQUIRE_PREFERENCE_TYPE)) ) {
        if (rhs_value_is_symbol(a->value))
          add_symbol_to_tc (rhs_value_to_symbol(a->value), tc, NIL,NIL);
      }
    }
    add_tc_through_lhs_and_rhs (lhs, rhs, tc, NIL, NIL);

    /* --- any action with id in the TC gets support --- */
    for (a=rhs; a!=NIL; a=a->next)
      if (action_is_in_tc (a, tc)) a->support = O_SUPPORT;
  }

  if (lhs_om_support == YES) {    /* --- look for RHS o-m support --- */
    /* --- do TC(lhs operators) --- */
    tc = get_new_tc_number();
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if (cond->type != POSITIVE_CONDITION) continue;
      if (test_is_for_symbol (cond->data.tests.id_test, match_goal) == YES)
        if (test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol))
            == YES)
          add_bound_variables_in_test (cond->data.tests.value_test, tc, NIL);
    }
    add_tc_through_lhs_and_rhs (lhs, rhs, tc, NIL, NIL);

    /* --- any action with id in the TC gets support --- */
    for (a=rhs; a!=NIL; a=a->next)
      if (action_is_in_tc (a, tc)) a->support = O_SUPPORT;
  }
}
