;;;; -*- Mode: emacs-lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde.el,v $
;;;; $Id: sde.el,v 0.26 1993/06/10 20:13:09 hucka Exp hucka $
;;;; 
;;;; Description       : Main load file and common functions for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;;
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs 18.58: Copyright (C) 1985-1991 Free Software Foundation, Inc.
;;;; Soar-mode 5.0    Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:      Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.46:       Copyright (C) 1991-1992 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25    Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:     Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu

(defconst sde-el-version "$Revision: 0.26 $"
  "The revision number of sde.el.  The complete RCS id is:
      $Id: sde.el,v 0.26 1993/06/10 20:13:09 hucka Exp hucka $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Simple gensym.  Code based on cl.el of Emacs 18.58.
;;;; 5.  Often-used macros.
;;;; 6.  Hash table routines
;;;; 7.  Miscellaneous basic functions
;;;; 8.  Buffer and window handling
;;;; 9.  Search-related functions.
;;;; 10. Basic movement and production interpretation.
;;;; 11. Recordkeeping
;;;; 12. Error and diagnostic handling.
;;;; 13. Parenthesis handling.
;;;; 14. Indentation support.
;;;; 15. Sending, excising, etc., productions and regions of productions.
;;;; 16. Miscellaneous editing commands
;;;; 17. Comment support
;;;; 18. SDE mode
;;;; 19. Help support.
;;;; 20. Load commands for rest of SDE
;;;; 21. Functions for reporting bugs
;;;; 22. Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; ----------------
;;;; 0. Documentation
;;;; ----------------
;;;;
;;;; This is the main file of the Soar Development Environment (SDE), an
;;;; editing and debugging environment featuring special tools to help manage
;;;; the complexities of writing Soar code and interacting with Soar.  SDE is
;;;; designed to implement the following features:
;;;;
;;;; * Task definition and manipulation at the level of goals and problem
;;;;   spaces, and the ability to generate skeleton Soar programs.
;;;;
;;;; * Data object definition, allowing the program to specify the attributes
;;;;   intended to appear on Soar data objects.
;;;;
;;;; * Global consistency checking and updating of production source code
;;;;   using knowledge of task structure and data objects.
;;;;
;;;; * Template production creation based on the PSCM operation that a
;;;;   production is intended to implement.
;;;;
;;;; * Context-sensitive point-and-click creation of production condition and
;;;;   action elements.
;;;;
;;;; * Viewing and editing productions in an abbreviated form, showing only
;;;;   the most important text of a production while hiding text that is
;;;;   necessary only to link different attribute and value elements together.
;;;;
;;;; * Locating the code of a production given its name or part of its name,
;;;;   and locating all productions that have specified attributes in their
;;;;   condition or action sides. 
;;;;
;;;; SDE is an integrated environment and provides the following specialized
;;;; tools to help implement the features listed above:
;;;;
;;;;     Soar Mode     -- for interacing with Soar
;;;;
;;;;     Goal Editor   -- for visually structuring Soar programs in terms 
;;;;                      of their goals and problem spaces
;;;;
;;;;     Object Editor -- for visualizing and editing Soar data objects
;;;;
;;;; This file contains the core of SDE, implemented in Emacs as a mode
;;;; called "sde-mode".  Also contained here are basic macros and functions
;;;; used in the other SDE modules, and load instructions for the rest of
;;;; SDE.


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Fabulous hack to figure out where this file is located and add that
;; directory to the load-path.  This allows users simply to issue a
;;      (load "/foo/bar/soar")
;; and automatically get the /foo/bar directory added to their load-path if
;; it's not in load-path already.  Algorithm: look at where this file is
;; being loaded from, look at load-path, and if the current directory isn't
;; on load-path, add it.
;;
;; sde-directory is based on original code from Andy Norman:
;;
;;   From: Andy Norman <ange@hplb.hpl.hp.com>
;;   To: hucka@engin.umich.edu
;;   Subject: Re: How to get path info during load? 
;;   Date: Tue, 28 Jan 92 10:40:28 GMT

(defun sde-directory ()
  "Guess the directory of the file currently being loaded, or return nil."
  (let* ((buf (get-buffer-create " *backtrace*"))
	 (standard-output buf)
	 file)
    (condition-case error
	(save-excursion
	  (set-buffer buf)
	  (erase-buffer)
	  (backtrace)
	  (goto-char (point-min))
	  (if (re-search-forward "load(\"\\([^\"]+\\)\"" nil t)
	      (setq file (buffer-substring (match-beginning 1) (match-end 1))))
	  (kill-buffer buf))
      (error nil))
    (if (and file (file-name-directory file))
	(directory-file-name (file-name-directory file)))))
   
(let ((path (sde-directory))
      (lpath load-path))
  (while (and lpath (not (equal path (car lpath))))
    (setq lpath (cdr lpath)))
  (if (and path (null lpath))
      (setq load-path (cons path load-path))))
   
;; Requirements

(require 'gmhist)
(require 'help-screen)
(require 'sde-version)

;; Provide

(provide 'sde)


;;;-----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;-----------------------------------------------------------------------------


(defvar sde-site-hook nil
  "*Hook run after loading the Soar Development Environment.
This is the place to put site-specific initializations, such as local
menus.")

(defvar sde-load-hook nil
  "*Hook run after loading the Soar Development Environment.
This is a good place to put customizations and key bindings.")

(defvar sde-mode-hook nil
  "*Hook run after starting sde-mode.  
This is a good place to put customizations and key bindings.")

(defvar sde-soar-program nil
  "*String indicating the default Soar image to invoke from `run-soar'.
Unless this pathname is absolute (i.e., beginning with a '/'), it is taken to be
relative to the directory indicated by variable 'sde-soar-starting-directory'.")

(defvar sde-soar-starting-directory nil
  "*String indicating the default directory to start Soar in.  
This is a useful variable to set because Soar looks in the starting directory
for init files.  Unless 'ESC x soar' is invoked with a prefix argument, it will
cd to 'sde-soar-default-directory' before starting Soar.")

(defvar sde-file-types '("\\.soar$"  "\\.soar5$" "\\.soar6$" "\\.init\\.soar")
  "*List of file extensions that should use SDE.")

(defvar sde-production-indent-offset 2
  "*Integer value indicating how far to indent production clauses.")

(defvar sde-arrow-indent-offset 0
  "*Integer value indicating how far to indent the arrow in a Soar production.
This value is relative to the indentation of the production as a whole.
Default: 0.  A negative value moves the arrow leftward.  A positive value
indents the arrow further to the right relative to the surrounding clauses.")

(defvar sde-sp-name-regexp "^[ \t]*(sp\\s +\\(\\s_\\|\\sw\\)*"
  "*Regexp to match the name part of a production.  
This must be a regular expression containing a grouping construct such that
SDE code can use \(match-beginning 1) and \(match-end 1) to extract the name
from within a matched string.")

(defvar sde-sp-name-predicate
  '(lambda (sym) (string-match "\*\\|chunk" sym))     
  "*Predicate for determining whether a symbol likely is a production name.
It is used by the function that extracts production names near the cursor in
a buffer.  By default the predicate tests that an extracted name either
contains the character '*' or the string \"chunk\".")


;;;-----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;-----------------------------------------------------------------------------


(defconst sde-running-epoch (and (boundp 'epoch::version) epoch::version)
  "Non-nil if we're running in Epoch.")

(defconst sde-running-lemacs (and (string-match "Lucid" emacs-version) t)
  "Non-nil if we're running in Lucid Emacs.")

(defconst sde-source-modes '(sde-mode)
  "List of mode names in SDE that deal with source files.")

(defconst sde-file-extensions-regexp (mapconcat 'identity sde-file-types "\\|")
  "Regexp for matching the extensions of files that enter SDE mode.")

(defvar sde-soar-version "6"		; This gets reset once Soar starts.
  "Soar's version number in string form.")

(defvar sde-last-buffer nil
  "The last used Soar buffer.")

(defvar sde-known-files nil
  "Hash table of files known to have been seen during a session with SDE.  
File names are hashed directly and an entry's value is just its string name.
Used in facilities like `sde-find-production'.")

(defvar sde-soar-buffer-mark nil
  "Buffer-local mark for current start of input in a Soar Mode buffer.")

(make-variable-buffer-local 'sde-soar-buffer-mark)

(defvar sde-soar-status-string "no process"
  "String indicating the current status of Soar.  Used in mode lines.")


;;;-----------------------------------------------------------------------------
;;; 4.  Simple gensym.  Code based on cl.el of Emacs 18.58.
;;;-----------------------------------------------------------------------------

;; Evil code.  I need sde-gensym later on in a macro, but there is a
;; bootstrapping problem unless one loads this file uncompiled first, then
;; compiles this file, and then loads the compiled file.  Zawinski's improved
;; byte compiler provides a eval-when-compile facility that would solve this
;; problem cleanly, but we can't count on having that byte compiler
;; available.  So, resort to this hack using require.  In the approach below,
;; sde-gensym will end up uncompiled, which is okay given how it's used
;; currently.

(require				; Evaluated at load and compile time.
 (progn
   (provide 'sde-gensym-require-hack)

   (defvar sde-gensym-index 0
     "Integer used by gensym to produce new names.")

   (defvar sde-gensym-prefix "sde$$"
     "Names generated by gensym begin with this string by default.")

   (defun sde-gensym (&optional prefix)
     "Generate a fresh uninterned symbol.  
Optional argument PREFIX becomes the string that begins the new name.
Warning: this could conflict with other gensyms.  Care should be taken to
insure that sde-gensym-prefix is unique to a given set of files."
     (setq prefix (concat (or prefix sde-gensym-prefix) "_"))
     (let ((newsymbol nil)
	   (newname   ""))
       (while (not newsymbol)
	 (setq newname          (concat prefix sde-gensym-index)
	       sde-gensym-index (+ sde-gensym-index 1))
	 (if (not (intern-soft newname))
	     (setq newsymbol (make-symbol newname))))
       newsymbol))

   'sde-gensym-require-hack))


;;;-----------------------------------------------------------------------------
;;; 5.  Often-used macros.
;;;-----------------------------------------------------------------------------

;; Warning: the CL equivalents are not completely compatible with the true CL
;; versions.  They evaluate the second arg twice, which violates the CL
;; standard.

(defmacro sde-member (elt list)		; Lifted from Zawinski's bytecomp.el
  "Look for ELT in LIST; return cdr whose car is `equal' to ITEM."
  (let ((x (sde-gensym "SDE")))
    (` (let (((, x) (, list)))
	 (while (and (, x) (not (equal (, elt) (car (, x)))))
	   (setq (, x) (cdr (, x))))
	 (, x)))))


(defmacro sde-push (elt lst)		; Lifted from EDB
  "Do like \(setq ELT \(cons ELT LST))."
  (if (symbolp lst)
      (list 'setq lst (list 'cons elt lst))
      (error "Second argument of push must be a symbol, not an expression:  %s"
	     lst)))
  

(defmacro sde-pushnew (elt list)
  "If ELT is not already a member of LIST, do \(setq ELT \(cons ELT LIST))."
  (` (if (not (sde-member (, elt) (, list)))
	 (setq (, list) (cons (, elt) (, list)))
	 (, list))))


(defmacro sde-substring (string match-number &optional match-end)
  "Do `substring' on the given STRING.  MATCH-NUMBER specifies the subexpression.
If MATCH-NUMBER is not followed with optional argument MATCH-END, this macro
expands into
    \(substring STRING \(match-beginning MATCH-NUMBER) \(match-end MATCH-NUMBER))
Otherwise, if optional MATCH-END is supplied, this macro becomes
    \(substring STRING \(match-beginning MATCH-NUMBER) \(match-end MATCH-END))"
  (if match-end
      (list 'substring string
	    (list 'match-beginning match-number)
	    (list 'match-end match-end))
      (list 'substring string
	    (list 'match-beginning match-number)
	    (list 'match-end match-number))))


(defmacro sde-buffer-substring (num)
  "Expands into \(buffer-substring \(match-beginning NUM) \(match-end NUM))."
  (` (buffer-substring (match-beginning (, num)) (match-end (, num)))))


(defmacro sde-interactive (prompt hist &optional default)
  "If `current-prefix-arg' is non-nil, prompt with PROMPT string and history HIST.
Optional DEFAULT specifies the value to return if `current-prefix-arg' is null."
  (` (if current-prefix-arg
	 (gmhist-interactive (, prompt) (, hist))
	 (list (or (, default) (car (eval (, hist))))))))


(defmacro sde-prompt (prompt hist)
  "If `current-prefix-arg' is non-nil, prompt user with PROMPT and history HIST.  
Otherwise, return the first item on the history list HIST."
  (` (if current-prefix-arg
	 (read-with-history-in (, hist) (, prompt))
	 (car (eval (, hist))))))


(defmacro sde-get-buffer-create (name)
  "Like `get-buffer-create', but binds `default-major-mode' to `fundamental-mode'.
This is a hack necessary to counteract a problem for users who set their
default major mode to `text-mode'.  It prevents Emacs from running
`text-mode-hook' when a buffer is created."
  (` (let ((default-major-mode 'fundamental-mode))
       (get-buffer-create (, name)))))


(defmacro sde-buffer-file-name (buffer)
  "Return the value of variable `buffer-file-name' in BUFFER."
  (` (save-excursion (set-buffer (, buffer)) buffer-file-name)))


(defmacro sde-make-name-regexp (name)
  "Take a production name and make a suitable regexp out of it."
  (` (format "^[ \t]*(sp\\s +\\(\\s_\\|\\sw\\)*%s" (regexp-quote (, name)))))


(defmacro sde-get-sde-buffers ()
  "Return a list of buffers that are SDE source mode buffers."
  (` (let ((buffers (buffer-list))
	   result)
       (while buffers
	 (save-excursion
	   (set-buffer (car buffers))
	   (if (memq major-mode sde-source-modes)
	       (sde-push (car buffers) result)))
	 (setq buffers (cdr buffers)))
       (nreverse result))))


(defmacro sde-in-process-buffer ()
  "Return non-nil if current buffer is a Soar process buffer."
  (` (and (eq major-mode 'sde-soar-mode)
	  (boundp 'sde-soar-buffer-mark)
	  sde-soar-buffer-mark)))


;; Temporary hack -- this belongs in sde-soar-mode.el, but it's used here
;; too.  Something needs to be moved.

(defmacro sde-soar-wait ()
  "Wait for Soar to be ready for input."
  (` (while (not (eq sde-soar-status 'ready))
       (accept-process-output)
       (sit-for 0))))


;;;-----------------------------------------------------------------------------
;;; 6.  Hash table routines
;;;-----------------------------------------------------------------------------


(defvar sde-name-table-size 2027
  "Size of the sde-name-table hash table.  Should be a prime number.")

(defvar sde-name-table nil
  "Hash table mapping production names to data about the productions.")


(defun sde-make-hashtable (&optional size)
  "Make an obarray suitable for use as a hashtable.
SIZE, if supplied, should be a prime number."
  (make-vector (or size sde-name-table-size) 0))


(defmacro sde-make-hash-key (key)
  "Return a string hash key made from KEY."
  (` (if (stringp (, key))
	 (, key)
	 (prin1-to-string (, key)))))


(defmacro sde-puthash (name data &optional ht)
  "Hash the string NAME and sets its data value to DATA.
Optional argument HT specifies the hash table to use.  The default hash table
is `sde-name-table'."
  (` (set
      (intern (sde-make-hash-key (, name)) (or (, ht) sde-name-table))
      (, data))))


(defmacro sde-gethash (name &optional ht)
  "Return the value associated with the hashed string NAME.
Optional argument HT specifies the hash table to use.  The default hash table
is `sde-name-table'.  Returns nil if NAME is not in the hash table."
  (let ((sym (sde-gensym "SDE")))
    (` (let (((, sym) (intern-soft (sde-make-hash-key (, name)) (or (, ht) sde-name-table))))
	 (and (, sym) (symbol-value (, sym)))))))


(defun sde-map-hashtable (fun tbl)
  "Call function FUN on each key and value in hash table TBL."
  (mapatoms
   (function 
    (lambda (sym)
     (funcall fun sym (symbol-value sym))))
   tbl))


;;;-----------------------------------------------------------------------------
;;; 7.  Miscellaneous basic functions
;;;-----------------------------------------------------------------------------

;; Emacs 18 doesn't provide add-hook.  Things are safer if we define our own,
;; even though it's duplicating code available in many other packages.  Based
;; on code posted by Tom May <uw-nsr!uw-warp!tom@beaver.cs.washington.edu>.
;; Differs in that it puts the new function on the front of the list.

(defun sde-add-hook (hook-var hook-fun)
  "Two arguments, HOOK-VAR and HOOK-FUN.  Adds HOOK-FUN to the *front* of the
list of hooks in HOOK-VAR if it is not already present.  `sde-add-hook' is very
tolerant: HOOK-VAR need not be previously defined, its value doesn't have to
be a list, lamda expressions are cool, etc."
  (or (boundp hook-var)
      (set hook-var nil))
  (let ((hook-var-val (symbol-value hook-var)))
    (or (listp hook-var-val)
	(setq hook-var-val (cons hook-var-val nil)))
    (if (eq (car hook-var-val) 'lambda)
	(setq hook-var-val (cons hook-var-val nil)))
    (or (sde-member hook-fun hook-var-val)
	(set hook-var (cons hook-fun hook-var-val)))))


;; Originally bbdb-string-trim in bbdb.el.

(defun sde-string-trim (string)
  "Remove leading and trailing whitepsace from STRING."
  (if (string-match "\\`[ \t\n]+" string)
      (setq string (substring string (match-end 0))))
  (if (string-match "[ \t\n]+\\'" string)
      (substring string 0 (match-beginning 0))
    string))


(defun sde-adelete (key alist)
  "Return a new version with ALIST minus any instances that have KEY."
  (let (tmp)
    (while alist
      (if (equal key (car (car alist)))
	  (setq alist (cdr alist))
	  (setq tmp (cons (car alist) tmp)
		alist (cdr alist))))
    (and tmp (nreverse tmp))))


;; Based on lisp-minus-prefix from ilisp.el 4.12.

(defun sde-minus-prefix-arg ()
  "Set `current-prefix-arg' to its absolute value and return t if negative."
  (if current-prefix-arg
      (if (symbolp current-prefix-arg)
	  (progn (setq current-prefix-arg nil) t)
	  (if (< (setq current-prefix-arg
		       (prefix-numeric-value current-prefix-arg))
		 0)
	      (progn (setq current-prefix-arg (- current-prefix-arg)) t)))))


(defun sde-cd (dir)
  "Make DIR become the current buffer's default directory.  
This function is like Emacs's `cd' but does its work silently."
  (interactive "DChange default directory: ")
  (setq dir (expand-file-name dir))
  (if (not (eq system-type 'vax-vms))
      (setq dir (file-name-as-directory dir)))
  (if (not (file-directory-p dir))
      (error "%s is not a directory" dir)
      (setq default-directory dir)))


;; Modified from lisp-in-string from ilisp-ext.el

(defun sde-in-string (&optional begin end)
  "Return the string region that immediately follows/precedes point or that
contains point in optional region BEGIN to END.  If point is in region, t
will be returned as well."
  (save-excursion
    (if (not begin)
	(save-excursion
	  (setq begin (sde-beginning-of-sp)
		end (sde-end-sp-text))))
    (let* ((point (progn (skip-chars-forward " \t") (point)))
	   (done nil))
      (goto-char begin)
      (while (and (< (point) end) (not done))
	(skip-chars-forward "^\"" end)
	(setq begin (point))
	(if (< begin end)
	    (if (and (not (bobp)) (= (char-after (1- begin)) ??))
		(forward-char)
		(if (condition-case () (progn (forward-sexp) (<= (point) end))
		      (error nil))
		    (progn		;After string
		      (skip-chars-forward " \t")
		      (if (or (= begin point) (= point (point)))
			  (setq done (list begin (point) nil))
			  (if (and (< begin point) (< point (point)))
			      (setq done (list begin (point) t)))))
		    ;; In string at end of buffer
		    (setq done (list begin end t))))))
      done)))


(defun sde-point-in-comment (test-regexp)
  "Return t if in a comment determined by TEST-REGEXP."
  (beginning-of-line)
  (and (looking-at test-regexp)
       (not (= (match-end 0)
	       (progn (end-of-line) (point))))))


;;;-----------------------------------------------------------------------------
;;; 8.  Buffer and window handling
;;;-----------------------------------------------------------------------------


(defun sde-killed-buffer-p (buffer)
  "Return t if BUFFER is killed."
  (not (buffer-name buffer)))


(defun sde-buffer-window (buffer)
  "Return the window in which BUFFER is shown, or return nil."
  (if sde-running-epoch
      (epoch::get-buffer-window buffer)
      (get-buffer-window buffer)))


(defun sde-pop-to-buffer (buffer)
  "Display BUFFER in a different window and switch to that window.
This function sets `sde-last-buffer' to the current buffer."
  (if (not (equal (current-buffer) sde-last-buffer))
      (setq sde-last-buffer (current-buffer)))
  (if (and (bufferp buffer) (not (sde-killed-buffer-p buffer)))
      (sde-show-buffer buffer t)))


(defun sde-show-buffer (buffer &optional switch here)
  "Display BUFFER, preferring a screen in which BUFFER has been shown previously.
If optional arg SWITCH is non-nil, switch to the buffer too.  If optional arg
HERE is non-nil, show the buffer in this window unless it is already being
shown in another window."
  (let ((buffer-window (sde-buffer-window buffer)))
    (if buffer-window
	;; There is a window currently displaying the buffer.
	(if switch			; Only switch to it if desired.
	    (progn
	      (select-window buffer-window)
	      (set-buffer buffer)))
	;; Ignore epoch for now.
	(if here
	    (switch-to-buffer buffer)
	    (if switch
		(progn
		  (pop-to-buffer buffer)
		  (set-buffer buffer))
		(display-buffer buffer))))))


;;;-----------------------------------------------------------------------------
;;; 9.  Search-related functions.
;;;-----------------------------------------------------------------------------


(defun sde-skip-chars (end)
  "Skip past whitespace, comments, backslashed characters and strings in the
current buffer as long as you are before END.  This does move the point."
  (if (< (point) end)
      (let ((comment (and comment-start (string-to-char comment-start)))
	    (done nil)
	    char)
	(while (and (< (point) end)
		    (not done))
	  (skip-chars-forward "\n\t " end)
	  (setq char (char-after (point)))
	  (cond ((eq char ?\")
		 (forward-sexp))
		((eq char comment)
		 (forward-char)
		 (skip-chars-forward "^\n" end))
		((eq char ?\\)
		 (forward-char 2))
		(t
		 (setq done t)))))))


(defun sde-skip-regexp-forward (regexp)
  "Keep moving point forward over REGEXP, stopping when the next characters in
the buffer do not match REGEXP."
  (while (looking-at regexp)
    (goto-char (match-end 0))))


(defun sde-skip-regexp-backward (regexp)
  "Keep moving point backward over REGEXP, with REGEXP *ending* at the current
point, until a character that is not matched by REGEXP."
  (let ((point (point)))
    (while (progn
	     (re-search-backward regexp (point-min) t)
	     (looking-at regexp)	; Get the match data
	     (= (match-end 0) point))
      (goto-char (match-beginning 0))
      (setq point (point)))
    (goto-char point)))


(defun sde-search-buffer (buffer regexp &optional start)
  "Search forward in BUFFER for REGEXP, starting from the top.
Optional arg START specifies a starting position.  Returns the value of point
where pattern found, or nil."
  (save-excursion
    (set-buffer buffer)
    (if start
	(goto-char start)
	(goto-char 1))
    (if (re-search-forward regexp nil t)
	(point))))


(defun sde-search-file (filename regexp &optional inhibit-msg)
  "Read FILENAME into Emacs and search for REGEXP.
This reads the raw file, not setting its mode or reading its local vars.
FILENAME must not already be in a buffer somewhere.  Prints a message
unless optional argument INHIBIT-MSG is non-nil."
  (let (results buffer)
    (save-excursion
      (setq buffer (create-file-buffer filename))
      (set-buffer buffer)
      (erase-buffer)
      (if (condition-case ()
	      (insert-file-contents filename t)
	    (file-error nil))
	  (progn
	    (goto-char (point-min))
	    (or inhibit-msg (message "Scanning file %s..." buffer-file-name))
	    (if (re-search-forward regexp nil t)
		(setq results (point)))))
      (kill-buffer buffer))
    results))


;;;-----------------------------------------------------------------------------
;;; 10. Basic movement and production interpretation.
;;; -----------------------------------------------------------------------------

;; Modified from Emacs' beginning-of-defun.

(defun sde-beginning-of-sp (&optional count stay)
  "Move backward to next beginning of an sp form.
Optional first argument COUNT means do it that many times.
Optional second argument STAY means don't move if already at the beginning
of an sp form.  Returns t unless search stops due to end of buffer."
  (interactive "p")
  (and count (< count 0) (forward-char 1))
  (if (sde-in-process-buffer)
      (if (and (= (point) (sde-process-input-start)) (not stay))
	  (progn (forward-line -1)
		 (sde-process-input-start)))
      (if (or (and stay (looking-at "^("))
	      (re-search-backward "^(" nil 'move (or count 1)))
	  (beginning-of-line)))
  (point))
  

(defun sde-end-of-sp (&optional arg must-end)
  "Move forward to next end of an sp form.
An end of an sp form is found by moving forward from the beginning of one.
Optional first argument COUNT means do it that many times.  Optional second
argument MUST-END means signal an error if sp form isn't terminated
properly."
  (interactive "p")
  (let ((point (point)))
    (if (or (sde-in-string)
	    (progn (beginning-of-line)
		   (re-search-forward "^[ \t\n]*[^; \t\n]" nil t)
		   (back-to-indentation)
		   (not (bolp))))
	(sde-beginning-of-sp))
    (sde-end-sp-text t)
    (if (= point (point))		; Already at end so move to next end
	(progn
	  (sde-skip-chars (point-max))
	  (if (not (or (eobp)
		       (= (char-after (point)) ?\n)))
	      (sde-end-sp-text t))))))


;; Modified from lisp-end-defun-text from ilisp-ext.el

(defun sde-end-sp-text (&optional at-start) 
  ;; Go the end of the text associated with the current sp and return
  ;; point.  The end is the last character before whitespace leading to
  ;; a left paren or ;;; at the left margin unless it is in a string.
  (if (not at-start)
      (sde-beginning-of-sp))
  (let ((point (point))
	(boundary (sde-find-next-start))
	(final (save-excursion
		 (condition-case ()
		     (progn (forward-sexp) (point))
		   (error (point-max))))))
    ;; Find the next line starting at the left margin and then check
    ;; to see if it is in a string. 
    (while (progn
	     (skip-chars-forward "^\"" boundary) ;To the next string
	     (if (= (point) boundary)	
		 nil			;No quote found and at limit
		 (let ((string-boundary ;Start of next defun
			(save-excursion
			  (if (re-search-forward "^\(\\|^;;;" nil t)
			      (match-beginning 0)
			      (point-max)))))
		   (if (condition-case ()
			   (progn (forward-sexp) t)
			 (error (goto-char string-boundary) nil))
		       (if (>= (point) boundary)
			   ;; Boundary was in string
			   (if (> (point) string-boundary)
			       (progn	;String ended in next defun
				 (goto-char string-boundary)
				 nil)
			       (if (> (setq boundary
					    (sde-find-next-start))
				      final)
				   ;; Normal defun
				   (progn (goto-char final) nil)
				   t))
			   t)
		       ;; Unclosed string
		       nil)))))
    (re-search-backward  "^[^; \t\n]\\|^[^;\n][ \t]*[^ \t\n]" point t)
    (end-of-line)
    (skip-chars-backward " \t")
    (if (< (point) point)
	(goto-char point)
	(if (save-excursion
	      (let ((point (point)))
		(beginning-of-line)
		(if comment-start (search-forward comment-start point t))))
	    (progn (next-line 1) (sde-indent-line)))
	(point))))


(defun sde-find-next-start ()
  ;; Find the start of the next line at the left margin that starts with
  ;; something other than whitespace, a -, a \), three or more semicolons, or
  ;; a \( not followed by <, and return point.
  (if (eobp)
      (point-max)
      (save-excursion
	(forward-char)
	(if (re-search-forward "^\\(\\(;;;\\)\\|\\(\([^<]\\)\\|\\([^-^( \t\n\);]\\)\\)" nil t)
	    (match-beginning 0)
	    (point-max)))))


;; Modified from Emacs' mark-defun.

(defun sde-mark-sp ()
  "Put mark at end of sp form, point at beginning."
  (interactive)
  (push-mark (point))
  (sde-end-of-sp)
  (push-mark (point))
  (sde-beginning-of-sp)
  (re-search-backward "^\n" (- (point) 1) t))


(defun sde-sp-body ()
  "Return as a string the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil."
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-sp 1 t))
	  (sp-ending (progn (forward-sexp 1) (point))))
      (if (and (>= pt sp-beginning) (<= pt sp-ending))
	  (buffer-substring sp-beginning sp-ending)))))


(defun sde-sp-name (&optional body)
  "Extract the name of the sp form that point is in or immediately next to.  If
point isn't in an sp form, nor immediately before \"(sp\", nor immediately
after the closing parenthesis of an sp form, return nil.  

Algorithm: store current point, scan backward for the beginning of an sp
form, then from there, scan forward for the end of that form.  If point is
within these bounds, extract the name of the production, else return nil.  "
  (save-excursion
    (let ((pt (point))
	  (sp-beginning (sde-beginning-of-sp 1 t))
	  (sp-ending (progn (forward-sexp 1) (point))))
      (if (and (>= pt sp-beginning) (<= pt sp-ending))
	  (progn
	    (goto-char sp-beginning)
	    (forward-char 4)
	    (buffer-substring (point) (progn (forward-sexp 1) (point))))))))


(defun sde-region-name (start end)
  "Return a descriptive name for a region of productions or other sexps.
Format of the name is \"From START-SP to END-SP\", where START-SP and END-SP
are the names of the productions at the beginning and end." 
  (let (from to expr-start)
    (save-excursion
      (goto-char start)
      (sde-skip-chars end)
      (if (looking-at sde-sp-name-regexp)
	  (setq from (buffer-substring (match-beginning 1) (match-end 1)))
	  (setq from (buffer-substring (point) (progn (end-of-line) (point)))))
      (goto-char end)
      ;; See if we're in the blank space or comments following some expression
      (re-search-backward "^(" nil 'move)
      (forward-sexp 1)
      (sde-skip-chars end)
      (if (= (point) end)
	  (progn
	    (re-search-backward "^(" nil 'move)
	    (if (looking-at sde-sp-name-regexp)
		(setq to (buffer-substring (match-beginning 1) (match-end 1)))
		(setq to (buffer-substring (point) (progn (end-of-line) (point))))))
	  ;; Can't recognize where we are, so punt.
	  (progn
	    (beginning-of-line)
	    (setq to (buffer-substring (point) (progn (end-of-line) (point))))))
      (concat "From " (sde-string-trim from) " to " (sde-string-trim to)))))


;; Extracting symbols and production names
;;
;; Extracting symbols is most easily done by using the trick of temporarily
;; changing the syntax table in the buffer so that sexp scanning can be used
;; effectively.  We only need to distinguish 3 character syntax classes:
;; word, space and "other".  
;;
;; Extracting production names is trickier business.  The cursor could be
;; over literally anything in a buffer, and it's impossible to tell 100%
;; correctly whether something is a production name or not unless the system
;; already knows the names of every production.  The heuristic approach used
;; here is:
;;
;;   1) try to use sde-sp-name, which will return a production name if the
;;      point is sitting inside a full sp form.
;;
;;   2) if that fails, call sde-symbol-near-point, and inspect the result
;;      to see if it "looks" like a production name.
;;
;;   3) if that fails, prompt the user.


(defvar sde-symbol-syntax-table
  (let ((table (copy-syntax-table lisp-mode-syntax-table)))
    (modify-syntax-entry ?\{ "(}" table)
    (modify-syntax-entry ?\} "){" table)
    (modify-syntax-entry ?\[ "(]" table)
    (modify-syntax-entry ?\] ")[" table)
    (modify-syntax-entry ?\( "()" table)
    (modify-syntax-entry ?\) ")(" table)
    (modify-syntax-entry ?*  "w"  table)
    (modify-syntax-entry ?:  "_"  table)
    (modify-syntax-entry ?-  "w"  table) 
    (modify-syntax-entry ?/  "w"  table) 
    (modify-syntax-entry ?\  "w"  table) 
    (modify-syntax-entry ?$  "w"  table) 
    (modify-syntax-entry ?%  "w"  table) 
    (modify-syntax-entry ?+  "w"  table) 
    (modify-syntax-entry ?=  "w"  table)
    (modify-syntax-entry ?   " "  table)
    (modify-syntax-entry ?\t " "  table)
    (modify-syntax-entry ?\f " "  table)
    (modify-syntax-entry ?\n " "  table)
    (modify-syntax-entry ?\; " "  table)
    (modify-syntax-entry ?'  " "  table)
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?\| "\"" table)
    (modify-syntax-entry ?^  " "  table)
    table)
  "Syntax table used for extracting Soar symbols from buffers.")


(defun sde-sp-name-near-point ()
  "Extract the name of the production under or before point.
If point is inside a full sp expression, the production name is unambiguous.
If point is elsewhere, this function extracts the symbol under or before
point, and heuristically judges whether it is likely to be a production
name."
  (interactive)
  (if (eobp)				   ; End of buffer.  Back up and try
      (if (not (bobp))			   ;  again, or if buffer is empty,
	  (save-excursion		   ;  complain.
	    (backward-char 1)
	    (sde-sp-name-near-point))
	  (error "Empty buffer."))
      (or (sde-sp-name)
	  (sde-symbol-near-point sde-sp-name-predicate "Production: "))))


(defun sde-symbol-near-point (&optional predicate prompt-str)
  "Extract the symbol \(production name or otherwise) under or before point.  
If optional arg PREDICATE is non-nil, it must be a function.  PREDICATE is
funcall'ed with the extracted symbol string as its single argument, and it
should return non-nil if the string is acceptable, nil otherwise.  If
PREDICATE returns nil, the user is prompted with optional argument PROMPT-STR
and the extracted symbol as a default answer which the user can select just
by typing return.  If no symbol is found near the point, and PROMPT-STR is
nil, the user is prompted with the default prompt \"Target: \".  This
function returns the symbol string found near point or a new string typed by
the user.  If nothing is found and the user doesn't supply a symbol, an error
is signaled."
  (interactive)
  (let ((old-syntax-table (syntax-table))
	(prompt-str (or prompt-str "Target: "))
	sym)
    (set-syntax-table sde-symbol-syntax-table)
    (setq sym (sde-extract-symbol))
    (set-syntax-table old-syntax-table)
    ;; Check special cases.
    (cond ((string-match sde-soar-prompt-regexp (concat sym " "))
	   ;; We extracted a prompt string -- bogus.
	   (setq sym nil))
	  ((and (eq major-mode 'sde-soar-mode)
		(string-match "\\`[0-9]+:\\'" sym))
	   ;; Extracted an integer followed by ":" -- probably a time tag in
	   ;; a wme.  Drop the trailing colon.
	   (setq sym (substring sym 0 -1))))
    ;; Check that we have something, and that if predicate is supplied it
    ;; returns non-nil on the candidate symbol.
    (if (or (null sym) (and predicate (not (funcall predicate sym))))
	(progn
	  (setq sym (read-string (if sym
				     (format "%s(default \"%s\") " prompt-str sym)
				     (format "%s" prompt-str))))
	  (if (string= "" sym)
	      (error "No symbol found or provided."))))
    sym))


(defun sde-extract-symbol ()
  ;; Return a string representation of the symbol under the point, or nil.
  (condition-case nil
      (let (start sym)
	(if (and (not (= (char-syntax (preceding-char)) ?w))
		 (= (char-syntax (following-char)) ?w))
	    ;; Sitting at the beginning of a symbol.
	    (setq sym (buffer-substring (point) (scan-sexps (point) 1)))
	    ;; If not at beginning of a symbol, go backward one sexp, mark
	    ;; there, and from there move forward one sexp.
	    (setq start (scan-sexps (point) -1)
		  sym   (buffer-substring start (scan-sexps start 1))))
	(and sym (not (string= sym "")) (sde-string-trim sym)))
    (error nil)))


;;;-----------------------------------------------------------------------------
;;; 11. Recordkeeping
;;;-----------------------------------------------------------------------------


(defun sde-record-file (filename)
  "Record file FILENAME on sde-known-files."
  (sde-puthash filename filename sde-known-files))



;;;-----------------------------------------------------------------------------
;;; 12. Error and diagnostic handling.
;;;-----------------------------------------------------------------------------


(defun sde-error-soar-not-multi-agent ()
  ;; Generate an error that Soar isn't running in multi-agent mode.
  (error "Soar is not running in multiple-agent mode."))


;;;-----------------------------------------------------------------------------
;;; 13. Parenthesis handling.
;;;-----------------------------------------------------------------------------

(defun sde-close-all-sp (&optional arg) ; From ilisp.el's close-all-lisp 
  "Unless you are in a string or comment, insert right parentheses as necessary
to balance unmatched left parentheses back to the start of the current Soar
sp form or to a previous left bracket which is then replaced with a left
parenthesis.  If there are too many right parentheses, remove them unless
there is text after the extra right parentheses.  If called with a prefix,
the entire expression will be closed and all open left brackets will be
replaced with left parentheses."
  (interactive "P")
  (let* ((point (point))
	 (begin (sde-beginning-of-sp))
	 (end (sde-end-sp-text t))
	 inserted
	 (closed nil))
    (goto-char point)
    (if (or (car (cdr (cdr (sde-in-string begin end))))
	    (save-excursion (beginning-of-line)
			    (looking-at "[ \t]*;")))
	(insert "]")
	(if (= begin end)
	    (error "No sexp to close.")
	    (save-restriction
	      (narrow-to-region begin end)
	      (if (< point begin) 
		  (setq point begin)
		  (if (> point end)
		      (setq point end)))
	      ;; Add parens at point until either the defun is closed, or we
	      ;; hit a square bracket.
	      (goto-char point)
	      (insert ?\))		;So we have an sexp
	      (while (progn
		       (setq inserted (point))
		       (condition-case () 
			   (progn (backward-sexp)
				  (or arg 
				      (not (eq (char-after (point)) ?\[))))
			 (error (setq closed t) nil)))
		;; With an arg replace all left brackets
		(if (and arg (= (char-after (point)) ?\[))
		    (progn
		      (delete-char 1)
		      (insert ?\()
		      (backward-char)))
		(forward-sexp)
		(insert ?\)))
	      (if (< (point) point)
		  ;; We are at a left bracket
		  (let ((left (point)))
		    (delete-char 1)
		    (insert ?\()
		    (backward-char)
		    (forward-sexp))
		  ;; There was not an open left bracket so close at end
		  (delete-region point inserted)
		  (goto-char begin)
		  (if (condition-case () (progn
					   (forward-sexp)
					   (<= (point) end))
			(error nil))
		      ;; Delete extra right parens
		      (let ((point (point)))
			;; Skip over trailing extra parens, comments & blanks.
			(sde-skip-regexp-forward "\\([ \t]*)\\|\n\\|[ \t]*;+[^\n]*\\)")
			(if (or (bolp) (eobp))
			    (progn
			      (sde-skip-regexp-backward ";+[^\n]*\n*")
			      (delete-region point (point)))
			    (error
			     "There is text after the last right parentheses.")))
		      ;; Insert parens at end changing any left brackets
		      (goto-char end)
		      (while 
			  (progn
			    (insert ?\))
			    (save-excursion
			      (condition-case ()
				  (progn (backward-sexp)
					 (if (= (char-after (point)) ?\[)
					     (progn
					       (delete-char 1)
					       (insert ?\()
					       (backward-char)))
					 (> (point) begin))
				(error (delete-backward-char 1)
				       nil))))))))))))


;;;-----------------------------------------------------------------------------
;;; 14. Indentation support.
;;;-----------------------------------------------------------------------------

;; Marker to keep track of point so that it does not move during a
;; sde-reindent.  Made a global var to avoid calling make-marker repeatedly.

(defvar sde-fill-marker (make-marker))

(defun sde-reindent ()
  "Intelligently reindent the text under the cursor.
If in a comment, indent the comment paragraph bounded by non-comments, blank
lines or empty comment lines.  If in a string, indent the paragraph bounded
by string delimiters or blank lines.  Otherwise go to the containing sp form,
close it and reindent the code block."
  (interactive)
  (let ((region (sde-in-string))
	(comment (concat "[ \t]*" comment-start "+[ \t]*")))
    (set-marker sde-fill-marker (point))
    (back-to-indentation)
    (cond (region
	   (or (= (char-after (point)) ?\")
	       (and (< (point) (car region)) (goto-char (car region)))
	       (re-search-backward "^$" (car region) 'end))
	   (let ((begin (point))
		 (end (car (cdr region)))
		 (fill-prefix nil))
	     (forward-char)
	     (re-search-forward "^$" end 'end)
	     (if (= (point) end)
		 (progn (skip-chars-forward "^\n")
			(if (not (eobp)) (forward-char))))
	     (fill-region-as-paragraph begin (point))))
	  ((looking-at comment)
	   (let ((fill-prefix
		  (buffer-substring
		   (progn (beginning-of-line) (point))
		   (match-end 0))))
	     (while (and (not (bobp)) (sde-point-in-comment comment))
	       (forward-line -1))
	     (if (not (bobp)) (forward-line 1))
	     (let ((begin (point)))
	       (while (and (sde-point-in-comment comment) (not (eobp)))
		 (replace-match fill-prefix)
		 (forward-line 1))
	       (if (not (eobp))
		   (progn (forward-line -1)
			  (end-of-line)
			  (forward-char 1)))
	       (fill-region-as-paragraph begin (point)))))
	  (t
	   (goto-char sde-fill-marker)
	   (sde-close-all-sp)
	   (sde-beginning-of-sp)
	   (sde-indent-sexp)))
  (goto-char sde-fill-marker)
  (set-marker sde-fill-marker nil)
  (message "Done.")))


(defun sde-indent-sexp ()
  "Indent each line of the list starting just after point."
  (interactive)
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region (save-excursion (sde-process-input-start)) (point-max)))
    (indent-sexp)))


(defun sde-indent-line (&optional whole-exp)
  "Indent current line as Soar code.  
With argument, indent any additional lines of the same expression rigidly
along with this one.  This is restricted to the current buffer input."
  (interactive "P")
  (save-restriction
    (if (eq major-mode 'sde-soar-mode)
	(narrow-to-region (save-excursion (sde-process-input-start)) (point-max)))
    (lisp-indent-line whole-exp)))


(defun sde-newline-and-indent ()
  "If at the end of the buffer and end of an sp, send the string back to the
process mark with no newline.  Otherwise, insert a newline, then indent.  In
a Soar process buffer the region is narrowed first.  See `newline-and-indent'
for more information."
  (interactive "*")
  (let (input)
    (if (and (= (point) (point-max)) 
	     (eq major-mode 'sde-soar-mode)
	     (setq input (sde-get-old-input)))
	(let ()
	  (sde-soar-cmd input)
	  (set-marker (process-mark sde-soar-process) (point)))
	(save-restriction
	  (if (eq major-mode 'sde-soar-mode)
	      (narrow-to-region (save-excursion (sde-process-input-start))
				(point-max)))
	  (newline-and-indent)))))


;; Indent hook for Soar code.  This is used by the standard Emacs
;; indentation routines to determine the column to which the current
;; line should be indented.
;;
;; From the doc string of parse-partial-sexp:
;;
;; State is a list of seven elements describing final state of parsing:
;; 0. depth in parens.
;; 1. character address of start of innermost containing list; nil if none.
;; 2. character address of start of last complete sexp terminated.
;; 3. non-nil if inside a string.
;;    (it is the character that will terminate the string.)
;; 4. t if inside a comment.
;; 5. t if following a quote character.
;; 6. the minimum paren-depth encountered during this scan.

(defun sde-indent-hook (indent-point state)
  (let ((normal-indent (current-column))
	(containing-form-start (elt state 1)))
    (goto-char containing-form-start)
    (cond ((looking-at "(sp")		; Start of sp.
	   (if (eq major-mode 'sde-soar-mode)
	       (+ 6 sde-production-indent-offset)
	       sde-production-indent-offset))
	  ((looking-at "(<[^>]+>")	; Variable name after "("
	   (forward-char 1)
	   (forward-sexp 1)
	   (skip-chars-forward " \t\n")
	   (list (current-column) containing-form-start))
	  ((looking-at "(goal")		; Special case: "(goal"
	   (forward-char 1)
	   (forward-sexp 1)
	   (skip-chars-forward " \t\n")
	   (if (looking-at "<[^>]+>")
	       (progn
		 (forward-sexp 1)
		 (skip-chars-forward " \t\n")
		 (current-column))
	       (current-column)))
	  (t
	   (current-column)))))  


;;;-----------------------------------------------------------------------------
;;; 15. Sending, excising, etc., productions and regions of productions.
;;;-----------------------------------------------------------------------------


(defun sde-eval-production ()
  "Evaluate the production under the cursor in the Soar process.  
With prefix argument, evaluate production and switch to Soar.  If the new
production is a changed version of an already-known production of the same
name, Soar will automatically excise the previous definition and update it
with the new one.  Note that evaluating a production will unpbreak the
production, if a pbreak was in effect."
  (interactive)
  (sde-check-soar)
  (let ((body (sde-sp-body)))
    (if body
	(let* ((name (sde-sp-name))
	       (msg (concat "sp " name))
	       (agent (sde-agent)))
	  (sde-soar-cmd agent body nil nil current-prefix-arg msg msg)
	  ;; Clear possible pbreaks for this production.
	  (if (sde-pbreak-in-effect agent name)
	      (sde-unpbreak-production agent name)))
	(error "Point is not inside a production."))))


;; The next function actually sends the whole region, instead of extracting
;; the productions in the region and sending only those.  This seems more
;; flexible, allowing users to evaluate more than productions in a buffer.
;; E.g., they may have load statements.  Without this facility, there would
;; be no way to evaluate a non-production statement, except by sending the
;; whole file.

(defun sde-region-eval (start end)
  "Send the region between point and mark to the Soar process.  
With prefix argument, send the region and switch to Soar.  Note that
evaluating a region will unpbreak all productions in the region, if pbreaks
were in effect for any of them."
  (interactive "r")
  (sde-check-soar)
  (let ((name (sde-region-name start end)))
    (sde-soar-cmd (sde-agent) (buffer-substring start end) 
		  nil nil current-prefix-arg name name)
    (sde-update-pbreaks)))


(defun sde-region-pbreak (start end)
  "Pbreak each production in the region.  
I.e., redefine each production to interrupt Soar when it fires.  If given a
positive prefix arg, undoes the pbreaks on each production in the region.  If
given a negative prefix arg, undoes all currently active pbreaks (whether for
productions in the region or not).  Pbreaks are remembered on a per-agent
basis, so pbreak'ing a production in one agent will not automatically pbreak
that production in any other agents that may also share the same production.
To list the currently active pbreaks, use \\[sde-view-pbreaks]"
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (pbreak (sde-agent) name current-prefix-arg)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-ptrace (start end)
  "Ptrace each production in the region.  
If given a positive prefix arg, undoes ptraces on all the productions in the
region.  If given a negative prefix arg, undoes all currently active ptraces
(whether for productions in the region or not).  To list all the currently
active ptraces, use \\[sde-view-ptraces]"
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (ptrace (sde-agent) name current-prefix-arg)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-excise (start end)
  "Excise each production in the region."
  (interactive "r")
  (sde-check-soar)
  (mapcar '(lambda (name)
	    (excise (sde-agent) name)
	    (sde-soar-wait))
	  (sde-region-sp-names start end)))


(defun sde-region-sp-names (start end)
  "Return a list of the names of all the productions in the given region."
  (if (and start end)			; Check for non-nil param
      (save-excursion
	(save-restriction
	  (narrow-to-region start end)
	  (goto-char (point-max))
	  (let (lst done pt)
	    (while (not done)		; Work backwards from end of region.
	      (setq pt (sde-beginning-of-sp)) ; This also moves point.
	      (if (looking-at "^(sp\\s +")
		  (progn
		    (goto-char (match-end 0))
		    (setq lst (cons (buffer-substring (point)
						      (progn (forward-sexp 1)
							     (point)))
				    lst))
		    (goto-char pt)))
	      (setq done (= (point) (point-min)))) ; Reached top of region yet?
	    ;; Return list.
	    lst)))))


;;;-----------------------------------------------------------------------------
;;; 16. Miscellaneous editing commands
;;;-----------------------------------------------------------------------------

(defun sde-close-and-send (arg)
  "Close and indent the current sp form, then send it to Soar."
  (interactive "P")
  (sde-reindent)
  (sde-close-all-sp arg)
  (if (eq major-mode 'sde-soar-mode)
      (sde-return)
      (save-excursion			; We're past last paren of sp,
	(backward-char 1)		;  so back up one char before
	(sde-eval-production)		;  trying to eval.
	(forward-char 1))))


;; Find unbalanced delimiters.
;; Originally from ilisp-ext.el by Chris McConnell.

(defun sde-find-unbalanced (arg)
  "Go to the point in buffer where there exists an extra delimiter.  
Point will be on the offending delimiter.  If called with a prefix, use the
current region.  Checks for '{' '}' and '(' ')' delimiters."
  (interactive "P")
  (if arg
      (call-interactively 'sde-find-unbalanced-region)
      (sde-find-unbalanced-region (point-min) (point-max))))


(defun sde-find-unbalanced-region (start end)
  "Go to the point in region where LEFT-DELIMITER and RIGHT-DELIMITER
become unbalanced.  Point will be on the offending delimiter."
  (interactive "r")
  (sde-count-pairs start end ?\{ ?\} )
  (sde-count-pairs start end ?\( ?\) )
  (beep)
  (message "Delimiters balance."))


(defun sde-count-pairs (begin end left-delimiter right-delimiter)
  "Return the number of top-level pairs of LEFT-DELIMITER and
RIGHT-DELIMITER between BEGIN and END.  If they don't match, the point
will be placed on the offending entry."
  (let ((old-point (point))
	(sexp 0)
	left)
    (goto-char begin)
    (sde-skip-chars end)
    (while (< (point) end)
      (let ((char (char-after (point))))
	(cond ((or (eq char left-delimiter)
		   ;; For things other than lists
		   (eq (char-after (1- (point))) ?\n))
	       (setq sexp (1+ sexp))
	       (if (condition-case ()
		       (progn (forward-sexp) nil)
		     (error t))
		   (error "Extra %s" (char-to-string left-delimiter))))
	      ((eq char right-delimiter)
	       (error "Extra %s" (char-to-string right-delimiter)))
	      ((< (point) end) (forward-char))))
      (sde-skip-chars end))
    (goto-char old-point)
    sexp))



;;;-----------------------------------------------------------------------------
;;; 17. Comment support
;;;-----------------------------------------------------------------------------

;; Comment out regions.
;; Originally from ilisp-ext.el by Chris McConnell.

(defvar sde-comment-marker (make-marker) ; Declare global to avoid calling
  "Marker for end of a comment region.") ;  make-marker repeatedly.

(defun sde-comment-region (start end prefix)
  "If prefix is positive, insert prefix copies of `comment-start' at the
start and `comment-end' at the end of each line in region.  If prefix is
negative, remove all `comment-start' and `comment-end' strings from the
region."
  (interactive "r\np")
  (save-excursion
    (goto-char end)
    (if (and (not (= start end)) (bolp))
	(setq end (1- end)))
    (goto-char end)
    (beginning-of-line)
    (set-marker sde-comment-marker (point))
    (untabify start end)
    (goto-char start)
    (beginning-of-line)
    (let* ((count 1)
	   (comment comment-start)
	   (comment-end (if (not (equal comment-end "")) comment-end)))
      (if (> prefix 0)
	  (progn
	    (while (< count prefix)
	      (setq comment (concat comment-start comment)
		    count (1+ count)))
	    (while (<= (point) sde-comment-marker)
	      (beginning-of-line)
	      (insert comment)
	      (if comment-end (progn (end-of-line) (insert comment-end)))
	      (forward-line 1)))
	  (setq comment (concat comment "+"))
	  (while (<= (point) sde-comment-marker)
	    (back-to-indentation)
	    (if (looking-at comment) (replace-match ""))
	    (if comment-end
		(progn
		  (re-search-backward comment-end)
		  (replace-match "")))
	    (forward-line 1)))
      (set-marker sde-comment-marker nil))))


;;;-----------------------------------------------------------------------------
;;; 18. SDE mode
;;;-----------------------------------------------------------------------------


(defvar sde-view-cmds-map 
  (let ((map (make-sparse-keymap)))
    (define-key map "?"    'sde-help-for-view-cmds)
    (define-key map "\C-h" 'sde-help-for-view-cmds)
    (define-key map "w"    'sde-view-working-memory)
    (define-key map "t"    'sde-view-ptraces)
    (define-key map "s"    'sde-view-stats)
    (define-key map "p"    'sde-view-productions)
    (define-key map "m"    'sde-view-ms)
    (define-key map "j"    'sde-view-justifications)
    (define-key map "g"    'sde-view-pgs)
    (define-key map "c"    'sde-view-chunks)
    (define-key map "b"    'sde-view-pbreaks)
    map)
  "Keymap for view commands.")


(defvar sde-agent-cmds-map 
  (let ((map (make-sparse-keymap)))
    (define-key map "?"    'sde-help-for-agent-cmds)
    (define-key map "\C-h" 'sde-help-for-agent-cmds)
    (define-key map "s"    'select-agent)
    (define-key map "g"    'agent-go)
    (define-key map "d"    'destroy-agents)
    (define-key map "c"    'create-agents)
    map)
  "Keymap for agent commands.")


(defvar sde-region-cmds-map
  (let ((map (make-sparse-keymap)))
    (define-key map "?"    'sde-help-for-region-cmds)
    (define-key map "\C-h" 'sde-help-for-region-cmds)
    (define-key map ";"    'sde-comment-region)
    (define-key map "x"    'sde-region-excise)
    (define-key map "t"    'sde-region-ptrace)
    (define-key map "e"    'sde-region-eval)
    (define-key map "b"    'sde-region-pbreak)
    map)
  "Keymap for region commands.")


;; Certain keys are reserved for use in Soar Mode:
;; 
;; c-c c-w  backward-kill-word    	   Like ^w in shells
;; C-c C-o  sde-kill-output                Delete last batch of process output
;; C-c C-u  sde-kill-input                 Like ^u in shells
;; M-p      sde-previous-input             Cycle backwards in input history
;; M-n      sde-next-input                 Cycle forwards
;; M-s      sde-previous-similar-input     Previous similar input
;; M-C-r    sde-previous-input-matching    Search backwards in input history
;;
;; Two key bindings get rebound in Soar Mode to something entirely different:
;;
;; C-c C-r  sde-show-output
;; M-C-r    sde-previous-input-matching

(defvar sde-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map "\C-c\C-z" 'sde-switch-to-soar)
    (define-key map "\C-c\C-y" 'wm)
    (define-key map "\C-c\C-x" 'excise)
    (define-key map "\C-c\C-v" sde-view-cmds-map)
    (define-key map "\C-c\C-t" 'ptrace)
    (define-key map "\C-c\C-r" sde-region-cmds-map)
    (define-key map "\C-c\C-q" 'firing-counts)
    (define-key map "\C-c\C-p" 'print-soar)
    (define-key map "\C-c\C-n" 'sde-find-next-production)
    (define-key map "\C-c\C-m" 'sde-soar-command-menu)
    (define-key map "\C-c\C-l" 'load-soar)
    (define-key map "\C-c\C-h" 'sde-describe-mode) ; Eventually make soar mode specific.
    (define-key map "\C-c\C-f" 'sde-find-production)
    (define-key map "\C-c\C-e" 'sde-eval-production)
    (define-key map "\C-c\C-c" 'sde-interrupt-soar)
    (define-key map "\C-c\C-b" 'pbreak)
    (define-key map "\C-c\C-a" sde-agent-cmds-map)
    (define-key map "\C-c0"    'init-soar)
    (define-key map "\C-c)"    'sde-find-unbalanced)
    (define-key map "\C-c]"    'sde-close-all-sp)
    (define-key map "\C-c\;"   'sde-comment-region)
    (define-key map "\C-c/"    'preferences)
    (define-key map "\C-c'"    'matches-1)
    (define-key map "\C-c\""   'matches-2)
    (define-key map "\C-c,"    'schedule)
    (define-key map "\C-c."    'go)
    (define-key map "\C-c?"    'sde-describe-mode)

    (define-key map "\C-hm"    'sde-describe-mode) ; Replace standard binding.

    (define-key map "\C-m"     'sde-newline-and-indent)
    (define-key map "\C-j"     'sde-newline-and-indent)
    (define-key map "\t"       'sde-indent-line)

    (define-key map "\eq"      'sde-reindent)
    (define-key map "\ej"      'indent-new-comment-line)
    (define-key map "\e\;"     'indent-for-comment)
    (define-key map "\e\C-x"   'sde-eval-production) ; Like eval-defun
    (define-key map "\e\C-r"   'sde-reposition-window)
    (define-key map "\e\C-q"   'sde-indent-sexp)
    (define-key map "\e\C-j"   'indent-new-comment-line)
    (define-key map "\e\C-e"   'sde-end-of-sp)
    (define-key map "\e\C-a"   'sde-beginning-of-sp)
    map)
  "Keymap for Soar Mode.")


;; Syntax table.

(defvar sde-mode-syntax-table
  (let ((table (copy-syntax-table lisp-mode-syntax-table)))
    (modify-syntax-entry ?\{ "(}" table)
    (modify-syntax-entry ?\} "){" table)
    (modify-syntax-entry ?\[ "(]" table)
    (modify-syntax-entry ?\] ")[" table)
    (modify-syntax-entry ?*  "_"  table)
    (modify-syntax-entry ?:  "_"  table)
    (modify-syntax-entry ?-  "_"  table) 
    (modify-syntax-entry ?/  "_"  table) 
    (modify-syntax-entry ?\  "_"  table) 
    (modify-syntax-entry ?$  "_"  table) 
    (modify-syntax-entry ?%  "_"  table) 
    (modify-syntax-entry ?+  "_"  table) 
    (modify-syntax-entry ?=  "_"  table)
    (modify-syntax-entry ?   " "  table)
    (modify-syntax-entry ?\t " "  table)
    (modify-syntax-entry ?\f " "  table)
    (modify-syntax-entry ?\; "<"  table)
    ;; Need newline to be comment-end for some of the functions that look
    ;; for productions components to work properly.  Unfortunately, this
    ;; causes "\\S " regexp's to fail to consider newline to be whitespace!
    (modify-syntax-entry ?\n ">"  table)
    (modify-syntax-entry ?'  "'"  table)
    (modify-syntax-entry ?\" "\"" table)
    (modify-syntax-entry ?\| "\"" table)
    (modify-syntax-entry ?^  "_"  table)
    table)
  "Syntax table used while in Soar Mode.")


;; Abbrev table.

(defvar sde-mode-abbrev-table nil
  "Table of abbreviations for Soar Mode.")

(if (not sde-mode-abbrev-table)
    (define-abbrev-table 'sde-mode-abbrev-table ()))


;; The actual SDE mode.

(defun sde-mode ()
  "The Soar Development Environment, SDE, a major mode for writing Soar programs.

\\[sde-beginning-of-sp] moves the cursor to the beginning of an sp form.
\\[sde-end-of-sp] moves the cursor to the end of an sp form.

Commands that move the cursor by words (e.g., forward-word, \\[forward-word], and
backward-word, \\[backward-word]) consider whitespace and the following characters to
be word delimiters: '^', '*', '-', '_', '<', '>' and '.'.  In addition, such
commands ignore parentheses, curly braces and whitespace.  Commands that move
the cursor by balanced expressions (e.g., forward-sexp, \\[forward-sexp], and
backward-sexp, \\[backward-sexp]) consider parentheses and curly braces to
be parts of expressions.

\\[sde-reindent] reindents whatever the cursor is in.  If the cursor is in a comment,
it reindents the comment.  If in an sp, it reindents the whole sp, closing the
sp first if necessary.  If in the documentation string of an sp, it reindents
the documentation string.  \\[sde-indent-line] indents just the current line.
RET automatically moves to the correct indentation level.  Variable
sde-production-indent-offset controls how far clauses are indented underneath
the first line of the production.  The default is 2 spaces.

\\[sde-eval-production] will send the current production (or preceeding production)
to the current Soar process.  With a prefix argument (such as \\[universal-argument]),
it will also switch to Soar after sending the production.  

The region commands perform common operations on a region in the buffer.
\\[sde-region-eval] sends everything within a region to Soar.
\\[sde-region-pbreak] issues a pbreak for every production in a region.
\\[sde-region-ptrace] issues a ptrace for every production in a region.
\\[sde-region-excise] excises every production in a region.

If Soar is running in multi-agent mode, and the current buffer is not
associated with any agent, the first command that needs a target agent will
prompt for an agent name and will set the buffer's default to the answer.
To change default agents, or to set the initial agent explicitly, use the
select-agent command (\\[select-agent]).

All interactive Soar commands are available via ESC x COMMANDNAME.  Some
commands in SDE have no Soar equivalents, e.g., excise-file.  Many key
bindings give direct access to the most common commands.  Some notable
bindings include the following, and a complete list appears later below.  
Use describe-key (\\[describe-key]) \(the standard Emacs help facility) to get
detailed help on each command.

  Controlling Soar execution:

    go                     \\[go]
    schedule               \\[schedule]

  Manipulating Soar memory:

    evaluate production    \\[sde-eval-production]
    excise production      \\[excise]
    load file              \\[load-soar]
    init-soar              \\[init-soar]
    
  Tracing and breaking:

    ptrace production      \\[ptrace]
    break production       \\[pbreak]

  Performing actions on regions:

    eval region            \\[sde-region-eval]
    pbreak region          \\[sde-region-pbreak]
    ptrace region          \\[sde-region-ptrace]
    excise region          \\[sde-region-excise]

  Querying Soar for information about specific objects:

    print                  \\[print-soar]
    matches 1              \\[matches-1]	
    matches 2              \\[matches-2]	
    preferences            \\[preferences]	
    wm                     \\[wm]	
    firing-counts          \\[firing-counts]	

  Querying Soar for general information:

    view ptraces           \\[sde-view-ptraces]
    view pbreaks           \\[sde-view-pbreaks]
    view goal stack (pgs)  \\[sde-view-pgs]
    view match set (ms)    \\[sde-view-ms]
    view Soar stats        \\[sde-view-stats]
    view chunks            \\[sde-view-chunks]
    view productions       \\[sde-view-productions]
    view justifications    \\[sde-view-justifications]
    view Soar WM           \\[sde-view-working-memory]

  Finding source code:

    find production source \\[sde-find-production]
    find next source       \\[sde-find-next-production]

  Controlling multiple agents:
  
    create-agents          \\[create-agents]
    destroy-agents         \\[destroy-agents] (note: currently unimplementable)
    agent-go               \\[agent-go]
    select-agent           \\[select-agent]

Many Soar commands, such as print, require \"target\" arguments, i.e.,
something to act on.  Those commands that need a production name as target
will use the name of the production that the cursor is in currently, or else
the name of the preceeding sp form.  Those commands that can take any symbol
as target will try to use the symbol under the cursor regardless of whether
it looks like a production name.

Many of the standard Soar commands also take optional arguments -- for
example, the go and print commands.  Giving a prefix argument (such as \\[universal-argument])
to most commands makes them prompt for optional arguments.  Optional
arguments are remembered on history lists.  Typing ESC p and ESC n at a 
prompt moves backward and forward through the history.

Optional arguments are automatically reused upon subsequent invocations of a
command until they are changed.  This allows you to set an option and then
apply a command repeatedly to different target arguments without having to
respecify the optional arguments each time.  For example, you can type
\\[universal-argument] \\[print-soar], type \":depth 3\" to the resulting prompt for optional
arguments, and then repeat this command with the same arguments using
just \\[print-soar] on different targets.

The ptrace (\\[ptrace]) and pbreak (\\[pbreak]) commands have different
conventions.  Without arguments, ptrace and pbreak act on the production
under the cursor.  With a positive argument (such as \\[universal-argument]) they
perform unptrace or unpbreak, respectively, on the production under the
cursor.  With a negative argument (e.g., \\[negative-argument]), they undo all
ptraces or pbreaks, respectively.

Commands that only query Soar for information place the output in in a
separate buffer for viewing, unless the variable sde-soar-use-output-buffer
is nil, in which case all output goes to the agent process buffer.  By
default the \"output buffer\" will be shown in an existing window if
possible, unless the variable sde-soar-pop-up-output-buffer is set to nil.
Also, the output buffer's existing contents will be cleared before new
output is written out, unless the variable sde-soar-erase-output-buffer is
set to nil.

If the popper package is available, Soar Mode will use it for popping up
the output buffer.

The find-production facility locates the source code for a production
whose name is under the cursor or given as argument.  Put the cursor on a
production in your task, either in a Soar process buffer or an editing buffer,
type \\[sde-find-production], and it will jump to the place in your files where the
production is defined.  With a prefix argument, it will prompt for a string
instead of using the production name under the cursor. You don't need to have
created a tags file, you don't even need to have visited the file in question
yet.  It uses several heuristics to locate productions: (1) search through
existing Emacs buffers that are Soar source code buffers; (2) if fail, and a
tag file exists, do tags search; (3) if fail, search through files known to
have been loaded into Soar (it watches Soar's output for signs of file
loading); (4) if fail, search through files found in directories of Soar
files that have been loaded or visited so far.  In addition,
sde-find-production caches information about the location of productions that
it finds, so that subsequent finds on the same productions are much faster.
\\[sde-find-next-production] goes to the next occurrence of the last searched-for production
name string in your files.

ESC x excise-file will prompt for a file name and will excise all of the
productions found in that file.

\\[sde-close-all-sp] will close the current sp form by adding or removing right parentheses
as necessary.

\\[sde-comment-region] comments out the region between mark and point.  If given a positive
prefix arg, it inserts that many semicolons at the beginning of each line
\(defaulting to 1).  If given a negative prefix argument, it uncomments
the current region.

\\[backward-delete-char-untabify] converts tabs to spaces as it moves back.

\\[sde-reposition-window] will make the current production and/or comment visible.
If the production is fully onscreen, it is moved to the top of the window.
If it is partly offscreen, the window is scrolled to get the definition (or
as much as will fit) onscreen, unless point is in a comment which is also
partly offscreen, in which case the scrolling attempts to get as much of the
comment onscreen as possible.  Repeated invocations of \\[sde-reposition-window] move the production
to the top of the window or toggle the visibility of comments that precede it.

\\[sde-find-unbalanced] searches for unbalanced parentheses in the current buffer.

\\[sde-interrupt-soar] interrupts whatever Soar is doing currently.

\\[sde-switch-to-soar] will switch you to the Soar process buffer.

To send a bug report, questions or other feedback to the authors and maintainers
of the Soar Development Environment, please use ESC x sde-feedback.

Here is a list of all the special bindings in Soar mode.
\\{sde-mode-map}
Entry to this mode will run the hooks on sde-mode-hook."

  (interactive)
  ;; Standard Emacs variables.
  (setq mode-name "SDE"
	major-mode 'sde-mode)
  (use-local-map sde-mode-map)
  (set-syntax-table sde-mode-syntax-table)
  (setq local-abbrev-table sde-mode-abbrev-table)

  (make-local-variable 'comment-column)
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'lisp-indent-hook)
  (make-local-variable 'indent-line-function)
  (make-local-variable 'paragraph-start)
  (make-local-variable 'paragraph-separate)
  (setq comment-column       40
	comment-start        ";"
	comment-start-skip   ";+ *"
	lisp-indent-hook     'sde-indent-hook
	indent-line-function 'sde-indent-line
	paragraph-start      "^(sp\\|^;+\\|^[ \t\f\n]*$"
	paragraph-separate   "^;+\\|^[ \t\f\n]*$"
	parse-sexp-ignore-comments nil)

  ;; SDE Soar Mode variables
  (make-local-variable 'sde-soar-buffer-agent)
  (setq sde-soar-buffer-agent nil)

  ;; Set up the mode line.
  (setq	mode-line-modified '("-%1*%1* ")
	mode-line-buffer-identification '("%24b")
	mode-line-format
	(list "" 'mode-line-modified 'mode-line-buffer-identification
	      "  {"
	      '(sde-soar-agents
		;; Multi-agent case
		((sde-soar-buffer-agent sde-soar-buffer-agent "no agent")
		 (sde-soar-buffer-agent ": ")
		 (sde-soar-buffer-agent sde-soar-status-string))
		;; Single-agent case
		sde-soar-status-string)
	      "}  "
	      'global-mode-string
	      " %[("
	      'mode-name 'minor-mode-alist
	      ")%n%] --"
	      '(-3 . "%p") "-%-"))
  ;; Internal bookeeping for SDE facilities.
  (sde-record-file buffer-file-name)

  ;; !!! This is a hack that must be removed when the header code
  ;; is cleaned up.  This is to enable users to set sde-header-hooks
  ;; now, even though header.el uses "make-header-hooks".
  (make-local-variable 'make-header-hooks)
  (if (boundp 'sde-header-hooks)
      (setq make-header-hooks sde-header-hooks))
  (run-hooks 'sde-mode-hook))


;;;-----------------------------------------------------------------------------
;;; 19. Help support.
;;;-----------------------------------------------------------------------------

;; Help screens for basic bindings.

(make-help-screen sde-help-for-view-cmds
  "b=pbreaks c=chunks g=pgs j=justif. m=ms p=productions s=stats t=traces w=WM"
  "Commands for viewing various Soar data:

b    view pbreaks
c    view chunks
g    view goal stack (using the \"pgs\" command)
j    view justifications
m    view match set 
p    view productions
s    view Soar stats
t    view ptraces
w    view all Soar working memory contents
?    help on view commands
C-h  help on view commands
C-g  cancel

Please use `\\[describe-key]' to find out more about any of these commands.
(Type `\\[describe-key]' followed by the key you want to find out more about.)"
  sde-view-cmds-map)


(make-help-screen sde-help-for-agent-cmds
  "c=create d=destroy g=agent-go s=select-agent ?=help"
  "Commands for Soar multi-agent commands:

c    create agents
d    destroy agents
g    set agent \"go\" parameters
s    select agent
?    help on agent commands
C-h  help on agent commands
C-g  cancel

Please use `\\[describe-key]' to find out more about any of these commands.
(Type `\\[describe-key]' followed by the key you want to find out more about.)"
  sde-agent-cmds-map)


(make-help-screen sde-help-for-region-cmds
  "b=pbreak e=eval t=ptrace x=excise \;=comment ?=help"
  "Commands for operating on regions of Soar code:

b    pbreak all productions in region
e    eval region (send to Soar)
t    ptrace all productions in region
x    excise all productions in region
;    comment or uncomment region
?    help on region commands
C-h  help on region commands
C-g  cancel

Please use `\\[describe-key]' to find out more about any of these commands.
(Type `\\[describe-key]' followed by the key you want to find out more about.)"
  sde-region-cmds-map)


;; Modifed from standard Emacs help.el.  Prints message at beginning and end,
;; to let user know that *something* is happening.  Some mode docs are long
;; and Emacs takes forever to format them.

(defvar sde-describe-mode-text-alist nil
  "Association list of mode-names and their associated documentation strings,
storing the result of doing a sde-describe-mode for each mode.  The list
consists of dotted pairs of the form \(major-mode . documentation-string).")

(defun sde-describe-mode ()
  "Display documentation of current major mode."
  (interactive)
  (with-output-to-temp-buffer "*Help*"
    (let ((mode-help (assoc major-mode sde-describe-mode-text-alist)))
      (if mode-help
	  (princ (cdr mode-help))
	  (let (tmp)
	    (message "Formatting documentation ...")
	    (princ mode-name)
	    (princ " Mode:\n")
	    (princ (documentation major-mode))
	    (save-excursion
	      (set-buffer "*Help*")
	      (setq tmp (buffer-string)))
	    (sde-push (cons major-mode tmp) sde-describe-mode-text-alist))))
    (print-help-return-message)))


;;;-----------------------------------------------------------------------------
;;; 20. Load commands for rest of SDE
;;;-----------------------------------------------------------------------------

(defvar sde-blurb "A yet-to-be-loaded Soar Development Environment command.")

;; From sde-soar-mode.el:

(autoload 'soar      		     "sde-soar-mode" sde-blurb t)
(autoload 'run-soar  		     "sde-soar-mode" sde-blurb t)
(autoload 'sde-soar-mode	     "sde-soar-mode" sde-blurb t)
(autoload 'go        		     "sde-soar-mode" sde-blurb t)
(autoload 'run       		     "sde-soar-mode" sde-blurb t)
(autoload 'init-soar 		     "sde-soar-mode" sde-blurb t)
(autoload 'sde-interrupt-subjob      "sde-soar-mode" sde-blurb t)
(autoload 'pbreak		     "sde-soar-mode" sde-blurb t)
(autoload 'ptrace		     "sde-soar-mode" sde-blurb t)
(autoload 'firing-counts 	     "sde-soar-mode" sde-blurb t)
(autoload 'list-chunks	       	     "sde-soar-mode" sde-blurb t)
(autoload 'list-productions    	     "sde-soar-mode" sde-blurb t)
(autoload 'list-justifications 	     "sde-soar-mode" sde-blurb t)
(autoload 'matches		     "sde-soar-mode" sde-blurb t)
(autoload 'matches-1		     "sde-soar-mode" sde-blurb t)
(autoload 'matches-2		     "sde-soar-mode" sde-blurb t)
(autoload 'memory-stats		     "sde-soar-mode" sde-blurb t)
(autoload 'ms			     "sde-soar-mode" sde-blurb t)
(autoload 'pgs			     "sde-soar-mode" sde-blurb t)
(autoload 'preferences		     "sde-soar-mode" sde-blurb t)
(autoload 'print-soar		     "sde-soar-mode" sde-blurb t)
(autoload 'print-stats		     "sde-soar-mode" sde-blurb t)
(autoload 'rete-stats		     "sde-soar-mode" sde-blurb t)
(autoload 'wm			     "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-chunks	     "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-pgs              "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-justifications   "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-ms		     "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-productions	     "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-stats            "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-pbreaks	     "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-ptraces	     "sde-soar-mode" sde-blurb t)
(autoload 'sde-view-working-memory   "sde-soar-mode" sde-blurb t)
(autoload 'excise		     "sde-soar-mode" sde-blurb t)
(autoload 'excise-chunks	     "sde-soar-mode" sde-blurb t)
(autoload 'excise-all		     "sde-soar-mode" sde-blurb t)
(autoload 'excise-task		     "sde-soar-mode" sde-blurb t)
(autoload 'excise-file		     "sde-soar-mode" sde-blurb t)
(autoload 'init-soar 		     "sde-soar-mode" sde-blurb t)
(autoload 'load-soar                 "sde-soar-mode" sde-blurb t)
(autoload 'pbreak                    "sde-soar-mode" sde-blurb t)
(autoload 'ptreak                    "sde-soar-mode" sde-blurb t)
(autoload 'soarnews                  "sde-soar-mode" sde-blurb t)
(autoload 'switch-to-soar 	     "sde-soar-mode" sde-blurb t)
(autoload 'select-agent		     "sde-soar-mode" sde-blurb t)
(autoload 'agent-go		     "sde-soar-mode" sde-blurb t)
(autoload 'create-agents	     "sde-soar-mode" sde-blurb t)
(autoload 'destroy-agents	     "sde-soar-mode" sde-blurb t)
(autoload 'sde-check-soar            "sde-soar-mode" sde-blurb t)
(autoload 'sde-soar-wait             "sde-soar-mode" sde-blurb t)
(autoload 'sde-agent                 "sde-soar-mode" sde-blurb t)
(autoload 'sde-switch-to-soar        "sde-soar-mode" sde-blurb t)
(autoload 'sde-process-input-start   "sde-soar-mode" sde-blurb t)
(autoload 'sde-get-old-input         "sde-soar-mode" sde-blurb t)
(autoload 'sde-soar-cmd              "sde-soar-mode" sde-blurb t)
(autoload 'sde-check-soar            "sde-soar-mode" sde-blurb t)
(autoload 'sde-agent                 "sde-soar-mode" sde-blurb t)
(autoload 'sde-pbreak-in-effect      "sde-soar-mode" sde-blurb t)
(autoload 'sde-unpbreak-production   "sde-soar-mode" sde-blurb t)
(autoload 'sde-update-pbreaks        "sde-soar-mode" sde-blurb t)
(autoload 'sde-soar-wait             "sde-soar-mode" sde-blurb t)
(autoload 'sde-return                "sde-soar-mode" sde-blurb t)

;; find-production

(autoload 'sde-find-production       "sde-find" sde-blurb t)
(autoload 'sde-next-find-production  "sde-find" sde-blurb t)

;; Compatibility with previous releases of SDE and with soar-mode v5.0

(autoload 'soar-mode                 "sde-compat" sde-blurb t)
(autoload 'production-editor-mode    "sde-compat" sde-blurb t)

(makunbound 'sde-blurb)

;; X mouse buttons

(if (eq window-system 'x)
    (load "sde-x-mouse" 'failure-ok))


;;;-----------------------------------------------------------------------------
;;; 21. Functions for reporting bugs
;;;-----------------------------------------------------------------------------

(autoload 'reporter-submit-bug-report "reporter")

(defconst sde-feedback-pkgname (concat "SDE " sde-version)
  "Package identifier for SDE for feedback messages.")

(defvar sde-feedback-address nil
  "Who to send bug reports to.  This should be set in the site file.")

(defvar sde-feedback-interesting-variables
  '((sde-version  		   . sde-feedback-safe-dump)
    (sde-el-version		   . sde-feedback-safe-dump)
    (sde-compat-el-version  	   . sde-feedback-safe-dump)
    (sde-find-el-version  	   . sde-feedback-safe-dump)
    (sde-header-el-version  	   . sde-feedback-safe-dump)
    (sde-soar-mode-el-version  	   . sde-feedback-safe-dump)
    (sde-x-mouse-el-version  	   . sde-feedback-safe-dump)
    (sde-site-hook  		   . sde-feedback-safe-dump)
    (sde-load-hook  		   . sde-feedback-safe-dump)
    (sde-mode-hook  		   . sde-feedback-safe-dump)
    (sde-soar-mode-hook		   . sde-feedback-safe-dump)
    (sde-soar-mode-soar-hook  	   . sde-feedback-safe-dump)
    (sde-soar-program  	           . sde-feedback-safe-dump)
    (sde-soar-starting-directory   . sde-feedback-safe-dump)
    (sde-file-types  		   . sde-feedback-safe-dump)
    (sde-soar-version  	           . sde-feedback-safe-dump)
    (sde-soar-beep-after-setup     . sde-feedback-safe-dump)
    (sde-soar-use-output-buffer    . sde-feedback-safe-dump)
    (sde-soar-pop-up-output-buffer . sde-feedback-safe-dump)
    (sde-soar-erase-output-buffer  . sde-feedback-safe-dump)
    (sde-soar-track-cd  	   . sde-feedback-safe-dump)
    (sde-soar-input-ring-filter    . sde-feedback-safe-dump)
    (sde-soar-input-ring-size      . sde-feedback-safe-dump)
    (sde-soar-default-name  	   . sde-feedback-safe-dump)
    (sde-soar-use-ptys  	   . sde-feedback-safe-dump)
    (sde-show-soar-status  	   . sde-feedback-safe-dump)
    (comint-output-filter  	   . sde-feedback-safe-dump)
    (comint-version  		   . sde-feedback-safe-dump)
    (comint-prompt-regexp  	   . sde-feedback-safe-dump)
    window-system
    window-system-version)
  "List of variables whose state is included in messages sent by sde-feedback.")


(defun sde-feedback ()
  "Mail a feedback message to the authors and maintainers of SDE.
Invoking this command will create a mail buffer and insert the values of a 
number of state variables useful in bug reports.  Simply write your comments,
questions, bug reports or other information, then type \\[mail-send-and-exit]
to send the message."
  (interactive)
  (reporter-submit-bug-report sde-feedback-address
			      sde-feedback-pkgname
			      sde-feedback-interesting-variables
			      "Feedback about SDE"
			      "SDE maintainer"
			      nil	; pre hooks
			      nil))	; post hooks


(defun sde-feedback-safe-dump (varsym)
  "Pretty-print the value of the variable in symbol VARSYM."
  (let ((val (if (and (symbolp varsym) (boundp varsym))
		 (eval varsym)
		 'unbound))
	(sym (symbol-name varsym))
	(print-escape-newlines t))
    (insert "     " sym " "
	    (if (or (listp val) (symbolp val)) "'" "")
	    (prin1-to-string val)
	    "\n")))


;;;-----------------------------------------------------------------------------
;;; 22. Closing statements.
;;;-----------------------------------------------------------------------------

;; Attempt to load site-specific file.  If it doesn't exist, don't complain.

(load "sde-site.el" 'failure-ok)

;; Run user hooks.

(run-hooks 'sde-site-hook)
(run-hooks 'sde-load-hook)

;; Set up the file extensions which invoke soar-mode.  The variable
;; `soar-file-types' contains the list of file suffixes.  If you wish to add
;; to this list, do a setq of `soar-file-types' prior to loading this file.
;; (For example, you can do this in the site-hook or load-hook.

(mapcar '(lambda (suffix)
	  (if (not (assoc suffix auto-mode-alist))
	      (setq auto-mode-alist (cons (cons suffix 'sde-mode)
					  auto-mode-alist))))
	sde-file-types)

;; Create various hash tables.

(setq sde-name-table (sde-make-hashtable)
      sde-known-files (sde-make-hashtable 503))

;; Emacs indentation support for macros.

(put 'make-help-screen         'lisp-indent-hook 1)

;; Form hooks for debugging with Edebug.

(put 'sde-make-hash-key       'edebug-form-spec '(form))
(put 'sde-puthash             'edebug-form-spec '(form form &optional form))
(put 'sde-gethash             'edebug-form-spec '(form &optional form))
(put 'sde-map-hashtable       'edebug-form-spec '(form form))
(put 'sde-member              'edebug-form-spec '(form form))
(put 'sde-push                'edebug-form-spec '(form form))
(put 'sde-pushnew             'edebug-form-spec '(form form))
(put 'sde-substring           'edebug-form-spec '(form integerp &optional form))
(put 'sde-buffer-substring    'edebug-form-spec '(integerp))
(put 'sde-interactive         'edebug-form-spec '(form form &optional form))
(put 'sde-prompt              'edebug-form-spec '(form form))
(put 'sde-get-buffer-create   'edebug-form-spec '(form))
(put 'sde-buffer-file-name    'edebug-form-spec '(form))
(put 'sde-make-name-regexp    'edebug-form-spec '(form))
