;;;; -*- Mode: emacs-lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-soar-mode.el,v $
;;;; $Id: sde-soar-mode.el,v 0.84 1993/06/11 04:56:55 hucka Exp hucka $
;;;; 
;;;; Description       : Soar Mode portion of SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;; 
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs 18.58: Copyright (C) 1985-1991 Free Software Foundation, Inc.
;;;; Soar-mode 5.0    Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:      Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.46:       Copyright (C) 1991-1992 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25    Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:     Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu

(defconst sde-soar-mode-el-version "$Revision: 0.84 $"
  "The revision number of sde-soar-mode.el.  The complete RCS id is:
      $Id: sde-soar-mode.el,v 0.84 1993/06/11 04:56:55 hucka Exp hucka $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Macros
;;;; 5.  Soar Mode
;;;; 6.  Interactive Soar commands
;;;; 6.A.  Commands for execution control
;;;; 6.B.  Commands for tracing and breaking
;;;; 6.C.  Commands for querying Soar
;;;; 6.D.  Commands for manipulating Soar memories
;;;; 6.E.  Commands for dealing with multiple agents
;;;; 6.F.  Help commands
;;;; 6.G.  Misc. commands
;;;; 7.  Command line interface in Soar process buffers.
;;;; 8.  Interface to Soar
;;;; 9.  Support for pbreak
;;;; 10. Support for tracking productions
;;;; 11. Miscellaneous support code
;;;; 12. Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; -----------------
;;;; 0.  Documentation
;;;; -----------------
;;;;
;;;; Soar process interaction and basic Soar editing commands.
;;;;
;;;; This was at first implement using comint-mode, but when multi-agent Soar
;;;; handling was added, comint-mode functions had to be replaced.  The
;;;; problem is that the comint functions assume there is only one buffer per
;;;; process.  To handle multi-agent Soar, we need to provide multiple
;;;; "pseudo" process buffers, one for each Soar agent.  The comint functions
;;;; make assumptions about being able to do (get-process (current-buffer))
;;;; in the current buffer, which then wouldn't work.  So, nearly all the
;;;; comint functions had to be rewritten.
;;;;
;;;; This module therefore reimplements comint-like functionality in the
;;;; process buffers, at the unfortunate cost of recreating a lot of code.
;;;;
;;;; .... (more documentation under construction here)...
;;;; 


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Requirements

(require 'gmhist)
(require 'comint)
(require 'help-screen)
(require 'sde)

;; Provide

(provide 'sde-soar-mode)


;;;-----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;-----------------------------------------------------------------------------


(defvar sde-soar-beep-after-setup nil
  "*Non-nil means beep after Soar has started up and is ready.  Default: nil.")

(defvar sde-soar-use-output-buffer t
  "*Non-nil means most output from Soar commands will be placed in a separate
buffer instead of being dumped into the Soar process buffer.  \(Exception:
output from the commands go and run is always placed in the Soar process
buffer.)  If nil, all output goes to the Soar process buffer.  Default: t")

(defvar sde-soar-pop-up-output-buffer t
  "*Non-nil (the default) means the Soar output buffer is automatically shown
in a window on the screen when new output arrives.  If nil, the buffer is not
shown and the user must switch to the buffer manually to see output.")

(defvar sde-soar-erase-output-buffer t
  "*Non-nil (the default) means the Soar output buffer is erased before each
new output appears in it.")

(defvar sde-soar-track-cd t
  "*Non-nil (the default) means track cd and pwd commands in the Soar process.")

(defvar sde-soar-input-ring-size 30
  "*Size of input history ring.")

(defvar sde-soar-input-ring-filter
  '(lambda (str) (not (string-match "\\`\\s *\\'" str)))
  "*Predicate for filtering additions to input history.
Only inputs answering true to this function are saved on the input
history list.  Default is to save anything that isn't all whitespace.")


;; Hook variables

(defvar sde-soar-mode-hook nil
  "*Hook for functions to be run after starting up a Soar process.  Good for
customizing the interaction environment.")

(defvar sde-soar-hook nil
  "*Hook of functions to be run after first prompt appears in Soar process
buffer.")


;; Defaults for optional arguments to commands

(defvar sde-go-args nil
  "*String containing default optional args for \"go\".")

(defvar sde-run-args nil
  "*String containing default optional args for \"run\".")

(defvar sde-matches-args nil
  "*String containing default optional args for \"matches\".")

(defvar sde-ms-args nil
  "*String containing default optional args for \"ms\".")

(defvar sde-firing-counts-args nil
  "*String containing default optional args for \"firing-counts\".")

(defvar sde-print-args nil
  "*String containing default optional args for \"soar-print\".")

(defvar sde-preferences-args nil
  "*String containing default optional args for \"preferences\".")

(defvar sde-list-productions-args nil
  "*String containing default optional args for \"list-productions\".")

(defvar sde-list-chunks-args nil
  "*String containing default optional args for \"list-chunks\".")

(defvar sde-list-justifications-args nil
  "*String containing default optional args for \"list-justifications\".")

(defvar sde-agent-go-args nil
  "*String containing default optional args for \"agent-go\".")

(defvar sde-schedule-args nil
  "*String containing default optional args for \"schedule\".")


;;;-----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;-----------------------------------------------------------------------------


(defvar sde-soar-use-ptys nil
  "Non-nil if SDE should make Emacs use a pty for communicating with Soar.
Default: nil.  A pty is not necessary in SDE because the Soar Mode controls
all aspects of communication and doesn't rely on features such as job
control, for which ptys are necessary.  It is more efficient to use pipes,
hence you will normally want this variable to be nil.")

(defvar sde-soar-default-name "soar"
  "*String that is the default name for single-agent Soar communication buffers.
If Soar is running in single agent mode, the buffer is named
\"*SDE-SOAR-DEFAULT-NAME*\".  If in multi-agent mode, agent buffers are named
\"*AGENT-NAME*\" where AGENT-NAME is the name of the agent.  For example, a
multi-agent Soar running two agents named \"egg-head\" and \"tough-guy\"
might have two associated buffers named \"*egg-head*\" and \"*tough-guy*\".")

(defvar sde-soar-process nil
  "The Soar process.")

(defvar sde-soar-buffer nil
  "The Soar process buffer.")

(defvar sde-soar-status 'ready
  "The current status of Soar.")

(defvar sde-show-soar-status t
  "If non-nil, immediately update process status in modelines.")

(defvar sde-soar-started-multi-regexp "Multiple agent mode selected"
  "Regexp to recognize when Soar has started up in multi-agent mode.")

(defvar sde-soar-prompt-regexp "^Soar\\( agent \\([^>]+\\)\\)?> "
  "Regular expression to recognize prompts in the Soar process buffer.
If Soar is running in multi-agent mode, match-beginning 2 returns the start
of the current agent's name in the prompt string.") 

(defvar sde-soar-sched-regexp "\nSelecting agent \\(\\s_\\|\\sw\\)*"
  "Regexp to recognize when multi-agent Soar switches agents")

(defvar sde-soar-agent-creation-regexp "Creating agent \\([^.]+\\)."
  "Regexp to recognize the names of agents when Soar says they're created.")

(defvar sde-soar-loading-regexp "^Loading \\([^ \t\n]+\\)"
  "Regexp to match Soar messages of loading files.")

(defvar sde-soar-chdir-regexp "^Changing to directory\\s *\\([^ \t\n]+\\)"
  "Regexp to match Soar saying it changed to a directory.")

(defvar sde-soar-returning-regexp "^Returning from directory\\s *\\([^ \t\n]+\\)"
  "Regexp to match Soar saying it returned from a directory.")

(defvar sde-soar-loading-or-chdir-regexp
  (concat sde-soar-loading-regexp "\\|" sde-soar-chdir-regexp "\\|"
	  sde-soar-returning-regexp)
  "Regexp to match either a loading file message or a chdir message.")

(defvar sde-soar-pwd-output-regexp "Current directory:\\s *\\([^ \t\n]+\\)"
  "Regexp to match Soar reply to pwd commands.")

(defvar sde-soar-first-goal-regexp "^[ \t]*0: ==>G:\\|^=>WM:"
  "Regexp to match sign of Soar starting a run.")

(defvar sde-soar-running-regexp
  (concat sde-soar-sched-regexp "\\|" sde-soar-first-goal-regexp)
  "Regexp to recognize when Soar has starting running without immediately
issuing a prompt.  This happens if the user has a .init file containing load,
run, go or schedule statements.")

(defvar sde-soar-error-regexp
  "^Error: \\|^Internal error: \\|^Expected \\|^No production named \
\\|^Ignoring \\|^There is no \\|^The current [^ \t]+ is not\
\\|^Matches 'level' must be \\|^Illegal argument\\|^Bad argument\
\\|^Invalid argument\\|^Illegal value\\|^'Level' argument must be \
\\|^MS level of detail must be \\|^Warning:\
\\|^That goal stack level doesn't exist right now\
\\|^No RHS function named\\|^Function [^ \t]+ cannot be used\
\\|^Function [ \t]+ can only be used\
\\|^Wrong number of arguments"
  "Regular expression to match error messages from Soar.")

(defvar sde-soar-startup-error-regexp
  "Couldn't exec\\|syntax error"
  "Regexp to match errors in starting up Soar.")

(defvar sde-soar-prompt-str "Soar> "
  "Format string for fake prompts in a single agent process buffer.")

(defvar sde-soar-agent-prompt-fmt "Soar agent %s> "
  "Format string for fake prompts issued in Soar agent interaction buffers.
This should imitate whatever Soar really uses for a prompt string.")

(defvar sde-soar-agent-select-fmt "select-agent %s"
  "Format string for commands to select the current agent in Soar.")

(defvar sde-soar-agents nil
  "Alist of names of Soar agents and their associated buffers.
When running in single-agent mode, this variable is nil.  When running in
multi-agent mode, there are multiple agents and a separate buffer for each
agent, and the first name on the alist is the first agent created.  Format of
elements on the list is \( name .  buffer-name ).")

;; A bit of a hack: leave the next var unbound by default, so that buffers
;; that aren't related to SDE don't have any value at all.  Functions can use
;; that to determine whether the current buffer is of SDE or not.

(defvar sde-soar-buffer-agent)		      
(put   'sde-soar-buffer-agent 'variable-documentation
  "Buffer-local variable indicating the agent responsible for i/o in
current Soar Mode buffer.")

(defvar sde-soar-input-ring nil
  "Input ring for process interaction buffers.")

(defvar sde-soar-input-ring-index
  "Index variable for process input history ring.")

(defvar sde-soar-last-input-match ""
  "Last string searched for by comint input history search, for defaulting.
Buffer local variable.") 

(defvar sde-soar-last-input-end 0
  "Location of last input in buffer.")

(defvar sde-soar-pbreak-list nil
  "List of productions currently under the effect of a pbreak.  Each element of
the list is a quadruples of the form (NAME ORIGINAL AGENT NEW), where NAME is
the name of the production, ORIGINAL is the full original production
definition, AGENT is the name of the agent in which the pbreak is in effect,
and NEW is the new definition of the production according to Soar.")

(defvar sde-soar-output-buffer-name "*Soar output*"
  "Name of pop-up output buffer.")

(defvar sde-soar-output-buffer nil
  "The Soar output buffer.")


;;;-----------------------------------------------------------------------------
;;; 4.  Macros
;;;-----------------------------------------------------------------------------


(defmacro sde-first-agent ()
  ;; Return the first agent's name.
  (` (car (car sde-soar-agents))))


(defmacro sde-first-agent-buffer ()
  ;; Return the first agent's buffer.
  (` (cdr (car sde-soar-agents))))


(defmacro sde-agent-buffer (name)
  (` (cdr (assoc (, name) sde-soar-agents))))


(defmacro sde-within-output-buffer (&rest forms)
  (` (let ((--sde-obuf-- (current-buffer)))
       (set-buffer sde-soar-output-buffer)
       (unwind-protect
	    (progn (,@ forms))
	 (and (not (eq --sde-obuf-- (current-buffer)))
	      (buffer-name --sde-obuf--)
	      (set-buffer --sde-obuf--))))))


(defmacro sde-gmhist-default (item hist-symbol)
  (` (progn
       (put (, hist-symbol) 'default (, item))
       (set (, hist-symbol) (append (list (, item)) (eval (, hist-symbol)))))))



;;;-----------------------------------------------------------------------------
;;; 5.  Soar Mode
;;;-----------------------------------------------------------------------------

;; Soar Mode is patterned after comint and uses the same basic functional key
;; bindings.  Unfortunately nearly all the comint commands needed to be 
;; rewritten because they will not work in a multi-agent Soar environment.
;;
;; The following functions and bindings are taken straight from comint-mode:
;;
;; c-a     comint-bol                      Beginning of line; skip prompt.
;; c-c c-w backward-kill-word    	   Like ^w in shells
;;
;; The following key bindings are the same as comint-mode's but use
;; new functions specific for Soar Mode:
;;
;; M-p      sde-previous-input             Cycle backwards in input history
;; M-n      sde-next-input                 Cycle forwards
;; M-s      sde-previous-similar-input     Previous similar input
;; M-C-r    sde-previous-input-matching    Search backwards in input history
;; C-c C-c  sde-interrupt-soar             Send ^c to Soar
;; C-c C-o  sde-kill-output                Delete last batch of process output
;; C-c C-r  sde-show-output                Show last batch of process output
;; C-c C-u  sde-kill-input                 Like ^u in shells


(defvar sde-soar-mode-map
  (let ((map (full-copy-sparse-keymap sde-mode-map)))
    (define-key map "\C-c\C-w" 'backward-kill-word)
    (define-key map "\C-c\C-u" 'sde-kill-input)
    (define-key map "\C-c\C-r" nil)    
    (define-key map "\C-c\C-r" 'sde-show-output)
    (define-key map "\C-c\C-o" 'sde-kill-output)
    (define-key map "\C-c>"    'sde-forward-prompt)
    (define-key map "\C-c<"    'sde-backward-prompt)

    (define-key map "\C-m"     'sde-return)
    (define-key map "\C-a"     'comint-bol)

    (define-key map "\es"      'sde-previous-similar-input)
    (define-key map "\ep"      'sde-previous-input)
    (define-key map "\en"      'sde-next-input)
    (define-key map "\e\C-r"   'sde-previous-input-matching)
    map)
  "Keymap for Soar Mode.")


;; Soar Mode.

(defun sde-soar-mode ()
  "The Soar interaction component of the Soar Development Environment (SDE).

Typing \\[sde-return] after the end of the process' output sends the text
from the end of the last Soar prompt to the end of the current line.
\\[sde-kill-input] erases the input typed so far (like ^u in Unix shells),
while \\[backward-kill-word] erases the last word typed (like ^w in Unix shells).

\\[sde-interrupt-soar] interrupts whatever Soar is doing currently.

The Soar process buffer has a command history associated with it.
\\[sde-previous-input] and \\[sde-next-input] cycle forward and backward through the input history.
\\[sde-previous-similar-input] searches backward for a past input that has the string
typed so far as a prefix.  The size of the input history is determined
by variable sde-soar-input-ring-size.

All interactive Soar commands are available via ESC x COMMANDNAME.  Some
commands in SDE have no Soar equivalents, e.g., excise-file.  Many key
bindings give direct access to the most common commands.  Some notable
bindings include the following, and a complete list appears later below.  
Use describe-key (\\[describe-key]) \(the standard Emacs help facility) to get
detailed help on each command.

  Controlling Soar execution:

    go                      \\[go]
    schedule                \\[schedule]

  Manipulating Soar memory:

    excise production       \\[excise]
    load file               \\[load-soar]
    init-soar               \\[init-soar]
    
  Tracing and breaking:

    ptrace production       \\[ptrace]
    break production        \\[pbreak]

  Querying Soar for information about specific objects:

    print                   \\[print-soar]
    matches 1               \\[matches-1]	
    matches 2               \\[matches-2]	
    preferences             \\[preferences]	
    wm                      \\[wm]	
    firing-counts           \\[firing-counts]	

  Querying Soar for general information:

    view ptraces            \\[sde-view-ptraces]
    view pbreaks            \\[sde-view-pbreaks]
    view goal stack (pgs)   \\[sde-view-pgs]
    view match set (ms)     \\[sde-view-ms]
    view Soar stats         \\[sde-view-stats]
    view chunks             \\[sde-view-chunks]
    view productions        \\[sde-view-productions]
    view justifications     \\[sde-view-justifications]
    view Soar WM            \\[sde-view-working-memory]

  Finding source code:

    find production source  \\[sde-find-production]
    find next source        \\[sde-find-next-production]

  Controlling multiple agents:
  
    create-agents           \\[create-agents]
    destroy-agents          \\[destroy-agents] (note: currently unimplementable)
    agent-go                \\[agent-go]
    select-agent            \\[select-agent]

  Moving around in the process buffer:

    show last output        \\[sde-show-output]
    move to previous prompt \\[sde-backward-prompt]
    move to next prompt     \\[sde-forward-prompt]

Many Soar commands, such as print, require \"target\" arguments, i.e.,
something to act on.  Those commands that need a production name as target
will try to use the symbol under the cursor, if that looks like a production
name, or the nearest previous symbol that looks like a production name.
Those commands that can take any symbol as target will try to use the symbol
under the cursor regardless if it looks like a production name.  Symbols are
deemed to \"look\" like production names if they consist of alphanumeric
characters preceeded by or followed by at least one '*' character, optionally
followed by more alphanumeric and '*' characters.

Many of the standard Soar commands also take optional arguments -- for
example, the go and print commands.  Giving a prefix argument (such as \\[universal-argument])
to most commands makes them prompt for optional arguments.  Optional
arguments are remembered on history lists.  Typing ESC p and ESC n at a 
prompt moves backward and forward through the history.

Optional arguments are automatically reused upon subsequent invocations of a
command until they are changed.  This allows you to set an option and then
apply a command repeatedly to different target arguments without having to
respecify the optional arguments each time.  For example, you can type
\\[universal-argument] \\[print-soar], type \":depth 3\" to the resulting prompt for optional
arguments, and then repeat this command with the same arguments using
just \\[print-soar] on different targets.

The ptrace (\\[ptrace]) and pbreak (\\[pbreak]) commands have different 
conventions.  Without arguments, ptrace and pbreak act on the production 
under the cursor.  With a positive argument (e.g., such as \\[universal-argument]) they perform 
unptrace or unpbreak, respectively.  With a negative argument (e.g., \\[negative-argument]), 
they undo all ptraces or pbreaks, respectively.  Pbreaks and ptraces are
remembered on a per-agent basis.  You can print out the current
list of active ptraces and pbreaks using the view commands \\[sde-view-ptraces] and
\\[sde-view-pbreaks], respectively.

Commands that only query Soar for information place the output in in a
separate buffer for viewing, unless the variable sde-soar-use-output-buffer
is nil, in which case all output goes to the agent process buffer.  The
output buffer is named \"*Soar output*\", and by default it will brought up
whenever a query command produces output, unless the variable
sde-soar-pop-up-output-buffer is set to nil, in which case you must manually
switch to the \"*Soar output*\" window to view the output.  Also, the output
buffer's existing contents will be cleared before new output is written out,
unless the variable sde-soar-erase-output-buffer is set to nil.

If the popper package is available, Soar Mode will use it for popping up
the output buffer.

The find-production facility locates the source code for a production
whose name is under the cursor or given as argument.  Put the cursor on a
production in your task, either in a Soar process buffer or an editing buffer,
type \\[sde-find-production], and it will jump to the place in your files where the
production is defined.  With a prefix argument, it will prompt for a string
instead of using the production name under the cursor. You don't need to have
created a tags file, you don't even need to have visited the file in question
yet.  It uses several heuristics to locate productions: (1) search through
existing Emacs buffers that are Soar source code buffers; (2) if fail, and a
tag file exists, do tags search; (3) if fail, search through files known to
have been loaded into Soar (it watches Soar's output for signs of file
loading); (4) if fail, search through files found in directories of Soar
files that have been loaded or visited so far.  In addition,
sde-find-production caches information about the location of productions that
it finds, so that subsequent finds on the same productions are much faster.
\\[sde-find-next-production] goes to the next occurrence of the last searched-for production
name string in your files.

ESC x excise-file will prompt for a file name and will excise all of the
productions found in that file.

The cd command, when given to Soar, is watched by Emacs to keep this buffer's
default directory the same as Soar's working directory.  ESC x track-cd-toggle
turns directory tracking on and off.  If Emacs ever gets confused, you can
resync with Soar by just typing 'pwd' in the Soar process buffer.

\\[comint-bol] will move the cursor to the beginning of the current line, and
will skip the Soar prompt if the current line is at a prompt.

\\[sde-return] knows about prompts and Soar productions.  If a production is not
complete, \\[sde-return] will indent it properly.  When an entire sp form is complete,
\\[sde-return] sends it to the Soar process together with a new line.  If you edit
old input, the input will be copied to the end of the buffer first and then
sent to Soar.

\\[sde-backward-prompt] moves the cursor backward to the next previous prompt line in the
current buffer, and displays that line at the top of the window.
\\[sde-forward-prompt] moves the cursor forward to the next prompt line and displays that
line at the top of the window.  These commands are handy for navigating 
through Soar output in the process buffer.

\\[sde-close-all-sp] will close the current sp form by adding or removing right parentheses
as necessary.

\\[backward-delete-char-untabify] converts tabs to spaces as it moves back.

The command history in the process interaction buffer is filtered by the
function on the variable sde-soar-input-ring-filter.  (This variable is
analogous to comint mode's comint-input-filter.)  Commands for which the
function returns true, and that do not match the immediately prior command,
will be added to the input history.  The default function on
sde-soar-input-ring-filter checks that the command is not all whitespace.

Soar Mode is modeled after comint-mode but does not use it directly.
Comint-mode's basic key bindings and functions are available in new forms
adapted for multi-agent Soar interaction.

To send a bug report, questions or other feedback to the authors and maintainers
of the Soar Development Environment, please use ESC x sde-feedback.

Here is a list of all the special bindings in Soar mode.  For further help on
individual commands, type `\\[describe-key] KEY' where KEY is the keystroke.
\\{sde-mode-map}
Customization: 

The name of the key map is sde-soar-mode-map.
Entry to this mode runs the hooks on sde-soar-mode-hook."

  (interactive)
  (let ((old-ring (and (assq 'sde-soar-input-ring (buffer-local-variables))
		       (boundp 'sde-soar-input-ring)
		       sde-soar-input-ring)))
    (sde-mode)
    (setq mode-name "SDE Soar"
	  major-mode 'sde-soar-mode)
    (use-local-map sde-soar-mode-map)
    ;; Set up the mode line for Soar Mode buffers.
    (setq mode-line-modified "--- "
	  mode-line-buffer-identification '("%20b")
	  mode-line-format 
	  (list "" 'mode-line-modified 'mode-line-buffer-identification
		" {"
		'sde-soar-status-string
		"}  "
		'global-mode-string
		" %[("
		'mode-name 'minor-mode-alist
		")%n%] --"
		'(-3 . "%p") "-%-"))
    ;; Comint.  We only need this for comint-bol.
    (make-local-variable 'comint-prompt-regexp)
    (setq comint-prompt-regexp sde-soar-prompt-regexp)
    ;; Soar
    (make-local-variable 'sde-soar-buffer-agent)
    (make-local-variable 'sde-soar-last-input-match)
    (make-local-variable 'sde-soar-last-input-end)
    (make-local-variable 'sde-soar-input-ring-size)
    (make-local-variable 'sde-soar-input-ring)
    (make-local-variable 'sde-soar-input-ring-index)
    (setq sde-soar-buffer-agent nil
	  sde-soar-last-input-match  ""
	  sde-soar-input-ring-index 0
	  sde-soar-last-input-end 0)
    ;; Run hooks first and then create input ring, to allow user to
    ;; set the SDE-SOAR-INPUT-RING-SIZE in the hook.  The test is to prevent
    ;; loosing history if sde-soar-mode is run twice in a buffer.
    (run-hooks 'sde-soar-mode-hook)
    (setq sde-soar-input-ring (if (ring-p old-ring) old-ring
				  (make-ring sde-soar-input-ring-size)))))



;;;-----------------------------------------------------------------------------
;;; 6.  Interactive Soar commands
;;;
;;; The various interactive Soar commands are very idiosyncratic in their use
;;; of arguments.  For the purposes of implementing a sensible interface to
;;; them in Emacs, I divided arguments into two categories: target arguments
;;; and optional arguments.  "Target arguments" are arguments that a Soar
;;; command directly acts on.  For example, print needs a target argument.
;;; All other arguments are "optional arguments"; they modify the behavior of
;;; a command.  For example, print takes optional args ":depth" and
;;; ":internal".
;;;
;;; The interface implemented here attempts to remember optional args from
;;; one invocation of a command to the next, so that if in subsequent
;;; invocations an optional arg is not given to a command, the optional args
;;; from the last invocation are reused.  Target arguments are never
;;; remembered.  The idea that a user is likely to want to set some behavior
;;; a command and then execute that same command on several different objects
;;; before changing the optional args.
;;;
;;; Associated with each command that can take an optional argument is a
;;; variable that the user can set to a string that is to be the initial
;;; value of the argument (i.e., before a new value is supplied in a
;;; session).  These variables are "sde-go-args", etc., in Section ii at the
;;; top of this file.
;;;
;;; The general scheme for supplying arguments summarized in the following
;;; rules; the rules are not mutually exclusive:
;;;
;;; 1) If a command takes optional args, then:
;;;
;;;      a) If this is the first time the command is invoked and no prefix
;;;         argument is given, the value of the command's associated
;;;         "sde-xxx-args" variable is used as the argument.  If this
;;;         variable is nil, the command is issued to Soar without any
;;;         arguments and Soar will do whatever its default behavior is.
;;;
;;;      b) If given a prefix argument, it prompts for the arguments to be
;;;         passed to the corresponding Soar command.  (Exception: ptrace,
;;;         pbreak, and firing-counts have different conventions.)
;;;
;;;      c) If no prefix argument is given, then the last optional argument
;;;         given to that particular command is used.
;;;
;;; 2) If a command takes target args, then the symbol under the cursor or
;;;    the nearest symbol left of the cursor is examined as a potential
;;;    target. 
;;;
;;;      a) If the command needs a production name, the symbol is
;;;         checked to see if it looks like a production name.  If so,
;;;         it is automatically used as the target argument.  If it does
;;;         not seem like a production name, the user is prompted for a
;;;         production name.
;;;
;;;      b) If the command just needs any symbol (e.g., an id or "<o>"
;;;         or whatever), then the symbol under the cursor is
;;;         automatically used.
;;;
;;; Note that all commands can be called from other elisp functions, and
;;; nearly all take as first argument the destination agent name.
;;;
;;; The output filter sde-soar-output-filter (see code section above) watches
;;; commands typed directly to the Soar process and remembers the optional
;;; arguments.  Thus, if a user types a command directly in the process
;;; window and subsequently uses the key binding for that command, the same
;;; optional args are reused.  The goal is to maximize consistency.
;;;
;;; The general output behavior of commands is as follows:
;;;
;;; 1. All commands are echoed in the minibuffer, to provide feedback to the
;;;    user.
;;;
;;; 2. When sde-soar-use-output-buffer is t, the output produced by query
;;;    commands is caught and displayed in the Soar output buffer; nothing
;;;    appears in the process buffer.  Query commands are things like print,
;;;    etc.  All other commands' output is shown directly in the process
;;;    buffer.
;;;
;;;    Exceptions:  pbreak, select-agent, create-agents are special.
;;;
;;; 3. The commands for execution control always switch the user to the
;;;    process buffer (because they are important to understanding the current
;;;    state of Soar).
;;;
;;;-----------------------------------------------------------------------------

;; Support for command histories for the interactive Soar commands.

(defvar sde-go-hist		     nil)
(defvar sde-run-hist		     nil)
(defvar sde-matches-hist	     nil)
(defvar sde-ms-hist		     nil)
(defvar sde-firing-counts-hist	     nil)
(defvar sde-print-hist		     nil)
(defvar sde-preferences-hist	     nil)
(defvar sde-list-productions-hist    nil)
(defvar sde-list-chunks-hist	     nil)     
(defvar sde-list-justifications-hist nil)
(defvar sde-agent-go-hist	     nil)
(defvar sde-schedule-hist            nil)


(defmacro sde-reset-hist (hist-symbol hist-initial)
  (` (progn
       (set (quote (, hist-symbol)) nil)
       (sde-gmhist-default (, hist-initial) (quote (, hist-symbol))))))


(defun sde-soar-reset-histories ()
  ;; Reset to initial defaults all of the Soar command arg history lists.
  (sde-reset-hist sde-go-hist		       sde-go-args)
  (sde-reset-hist sde-run-hist		       sde-run-args)
  (sde-reset-hist sde-matches-hist	       sde-matches-args)
  (sde-reset-hist sde-ms-hist		       sde-ms-args)
  (sde-reset-hist sde-firing-counts-hist       sde-firing-counts-args)
  (sde-reset-hist sde-print-hist	       sde-print-args)
  (sde-reset-hist sde-preferences-hist	       sde-preferences-args)
  (sde-reset-hist sde-list-productions-hist    sde-list-productions-args)
  (sde-reset-hist sde-list-chunks-hist	       sde-list-chunks-args)
  (sde-reset-hist sde-list-justifications-hist sde-list-justifications-args)
  (sde-reset-hist sde-agent-go-hist	       sde-agent-go-args)
  (sde-reset-hist sde-schedule-hist	       sde-schedule-args))


(defmacro sde-args (&rest rest)
  ;; Used in calls to (interactive) to check Soar is okay before attempting
  ;; to prompt or extract arguments.
  (` (progn (sde-check-soar) (list (,@ rest)))))



;;;
;;; 6.A.  Commands for execution control
;;;
;;; All commands that start & stop executing are here.
;;;


(defvar sde-soar-program-args nil)
(defvar sde-soar-dir-args     nil)

(defun soar (&optional dir program)
  "Run Soar in a buffer.
Unless given a prefix argument, starts the Soar program indicated by the
variable `sde-soar-program' in the directory indicated by the variable
`sde-soar-starting-directory'.  The pathname `sde-soar-program' is relative
to the directory `sde-soar-starting-directory', or it can be an absolute
pathname.  If given a prefix argument, prompts for a directory to cd to, and
the pathname of the Soar program, before starting Soar.  If no prefix
argument is given, and one or both of `sde-soar-program' and
`sde-soar-starting-directory' are not set, prompts for the missing values.

Do a describe-mode command (\\[sde-describe-mode]) in the Soar buffer once Soar
is running to get help on Soar Mode."
  (interactive)
  (if (sde-soar-is-alive)
      (sde-pop-to-buffer (if sde-soar-agents
			     (sde-first-agent-buffer)
			     sde-soar-buffer))
      (let ((default-dir (file-name-as-directory
			  (substitute-in-file-name
			   (or dir sde-soar-starting-directory default-directory))))
	    (program (or program sde-soar-program "")))
	(if (or current-prefix-arg
		(and (null dir) (null sde-soar-starting-directory)))
	    (setq dir (read-file-name-with-history-in
		       'sde-soar-dir-args "Starting directory: "
		       default-dir default-dir)))
	(setq dir (file-name-as-directory
		   (substitute-in-file-name (or dir default-dir))))
	(if (or current-prefix-arg (string= program ""))
	    (setq program (read-file-name-with-history-in
			   'sde-soar-program-args "Program to run: "
			   dir program)))
	;; read-file-name doesn't check for files vs. directories, so we must. 
	(if (file-directory-p program)
	    (error "A directory does not make sense as a program name."))
	(setq program (expand-file-name (substitute-in-file-name program) dir))
	(sde-start-soar program dir))))


(defun go (&optional agent args)
  "Execute Soar's \"go\" command.
If given a prefix argument, prompts for optional arguments to give to \"go\".
If not given a prefix argument, uses the same argument given to this command
the last time it was invoked.

This is the most general command for running Soar.  It takes two optional
arguments, one specifying how many things to run, and one specifying what
type of things to run.  The following types are available:

p  - run Soar for n phases.  A phase is either an input phase, preference
     phase, working memory phase, output phase, or quiescence phase.
e  - run Soar for n elaboration cycles.  (For purposes of this command,
     quiescence phase is counted as an elaboration cycle.)
d  - run Soar for n decision cycles.
g  - run Soar until the nth time a goal is selected.
ps - run Soar until the nth time a problem space is selected.
s  - run Soar until the nth time a state is selected.
o  - run Soar until the nth time an operator is selected.
context-variable - run Soar until the nth time a selection is made for that
    particular context slot, or until the context stack pops to above that
    context.

Examples:
  (go 5 d)   --> run for 5 decision cycles
  (go e)     --> run for another 5 elaboration cycles
  (go 1 g)   --> run until the next goal is selected (i.e., until the next
                 time an impasse arises)
  (go <so>)  --> run until the next superoperator is selected (or until the
                 supergoal goes away)
  (go 3 <o>) --> run for 3 operator selections at this level (continuing
                 through any subgoals that arise)
"
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'go': " 'sde-go-hist)))
  (sde-soar-cmd agent (concat "go " args) nil nil t))


(defun run (&optional agent args)
  "Execute Soar's \"run\" command.  
If given a prefix argument, prompts for optional arguments to give to
\"run\".  If not given a prefix argument, uses the same argument given to
this command the last time it was invoked.

This command runs Soar for that number of elaboration cycles.  (For this
command, quiescence phase is counted as an elaboration cycle.)  With no
arguments, this runs Soar forever (or until Soar halts, receives an
interrupt, etc.)."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'run': " 'sde-run-hist)))
  (sde-soar-cmd agent (concat "run " args) nil nil t))



;;;
;;; 6.B.  Commands for tracing and breaking
;;; 


(defun pbreak (&optional agent name arg)
  "Cause a production to issue a break.  I.e., redefine the production to
interrupt Soar when it fires.  Uses as target the name of the production that
the cursor is in, or the next-previous production, in the current buffer.  If
given a positive prefix arg, undoes the pbreak on the production.  If given a
negative prefix arg, undoes all currently active pbreaks.  Pbreaks are
remembered on a per-agent basis, so pbreak'ing a production in one agent will
not automatically pbreak that production in any other agents that may also
share the same production.  To list the currently active pbreaks, use
\\[sde-view-pbreaks]"
  (interactive)
  (sde-check-soar)
  (let ((agent (or agent (sde-agent))))
    (cond ((or (and arg (< arg 0))
	       (sde-minus-prefix-arg))	; Undo all pbreaks.
	   (sde-unpbreak-all)
	   (message "All pbreaks disabled."))
	  ((or arg current-prefix-arg)	; Undo one pbreak.
	   (let ((name (or name (sde-sp-name-near-point))))
	     (if (sde-pbreak-in-effect agent name)
		 (progn
		   (sde-unpbreak-production agent name)
		   (message "unpbreak %s" name))
		 (error "Production \"%s\" is not under the effect of a pbreak." name))))
	  (t				; Pbreak current production
	   (let ((name (or name (sde-sp-name-near-point))))
	     (if (sde-pbreak-in-effect agent name)
		 (error "Production \"%s\" is already being pbreak'ed." name)
		 (progn
		   (sde-pbreak-production agent name)
		   (message "pbreak %s" name))))))))


(defun ptrace (&optional agent name arg)
  "Execute the Soar \"ptrace\" command.
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer.  If given a positive prefix
arg, undoes the effects of a ptrace on the production.  If given a negative
prefix arg, disables all currently active ptraces.  To list all the currently
active ptraces, use \\[sde-view-ptraces]

Ptrace enables and disables tracing the firings and retractions of individual
productions.  (This mechanism is orthogonal to the watch :firings mechanism.)
Tracing persists until disabled by an unptrace command, or until the
production is excised."
  (interactive)
  (sde-check-soar)
  (let ((agent (or agent (sde-agent))))
    (cond ((or (and arg (< arg 0))
	       (sde-minus-prefix-arg))	; Unptrace all.
	   (sde-soar-query (sde-agent) "unptrace"))
	  ((or arg current-prefix-arg)	; Unptrace one production.
	   (let ((name (or name (sde-sp-name-near-point))))
	     (sde-soar-cmd (sde-agent) (concat "unptrace " name))))
	  (t				; Ptrace current production.
	   (let ((name (or name (sde-sp-name-near-point))))
	     (sde-soar-cmd (sde-agent) (concat "ptrace " name)))))))



;;;
;;; 6.C.  Commands for querying Soar
;;; 
;;; This includes the following Soar commands:
;;;   pgs                     print, p, spr, wm
;;;   ms                      preferences
;;;   firing-counts           stats, memory-stats, rete-stats
;;;   matches                 watch, stack-trace-format, object-trace-format
;;;   list-chunks, -productions, -justifications
;;;
;;; Additional special commands implemented:
;;;   print-soar-working-memory
;;;   


(defun firing-counts (&optional agent args)
  "Execute the Soar \"firing-counts\" command.
This command prints how many times certain productions have fired.  With no
prefix argument, executes 'firing-counts' on the production under or immediately
before the cursor.  With prefix argument, prompts for a number (call it k)
and does `firing-counts k'.  If an integer argument (k) is given, only the
top k productions are listed.  If k=0, only the productions which haven't
fired at all are listed.  Note that firing counts are not reset by an
\(init-soar), the counts indicate the number of firings since the productions
were loaded or built.

Note:  this is slow, because the sorting takes time O(n*log n)."
  (interactive (sde-args
		 (sde-agent)
		 (if current-prefix-arg
		     (sde-prompt "Integer for 'firing-counts n': "
				 'sde-firing-counts-hist)
		     (sde-sp-name-near-point))))
  (sde-soar-query agent (concat "firing-counts " args)))


(defun list-chunks (&optional agent args)
  "Execute the Soar \"list-chunks\" command.
If given a prefix arg, prompts for optional arguments to list-chunks.

List-chunks lists all the chunks in the system.

The optional [:internal] argument tells Soar to print productions in their
internal reordered (RETE net) form.

If a filename is given, the productions are printed to that file, otherwise
they are printed to the screen.  If :append is given, the file is appended
to, rather than overwritten."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'list-chunks': "
			     'sde-list-chunks-hist)))
  (sde-soar-query agent (concat "list-chunks " args)))


(defun list-justifications (&optional agent args)
  "Execute the Soar \"list-justifications\" command.
If given a prefix arg, prompts for optional arguments to list-justifications.

List-justifications lists all the justifications in the system.

The optional [:internal] argument tells Soar to print productions in their
internal reordered (RETE net) form.

If a filename is given, the productions are printed to that file, otherwise
they are printed to the screen.  If :append is given, the file is appended
to, rather than overwritten."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'list-justifications': "
			     'sde-list-justifications-hist)))
  (sde-soar-query agent (concat "list-justifications " args)))
  

(defun list-productions (&optional agent args)
  "Execute the Soar \"list-productions\" command.  
If given a prefix arg, prompts for optional arguments to list-productions.

List-productions lists all the productions in the system.

The optional [:internal] argument tells Soar to print productions in their
internal reordered (RETE net) form.

If a filename is given, the productions are printed to that file, otherwise
they are printed to the screen.  If :append is given, the file is appended
to, rather than overwritten."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'list-productions': "
			     'sde-list-productions-hist)))
  (sde-soar-query agent (concat "list-productions " args)))  


(defun matches (&optional agent name args)
  "Execute the Soar \"matches\" command.
Print partial match information for a production.  By default, uses the name
of the production that the cursor is in.  If given a prefix argument, prompts
for an additional argument to specify the level of detail wanted:

  0 (default) prints out just the partial match counts
  1 also prints the timetags of wmes at the first failing condition
  2 prints the wmes rather than just their timetags.

If not given a prefix argument, uses the same level argument given to this
command the last time it was invoked."
  (interactive (sde-args
		 (sde-agent)
		 (sde-sp-name-near-point)
		 (sde-prompt "Arguments to 'matches': " 'sde-matches-hist)))
  (sde-soar-query agent (concat "matches " name " " args)))


(defun matches-1 (&optional agent name)
  "Execute the Soar \"matches\" command with a \"level\" argument of 1.  
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer."
  (interactive (sde-args (sde-agent) (sde-sp-name-near-point)))
  (sde-soar-query agent (concat "matches " name " 1")))


(defun matches-2 (&optional agent name)
  "Execute the Soar \"matches\" command with \"level\" argument of 2.  
Use as target the name of the production that the cursor is in, or the
next-previous production, in the current buffer."
  (interactive (sde-args (sde-agent) (sde-sp-name-near-point)))
  (sde-soar-query agent (concat "matches " name " 2")))


(defun memory-stats (&optional agent)
  "Execute the Soar \"memory-stats\" command.
This command prints out statistics on memory usage.  See also
\"rete-stats\" and \"stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "memory-stats"))


(defun ms (&optional agent args)
  "Execute the Soar \"ms\" command.
This command prints the current \"match set\", i.e., a list of productions
that are about to fire or retract in the next preference phase.  Given a
prefix argument, prompts for an optional argument to give to 'ms'; without a
prefix, reuses the last-given optional argument.  The optional integer
specifies the level of detail wanted: 0 (the default) prints out just the
production names, 1 also prints the timetags of wmes matched, and 2 prints
the wmes rather than just their timetags."
  (interactive (sde-args
		 (sde-agent)
		 (sde-prompt "Arguments to 'ms': " 'sde-ms-hist)))
  (sde-soar-query agent (concat "ms " args) ))


(defun pgs (&optional agent)
  "Pgs (\"print goal stack\") prints Soar's current context stack."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "pgs"))


(defun preferences (&optional agent symbol attrib level)
  "Execute the Soar \"preferences\" command.
Use as target argument the symbol under the cursor, or the symbol nearest
the cursor from the left.  Always prompts for the attribute.  If given a
prefix argument, prompts for optional \"level\" argument to \"preferences\".

This command prints all the preferences for the slot given by the \"id\" and
\"attribute\" arguments.  The optional \"level\" argument must be 0, 1, 2, or 3
\(0 is the default) and it indicates the level of detail requested:
  level 0 -- prints just the preferences themselves
  level 1 -- also prints the names of the productions that generated them
  level 2 -- also prints the timetags of the wmes matched by the productions
  level 3 -- prints the whole wmes, not just their timetags."
  (interactive (sde-args
		 (sde-agent)
		 (sde-symbol-near-point)
		 (read-string "Attribute: ")
		 (sde-prompt "Level of detail (0,1,2,3): " 'sde-preferences-hist)))
  (sde-soar-query agent (concat "preferences " symbol " " attrib " " level)))


(defun print-soar (&optional agent symbol args)
  "Execute the Soar \"print\" command.
Use as target the symbol under the cursor, or the symbol nearest the cursor
from the left.  If given a prefix argument, prompts for optional arguments to
\"print\".

The print command is used to print items from production memory or working
memory.  It can take several kinds of arguments:

  arg ::= production-name  (print that production)
  arg ::= identifier       (id of the object to print)
  arg ::= integer          (timetag of wme--the identifier from the wme
                            indicates the object to be printed)
  arg ::= pattern          (pattern--same as if you listed as arguments
                            the timetags of all wmes matching the pattern)

  pattern ::= ( {identifier | '*'} ^ { attribute | '*'} { value | '*' } [+])

The optional [:depth n] argument overrides default-print-depth.

The optional [:internal] argument tells Soar to print things in their
internal form.  For productions, this means leaving conditions in their
reordered (rete net) form.  For wmes, this means printing the individual
wmes with their timetags, rather than the objects.

:depth 0 is meaningful only for integer and pattern arguments, and only
when used along with :internal.  It causes just the matching wmes to be
printed, instead of all wmes whose id is an id in one of the matching wmes."
  (interactive (sde-args
		 (sde-agent)
		 (sde-symbol-near-point)
		 (sde-prompt "Arguments to 'print': " 'sde-print-hist)))
  (sde-soar-query agent (concat "print " args " " symbol)))


(defun print-stats (&optional agent)
  "Execute the Soar \"print-stats\" command.  This command prints out some
statistics on the current Soar run.  See also:  \"memory-stats\",
\"rete-stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "print-stats"))


(defun rete-stats (&optional agent)
  "Execute the Soar \"rete-stats\".  This command prints out statistics on
the rete net.  See also:  \"memory-stats\", \"stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "rete-stats"))


(defun wm (&optional agent symbol)
  "Execute the Soar \"wm\" command, which is shorthand for
     \"print :depth 0 :internal ...\"."
  (interactive (sde-args (sde-agent) (sde-symbol-near-point)))
  (sde-soar-query agent (concat "wm " symbol)))


;;;
;;; The view commands.  These are query commands grouped together to
;;; provide a more common interface.  Some are identical to other commands.
;;; 


(fset 'sde-view-chunks         'list-chunks)
(fset 'sde-view-pgs            'pgs)
(fset 'sde-view-justifications 'list-justifications)
(fset 'sde-view-ms             'ms)
(fset 'sde-view-productions    'list-productions)


(defun sde-view-stats (&optional agent)
  "View the output of \"print-stats\", \"memory-stats\" and \"rete-stats\"."
  (interactive (sde-args (sde-agent)))
  (sde-init-output-buffer)
  (sde-within-output-buffer
    (insert (sde-soar-silent-cmd agent "print-stats"))
    (insert (sde-soar-silent-cmd agent "memory-stats"))
    (insert (sde-soar-silent-cmd agent "rete-stats"))
    (if sde-soar-pop-up-output-buffer
	(sde-show-output-buffer))))


(defun sde-view-pbreaks (&optional agent)
  "List all the productions currently under the effect of a pbreak."
  (interactive (sde-args (sde-agent)))
  (sde-print-pbreak-list))


(defun sde-view-ptraces (&optional agent)
  "List all the productions currently under the effect of a ptrace."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "ptrace"))


(defun sde-view-working-memory (&optional agent)
  "Print the contents of working memory, as would \"print (* ^ * *)\"."
  (interactive (sde-args (sde-agent)))
  (sde-soar-query agent "print (* ^ * *)"))



;;;
;;; 6.D.  Commands for manipulating Soar memories
;;; 

(defun excise (&optional agent name)
  "Execute the Soar \"excise\" command on the production under the cursor.
This removes the given production from Soar's memory.  
See also: \"excise-chunks\", \"excise-task\", \"excise-all\"." 
  (interactive (sde-args (sde-agent) (sde-sp-name-near-point)))
  (sde-soar-cmd agent (concat "excise " name)))


(defun excise-chunks (&optional agent)
  "Execute the Soar \"excise-chunks\" command.
This command removes all chunks and justifications from the system.
See also: \"excise\", \"excise-task\", \"excise-all\"." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "excise-chunks"))


(defun excise-task (&optional agent)
  "Execute the Soar \"excise-task\" command. This command removes all
non-default productions from the system.  It also does an (init-soar).
See also: \"excise\", \"excise-chunks\", \"excise-all\"." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "excise-task"))


(defun excise-all (&optional agent)
  "Execute the Soar \"excise-all\" command.  This command removes all
productions from the system.  It also does an (init-soar).
See also: \"excise\", \"excise-chunks\", \"excise-task\"." 
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "excise-all"))


(defvar sde-soar-last-dir/file nil
  "List of previously accessed files in Soar process or edit buffers.")

(defun excise-file (&optional agent file)
  "Excise the productions found in a file.  Prompts for file name.
This function searches the given file for all production definitions and
executes the Soar \"excise\" command on them in the current Soar agent."
  (interactive (sde-args
		 (sde-agent)
		 (car (comint-get-source "Excise file: " sde-soar-last-dir/file
					 sde-source-modes t))))
  (comint-check-source file)
  ;; Read file into Emacs, but don't bother setting mode, etc.  Loop,
  ;; searching for production names, create a list of them.  Then send excise
  ;; command to Soar.
  (let (buffer excise-list i names msg)
    (unwind-protect 
	 (save-excursion
	   (setq buffer (create-file-buffer file))
	   (set-buffer buffer)
	   (erase-buffer)
	   (if (condition-case ()
		   (insert-file-contents file t)
		 (file-error nil))
	       (progn
		 (goto-char (point-min))
		 (message "Scanning file %s..." file)
		 (while (re-search-forward sde-sp-name-regexp nil t)
		   (setq excise-list (cons (sde-buffer-substring 1) excise-list)))
		 (if (null excise-list)
		     (error "No productions found in %s" file))
		 (message "Excising file %s..." file)
		 (setq msg (format "Excising file %s " file))
		 ;; Soar cannot accept lines longer than 1000 chars.  Have
		 ;; to break up excises into small groups.  Some people have very
		 ;; long production names.  Groups of 10 seems reasonable.
		 (while excise-list
		   (setq i 0
			 names "")
		   (while (and (< i 10) excise-list)
		     (setq i           (+ i 1)
			   names       (concat names " " (car excise-list))
			   excise-list (cdr excise-list)))
		   (sde-soar-cmd agent (concat "excise" names)
				 nil nil nil nil msg)
		   (setq msg 'no-message)
		   (sde-soar-wait))
		 (message "Excising file %s...Done" file))))
      (kill-buffer buffer))))


(defun init-soar (&optional agent)
  "Execute Soar's \"init-soar\" command.  Takes no arguments.
This command re-initializes Soar.  It removes all wmes from working memory,
wiping out the goal stack, and resets all statistics (except the counts of
how many times each individual production has fired, used by the
\"firing-counts\" command)."
  (interactive (sde-args (sde-agent)))
  (sde-soar-cmd agent "init-soar"))


(defun load-soar (&optional agent file switch)
  "Invoke the Soar \"load\" command on a file.  Prompts for the file name.
Automatically switches to the process buffer unless given a prefix argument.

The load command tells Soar to read commands from the given file instead of
the keyboard.  Soar will read and execute each command in the file, and then
go back to the keyboard.  Loads may be nested, i.e., the given file may
contain a command to load another file, and so on."
  (interactive (sde-args
		 (sde-agent)
		 (car (comint-get-source "Load file: " sde-soar-last-dir/file
					 sde-source-modes t))
		 (and current-prefix-arg t)))
  (comint-check-source file)
  (sde-soar-cmd agent (concat "load \"" file "\"")
		nil nil			   ; wait-p dest
		(not switch)		   ; switch
		nil nil			   ; m-msg b-msg
		(list 'sde-update-pbreaks 'sde-soar-record-loads)))


;;;
;;; 6.E.  Commands for dealing with multiple agents
;;; 

(defun select-agent (&optional agent-name)
  "Select an agent to be the current recipient of commands in this buffer."
  (interactive)
  (sde-check-soar)
  (cond ((null sde-soar-agents)
	 (sde-error-soar-not-multi-agent))
	((eq major-mode 'sde-soar-mode)
	 ;; Don't allow the user to reset a process buffer's agent.
	 ;; Instead, switch the agent buffer.
	 (switch-to-buffer (sde-agent-buffer (sde-ask-for-agent))))
	((boundp 'sde-soar-buffer-agent)
	 ;; A mode derived from Soar Mode.  It's okay to set an agent
	 (if (null agent-name)
	     (setq agent-name (sde-ask-for-agent)))
	 (setq sde-soar-buffer-agent agent-name)
	 (message "Agent %s selected." agent-name))
	(t
	 ;; We're not in a Soar Mode-based mode.  Makes no sense.
	 (error "Cursor is not in a buffer that communicates with Soar."))))


(defun schedule (&optional cycles)
  "Execute the Soar \"schedule\" command in multi-agent Soar.
If given a prefix argument, prompts for optional argument specifying number of
times to run each agent."
  (interactive (progn
		 (sde-check-soar-multi-agent)
		 (sde-args (sde-prompt "Cycles to run: " 'sde-schedule-hist))))
  (sde-soar-cmd (sde-first-agent) (concat "schedule " cycles) nil nil t))


(defun agent-go (&optional agent args)
  "Execute the Soar \"agent-go\" command.  Uses the current buffer's agent
by default; if given a command prefix, prompts the user for an agent name
as well as the go arguments.

This command operates exactly the same as the go command, EXCEPT that
this only defines the go settings for a agent when it is selected to
run by the multi-agent scheduler.  Hence, the go settings are defined
for subsequent runs, but no runs are made when the command is read.
See also:  \"go\", \"run\"."
  (interactive (progn
		 (sde-check-soar-multi-agent)
		 (sde-args
		   (if current-prefix-arg
		       (sde-ask-for-agent)
		       (sde-agent))
		   (read-with-history-in 'sde-agent-go-hist
					 "Arguments to 'agent-go': "))))
  (sde-soar-cmd agent (concat "agent-go " agent " " args)))


(defun create-agents (&optional agents)
  "Execute the Soar \"create-agents\" command.  Prompts for the names of the 
agents.  This command creates Soar agents identified by the given names, 
and also creates interaction buffers for each agent."
  (interactive (progn
		 (sde-check-soar-multi-agent)
		 (sde-args (read-string "Agent names: "))))
  (sde-soar-cmd (sde-first-agent) (concat "create-agents " agents))
  (sde-create-agents-from-string agents)
  (if (string= "*control*" (buffer-name (current-buffer)))
      (progn
	(bury-buffer (current-buffer))
	(switch-to-buffer (sde-first-agent-buffer)))))


(defun destroy-agents (&rest agents)
  "The destroy-agents command is currently unimplementable in Soar."
  (interactive)
  (sde-check-soar)
  (message "Sorry, Soar does not yet implement destroy-agents.  Harrass soar-bugs."))


(defun sde-create-agents-from-string (agent-names)
  (let ((begin 0))
    (while (string-match "\\(\\s_\\|\\sw\\)+\\s *" agent-names begin)
      ;; Note that default-directory is taken from caller's current buffer.
      (sde-new-agent (sde-substring agent-names 1) default-directory)
      (setq begin (match-end 0)))))


;;;
;;; 6.F.  Help commands
;;; 


(defun sde-soar-command-menu ()
  (interactive)
  (sde-check-soar)
  (message "Sorry, the help facility is not implemented yet."))


(defun soarnews ()
  "Invoke the Soar \"soarnews\" command and place the result in a buffer.
This command prints news about the current release of Soar."
  (interactive)
  (sde-check-soar)
  (sde-soar-query (sde-agent) "soarnews"))



;;;
;;; 6.G.  Misc. commands
;;; 

(defun sde-switch-to-soar (eob-p)
  "Switch to the current Soar process.  
If already in a Soar process buffer, switch back to the buffer that last
switched to the process buffer.  With argument, position cursor at end of
buffer.  In cases where it is hard to determine from which buffer the user
last switched to the process buffer (e.g., because sde-switch-to-soar wasn't
used to do it) it will heuristically attempt to find the last-used SDE source
buffer."
  (interactive "P")
  (sde-check-soar)
  (if (eq major-mode 'sde-soar-mode)
      (if (and sde-last-buffer (not (equal sde-last-buffer (current-buffer))))
	  ;; In a soar buffer and we know where we were last.
	  (sde-pop-to-buffer sde-last-buffer)
	  ;; In a soar buffer but we don't know where we were last.  Punt.
	  (sde-pop-to-buffer (or (sde-previous-sde-buffer)
				 (other-buffer (current-buffer)))))
      ;; Not in a Soar mode buffer.
      (progn
	(setq sde-last-buffer (current-buffer))
 	(sde-show-buffer (or (sde-agent-buffer (sde-agent)) sde-soar-buffer) t)
 	(if eob-p
	    (goto-char (point-max))))))


(defun sde-previous-sde-buffer ()
  ;; Return the first buffer after this buffer in the buffer list that is
  ;; an SDE mode buffer.
  (let ((buffers (sde-get-sde-buffers)))
    (if (equal (current-buffer) (car buffers))
	(cdr buffers)
	(car buffers))))


;;;-----------------------------------------------------------------------------
;;; 7.  Command line interface in Soar process buffers.
;;;-----------------------------------------------------------------------------

;;; Input history.
;;;
;;; These are modifications of commands from comint.el.  The comint mode
;;; commands can't be used directly since they assume that there is only
;;; one buffer associated with a process.  In Soar Mode we may have multiple
;;; Soar agents, hence multiple pseudo-process-buffers in which we want these
;;; commands available.
;;;
;;; M-p     sde-previous-input             Cycle backwards in input history
;;; M-n     sde-next-input                 Cycle forwards
;;; M-s     sde-previous-similar-input     Previous similar input
;;; M-C-r   sde-previous-input-matching    Search backwards in input history


(defun sde-previous-input (arg)
  "Cycle backwards through input history."
  (interactive "*p")
  (let ((len (ring-length sde-soar-input-ring)))
    (cond ((<= len 0)
	   (message "Empty input ring.")
	   (ding))
	  ((< (point) (marker-position sde-soar-buffer-mark))
	   (message "Not after process mark.")
	   (ding))
	  (t
	   (cond ((eq last-command 'sde-previous-input)
		  (delete-region (mark) (point)))
		 ((eq last-command 'sde-previous-similar-input)
		  (delete-region (marker-position sde-soar-buffer-mark) (point)))
		 (t                          
		  (setq sde-soar-input-ring-index
			(if (> arg 0) -1
			    (if (< arg 0) 1 0)))
		  (push-mark (point))))
	   (setq sde-soar-input-ring-index (comint-mod (+ sde-soar-input-ring-index arg) len))
	   (message "%d" (1+ sde-soar-input-ring-index))
	   (insert (ring-ref sde-soar-input-ring sde-soar-input-ring-index))
	   (setq this-command 'sde-previous-input)))))

	 
(defun sde-next-input (arg)
  "Cycle forwards through input history."
  (interactive "*p")
  (sde-previous-input (- arg)))


(defun sde-previous-input-matching (str)
  "Searches backwards through input history for substring match."
  (interactive (let* ((last-command last-command) ; Preserve around r-f-m.
		      (s (read-from-minibuffer 
			  (format "Command substring (default %s): "
				  sde-soar-last-input-match))))
		 (list (if (string= s "") sde-soar-last-input-match s))))
  (setq sde-soar-last-input-match str)	; Update default.
  (if (not (eq last-command 'sde-previous-input))
      (setq sde-soar-input-ring-index -1))
  (let ((str (regexp-quote str))
        (len (ring-length sde-soar-input-ring))
	(n (+ sde-soar-input-ring-index 1)))
    (while (and (< n len) (not (string-match str (ring-ref sde-soar-input-ring n))))
      (setq n (+ n 1)))
    (cond ((< n len)
	   (sde-previous-input (- n sde-soar-input-ring-index)))
	  (t (if (eq last-command 'sde-previous-input) 
		 (setq this-command 'sde-previous-input))
	     (message "Not found.")
	     (ding)))))


(defvar sde-last-similar-string "" 
  "The string last used in an sde-previous-similar-input search.")

(defun sde-previous-similar-input (arg)
  "Reenters the last input that matches the string typed so far.  If repeated
successively older inputs are reentered.  If arg is 1, it will go back in the
history, if -1 it will go forward."
  (interactive "p")
  (if (< (point) (marker-position sde-soar-buffer-mark))
      (error "Not after process mark."))
  (if (not (eq last-command 'sde-previous-similar-input))
      (setq sde-soar-input-ring-index -1
	    sde-last-similar-string (buffer-substring
				     (marker-position sde-soar-buffer-mark)
				     (point))))
  (let* ((size (length sde-last-similar-string))
	 (len (ring-length sde-soar-input-ring))
	 (n (+ sde-soar-input-ring-index arg))
	 entry)
    (while (and (< n len) 
		(or (< (length (setq entry (ring-ref sde-soar-input-ring n))) size)
		    (not (equal sde-last-similar-string 
				(substring entry 0 size)))))
      (setq n (+ n arg)))
    (cond ((< n len)
	   (setq sde-soar-input-ring-index n)
	   (if (eq last-command 'sde-previous-similar-input)
	       (delete-region (mark) (point)) ; repeat
	       (push-mark (point)))	      ; 1st time
	   (insert (substring entry size)))
	  (t (message "Not found.") (ding) (sit-for 1)))
    (message "%d" (1+ sde-soar-input-ring-index))))


;;; Direct I/O in Soar buffers.
;;;
;;; C-c C-c sde-interrupt-soar
;;; C-c C-o sde-kill-output
;;; C-c C-r sde-show-output
;;; C-c C-u sde-kill-input
;;; RET     sde-return


(defun sde-interrupt-soar ()
  "Interrupt the Soar process."
  (interactive)
  (sde-check-soar)
  (interrupt-process sde-soar-process comint-ptyp)
  (sde-soar-wait)
  (sde-soar-show-prompts))


(defun sde-kill-output ()
  "Kill all output from interpreter since last input."
  (interactive)
  (kill-region sde-soar-last-input-end (marker-position sde-soar-buffer-mark))
  (goto-char (marker-position sde-soar-buffer-mark))
  (insert "*** output flushed ***\n")
  (sde-soar-show-prompts)
  (set-marker sde-soar-buffer-mark (point)))


(defun sde-show-output ()
  "Display start of this batch of interpreter output at top of window.
Also put cursor there."
  (interactive)
  (goto-char sde-soar-last-input-end)
  (backward-char)
  (beginning-of-line)
  (set-window-start (selected-window) (point))
  (end-of-line))


(defun sde-kill-input ()
  "Kill all text from last stuff output by interpreter to point."
  (interactive)
  (if (> (point) (marker-position sde-soar-buffer-mark))
      (kill-region sde-soar-buffer-mark (point))))


;; The following are based on code from "Joseph S. Mertz" <jm7l+@andrew.cmu.edu>
;; submitted for CMU-released soar-mode v5.0.  Originally called
;; last-cl-top-window and next-cl-top-window.

(defun sde-backward-prompt ()
  "Move cursor to the previous prompt in the buffer and display that line at top."
  (interactive)
  (beginning-of-line)
  (if (not (re-search-backward sde-soar-prompt-regexp (point-min) t))
      (error "Cannot find any more prompts in this buffer."))
  (recenter 0)
  (end-of-line)
  (comint-bol nil))


(defun sde-forward-prompt ()
  "Move cursor to the next prompt in the buffer and display that line at the top."
  (interactive)
  (end-of-line)
  (if (not (re-search-forward sde-soar-prompt-regexp (point-max) t))
      (error "Cannot find any more prompts in this buffer."))
  (recenter 0)
  (end-of-line)
  (comint-bol nil))


(defun sde-return ()
  "Grab the current expression and send it to Soar.
If we have a complete sexp, send it.  Otherwise, indent appropriately."
  (interactive)
  (sde-check-soar)
  (let ((input (sde-get-old-input)))
    (if input
	;; Input is complete
	(if (sde-soar-input-filter input)
	    ;; Only proceed if filter returns true.
	    (progn
	      (if (>= (point) (marker-position sde-soar-buffer-mark))
		  (goto-char (point-max))
		  (progn
		    (goto-char (marker-position sde-soar-buffer-mark))
		    (insert input)))
	      (insert ?\n)
	      (set-marker sde-soar-buffer-mark (point))
	      (setq sde-soar-last-input-end (point))
	      (sde-add-to-input-history input)
	      (sde-update-args input)
	      (sde-soar-cmd (sde-agent) input nil (current-buffer) nil
			    'no-message 'no-message 'sde-soar-output-filter)))
	;; Input is not complete
	(if (= (marker-position sde-soar-buffer-mark) (point-max))
	    ;; Send a blank line
	    (progn
	      (insert ?\n)
	      (set-marker sde-soar-buffer-mark (point))
	      (setq sde-soar-last-input-end (point))
	      (sde-soar-cmd (sde-agent) "" nil (current-buffer) nil
			    'no-message 'no-message)
	      ;; Stupid hack because Soar doesn't echo a prompt after a blank line
	      (sde-soar-show-prompts)
	      (sde-soar-update-status 'ready))
	    ;; Dont' send anything, just reindent.
	    (progn
	      (insert ?\n)
	      (save-restriction
		(narrow-to-region (marker-position sde-soar-buffer-mark)
				  (point-max))
		(funcall indent-line-function)))))))


(defun sde-add-to-input-history (cmd)
  ;; Add the CMD to this buffer's sde-soar-input-ring.
  (if (and (funcall sde-soar-input-ring-filter cmd)
	   (or (ring-empty-p sde-soar-input-ring)
	       (not (string= (ring-ref sde-soar-input-ring 0) cmd))))
      (ring-insert sde-soar-input-ring cmd)))


(defun sde-get-old-input ()
  ;; Return the sexp starting at the nearest previous prompt, or nil if none.
  (save-excursion
    (let* ((begin (sde-beginning-of-sp 1 t))
	   (once (if (< (point) (marker-position sde-soar-buffer-mark))
		     (save-excursion (end-of-line) (point))))
	   (end nil)
	   (done nil))
      (condition-case ()
	  (while (and (not done) (< (point) (point-max)))
	    (forward-sexp)
	    (setq end (point))
	    (skip-chars-forward " \t\n")
	    (if (and once (>= (point) once))
		(setq done t)))
 	(error (setq end nil)))
      (if end
	  (buffer-substring begin end)))))


(defun sde-process-input-start ()
  ;; Go to the start of the input region and return point.
  (if (>= (point) (marker-position sde-soar-buffer-mark))
      (goto-char (marker-position sde-soar-buffer-mark))
      (progn
	(end-of-line)
	(if (re-search-backward sde-soar-prompt-regexp (point-min) 'stay)
	    (comint-skip-prompt)
	    (point)))))


;;; Command filtering, based in part on directory tracking code of cmushell.el.
;;;
;;; sde-soar-input-filter checks and validates inputs, and possibly signals
;;; errors.  For example, in multi-agent Soar we don't want the user to
;;; select a different agent in an agent buffer (accidentally or
;;; deliberately).  The input filter checks for that.
;;;
;;; sde-update-args looks at the arguments passed to the Soar command before
;;; it's sent to Soar, and updates the argument history list for the
;;; particular command.  This is necessary to insure that when the user next
;;; invokes a particular command from a key binding, the default arguments
;;; are consistent with what they last typed to the process.  Also, the user
;;; may have typed commands into the Soar process buffer before ever invoking
;;; the command via key binding or M-X.  We want the defaults to be
;;; consistent no matter which way the command is invoked first.
;;;
;;; Note that it is necessary to call sde-update-args before Soar gets the
;;; command, to insure that the argument history is set prior to commands
;;; that may generate long output.  This is important if the user wants to
;;; type a *new* command while a command is still executing.  Otherwise, if
;;; sde-update-args was made part of the sde-soar-output-filter, the command
;;; argument histories wouldn't get updated until Soar finished.
;;;
;;; sde-soar-output-filter tracks commands given to Soar and does any
;;; necessary post-command processing.  Other modes, such as cmushell, simply
;;; use the comint-input-sentinel to match commands like cd and then extract
;;; the arguments to the command that the user typed.  However, we have to
;;; get Soar's actual response to the commands.  For example, we don't want
;;; to just change directories to the directory specified in a cd command,
;;; because while the user may have typed a valid directory, Soar may not be
;;; able to cd to it.  So we have to get Soar's own output.  A simple way to
;;; do this is to run this postprocessing function.  sde-return invokes
;;; sde-soar-cmd with sde-soar-output-filter as a hook function to be called
;;; by the process filters.


(defun sde-match-soar-cmd (regexp input)
  (string-match "^(?\\s *" input)	; SKip whitespace
  (let ((start (match-end 0)))
    (and (string-match regexp input start)
	 (let ((end (match-end 0)))
	   (cond ((eq end (string-match "\\s *\\(\)\\|$\\)" input end))
		  "")			; No arg.
		 ((eq end (string-match "\\s +\\([^)]*\\)\)?$" input end))
		  (sde-substring input 1)) ; Have arg, return it.
		 (t nil))))))           ; Something else.


(defun sde-soar-input-filter (input)
  ;; The input filter checks and validates inputs, and possibly signals
  ;; errors.  It is called on every input to Soar.
  (cond ((and (sde-match-soar-cmd "select-agent" input)
	      sde-soar-agents
	      (not (string= (buffer-name (current-buffer)) "*control*")))
	 (error "This buffer's agent is fixed."))
	(t t)))


(defun sde-update-args (input)
  ;; Update the arguments history list for the command in the INPUT string.
  ;; This function is called on each input passed to the Soar process.
  (condition-case err
      (let (args)
	(cond ((setq args (sde-match-soar-cmd "go" input))
	       (sde-gmhist-default args 'sde-go-hist))

	      ((setq args (sde-match-soar-cmd "run" input))
	       (sde-gmhist-default args 'sde-run-hist))

	      ((setq args (sde-match-soar-cmd "p\\|print" input))
	       ;; Ugly match string to find :depth and :internal arguments
	       ;; regardless of their order in the argument.
	       (if (string-match "\\(:depth\\s +[0-9]\\|:internal\\)\\s +\\(:depth\\s +[0-9]\\|:internal\\)?" args)
		   (if (match-beginning 2)
		       (sde-gmhist-default (sde-substring args 1 2) 'sde-print-hist)
		       (sde-gmhist-default (sde-substring args 1) 'sde-print-hist))))

	      ((setq args (sde-match-soar-cmd "matches" input))
	       (if (string-match "\\(\\S +\\)\\s +\\([0-9]+\\)" args)
		   (sde-gmhist-default (sde-substring args 2)
				       'sde-matches-hist)))

	      ((setq args (sde-match-soar-cmd "ms" input))
	       (sde-gmhist-default args 'sde-ms-hist))

	      ((setq args (sde-match-soar-cmd "firing-counts" input))
	       (let ((first-elem (car (read-from-string args))))
		 (if (integerp first-elem)
		     (sde-gmhist-default first-elem 'sde-firing-counts-hist))))

	      ((setq args (sde-match-soar-cmd "preferences" input))
	       ;; 3rd arg, if present, is the optional level specification
	       (if (and (string-match "\\(\\S +\\)\\s +\\(\\S +\\)\\s +\\([0-3]+\\)" args)
			(match-beginning 3))
		   (sde-gmhist-default (sde-substring args 3)
				       'sde-preferences-hist)))

	      ((setq args (sde-match-soar-cmd "list-chunks" input))
	       (sde-gmhist-default args 'sde-list-chunks-hist))

	      ((setq args (sde-match-soar-cmd "list-justifications" input))
	       (sde-gmhist-default args 'sde-list-justifications-hist))

	      ((setq args (sde-match-soar-cmd "list-productions" input))
	       (sde-gmhist-default args 'sde-list-productions-hist))
	      ))
    (error
     (message (car (cdr err))))))


(defun sde-soar-output-filter ()
  ;; This function is called on each input passed to the Soar process,
  ;; *after* the input has been processed by Soar.  It watches for commands
  ;; such as pwd and performs special-case processing, such as setting this
  ;; buffer's directory to match the output of pwd.
  (condition-case err
      (let (args)
	(cond ((setq args (sde-match-soar-cmd "load" sde-soar-cmd-input))
	       (sde-update-pbreaks)
	       (sde-soar-record-loads))

	      ((and sde-soar-track-cd
		    (sde-match-soar-cmd "cd\\|chdir\\|pwd" sde-soar-cmd-input))
	       (sde-soar-check-pwd))

	      ((setq args (sde-match-soar-cmd "create-agents" sde-soar-cmd-input))
	       ;; This is only available in a Soar compiled for multiple
	       ;; agents, which implies that we must already have a control
	       ;; agent.  So only need to create new agent buffers.
	       (if sde-soar-agents
		   (sde-create-agents-from-string args)
		   (message "Cannot create agents.")))
	      ))
    (error
     (message (car (cdr err))))))


(defun sde-soar-track-cd-toggle ()
  "Turn directory tracking on and off in a Soar process buffer."
  (interactive)
  (setq sde-soar-track-cd (not sde-soar-track-cd))
  (message "directory tracking %s."
	   (if sde-soar-track-cd "on" "off")))


(defun sde-soar-check-pwd ()
  (condition-case nil
      (let ((output (sde-soar-silent-cmd (sde-agent) "pwd")))
	(string-match sde-soar-pwd-output-regexp output)
	(setq default-directory (file-name-as-directory (sde-substring output 1)))
	(sde-soar-pwd-message))
    (error nil)))


(defun sde-soar-pwd-message ()
  ;; Show the current directory on the message line.
  ;; Pretty up dirs a bit by changing "/usr/jqr/foo" to "~/foo".
  (let ((dir default-directory))
    (if (string-match (format "^%s\\(/\\|$\\)" (getenv "HOME")) dir)
	(setq dir (concat "~/" (substring default-directory (match-end 0)))))
    (if (string-equal dir "~/")
	(setq dir "~"))
    (message dir)))



;;;-----------------------------------------------------------------------------
;;; 8.  Interface to Soar
;;;-----------------------------------------------------------------------------
;;;
;;; General notes:
;;; 
;;; This interface uses asynchronous communication with Soar.  This means
;;; that Emacs functions that send input to Soar come back immediately, and
;;; Soar's output must be gathered by a process filter.  The control of
;;; communication with Soar is therefore split between several routines,
;;; which makes it harder to understand and debug.
;;;
;;; There are different filters for different command situations.  Output
;;; from Soar is not contiguous.  Each filter may get called many, many, many
;;; times before Soar is finished with a given command.  The only way for a
;;; filter to know that Soar has finished production output for a particular
;;; command is to watch the output stream and look for some sign, such
;;; as the appearance of a Soar prompt.
;;;
;;; In the main (real) process buffer, `sde-soar-buffer-mark' is the
;;; process-mark.  In agent buffers, `sde-soar-buffer-mark' is a marker for
;;; where to dump new output.  In both cases, code uniformly refers to
;;; `sde-soar-buffer-mark' rather than making a distinction between the
;;; buffer types.
;;;
;;; `sde-soar-status' indicates the status of the current Soar command.  It
;;; is a superset of the process-status of the process and should be set by
;;; calling the function sde-soar-update-status. Possible values are:
;;;
;;;      'starting -- Soar in the process of starting up
;;;      'ready    -- no commands pending; Soar ready for input
;;;      'running  -- command running; Soar may be producing output
;;;      'stop     -- process is stopped but continuable
;;;      'exit     -- process has exited
;;;      'signal   -- process received a fatal signal
;;;
;;;-----------------------------------------------------------------------------

(defvar sde-soar-output-start 0
  "Starting pos of output in sde-soar-buffer from new Soar processes.")

;; The Soar process is started with sde-start-soar.  It uses a temporary,
;; initial process filter to gather up Soar's initial output and initialize
;; the interface in various ways based on the contents of that output.  For
;; example, it detects whether Soar started in single agent or multi agent
;; mode, and sets things up appropriately.

(defun sde-start-soar (program start-dir)
  ;; Starts soar program PROGRAM in directory START-DIR. 
  ;; First kills any old Soar buffers.
  (if (and sde-soar-buffer (get-buffer sde-soar-buffer))
      (kill-buffer sde-soar-buffer))
  ;; If there are no multiple agents, sde-soar-agents will be nil. 
  (while sde-soar-agents
    (kill-buffer (cdr (car sde-soar-agents)))
    (setq sde-soar-agents (cdr sde-soar-agents)))
  (message "Starting Soar (\"%s\") . . ." program)
  (save-excursion
    (setq sde-soar-buffer
	  (sde-get-buffer-create (concat "*" sde-soar-default-name "*")))
    (set-buffer sde-soar-buffer)
    ;; Make sure to start in the right directory.
    (sde-cd (file-name-as-directory start-dir))
    (sde-soar-mode)
    (goto-char (point-max))		; Move point where new output will go.
    (setq sde-soar-output-start (point)) ; Must set this before starting Soar.
    (sde-soar-reset-histories)
    (let ((process-connection-type sde-soar-use-ptys))
      (comint-exec sde-soar-buffer sde-soar-default-name program nil nil))
    (setq sde-soar-process     (get-buffer-process sde-soar-buffer)
	  sde-soar-buffer-mark (process-mark sde-soar-process)
	  sde-soar-agents      nil)
    (set-process-filter sde-soar-process 'sde-soar-initial-process-filter)
    (set-process-sentinel sde-soar-process 'sde-soar-process-sentinel)
    (sde-soar-update-status 'starting)))


;; When a Soar process is first created, the process filter is
;; `sde-soar-initial-process-filter'.
;;
;; The initial filter handles the first output from Soar, creates agent
;; buffers if needed, etc.  It has to be aware of the possibility that Soar
;; may not come back with a prompt after it starts: the user may have a
;; .init.soar or .init.soar.multi file that executes a run, go, or schedule
;; command.  Hence, this resets the process filter to
;; sde-soar-process-display-filter as early as possible, once it determines
;; that Soar is done starting.  Doing this insures that if Soar doesn't
;; immediately come back with a prompt (e.g., as when happens if a .init file
;; does a run or schedule), this initial filter function doesn't keep getting
;; called on the rest of the output.

(defun sde-soar-initial-process-filter (process output)
  ;; Gather first output from Soar and set up Soar Mode properly.
  ;; Check if Soar is in multi-agent mode.  If so, set up appropriately.
  ;; Pop up the buffer when Soar finishes starting.
  ;; Run the user hooks after Soar finishes starting.
  ;; When done, install sde-soar-process-display-filter.
  (let ((inhibit-quit t)
	(orig-buffer (current-buffer))
	(match-data (match-data)))
    (unwind-protect
	 (progn
	   (set-buffer (process-buffer process))
	   (goto-char (point-max))
	   (insert output)
	   (set-marker sde-soar-buffer-mark (point-max))
	   ;; Only do the next stmts if have an error or have finished starting.
	   (cond ((string-match sde-soar-startup-error-regexp output)
		  ;; Error during startup.
		  (set-process-filter process 'sde-soar-process-display-filter)
		  (sde-show-buffer sde-soar-buffer)
		  (error "Error occurred while starting up Soar."))

		 ((string-match sde-soar-running-regexp output)
		  ;; Finished starting but Soar didn't stop for a prompt --
		  ;;  maybe it's loading files, or there was a run stmt, etc.
		  (setq sde-soar-cmd-last-dest  sde-soar-buffer
			sde-soar-last-input-end sde-soar-output-start
			sde-soar-cmd-hook       'sde-soar-record-loads)
		  ;; Reset process filter immediately, in case Soar is
		  ;; running and producing more output.
		  (set-process-filter process 'sde-soar-process-display-filter)
		  (sde-soar-init output))

		 ((string-match sde-soar-prompt-regexp output)
		  ;; Finished starting & waiting at a prompt.  This case
		  ;; won't match if the previous case ever matches, because
		  ;; the previous case will reset the process filter.
		  (setq sde-soar-cmd-last-dest  sde-soar-buffer
			sde-soar-last-input-end sde-soar-output-start)
		  ;; Have to call record-soar-loads explicitly, since nothing
		  ;; is going to get called on the initial output after this
		  ;; point.
		  (sde-soar-record-loads) 
		  (sde-soar-init output))))
      (store-match-data match-data)
      (set-buffer orig-buffer))))


(defun sde-soar-init (output)
  ;; Finish initializations at Soar startup time.
  (let* ((output-end (save-excursion
		       (goto-char sde-soar-output-start)
		       (if (re-search-forward sde-soar-running-regexp nil 'move)
			   (beginning-of-line))
		       (point)))
	 (initial-output (buffer-substring sde-soar-output-start output-end)))
    (if (string-match sde-soar-started-multi-regexp initial-output)
	(progn
	  (message "Soar running in multi-agent mode.")
	  (sde-soar-init-agents initial-output default-directory)
	  (rename-buffer "*control*")
	  (setq sde-soar-buffer       (current-buffer)
		sde-soar-buffer-agent "control")
	  (sde-add-agent-and-buffer "control" sde-soar-buffer)
	  (bury-buffer sde-soar-buffer))
	;; Didn't find indication of multiple agents.
	(message "Soar running in single-agent mode."))
    (sde-soar-update-status 'ready)
    ;; The following only matches if Soar continued running.
    (if (string-match sde-soar-sched-regexp output)
	;; Set this so that process-display-filter knows where to put the output.
	(setq sde-soar-cmd-last-dest (sde-agent-buffer (sde-substring output 1))))
    (sde-show-buffer (or (sde-first-agent-buffer) sde-soar-buffer))
    (if sde-soar-beep-after-setup
	(beep t))
    (run-hooks 'sde-soar-hook)))


(defun sde-soar-init-agents (initial-output current-dir)
  ;; Set up multiple agent buffers, based on Soar's INITIAL-OUTPUT and knowing
  ;; that Soar started in CURRENT-DIR.  Basic algorithm:
  ;;  look for the names of the agents, create buffers, and initialize
  ;;  appropriately.
  (let* ((prompt-start (string-match sde-soar-prompt-regexp initial-output))
	 (banner (substring initial-output 0 prompt-start))
	 (next-start 1))
    (while (string-match sde-soar-agent-creation-regexp initial-output next-start)
      (sde-new-agent (sde-substring initial-output 1) current-dir banner)
      (setq next-start (match-end 0)))))


;; The agent's buffer is set to STARTING-DIR/AGENT-NAME if possible.  This is
;; done unconditionally if the directory exists, regardless of whether Soar
;; actually prints any messages about changing directory to the agent's
;; subdirectory.  When Soar starts up, it does print a message like this:
;;
;;   Creating agent one.
;; 
;;   Changing to directory one/.
;;   Loading .init.soar
;;
;; However, if there is no .init.soar file in the agent subdirectory, Soar
;; does not print any messages about changing directory.  So we set the
;; agent directory based on whether it exists, rather than based on any
;; messages from Soar.

(defun sde-new-agent (name root-dir &optional initial-output)
  ;; Create a new agent buffer for agent NAME and initialize appropriately.
  ;; Directory ROOT-DIR is the directory in which Soar was started and where
  ;; there may be a subdirectory for the agent.  If there is a subdirectory
  ;; for the agent, the agent's buffer is set to it; if not, the agent's
  ;; buffer is set to ROOT-DIR.  
  ;; Optional arg INITIAL-OUTPUT specifies what to print in the
  ;; buffer when it first comes up.  This is for Soar's initial output
  ;; message.
  (save-excursion
    (let* ((new-buf (sde-get-buffer-create (concat "*" name "*")))
	   (root-dir (file-name-as-directory root-dir))
	   (agent-dir (concat root-dir name)))
      (set-buffer new-buf)
      (if (file-directory-p agent-dir)
	  (sde-cd agent-dir))
      (sde-soar-mode)
      (goto-char (point-max))
      (if initial-output
	  (insert initial-output))
      (insert "\n")
      (insert (format sde-soar-agent-prompt-fmt name))
      (setq sde-soar-buffer-mark    (point-marker)
	    sde-soar-last-input-end (point)
	    sde-soar-buffer-agent   name)
      (sde-add-agent-and-buffer name new-buf)
      (run-hooks 'sde-soar-hook))))


(defun sde-add-agent-and-buffer (name buffer)
  (setq sde-soar-agents (append sde-soar-agents (list (cons name buffer)))))


;; Sending commands to the Soar process.
;;
;; The main function for sending commands to Soar is `sde-soar-cmd'.
;; It calls a low-level function `sde-soar-send' that does the real work of
;; sending input to Soar.  Several global variables are used to communicate
;; between `sde-soar-send' and the process filters.  The main global
;; variables are below and have names all prefixed with `sde-soar-cmd-'.
;;
;; All of these variables are part of the current command being sent to Soar.
;; Using such separate variables is stylistically poor.  It would have been
;; better to gather up these separate variables into a defstruct, but the
;; defstruct facility in Emacs Lisp is inefficient and would require loading
;; a lot more than we need.  Using a vector is another approach but that
;; complicates the code more.

(defvar sde-soar-cmd-input     ""  "String sent to Soar.")
(defvar sde-soar-cmd-action    nil "What should be done with Soar's output.")
(defvar sde-soar-cmd-dest      nil "Initial destination buffer of Soar's output.")
(defvar sde-soar-cmd-output    nil "Output from Soar after a command is sent.")
(defvar sde-soar-cmd-last-dest nil "Last buffer that received output from Soar.")
(defvar sde-soar-cmd-error     nil "Whether Soar signaled an error.")
(defvar sde-soar-cmd-hook      nil "List of functions called after Soar finishes.")


(defun sde-soar-query (agent input)
  ;; Wrapper around sde-soar-cmd for a common operation:  querying Soar
  ;; for something and dumping the output into the Soar output buffer.
  (sde-soar-cmd agent input nil 'special))


(defun sde-soar-silent-cmd (agent input)
  ;; Send command to Soar, silently.  Useful for querying Soar behind the
  ;; scenes.  Ignores errors.
  (sde-soar-cmd agent input 'wait 'silent))


;; sde-soar-cmd has a horrible parameter scheme, but it was hard to figure
;; out how to get the necessary functionality without resorting to many
;; similar-but-different versions of sde-soar-cmd.  Still, this is a good
;; candidate for revision sometime in the future.
;;
;; A tough issue here is how to handle errors.  If sde-soar-cmd doesn't wait,
;; it can't catch errors.  However, you don't want the process filters
;; necessarily to catch errors, because in some cases (e.g., sde-return) you
;; never want an error popup but instead want errors just to be passed to the
;; buffer.  In other cases, we want the errors to be ignored.  The solution
;; adopted here is (1) for "special" sends, errors are caught & displayed by
;; process-special-filter; (2) for all other sends, errors are handled by
;; sde-soar-cmd and not by process filters, and then only if WAIT-P is
;; non-nil.

(defun sde-soar-cmd (agent input &optional wait-p dest switch m-msg b-msg hook)
  ;; To agent AGENT, send INPUT. Optional arguments:
  ;; Wait until Soar returns if WAIT-P.
  ;; If WAIT-P is nil, errors are only caught if DEST is 'special.
  ;; If DEST is nil, dumps output to appropriate agent buffers.
  ;; If DEST is 'special, dumps output to Soar output buffer, doesn't print
  ;;   any messages in the process buffer.  If DEST is 'special but
  ;;   sde-soar-use-output-buffer is nil, the 'special setting is ignored.
  ;; If DEST is 'silent, doesn't dump output anywhere, doesn't print any
  ;;   messages in the process buffer.
  ;; Switch to destination buffer if SWITCH is non-nil.
  ;; M-MSG specifies a string to echo in the minibuffer.
  ;; If M-MSG is nil, INPUT is echoed in the minibuffer instead.
  ;; If M-MSG is 'no-message, no message is echoed under any circumstance.
  ;; B-MSG specifies a string to print in the agent process buffer.
  ;; If B-MSG is nil, INPUT is printed in the process buffer, unless
  ;;   DEST is 'special or 'silent.
  ;; If B-MSG is 'no-message, nothing is echoed under any circumstance.
  ;; If HOOK is defined, it should be a function or list of functions that are
  ;;  to be called after the command finishes.  Note that this may happen after
  ;;  sde-soar-cmd itself returns.  HOOK functions are called by the process 
  ;;  filter with no arguments.
  (let* ((inhibit-quit t)
	 (orig-buffer (current-buffer))
	 (special (and (eq dest 'special) sde-soar-use-output-buffer))
	 (silent (eq dest 'silent))
	 (output-action (cond (wait-p 'gather)
			      (special 'display-special)
			      (t 'display-buffers)))
	 (dest (cond ((or silent special) nil)
		     (sde-soar-agents (sde-agent-buffer agent))
		     (t sde-soar-buffer)))
	 (m-msg (and (not silent) (not (eq m-msg 'no-message)) (or m-msg input)))
	 (b-msg (and (not silent) (not (eq b-msg 'no-message)) (or b-msg input))))
    ;; If we're running multiple agents, first have to specify which
    ;; agent is to receive this command.
    (if (and sde-soar-agents (null agent))
	(error "Destination agent unknown; command aborted."))
    (if sde-soar-agents
	(sde-soar-send (format sde-soar-agent-select-fmt agent) nil 'wait 'gather nil))
    (if (and special m-msg)
	(sde-init-output-buffer m-msg))
    (if m-msg
	(message m-msg))
    ;; Go to the destination buffer, move point to the start of where new
    ;; output should go, and if there's a b-msg to display, print it.  If
    ;; point is in the destination buffer and it is not at the end, do not
    ;; move point.  Otherwise, move point.  Gross code.
    (if (and (not silent) (not special))
	(let ((in-window (eq dest (window-buffer (selected-window))))
	      (at-end (save-excursion
			(set-buffer dest)
			(= (point) (marker-position sde-soar-buffer-mark))))
	      (update '(lambda ()
			(set-buffer dest)
			(goto-char (marker-position sde-soar-buffer-mark))
			(setq sde-soar-last-input-end (point))
			(if b-msg
			    (progn
			      (insert-before-markers (concat ";;; " b-msg "\n"))
			      (set-marker sde-soar-buffer-mark (point))))
			(set-buffer orig-buffer))))
	  (if (and in-window (not at-end))
	      (save-excursion (funcall update))
	      (funcall update))))
    ;; Send command.
    (sde-soar-send input dest wait-p output-action hook)
    ;; Post-processing of output.
    (cond ((and (eq output-action 'gather) sde-soar-cmd-error (not silent))
	   (sde-soar-error sde-soar-cmd-output (or m-msg b-msg "")))
	  ((and switch (not silent) (not special))
	   (sde-pop-to-buffer dest))
	  ((and (eq output-action 'gather) (not silent))
	   (sde-soar-show-prompts)
	   sde-soar-cmd-output)
	  ((eq output-action 'gather)
	   ;; If we get an error in this mode, just return nil.
	   (and (not sde-soar-cmd-error) sde-soar-cmd-output)))))


;; The different filters interact closely with `sde-soar-send'.  When
;; `sde-soar-send' is invoked it sets global variables that then affect the
;; process filters' operation.  This is necessary because the communication
;; with Soar is asynchronous.

(defun sde-soar-send (input dest wait-p output-action hook)
  ;; Low-level send function.
  ;; We set the status here to 'running, create the sde-soar-input record,
  ;; send the input to Soar, and let the process-filter gather the results
  ;; and reset the status to nil when Soar returns with a prompt.
  (sde-soar-update-status 'running)
  (setq sde-soar-cmd-input     input
	sde-soar-cmd-action    output-action
	sde-soar-cmd-dest      dest
	sde-soar-cmd-output    ""
	sde-soar-cmd-last-dest nil
	sde-soar-cmd-error     nil
	sde-soar-cmd-hook      hook)
  (cond ((eq output-action 'display-buffers)
	 (set-process-filter sde-soar-process 'sde-soar-process-display-filter))
	((eq output-action 'display-special)
	 (set-process-filter sde-soar-process 'sde-soar-process-special-filter))
	((eq output-action 'gather)
	 (set-process-filter sde-soar-process 'sde-soar-process-gather-filter)))
  (comint-send-string sde-soar-process (concat input "\n"))
  (let ((inhibit-quit nil))
    (if wait-p
	(sde-soar-wait))))


;; Emacs receives the output from Soar in spurts.  The process filters have
;; to gather up all the output (or write it out to buffers, as appropriate)
;; and keep checking for the appearance of a prompt in the output from Soar.
;; Once the prompt is seen the process filter assumes that the current
;; command has been finished.  Of course, the filter also needs to check the
;; output stream for signs of errors.
;;
;; Emacs apparently will queue the calls to process filters in a smart way,
;; so that even if a new filter has been installed, this invocation of the
;; current filter will finish before the next filter function gets called on
;; the continuing output.
;;
;; Prompts have to be handled specially in multi-agent case.  Since there is
;; only one physical Soar process, Soar only prints one prompt after each
;; command is finished, even if that command is "schedule".  This is
;; problematic if we're trying to give each agent its own buffer; we need a
;; separate prompt for each buffer.  The solution here is to ignore the
;; prompt printed by Soar and instead stuff an explicit prompt in each agent
;; buffer that needs it after each command.

(defun sde-soar-process-display-filter (process output)
  ;; Filter for when output is going straight to agent buffers.
  ;; This has to be as fast and short as possible.
  ;; This does not need to check for errors -- just let them show
  ;; up in the process buffer.
  (let ((inhibit-quit t)
	(match-data (match-data))
	(last-dest (or sde-soar-cmd-last-dest sde-soar-cmd-dest))
	(orig-buffer (current-buffer))
	prompt)
    (unwind-protect
	 (progn
	   ;; If we're supposed to dump output to a buffer somewhere,
	   ;; loop on output string, looking for indication of which
	   ;; agent is supposed to receive each piece of the output.
	   (while (and (not (string= output ""))
		       (string-match sde-soar-sched-regexp output))
	     (sde-insert-in-buffer last-dest (substring output 0 (match-beginning 0)))
	     (setq last-dest (sde-agent-buffer (sde-substring output 1))
		   output    (substring output (match-end 0))))
	   (setq sde-soar-cmd-last-dest last-dest
		 prompt (string-match sde-soar-prompt-regexp output))
	   ;; Following handles both left-over output in
	   ;; multi-agents case and all output in single agent case.
	   (if (not (string= output ""))
	       (sde-insert-in-buffer last-dest (substring output 0 prompt)))
	   (if prompt
	       ;; Done with this command.  Show prompts & set status.
	       ;; This last code doesn't have to be as efficient as the rest.
	       (progn
		 (sde-soar-show-prompts)
		 (sde-soar-update-status 'ready)
		 (if sde-soar-cmd-hook
		     (run-hooks 'sde-soar-cmd-hook)))))
      (store-match-data match-data)
      (set-buffer orig-buffer))))


(defun sde-soar-process-special-filter (process output)
  ;; This has to catch errors, because there may not be a caller waiting.
  (let* ((inhibit-quit t)
	 (match-data (match-data))
	 (prompt (string-match sde-soar-prompt-regexp output))
	 (error (string-match sde-soar-error-regexp output)))
    (unwind-protect
	 ;; If no error, dump to output buffer.
	 ;; If prompt or error, update status and run the cmd-hook. 
	 ;; If error, flag it and call sde-soar-error.
	 (progn
	   (if (not error)
	       (progn
		 (sde-within-output-buffer
		   (goto-char (marker-position sde-soar-buffer-mark))
		   (insert-before-markers (if prompt (substring output 0 prompt) output)))
		 (if (and prompt sde-soar-pop-up-output-buffer)
		     (sde-show-output-buffer))))
	   (if (or prompt error)
	       (progn
		 (sde-soar-update-status 'ready)
		 (if sde-soar-cmd-hook
		     (run-hooks sde-soar-cmd-hook))))
	   ;; I hope that when errors occur, the output contains the complete
	   ;; error message!
	   (if error
	       (progn
		 (setq sde-soar-cmd-error  t
		       sde-soar-cmd-output (substring output 0 prompt))
		 (sde-soar-error sde-soar-cmd-output sde-soar-cmd-input))))
      (store-match-data match-data))))


(defun sde-soar-process-gather-filter (process output)
  ;; This routine must always have a caller waiting.
  ;; It must recognize and flag errors, but not do anything about them.
  ;; The caller is assumed responsible for error handling. 
  (let* ((inhibit-quit t)
	 (match-data (match-data))
	 (prompt (string-match sde-soar-prompt-regexp output)))
    (unwind-protect
	 (progn
	   ;; Don't do substring unless really necessary, to reduce consing.
	   (setq sde-soar-cmd-output
		 (concat sde-soar-cmd-output (if prompt
						 (substring output 0 prompt)
						 output)))
	   (if (string-match sde-soar-error-regexp output)
	       (setq sde-soar-cmd-error t))
	   (if prompt
	       (progn
		 (sde-soar-update-status 'ready)
		 (if sde-soar-cmd-hook
		     (run-hooks sde-soar-cmd-hook)))))
      (store-match-data match-data))))


;; The tricky thing about the next function is that it has to handle the case
;; where the user is running multiple agents but is only interacting with one
;; agent.  Then, the remaining agents' prompts will (probably) still be
;; showing, and we don't want to insert another prompt string.  So this
;; function has to find where a prompt should go, and then check that there
;; isn't already a prompt there.

(defun sde-soar-show-prompts ()
  ;; Cycle through agent buffers and check that they all show a prompt for
  ;; the user, if appropriate.    
  (save-excursion
    (if sde-soar-agents
	(let ((agents sde-soar-agents))
	  (while agents
	    (set-buffer (cdr (car agents)))
	    (goto-char (point-max))
	    (beginning-of-line)
	    (if (not (looking-at sde-soar-prompt-regexp))
		(progn
		  (goto-char (point-max))
		  (insert-before-markers "\n")
		  (insert-before-markers
		   (format sde-soar-agent-prompt-fmt (car (car agents))))
		  (set-marker sde-soar-buffer-mark (point)))
		(goto-char (point-max)))
	    (setq agents (cdr agents))))
	;; Single agent case
	(progn
	  (set-buffer sde-soar-buffer)
	  (goto-char (point-max))
	  (insert-before-markers "\n")
	  (insert-before-markers sde-soar-prompt-str)
	  (set-marker sde-soar-buffer-mark (point))))))


;; The process sentinel is called by Emacs whenever the Soar process changes
;; state, and it is also called throughout Soar Mode explicitly.  It sets
;; 'sde-soar-status' to the current status of the process and
;; 'sde-soar-status-string' to a string equivalent of this status indicator.


(defun sde-soar-process-sentinel (process status)
  "Main process sentinel for Soar process."
  (let ((inhibit-quit t))
    (sde-soar-update-status status)))


(defun sde-soar-update-status (status)
  ;; Update the PROCESS STATUS globally in Soar Mode mode lines.
  (if (stringp status)
      (setq status (process-status sde-soar-process)))
  (setq sde-soar-status status)
  (setq sde-soar-status-string (format "%s" status))
  (if sde-show-soar-status
      (let ((orig-buffer (current-buffer)))
	(unwind-protect
	     (set-buffer (other-buffer))
	  (set-buffer orig-buffer))))
  sde-soar-status)


;; Functions for output handling.
;; This could use some cleaning up.

(defun sde-insert-in-buffer (buffer output)
  ;; Insert into BUFFER the given OUTPUT.  Based on bridge-insert from
  ;; bridge.el.  This depends on each buffer having its own local value of
  ;; sde-soar-buffer-mark so that this function can know where to put output.
  (let ((window (selected-window))
	(orig-buffer (current-buffer))
	(at-end nil))
    (unwind-protect
	 (progn
	   ;; Switch to buffer to access its local vars.
	   (set-buffer buffer)
	   (if (eq (window-buffer window) buffer)
	       (setq at-end (= (point) sde-soar-buffer-mark))
	       (setq window (get-buffer-window buffer)))
	   (save-excursion
	     (goto-char sde-soar-buffer-mark)
	     (insert-before-markers output)
	     (set-marker sde-soar-buffer-mark (point)))
	   (if window 
	       (progn
		 (if at-end
		     (goto-char sde-soar-buffer-mark))
		 (if (not (pos-visible-in-window-p (point) window))
		     (let ((original (selected-window)))
		       (save-excursion
			 (select-window window)
			 (recenter '(center))
			 (select-window original)))))))
      (set-buffer orig-buffer))))


;; This function must be called prior to letting anything print new
;; output in the output buffer.  It handles recreating the buffer
;; in case the user deleted it, and also does things like erase the
;; buffer if sde-soar-erase-output-buffer is non-nil.

(defun sde-init-output-buffer (&optional message)
  (if (or (null sde-soar-output-buffer)
	  (sde-killed-buffer-p sde-soar-output-buffer))
      ;; Recreate the output buffer.
      (save-excursion
	(setq sde-soar-output-buffer
	      (sde-get-buffer-create sde-soar-output-buffer-name))
	(set-buffer sde-soar-output-buffer)
	(sde-soar-mode)			; Set key bindings, etc.
	(setq sde-soar-buffer-mark (make-marker))))
  (save-excursion
    (set-buffer sde-soar-output-buffer)
    (goto-char (point-min))
    (if sde-soar-erase-output-buffer	; Erase if desired
	(erase-buffer)
	(insert (concat "---- " (or message "") " ----\n\n")))
    (set-marker sde-soar-buffer-mark (point))))


;; The following gets called in some process filters and in regular functions.

(defun sde-show-output-buffer ()
  (if (boundp 'popper-pop-buffers)	; Pop up using popper
      (progn
	(popper-show sde-soar-output-buffer)
	(set-window-start (get-buffer-window sde-soar-output-buffer) 1))
      (if sde-running-epoch		; Pop up without popper.
	  ;; Select a screen that the buffer has been displayed in before
	  ;; or the current screen otherwise.
	  (epoch::select-screen
	   ;; allowed-screens in epoch 3.2, was called screens before that
	   (or (car (symbol-buffer-value 'allowed-screens sde-soar-output-buffer))
	       (epoch::current-screen)))
	  ;; We're not in Epoch, so do standard Emacs code.
	  ;; The following is unusual code -- more natural would be to
	  ;; set-buffer to the output buffer and do a goto-char (point-min)
	  ;; to make the top of the buffer be visible.  But for some reason,
	  ;; when this function gets called from a process filter, point is
	  ;; restored in the output buffer upon exit.  I think there is a bug
	  ;; in select-window, and that select-window gets called by the top
	  ;; level at some point after a process filter is run, with the end
	  ;; result apparently that no matter what you do to point in here,
	  ;; it won't end up moved unless you fiddle with the window point.
	  (set-window-start (display-buffer sde-soar-output-buffer) 1))))


(defvar sde-soar-error-buffer-name "*Soar error*" "Name of pop-up error buffer.")
(defvar sde-soar-error-buffer  nil "Process error buffer.")

(defun sde-soar-error (output msg)
  ;; Generate an error, showing Soar's OUTPUT in a pop-up buffer and
  ;; signaling error with message MSG.
  (progn
    (save-excursion
      (with-output-to-temp-buffer sde-soar-error-buffer-name
	(set-buffer (get-buffer sde-soar-error-buffer-name))
	(sde-soar-mode)
	(insert output)))
    (error msg)))



;;;-----------------------------------------------------------------------------
;;; 9.  Support for pbreak
;;;-----------------------------------------------------------------------------


(defun sde-pbreak-in-effect (agent name)
  ;; Return entry from sde-soar-pbreak-list if there is a pbreak in effect
  ;; in AGENT for production named NAME.
  (let ((pbreaks sde-soar-pbreak-list)
	(data nil)
	(found nil))
    ;; There may be more than one instance of the same production having
    ;; a pbreak, but for different agents; this has to work in such cases.
    (while
	(not (if (and (setq data (car pbreaks))
		      (string= name (nth 0 data)) 
		      (string= agent (nth 2 data)))
		 (setq found data)
		 (null (setq pbreaks (cdr pbreaks))))))
    found))


(defun sde-print-pbreak-list ()
  ;; Show which productions are under the effect of a pbreak.
  (sde-init-output-buffer "pbreak")
  (sde-within-output-buffer
    (insert "List of pbreaks:\n")
    (let ((pbreaks sde-soar-pbreak-list))
      (while pbreaks
	(if sde-soar-agents
	    (insert "  [agent: " (nth 2 (car pbreaks)) "]"))
	(insert "  " (nth 0 (car pbreaks)) "\n")
	(setq pbreaks (cdr pbreaks)))
      (insert "\n")))
  (if sde-soar-pop-up-output-buffer
      (sde-show-output-buffer)))


(defun sde-pbreak-production (agent name)
  ;; In agent AGENT, establish a pbreak for production NAME, and add it to
  ;; the sde-soar-pbreak-list.
  (let ((sp (sde-get-production agent name)))
    (if (not sp)
	(error "Production \"%s\" not yet defined in Soar." name))
    ;; Search for last closing paren and add interrupt stmt. 
    (sde-soar-silent-cmd agent (concat
				(substring sp 0 (string-match ")[ \t\n]*\\'" sp))
				"(interrupt) )"))
    ;; Store away info, including Soar's idea of the new production.
    (setq sde-soar-pbreak-list
	  (cons (list name sp agent (sde-get-production agent name)) sde-soar-pbreak-list))))


(defun sde-unpbreak-production (agent name)
  ;; Undo the pbreak on the named production.
  (let ((pbreaks sde-soar-pbreak-list)
	data new)
    (while (setq data (car pbreaks))
      (if (and (string= name (nth 0 data)) (string= agent (nth 2 data)))
	  ;; Redefine back to original.
	  (sde-soar-silent-cmd (nth 2 data) (nth 1 data))
	  (setq new (cons data new)))
      (setq pbreaks (cdr pbreaks)))
    (setq sde-soar-pbreak-list new)))
	  

(defun sde-unpbreak-all ()
  ;; Undo all pbreaks.
  (while sde-soar-pbreak-list
    (sde-soar-silent-cmd (nth 2 (car sde-soar-pbreak-list))  ; agent
			 (nth 1 (car sde-soar-pbreak-list))) ; sp body
    (setq sde-soar-pbreak-list (cdr sde-soar-pbreak-list))))


(defun sde-update-pbreaks ()
  ;; Check with Soar the status of each of the productions thought to be
  ;; in pbreak mode.  If Soar's definition doesn't match the stored definition,
  ;; assume something has redefined the production, and clear its pbreak list
  ;; entry.
  (let ((pbreaks sde-soar-pbreak-list)
	data new)
    (while (setq data (car pbreaks))
      (if (string= (nth 3 data) (sde-get-production (nth 2 data) (nth 0 data)))
	  (setq new (cons data new)))
      (setq pbreaks (cdr pbreaks)))
    (setq sde-soar-pbreak-list new)))



;;;-----------------------------------------------------------------------------
;;; 10. Support for tracking productions
;;;-----------------------------------------------------------------------------

;;; All files loaded into Soar are tracked, so that operations such as finding
;;; production source code can work more effectively.  A hash table is used
;;; instead of a list, because the entries have to be unique, and Emacs doesn't
;;; have any built-in functions for efficiently doing member with string= on
;;; a long list of elements.  Using a hash table results in a set of unique
;;; entries, and we can use mapatoms to iterate over the elements.

(defun sde-soar-record-loads ()
  ;; Check through current Soar buffer for signs of Soar having loaded files.
  ;; Record their pathnames.
  (save-excursion
    (set-buffer sde-soar-buffer)
    (goto-char sde-soar-last-input-end)
    (let ((dir default-directory)
	  trailing name)
      (while (re-search-forward sde-soar-loading-or-chdir-regexp nil t)
	(beginning-of-line)
	(cond ((looking-at sde-soar-loading-regexp)
	       ;; Loading a file.
	       (setq name (sde-buffer-substring 1))
	       (if (string-match sde-file-extensions-regexp name)
		   ;; Store the string name in the hash table.
		   (progn
		     (setq name (expand-file-name name dir))
		     (sde-puthash name name sde-known-files))))

	      ((looking-at sde-soar-returning-regexp)
	       ;; Soar moved up from a directory.
	       ;; The math in the substring below removes the trailing "/."
	       (setq trailing (buffer-substring (match-beginning 1) (- (match-end 1) 2)))
	       (if (string-match (concat "\\`\\(.*/\\)" trailing "\\'") dir)
		   (setq dir (sde-substring dir 1))))

	      ((looking-at sde-soar-chdir-regexp)
	       ;; Soar moved down into a directory.
	       (setq dir (expand-file-name (sde-buffer-substring 1) dir))))
	;; Move on, to avoid getting stuck on same line!
	(end-of-line)))))


;; If user does excise-task, we're in trouble.  Need a resync loads list.
;; Keep a hash table on production names, and walk down it, compare it to
;; what's in memory.  Keep those files that had default productions.



;;;-----------------------------------------------------------------------------
;;; 11. Miscellaneous support code
;;;-----------------------------------------------------------------------------


(defun sde-get-production (agent name)
  ;; Get named production from Soar.
  (sde-soar-silent-cmd agent (concat "print " name)))


(defun sde-agent ()
  ;; Determine the current agent.  Used prior to sending certain commands to
  ;; Soar.
  (if sde-soar-agents
      (if (or (eq major-mode 'sde-soar-mode) (boundp 'sde-soar-buffer-agent))
	  ;; Case of Soar Mode buffers, or buffers derived from Soar Mode.
	  (or sde-soar-buffer-agent
	      (setq sde-soar-buffer-agent (sde-ask-for-agent)))
	  ;; Case of non-Soar Mode buffers.  We don't set a default.
	  (sde-ask-for-agent))))


(defun sde-agent-name-alist ()
  ;; Return an alist of the names of all known agents suitable for
  ;; use in calls to completing-read.
  (let ((n 0))
    (mapcar '(lambda (agent-buffer-pair)
	      (setq n (1+ n))
	      (cons (car agent-buffer-pair) (list n)))
	    sde-soar-agents)))


(defvar sde-select-agent-hist nil)

(defun sde-ask-for-agent ()
  ;; Prompt the user to choose an agent, and return their choice.
  (interactive)
  (if sde-soar-buffer-agent
      (sde-gmhist-default sde-soar-buffer-agent 'sde-select-agent-hist))
  (completing-read-with-history-in 'sde-select-agent-hist
				   "Agent name: "         ; Prompt.
				   (sde-agent-name-alist) ; Possible choices.
				   nil t))		  ; Require a match.


(defun sde-soar-is-alive ()
  ;; Return non-nil if a Soar process is running, nil otherwise.
  (and sde-soar-process
       (memq (process-status sde-soar-process) '(run stop))))


(defun sde-check-soar ()
  "Check if there's a Soar running, and complain if there isn't."
  (or (sde-soar-is-alive) (sde-complain-no-soar-process)))


(defun sde-check-soar-multi-agent ()
  "Check that Soar is running in multi-agent mode.  If not, signal error."
  (or sde-soar-agents (sde-error-soar-not-multi-agent)))


(defun sde-check-soar-in-progress ()
  (if (eq sde-soar-status 'running)
      (let ((inhibit-quit nil))
	(beep)
	(if (y-or-n-p "Soar command in progress.  Abort it? ")
	    (sde-interrupt-soar)
	    (progn
	      (message "Waiting for Soar to finish....")
	      (sde-soar-wait))))))


(defun sde-complain-no-soar-process ()
  "Complain that there isn't a running Soar, and ask to start one."
  (interactive)
  (beep)
  (if (cond ((null sde-soar-process)
	     (y-or-n-p "No Soar process running.  Start one? "))
	    ((eq sde-soar-status 'exit)
	     (y-or-n-p "Soar has quit.  Restart Soar? "))
	    ((eq sde-soar-status 'signal)
	     (y-or-n-p "Soar has received a fatal signal.  Restart Soar? ")))
      (progn
	(soar)
	(sde-soar-wait))
      (error "No Soar process running.")))


;;;
;;; Debugging aids
;;; 

(defun sde-hand-process (output dest output-action)
  (interactive "sOutput: \nbDest: \nSOutput-action: ")
  (sde-soar-update-status 'running)
  (setq sde-soar-cmd-input    "input"
	sde-soar-cmd-action    output-action
	sde-soar-cmd-dest      dest
	sde-soar-cmd-output    ""
	sde-soar-cmd-last-dest nil
	sde-soar-cmd-error     nil)
  (cond ((eq output-action 'display-buffers)
	 (sde-soar-process-display-filter nil output)
	 )
	((eq output-action 'display-special)
	 (sde-soar-process-special-filter nil output)
	 )
	((eq output-action 'gather)
	 (sde-soar-process-gather-filter nil output)
	 )))



;;;-----------------------------------------------------------------------------
;;; 12. Closing statements.
;;;-----------------------------------------------------------------------------

;; Aliases for common commands

(fset 'run-soar 'soar)
(fset 'track-cd-toggle 'sde-soar-track-cd-toggle)

;; Indentation for macros in Emacs.

(put 'sde-within-output-buffer 'lisp-indent-hook 0)
(put 'sde-args                 'lisp-indent-hook 0)

;; Hooks for debugging with Edebug.

(put 'sde-agent-buffer         'edebug-form-spec '(form))
(put 'sde-within-output-buffer 'edebug-form-spec '(&rest form))
(put 'sde-gmhist-default       'edebug-form-spec '(form symbol))
(put 'sde-args                 'edebug-form-spec '(&rest form))
(put 'sde-prompt               'edebug-form-spec '(form form))
