;;;; -*- Mode: emacs-lisp -*-
;;;; 
;;;; $Source: /n/manic/u/hucka/Projects/Soar/Interface/Src/RCS/sde-find.el,v $
;;;; $Id: sde-find.el,v 0.6 1993/06/10 07:35:09 hucka Exp hucka $
;;;; 
;;;; Description       : Production findinding facilities for SDE.
;;;; Original author(s): Michael Hucka <hucka@eecs.umich.edu>
;;;; Organization      : University of Michigan AI Lab
;;;; 
;;;; Copyright (C) 1993 Michael Hucka.
;;;;
;;;; This program (SDE) is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU General Public License as published
;;;; by the Free Software Foundation; either version 1 of the License, or (at
;;;; your option) any later version.
;;;; 
;;;; SDE is distributed in the hope that it will be useful, but WITHOUT ANY
;;;; WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;; for more details.
;;;; 
;;;; You should have received a copy of the GNU General Public License along
;;;; with this program; see the file COPYING.  If not, write to the Free
;;;; Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;;; Portions of SDE were derived from copyrighted code that permits copying
;;;; as long as the copyrights are preserved.  Here are the copyrights from
;;;; the relevant packages:
;;;;
;;;; GNU Emacs 18.58: Copyright (C) 1985-1991 Free Software Foundation, Inc.
;;;; Soar-mode 5.0    Copyright (C) 1990-1991 Frank Ritter, frank.ritter@cmu.edu
;;;; Ilisp 4.12:      Copyright (C) 1990-1992 Chris McConnell, ccm@cs.cmu.edu
;;;; BBDB 1.46:       Copyright (C) 1991-1992 Jamie Zawinski, jwz@lucid.com
;;;; Ange-ftp 4.25    Copyright (C) 1989-1992 Andy Norman, ange@hplb.hpl.hp.com
;;;; Comint 2.03:     Copyright (C) 1988 Olin Shivers, shivers@cs.cmu.edu

(defconst sde-find-el-version "$Revision: 0.6 $"
  "The revision number of sde-find.el.  The complete RCS id is:
      $Id: sde-find.el,v 0.6 1993/06/10 07:35:09 hucka Exp hucka $")

;;;; -----------------
;;;; Table of contents
;;;; -----------------
;;;; 0.  Documentation
;;;; 1.  Require, provide, and miscellaneous setup.
;;;; 2.  Global parameters and configuration variables
;;;; 3.  Internal constants and variables
;;;; 4.  Main code.
;;;; 5.  Closing statements.
;;;;
;;;; Suggestion for navigating this file: use the page movement commands in
;;;; Emacs (`C-x [' and `C-x ]') to move from section to section.  Also, get
;;;; the "page-menu" Emacs package from archive.cis.ohio-state.edu
;;;; (File /pub/gnu/emacs/elisp-archive/as-is/page-menu.el.Z).


;;;; -----------------
;;;; 0.  Documentation
;;;; -----------------
;;;;
;;;; The control of execution here is messy, because we're using multiple
;;;; heuristics one after another as each one fails.  It is further
;;;; complicated by the need to allow continuation of searches from the last
;;;; place where a string was found.
;;;;
;;;; The overall search procedure is split among several functions, and global
;;;; variables are used to maintain state information and communicate between
;;;; the functions.
;;;;
;;;; The general algorithm is:
;;;;
;;;; 1) Every production that is found is recorded in a hash table along with
;;;;    the buffer and file pathname where it was found.  find-production
;;;;    first checks to see if hashed info is available for a production.  If
;;;;    info is available, it tries to find the production in the buffer that
;;;;    it was last found in.  If the buffer has been killed or the production
;;;;    isn't in the buffer any more (e.g., the user deleted it), it looks in
;;;;    the file.  If the production isn't in the file anymore, it removes
;;;;    the production's hashed info entirely and resorts to searching as if
;;;;    the production was never encountered before.
;;;;
;;;; 2) If a production's info hasn't been cached, or the buffer it was last
;;;;    in no longer exists, or the production isn't in the file it was last
;;;;    found in, find-production next searches all Emacs buffers that are in
;;;;    SDE mode.  If found, the production location is cached.
;;;;
;;;; 3) If the production isn't found in any buffer, find-production next
;;;;    tries using tags.  If found, the production location is cached.
;;;;
;;;; 4) If the tags lookup fails, find-production next searches all files in
;;;;    the table sde-known-files.  If found, the production location is
;;;;    cached. 
;;;;
;;;; 5) If this fails, find-production derives a set of directories from the
;;;;    file pathnames in sde-known-files, then derives a table of all files
;;;;    in those directories that have SDE file extensions, and then performs
;;;;    file search as in step (4).  If found, the production location is
;;;;    cached.
;;;;
;;;; Searched-for strings are always cached even if they are not complete
;;;; production names.  There is no certain way to determine whether a given
;;;; string is part of a production name or a full name.
;;;;
;;;; When doing continuation of a search, the last-found location is
;;;; cached, instead of the first.  Not only is it slightly easier to
;;;; implement, but this also seems like the natural thing to do, since the
;;;; user is likely to continue searching mainly if the first thing found is
;;;; not the desired one.
;;;;
;;;; The use of tags is limited to the standard Emacs tags facility.  I was
;;;; tempted to use some of the enhanced tags packages available (particularly
;;;; Joe Wells' version) but that would force the user to adopt some changes
;;;; to the tags behavior that they may not want.  The user should be able
;;;; to use their favorite tags enhancement package if they wish.
;;;;
;;;; Notes and lessons from failed attempts:
;;;;
;;;; It would be nice to be able to hash the names of productions in files
;;;; visited, storing the buffer and file name where each production is found,
;;;; so that subsequent find-production calls on known productions could be
;;;; sped up.  Unfortunately, doing this is expensive even for a moderately
;;;; sized file (60k) on a fast machine.  The reason is that you have to loop
;;;; through the file looking for regexp matches of production names and then
;;;; do a buffer-substring for each name.  This results in a lot of consing
;;;; since buffer-substring allocates a new string each time.
;;;;
;;;; Another idea that didn't pay off is having the Production Editor add each
;;;; new file buffer to a special list in order to make searching through
;;;; buffers a little faster.  Reason: there are no hooks or other
;;;; straightforward mechanism in Emacs that could be used to remove a buffer
;;;; name from the list when the buffer is killed.  You're forced to do
;;;; significant bookkeeping.  In the end it seems better to just run through
;;;; (buffer-list) each time that sde-find-production needs to search through
;;;; buffers.
;;;;
;;;; When constructing lists of file and directory names, we generally want
;;;; only unique entries.  Emacs doesn't have good primitives for this -- it
;;;; doesn't even have a built-in member function (one that uses equal for
;;;; comparison) prior to Emacs 19.  In the end I decided to use hash tables,
;;;; which can be implemented cheaply using Emacs' symbol table functions.
;;;; The hash table functions are in the main file, sde.el.


;;;-----------------------------------------------------------------------------
;;; 1.  Require, provide, and miscellaneous setup.
;;;     Do not modify these.
;;;-----------------------------------------------------------------------------

;; Requirements

(require 'sde)


;;;-----------------------------------------------------------------------------
;;; 2.  Global parameters and configuration variables
;;;
;;; Users may wish to customize the values of these symbols, by resetting their
;;; values via setq in their .emacs files.
;;;-----------------------------------------------------------------------------



;;;-----------------------------------------------------------------------------
;;; 3.  Internal constants and variables
;;;-----------------------------------------------------------------------------

(if (not (boundp 'tags-file-name))	; Avoid errors if tags-file-name not set
    (defvar tags-file-name nil))


;;;-----------------------------------------------------------------------------
;;; 4.  Main code.
;;;-----------------------------------------------------------------------------

;; Global variables to keep state information across calls.  The
;; sde-find-last-xxx variables are for implementing sde-find-next functionality.

(defvar sde-find-production-args nil)
(defvar sde-find-last-buffer nil)
(defvar sde-find-last-point  nil)
(defvar sde-find-last-name   nil)
(defvar sde-find-buffer-list nil)

(defmacro sde-show-in-buffer (buffer where)
  (` (progn
       (if (sde-buffer-window (, buffer))
	   (progn
	     (sde-pop-to-buffer (, buffer))
	     (goto-char (, where))
	     (beginning-of-line))
	   (progn
	     (set-buffer (, buffer))
	     (goto-char (, where))
	     (beginning-of-line)
	     (sde-pop-to-buffer (, buffer))
	     (recenter '(center)))))))


;; The next two functions are the main entry points to this module.

(defun sde-find-production (name &optional continue)
  "Find the source code of the production whose name is under the cursor.
This will also find productions whose names are longer than and contain the
name under the cursor.  If given a prefix argument, prompts for a string and
searches for productions whose names contain the given string.

This command uses a number of heuristics to attempt to locate productions in
their source files.  The following describes the search procedure:

  \(a) Search through existing Emacs buffers for the first match.
  \(b) If fail, attempt to look up the name using the standard Emacs tag
       facility.  You must have built a tags table for your Soar files in
       order for this to work.
  \(c) If fail, search through all files known to have either been edited in
       SDE or loaded into a Soar subprocess in Soar Mode.
  \(d) If fail, search through all files found in the directories of the
       files from (c).

Since some production names may be substrings of other, longer production
names, the first match found by this command may not be the production you
are looking for.  Use the command `sde-find-next-production' \(\\[sde-find-next-production])
to continue searching for the same name from the current file and file
position."
  (interactive (list (if current-prefix-arg
			 (sde-prompt "Find production: " 'sde-find-production-args)
			 (sde-sp-name-near-point))
		     nil))
  (let ((hashdata (sde-gethash name))
	(regexp (sde-make-name-regexp name))
	buffer file pos)
    (setq sde-find-last-name        name
	  sde-find-last-buffer      nil
	  sde-find-last-point       nil
	  sde-find-buffer-list (sde-get-sde-buffers))
    (if hashdata
	(if (and (setq buffer (car hashdata))
		 (not (sde-killed-buffer-p buffer))
		 (setq pos (sde-search-buffer buffer regexp)))
	    (progn
	      (sde-show-in-buffer buffer pos)
	      (setq sde-find-last-buffer buffer
		    sde-find-last-point  pos))
	    ;; Buffer doesn't exist or production not in that buffer anymore.
	    ;; Try the cached file name.
	    (if (and (setq file (car (cdr hashdata)))
		     (setq buffer (find-file-noselect file))
		     (setq pos (sde-search-buffer buffer regexp)))
		(progn
		  (sde-puthash name (list buffer file))
		  (sde-show-in-buffer buffer pos)
		  (setq sde-find-last-buffer buffer
			sde-find-last-point  pos))
		;; Not in buffer, not in file.  Remove the hash entry and
		;; resort to searching.
		(progn
		  (sde-puthash name nil)
		  (sde-find-by-search name))))
	;; No hash entry, so just search.
	(sde-find-by-search name))))


(defun sde-find-next-production ()
  "Find the next production whose name was last searched-for.  
Strings searched for using `sde-find-production' (\\[sde-find-production]) are remembered.
This function allows you to search for the next occurrence of a string
in your files.  This is useful because production names may not be unique
and/or may be substrings of other, longer production names, and the first
match found may not be the desired one."
  (interactive)
  (sde-find-by-search sde-find-last-name 'continue))


(defun sde-find-by-search (name &optional continue)
  ;; The main control loop of the find-production search operations.
  ;; Searches for production named NAME through buffers, files and
  ;; directories.  Optional arg CONTINUE indicates that this is a
  ;; continuation of the last find-production search.
  (let ((regexp (sde-make-name-regexp name)))
    (cond ((sde-find-in-buffers     name regexp continue))
	  ((sde-find-tag            name regexp continue))
	  ((sde-find-in-files       name regexp))
	  ((sde-find-in-directories name regexp))
	  ((if continue
	       (error "No more occurrences of %s" name)
	       (error "Cannot find %s" name))))))


(defun sde-find-in-buffers (name pattern &optional continue)
  (let (found)
    (if continue
	(cond ((null sde-find-last-buffer)
	       (setq sde-find-buffer-list (sde-get-sde-buffers)))
	      
	      ((sde-killed-buffer-p sde-find-last-buffer)
	       ;; We're continuing, but the buffer has been killed.
	       (setq sde-find-last-buffer nil)
	       (setq sde-find-buffer-list (cdr sde-find-buffer-list)))
	      
	      ((setq sde-find-last-point
		     (sde-search-buffer sde-find-last-buffer pattern
					sde-find-last-point))
	       (sde-show-in-buffer sde-find-last-buffer sde-find-last-point)
	       (setq found t))

	      (t
	       (setq sde-find-buffer-list
		     (cdr (sde-member sde-find-last-buffer sde-find-buffer-list))))))
    (if (not found)
	(let ((buffers sde-find-buffer-list)
	      buf pos)
	  (while (setq buf (car buffers))
	    (if (setq pos (sde-search-buffer buf pattern))
		(progn
		  (sde-puthash name (list buf (sde-buffer-file-name buf)))
		  (sde-show-in-buffer buf pos)
		  (setq sde-find-last-buffer buf
			sde-find-last-point  pos
			buffers              nil
			found                t))
		(setq buffers (cdr buffers))))))
    found))


(defun sde-find-tag (name pattern &optional continue)
  ;; Look up the string NAME in the current tag table.  If optional argument
  ;; CONTINUE is non-nil, do a tags-loop-continue.  Returns t if found
  ;; definition.
  (if tags-file-name
      ;; Try tags.
      (if continue
	  (if (tags-loop-continue)
	      (progn
		(sde-puthash name (list (current-buffer) buffer-file-name))
		(setq sde-find-last-buffer (current-buffer)
		      sde-find-last-point  (point))
		t))
	  ;; Else, first time using tags.  Don't let errors stop the whole
	  ;; search -- just return nil.
	  (condition-case nil
	      (find-tag pattern continue)
	    (error nil)))))


(defun sde-find-in-files (name pattern &optional file-table)
  ;; Search for string NAME through the files known to SDE to be related to
  ;; the current task.  Does this by searching all the files in
  ;; sde-known-files which don't already have buffers.  Returns non-nil if
  ;; found.
  (let* ((where nil)
	 (found
	  (catch 'found
	    (sde-map-hashtable (function 
				(lambda (entry val)
				 (if (and val
					  (not (get-file-buffer val))
					  (setq where (sde-search-file val pattern)))
				     (throw 'found val))))
			       (or file-table sde-known-files))
	    nil)))
    (if found
	(if (condition-case nil
		(set-buffer (find-file-noselect found))
	      (file-error nil))
	    ;; Found file and was able to read it okay.
	    (progn
	      (setq sde-find-last-buffer (current-buffer)
		    sde-find-last-point  where)
	      (sde-puthash name (list sde-find-last-buffer buffer-file-name))
	      (sde-show-in-buffer sde-find-last-buffer where)
	      (message "Found in file %s" found))
	    ;; !!! Should do something if found file but couldn't read it.
	    ))
    found))
    

(defun sde-find-in-directories (name pattern)
  (let ((dirs-table (sde-make-hashtable 103))
	(files-table (sde-make-hashtable 103))
	files)
    ;; Create a table of directories.
    (sde-map-hashtable (function
			(lambda (entry val)
			 (if val
			     (let ((dir-name (file-name-directory val)))
			       (sde-puthash dir-name dir-name dirs-table)))))
		       sde-known-files)
    ;; Create a list of files in those directories.
    (sde-map-hashtable (function
			(lambda (entry val)
			 (let ((files (directory-files val t sde-file-extensions-regexp)))
			   (while files
			     (sde-puthash (car files) (car files) files-table)
			     (setq files (cdr files))))))
		       dirs-table)
    ;; Search through files for pattern.
    (sde-find-in-files name pattern files-table)))


;;;-----------------------------------------------------------------------------
;;; 5.  Closing statements.
;;;-----------------------------------------------------------------------------

;; Form hooks for debugging with Edebug.

(put 'sde-show-in-buffer    'edebug-form-spec '(form form))
