;;; reporter.el --- customizable bug reporting of lisp programs

;; Author: 1993 Barry A. Warsaw, Century Computing Inc. <bwarsaw@cen.com>
;; Maintainer:      bwarsaw@cen.com
;; Created:         19-Apr-1993
;; Version:         1.10
;; Last Modified:   1993/05/17 00:07:37
;; Keywords: bug reports lisp

;; Copyright (C) 1993 Barry A. Warsaw

;; This file is not yet part of GNU Emacs.
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Introduction
;; ============
;; This program is for lisp package authors and is used to ease
;; reporting of bugs.  When envoked, reporter-submit-bug-report will
;; set up a mail buffer with the appropriate bug report address,
;; including a lisp expression the maintainer of the package can use
;; to completely reproduce the environment in which the bug was
;; observed (e.g. by using eval-last-sexp). This package is especially
;; useful for my development of c++-mode.el, which is highly dependent
;; on its configuration variables.
;;
;; Do a "C-h f reporter-submit-bug-report" for more information.

;;; Code:


;; vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
;; user defined variables

(defvar reporter-mailer 'mail
  "*Mail package to use to generate bug report buffer.")

(defvar reporter-show-help t
  "*Whether the report function should print a help message when invoked.
Users or package may set this to nil to turn off the printing of the 
help message.")

;; ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;; end of user defined variables


(defun reporter-dump-variable (varsym)
  "Pretty-print the value of the variable in symbol VARSYM."
  (let ((val (eval varsym))
	(sym (symbol-name varsym))
	(print-escape-newlines t))
    (insert "     " sym " "
	    (if (or (listp val) (symbolp val)) "'" "")
	    (prin1-to-string val)
	    "\n")))

(defun reporter-dump-state (pkgname varlist pre-hooks post-hooks)
  "Dump the state of the mode specific variables.
PKGNAME contains the name of the mode as it will appear in the bug
report (you must explicitly add any version numbers).

VARLIST is the list of variables to dump.  Each element in VARLIST can
be a variable symbol, or a cons cell.  If a symbol, this will be
passed to reporter-dump-variable for insertion into the mail buffer.
If a cons cell, the car must be a variable symbol and the cdr must be
a function which will be funcall'd with the symbol. Use this to write
your own custom variable value printers for specific variables.

PRE-HOOKS is run after the emacs-version and PKGNAME are inserted, but
before the VARLIST is dumped.  POST-HOOKS is run after the VARLIST is
dumped."
  (let ((buffer (current-buffer)))
    (set-buffer buffer)
    (insert "Emacs  : " (emacs-version) "\nPackage: " pkgname "\n")
    (run-hooks 'pre-hooks)
    (insert "\ncurrent state:\n==============\n(setq\n")
    (mapcar
     (function
      (lambda (varsym-or-cons-cell)
	(let ((varsym (or (car-safe varsym-or-cons-cell)
			  varsym-or-cons-cell))
	      (printer (or (cdr-safe varsym-or-cons-cell)
			   'reporter-dump-variable)))
	  (funcall printer varsym)
	  )))
     varlist)
    (insert "     )\n")
    (run-hooks 'post-hooks)
    ))

(defun reporter-submit-bug-report
    (address pkgname varlist &optional subject salutation pre-hooks post-hooks)
  "Submit a bug report via mail.

ADDRESS is the email address for the package's maintainer. 
PKGNAME is the name of the package.  You must explicitly include any 
desired version information in this string.
VARLIST is the list of variables to dump (do a \"\\[describe-function] reporter-dump-state\"
for details).  Optional argument SUBJECT is the subject string to
use in the mail message.  By default, reporter uses the subject line

   \"Report on package PKGNAME\"

where PKGNAME is the 2nd argument as before.  Providing a non-nil
SUBJECT argument allows you to supply a custom subject line.  Optional
argument SALUTATION is inserted at the top of the mail buffer, and point
is left after the saluation.  Optional arguments PRE-HOOKS and
POST-HOOKS are lists of hook functions passed to reporter-dump-state.

The mailer used is described in the variable reporter-mailer."

  (let ((curbuf (current-buffer))
	(mailbuf (progn (call-interactively reporter-mailer)
			(current-buffer))))
    (require 'sendmail)
    (pop-to-buffer curbuf)
    (pop-to-buffer mailbuf)
    ;; different mailers use different separators, some may not even
    ;; use m-h-s, but sendmail.el stuff must have m-h-s bound.
    (let* ((mail-header-separator
	    (save-excursion
	      (re-search-forward
	       (concat
		"^\\("			;beginning of line
		(mapconcat
		 'identity
		 (list "[        ]*"	;simple SMTP form
		       "-+"		;mh-e form
		       mail-header-separator) ;sendmail.el form
		 "\\|")			;or them together
		"\\)$")			;end of line
	       nil
	       'move)			;search for and move
	      (buffer-substring (match-beginning 0) (match-end 0))))
	   (buf-id (and (stringp (car mode-line-buffer-identification))
			(car mode-line-buffer-identification)))
	   (region-end nil)
	   (sendkey "C-c C-c")		;can this be generalized like below?
	   (killkey-whereis (where-is-internal 'kill-buffer nil t))
	   (killkey (if killkey-whereis
			(key-description killkey-whereis)
			"ESC x kill-buffer")))
      (mail-position-on-field "to")
      (insert address)
      (mail-position-on-field "subject")
      (insert (or subject "Report on package " pkgname))
      (re-search-forward mail-header-separator (point-max) 'move)
      (forward-line 1)
      (insert "\nDear " (or salutation address) ",\n\n")
      (set-mark (point))                ;user should see mark change
      (insert "\n\n")
      (setq region-end (point))
      (reporter-dump-state pkgname varlist pre-hooks post-hooks)
      (exchange-point-and-mark)
      ;; Hide the state info
      (narrow-to-region (point-min) region-end)
      ;; For some reason, setting these in the let stmt doesn't work.
      ;; Need to setq them explicitly.
      (setq mode-line-buffer-identification
	    ;; Try to shorten the space allocated to the buffer name.
	    (if (and buf-id
		     (string-match "\\`\\([^0-9]*\\)[0-9]+\\(.*\\)\\'" buf-id))
		(list (concat
		       (substring buf-id (match-beginning 1) (match-end 1))
		       "8"
		       (substring buf-id (match-beginning 2) (match-end 2))))
		mode-line-buffer-identification))
      (setq mode-line-format
	    (list "" 'mode-line-modified 'mode-line-buffer-identification
		  (concat "(Type C-c C-c to send message, "
			  (substitute-command-keys "\\[kill-buffer]")
			  " to abort.)")
		  "  %[("
		  'mode-name 'minor-mode-alist "%n"
		  ")%]----"
		  '(-3 . "%p") "-%-"))
      (message "Please type in your report.  Type %s to send, %s to abort. "
	       sendkey killkey))))


;; this is useful
(provide 'reporter)

;;; reporter.el ends here
