;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; help-screen.el -- Macro to build help-for-* style functions
;; 
;; Copyright (c) 1990 Lynn Randolph Slater, Jr
;;   Copying terms identical to that of Gnu Emacs
;; 
;; Author          : Lynn Slater (lrs@indetech.com)
;; Created On      : Sun Oct 28 14:54:16 1990
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Wed Aug 28 09:33:32 1991
;; Update Count    : 33
;; Status          : GEET General Release 2d Patch 0
;; 
;; HISTORY
;; PURPOSE
;; 	This file contains functions that construct help-lines and help
;; screens such as that used by help-for-help. These screens are
;; characterized by a prefix character, a 1 line list of initial
;; characters that appears if the help character is pressed, and a help
;; description screen that appears if the help character is pressed again.
;;
;; These functions should be placed into help.el
;;
;; This code is used buy the Gnu Emacs Empire Tool (GEET) but nothing in
;; this file is special to empire.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; COPYING which is distributed with GnU Emacs. These terms constitute what
;; the Free Software Foundation calls a COPYLEFT. 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'help-screen)
(require 'backquote)

;; Support changable help characters (per Mark Baushke, Rich Berlin, and Lynn Slater)
(defvar help-character (format "%c" help-char))	; current command string
(defvar help-ch help-char)			; current command character
						; (an integer)

(defmacro make-help-screen (fname help-line help-text helped-map)
  "Constructs function FNAME that when invoked shows HELP-LINE and if a help
character is requested, shows HELP-TEXT. The user is prompted for a character
from the HELPED-MAP.

This operation is the same as used by help-for-help.
This fcn supports the extended help character choices advocated by 
Mark Baushke, Rich Berlin, and Lynn Slater"
  (` (defun (, fname) ()
	   (, help-text)
	   (interactive)
	   (let ((line-prompt
		  (substitute-command-keys (, help-line))))
	     (message line-prompt)
	     (let ((char (read-char)))
	       (if (or (= char ??) (= char help-ch))
		   (save-window-excursion
		     (switch-to-buffer-other-window "*Help*")
		     (erase-buffer)
		     (set-window-hscroll (selected-window) 0)
		     (insert (documentation (quote (, fname))))
		     (goto-char (point-min))
		     (while (memq char (cons help-ch '(?? ?\C-v ?\ ?\177 ?\M-v)))
		       (if (memq char '(?\C-v ?\ ))
			   (scroll-up))
		       (if (memq char '(?\177 ?\M-v))
			   (scroll-down))
		       (message "%s%s: "
				line-prompt
				(if (pos-visible-in-window-p (point-max))
				    "" " or Space to scroll"))
		       (let ((cursor-in-echo-area t))
			 (setq char (read-char))))))
	       (let ((defn (if (vectorp (, helped-map))
			       (aref (, helped-map) char)
			     (cdr (assq char (, helped-map)))
			     )))
		 (if defn
		     (if (keymapp defn)
			 (error "Sorry, this command cannot be run from the help screen.  Start over.")
		       (call-interactively defn))
		   (ding))))))
     ))

;; Here is an example defn of a help screen and its mode map

;;example-> ;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;example-> ;; Redistribution map
;;example-> ;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;example-> (defvar empire-shell-redistribute-map nil
;;example->   "Keymap used in shell or empire shell mode with redistribution commands.")
;;example-> 
;;example-> (let ((c-mp (make-sparse-keymap)))
;;example->   (define-key c-mp "?"	  'help-for-empire-redistribute-map)
;;example->   (define-key c-mp "\C-h" 'help-for-empire-redistribute-map)
;;example->   (define-key c-mp help-character 'help-for-empire-redistribute-map)
;;example->   (define-key c-mp "m"		'redistribute-mil)
;;example->   (define-key c-mp "f"		'redistribute-food)
;;example->   (define-key c-mp "p"		'redistribute-population)
;;example->   (define-key c-mp "c"		'redistribute-civ)
;;example->   (setq empire-shell-redistribute-map c-mp))
;;example-> 
;;example-> (make-help-screen help-for-empire-redistribute-map
;;example-> 		  "c:civ m:mil p:population f:food ?"
;;example-> 		  "You have discovered the GEET redistribution commands
;;example->    From here, you can use the following options:
;;example-> 
;;example-> c	Redistribute civs from overfull sectors into connected underfull ones
;;example-> 	  The functions typically named by empire-ideal-civ-fcn control
;;example->           based in part on empire-sector-civ-threshold  
;;example-> m	Redistribute military using levels given by empire-ideal-mil-fcn
;;example-> p	Redistribute excess population to highways for max pop growth
;;example-> 	  Excess is any sector so full babies will not be born.
;;example-> f	Even out food on highways to highway min and leave levels
;;example-> 	  This is good to pump max food to all warehouses/dist pts
;;example-> 
;;example-> 
;;example-> Use \\[help-for-empire-redistribute-map] for help on redistribution.
;;example-> Use \\[help-for-empire-extract-map] for help on data extraction.
;;example-> Please use \\[describe-key] to find out more about any of the other keys."
;;example-> 		  empire-shell-redistribute-map)
;;example-> 

;; and in another mode map, the line
;;example->   (define-key c-mp "\C-d" empire-shell-redistribute-map)
;; is used to bind the help line commands under control-d. When ? or c-H is
;; pressed, the help line is shown, and when the same char is pressed
;; again, the help screen is shown.
