/* ===================================================================
                       Main file for Soar 6
=================================================================== */

#include <signal.h>         /* used for control-c hanler */
#ifdef __hpux
#include <sys/types.h>
#define _INCLUDE_HPUX_SOURCE
#endif /* __hpux */
#include <sys/time.h>       /* used for timing stuff */
#include <sys/resource.h>   /* used for timing stuff */
#ifdef __hpux
#undef _INCLUDE_HPUX_SOURCE
#endif /* __hpux */
#include "soar.h"

#ifdef __hpux
#include <sys/syscall.h>
#define getrusage(a, b) syscall(SYS_GETRUSAGE, a, b)
#endif /* __hpux */
/* ===================================================================

                            Exiting Soar

   Exit_soar() and abort_with_fatal_error() both terminate Soar, closing
   the log file before exiting.  Abort_with_fatal_error() also prints
   an error message before exiting.
=================================================================== */

void exit_soar (void) {
  system_termination_hook (TRUE);
  if (logging_to_file) stop_log_file ();
  exit (0);
}

void abort_with_fatal_error (void) {
  print ("Soar cannot recover from this error.  Aborting...\n");
  system_termination_hook (FALSE);
  if (logging_to_file) stop_log_file ();
  exit (1);
}

/* ===================================================================
   
                        Signal Handling

   Setup things so control_c_handler() gets control whenever the program
   receives a SIGINT (e.g., from a ctrl-c at the keyboard).  The handler
   just sets the stop_soar flag.
=================================================================== */

void control_c_handler (int the_signal) {
  stop_soar = TRUE;
  reason_for_stopping = "*** Ctrl-C Interrupt ***";
  /* --- reinstall this signal handler -- some brain-damaged OS's uninstall
     it after delivering the signal --- */
  signal (SIGINT, control_c_handler);
}

void setup_signal_handling (void) {
  signal (SIGINT, control_c_handler);
/* BUGBUG according to the ANSI standard, we're supposed to check whether
   the result of the signal() call is SIG_ERR--if so, it means that the
   signal handler was not properly set up, and thus ctrl-c won't work.
   I couldn't seem to get this to compile right at CMU or U-M, though.
*/
}

/* ===================================================================

                       Timer Utility Routines

   These are utility routines for using timers.  We use (struct timeval)'s
   (defined in a system include file) for keeping track of the cumulative
   time spent in one part of the system or another.  Reset_timer()
   clears a timer to 0.  Start_timer() and stop_timer() are used for
   timing an interval of code--the usage is:
   
     start_timer (&timeval_to_record_the_start_time_in); 
     ... other code here ...
     stop_timer (&timeval_to_record_the_start_time_in,
                 &timeval_holding_accumulated_time_for_this_code);

   Finally, timer_value() returns the accumulated value of a timer
   (in seconds).
=================================================================== */

#define ONE_MILLION (1000000)

void reset_timer (struct timeval *tv_to_reset) {
  tv_to_reset->tv_sec = 0;
  tv_to_reset->tv_usec = 0;
}

void get_cputime_from_rusage (struct rusage *r, struct timeval *dest_tv) {
  dest_tv->tv_sec = r->ru_utime.tv_sec + r->ru_stime.tv_sec;
  dest_tv->tv_usec = r->ru_utime.tv_usec + r->ru_stime.tv_usec;
  if (dest_tv->tv_usec >= ONE_MILLION) {
    dest_tv->tv_usec -= ONE_MILLION;
    dest_tv->tv_sec++;
  }
}

void start_timer (struct timeval *tv_for_recording_start_time) {
  struct rusage temp_rusage;
  
  getrusage (RUSAGE_SELF, &temp_rusage);
  get_cputime_from_rusage (&temp_rusage, tv_for_recording_start_time);
}

void stop_timer (struct timeval *tv_with_recorded_start_time,
                 struct timeval *tv_with_accumulated_time) {
  struct rusage end_rusage;
  struct timeval end_tv;
  long delta_sec, delta_usec;
  
  getrusage (RUSAGE_SELF, &end_rusage);
  get_cputime_from_rusage (&end_rusage, &end_tv);

  delta_sec = end_tv.tv_sec - tv_with_recorded_start_time->tv_sec;
  delta_usec = end_tv.tv_usec - tv_with_recorded_start_time->tv_usec;
  if (delta_usec < 0) {
    delta_usec += ONE_MILLION;
    delta_sec--;
  }

  tv_with_accumulated_time->tv_sec += delta_sec;
  tv_with_accumulated_time->tv_usec += delta_usec;
  if (tv_with_accumulated_time->tv_usec >= ONE_MILLION) {
    tv_with_accumulated_time->tv_usec -= ONE_MILLION;
    tv_with_accumulated_time->tv_sec++;
  }
}

double timer_value (struct timeval *tv) {
  return (double)(tv->tv_sec) + (double)(tv->tv_usec)/(double)ONE_MILLION;
}

/* ===================================================================
   
                            Sysparams

=================================================================== */

long sysparams[HIGHEST_SYSPARAM_NUMBER+1];

void set_sysparam (int param_number, long new_value) {
  if ((param_number < 0) || (param_number > HIGHEST_SYSPARAM_NUMBER)) {
    print ("Internal error: tried to set bad sysparam #: %d\n", param_number);
    return;
  }
  sysparams[param_number] = new_value;
  system_parameter_changed_hook (param_number);
}

void init_sysparams (void) {
  int i;

  for (i=0; i<HIGHEST_SYSPARAM_NUMBER+1; i++) sysparams[i] = 0;
  
  /* --- set all params to zero, except the following: --- */
  sysparams[TRACE_CONTEXT_DECISIONS_SYSPARAM] = TRUE;
  sysparams[TRACE_FIRINGS_OF_CHUNKS_SYSPARAM] = TRUE;
  sysparams[TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM] = NONE_WME_TRACE;
  sysparams[TRACE_CHUNK_NAMES_SYSPARAM] = TRUE;
  sysparams[TRACE_JUSTIFICATION_NAMES_SYSPARAM] = TRUE;
  sysparams[MAX_ELABORATIONS_SYSPARAM] = 100;
  sysparams[LEARNING_ON_SYSPARAM] = TRUE;
  sysparams[LEARNING_ALL_GOALS_SYSPARAM] = TRUE;
  sysparams[USER_SELECT_MODE_SYSPARAM] = USER_SELECT_FIRST;
  sysparams[PRINT_WARNINGS_SYSPARAM] = TRUE;
}

/* ===================================================================
   
                     Adding and Removing Ptraces

   Productions_being_traced is a (consed) list of all productions
   on which a ptrace has been set.  Ptraces are added/removed via
   calls to add_ptrace() and remove_ptrace().
=================================================================== */

list *productions_being_traced = NIL; /* list of production structures */

void add_ptrace (production *prod) {
  if (prod->trace_firings) return;
  prod->trace_firings = TRUE;
  push (prod, productions_being_traced);
}

production *prod_to_remove_ptrace_of;

bool remove_ptrace_test_fn (cons *c) {
  return (c->first == prod_to_remove_ptrace_of);
}

void remove_ptrace (production *prod) {
  if (! prod->trace_firings) return;
  prod->trace_firings = FALSE;
  prod_to_remove_ptrace_of = prod;
  free_list (extract_list_elements (&productions_being_traced,
                                    remove_ptrace_test_fn));
}

/* ===================================================================
   
                        Global Variables, Etc.

=================================================================== */

/* --- list of symbols (problem space names) declared chunk-free --- */
list *chunk_free_problem_spaces = NIL;

/* --- during firing, this points to the production being fired --- */
production *production_being_fired = NIL;

/* --- to interrupt at the end of the current phase, set stop_soar to TRUE
   and reason_for_stopping to some appropriate string --- */
bool stop_soar = FALSE;           
char *reason_for_stopping = "";

/* --- current top level phase --- */
enum top_level_phase current_phase = INPUT_PHASE;

/* --- the RHS action (halt) sets this TRUE --- */
bool system_halted = FALSE;

unsigned long d_cycle_count;            /* # of DC's run so far */
unsigned long e_cycle_count;            /* # of EC's run so far */
unsigned long e_cycles_this_d_cycle;    /* # of EC's run so far this DC */
unsigned long production_firing_count;  /* # of production firings */
unsigned long wme_addition_count;       /* # of wmes added to WM */
unsigned long wme_removal_count;        /* # of wmes removed from WM */

unsigned long max_wm_size;    /* maximum size of WM so far */
double cumulative_wm_size;    /* running total of WM sizes at end of phases */
unsigned long num_wm_sizes_accumulated; /* # of items included in above sum */

/* --- accumulated cpu time spent in various parts of the system --- */
struct timeval total_cpu_time;
struct timeval match_cpu_time;
#ifdef DETAILED_TIMING_STATS
struct timeval ownership_cpu_time;
struct timeval chunking_cpu_time;
struct timeval preference_phase_cpu_time;
struct timeval create_instantiations_cpu_time;
struct timeval o_support_cpu_time;
#endif

/* ===================================================================
   
                         Reinitializing Soar

   Reset_statistics() resets all the statistics (except the firing counts
   on each individual production).  Reinitialize_soar() does all the 
   work for an init-soar.
=================================================================== */

void reset_statistics (void) {
  d_cycle_count = 0;
  e_cycle_count = 0;
  e_cycles_this_d_cycle = 0;
  production_firing_count = 0;
  wme_addition_count = 0;
  wme_removal_count = 0;
  max_wm_size = 0;
  cumulative_wm_size = 0.0;
  num_wm_sizes_accumulated = 0;
  reset_timer (&total_cpu_time);
  reset_timer (&match_cpu_time);
#ifdef DETAILED_TIMING_STATS
  reset_timer (&ownership_cpu_time);
  reset_timer (&chunking_cpu_time);
  reset_timer (&preference_phase_cpu_time);
  reset_timer (&create_instantiations_cpu_time);
  reset_timer (&o_support_cpu_time);
#endif
}

void reinitialize_soar (void) {
  before_init_soar_hook();
  /* BUGBUG ought to temporarily set tracing flags to FALSE so we don't
     get tons of extra info being printed out during the call to
     do_preference_phase(). */
  clear_goal_stack ();
  do_preference_phase ();   /* allow all instantiations to retract */
  reset_id_counters ();
  reset_wme_timetags ();
  reset_statistics ();
  system_halted = FALSE;
  after_init_soar_hook();
}

/* ===================================================================
   
                            Running Soar

   Do_one_top_level_phase() runs Soar one top-level phase.  Note that
   this does not start/stop the total_cpu_time timer--the caller must
   do this.

   Each of the following routines runs Soar for a certain duration,
   or until stop_soar gets set to TRUE.
     - Run_forever() runs Soar forever.
     - Run_for_n_phases() runs Soar for a given number (n) of top-level
       phases.  (If n==-1, it runs forever.)
     - Run_for_n_elaboration_cycles() runs Soar for a given number (n)
       of elaboration cycles.  (Here, quiescence phase is counted as
       an elaboration cycle.)  (If n==-1, it runs forever.)
     - Run_for_n_decision_cycles() runs Soar for a given number (n) of
       decision cycles.  (If n==-1, it runs forever.)
     - Run_for_n_selections_of_slot (long n, symbol *attr_of_slot): this
       runs Soar until the nth time a selection is made for a given
       type of slot.  Attr_of_slot should be either goal_symbol,
       problem_space_symbol, state_symbol, or operator_symbol.
     - Run_for_n_selections_of_slot_at_level (long n, symbol *attr_of_slot,
       goal_stack_level level):  this runs Soar for n selections of the
       given slot at the given level, or until the goal stack is popped
       so that level no longer exists.
=================================================================== */

void do_one_top_level_phase (void) {
  if (system_halted) {
    print ("\nSystem halted.  Use (init-soar) before running Soar again.");
    stop_soar = TRUE;
    reason_for_stopping = "System halted.";
    return;
  }
  
  if (! top_goal) {
    create_top_goal();
    if (sysparams[TRACE_CONTEXT_DECISIONS_SYSPARAM]) {
      print_string ("\n");
      print_lowest_slot_in_context_stack ();
    }
    current_phase = INPUT_PHASE;
  }

  switch (current_phase) {
  case INPUT_PHASE:
    if (e_cycles_this_d_cycle==0) before_decision_cycle_hook ();
    before_input_phase_hook ();
    do_input_cycle();
    after_input_phase_hook ();
    if (any_assertions_or_retractions_ready())
      current_phase = PREFERENCE_PHASE;
    else
      current_phase = QUIESCENCE_PHASE;
    break;
    
  case PREFERENCE_PHASE:
    before_preference_phase_hook ();
    do_preference_phase();
    after_preference_phase_hook ();
    current_phase = WM_PHASE;
    break;
    
  case WM_PHASE:
    before_wm_phase_hook ();
    do_working_memory_phase();
    after_wm_phase_hook ();
    current_phase = OUTPUT_PHASE;
    break;
    
  case OUTPUT_PHASE:
    before_output_phase_hook ();
    do_output_cycle();
    after_output_phase_hook ();
    e_cycle_count++;
    e_cycles_this_d_cycle++;
    if (e_cycles_this_d_cycle >= sysparams[MAX_ELABORATIONS_SYSPARAM]) {
      if (sysparams[PRINT_WARNINGS_SYSPARAM])
        print ("\nWarning: reached max-elaborations; proceeding to quiescence phase.");
      current_phase = QUIESCENCE_PHASE;
    } else {
      current_phase = INPUT_PHASE;
    }
    break;
    
  case QUIESCENCE_PHASE:
    d_cycle_count++;
    before_quiescence_phase_hook ();
    do_quiescence_phase();
    after_quiescence_phase_hook ();
    after_decision_cycle_hook ();
    if (sysparams[TRACE_CONTEXT_DECISIONS_SYSPARAM]) {
      print_string ("\n");
      print_lowest_slot_in_context_stack ();
    }
    e_cycles_this_d_cycle = 0;
    current_phase = INPUT_PHASE;
    break;
  }

  /* --- update WM size statistics --- */
  if (num_wmes_in_rete > max_wm_size) max_wm_size = num_wmes_in_rete;
  cumulative_wm_size += num_wmes_in_rete;
  num_wm_sizes_accumulated++;
  
  if (system_halted) {
    stop_soar = TRUE;
    reason_for_stopping = "System halted.";
    after_halt_soar_hook ();
  }
  
  if (stop_soar) print ("\n%s", reason_for_stopping);
}

struct timeval start_total_tv;

void run_forever (void) {
  start_timer (&start_total_tv);
  stop_soar = FALSE;
  reason_for_stopping = "";
  while (! stop_soar) {
    do_one_top_level_phase();
  }
  stop_timer (&start_total_tv, &total_cpu_time);
}

void run_for_n_phases (long n) {
  if (n == -1) { run_forever(); return; }
  if (n < -1) return;
  start_timer (&start_total_tv);
  stop_soar = FALSE;
  reason_for_stopping = "";
  while (!stop_soar && n) {
    do_one_top_level_phase();
    n--;
  }
  stop_timer (&start_total_tv, &total_cpu_time);
}

void run_for_n_elaboration_cycles (long n) {
  long e_cycles_at_start, d_cycles_at_start, elapsed_cycles;
  
  if (n == -1) { run_forever(); return; }
  if (n < -1) return;
  start_timer (&start_total_tv);
  stop_soar = FALSE;
  reason_for_stopping = "";
  e_cycles_at_start = e_cycle_count;
  d_cycles_at_start = d_cycle_count;
  while (!stop_soar) {
    elapsed_cycles = (d_cycle_count-d_cycles_at_start) +
                     (e_cycle_count-e_cycles_at_start);
    if (n==elapsed_cycles) break;
    do_one_top_level_phase();
  }
  stop_timer (&start_total_tv, &total_cpu_time);
}

void run_for_n_decision_cycles (long n) {
  long d_cycles_at_start;
  
  if (n == -1) { run_forever(); return; }
  if (n < -1) return;
  start_timer (&start_total_tv);
  stop_soar = FALSE;
  reason_for_stopping = "";
  d_cycles_at_start = d_cycle_count;
  while (!stop_soar) {
    if (n==d_cycle_count-d_cycles_at_start) break;
    do_one_top_level_phase();
  }
  stop_timer (&start_total_tv, &total_cpu_time);
}

symbol *attr_of_slot_just_decided (void) {
  if (bottom_goal->id.operator_slot->wmes) return operator_symbol;
  if (bottom_goal->id.state_slot->wmes) return state_symbol;
  if (bottom_goal->id.problem_space_slot->wmes) return problem_space_symbol;
  return goal_symbol;
}

void run_for_n_selections_of_slot (long n, symbol *attr_of_slot) {
  long count;
  bool was_quiescence_phase;
  
  if (n == -1) { run_forever(); return; }
  if (n < -1) return;
  start_timer (&start_total_tv);
  stop_soar = FALSE;
  reason_for_stopping = "";
  count = 0;
  while (!stop_soar && (count < n)) {
    was_quiescence_phase = (current_phase==QUIESCENCE_PHASE);
    do_one_top_level_phase();
    if (was_quiescence_phase)
      if (attr_of_slot_just_decided()==attr_of_slot) count++;
  }
  stop_timer (&start_total_tv, &total_cpu_time);
}

void run_for_n_selections_of_slot_at_level (long n,
                                            symbol *attr_of_slot,
                                            goal_stack_level level) {
  long count;
  bool was_quiescence_phase;
  
  if (n == -1) { run_forever(); return; }
  if (n < -1) return;
  start_timer (&start_total_tv);
  stop_soar = FALSE;
  reason_for_stopping = "";
  count = 0;
  while (!stop_soar && (count < n)) {
    was_quiescence_phase = (current_phase==QUIESCENCE_PHASE);
    do_one_top_level_phase();
    if (was_quiescence_phase) {
      if (bottom_goal->id.level < level) break;
      if (bottom_goal->id.level==level) {
        if (attr_of_slot_just_decided()==attr_of_slot) count++;
      }
    }
  }
  stop_timer (&start_total_tv, &total_cpu_time);
}

/* ===================================================================
   
             Loading the Initialization File ".init.soar"

   This routine looks for a file ".init.soar" in either the current
   directory or $HOME, and if found, loads it.
=================================================================== */

extern char *getenv();
void load_init_file (void) {
  char filename[1000];
  char *home_directory;
  FILE *initfile;

  strcpy (filename, ".init.soar");
  initfile = fopen (filename, "r");
  if (!initfile) {
    home_directory = getenv ("HOME");
    if (home_directory) {
      strcpy (filename, home_directory);
      strcat (filename, "/.init.soar");
      initfile = fopen (filename, "r");
    }
  }
  if (initfile) {
    print ("\nLoading %s\n",filename);
    load_file (filename, initfile);
    fclose (initfile);
  }
}

/* ===================================================================

                     Print the Startup Banner

=================================================================== */

void print_startup_banner (void) {
  /*
   * only show major and minor versions here.. this keeps us from 
   * having to update the tests every time we make a fix.
   */
  print ("Soar %d.%d\n", MAJOR_VERSION_NUMBER, MINOR_VERSION_NUMBER);
  print ("\n");
  print ("Bugs and questions should be sent to Soar-bugs@cs.cmu.edu\n");
  print ("The current bug-list may be obtained by sending mail to\n");
  print ("Soar-bugs@cs.cmu.edu with the Subject: line \"bug list\"\n");
  print ("This software is in the public domain.\n");
  print ("\n");
  print ("This software is made available AS IS, and Carnegie Mellon\n");
  print ("University and the University of Michigan make no warranty\n");
  print ("about the software or its performance.\n");
  print ("\n");
  print ("Type \"help\" for information on various topics.\n");
  print ("Type \"quit\" to exit.  Use ctrl-c to stop a Soar run.\n");
  print ("Type \"soarnews\" for news.\n");
  print ("Type \"version\" for complete version information.\n");
}

/* ===================================================================
   
                           Main Function

=================================================================== */

int main (int argc, char *(argv[]))
{
  /* --- set the random number generator seed to a "random" value --- */
  {
     struct timeval tv;
     gettimeofday (&tv, NIL);
     srand (tv.tv_usec);
   }

  /* --- initialize everything --- */
  setup_signal_handling();
  init_memory_utilities();
  init_symbol_tables();
  create_predefined_symbols();
  init_production_utilities();
  init_rete ();
  init_lexer ();
  init_parser ();
  init_firer ();
  init_decider ();
  init_soar_io ();
  init_text_io ();
  init_built_in_commands ();
  init_built_in_rhs_functions ();
  init_chunker ();
  init_sysparams ();
  init_tracing ();
  
  /* --- add default object trace formats --- */
  add_trace_format (FALSE, FOR_ANYTHING_TF, NIL,
                    "%id %ifdef[(%v[name])]");
  add_trace_format (FALSE, FOR_GOALS_TF, NIL,
                    "%id %ifdef[(%v[attribute] %v[impasse])]");
  { symbol *evaluate_object_sym;
    evaluate_object_sym = make_sym_constant ("evaluate-object");
    add_trace_format (FALSE, FOR_OPERATORS_TF, evaluate_object_sym,
                      "%id (evaluate-object %o[object])");
    symbol_remove_ref (evaluate_object_sym);
  }
  /* --- add default stack trace formats --- */
  add_trace_format (TRUE, FOR_GOALS_TF, NIL,
                    "%right[6,%dc]: %rsd[   ]==>G: %cg");
  add_trace_format (TRUE, FOR_PROBLEM_SPACES_TF, NIL,
                    "%right[6,%dc]: %rsd[   ]   P: %cp");
  add_trace_format (TRUE, FOR_STATES_TF, NIL,
                    "%right[6,%dc]: %rsd[   ]   S: %cs");
  add_trace_format (TRUE, FOR_OPERATORS_TF, NIL,
                    "%right[6,%dc]: %rsd[   ]   O: %co");
  
  print_startup_banner ();

  reset_statistics ();

  system_startup_hook ();

  load_init_file ();

  repeatedly_read_and_dispatch_commands (TRUE);

  exit_soar();  
  return 0; /* unreachable, but without it, gcc -Wall warns here */
}
