/****************************************************************************
 *	Windows Soar v1.0
 *	GUI code (c) 1993 Price Waterhouse,
 *	Soar 6.1 kernel (c) Carnegie Mellon University
 *
 * 	Windows Stubs
 */

#include <stdarg.h>
#include <stdlib.h>

#include "stdafx.h"
#include "../resource.h"
#include "../gui/soar.h"
#include "../gui/machdoc.h"
#include "../gui/accptdlg.h"
#include "../src/soar.h"

#include "windows.h"
#include "windowsx.h"
#include "toolhelp.h"
#include "string.h"

/* 
 * External soar routines I need to interface with
 */
extern "C" int soar_is_halted();
extern "C" int print_string_to_window(char *);
extern "C" char *get_line_from_window(char *buff,int maxchar);
extern "C" void get_next_char(void);
extern "C" int getwd(char *);
extern "C" void add_production_to_stat_lists(production *prod);
extern "C" void remove_production_from_stat_lists(production *prod);

struct timeval total_cpu_time;

// And a pointer to the application
extern CSoarApp FAR theApp;      

/**************************
 * UNIX EMULATION SECTION
 **************************/

/*
 * Timing stuff
 */
int getrusage(int type, struct rusage *use)
{
  TIMERINFO timer;
        
  timer.dwSize=sizeof(TIMERINFO);
  if (!TimerCount(&timer)) {
    print("getrusage: Call to TimerCount unsuccessful.\n");
    return FALSE;
  }
  if (type==RUSAGE_SELF) {
    use->ru_utime.tv_sec=timer.dwmsThisVM/1000L;
    use->ru_utime.tv_usec=timer.dwmsThisVM%1000L;
    use->ru_stime.tv_sec=timer.dwmsSinceStart/1000L;
    use->ru_stime.tv_usec=timer.dwmsSinceStart%1000L;
  }
  return TRUE;
}

int gettimeofday(struct timeval *o_time, void *dummy)
{
  time_t s_time;
  
  s_time=time(NULL);
  o_time->tv_sec=s_time;
  o_time->tv_usec=0;
  return 0;
}

int getwd(char *buff)
{ 
  return (_fullpath(buff,".",MISC_STRING_SIZE)!=NULL);
}   

/********************************************************
 * Windows Hooks... 
 ********************************************************
 */

/*
 * This routine is called every decision cycle,
 * checks to see if an interrupt has ocuured, the
 * reason, and what to do
 */
int soar_is_halted(void)
{                 
  MSG msg;
  
  switch (current_agent(stop_soar)) {
    case STOP_NOT:
      if (PeekMessage(&msg,NULL,0,0,PM_NOREMOVE)) {
        GetMessage(&msg,NULL,0,0);
        TranslateMessage(&msg);
        DispatchMessage(&msg);
      }
      return FALSE;
    case STOP_HALT:
    case STOP_INTERRUPT:
      return TRUE;
    case STOP_PAUSE:
      while (current_agent(stop_soar)==STOP_PAUSE) {
        GetMessage(&msg,NULL,0,0);
        TranslateMessage(&msg);
        DispatchMessage(&msg);
      }
      return FALSE;
    default:
      return TRUE;
  }
}   
      

/*********************************************
 * D O C U M E N T   M A I N T E N A N C E
 *********************************************/

/*
 * Maintain the production lists
 * (both are tied into src/hooks.c)
 */
void add_production_to_stat_lists(production *prod)
{
  switch (prod->type) {
    case CHUNK_PRODUCTION_TYPE:
	  theApp.m_pSoarMachine->AddChunk(prod);
      break;
    case USER_PRODUCTION_TYPE:
	  theApp.m_pSoarMachine->AddProduction(prod);
      break;
    case JUSTIFICATION_PRODUCTION_TYPE:
    case DEFAULT_PRODUCTION_TYPE:
    default:
      break;
  }
}

void remove_production_from_stat_lists(production *prod)
{             
  switch (prod->type) {
    case CHUNK_PRODUCTION_TYPE:
	  theApp.m_pSoarMachine->RemoveChunk(prod);
      break;
    case USER_PRODUCTION_TYPE:
	  theApp.m_pSoarMachine->RemoveProduction(prod);
      break;
    case JUSTIFICATION_PRODUCTION_TYPE:
    case DEFAULT_PRODUCTION_TYPE:
    default:
      break;
  }
}

/***************************************
 * Replaced Soar kernel code
 ***************************************/

/****************
 * from src/io.c
 ****************/


// Handles "string input"
void get_next_char (void) 
{
  char *s; 

  if (current_agent(using_input_string)) {
    if (*(current_agent(input_string)))
      current_agent(current_char) = *(current_agent(input_string)++);
    return;
  }
    
  if (current_agent(current_file)->file==WINDOW_FILE) { 
    switch (current_agent(current_line)[current_agent(current_line_index)]) {
      case EOF_AS_CHAR:
      case 0:
        current_agent(current_line)[current_agent(current_line_index)]=current_agent(current_char)=EOF_AS_CHAR;
        break;
      default:
        current_agent(current_char)=current_agent(current_line)[current_agent(current_line_index)++];
    }
  } else 
    current_agent(current_char) = current_agent(current_file)->buffer 
                           [current_agent(current_file)->current_column++];
  if (current_agent(current_char)) return;

  if ((current_agent(current_file)->current_column == BUFSIZE) &&
      (current_agent(current_file)->buffer[BUFSIZE-2] != '\n') &&
      (current_agent(current_file)->buffer[BUFSIZE-2] != EOF_AS_CHAR)) {
    print ("Error:  line too long (max allowed is %d chars)\n",
           MAX_LEXER_LINE_LENGTH);
    print ("File %s, line %lu\n", current_agent(current_file)->filename,
           current_agent(current_file)->current_line);
    abort_with_fatal_error();
  }

  s = fgets (current_agent(current_file)->buffer, BUFSIZE, current_agent(current_file)->file);
  
  if (s) {
    current_agent(current_file)->current_line++;
    if (reading_from_top_level()) {
      tell_printer_that_output_column_has_been_reset ();
      if (current_agent(logging_to_file))
        print_string_to_log_file_only (current_agent(current_file)->buffer);
    }
  } else {

    /* s==NIL means immediate eof encountered or read error occurred */
    if (! feof(current_agent(current_file)->file)) {
      if(reading_from_top_level()) {
        return;
      } else {
        print ("I/O error while reading file %s; ignoring the rest of it.\n",
               current_agent(current_file)->filename);
      }
    }
    current_agent(current_file)->buffer[0] = EOF_AS_CHAR;
    current_agent(current_file)->buffer[1] = 0;
  }
  current_agent(current_char) = current_agent(current_file)->buffer[0];
  current_agent(current_file)->current_column = 1;
}

// Calls the gui/accptdlg module to get a line from the screen.
// fgets on steroids              
char *get_line_from_window(char *buff,int maxchar)
{             
	CAcceptDialog dialog;
	dialog.DoModal();
	strncpy(buff,dialog.m_Value,maxchar);
	return buff;
}

// Prints a string to the log
int print_string_to_window(char *s)
{                                
  int len;
  char *cp;
  static char buffer[1024];
           
  if (s==NULL)
    return 0;
                     
  len=strlen(s);
  while ((cp=strchr(s,'\n'))!=NULL) {
    if (cp-s) strncat(buffer,s,cp-s);
    s=cp+1;
    theApp.m_pSoarMachine->AddLog(CString(buffer));
    buffer[0]=0;
  }            

  strcat(buffer,s);
  return len;   
}


/*****************
 * from src/mem.c
 *****************/


/* ====================================================================

                   Basic Memory Allocation Utilities

   All memory blocks are allocated via calls to allocate_memory().  It
   calls malloc() and aborts if we run out of memory.  Free_memory() is
   the inverse of allocate_memory().  Allocate_memory_and_zerofill()
   does the obvious thing.  These routines take a usage_code indicating
   what purpose the memory is for (hash tables, strings, etc.).  This
   is used purely for statistics keeping.

   Print_memory_statistics() prints out stats on the memory usage.
==================================================================== */

void *allocate_memory (unsigned long size, int usage_code) {
  char *p;
  HGLOBAL hGlbl;

  size += sizeof(HGLOBAL);

  if (!(hGlbl=GlobalAlloc(GPTR,size))) {
    print ("\nError:  Tried but failed to allocate %lu bytes of memory.\n",
             size);
    abort_with_fatal_error ();
  }
  
  p=(char *)GlobalLock(hGlbl);
  
  *(HGLOBAL *)(p)=hGlbl;
  
  current_agent(memory_for_usage)[STATS_OVERHEAD_MEM_USAGE] += size;
  current_agent(memory_for_usage)[usage_code] += size;
               
  return (void *)(p+sizeof(HGLOBAL));
}

void *allocate_memory_and_zerofill (unsigned long size, int usage_code) {
  char *cp;
  if ((cp=(char *)allocate_memory (size, usage_code))!=NULL)
    memset(cp,0,(size_t)size);
  return (void *)cp;
}

void free_memory (void *p, int usage_code) {
        
  unsigned long size;
  HGLOBAL hGlbl;
  char *mem;

  mem=(char *)p - sizeof(HGLOBAL);
  
  hGlbl = *(HGLOBAL *)(mem);
  GlobalUnlock(hGlbl);
  GlobalFree(hGlbl);
  size=GlobalSize(hGlbl);

  current_agent(memory_for_usage)[STATS_OVERHEAD_MEM_USAGE] -= size;
  current_agent(memory_for_usage)[usage_code] -= size;
}

void print_memory_statistics (void) {
  unsigned long total;
  int i;

  total = 0;
  for (i=0; i<NUM_MEM_USAGE_CODES; i++) total += current_agent(memory_for_usage)[i];
  
  print ("%8lu bytes total memory allocated\n", total);
  print ("%8lu bytes statistics overhead\n",
         current_agent(memory_for_usage)[STATS_OVERHEAD_MEM_USAGE]);
  print ("%8lu bytes for strings\n",
         current_agent(memory_for_usage)[STRING_MEM_USAGE]);
  print ("%8lu bytes for hash tables\n",
         current_agent(memory_for_usage)[HASH_TABLE_MEM_USAGE]);
  print ("%8lu bytes for various memory pools\n",
         current_agent(memory_for_usage)[POOL_MEM_USAGE]);
  print ("%8lu bytes for miscellaneous other things\n",
         current_agent(memory_for_usage)[MISCELLANEOUS_MEM_USAGE]);
}



