/* ====================================================================
                    Production Utilities for Soar 6

   This file contains various utility routines for manipulating 
   productions and parts of productions:  tests, conditions, actions,
   etc.  It also includes the reorderer and compile-time o-support
   calculations.

   Init_production_utilities() should be called before anything else here.
==================================================================== */

#include <ctype.h>
#include "soar.h"

/* comment out the following line to supress compile-time o-support
   calculations */
#define DO_COMPILE_TIME_O_SUPPORT_CALCS

/* uncomment the following line to get printouts of names of productions
   that can't be fully compile-time o-support evaluated */
/* #define LIST_COMPILE_TIME_O_SUPPORT_FAILURES */


void init_reorderer (void);

void init_production_utilities (void) {
  init_memory_pool (&current_agent(complex_test_pool), sizeof(complex_test), "complex test");
  init_memory_pool (&current_agent(condition_pool), sizeof(condition), "condition");
  init_memory_pool (&current_agent(production_pool), sizeof(production), "production");
  init_memory_pool (&current_agent(action_pool), sizeof(action), "action");
  init_memory_pool (&current_agent(not_pool), sizeof(not), "not");
  init_reorderer();
}

/* ********************************************************************

           Utility Routines for Various Parts of Productions

******************************************************************** */

/* ====================================================================

                   Utilities for Lists of Symbols

==================================================================== */

/* ----------------------------------------------------------------
   Takes a list of symbols and returns a copy of the same list,
   incrementing the reference count on each symbol in the list.
---------------------------------------------------------------- */

list *copy_symbol_list_adding_references (list *sym_list) {
  cons *c, *first, *prev;

  if (! sym_list) return NIL;
  allocate_cons (&first);
  first->first = sym_list->first;
  symbol_add_ref ((symbol *)(first->first));
  sym_list = sym_list->rest;
  prev = first;
  while (sym_list) {
    allocate_cons (&c);
    prev->rest = c;
    c->first = sym_list->first;
    symbol_add_ref ((symbol *)(c->first));
    sym_list = sym_list->rest;
    prev = c;
  }
  prev->rest = NIL;
  return first;
}

/* ----------------------------------------------------------------
   Frees a list of symbols, decrementing their reference counts.
---------------------------------------------------------------- */

void deallocate_symbol_list_removing_references (list *sym_list) {
  cons *c;
  
  while (sym_list) {
    c = sym_list;
    sym_list = sym_list->rest;
    symbol_remove_ref ((symbol *)(c->first));
    free_cons (c);
  }
}
  
/* =================================================================

                      Utility Routines for Tests

================================================================= */

/* --- This just copies a consed list of tests. --- */
list *copy_test_list (cons *c) {
  cons *new_c;

  if (!c) return NIL;
  allocate_cons (&new_c);
  new_c->first = copy_test (c->first);
  new_c->rest = copy_test_list (c->rest);
  return new_c;
}

/* ----------------------------------------------------------------
   Takes a test and returns a new copy of it.
---------------------------------------------------------------- */

test copy_test (test t) {
  symbol *referent;
  complex_test *ct, *new_ct;
  
  if (test_is_blank_test(t))
    return make_blank_test();

  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    return make_equality_test(referent);
  }
  
  ct = complex_test_from_test(t);
  
  allocate_with_pool (&current_agent(complex_test_pool), &new_ct);
  new_ct->type = ct->type;
  switch(ct->type) {
  case GOAL_ID_TEST:
  case IMPASSE_ID_TEST:
    break;
  case DISJUNCTION_TEST:
    new_ct->data.disjunction_list =
      copy_symbol_list_adding_references (ct->data.disjunction_list);
    break;
  case CONJUNCTIVE_TEST:
    new_ct->data.conjunct_list = copy_test_list (ct->data.conjunct_list);
    break;
  default:  /* relational tests other than equality */
    new_ct->data.referent = ct->data.referent;
    symbol_add_ref (ct->data.referent);
    break;
  }
  return make_test_from_complex_test(new_ct);
}

/* ----------------------------------------------------------------
   Same as copy_test(), only it doesn't include goal or impasse tests
   in the new copy.  The caller should initialize the two flags to FALSE
   before calling this routine; it sets them to TRUE if it finds a goal
   or impasse test.
---------------------------------------------------------------- */

test copy_test_removing_goal_impasse_tests (test t,
                                            bool *removed_goal,
                                            bool *removed_impasse) {
  complex_test *ct, *new_ct;
  cons *c;
  test new_t, temp;
  
  if (test_is_blank_or_equality_test(t)) return copy_test (t);
  
  ct = complex_test_from_test(t);
  
  switch(ct->type) {
  case GOAL_ID_TEST:
    *removed_goal = TRUE;
    return make_blank_test();
  case IMPASSE_ID_TEST:
    *removed_impasse = TRUE;
    return make_blank_test();

  case CONJUNCTIVE_TEST:
    new_t = make_blank_test();
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest) {
      temp = copy_test_removing_goal_impasse_tests (c->first,
                                                    removed_goal,
                                                    removed_impasse);
      if (! test_is_blank_test(temp))
        add_new_test_to_test (&new_t, temp);
    }
    if (test_is_complex_test(new_t)) {
      new_ct = complex_test_from_test(new_t);
      if (new_ct->type==CONJUNCTIVE_TEST)
        new_ct->data.conjunct_list =
          destructively_reverse_list (new_ct->data.conjunct_list);
    }
    return new_t;

  default:  /* relational tests other than equality */
    return copy_test (t);
  }
}

/* ----------------------------------------------------------------
   Deallocates a test.
---------------------------------------------------------------- */

void deallocate_test (test t) {
  cons *c, *next_c;
  complex_test *ct;

  if (test_is_blank_test(t)) return;
  if (test_is_blank_or_equality_test(t)) {
    symbol_remove_ref (referent_of_equality_test(t));
    return;
  }

  ct = complex_test_from_test(t);
  
  switch (ct->type) {
  case GOAL_ID_TEST:
  case IMPASSE_ID_TEST:
    break;
  case DISJUNCTION_TEST:
    deallocate_symbol_list_removing_references (ct->data.disjunction_list);
    break;
  case CONJUNCTIVE_TEST:
    c = ct->data.conjunct_list;
    while (c) {
      next_c = c->rest;
      deallocate_test (c->first);
      free_cons (c);
      c = next_c;
    }
    break;
  default: /* relational tests other than equality */
    symbol_remove_ref (ct->data.referent);
    break;
  }
  free_with_pool (&current_agent(complex_test_pool), ct);
}

/* --- Macro for doing this (usually) without procedure call overhead. --- */
#define quickly_deallocate_test(t) { \
  if (! test_is_blank_test(t)) { \
    if (test_is_blank_or_equality_test(t)) { \
      symbol_remove_ref (referent_of_equality_test(t)); \
    } else { \
      deallocate_test (t); } } }

/* ----------------------------------------------------------------
   Destructively modifies the first test (t) by adding the second
   one (add_me) to it (usually as a new conjunct).  The first test
   need not be a conjunctive test.
---------------------------------------------------------------- */

void add_new_test_to_test (test *t, test add_me) {
  complex_test *ct;
  cons *c;
  bool already_a_conjunctive_test;

  if (test_is_blank_test(add_me)) return;

  if (test_is_blank_test(*t)) {
    *t = add_me;
    return;
  }

  /* --- if *t isn't already a conjunctive test, make it into one --- */
  already_a_conjunctive_test = FALSE;
  if (test_is_complex_test(*t)) {
    ct = complex_test_from_test (*t);
    if (ct->type==CONJUNCTIVE_TEST) already_a_conjunctive_test = TRUE;
  }

  if (! already_a_conjunctive_test)  {
    allocate_with_pool (&current_agent(complex_test_pool), &ct);
    ct->type = CONJUNCTIVE_TEST;
    allocate_cons (&c);
    ct->data.conjunct_list = c;
    c->first = *t;
    c->rest = NIL;
    *t = make_test_from_complex_test (ct);
  }
  /* --- at this point, ct points to the complex test structure for *t --- */
    
  /* --- now add add_me to the conjunct list --- */
  allocate_cons (&c);
  c->first = add_me;
  c->rest = ct->data.conjunct_list;
  ct->data.conjunct_list = c;
}

/* ----------------------------------------------------------------
   Same as add_new_test_to_test(), only has no effect if the second
   test is already included in the first one.
---------------------------------------------------------------- */

void add_new_test_to_test_if_not_already_there (test *t, test add_me) {
  complex_test *ct;
  cons *c;

  if (tests_are_equal (*t, add_me)) {
    deallocate_test (add_me);
    return;
  }

  if (test_is_complex_test (*t)) {
    ct = complex_test_from_test (*t);
    if (ct->type == CONJUNCTIVE_TEST)
      for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
        if (tests_are_equal (c->first, add_me)) {
          deallocate_test (add_me);
          return;
        }
  }

  add_new_test_to_test (t, add_me);
}

/* ----------------------------------------------------------------
   Returns TRUE iff the two tests are identical.
---------------------------------------------------------------- */

bool tests_are_equal (test t1, test t2) {
  cons *c1, *c2;
  complex_test *ct1, *ct2;

  if (test_is_blank_or_equality_test(t1))
    return (t1==t2); /* Warning: this relies on the representation of tests */

  ct1 = complex_test_from_test(t1);
  ct2 = complex_test_from_test(t2);
  
  if (ct1->type != ct2->type) return FALSE;

  switch(ct1->type) {
  case GOAL_ID_TEST: return TRUE;
  case IMPASSE_ID_TEST: return TRUE;

  case DISJUNCTION_TEST:
    for (c1=ct1->data.disjunction_list, c2=ct2->data.disjunction_list;
         ((c1!=NIL)&&(c2!=NIL));
         c1=c1->rest, c2=c2->rest)
      if (c1->first != c2->first) return FALSE;
    if (c1==c2) return TRUE;  /* make sure they both hit end-of-list */
    return FALSE;

  case CONJUNCTIVE_TEST:
    for (c1=ct1->data.conjunct_list, c2=ct2->data.conjunct_list;
         ((c1!=NIL)&&(c2!=NIL));
         c1=c1->rest, c2=c2->rest)
      if (! tests_are_equal(c1->first,c2->first)) return FALSE;
    if (c1==c2) return TRUE;  /* make sure they both hit end-of-list */
    return FALSE;

  default:  /* relational tests other than equality */
    if (ct1->data.referent == ct2->data.referent) return TRUE;
    return FALSE;
  }
}

/* ----------------------------------------------------------------
   Returns a hash value for the given test.
---------------------------------------------------------------- */

unsigned long hash_test (test t) {
  complex_test *ct;
  cons *c;
  unsigned long result;
  
  if (test_is_blank_test(t))
    return 0;

  if (test_is_blank_or_equality_test(t))
    return (referent_of_equality_test(t))->common.hash_id;

  ct = complex_test_from_test(t);

  switch (ct->type) {
  case GOAL_ID_TEST: return 34894895;  /* just use some unusual number */
  case IMPASSE_ID_TEST: return 2089521;
  case DISJUNCTION_TEST:
    result = 7245;
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      result = result + ((symbol *)(c->first))->common.hash_id;
    return result;
  case CONJUNCTIVE_TEST:
    result = 100276;
    for (c=ct->data.disjunction_list; c!=NIL; c=c->rest)
      result = result + hash_test (c->first);
    return result;
  case NOT_EQUAL_TEST:
  case LESS_TEST:
  case GREATER_TEST:
  case LESS_OR_EQUAL_TEST:
  case GREATER_OR_EQUAL_TEST:
  case SAME_TYPE_TEST:
    return (ct->type << 24) + ct->data.referent->common.hash_id;
  default:
    print ("Error: bad test type in hash_test\n");
    abort_with_fatal_error();
  }
  return 0; /* unreachable, but without it, gcc -Wall warns here */
}

/* ----------------------------------------------------------------
   Returns TRUE iff the test contains an equality test for the given
   symbol.  If sym==NIL, returns TRUE iff the test contains any
   equality test.
---------------------------------------------------------------- */

bool test_includes_equality_test_for_symbol (test t, symbol *sym) {
  cons *c;
  complex_test *ct;

  if (test_is_blank_test(t)) return FALSE;
  
  if (test_is_blank_or_equality_test(t)) {
    if (sym) return (referent_of_equality_test(t) == sym);
    return TRUE;
  }
  
  ct = complex_test_from_test(t);

  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if (test_includes_equality_test_for_symbol (c->first, sym)) return TRUE;
  }
  return FALSE;
}

/* ----------------------------------------------------------------
   Looks for goal or impasse tests (as directed by the two flag
   parameters) in the given test, and returns TRUE if one is found.
---------------------------------------------------------------- */

bool test_includes_goal_or_impasse_id_test (test t,
                                            bool look_for_goal,
                                            bool look_for_impasse) {
  complex_test *ct;
  cons *c;
  
  if (test_is_blank_or_equality_test(t)) return FALSE;
  ct = complex_test_from_test(t);
  if (look_for_goal && (ct->type==GOAL_ID_TEST)) return TRUE;
  if (look_for_impasse && (ct->type==IMPASSE_ID_TEST)) return TRUE;
  if (ct->type == CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if (test_includes_goal_or_impasse_id_test (c->first,
                                                 look_for_goal,
                                                 look_for_impasse))
        return TRUE;
    return FALSE;
  }
  return FALSE;
}

/* ----------------------------------------------------------------
   Looks through a test, and returns a new copy of the first equality
   test it finds.  Signals an error if there is no equality test in
   the given test.
---------------------------------------------------------------- */

test copy_of_equality_test_found_in_test (test t) {
  complex_test *ct;
  cons *c;
  
  if (test_is_blank_test(t)) {
    print ("Internal error: can't find equality test in test\n");
    abort_with_fatal_error();
  }
  if (test_is_blank_or_equality_test(t)) return copy_test (t);
  ct = complex_test_from_test(t);
  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if ( (! test_is_blank_test ((test)(c->first))) &&
           (test_is_blank_or_equality_test ((test)(c->first))) )
        return copy_test (c->first);
  }
  print ("Internal error: can't find equality test in test\n");
  abort_with_fatal_error();
  return 0; /* unreachable, but without it, gcc -Wall warns here */
}

/* =================================================================

                  Utility Routines for Conditions

================================================================= */

/* ----------------------------------------------------------------
   Deallocates a condition list (including any NCC's and tests in it).
---------------------------------------------------------------- */

void deallocate_condition_list (condition *cond_list) {
  condition *c;
  
  while (cond_list) {
    c = cond_list;
    cond_list = cond_list->next;
    if (c->type==CONJUNCTIVE_NEGATION_CONDITION) {
      deallocate_condition_list (c->data.ncc.top);
    } else { /* positive and negative conditions */
      quickly_deallocate_test (c->data.tests.id_test);
      quickly_deallocate_test (c->data.tests.attr_test);
      quickly_deallocate_test (c->data.tests.value_test);
    }
    free_with_pool (&current_agent(condition_pool), c);
  }
}

/* ----------------------------------------------------------------
   Returns a new copy of the given condition.
---------------------------------------------------------------- */

condition *copy_condition (condition *cond) {
  condition *new;

  if (!cond) return NIL;
  allocate_with_pool (&current_agent(condition_pool), &new);
  new->type = cond->type;
  
  switch (cond->type) {
  case POSITIVE_CONDITION:
    new->bt = cond->bt;
    /* ... and fall through to next case */
  case NEGATIVE_CONDITION:
    new->data.tests.id_test = copy_test (cond->data.tests.id_test);
    new->data.tests.attr_test = copy_test (cond->data.tests.attr_test);
    new->data.tests.value_test = copy_test (cond->data.tests.value_test);
    new->test_for_acceptable_preference = cond->test_for_acceptable_preference;
    break;
  case CONJUNCTIVE_NEGATION_CONDITION:
    copy_condition_list (cond->data.ncc.top, &(new->data.ncc.top),
                         &(new->data.ncc.bottom));
    break;
  }
  return new;
}

/* ----------------------------------------------------------------
   Copies the given condition list, returning pointers to the
   top-most and bottom-most conditions in the new copy.
---------------------------------------------------------------- */

void copy_condition_list (condition *top_cond,
                          condition **dest_top,
                          condition **dest_bottom) {
  condition *new, *prev;

  prev = NIL;
  while (top_cond) {
    new = copy_condition (top_cond);
    if (prev) prev->next = new; else *dest_top = new;
    new->prev = prev;
    prev = new;
    top_cond = top_cond->next;
  }
  if (prev) prev->next = NIL; else *dest_top = NIL;
  *dest_bottom = prev;
}

/* ----------------------------------------------------------------
   Returns TRUE iff the two conditions are identical.
---------------------------------------------------------------- */

bool conditions_are_equal (condition *c1, condition *c2) {
  if (c1->type != c2->type) return FALSE;
  switch (c1->type) {
  case POSITIVE_CONDITION:
  case NEGATIVE_CONDITION:
    if (! tests_are_equal (c1->data.tests.id_test,
                           c2->data.tests.id_test))
      return FALSE;
    if (! tests_are_equal (c1->data.tests.attr_test,
                           c2->data.tests.attr_test))
      return FALSE;
    if (! tests_are_equal (c1->data.tests.value_test,
                           c2->data.tests.value_test))
      return FALSE;
    if (c1->test_for_acceptable_preference !=
        c2->test_for_acceptable_preference)
      return FALSE;
    return TRUE;
    
  case CONJUNCTIVE_NEGATION_CONDITION:
    for (c1=c1->data.ncc.top, c2=c2->data.ncc.top;
         ((c1!=NIL)&&(c2!=NIL));
         c1=c1->next, c2=c2->next)
      if (! conditions_are_equal (c1,c2)) return FALSE;
    if (c1==c2) return TRUE;  /* make sure they both hit end-of-list */
    return FALSE;
  }
  return FALSE; /* unreachable, but without it, gcc -Wall warns here */
}

/* ----------------------------------------------------------------
   Returns a hash value for the given condition.
---------------------------------------------------------------- */

unsigned long hash_condition (condition *cond) {
  unsigned long result;
  condition *c;

  switch (cond->type) {
  case POSITIVE_CONDITION:
    result = hash_test (cond->data.tests.id_test);
    result = (result << 24) | (result >>  8);
    result ^= hash_test (cond->data.tests.attr_test);
    result = (result << 24) | (result >>  8);
    result ^= hash_test (cond->data.tests.value_test);
    if (cond->test_for_acceptable_preference) result++;
    break;
  case NEGATIVE_CONDITION:
    result = 1267818;
    result ^= hash_test (cond->data.tests.id_test);
    result = (result << 24) | (result >>  8);
    result ^= hash_test (cond->data.tests.attr_test);
    result = (result << 24) | (result >>  8);
    result ^= hash_test (cond->data.tests.value_test);
    if (cond->test_for_acceptable_preference) result++;
    break;
  case CONJUNCTIVE_NEGATION_CONDITION:
    result = 82348149;
    for (c=cond->data.ncc.top; c!=NIL; c=c->next) {
      result ^= hash_condition (c);
      result = (result << 24) | (result >>  8);
    }
    break;
  }
  return result;
}

/* =================================================================

              Utility Routines for Actions and RHS Values

================================================================= */

/* ----------------------------------------------------------------
   Deallocates the given rhs_value.
---------------------------------------------------------------- */

void deallocate_rhs_value (rhs_value rv) {
  cons *c;
  list *fl;

  if (rhs_value_is_funcall(rv)) {
    fl = rhs_value_to_funcall_list(rv);
    for (c=fl->rest; c!=NIL; c=c->rest)
      deallocate_rhs_value (c->first);
    free_list (fl);
  } else {
    symbol_remove_ref (rhs_value_to_symbol(rv));
  }
}

/* ----------------------------------------------------------------
   Returns a new copy of the given rhs_value.
---------------------------------------------------------------- */

rhs_value copy_rhs_value (rhs_value rv) {
  cons *c, *new_c, *prev_new_c;
  list *fl, *new_fl;

  if (rhs_value_is_funcall(rv)) {
    fl = rhs_value_to_funcall_list(rv);
    allocate_cons (&new_fl);
    new_fl->first = fl->first;
    prev_new_c = new_fl;
    for (c=fl->rest; c!=NIL; c=c->rest) {
      allocate_cons (&new_c);
      new_c->first = copy_rhs_value (c->first);
      prev_new_c->rest = new_c;
      prev_new_c = new_c;
    }
    prev_new_c->rest = NIL;
    return funcall_list_to_rhs_value (new_fl);
  } else {
    symbol_add_ref (rhs_value_to_symbol(rv));
    return rv;
  }
}
  
/* ----------------------------------------------------------------
   Deallocates the given action (singly-linked) list.
---------------------------------------------------------------- */

void deallocate_action_list (action *actions) {
  action *a;
  
  while (actions) {
    a = actions;
    actions = actions->next;
    if (a->type==FUNCALL_ACTION) {
      deallocate_rhs_value (a->value);
    } else {
      /* --- make actions --- */
      symbol_remove_ref (a->id);
      symbol_remove_ref (a->attr);
      deallocate_rhs_value (a->value);
      if (preference_is_binary(a->preference_type))
        deallocate_rhs_value (a->referent);
    }
    free_with_pool (&current_agent(action_pool),a);
  }
}

/* =================================================================

                    Utility Routines for Nots

================================================================= */

/* ----------------------------------------------------------------
   Deallocates the given (singly-linked) list of Nots.
---------------------------------------------------------------- */

void deallocate_list_of_nots (not *nots) {
  not *temp;

  while (nots) {
    temp = nots;
    nots = nots->next;
    symbol_remove_ref (temp->s1);
    symbol_remove_ref (temp->s2);
    free_with_pool (&current_agent(not_pool), temp);
  }
}

/* *********************************************************************

                    Transitive Closure Utilities

********************************************************************* */

/* =====================================================================

              Increment TC Counter and Return New TC Number

   Get_new_tc_number() is called from lots of places.  Any time we need
   to mark a set of identifiers and/or variables, we get a new tc_number
   by calling this routine, then proceed to mark various ids or vars
   by setting the sym->id.tc_num or sym->var.tc_num fields.

   A global tc number counter is maintained and incremented by this
   routine in order to generate a different tc_number each time.  If
   the counter ever wraps around back to 0, we bump it up to 1 and
   reset the the tc_num fields on all existing identifiers and variables
   to 0.
===================================================================== */

tc_number current_tc_number = 0;

tc_number get_new_tc_number (void) {
  current_tc_number++;
  if (current_tc_number==0) {
    reset_id_and_variable_tc_numbers ();
    current_tc_number = 1;
  }
  return current_tc_number;
}

/* =====================================================================

               Marking, Unmarking, and Collecting Symbols

   Sometimes in addition to marking symbols using their tc_num fields,
   we also want to build up a list of the symbols we've marked.  So,
   many routines in this file take an "id_list" or "var_list" argument.
   This argument should be NIL if no such list is desired.  If non-NIL,
   it should point to the header of the linked list being built.

   Mark_identifier_if_unmarked() and mark_variable_if_unmarked() are
   macros for adding id's and var's to the set of symbols.

   Unmark_identifiers_and_free_list() unmarks all the id's in the given
   list, and deallocates the list.  Unmark_variables_and_free_list()
   is similar, only the list should be a list of variables rather than
   identifiers.

   Symbol_is_constant_or_marked_variable() tests whether the given symbol
   is either a constant (non-variable) or a variable marked with the
   given tc number.
===================================================================== */

#define mark_identifier_if_unmarked(ident,tc,id_list) { \
  if ((ident)->id.tc_num != (tc)) { \
    (ident)->id.tc_num = (tc); \
    if (id_list) push ((ident),(*(id_list))); } }

#define mark_variable_if_unmarked(v,tc,var_list) { \
  if ((v)->var.tc_num != (tc)) { \
    (v)->var.tc_num = (tc); \
    if (var_list) push ((v),(*(var_list))); } }

void unmark_identifiers_and_free_list (list *id_list) {
  cons *next;
  symbol *sym;

  while (id_list) {
    sym = id_list->first;
    next = id_list->rest;
    free_cons (id_list);
    sym->id.tc_num = 0;
    id_list = next;
  }
}

void unmark_variables_and_free_list (list *var_list) {
  cons *next;
  symbol *sym;

  while (var_list) {
    sym = var_list->first;
    next = var_list->rest;
    free_cons (var_list);
    sym->var.tc_num = 0;
    var_list = next;
  }
}

#define symbol_is_constant_or_marked_variable(sym,tc) \
  ( ((sym)->common.symbol_type!=VARIABLE_SYMBOL_TYPE) || \
    ((sym)->var.tc_num == (tc)) )

/* =====================================================================

   Finding bound variables from tests, conditions, and condition lists

   These routines collect the bound variables in tests, etc.  Their
   "var_list" arguments should either be NIL or else should point to
   the header of the list of marked variables being constructed.
===================================================================== */

void add_bound_variables_in_test (test t, tc_number tc, list **var_list) {
  cons *c;
  symbol *referent;
  complex_test *ct;
  
  if (test_is_blank_test(t)) return;

  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      mark_variable_if_unmarked (referent, tc, var_list);
    return;
  }

  ct = complex_test_from_test(t);
  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      add_bound_variables_in_test (c->first, tc, var_list);
  }
}

void add_bound_variables_in_condition (condition *c, tc_number tc,
                                       list **var_list) {
  if (c->type!=POSITIVE_CONDITION) return;
  add_bound_variables_in_test (c->data.tests.id_test, tc, var_list);
  add_bound_variables_in_test (c->data.tests.attr_test, tc, var_list);
  add_bound_variables_in_test (c->data.tests.value_test, tc, var_list);
}

void add_bound_variables_in_condition_list (condition *cond_list,
                                            tc_number tc, list **var_list) {
  condition *c;
  
  for (c=cond_list; c!=NIL; c=c->next)
    add_bound_variables_in_condition (c, tc, var_list);
}

/* =====================================================================

   Finding all variables from tests, conditions, and condition lists

   These routines collect all the variables in tests, etc.  Their
   "var_list" arguments should either be NIL or else should point to
   the header of the list of marked variables being constructed.
===================================================================== */

void add_all_variables_in_test (test t, tc_number tc, list **var_list) {
  cons *c;
  symbol *referent;
  complex_test *ct;

  if (test_is_blank_test(t)) return;

  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      mark_variable_if_unmarked (referent, tc, var_list);
    return;
  }

  ct = complex_test_from_test(t);
  
  switch (ct->type) {
  case GOAL_ID_TEST:
  case IMPASSE_ID_TEST:
  case DISJUNCTION_TEST:
    break;

  case CONJUNCTIVE_TEST:
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      add_all_variables_in_test (c->first, tc, var_list);
    break;

  default:
    /* --- relational tests other than equality --- */
    referent = ct->data.referent;
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      mark_variable_if_unmarked (referent, tc, var_list);
    break;
  }
}

void add_all_variables_in_condition_list (condition *cond_list,
                                          tc_number tc, list **var_list);

void add_all_variables_in_condition (condition *c, tc_number tc,
                                     list **var_list) {
  if (c->type==CONJUNCTIVE_NEGATION_CONDITION) {
    add_all_variables_in_condition_list (c->data.ncc.top, tc, var_list);
  } else {
    add_all_variables_in_test (c->data.tests.id_test, tc, var_list);
    add_all_variables_in_test (c->data.tests.attr_test, tc, var_list);
    add_all_variables_in_test (c->data.tests.value_test, tc, var_list);
  }
}

void add_all_variables_in_condition_list (condition *cond_list,
                                          tc_number tc, list **var_list) {
  condition *c;

  for (c=cond_list; c!=NIL; c=c->next)
    add_all_variables_in_condition (c, tc, var_list);
}

/* =====================================================================

   Finding all variables from rhs_value's, actions, and action lists

   These routines collect all the variables in rhs_value's, etc.  Their
   "var_list" arguments should either be NIL or else should point to
   the header of the list of marked variables being constructed.
===================================================================== */

void add_all_variables_in_rhs_value (rhs_value rv, tc_number tc,
                                     list **var_list) {
  list *fl;
  cons *c;
  symbol *sym;

  if (rhs_value_is_symbol(rv)) {
    /* --- ordinary values (i.e., symbols) --- */
    sym = rhs_value_to_symbol(rv);
    if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      mark_variable_if_unmarked (sym, tc, var_list);
  } else {
    /* --- function calls --- */
    fl = rhs_value_to_funcall_list(rv);
    for (c=fl->rest; c!=NIL; c=c->rest)
      add_all_variables_in_rhs_value (c->first, tc, var_list);
  }
}

void add_all_variables_in_action (action *a, tc_number tc, list **var_list){
  if (a->type==MAKE_ACTION) {
    /* --- ordinary make actions --- */
    if (a->id->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      mark_variable_if_unmarked (a->id, tc, var_list);
    if (a->attr->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      mark_variable_if_unmarked (a->attr, tc, var_list);
    add_all_variables_in_rhs_value (a->value, tc, var_list);
    if (preference_is_binary(a->preference_type))
      add_all_variables_in_rhs_value (a->referent, tc, var_list);
  } else {
    /* --- function call actions --- */
    add_all_variables_in_rhs_value (a->value, tc, var_list);
  }
}

void add_all_variables_in_action_list (action *actions, tc_number tc,
                                       list **var_list) {
  action *a;

  for (a=actions; a!=NIL; a=a->next)
    add_all_variables_in_action (a, tc, var_list);
}

/* ====================================================================

              Transitive Closure for Conditions and Actions

   These routines do transitive closure calculations for tests,
   conditions, actions, etc.

   Usage: 
     1. Set my_tc = get_new_tc_number() to start a new TC
     2. (optional) If you want linked lists of symbols in the TC, initialize
        id_list=NIL and var_list=NIL.
        If you're not using id_list and/or var_list, give NIL for "&id_list"
        and/or "&var_list" in the function calls below.
     3. (optional) setup any id's or var's that you want to include in the
        initial TC, by calling 
           add_symbol_to_tc (sym, my_tc, &id_list, &var_list)
        (If not using id_list or var_list, you can just mark
         sym->{id,var}.tc_num = my_tc instead.)
     4. To do the work you want, use any of the following any number of times:
            add_cond_to_tc (cond, my_tc, &id_list, &var_list);
            add_action_to_tc (cond, my_tc, &id_list, &var_list);
            result = cond_is_in_tc (cond, my_tc);
            result = action_is_in_tc (action, my_tc);
     5. When finished, free the cons cells in id_list and var_list (but
        don't call symbol_remove_ref() on the symbols in them).

==================================================================== */

void add_symbol_to_tc (symbol *sym, tc_number tc,
                       list **id_list, list **var_list) {
  if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE) {
    mark_variable_if_unmarked (sym, tc, var_list);
  } else if (sym->common.symbol_type==IDENTIFIER_SYMBOL_TYPE) {
    mark_identifier_if_unmarked (sym, tc, id_list);
  }
}

void add_test_to_tc (test t, tc_number tc,
                     list **id_list, list **var_list) {
  cons *c;
  complex_test *ct;
  
  if (test_is_blank_test(t)) return;
  
  if (test_is_blank_or_equality_test(t)) {
    add_symbol_to_tc (referent_of_equality_test(t), tc, id_list, var_list);
    return;
  }

  ct = complex_test_from_test(t);
  if (ct->type == CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      add_test_to_tc (c->first, tc, id_list, var_list);
  }
}

void add_cond_to_tc (condition *c, tc_number tc,
                     list **id_list, list **var_list) {
  if (c->type==POSITIVE_CONDITION) {
    add_test_to_tc (c->data.tests.id_test, tc, id_list, var_list);
    add_test_to_tc (c->data.tests.value_test, tc, id_list, var_list);
  }
}

void add_action_to_tc (action *a, tc_number tc,
                       list **id_list, list **var_list) {
  if (a->type != MAKE_ACTION) return;
  add_symbol_to_tc (a->id, tc, id_list, var_list);
  if (rhs_value_is_symbol(a->value))
    add_symbol_to_tc (rhs_value_to_symbol(a->value), tc, id_list, var_list);
  if (preference_is_binary(a->preference_type))
    if (rhs_value_is_symbol(a->referent))
      add_symbol_to_tc (rhs_value_to_symbol(a->referent),tc,id_list,var_list);
}

bool symbol_is_in_tc (symbol *sym, tc_number tc) {
  if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE)
    return (sym->var.tc_num == tc);
  if (sym->common.symbol_type==IDENTIFIER_SYMBOL_TYPE)
    return (sym->id.tc_num == tc);
  return FALSE;
}

bool test_is_in_tc (test t, tc_number tc) {
  cons *c;
  complex_test *ct;

  if (test_is_blank_test(t)) return FALSE;
  if (test_is_blank_or_equality_test(t)) {
    return symbol_is_in_tc (referent_of_equality_test(t), tc);
  }

  ct = complex_test_from_test(t);
  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if (test_is_in_tc (c->first, tc)) return TRUE;
    return FALSE;
  }
  return FALSE;
}

bool cond_is_in_tc (condition *cond, tc_number tc) {
  condition *c;
  bool anything_changed;
  bool result;
  list *new_ids, *new_vars;

  if (cond->type != CONJUNCTIVE_NEGATION_CONDITION)
    return test_is_in_tc (cond->data.tests.id_test, tc);
  
  /* --- conjunctive negations:  keep trying to add stuff to the TC --- */
  new_ids = NIL;
  new_vars = NIL;
  for (c=cond->data.ncc.top; c!=NIL; c=c->next)
    c->already_in_tc = FALSE;
  while (TRUE) {
    anything_changed = FALSE;
    for (c=cond->data.ncc.top; c!=NIL; c=c->next)
      if (! c->already_in_tc)
        if (cond_is_in_tc (c, tc)) {
          add_cond_to_tc (c, tc, &new_ids, &new_vars);
          c->already_in_tc = TRUE;
          anything_changed = TRUE;
        }
    if (! anything_changed) break;
  }

  /* --- complete TC found, look for anything that didn't get hit --- */
  result = TRUE;
  for (c=cond->data.ncc.top; c!=NIL; c=c->next)
    if (! c->already_in_tc) result = FALSE;
  
  /* --- unmark identifiers and variables that we just marked --- */
  unmark_identifiers_and_free_list (new_ids);
  unmark_variables_and_free_list (new_vars);

  return result;
}

bool action_is_in_tc (action *a, tc_number tc) {
  if (a->type != MAKE_ACTION) return FALSE;
  return symbol_is_in_tc (a->id, tc);
}

/* *********************************************************************

                         Variable Generator

   These routines are used for generating new variables.  The variables
   aren't necessarily "completely" new--they might occur in some existing
   production.  But we usually need to make sure the new variables don't
   overlap with those already used in a *certain* production--for instance,
   when variablizing a chunk, we don't want to introduce a new variable that
   conincides with the name of a variable already in an NCC in the chunk.
   
   To use these routines, first call reset_variable_generator(), giving
   it lists of conditions and actions whose variables should not be
   used.  Then call generate_new_variable() any number of times; each
   time, you give it a string to use as the prefix for the new variable's
   name.  The prefix string should not include the opening "<".
********************************************************************* */


void reset_variable_generator (condition *conds_with_vars_to_avoid,
                               action *actions_with_vars_to_avoid) {
  tc_number tc;
  list *var_list;
  cons *c;
  int i;

  /* --- reset counts, and increment the gensym number --- */
  for (i=0; i<26; i++) current_agent(gensymed_variable_count)[i] = 1;
  current_agent(current_variable_gensym_number)++;
  if (current_agent(current_variable_gensym_number)==0) {
    reset_variable_gensym_numbers ();
    current_agent(current_variable_gensym_number) = 1;
  }

  /* --- mark all variables in the given conds and actions --- */
  tc = get_new_tc_number();
  var_list = NIL;
  add_all_variables_in_condition_list (conds_with_vars_to_avoid,tc, &var_list);
  add_all_variables_in_action_list (actions_with_vars_to_avoid, tc, &var_list);
  for (c=var_list; c!=NIL; c=c->rest)
    ((symbol *)(c->first))->var.gensym_number = current_agent(current_variable_gensym_number);
  free_list (var_list);
}

symbol *generate_new_variable (char *prefix) {
  char name[200];  /* that ought to be long enough! */
  symbol *new;
  char first_letter;

  first_letter = *prefix;
  if (isalpha(first_letter)) {
    if (isupper(first_letter)) first_letter = tolower(first_letter);
  } else {
    first_letter = 'v';
  }
  first_letter -= 'a';

  while (TRUE) {
    sprintf (name, "<%s%lu>", prefix, current_agent(gensymed_variable_count)[first_letter]++);
    new = make_variable (name);
    if (new->var.gensym_number != current_agent(current_variable_gensym_number)) break;
    symbol_remove_ref (new);
  }
  
  new->var.current_binding_value = NIL;
  new->var.gensym_number = current_agent(current_variable_gensym_number);
  return new;
}

/* *********************************************************************

                             Reordering

********************************************************************* */

/* =====================================================================

                  Name of production being reordered

   In case any errors are encountered during reordering, this variable
   holds the name of the production currently being reordered, so it
   can be printed with the error message.
===================================================================== */

char *name_of_production_being_reordered;

/* =====================================================================

                      Reordering for RHS Actions

  Whenever a new identifier is created, we need to know (at creation time)
  what level of the goal stack it's at.  If the <newid> appears in the
  attribute or value slot of a make, we just give it the same level as
  whatever's in the id slot of that make.  But if <newid> appears in the
  id slot of a make, we can't tell what level it goes at.  

  To avoid this problem, we reorder the list of RHS actions so that when
  the actions are executed (in the resulting order), each <newid> is
  encountered in an attribute or value slot *before* it is encountered
  in an id slot.
  
  Furthermore, we make sure all arguments to function calls are bound
  before the function call is executed.
  
  Reorder_action_list() does the reordering.  Its parameter action_list
  is reordered in place (destructively modified).  It also requires at entry
  that the variables bound on the LHS are marked.  The function returns
  TRUE if successful, FALSE if it was unable to produce a legal ordering.
===================================================================== */

bool legal_to_execute_action (action *a, tc_number tc);

bool reorder_action_list (action **action_list, tc_number lhs_tc) {
  list *new_bound_vars;
  action *remaining_actions;
  action *first_action, *last_action;
  action *a, *prev_a;
  bool result_flag;

  new_bound_vars = NIL;
  remaining_actions = *action_list;
  first_action = NIL;
  last_action = NIL;
  
  while (remaining_actions) {
    /* --- scan through remaining_actions, look for one that's legal --- */
    prev_a = NIL;
    a = remaining_actions;
    while (TRUE) {
      if (!a) break; /* looked at all candidates, but none were legal */
      if (legal_to_execute_action (a, lhs_tc)) break;
      prev_a = a;
      a = a->next;
    }
    if (!a) break;
    /* --- move action a from remaining_actions to reordered list --- */
    if (prev_a) prev_a->next = a->next; else remaining_actions = a->next;
    a->next = NIL;
    if (last_action) last_action->next = a; else first_action = a;
    last_action = a;
    /* --- add new variables from a to new_bound_vars --- */
    add_all_variables_in_action (a, lhs_tc, &new_bound_vars);
  }

  if (remaining_actions) {
    /* --- there are remaining_actions but none can be legally added --- */
    print ("Error: production %s has a bad RHS--\n",
           name_of_production_being_reordered);
    print ("       Either it creates structure not connected to anything\n");
    print ("       else in WM, or it tries to pass an unbound variable as\n");
    print ("       an argument to a function.\n");
    /* --- reconstruct list of all actions --- */
    if (last_action)
      last_action->next = remaining_actions;
    else
      first_action = remaining_actions;
    result_flag = FALSE;
  } else {
    result_flag = TRUE;
  }

  /* --- unmark variables that we just marked --- */
  unmark_variables_and_free_list (new_bound_vars);

  /* --- return final result --- */
  *action_list = first_action;
  return result_flag;
}

bool all_variables_in_rhs_value_bound (rhs_value rv, tc_number tc) {
  cons *c;
  list *fl;
  symbol *sym;
  
  if (rhs_value_is_funcall(rv)) {
    /* --- function calls --- */
    fl = rhs_value_to_funcall_list (rv);
    for (c=fl->rest; c!=NIL; c=c->rest)
      if (! all_variables_in_rhs_value_bound (c->first, tc))
        return FALSE;
    return TRUE;
  } else {
    /* --- ordinary (symbol) rhs values --- */
    sym = rhs_value_to_symbol (rv);
    if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      return (sym->var.tc_num == tc);
    return TRUE;
  }
}

bool legal_to_execute_action (action *a, tc_number tc) {
  if (a->type==MAKE_ACTION) {
    if (a->id->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      return (a->id->var.tc_num == tc);
    return TRUE;
  }
  /* --- otherwise it's a function call; make sure args are all bound  --- */
  return all_variables_in_rhs_value_bound (a->value, tc);
}

/* =====================================================================

                 Condition Simplification / Restoration

  In order to be able to move tests from one condition to another, we
  reorder using the following high-level technique.  (This procedure is
  applied separately at each level of nesting.)

    1. Simplify the positive conditions, by stripping off all tests other
       than equality.  When this is done, all tests in positive conditions
       are either equality tests or conjunctions of equality tests.  All
       other tests are saved away for later restoration.
    2. Then do the reordering...
    3. Then go back and restore all the tests that were previously saved
       away.  The restored tests might end up on different conditions
       than they started--they're inserted in the first place possible
       (i.e., as soon as all the necessary things are bound).

  The two main routines here are simplify_condition_list() and
  restore_and_deallocate_saved_tests().
       
===================================================================== */

typedef struct saved_test_struct {
  struct saved_test_struct *next;
  symbol *var;
  test the_test;
} saved_test;


void print_saved_test (saved_test *st) {
  print_with_symbols ("  Symbol: %y  Test: ", st->var);
  print_string (test_to_string (st->the_test, NULL));
}

void print_saved_test_list (saved_test *st) {
  while (st) {
    print_saved_test (st);
    print ("\n");
    st = st->next;
  }
}

saved_test *simplify_test (test *t, saved_test *old_sts) {
  test new, subtest;
  saved_test *saved;
  symbol *var, *sym;
  cons *c, *prev_c, *next_c;
  complex_test *ct;

  if (test_is_blank_test(*t)) {
    sym = generate_new_variable ("dummy-");
    *t = make_equality_test_without_adding_reference (sym);
    return old_sts;
  }

  if (test_is_blank_or_equality_test(*t)) {
    return old_sts;
  }

  ct = complex_test_from_test(*t);
  
  switch (ct->type) {
    
  case CONJUNCTIVE_TEST:
    /* --- look at subtests for an equality test --- */
    sym = NIL;
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest) {
      subtest = c->first;
      if (test_is_blank_or_equality_test(subtest))
        sym = referent_of_equality_test(subtest);
    }
    /* --- if no equality test was found, generate a variable for it --- */
    if (!sym) {
      sym = generate_new_variable ("dummy-");
      new = make_equality_test_without_adding_reference (sym);
      allocate_cons (&c);
      c->first = new;
      c->rest = ct->data.conjunct_list;
      ct->data.conjunct_list = c;
    }
    /* --- scan through, create saved_test for subtests except equality --- */
    prev_c = NIL;
    c = ct->data.conjunct_list;
    while (c) {
      next_c = c->rest;
      subtest = c->first;
      if (! test_is_blank_or_equality_test(subtest)) {
        /* --- create saved_test, splice this cons out of conjunct_list --- */
        allocate_with_pool (&current_agent(saved_test_pool), &saved);
        saved->next = old_sts;
        old_sts = saved;
        saved->var = sym;
        symbol_add_ref (sym);
        saved->the_test = subtest;
        if (prev_c)
          prev_c->rest = next_c;
        else
          ct->data.conjunct_list = next_c;
        free_cons (c);
      } else {
        prev_c = c;
      }
      c = next_c;
    }
    break;
    
  default:
    /* --- goal/impasse, disjunction, and non-equality relational tests --- */
    var = generate_new_variable ("dummy-");
    new = make_equality_test_without_adding_reference (var);
    allocate_with_pool (&current_agent(saved_test_pool), &saved);
    saved->next = old_sts;
    old_sts = saved;
    saved->var = var;
    symbol_add_ref (var);
    saved->the_test = *t;
    *t = new;
    break;
  }
  return old_sts;
}

saved_test *simplify_condition_list (condition *conds_list) {
  condition *c;
  saved_test *sts;

  sts = NIL;
  for (c=conds_list; c!=NIL; c=c->next) {
    if (c->type==POSITIVE_CONDITION) {
      sts = simplify_test (&(c->data.tests.id_test), sts);
      sts = simplify_test (&(c->data.tests.attr_test), sts);
      sts = simplify_test (&(c->data.tests.value_test), sts);
    }
  }
  return sts;
}

byte reverse_direction_of_relational_test (byte type) {
  switch (type) {
    case NOT_EQUAL_TEST: return NOT_EQUAL_TEST;
    case LESS_TEST: return GREATER_TEST;
    case GREATER_TEST: return LESS_TEST;
    case LESS_OR_EQUAL_TEST: return GREATER_OR_EQUAL_TEST;
    case GREATER_OR_EQUAL_TEST: return LESS_OR_EQUAL_TEST;
    case SAME_TYPE_TEST: return SAME_TYPE_TEST;
    default:
      print ("Internal error: arg to reverse_direction_of_relational_test\n");
      abort_with_fatal_error();
  }
  return 0; /* unreachable, but without it, gcc -Wall warns here */
}

saved_test *restore_saved_tests_to_test (test *t,
                                         bool is_id_field,
                                         tc_number bound_vars_tc_number,
                                         saved_test *tests_to_restore) {
  saved_test *st, *prev_st, *next_st;
  bool added_it;
  symbol *referent;
  complex_test *ct;
  
  prev_st = NIL;
  st = tests_to_restore;
  while (st) {
    next_st = st->next;
    added_it = FALSE;
    ct = complex_test_from_test(st->the_test);
    switch (ct->type) {
    case GOAL_ID_TEST:
    case IMPASSE_ID_TEST:
      if (! is_id_field) break; /* goal/impasse tests only go in id fields */
      /* ... otherwise fall through to the next case below ... */
    case DISJUNCTION_TEST:
      if (test_includes_equality_test_for_symbol (*t, st->var)) {
        add_new_test_to_test_if_not_already_there (t, st->the_test);
        added_it = TRUE;
      }
      break;
    default:  /* --- st->test is a relational test other than equality --- */
      referent = ct->data.referent;
      if (test_includes_equality_test_for_symbol (*t, st->var)) {
        if (symbol_is_constant_or_marked_variable (referent,
                                                   bound_vars_tc_number) ||
           (st->var == referent)) {
          add_new_test_to_test_if_not_already_there (t, st->the_test);
          added_it = TRUE;
        } 
      } else if (test_includes_equality_test_for_symbol (*t, referent)) {
        if (symbol_is_constant_or_marked_variable (st->var,
                                                   bound_vars_tc_number) ||
           (st->var == referent)) {
          ct->type = reverse_direction_of_relational_test (ct->type);
          ct->data.referent = st->var;
          st->var = referent;
          add_new_test_to_test_if_not_already_there (t, st->the_test);
          added_it = TRUE;
        }
      }
      break;
    } /* end of switch statement */
    if (added_it) {
      if (prev_st) prev_st->next = next_st; else tests_to_restore = next_st;
      symbol_remove_ref (st->var);
      free_with_pool (&current_agent(saved_test_pool), st);
    } else {
      prev_st = st;
    }
    st = next_st;
  } /* end of while (st) */
  return tests_to_restore;
}

void restore_and_deallocate_saved_tests (condition *conds_list,
                                         /* tc number for vars bound outside */
                                         tc_number tc, 
                                         saved_test *tests_to_restore) {
  condition *cond;
  list *new_vars;

  new_vars = NIL;
  for (cond=conds_list; cond!=NIL; cond=cond->next) {
    if (cond->type!=POSITIVE_CONDITION) continue;
    tests_to_restore = restore_saved_tests_to_test
      ((&cond->data.tests.id_test), TRUE, tc, tests_to_restore);
    add_bound_variables_in_test (cond->data.tests.id_test, tc, &new_vars);
    tests_to_restore = restore_saved_tests_to_test
      ((&cond->data.tests.attr_test), FALSE, tc, tests_to_restore);
    add_bound_variables_in_test (cond->data.tests.attr_test, tc, &new_vars);
    tests_to_restore = restore_saved_tests_to_test
      ((&cond->data.tests.value_test), FALSE, tc, tests_to_restore);
    add_bound_variables_in_test (cond->data.tests.value_test, tc, &new_vars);
  }
  if (tests_to_restore) {
    if (current_agent(sysparams)[PRINT_WARNINGS_SYSPARAM]) {
      print ("\nWarning:  in production %s,\n",
             name_of_production_being_reordered);
      print ("      ignoring test(s) whose referent is unbound:\n");
      print_saved_test_list (tests_to_restore);
    }
    /* ought to deallocate the saved tests, but who cares */
  }
  unmark_variables_and_free_list (new_vars);
}

/* =====================================================================

           Finding The Variables in a Negated Condition (or NCC)
                That Refer to Variables Bound Outside

  If a variable occurs within a negated condition (or NCC), and that 
  same variable is bound by some positive condition outside the negation,
  then the reorderer must ensure that the positive (binding) condition 
  comes before the negated (testing) condition.  To do this, we put
  a list on every NC or NCC of all the variables whose bindings it requires.

  When the reorderer is finished, these lists are removed.
  
  The main routines here are fill_in_vars_requiring_bindings() and
  remove_vars_requiring_bindings().  Each of these recursively traverses
  the lhs and does its work at all nesting levels.
  Fill_in_vars_requiring_bindings() takes a tc_number parameter which
  indicates the variables that are bound outside the give condition list.
  (At the top level, this should be *no* variables.)
  
===================================================================== */

list *collect_vars_tested_by_test_that_are_bound (test t,
                                                  tc_number tc,
                                                  list *starting_list) {
  cons *c;
  complex_test *ct;
  symbol *referent;

  if (test_is_blank_test(t)) return starting_list;

  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      if (referent->var.tc_num == tc)
        starting_list = add_if_not_member (referent, starting_list);
    return starting_list;
  }

  ct = complex_test_from_test(t);
  
  switch (ct->type) {
  case GOAL_ID_TEST:
  case IMPASSE_ID_TEST:
  case DISJUNCTION_TEST:
    return starting_list; 
    
  case CONJUNCTIVE_TEST:
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      starting_list = collect_vars_tested_by_test_that_are_bound
        (c->first, tc, starting_list);
    return starting_list;
    
  default:
    /* --- relational tests other than equality --- */
    referent = ct->data.referent;
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE)
      if (referent->var.tc_num == tc)
        starting_list = add_if_not_member (referent, starting_list);
    return starting_list; 
  }
}
                          
list *collect_vars_tested_by_cond_that_are_bound (condition *cond,
                                                  tc_number tc,
                                                  list *starting_list) {
  condition *c;

  if (cond->type==CONJUNCTIVE_NEGATION_CONDITION) {
    /* --- conjuctive negations --- */
    for (c=cond->data.ncc.top; c!=NIL; c=c->next)
      starting_list = collect_vars_tested_by_cond_that_are_bound
        (c, tc, starting_list);
  } else {
    /* --- positive, negative conditions --- */
    starting_list = collect_vars_tested_by_test_that_are_bound
      (cond->data.tests.id_test, tc, starting_list);
    starting_list = collect_vars_tested_by_test_that_are_bound
      (cond->data.tests.attr_test, tc, starting_list);
    starting_list = collect_vars_tested_by_test_that_are_bound
      (cond->data.tests.value_test, tc, starting_list);
  }
  return starting_list;
}

void fill_in_vars_requiring_bindings (condition *cond_list, tc_number tc) {
  list *new_bound_vars;
  condition *c;

  /* --- add anything bound in a positive condition at this level --- */
  new_bound_vars = NIL;
  for (c=cond_list; c!=NIL; c=c->next)
    if (c->type==POSITIVE_CONDITION)
      add_bound_variables_in_condition (c, tc, &new_bound_vars);

  /* --- scan through negated and NC cond's, fill in stuff --- */
  for (c=cond_list; c!=NIL; c=c->next) {
    if (c->type!=POSITIVE_CONDITION)
      c->reorder.vars_requiring_bindings =
        collect_vars_tested_by_cond_that_are_bound (c, tc, NIL);
    if (c->type==CONJUNCTIVE_NEGATION_CONDITION)
      fill_in_vars_requiring_bindings (c->data.ncc.top, tc);
  }

  unmark_variables_and_free_list (new_bound_vars);
}

void remove_vars_requiring_bindings (condition *cond_list) {
  condition *c;

  /* --- scan through negated and NC cond's, remove lists from them --- */
  for (c=cond_list; c!=NIL; c=c->next) {
    if (c->type!=POSITIVE_CONDITION)
      free_list (c->reorder.vars_requiring_bindings);
    if (c->type==CONJUNCTIVE_NEGATION_CONDITION)
      remove_vars_requiring_bindings (c->data.ncc.top);
  }
}

/* =====================================================================

             Finding the Root Variables in a Condition List

   This routine finds the root variables in a given condition list.
   The caller should setup the current tc to be the set of variables
   bound outside the given condition list.  (This should normally be
   an empty TC, except when the condition list is the subconditions
   of an NCC.)  If the "allow_printing_warnings" flag is TRUE, then
   this routine makes sure each root variable is accompanied by a
   goal or impasse id test, and prints a warning message if it isn't.
===================================================================== */

list *collect_root_variables (condition *cond_list,
                              tc_number tc, /* for vars bound outside */
                              bool allow_printing_warnings) {

  list *new_vars_from_value_slot;
  list *new_vars_from_id_slot;
  cons *c;
  condition *cond;
  bool found_goal_impasse_test;
 
  /* --- find everthing that's in the value slot of some condition --- */
  new_vars_from_value_slot = NIL;
  for (cond=cond_list; cond!=NIL; cond=cond->next)
    if (cond->type==POSITIVE_CONDITION)
      add_bound_variables_in_test (cond->data.tests.value_test, tc,
                                   &new_vars_from_value_slot);

  /* --- now see what else we can add by throwing in the id slot --- */
  new_vars_from_id_slot = NIL;
  for (cond=cond_list; cond!=NIL; cond=cond->next)
    if (cond->type==POSITIVE_CONDITION)
      add_bound_variables_in_test (cond->data.tests.id_test, tc,
                                   &new_vars_from_id_slot);

  /* --- unmark everything we just marked --- */
  unmark_variables_and_free_list (new_vars_from_value_slot);
  for (c=new_vars_from_id_slot; c!=NIL; c=c->rest)
    ((symbol *)(c->first))->var.tc_num = 0;
  
  /* --- make sure each root var has some condition with goal/impasse --- */
  if (allow_printing_warnings && current_agent(sysparams)[PRINT_WARNINGS_SYSPARAM]) {
    for (c=new_vars_from_id_slot; c!=NIL; c=c->rest) {
      found_goal_impasse_test = FALSE;
      for (cond=cond_list; cond!=NIL; cond=cond->next) {
        if (cond->type!=POSITIVE_CONDITION) continue;
        if (test_includes_equality_test_for_symbol (cond->data.tests.id_test,
                                                    c->first))
          if (test_includes_goal_or_impasse_id_test (cond->data.tests.id_test,
                                                     TRUE, TRUE)) {
            found_goal_impasse_test = TRUE;
            break;
          }
      }
      if (! found_goal_impasse_test) {
        print ("\nWarning: On the LHS of production %s, identifier ",
               name_of_production_being_reordered);
        print_with_symbols ("%y is not connected to any goal or impasse.\n",
                            (symbol *)(c->first));
      }
    }
  }
  
  return new_vars_from_id_slot;
}

/* =====================================================================

                     Reordering for LHS Conditions

   (Sorry for the poor comments here.  I think the reorderer needs
   substantial revisions in order to make Soar reliably scalable, so
   most of this code will eventually get thrown out anyway...)
===================================================================== */

/* --- estimated k-search branching factors --- */ 
#define MAX_COST 10000              /* cost of a disconnected condition */
#define BF_FOR_ACCEPTABLE_PREFS 8   /* cost of (. ^. <var> +) */
#define BF_FOR_VALUES 8             /* cost of (. ^. <var>) */
#define BF_FOR_ATTRIBUTES 8         /* cost of (. ^<var> .) */

/* -------------------------------------------------------------
   Return TRUE iff the given test is covered by the previously
   bound variables.  The set of previously bound variables is
   given by the current TC, PLUS any variables in the list
   "extra_vars."
------------------------------------------------------------- */

bool test_covered_by_bound_vars (test t, tc_number tc, list *extra_vars) {
  cons *c;
  symbol *referent;
  complex_test *ct;

  if (test_is_blank_test(t)) return FALSE;
  
  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (symbol_is_constant_or_marked_variable (referent, tc))
      return TRUE;
    if (extra_vars) return member_of_list (referent, extra_vars);
    return FALSE;
  }

  ct = complex_test_from_test(t);
  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if (test_covered_by_bound_vars (c->first, tc, extra_vars))
        return TRUE;
  }
  return FALSE;
}

/* -------------------------------------------------------------
   Return an estimate of the "cost" of the given condition.
   The current TC should be the set of previously bound variables;
   "root_vars_not_bound_yet" should be the set of other root
   variables.
------------------------------------------------------------- */

long cost_of_adding_condition (condition *cond,
                               tc_number tc,
                               list *root_vars_not_bound_yet) {
  cons *c;
  long result;

  /* --- handle the common simple case quickly up front --- */
  if ((! root_vars_not_bound_yet) &&
      (cond->type==POSITIVE_CONDITION) &&
      (test_is_blank_or_equality_test (cond->data.tests.id_test)) &&
      (test_is_blank_or_equality_test (cond->data.tests.attr_test)) &&
      (test_is_blank_or_equality_test (cond->data.tests.value_test)) &&
      (! test_is_blank_test (cond->data.tests.id_test)) &&
      (! test_is_blank_test (cond->data.tests.attr_test)) &&
      (! test_is_blank_test (cond->data.tests.value_test)) ) {

    if (! symbol_is_constant_or_marked_variable
          (referent_of_equality_test (cond->data.tests.id_test), tc))
      return MAX_COST;
    if (symbol_is_constant_or_marked_variable
          (referent_of_equality_test (cond->data.tests.attr_test), tc))
      result = 1;
    else
      result =  BF_FOR_ATTRIBUTES;

    if (! symbol_is_constant_or_marked_variable
          (referent_of_equality_test(cond->data.tests.value_test),tc)){
      if (cond->test_for_acceptable_preference)
        result = result * BF_FOR_ACCEPTABLE_PREFS;
      else
        result = result * BF_FOR_VALUES;
    }
    return result;
  } /* --- end of common simple case --- */

  if (cond->type==POSITIVE_CONDITION) {
    /* --- for pos cond's, check what's bound, etc. --- */
    if (! test_covered_by_bound_vars (cond->data.tests.id_test, tc,
                                      root_vars_not_bound_yet))
      return MAX_COST;
    if (test_covered_by_bound_vars (cond->data.tests.attr_test, tc,
                                    root_vars_not_bound_yet))
      result = 1;
    else
      result =  BF_FOR_ATTRIBUTES;
    if (! test_covered_by_bound_vars (cond->data.tests.value_test, tc,
                                      root_vars_not_bound_yet)) {
      if (cond->test_for_acceptable_preference)
        result = result * BF_FOR_ACCEPTABLE_PREFS;
      else
        result = result * BF_FOR_VALUES;
    }
    return result;
  }
  /* --- negated or NC conditions:  just check whether all variables
     requiring bindings are actually bound.  If so, return 1, else
     return MAX_COST --- */
  for (c=cond->reorder.vars_requiring_bindings; c!=NIL; c=c->rest) {
    if (((symbol *)(c->first))->var.tc_num != tc) return MAX_COST;
  }
  return 1;
}

/* -------------------------------------------------------------
   Return an estimate of the "cost" of the lowest-cost condition
   that could be added next, IF the given "chosen" condition is
   added first.
------------------------------------------------------------- */

long find_lowest_cost_lookahead (condition *candidates,
                                 condition *chosen,
                                 tc_number tc,
                                 list *root_vars_not_bound_yet) {
  condition *c;
  long min_cost, cost;
  list *new_vars;

  new_vars = NIL;
  add_bound_variables_in_condition (chosen, tc, &new_vars);
  min_cost = MAX_COST + 1;
  for (c=candidates; c!=NIL; c=c->next) {
    if (c==chosen) continue;
    cost = cost_of_adding_condition (c, tc, root_vars_not_bound_yet);
    if (cost < min_cost) {
      min_cost = cost;
      if (cost <= 1) break;
    }
  }
  unmark_variables_and_free_list (new_vars);
  return min_cost;
}

/* -------------------------------------------------------------
   Reorder the given list of conditions.  The "top_of_conds" and
   "bottom_of_conds" arguments are destructively modified to reflect
   the reordered conditions.  The "bound_vars_tc_number"
   should reflect the variables bound outside the given condition list.
   The "reorder_nccs" flag indicates whether it is necessary to
   recursively reorder the subconditions of NCC's.  (For newly
   built chunks, this is never necessary.)
------------------------------------------------------------- */

void reorder_condition_list (condition **top_of_conds,
                             condition **bottom_of_conds,
                             list *roots,
                             tc_number tc,
                             bool reorder_nccs);

void reorder_simplified_conditions (condition **top_of_conds,
                                    condition **bottom_of_conds,
                                    list *roots,
                                    tc_number bound_vars_tc_number,
                                    bool reorder_nccs) {
  condition *remaining_conds;           /* header of dll */
  condition *first_cond, *last_cond;
  condition *cond;
  condition *min_cost_conds, *chosen;
  long cost, min_cost;
  list *new_vars;

  remaining_conds = *top_of_conds;
  first_cond = NIL;
  last_cond = NIL;
  new_vars = NIL;
  
  /* repeat:  scan through remaining_conds
              rate each one
              if tie, call lookahead routine
              add min-cost item to conds
  */
  
  while (remaining_conds) {
    /* --- find min-cost set --- */
    min_cost_conds = NIL;
    min_cost = MAX_COST + 1;
    for (cond=remaining_conds; cond!=NIL; cond=cond->next) {
      cost = cost_of_adding_condition (cond, bound_vars_tc_number, roots);
      if (cost < min_cost) {
        min_cost = cost;
        min_cost_conds = cond;
        cond->reorder.next_min_cost = NIL;
      } else if (cost==min_cost) {
        cond->reorder.next_min_cost = min_cost_conds;
        min_cost_conds = cond;
      }
      if (min_cost <= 1) break;
    }
    /* --- if min_cost==MAX_COST, print error message --- */
    if ((min_cost==MAX_COST) && current_agent(sysparams)[PRINT_WARNINGS_SYSPARAM]) {
      print ("Warning:  in production %s,\n",
             name_of_production_being_reordered);
      print ("     The LHS conditions are not all connected.\n");
      /* BUGBUG I'm not sure whether this can ever happen. */
    }
    /* --- if more than one min-cost item, and cost>1, do lookahead --- */
    if ((min_cost > 1) && (min_cost_conds->reorder.next_min_cost)) {
      min_cost = MAX_COST + 1;
      for (cond=min_cost_conds; cond!=NIL; cond=cond->reorder.next_min_cost) {
        cost = find_lowest_cost_lookahead (remaining_conds, cond,
                                           bound_vars_tc_number, roots);
        if (cost < min_cost) {
          min_cost = cost;
          min_cost_conds = cond;
        }
      }
    }
    /* --- install the first item in the min-cost set --- */
    chosen = min_cost_conds;
    remove_from_dll (remaining_conds, chosen, next, prev);
    if (!first_cond) first_cond = chosen;
    /* Note: args look weird on the next line, because we're really
       inserting the chosen item at the *end* of the list */
    insert_at_head_of_dll (last_cond, chosen, prev, next);

    /* --- if a conjunctive negation, recursively reorder its conditions --- */
    if ((chosen->type==CONJUNCTIVE_NEGATION_CONDITION) && reorder_nccs) {
      list *ncc_roots;
      ncc_roots = collect_root_variables (chosen->data.ncc.top,
                                          bound_vars_tc_number, TRUE);
      reorder_condition_list (&(chosen->data.ncc.top),
                              &(chosen->data.ncc.bottom),
                              ncc_roots,
                              bound_vars_tc_number,
                              reorder_nccs);
      free_list (ncc_roots);
    }

    /* --- update set of bound variables for newly added condition --- */
    add_bound_variables_in_condition (chosen, bound_vars_tc_number, &new_vars);
    
    /* --- if all roots are bound, set roots=NIL: don't need 'em anymore --- */
    if (roots) {
      cons *c;
      for (c=roots; c!=NIL; c=c->rest)
        if (((symbol *)(c->first))->var.tc_num != bound_vars_tc_number)
          break;
      if (!c) roots=NIL;
    }

  } /* end of while (remaining_conds) */

  unmark_variables_and_free_list (new_vars);
  *top_of_conds = first_cond;
  *bottom_of_conds = last_cond;
}

void reorder_condition_list (condition **top_of_conds,
                             condition **bottom_of_conds,
                             list *roots,
                             tc_number tc, /* for vars bound outside */
                             bool reorder_nccs) {
  saved_test *saved_tests;

  saved_tests = simplify_condition_list (*top_of_conds);
  reorder_simplified_conditions (top_of_conds, bottom_of_conds, roots, tc,
                                 reorder_nccs);
  restore_and_deallocate_saved_tests (*top_of_conds, tc, saved_tests);
}

/* -------------------------------------------------------------
   Reorders the LHS.
------------------------------------------------------------- */

bool reorder_lhs (condition **lhs_top, condition **lhs_bottom,
                  bool reorder_nccs) {
  tc_number tc;
  list *roots;

  tc = get_new_tc_number ();
  /* don't mark any variables, since nothing is bound outside the LHS */

  roots = collect_root_variables (*lhs_top, tc, TRUE);
  if (!roots) {
    print ("Error:  in production %s,\n", name_of_production_being_reordered);
    print ("        The LHS has no roots.\n");
    /* BUGBUG most people aren't going to understand this error message */
    return FALSE;
  }

  fill_in_vars_requiring_bindings (*lhs_top, tc);
  reorder_condition_list (lhs_top, lhs_bottom, roots, tc, reorder_nccs);
  remove_vars_requiring_bindings (*lhs_top);
  free_list (roots);
  return TRUE;
}

void init_reorderer (void) {  /* called from init_production_utilities() */
  init_memory_pool (&current_agent(saved_test_pool), sizeof(saved_test), "saved test");
}

/* *********************************************************************

                   Compile-Time O-Support Calculations

********************************************************************* */

/* ------------------------------------------------------------------
                         Test Is For Symbol

   This function determines whether a given symbol could be the match
   for a given test.  It returns YES if the symbol is the only symbol
   that could pass the test (i.e., the test *forces* that symbol to be
   present in WM), NO if the symbol couldn't possibly pass the test,
   and MAYBE if it can't tell for sure.  The symbol may be a variable;
   the test may contain variables.
------------------------------------------------------------------ */

typedef enum yes_no_maybe_enum { YES, NO, MAYBE } yes_no_maybe;

yes_no_maybe test_is_for_symbol (test t, symbol *sym) {
  cons *c;
  yes_no_maybe temp;
  bool maybe_found;
  complex_test *ct;
  symbol *referent;

  if (test_is_blank_test(t)) return MAYBE;

  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent==sym) return YES;
    if (referent->common.symbol_type==VARIABLE_SYMBOL_TYPE) return MAYBE;
    if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE) return MAYBE;
    return NO;
  }

  ct = complex_test_from_test(t);
  
  switch (ct->type) {
  case DISJUNCTION_TEST:
    if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE) return MAYBE;
    if (member_of_list (sym, ct->data.disjunction_list)) return MAYBE;
    return NO;
  case CONJUNCTIVE_TEST:
    maybe_found = FALSE;
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest) {
      temp = test_is_for_symbol (c->first, sym);
      if (temp==YES) return YES;
      if (temp==MAYBE) maybe_found = TRUE;
    }
    if (maybe_found) return MAYBE;
    return NO;
  default:  /* goal/impasse tests, relational tests other than equality */
    return MAYBE;
  }
}

/* ------------------------------------------------------------------
                         Find Known Goals

   This routine looks at the LHS and returns a list of variables that
   are certain to be bound to goals.

   Note:  this uses the TC routines and clobbers any existing TC.
                         
   BUGBUG should follow ^object links up the goal stack if possible
------------------------------------------------------------------ */

list *find_known_goals (condition *lhs) {
  tc_number tc;
  list *vars;
  condition *c;

  tc = get_new_tc_number();
  vars = NIL;
  for (c=lhs; c!=NIL; c=c->next) {
    if (c->type != POSITIVE_CONDITION) continue;
    if (test_includes_goal_or_impasse_id_test (c->data.tests.id_test,
                                               TRUE,
                                               FALSE))
      add_bound_variables_in_test (c->data.tests.id_test, tc, &vars);
  }
  return vars;
}

/* ------------------------------------------------------------------
                  Find Compile Time Match Goal

   Given the LHS and a list of known goals (i.e., variables that must
   be bound to goals at run-time), this routine tries to determine
   which variable will be the match goal.  If successful, it returns
   that variable; if it can't tell which variable will be the match
   goal, it returns NIL.

   Note:  this uses the TC routines and clobbers any existing TC.
------------------------------------------------------------------ */

symbol *find_compile_time_match_goal (condition *lhs, list *known_goals) {
  tc_number tc;
  list *roots;
  list *root_goals;
  int num_root_goals;
  cons *c, *prev_c, *next_c;
  symbol *result;
  condition *cond;
  
  /* --- find root variables --- */
  tc = get_new_tc_number();
  roots = collect_root_variables (lhs, tc, FALSE);
  
  /* --- intersect roots with known_goals, producing root_goals --- */
  root_goals = NIL;
  num_root_goals = 0;
  for (c=roots; c!=NIL; c=c->rest)
    if (member_of_list (c->first, known_goals)) {
      push (c->first, root_goals);
      num_root_goals++;
    }
  free_list (roots);

  /* --- if more than one goal, remove any with "^object nil" --- */
  if (num_root_goals > 1) {
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if ((cond->type==POSITIVE_CONDITION) &&
          (test_is_for_symbol(cond->data.tests.attr_test,current_agent(object_symbol))==YES)&&
          (test_is_for_symbol(cond->data.tests.value_test,current_agent(nil_symbol))==YES)) {
        prev_c = NIL;
        for (c=root_goals; c!=NIL; c=next_c) {
          next_c = c->rest;
          if (test_is_for_symbol (cond->data.tests.id_test, c->first)==YES) {
            /* --- remove c from the root_goals list --- */
            if (prev_c) prev_c->rest = next_c; else root_goals = next_c;
            free_cons (c);
            num_root_goals--;
            if (num_root_goals==1) break; /* be sure not to remove them all */
          } else {
            prev_c = c;
          }
        } /* end of for (c) loop */
        if (num_root_goals==1) break; /* be sure not to remove them all */
      }
    } /* end of for (cond) loop */
  }
  
  /* --- if there's only one root goal, that's it! --- */
  if (num_root_goals==1)
    result = root_goals->first;
  else
    result = NIL;

  /* --- clean up and return result --- */
  free_list (root_goals);
  return result;      
}

/* ------------------------------------------------------------------
                       Find Thing Off Goal

   Given the LHS and a the match goal variable, this routine looks
   for a positive condition testing (goal ^attr) for the given attribute
   "attr".  If such a condition exists, and the value field contains
   an equality test for a variable, then that variable is returned.
   (If more than one such variable exists, one is chosen arbitrarily
   and returned.)  Otherwise the function returns NIL.

   Note:  this uses the TC routines and clobbers any existing TC.
------------------------------------------------------------------ */

symbol *find_thing_off_goal (condition *lhs, symbol *goal, symbol *attr) {
  condition *c;
  list *vars;
  tc_number tc;
  symbol *result;

  for (c=lhs; c!=NIL; c=c->next) {
    if (c->type != POSITIVE_CONDITION) continue;
    if (test_is_for_symbol (c->data.tests.id_test, goal) != YES) continue;
    if (test_is_for_symbol (c->data.tests.attr_test, attr) != YES) continue;
    if (c->test_for_acceptable_preference) continue;
    tc = get_new_tc_number();
    vars = NIL;
    add_bound_variables_in_test (c->data.tests.value_test, tc, &vars);
    if (vars) {
      result = vars->first;
      free_list (vars);
      return result;
    }
  }
  return NIL;
}

/* ------------------------------------------------------------------
                 Condition List Has Id Test For Sym

   This checks whether a given condition list has an equality test for
   a given symbol in the id field of any condition (at any nesting level
   within NCC's).
------------------------------------------------------------------ */

bool condition_list_has_id_test_for_sym (condition *conds, symbol *sym) {
  for ( ; conds!=NIL; conds=conds->next) {
    switch (conds->type) {
    case POSITIVE_CONDITION:
    case NEGATIVE_CONDITION:
      if (test_includes_equality_test_for_symbol (conds->data.tests.id_test,
                                                  sym))
        return TRUE;
      break;
    case CONJUNCTIVE_NEGATION_CONDITION:
      if (condition_list_has_id_test_for_sym (conds->data.ncc.top, sym))
        return TRUE;
      break;
    }
  }
  return FALSE;
}

/* ------------------------------------------------------------------
                      Add TC Through LHS and RHS

   This enlarges a given TC by adding to it any connected conditions
   in the LHS or actions in the RHS.
------------------------------------------------------------------ */

void add_tc_through_lhs_and_rhs (condition *lhs, action *rhs, tc_number tc,
                                 list **id_list, list **var_list) {
  condition *c;
  action *a;
  bool anything_changed;
  
  for (c=lhs; c!=NIL; c=c->next) c->already_in_tc = FALSE;
  for (a=rhs; a!=NIL; a=a->next) a->already_in_tc = FALSE;

  /* --- keep trying to add new stuff to the tc --- */  
  while (TRUE) {
    anything_changed = FALSE;
    for (c=lhs; c!=NIL; c=c->next)
      if (! c->already_in_tc)
        if (cond_is_in_tc (c, tc)) {
          add_cond_to_tc (c, tc, id_list, var_list);
          c->already_in_tc = TRUE;
          anything_changed = TRUE;
        }
    for (a=rhs; a!=NIL; a=a->next)
      if (! a->already_in_tc)
        if (action_is_in_tc (a, tc)) {
          add_action_to_tc (a, tc, id_list, var_list);
          a->already_in_tc = TRUE;
          anything_changed = TRUE;
        }
    if (! anything_changed) break;
  }
}

/* -----------------------------------------------------------------------
                   Calculate Compile Time O-Support

   This takes the LHS and RHS, and fills in the a->support field in each
   RHS action with either UNKNOWN_SUPPORT, O_SUPPORT, or NO_O_SUPPORT.
   (Actually, it only does this for MAKE_ACTION's--for FUNCALL_ACTION's,
   the support doesn't matter.)
----------------------------------------------------------------------- */

void calculate_compile_time_o_support (condition *lhs, action *rhs) {
  list *known_goals;
  cons *c;
  symbol *match_goal, *match_state, *match_operator;
  yes_no_maybe lhs_oa_support, lhs_oc_support, lhs_om_support;
  action *a;
  condition *cond;
  yes_no_maybe ynm;
  bool operator_found, possible_operator_found;
  tc_number tc;

  /* --- initialize:  mark all rhs actions as "unknown" --- */
  for (a=rhs; a!=NIL; a=a->next)
    if (a->type==MAKE_ACTION) a->support=UNKNOWN_SUPPORT;

  /* --- if "operator" doesn't appear in any LHS attribute slot, and there
         are no RHS +/! makes for "operator", then nothing gets support --- */
  operator_found = FALSE;
  possible_operator_found = FALSE;
  for (cond=lhs; cond!=NIL; cond=cond->next) {
    if (cond->type != POSITIVE_CONDITION) continue;
    ynm = test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol));
    if (ynm==YES) { operator_found = possible_operator_found = TRUE; break; }
    if (ynm==MAYBE) possible_operator_found = TRUE;
  }
  if (! operator_found)
    for (a=rhs; a!=NIL; a=a->next) {
      if (a->type != MAKE_ACTION) continue;
      if (a->attr==current_agent(operator_symbol))
        { operator_found = possible_operator_found = TRUE; break; }
      if (a->attr->common.symbol_type==VARIABLE_SYMBOL_TYPE)
        possible_operator_found = TRUE;
    }
  if (! possible_operator_found) {
    for (a=rhs; a!=NIL; a=a->next)
      if (a->type == MAKE_ACTION) a->support=NO_O_SUPPORT;
    return;
  }

  /* --- find known goals; RHS augmentations of goals get no support --- */
  known_goals = find_known_goals (lhs);
  for (c=known_goals; c!=NIL; c=c->rest)
    for (a=rhs; a!=NIL; a=a->next)
      if (a->type == MAKE_ACTION)
        if (a->id == c->first) a->support = NO_O_SUPPORT;

  /* --- find match goal, state, and operator --- */
  match_goal = find_compile_time_match_goal (lhs, known_goals);
  free_list (known_goals);
  if (!match_goal) return;
  match_state = find_thing_off_goal (lhs, match_goal, current_agent(state_symbol));
  if (!match_state) return;
  match_operator = find_thing_off_goal (lhs, match_goal, current_agent(operator_symbol));

  /* --- If when checking (above) for "operator" appearing anywhere, we
     found a possible operator but not a definite operator, now go back and
     see if the possible operator was actually the match goal or match state;
     if so, it's not a possible operator.  (Note:  by "possible operator" I
     mean something appearing in the *attribute* field that might get bound
     to the symbol "operator".)  --- */
  if (possible_operator_found && !operator_found) {
    possible_operator_found = FALSE;
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if (cond->type != POSITIVE_CONDITION) continue;
      ynm = test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol));
      if ((ynm!=NO) &&
          (test_is_for_symbol (cond->data.tests.attr_test, match_goal)!=YES) &&
          (test_is_for_symbol (cond->data.tests.attr_test, match_state)!=YES))
        { possible_operator_found = TRUE; break; }
    }
    if (! possible_operator_found) {
      for (a=rhs; a!=NIL; a=a->next) {
        if (a->type != MAKE_ACTION) continue;
        /* we're looking for "operator" augs of goals only, and match_state
           couldn't get bound to a goal */
        if (a->id == match_state) continue;
        if ((a->attr->common.symbol_type==VARIABLE_SYMBOL_TYPE) &&
            (a->attr != match_goal) &&
            (a->attr != match_state))
          { possible_operator_found = TRUE; break; }
      }
    }
    if (! possible_operator_found) {
      for (a=rhs; a!=NIL; a=a->next)
        if (a->type == MAKE_ACTION) a->support=NO_O_SUPPORT;
      return;
    }
  }
  
  /* --- calculate LHS support predicates --- */
  lhs_oa_support = MAYBE;
  if (match_operator)
    if (condition_list_has_id_test_for_sym (lhs, match_operator))
      lhs_oa_support = YES;

  lhs_oc_support = MAYBE;
  lhs_om_support = MAYBE;
  if (condition_list_has_id_test_for_sym (lhs, match_state)) {
    lhs_oc_support = YES;
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if (cond->type != POSITIVE_CONDITION) continue;
      if (test_is_for_symbol (cond->data.tests.id_test, match_goal) != YES)
        continue;
      if (test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol))
          != YES)
        continue;
      lhs_om_support = YES;
      break;
    }
  }     

  if (lhs_oa_support == YES) {    /* --- look for RHS o-a support --- */
    /* --- do TC(match_state) --- */
    tc = get_new_tc_number();
    add_symbol_to_tc (match_state, tc, NIL, NIL);
    add_tc_through_lhs_and_rhs (lhs, rhs, tc, NIL, NIL);

    /* --- any action with id in the TC gets support --- */
    for (a=rhs; a!=NIL; a=a->next)
      if (action_is_in_tc (a, tc)) a->support = O_SUPPORT;
  }

  if (lhs_oc_support == YES) {    /* --- look for RHS o-c support --- */
    /* --- do TC(rhs operators) --- */
    tc = get_new_tc_number();
    for (a=rhs; a!=NIL; a=a->next) {
      if (a->type != MAKE_ACTION) continue;
      if ((a->id==match_goal) &&
          (a->attr==current_agent(operator_symbol)) &&
          ((a->preference_type==ACCEPTABLE_PREFERENCE_TYPE) ||
           (a->preference_type==REQUIRE_PREFERENCE_TYPE)) ) {
        if (rhs_value_is_symbol(a->value))
          add_symbol_to_tc (rhs_value_to_symbol(a->value), tc, NIL,NIL);
      }
    }
    add_tc_through_lhs_and_rhs (lhs, rhs, tc, NIL, NIL);

    /* --- any action with id in the TC gets support --- */
    for (a=rhs; a!=NIL; a=a->next)
      if (action_is_in_tc (a, tc)) a->support = O_SUPPORT;
  }

  if (lhs_om_support == YES) {    /* --- look for RHS o-m support --- */
    /* --- do TC(lhs operators) --- */
    tc = get_new_tc_number();
    for (cond=lhs; cond!=NIL; cond=cond->next) {
      if (cond->type != POSITIVE_CONDITION) continue;
      if (test_is_for_symbol (cond->data.tests.id_test, match_goal) == YES)
        if (test_is_for_symbol (cond->data.tests.attr_test, current_agent(operator_symbol))
            == YES)
          add_bound_variables_in_test (cond->data.tests.value_test, tc, NIL);
    }
    add_tc_through_lhs_and_rhs (lhs, rhs, tc, NIL, NIL);

    /* --- any action with id in the TC gets support --- */
    for (a=rhs; a!=NIL; a=a->next)
      if (action_is_in_tc (a, tc)) a->support = O_SUPPORT;
  }
}

/* *********************************************************************

                         Production Management

    Make_production() does reordering, compile-time o-support calc's,
    and builds and returns a production structure for a new production.
    It does not enter the production into the Rete net, however.
    The "type" argument should be one of USER_PRODUCTION_TYPE, etc.
    The flag "reorder_nccs" tells whether to recursively reorder
    the subconditions of NCC's--this is not necessary for newly
    built chunks, as their NCC's are copies of other NCC's in SP's that
    have already been reordered.  If any error occurs, make_production()
    returns NIL.

    Deallocate_production() and excise_production() do just what they
    say.  Normally deallocate_production() should be invoked only via
    the production_remove_ref() macro.
********************************************************************* */

production *make_production (byte type,
                             symbol *name,
                             condition **lhs_top,
                             condition **lhs_bottom,
                             action **rhs_top,
                             bool reorder_nccs) {
  production *p;
  list *rhs_unbound_vars;
  cons *c;
  tc_number tc;
  action *a;

  name_of_production_being_reordered = name->sc.name;

  reset_variable_generator (*lhs_top, *rhs_top);

  tc = get_new_tc_number();
  add_bound_variables_in_condition_list (*lhs_top, tc, NIL);

  rhs_unbound_vars = NIL;
  add_all_variables_in_action_list (*rhs_top, tc, &rhs_unbound_vars);
  for (c=rhs_unbound_vars; c!=NIL; c=c->rest)
    (((symbol *)(c->first))->var.tc_num = 0);

  if (type!=JUSTIFICATION_PRODUCTION_TYPE) {
    if (! reorder_action_list (rhs_top, tc)) {
      free_list (rhs_unbound_vars);
      return NIL;
    }
    if (! reorder_lhs (lhs_top, lhs_bottom, reorder_nccs)) {
      free_list (rhs_unbound_vars);
      return NIL;
    }
    
#ifdef DO_COMPILE_TIME_O_SUPPORT_CALCS
    calculate_compile_time_o_support (*lhs_top, *rhs_top);
#ifdef LIST_COMPILE_TIME_O_SUPPORT_FAILURES
    for (a = *rhs_top; a!=NIL; a=a->next)
      if ((a->type==MAKE_ACTION) && (a->support==UNKNOWN_SUPPORT)) break;
    if (a) print_with_symbols ("\nCan't classify %y\n", name);
#endif
#else
    for (a = *rhs_top; a!=NIL; a=a->next) a->support = UNKNOWN_SUPPORT;
#endif
  } else {
    /* --- for justifications --- */
    /* force run-time o-support (it'll only be done once) */
    for (a = *rhs_top; a!=NIL; a=a->next) a->support = UNKNOWN_SUPPORT;
  }

  allocate_with_pool (&current_agent(production_pool), &p);
  p->name = name;
  if (name->sc.production) {
    print ("Internal error: make_production called with name %s\n",
           name_of_production_being_reordered);
    print ("for which a production already exists\n");
  }
  name->sc.production = p;
  p->documentation = NIL;
  p->firing_count = 0;
  p->reference_count = 1;
  insert_at_head_of_dll (current_agent(all_productions_of_type)[type], p, next, prev);
  current_agent(num_productions_of_type)[type]++;
  p->type = type;
  p->declared_support = UNDECLARED_SUPPORT;
  p->trace_firings = FALSE;
  p->p_node = NIL;               /* it's not in the rete yet */
  p->action_list = *rhs_top;
  p->rhs_unbound_variables = rhs_unbound_vars;
  p->instantiations = NIL;

  return p;
}

void deallocate_production (production *prod) {
  if (prod->instantiations) {
    print ("Internal error: deallocating prod. that still has inst's\n");
    abort_with_fatal_error();
  }
  deallocate_action_list (prod->action_list);
  free_list (prod->rhs_unbound_variables);
  symbol_remove_ref (prod->name);
  if (prod->documentation) free_memory_block_for_string (prod->documentation);
  free_with_pool (&current_agent(production_pool), prod);
}

void excise_production (production *prod, bool print_sharp_sign) {
  if (prod->trace_firings) remove_ptrace (prod);
  remove_from_dll (current_agent(all_productions_of_type)[prod->type], prod, next, prev);
  current_agent(num_productions_of_type)[prod->type]--;
  if (print_sharp_sign) print ("#");
  if (prod->p_node) excise_production_from_rete (prod);
  prod->name->sc.production = NIL;
  production_remove_ref (prod);
}
