/* =======================================================================

             Recognition Memory (Firer and Chunker) Routines
                   (Does not include the Rete net)

======================================================================= */

#include <ctype.h>
#include <sys/time.h>
#include "soar.h"

/* -----------------------------------------------------------------------
                         Find Clone For Level

   This routines take a given preference and finds the clone of it whose
   match goal is at the given goal_stack_level.  (This is used to find the
   proper preference to backtrace through.)  If the given preference
   itself is at the right level, it is returned.  If there is no clone at
   the right level, NIL is returned.
----------------------------------------------------------------------- */

preference *find_clone_for_level (preference *p, goal_stack_level level) {
  preference *clone;

  if (! p) {
    /* --- if the wme doesn't even have a preference on it, we can't backtrace
       at all (this happens with I/O and some architecture-created wmes --- */
    return NIL;
  }

  /* --- look at pref and all of its clones, find one at the right level --- */

  if (p->inst->match_goal_level == level) return p;

  for (clone=p->next_clone; clone!=NIL; clone=clone->next_clone)
    if (clone->inst->match_goal_level==level) return clone;

  for (clone=p->prev_clone; clone!=NIL; clone=clone->prev_clone)
    if (clone->inst->match_goal_level==level) return clone;

  /* --- if none was at the right level, we can't backtrace at all --- */
  return NIL;
}
  
/* =======================================================================

                           Firer Utilities

======================================================================= */

/* -----------------------------------------------------------------------
                             Find Match Goal

   Given an instantiation, this routines looks at the instantiated
   conditions to find its match goal.  It fills in inst->match_goal and
   inst->match_goal_level.  If there is a match goal, match_goal is set
   to point to the goal identifier.  If no goal was matched, match_goal
   is set to NIL and match_goal_level is set to ATTRIBUTE_IMPASSE_LEVEL.
----------------------------------------------------------------------- */

void find_match_goal (instantiation *inst) {
  symbol *lowest_goal_so_far;
  goal_stack_level lowest_level_so_far;
  condition *cond;
  symbol *id;
  
  lowest_goal_so_far = NIL;
  lowest_level_so_far = -1;
  for (cond=inst->top_of_instantiated_conditions; cond!=NIL; cond=cond->next)
    if (cond->type==POSITIVE_CONDITION) {
      id = cond->bt.wme->id;
      if (id->id.isa_goal)
        if (cond->bt.level > lowest_level_so_far) {
          lowest_goal_so_far = id;
          lowest_level_so_far = cond->bt.level;
        }
    }
  
  inst->match_goal = lowest_goal_so_far;
  if (lowest_goal_so_far)
    inst->match_goal_level = lowest_level_so_far;
  else
    inst->match_goal_level = ATTRIBUTE_IMPASSE_LEVEL;
}

/* -----------------------------------------------------------------------
                  O-Support Transitive Closure Routines

   These routines are used by the o-support calculations to mark transitive
   closures through TM (= WM+PM) plus (optionally) the RHS-generated pref's.

   The caller should first call begin_os_tc (rhs_prefs_or_nil).  Then
   add_to_os_tc (id) should be called any number of times to add stuff
   to the TC.  (Note that the rhs_prefs shouldn't be modified between the
   begin_os_tc() call and the last add_to_os_tc() call.)

   Each identifier in the TC is marked with id.tc_num=o_support_tc; the
   caller can check for TC membership by looking at id.tc_num on any id.
----------------------------------------------------------------------- */

tc_number o_support_tc;   /* current tc number in use by o-support calc's */
preference *rhs_prefs_from_instantiation; /* RHS-generated prefs, or NIL */

#define add_to_os_tc_if_needed(sym) \
  { if ((sym)->common.symbol_type==IDENTIFIER_SYMBOL_TYPE) \
      add_to_os_tc (sym); }

void add_to_os_tc (symbol *id) {
  slot *s;
  preference *pref;
  wme *w;

  /* --- if id is already in the TC, exit; else mark it as in the TC --- */
  if (id->id.tc_num==o_support_tc) return;
  id->id.tc_num = o_support_tc;
  
  /* --- scan through all preferences and wmes for all slots for this id --- */
  for (w=id->id.input_wmes; w!=NIL; w=w->next)
    add_to_os_tc_if_needed (w->value);
  for (s=id->id.slots; s!=NIL; s=s->next) {
    for (pref=s->all_preferences; pref!=NIL; pref=pref->all_of_slot_next) {
      add_to_os_tc_if_needed (pref->value);
      if (preference_is_binary(pref->type))
        add_to_os_tc_if_needed (pref->referent);
    }
    for (w=s->wmes; w!=NIL; w=w->next)
      add_to_os_tc_if_needed (w->value);
  } /* end of for slots loop */
  /* --- now scan through RHS prefs and look for any with this id --- */
  for (pref=rhs_prefs_from_instantiation; pref!=NIL; pref=pref->inst_next) {
    if (pref->id==id) {
      add_to_os_tc_if_needed (pref->value);
      if (preference_is_binary(pref->type))
        add_to_os_tc_if_needed (pref->referent);
    }
  }
  /* We don't need to worry about goal/impasse wmes here, since o-support tc's
     never start there and there's never a pointer to a goal or impasse from
     something else. */
}

void begin_os_tc (preference *rhs_prefs_or_nil) {
  o_support_tc = get_new_tc_number();
  rhs_prefs_from_instantiation = rhs_prefs_or_nil;
}

/* -----------------------------------------------------------------------
           Utilities for Testing Inclusion in the O-Support TC

   After a TC has been marked with the above routine, these utility
   routines are used for checking whether certain things are in the TC.
   Test_has_id_in_os_tc() checks whether a given test contains an equality
   test for any identifier in the TC, other than the identifier
   "excluded_sym".  Id_or_value_of_condition_list_is_in_os_tc() checks whether
   any id or value test in the given condition list (including id/value tests
   inside NCC's) has a test for an id in the TC.  In the case of value tests,
   the id is not allowed to be "sym_excluded_from_value".
----------------------------------------------------------------------- */

bool test_has_id_in_os_tc (test t, symbol *excluded_sym) {
  cons *c;
  symbol *referent;
  complex_test *ct;

  if (test_is_blank_test(t)) return FALSE;
  if (test_is_blank_or_equality_test(t)) {
    referent = referent_of_equality_test(t);
    if (referent->common.symbol_type==IDENTIFIER_SYMBOL_TYPE)
      if (referent->id.tc_num==o_support_tc)
        if (referent!=excluded_sym)
          return TRUE;
    return FALSE;
  }
  ct = complex_test_from_test(t);
  if (ct->type==CONJUNCTIVE_TEST) {
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      if (test_has_id_in_os_tc (c->first, excluded_sym)) return TRUE;
    return FALSE;
  }
  return FALSE;
}

bool id_or_value_of_condition_list_is_in_os_tc (condition *conds,
                                        symbol *sym_excluded_from_value) {
  for ( ; conds!=NIL; conds=conds->next) {
    switch (conds->type) {
    case POSITIVE_CONDITION:
    case NEGATIVE_CONDITION:
      if (test_has_id_in_os_tc (conds->data.tests.id_test, NIL))
        return TRUE;
      if (test_has_id_in_os_tc (conds->data.tests.value_test,
                                sym_excluded_from_value))
        return TRUE;
      break;
    case CONJUNCTIVE_NEGATION_CONDITION:
      if (id_or_value_of_condition_list_is_in_os_tc (conds->data.ncc.top,
                                              sym_excluded_from_value))
        return TRUE;
      break;
    }
  }
  return FALSE;
}

/* -----------------------------------------------------------------------
                    Run-Time O-Support Calculation

   This routine calculates o-support for each preference for the given
   instantiation, filling in pref->o_supported (TRUE or FALSE) on each one.

   The following predicates are used for support calculations.  In the
   following, "lhs has some elt. ..." means the lhs has some id or value
   at any nesting level.

     lhs_oa_support:
       (1) does lhs test (match_goal ^operator match_operator NO) ?
       (2) mark TC (match_operator) using TM;
           does lhs has some elt. in TC but != match_operator ?
       (3) mark TC (match_state) using TM;
           does lhs has some elt. in TC ?
     lhs_oc_support:
       (1) mark TC (match_state) using TM;
           does lhs has some elt. in TC but != match_state ?
     lhs_om_support:
       (1) does lhs tests (match_goal ^operator) ?
       (2) mark TC (match_state) using TM;
           does lhs has some elt. in TC but != match_state ?

     rhs_oa_support:
       mark TC (match_state) using TM+RHS;
       if pref.id is in TC, give support
     rhs_oc_support:
       mark TC (inst.rhsoperators) using TM+RHS;
       if pref.id is in TC, give support
     rhs_om_support:
       mark TC (inst.lhsoperators) using TM+RHS;
       if pref.id is in TC, give support

   BUGBUG the code does a check of whether the lhs tests the match state via
          looking just at id and value fields of top-level positive cond's.
          It doesn't look at the attr field, or at any negative or NCC's.
          I'm not sure whether this is right or not.  (It's a pretty
          obscure case, though.)
----------------------------------------------------------------------- */

void calculate_support_for_instantiation_preferences (instantiation *inst) {
  symbol *match_goal, *match_state, *match_operator;
  wme *match_state_wme, *match_operator_wme;
  bool lhs_tests_operator_installed;
  bool lhs_tests_operator_acceptable_or_installed;
  bool lhs_tests_match_state;
  bool lhs_is_known_to_test_something_off_match_state;
  bool lhs_is_known_to_test_something_off_match_operator;
  bool rhs_has_some_non_goal_preference;
  bool rhs_does_an_operator_creation;
  bool oc_support_possible;
  bool om_support_possible;
  bool oa_support_possible;
  preference *rhs, *pref;
  wme *w;
  condition *lhs, *c;
#ifdef DETAILED_TIMING_STATS
  struct timeval saved_start_tv;
#endif

#ifdef DETAILED_TIMING_STATS
  start_timer (&saved_start_tv);
#endif

  /* --- initialize by giving everything NO o_support --- */  
  for (pref=inst->preferences_generated; pref!=NIL; pref=pref->inst_next)
    pref->o_supported = FALSE;

  /* --- find the match goal, match state, and match operator --- */
  match_goal = inst->match_goal;
  if (!match_goal) goto o_support_done;  /* nothing gets o-support */

  match_state_wme = match_goal->id.state_slot->wmes;
  if (! match_state_wme) goto o_support_done; /* no state --> no o-support */
  match_state = match_state_wme->value;

  match_operator_wme = match_goal->id.operator_slot->wmes;
  if (match_operator_wme)
    match_operator = match_operator_wme->value;
  else
    match_operator = NIL;

  lhs = inst->top_of_instantiated_conditions;
  rhs = inst->preferences_generated;
  
  /* --- scan through rhs to look for various things --- */
  rhs_has_some_non_goal_preference = FALSE;
  rhs_does_an_operator_creation = FALSE;  

  for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
    if (! pref->id->id.isa_goal) rhs_has_some_non_goal_preference = TRUE;
    if ((pref->id==match_goal) &&
        (pref->attr==operator_symbol) &&
        ((pref->type==ACCEPTABLE_PREFERENCE_TYPE) ||
         (pref->type==REQUIRE_PREFERENCE_TYPE)) )
      rhs_does_an_operator_creation = TRUE;
  }

  /* --- if all rhs preferences are goal aug's, there's no o-support --- */
  if (! rhs_has_some_non_goal_preference) goto o_support_done;
  
  /* --- scan through lhs to look for various tests --- */
  lhs_tests_operator_acceptable_or_installed = FALSE;
  lhs_tests_operator_installed = FALSE;
  lhs_tests_match_state = FALSE;
  lhs_is_known_to_test_something_off_match_state = FALSE;
  lhs_is_known_to_test_something_off_match_operator = FALSE;

  for (c=lhs; c!=NIL; c=c->next) {
    if (c->type!=POSITIVE_CONDITION) continue;
    w = c->bt.wme;
    if (w->value==match_state) lhs_tests_match_state = TRUE;
    if (w->id==match_state)
      lhs_is_known_to_test_something_off_match_state = TRUE;
    if (w->id==match_operator)
      lhs_is_known_to_test_something_off_match_operator = TRUE;
    if (w==match_operator_wme) lhs_tests_operator_installed = TRUE;
    if ((w->id==match_goal)&&(w->attr==operator_symbol))
      lhs_tests_operator_acceptable_or_installed = TRUE;
  }

  /* --- calcluate lhs support flags --- */
  oa_support_possible = lhs_tests_operator_installed;
  oc_support_possible = rhs_does_an_operator_creation; 
  om_support_possible = lhs_tests_operator_acceptable_or_installed;

  if ((!oa_support_possible)&&(!oc_support_possible)&&(!om_support_possible))
    goto o_support_done;

  if (! lhs_is_known_to_test_something_off_match_state) {
    begin_os_tc (NIL);
    add_to_os_tc (match_state);
    if (! id_or_value_of_condition_list_is_in_os_tc (lhs, match_state)) {
      oc_support_possible = FALSE;
      om_support_possible = FALSE;
      if (! lhs_tests_match_state) oa_support_possible = FALSE;
    }
  }

  if (oa_support_possible) {
    if (! lhs_is_known_to_test_something_off_match_operator) {
      begin_os_tc (NIL);
      add_to_os_tc (match_operator);
      if (! id_or_value_of_condition_list_is_in_os_tc (lhs, match_operator))
        oa_support_possible = FALSE;
    }
  }

  /* --- look for rhs oa support --- */
  if (oa_support_possible) {
    begin_os_tc (rhs);
    add_to_os_tc (match_state);
    for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
      if (pref->id->id.tc_num==o_support_tc)
        pref->o_supported = TRUE;
    }
  }

  /* --- look for rhs oc support --- */
  if (oc_support_possible) {
    begin_os_tc (rhs);
    for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
      if ((pref->id==match_goal) &&
          (pref->attr==operator_symbol) &&
          ((pref->type==ACCEPTABLE_PREFERENCE_TYPE) ||
           (pref->type==REQUIRE_PREFERENCE_TYPE)) ) {
          add_to_os_tc (pref->value);
      }
    }
    for (pref=rhs; pref!=NIL; pref=pref->inst_next) {
      if (pref->id->id.tc_num==o_support_tc)
        pref->o_supported = TRUE;
    }
  }
  
  /* --- look for rhs om support --- */
  if (om_support_possible) {
    begin_os_tc (rhs);
    for (c=inst->top_of_instantiated_conditions; c!=NIL; c=c->next)
      if (c->type==POSITIVE_CONDITION) {
        w = c->bt.wme;
        if ((w->id==match_goal) && (w->attr==operator_symbol))
          add_to_os_tc (w->value);
      }
    for (pref=rhs; pref!=NIL; pref=pref->inst_next)
      if (pref->id->id.tc_num==o_support_tc)
        pref->o_supported = TRUE;
  }

  o_support_done:  {}
#ifdef DETAILED_TIMING_STATS
  stop_timer (&saved_start_tv, &o_support_cpu_time);
#endif
}

/* -----------------------------------------------------------------------

               Executing the RHS Actions of an Instantiation

   Execute_action() executes a given RHS action.  For MAKE_ACTION's, it
   returns the created preference structure, or NIL if an error occurs.
   For FUNCALL_ACTION's, it returns NIL.

   Instantiate_symbol() and instantiate_rhs_value() return the (symbol)
   instantiation of a symbol and rhs_value, respectively.  They return
   NIL if an error occurs.  These two routines take a new_id_level
   argument indicating what goal_stack_level a new id is to be created
   at, in case a gensym is needed for the instantiation of a variable.
   (BUGBUG I'm not sure this is really needed.)
----------------------------------------------------------------------- */

symbol *instantiate_symbol (symbol *sym, goal_stack_level new_id_level) {
  char new_id_letter;
  
  if (sym->common.symbol_type==VARIABLE_SYMBOL_TYPE) {
    if (sym->var.current_binding_value) {
      symbol_add_ref (sym->var.current_binding_value);
      return sym->var.current_binding_value;
    }
    new_id_letter = *(sym->var.name + 1);
    sym->var.current_binding_value = make_new_identifier (new_id_letter,
                                                          new_id_level);
    return sym->var.current_binding_value;
  }
  symbol_add_ref (sym);
  return sym;
}

symbol *instantiate_rhs_value (rhs_value rv, goal_stack_level new_id_level) {
  list *fl;
  list *arglist;
  cons *c, *prev_c, *arg_cons;
  rhs_function *rf;
  symbol *result;
  bool nil_arg_found;
  
  if (rhs_value_is_symbol(rv))
    return instantiate_symbol (rhs_value_to_symbol(rv), new_id_level);

  fl = rhs_value_to_funcall_list(rv);
  rf = fl->first;

  /* --- build up a list of the argument values --- */
  prev_c = NIL;
  nil_arg_found = FALSE;
  for (arg_cons=fl->rest; arg_cons!=NIL; arg_cons=arg_cons->rest) {
    allocate_cons (&c);
    c->first = instantiate_rhs_value (arg_cons->first, new_id_level);
    if (! c->first) nil_arg_found = TRUE;
    if (prev_c) prev_c->rest = c; else arglist = c;
    prev_c = c;
  }
  if (prev_c) prev_c->rest = NIL; else arglist = NIL;

  /* --- if all args were ok, call the function --- */
  if (!nil_arg_found)
    result = (*(rf->f))(arglist);
  else
    result = NIL;

  /* --- scan through arglist, dereference symbols and deallocate conses --- */
  for (c=arglist; c!=NIL; c=c->rest)
    if (c->first) symbol_remove_ref ((symbol *)(c->first));
  free_list (arglist);

  return result;
}

preference *execute_action (action *a) {
  symbol *id, *attr, *value, *referent;
  
  if (a->type==FUNCALL_ACTION) {
    value = instantiate_rhs_value (a->value, -1);
    if (value) symbol_remove_ref (value);
    return NIL;
  }

  id = instantiate_symbol (a->id, -1);
  if (id->common.symbol_type!=IDENTIFIER_SYMBOL_TYPE) {
    print_with_symbols ("Error: RHS makes a preference for %y (not an identifier)\n", id);
    symbol_remove_ref (id);
    return NIL;
  }
  
  attr = instantiate_symbol (a->attr, id->id.level);

  value = instantiate_rhs_value (a->value, id->id.level);
  if (!value) {
    symbol_remove_ref (id);
    symbol_remove_ref (attr);
    return NIL;
  }

  if (preference_is_binary(a->preference_type)) {
    referent = instantiate_rhs_value (a->referent, id->id.level);
    if (!referent) {
      symbol_remove_ref (id);
      symbol_remove_ref (attr);
      symbol_remove_ref (value);
      return NIL;
    }
  } else {
    referent = NIL;
  }

  return make_preference (a->preference_type, id, attr, value, referent);
}

/* -----------------------------------------------------------------------
                    Fill In New Instantiation Stuff

   This routine fills in a newly created instantiation structure with
   various information.   At input, the instantiation should have:
     - preferences_generated filled in; 
     - instantiated conditions filled in;
     - top-level positive conditions should have bt.wme, bt.level, and
       bt.trace filled in, but bt.wme and bt.trace shouldn't have their
       reference counts incremented yet.

   This routine does the following:
     - increments reference count on production;
     - fills in match_goal and match_goal_level;
     - for each top-level positive cond:
         replaces bt.trace with the preference for the correct level,
         updates reference counts on bt.pref and bt.wmetraces and wmes
     - for each preference_generated, adds that pref to the list of all
       pref's for the match goal
     - fills in backtrace_number;   
     - if "need_to_do_support_calculations" is TRUE, calculates o-support
       for preferences_generated;
----------------------------------------------------------------------- */

void fill_in_new_instantiation_stuff (instantiation *inst,
                                      bool need_to_do_support_calculations) {
  condition *cond;
  preference *p;
  goal_stack_level level;

  production_add_ref (inst->prod);
  
  find_match_goal (inst);

  level = inst->match_goal_level;

  /* Note: since we'll never backtrace through instantiations at the top
     level, it might make sense to not increment the reference counts
     on the wmes and preferences here if the instantiation is at the top
     level.  As it stands now, we could gradually accumulate garbage at
     the top level if we have a never-ending sequence of production
     firings at the top level that chain on each other's results.  (E.g.,
     incrementing a counter on every decision cycle.)  I'm leaving it this
     way for now, because if we go to S-Support, we'll (I think) need to
     save these around (maybe??). */

  for (cond=inst->top_of_instantiated_conditions; cond!=NIL; cond=cond->next)
    if (cond->type==POSITIVE_CONDITION) {
      wme_add_ref (cond->bt.wme);
      /* --- if trace is for a lower level, find one for this level --- */
      if (cond->bt.trace)
        if (cond->bt.trace->inst->match_goal_level > level)
          cond->bt.trace = find_clone_for_level (cond->bt.trace, level);
      if (cond->bt.trace) preference_add_ref (cond->bt.trace);
    }

  if (inst->match_goal) {
    for (p=inst->preferences_generated; p!=NIL; p=p->inst_next) {
      insert_at_head_of_dll (inst->match_goal->id.preferences_from_goal, p,
                             all_of_goal_next, all_of_goal_prev);
      p->on_goal_list = TRUE;
    }
  }
  inst->backtrace_number = 0;

  if (need_to_do_support_calculations)
    calculate_support_for_instantiation_preferences (inst);
}

/* =======================================================================

                          Main Firer Routines

   Init_firer() should be called at startup time.  Do_preference_phase()
   is called from the top level to run the whole preference phase.

   Preference phase follows this sequence:

   (1) Productions are fired for new matches.  As productions are fired,
   their instantiations are stored on the list newly_created_instantiations,
   linked via the "next" fields in the instantiation structure.  No
   preferences are actually asserted yet.
   
   (2) Instantiations are retracted; their preferences are retracted.

   (3) Preferences (except o-rejects) from newly_created_instantiations
   are asserted, and these instantiations are removed from the 
   newly_created_instantiations list and moved over to the per-production
   lists of instantiations of that production.

   (4) Finally, o-rejects are processed.
======================================================================= */

memory_pool instantiation_pool;

instantiation *newly_created_instantiations;

void init_firer (void) {
  init_memory_pool (&instantiation_pool, sizeof(instantiation),
                    "instantiation");
}

/* --- Macro returning TRUE iff we're supposed to trace firings for the
   given instantiation, which should have the "prod" field filled in. --- */

#define trace_firings_of_inst(inst) \
  ((inst)->prod && \
   (sysparams[TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM+(inst)->prod->type] || \
    ((inst)->prod->trace_firings)))

/* -----------------------------------------------------------------------
                         Create Instantiation

   This builds the instantiation for a new match, and adds it to
   newly_created_instantiations.  It also calls chunk_instantiation() to
   do any necessary chunk or justification building.
----------------------------------------------------------------------- */

void chunk_instantiation (instantiation *inst, bool variablize_it);

void create_instantiation (production *prod,
                           struct token_struct *tok,
                           wme *w) {
  instantiation *inst;
  condition *cond;
  preference *pref;
  action *a;
  cons *c;
  bool need_to_do_support_calculations;
  bool trace_it;
  
  allocate_with_pool (&instantiation_pool, &inst);
  inst->next = newly_created_instantiations;
  newly_created_instantiations = inst;
  inst->prod = prod;
  inst->rete_token = tok;
  inst->rete_wme = w;
  inst->okay_to_variablize = TRUE;
  inst->in_ms = TRUE;

  production_being_fired = inst->prod;
  prod->firing_count++;
  production_firing_count++;
  
  /* --- build the instantiated conditions, and bind LHS variables --- */
  p_node_to_conditions_and_nots (prod->p_node, tok, w,
                                 &(inst->top_of_instantiated_conditions),
                                 &(inst->bottom_of_instantiated_conditions),
                                 &(inst->nots));

  /* --- record the level of each of the wmes that was positively tested --- */
  for (cond=inst->top_of_instantiated_conditions; cond!=NIL; cond=cond->next) {
    if (cond->type==POSITIVE_CONDITION) {
      cond->bt.level = cond->bt.wme->id->id.level;
      cond->bt.trace = cond->bt.wme->preference;
    }
  }

  /* --- mark RHS unbound variables as "unbound" by setting binding=NIL --- */
  for (c=prod->rhs_unbound_variables; c!=NIL; c=c->rest)
    ((symbol *)(c->first))->var.current_binding_value = NIL;

  /* --- print trace info --- */
  trace_it = trace_firings_of_inst (inst);
  if (trace_it) {
    print ("\nFiring ");
    print_instantiation_with_wmes
      (inst, sysparams[TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM]);
  }

  /* --- execute the RHS actions, collect the results --- */
  inst->preferences_generated = NIL;
  need_to_do_support_calculations = FALSE;
  for (a=prod->action_list; a!=NIL; a=a->next) {
    pref = execute_action (a);
    if (pref) {
      pref->inst = inst;
      insert_at_head_of_dll (inst->preferences_generated, pref,
                             inst_next, inst_prev);
      if (inst->prod->declared_support==DECLARED_O_SUPPORT)
        pref->o_supported = TRUE;
      else if (inst->prod->declared_support==DECLARED_NO_O_SUPPORT)
        pref->o_supported = FALSE;
      else if (a->support==O_SUPPORT) pref->o_supported = TRUE;
      else if (a->support==NO_O_SUPPORT) pref->o_supported = FALSE;
      else need_to_do_support_calculations = TRUE;
    }
  }

  /* --- fill in lots of other stuff --- */
  fill_in_new_instantiation_stuff (inst, need_to_do_support_calculations);

  /* --- print trace info: printing preferences --- */
  /* Note: can't move this up, since fill_in_new_instantiation_stuff gives
     the o-support info for the preferences we're about to print */
  if (trace_it && sysparams[TRACE_FIRINGS_PREFERENCES_SYSPARAM]) {
    print (" -->");
    for (pref=inst->preferences_generated; pref!=NIL; pref=pref->inst_next) {
      print (" ");
      print_preference (pref);
    }
  }

  production_being_fired = NIL;

  /* --- build chunks/justifications if necessary --- */
  chunk_instantiation (inst, sysparams[LEARNING_ON_SYSPARAM]);

  /* --- call hook function --- */
  firing_hook (inst);
}

/* -----------------------------------------------------------------------
                        Deallocate Instantiation

   This deallocates the given instantiation.  This should only be invoked
   via the possibly_deallocate_instantiation() macro.
----------------------------------------------------------------------- */

void deallocate_instantiation (instantiation *inst) {
  condition *cond;

#ifdef DEBUG_INSTANTIATIONS
  if (inst->prod)
    print_with_symbols ("\nDeallocate instantiation of %y",inst->prod->name);
#endif

  for (cond=inst->top_of_instantiated_conditions; cond!=NIL; cond=cond->next)
    if (cond->type==POSITIVE_CONDITION) {
      wme_remove_ref (cond->bt.wme);
      if (cond->bt.trace) preference_remove_ref (cond->bt.trace);
    }

  deallocate_condition_list (inst->top_of_instantiated_conditions);
  deallocate_list_of_nots (inst->nots);
  if (inst->prod) production_remove_ref (inst->prod);
  free_with_pool (&instantiation_pool, inst);
}

/* -----------------------------------------------------------------------
                         Retract Instantiation

   This retracts the given instantiation.
----------------------------------------------------------------------- */

void retract_instantiation (instantiation *inst) {
  preference *pref, *next;
  bool retracted_a_preference;
  bool trace_it;

  /* --- call hook function --- */
  retraction_hook (inst);
  
  retracted_a_preference = FALSE;
  
  trace_it = trace_firings_of_inst (inst);

  /* --- retract any preferences that are in TM and aren't o-supported --- */
  pref = inst->preferences_generated;
  while (pref!=NIL) {
    next = pref->inst_next;
    if (pref->in_tm && (! pref->o_supported)) {

      if (trace_it) {
        if (!retracted_a_preference) {
          print ("\nRetracting ");
          print_instantiation_with_wmes
            (inst, sysparams[TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM]);
          if (sysparams[TRACE_FIRINGS_PREFERENCES_SYSPARAM]) print (" -->");
        }
        if (sysparams[TRACE_FIRINGS_PREFERENCES_SYSPARAM]) {
          print (" ");
          print_preference (pref);
        }
      }

      remove_preference_from_tm (pref);
      retracted_a_preference = TRUE;
    }
    pref = next;
  }

  /* --- remove inst from list of instantiations of this production --- */
  remove_from_dll (inst->prod->instantiations, inst, next, prev);

  /* --- if retracting a justification, excise it --- */
  /*
   * if the reference_count on the production is 1 (or less) then the
   * only thing supporting this justification is the instantiation, hence
   * it has already been excised, and doing it again is wrong.
   */
  if (inst->prod->type==JUSTIFICATION_PRODUCTION_TYPE &&
      inst->prod->reference_count > 1)
    excise_production (inst->prod, FALSE);
  
  /* --- mark as no longer in MS, and possibly deallocate  --- */
  inst->in_ms = FALSE;
  possibly_deallocate_instantiation (inst);
}

/* -----------------------------------------------------------------------
                         Assert New Preferences

   This routine scans through newly_created_instantiations, asserting
   each preference generated except for o-rejects.  It also removes
   each instantiation from newly_created_instantiations, linking each
   onto the list of instantiations for that particular production.
   O-rejects are bufferred and handled after everything else.

   Note that some instantiations on newly_created_instantiations are not
   in the match set--for the initial instantiations of chunks/justifications,
   if they don't match WM, we have to assert the o-supported preferences
   and throw away the rest.
----------------------------------------------------------------------- */

void assert_new_preferences (void) {
  instantiation *inst, *next_inst;
  preference *pref, *next_pref;
  preference *o_rejects;

  o_rejects = NIL;  

  for (inst=newly_created_instantiations; inst!=NIL; inst=next_inst) {
    next_inst = inst->next;
    if (inst->in_ms)
      insert_at_head_of_dll (inst->prod->instantiations, inst, next, prev);
    for (pref=inst->preferences_generated; pref!=NIL; pref=next_pref) {
      next_pref = pref->inst_next;
      if ((pref->type==REJECT_PREFERENCE_TYPE)&&(pref->o_supported)) {
        /* --- o-reject: just put it in the buffer for later --- */
        pref->next = o_rejects;
        o_rejects = pref;
      } else if (inst->in_ms || pref->o_supported) {
        /* --- normal case --- */
        add_preference_to_tm (pref);
      } else {
        /* --- inst. is refracted chunk, and pref. is not o-supported:
           remove the preference --- */
        /* --- first splice it out of the clones list--otherwise we might
           accidentally deallocate some clone that happens to have refcount==0
           just because it hasn't been asserted yet --- */
        if (pref->next_clone) pref->next_clone->prev_clone = pref->prev_clone;
        if (pref->prev_clone) pref->prev_clone->next_clone = pref->next_clone;
        pref->next_clone = pref->prev_clone = NIL;
        /* --- now add then remove ref--this should result in deallocation */
        preference_add_ref (pref);
        preference_remove_ref (pref);
      }
    }
  }

  if (o_rejects) process_o_rejects_and_deallocate_them (o_rejects);
}

/* -----------------------------------------------------------------------
                          Do Preference Phase

   This routine is called from the top level to run the preference phase.
----------------------------------------------------------------------- */

void do_preference_phase (void) {
  production *prod;
  struct token_struct *tok;
  wme *w;
  instantiation *inst;
#ifdef DETAILED_TIMING_STATS
  struct timeval saved_start_tv;
#endif

#ifdef DETAILED_TIMING_STATS
  start_timer (&saved_start_tv);
#endif

  if (sysparams[TRACE_PHASES_SYSPARAM]) print ("\n--- Preference Phase ---");

  newly_created_instantiations = NIL;
  while (get_next_assertion (&prod, &tok, &w))
    create_instantiation (prod, tok, w);

#ifdef DETAILED_TIMING_STATS
  stop_timer (&saved_start_tv, &create_instantiations_cpu_time);
#endif

  assert_new_preferences ();
  
  while (get_next_retraction (&inst))
    retract_instantiation (inst);

#ifdef DETAILED_TIMING_STATS
  stop_timer (&saved_start_tv, &preference_phase_cpu_time);
#endif
}

/* ====================================================================

                          Chunking Routines

==================================================================== */

extern memory_pool action_pool, complex_test_pool;

/* =====================================================================

                           Results Calculation

   Get_results_for_instantiation() finds and returns the result preferences
   for a given instantiation.  This is the main routine here.

   The results are accumulated in the list "results," linked via the
   "next_result" field of the preference structures.  (BUGBUG: to save
   space, just use conses for this.)

   Add_pref_to_results() adds a preference to the results.
   Add_results_for_id() adds any preferences for the given identifier.
   Identifiers are marked with results_tc_number as they are added.
===================================================================== */

tc_number results_tc_number;  /* used for marking id's included in results */
preference *extra_result_prefs_from_instantiation; /* RHS of instantiation */
goal_stack_level results_match_goal_level; /* match goal level of inst. */
preference *results;                       /* accumulated result pref's */

void add_results_for_id (symbol *id);

#define add_results_if_needed(sym) \
  { if ((sym)->common.symbol_type==IDENTIFIER_SYMBOL_TYPE) \
      if ( ((sym)->id.level >= results_match_goal_level) && \
           ((sym)->id.tc_num != results_tc_number) ) \
        add_results_for_id(sym); }
                                    
void add_pref_to_results (preference *pref) {
  preference *p;

  /* --- if an equivalent pref is already a result, don't add this one --- */
  for (p=results; p!=NIL; p=p->next_result) {
    if (p->id!=pref->id) continue;
    if (p->attr!=pref->attr) continue;
    if (p->value!=pref->value) continue;
    if (p->type!=pref->type) continue;
    if (preference_is_unary(pref->type)) return;
    if (p->referent!=pref->referent) continue;
    return;
  }

  /* --- if pref isn't at the right level, find a clone that is --- */
  if (pref->inst->match_goal_level != results_match_goal_level) {
    for (p=pref->next_clone; p!=NIL; p=p->next_clone)
      if (p->inst->match_goal_level == results_match_goal_level) break;
    if (!p)
      for (p=pref->prev_clone; p!=NIL; p=p->prev_clone)
        if (p->inst->match_goal_level == results_match_goal_level) break;
    if (!p) return;  /* if can't find one, it isn't a result */
    pref = p;
  }

  /* --- add this preference to the result list --- */
  pref->next_result = results; 
  results = pref;

  /* --- follow transitive closuse through value, referent links --- */
  add_results_if_needed (pref->value);
  if (preference_is_binary(pref->type))
    add_results_if_needed (pref->referent);
}

void add_results_for_id (symbol *id) {
  slot *s;
  preference *pref;
  wme *w;

  id->id.tc_num = results_tc_number;

  /* --- scan through all preferences and wmes for all slots for this id --- */
  for (w=id->id.input_wmes; w!=NIL; w=w->next)
    add_results_if_needed (w->value);
  for (s=id->id.slots; s!=NIL; s=s->next) {
    for (pref=s->all_preferences; pref!=NIL; pref=pref->all_of_slot_next)
      add_pref_to_results(pref);
    for (w=s->wmes; w!=NIL; w=w->next)
      add_results_if_needed (w->value);
  } /* end of for slots loop */
  /* --- now scan through extra prefs and look for any with this id --- */
  for (pref=extra_result_prefs_from_instantiation; pref!=NIL;
       pref=pref->inst_next) {
    if (pref->id==id) add_pref_to_results(pref);
  }
}

preference *get_results_for_instantiation (instantiation *inst) {
  preference *pref;

  results = NIL;
  results_match_goal_level = inst->match_goal_level;
  results_tc_number = get_new_tc_number();
  extra_result_prefs_from_instantiation = inst->preferences_generated;
  for (pref=inst->preferences_generated; pref!=NIL; pref=pref->inst_next)
    if ( (pref->id->id.level < results_match_goal_level) &&
         (pref->id->id.tc_num != results_tc_number) ) {
      add_pref_to_results(pref);
    }
  return results;
}

/* =====================================================================

                  Variablizing Conditions and Results

   Variablizing of conditions is done by walking over a condition list
   and destructively modifying it, replacing tests of identifiers with
   tests of tests of variables.  The identifier-to-variable mapping is
   built as we go along:  identifiers that have already been assigned
   a variablization are marked with id.tc_num==variablization_tc, and
   id.variablization points to the corresponding variable.

   Variablizing of results can't be done destructively because we need
   to convert the results--preferences--into actions.  This is done
   by copy_and_variablize_result_list(), which takes the result preferences
   and returns an action list.

   The global variable "variablize_this_chunk" indicates whether to
   variablize at all.  This flag is set to TRUE or FALSE before and during
   backtracing.  FALSE means the new production will become a justification;
   TRUE means it will be a chunk.
===================================================================== */

bool variablize_this_chunk;

tc_number variablization_tc;

void variablize_symbol (symbol **sym) {
  char prefix[2];
  symbol *var;
  
  if ((*sym)->common.symbol_type!=IDENTIFIER_SYMBOL_TYPE) return;
  if (! variablize_this_chunk) return;
  
  if ((*sym)->id.tc_num == variablization_tc) {
    /* --- it's already been variablized, so use the existing variable --- */
    var = (*sym)->id.variablization;
    symbol_remove_ref (*sym);
    *sym = var;
    symbol_add_ref (var);
    return;
  }

  /* --- need to create a new variable --- */
  (*sym)->id.tc_num = variablization_tc;
  prefix[0] = tolower((*sym)->id.name_letter);
  prefix[1] = 0;
  var = generate_new_variable (prefix);
  (*sym)->id.variablization = var;
  symbol_remove_ref (*sym);
  *sym = var;
}

void variablize_test (test *t) {
  cons *c;
  complex_test *ct;

  if (test_is_blank_test(*t)) return;
  if (test_is_blank_or_equality_test(*t)) {
    variablize_symbol ((symbol **) t);
    /* Warning: this relies on the representation of tests */
    return;
  }

  ct = complex_test_from_test(*t);
  
  switch (ct->type) {
  case GOAL_ID_TEST:
  case IMPASSE_ID_TEST:
  case DISJUNCTION_TEST:
    return;
  case CONJUNCTIVE_TEST:
    for (c=ct->data.conjunct_list; c!=NIL; c=c->rest)
      variablize_test ((test *)(&(c->first)));
    return;
  default:  /* relational tests other than equality */
    variablize_symbol (&(ct->data.referent));
    return;
  }
}

void variablize_condition_list (condition *cond) {
  for ( ; cond!=NIL; cond=cond->next) {
    switch (cond->type) {
    case POSITIVE_CONDITION:
    case NEGATIVE_CONDITION:
      variablize_test (&(cond->data.tests.id_test));
      variablize_test (&(cond->data.tests.attr_test));
      variablize_test (&(cond->data.tests.value_test));
      break;
    case CONJUNCTIVE_NEGATION_CONDITION:
      variablize_condition_list (cond->data.ncc.top);
      break;
    }
  }
}

action *copy_and_variablize_result_list (preference *pref) {
  action *a;
  symbol *temp;
  
  if (!pref) return NIL;
  allocate_with_pool (&action_pool, &a);
  a->type = MAKE_ACTION;

  a->id = pref->id;
  symbol_add_ref (pref->id);
  variablize_symbol (&(a->id));

  a->attr = pref->attr;
  symbol_add_ref (pref->attr);
  variablize_symbol (&(a->attr));

  temp = pref->value;
  symbol_add_ref (temp);
  variablize_symbol (&temp);
  a->value = symbol_to_rhs_value (temp);

  a->preference_type = pref->type;

  if (preference_is_binary(pref->type)) {
    temp = pref->referent;
    symbol_add_ref (temp);
    variablize_symbol (&temp);
    a->referent = symbol_to_rhs_value (temp);
  }
  
  a->next = copy_and_variablize_result_list (pref->next_result);
  return a;  
}

/* ====================================================================

     Chunk Conditions, and Chunk Conditions Set Manipulation Routines

   These structures have two uses.  First, for every ground condition,
   one of these structures maintains certain information about it--
   pointers to the original (instantiation's) condition, the chunks's
   instantiation's condition, and the variablized condition, etc.

   Second, for negated conditions, these structures are entered into
   a hash table with keys hash_condition(this_cond).  This hash table
   is used so we can add a new negated condition to the set of negated
   potentials quickly--we don't want to add a duplicate of a negated
   condition that's already there, and the hash table lets us quickly
   determine whether a duplicate is already there.

   I used one type of structure for both of these uses, (1) for simplicity
   and (2) to avoid having to do a second allocation when we move
   negated conditions over to the ground set.
==================================================================== */

typedef struct chunk_cond_struct {
  condition *cond;                /* points to the original condition */

  condition *instantiated_cond;   /* points to cond in chunk instantiation */
  condition *variablized_cond;    /* points to cond in the actual chunk */
  condition *saved_prev_pointer_of_variablized_cond; /* don't ask */

  /* dll of all cond's in a set (i.e., a chunk_cond_set, or the grounds) */
  struct chunk_cond_struct *next, *prev;

  /* dll of cond's in this particular hash bucket for this set */
  struct chunk_cond_struct *next_in_bucket, *prev_in_bucket; 

  unsigned long hash_value;             /* equals hash_condition(cond) */
  unsigned long compressed_hash_value;  /* above, compressed to a few bits */
} chunk_cond;

memory_pool chunk_cond_pool;

/* --------------------------------------------------------------------
                      Chunk Cond Set Routines

   Init_chunk_cond_set() initializes a given chunk_cond_set to be empty.
   
   Make_chunk_cond_for_condition() takes a condition and returns a
   chunk_cond for it, for use in a chunk_cond_set.  This is used only
   for the negated conditions, not grounds.

   Add_to_chunk_cond_set() adds a given chunk_cond to a given chunk_cond_set
   and returns TRUE if the condition isn't already in the set.  If the 
   condition is already in the set, the routine deallocates the given
   chunk_cond and returns FALSE.

   Remove_from_chunk_cond_set() removes a given chunk_cond from a given
   chunk_cond_set, but doesn't deallocate it.
-------------------------------------------------------------------- */

#define CHUNK_COND_HASH_TABLE_SIZE 1024
#define LOG_2_CHUNK_COND_HASH_TABLE_SIZE 10

typedef struct chunk_cond_set_struct {
  chunk_cond *all;       /* header for dll of all chunk_cond's in the set */
  chunk_cond *table[CHUNK_COND_HASH_TABLE_SIZE];  /* hash table buckets */
} chunk_cond_set;

chunk_cond_set negated_set;  /* set of all negated conditions we encounter
                                during backtracing--these are all potentials
                                and (some of them) are added to the grounds
                                in one pass at the end of the backtracing */

void init_chunk_cond_set (chunk_cond_set *set) {
  int i;
  
  set->all = NIL;
  for (i=0; i<CHUNK_COND_HASH_TABLE_SIZE; i++) set->table[i] = NIL;
}

chunk_cond *make_chunk_cond_for_condition (condition *cond) {
  chunk_cond *cc;
  unsigned long remainder, hv;
  
  allocate_with_pool (&chunk_cond_pool, &cc);
  cc->cond = cond;
  cc->hash_value = hash_condition (cond);
  remainder = cc->hash_value;
  hv = 0;
  while (remainder) {
    hv ^= (remainder &
           masks_for_n_low_order_bits[LOG_2_CHUNK_COND_HASH_TABLE_SIZE]);
    remainder = remainder >> LOG_2_CHUNK_COND_HASH_TABLE_SIZE;
  }
  cc->compressed_hash_value = hv;
  return cc;
}

bool add_to_chunk_cond_set (chunk_cond_set *set, chunk_cond *new_cc) {
  chunk_cond *old;
  
  for (old=set->table[new_cc->compressed_hash_value]; old!=NIL;
       old=old->next_in_bucket)
    if (old->hash_value==new_cc->hash_value)
      if (conditions_are_equal (old->cond, new_cc->cond))
        break;
  if (old) {
    /* --- the new condition was already in the set; so don't add it --- */
    free_with_pool (&chunk_cond_pool, new_cc);
    return FALSE;
  }
  /* --- add new_cc to the table --- */
  insert_at_head_of_dll (set->all, new_cc, next, prev);
  insert_at_head_of_dll (set->table[new_cc->compressed_hash_value], new_cc,
                         next_in_bucket, prev_in_bucket);
  return TRUE;
}

void remove_from_chunk_cond_set (chunk_cond_set *set, chunk_cond *cc) {
  remove_from_dll (set->all, cc, next, prev);
  remove_from_dll (set->table[cc->compressed_hash_value],
                   cc, next_in_bucket, prev_in_bucket);
}

/* ====================================================================

                            Backtracing

   Four sets of conditions are maintained during backtracing:  locals,
   grounds, positive potentials, and negateds.  Negateds are really
   potentials, but we keep them separately throughout backtracing, and
   ground them at the very end.  Note that this means during backtracing,
   the grounds, positive potentials, and locals are all instantiated
   top-level positive conditions, so they all have a bt.wme on them.

   In order to avoid backtracing through the same instantiation twice,
   we mark each instantiation as we BT it, by setting
   inst->backtrace_number = backtrace_number (this is a global variable
   which gets incremented each time we build a chunk).

   Locals, grounds, and positive potentials are kept on lists (see the
   global variables below).  These are consed lists of the conditions
   (that is, the original instantiated conditions).  Furthermore,
   we mark the bt.wme's on each condition so we can quickly determine
   whether a given condition is already in a given set.  The "grounds_tc",
   "potentials_tc", "locals_tc", and "chunker_bt_pref" fields on wme's
   are used for this.  Wmes are marked as "in the grounds" by setting
   wme->grounds_tc = grounds_tc.  For potentials and locals, we also
   must set wme->chunker_bt_pref:  if the same wme was tested by two
   instantiations created at different times--times at which the wme
   was supported by two different preferences--then we really need to
   BT through *both* preferences.  Marking the wmes with just "locals_tc"
   or "potentials_tc" alone would prevent the second preference from
   being BT'd.

   The add_to_grounds(), add_to_potentials(), and add_to_locals()
   macros below are used to add conditions to these sets.  The negated
   conditions are maintained in the chunk_cond_set "negated_set."

   As we backtrace, each instantiation that has some Nots is added to
   the list instantiations_with_nots.  We have to go back afterwards
   and figure out which Nots are between identifiers that ended up in
   the grounds.
==================================================================== */

tc_number backtrace_number = 0;

tc_number grounds_tc = 0, potentials_tc = 0, locals_tc = 0;
list *locals, *grounds, *positive_potentials;

list *instantiations_with_nots;

#define add_to_grounds(cond) { \
  if ((cond)->bt.wme->grounds_tc != grounds_tc) { \
    (cond)->bt.wme->grounds_tc = grounds_tc; \
    push ((cond), grounds); } }

#define add_to_potentials(cond) { \
  if ((cond)->bt.wme->potentials_tc != potentials_tc) { \
    (cond)->bt.wme->potentials_tc = potentials_tc; \
    (cond)->bt.wme->chunker_bt_pref = (cond)->bt.trace; \
    push ((cond), positive_potentials); \
  } else if ((cond)->bt.wme->chunker_bt_pref != (cond)->bt.trace) { \
    push ((cond), positive_potentials); } }

#define add_to_locals(cond) { \
  if ((cond)->bt.wme->locals_tc != locals_tc) { \
    (cond)->bt.wme->locals_tc = locals_tc; \
    (cond)->bt.wme->chunker_bt_pref = (cond)->bt.trace; \
    push ((cond), locals); \
  } else if ((cond)->bt.wme->chunker_bt_pref != (cond)->bt.trace) { \
    push ((cond), locals); } }

/* -------------------------------------------------------------------
                     Backtrace Through Instantiation

   This routine BT's through a given instantiation.  The general method
   is as follows:

     1. If we've already BT'd this instantiation, then skip it.
     2. Mark the TC (in the instantiated conditions) of all higher goal
        ids tested in top-level positive conditions
     3. Scan through the instantiated conditions; add each one to the
        appropriate set (locals, positive_potentials, grounds, negated_set).
     4. If the instantiation has any Nots, add this instantiation to
        the list of instantiations_with_nots.
------------------------------------------------------------------- */

void print_consed_list_of_conditions (list *c) {
  for ( ; c!=NIL; c=c->rest) {
    if (get_printer_output_column() >= COLUMNS_PER_LINE-20) print ("\n      ");
    print_condition (c->first);
  }
}

void print_consed_list_of_condition_wmes (list *c) {
  for ( ; c!=NIL; c=c->rest) {
    if (get_printer_output_column() >= COLUMNS_PER_LINE-20) print ("\n      ");
    print_wme (((condition *)(c->first))->bt.wme);
  }
}

void backtrace_through_instantiation (instantiation *inst,
                            goal_stack_level grounds_level) {
  tc_number tc;   /* use this to mark ids in the ground set */
  tc_number tc2;  /* use this to mark other ids we see */
  condition *c;
  list *grounds_to_print, *pots_to_print, *locals_to_print, *negateds_to_print;
  bool need_another_pass;

  if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
    print ("... BT through instantiation of ");
    if (inst->prod) print_with_symbols ("%y\n",inst->prod->name);
    else print_string ("[dummy production]\n");
  }

  /* --- if the instantiation has already been BT'd, don't repeat it --- */
  if (inst->backtrace_number == backtrace_number) {
    if (sysparams[TRACE_BACKTRACING_SYSPARAM])
      print_string ("(We already backtraced through this instantiation.)\n");
    return;
  }
  inst->backtrace_number = backtrace_number;

  /* --- check okay_to_variablize flag --- */
  if (! inst->okay_to_variablize) variablize_this_chunk = FALSE;

  /* --- mark transitive closure of each higher goal id that was tested in
     the id field of a top-level positive condition --- */
  tc = get_new_tc_number ();
  tc2 = get_new_tc_number ();
  need_another_pass = FALSE;

  for (c=inst->top_of_instantiated_conditions; c!=NIL; c=c->next) {
    symbol *id, *value;
    
    if (c->type!=POSITIVE_CONDITION) continue;
    id = referent_of_equality_test (c->data.tests.id_test);
    if (id->id.tc_num == tc) {
      /* --- id is already in the TC, so add in the value --- */
      value = referent_of_equality_test (c->data.tests.value_test);
      if (value->common.symbol_type==IDENTIFIER_SYMBOL_TYPE) {
        /* --- if we already saw it before, we're going to have to go back
           and make another pass to get the complete TC --- */
        if (value->id.tc_num == tc2) need_another_pass = TRUE;
        value->id.tc_num = tc;
      }
    } else if ((id->id.isa_goal) && (c->bt.level <= grounds_level)) {
      /* --- id is a higher goal id that was tested: so add id to the TC --- */
      id->id.tc_num = tc;
      value = referent_of_equality_test (c->data.tests.value_test);
      if (value->common.symbol_type==IDENTIFIER_SYMBOL_TYPE) {
        /* --- if we already saw it before, we're going to have to go back
           and make another pass to get the complete TC --- */
        if (value->id.tc_num == tc2) need_another_pass = TRUE;
        value->id.tc_num = tc;
      }
    } else {
      /* --- as far as we know so far, id shouldn't be in the tc: so mark it
         with number "tc2" to indicate that it's been seen already --- */
      id->id.tc_num = tc2;
    }
  }
  
  /* --- if necessary, make more passes to get the complete TC through the
     top-level positive conditions (recall that they're all super-simple
     wme tests--all three fields are equality tests --- */
  while (need_another_pass) {
    symbol *value;
    
    need_another_pass = FALSE;
    for (c=inst->top_of_instantiated_conditions; c!=NIL; c=c->next) {
      if (c->type!=POSITIVE_CONDITION)
        continue;
      if (referent_of_equality_test(c->data.tests.id_test)->id.tc_num != tc)
        continue;
      value = referent_of_equality_test(c->data.tests.value_test);
      if (value->common.symbol_type==IDENTIFIER_SYMBOL_TYPE)
        if (value->id.tc_num != tc) {
          value->id.tc_num = tc;
          need_another_pass = TRUE;
        }
    } /* end of for loop */
  } /* end of while loop */

  /* --- scan through conditions, collect grounds, potentials, & locals --- */
  grounds_to_print = NIL;
  pots_to_print = NIL;
  locals_to_print = NIL;
  negateds_to_print = NIL;
  
  for (c=inst->top_of_instantiated_conditions; c!=NIL; c=c->next) {
    if (c->type==POSITIVE_CONDITION) {
      /* --- positive cond's are grounds, potentials, or locals --- */
      if (referent_of_equality_test(c->data.tests.id_test)->id.tc_num == tc) {
        add_to_grounds (c);
        if (sysparams[TRACE_BACKTRACING_SYSPARAM]) push (c, grounds_to_print);
      } else if (c->bt.level <= grounds_level) {
        add_to_potentials (c);
        if (sysparams[TRACE_BACKTRACING_SYSPARAM]) push (c, pots_to_print);
      } else {
        add_to_locals (c);
        if (sysparams[TRACE_BACKTRACING_SYSPARAM]) push (c, locals_to_print);
      }
    } else {
      /* --- negative or nc cond's are either grounds or potentials --- */
      add_to_chunk_cond_set (&negated_set, make_chunk_cond_for_condition(c));
      if (sysparams[TRACE_BACKTRACING_SYSPARAM]) push (c, negateds_to_print);
    }
  } /* end of for loop */

  /* --- add new nots to the not set --- */
  if (inst->nots) push (inst, instantiations_with_nots);

  /* --- if tracing BT, print the resulting conditions, etc. --- */
  if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
    not *not1;

    print_string ("  -->Grounds:  ");
    print_consed_list_of_condition_wmes (grounds_to_print);
    print_string ("\n  -->Potentials:  ");
    print_consed_list_of_condition_wmes (pots_to_print);
    print_string ("\n  -->Locals:  ");
    print_consed_list_of_condition_wmes (locals_to_print);
    print_string ("\n  -->Negated:  ");
    print_consed_list_of_conditions (negateds_to_print);
    free_list (grounds_to_print);
    free_list (pots_to_print);
    free_list (locals_to_print);
    free_list (negateds_to_print);
    print_string ("\n  -->Nots:\n");
    for (not1=inst->nots; not1!=NIL; not1=not1->next)
      print_with_symbols ("    %y <> %y\n", not1->s1, not1->s2);
  }
}

/* ---------------------------------------------------------------
                             Trace Locals

   This routine backtraces through locals, and keeps doing so until
   there are no more locals to BT.
--------------------------------------------------------------- */

void trace_locals (goal_stack_level grounds_level) {
  cons *c;
  condition *cond;
  preference *bt_pref;

  if (sysparams[TRACE_BACKTRACING_SYSPARAM])
    print_string ("\n\n*** Tracing Locals ***\n");
  
  while (locals) {
    c = locals;
    locals = locals->rest;
    cond = c->first;
    free_cons (c);

    if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
      print_string ("\nFor local ");
      print_wme (cond->bt.wme);
      print_string (" ");
    }

    bt_pref = find_clone_for_level (cond->bt.trace, grounds_level+1);
    /* --- if it has a trace at this level, backtrace through it --- */
    if (bt_pref) {
      backtrace_through_instantiation (bt_pref->inst, grounds_level);
      continue;
    }

    if (sysparams[TRACE_BACKTRACING_SYSPARAM])
      print_string ("...no trace, can't BT");
    /* --- for augmentations of the local goal id, either handle the
       "^quiescence t" test or discard it --- */
    if (referent_of_equality_test(cond->data.tests.id_test)->id.isa_goal) {
      if ((referent_of_equality_test(cond->data.tests.attr_test) ==
           quiescence_symbol) &&
          (referent_of_equality_test(cond->data.tests.value_test) ==
           t_symbol) &&
          (! cond->test_for_acceptable_preference))
        variablize_this_chunk = FALSE;
      continue;
    }
    
    /* --- otherwise add it to the potential set --- */
    if (sysparams[TRACE_BACKTRACING_SYSPARAM])
      print_string (" --> make it a potential.");
    add_to_potentials (cond);

  } /* end of while locals loop */
}

/* ---------------------------------------------------------------
                       Trace Grounded Potentials

   This routine looks for positive potentials that are in the TC
   of the ground set, and moves them over to the ground set.  This
   process is repeated until no more positive potentials are in
   the TC of the grounds.
--------------------------------------------------------------- */

void trace_grounded_potentials (void) {
  tc_number tc;
  cons *c, *next_c, *prev_c;
  condition *pot;
  bool need_another_pass;
  
  if (sysparams[TRACE_BACKTRACING_SYSPARAM])
    print_string ("\n\n*** Tracing Grounded Potentials ***\n");
  
  /* --- setup the tc of the ground set --- */
  tc = get_new_tc_number();
  for (c=grounds; c!=NIL; c=c->rest) add_cond_to_tc (c->first, tc, NIL, NIL); 

  need_another_pass = TRUE;
  while (need_another_pass) {
    need_another_pass = FALSE;
    /* --- look for any potentials that are in the tc now --- */
    prev_c = NIL;
    for (c=positive_potentials; c!=NIL; c=next_c) {
      next_c = c->rest;
      pot = c->first;
      if (cond_is_in_tc (pot, tc)) {
        /* --- pot is a grounded potential, move it over to ground set --- */
        if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
          print_string ("\n-->Moving to grounds: ");
          print_wme (pot->bt.wme);
        }
        if (prev_c) prev_c->rest = next_c; else positive_potentials = next_c;
        if (pot->bt.wme->grounds_tc != grounds_tc) { /* add pot to grounds */
          pot->bt.wme->grounds_tc = grounds_tc;
          c->rest = grounds; grounds = c;
          add_cond_to_tc (pot, tc, NIL, NIL);
          need_another_pass = TRUE;
        } else { /* pot was already in the grounds, do don't add it */
          free_cons (c);
        }
      } else {
        prev_c = c;
      }
    } /* end of for c */
  } /* end of while need_another_pass */
}

/* ---------------------------------------------------------------
                     Trace Ungrounded Potentials

   This routine backtraces through ungrounded potentials.  At entry,
   all potentials must be ungrounded.  This BT's through each
   potential that has some trace (at the right level) that we can
   BT through.  Other potentials are left alone.  TRUE is returned
   if anything was BT'd; FALSE if nothing changed.
--------------------------------------------------------------- */

bool trace_ungrounded_potentials (goal_stack_level grounds_level) {
  cons *c, *next_c, *prev_c;
  cons *pots_to_bt;
  condition *potential;
  preference *bt_pref;

  if (sysparams[TRACE_BACKTRACING_SYSPARAM])
    print_string ("\n\n*** Tracing Ungrounded Potentials ***\n");
  
  /* --- scan through positive potentials, pick out the ones that have
     a preference we can backtrace through --- */
  pots_to_bt = NIL;
  prev_c = NIL;
  for (c=positive_potentials; c!=NIL; c=next_c) {
    next_c = c->rest;
    potential = c->first;
    bt_pref = find_clone_for_level (potential->bt.trace, grounds_level+1);
    if (bt_pref) {
      if (prev_c) prev_c->rest = next_c; else positive_potentials = next_c;
      c->rest = pots_to_bt; pots_to_bt = c;
    } else {
      prev_c = c;
    }
  }
  
  /* --- if none to BT, exit --- */
  if (!pots_to_bt) return FALSE;

  /* --- backtrace through each one --- */
  while (pots_to_bt) {
    c = pots_to_bt;
    pots_to_bt = pots_to_bt->rest;
    potential = c->first;
    free_cons (c);
    if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
      print_string ("\nFor ungrounded potential ");
      print_wme (potential->bt.wme);
      print_string (" ");
    }
    bt_pref = find_clone_for_level (potential->bt.trace, grounds_level+1);
    backtrace_through_instantiation (bt_pref->inst, grounds_level);
  }

  return TRUE;
}

/* ==================================================================== 

                 Other Miscellaneous Chunking Routines

==================================================================== */

/* --------------------------------------------------------------------
            Build Chunk Conds For Grounds And Add Negateds
       
   This routine is called once backtracing is finished.  It goes through
   the ground conditions and builds a chunk_cond (see above) for each
   one.  The chunk_cond includes two new copies of the condition:  one
   to be used for the initial instantiation of the chunk, and one to
   be (variablized and) used for the chunk itself.

   This routine also goes through the negated conditions and adds to
   the ground set (again building chunk_cond's) any negated conditions
   that are connected to the grounds.

   At exit, the "dest_top" and "dest_bottom" arguments are set to point
   to the first and last chunk_cond in the ground set.  The "tc_to_use"
   argument is the tc number that this routine will use to mark the
   TC of the ground set.  At exit, this TC indicates the set of identifiers
   in the grounds.  (This is used immediately afterwards to figure out
   which Nots must be added to the chunk.)
-------------------------------------------------------------------- */

void build_chunk_conds_for_grounds_and_add_negateds (chunk_cond **dest_top,
                                                     chunk_cond **dest_bottom,
                                                     tc_number tc_to_use) {
  cons *c;
  condition *ground;
  chunk_cond *cc, *first_cc, *prev_cc;

  /* --- build instantiated conds for grounds and setup their TC --- */
  prev_cc = NIL;
  while (grounds) {
    c = grounds;
    grounds = grounds->rest;
    ground = c->first;
    free_cons (c);
    /* --- make the instantiated condition --- */
    allocate_with_pool (&chunk_cond_pool, &cc);
    cc->cond = ground;
    cc->instantiated_cond = copy_condition (cc->cond);
    cc->variablized_cond = copy_condition (cc->cond);
    if (prev_cc) {
      prev_cc->next = cc;
      cc->prev = prev_cc;
      cc->variablized_cond->prev = prev_cc->variablized_cond;
      prev_cc->variablized_cond->next = cc->variablized_cond;
    } else {
      first_cc = cc;
      cc->prev = NIL;
      cc->variablized_cond->prev = NIL;
    }
    prev_cc = cc;
    /* --- add this in to the TC --- */
    add_cond_to_tc (ground, tc_to_use, NIL, NIL);
  }

  /* --- scan through negated conditions and check which ones are connected
     to the grounds --- */
  if (sysparams[TRACE_BACKTRACING_SYSPARAM])
    print_string ("\n\n*** Adding Grounded Negated Conditions ***\n");
  
  while (negated_set.all) {
    cc = negated_set.all;
    remove_from_chunk_cond_set (&negated_set, cc);
    if (cond_is_in_tc (cc->cond, tc_to_use)) {
      /* --- negated cond is in the TC, so add it to the grounds --- */
      if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
        print_string ("\n-->Moving to grounds: ");
        print_condition (cc->cond);
      }
      cc->instantiated_cond = copy_condition (cc->cond);
      cc->variablized_cond = copy_condition (cc->cond);
      if (prev_cc) {
        prev_cc->next = cc;
        cc->prev = prev_cc;
        cc->variablized_cond->prev = prev_cc->variablized_cond;
        prev_cc->variablized_cond->next = cc->variablized_cond;
      } else {
        first_cc = cc;
        cc->prev = NIL;
        cc->variablized_cond->prev = NIL;
      }
      prev_cc = cc;
    } else {
      /* --- not in TC, so discard the condition --- */
      free_with_pool (&chunk_cond_pool, cc);
    }
  }

  if (prev_cc) {
    prev_cc->next = NIL;
    prev_cc->variablized_cond->next = NIL;
  } else {
    first_cc = NIL;
  }
  
  *dest_top = first_cc;
  *dest_bottom = prev_cc;
}

/* --------------------------------------------------------------------
                  Get Nots For Instantiated Conditions

   This routine looks through all the Nots in the instantiations in
   instantiations_with_nots, and returns copies of the ones involving
   pairs of identifiers in the grounds.  Before this routine is called,
   the ids in the grounds must be marked with "tc_of_grounds."  
-------------------------------------------------------------------- */

extern memory_pool not_pool;

not *get_nots_for_instantiated_conditions (list *instantiations_with_nots,
                                           tc_number tc_of_grounds) {
  cons *c;
  instantiation *inst;
  not *n1, *n2, *new_not, *collected_nots;

  /* --- collect nots for which both id's are marked --- */
  collected_nots = NIL;
  while (instantiations_with_nots) {
    c = instantiations_with_nots;
    instantiations_with_nots = c->rest;
    inst = c->first;
    free_cons (c);
    for (n1=inst->nots; n1 != NIL; n1=n1->next) {
      /* --- Are both id's marked? If no, goto next loop iteration --- */
      if (n1->s1->id.tc_num != tc_of_grounds) continue;
      if (n1->s2->id.tc_num != tc_of_grounds) continue;
      /* --- If the pair already in collected_nots, goto next iteration --- */
      for (n2=collected_nots; n2!=NIL; n2=n2->next) {
        if ((n2->s1 == n1->s1) && (n2->s2 == n1->s2)) break;
        if ((n2->s1 == n1->s2) && (n2->s2 == n1->s1)) break;
      }
      if (n2) continue;
      /* --- Add the pair to collected_nots --- */
      allocate_with_pool (&not_pool, &new_not);
      new_not->next = collected_nots;
      collected_nots = new_not;
      new_not->s1 = n1->s1;
      symbol_add_ref (new_not->s1);
      new_not->s2 = n1->s2;
      symbol_add_ref (new_not->s2);
    } /* end of for n1 */
  } /* end of while instantiations_with_nots */

  return collected_nots;
}

/* --------------------------------------------------------------------
              Variablize Nots And Insert Into Conditions
             
   This routine goes through the given list of Nots and, for each one,
   inserts a variablized copy of it into the given condition list at
   the earliest possible location.  (The given condition list should
   be the previously-variablized condition list that will become the
   chunk's LHS.)  The given condition list is destructively modified;
   the given Not list is unchanged.
-------------------------------------------------------------------- */

void variablize_nots_and_insert_into_conditions (not *nots,
                                                 condition *conds) {
  not *n;
  symbol *var1, *var2;
  test t;
  complex_test *ct;
  condition *c;
  bool added_it;

  /* --- don't bother Not-ifying justifications --- */
  if (! variablize_this_chunk) return;
  
  for (n=nots; n!=NIL; n=n->next) {
    var1 = n->s1->id.variablization;
    var2 = n->s2->id.variablization;
    /* --- find where var1 is bound, and add "<> var2" to that test --- */
    allocate_with_pool (&complex_test_pool, &ct);
    t = make_test_from_complex_test (ct);
    ct->type = NOT_EQUAL_TEST;
    ct->data.referent = var2;
    symbol_add_ref (var2);
    added_it = FALSE;
    for (c=conds; c!=NIL; c=c->next) {
      if (c->type != POSITIVE_CONDITION) continue;
      if (test_includes_equality_test_for_symbol (c->data.tests.id_test,
                                                  var1)) {
        add_new_test_to_test (&(c->data.tests.id_test), t);
        added_it = TRUE;
        break;
      }
      if (test_includes_equality_test_for_symbol (c->data.tests.attr_test,
                                                  var1)) {
        add_new_test_to_test (&(c->data.tests.attr_test), t);
        added_it = TRUE;
        break;
      }
      if (test_includes_equality_test_for_symbol (c->data.tests.value_test,
                                                  var1)) {
        add_new_test_to_test (&(c->data.tests.value_test), t);
        added_it = TRUE;
        break;
      }
    }
    if (!added_it) {
      print ("Internal error: couldn't add Not test to chunk\n");
      abort_with_fatal_error();
    }
  } /* end of for n=nots */
}

/* --------------------------------------------------------------------
                     Add Goal or Impasse Tests

   This routine adds goal id or impasse id tests to the variablized
   conditions.  For each id in the grounds that happens to be the
   identifier of a goal or impasse, we add a goal/impasse id test
   to the variablized conditions, to make sure that in the resulting
   chunk, the variablization of that id is constrained to match against
   a goal/impasse.  (Note:  actually, in the current implementation of
   chunking, it's impossible for an impasse id to end up in the ground
   set.  So part of this code is unnecessary.)
-------------------------------------------------------------------- */

void add_goal_or_impasse_tests (chunk_cond *all_ccs) {
  chunk_cond *cc;
  tc_number tc;   /* mark each id as we add a test for it, so we don't add
                     a test for the same id in two different places */
  symbol *id;
  test t;
  complex_test *ct;

  tc = get_new_tc_number();
  for (cc=all_ccs; cc!=NIL; cc=cc->next) {
    if (cc->instantiated_cond->type!=POSITIVE_CONDITION) continue;
    id = referent_of_equality_test (cc->instantiated_cond->data.tests.id_test);
    if ( (id->id.isa_goal || id->id.isa_impasse) &&
         (id->id.tc_num != tc) ) {
      allocate_with_pool (&complex_test_pool, &ct);
      ct->type = (id->id.isa_goal) ? GOAL_ID_TEST : IMPASSE_ID_TEST;
      t = make_test_from_complex_test(ct);
      add_new_test_to_test (&(cc->variablized_cond->data.tests.id_test), t);
      id->id.tc_num = tc;
    }
  }
}

/* --------------------------------------------------------------------
                    Reorder Instantiated Conditions

   The Rete routines require the instantiated conditions (on the
   instantiation structure) to be in the same order as the original
   conditions from which the Rete was built.  This means that the
   initial instantiation of the chunk must have its conditions in
   the same order as the variablized conditions.  The trouble is,
   the variablized conditions get rearranged by the reorderer.  So,
   after reordering, we have to rearrange the instantiated conditions
   to put them in the same order as the now-scrambled variablized ones.
   This routine does this.

   Okay, so the obvious way is to have each variablized condition (VCond)
   point to the corresponding instantiated condition (ICond).  Then after
   reordering the VConds, we'd scan through the VConds and say
      VCond->Icond->next = VCond->next->Icond
      VCond->Icond->prev = VCond->prev->Icond
   (with some extra checks for the first and last VCond in the list).

   The problem with this is that it takes an extra 4 bytes per condition,
   for the "ICond" field.  Conditions were taking up a lot of memory in
   my test cases, so I wanted to shrink them.  This routine avoids needing
   the 4 extra bytes by using the following trick:  first "swap out" 4
   bytes from each VCond; then use that 4 bytes for the "ICond" field.
   Now run the above algorithm.  Finally, swap those original 4 bytes
   back in.
-------------------------------------------------------------------- */

void reorder_instantiated_conditions (chunk_cond *top_cc,
                                      condition **dest_inst_top,
                                      condition **dest_inst_bottom) {
  chunk_cond *cc;

  /* --- Step 1:  swap prev pointers out of variablized conds into chunk_conds,
     and swap pointer to the corresponding instantiated conds into the
     variablized conds' prev pointers --- */
  for (cc=top_cc; cc!=NIL; cc=cc->next) {
    cc->saved_prev_pointer_of_variablized_cond = cc->variablized_cond->prev;
    cc->variablized_cond->prev = cc->instantiated_cond;
  }

  /* --- Step 2:  do the reordering of the instantiated conds --- */
  for (cc=top_cc; cc!=NIL; cc=cc->next) {
    if (cc->variablized_cond->next) {
      cc->instantiated_cond->next = cc->variablized_cond->next->prev;
    } else {
      cc->instantiated_cond->next = NIL;
      *dest_inst_bottom = cc->instantiated_cond;
    }
    
    if (cc->saved_prev_pointer_of_variablized_cond) {
      cc->instantiated_cond->prev =
        cc->saved_prev_pointer_of_variablized_cond->prev;
    } else {
      cc->instantiated_cond->prev = NIL;
      *dest_inst_top = cc->instantiated_cond;
    }
  }

  /* --- Step 3:  restore the prev pointers on variablized conds --- */
  for (cc=top_cc; cc!=NIL; cc=cc->next) {
    cc->variablized_cond->prev = cc->saved_prev_pointer_of_variablized_cond;
  }
}

/* --------------------------------------------------------------------
                       Make Clones of Results

   When we build the initial instantiation of the new chunk, we have
   to fill in preferences_generated with *copies* of all the result
   preferences.  These copies are clones of the results.  This routine
   makes these clones and fills in chunk_inst->preferences_generated.
-------------------------------------------------------------------- */

void make_clones_of_results (preference *results, instantiation *chunk_inst) {
  preference *p, *result_p;

  chunk_inst->preferences_generated = NIL;
  for (result_p=results; result_p!=NIL; result_p=result_p->next_result) {
    /* --- copy the preference --- */
    p = make_preference (result_p->type, result_p->id, result_p->attr,
                         result_p->value, result_p->referent);
    symbol_add_ref (p->id);
    symbol_add_ref (p->attr);
    symbol_add_ref (p->value);
    if (preference_is_binary(p->type))
      symbol_add_ref (p->referent);
    /* --- put it onto the list for chunk_inst --- */
    p->inst = chunk_inst;
    insert_at_head_of_dll (chunk_inst->preferences_generated, p,
                           inst_next, inst_prev);
    /* --- insert it into the list of clones for this preference --- */
    p->next_clone = result_p;
    p->prev_clone = result_p->prev_clone;
    result_p->prev_clone = p;
    if (p->prev_clone) p->prev_clone->next_clone = p;
  }
}

/* ====================================================================

                        Chunk Instantiation

   This the main chunking routine.  It takes an instantiation, and a
   flag "allow_variablization"--if FALSE, the chunk will not be
   variablized.  (If TRUE, it may still not be variablized, due to
   chunk-free-problem-spaces, ^quiescence t, etc.)
==================================================================== */

unsigned long chunk_count = 1;          /* counters used for names of chunks */
unsigned long justification_count = 1;

void chunk_instantiation (instantiation *inst, bool allow_variablization) {
  goal_stack_level grounds_level;
  preference *results, *pref;
  action *rhs;
  production *prod;
  instantiation *chunk_inst;
  symbol *prod_name;
  byte prod_type;
  bool print_name, print_prod;
  byte rete_addition_result;
  condition *lhs_top, *lhs_bottom;
  not *nots;
  chunk_cond *top_cc, *bottom_cc;
#ifdef DETAILED_TIMING_STATS
  struct timeval saved_start_tv;
#endif
  
  /* --- if it only matched an attribute impasse, don't chunk --- */
  if (! inst->match_goal) return; 

  /* --- if no preference is above the match goal level, exit --- */
  for (pref=inst->preferences_generated; pref!=NIL; pref=pref->inst_next)
    if (pref->id->id.level < inst->match_goal_level)
      break;
  if (! pref) return;
  
#ifdef DETAILED_TIMING_STATS
  start_timer (&saved_start_tv);
#endif

  results = get_results_for_instantiation (inst);
  if (!results) goto chunking_done;

  /* --- update flags on goal stack for bottom-up chunking --- */
  { symbol *g;
    for (g=inst->match_goal->id.higher_goal;
         g && g->id.allow_bottom_up_chunks;
         g=g->id.higher_goal)
      g->id.allow_bottom_up_chunks = FALSE;
  }

  grounds_level = inst->match_goal_level - 1;

  backtrace_number++; if (backtrace_number==0) backtrace_number=1;
  grounds_tc++; if (grounds_tc==0) grounds_tc=1;
  potentials_tc++; if (potentials_tc==0) potentials_tc=1;
  locals_tc++; if (locals_tc==0) locals_tc=1;
  grounds = NIL;
  positive_potentials = NIL;
  locals = NIL;
  instantiations_with_nots = NIL;

  if (allow_variablization && (! sysparams[LEARNING_ALL_GOALS_SYSPARAM]))
    allow_variablization = inst->match_goal->id.allow_bottom_up_chunks;

  /* --- check whether ps name is in chunk_free_problem_spaces --- */
  if (allow_variablization) {
    if (inst->match_goal->id.problem_space_slot->wmes) {
      symbol *ps_id, *ps_name;
      ps_id = inst->match_goal->id.problem_space_slot->wmes->value;
      ps_name = find_name_of_object (ps_id);
      if (ps_name)
        if (member_of_list (ps_name, chunk_free_problem_spaces))
          allow_variablization = FALSE;
    }
  }

  variablize_this_chunk = allow_variablization;
  
  /* --- backtrace through the instantiation that produced each result --- */
  for (pref=results; pref!=NIL; pref=pref->next_result) {
    if (sysparams[TRACE_BACKTRACING_SYSPARAM]) {
      print_string ("\nFor result preference ");
      print_preference (pref);
      print_string (" ");
    }
    backtrace_through_instantiation (pref->inst, grounds_level);
  }

  while (TRUE) {
    trace_locals (grounds_level);
    trace_grounded_potentials ();
    if (! trace_ungrounded_potentials (grounds_level)) break;
  }
  free_list (positive_potentials);

  /* --- backtracing done; collect the grounds into the chunk --- */
  { tc_number tc_for_grounds;
    tc_for_grounds = get_new_tc_number();
    build_chunk_conds_for_grounds_and_add_negateds (&top_cc, &bottom_cc,
                                                    tc_for_grounds);
    nots = get_nots_for_instantiated_conditions (instantiations_with_nots,
                                                 tc_for_grounds);
  }

  /* --- get symbol for name of new chunk or justification --- */
  if (variablize_this_chunk) {
    prod_name = generate_new_sym_constant ("chunk-",&chunk_count);
    prod_type = CHUNK_PRODUCTION_TYPE;
    print_name = sysparams[TRACE_CHUNK_NAMES_SYSPARAM];
    print_prod = sysparams[TRACE_CHUNKS_SYSPARAM];
  } else {
    prod_name = generate_new_sym_constant ("justification-",
                                           &justification_count);
    prod_type = JUSTIFICATION_PRODUCTION_TYPE;
    print_name = sysparams[TRACE_JUSTIFICATION_NAMES_SYSPARAM];
    print_prod = sysparams[TRACE_JUSTIFICATIONS_SYSPARAM];
  }
  if (print_name) print_with_symbols ("\nBuild: %y", prod_name);

  /* --- if there aren't any grounds, exit --- */
  if (! top_cc) {
    if (sysparams[PRINT_WARNINGS_SYSPARAM])
      print_string (" Warning: chunk has no grounds, ignoring it.");
    goto chunking_done;
  }

  /* --- variablize it --- */
  lhs_top = top_cc->variablized_cond;
  lhs_bottom = bottom_cc->variablized_cond;
  reset_variable_generator (lhs_top, NIL);
  variablization_tc = get_new_tc_number();
  variablize_condition_list (lhs_top);
  variablize_nots_and_insert_into_conditions (nots, lhs_top);
  rhs = copy_and_variablize_result_list (results);

  /* --- add goal/impasse tests to it --- */
  add_goal_or_impasse_tests (top_cc);

  /* --- reorder lhs and make the production --- */

  prod = make_production (prod_type, prod_name, &lhs_top, &lhs_bottom, &rhs,
                          FALSE);

  if (!prod) {
    print ("\nUnable to reorder this chunk:\n  ");
    print_condition_list (lhs_top, 2, FALSE);
    print ("\n  -->\n   ");
    print_action_list (rhs, 3, FALSE);
    print ("\n\n(Ignoring this chunk.  Weird things could happen from now on...)\n");
    goto chunking_done; /* this leaks memory but who cares */
  }

  { condition *inst_lhs_top, *inst_lhs_bottom;

    reorder_instantiated_conditions (top_cc, &inst_lhs_top, &inst_lhs_bottom);
 
    allocate_with_pool (&instantiation_pool, &chunk_inst);
    chunk_inst->prod = prod;
    chunk_inst->top_of_instantiated_conditions = inst_lhs_top;
    chunk_inst->bottom_of_instantiated_conditions = inst_lhs_bottom;
    chunk_inst->nots = nots;
    chunk_inst->okay_to_variablize = variablize_this_chunk;
    chunk_inst->in_ms = TRUE;  /* set TRUE for now, we'll find out later... */
    make_clones_of_results (results, chunk_inst);
    fill_in_new_instantiation_stuff (chunk_inst, TRUE);
  }

  rete_addition_result = add_production_to_rete (prod, lhs_top, chunk_inst,
                                                 print_name);

  /* --- deallocate chunks conds and variablized conditions --- */
  deallocate_condition_list (lhs_top);
  { chunk_cond *cc;
    while (top_cc) {
      cc = top_cc;
      top_cc = cc->next;
      free_with_pool (&chunk_cond_pool, cc);
    }
  }

  if (print_prod && (rete_addition_result!=DUPLICATE_PRODUCTION)) {
    print_string ("\n");
    print_production (prod, FALSE);
  }
  
  if (rete_addition_result==DUPLICATE_PRODUCTION) {
    excise_production (prod, FALSE);
  } else if ((prod_type==JUSTIFICATION_PRODUCTION_TYPE) &&
             (rete_addition_result==REFRACTED_INST_DID_NOT_MATCH)) {
    excise_production (prod, FALSE);
  }

  if (rete_addition_result!=REFRACTED_INST_MATCHED) {
    /* --- it didn't match, or it was a duplicate production --- */
    /* --- tell the firer it didn't match, so it'll only assert the
       o-supported preferences --- */
    chunk_inst->in_ms = FALSE;
  }

  /* --- assert the preferences --- */
  chunk_inst->next = newly_created_instantiations;
  newly_created_instantiations = chunk_inst;

#ifdef DETAILED_TIMING_STATS
  stop_timer (&saved_start_tv, &chunking_cpu_time);
#endif
  chunk_instantiation (chunk_inst, variablize_this_chunk);
  return;

  chunking_done: {}
#ifdef DETAILED_TIMING_STATS
  stop_timer (&saved_start_tv, &chunking_cpu_time);
#endif
}

/* --------------------------------------------------------------------

                        Chunker Initialization

   Init_chunker() is called at startup time to do initialization here.
-------------------------------------------------------------------- */

void init_chunker (void) {
  init_memory_pool (&chunk_cond_pool, sizeof(chunk_cond), "chunk condition");
  init_chunk_cond_set (&negated_set);
}
