/* ====================================================================
                   RHS Function Management for Soar 6

   The system maintains a list of available RHS functions.  Functions
   can appear on the RHS of productions either as values (in make actions
   or as arguments to other function calls) or as stand-alone actions
   (e.g., "write" and "halt").  When a function is executed, its C code
   is called with one parameter--a (consed) list of the arguments (symbols).
   The C function should return either a symbol (if all goes well) or NIL
   (if an error occurred, or if the function is a stand-alone action).

   All available RHS functions should be setup at system startup time via
   calls to add_rhs_function().  It takes as arguments the name of the
   function (a symbol), a pointer to the corresponding C function, the
   number of arguments the function expects (-1 if the function can take
   any number of arguments), and flags indicating whether the function can
   be a RHS value or a stand-alone action.

   Lookup_rhs_function() takes a symbol and returns the corresponding
   rhs_function structure (or NIL if there is no such function).

   Init_built_in_rhs_functions() should be called at system startup time
   to setup all the built-in functions.
==================================================================== */

#include "soar.h"
#include <time.h>

rhs_function *available_rhs_functions = NIL;

void add_rhs_function (symbol *name,
                       rhs_function_routine f,
                       int num_args_expected,
                       bool can_be_rhs_value,
                       bool can_be_stand_alone_action) {
  rhs_function *rf;

  if ((!can_be_rhs_value) && (!can_be_stand_alone_action)) {
    print ("Internal error: attempt to add_rhs_function that can't appear anywhere\n");
    return;
  }
  for (rf=available_rhs_functions; rf!=NIL; rf=rf->next)
    if (rf->name==name) break;
  if (rf) {
    print_with_symbols ("Internal error: attempt to add_rhs_function that already exists: %y\n", name);
    return;
  }
  rf = allocate_memory (sizeof(rhs_function), MISCELLANEOUS_MEM_USAGE);
  rf->next = available_rhs_functions;
  available_rhs_functions = rf;
  rf->name = name;
  rf->f = f;
  rf->num_args_expected = num_args_expected;
  rf->can_be_rhs_value = can_be_rhs_value;
  rf->can_be_stand_alone_action = can_be_stand_alone_action;
}

rhs_function *lookup_rhs_function (symbol *name) {
  rhs_function *rf;

  for (rf=available_rhs_functions; rf!=NIL; rf=rf->next)
    if (rf->name==name) return rf;
  return NIL;
}

/* ====================================================================

               Code for Executing Built-In RHS Functions

====================================================================  */

/* --------------------------------------------------------------------
                                Write

   Takes any number of arguments, and prints each one.
-------------------------------------------------------------------- */

symbol *write_rhs_function_code (list *args) {
  symbol *arg;
  char *string;
  
  for ( ; args!=NIL; args=args->rest) {
    arg = args->first;
    /* --- Note use of FALSE here--print the symbol itself, not a rereadable
       version of it --- */
    string = symbol_to_string (arg, FALSE, NIL);
    print_string (string);
  }
  return NIL;
}

/* --------------------------------------------------------------------
                                Crlf

   Just returns a sym_constant whose print name is a line feed.
-------------------------------------------------------------------- */

symbol *crlf_rhs_function_code (list *args) {
  return make_sym_constant ("\n");
}

/* --------------------------------------------------------------------
                                Halt

   Just sets a flag indicating that the system has halted.
-------------------------------------------------------------------- */

symbol *halt_rhs_function_code (list *args) {
  system_halted = TRUE;
  return NIL;
}

/* --------------------------------------------------------------------
                              Interrupt

   This causes an interrupt at the end of the current preference phase.
   It sets stop_soar to TRUE, and reason_for_stopping to an appropriate
   string.
-------------------------------------------------------------------- */

char interrupt_source[2*MAX_LEXEME_LENGTH+100];

symbol *interrupt_rhs_function_code (list *args) {
  char *ch;
  
  stop_soar = TRUE;
  strcpy (interrupt_source, "*** Interrupt from production ");
  ch = interrupt_source;
  while (*ch) ch++;
  symbol_to_string (production_being_fired->name, TRUE, ch); 
  while (*ch) ch++;
  strcpy (ch, " ***");
  reason_for_stopping = interrupt_source;
  return NIL;
}

/* --------------------------------------------------------------------
                                Plus

   Takes any number of int_constant or float_constant arguments, and
   returns their sum.
-------------------------------------------------------------------- */

symbol *plus_rhs_function_code (list *args) {
  bool float_found;
  long i;
  float f;
  symbol *arg;
  cons *c;

  for (c=args; c!=NIL; c=c->rest) {
    arg = c->first;
    if ((arg->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) &&
        (arg->common.symbol_type != FLOAT_CONSTANT_SYMBOL_TYPE)) {
      print_with_symbols ("Error: non-number (%y) passed to + function\n",
                          arg);
      return NIL;
    }
  }

  i = 0;
  float_found = FALSE;
  while (args) {
    arg = args->first;
    if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) {
      if (float_found) f += arg->ic.value;
      else i += arg->ic.value;
    } else {
      if (float_found) f += arg->fc.value;
      else { float_found = TRUE; f = arg->fc.value + i; }
    }
    args = args->rest;
  }
  if (float_found) return make_float_constant (f);
  return make_int_constant (i);
}

/* --------------------------------------------------------------------
                                Times

   Takes any number of int_constant or float_constant arguments, and
   returns their product.
-------------------------------------------------------------------- */

symbol *times_rhs_function_code (list *args) {
  bool float_found;
  long i;
  float f;
  symbol *arg;
  cons *c;
  
  for (c=args; c!=NIL; c=c->rest) {
    arg = c->first;
    if ((arg->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) &&
        (arg->common.symbol_type != FLOAT_CONSTANT_SYMBOL_TYPE)) {
      print_with_symbols ("Error: non-number (%y) passed to * function\n",
                          arg);
      return NIL;
    }
  }

  i = 1;
  float_found = FALSE;
  while (args) {
    arg = args->first;
    if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) {
      if (float_found) f *= arg->ic.value;
      else i *= arg->ic.value;
    } else {
      if (float_found) f *= arg->fc.value;
      else { float_found = TRUE; f = arg->fc.value * i; }
    }
    args = args->rest;
  }
  if (float_found) return make_float_constant (f);
  return make_int_constant (i);
}

/* --------------------------------------------------------------------
                                Minus

   Takes one or more int_constant or float_constant arguments.
   If 0 arguments, returns NIL (error).
   If 1 argument (x), returns -x.
   If >=2 arguments (x, y1, ..., yk), returns x - y1 - ... - yk.
-------------------------------------------------------------------- */

symbol *minus_rhs_function_code (list *args) {
  symbol *arg;
  float f;
  long i;
  cons *c;
  bool float_found;

  if (!args) {
    print ("Error: '-' function called with no arguments\n");
    return NIL;
  }
  
  for (c=args; c!=NIL; c=c->rest) {
    arg = c->first;
    if ((arg->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) &&
        (arg->common.symbol_type != FLOAT_CONSTANT_SYMBOL_TYPE)) {
      print_with_symbols ("Error: non-number (%y) passed to - function\n",
                          arg);
      return NIL;
    }
  }

  if (! args->rest) {
    /* --- only one argument --- */
    arg = args->first;
    if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE)
      return make_int_constant (- arg->ic.value);
    return make_float_constant (- arg->fc.value);
  }

  /* --- two or more arguments --- */
  arg = args->first;
  float_found = FALSE;
  if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) i = arg->ic.value;
  else { float_found = TRUE; f = arg->fc.value; }
  for (c=args->rest; c!=NIL; c=c->rest) {
    arg = c->first;
    if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) {
      if (float_found) f -= arg->ic.value;
      else i -= arg->ic.value;
    } else {
      if (float_found) f -= arg->fc.value;
      else { float_found = TRUE; f = i - arg->fc.value; }
    }
  }
 
  if (float_found) return make_float_constant (f);
  return make_int_constant (i);
}

/* --------------------------------------------------------------------
                     Floating-Point Division

   Takes one or more int_constant or float_constant arguments.
   If 0 arguments, returns NIL (error).
   If 1 argument (x), returns 1/x.
   If >=2 arguments (x, y1, ..., yk), returns x / y1 / ... / yk.
-------------------------------------------------------------------- */

symbol *fp_divide_rhs_function_code (list *args) {
  symbol *arg;
  float f;
  cons *c;

  if (!args) {
    print ("Error: '/' function called with no arguments\n");
    return NIL;
  }
  
  for (c=args; c!=NIL; c=c->rest) {
    arg = c->first;
    if ((arg->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) &&
        (arg->common.symbol_type != FLOAT_CONSTANT_SYMBOL_TYPE)) {
      print_with_symbols ("Error: non-number (%y) passed to / function\n",
                          arg);
      return NIL;
    }
  }

  if (! args->rest) {
    /* --- only one argument --- */
    arg = args->first;
    if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) f = arg->ic.value;
    else f = arg->fc.value;
    if (f != 0.0) return make_float_constant (1.0 / f);
    print ("Error: attempt to divide ('/') by zero.\n");
    return NIL;
  }

  /* --- two or more arguments --- */
  arg = args->first;
  if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) f = arg->ic.value;
  else f = arg->fc.value;
  for (c=args->rest; c!=NIL; c=c->rest) {
    arg = c->first;
    if (arg->common.symbol_type==INT_CONSTANT_SYMBOL_TYPE) {
      if (arg->ic.value) f /= arg->ic.value;
      else { print ("Error: attempt to divide ('/') by zero.\n"); return NIL; }
    } else {
      if (arg->fc.value != 0.0) f /= arg->fc.value;
      else { print ("Error: attempt to divide ('/') by zero.\n"); return NIL; }
    }
  }
 
  return make_float_constant (f);
}

/* --------------------------------------------------------------------
                     Integer Division (Quotient)

   Takes two int_constant arguments, and returns their quotient.
-------------------------------------------------------------------- */

symbol *div_rhs_function_code (list *args) {
  symbol *arg1, *arg2;

  arg1 = args->first;
  arg2 = args->rest->first;
  
  if (arg1->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) {
    print_with_symbols ("Error: non-integer (%y) passed to div function\n",
                        arg1);
    return NIL;
  }
  if (arg2->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) {
    print_with_symbols ("Error: non-integer (%y) passed to div function\n",
                        arg2);
    return NIL;
  }

  if (arg2->ic.value == 0) {
    print ("Error: attempt to divide ('div') by zero.\n");
    return NIL;
  }
  
  return make_int_constant (arg1->ic.value / arg2->ic.value);
 /* Warning: ANSI doesn't say precisely what happens if one or both of the
    two args is negative. */
}

/* --------------------------------------------------------------------
                          Integer Modulus

   Takes two int_constant arguments (x,y) and returns (x mod y), i.e.,
   the remainder after dividing x by y.
-------------------------------------------------------------------- */

symbol *mod_rhs_function_code (list *args) {
  symbol *arg1, *arg2;

  arg1 = args->first;
  arg2 = args->rest->first;
  
  if (arg1->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) {
    print_with_symbols ("Error: non-integer (%y) passed to mod function\n",
                        arg1);
    return NIL;
  }
  if (arg2->common.symbol_type != INT_CONSTANT_SYMBOL_TYPE) {
    print_with_symbols ("Error: non-integer (%y) passed to mod function\n",
                        arg2);
    return NIL;
  }

  if (arg2->ic.value == 0) {
    print ("Error: attempt to divide ('mod') by zero.\n");
    return NIL;
  }
  
  return make_int_constant (arg1->ic.value % arg2->ic.value);
 /* Warning:  ANSI guarantees this does the right thing if both args are
    positive.  If one or both is negative, it only guarantees that
    (a/b)*b + a%b == a. */
}

/* --------------------------------------------------------------------
                         Make-constant-symbol

   Returns a newly generated sym_constant.  If no arguments are given,
   the constant will start with "constant".  If one or more arguments
   are given, the constant will start with a string equal to the
   concatenation of those arguments.
-------------------------------------------------------------------- */

unsigned long mcs_counter = 1;

symbol *make_constant_symbol_rhs_function_code (list *args) {
  char buf[1000]; /* that ought to be long enough */
  char *string;
  cons *c;

  if (!args) {
    strcpy (buf, "constant");
  } else {
    buf[0] = 0;
    for (c=args; c!=NIL; c=c->rest) {
      string = symbol_to_string (c->first, FALSE, NIL);
      strcat (buf, string);
    }
  }
  if ((!args) && (!find_sym_constant (buf))) return make_sym_constant (buf);
  return generate_new_sym_constant (buf, &mcs_counter);
}

/* --------------------------------------------------------------------
                               Timestamp

   Returns a newly generated sym_constant whose name is a representation
   of the current local time.
-------------------------------------------------------------------- */

symbol *timestamp_rhs_function_code (list *args) {
  long now;
  struct tm *temp;
  char buf[100];

  now = time(NULL);
  temp = localtime (&now);
  sprintf (buf, "%d/%d/%d-%02d:%02d:%02d",
           temp->tm_mon + 1, temp->tm_mday, temp->tm_year,
           temp->tm_hour, temp->tm_min, temp->tm_sec);
  return make_sym_constant (buf);
}

/* --------------------------------------------------------------------
                              Accept

   Waits for the user to type a line of input; then returns the first
   symbol from that line.
-------------------------------------------------------------------- */

symbol *accept_rhs_function_code (list *args) {
  char buf[2000], *s;
  symbol *sym;

  while (TRUE) {
    s = fgets (buf, 2000, stdin);
    if (!s) {
      /* s==NIL means immediate eof encountered or read error occurred */
      return NIL;
    }
    s = buf;
    sym = get_next_io_symbol_from_text_input_line (&s);
    if (sym) break;
  }
  symbol_add_ref (sym);
  release_io_symbol (sym); /* because it was obtained using get_io_... */
  return sym;
}

/* ====================================================================

                  Initialize the Built-In RHS Functions

====================================================================  */

void init_built_in_rhs_functions (void) {
  add_rhs_function (make_sym_constant ("write"), write_rhs_function_code,
                    -1, FALSE, TRUE);
  add_rhs_function (make_sym_constant ("crlf"), crlf_rhs_function_code,
                    0, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("halt"), halt_rhs_function_code,
                    0, FALSE, TRUE);
  add_rhs_function (make_sym_constant ("interrupt"),
                    interrupt_rhs_function_code,
                    0, FALSE, TRUE);
  add_rhs_function (make_sym_constant ("+"), plus_rhs_function_code,
                    -1, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("*"), times_rhs_function_code,
                    -1, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("-"), minus_rhs_function_code,
                    -1, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("/"), fp_divide_rhs_function_code,
                    -1, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("div"), div_rhs_function_code,
                    2, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("mod"), mod_rhs_function_code,
                    2, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("make-constant-symbol"),
                    make_constant_symbol_rhs_function_code,
                    -1, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("timestamp"),
                    timestamp_rhs_function_code,
                    0, TRUE, FALSE);
  add_rhs_function (make_sym_constant ("accept"), accept_rhs_function_code,
                    0, TRUE, FALSE);
}
