/* =================================================================
                             interface.c                             

See comments in soar.h for an overview.
================================================================= */

#include <sys/time.h>       /* used for "time" command */
#include <sys/resource.h>
#include <time.h>
#include "soar.h"

/* ===================================================================

                          Command Management

  Each user interface command has a corresponding function
  (user_interface_routine) to handle it.  These commands/functions
  should be installed at system startup time via add_command().  The
  command name string passed to add_command() must be permanently
  available (e.g., a constant in global data memory).

  When a user interface routine is called, the current lexeme is the
  command name.  The function should call the lexer to read its arguments,
  etc.  If successful, the function should return TRUE and should exit
  with the current lexeme being the closing right parenthesis (otherwise
  the dispatching function will print an error message about extra
  arguments being given).  If unsuccessful, the function should
  return FALSE.
  
  The command dispatcher calls set_lexer_allow_ids(TRUE) before dispatching
  any command.
=================================================================== */

typedef struct interface_routine_struct {
  struct interface_routine_struct *next;
  char *command_name;
  user_interface_routine f;
} interface_routine;

interface_routine *interface_routines = NIL;

void add_command (char *command_name, user_interface_routine f) {
  interface_routine *ir;

  /* --- make sure we don't already have a routine with the same name --- */
  for (ir=interface_routines; ir!=NIL; ir=ir->next)
    if (! strcmp(ir->command_name,command_name)) break;
  if (ir) {
    print ("Internal error: tried to add_command that already exists: %s\n",
           command_name);
    return;
  }
  /* --- create new interface routine structure --- */
  ir = allocate_memory (sizeof(interface_routine), MISCELLANEOUS_MEM_USAGE);
  ir->next = interface_routines;
  interface_routines = ir;
  ir->command_name = command_name;
  ir->f = f;
}

/* ===================================================================
   
                         Dispatching Commands

  Dispatch_command() dispatches the appropriate user interface routine
  for the current command (i.e., the command named by the current lexeme).
  It calls set_lexer_allow_ids(TRUE) before dispatching the command,
  so if the command doesn't allow id's, it should call 
  set_lexer_allow_ids(FALSE) immediately.  Dispatch_command() returns 
  TRUE if the command was successful, FALSE if any error occurred.
   
  Repeatedly_read_and_dispatch_commands() keeps calling dispatch_command()
  until end-of-file is reached on the current input file.
  
  Load_file() sets up the lexer to read from a given open file, executes
  all the commands in that file, and then restore the lexer to reading
  the previous file.
=================================================================== */

bool dispatch_command (void) {
  interface_routine *ir;
  int parentheses_level;
  bool result;
  
  parentheses_level = current_lexer_parentheses_level();
  for (ir=interface_routines; ir!=NIL; ir=ir->next)
    if (! strcmp(ir->command_name,lexeme.string)) break;
  if (! ir) {
    /* --- no such command --- */
    print ("Error:  unknown command %s\n", lexeme.string);
    print_location_of_most_recent_lexeme();
    if (current_lexer_parentheses_level() != parentheses_level-1)
      skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return FALSE;
  }

  set_lexer_allow_ids (TRUE);
  result = (*(ir->f))();
  if (current_lexer_parentheses_level() != parentheses_level-1) {
    if (result) {
      print ("Ignoring extra argument(s)\n");
      print_location_of_most_recent_lexeme();
      result = FALSE;
    }
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
  }
  return result;
}

void repeatedly_read_and_dispatch_commands (bool print_prompt) {
  while (TRUE) {
    if (print_prompt) print ("\nSoar> ");

    if (current_lexer_parentheses_level()!=0) {
      print ("Internal error:  misbalanced parentheses in main loop.\n");
      abort_with_fatal_error();
    }
    
    /* --- consume rparen from previous command, get start of next cmd. --- */
    get_lexeme();
    if (lexeme.type==EOF_LEXEME) return;

    /* --- if not lparen, fake one at end of the current line --- */
    if (lexeme.type==L_PAREN_LEXEME) {
      get_lexeme(); /* consume lparen */
    } else {
      fake_rparen_at_next_end_of_line ();
    }
    
    if (lexeme.type==SYM_CONSTANT_LEXEME) {
      dispatch_command();
    } else {
      print ("Error:  unknown command %s\n", lexeme.string);
      print_location_of_most_recent_lexeme();
      skip_ahead_to_balanced_parentheses(0);
    }
    if (lexeme.type==EOF_LEXEME) return;
  } /* end of while TRUE */
}

void load_file (char *file_name, FILE *already_open_file) {
  start_lex_from_file (file_name, already_open_file);
  repeatedly_read_and_dispatch_commands (FALSE);
  stop_lex_from_file ();
}

/* ===================================================================
   
               Help Information Management Routines

   Add_help() should be called at system startup time to specify to the
   "help" command what help info is available.  It takes a topic name and
   an array of lines of text for the helpscreen.  All these strings should
   be permanently available (e.g., constants in global data memory).

   Help_interface_routine() is called when the user types "help".  It
   look in a table for the appropriate help screen, and prints it.
=================================================================== */

typedef struct help_screen_info_struct {
  struct help_screen_info_struct *next;
  char *topic;
  char **lines_of_text;
} help_screen_info;

help_screen_info *available_helpscreens = NIL;

help_screen_info *lookup_helpscreen (char *topic) {
  help_screen_info *hsi;

  for (hsi=available_helpscreens; hsi!=NIL; hsi=hsi->next)
    if (! strcmp (topic, hsi->topic)) return hsi;
  return NIL;
}

void add_help (char *topic, char **lines_of_text) {
  help_screen_info *hsi, *prev;

  if (lookup_helpscreen (topic)) {
    print ("Internal error: attempt to add_help to existing topic %s\n",
           topic);
    return;
  }
  hsi = allocate_memory (sizeof(help_screen_info), MISCELLANEOUS_MEM_USAGE);
  hsi->topic = topic;
  hsi->lines_of_text = lines_of_text;
  /* --- insert into list available_helpscreens in alphabetical order --- */
  if ((! available_helpscreens) ||
      (strcmp (topic, available_helpscreens->topic) < 0)) {
    hsi->next = available_helpscreens;
    available_helpscreens = hsi;
  } else {
    for (prev = available_helpscreens; prev->next!=NIL; prev=prev->next)
      if (strcmp (topic, prev->next->topic) < 0) break;
    hsi->next = prev->next;
    prev->next = hsi;
  }
}

char *help_on_help[] = {
"For help on a specific command, type either \"help\" or \"?\", followed by",
"the command name.",
"For a list of all available help topics, type \"list-help-topics\".",
"To print all the help screens to a file, type (print-all-help \"filename\").",
0 };

char *no_help_available[] = {
"No help on that subject is available.",
"For a list of all available help topics, type \"list-help-topics\".",
0 };

bool help_interface_routine (void) {
  help_screen_info *hsi;
  char **line;
  
  get_lexeme(); /* consume "help", look for topic name */
  if (lexeme.type!=R_PAREN_LEXEME) { /* get topic name */
    hsi = lookup_helpscreen (lexeme.string);
    get_lexeme(); /* consume topic */
  } else {
    /* if user didn't give a topic, give help on help */
    hsi = lookup_helpscreen ("help");
  }

  if (hsi) line = hsi->lines_of_text; else line = no_help_available;
  print ("\n");
  while (*line) {
    print ("%s\n", *line);
    line++;
  }
  
  return TRUE;
}

char *help_on_list_help_topics[] = {
"Command: list-help-topics",
"",
"Syntax: (list-help-topics)",
"",
"This prints out the names of all topics on which help information is",
"available.",
0 };

bool list_help_topics_interface_routine (void) {
  help_screen_info *hsi;

  print ("\nHelp is available on the following topics:\n\n");
  for (hsi=available_helpscreens; hsi!=NIL; hsi=hsi->next) {
    if (get_printer_output_column()+strlen(hsi->topic)+2 >= COLUMNS_PER_LINE)
      print_string ("\n");
    print_string (hsi->topic);
    if (hsi->next) print_string (", ");
  }
  print_string ("\n");
  get_lexeme(); /* consume "list-help-topics" */
  return TRUE;
}

char *help_on_print_all_help[] = {
"Command: print-all-help",
"",
"Syntax: (print-all-help \"filename\")",
"",
"This prints all available help screens to the indicated file.  This is a",
"quick way to produce a reference manual (well, sort of).",
0 };

bool print_all_help_interface_routine (void) {
  FILE *output_file;
  help_screen_info *hsi;
  char **line;

  get_lexeme();  /* consume "print-all-help" */

  /* --- look for filename --- */
  if (lexeme.type!=QUOTED_STRING_LEXEME) {
    print ("Expected string in quotes for filename\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }

  /* --- open the file --- */
  output_file = fopen (lexeme.string, "w");
  if (!output_file) {
    /* --- error when opening the file --- */
    print ("Error: unable to open file %s\n", lexeme.string);
    return FALSE;
  }
  print ("Writing help screens to file %s\n", lexeme.string);

  /* --- print each helpscreen --- */
  for (hsi=available_helpscreens; hsi!=NIL; hsi=hsi->next) {
    fprintf (output_file, "TOPIC:  %s\n\n", hsi->topic);
    line = hsi->lines_of_text;
    while (*line) {
      fprintf (output_file, "%s\n", *line);
      line++;
    }
    fprintf (output_file, "------------------------------------");
    fprintf (output_file, "------------------------------------\n");
  }

  /* --- clean up and exit --- */
  fclose (output_file);
  get_lexeme();  /* consume filename */
  return TRUE;
}

/* ===================================================================
                       Get Context Var Info

   This utility routine is used by interface routines that take context
   variable arguments (e.g., <s> for the current state).  It looks at
   the current lexeme (which must be of type VARIABLE_LEXEME), and
   checks to see if it's a context variable.  Returns:

    if lexeme is not a context variable, dest_attr_of_slot=NIL; else
      dest_attr_of_slot = {goal_symbol, problem_space_symbol, etc.}.
      dest_goal = goal identifier for the given slot (NIL if no such goal)
      dest_current_value = currently installed value (goal id itself for goals,
                           NIL if no installed value)
=================================================================== */

void get_context_var_info (symbol **dest_goal,
                           symbol **dest_attr_of_slot,
                           symbol **dest_current_value) {
  symbol *v, *g;
  int levels_up;
  wme *w;
  
  v = find_variable (lexeme.string);
  if (v==g_context_variable) {
    levels_up = 0;
    *dest_attr_of_slot = goal_symbol;
  } else if (v==p_context_variable) {
    levels_up = 0;
    *dest_attr_of_slot = problem_space_symbol;
  } else if (v==s_context_variable) {
    levels_up = 0;
    *dest_attr_of_slot = state_symbol;
  } else if (v==o_context_variable) {
    levels_up = 0;
    *dest_attr_of_slot = operator_symbol;
  } else if (v==sg_context_variable) {
    levels_up = 1;
    *dest_attr_of_slot = goal_symbol;
  } else if (v==sp_context_variable) {
    levels_up = 1;
    *dest_attr_of_slot = problem_space_symbol;
  } else if (v==ss_context_variable) {
    levels_up = 1;
    *dest_attr_of_slot = state_symbol;
  } else if (v==so_context_variable) {
    levels_up = 1;
    *dest_attr_of_slot = operator_symbol;
  } else if (v==ssg_context_variable) {
    levels_up = 2;
    *dest_attr_of_slot = goal_symbol;
  } else if (v==ssp_context_variable) {
    levels_up = 2;
    *dest_attr_of_slot = problem_space_symbol;
  } else if (v==sss_context_variable) {
    levels_up = 2;
    *dest_attr_of_slot = state_symbol;
  } else if (v==sso_context_variable) {
    levels_up = 2;
    *dest_attr_of_slot = operator_symbol;
  } else if (v==tg_context_variable) {
    levels_up = top_goal ? bottom_goal->id.level-top_goal->id.level : 0;
    *dest_attr_of_slot = goal_symbol;
  } else if (v==tp_context_variable) {
    levels_up = top_goal ? bottom_goal->id.level-top_goal->id.level : 0;
    *dest_attr_of_slot = problem_space_symbol;
  } else if (v==ts_context_variable) {
    levels_up = top_goal ? bottom_goal->id.level-top_goal->id.level : 0;
    *dest_attr_of_slot = state_symbol;
  } else if (v==to_context_variable) {
    levels_up = top_goal ? bottom_goal->id.level-top_goal->id.level : 0;
    *dest_attr_of_slot = operator_symbol;
  } else {
    *dest_goal = NIL;
    *dest_attr_of_slot = NIL;
    *dest_current_value = NIL;
    return;
  }

  g = bottom_goal;
  while (g && levels_up) {
    g = g->id.higher_goal;
    levels_up--;
  }
  *dest_goal = g;

  if (!g) {
    *dest_current_value = NIL;
    return;
  }
  
  if (*dest_attr_of_slot==goal_symbol) {
    *dest_current_value = g;
  } else {
    if (*dest_attr_of_slot==problem_space_symbol)
      w = g->id.problem_space_slot->wmes;
    else if (*dest_attr_of_slot==state_symbol)
      w = g->id.state_slot->wmes;
    else
      w = g->id.operator_slot->wmes;
    *dest_current_value = w ? w->value : NIL;
  }
}

/* ===================================================================
                  Read Identifier or Context Variable

   Many interface routines take identifiers as arguments.  These ids
   can be given as normal ids, or as special variables such as <s> for
   the current state, etc.  This routine reads (without consuming it)
   an identifier or context variable, and returns a pointer (symbol *)
   to the id.  (In the case of context variables, the instantiated
   variable is returned.  If any error occurs (e.g., no such id, no
   instantiation of the variable), an error message is printed and
   NIL is returned.
=================================================================== */

symbol *read_identifier_or_context_variable (void) {
  symbol *id;
  symbol *g, *attr, *value;

  if (lexeme.type==IDENTIFIER_LEXEME) {
    id = find_identifier (lexeme.id_letter, lexeme.id_number);
    if (!id) {
      print ("There is no identifier %c%lu.\n", lexeme.id_letter,
             lexeme.id_number);
      print_location_of_most_recent_lexeme();
      return NIL;
    }
    return id;
  }
  if (lexeme.type==VARIABLE_LEXEME) {
    get_context_var_info (&g, &attr, &value);
    if (!attr) {
      print ("Expected identifier (or context variable)\n");
      print_location_of_most_recent_lexeme();
      return NIL;
    }
    if (!value) {
      print ("There is no current %s.\n", lexeme.string);
      print_location_of_most_recent_lexeme();
      return NIL;
    }
    if (value->common.symbol_type!=IDENTIFIER_SYMBOL_TYPE) {
      print ("The current %s ", lexeme.string);
      print_with_symbols ("(%y) is not an identifier.\n", value);
      print_location_of_most_recent_lexeme();
      return NIL;
    }
    return value;
  }
  print ("Expected identifier (or context variable)\n");
  print_location_of_most_recent_lexeme();
  return NIL;
}

/* ===================================================================
                Read Pattern And Get Matching Wmes

   This routine reads a pattern and returns a list of all wmes that
   match it.  At entry, the current lexeme should be the "("; at exit,
   the current lexeme will be the ")".  If any error occurs or if no
   wmes match the pattern, the function returns NIL.

   pattern ::= ( {identifier | '*'} ^ { attribute | '*'} { value | '*' } [+])

=================================================================== */

int read_pattern_component (symbol **dest_sym) {
  /* --- Read and consume one pattern element.  Return 0 if error, 1 if "*",
     otherwise return 2 and set dest_sym to find_symbol() result. --- */
  if (strcmp(lexeme.string,"*") == 0) return 1;
  switch (lexeme.type) {
  case SYM_CONSTANT_LEXEME:
    *dest_sym = find_sym_constant (lexeme.string); return 2;
  case INT_CONSTANT_LEXEME:
    *dest_sym = find_int_constant (lexeme.int_val); return 2;
  case FLOAT_CONSTANT_LEXEME:
    *dest_sym = find_float_constant (lexeme.float_val); return 2;
  case IDENTIFIER_LEXEME:
    *dest_sym = find_identifier (lexeme.id_letter, lexeme.id_number); return 2;
  case VARIABLE_LEXEME:
    *dest_sym = read_identifier_or_context_variable();
    if (*dest_sym) return 2;
    return 0;
  default:
    print ("Expected identifier or constant in wme pattern\n");
    print_location_of_most_recent_lexeme();
    return 0;
  }
}

list *read_pattern_and_get_matching_wmes (void) {
  int parentheses_level;
  list *wmes;
  wme *w;
  symbol *id, *attr, *value;
  int id_result, attr_result, value_result;
  bool acceptable;
  
  if (lexeme.type!=L_PAREN_LEXEME) {
    print ("Expected '(' to begin wme pattern\n");
    print_location_of_most_recent_lexeme();
    return NIL;
  }
  parentheses_level = current_lexer_parentheses_level();

  get_lexeme();
  id_result = read_pattern_component (&id);
  if (! id_result) {
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return NIL;
  }
  get_lexeme();
  if (lexeme.type!=UP_ARROW_LEXEME) {
    print ("Expected ^ in wme pattern\n");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return NIL;
  }
  get_lexeme();
  attr_result = read_pattern_component (&attr);
  if (! attr_result) {
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return NIL;
  }
  get_lexeme();
  value_result = read_pattern_component (&value);
  if (! value_result) {
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return NIL;
  }
  get_lexeme();
  if (lexeme.type==PLUS_LEXEME) {
    acceptable = TRUE;
    get_lexeme();
  } else {
    acceptable = FALSE;
  }
  if (lexeme.type!=R_PAREN_LEXEME) {
    print ("Expected ')' to end wme pattern\n");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
    return NIL;
  }

  wmes = NIL;
  for (w=all_wmes_in_rete; w!=NIL; w=w->rete_next) {
    if ((id_result==1) || (id==w->id))
      if ((attr_result==1) || (attr==w->attr))
        if ((value_result==1) || (value==w->value))
          if (acceptable==w->acceptable)
            push (w, wmes);
  }
  return wmes;  
}

/* ===================================================================
   
                   Built-In User Interface Commands

=================================================================== */

/* -------------------------------------------------------------------
   
                          "Log" Command

   Syntax: (log ["filename" [:append]])
------------------------------------------------------------------- */

char *help_on_log[] = {
"Command: log",
"",
"Syntax: (log [\"filename\" [:append]])",
"",
"The log command turns on and off logging to a file.  When Soar is logging",
"to a file, everything you type and everything Soar prints is written to",
"the file (in addition to the screen).  This is like the (dribble) function",
"in Common Lisp.",
"",
"To start a new log file, type (log \"filename\").",
"To append to an existing file, type (log \"filename\" :append).",
"To stop logging to the current file, type (log).",
0 };

bool log_interface_routine (void) {
  char log_file_name[MAX_LEXEME_LENGTH+1];
  bool append;
  
  if (logging_to_file) stop_log_file (); /* close existing log */
  get_lexeme();  /* consume "log" */
  if (lexeme.type==R_PAREN_LEXEME) return TRUE;
  
  if (lexeme.type!=QUOTED_STRING_LEXEME) {
    print ("Expected string in quotes for log filename\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  strcpy (log_file_name, lexeme.string);
  get_lexeme();
  append = FALSE;
  if (!strcmp(lexeme.string, ":append")) { append=TRUE; get_lexeme(); }

  start_log_file (log_file_name, append);
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Load" Command

   Syntax:  (load "filename")
------------------------------------------------------------------- */

char *help_on_load[] = {
"Command: load",
"",
"Syntax: (load \"filename\")",
"",
"Load tells Soar to read commands from the given file instead of the",
"keyboard.  Soar will read and execute each command in the file, and then",
"go back to the keyboard.  Loads may be nested; i.e., the given file may",
"contain a command to load another file, and so on.",
0 };

bool load_interface_routine (void) {
  FILE *f;
  char *expanded_string;
  char *tilde_expand();
  
  get_lexeme();  /* consume "load", advance to quoted file name */
  if (lexeme.type!=QUOTED_STRING_LEXEME) {
    print ("Expected string in quotes for filename to load\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  expanded_string = tilde_expand(lexeme.string);
  f = fopen (expanded_string,"r");
  if (!f) {
    /* --- error when opening the file --- */
    print ("Error: unable to open file %s\n",lexeme.string);
    return FALSE;
  }
  print ("\nLoading %s\n",lexeme.string);
  load_file (expanded_string, f);
  fclose (f);
  print ("\n");
  free(expanded_string);
  get_lexeme();  /* consume filename, advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                      "Chdir" and "cd" Commands

   Syntax:  (chdir "pathname") or (cd "pathname")
------------------------------------------------------------------- */

/* AUTHOR: SCOTT HUFFMAN
 * CREATED: 8-31-92
 *
 * User interface command to allow changing the current (to-be-loaded-from)
 * directory. 
 *
 */

char *help_on_chdir[] = {
"Commands: chdir, cd",
"",
"Syntax: (chdir \"path\") or (cd \"path\")",
"",
"Change the current directory (which files will be loaded from) to the specified",
"directory.",
"",
"See also:  pwd",
0 };

bool chdir_interface_routine (void) {
  int chdir_res;
  
  get_lexeme();  /* consume "chdir", advance to quoted path name */
  if (lexeme.type!=QUOTED_STRING_LEXEME) {
    print ("Expected string in quotes for directory pathname\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  print ("Changing to directory: %s\n", lexeme.string);

  chdir_res =  chdir(lexeme.string);
  if (chdir_res)
    printf("  FAILED.\n");

  get_lexeme();  /* consume pathname, advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                           "Pwd" Command

   Syntax:  (pwd)
------------------------------------------------------------------- */

char *help_on_pwd[] = {
"Command: pwd",
"",
"Syntax: (pwd)",
"",
"[Print Working Directory]  Prints the current working directory.",
"",
"See also:  chdir",
0 };

bool pwd_interface_routine (void) {
  char pathname[10000];
  char *getwd_result;
  
  get_lexeme();  /* consume "pwd" */

  if (getwd (pathname)) {
    print ("Current directory: %s\n", pathname);
  } else {
    print ("Error: unable to determine current working directory.\n");
  }
  
  return TRUE;
}

/* -------------------------------------------------------------------
   
                    "Exit" and "Quit" Commands

   Syntax: (exit) or (quit)
------------------------------------------------------------------- */

char *help_on_exit[] = {
"Commands: exit, quit",
"",
"Syntax: (exit) or (quit)",
"",
"These two commands are synonymous; they cause Soar to terminate and return",
"control to the shell.",
0 };

bool exit_interface_routine (void) {
  print ("Exiting Soar...\n");
  exit_soar();
  return FALSE; /* unreachable, but without it, gcc -Wall warns here */
}

/* -------------------------------------------------------------------
   
                            "Pgs" Command

   Syntax: (pgs)
------------------------------------------------------------------- */

char *help_on_pgs[] = {
"Command: pgs",
"",
"Syntax: (pgs)",
"",
"Pgs (\"print goal stack\") prints Soar's current context stack.",
0 };

bool pgs_interface_routine (void) {
  symbol *g;

  for (g=top_goal; g!=NIL; g=g->id.lower_goal) {
    print_stack_trace (g, g, FOR_GOALS_TF, FALSE);
    print ("\n");
    if (g->id.problem_space_slot->wmes) {
      print_stack_trace (g->id.problem_space_slot->wmes->value,
                         g, FOR_PROBLEM_SPACES_TF, FALSE);
      print ("\n");
    }
    if (g->id.state_slot->wmes) {
      print_stack_trace (g->id.state_slot->wmes->value,
                         g, FOR_STATES_TF, FALSE);
      print ("\n");
    }
    if (g->id.operator_slot->wmes) {
      print_stack_trace (g->id.operator_slot->wmes->value,
                         g, FOR_OPERATORS_TF, FALSE);
      print ("\n");
    }
  }
  get_lexeme();  /* consume "pgs", advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                  "Excise" and "Excise-xxx" Commands

   Syntax: (excise production-name*)
           (excise-chunks)
           (excise-task)
           (excise-all)
------------------------------------------------------------------- */

char *help_on_excise[] = {
"Command: excise",
"",
"Syntax: (excise production-name*)",
"",
"This command removes the given production(s) from the system.",
"",
"See also: excise-chunks, excise-task, excise-all",
0 };

char *help_on_excise_chunks[] = {
"Command: excise-chunks",
"",
"Syntax: (excise-chunks)",
"",
"This command removes all chunks and justifications from the system.",
"",
"See also: excise, excise-task, excise-all",
0 };

char *help_on_excise_task[] = {
"Command: excise-task",
"",
"Syntax: (excise-task)",
"",
"This command removes all non-default productions from the system.  It also",
"does an (init-soar).",
"",
"See also: excise, excise-chunks, excise-all",
0 };

char *help_on_excise_all[] = {
"Command: excise-all",
"",
"Syntax: (excise-all)",
"",
"This command removes all productions from the system.  It also does an",
"(init-soar)",
"",
"See also: excise, excise-chunks, excise-task",
0 };

bool excise_interface_routine (void) {
  symbol *sym;
  
  set_lexer_allow_ids (FALSE);
  get_lexeme();  /* consume "excise", advance to production name(s) */
  while (lexeme.type==SYM_CONSTANT_LEXEME) {
    sym = find_sym_constant (lexeme.string);
    if (sym && sym->sc.production) {
      excise_production (sym->sc.production, TRUE);
    } else {
      print ("No production named %s\n", lexeme.string);
      print_location_of_most_recent_lexeme();
    }
    get_lexeme(); /* consume this one, advance to next production name */
  }
  if (lexeme.type!=R_PAREN_LEXEME) {
    print ("Expected symbol for name of production to excise\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  return TRUE;
}

void excise_all_productions_of_type (byte type) {
  while (all_productions_of_type[type])
    excise_production (all_productions_of_type[type], TRUE);
}

bool excise_chunks_interface_routine (void) {
  excise_all_productions_of_type (CHUNK_PRODUCTION_TYPE);
  excise_all_productions_of_type (JUSTIFICATION_PRODUCTION_TYPE);
  get_lexeme();  /* consume "excise-chunks" */
  return TRUE;
}

bool excise_task_interface_routine (void) {
  excise_all_productions_of_type (USER_PRODUCTION_TYPE);
  excise_all_productions_of_type (CHUNK_PRODUCTION_TYPE);
  excise_all_productions_of_type (JUSTIFICATION_PRODUCTION_TYPE);
  reinitialize_soar();  /* for excise-task, also do an init-soar */
  get_lexeme();  /* consume "excise-task" */
  return TRUE;
}

bool excise_all_interface_routine (void) {
  excise_all_productions_of_type (DEFAULT_PRODUCTION_TYPE);
  excise_all_productions_of_type (USER_PRODUCTION_TYPE);
  excise_all_productions_of_type (CHUNK_PRODUCTION_TYPE);
  excise_all_productions_of_type (JUSTIFICATION_PRODUCTION_TYPE);
  reinitialize_soar();  /* for excise-all, also do an init-soar */
  get_lexeme();  /* consume "excise-all" */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Matches" Command

   Syntax: (matches production-name [ 0 | 1 | 2 ])
------------------------------------------------------------------- */

char *help_on_matches[] = {
"Command: matches",
"",
"Syntax: (matches production-name [ 0 | 1 | 2 ])",
"",
"This command prints partial match information for the given production.",
"The optional integer specifies the level of detail wanted:  0 (the default)",
"prints out just the partial match counts, ala 'smatches'; 1 also prints",
"the timetags of wmes at the first failing condition, ala 'full-matches';",
"and 2 prints the wmes rather than just their timetags.",
0 };

bool matches_interface_routine (void) {
  symbol *sym;
  wme_trace_type wtt;
  
  set_lexer_allow_ids (FALSE);
  get_lexeme();  /* consume "matches", advance to production name(s) */
  if (lexeme.type!=SYM_CONSTANT_LEXEME) {
    print ("Expected symbol for name of production for 'matches' command\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  sym = find_sym_constant (lexeme.string);
  if ((!sym) || (! sym->sc.production)) {
    print ("No production named %s\n", lexeme.string);
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme(); /* consume production name, look for level */
  wtt = NONE_WME_TRACE;
  if (lexeme.type==INT_CONSTANT_LEXEME) {
    if ((lexeme.int_val>=0) && (lexeme.int_val<=2)) {
      if (lexeme.int_val==0) wtt = NONE_WME_TRACE;
      if (lexeme.int_val==1) wtt = TIMETAG_WME_TRACE;
      if (lexeme.int_val==2) wtt = FULL_WME_TRACE;
      get_lexeme();
    } else {
      print ("Matches 'level' must be 0, 1, or 2.\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
  }
  print_partial_match_information (sym->sc.production->p_node, wtt);
  return TRUE;
}

/* -------------------------------------------------------------------
   
                    "Default-print-depth" Command
 
   Syntax:  (default-print-depth [integer])
------------------------------------------------------------------- */

int default_print_depth = 1;  /* global variable containing it */

char *help_on_default_print_depth[] = {
"Command: default-print-depth",
"",
"Syntax: (default-print-depth [integer])",
"",
"With no arguments, this command prints the current default print depth used",
"by the (print) command.  With an integer argument, it sets the current",
"default print depth.   This default print depth can be overridden on any",
"particular invocation of the (print) command by using the :depth flag,",
"e.g., (print :depth 10 args...).  The default print depth is initially 1.",
"",
"See also:  print",
0 };

bool default_print_depth_interface_routine (void) {
  get_lexeme();  /* consume "default-print-depth", advance to integer */
  if (lexeme.type==R_PAREN_LEXEME) {
    /* --- invoked with no arguments, so just print the default depth --- */
    print ("The current default print depth is %d.\n", default_print_depth);
    return TRUE;
  }
  if (lexeme.type!=INT_CONSTANT_LEXEME) {
    print ("Expected integer for new default print depth\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  default_print_depth = lexeme.int_val;
  get_lexeme(); /* consume the integer */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Print" Command

   Syntax:  see help screen below.
------------------------------------------------------------------- */

char *help_on_print[] = {
"Commands: print, p, spr, wm",
"",
"Syntax:  (print [:depth n] [:internal] arg*)",
"         (p ...) and (spr ...) are shorthand for (print ...)",
"         (wm ...) is shorthand for (print :depth 0 :internal ...)",
"",
"The print command is used to print items from production memory or working",
"memory.  It can take several kinds of arguments:",
"",
"  arg ::= production-name  (print that production)",
"  arg ::= identifier       (id of the object to print)",
"  arg ::= integer          (timetag of wme--the identifier from the wme",
"                            indicates the object to be printed)",
"  arg ::= pattern          (pattern--same as if you listed as arguments",
"                            the timetags of all wmes matching the pattern)",
"",
"  pattern ::= ( {identifier | '*'} ^ { attribute | '*'} { value | '*' } [+])",
"",
"The optional [:depth n] argument overrides default-print-depth.",
"",
"The optional [:internal] argument tells Soar to print things in their",
"internal form.  For productions, this means leaving conditions in their",
"reordered (rete net) form.  For wmes, this means printing the individual",
"wmes with their timetags, rather than the objects.",
"",
":depth 0 is meaningful only for integer and pattern arguments, and only",
"when used along with :internal.  It causes just the matching wmes to be",
"printed, instead of all wmes whose id is an id in one of the matching wmes.",
"",
"See also:  default-print-depth",
0 };

void neatly_print_wme_augmentation_of_id (wme *w, int indentation) {
  char buf[10000], *ch;

  strcpy (buf, " ^");
  ch = buf;
  while (*ch) ch++;
  symbol_to_string (w->attr, TRUE, ch); while (*ch) ch++;
  *(ch++) = ' ';
  symbol_to_string (w->value, TRUE, ch); while (*ch) ch++;
  if (w->acceptable) { strcpy (ch, " +"); while (*ch) ch++; }

  if (get_printer_output_column() + (ch - buf) >= 80) {
    print ("\n");
    print_spaces (indentation+6);
  }
  print_string (buf);
}

void print_augs_of_id (symbol *id, int depth, bool internal,
                       int indent, tc_number tc) {
  slot *s;
  wme *w;

  if (id->common.symbol_type != IDENTIFIER_SYMBOL_TYPE) return;
  if (id->id.tc_num==tc) return;
  id->id.tc_num = tc;

  /* --- first, print all direct augmentation of this id --- */
  if (internal) {
    for (w=id->id.impasse_wmes; w!=NIL; w=w->next)
      { print_spaces (indent); print_wme (w); print ("\n"); }
    for (w=id->id.input_wmes; w!=NIL; w=w->next)
      { print_spaces (indent); print_wme (w); print ("\n"); }
    for (s=id->id.slots; s!=NIL; s=s->next) {
      for (w=s->wmes; w!=NIL; w=w->next)
        { print_spaces (indent); print_wme (w); print ("\n"); }
      for (w=s->acceptable_preference_wmes; w!=NIL; w=w->next)
        { print_spaces (indent); print_wme (w); print ("\n"); }
    }
  } else {
    print_spaces (indent); 
    print_with_symbols ("(%y", id);
    for (w=id->id.impasse_wmes; w!=NIL; w=w->next)
      neatly_print_wme_augmentation_of_id (w, indent);
    for (w=id->id.input_wmes; w!=NIL; w=w->next)
      neatly_print_wme_augmentation_of_id (w, indent);
    for (s=id->id.slots; s!=NIL; s=s->next) {
      for (w=s->wmes; w!=NIL; w=w->next)
        neatly_print_wme_augmentation_of_id (w, indent);
      for (w=s->acceptable_preference_wmes; w!=NIL; w=w->next)
        neatly_print_wme_augmentation_of_id (w, indent);
    }
    print (")\n");
  }

  /* --- if depth<=1, we're done --- */
  if (depth<=1) return;

  /* --- call this routine recursively --- */
  for (w=id->id.input_wmes; w!=NIL; w=w->next) {
    print_augs_of_id (w->attr, depth-1, internal, indent+2, tc);
    print_augs_of_id (w->value, depth-1, internal, indent+2, tc);
  }
  for (w=id->id.impasse_wmes; w!=NIL; w=w->next) {
    print_augs_of_id (w->attr, depth-1, internal, indent+2, tc);
    print_augs_of_id (w->value, depth-1, internal, indent+2, tc);
  }
  for (s=id->id.slots; s!=NIL; s=s->next) {
    for (w=s->wmes; w!=NIL; w=w->next) {
      print_augs_of_id (w->attr, depth-1, internal, indent+2, tc);
      print_augs_of_id (w->value, depth-1, internal, indent+2, tc);
    }
    for (w=s->acceptable_preference_wmes; w!=NIL; w=w->next) {
      print_augs_of_id (w->attr, depth-1, internal, indent+2, tc);
      print_augs_of_id (w->value, depth-1, internal, indent+2, tc);
    }
  }
}

void do_print_for_production_name (char *prod_name, bool internal) {
  symbol *sym;
  
  sym = find_sym_constant (lexeme.string);
  if (sym && sym->sc.production) {
    print_production (sym->sc.production, internal);
    print ("\n");
  } else {
    print ("No production named %s\n", prod_name);
    print_location_of_most_recent_lexeme();
  }
}

void do_print_for_identifier (symbol *id, int depth, bool internal) {
  tc_number tc;

  tc = get_new_tc_number();
  print_augs_of_id (id, depth, internal, 0, tc);
}

void do_print_for_wme (wme *w, int depth, bool internal) {
  tc_number tc;
  
  if (internal && (depth==0)) {
    print_wme (w);
    print ("\n");
  } else {
    tc = get_new_tc_number();
    print_augs_of_id (w->id, depth, internal, 0, tc);
  }
}

bool print_interface_routine (void) {
  bool internal;
  int depth;
  symbol *id;
  wme *w;
  list *wmes;
  cons *c;

  internal = FALSE;
  depth = default_print_depth;

  /* --- if the user typed "wm", change initial internal, depth values --- */
  if (!strcmp(lexeme.string,"wm")) {
    internal = TRUE;
    depth = 0;
  }

  get_lexeme();  /* consume command name, advance to optional flags */

  /* --- read optional :depth and :internal flags --- */
  while (TRUE) {
    if (lexeme.type!=SYM_CONSTANT_LEXEME) break;
    if (!strcmp(lexeme.string,":depth")) {
      get_lexeme();
      if (lexeme.type!=INT_CONSTANT_LEXEME) {
        print ("Expected integer for value of :depth argument\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      }
      depth = lexeme.int_val;
      get_lexeme();
      continue;
    }
    if (!strcmp(lexeme.string,":internal")) {
      internal = TRUE;
      get_lexeme();
      continue;
    }
    break;
  }

  /* --- repeat: read one arg and print it --- */
  while (lexeme.type!=R_PAREN_LEXEME) {
    switch (lexeme.type) {
    case SYM_CONSTANT_LEXEME:
      do_print_for_production_name (lexeme.string, internal);
      get_lexeme();
      break;
      
    case INT_CONSTANT_LEXEME:
      for (w=all_wmes_in_rete; w!=NIL; w=w->rete_next)
        if (w->timetag == lexeme.int_val) break;
      if (w) {
        do_print_for_wme (w, depth, internal);
      } else {
        print ("No wme %ld in working memory\n", lexeme.int_val);
      }
      get_lexeme();
      break;

    case IDENTIFIER_LEXEME:
    case VARIABLE_LEXEME:
      id = read_identifier_or_context_variable();
      if (id) do_print_for_identifier (id, depth, internal);
      get_lexeme();
      break;
      
    case L_PAREN_LEXEME:
      wmes = read_pattern_and_get_matching_wmes ();
      for (c=wmes; c!=NIL; c=c->rest)
        do_print_for_wme (c->first, depth, internal);
      free_list (wmes);
      get_lexeme();
      break;

    default:
      print ("Illegal argument to 'print' command\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    } /* end of switch statement */
  } /* end of while loop */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "D" Command

   Syntax:  (d [integer])
------------------------------------------------------------------- */

char *help_on_d[] = {
"Command: d",
"",
"Syntax: (d [integer])",
"",
"With an integer argument, this command runs Soar for that number of decision",
"cycles.  With no arguments, it runs Soar forever (or until Soar halts,",
"receives an interrupt, etc.).",
"",
"See also:  run, go",
0 };

bool d_interface_routine (void) {
  long num_requested;

  get_lexeme();  /* consume "d" */
  if (lexeme.type==INT_CONSTANT_LEXEME) {
    num_requested = lexeme.int_val;
    get_lexeme(); /* consume the integer */
  } else if (lexeme.type==R_PAREN_LEXEME) { 
    num_requested = -1;
  } else {
    print ("Bad argument for 'd' command.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  run_for_n_decision_cycles (num_requested);
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Run" and "R" Commands

   Syntax:  (run [integer])
            (r ...) is shorthand for (run ...)
------------------------------------------------------------------- */

char *help_on_run[] = {
"Commands: run, r",
"",
"Syntax: (run [integer])",
"        (r ...) is shorthand for (run ...)",
"",
"With an integer argument, this command runs Soar for that number of",
"elaboration cycles.  (For this command, quiescence phase is counted as an",
"elaboration cycle.)  With no arguments, this runs Soar forever (or until",
"Soar halts, receives an interrupt, etc.).",
"",
"See also:  d, go",
0 };

bool run_interface_routine (void) {
  long num_requested;

  get_lexeme();  /* consume "run" */
  if (lexeme.type==INT_CONSTANT_LEXEME) {
    num_requested = lexeme.int_val;
    get_lexeme(); /* consume the integer */
  } else if (lexeme.type==R_PAREN_LEXEME) { 
    num_requested = -1;
  } else {
    print ("Bad argument for 'run' command.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  run_for_n_elaboration_cycles (num_requested);
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Go" Command

   Syntax:  (go [integer | 'forever'] [type])
            type ::= p | e | d | g | ps | s | o | context-variable
------------------------------------------------------------------- */

char *help_on_go[] = {
"Command: go",
"",
"Syntax: (go [integer | 'forever'] [type])",
"        type ::= 'p' | 'e' | 'd' | 'g' | 'ps' | 's' | 'o' | context-variable",
"",
"This is the most general command for running Soar.  It takes two optional",
"arguments, one specifying how many things to run, and one specifying what",
"type of things to run.  The following types are available:",
"",
"p - run Soar for n phases.  A phase is either an input phase, preference",
"    phase, working memory phase, output phase, or quiescence phase.",
"e - run Soar for n elaboration cycles.  (For purposes of this command,",
"    quiescence phase is counted as an elaboration cycle.)",
"d - run Soar for n decision cycles.",
"g - run Soar until the nth time a goal is selected.",
"ps - run Soar until the nth time a problem space is selected.",
"s - run Soar until the nth time a state is selected.",
"o - run Soar until the nth time an operator is selected.",
"context-variable - run Soar until the nth time a selection is made for that",
"    particular context slot, or until the context stack pops to above that",
"    context.",
"",
"Go remembers each argument you give it each time.  If you don't give",
"arguments next time, it uses the ones from the previous time.",
"",
"Examples:",
"  (go 5 d)  --> run for 5 decision cycles",
"  (go e)    --> run for another 5 elaboration cycles",
"  (go 1 g)  --> run until the next goal is selected (i.e., until the next",
"                time an impasse arises)",
"  (go <so>) --> run until the next superoperator is selected (or until the",
"                supergoal goes away)",
"  (go 3 <o>) --> run for 3 operator selections at this level (continuing",
"                 through any subgoals that arise)",
"",
"See also:  d, run",
0 };

long go_number = 1;
enum go_type_enum { GO_PHASE, GO_ELABORATION, GO_DECISION,
                    GO_GOAL, GO_PROBLEM_SPACE, GO_STATE, GO_OPERATOR,
                    GO_SLOT } go_type = GO_DECISION;
goal_stack_level go_slot_level;
symbol *go_slot_attr;

bool go_interface_routine (void) {
  symbol *g, *attr, *value;
  
  get_lexeme();  /* consume "go" */
  while (TRUE) {
    if (lexeme.type==INT_CONSTANT_LEXEME) {
      go_number = lexeme.int_val;
      get_lexeme();
      continue;
    }
    if (lexeme.type==SYM_CONSTANT_LEXEME) {
      if (!strcmp(lexeme.string,"forever")) {
        go_number = -1;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"p")) {
        go_type = GO_PHASE;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"e")) {
        go_type = GO_ELABORATION;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"d")) {
        go_type = GO_DECISION;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"g")) {
        go_type = GO_GOAL;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"ps")) {
        go_type = GO_PROBLEM_SPACE;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"s")) {
        go_type = GO_STATE;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"o")) {
        go_type = GO_OPERATOR;
        get_lexeme();
        continue;
      }
    }
    if (lexeme.type==VARIABLE_LEXEME) {
      get_context_var_info (&g, &attr, &value);
      if (!attr) {
        print ("Expected a context variable.\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      }
      if (!g) {
        print ("That goal stack level doesn't exist right now.\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      }
      go_type = GO_SLOT;
      go_slot_level = g->id.level;
      go_slot_attr = attr;
      get_lexeme();
      continue;
    }
    break; /* if it didn't match anything so far, break out of the loop */
  } /* end of while (TRUE) */

  if (lexeme.type != R_PAREN_LEXEME) {
    print ("Bad argument for 'go' command.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }

  switch (go_type) {
  case GO_PHASE:
    run_for_n_phases (go_number);
    break;
  case GO_ELABORATION:
    run_for_n_elaboration_cycles (go_number);
    break;
  case GO_DECISION:
    run_for_n_decision_cycles (go_number);
    break;
  case GO_GOAL:
    run_for_n_selections_of_slot (go_number, goal_symbol);
    break;
  case GO_PROBLEM_SPACE:
    run_for_n_selections_of_slot (go_number, problem_space_symbol);
    break;
  case GO_STATE:
    run_for_n_selections_of_slot (go_number, state_symbol);
    break;
  case GO_OPERATOR:
    run_for_n_selections_of_slot (go_number, operator_symbol);
    break;
  case GO_SLOT:
    run_for_n_selections_of_slot_at_level (go_number, go_slot_attr,
                                           go_slot_level);
    break;
  }
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Init-Soar" Command

   Syntax:  (init-soar)
------------------------------------------------------------------- */

char *help_on_init_soar[] = {
"Command: init-soar",
"",
"Syntax: (init-soar)",
"",
"This command re-initializes Soar.  It removes all wmes from working memory,",
"wiping out the goal stack, and resets all statistics (except the counts of",
"how many times each individual production has fired, used by the",
"\"firing-counts\" command).",
0 };

bool init_soar_interface_routine (void) {
  reinitialize_soar();
  get_lexeme();  /* consume "init-soar", advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                            "Learn" Command

   Syntax:  (learn arg* )
            arg  ::=  on | off
            arg  ::=  all-goals | bottom-up
            arg  ::=  noprint | print | full-print
            arg  ::=  notrace | trace | full-trace
------------------------------------------------------------------- */

char *help_on_learn[] = {
"Command: learn",
"",
"Syntax: (learn arg* )",
"        arg  ::=  on | off",
"        arg  ::=  all-goals | bottom-up",
"        arg  ::=  noprint | print | full-print",
"        arg  ::=  notrace | trace | full-trace",
"",
"With no arguments, this command prints out the current learning status.",
"Any of the following arguments may be given:",
"   on         - turns learning on",
"   off        - turns all learning off",
"   all-goals  - when learning is on, this allows learning at all goal stack",
"                levels (in contrast to bottom-up learning)",
"   bottom-up  - when learning is on, this allows learning at only the lowest",
"                goal stack level; i.e., a chunk is learned at a given level",
"                only if no chunk has yet been learned at a lower level.",
"   noprint    - equivalent to (watch :chunk-names off :chunks off)",
"   print      - equivalent to (watch :chunk-names  on :chunks off)",
"   full-print - equivalent to (watch :chunk-names  on :chunks  on)",
"   notrace    - equivalent to (watch :firings chunk off :backtracing off)",
"   trace      - equivalent to (watch :firings chunk  on :backtracing off)",
"   full-trace - equivalent to (watch :firings chunk  on :backtracing  on)",
"",
"See also: chunk-free-problem-spaces, watch",
0 };

bool learn_interface_routine (void) {
  get_lexeme();  /* consume "learn" */
  
  if (lexeme.type==R_PAREN_LEXEME) {
    print ("Learning status:  %s, %s\n",
           sysparams[LEARNING_ON_SYSPARAM] ? "on" : "off",
           sysparams[LEARNING_ALL_GOALS_SYSPARAM] ? "all-goals" : "bottom-up");
    return TRUE;
  }

  while (lexeme.type!=R_PAREN_LEXEME) {
    if (lexeme.type==SYM_CONSTANT_LEXEME) {
      if (!strcmp(lexeme.string,"on")) {
        set_sysparam (LEARNING_ON_SYSPARAM, TRUE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"off")) {
        set_sysparam (LEARNING_ON_SYSPARAM, FALSE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"all-goals")) {
        set_sysparam (LEARNING_ALL_GOALS_SYSPARAM, TRUE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"bottom-up")) {
        set_sysparam (LEARNING_ALL_GOALS_SYSPARAM, FALSE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"noprint")) {
        set_sysparam (TRACE_CHUNK_NAMES_SYSPARAM, FALSE);
        set_sysparam (TRACE_CHUNKS_SYSPARAM, FALSE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"print")) {
        set_sysparam (TRACE_CHUNK_NAMES_SYSPARAM, TRUE);
        set_sysparam (TRACE_CHUNKS_SYSPARAM, FALSE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"full-print")) {
        set_sysparam (TRACE_CHUNK_NAMES_SYSPARAM, TRUE);
        set_sysparam (TRACE_CHUNKS_SYSPARAM, TRUE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"notrace")) {
        set_sysparam (TRACE_FIRINGS_OF_CHUNKS_SYSPARAM, FALSE);
        set_sysparam (TRACE_BACKTRACING_SYSPARAM, FALSE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"trace")) {
        set_sysparam (TRACE_FIRINGS_OF_CHUNKS_SYSPARAM, TRUE);
        set_sysparam (TRACE_BACKTRACING_SYSPARAM, FALSE);
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"full-trace")) {
        set_sysparam (TRACE_FIRINGS_OF_CHUNKS_SYSPARAM, TRUE);
        set_sysparam (TRACE_BACKTRACING_SYSPARAM, TRUE);
        get_lexeme();
        continue;
      }
    }
    print ("Error: unrecognized argument to 'learn' command\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  return TRUE;
}

/* -------------------------------------------------------------------
   
                    "Chunk-free-problem-spaces" Command

   Syntax:  (chunk-free-problem-spaces)
            (chunk-free-problem-spaces :add space-name)
            (chunk-free-problem-spaces :remove space-name)
------------------------------------------------------------------- */

char *help_on_chunk_free_problem_spaces[] = {
"Command: chunk-free-problem-spaces",
"",
"Syntax: (chunk-free-problem-spaces)",
"        (chunk-free-problem-spaces :add space-name)",
"        (chunk-free-problem-spaces :remove space-name)",
"",
"With no arguments, this command prints the current list of problem spaces",
"declared chunk-free.  With arguments, it adds or removes a given problem",
"space from this list.  No chunks will be built in a problem space if that",
"space is declared chunk-free.",
"",
"See also: learn",
0 };

symbol *space_to_remove_from_cfps;

bool cfps_removal_test_function (cons *c) {
  return (c->first == space_to_remove_from_cfps);
}

bool chunk_free_problem_spaces_interface_routine (void) {
  cons *c;
  symbol *sym;
  list *extracted_stuff;
  
  get_lexeme();  /* consume "chunk-free-problem-spaces" */
  
  if (lexeme.type==R_PAREN_LEXEME) {
    print ("Problem spaces declared chunk-free:\n");
    for (c=chunk_free_problem_spaces; c!=NIL; c=c->rest)
      print_with_symbols ("  %y\n", (symbol *)(c->first));
    return TRUE;
  }

  if (lexeme.type==SYM_CONSTANT_LEXEME) {
    if (!strcmp(lexeme.string,":add")) {
      get_lexeme();  /* consume ":add", advance to the space name */
      if (lexeme.type!=SYM_CONSTANT_LEXEME) {
        print ("Error: expected a symbol for the problem space name\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      }
      sym = make_sym_constant (lexeme.string);
      if (! member_of_list (sym, chunk_free_problem_spaces)) {
        symbol_add_ref (sym);
        push (sym, chunk_free_problem_spaces);
      }
      symbol_remove_ref (sym);
      get_lexeme();
      return TRUE;
    }
    if (!strcmp(lexeme.string,":remove")) {
      get_lexeme();  /* consume ":remove", advance to the space name */
      if (lexeme.type!=SYM_CONSTANT_LEXEME) {
        print ("Error: expected a symbol for the problem space name\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      }
      sym = find_sym_constant (lexeme.string);
      if (sym) {
        space_to_remove_from_cfps = sym;
        extracted_stuff = extract_list_elements (&chunk_free_problem_spaces,
                                                 cfps_removal_test_function);
        deallocate_symbol_list_removing_references (extracted_stuff);
      }
      get_lexeme();
      return TRUE;
    }
  }
  print ("Error: unrecognized argument to 'chunk-free-problem-spaces'\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

/* -------------------------------------------------------------------
   
                          "Watch" Command

   Syntax:  see helpscreen below
------------------------------------------------------------------- */

char *help_on_watch[] = {
"Command: watch",
"",
"Syntax: (watch arg*)",
"        arg  ::=  -1 | 0 | 0.5 | 1 | 1.5 | 2 | 3 | task",
"        arg  ::=  :context {on|off}", 
"        arg  ::=  :phases {on|off}", 
"        arg  ::=  :firings [default|user|chunk|nonchunk|all] {on|off}",
"        arg  ::=     :firings-wmes {0|1|2}",
"        arg  ::=     :firings-preferences {on|off}",
"        arg  ::=  :wmes {on|off}",
"        arg  ::=  :chunk-names {on|off}",
"        arg  ::=  :justification-names {on|off}",
"        arg  ::=  :chunks {on|off}",
"        arg  ::=  :justifications {on|off}",
"        arg  ::=  :backtracing {on|off}",
"",
"This command controls what information gets printed in the run-time trace.",
"With no arguments, it just prints out the current watch status.  The numeric",
"arguments have roughly the same semantics as in Soar 5; for details, see",
"(help watch-levels).  The various :keyword arguments are used to modify the",
"current watch settings.  For example, (watch :context on) turns on the",
"tracing of context slot decisions; (watch :context off) turns it off again.",
"For information about what each keyword does, see (help watch-keywords).",
"",
"See also:  watch-keywords, watch-levels, learn, ptrace",
0 };

char *help_on_watch_keywords[] = {
"The following keyword arguments may be given to the 'watch' command:",
"",
"  :context -- controls whether context slot decisions are printed ",
"  :phases -- controls whether phase names are printed",
"  :firings [default|user|chunk|nonchunk|all] -- controls which production",
"      firings and retractions are printed.  The optional argument (which",
"      defaults to 'all') specifies which types of productions the 'on/off'",
"      argument refers to.",
"  :firings-wmes -- controls the level of detail given about the wmes matched",
"      by productions whose firings and retractions are being traced.  Level",
"      0 means no information about the wmes is printed.  Level 1 means the",
"      wme timetags are printed.  Level 2 means the whole wmes are printed.",
"  :firings-preferences -- controls whether the preferences generated by",
"      the traced productions are printed when those productions fire or",
"      retract.  When a production fires, all the preferences it generates",
"      are printed.  When it retracts, only the ones being removed from",
"      preference memory are printed (i.e., the i-supported ones).",
"  :wmes -- controls whether changes to working memory are printed",
"  :chunk-names -- controls whether names of newly built chunks are printed",
"  :justification-names -- ditto, only for justifications (internal chunks)",
"  :chunks -- controls whether newly built chunks are printed",
"  :justifications -- ditto, only for justifications",
"  :backtracing -- controls whether backtracing information is printed",
"",
"These keyword arguments are provided so you can have more complete control",
"over what gets traced.  The numeric arguments to 'watch', as well as the",
"'print' and 'trace' arguments to 'learn', provide simple ways of getting",
"some useful watch settings.  See (help watch-levels) for details.",
"",
"See also:  watch, watch-levels, learn",
0 };

char *help_on_watch_levels[] = {
"The watch levels (-1, 0, 0.5, 1, 1.5, 2, 3) in Soar 6 have roughly the same",
"semantics as in Soar 5.  The table below gives the corresponding keyword",
"parameter settings for each level.",
"",
"            Watch Level:    -1    0  0.5    1  1.5    2    3",
"                          ---- ---- ---- ---- ---- ---- ----",
"  :context                 off   on   on   on   on   on   on",
"  :phases                  off  off   on   on   on   on   on",
"  :firings nonchunk        off  off   on   on   on   on   on",
"    :firings-wmes            0    0    0    1    2    2    2",
"    :firings-preferences   off  off  off  off  off  off   on",
"  :wmes                    off  off  off  off  off   on   on",
"",
"(watch task) is also provided in Soar 6 for Soar 5 compatibility, and is",
"equivalent to (watch :firings default off).",
"",
"Learn noprint/print/full-print and notrace/trace/full-trace also translate",
"into keyword parameter settings, as shown in the tables below:",
"",
"         Learn:   noprint  print  full-print",
"                  -------  -----  ----------",
"  :chunk-names        off     on          on",
"  :chunks             off    off          on",
"",
"         Learn:   notrace  trace  full-trace",
"                  -------  -----  ----------",
"  :firings chunk      off     on          on",
"  :backtracing        off    off          on",
"",
"See also:  watch, watch-keywords, learn",
0 };

bool set_watch_setting (int dest_sysparam_number) {
  get_lexeme(); /* consume :keyword name */
  if (lexeme.type==SYM_CONSTANT_LEXEME) {
    if (!strcmp(lexeme.string,"on")) {
      set_sysparam (dest_sysparam_number, TRUE);
      get_lexeme(); /* consume on/off flag */
      return TRUE;
    }
    if (!strcmp(lexeme.string,"off")) {
      set_sysparam (dest_sysparam_number, FALSE);
      get_lexeme(); /* consume on/off flag */
      return TRUE;
    }
  }
  print ("Expected 'on' or 'off' for new watch setting\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

bool watch_interface_routine (void) {
  get_lexeme();  /* consume "watch" */
  if (lexeme.type==R_PAREN_LEXEME) {
    print ("Current watch settings:\n");
    print ("  Context decisions:  %s\n",
           sysparams[TRACE_CONTEXT_DECISIONS_SYSPARAM] ? "on" : "off");
    print ("  Phases:  %s\n",
           sysparams[TRACE_PHASES_SYSPARAM] ? "on" : "off");
    print ("  Firings/retractions\n");
    print ("    default productions:  %s\n",
           sysparams[TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM] ? "on" : "off");
    print ("    user productions:  %s\n",
           sysparams[TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM] ? "on" : "off");
    print ("    chunks:  %s\n",
           sysparams[TRACE_FIRINGS_OF_CHUNKS_SYSPARAM] ? "on" : "off");
    print ("    justifications:  %s\n",
           sysparams[TRACE_FIRINGS_OF_JUSTIFICATIONS_SYSPARAM] ? "on" : "off");
    print ("  WME detail level:  %d\n",
           sysparams[TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM]);
    print ("  Preferences generated by firings/retractions:  %s\n",
           sysparams[TRACE_FIRINGS_PREFERENCES_SYSPARAM] ? "on" : "off");
    print ("  Working memory changes:  %s\n",
           sysparams[TRACE_WM_CHANGES_SYSPARAM] ? "on" : "off");
    print ("  Chunk names:  %s\n",
           sysparams[TRACE_CHUNK_NAMES_SYSPARAM] ? "on" : "off");
    print ("  Justification names:  %s\n",
           sysparams[TRACE_JUSTIFICATION_NAMES_SYSPARAM] ? "on" : "off");
    print ("  Chunks:  %s\n",
           sysparams[TRACE_CHUNKS_SYSPARAM] ? "on" : "off");
    print ("  Justifications:  %s\n",
           sysparams[TRACE_JUSTIFICATIONS_SYSPARAM] ? "on" : "off");
    print ("  Backtracing:  %s\n",
           sysparams[TRACE_BACKTRACING_SYSPARAM] ? "on" : "off");
    return TRUE;
  }
  while (lexeme.type!=R_PAREN_LEXEME) {
    if (lexeme.type==SYM_CONSTANT_LEXEME) {
      if (!strcmp(lexeme.string,":context")) {
        if (set_watch_setting (TRACE_CONTEXT_DECISIONS_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":phases")) {
        if (set_watch_setting (TRACE_PHASES_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":wmes")) {
        if (set_watch_setting (TRACE_WM_CHANGES_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":chunk-names")) {
        if (set_watch_setting (TRACE_CHUNK_NAMES_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":justification-names")) {
        if (set_watch_setting (TRACE_JUSTIFICATION_NAMES_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":chunks")) {
        if (set_watch_setting (TRACE_CHUNKS_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":justifications")) {
        if (set_watch_setting (TRACE_JUSTIFICATIONS_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":backtracing")) {
        if (set_watch_setting (TRACE_BACKTRACING_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,":firings-preferences")) {
        if (set_watch_setting (TRACE_FIRINGS_PREFERENCES_SYSPARAM)) continue;
        return FALSE;
      }
      if (!strcmp(lexeme.string,"task")) {
        get_lexeme(); /* consume "task" */
        set_sysparam (TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, FALSE);
        continue;
      }
      if (!strcmp(lexeme.string,":firings")) {
        bool types[NUM_PRODUCTION_TYPES];
        int i;
        for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=TRUE;
        get_lexeme();
        /* --- read optional type indicator --- */
        if (lexeme.type==SYM_CONSTANT_LEXEME) {
          if (!strcmp(lexeme.string,"default")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=FALSE;
            types[DEFAULT_PRODUCTION_TYPE]=TRUE;
            get_lexeme();
          } else if (!strcmp(lexeme.string,"user")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=FALSE;
            types[USER_PRODUCTION_TYPE]=TRUE;
            get_lexeme();
          } else if (!strcmp(lexeme.string,"chunk")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=FALSE;
            types[CHUNK_PRODUCTION_TYPE]=TRUE;
            get_lexeme();
          } else if (!strcmp(lexeme.string,"nonchunk")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=FALSE;
            types[DEFAULT_PRODUCTION_TYPE]=TRUE;
            types[USER_PRODUCTION_TYPE]=TRUE;
            get_lexeme();
          } else if (!strcmp(lexeme.string,"all")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=TRUE;
            get_lexeme();
          }
        }
        /* --- read on/off flag --- */
        if (lexeme.type==SYM_CONSTANT_LEXEME) {
          if (!strcmp(lexeme.string,"on")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++)
              if (types[i])
                set_sysparam (TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM+i,TRUE);
            get_lexeme(); /* consume on/off flag */
            continue;
          }
          if (!strcmp(lexeme.string,"off")) {
            for (i=0; i<NUM_PRODUCTION_TYPES; i++)
              if (types[i])
                set_sysparam (TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM+i,FALSE);
            get_lexeme(); /* consume on/off flag */
            continue;
          }
        }
        print ("Expected 'on' or 'off' for new watch setting\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      } /* end of if lexeme.string is "firings" */

      if (!strcmp(lexeme.string,"firings-wmes")) {
        get_lexeme(); /* consume "firings-wmes" */
        if (lexeme.type==INT_CONSTANT_LEXEME) {
          if (lexeme.int_val==0) {
            set_sysparam (TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM,
                          NONE_WME_TRACE);
            get_lexeme();
            continue;
          }
          if (lexeme.int_val==1) {
            set_sysparam (TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM,
                          TIMETAG_WME_TRACE);
            get_lexeme();
            continue;
          }
          if (lexeme.int_val==2) {
            set_sysparam (TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM,
                          FULL_WME_TRACE);
            get_lexeme();
            continue;
          }
        }
        print ("Expected 0, 1, or 2 for new watch :firings-wmes setting\n");
        print_location_of_most_recent_lexeme();
        return FALSE;
      }
    }
    if (lexeme.type==INT_CONSTANT_LEXEME) {
      if (lexeme.int_val==-1) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, FALSE);
        set_sysparam(TRACE_PHASES_SYSPARAM, FALSE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, FALSE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, FALSE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, NONE_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, FALSE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, FALSE);
        get_lexeme(); /* consume the number */
        continue;
      }
      if (lexeme.int_val==0) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, TRUE);
        set_sysparam(TRACE_PHASES_SYSPARAM, FALSE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, FALSE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, FALSE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, NONE_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, FALSE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, FALSE);
        get_lexeme(); /* consume the number */
        continue;
      }
      if (lexeme.int_val==1) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, TRUE);
        set_sysparam(TRACE_PHASES_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, TIMETAG_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, FALSE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, FALSE);
        get_lexeme(); /* consume the number */
        continue;
      }
      if (lexeme.int_val==2) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, TRUE);
        set_sysparam(TRACE_PHASES_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, FULL_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, FALSE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, TRUE);
        get_lexeme(); /* consume the number */
        continue;
      }
      if (lexeme.int_val==3) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, TRUE);
        set_sysparam(TRACE_PHASES_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, FULL_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, TRUE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, TRUE);
        get_lexeme(); /* consume the number */
        continue;
      }
    }
    if (lexeme.type==FLOAT_CONSTANT_LEXEME) {
      if (lexeme.float_val==0.5) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, TRUE);
        set_sysparam(TRACE_PHASES_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, NONE_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, FALSE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, FALSE);
        get_lexeme(); /* consume the number */
        continue;
      }
      if (lexeme.float_val==1.5) {
        set_sysparam(TRACE_CONTEXT_DECISIONS_SYSPARAM, TRUE);
        set_sysparam(TRACE_PHASES_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_DEFAULT_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_OF_USER_PRODS_SYSPARAM, TRUE);
        set_sysparam(TRACE_FIRINGS_WME_TRACE_TYPE_SYSPARAM, FULL_WME_TRACE);
        set_sysparam(TRACE_FIRINGS_PREFERENCES_SYSPARAM, FALSE);
        set_sysparam(TRACE_WM_CHANGES_SYSPARAM, FALSE);
        get_lexeme(); /* consume the number */
        continue;
      }
    }
    print ("Invalid argument to 'watch' command.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  return TRUE;
}

/* -------------------------------------------------------------------
   
                        "Preferences" Command

   Syntax:  (preferences id attribute [level])
------------------------------------------------------------------- */

char *help_on_preferences[] = {
"Command: preferences",
"",
"Syntax: (preferences id attribute [level])",
"",
"This command prints all the preferences for the slot given by the 'id' and",
"'attribute' arguments.  The optional 'level' argument must be 0, 1, 2, or 3",
"(0 is the default); it indicates the level of detail requested:",
"  level 0 -- prints just the preferences themselves",
"  level 1 -- also prints the names of the productions that generated them",
"  level 2 -- also prints the timetags of the wmes matched by the productions",
"  level 3 -- prints the whole wmes, not just their timetags.",
0 };

void print_preference_and_source (preference *pref,
                                  bool print_source,
                                  wme_trace_type wtt) {
  print_string ("  ");
  print_object_trace (pref->value);
  print (" %c", preference_type_indicator (pref->type));
  if (preference_is_binary(pref->type)) print_object_trace (pref->referent);
  if (pref->o_supported) print (" [O]");
  print ("\n");
  if (print_source) {
    print ("    From ");
    print_instantiation_with_wmes (pref->inst, wtt);
    print ("\n");
  }
}

bool preferences_interface_routine (void) {
  symbol *id, *attr;
  bool print_productions;
  wme_trace_type wtt;
  slot *s;
  preference *p;
  
  get_lexeme();  /* consume "preferences", advance to production name(s) */

  /* --- read id --- */
  id = read_identifier_or_context_variable();
  if (!id) return FALSE;
  get_lexeme();  /* consume the id */

  /* --- read attribute --- */
  switch (lexeme.type) {
  case SYM_CONSTANT_LEXEME:
    attr = find_sym_constant (lexeme.string); break;
  case INT_CONSTANT_LEXEME:
    attr = find_int_constant (lexeme.int_val); break;
  case FLOAT_CONSTANT_LEXEME:
    attr = find_float_constant (lexeme.float_val); break;
  case IDENTIFIER_LEXEME:
    attr = find_identifier (lexeme.id_letter, lexeme.id_number); break;
  case VARIABLE_LEXEME:
    attr = read_identifier_or_context_variable();
    if (!attr) return FALSE;
    break;
  default:
    print ("Expected either an identifier or a constant for the attribute\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();  /* consume the attribute */

  /* --- read the optional level indicator --- */
  print_productions = FALSE;
  wtt = NONE_WME_TRACE;
  if (lexeme.type==INT_CONSTANT_LEXEME) {
    if (lexeme.int_val==0) {
      print_productions = FALSE;
      wtt = NONE_WME_TRACE;
    } else if (lexeme.int_val==1) {
      print_productions = TRUE;
      wtt = NONE_WME_TRACE;
    } else if (lexeme.int_val==2) {
      print_productions = TRUE;
      wtt = TIMETAG_WME_TRACE;
    } else if (lexeme.int_val==3) {
      print_productions = TRUE;
      wtt = FULL_WME_TRACE;
    } else {
      print ("'Level' argument must be 0, 1, 2, or 3\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
    get_lexeme();
  }

  /* --- print the preferences --- */
  s = find_slot (id, attr);
  if (!s) {
    print ("There is no such slot.\n");
    return TRUE;
  }

  print_with_symbols ("Preferences for %y ^%y:\n", id, attr);

  if (s->preferences[REQUIRE_PREFERENCE_TYPE]) {
    print ("\nRequires:\n");
    for (p=s->preferences[REQUIRE_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[PROHIBIT_PREFERENCE_TYPE]) {
    print ("\nProhibits:\n");
    for (p=s->preferences[PROHIBIT_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[ACCEPTABLE_PREFERENCE_TYPE]) {
    print ("\nAcceptables:\n");
    for (p=s->preferences[ACCEPTABLE_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[REJECT_PREFERENCE_TYPE]) {
    print ("\nRejects:\n");
    for (p=s->preferences[REJECT_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[BEST_PREFERENCE_TYPE]) {
    print ("\nBests:\n");
    for (p=s->preferences[BEST_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[WORST_PREFERENCE_TYPE]) {
    print ("\nWorsts:\n");
    for (p=s->preferences[WORST_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[BETTER_PREFERENCE_TYPE]) {
    print ("\nBetters:\n");
    for (p=s->preferences[BETTER_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[WORSE_PREFERENCE_TYPE]) {
    print ("\nWorses:\n");
    for (p=s->preferences[WORSE_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[UNARY_INDIFFERENT_PREFERENCE_TYPE]) {
    print ("\nUnary Indifferents:\n");
    for (p=s->preferences[UNARY_INDIFFERENT_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[BINARY_INDIFFERENT_PREFERENCE_TYPE]) {
    print ("\nBinary Indifferents:\n");
    for (p=s->preferences[BINARY_INDIFFERENT_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[UNARY_PARALLEL_PREFERENCE_TYPE]) {
    print ("\nUnary Parallels:\n");
    for (p=s->preferences[UNARY_PARALLEL_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[BINARY_PARALLEL_PREFERENCE_TYPE]) {
    print ("\nBinary Parallels:\n");
    for (p=s->preferences[BINARY_PARALLEL_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  if (s->preferences[RECONSIDER_PREFERENCE_TYPE]) {
    print ("\nReconsiders:\n");
    for (p=s->preferences[RECONSIDER_PREFERENCE_TYPE]; p; p=p->next)
      print_preference_and_source (p, print_productions, wtt);
  }

  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Ms" Command

   Syntax: (ms [ 0 | 1 | 2 ])
------------------------------------------------------------------- */

char *help_on_ms[] = {
"Command: ms",
"",
"Syntax: (ms [ 0 | 1 | 2 ])",
"",
"This command prints the current \"match set\", i.e., a list of productions",
"that are about to fire or retract in the next preference phase.  The",
"optional integer specifies the level of detail wanted:  0 (the default)",
"prints out just the production names; 1 also prints the timetags of wmes",
"matched; and 2 prints the wmes rather than just their timetags.",
0 };

bool ms_interface_routine (void) {
  wme_trace_type wtt;
  
  get_lexeme();  /* consume "ms", look for level */
  wtt = NONE_WME_TRACE;
  if (lexeme.type==INT_CONSTANT_LEXEME) {
    if ((lexeme.int_val>=0) && (lexeme.int_val<=2)) {
      if (lexeme.int_val==0) wtt = NONE_WME_TRACE;
      if (lexeme.int_val==1) wtt = TIMETAG_WME_TRACE;
      if (lexeme.int_val==2) wtt = FULL_WME_TRACE;
      get_lexeme();
    } else {
      print ("MS level of detail must be 0, 1, or 2.\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
  }
  print_match_set (wtt);
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Sp" Command

   Syntax:  see helpscreen below.
------------------------------------------------------------------- */

char *help_on_sp[] = {
"Command: sp",
"",
"Syntax: (sp production-name",
"          [ \"optional-documentation-string\" ]",
"          [ flag ]*",
"          LHS",
"          -->",
"          RHS)",       
"",
"       flag  ::=  :o-support",
"       flag  ::=  :no-o-support",
"       flag  ::=  :default",
"       flag  ::=  :chunk",
"",
"This command adds a new production to the system.  (If another production",
"with the same name already exists, it is excised.)  The optional flags",
"are as follows:",
"   :o-support -- specifies that all the RHS actions are to be given",
"                 o-support when the production fires",
"   :no-support -- specifies that all the RHS actions are only to be given",
"                  i-support when the production fires",
"   :default -- specifies that this production is a default production (this",
"               matters for (excise-task) and (watch task))",
"   :chunk -- specifies that this production is a chunk (this matters for",
"             (learn trace))",
"",
"See also:  lhs-grammar, rhs-grammar",
0 };

bool sp_interface_routine (void) {
  production *p;

  set_lexer_allow_ids (FALSE);
  get_lexeme();  /* consume "sp", advance to production name */
  p = parse_production();
  if (p) print ("*");
  if (p) return TRUE; else return FALSE;
}

/* -------------------------------------------------------------------
   
                    "Max-elaborations" Command
 
   Syntax:  (max-elaborations [integer])
------------------------------------------------------------------- */

char *help_on_max_elaborations[] = {
"Command: max-elaborations",
"",
"Syntax: (max-elaborations [integer])",
"",
"With no arguments, this command prints the current value of the system",
"variable 'max-elaborations'.  With an integer argument, it sets the current",
"value.   This variable controls the maximum number of elaboration cycles",
"allowed in a single decision cycle.  After this many elabloration cycles",
"have been executed, Soar proceeds to quiescence phase even if quiescence",
"hasn't really been reached yet.  (Max-elaborations is initially 100.)",
0 };

bool max_elaborations_interface_routine (void) {
  get_lexeme();  /* consume "max-elaborations", advance to integer */
  if (lexeme.type==R_PAREN_LEXEME) {
    /* --- invoked with no arguments, so just print the current value --- */
    print ("Max-elaborations is %ld.\n", sysparams[MAX_ELABORATIONS_SYSPARAM]);
    return TRUE;
  }
  if (lexeme.type!=INT_CONSTANT_LEXEME) {
    print ("Expected integer for new value of max-elaborations.\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  set_sysparam (MAX_ELABORATIONS_SYSPARAM, lexeme.int_val);
  get_lexeme(); /* consume the integer */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                        "User-select" Command
 
   Syntax:  (user-select [first | ask | random | t | nil])
------------------------------------------------------------------- */

char *help_on_user_select[] = {
"Command: user-select",
"",
"Syntax: (user-select [first | ask | random | t | nil])",
"",
"With no arguments, this command prints the current setting of user-select.",
"With an argument, it sets user-select to the given value.  This controls",
"how Soar's decision procedure chooses between multiple indifferent items:",
"   first -- just choose the first one found (deterministically)",
"   ask -- ask the user to choose one of the items",
"   random -- choose one randomly",
"   t -- synonymous with 'ask'",
"   nil -- synonymous with 'random'",
0 };

bool user_select_interface_routine (void) {
  get_lexeme();  /* consume "user-select", advance to mode */
  if (lexeme.type==R_PAREN_LEXEME) {
    /* --- invoked with no arguments, so just print the current value --- */
    print ("User-select is currently set to:  ");
    switch (sysparams[USER_SELECT_MODE_SYSPARAM]) {
    case USER_SELECT_FIRST: print ("first"); break;
    case USER_SELECT_ASK: print ("ask"); break;
    case USER_SELECT_RANDOM: print ("random"); break;
    }
    print ("\n");
    return TRUE;
  }
  if (lexeme.type==SYM_CONSTANT_LEXEME) {
    if (!strcmp(lexeme.string,"first")) {
      set_sysparam (USER_SELECT_MODE_SYSPARAM, USER_SELECT_FIRST);
      get_lexeme();
      return TRUE;
    }
    if ( (!strcmp(lexeme.string,"ask")) ||
         (!strcmp(lexeme.string,"t")) ) {
      set_sysparam (USER_SELECT_MODE_SYSPARAM, USER_SELECT_ASK);
      get_lexeme();
      return TRUE;
    }
    if ( (!strcmp(lexeme.string,"random")) ||
         (!strcmp(lexeme.string,"nil")) ) {
      set_sysparam (USER_SELECT_MODE_SYSPARAM, USER_SELECT_RANDOM);
      get_lexeme();
      return TRUE;
    }
  }
  print ("Expected first, ask, or random for new value of user-select.\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

/* -------------------------------------------------------------------
   
                        "Soarnews" Command
 
   Syntax:  (soarnews)
------------------------------------------------------------------- */

char *help_on_soarnews[] = {
"Command: soarnews",
"",
"Syntax: (soarnews)",
"",
"This command prints news about the current release of Soar.",
0 };

bool soarnews_interface_routine (void) {
  get_lexeme();  /* consume "soarnews" */
  /* BUGBUG update soarnews printout on successive versions */
  print ("News for Soar %d.%d.%d\n", MAJOR_VERSION_NUMBER,
         MINOR_VERSION_NUMBER, MICRO_VERSION_NUMBER);
  print ("\n");
  print ("Bugs and questions should be sent to Soar-bugs@cs.cmu.edu\n");
  print ("The current bug-list may be obtained by sending mail to\n");
  print ("Soar-bugs@cs.cmu.edu with the Subject: line \"bug list\"\n");
  print ("This software is in the public domain.\n");
  print ("\n");
  print ("This software is made available AS IS, and Carnegie Mellon\n");
  print ("University and the University of Michigan make no warranty\n");
  print ("about the software or its performance.\n");

  return TRUE;
}

/* -------------------------------------------------------------------
  
  "List-productions", "list-chunks", and "list-justifications" Commands

   Syntax: (list-productions [prod-type*] [:internal] ["filename" [:append]])
           prod-type ::= default | user | chunk | justification
           (list-chunks [:internal] ["filename" [:append]])
           (list-justifications [:internal] ["filename" [:append]])
------------------------------------------------------------------- */

char *help_on_list_productions[] = {
"Commands: list-productions, list-chunks, list-justifications",
"",
"Syntax: (list-productions [prod-type*] [:internal] [\"filename\" [:append]])",
"    prod-type ::= default | user | chunk | justification",
"    (list-chunks) is shorthand for (list-productions chunk)",
"    (list-justifications) is shorthand for (list-productions justification)",
"",
"This command prints all productions of the indicated types.  (If no",
"prod-type's are given, all productions except justifications are printed.)",
"",
"The optional [:internal] argument tells Soar to print productions in their",
"internal reordered (rete net) form.",
"",
"If a filename is given, the productions are printed to that file; otherwise",
"they are printed to the screen.  If :append is given, the file is appended",
"to, rather than overwritten.",
0 };

bool list_productions_interface_routine (void) {
  char filename[MAX_LEXEME_LENGTH+1];
  bool prod_type_present, filename_present, internal, append;
  bool types[NUM_PRODUCTION_TYPES]; /* tells which types of prod's to list */
  int i;

  /* --- get settings of types[] --- */
  for (i=0; i<NUM_PRODUCTION_TYPES; i++) types[i]=FALSE; /* init to FALSE */
  if (!strcmp(lexeme.string,"list-chunks")) {
    types[CHUNK_PRODUCTION_TYPE] = TRUE;
    get_lexeme(); /* consume "list-chunks" */
  } else if (!strcmp(lexeme.string,"list-justifications")) {
    types[JUSTIFICATION_PRODUCTION_TYPE] = TRUE;
    get_lexeme(); /* consume "list-justifications" */
  } else {
    get_lexeme(); /* consume "list-productions" */
    /* --- read prod-type* --- */
    prod_type_present = FALSE;
    while (TRUE) {
      if (lexeme.type!=SYM_CONSTANT_LEXEME) break;
      if (!strcmp(lexeme.string,"default")) {
        prod_type_present = TRUE;
        types[DEFAULT_PRODUCTION_TYPE] = TRUE;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"user")) {
        prod_type_present = TRUE;
        types[USER_PRODUCTION_TYPE] = TRUE;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"chunk")) {
        prod_type_present = TRUE;
        types[CHUNK_PRODUCTION_TYPE] = TRUE;
        get_lexeme();
        continue;
      }
      if (!strcmp(lexeme.string,"justification")) {
        prod_type_present = TRUE;
        types[JUSTIFICATION_PRODUCTION_TYPE] = TRUE;
        get_lexeme();
        continue;
      }
      break;
    } /* end of while (TRUE) */
    if (!prod_type_present) {
      /* --- no prod-type* was present --- */
      types[DEFAULT_PRODUCTION_TYPE] = TRUE;
      types[USER_PRODUCTION_TYPE] = TRUE;
      types[CHUNK_PRODUCTION_TYPE] = TRUE;
    }
  }

  /* --- got types[], so now look for :internal flag --- */
  internal = FALSE;
  if ((lexeme.type==SYM_CONSTANT_LEXEME)&&(!strcmp(lexeme.string,":internal")))
    { internal = TRUE; get_lexeme(); }

  /* --- look for filename and :append --- */
  if (lexeme.type==R_PAREN_LEXEME) {
    filename_present = FALSE;
  } else if (lexeme.type==QUOTED_STRING_LEXEME) {
    filename_present = TRUE;
    strcpy (filename, lexeme.string);
    get_lexeme();
    append = FALSE;
    if (!strcmp(lexeme.string, ":append")) { append=TRUE; get_lexeme(); }
  } else {
    print ("Expected string in quotes for filename\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }

  /* --- all set, now print all the productions --- */
  {
    FILE *output_file;
    production *prod;

    if (filename_present) {
      output_file = fopen (filename, (append ?  "a" : "w") );
      if (!output_file) {
        /* --- error when opening the file --- */
        print ("Error: unable to open file %s\n",filename);
        return FALSE;
      }
      print ("Writing productions to file %s\n", filename);
      start_redirection_to_file (output_file);
    }

    for (i=0; i<NUM_PRODUCTION_TYPES; i++)
      if (types[i])
        for (prod=all_productions_of_type[i]; prod!=NIL; prod=prod->next) {
          print_production (prod, internal);
          print ("\n");
        }

    if (filename_present) {
      stop_redirection_to_file ();
      fclose (output_file);
    }
  }

  return TRUE;
}

/* -------------------------------------------------------------------
   
                   "Add-wme" and "Remove-wme" Commands

   Syntax: (add-wme id ^ { attribute | '*'} { value | '*' } [+])
           (remove-wme integer)
------------------------------------------------------------------- */

char *help_on_add_or_remove_wme[] = {
"Commands: add-wme, remove-wme",
"",
"Syntax: (add-wme id ^ { attribute | '*'} { value | '*' } [+])",
"        (remove-wme integer)",
"",
"These commands surgically modify Soar's working memory.  Add-wme adds a",
"new wme with the given id, attribute, value, and optional acceptable",
"preference.  The given id must be an existing identifier.  If '*' is given",
"in place of the attribute or value, Soar creates a new identifier (gensym)",
"for that field.  Remove-wme removes the wme with the given timetag.",
"",
"WARNING: these commands are inherently unstable and may have weird side",
"effects (possibly even including system crashes).  For example, the chunker",
"can't backtrace through wmes created via add-wme.  Removing input wmes or",
"context/impasse wmes may have unexpected side effects.  You've been warned.",
0 };

bool add_wme_interface_routine (void) {
  symbol *id, *attr, *value;
  bool acceptable_preference;
  wme *w;

  get_lexeme();  /* consume "add-wme" */
  id = read_identifier_or_context_variable();
  if (!id) return FALSE;
  get_lexeme();  /* consume id */
  if (lexeme.type!=UP_ARROW_LEXEME) {
    print ("Expected ^ between id and attribute\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();  /* consume ^ */
  
  /* --- get attribute or '*' --- */
  if (strcmp(lexeme.string,"*") == 0) {
    attr = make_new_identifier ('I', id->id.level);
  } else {
    switch (lexeme.type) {
    case SYM_CONSTANT_LEXEME:
      attr = make_sym_constant (lexeme.string); break;
    case INT_CONSTANT_LEXEME:
      attr = make_int_constant (lexeme.int_val); break;
    case FLOAT_CONSTANT_LEXEME:
      attr = make_float_constant (lexeme.float_val); break;
    case IDENTIFIER_LEXEME:
    case VARIABLE_LEXEME:
      attr = read_identifier_or_context_variable();
      if (!attr) return FALSE;
      symbol_add_ref (attr);
      break;
    default:
      print ("Expected constant, identifier, or '*' for attribute\n");
      print_location_of_most_recent_lexeme();
      return FALSE;
    }
  }
  get_lexeme(); /* consume attribute */

  /* --- get value or '*' --- */
  if (strcmp(lexeme.string,"*") == 0) {
    value = make_new_identifier ('I', id->id.level);
  } else {
    switch (lexeme.type) {
    case SYM_CONSTANT_LEXEME:
      value = make_sym_constant (lexeme.string); break;
    case INT_CONSTANT_LEXEME:
      value = make_int_constant (lexeme.int_val); break;
    case FLOAT_CONSTANT_LEXEME:
      value = make_float_constant (lexeme.float_val); break;
    case IDENTIFIER_LEXEME:
    case VARIABLE_LEXEME:
      value = read_identifier_or_context_variable();
      if (!value) { symbol_remove_ref (attr); return FALSE; }
      symbol_add_ref (value);
      break;
    default:
      print ("Expected constant, identifier, or '*' for value\n");
      print_location_of_most_recent_lexeme();
      symbol_remove_ref (attr);
      return FALSE;
    }
  }
  get_lexeme(); /* consume value */

  /* --- get optional acceptable preference indicator --- */
  acceptable_preference = FALSE;
  if (lexeme.type==PLUS_LEXEME) { acceptable_preference = TRUE; get_lexeme(); }

  /* --- now create and add the wme --- */
  w = make_wme (id, attr, value, acceptable_preference);
  symbol_remove_ref (attr);
  symbol_remove_ref (value);
  insert_at_head_of_dll (id->id.input_wmes, w, next, prev);
  add_wme_to_wm (w);
  do_buffered_wm_and_ownership_changes();
  
  return TRUE;
}

bool remove_wme_interface_routine (void) {
  wme *w, *w2;
  symbol *id;
  slot *s;
  
  get_lexeme();  /* consume "remove-wme" */
  if (lexeme.type!=INT_CONSTANT_LEXEME) {
    print ("Expected integer for timetag of wme to remove\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  for (w=all_wmes_in_rete; w!=NIL; w=w->rete_next)
    if (w->timetag == lexeme.int_val) break;
  if (!w) {
    print ("No wme %ld in working memory\n", lexeme.int_val);
    return FALSE;
  }
  get_lexeme();  /* consume timetag */

  id = w->id;

  /* --- remove w from whatever list of wmes it's on --- */
  for (w2=id->id.input_wmes; w2!=NIL; w2=w2->next)
    if (w==w2) break;
  if (w2) remove_from_dll (id->id.input_wmes, w, next, prev);
  for (w2=id->id.impasse_wmes; w2!=NIL; w2=w2->next)
    if (w==w2) break;
  if (w2) remove_from_dll (id->id.impasse_wmes, w, next, prev);
  for (s=id->id.slots; s!=NIL; s=s->next) {
    for (w2=s->wmes; w2!=NIL; w2=w2->next)
      if (w==w2) break;
    if (w2) remove_from_dll (s->wmes, w, next, prev);    
    for (w2=s->acceptable_preference_wmes; w2!=NIL; w2=w2->next)
      if (w==w2) break;
    if (w2) remove_from_dll (s->acceptable_preference_wmes, w, next, prev);
  }

  /* --- now remove w from working memory --- */
  remove_wme_from_wm (w);
  do_buffered_wm_and_ownership_changes();

  return TRUE;
}

/* -------------------------------------------------------------------
   
                        "Firing-counts" Command

   Syntax: (firing-counts [integer])
           (firing-counts production-name ...)
------------------------------------------------------------------- */

char *help_on_firing_counts[] = {
"Command: firing-counts",
"",
"Syntax: (firing-counts [integer])",
"        (firing-counts production-name ...)",
"",
"This command prints how many times certain productions have fired.  With",
"no arguments, it lists all the productions sorted according to how many",
"times they have fired.  If an integer argument (call it k) is given, only",
"the top k productions are listed.  If k=0, only the productions which",
"haven't fired at all are listed.  Note that firing counts are not reset",
"by an (init-soar); the counts indicate the number of firings since the",
"productions were loaded or built.",
"",
"Note:  this is slow, because the sorting takes time O(n*log n)",
"",
"With one or more production names as arguments, this command prints how",
"many times each of those productions fired.",
0 };

int compare_firing_counts (e1,e2)
     void *e1;
     void *e2;
{
  production *p1, *p2;
  unsigned long count1, count2;
  p1 = *((production **)e1);
  p2 = *((production **)e2);
  count1 = p1->firing_count;
  count2 = p2->firing_count;
  return (count1<count2) ? -1 : (count1>count2) ? 1 : 0;
}

bool firing_counts_interface_routine (void) {
  symbol *sym;
  long num_prods, num_requested;
  production *((*all_prods)[]), **ap_item, *p;
  
  set_lexer_allow_ids (FALSE); /* only takes production names, never ids */
  get_lexeme();  /* consume "firing-counts" */

  /* --- handle production name arguments --- */
  if (lexeme.type==SYM_CONSTANT_LEXEME) {
    while (lexeme.type==SYM_CONSTANT_LEXEME) {
      sym = find_sym_constant (lexeme.string);
      if (sym && sym->sc.production) {
        print ("%6lu:  %s\n", sym->sc.production->firing_count, lexeme.string);
      } else {
        print ("No production named %s\n", lexeme.string);
        print_location_of_most_recent_lexeme();
      }
      get_lexeme();
    }
    return TRUE;
  }

  /* --- handle integer (or no) arguments --- */
  if (lexeme.type==R_PAREN_LEXEME) {
    num_requested = -1;
  } else if (lexeme.type==INT_CONSTANT_LEXEME) {
    num_requested = lexeme.int_val;
    get_lexeme();
  } else {
    print ("Illegal argument to 'firing-counts' command\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }

  num_prods = num_productions_of_type[DEFAULT_PRODUCTION_TYPE] +
              num_productions_of_type[USER_PRODUCTION_TYPE] +
              num_productions_of_type[CHUNK_PRODUCTION_TYPE];
  if (num_prods==0) return TRUE;  /* so we don't barf on zero later */

  /* --- make an array of pointers to all the productions --- */
  all_prods = allocate_memory (num_prods * sizeof (production *),
                               MISCELLANEOUS_MEM_USAGE);
  ap_item = &((*all_prods)[0]);
  for (p=all_productions_of_type[DEFAULT_PRODUCTION_TYPE]; p!=NIL; p=p->next)
    *(ap_item++) = p;
  for (p=all_productions_of_type[USER_PRODUCTION_TYPE]; p!=NIL; p=p->next)
    *(ap_item++) = p;
  for (p=all_productions_of_type[CHUNK_PRODUCTION_TYPE]; p!=NIL; p=p->next)
    *(ap_item++) = p;

  /* --- sort that array according to firing counts --- */
  qsort (all_prods, num_prods, sizeof (production *), compare_firing_counts);

  /* --- now print out the results --- */
  if (num_requested==0) {
    ap_item = &((*all_prods)[0]);
    while ((*ap_item)->firing_count==0) {
      print_with_symbols ("     0:  %y\n", (*ap_item)->name);
      ap_item++;
    }
    return TRUE;
  }
  if ((num_requested < 0) || (num_requested > num_prods))
    num_requested = num_prods;
  ap_item = &((*all_prods)[num_prods-1]);
  while (num_requested) {
    print ("%6lu:  ", (*ap_item)->firing_count);
    print_with_symbols ("%y\n", (*ap_item)->name);
    ap_item--;
    num_requested--;
  }

  return TRUE;
}

/* -------------------------------------------------------------------
   
                      "Ptrace" and "Unptrace" Commands

   Syntax: (ptrace [production-name ...])
   Syntax: (unptrace [production-name ...])
------------------------------------------------------------------- */

char *help_on_ptrace_and_unptrace[] = {
"Commands: ptrace, unptrace",
"",
"Syntax: (ptrace [production-name ...])",
"        (unptrace [production-name ...])",
"",
"These commands enable and disable tracing the firings and retractions of",
"individual productions.  (This mechanism is orthogonal to the watch :firings",
"mechanism.  See (help watch) for more information.)",
"",
"Ptrace, with no arguments, lists the productions currently being traced.",
"With one or more production name arguments, it enables tracing of those",
"productions.  Tracing persists until disabled by an unptrace command, or",
"until the production is excised.",
"",
"Unptrace undoes the effects of ptrace.  With no arguments, it disables all",
"previously enabled production traces.  With one or more production name",
"arguments, it disables just those traces.",
"",
"See also:  watch",
0 };

bool ptrace_interface_routine (void) {
  symbol *sym;
  cons *c;
  
  set_lexer_allow_ids (FALSE); /* only takes production names, never ids */
  get_lexeme();  /* consume "ptrace" */

  if (lexeme.type==R_PAREN_LEXEME) {
    /* --- list current ptraces --- */
    for (c=productions_being_traced; c!=NIL; c=c->rest)
      print_with_symbols (" %y\n", ((production *)(c->first))->name);
    return TRUE;
  }

  /* --- handle production name arguments --- */
  while (lexeme.type==SYM_CONSTANT_LEXEME) {
    sym = find_sym_constant (lexeme.string);
    if (sym && sym->sc.production) {
      add_ptrace (sym->sc.production);
    } else {
      print ("No production named %s\n", lexeme.string);
      print_location_of_most_recent_lexeme();
    }
    get_lexeme();
  }

  if (lexeme.type!=R_PAREN_LEXEME) {
    print ("Bad argument to 'ptrace' command--expected a production name\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  return TRUE;
}

bool unptrace_interface_routine (void) {
  production *prod;
  symbol *sym;
  
  set_lexer_allow_ids (FALSE); /* only takes production names, never ids */
  get_lexeme();  /* consume "unptrace" */

  if (lexeme.type==R_PAREN_LEXEME) {
    /* --- remove all current ptraces --- */
    while (productions_being_traced) {
      prod = productions_being_traced->first;
      remove_ptrace (prod);
    }
    return TRUE;
  }

  /* --- handle production name arguments --- */
  while (lexeme.type==SYM_CONSTANT_LEXEME) {
    sym = find_sym_constant (lexeme.string);
    if (sym && sym->sc.production) {
      remove_ptrace (sym->sc.production);
    } else {
      print ("No production named %s\n", lexeme.string);
      print_location_of_most_recent_lexeme();
    }
    get_lexeme();
  }

  if (lexeme.type!=R_PAREN_LEXEME) {
    print ("Bad argument to 'unptrace' command--expected a production name\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Warnings" Command
 
   Syntax:  (warnings [on|off])
------------------------------------------------------------------- */

char *help_on_warnings[] = {
"Command: warnings",
"",
"Syntax: (warnings [on|off])",
"",
"(Warnings on) enables the printing of warning messages.  This is the",
"default.  (Warnings off) turns off most warning messages.  (Warnings)",
"prints an indication of whether warning messages are enabled or not.",
0 };

bool warnings_interface_routine (void) {
  get_lexeme();  /* consume "warnings", advance to integer */
  if (lexeme.type==R_PAREN_LEXEME) {
    /* --- invoked with no arguments, so just print the currents status --- */
    print ("Warnings are %s.\n",
           sysparams[PRINT_WARNINGS_SYSPARAM] ? "on" : "off");
    return TRUE;
  }
  if (lexeme.type!=SYM_CONSTANT_LEXEME) {
    print ("Expected 'on' or 'off' for argument to 'warnings' command\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  if (!strcmp (lexeme.string, "on")) {
    set_sysparam (PRINT_WARNINGS_SYSPARAM, TRUE);
    get_lexeme();
    return TRUE;
  }
  if (!strcmp (lexeme.string, "off")) {
    set_sysparam (PRINT_WARNINGS_SYSPARAM, FALSE);
    get_lexeme();
    return TRUE;
  }
  print ("Expected 'on' or 'off' for argument to 'warnings' command\n");
  print_location_of_most_recent_lexeme();
  return FALSE;
}

/* -------------------------------------------------------------------
   
                          "Time" Command

------------------------------------------------------------------- */

char *help_on_time[] = {
"Command: time",
"",
"Syntax: (time (command-1 its-args...) (command-2 its-args...) ...)",
"",
"This command executes one or more other commands, then gives a detailed",
"report on how much time it took to execute them.  Note that the other",
"commands *must* be enclosed in parentheses--this is one case where you",
"can't leave off the parentheses.",
0 };

double time_difference (struct timeval *start, struct timeval *end) {
  long seconds, micros;

  seconds = end->tv_sec - start->tv_sec;
  micros = end->tv_usec - start->tv_usec;
  if (micros < 0) {
    micros += 1000000;
    seconds--;
  }
  return (double)(seconds) + (double)(micros)/1000000.0;
}

bool time_interface_routine (void) {
  int parentheses_level;
  struct rusage start_rusage, end_rusage;
  struct timeval start_real_time, end_real_time;
  double user_cpu_time, sys_cpu_time, total_cpu_time, real_time;
  
  /* --- get initial time statistics --- */
  getrusage (RUSAGE_SELF, &start_rusage);
  gettimeofday (&start_real_time, NIL);
  
  /* --- read and dispatch the series of commands --- */
  parentheses_level = current_lexer_parentheses_level();  
  get_lexeme();  /* consume "time", advance to first command */
  while (lexeme.type==L_PAREN_LEXEME) {
    /* --- read one command, dispatch it --- */
    get_lexeme(); /* consume lparen */
    if (lexeme.type==SYM_CONSTANT_LEXEME) {
      dispatch_command();
      if (lexeme.type==EOF_LEXEME) break;
    } else {
      print ("Error:  unknown command\n");
      print_location_of_most_recent_lexeme();
      /* consume just the single bad command */
      skip_ahead_to_balanced_parentheses (parentheses_level);
    }
    get_lexeme();  /* consume rparen, advance to next command */
  } /* end of while loop */
  if (lexeme.type!=R_PAREN_LEXEME) {
    print ("Error: expected ) to end time command or ( to begin next thing to time\n");
    print_location_of_most_recent_lexeme();
    skip_ahead_to_balanced_parentheses (parentheses_level-1);
  }
  
  /* --- print out the final time statistics --- */
  gettimeofday (&end_real_time, NIL);
  getrusage (RUSAGE_SELF, &end_rusage);
  user_cpu_time = time_difference (&(start_rusage.ru_utime),
                                   &(end_rusage.ru_utime));
  sys_cpu_time = time_difference (&(start_rusage.ru_stime),
                                  &(end_rusage.ru_stime));
  total_cpu_time = user_cpu_time + sys_cpu_time;
  real_time = time_difference (&start_real_time, &end_real_time);
  print ("\nCPU time (seconds): %.3f (%.3f user, %.3f system)",
         total_cpu_time, user_cpu_time, sys_cpu_time);
  print ("\nReal time: %.3f  (%.1f%%)\n",
         real_time,
         100.0 * total_cpu_time / ((real_time>0.0) ? real_time : 0.1));
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Memory-stats" Command

   Syntax:  (memory-stats)
------------------------------------------------------------------- */

char *help_on_memory_stats[] = {
"Command: memory-stats",
"",
"Syntax: (memory-stats)",
"",
"This command prints out statistics on memory usage.",
"",
"See also:  rete-stats, stats",
0 };

bool memory_stats_interface_routine (void) {
  print_memory_statistics ();
  print_memory_pool_statistics ();
  get_lexeme();  /* consume "memory-stats", advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Rete-stats" Command

   Syntax:  (rete-stats)
------------------------------------------------------------------- */

char *help_on_rete_stats[] = {
"Command: rete-stats",
"",
"Syntax: (rete-stats)",
"",
"This command prints out statistics on the rete net.",
"",
"See also:  memory-stats, stats",
0 };

bool rete_stats_interface_routine (void) {
  print_rete_statistics ();
  get_lexeme();  /* consume "rete-stats", advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
                  "Stats" and "Print-Stats" Commands

   Syntax:  (stats) or (print-stats)
------------------------------------------------------------------- */

char *help_on_stats[] = {
"Commands: stats, print-stats",
"",
"Syntax: (stats) or (print-stats) [they are synonymous]",
"",
"This command prints out some statistics on the current Soar run.",
"",
"See also:  memory-stats, rete-stats",
0 };

bool stats_interface_routine (void) {
  double total_time, total_msec;
  unsigned long wme_changes;
#ifdef DETAILED_TIMING_STATS
  double match_time, match_msec;
  double ownership_time, o_support_time, chunking_time, pp_time, ci_time;
#endif
  char hostname[1000];
  long current_time;  

  if (gethostname (hostname, 1000)) strcpy (hostname, "[host name unknown]");
  current_time = time(NULL);
  print ("Soar %d.%d.%d on %s at %s\n", MAJOR_VERSION_NUMBER,
         MINOR_VERSION_NUMBER, MICRO_VERSION_NUMBER, hostname,
         ctime(&current_time));

  print ("%lu productions (%lu default, %lu user, %lu chunks)\n",
         num_productions_of_type[DEFAULT_PRODUCTION_TYPE] +
         num_productions_of_type[USER_PRODUCTION_TYPE] +
         num_productions_of_type[CHUNK_PRODUCTION_TYPE],
         num_productions_of_type[DEFAULT_PRODUCTION_TYPE],
         num_productions_of_type[USER_PRODUCTION_TYPE],
         num_productions_of_type[CHUNK_PRODUCTION_TYPE]);
  print ("   + %lu justifications\n",
         num_productions_of_type[JUSTIFICATION_PRODUCTION_TYPE]);

  total_time = timer_value (&total_cpu_time);
  total_msec = total_time * 1000.0;
  print ("Total cpu time: %.3f seconds\n", total_time);

#ifdef DETAILED_TIMING_STATS
  match_time = timer_value (&match_cpu_time);
  match_msec = match_time * 1000.0;
  ownership_time = timer_value (&ownership_cpu_time);
  o_support_time = timer_value (&o_support_cpu_time);  
  chunking_time = timer_value (&chunking_cpu_time);
  pp_time = timer_value (&preference_phase_cpu_time);
  ci_time = timer_value (&create_instantiations_cpu_time);
  
  print ("  ( %.3f match, %.3f ownership, %.3f chunking, %.3f o-support,\n",
         match_time, ownership_time, chunking_time, o_support_time);
  print ("    %.3f build instantiations, %.3f other pref. phase )\n",
         ci_time - chunking_time, pp_time - ci_time);
#endif

  print ("%lu decision cycles (%.3f msec/dc)\n",
         d_cycle_count,
         d_cycle_count ? total_msec/d_cycle_count : 0.0);
  print ("%lu elaboration cycles (%.3f ec's per dc, %.3f msec/ec)\n",
         e_cycle_count,
         d_cycle_count ? (double)e_cycle_count/d_cycle_count : 0,
         e_cycle_count ? total_msec/e_cycle_count : 0);
  print ("%lu production firings (%.3f pf's per ec, %.3f msec/pf)\n",
         production_firing_count,
         e_cycle_count ? (double)production_firing_count/e_cycle_count : 0.0,
         production_firing_count ? total_msec/production_firing_count : 0.0);
  
  wme_changes = wme_addition_count + wme_removal_count;
  print ("%lu wme changes (%lu additions, %lu removals)\n",
         wme_changes, wme_addition_count, wme_removal_count);
#ifdef DETAILED_TIMING_STATS
  print ("    match time: %.3f msec/wm change\n",
         wme_changes ? match_msec/wme_changes : 0.0);
#endif

  print ("WM size: %lu current, %.3f mean, %lu maximum\n",
         num_wmes_in_rete,
         (num_wm_sizes_accumulated ?
          (cumulative_wm_size / num_wm_sizes_accumulated) :
          0.0),
         max_wm_size);

  get_lexeme();  /* consume "stats", advance to rparen */
  return TRUE;
}

/* -------------------------------------------------------------------
   
         "Object-trace-format" and "Stack-trace-format" Commands

   Syntax:  (see below)
------------------------------------------------------------------- */

char *help_on_object_and_stack_traces[] = {
"Commands: object-trace-format, stack-trace-format",
"",
"Syntax:",
"    (object-trace-format :add {g|p|s|o|*} [object-name] \"format-string\")",
"    (object-trace-format :remove {g|p|s|o|*} [object-name])",
"    (object-trace-format)",
"    (stack-trace-format :add {g|p|s|o|*} [ps-name] \"format-string\")",
"    (stack-trace-format :remove {g|p|s|o|*} [ps-name])",
"    (stack-trace-format)",
"",
"Object trace formats control how Soar prints an object--e.g., a certain",
"operator, problem-space, etc.  (This is like trace-attributes in Soar 5.)",
"Stack trace formats control how Soar prints its context stack selections",
"in 'watch 0' and 'pgs' printouts.  You specify a trace format by indicating",
"two things:",
"  - a format string, indicating the printout format to be used",
"  - what things this format string can be applied to",
"",
"The format string can be any string in quotation marks.  Certain 'escape",
"sequences' can be used within the string; for example, '%dc' means print",
"the current decision cycle number.  For a list of escape sequences, see",
"(help trace-format-escapes).",
"",
"There are two ways to restrict what objects a format string applies to.  The",
"{g|p|s|o|*} argument restricts the types of objects:  'g' indicates that the",
"format only applies to goals; 'p' means it only applies to problem spaces;",
"etc.  '*' means it applies to any type of object.  The [object-name]",
"argument (for object trace formats), if given, means it only applies to",
"objects with that ^name.  The [ps-name] argument (for stack trace formats)",
"means it only applies within problem spaces with that ^name.",
"",
"With an :add argument, these commands add new trace formats (replacing any",
"existing ones with identical applicability conditions).  With a :remove",
"argument, they remove trace formats with the given applicability conditions.",
"With no arguments, they print out all current trace formats.",
"",
"See also:  trace-format-escapes",
0 };

bool trace_format_interface_routine (void) {
  bool stack_trace;
  int type_restriction;
  symbol *name_restriction;
  bool remove;
  
  /* --- set stack_trace depending on which command name was given --- */
  stack_trace = FALSE;
  if (! strcmp (lexeme.string, "stack-trace-format")) stack_trace = TRUE;
  get_lexeme();  /* consume command name */

  /* --- if no args, print all trace formats of that type --- */
  if (lexeme.type==R_PAREN_LEXEME) {
    print_all_trace_formats (stack_trace);
    return TRUE;
  }

  /* --- first argument must be either :add or :remove --- */
  remove = FALSE;
  if (! strcmp (lexeme.string, ":add")) remove = FALSE;
  else if (! strcmp (lexeme.string, ":remove")) remove = TRUE;
  else {
    print ("Expected :add or :remove in trace format command\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();  /* consume :add or :remove */

  /* --- read second argument: g, p, s, o, or '*' --- */
  if (! strcmp (lexeme.string, "g"))
    type_restriction = FOR_GOALS_TF;
  else if (! strcmp (lexeme.string, "p"))
    type_restriction = FOR_PROBLEM_SPACES_TF;
  else if (! strcmp (lexeme.string, "s"))
    type_restriction = FOR_STATES_TF;
  else if (! strcmp (lexeme.string, "o"))
    type_restriction = FOR_OPERATORS_TF;
  else if (! strcmp (lexeme.string, "*"))
    type_restriction = FOR_ANYTHING_TF;
  else {
    print ("Expected g, p, s, o, or * in trace format command\n");
    print_location_of_most_recent_lexeme();
    return FALSE;
  }
  get_lexeme();  /* consume *|g|p|s|o */      

  /* --- read optional name restriction --- */
  if (lexeme.type==SYM_CONSTANT_LEXEME) {
    name_restriction = make_sym_constant (lexeme.string);
    get_lexeme();
  } else {
    name_restriction = NIL;
  }

  /* --- finally, execute the command --- */
  if (remove) {
    remove_trace_format (stack_trace, type_restriction, name_restriction);
  } else {
    if (lexeme.type!=QUOTED_STRING_LEXEME) {
      print ("Expected string in quotes for trace format to add\n");
      print_location_of_most_recent_lexeme();
      if (name_restriction) symbol_remove_ref (name_restriction);
      return FALSE;
    }
    add_trace_format (stack_trace, type_restriction, name_restriction,
                      lexeme.string);
    get_lexeme();
  }
  if (name_restriction) symbol_remove_ref (name_restriction);
  return TRUE;
}

/* -------------------------------------------------------------------
   
                          "Echo" Command

   Syntax:  (echo [echoed argument ...])
------------------------------------------------------------------- */

char *help_on_echo[] = {
"Command: echo",
"",
"Syntax: (echo [echoed argument ...])",
"",
"This command echos its arguments to the screen.",
0 };

bool echo_interface_routine (void) {
  int paren_depth;

  paren_depth = current_lexer_parentheses_level() - 1;
  
  get_lexeme();

  while(lexeme.type != R_PAREN_LEXEME 
        || current_lexer_parentheses_level() != paren_depth) {
    print ("%s", lexeme.string);
    get_lexeme();
  }

  return TRUE;
}

/* ===================================================================
   
                     Built-In Debugging Commands

=================================================================== */

bool print_all_symbols_interface_routine (void) {
  print_all_symbols ();
  get_lexeme();  /* consume "debug:print-all-symbols", advance to rparen */
  return TRUE;
}

/* ===================================================================
   
             Initialize Built-In User Interface Commands

=================================================================== */

void init_built_in_commands (void) {

  add_command ("help", help_interface_routine);
  add_command ("?", help_interface_routine);
  add_help ("help", help_on_help);
  add_help ("?", help_on_help);

  add_command ("list-help-topics", list_help_topics_interface_routine);
  add_help ("list-help-topics", help_on_list_help_topics);

  add_command ("print-all-help", print_all_help_interface_routine);
  add_help ("print-all-help", help_on_print_all_help);

  add_command ("exit", exit_interface_routine);
  add_command ("quit", exit_interface_routine);
  add_help ("exit", help_on_exit);
  add_help ("quit", help_on_exit);

  add_command ("log", log_interface_routine);
  add_help ("log", help_on_log);

  add_command ("load", load_interface_routine);
  add_help ("load", help_on_load);

  add_command ("chdir", chdir_interface_routine);
  add_command ("cd", chdir_interface_routine);
  add_help ("chdir", help_on_chdir);
  add_help ("cd", help_on_chdir);

  add_command ("pwd", pwd_interface_routine);
  add_help ("pwd", help_on_pwd);

  add_command ("pgs", pgs_interface_routine);
  add_help ("pgs", help_on_pgs);

  add_command ("excise", excise_interface_routine);
  add_help ("excise", help_on_excise);
  add_command ("excise-chunks", excise_chunks_interface_routine);
  add_help ("excise-chunks", help_on_excise_chunks);
  add_command ("excise-task", excise_task_interface_routine);
  add_help ("excise-task", help_on_excise_task);
  add_command ("excise-all", excise_all_interface_routine);
  add_help ("excise-all", help_on_excise_all);

  add_command ("matches", matches_interface_routine);
  add_help ("matches", help_on_matches);

  add_command ("default-print-depth", default_print_depth_interface_routine);
  add_help ("default-print-depth", help_on_default_print_depth);
  add_command ("print", print_interface_routine);
  add_command ("p", print_interface_routine);
  add_command ("spr", print_interface_routine);
  add_command ("wm", print_interface_routine);
  add_help ("print", help_on_print);
  add_help ("p", help_on_print);
  add_help ("spr", help_on_print);
  add_help ("wm", help_on_print);

  add_command ("go", go_interface_routine);
  add_help ("go", help_on_go);
  add_command ("d", d_interface_routine);
  add_help ("d", help_on_d);
  add_command ("run", run_interface_routine);
  add_command ("r", run_interface_routine);
  add_help ("run", help_on_run);
  add_help ("r", help_on_run);

  add_command ("init-soar", init_soar_interface_routine);
  add_help ("init-soar", help_on_init_soar);  

  add_command ("learn", learn_interface_routine);
  add_help ("learn", help_on_learn);

  add_command ("chunk-free-problem-spaces",
               chunk_free_problem_spaces_interface_routine);
  add_help ("chunk-free-problem-spaces", help_on_chunk_free_problem_spaces);

  add_command ("watch", watch_interface_routine);
  add_help ("watch", help_on_watch);
  add_help ("watch-keywords", help_on_watch_keywords);
  add_help ("watch-levels", help_on_watch_levels);

  add_command ("preferences", preferences_interface_routine);
  add_help ("preferences", help_on_preferences);

  add_command ("ms", ms_interface_routine);
  add_help ("ms", help_on_ms);

  add_command ("sp", sp_interface_routine);
  add_help ("sp", help_on_sp);

  add_command ("max-elaborations", max_elaborations_interface_routine);
  add_help ("max-elaborations", help_on_max_elaborations);

  add_command ("user-select", user_select_interface_routine);
  add_help ("user-select", help_on_user_select);

  add_command ("soarnews", soarnews_interface_routine);
  add_help ("soarnews", help_on_soarnews);

  add_command ("list-productions", list_productions_interface_routine);
  add_command ("list-chunks", list_productions_interface_routine);
  add_command ("list-justifications", list_productions_interface_routine);
  add_help ("list-productions", help_on_list_productions);
  add_help ("list-chunks", help_on_list_productions);
  add_help ("list-justifications", help_on_list_productions);

  add_command ("add-wme", add_wme_interface_routine);
  add_command ("remove-wme", remove_wme_interface_routine);
  add_help ("add-wme", help_on_add_or_remove_wme);
  add_help ("remove-wme", help_on_add_or_remove_wme);

  add_command ("firing-counts", firing_counts_interface_routine);
  add_help ("firing-counts", help_on_firing_counts);

  add_command ("ptrace", ptrace_interface_routine);
  add_command ("unptrace", unptrace_interface_routine);
  add_help ("ptrace", help_on_ptrace_and_unptrace);
  add_help ("unptrace", help_on_ptrace_and_unptrace);

  add_command ("warnings", warnings_interface_routine);
  add_help ("warnings", help_on_warnings);

  add_command ("time", time_interface_routine);
  add_help ("time", help_on_time);

  add_command ("memory-stats", memory_stats_interface_routine);
  add_help ("memory-stats", help_on_memory_stats);
  add_command ("rete-stats", rete_stats_interface_routine);
  add_help ("rete-stats", help_on_rete_stats);
  add_command ("stats", stats_interface_routine);
  add_command ("print-stats", stats_interface_routine);
  add_help ("stats", help_on_stats);
  add_help ("print-stats", help_on_stats);

  add_command ("object-trace-format", trace_format_interface_routine);
  add_command ("stack-trace-format", trace_format_interface_routine);
  add_help ("object-trace-format", help_on_object_and_stack_traces);
  add_help ("stack-trace-format", help_on_object_and_stack_traces);

  add_command ("echo", echo_interface_routine);
  add_help ("echo", help_on_echo);

  /* --- additional undocumented commands for debugging purposes --- */
  add_command ("debug:print-all-symbols", print_all_symbols_interface_routine);

}
