(in-package "PRODIGY4")

(export '(state goal igoal create-problem current-problem))

;; It is best to have a canonical description for problems, however,
;; we originally did not have this and so problems were just loaded
;; a-la-nolimit with STATE, GOAL, and INSTANTIATION macros in a file.
;; This makes deleting an old problem tricky and makes manipulating
;; the internals of a problem an exercise in reading and writing
;; files.  Some of the old macros are left around and used via calls
;; to EVAL, this is not pretty but it is there.  I would suggest
;; always using the canonical form.

;; LOAD-PROBLEM does the following consistancy checks:

;; No two objects may be given the same name (a symbol).  This is checked.

(defun instantiate-consed-exp (exp)
  (if (eq (car exp) 'user::~)
      (let ((literal (instantiate-consed-literal (second exp))))
	(setf (literal-state-p literal) nil))
      (let ((literal (instantiate-consed-literal exp)))
	(setf (literal-state-p literal) t))))

;; The STATE macro state loads the state into the system.  The state
;; may be a literal, a list of literals or a list of literals whose
;; car is AND.  Anything else will generate an error.

(defmacro state (consed-literals)
  `(prog nil
    (clear-assertion-hash *current-problem-space*)
    ,(cond ((eq (car consed-literals) 'user::and)
	    `(mapcar #'instantiate-consed-exp ',(cdr consed-literals)))
	   ((consp (car consed-literals))
	    `(mapcar #'instantiate-consed-exp ',consed-literals))
	   ((symbolp (car consed-literals))
	    `(instantiate-consed-exp ',consed-literals))
	   (t
	    (error "~&Bad state in ~S.~%" consed-literals)))))

;; Because a prodigy GOAL is used to create a ficticious operator with
;; the goal of the problem goal and the effect of adding the predicate
;; DONE we must allow for the construction of an operator from our
;; goal.  This is to be done by allowing a binding generator to be
;; part of the description of the goal state.

;; This fictious operator is called *FINISH* to maintain consistancy
;; with the rest of the prodigy system.

;; This way goal works whether preconds is  instantantiated literals
;; or it has variables. 
;; for example, we can say (goal (and (on A B) (on B C))), we can also
;; do (goal ((<x> Object) (<y> Object)) (on <x> <y>))
(defmacro goal (&rest preconds)
  (cond ((= 1 (length preconds))
	 `(vgoal nil ,(first preconds)))
	((= 2 (length preconds))
	 `(vgoal ,(first preconds) ,(second preconds)))
	(t
	 (error "Wrong number of subfields in the goal specification
(should be 2 or 3)~%"))))


(defmacro vgoal (binding-generator preconds)
  `(foperator '*FINISH*
    '(user::params ,.(mapcar #'car binding-generator))
    '(user::preconds ,binding-generator ,preconds)
    '(user::effects ();; No variables need defined.
      ((user::add (done))))))

;; IGOAL is a macro that permits instantiated goals.  The more general
;; goal macro is useful when you have a uninstantiated variable in the
;; goal, for example you might want (on A <x>) which puts A on top of
;; some other object, but it doesn't matter which one.  Generally the
;; entire goal is instantiated and so it is more convenient to use the
;; igoal macro.

;;; This has been made obsolete, by allowing "goal" to take both kinds
;;; of syntax by checking the number of its arguments.
(defmacro Igoal (preconds)
  `(goal nil ,preconds))

;; A Prodigy problem should be represented in the below structure.
;; The macro create-problem can be used to build this structure.  To
;; run an experiment over lists of operators you should use this
;; method.  The problem space has the idea of a current-problem (what
;; you solve by typing (run).

(defstruct (problem (:print-function print-problem))
    (name 'no-name)
    (objects nil)
    (goal nil)
    (state nil)
    (plist nil))

(defun print-problem (prob stream z)
  (declare (type problem prob)
	   (stream stream)
	   (ignore z))

  (let ((*standard-output* stream))
    (princ "#<PROB: ")
    (princ (problem-name prob))
    (princ ">")))

(defmacro current-problem ()
  "Returns the current problem and (using setf) sets the problem to a
particular value."
  (declare (special *current-problem-space*))
  '(getf (problem-space-plist *current-problem-space*) :current-problem))

(defmacro create-problem (&rest args)
  "This macro creates a Prodigy PROBLEM lisp object."
    `(make-problem :name ',(second (assoc 'user::name args))
                   :objects ',(cdr (assoc 'user::objects args))
                   :goal ',(assure-goal args)
                   :state ',(assoc 'user::state args)
                   :plist ',(mapcan #'(lambda (x)
					(unless (member (car x)
							'(user::name
							  user::objects
							  user::goal
							  user::state))
					  (copy-list x)))
			     args)))

(defun assure-goal (a-list)
   (cond ((and (assoc 'user::igoal a-list)
	       (assoc 'user::goal a-list))
	  (error "~&Problem ~S has both goal and igoal.~%"))
	 ((assoc 'igoal a-list))
	 (t (assoc 'goal a-list))))

;; This loads the problem into lisp so it can be solved by the
;; planner.  It should only be called by the run routine.
(defun load-problem (problem &optional no-new-abstractions)
  (declare (type problem problem)
	   (special *current-problem-space*))

  (reset-problem-space *current-problem-space*)
  
  (dolist (ob (problem-objects problem))
    (cond ((eq (car ob) 'object-is)
	   (output 2 t "Creating object ~S of type ~S~%"
		   (second ob) (third ob))
	   (eval ob))
	  ((eq (car ob) 'objects-are)
	   (output 2 t "Creating objects ~S of type ~S~%"
		   (butlast (cdr ob)) (car (last ob)))
	   (eval ob))
	  ((= (length ob) 2)
	   (output 2 t "Creating object ~S of type ~S~%"
		   (first ob) (second ob))
	   (eval (cons 'object-is ob)))
	  (t
	   (output 2 t "Creating objects ~S of type ~S~%"
		   (butlast ob) (car (last ob)))
	   (eval (cons 'objects-are ob)))))

  (eval (problem-state problem))
  (eval (problem-goal problem))

  ;; Must also create generators for the *finish* operator, in case it
  ;; has any.
  (let ((finish-op (rule-name-to-rule '*finish* *current-problem-space*)))
    (create-tests-for-operator finish-op)
    (create-generator-for-operator finish-op))

  ;; Set up the abstraction hierarchy for this problem.
  (unless no-new-abstractions ; read my lisp - no new abstractions
    (setf (problem-space-property :abs-hier)
	  (create-hierarchy *current-problem-space* (current-problem)))
    (setf (problem-space-property :control-rule-abstraction-levels)
	  (compute-control-abstraction-levels *current-problem-space*))))


(defun compute-control-abstraction-levels (pspace)
  (declare (type problem-space pspace))
  ;; Yuck!
  (let ((res nil))
    (declare (special res))
    (dolist (slot '(problem-space-select-nodes
		    problem-space-select-goals
		    problem-space-select-operators
		    problem-space-select-bindings
		    problem-space-reject-nodes
		    problem-space-reject-goals
		    problem-space-reject-operators
		    problem-space-reject-bindings
		    problem-space-prefer-nodes
		    problem-space-prefer-goals
		    problem-space-prefer-operators
		    problem-space-prefer-bindings
                    problem-space-apply-or-subgoal))
      (add-in-c-rules pspace slot))
    res))

(defun add-in-c-rules (pspace slot)
  (declare (type problem-space pspace)
	   (symbol slot)
           (special res))
  (dolist (rule (apply slot (list pspace)))
    (push (cons rule (c-rule-abstraction-level (control-rule-if rule) pspace))
	  res)))

(defun c-rule-abstraction-level (exp pspace)
  ;; (format t "~%Abs on ~S" exp)
  (let ((result
         (cond ((not (listp exp)) 0)
               ((eq (car exp) 'user::~)
                (c-rule-abstraction-level (second exp) pspace))
               ((member (car exp) (problem-space-all-preds pspace))
                (c-rule-pred-abs-level exp pspace))
               ((listp (cdr exp))
                (apply #'min
                       (cons 0
                             (mapcar #'(lambda (s)
					 (c-rule-abstraction-level s pspace))
                                     (cdr exp)))))
	       (t 0))))
    ;;(format t "~%  res ~S" result)
    result))

;;; This calculates the abstraction level of a domain-level predicate in
;;; the left hand side of a control rule as the maximum abstraction level
;;; of all predicates with the same first name and the same length in the
;;; problem space. This is because the control rule has no type information.
(defun c-rule-pred-abs-level (exp pspace)
  (or
   (position-if
    #'(lambda (group)
        (member-if #'(lambda (schema)
                       (and (eq (abs-node-name schema) (car exp))
                            (= (length (abs-node-args schema))
                               (length (cdr exp)))))
                   group))
    (getf (problem-space-plist pspace) :abs-hier) :from-end t)
   0))

       

